/* $Id: mat_oifits.c,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oifits.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/

#include <string.h>

#include "mat_const.h"
#include "mat_error.h"
#include "mat_utils.h"
#include "mat_oifits.h"

/*-----------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/

/**
   @ingroup oifits
   @brief This method creates a new mat_oifits structure with all pointers set to NULL.
   @returns Pointer to new allocated mat_oifits structure on success or NULL on failure.
 
   This method creates a new mat_oifits structure and sets all values to the default. For
   pointers the value is NULL for primitive data types the value is set to 0.
*/
mat_oifits* mat_oifits_new(void)
{
    cpl_msg_info(cpl_func,"New oifits");
    mat_oifits* oifits = cpl_calloc(1, sizeof(mat_oifits));
    if(oifits == NULL)
	{
	cpl_msg_error(cpl_func, "could not allocate mat_oifits");
	return NULL;
	}
    return oifits;
}

/**
   @ingroup oifits
   @brief Delete one whole mat_oifits structure (including nested data structures).
   @param oifits   A OI-FITS data structure.
   @return An error code

   This function frees all memory assocciated with one OI-FITS data structure. This includes all referenced
   data structures representing the binary tables and the property list in the primary header. All existing
   references to these structures are invalid after this function.
*/
cpl_error_code mat_oifits_delete(mat_oifits *oifits)
{
    cpl_msg_info(cpl_func,"Delete oifits");
    int i;

    if (oifits == NULL) return CPL_ERROR_NONE;

    if (oifits->keywords != NULL) 
	{
	cpl_propertylist_delete(oifits->keywords);
	oifits->keywords = NULL;
	}
    if (oifits->oitarget != NULL) 
	{
	mat_oitarget_delete(oifits->oitarget);
	oifits->oitarget = NULL;
	}
    /* remove all OI_ARRAY data structures (oifits->array_list) */
    if (oifits->array_list != NULL)
	{
	for (i = 0; i < oifits->nbarray; i++)
	    {
	    if (oifits->array == oifits->array_list[i]) oifits->array = NULL;
	    mat_array_delete(oifits->array_list[i]);
	    }
	cpl_free(oifits->array_list);
	oifits->array_list = NULL;
	oifits->nbarray = 0;
	}
    /* should not happen with the new interface! */
    if (oifits->array != NULL) 
	{
	mat_array_delete(oifits->array);
	oifits->array = NULL;
	}
    /* remove all OI_WAVELENGTH data structures (oifits->wave_list) */
    if (oifits->wave_list != NULL)
	{
	for (i = 0; i < oifits->nbwave; i++)
	    {
	    if (oifits->oiwave == oifits->wave_list[i]) oifits->oiwave = NULL;
	    mat_oiwavelength_delete(oifits->wave_list[i]);
	    }
	cpl_free(oifits->wave_list);
	oifits->wave_list = NULL;
	oifits->nbwave = 0;
	}
    /* should not happen with the new interface! */
    if (oifits->oiwave != NULL) 
	{
	mat_oiwavelength_delete(oifits->oiwave);
	oifits->oiwave = NULL;
	}
    /* remove all OI_WAVELENGTH data structures (oifits->wave_list) */
    if (oifits->spect_list != NULL)
	{
	for (i = 0; i < oifits->nbspect; i++)
	    {
	    if (oifits->oispect == oifits->spect_list[i]) oifits->oispect = NULL;
	    mat_oispectrum_delete(oifits->spect_list[i]);
	    }
	cpl_free(oifits->spect_list);
	oifits->spect_list = NULL;
	oifits->nbspect = 0;
	}

    /* should not happen with the new interface! */
    if (oifits->oispect != NULL)
	{
	mat_oispectrum_delete(oifits->oispect);
	oifits->oispect = NULL;
	}
    /* we have to remove the remaining data structures and all tupels */
    if (oifits->tupel_list != NULL)
	{
	for (i = 0; i < oifits->nbtupel; i++)
	    {
	    mat_oifits_tupel *tupel = oifits->tupel_list[i];
	    cpl_free(tupel->arrname);
	    cpl_free(tupel->insname);
	    if (tupel->oivis != NULL)
		{
		if (oifits->oivis == tupel->oivis) oifits->oivis = NULL;
		mat_oivis_delete(tupel->oivis);
		}
	    if (tupel->oivis2 != NULL)
		{
		if (oifits->oivis2 == tupel->oivis2) oifits->oivis2 = NULL;
		mat_oivis2_delete(tupel->oivis2);
		}
	    if (tupel->oit3 != NULL)
		{
		if (oifits->oit3 == tupel->oit3) oifits->oit3 = NULL;
		mat_oit3_delete(tupel->oit3);
		}
	    if (tupel->oitf2 != NULL)
		{
		if (oifits->oitf2 == tupel->oitf2) oifits->oitf2 = NULL;
		mat_oitf2_delete_nt(tupel->oitf2);
		}
	    cpl_free(tupel);
	    }
	cpl_free(oifits->tupel_list);
	oifits->tupel_list = NULL;
	oifits->nbtupel = 0;
	oifits->sel_tupel = 0;
	}
    /* should not happen with the new interface! */
    if (oifits->oivis != NULL)
	{
	mat_oivis_delete(oifits->oivis);
	oifits->oivis = NULL;
	}
    if (oifits->oivis2 != NULL)
	{
	mat_oivis2_delete(oifits->oivis2);
	oifits->oivis2 = NULL;
	}
    if (oifits->oit3 != NULL)
	{
	mat_oit3_delete(oifits->oit3);
	oifits->oit3 = NULL;
	}
    if (oifits->oitf2 != NULL)
	{
	mat_oitf2_delete_nt(oifits->oitf2);
	oifits->oitf2 = NULL;
	}
    cpl_free(oifits);
    return cpl_error_get_code();
}

/**
   @ingroup oifits
   @brief Frees the complete memory of a vector of mat_oifits structures
   @param oifits Vector of pointer to the mat_oifits structure.
   @param nbfits Number of elements in the vector.
 
   This method frees the complete memory of a vector of the mat_oifits structures by calling
   the delete method on every element.
*/
cpl_error_code mat_oifits_list_delete(mat_oifits **oifits, int nbfits)
{
    cpl_msg_info(cpl_func,"Delete oifits list");
    int i = 0;

    if(oifits == NULL) return CPL_ERROR_NONE;

    for(i = 0; i < nbfits; ++i)
	{
	if (oifits[i] != NULL)
	    {
	    mat_oifits_delete(oifits[i]);
	    oifits[i] = NULL;
	    }
	}
    cpl_free(oifits);
    return CPL_ERROR_NONE;
}

/**
 * @ingroup oifits
 * @brief Cleanup the list of tuples by removing empty entries (no binary table).
 * @param oifits Contains an OI-FITS file.
 *
 * All empty tuples (data structure mat_oifits_tupel in the mat_oifits->tupel_list)
 * are removed from the list. If such a tupel is the current selected tupel, the
 * next available tupel is automatically selected.
 */
static void mat_oifits_cleanup_tupel_list(mat_oifits *oifits)
{
    cpl_msg_info(cpl_func,"Cleanup tuple list");
    int     i, j;

    for (i = 0; i < oifits->nbtupel; i++)
	{
	if (oifits->tupel_list[i]->oivis != NULL) continue;
	if (oifits->tupel_list[i]->oivis2 != NULL) continue;
	if (oifits->tupel_list[i]->oit3 != NULL) continue;
	if (oifits->tupel_list[i]->oitf2 != NULL) continue;
	/* all tables are empty -> remove this tupel from the list */
	cpl_free(oifits->tupel_list[i]->arrname);
	cpl_free(oifits->tupel_list[i]->insname);
	cpl_free(oifits->tupel_list[i]);
	for (j = 0; j < oifits->nbtupel - 1; j++)
	    {
	    oifits->tupel_list[j] = oifits->tupel_list[j + 1];
	    }
	oifits->tupel_list[oifits->nbtupel - 1] = NULL;
	oifits->nbtupel--;
	if (oifits->sel_tupel == i)
	    {
	    mat_oifits_select_tupel(oifits, i + 1);
	    }
	else if (oifits->sel_tupel > i)
	    {
	    oifits->sel_tupel--;
	    }
	}
}

/**
   @ingroup oifits
   @brief Load all information of an OI-FITS file and store it as internal data structures.
   @param frame  This frame represents the OI-FITS file (by name).
   @return A allocated and filled mat_oifits data structure or NULL.

   This function creates a new mat_oifits data structure and loads the property list from the primary header.
   All binary tables which have a representation as a MATISSE data structure (for example mat_oivis2) are loaded
   and transferred into the internal data structure. The ordering of the binary tables does not matter. A mat_oifits
   data structure can contain several mat_oivis2, mat_oit3 and so on data structures (as defined in the OI-FITS standard
   document). They are organized as mat_oifits_tupel (OI_VIS, OI_VIS2, OI_T3 and TF2) or lists
   (OI_ARRAY, OI_WAVELENGTH and OI_FLUX).
*/
mat_oifits *mat_oifits_load(cpl_frame *frame)
{
    cpl_msg_info(cpl_func,"Load oifits");
    mat_oifits         *oifits;
    const char         *fname;
    int                 nbextent;
    int                 i;
    const char         *extname;
    cpl_propertylist   *plist;
    cpl_table          *table;

    mat_assert_value((frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,
		     "no cpl_frame (frame) argument given");
    oifits = (mat_oifits *)cpl_calloc(1, sizeof(mat_oifits));
    if (oifits == NULL)
	{
	cpl_msg_error(cpl_func,"could not allocat int : e memory for mat_oifits");
	return NULL;
	}
    // load the primary keywords (always)
    oifits->keywords = cpl_propertylist_load(cpl_frame_get_filename(frame), 0);
    // we load everything which is in the input file and listed in the mat_oifits data structure
    fname = cpl_frame_get_filename(frame);
    nbextent = cpl_frame_get_nextensions(frame);
    for(i = 0; i < nbextent; i++)
	{
	plist = cpl_propertylist_load(fname, i + 1);
	extname = cpl_propertylist_get_string(plist,"EXTNAME");
	if (extname == NULL)
	    {
	    cpl_propertylist_delete(plist);
	    continue;
	    }
	if (!strcmp(extname,"OI_TARGET"))
	    {
	    table = cpl_table_load(fname, i + 1, 0);
	    oifits->oitarget = mat_oitarget_from_table(plist, table);
	    cpl_table_delete(table);
	    }
	if (!strcmp(extname,"OI_ARRAY") || !strcmp(extname,"ARRAY_GEOMETRY"))
	    {
	    table = cpl_table_load(fname, i + 1, 0);
	    mat_oifits_add_array(oifits, mat_array_from_table(plist, table));

	    cpl_table_delete(table);
	    }
	if (!strcmp(extname,"OI_WAVELENGTH"))
	    {
	    table = cpl_table_load(fname, i + 1, 0);
	    mat_oifits_add_wavelength(oifits, mat_oiwavelength_from_table(plist, table));
	    cpl_table_delete(table);
	    }
	if (!strcmp(extname,"OI_FLUX"))
	    {
	    table = cpl_table_load(fname, i + 1, 0);
	    mat_oifits_add_spectrum(oifits, mat_oispectrum_from_table(plist, table));
	    cpl_table_delete(table);
	    }
	if (!strcmp(extname,"OI_VIS"))
	    {
	    table = cpl_table_load(fname, i + 1, 0);
	    mat_oifits_add_vis(oifits, mat_oivis_from_table(plist, table));
	    //printf("oifits0 oivis, oifits->nbtupel %d\n",oifits->nbtupel );
	    cpl_table_delete(table);
	    }
	if (!strcmp(extname,"OI_VIS2"))
	    {
	    table = cpl_table_load(fname, i + 1, 0);
	    mat_oifits_add_vis2(oifits, mat_oivis2_from_table(plist, table));
	    //printf("oifits0 oivis2, oifits->nbtupel %d\n",oifits->nbtupel );//nbtupel = 1 ICI
	    cpl_table_delete(table);
	    }
	if (!strcmp(extname,"OI_T3"))
	    {
	    table = cpl_table_load(fname, i + 1, 0);
	    mat_oifits_add_t3(oifits, mat_oit3_from_table(plist, table));
	    cpl_table_delete(table);
	    }
	if (!strcmp(extname,"TF2"))//interptf2?
	    {
	    table = cpl_table_load(fname, i + 1, 0);
	    mat_oifits_add_tf2(oifits, mat_oitf2_from_table(plist, table));
	    cpl_table_delete(table);
	    }
	cpl_propertylist_delete(plist);
	} 
    return oifits;
}

/**
   @ingroup oifits
   @brief Save a mat_oifits structure in a FITS file.
   @param oifits            data structure to save
   @param frameset          input frameset
   @param usedframes        selected frameset
   @param parlist           recipe parameter list
   @param recipe_name       name of recipe
   @param filename          name of the FITS file
   @param type              1:RAW_VIS2, 2:RAW_CPHASE, 3:RAW_SPECTRUM,
   4:RAW_DPHASE, 5: TARGET_RAW_INT 6: CAL_VIS2 
   7: CAL_CPHASE 8:CAL_DPHASE 9: TARGET_CAL_INT
   @return cpl_error_code

   This function stores a mat_oifits data structure and the related binary tables in one OI-FITS file.
   The OI_TARGET binary table and the property list for the primary header are stored directly
   from this function. All other tables are stored using the mat_*_append function of the data structures.
   The type parameter specifies which tables are stored in the file:

   <table>
   <tr>
   <th>type</th>
   <th>OI_VIS</th>
   <th>OI_VIS2</th>
   <th>OI_T3</th>
   <th>TF2</th>
   <th>OI_FLUX</th>
   </tr>
   <tr><td>0</td>               <td>Yes</td><td>Yes</td><td>Yes</td><td>Yes</td><td>Yes</td></tr>
   <tr><td>1:RAW_VIS2</td>      <td>No </td><td>Yes</td><td>No </td><td>Yes</td><td>No </td></tr>
   <tr><td>2:RAW_CPHASE</td>    <td>No </td><td>No </td><td>Yes</td><td>Yes</td><td>No </td></tr>
   <tr><td>3:RAW_SPECTRUM</td>  <td>No </td><td>No </td><td>No </td><td>Yes</td><td>Yes</td></tr>
   <tr><td>4:RAW_DPHASE</td>    <td>Yes</td><td>No </td><td>No </td><td>Yes</td><td>No </td></tr>
   <tr><td>5:TARGET_RAW_INT</td><td>Yes</td><td>Yes</td><td>Yes</td><td>Yes</td><td>Yes</td></tr>
   <tr><td>6:CAL_VIS2</td>      <td>No </td><td>Yes</td><td>No </td><td>Yes</td><td>No </td></tr>
   <tr><td>7:CAL_CPHASE</td>    <td>No </td><td>No </td><td>Yes</td><td>Yes</td><td>No </td></tr>
   <tr><td>8:CAL_DPHASE</td>    <td>Yes</td><td>No </td><td>No </td><td>Yes</td><td>No </td></tr>
   <tr><td>9:TARGET_CAL_INT</td><td>Yes</td><td>Yes</td><td>Yes</td><td>Yes</td><td>Yes</td></tr>
   <tr><td>10:RAW_TF2</td>      <td>No </td><td>No </td><td>No </td><td>Yes</td><td>No </td></tr>
   <tr><td>11:CAL_RAW_INT</td>  <td>Yes</td><td>Yes</td><td>Yes</td><td>Yes</td><td>Yes</td></tr>
   <tr><td>12:INTERP_TF2</td>   <td>No </td><td>No </td><td>No </td><td>Yes</td><td>No </td></tr>
   </table>

   The binary tables OI_ARRAY and OI_WAVELENGTH are always stored in a OI-FITS file.
   The OI_TARGET binary table is added to all OI-FITS files except for the RAW_TF2 and INTERP_TF2 files.
   Because an OI-FITS file can contain several instances of some binary tables, an individual
   EXTVER value is assigned for each binary table instance of a specific binary table type.

   If type is specified as 0, all binary tables which do exist in the mat_oifits data structure are stored in
   an OI_FITS file. In this case, the property list must contain the "PRO CATG" and "PRO TECH" keywords.
*/
cpl_error_code  mat_oifits_save(mat_oifits *oifits,
				cpl_frameset *frameset,
				cpl_frameset *usedframes,
				const cpl_parameterlist *parlist,
				const char *recipe_name,
				char *filename,
				int type)
{
    cpl_msg_info(cpl_func,"Save oifits");
    cpl_propertylist *keyTab = NULL;
    int               i, j;
    cpl_error_code    ec;
    cpl_errorstate    prestate = cpl_errorstate_get();

    //cpl_msg_info(cpl_func,"bla");

    /*Check input parameters*/
    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((frameset!=NULL), CPL_ERROR_NULL_INPUT, 
		     CPL_ERROR_NULL_INPUT, 
		     "no cpl_frameset (frameset) argument given");
    mat_assert_value((usedframes!=NULL), CPL_ERROR_NULL_INPUT, 
		     CPL_ERROR_NULL_INPUT, 
		     "no cpl_frameset (usedframes) argument given");
    mat_assert_value((filename!=NULL), CPL_ERROR_NULL_INPUT, 
		     CPL_ERROR_NULL_INPUT, "no filename given");
    mat_assert_value((recipe_name!=NULL), CPL_ERROR_NULL_INPUT, 
		     CPL_ERROR_NULL_INPUT, "no recipe_name given");

    //cpl_msg_info(cpl_func,"bla..");

    keyTab=cpl_propertylist_new();
    cpl_propertylist_erase(oifits->keywords,"RADECSYS");
    
    /*Remove the keywords from the DPR categories*/
    if (type != 0)
      {
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_TECH, "IMAGE");
	cpl_propertylist_erase_regexp(oifits->keywords, "DPR", 0);
	if ( cpl_propertylist_has(oifits->keywords,"CONTENT") )
	  {
	    cpl_propertylist_erase(oifits->keywords,"CONTENT");
	  }
	cpl_propertylist_append_string(oifits->keywords, "CONTENT","OIFITS2");
	cpl_propertylist_append_string(oifits->keywords, "INSMODE","HYBRID");
      }
    switch (type)
	{
    case 0:
	/* the "PRO CATG" and "PRO TECH" must have their value set in the property list */
	break;
    case 1:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "RAW_VIS2");
	break;
    case 2:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "RAW_CPHASE");
	break;
    case 3:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "RAW_SPECTRUM");
	break;
    case 4:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "RAW_DPHASE");
	break;
    case 5:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "TARGET_RAW_INT");
	break;
    case 6:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "CAL_VIS2");
	break;
    case 7:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "CAL_CPHASE");
	break;
    case 8:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "CAL_DPHASE");
	break;
    case 9:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "TARGET_CAL_INT");
	break;
    case 10:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "RAW_TF2");
	break;
    case 11:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "CALIB_RAW_INT");
	break;
    case 12:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "INTERP_TF2");
	break;
    default:
	cpl_propertylist_update_string(oifits->keywords, CPL_DFS_PRO_CATG, "RAW_OIFITS");
	cpl_msg_warning(cpl_func,"This type is not allowed. We put RAW_OIFITS as PRO_CATG");
	break;
    }
    /* update the EXTVER values for all tables */
    /* needed for the old interface of mat_oifits */
    if (oifits->array != NULL)
	{
	oifits->array->extver = 1;
	}
    if (oifits->oiwave != NULL)
	{
	oifits->oiwave->extver = 1;
	}
    if (oifits->oispect != NULL)
	{
	oifits->oispect->extver = 1;
	}
    if (oifits->oivis != NULL)
	{
	oifits->oivis->extver = 1;
	}
    if (oifits->oivis2 != NULL)
	{
	oifits->oivis2->extver = 1;
	}
    if (oifits->oit3 != NULL)
	{
	oifits->oit3->extver = 1;
	}
    if (oifits->oitf2 != NULL)
	{
	oifits->oitf2->extver = 1;
	}
    //cpl_msg_info(cpl_func,"bla.../");
    /* needed for the new interface of mat_oifits */
    for (i = 0; i < oifits->nbarray; i++)
	{
	oifits->array_list[i]->extver = (i + 1);
	}
    for (i = 0; i < oifits->nbwave; i++)
	{
	oifits->wave_list[i]->extver = (i + 1);
	}
    for (i = 0; i < oifits->nbspect; i++)
	{
	oifits->spect_list[i]->extver = (i + 1);
	}
    for (i = 0; i < oifits->nbtupel; i++)
	{
	mat_oifits_tupel *tupel = oifits->tupel_list[i];
	if (tupel->oivis != NULL)
	    {
	    tupel->oivis->extver = (i + 1);
	    }
	if (tupel->oivis2 != NULL)
	    {
	    tupel->oivis2->extver = (i + 1);
	    }
	if (tupel->oit3 != NULL)
	    {
	    tupel->oit3->extver = (i + 1);
	    }
	if (tupel->oitf2 != NULL)
	    {
	    tupel->oitf2->extver = (i + 1);
	    }
	}

    /* Create the OI-FITS file using only the property list, the data part will be empty. */
    if (cpl_propertylist_has(oifits->keywords,"ESO DRS BCD COMBINED"))
      {
	if (cpl_propertylist_get_bool(oifits->keywords,"ESO DRS BCD COMBINED") != 0)
	  {
	    ec = cpl_dfs_save_propertylist(frameset, NULL, parlist, usedframes, NULL,
					   recipe_name, oifits->keywords,
					   "^ESO CFG", PACKAGE "/" PACKAGE_VERSION, filename);
 	  }
	else
	  {
	    ec = cpl_dfs_save_propertylist(frameset, NULL, parlist, usedframes, NULL,
					   recipe_name, oifits->keywords,
					   NULL, PACKAGE "/" PACKAGE_VERSION, filename);
	  }
      }
    else
      {
	ec = cpl_dfs_save_propertylist(frameset, NULL, parlist, usedframes, NULL,
				       recipe_name, oifits->keywords,
				       NULL, PACKAGE "/" PACKAGE_VERSION, filename);
      }
	
   if (ec != CPL_ERROR_NONE)
    	{
    	cpl_msg_error(cpl_func,
    		      "An error occurred during property list writing: %s",
    		      cpl_error_get_message_default(ec));
    	cpl_propertylist_delete(keyTab);
    	return CPL_ERROR_UNSPECIFIED;
    	}
    /* If the OI-FITS file does not contain a transfer function, add the OI_TARGET binary table. */
    if ((type != 10) && (type != 12))
    	{
    	mat_oitarget_append(filename, oifits->oitarget);
    	}

    /* Append all OI_ARRAY binary tables to the OI-FITS file. */
    if (oifits->nbarray != 0)
    	{ // new interface
    	for (i = 0; i < oifits->nbarray; i++)
    	    {
    	    mat_array_append(filename, oifits->array_list[i]);
    	    }
    	}
    else if (oifits->array != NULL)
    	{ // old interface
    	mat_array_append(filename, oifits->array);
    	}

    /* Append all OI_WAVELENGTH binary tables to the OI_FITS file. */
    if (oifits->nbwave != 0)
    	{ // new interface
    	for (i = 0; i < oifits->nbwave; i++)
    	    {
    	    mat_oiwavelength_append(filename, oifits->wave_list[i]);
    	    }
    	}
    else if (oifits->oiwave != NULL)
    	{ // old interface
    	mat_oiwavelength_append(filename,oifits->oiwave);
    	}

    /* Append all binary tables with interferometric data to the OI_FITS file. */
    if (oifits->nbtupel != 0)
    	{ // new interface
    	for (i = 0; i < oifits->nbtupel; i++)
    	    {
    	    mat_oifits_tupel *tupel = oifits->tupel_list[i];
    	    if ((tupel->oivis2 != NULL) && (type == 0 || type == 1 || type == 5 || type == 6 || type == 9))
    		{
    		mat_oivis2_append(filename, tupel->oivis2);
    		}
    	    if ((tupel->oit3 != NULL) && (type == 0 || type == 2 || type == 5 || type == 7 || type == 9))
    		{
    		mat_oit3_append(filename, tupel->oit3);
    		}
    	    if ((tupel->oivis != NULL) && (type == 0 || type == 4 || type == 5 || type == 8 || type == 9))
    		{
    		mat_oivis_append(filename, tupel->oivis);
    		}
    	    if (tupel->oitf2 != NULL)
    		{
    		mat_oitf2_append_nt(filename, tupel->oitf2);
    		}
    	    }
    	}
    else
    	{
    	if ((oifits->oivis2 != NULL) && (type == 0 || type == 1 || type == 5 || type == 6 || type == 9 || type == 11))
    	    {
    	    mat_oivis2_append(filename,oifits->oivis2);
    	    }
    	if ((oifits->oit3 != NULL) && (type == 0 || type == 2 || type == 5 || type == 7 || type == 9 || type == 11))
    	    {
    	    mat_oit3_append(filename,oifits->oit3);
    	    }
    	if ((oifits->oivis != NULL) && (type == 0 || type == 4 || type == 5 || type == 8 || type == 9 || type == 11))
    	    {
    	    mat_oivis_append(filename,oifits->oivis);
    	    }
    	if (oifits->oitf2 != NULL )
    	    {
    	    mat_oitf2_append_nt(filename, oifits->oitf2);
    	    }
    	}

    /* Append all OI_FLUX binary tables to the OI_FITS file. */
    if (type == 0 || type == 3 || type == 5 || type == 9 || type == 11)
    	{
    	if (oifits->nbspect != 0)
    	    { // new interface
    	    for (i = 0; i < oifits->nbspect; i++)
    		{
    		for (j = 0; j < oifits->nbwave; j++)
    		    {
    		    if (strcmp(oifits->spect_list[i]->insname, oifits->wave_list[j]->insname) == 0)
    			{
			
    			mat_oispectrum_append(filename, oifits->spect_list[i]);
    			break;
    			}
    		    }
    		}
    	    }
    	else if ((oifits->oispect != NULL) && (oifits->oiwave != NULL))
    	    { // old interface
    	    mat_oispectrum_append(filename,oifits->oispect);
    	    }
    	}

    //cpl_msg_info(cpl_func,"bla......");
    /*Free memory*/
    cpl_propertylist_delete(keyTab);
 
    if (!cpl_errorstate_is_equal(prestate))
	{
	cpl_errorstate_set(prestate);
	return CPL_ERROR_UNSPECIFIED;
	} 
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Searches (optionally creates) a tupel for one ARRNAME/INSNAME combination.
   @param oifits  Contains a OI-FITS file.
   @param arrname The array name (keyword ARRNAME) in the extension.
   @param insname The instrument name (keyword INSNAME) in the extension.
   @param min_mjd The beginning of a data set as MJD.
   @param max_mjd The end of a data set as MJD.
   @param cflag Flag if the tupel should be created (1) or not (0).
   @returns The index of the tupel or -1.

   Searches for a mat_oifits_tupel data structure for a given ARRNAME/INSNAME combination
   and a given observation time interval (min_mjd and max_mjd).
   If it is found, the index is returned, otherwise -1 is returned. The cflag parameter
   allows it to create the specified tupel even if it does not exist.

   The found tupel is not selected. It must be selected using mat_oifits_select_tupel.
*/
int mat_oifits_find_tupel(mat_oifits *oifits,
			  const char *arrname,
			  const char *insname,
			  double min_mjd,
			  double max_mjd,
			  int cflag)
{
    cpl_msg_info(cpl_func,"Find tuple");
    int               i;
    
    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, -1,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((arrname!=NULL), CPL_ERROR_NULL_INPUT, -1,
		     " no array name (arrname) argument given");
    mat_assert_value((insname!=NULL), CPL_ERROR_NULL_INPUT, -1,
		     " no instrument name (insname) argument given");
    //printf("nbtupel %d\n",oifits->nbtupel);
    for (i = 0; i < oifits->nbtupel; i++)
	{
	if (strcmp(oifits->tupel_list[i]->arrname, arrname) != 0) continue;
	if (strcmp(oifits->tupel_list[i]->insname, insname) != 0) continue;
	//if ((min_mjd != 0) && (fabs(oifits->tupel_list[i]->min_mjd - min_mjd) > 1.0/24.0)) continue;
	//if ((max_mjd != 0) && (fabs(oifits->tupel_list[i]->max_mjd - max_mjd) > 1.0/24.0)) continue;
	if (CPL_MAX(min_mjd, oifits->tupel_list[i]->min_mjd) - CPL_MIN(max_mjd, oifits->tupel_list[i]->max_mjd) > 1.E-6 ) continue;
	//if (CPL_MAX(min_mjd, oifits->tupel_list[i]->min_mjd) - CPL_MIN(max_mjd, oifits->tupel_list[i]->max_mjd) < -1.E-6 ) continue;
	return i;
	}
    /* we did not find the tupel -> create a new one if cflag == 1 */
    if (cflag)
	{
	mat_oifits_tupel *tupel;

	oifits->tupel_list = cpl_realloc(oifits->tupel_list, (oifits->nbtupel + 1)*sizeof(mat_oifits_tupel *));
	mat_assert_not_null(oifits->tupel_list, "cannot extend the list of OI-FITS tupels, fatal");
	oifits->tupel_list[oifits->nbtupel] = NULL; /* initialize it (important for deleting the data structure) */
	tupel = cpl_calloc(1, sizeof(mat_oifits_tupel));
	mat_assert_value((tupel!=NULL), CPL_ERROR_NULL_INPUT, -1,
			 "cannot allocate memory for a new OI-FITS tupel, fatal");
	tupel->arrname = cpl_strdup(arrname);
	tupel->insname = cpl_strdup(insname);
	tupel->min_mjd = min_mjd;
	tupel->max_mjd = max_mjd;
	oifits->tupel_list[oifits->nbtupel] = tupel;
	oifits->nbtupel++;
	return (oifits->nbtupel - 1);
	}
    return -1;
}

/**
   @ingroup oifits
   @brief Selects a specific tupel (combination of ARRNAME/INSNAME).
   @param oifits Contains a OI-FITS file.
   @param itupel Index of a tupel.
   @returns An error code.

   A specific tupel (combination of ARRNAME/INSNAME) is selected. If the parameter itupel is negative,
   the first tupel in the list is selected. If the itupel parameter is greater than the number of tupels
   minus one, the last tupel in the list is selected. The pointers to the data structures corresponding
   with OI_ARRAY, OI_WAVELENGTH, OI_FLUX, OI_VIS, OI_VIS2, OI_T3 and TF2 are put into elements of
   the mat_oifits data structure and they can be easily used with a direct access.
*/
cpl_error_code mat_oifits_select_tupel(mat_oifits *oifits,
				       int itupel)
{
    cpl_msg_info(cpl_func,"Select tuple");
    mat_oifits_tupel *tupel;
    int               i;

    /*Check input parameters*/
    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    if (oifits->nbtupel == 0)
	{ /* we do not have any tupel available -> set all pointers to NULL */
	oifits->sel_tupel = 0;
	if (oifits->array_list != NULL)
	    { /* the new interface was used to add an OI_ARRAY
	       * -> we can set the pointer to NULL without a memory leak
	       */
	    oifits->array = NULL;
	    }
	if (oifits->wave_list != NULL)
	    { /* the new interface was used to add an OI_WAVELENGTH
	       * -> we can set the pointer to NULL without a memory leak
	       */
	    oifits->oiwave = NULL;
	    }
	if (oifits->spect_list != NULL)
	    { /* the new interface was used to add an OI_FLUX
	       * -> we can set the pointer to NULL without a memory leak
	       */
	    oifits->oispect = NULL;
	    }
	if (oifits->tupel_list != NULL)
	    { /* the new interface was used to add OI_VIS, OI_VIS2, OI_T3 or an TF2 binary table
	       * -> we can set the pointer to NULL without a memory leak
	       */
	    oifits->oivis  = NULL;
	    oifits->oivis2 = NULL;
	    oifits->oit3   = NULL;
	    oifits->oitf2  = NULL;
	    }
	return CPL_ERROR_NONE;
	}
    if (itupel < 0) itupel = 0;
    if (itupel >= oifits->nbtupel) itupel = oifits->nbtupel - 1;
    oifits->sel_tupel = itupel;
    tupel = oifits->tupel_list[itupel];
    /* select the OI_ARRAY table associated with this tupel */
    oifits->array = NULL;
    for (i = 0; i < oifits->nbarray; i++)
	{
	if (strcmp(oifits->array_list[i]->arrayname, tupel->arrname) == 0)
	    {
	    oifits->array = oifits->array_list[i];
	    break;
	    }
	}
    /* select the OI_WAVELENGTH table associated with this tupel */
    oifits->oiwave = NULL;
    for (i = 0; i < oifits->nbwave; i++)
	{
	if (strcmp(oifits->wave_list[i]->insname, tupel->insname) == 0)
	    {
	    oifits->oiwave = oifits->wave_list[i];
	    break;
	    }
	}
    /* select the OI_FLUX table associated with this tupel */
    oifits->oispect = NULL;
    for (i = 0; i < oifits->nbspect; i++)
	{
	if (strcmp(oifits->spect_list[i]->insname, tupel->insname) == 0)
	    {
	    oifits->oispect = oifits->spect_list[i];
	    break;
	    }
	}
    /* set all remaining OI_* tables */
    oifits->oivis  = tupel->oivis;
    oifits->oivis2 = tupel->oivis2;
    oifits->oit3   = tupel->oit3;
    oifits->oitf2  = tupel->oitf2;
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Adds an OI_TARGET to an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param target  Contains an OI_TARGET binary table as mat_oitarget data structure.
   @returns An error code.

   An OI_TARGET binary table is added to an OI_FITS file.
   If another OI_TARGET binary table exists for this OI_FITS file, it is deleted.
*/
cpl_error_code mat_oifits_add_target(mat_oifits *oifits, mat_oitarget *target)
{
    cpl_msg_info(cpl_func,"Add target");
    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((target!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oitarget (target) argument given");
    if (oifits->oitarget != NULL)
	{
	mat_oitarget_delete(oifits->oitarget);
	}
    oifits->oitarget = target;
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Adds an OI_ARRAY to an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param array  Contains an OI_ARRAY binary table as mat_array data structure.
   @returns An error code.

   An OI_ARRAY binary table is added to the list of OI_ARRAYs of an OI_FITS file.
   It is not checked if another OI_ARRAY with the same ARRNAME already exists.
   If it is the first table of an OI_FITS file or it corresponds to the current tupel
   (ARRNAME of the OI_ARRAY table and ARRNAME of the current tupel is equal), the
   OI_ARRAY is set as current OI_ARRAY and can directly accessed.
*/
cpl_error_code mat_oifits_add_array(mat_oifits *oifits, mat_array *array)
{
    cpl_msg_info(cpl_func,"Add array");
    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((array!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_array (array) argument given");
    mat_assert_value(!((oifits->array_list == NULL) && (oifits->array != NULL)), CPL_ERROR_UNSPECIFIED, CPL_ERROR_UNSPECIFIED,
		     " there exists an OI_ARRAY binary table which was directly set but not added");
    /* extend the current list and add the new element at the end */
    oifits->array_list = cpl_realloc(oifits->array_list, (oifits->nbarray + 1)*sizeof(mat_array *));
    mat_assert_not_null(oifits->array_list, "cannot extend the list of OI_ARRAY, fatal");
    oifits->array_list[oifits->nbarray] = array;
    oifits->nbarray++;
    if (oifits->nbtupel == 0)
	{ /* we have no tupel */
	oifits->array = array;
	}
    else if (strcmp(oifits->tupel_list[oifits->sel_tupel]->arrname, array->arrayname) == 0)
	{ /* the spectrum table is related to the current tupel */
	oifits->array = array;
	}
    //return mat_oifits_select_tupel(oifits, oifits->sel_tupel);
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an OI_ARRAY from an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param array  Contains an OI_ARRAY binary table as mat_array data structure.
   @returns An error code.

   An OI_ARRAY binary table is removed from the list of OI_ARRAYs of an OI_FITS file.
   The data structure of the OI_ARRAY binary table is not deleted!
*/
cpl_error_code mat_oifits_remove_array(mat_oifits *oifits, mat_array *array)
{
    int     i, j;
    cpl_msg_info(cpl_func,"Remove array");

    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((array!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_array (array) argument given");
    mat_assert_value(!((oifits->array_list == NULL) && (oifits->array != NULL)), CPL_ERROR_UNSPECIFIED, CPL_ERROR_UNSPECIFIED,
		     " there exists an OI_ARRAY binary table which was directly set but not added");
    /* remove the element from the list */
    for (i = 0; i < oifits->nbarray; i++)
	{
	if (oifits->array_list[i] == array)
	    {
	    for (j = i; j < oifits->nbarray - 1; j++)
		{
		oifits->array_list[j] = oifits->array_list[j + 1];
		}
	    oifits->array_list[oifits->nbarray - 1] = NULL;
	    break;
	    }
	}
    oifits->nbarray--;
    return mat_oifits_select_tupel(oifits, oifits->sel_tupel);
}

/**
   @ingroup oifits
   @brief Removes an OI_ARRAY from an OI_FITS file and delete it.
   @param oifits Contains an OI-FITS file.
   @param array  Contains an OI_ARRAY binary table as mat_array data structure.
   @returns An error code.

   An OI_ARRAY binary table is removed from the list of OI_ARRAYs of an OI_FITS file.
   The OI_ARRAY data structure is deleted.
*/
cpl_error_code mat_oifits_delete_array(mat_oifits *oifits, mat_array *array)
{
    cpl_msg_info(cpl_func,"Delete array");
    cpl_error_code ec = mat_oifits_remove_array(oifits, array);
    if (ec != CPL_ERROR_NONE) return ec;
    return mat_array_delete(array);
}

/**
   @ingroup oifits
   @brief Adds an OI_WAVELENGTH to an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param wave  Contains an OI_WAVELENGTH binary table as mat_oiwavelength data structure.
   @returns An error code.

   An OI_WAVELENGTH binary table is added to the list of OI_WAVELENGTHs of an OI_FITS file.
   It is not checked if another OI_WAVELENGTH with the same INSNAME already exists.
   If it is the first table of an OI_FITS file or it corresponds to the current tupel
   (INSNAME of the OI_WAVELENGTH table and INSNAME of the current tupel is equal), the
   OI_WAVELENGTH is set as current OI_WAVELENGTH and can directly accessed.
*/
cpl_error_code mat_oifits_add_wavelength(mat_oifits *oifits, mat_oiwavelength *wave)
{
    cpl_msg_info(cpl_func,"Add wavelength");
    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((wave!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oiwavelength (wave) argument given");
    /* extend the current list and add the new element at the end */
    oifits->wave_list = cpl_realloc(oifits->wave_list, (oifits->nbwave + 1)*sizeof(mat_oiwavelength *));
    mat_assert_not_null(oifits->wave_list, "cannot extend the list of OI_WAVELENGTH, fatal");
    oifits->wave_list[oifits->nbwave] = wave;
    oifits->nbwave++;
    if (oifits->nbtupel == 0)
	{ /* we have no tupel */
	oifits->oiwave = wave;
	}
    else if (strcmp(oifits->tupel_list[oifits->sel_tupel]->insname, wave->insname) == 0)
	{ /* the wavelength table is related to the current tupel */
	oifits->oiwave = wave;
	}
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an OI_WAVELENGTH from an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param wave  Contains an OI_WAVELENGTH binary table as mat_oiwavelength data structure.
   @returns An error code.

   An OI_WAVELENGTH binary table is removed from the list of OI_WAVELENGTHs of an OI_FITS file.
   The data structure of the OI_WAVELENGTH binary table is not deleted!
*/
cpl_error_code mat_oifits_remove_wavelength(mat_oifits *oifits, mat_oiwavelength *wave)
{
    int     i, j;
    cpl_msg_info(cpl_func,"Remove wavelength");

    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((wave!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oiwave (wave) argument given");
    /* remove the element from the list */
    for (i = 0; i < oifits->nbwave; i++)
	{
	if (oifits->wave_list[i] == wave)
	    {
	    for (j = i; j < oifits->nbwave - 1; j++)
		{
		oifits->wave_list[j] = oifits->wave_list[j + 1];
		}
	    oifits->wave_list[oifits->nbwave - 1] = NULL;
	    break;
	    }
	}
    oifits->nbwave--;
    if (oifits->oiwave == wave) oifits->oiwave = NULL;
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an OI_WAVELENGTH from an OI_FITS file and delete it.
   @param oifits Contains an OI-FITS file.
   @param wave  Contains an OI_WAVELENGTH binary table as mat_oiwavelength data structure.
   @returns An error code.

   An OI_WAVELENGTH binary table is removed from the list of OI_WAVELENGTHs of an OI_FITS file.
   The OI_WAVELENGTH data structure is deleted.
*/
cpl_error_code mat_oifits_delete_wavelength(mat_oifits *oifits, mat_oiwavelength *wave)
{
    cpl_msg_info(cpl_func,"Delete wavelength");
    cpl_error_code ec = mat_oifits_remove_wavelength(oifits, wave);
    if (ec != CPL_ERROR_NONE) return ec;
    return mat_oiwavelength_delete(wave);
}

/**
   @ingroup oifits
   @brief Adds an OI_FLUX to an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param spect  Contains an OI_FLUX binary table as mat_array data structure.
   @returns An error code.

   An OI_FLUX binary table is added to the list of OI_FLUXs of an OI_FITS file.
   It is not checked if another OI_FLUX with the same ARRNAME already exists.
   If it is the first table of an OI_FITS file or it corresponds to the current tupel
   (ARRNAME and INSNAME of the OI_FLUX table and ARRNAME and INSNAME of the current tupel is equal), the
   OI_FLUX is set as current OI_FLUX and can directly accessed.
*/
cpl_error_code mat_oifits_add_spectrum(mat_oifits *oifits, mat_oispectrum *spect)
{
    cpl_msg_info(cpl_func,"Add spectrum");
    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((spect!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oispectrum (spect) argument given");
    /* extend the current list and add the new element at the end */
    oifits->spect_list = cpl_realloc(oifits->spect_list, (oifits->nbspect + 1)*sizeof(mat_oispectrum *));
    mat_assert_not_null(oifits->spect_list, "cannot extend the list of OI_FLUX, fatal");
    oifits->spect_list[oifits->nbspect] = spect;
    oifits->nbspect++;
    if (oifits->nbtupel == 0)
	{ /* we have no tupel */
	oifits->oispect = spect;
	}
    else if ((strcmp(oifits->tupel_list[oifits->sel_tupel]->insname,
		     spect->insname) == 0)
	     &&
	     (strcmp(oifits->tupel_list[oifits->sel_tupel]->arrname,
		     spect->arrayname) == 0))
	{ /* the spectrum table is related to the current tupel */
	oifits->oispect = spect;
	}
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an OI_FLUX from an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param spect  Contains an OI_FLUX binary table as mat_oispectrum data structure.
   @returns An error code.

   An OI_FLUX binary table is removed from the list of OI_FLUXs of an OI_FITS file.
   The data structure of the OI_FLUX binary table is not deleted!
*/
cpl_error_code mat_oifits_remove_spectrum(mat_oifits *oifits, mat_oispectrum *spect)
{
    cpl_msg_info(cpl_func,"Remove spectrum");
    int     i, j;

    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((spect!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oispectrum (spect) argument given");
    /* remove the element from the list */
    for (i = 0; i < oifits->nbspect; i++)
	{
	if (oifits->spect_list[i] == spect)
	    {
	    for (j = i; j < oifits->nbspect - 1; j++)
		{
		oifits->spect_list[j] = oifits->spect_list[j + 1];
		}
	    oifits->spect_list[oifits->nbspect - 1] = NULL;
	    break;
	    }
	}
    oifits->nbspect--;
    if (oifits->oispect == spect) oifits->oispect = NULL;
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an OI_FLUX from an OI_FITS file and delete it.
   @param oifits Contains an OI-FITS file.
   @param spect  Contains an OI_FLUX binary table as mat_oispectrum data structure.
   @returns An error code.

   An OI_FLUX binary table is removed from the list of OI_FLUXs of an OI_FITS file.
   The OI_FLUX data structure is deleted.
*/
cpl_error_code mat_oifits_delete_spectrum(mat_oifits *oifits, mat_oispectrum *spect)
{
    cpl_msg_info(cpl_func,"Delete spectrum");
    cpl_error_code ec = mat_oifits_remove_spectrum(oifits, spect);
    if (ec != CPL_ERROR_NONE) return ec;
    return mat_oispectrum_delete(spect);
}

/**
   @ingroup oifits
   @brief Adds an OI_VIS to an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param vis  Contains an OI_VIS binary table as mat_oivis data structure.
   @returns An error code.

   An OI_VIS binary table is added to the corresponding tupel of an OI_FITS file.
   If another OI_VIS with the same INSNAME/ARRNAME already exists, an error is returned.
   If it corresponds to the current tupel (ARRNAME and INSNAME of the OI_VIS table and
   ARRNAME and INSNAME of the current tupel is equal), the OI_VIS can directly be accessed.
*/
cpl_error_code mat_oifits_add_vis(mat_oifits *oifits, mat_oivis *vis)
{
    int               itupel;
    double            min_mjd;
    double            max_mjd;
    cpl_msg_info(cpl_func,"Add vis");

    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((vis!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oivis (vis) argument given");
    mat_oivis_get_mjd_span(vis, &min_mjd, &max_mjd);
    itupel = mat_oifits_find_tupel(oifits, vis->arrayname, vis->insname, min_mjd, max_mjd, 1);
    mat_assert_value((itupel >= 0), CPL_ERROR_UNSPECIFIED, CPL_ERROR_UNSPECIFIED,
		     "fatal memory allocation error (OI-FITS tupel inside a mat_oifits data structure)");
    mat_assert_value((oifits->tupel_list[itupel]->oivis == NULL), CPL_ERROR_UNSPECIFIED, CPL_ERROR_UNSPECIFIED,
		     " there exists already an OI_VIS binary table for the INSNAME/ARRNAME combination");
    oifits->tupel_list[itupel]->oivis = vis;
    if (oifits->sel_tupel == itupel)
	{ /* reselect the current tupel (update all pointers) */
	mat_oifits_select_tupel(oifits, oifits->sel_tupel);
	}
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an OI_VIS from an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param vis  Contains an OI_VIS binary table as mat_oivis data structure.
   @returns An error code.

   The OI_VIS binary table is removed from the associated tupel. If no binary table
   is in this tupel, it is removed from the OI_FITS file.
   The data structure of the OI_VIS binary table is not deleted!
*/
cpl_error_code mat_oifits_remove_vis(mat_oifits *oifits, mat_oivis *vis)
{
    int     i;
    cpl_msg_info(cpl_func,"Remove vis");

    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((vis!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oivis (vis) argument given");
    /* remove the element from the list */
    for (i = 0; i < oifits->nbtupel; i++)
	{
	if (oifits->tupel_list[i]->oivis == vis)
	    {
	    oifits->tupel_list[i]->oivis = NULL;
	    mat_oifits_cleanup_tupel_list(oifits);
	    break;
	    }
	}
    if (oifits->oivis == vis) oifits->oivis = NULL;
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an OI_VIS from an OI_FITS file and delete it.
   @param oifits Contains an OI-FITS file.
   @param vis  Contains an OI_VIS binary table as mat_oivis data structure.
   @returns An error code.

   An OI_VIS binary table is removed from the associated tupel.
   The OI_VIS data structure is deleted.
*/
cpl_error_code mat_oifits_delete_vis(mat_oifits *oifits, mat_oivis *vis)
{
    cpl_msg_info(cpl_func,"Delete vis");
    cpl_error_code ec = mat_oifits_remove_vis(oifits, vis);
    if (ec != CPL_ERROR_NONE) return ec;
    return mat_oivis_delete(vis);
}

/**
   @ingroup oifits
   @brief Adds an OI_VIS2 to an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param vis2  Contains an OI_VIS2 binary table as mat_oivis data structure.
   @returns An error code.

   An OI_VIS2 binary table is added to the corresponding tupel of an OI_FITS file.
   If another OI_VIS2 with the same INSNAME/ARRNAME already exists, an error is returned.
   If it corresponds to the current tupel (ARRNAME and INSNAME of the OI_VIS2 table and
   ARRNAME and INSNAME of the current tupel is equal), the OI_VIS2 can directly be accessed.
*/
cpl_error_code mat_oifits_add_vis2(mat_oifits *oifits, mat_oivis2 *vis2)
{
    int               itupel;
    double            min_mjd;
    double            max_mjd;
    cpl_msg_info(cpl_func,"Add vis2");

    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((vis2!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oivis2 (vis2) argument given");
    mat_oivis2_get_mjd_span(vis2, &min_mjd, &max_mjd);
    itupel = mat_oifits_find_tupel(oifits, vis2->arrayname, vis2->insname, min_mjd, max_mjd, 1);
    mat_assert_value((itupel >= 0), CPL_ERROR_UNSPECIFIED, CPL_ERROR_UNSPECIFIED,
		     "fatal memory allocation error (OI-FITS tupel inside a mat_oifits data structure)");
    oifits->tupel_list[itupel]->oivis2 = vis2;
    if (oifits->sel_tupel == itupel)
	{ /* reselect the current tupel (update all pointers) */
	mat_oifits_select_tupel(oifits, oifits->sel_tupel);
	}
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an OI_VIS2 from an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param vis2  Contains an OI_VIS2 binary table as mat_oivis data structure.
   @returns An error code.

   The OI_VIS2 binary table is removed from the associated tupel. If no binary table
   is in this tupel, it is removed from the OI_FITS file.
   The data structure of the OI_VIS2 binary table is not deleted!
*/
cpl_error_code mat_oifits_remove_vis2(mat_oifits *oifits, mat_oivis2 *vis2)
{
    cpl_msg_info(cpl_func,"Remove vis2");
    int     i;

    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((vis2!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oivis2 (vis2) argument given");
    /* remove the element from the list */
    for (i = 0; i < oifits->nbtupel; i++)
	{
	if (oifits->tupel_list[i]->oivis2 == vis2)
	    {
	    oifits->tupel_list[i]->oivis2 = NULL;
	    mat_oifits_cleanup_tupel_list(oifits);
	    break;
	    }
	}
    if (oifits->oivis2 == vis2) oifits->oivis2 = NULL;
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an OI_VIS2 from an OI_FITS file and delete it.
   @param oifits Contains an OI-FITS file.
   @param vis2  Contains an OI_VIS2 binary table as mat_oivis data structure.
   @returns An error code.

   An OI_VIS2 binary table is removed from the associated tupel.
   The OI_VIS2 data structure is deleted.
*/
cpl_error_code mat_oifits_delete_vis2(mat_oifits *oifits, mat_oivis2 *vis2)
{
    cpl_msg_info(cpl_func,"Delete vis2");
    cpl_error_code ec = mat_oifits_remove_vis2(oifits, vis2);
    if (ec != CPL_ERROR_NONE) return ec;
    return mat_oivis2_delete(vis2);
}

/**
   @ingroup oifits
   @brief Adds an OI_T3 to an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param t3  Contains an OI_T3 binary table as mat_oivis data structure.
   @returns An error code.

   An OI_T3 binary table is added to the corresponding tupel of an OI_FITS file.
   If another OI_T3 with the same INSNAME/ARRNAME already exists, an error is returned.
   If it corresponds to the current tupel (ARRNAME and INSNAME of the OI_T3 table and
   ARRNAME and INSNAME of the current tupel is equal), the OI_T3 can directly be accessed.
*/
cpl_error_code mat_oifits_add_t3(mat_oifits *oifits, mat_oit3 *t3)
{
    cpl_msg_info(cpl_func,"Add T3");
    int               itupel;
    double            min_mjd;
    double            max_mjd;

    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((t3!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oit3 (t3) argument given");
    mat_oit3_get_mjd_span(t3, &min_mjd, &max_mjd);
    itupel = mat_oifits_find_tupel(oifits, t3->arrayname, t3->insname, min_mjd, max_mjd, 1);
    mat_assert_value((itupel >= 0), CPL_ERROR_UNSPECIFIED, CPL_ERROR_UNSPECIFIED,
		     "fatal memory allocation error (OI-FITS tupel inside a mat_oifits data structure)");
    oifits->tupel_list[itupel]->oit3 = t3;
    if (oifits->sel_tupel == itupel)
	{ /* reselect the current tupel (update all pointers) */
	mat_oifits_select_tupel(oifits, oifits->sel_tupel);
	}
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an OI_T3 from an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param t3  Contains an OI_T3 binary table as mat_oit3 data structure.
   @returns An error code.

   The OI_T3 binary table is removed from the associated tupel. If no binary table
   is in this tupel, it is removed from the OI_FITS file.
   The data structure of the OI_T3 binary table is not deleted!
*/
cpl_error_code mat_oifits_remove_t3(mat_oifits *oifits, mat_oit3 *t3)
{
    cpl_msg_info(cpl_func,"Remove T3");
    int     i;

    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((t3!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oit3 (t3) argument given");
    /* remove the element from the list */
    for (i = 0; i < oifits->nbtupel; i++)
	{
	if (oifits->tupel_list[i]->oit3 == t3)
	    {
	    oifits->tupel_list[i]->oit3 = NULL;
	    mat_oifits_cleanup_tupel_list(oifits);
	    break;
	    }
	}
    if (oifits->oit3 == t3) oifits->oit3 = NULL;
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an OI_T3 from an OI_FITS file and delete it.
   @param oifits Contains an OI-FITS file.
   @param t3  Contains an OI_T3 binary table as mat_oit3 data structure.
   @returns An error code.

   An OI_T3 binary table is removed from the associated tupel.
   The OI_T3 data structure is deleted.
*/
cpl_error_code mat_oifits_delete_t3(mat_oifits *oifits, mat_oit3 *t3)
{
    cpl_msg_info(cpl_func,"Delete T3");
    cpl_error_code ec = mat_oifits_remove_t3(oifits, t3);
    if (ec != CPL_ERROR_NONE) return ec;
    return mat_oit3_delete(t3);
}

/**
   @ingroup oifits
   @brief Adds an TF2 to an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param tf2  Contains an TF2 binary table as mat_oivis data structure.
   @returns An error code.

   An TF2 binary table is added to the corresponding tupel of an OI_FITS file.
   If another TF2 with the same INSNAME/ARRNAME already exists, an error is returned.
   If it corresponds to the current tupel (ARRNAME and INSNAME of the TF2 table and
   ARRNAME and INSNAME of the current tupel is equal), the TF2 can directly be accessed.
*/
cpl_error_code mat_oifits_add_tf2(mat_oifits *oifits, mat_oitf2 *tf2)
{
    int               itupel;
    double            min_mjd;
    double            max_mjd;

    cpl_msg_info(cpl_func,"Add TF2");
      
    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((tf2!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oitf2 (tf2) argument given");
    mat_oitf2_get_mjd_span(tf2, &min_mjd, &max_mjd);
    itupel = mat_oifits_find_tupel(oifits, tf2->arrayname, tf2->insname, min_mjd, max_mjd, 1);
    mat_assert_value((itupel >= 0), CPL_ERROR_UNSPECIFIED, CPL_ERROR_UNSPECIFIED,
		     "fatal memory allocation error (OI-FITS tupel inside a mat_oifits data structure)");
    oifits->tupel_list[itupel]->oitf2 = tf2;
    if (oifits->sel_tupel == itupel)
	{ /* reselect the current tupel (update all pointers) */
	mat_oifits_select_tupel(oifits, oifits->sel_tupel);
	}
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an TF2 from an OI_FITS file.
   @param oifits Contains an OI-FITS file.
   @param tf2  Contains an TF2 binary table as mat_oitf2 data structure.
   @returns An error code.

   The TF2 binary table is removed from the associated tupel. If no binary table
   is in this tupel, it is removed from the OI_FITS file.
   The data structure of the TF2 binary table is not deleted!
*/
cpl_error_code mat_oifits_remove_tf2(mat_oifits *oifits, mat_oitf2 *tf2)
{
    int     i;

    cpl_msg_info(cpl_func,"Remove TF2");
      
    mat_assert_value((oifits!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oifits (oifits) argument given");
    mat_assert_value((tf2!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		     " no mat_oitf2 (tf2) argument given");
    /* remove the element from the list */
    for (i = 0; i < oifits->nbtupel; i++)
	{
	if (oifits->tupel_list[i]->oitf2 == tf2)
	    {
	    oifits->tupel_list[i]->oitf2 = NULL;
	    mat_oifits_cleanup_tupel_list(oifits);
	    break;
	    }
	}
    if (oifits->oitf2 == tf2) oifits->oitf2 = NULL;
    return CPL_ERROR_NONE;
}

/**
   @ingroup oifits
   @brief Removes an TF2 from an OI_FITS file and delete it.
   @param oifits Contains an OI-FITS file.
   @param tf2  Contains an TF2 binary table as mat_oitf2 data structure.
   @returns An error code.

   An TF2 binary table is removed from the associated tupel.
   The TF2 data structure is deleted.
*/
cpl_error_code mat_oifits_delete_tf2(mat_oifits *oifits, mat_oitf2 *tf2)
{
    cpl_msg_info(cpl_func,"Delete TF2");
      
    cpl_error_code ec = mat_oifits_remove_tf2(oifits, tf2);
    if (ec != CPL_ERROR_NONE) return ec;
    return mat_oitf2_delete_nt(tf2);
}

