/* $Id: mat_oifits.h,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oifits.h $
 */

#ifndef MAT_OIFITS_H
#define MAT_OIFITS_H

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "mat_array.h"
#include "mat_oitarget.h"
#include "mat_oiwavelength.h"
#include "mat_oivis.h"
#include "mat_oivis2.h"
#include "mat_oit3.h"
#include "mat_oitf2.h"
#include "mat_oispectrum.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  High Level Structures
  ----------------------------------------------------------------------------*/

/**
   @ingroup oifits
   @brief Contains all OI-FITS tables for one ARRNAME/INSNAME combination.

   The four OI-FITS tables OI_VIS, OI_VIS2, OI_T3 and OI_TF2 are identified by an unique
   ARRNAME/INSNAME combination and a common time intervall for the data. Some OI-FITS files
   do exist where the same ARRNAME/INSNAME combination is used several times inside such a file.
   Because the data is taken at different time intervals, the minimum and maximum MJD of these
   data sets is also put into a mat_oifits_tupel as a grouping criteria.
*/
typedef struct {
  char        *arrname; /*!< array name */
  char        *insname; /*!< instrument name */
  double       min_mjd; /*!< smallest MJD column value */
  double       max_mjd; /*!< largest MJD column value */
  mat_oivis   *oivis;   /*!< OI_VIS binary table for this ARRNAME/INSNAME combination */
  mat_oivis2  *oivis2;  /*!< OI_VIS2 binary table for this ARRNAME/INSNAME combination */
  mat_oit3    *oit3;    /*!< OI_T3 binary table for this ARRNAME/INSNAME combination */
  mat_oitf2   *oitf2;   /*!< OI_TF2 binary table for this ARRNAME/INSNAME combination */
} mat_oifits_tupel;

/**
   @ingroup oifits
   @brief Structure containing oifits files.

   This data structure provides the content of an OI-FITS file. This excludes
   the tables which are ignored by the MATISSE pipeline software. Between the tables
   in an OI-FITS file certain cross-references do exist:

   <table>
     <tr><th>Name</th><th>Cardinality</th><th>Keywords for cross reference</th></tr>
     <tr><td>OI_TARGET</td><td>1</td><td>only one table is allowed</td></tr>
     <tr><td>OI_ARRAY</td><td>N</td><td>ARRNAME</td></tr>
     <tr><td>OI_WAVELENGTH</td><td>N</td><td>INSNAME</td></tr>
     <tr><td>OI_VIS</td><td>N</td><td>ARRNAME, INSNAME</td></tr>
     <tr><td>OI_VIS2</td><td>N</td><td>ARRNAME, INSNAME</td></tr>
     <tr><td>OI_T3</td><td>N</td><td>ARRNAME, INSNAME</td></tr>
     <tr><td>OI_TF2</td><td>N</td><td>ARRNAME, INSNAME</td></tr>
     <tr><td>OI_SPECTRUM</td><td> N</td><td>INSNAME</td></tr>
   </table>

   Since an OI-FITS file can contain several binary tables (except OI_TARGET), they are managed in four
   lists. One list contains all OI_ARRAY, one all OI_WAVELENGTH and one all OI_SPECTRUM binary tables,
   each stored in the assocciated data structures. A fourth list contains a tupel of OI_VIS, OI_VIS2,
   OI_T3 and OI_TF2 binary tables for a specific ARRNAME/INSNAME combination with the same time intervall
   for the observation. Working with these tupels is supported by simply selecting a tupel, after that
   the binary tables can be accessed directly via the corresponding elements of the mat_oifits structure.

   An example using a mat_oifits data structure is the load and process loop inside the mat_cal_imarec plugin:

<div style="font-family:'monospace'">
&nbsp;&nbsp;mat_cal_imarec_info    info;<br/>
&nbsp;&nbsp;cpl_frameset_iterator *it;<br/>
&nbsp;&nbsp;cpl_frame             *frame;<br/>
&nbsp;&nbsp;mat_oifits            *oifits;<br/>
<br/>
&nbsp;&nbsp;it = cpl_frameset_iterator_new(frameset);<br/>
&nbsp;&nbsp;frame = cpl_frameset_iterator_get(it);<br/>
&nbsp;&nbsp;while (frame != NULL)<br/>
&nbsp;&nbsp;&nbsp;&nbsp;{&nbsp;<i>// we use only calibrated target interferometric data</i><br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;if (strcmp(cpl_frame_get_tag(frame), "TARGET_CAL_INT") == 0)<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;{<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;oifits = mat_oifits_load(frame);<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;if (oifits != NULL)<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;{&nbsp;<i>// iterate over all OI-FITS tupels</i><br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;for (i = 0; i < oifits->nbtupel; i++)<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;{<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;mat_oifits_select_tupel(oifits, i);<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<i>// calculate the baseline to pixel scale for the selected wavelengths</i><br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;mat_calc_baseline_scale(&info, oifits->oiwave);<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<i>// add the filtered squared visibilities to the internal data</i><br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;mat_add_vis2(&info, oifits->oivis2);<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<i>// add the filtered bispectra to the internal data</i><br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;mat_add_t3(&info, oifits->oit3);<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;mat_oifits_delete(oifits);<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;cpl_frameset_iterator_advance(it, 1);<br/>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;frame = cpl_frameset_iterator_get(it);<br/>
&nbsp;&nbsp;&nbsp;&nbsp;}<br/>
&nbsp;&nbsp;cpl_frameset_iterator_delete(it);
</div>

   This data structure can be used in two different ways (no mixture allowed!):

   <ol>
     <li>The members oitarget, array, oiwave, oispect, oivis, oivis2, oit3 and oitf2 are set and used directly.</li>
     <li>All binary tables are added using the mat_oifits_add_XXX methods. The binary tables stored in tuples are selected with mat_oifits_select_tupel.</li>
   </ol>

   As an example for distinguishing both use cases the following scenarios are described:

   <table>
     <tr>
       <th>array</th>
       <th>array_list</th>
       <th>scenario</th>
     </tr>
     <tr>
       <td>== NULL</td>
       <td>== NULL</td>
       <td>The OI-FITS file does not contain any OI_ARRAY binary table.</td>
     </tr>
     <tr>
       <td>== NULL</td>
       <td>!= NULL</td>
       <td>At least one OI_ARRAY binary table was added using the mat_oifits_add_array() function, but no specific OI_ARRAY is currently selected.</td>
     </tr>
     <tr>
       <td>!= NULL</td>
       <td>== NULL</td>
       <td>The OI_ARRAY binary table was set directly.</td>
     </tr>
     <tr>
       <td>!= NULL</td>
       <td>!= NULL</td>
       <td>At least one OI_ARRAY binary table was added using the mat_oifits_add_array() function and a specific OI_ARRAY binary table is selected.</td>
     </tr>
   </table>

*/
typedef struct {
  cpl_propertylist      *keywords;     /*!< The keywords from the primary header. */
  mat_oitarget          *oitarget;     /*!< Structure containing target informations. */
  mat_array             *array;        /*!< Structure containing informations on the interferometric array. */
  mat_oiwavelength      *oiwave;       /*!< Structure containing informations on the spectral bands. */
  mat_oispectrum        *oispect;      /*!< Structure containing the spectrum. */
  mat_oivis             *oivis;        /*!< Structure containing informations on the complex differential visibility. */
  mat_oivis2            *oivis2;       /*!< Structure containing informations on the squared visibility. */
  mat_oit3              *oit3;         /*!< Structure containing informations on the closure phase. */
  mat_oitf2             *oitf2;        /*!< Structure containing informations on the transfer function. */
  /* support for OI-FITS files with multiples of the above tables as defined in the standard */
  /* only one OI_TARGET is allowed -> we use the previously defined oitarget member */
  /* several OI_ARRAY tables are allowed (distinguished by ARRNAME) */
  /* this list is also used to provide a list of array names (ARRNAME) */
  int                    nbarray;      /*!< The number of OI_ARRAY tables. */
  mat_array            **array_list;   /*!< The list of OI_ARRAY tables. */
  /* several OI_WAVELENGTH are allowed (distinguished by INSNAME) */
  /* this list is also used to provide a list of instrument names (INSNAME) */
  int                    nbwave;       /*!< The number of OI_WAVELENGTH tables. */
  mat_oiwavelength     **wave_list;    /*!< The list of OI_WAVELENGTH tables. */
  /* several spectrum tables are allowed (distinguished by INSNAME) */
  int                    nbspect;      /*!< The number of OI_SPECTRUM tables. */
  mat_oispectrum       **spect_list;   /*!< The list of OI_SPECTRUM tables. */
  /* several OI_VIS, OI_VIS2, OI_T3 and OI_TF2 tables are allowed (distinguished by ARRNAME, INSNAME) */
  int                    nbtupel;      /*!< The number of tupels (associated OI_* tables). */
  int                    sel_tupel;    /*!< The number (index) of the selected tupel. */
  mat_oifits_tupel     **tupel_list;   /*!< This tupel list contains all OI_* tables. */
} mat_oifits;


/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/
mat_oifits* mat_oifits_new(void);
cpl_error_code mat_oifits_delete(mat_oifits *oifits);
cpl_error_code mat_oifits_list_delete(mat_oifits **oifits, int nbfits);

mat_oifits *mat_oifits_load(cpl_frame *frame);

cpl_error_code mat_oifits_save(mat_oifits *oifits,
			       cpl_frameset *frameset,
			       cpl_frameset *usedframes,
			       const cpl_parameterlist *parlist,
			       const char *recipe_name,
			       char *filename,
			       int type);

int mat_oifits_find_tupel(mat_oifits *oifits,
			  const char *arrname,
			  const char *insname,
			  double min_mjd,
			  double max_mjd,
			  int cflag);

cpl_error_code mat_oifits_select_tupel(mat_oifits *oifits,
				       int itupel);

cpl_error_code mat_oifits_add_target(mat_oifits *oifits, mat_oitarget *target);

cpl_error_code mat_oifits_add_array(mat_oifits *oifits, mat_array *array);
cpl_error_code mat_oifits_remove_array(mat_oifits *oifits, mat_array *array);
cpl_error_code mat_oifits_delete_array(mat_oifits *oifits, mat_array *array);

cpl_error_code mat_oifits_add_wavelength(mat_oifits *oifits, mat_oiwavelength *wave);
cpl_error_code mat_oifits_remove_wavelength(mat_oifits *oifits, mat_oiwavelength *wave);
cpl_error_code mat_oifits_delete_wavelength(mat_oifits *oifits, mat_oiwavelength *wave);

cpl_error_code mat_oifits_add_spectrum(mat_oifits *oifits, mat_oispectrum *spect);
cpl_error_code mat_oifits_remove_spectrum(mat_oifits *oifits, mat_oispectrum *spect);
cpl_error_code mat_oifits_delete_spectrum(mat_oifits *oifits, mat_oispectrum *spect);

cpl_error_code mat_oifits_add_vis(mat_oifits *oifits, mat_oivis *vis);
cpl_error_code mat_oifits_remove_vis(mat_oifits *oifits, mat_oivis *vis);
cpl_error_code mat_oifits_delete_vis(mat_oifits *oifits, mat_oivis *vis);

cpl_error_code mat_oifits_add_vis2(mat_oifits *oifits, mat_oivis2 *vis2);
cpl_error_code mat_oifits_remove_vis2(mat_oifits *oifits, mat_oivis2 *vis2);
cpl_error_code mat_oifits_delete_vis2(mat_oifits *oifits, mat_oivis2 *vis2);

cpl_error_code mat_oifits_add_t3(mat_oifits *oifits, mat_oit3 *t3);
cpl_error_code mat_oifits_remove_t3(mat_oifits *oifits, mat_oit3 *t3);
cpl_error_code mat_oifits_delete_t3(mat_oifits *oifits, mat_oit3 *t3);

cpl_error_code mat_oifits_add_tf2(mat_oifits *oifits, mat_oitf2 *tf2);
cpl_error_code mat_oifits_remove_tf2(mat_oifits *oifits, mat_oitf2 *tf2);
cpl_error_code mat_oifits_delete_tf2(mat_oifits *oifits, mat_oitf2 *tf2);

#endif
