/* $Id: mat_oispectrum.c,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oispectrum.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <string.h>
#include "mat_error.h"
#include "mat_utils.h"
#include "mat_oispectrum.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/
/**
  @ingroup oispec
  @brief   This method creates a new mat_fluxelem structure with all pointers set to NULL
  @param   nbchannel Number of spectral channels
  @returns Pointer to new allocated mat_fluxelem structure on success or NULL on failure
 
  This method creates a new mat_fluxelem structure and sets all value to the default. For
  pointers the value is NULL for primitive data types the value is set to 0.
 */
mat_fluxelem *mat_fluxelem_new(int nbchannel)
{
  mat_fluxelem  *el = (mat_fluxelem *)cpl_calloc(1, sizeof(mat_fluxelem));

  if (el == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_fluxelem");
      return NULL;
    }
  el->fluxdata=NULL;
  el->fluxerr=NULL;
  el->bandflag=NULL;

  el->fluxdata = cpl_calloc(nbchannel, sizeof(double));
  if (el->fluxdata == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_fluxelem->fluxdata");
      mat_fluxelem_delete(el);
      return NULL;
    }
  el->fluxerr = cpl_calloc(nbchannel, sizeof(double));
  if (el->fluxerr == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_fluxelem->fluxerr");
      mat_fluxelem_delete(el);
      return NULL;
    }
  el->bandflag = cpl_calloc(nbchannel, sizeof(cpl_boolean));
  if (el->bandflag == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_fluxelem->bandflag");
      mat_fluxelem_delete(el);
      return NULL;
    }
  return el;
}
/**
   @ingroup oispec
   @brief Free memory of a single mat_fluxelem data structure.
   @param el Pointer to one mat_fluxelem data structure.
   @return A cpl_error_code
 
   This method frees the memory of a mat_vis2elem data structure.
 */
cpl_error_code mat_fluxelem_delete(mat_fluxelem *el)
{
  mat_assert_value((el != NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no mat_vis2elem (el) argument given");
  if (el->fluxdata != NULL)
    {
      cpl_free(el->fluxdata);
      el->fluxdata = NULL;
    }
  if (el->fluxerr != NULL)
    {
      cpl_free(el->fluxerr);
      el->fluxerr = NULL;
    }
  if (el->bandflag != NULL)
    {
      cpl_free(el->bandflag);
      el->bandflag = NULL;
    }
  cpl_free(el);
  return CPL_ERROR_NONE;
}
/**
  @ingroup oispec
  @brief This method creates a new mat_oispectrum structure with all pointers set to NULL
  @param nbflux    Number of flux
  @param nbchannel Number of spectral channels
  @returns Pointer to new allocated mat_oispectrum structure on success or NULL on failure
 
  This method creates a new mat_oispectrum structure and sets all value to the default. For
  pointers the value is NULL for primitive data types the value is set to 0.
 */
mat_oispectrum *mat_oispectrum_new(int nbflux, int nbchannel)
{
  mat_oispectrum *oispectrum;
  int i=0;

  oispectrum = (mat_oispectrum *)cpl_calloc(1, sizeof(mat_oispectrum));
  if (oispectrum == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oispectrum");
      return NULL;
    }
  oispectrum->fov=1.0;
  /* oispectrum->fovtype=cpl_sprintf("RADIUS"); */
  /* oispectrum->calstat=cpl_sprintf("U"); */
  oispectrum->nbflux=nbflux;
  oispectrum->nbchannel=nbchannel;
  oispectrum->list_flux = (mat_fluxelem **)cpl_calloc(nbflux, sizeof(mat_fluxelem *));
  if (oispectrum->list_flux == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oispectrum->list_flux");
      mat_oispectrum_delete(oispectrum);
      return NULL;
    }
  for (i = 0; i < nbflux; i++)
    {
      oispectrum->list_flux[i] = mat_fluxelem_new(nbchannel);
      if (oispectrum->list_flux[i] == NULL)
  	{
  	  mat_oispectrum_delete(oispectrum);
  	  return NULL;
  	}
    }
  return oispectrum;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oispec
  @brief delete structure mat_oispectrum
  @param  oispectrum          current structure
  @return error code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_oispectrum_delete(mat_oispectrum *oispectrum)
{
  int i=0;

  mat_assert_value((oispectrum!=NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no mat_oispectrum (oispectrum) argument given");
  if (oispectrum->dateobs != NULL)
    {
      cpl_free(oispectrum->dateobs);
      oispectrum->dateobs = NULL;
    }
  if (oispectrum->arrayname != NULL)
    {
      cpl_free(oispectrum->arrayname);
      oispectrum->arrayname = NULL;
    }
  if (oispectrum->insname != NULL)
    {
      cpl_free(oispectrum->insname);
      oispectrum->insname = NULL;
    }
  if (oispectrum->fovtype != NULL)
    {
      cpl_free(oispectrum->fovtype);
      oispectrum->fovtype = NULL;
    }
  if (oispectrum->calstat != NULL)
    {
      cpl_free(oispectrum->calstat);
      oispectrum->calstat = NULL;
    }
  if (oispectrum->list_flux != NULL)
    {
      for (i = 0; i < oispectrum->nbflux; i++)
	{
	  if (oispectrum->list_flux[i] != NULL)
	    {
	      mat_fluxelem_delete(oispectrum->list_flux[i]);
	      oispectrum->list_flux[i] = NULL;
	    }
	}
      cpl_free(oispectrum->list_flux);
      oispectrum->list_flux = NULL;
      oispectrum->nbflux = 0;
    }
  cpl_free(oispectrum);
  return CPL_ERROR_NONE;
}

/**
  @ingroup oispec
  @brief Load a  mat_oispectrum data structure from a cpl table and a propertylist
  @param plist          propertylist containing the header 
  @param table          cpl table containing the data
  @return mat_oispectrum
*/
mat_oispectrum *mat_oispectrum_from_table(cpl_propertylist *plist, cpl_table *table)
{
  mat_oispectrum *oispectrum;
  int             nbchannel;
  int             nbflux;
  int             j, k;

  nbflux = cpl_table_get_nrow(table);
  /* old code
  nbchannel = cpl_array_get_size(cpl_table_get_array (table, "FLUXDATA", 0));
  if (nbchannel == 0) nbchannel = 1; // no array!
  */
  if (cpl_table_get_column_depth(table, "FLUXDATA") == 0)
    {
      nbchannel = 1;
    }
  else
    {
      nbchannel = cpl_table_get_column_dimension(table, "FLUXDATA", 0);
    }
  cpl_msg_info(cpl_func, "SPECTRUM table has %d wavelength data", nbchannel);
  oispectrum = mat_oispectrum_new(nbflux, nbchannel);
  if (oispectrum == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oispectrum");
      return NULL;
    }

  oispectrum->extver    = mat_propertylist_get_int_default(plist, "EXTVER", 1);
  oispectrum->insname   = mat_propertylist_get_string_default(NULL, 0, plist, "INSNAME", "empty");
  oispectrum->arrayname = mat_propertylist_get_string_default(NULL, 0, plist, "ARRNAME", "empty");
  oispectrum->dateobs   = mat_propertylist_get_string_default(NULL, 0, plist, "DATE-OBS", "empty");
  oispectrum->fov       = cpl_propertylist_get_double(plist, "FOV");
  oispectrum->fovtype   = mat_propertylist_get_string_default(NULL, 0, plist, "FOVTYPE", "RADIUS");
  oispectrum->calstat   = mat_propertylist_get_string_default(NULL, 0, plist, "CALSTAT", "U");

  for(j = 0; j < oispectrum->nbflux; j++)
    {
      mat_fluxelem  *el = oispectrum->list_flux[j];
      // copy the values from the table into the vis2 element
      el->targetid = cpl_table_get_int (table, "TARGET_ID", j, NULL);
      el->time = cpl_table_get_double (table, "TIME", j, NULL);
      el->dateobsmjd = cpl_table_get_double (table, "MJD", j, NULL);
      el->exptime = cpl_table_get_double (table, "INT_TIME", j, NULL);
      el->stationindex= cpl_table_get_int(table, "STA_INDEX", j, NULL);
      if (nbchannel == 1)
  	{
  	  el->fluxdata[0] = cpl_table_get_double(table, "FLUXDATA", j, NULL);
  	  el->fluxerr[0] = cpl_table_get_double(table, "FLUXERR", j, NULL);
  	  el->bandflag[0] = (cpl_table_get_int(table,"FLAG", j, NULL) == 1);
  	}
      else
  	{
  	  for (k = 0; k < oispectrum->nbchannel; k++)
  	    {
  	      el->fluxdata[k] = cpl_array_get_double(cpl_table_get_array(table, "FLUXDATA", j), k, NULL);
  	      el->fluxerr[k] = cpl_array_get_double(cpl_table_get_array(table, "FLUXERR", j), k, NULL);
  	      el->bandflag[k] = (cpl_array_get_int(cpl_table_get_array(table,"FLAG", j), k, NULL) == 1);
  	    }
  	}
    }
  return oispectrum;
}

/**
  @ingroup oispec
  @brief copy a  mat_oispectrum data structure into a cpl table and a propertylist
  @param oispectrum     data structure to copy
  @param plist          propertylist containing the header 
  @return cpl table containing the data
*/
cpl_table *mat_oispectrum_to_table(mat_oispectrum *oispectrum, cpl_propertylist *plist)
{
  cpl_table    *table;
  cpl_array    *arr;
  int           i;
  int          *flags;

  if (plist != NULL)
    {
      cpl_propertylist_append_string(plist,"EXTNAME","OI_FLUX");
      cpl_propertylist_append_int(plist, "EXTVER", oispectrum->extver);
      if ( cpl_propertylist_has(plist,"OI_REVN") )
	{
	  cpl_propertylist_erase(plist,"OI_REVN");
	}
      cpl_propertylist_append_int(plist,"OI_REVN",1);
      cpl_propertylist_append_string(plist,"DATE-OBS",oispectrum->dateobs);
      cpl_propertylist_append_string(plist,"ARRNAME",oispectrum->arrayname);
      cpl_propertylist_append_string(plist,"INSNAME",oispectrum->insname);
      cpl_propertylist_append_double(plist,"FOV",oispectrum->fov);
      cpl_propertylist_append_string(plist,"FOVTYPE",oispectrum->fovtype);
      cpl_propertylist_append_string(plist,"CALSTAT",oispectrum->calstat);
    }
  flags = (int *)cpl_calloc(oispectrum->nbchannel, sizeof(int));
  if (flags == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for flag vector");
      return NULL;
    }
  table = cpl_table_new(oispectrum->nbflux);
  if (table == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_table");
      cpl_free(flags);
      return NULL;
    }
  // Add columns
  cpl_table_new_column(table, "TARGET_ID", CPL_TYPE_INT);
  cpl_table_new_column(table, "TIME", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "TIME","s");
  cpl_table_new_column(table, "MJD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "MJD","day");
  cpl_table_new_column(table, "INT_TIME", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "INT_TIME","s");
  if (oispectrum->nbchannel == 1)
    {
      cpl_table_new_column(table, "FLUXDATA", CPL_TYPE_DOUBLE);
      cpl_table_set_column_unit(table, "FLUXDATA", "ADU");
      cpl_table_new_column(table, "FLUXERR", CPL_TYPE_DOUBLE);
      cpl_table_set_column_unit(table, "FLUXERR", "ADU");
   }
  else
    {
      cpl_table_new_column_array(table, "FLUXDATA", CPL_TYPE_DOUBLE, oispectrum->nbchannel);
      cpl_table_set_column_unit(table, "FLUXDATA", "ADU");
      cpl_table_new_column_array(table, "FLUXERR", CPL_TYPE_DOUBLE, oispectrum->nbchannel);
      cpl_table_set_column_unit(table, "FLUXERR", "ADU");
    }
  cpl_table_new_column(table,"STA_INDEX", CPL_TYPE_INT);
  if (oispectrum->nbchannel == 1)
    {
      cpl_table_new_column(table, "FLAG", CPL_TYPE_INT);
    }
  else
    {
      cpl_table_new_column_array(table, "FLAG", CPL_TYPE_INT, oispectrum->nbchannel);
    }
  // Fill the table
  for(i=0;i<oispectrum->nbflux;i++)
    {
      mat_fluxelem  *el = oispectrum->list_flux[i];
      cpl_table_set_int(table, "TARGET_ID", i, el->targetid);
      cpl_table_set_double(table,"TIME",i, el->time);
      cpl_table_set_double(table,"MJD",i, el->dateobsmjd);
      cpl_table_set_double(table,"INT_TIME",i, el->exptime);
      if (oispectrum->nbchannel == 1)
	{
	  cpl_table_set_double(table,"FLUXDATA",i, el->fluxdata[0]);
	  cpl_table_set_double(table,"FLUXERR",i, el->fluxerr[0]);
	}
      else
	{
	  arr = cpl_array_wrap_double(el->fluxdata, oispectrum->nbchannel);
	  cpl_table_set_array(table,"FLUXDATA", i, arr);
	  cpl_array_unwrap(arr);
	  arr = cpl_array_wrap_double(el->fluxerr, oispectrum->nbchannel);
	  cpl_table_set_array(table,"FLUXERR",i,arr);
	  cpl_array_unwrap(arr);
	}
      cpl_table_set_int(table,"STA_INDEX",i,el->stationindex);
      if (oispectrum->nbchannel == 1)
	{
	  cpl_table_set_int(table, "FLAG", i, (el->bandflag[0] != 0));
	}
      else
	{
	  for(int j=0;j<oispectrum->nbchannel;j++)
	    {
	      if (el->bandflag[j])
		{
		  flags[j] = 1;
		}
	      else
		{
		  flags[j] = 0;
		}
	    }
	  arr = cpl_array_wrap_int(flags, oispectrum->nbchannel);
	  cpl_table_set_array(table,"FLAG",i, arr);
	  cpl_array_unwrap(arr);
	}
    }
  cpl_table_set_column_savetype(table,"TARGET_ID",CPL_TYPE_SHORT);
  cpl_table_set_column_savetype(table,"STA_INDEX",CPL_TYPE_SHORT);
  cpl_table_set_column_savetype(table,"FLAG",CPL_TYPE_BOOL);
  cpl_free(flags);
  return table;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oispec
  @brief Load a frame in mat_oispectrum structure
  @param frame           current frame
  @return mat_oispectrum
 */
/*-----------------------------------------------------------------------------*/

mat_oispectrum * mat_oispectrum_load(cpl_frame *frame)
{

  mat_oispectrum *oispectrum = NULL;  
  int nbExtent = 0;
  int i = 0;
  cpl_propertylist *plist = NULL;
  char *keyExtname = NULL;
  cpl_table *table = NULL;

  /*Allocate the structure and initialize its attributes*/

  mat_assert_value((frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_frame (frame) argument given");

  nbExtent = cpl_frame_get_nextensions(frame);
  for(i = 0; i<nbExtent; i++) 
    {
      plist = cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
      keyExtname = (char *)cpl_propertylist_get_string(plist,"EXTNAME");
      if (keyExtname != NULL) 
	{
	  table = cpl_table_load(cpl_frame_get_filename(frame), i+1, 
				 0);
	  /*Load data*/
	  if (!strcmp(keyExtname, "OI_FLUX") ) 
	    {
	      cpl_error_reset();
	      //aux = 1;
	      oispectrum = mat_oispectrum_from_table(plist, table);
		
	    }
	  cpl_table_delete(table);
	}
      cpl_propertylist_delete(plist);
    }
  return oispectrum;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oispec
  @brief Append an OI_FLUX binary table to an existing OIFITS file
  @param filename          name of the FITS file
  @param oispec            the mat_oispectrum structure containing the 
                           informations for filling the binary table
  @return cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code  mat_oispectrum_append(char *filename,mat_oispectrum *oispec)
{
  cpl_table *oispecTab = NULL;
  cpl_propertylist *keyTab = NULL;
  cpl_errorstate prestate = cpl_errorstate_get();


  /*Check input parameters*/
  mat_assert_value((oispec!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oispectrum (oispec) argument given");
  mat_assert_value((filename!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no filename given");

  keyTab=cpl_propertylist_new();
  oispecTab = mat_oispectrum_to_table(oispec, keyTab);
  cpl_table_save(oispecTab,NULL,keyTab,filename,CPL_IO_EXTEND);
  cpl_propertylist_delete(keyTab);
  cpl_table_delete(oispecTab);
  if (!cpl_errorstate_is_equal(prestate))
    {
      cpl_errorstate_set(prestate);
      return CPL_ERROR_UNSPECIFIED;
    }
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oispectrum
  @brief Get the time span of the observation as MJD values (beginning and end).
  @param oispectrum   the mat_oispectrum structure containing the informations for filling the binary table
  @param min_mjd The beginning of the observation.
  @param max_mjd The end of the observation.
  @return cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mat_oispectrum_get_mjd_span(mat_oispectrum *oispectrum, double *min_mjd, double *max_mjd)
{
  int i;

  /*Check input parameters*/
  mat_assert_value((oispectrum!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oispectrum (oispectrum) argument given");
  mat_assert_value((min_mjd!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no start MJD given");
  mat_assert_value((max_mjd!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no end MJD given");
  *min_mjd =  1.0e12;
  *max_mjd = -1.0e12;
  for (i = 0; i < oispectrum->nbflux; i++)
    {
      mat_fluxelem *el = oispectrum->list_flux[i];
      *min_mjd = fmin(*min_mjd, el->dateobsmjd);
      *max_mjd = fmax(*max_mjd, el->dateobsmjd);
    }
  return CPL_ERROR_NONE;
}
