/* $Id: mat_oispectrum.h,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oispectrum.h $
 */

#ifndef MAT_OISPECTRUM_H
#define MAT_OISPECTRUM_H

/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/

#include "cpl.h"
#include "mat_oiwavelength.h"

/*-----------------------------------------------------------------------------
  Defines
  ----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  Low Level Structures
  ----------------------------------------------------------------------------*/
/* Structure mat_fluxelem */
/**
   @ingroup oispec
   @brief Structure containing one measurement of an OI_SPECTRUM.
*/
typedef struct {
    int            targetid;              /* target ID                        */
    double         time;                  /* UTC time of observation          */
    double         dateobsmjd;            /* UTC date (MJD)                   */
    double         exptime;               /* integration time per frame       */
    double        *fluxdata;              /* array of spectrum                */
    double        *fluxerr;               /* array of spectrum error          */
    int            stationindex;          /* Station indexes                  */
    cpl_boolean    *bandflag;                  /*!< Flag if a data point is invalid (should be ignored). */
} mat_fluxelem;
/* Structure mat_oispectrum */
/**
   @ingroup oispec
   @brief Structure containing oispectrum.
*/
typedef struct {
  int              extver;                  /*!< extension version (optional)    */
  char            *dateobs;                /* observation date                */
  char            *arrayname;              /* array name                      */
  char            *insname;                /* instrument name                 */
  int              nbchannel;               /* number of spectral channels     */
  int              nbflux;                  /* number of spectrum */
  double           fov;                     /* Area on sky over which flux is integrated (arc sec)*/
  char            *fovtype;                /* Model for FOV: “FWHM” or “RADIUS” */
  char            *calstat;              /* "C" calibrated spectrum, "U" uncalibrated */
  mat_fluxelem   **list_flux;             /* vector of flux                  */
} mat_oispectrum;

/*-----------------------------------------------------------------------------
  High Level Structures
  ----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/
mat_fluxelem *mat_fluxelem_new(int nbchannel);

cpl_error_code mat_fluxelem_delete(mat_fluxelem *el);

mat_oispectrum *mat_oispectrum_new(int nbflux, int nbchannel);

cpl_error_code mat_oispectrum_delete(mat_oispectrum *oispectrum);

mat_oispectrum *mat_oispectrum_from_table(cpl_propertylist *plist,
					  cpl_table *table);

cpl_table *mat_oispectrum_to_table(mat_oispectrum *oispectrum,
				   cpl_propertylist *plist);

mat_oispectrum *mat_oispectrum_load(cpl_frame *frame);

cpl_error_code  mat_oispectrum_append(char *filename,mat_oispectrum *oispec);

cpl_error_code mat_oispectrum_get_mjd_span(mat_oispectrum *oispectrum, double *min_mjd, double *max_mjd);

#endif
