/* $Id: mat_oit3.c,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oit3.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/

#include "mat_const.h"
#include "mat_error.h"
#include "mat_utils.h"
#include "mat_oit3.h"

/*-----------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/

#define CHUNK_SIZE 1024

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/

/**
   @ingroup oit3
   @brief This method creates a new mat_t3elem structure with all pointers set to NULL.
   @param nbchannel Number of spectral channels.
   @returns Pointer to new allocated mat_t3elem structure on success or NULL on failure.
 
   This method creates a new mat_t3elem structure and sets all value to the default. For
   pointers the value is NULL for primitive data types the value is set to 0.
*/
mat_t3elem *mat_t3elem_new(int nbchannel)
{
  mat_t3elem  *el = (mat_t3elem *)cpl_calloc(1, sizeof(mat_t3elem));
  if (el == NULL)
    {
      return NULL;
    }
  el->t3phi = cpl_calloc(nbchannel, sizeof(double));
  if (el->t3phi == NULL)
    {
      mat_t3elem_delete(el);
      return NULL;
    }
  el->t3phierr = cpl_calloc(nbchannel, sizeof(double));
  if (el->t3phierr == NULL)
    {
      mat_t3elem_delete(el);
      return NULL;
    }
  el->t3amp = cpl_calloc(nbchannel, sizeof(double));
  if (el->t3amp == NULL)
    {
      mat_t3elem_delete(el);
      return NULL;
    }
  el->t3amperr = cpl_calloc(nbchannel, sizeof(double));
  if (el->t3amperr == NULL)
    {
      mat_t3elem_delete(el);
      return NULL;
    }
  el->bandflag = cpl_calloc(nbchannel, sizeof(cpl_boolean));
  if (el->bandflag == NULL)
    {
      mat_t3elem_delete(el);
      return NULL;
    }
  return el;
}

/**
   @ingroup oit3
   @brief Free memory of a single mat_t3elem data structure.
   @param el Pointer to one mat_t3elem data structure.
   @return A cpl_error_code
 
   This method frees the memory of a mat_t3elem data structure.
*/
cpl_error_code mat_t3elem_delete(mat_t3elem *el)
{
  mat_assert_value((el != NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no mat_t3elem (el) argument given");
  if (el->t3phi != NULL)
    {
      cpl_free(el->t3phi);
      el->t3phi = NULL;
    }
  if (el->t3phierr != NULL)
    {
      cpl_free(el->t3phierr);
      el->t3phierr = NULL;
    }
  if (el->t3amp != NULL)
    {
      cpl_free(el->t3amp);
      el->t3amp = NULL;
    }
  if (el->t3amperr != NULL)
    {
      cpl_free(el->t3amperr);
      el->t3amperr = NULL;
    }
  if (el->bandflag != NULL)
    {
      cpl_free(el->bandflag);
      el->bandflag = NULL;
    }
  cpl_free(el);
  return CPL_ERROR_NONE;
}

/**
   @ingroup oit3
   @brief This method creates a new mat_oit3 structure with all pointers set to NULL.
   @param nbt3       Number of measurements.
   @param nbchannel  Number of spectral channels.
   @returns Pointer to new allocated mat_oit3 structure on success or NULL on failure.
 
   This method creates a new mat_oit3 structure and sets all values to the default. For
   pointers the value is NULL for primitive data types the value is set to 0. The size
   of the element list (member list_t3 of the mat_oit3 data structure) is a multiple
   of CHUNK_SIZE (1024). This allows that after creation of a mat_oit3 data structure
   new elements (mat_t3elem) can be created an added to that structure.
*/
mat_oit3 *mat_oit3_new(int nbt3, int nbchannel)
{
  mat_oit3    *oit3;
  int          i;

  oit3 = (mat_oit3 *)cpl_calloc(1, sizeof(mat_oit3));
  if (oit3 == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oit3");
      return NULL;
    }
  oit3->nbt3 = nbt3;
  oit3->nbchannel = nbchannel;
  if (nbt3 != 0)
    { /* ensure that the vector of pointers is a multiple of CHUNK_SIZE */
      size_t size = ((nbt3 + CHUNK_SIZE - 1)/CHUNK_SIZE)*CHUNK_SIZE;
      oit3->list_t3 = (mat_t3elem **)cpl_calloc(size, sizeof(mat_t3elem *));
      if (oit3->list_t3 == NULL)
	{
	  mat_oit3_delete(oit3);
	  return NULL;
	}
      for(i = 0; i < nbt3; i++)
	{
	  oit3->list_t3[i] = mat_t3elem_new(nbchannel);
	  if (oit3->list_t3[i] == NULL)
	    {
	      mat_oit3_delete(oit3);
	      return NULL;
	    }
	}
    }
  return oit3;
}

/**
   @ingroup oit3
   @brief Creates and add a new mat_t3elem data structure to an mat_oit3 data structure.
   @param oit3  This represents an existing OI_T3 binary table.
   @returns A newly created and initialized mat_t3elem data structure.

   This function allocates a new mat_t3elem data structure representing a triple product.
   The number of wavelengths is given by the mat_oit3 data structure (member nbchannel).
   The new T3 element is added to the list of T3 elements in the specified OI-T3 binary
   table (parameter oit3). The size of the element list (member list_t3 of the mat_oit3
   data structure) is a multiple of CHUNK_SIZE (1024).
*/
mat_t3elem *mat_oit3_elem_new(mat_oit3 *oit3)
{
  mat_t3elem *elem = NULL;

  mat_assert_value((oit3!=NULL),CPL_ERROR_NULL_INPUT,NULL,"no mat_oit3 (oit3) argument given");
  if ((oit3->nbt3 % CHUNK_SIZE) == 0)
    { /* make the vector for element pointers bigger if neccessary */
      mat_t3elem **nlist = NULL;
      int          i;
      nlist = cpl_realloc(oit3->list_t3, (oit3->nbt3 + CHUNK_SIZE)*sizeof(mat_t3elem *));
      if (nlist == NULL)
	{
	  cpl_msg_error(cpl_func,"could not reallocate memory for mat_oit3->list_t3");
	  return NULL;
	}
      /* the memory is not initialized after a realloc */
      for (i = oit3->nbt3; i < oit3->nbt3 + CHUNK_SIZE; i++)
	{
	  nlist[i] = NULL;
	}
      oit3->list_t3 = nlist;
    }
  elem = mat_t3elem_new(oit3->nbchannel);
  if (elem == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oit3->list_t3[%d]", oit3->nbt3);
      return NULL;
    }
  oit3->list_t3[oit3->nbt3] = elem;
  oit3->nbt3++;
  return elem;
}

/**
   @ingroup oit3
   @brief Free memory of a single mat_oit3 data structure.
   @param oit3  Pointer to one mat_oit3 data structure.
   @return A cpl_error_code
 
   This method frees the memory of a mat_oit3 data structure. This includes
   all embedded data structures (strings and mat_t3elem data structures).
*/
cpl_error_code mat_oit3_delete(mat_oit3 *oit3)
{
  int i;

  mat_assert_value((oit3!=NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no mat_oit3 (oit3) argument given");
  if (oit3->dateobs != NULL)
    {
      cpl_free(oit3->dateobs);
      oit3->dateobs = NULL;
    }
  if (oit3->arrayname != NULL)
    {
      cpl_free(oit3->arrayname);
      oit3->arrayname = NULL;
    }
  if (oit3->insname != NULL)
    {
      cpl_free(oit3->insname);
      oit3->insname = NULL;
    }
  if (oit3->list_t3 != NULL)
    {
      for (i = 0; i < oit3->nbt3; i++)
	{
	  if (oit3->list_t3[i] != NULL)
	    {
	      mat_t3elem_delete(oit3->list_t3[i]);
	      oit3->list_t3[i] = NULL;
	    }
	}
      cpl_free(oit3->list_t3);
      oit3->list_t3 = NULL;
      oit3->nbt3 = 0;
    }
  cpl_free(oit3);
  return CPL_ERROR_NONE;
}

/**
   @ingroup oit3
   @brief Load a mat_oit3 data structure from a CPL table and a propertylist.
   @param plist  This CPL propertylist contais the header keywords.
   @param table  This CPL table contains the OI_T3 binary table.
   @return A newly allocated an filled mat_oit3 or NULL on failure.

   This function loads an entire OI_T3 binary table into an internal data structure.
   The data structure contains no members for the CORRINDX_T3AMP and CORRINDX_T3PHI columns from the OI-FITS version 2 standard.
*/
mat_oit3 *mat_oit3_from_table(cpl_propertylist *plist, cpl_table *table)
{
  mat_oit3    *oit3;
  int          j, k;
  int          nbt3;
  int          nbchannel;

  mat_assert_value((plist != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_propertylist (plist) argument given");
  mat_assert_value((table != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_table (table) argument given");

  nbt3 = cpl_table_get_nrow(table);
  /* old code
  nbchannel = cpl_array_get_size(cpl_table_get_array (table, "T3PHI",0));
  if (nbchannel == 0) nbchannel = 1; // no array!
  */
  if (cpl_table_get_column_depth(table, "T3PHI") == 0)
    {
      nbchannel = 1;
    }
  else
    {
      nbchannel = cpl_table_get_column_dimension(table, "T3PHI", 0);
    }
  cpl_msg_info(cpl_func, "T3 table has %d wavelength data", nbchannel);

  oit3 = mat_oit3_new(nbt3, nbchannel);
  if (oit3 == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oit3");
      return NULL;
    }
  oit3->extver    = mat_propertylist_get_int_default(plist, "EXTVER", 1);
  oit3->arrayname = mat_propertylist_get_string_default(NULL, 0, plist, "ARRNAME", "empty");
  oit3->dateobs   = mat_propertylist_get_string_default(NULL, 0, plist, "DATE-OBS", "empty");
  oit3->insname   = mat_propertylist_get_string_default(NULL, 0, plist, "INSNAME", "empty");
  for(j = 0; j < nbt3; j++)
    {
      mat_t3elem *el = oit3->list_t3[j];
      // copy the values from the table into the t3 element
      el->targetid = cpl_table_get_int (table, "TARGET_ID", j, NULL);
      el->time = cpl_table_get_double (table, "TIME", j, NULL);
      el->dateobsmjd = cpl_table_get_double (table, "MJD", j, NULL);
      el->exptime = cpl_table_get_double (table, "INT_TIME", j, NULL);
      el->u1coord = cpl_table_get_double (table, "U1COORD", j ,NULL);
      el->v1coord = cpl_table_get_double (table, "V1COORD", j, NULL);
      el->u2coord = cpl_table_get_double (table, "U2COORD", j ,NULL);
      el->v2coord = cpl_table_get_double (table, "V2COORD", j, NULL);
      el->stationindex[0]= cpl_array_get_int(cpl_table_get_array(table, "STA_INDEX", j), 0, NULL);
      el->stationindex[1] = cpl_array_get_int(cpl_table_get_array(table, "STA_INDEX", j), 1, NULL);
      el->stationindex[2] = cpl_array_get_int(cpl_table_get_array(table, "STA_INDEX", j), 2, NULL);
      if (nbchannel == 1)
	{
	  el->t3phi[0] = cpl_table_get_double(table,"T3PHI", j, NULL);
	  el->t3phierr[0] = cpl_table_get_double(table, "T3PHIERR", j, NULL);
	  el->t3amp[0] = cpl_table_get_double(table,"T3AMP", j, NULL);
	  el->t3amperr[0] = cpl_table_get_double(table, "T3AMPERR", j, NULL);
	  el->bandflag[0] = (cpl_table_get_int(table,"FLAG", j, NULL) == 1);
	}
      else
	{
	  for (k = 0; k < oit3->nbchannel; k++)
	    {
	      el->t3phi[k] = cpl_array_get_double(cpl_table_get_array(table,"T3PHI", j), k, NULL);
	      el->t3phierr[k] = cpl_array_get_double(cpl_table_get_array(table, "T3PHIERR", j), k, NULL);
	      el->t3amp[k] = cpl_array_get_double(cpl_table_get_array(table,"T3AMP", j), k, NULL);
	      el->t3amperr[k] = cpl_array_get_double(cpl_table_get_array(table, "T3AMPERR", j), k, NULL);
	      el->bandflag[k] = (cpl_array_get_int(cpl_table_get_array(table,"FLAG", j), k, NULL) == 1);
	    }
	}
    }
  return oit3;
}

/**
   @ingroup oit3
   @brief Convert a mat_oit3 data structure into a CPL table and a propertylist.
   @param oit3    The data structure containing a whole OI_T3 binary table.
   @param plist   This propertylist will contain the header keywords of the OI_T3 binary table.
   @return A CPL table containing the OI_T3 binary table of NULL on failure.

   A new CPL table will be created which will contain all triple products. The header keywords of
   the OI_T3 binary table will be stored in the provided propertylist (parameter plist).
   The CPL table does not contain the CORRINDX_T3AMP and CORRINDX_T3PHI columns from the OI-FITS version 2 standard.
*/
cpl_table *mat_oit3_to_table(mat_oit3 *oit3, cpl_propertylist *plist)
{
  cpl_table    *table;
  cpl_array    *arr;
  int           i, j;
  int          *flags;

  if (plist != NULL)
    {
      cpl_propertylist_append_string(plist,"EXTNAME","OI_T3");
      cpl_propertylist_append_int(plist, "EXTVER", oit3->extver);
      if ( cpl_propertylist_has(plist,"OI_REVN") )
	{
	  cpl_propertylist_erase(plist,"OI_REVN");
	}
      cpl_propertylist_append_int(plist,"OI_REVN",2);
      cpl_propertylist_append_string(plist,"DATE-OBS",oit3->dateobs);
      cpl_propertylist_append_string(plist,"ARRNAME",oit3->arrayname);
      cpl_propertylist_append_string(plist,"INSNAME",oit3->insname);
    }
  flags = (int *)cpl_calloc(oit3->nbchannel, sizeof(int));
  if (flags == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for flag vector");
      return NULL;
    }
  table = cpl_table_new(oit3->nbt3);
  if (table == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_table");
      cpl_free(flags);
      return NULL;
    }
  // Add columns
  cpl_table_new_column(table, "TARGET_ID", CPL_TYPE_INT);
  cpl_table_new_column(table, "TIME", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "TIME","s");
  cpl_table_new_column(table, "MJD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "MJD","day");
  cpl_table_new_column(table, "INT_TIME", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "INT_TIME","s");
  if (oit3->nbchannel == 1)
    {
      cpl_table_new_column(table,"T3AMP", CPL_TYPE_DOUBLE);
      cpl_table_new_column(table,"T3AMPERR", CPL_TYPE_DOUBLE);
      cpl_table_new_column(table,"T3PHI", CPL_TYPE_DOUBLE);
      cpl_table_new_column(table,"T3PHIERR", CPL_TYPE_DOUBLE);
    }
  else
    {
      cpl_table_new_column_array(table,"T3AMP", CPL_TYPE_DOUBLE, oit3->nbchannel);
      cpl_table_new_column_array(table,"T3AMPERR", CPL_TYPE_DOUBLE, oit3->nbchannel);
      cpl_table_new_column_array(table,"T3PHI", CPL_TYPE_DOUBLE, oit3->nbchannel);
      cpl_table_new_column_array(table,"T3PHIERR", CPL_TYPE_DOUBLE, oit3->nbchannel);
    }
  cpl_table_set_column_unit(table, "T3PHI","deg");
  cpl_table_set_column_unit(table, "T3PHIERR","deg");
  cpl_table_new_column(table, "U1COORD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "U1COORD","m");
  cpl_table_new_column(table, "V1COORD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "V1COORD","m");
  cpl_table_new_column(table, "U2COORD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "U2COORD","m");
  cpl_table_new_column(table, "V2COORD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "V2COORD","m");
  cpl_table_new_column_array(table,"STA_INDEX", CPL_TYPE_INT, 3);
  if (oit3->nbchannel == 1)
    {
      cpl_table_new_column(table, "FLAG", CPL_TYPE_INT);
    }
  else
    {
      cpl_table_new_column_array(table, "FLAG", CPL_TYPE_INT, oit3->nbchannel);
    }

  // Fill the table
  for(i = 0; i < oit3->nbt3; i++)
    {
      mat_t3elem  *el = oit3->list_t3[i];
      cpl_table_set_int(table, "TARGET_ID", i, el->targetid);
      cpl_table_set_double(table,"TIME",i, el->time);
      cpl_table_set_double(table,"MJD",i, el->dateobsmjd);
      cpl_table_set_double(table,"INT_TIME",i, el->exptime);
      if (oit3->nbchannel == 1)
	{
	  cpl_table_set_double(table,"T3AMP",i, el->t3amp[0]);
	  cpl_table_set_double(table,"T3AMPERR",i, el->t3amperr[0]);
	  cpl_table_set_double(table,"T3PHI",i, el->t3phi[0]);
	  cpl_table_set_double(table,"T3PHIERR",i, el->t3phierr[0]);
	}
      else
	{
	  arr = cpl_array_wrap_double(el->t3amp, oit3->nbchannel);
	  cpl_table_set_array(table,"T3AMP",i,arr);
	  cpl_array_unwrap(arr);
	  arr = cpl_array_wrap_double(el->t3amperr, oit3->nbchannel);
	  cpl_table_set_array(table,"T3AMPERR",i,arr);
	  cpl_array_unwrap(arr);
	  arr = cpl_array_wrap_double(el->t3phi, oit3->nbchannel);
	  cpl_table_set_array(table,"T3PHI",i,arr);
	  cpl_array_unwrap(arr);
	  arr = cpl_array_wrap_double(el->t3phierr, oit3->nbchannel);
	  cpl_table_set_array(table,"T3PHIERR",i,arr);
	  cpl_array_unwrap(arr);
	}
      cpl_table_set_double(table,"U1COORD",i, el->u1coord);
      cpl_table_set_double(table,"V1COORD",i, el->v1coord);
      cpl_table_set_double(table,"U2COORD",i, el->u2coord);
      cpl_table_set_double(table,"V2COORD",i, el->v2coord);
      arr = cpl_array_wrap_int(el->stationindex, 3);
      cpl_table_set_array(table, "STA_INDEX", i, arr);
      cpl_array_unwrap(arr);
      if (oit3->nbchannel == 1)
	{
	  cpl_table_set_int(table, "FLAG", i, (el->bandflag[0] != 0));
	}
      else
	{
	  for(j = 0; j < oit3->nbchannel; j++)
	    {
	      if (el->bandflag[j])
		{
		  flags[j] = 1;
		}
	      else
		{
		  flags[j] = 0;
		}
	    }
	  arr = cpl_array_wrap_int(flags, oit3->nbchannel);
	  cpl_table_set_array(table,"FLAG",i, arr);
	  cpl_array_unwrap(arr);
	}
    }


  cpl_table_set_column_savetype(table,"TARGET_ID",CPL_TYPE_SHORT);
  cpl_table_set_column_savetype(table,"STA_INDEX",CPL_TYPE_SHORT);
  cpl_table_set_column_savetype(table,"FLAG",CPL_TYPE_BOOL);
  cpl_free(flags);
  return table;
}

/**
   @ingroup oit3
   @brief Load a OI_T3 binary table from a frame into a mat_oit3 structure.
   @param frame  A frame containing a OI_T3 binary table.
   @return A newly allocated an filled mat_oit3 or NULL on failure.

   This function scans all extensions inside a frame and loads the first OI_T3 binary table
   into a newly created mat_oit3 data structure. If the frame does not contain any
   OI_T3 binary table, NULL is returned.
*/
mat_oit3 *mat_oit3_load(cpl_frame *frame)
{
  int nbExtent;
  int  iExtent;
 
  mat_assert_value((frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_frame (frame) argument given");
  nbExtent = cpl_frame_get_nextensions(frame);
  for(iExtent = 1; iExtent <= nbExtent; iExtent++)
    {
      cpl_propertylist *plist = cpl_propertylist_load(cpl_frame_get_filename(frame), iExtent);
      char *keyExtname = (char *)cpl_propertylist_get_string(plist, "EXTNAME");
      if ((keyExtname != NULL) && (strcmp(keyExtname, "OI_T3") == 0))
	{ /* Only the first OI_T3 binary table is loaded! */
	  cpl_table *table = cpl_table_load(cpl_frame_get_filename(frame), iExtent, 0);
	  mat_oit3 *oit3 = mat_oit3_from_table(plist, table);
	  cpl_table_delete(table);
	  cpl_propertylist_delete(plist);
	  return oit3;
      }
      cpl_propertylist_delete(plist);
    }
  return NULL;
}

/**
   @ingroup oit3
   @brief Append an OI_T3 binary table to an existing OI-FITS file.
   @param filename  Name of the FITS file.
   @param oit3      The mat_oit3 structure containing a whole OI_T3 binary table.
   @return cpl_error_code

   This function converts a mat_oit3 data structure into a CPL binary table and
   appends this at the end of an already existing FITS file.
*/
cpl_error_code mat_oit3_append(char *filename, mat_oit3 *oit3)
{
  cpl_table *oit3Tab = NULL;
  cpl_propertylist *keyTab = NULL;
  cpl_errorstate prestate = cpl_errorstate_get();

  /*Check input parameters*/
  mat_assert_value((oit3!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oit3 (oit3) argument given");
  mat_assert_value((filename!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no filename given");

  keyTab=cpl_propertylist_new();
  oit3Tab = mat_oit3_to_table(oit3, keyTab);
  cpl_table_save(oit3Tab,NULL,keyTab,filename,CPL_IO_EXTEND);

  cpl_propertylist_delete(keyTab);
  cpl_table_delete(oit3Tab);
  if (!cpl_errorstate_is_equal(prestate))
    {
      cpl_errorstate_set(prestate);
      return CPL_ERROR_UNSPECIFIED;
    }
  return CPL_ERROR_NONE;
}

/**
   @ingroup oit3
   @brief Get the time span of the observation as MJD values (beginning and end).
   @param oit3     The mat_oit3 structure containing a OI_T3 binary table.
   @param min_mjd  The beginning of the observation.
   @param max_mjd  The end of the observation.
   @return cpl_error_code

   The time span of a binary table is used to group OI_VIS, OI_VIS2 and OI_T3 binary
   tables into one tupel (see the mat_oifits data structure).
*/
cpl_error_code mat_oit3_get_mjd_span(mat_oit3 *oit3, double *min_mjd, double *max_mjd)
{
  int i;

  /*Check input parameters*/
  mat_assert_value((oit3!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oit3 (oit3) argument given");
  mat_assert_value((min_mjd!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no start MJD given");
  mat_assert_value((max_mjd!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no end MJD given");
  *min_mjd =  1.0e12;
  *max_mjd = -1.0e12;
  for (i = 0; i < oit3->nbt3; i++)
    {
      mat_t3elem *el = oit3->list_t3[i];
      *min_mjd = fmin(*min_mjd, el->dateobsmjd);
      *max_mjd = fmax(*max_mjd, el->dateobsmjd);
    }
  return CPL_ERROR_NONE;
}
