/* $Id: mat_oit3.h,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oit3.h $
 */

#ifndef MAT_OIT3_H
#define MAT_OIT3_H

/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/

#include "cpl.h"

/*-----------------------------------------------------------------------------
  Defines
  ----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  Low Level Structures
  ----------------------------------------------------------------------------*/

/* Structure mat_t3elem */
/**
   @ingroup oit3
   @brief Structure containing one measurement of an OI_T3.

   This data structure represents one entry of an OI_T3 binary table.
*/
typedef struct {
  int          targetid;         /*!< Target number as index into the OI_TARGET binary table (always 1 for MATISSE OI-FITS files). */
  double       time;             /*!< UTC time of observation. */
  double       dateobsmjd;       /*!< UTC date of observation (MJD). */
  double       exptime;          /*!< Integration time per measurement in seconds. */
  double      *t3phi;            /*!< Array of triple product phases in degree. */
  double      *t3phierr;         /*!< Array of error in triple product phases in degree. */
  double      *t3amp;            /*!< Array of triple product amplitudes. */
  double      *t3amperr;         /*!< Array of error in triple product amplitudes. */
  double       u1coord;          /*!< U coordinate of baseline AB of the triangle in meter. */
  double       v1coord;          /*!< V coordinate of baseline AB of the triangle in meter. */
  double       u2coord;          /*!< U coordinate of baseline BC of the triangle in meter. */
  double       v2coord;          /*!< V coordinate of baseline BC of the triangle in meter. */
  int          stationindex[3];  /*!< Station numbers contributing to the data */
  cpl_boolean *bandflag;         /*!< Flag if a data point is invalid (should be ignored). */
} mat_t3elem;

/* Structure mat_oit3 */
/**
   @ingroup oit3
   @brief Structure containing an OI_T3 binary table.

   This data structure represents one OI_T3 binary table of an OI-FITS file. A standard compliant OI-FITS file
   can contain several of these binary tables. This is implemented in the mat_oifits data structure.
*/
typedef struct {
  int          extver;     /*!< Unique extension version number (optional). */
  char        *dateobs;    /*!< UTC start date of observations. */
  char        *arrayname;  /*!< Identifies the corresponding OI_ARRAY binary table. */
  char        *insname;    /*!< Identifies the corresponding OI_WAVELENGTH binary table. */
  int          nbchannel;  /*!< Number of spectral channels (equivalent to the corresponding OI_WAVELENGTH binary table). */
  int          nbt3;       /*!< Number of closure phase data in this binary table (equivalent with the number of rows). */
  mat_t3elem **list_t3;    /*!< List containing all closure phase data stored in mat_t3elem data structures. */
} mat_oit3;


/*-----------------------------------------------------------------------------
  High Level Structures
  ----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/
mat_t3elem *mat_t3elem_new(int nbchannel);

cpl_error_code mat_t3elem_delete(mat_t3elem *el);

mat_oit3 *mat_oit3_new(int nbt3, int nbchannel);

mat_t3elem *mat_oit3_elem_new(mat_oit3 *oit3);

cpl_error_code mat_oit3_delete(mat_oit3 *oit3);

mat_oit3 *mat_oit3_from_table(cpl_propertylist *plist,
			      cpl_table *table);

cpl_table *mat_oit3_to_table(mat_oit3 *oit3,
			     cpl_propertylist *plist);

mat_oit3 *mat_oit3_load(cpl_frame *frame);

cpl_error_code  mat_oit3_append(char *filename,mat_oit3 *oit3);

cpl_error_code mat_oit3_get_mjd_span(mat_oit3 *oit3, double *min_mjd, double *max_mjd);


#endif
