/* $Id: mat_oitarget.c,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oitarget.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/
#include <string.h>
#include "mat_error.h"
#include "mat_oitarget.h"

/*-----------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/

/**
   @ingroup oitar
   @brief This method creates a new mat_oitarget structure with all pointers set to NULL.
   @returns Pointer to new allocated mat_oitarget structure on success or NULL on failure.
 
   This method creates a new mat_oitarget structure and sets all value to the default. For
   pointers the value is NULL for primitive data types the value is set to 0.
*/
mat_oitarget *mat_oitarget_new(void)
{
  mat_oitarget *oitarget;

  oitarget = (mat_oitarget *)cpl_calloc(1, sizeof(mat_oitarget));
  if (oitarget == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oitarget");
      return NULL;
    }
  return oitarget;
}

/**
   @ingroup oitar
   @brief This method deletes a mat_oitarget data structure and all embedded memory.
   @param  oitarget  A previously allocated mat_oitarget data structure.
   @return error code

   This method deletes a mat_oitarget data structure. All local strings (e.g. target name)
   are freed too.
*/
cpl_error_code mat_oitarget_delete(mat_oitarget *oitarget)
{
  mat_assert_value((oitarget != NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oitarget (oitarget) argument given");
  if (oitarget->targetname != NULL)
    {
      cpl_free(oitarget->targetname);
      oitarget->targetname = NULL;
    }
  if (oitarget->veltyp != NULL)
    {
      cpl_free(oitarget->veltyp);
      oitarget->veltyp = NULL;
    }
  if (oitarget->veldef != NULL)
    {
      cpl_free(oitarget->veldef);
      oitarget->veldef = NULL;
    }
  if (oitarget->spectyp != NULL)
    {
      cpl_free(oitarget->spectyp);
      oitarget->spectyp = NULL;
    }
  if (oitarget->category != NULL)
    {
      cpl_free(oitarget->category);
      oitarget->category = NULL;
    }
  cpl_free(oitarget);
  return CPL_ERROR_NONE;
}

/**
   @ingroup oitar
   @brief Load a mat_oitarget data structure from a CPL table and a propertylist.
   @param plist  This CPL propertylist contais the header keywords.
   @param table  This CPL table contains the OI_TARGET binary table.
   @return A newly allocated an filled mat_oitarget or NULL on failure.

   Despite the name of this function, only the first target in an OI_TARGET binary table is loaded.
   The data structure contains no member for the CATEGORY column from the OI-FITS version 2 standard.
*/
mat_oitarget *mat_oitarget_from_table(cpl_propertylist *plist, cpl_table *table)
{
  mat_oitarget *oitarget;

  mat_assert_value((plist != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_propertylist (plist) argument given");
  mat_assert_value((table != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_table (table) argument given");
  oitarget = mat_oitarget_new();
  if (oitarget == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oitarget");
      return NULL;
    }
  oitarget->targetname = cpl_strdup(cpl_table_get_string(table, "TARGET", 0));
  oitarget->targetid = cpl_table_get_int(table, "TARGET_ID", 0, NULL);
  oitarget->equinox = (double)cpl_table_get_float(table, "EQUINOX", 0, NULL);
  oitarget->raep0 = cpl_table_get_double(table, "RAEP0", 0, NULL);
  oitarget->decep0 = cpl_table_get_double(table, "DECEP0", 0,NULL);
  oitarget->raerr = cpl_table_get_double(table, "RA_ERR", 0, NULL);
  oitarget->decerr = cpl_table_get_double(table, "DEC_ERR", 0, NULL);
  oitarget->sysvel = cpl_table_get_double(table, "SYSVEL", 0, NULL);
  oitarget->veldef = cpl_strdup(cpl_table_get_string(table, "VELDEF", 0));
  oitarget->veltyp = cpl_strdup(cpl_table_get_string(table, "VELTYP", 0));
  oitarget->pmra = cpl_table_get_double(table, "PMRA", 0, NULL);
  oitarget->pmdec = cpl_table_get_double(table, "PMDEC", 0, NULL);
  oitarget->pmraerr = cpl_table_get_double(table, "PMRA_ERR", 0, NULL);
  oitarget->pmdecerr = cpl_table_get_double(table, "PMDEC_ERR", 0, NULL);
  oitarget->parallax = (double)cpl_table_get_float(table, "PARALLAX", 0, NULL);
  oitarget->paraerr = (double)cpl_table_get_float(table, "PARA_ERR", 0, NULL);
  oitarget->spectyp = cpl_strdup(cpl_table_get_string(table, "SPECTYP", 0));
  oitarget->category = cpl_strdup(cpl_table_get_string(table, "CATEGORY", 0));
  return oitarget;
}

/**
   @ingroup oitar
   @brief Convert a mat_oitarget data structure into a CPL table and a propertylist.
   @param oitarget  The data structure containing the first target of a OI_TARGET binary table.
   @param plist     This propertylist will contain the header keywords of the OI_TARGET binary table.
   @return A CPL table containing the OI_TARGET binary table of NULL on failure.

   A new CPL table will be created which will contain exactly one target. The header keywords of
   the OI_TARGET binary table will be stored in the provided propertylist (parameter plist).
   The CPL table does not contain the CATEGORY column from the OI-FITS version 2 standard.
*/
cpl_table *mat_oitarget_to_table(mat_oitarget *oitarget, cpl_propertylist *plist)
{
  cpl_table   *table;

  if (plist != NULL)
    {
      cpl_propertylist_append_string(plist,"EXTNAME","OI_TARGET");
      if ( cpl_propertylist_has(plist,"OI_REVN") )
	{
	  cpl_propertylist_erase(plist,"OI_REVN");
	}
      cpl_propertylist_append_int(plist,"OI_REVN",2);
    }
  table = cpl_table_new(1);
  if (table == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_table");
      return NULL;
    }
  /*Add columns*/
  cpl_table_new_column(table, "TARGET_ID", CPL_TYPE_INT);
  cpl_table_new_column(table, "TARGET", CPL_TYPE_STRING);
  cpl_table_new_column(table, "RAEP0", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "RAEP0", "deg");
  cpl_table_new_column(table, "DECEP0", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "DECEP0", "deg");
  cpl_table_new_column(table, "EQUINOX", CPL_TYPE_FLOAT);
  cpl_table_set_column_unit(table, "EQUINOX", "yr");
  cpl_table_new_column(table, "RA_ERR", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "RA_ERR", "deg");
  cpl_table_new_column(table, "DEC_ERR", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "DEC_ERR", "deg");
  cpl_table_new_column(table, "SYSVEL", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "SYSVEL", "m/s");
  cpl_table_new_column(table, "VELTYP", CPL_TYPE_STRING);
  cpl_table_new_column(table, "VELDEF", CPL_TYPE_STRING);
  cpl_table_new_column(table, "PMRA", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "PMRA", "deg/yr");
  cpl_table_new_column(table, "PMDEC", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "PMDEC", "deg/yr");
  cpl_table_new_column(table, "PMRA_ERR", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "PMRA_ERR", "deg/yr");
  cpl_table_new_column(table, "PMDEC_ERR", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "PMDEC_ERR", "deg/yr");
  cpl_table_new_column(table, "PARALLAX", CPL_TYPE_FLOAT);
  cpl_table_set_column_unit(table, "PARALLAX", "deg");
  cpl_table_new_column(table, "PARA_ERR", CPL_TYPE_FLOAT);
  cpl_table_set_column_unit(table, "PARA_ERR", "deg");
  cpl_table_new_column(table, "SPECTYP", CPL_TYPE_STRING);
  cpl_table_new_column(table, "CATEGORY", CPL_TYPE_STRING);

  /*Fill the table*/
  cpl_table_set_int(table,"TARGET_ID",0,oitarget->targetid);
  cpl_table_set_string(table,"TARGET",0,oitarget->targetname);
  cpl_table_set_double(table,"RAEP0",0,oitarget->raep0);
  cpl_table_set_double(table,"DECEP0",0,oitarget->decep0);
  cpl_table_set_float(table,"EQUINOX",0,(float)
		      oitarget->equinox);
  cpl_table_set_double(table,"RA_ERR",0,oitarget->raerr);
  cpl_table_set_double(table,"DEC_ERR",0,oitarget->decerr);
  cpl_table_set_double(table,"SYSVEL",0,oitarget->sysvel);
  cpl_table_set_string(table,"VELTYP",0,oitarget->veltyp);
  cpl_table_set_string(table,"VELDEF",0,oitarget->veldef);
  cpl_table_set_double(table,"PMRA",0,oitarget->pmra);
  cpl_table_set_double(table,"PMDEC",0,oitarget->pmdec);
  cpl_table_set_double(table,"PMRA_ERR",0,oitarget->pmraerr);
  cpl_table_set_double(table,"PMDEC_ERR",0,oitarget->pmdecerr);
  cpl_table_set_float(table,"PARALLAX",0,(float)
		      oitarget->parallax);
  cpl_table_set_float(table,"PARA_ERR",0,(float)
		      oitarget->parallax);
  cpl_table_set_string(table,"SPECTYP",0,oitarget->spectyp);
  cpl_table_set_string(table,"CATEGORY",0,oitarget->category);
  
  cpl_table_set_column_savetype(table, "TARGET_ID", CPL_TYPE_SHORT);
  return table;
}

/**
   @ingroup oitar
   @brief Load a OI_TARGET binary table from a frame into a mat_oitarget structure.
   @param frame  A frame containing a OI_TARGET binary table.
   @return A new and filled mat_oitarget data structure or NULL on failure.

   This function scans all extensions inside a frame and loads the first OI_TARGET binary table
   into a newly created mat_oitarget data structure. If the frame does not contain any
   OI_TARGET binary table, NULL is returned.
*/
mat_oitarget * mat_oitarget_load(cpl_frame *frame)
{
  int  nbExtent;
  int  iExtent;
   
  mat_assert_value((frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_frame (frame) argument given");
  nbExtent = cpl_frame_get_nextensions(frame);
  for(iExtent = 1; iExtent <= nbExtent; iExtent++)
    {
      cpl_propertylist *plist = cpl_propertylist_load(cpl_frame_get_filename(frame), iExtent);
      char *keyExtname = (char *)cpl_propertylist_get_string(plist, "EXTNAME");
      if ((keyExtname != NULL) && (strcmp(keyExtname, "OI_TARGET") == 0))
	{ /* Only the first OI_TARGET binary table is loaded! */
	  cpl_table *table = cpl_table_load(cpl_frame_get_filename(frame), iExtent, 0);
	  mat_oitarget *oitarget = mat_oitarget_from_table(plist, table);
	  cpl_table_delete(table);
	  cpl_propertylist_delete(plist);
	  return oitarget;
    	}
      cpl_propertylist_delete(plist);
    }
  return NULL;
}

/**
   @ingroup oitar
   @brief Append an OI_TARGET binary table to an existing OI-FITS file.
   @param filename  Name of the FITS file.
   @param oitarget  The mat_oitarget structure containing one target of an OI_TARGET binary table.
   @return cpl_error_code

   This function converts a mat_oitarget data structure into a CPL binary table and
   appends this at the end of an already existing FITS file.
*/
cpl_error_code  mat_oitarget_append(char *filename, mat_oitarget *oitarget)
{
  cpl_table *oitargetTab = NULL;
  cpl_propertylist *keyTab = NULL;
  cpl_errorstate prestate = cpl_errorstate_get();

  /*Check input parameters*/
  mat_assert_value((oitarget!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oitarget (oitarget) argument given");
  mat_assert_value((filename!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no filename given");

  keyTab=cpl_propertylist_new();
  oitargetTab = mat_oitarget_to_table(oitarget, keyTab);

  cpl_table_save(oitargetTab,NULL,keyTab,filename,CPL_IO_EXTEND);

  cpl_propertylist_delete(keyTab);
  cpl_table_delete(oitargetTab);
  if (!cpl_errorstate_is_equal(prestate))
    {
      cpl_errorstate_set(prestate);
      return CPL_ERROR_UNSPECIFIED;
    }
    
  return CPL_ERROR_NONE;
}

/**
   @ingroup oitar
   @brief Compare two mat_oitarget data structures.
   @param oitarget1  The first mat_oitarget data structure.
   @param oitarget2  The second mat_oitarget data structure.
   @return CPL_TRUE if oitarget1 and oitaget2 are identical, CPL_FALSE otherwise.
*/
cpl_boolean  mat_oitarget_compare(mat_oitarget *oitarget1, 
				  mat_oitarget *oitarget2)
{
  mat_assert_value((oitarget1!=NULL), CPL_ERROR_NULL_INPUT, CPL_FALSE,
		   "no mat_oitarget (oitarget1) argument given");
  mat_assert_value((oitarget2!=NULL), CPL_ERROR_NULL_INPUT, CPL_FALSE,
		   "no mat_oitarget (oitarget2) argument given");
  if (!strcmp(oitarget1->targetname, oitarget2->targetname)
      /* && (oitarget1->targetid == oitarget2->targetid) */
      /* && (oitarget1->equinox == oitarget2->equinox) */
      /* && (oitarget1->raep0 == oitarget2->raep0) */
      /* && (oitarget1->decep0 == oitarget2->decep0) */
      /* && (oitarget1->raerr == oitarget2->raerr) */
      /* && (oitarget1->decerr == oitarget2->decerr) */
      /* && (oitarget1->sysvel == oitarget2->sysvel) */
      /* && !strcmp(oitarget1->veltyp, oitarget2->veltyp) */
      /* && !strcmp(oitarget1->veldef, oitarget2->veldef) */
      /* && !strcmp(oitarget1->spectyp, oitarget2->spectyp) */
      /* && !strcmp(oitarget1->category, oitarget2->category) */
      /* && (oitarget1->pmra == oitarget2->pmra) */
      /* && (oitarget1->pmdec == oitarget2->pmdec) */
      /* && (oitarget1->pmraerr == oitarget2->pmraerr) */
      /* && (oitarget1->pmdecerr == oitarget2->pmdecerr) */
      /* && (oitarget1->parallax == oitarget2->parallax) */
      /* && (oitarget1->paraerr == oitarget2->paraerr) */
      )
    {
      return CPL_TRUE;
      
    }
  else
    {
      return CPL_FALSE;
    }
}
