/* $Id: mat_oivis.c,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oivis.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/

#include "mat_const.h"
#include "mat_error.h"
#include "mat_utils.h"
#include "mat_oivis.h"

/*-----------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/

#define CHUNK_SIZE 1024

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/

/**
   @ingroup oivis
   @brief This method creates a new mat_viselem structure with all pointers set to NULL.
   @param nbchannel Number of spectral channels.
   @returns Pointer to a new allocated mat_viselem structure on success or NULL on failure.
 
   This method creates a new mat_viselem structure and sets all values to the default. For
   pointers the value is NULL for primitive data types the value is set to 0.
*/
mat_viselem *mat_viselem_new(int nbchannel)
{
  mat_viselem *el;

  el = (mat_viselem *)cpl_calloc(1, sizeof(mat_viselem));
  if (el == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_viselem");
      return NULL;
    }
  
  el->visamp = cpl_calloc(nbchannel, sizeof(double));
  if (el->visamp == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_viselem->visamp");
      mat_viselem_delete(el);
      return NULL;
    }
  el->visamperr = cpl_calloc(nbchannel, sizeof(double));
  if (el->visamperr == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_viselem->visamperr");
      mat_viselem_delete(el);
      return NULL;
    }
  
  /* el->cfxamp = cpl_calloc(nbchannel, sizeof(double)); */
  /* if (el->cfxamp == NULL) */
  /*   { */
  /*     cpl_msg_error(cpl_func,"could not allocate memory for mat_viselem->cfxamp"); */
  /*     mat_viselem_delete(el); */
  /*     return NULL; */
  /*   } */
  /* el->cfxamperr = cpl_calloc(nbchannel, sizeof(double)); */
  /* if (el->cfxamperr == NULL) */
  /*   { */
  /*     cpl_msg_error(cpl_func,"could not allocate memory for mat_viselem->cfxamperr"); */
  /*     mat_viselem_delete(el); */
  /*     return NULL; */
  /*   } */
  
  el->visphi = cpl_calloc(nbchannel, sizeof(double));
  if (el->visphi == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_viselem->visphi");
      mat_viselem_delete(el);
      return NULL;
    }
  el->visphierr = cpl_calloc(nbchannel, sizeof(double));
  if (el->visphierr == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_viselem->visphierr");
      mat_viselem_delete(el);
      return NULL;
    }
  
  el->bandflag = cpl_calloc(nbchannel, sizeof(cpl_boolean));
  if (el->bandflag == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_viselem->bandflag");
      mat_viselem_delete(el);
      return NULL;
    }
  return el;
}

/**
   @ingroup oivis
   @brief Free memory of a single mat_viselem data structure.
   @param el Pointer to one mat_viselem data structure.
   @return A cpl_error_code
 
   This method frees the memory of a mat_viselem data structure.
*/
cpl_error_code mat_viselem_delete(mat_viselem *el)
{
  mat_assert_value((el != NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no mat_viselem (el) argument given");
  
  if (el->visamp != NULL)
    {
      cpl_free(el->visamp);
      el->visamp = NULL;
    }
  if (el->visamperr != NULL)
    {
      cpl_free(el->visamperr);
      el->visamperr = NULL;
    }
  
  /* if (el->cfxamp != NULL) */
  /*   { */
  /*     cpl_free(el->cfxamp); */
  /*     el->cfxamp = NULL; */
  /*   } */
  /* if (el->cfxamperr != NULL) */
  /*   { */
  /*     cpl_free(el->cfxamperr); */
  /*     el->cfxamperr = NULL; */
  /*   } */
  
  if (el->visphi != NULL)
    {
      cpl_free(el->visphi);
      el->visphi = NULL;
    }
  if (el->visphierr != NULL)
    {
      cpl_free(el->visphierr);
      el->visphierr = NULL;
    }
  if (el->bandflag != NULL)
    {
      cpl_free(el->bandflag);
      el->bandflag = NULL;
    }
  cpl_free(el);
  return CPL_ERROR_NONE;
}

/**
   @ingroup oivis
   @brief This method creates a new mat_oivis structure with all pointers set to NULL.
   @param nbvis      Number of measurements.
   @param nbchannel  Number of spectral channels.
   @returns Pointer to a new allocated mat_oivis structure on success or NULL on failure.
 
   This method creates a new mat_oivis structure and sets all values to the default. For
   pointers the value is NULL for primitive data types the value is set to 0. The size
   of the element list (member list_vis of the mat_oivis data structure) is a multiple
   of CHUNK_SIZE (1024). This allows that after creation of a mat_vis data structure
   new elements (mat_viselem) can be created and added to that structure.
   If nbvis is 0, the first call of mat_oivis_elem_new() also creates the first list of
   visibilities (member list_vis).
*/
mat_oivis *mat_oivis_new(int nbvis, int nbchannel)
{
  int i;
  mat_oivis *oivis;

  oivis = (mat_oivis *)cpl_calloc(1, sizeof(mat_oivis));
  if (oivis == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oivis");
      return NULL;
    }
  oivis->nbvis = nbvis;
  oivis->nbchannel = nbchannel;
  if (nbvis != 0)
    { /* ensure that the vector of pointers is a multiple of CHUNK_SIZE */
      size_t size = ((nbvis + CHUNK_SIZE - 1)/CHUNK_SIZE)*CHUNK_SIZE;
      oivis->list_vis = (mat_viselem **)cpl_calloc(size, sizeof(mat_viselem *));
      if (oivis->list_vis == NULL)
	{
	  cpl_msg_error(cpl_func,"could not allocate memory for mat_oivis->list_vis");
	  mat_oivis_delete(oivis);
	  return NULL;
	}
      for (i = 0; i < nbvis; i++)
	{
	  oivis->list_vis[i] = mat_viselem_new(nbchannel);
	  if (oivis->list_vis[i] == NULL)
	    {
	      cpl_msg_error(cpl_func,"could not allocate memory for mat_oivis->list_vis[i]");
	      mat_oivis_delete(oivis);
	      return NULL;
	    }
	}
    }
  return oivis;
}

/**
   @ingroup oivis
   @brief Creates and add a new mat_viselem data structure to an mat_oivis data structure.
   @param oivis  This represents an existing OI_VIS binary table.
   @returns A newly created and initialized mat_viselem data structure.

   This function allocates a new mat_viselem data structure representing a complex visibility.
   The number of wavelengths is given by the mat_oivis data structure (member nbchannel).
   The new VIS element is added to the list of VIS elements in the specified OI_VIS binary
   table (parameter oivis). The size of the element list (member list_vis of the mat_oivis
   data structure) is a multiple of CHUNK_SIZE (1024).
*/
mat_viselem *mat_oivis_elem_new(mat_oivis *oivis)
{
  mat_viselem *elem = NULL;

  mat_assert_value((oivis!=NULL),CPL_ERROR_NULL_INPUT,NULL,"no mat_oivis (oivis) argument given");
  if ((oivis->nbvis % CHUNK_SIZE) == 0)
    { /* make the vector for element pointers bigger if neccessary */
      mat_viselem **nlist = NULL;
      int           i;
      nlist = cpl_realloc(oivis->list_vis, (oivis->nbvis + CHUNK_SIZE)*sizeof(mat_viselem *));
      if (nlist == NULL)
	{
	  cpl_msg_error(cpl_func,"could not reallocate memory for mat_oivis->list_vis");
	  return NULL;
	}
      /* the memory is not initialized after a realloc */
      for (i = oivis->nbvis; i < oivis->nbvis + CHUNK_SIZE; i++)
	{
	  nlist[i] = NULL;
	}
      oivis->list_vis = nlist;
    }
  elem = mat_viselem_new(oivis->nbchannel);
  if (elem == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oivis->list_vis[%d]", oivis->nbvis);
      return NULL;
    }
  oivis->list_vis[oivis->nbvis] = elem;
  oivis->nbvis++;
  return elem;
}

/**
   @ingroup oivis
   @brief Free memory of a single mat_oivis data structure.
   @param oivis Pointer to one mat_oivis data structure.
   @return A cpl_error_code
 
   This method frees the memory of a mat_oivis data structure. This includes
   all embedded data structures (strings and mat_viselem data structures).
*/
cpl_error_code mat_oivis_delete(mat_oivis *oivis)
{
  int i;

  mat_assert_value((oivis!=NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no mat_oivis (oivis) argument given");
  if (oivis->dateobs != NULL)
    {
      cpl_free(oivis->dateobs);
      oivis->dateobs = NULL;
    }
  if (oivis->arrayname != NULL)
    {
      cpl_free(oivis->arrayname);
      oivis->arrayname = NULL;
    }
  if (oivis->insname != NULL)
    {
      cpl_free(oivis->insname);
      oivis->insname = NULL;
    }
  if (oivis->amptyp != NULL)
    {
      cpl_free(oivis->amptyp);
      oivis->amptyp = NULL;
    }
  if (oivis->phityp != NULL)
    {
      cpl_free(oivis->phityp);
      oivis->phityp = NULL;
    }
  if (oivis->list_vis != NULL)
    {
      for (i = 0; i < oivis->nbvis; i++)
	{
	  if (oivis->list_vis[i] != NULL)
	    {
	      mat_viselem_delete(oivis->list_vis[i]);
	      oivis->list_vis[i] = NULL;
	    }
	}
      cpl_free(oivis->list_vis);
      oivis->list_vis = NULL;
      oivis->nbvis = 0;
    }
  cpl_free(oivis);
  return CPL_ERROR_NONE;
}

/**
   @ingroup oivis
   @brief Load a mat_oivis data structure from a CPL table and a propertylist.
   @param plist  This CPL propertylist contains the header keywords.
   @param table  This CPL table contains the OI_VIS vinary table.
   @return A newly allocated an filled mat_oivis or NULL on failure.

   This function loads an entire OI_VIS binary table into an internal data structure.
   The data structure contains no members for the columns from the OI-FITS version 2 standard.
*/
mat_oivis *mat_oivis_from_table(cpl_propertylist *plist, cpl_table *table)
{
  mat_oivis  *oivis;
  int         nbvis;
  int         nbchannel;
  int         j, k;

  mat_assert_value((plist != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_propertylist (plist) argument given");
  mat_assert_value((table != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_table (table) argument given");

  nbvis = cpl_table_get_nrow (table);
  /* old code
  nbchannel = cpl_array_get_size(cpl_table_get_array (table, "VISAMP",0));
  if (nbchannel == 0) nbchannel = 1; // no array!
  */
  /* new code */
  if (cpl_table_get_column_depth(table, "VISAMP") == 0)
    {
      nbchannel = 1;
    }
  else
    {
      nbchannel = cpl_table_get_column_dimension(table, "VISAMP", 0);
    }
  cpl_msg_info(cpl_func, "VIS table has %d wavelength data", nbchannel);
  oivis = mat_oivis_new(nbvis, nbchannel);
  if (oivis == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oivis");
      return NULL;
    }

  oivis->extver    = mat_propertylist_get_int_default(plist, "EXTVER", 1);
  oivis->arrayname = mat_propertylist_get_string_default(NULL, 0, plist, "ARRNAME", "empty");
  oivis->dateobs   = mat_propertylist_get_string_default(NULL, 0, plist, "DATE-OBS", "empty");
  oivis->insname   = mat_propertylist_get_string_default(NULL, 0, plist, "INSNAME", "empty");
  if (cpl_propertylist_has(plist, "AMPTYP"))
    {
      oivis->amptyp = mat_propertylist_get_string_default(NULL, 0, plist, "AMPTYP", "empty");
    }
  if (cpl_propertylist_has(plist, "PHITYP"))
    {
      oivis->phityp = mat_propertylist_get_string_default(NULL, 0, plist, "PHITYP", "empty");
    }
      

  for(j=0;j<oivis->nbvis;j++)
    {
      mat_viselem  *el = oivis->list_vis[j];
      el->targetid = cpl_table_get_int (table,"TARGET_ID", j, NULL);
      el->time = cpl_table_get_double (table, "TIME", j, NULL);
      el->dateobsmjd = cpl_table_get_double (table, "MJD", j, NULL);
      el->exptime = cpl_table_get_double (table, "INT_TIME", j, NULL);
      el->ucoord = cpl_table_get_double (table, "UCOORD", j, NULL);
      el->vcoord = cpl_table_get_double (table, "VCOORD", j, NULL);
      el->stationindex[0]= cpl_array_get_int(cpl_table_get_array(table, "STA_INDEX", j), 0, NULL);
      el->stationindex[1] = cpl_array_get_int(cpl_table_get_array(table, "STA_INDEX", j), 1, NULL);
      if (nbchannel == 1)
	{
	  el->visamp[0] = cpl_table_get_double(table,"VISAMP", j, NULL);
	  el->visamperr[0] = cpl_table_get_double(table, "VISAMPERR", j, NULL);
	  
	  /* if ( cpl_table_has_column(table,"CFXAMP") ) { */
	  /*   el->cfxamp[0] = cpl_table_get_double(table,"CFXAMP", j, NULL); */
	  /* } */
	  /* if ( cpl_table_has_column(table,"CFXAMPERR") ) { */
	  /*   el->cfxamperr[0] = cpl_table_get_double(table, "CFXAMPERR", j, NULL); */
	  /* } */
	  
	  el->visphi[0] = cpl_table_get_double(table,"VISPHI", j, NULL);
	  el->visphierr[0] = cpl_table_get_double(table,"VISPHIERR", j, NULL);
	  
	  el->bandflag[0] = (cpl_table_get_int(table,"FLAG", j, NULL) == 1);
	}
      else
	{
	  for (k = 0; k<oivis->nbchannel; k++)
	    {
	      el->visamp[k] = cpl_array_get_double(cpl_table_get_array(table,"VISAMP", j), k, NULL);
	      el->visamperr[k] = cpl_array_get_double(cpl_table_get_array(table, "VISAMPERR", j), k, NULL);
	      
	      /* if ( cpl_table_has_column(table,"CFXAMP") ) { */
	      /* 	el->cfxamp[k] = cpl_array_get_double(cpl_table_get_array(table,"CFXAMP", j), k, NULL); */
	      /* } */
	      /* if ( cpl_table_has_column(table,"CFXAMPERR") ) { */
	      /* 	el->cfxamperr[k] = cpl_array_get_double(cpl_table_get_array(table, "CFXAMPERR", j), k, NULL); */
	      /* } */
	      
	      el->visphi[k] = cpl_array_get_double(cpl_table_get_array(table,"VISPHI", j), k, NULL);
	      el->visphierr[k] = cpl_array_get_double(cpl_table_get_array(table,"VISPHIERR", j), k, NULL);
	      
	      el->bandflag[k] = (cpl_array_get_int(cpl_table_get_array(table,"FLAG", j), k, NULL) == 1);
	    }
	}
    }
  return oivis;
}

/**
   @ingroup oivis
   @brief Convert a mat_oivis data structure into a CPL table and a propertylist.
   @param oivis   The data structure containing a whole OI_VIS binary table.
   @param plist   This propertylist will contain the header keywords of the OI_VIS binary table.
   @return A CPL table containing the OI_VIS binary table or NULL on failure.

   A new CPL table will be created which will contain all complex visibilities. The header keywords of
   the OI_VIS binary table will be stored in the provided propertylist (parameter plist).
   The CPL table does not contain the columns from the OI-FITS version 2 standard.
*/
cpl_table *mat_oivis_to_table(mat_oivis *oivis, cpl_propertylist *plist)
{
  cpl_table    *table;
  cpl_array    *arr;
  int           i, j;
  int          *flags;

  if (plist != NULL)
    {
      cpl_propertylist_append_string(plist, "EXTNAME", "OI_VIS");
      cpl_propertylist_append_int(plist, "EXTVER", oivis->extver);
      if ( cpl_propertylist_has(plist,"OI_REVN") )
	{
	  cpl_propertylist_erase(plist,"OI_REVN");
	}
      cpl_propertylist_append_int(plist, "OI_REVN", 2);
      cpl_propertylist_append_string(plist, "DATE-OBS", oivis->dateobs);
      cpl_propertylist_append_string(plist, "ARRNAME", oivis->arrayname);
      cpl_propertylist_append_string(plist, "INSNAME", oivis->insname);
      if (oivis->amptyp != NULL)
	{
	  cpl_propertylist_append_string(plist, "AMPTYP", oivis->amptyp);
	}
      if (oivis->phityp != NULL)
	{
	  cpl_propertylist_append_string(plist, "PHITYP", oivis->phityp);
	}
    }

  flags = (int *)cpl_calloc(oivis->nbchannel, sizeof(int));
  if (flags == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for flag vector");
      return NULL;
    }

  table = cpl_table_new(oivis->nbvis);
  if (table == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_table");
      cpl_free(flags);
      return NULL;
    }
  /*Add columns*/
  cpl_table_new_column(table, "TARGET_ID", CPL_TYPE_INT);
  cpl_table_new_column(table, "TIME", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "TIME","s");
  cpl_table_new_column(table, "MJD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "MJD","day");
  cpl_table_new_column(table, "INT_TIME", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "INT_TIME","s");
  if (oivis->nbchannel == 1)
    {
      cpl_table_new_column(table,"VISAMP", CPL_TYPE_DOUBLE);
      cpl_table_new_column(table,"VISAMPERR", CPL_TYPE_DOUBLE);
      /* cpl_table_new_column(table,"CFXAMP", CPL_TYPE_DOUBLE); */
      /* cpl_table_new_column(table,"CFXAMPERR", CPL_TYPE_DOUBLE); */
      cpl_table_new_column(table,"VISPHI", CPL_TYPE_DOUBLE);
      cpl_table_new_column(table,"VISPHIERR", CPL_TYPE_DOUBLE);
    }
  else
    {
      cpl_table_new_column_array(table,"VISAMP", CPL_TYPE_DOUBLE, oivis->nbchannel);
      cpl_table_new_column_array(table,"VISAMPERR", CPL_TYPE_DOUBLE, oivis->nbchannel);
      /* cpl_table_new_column_array(table,"CFXAMP", CPL_TYPE_DOUBLE, oivis->nbchannel); */
      /* cpl_table_new_column_array(table,"CFXAMPERR", CPL_TYPE_DOUBLE, oivis->nbchannel); */
      cpl_table_new_column_array(table,"VISPHI", CPL_TYPE_DOUBLE, oivis->nbchannel);
      cpl_table_new_column_array(table,"VISPHIERR", CPL_TYPE_DOUBLE, oivis->nbchannel);
    }
  cpl_table_set_column_unit(table, "VISPHI","deg");
  cpl_table_set_column_unit(table, "VISPHIERR","deg");
  cpl_table_new_column(table, "UCOORD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "UCOORD","m");
  cpl_table_new_column(table, "VCOORD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "VCOORD","m");
  cpl_table_new_column_array(table,"STA_INDEX", CPL_TYPE_INT,2);
  if (oivis->nbchannel == 1)
    {
      cpl_table_new_column(table, "FLAG", CPL_TYPE_INT);
    }
  else
    {
      cpl_table_new_column_array(table, "FLAG", CPL_TYPE_INT, oivis->nbchannel);
    }
  // Fill the table
  for(i = 0; i < oivis->nbvis; i++)
    {
      mat_viselem  *el = oivis->list_vis[i];
      cpl_table_set_int(table, "TARGET_ID", i, el->targetid);
      cpl_table_set_double(table,"TIME",i, el->time);
      cpl_table_set_double(table,"MJD",i, el->dateobsmjd);
      cpl_table_set_double(table,"INT_TIME",i, el->exptime);
      if (oivis->nbchannel == 1)
	{
	  cpl_table_set_double(table,"VISAMP",i, el->visamp[0]);
	  cpl_table_set_double(table,"VISAMPERR",i, el->visamperr[0]);
	  /* cpl_table_set_double(table,"CFXAMP",i, el->cfxamp[0]); */
	  /* cpl_table_set_double(table,"CFXAMPERR",i, el->cfxamperr[0]); */
	  cpl_table_set_double(table,"VISPHI",i, el->visphi[0]);
	  cpl_table_set_double(table,"VISPHIERR",i, el->visphierr[0]);
	}
      else
	{
	  arr = cpl_array_wrap_double(el->visamp, oivis->nbchannel);
	  cpl_table_set_array(table,"VISAMP",i, arr);
	  cpl_array_unwrap(arr);
	  arr = cpl_array_wrap_double(el->visamperr, oivis->nbchannel);
	  cpl_table_set_array(table,"VISAMPERR",i,arr);
	  cpl_array_unwrap(arr);
	  
	  /* arr = cpl_array_wrap_double(el->cfxamp, oivis->nbchannel); */
	  /* cpl_table_set_array(table,"CFXAMP",i, arr); */
	  /* cpl_array_unwrap(arr); */
	  /* arr = cpl_array_wrap_double(el->cfxamperr, oivis->nbchannel); */
	  /* cpl_table_set_array(table,"CFXAMPERR",i,arr); */
	  /* cpl_array_unwrap(arr); */
	  
	  arr = cpl_array_wrap_double(el->visphi, oivis->nbchannel);
	  cpl_table_set_array(table,"VISPHI",i,arr);
	  cpl_array_unwrap(arr);
	  arr = cpl_array_wrap_double(el->visphierr, oivis->nbchannel);
	  cpl_table_set_array(table,"VISPHIERR",i,arr);
	  cpl_array_unwrap(arr);
	}
      cpl_table_set_double(table,"UCOORD",i,el->ucoord);
      cpl_table_set_double(table,"VCOORD",i,el->vcoord);
      arr = cpl_array_wrap_int(el->stationindex, 2);
      cpl_table_set_array(table, "STA_INDEX", i, arr);
      cpl_array_unwrap(arr);
      if (oivis->nbchannel == 1)
	{
	  cpl_table_set_int(table, "FLAG", i, (el->bandflag[0] != 0));
	}
      else
	{
	  for(j=0;j<oivis->nbchannel;j++)
	    {
	      if (el->bandflag[j])
		{
		  flags[j] = 1;
		}
	      else
		{
		  flags[j] = 0;
		}
	    }
	  arr = cpl_array_wrap_int(flags, oivis->nbchannel);
	  cpl_table_set_array(table,"FLAG",i, arr);
	  cpl_array_unwrap(arr);
	}
    }

  cpl_table_set_column_savetype(table,"TARGET_ID",CPL_TYPE_SHORT);
  cpl_table_set_column_savetype(table,"STA_INDEX",CPL_TYPE_SHORT);
  cpl_table_set_column_savetype(table,"FLAG",CPL_TYPE_BOOL);
  cpl_free(flags);
  return table;
}

/**
   @ingroup oivis
   @brief Load a OI_VIS binary table from a frame into a mat_oivis structure.
   @param frame  A frame containing a OI_VIS binary table.
   @return A newly allocated an filled mat_oivis or NULL on failure.

   This function scans all extensions inside a frame and loads the first OI_VIS binary table
   into a newly created mat_oivis data structure. If the frame does not contain any
   OI_VIS binary table, NULL is returned.
*/
mat_oivis *mat_oivis_load(cpl_frame *frame)
{
  int nbExtent;
  int  iExtent;

  mat_assert_value((frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_frame (frame) argument given");
  nbExtent = cpl_frame_get_nextensions(frame);
  for(iExtent = 1; iExtent <= nbExtent; iExtent++)
    {
      cpl_propertylist *plist = cpl_propertylist_load(cpl_frame_get_filename(frame), iExtent);
      char *keyExtname = (char *)cpl_propertylist_get_string(plist, "EXTNAME");
      if ((keyExtname != NULL) && (strcmp(keyExtname, "OI_VIS") == 0))
	{ /* Only the first OI_VIS binary table is loaded! */
	  cpl_table *table = cpl_table_load(cpl_frame_get_filename(frame), iExtent, 0);
	  mat_oivis *oivis = mat_oivis_from_table(plist, table);
	  cpl_table_delete(table);
	  cpl_propertylist_delete(plist);
	  return oivis;
      }
      cpl_propertylist_delete(plist);
    }
  return NULL;
}

/**
   @ingroup oivis
   @brief Append an OI_VIS binary table to an existing OI-FITS file.
   @param filename  Name of the FITS file.
   @param oivis     The mat_oivis structure containing a whole OI_VIS binary table.
   @return cpl_error_code

   This function converts a mat_oivis data structure into a CPL binary table and
   appends this at the end of an already existing FITS file.
*/
cpl_error_code  mat_oivis_append(char *filename,mat_oivis *oivis)
{
  cpl_table *oivisTab = NULL;
  cpl_propertylist *keyTab = NULL;
  cpl_errorstate prestate = cpl_errorstate_get();

  /*Check input parameters*/
  mat_assert_value((oivis!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oivis (oivis) argument given");
  mat_assert_value((filename!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no filename given");

  keyTab=cpl_propertylist_new();

  oivisTab = mat_oivis_to_table(oivis, keyTab);
  cpl_table_save(oivisTab,NULL,keyTab,filename,CPL_IO_EXTEND);

  cpl_propertylist_delete(keyTab);
  cpl_table_delete(oivisTab);
  if (!cpl_errorstate_is_equal(prestate))
    {
      cpl_errorstate_set(prestate);
      return CPL_ERROR_UNSPECIFIED;
    }
  return CPL_ERROR_NONE;
}

/**
   @ingroup oivis
   @brief Get the time span of the observation as MJD values (beginning and end).
   @param oivis    The mat_oivis structure containing a OI_VIS binary table.
   @param min_mjd  The beginning of the observation.
   @param max_mjd  The end of the observation.
   @return cpl_error_code

   The time span of a binary table is used to group OI_VIS, OI_VIS2 and OI_T3 binary
   tables into one tupel (see the mat_oifits data structure).
*/
cpl_error_code mat_oivis_get_mjd_span(mat_oivis *oivis, double *min_mjd, double *max_mjd)
{
  int i;

  /*Check input parameters*/
  mat_assert_value((oivis!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oivis (oivis) argument given");
  mat_assert_value((min_mjd!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no start MJD given");
  mat_assert_value((max_mjd!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no end MJD given");
  *min_mjd =  1.0e12;
  *max_mjd = -1.0e12;
  for (i = 0; i < oivis->nbvis; i++)
    {
      mat_viselem *el = oivis->list_vis[i];
      *min_mjd = fmin(*min_mjd, el->dateobsmjd);
      *max_mjd = fmax(*max_mjd, el->dateobsmjd);
    }
  return CPL_ERROR_NONE;
}
