/* $Id: mat_oivis.h,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oivis.h $
 */

#ifndef MAT_OIVIS_H
#define MAT_OIVIS_H

/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/

#include "cpl.h"

/*-----------------------------------------------------------------------------
  Defines
  ----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  Low Level Structures
  ----------------------------------------------------------------------------*/

/* Structure mat_viselem */
/**
   @ingroup oivis
   @brief Structure containing one measurement of an OI_VIS.

   This data structure represents one entry of an OI_VIS binary table.
*/
typedef struct {
  int          targetid;         /*!< Target number as index into the OI_TARGET binary table (always 1 for MATISSE OI-FITS files). */
  double       time;             /*!< UTC time of observation. */
  double       dateobsmjd;       /*!< UTC date of observation (MJD). */
  double       exptime;          /*!< Integration time per measurement in seconds. */
  double      *visamp;           /*!< Array of visibility amplitudes. */
  double      *visamperr;        /*!< Array of errors in visibility amplitude. */
  /* double      *cfxamp;           /\*!< Array of correlated flux amplitudes. *\/ */
  /* double      *cfxamperr;        /\*!< Array of errors in correlated flux amplitude. *\/ */
  double      *visphi;           /*!< Array of visibility phases in degree. */
  double      *visphierr;        /*!< Array of errors in visibility phase in degree. */
  double       ucoord;           /*!< U coordinate of the data in meter. */
  double       vcoord;           /*!< V coordinate of the data in meter. */
  int          stationindex[2];  /*!< Station numbers contributing to the data. */
  cpl_boolean *bandflag;         /*!< Flag if a data point is invalid (should be ignored). */
} mat_viselem;

/* Structure mat_oivis */
/**
   @ingroup oivis
   @brief Structure containing an OI_VIS binary table.

   This data structure represents one OI_VIS binary table of an OI-FITS file. A standard compliant OI-FITS file
   can contain several of these binary tables. This is implemented in the mat_oifits data structure.
*/
typedef struct {
  int            extver;     /*!< Extension version (optional). */
  char          *dateobs;    /*!< UTC start date of observations. */
  char          *arrayname;  /*!< Identifies the corresponding OI_ARRAY binary table. */
  char          *insname;    /*!< Identifies the corresponding OI_WAVELENGTH binary table. */
  char          *amptyp;    /*!< "absolute", "differential", correlated flux". */
  char          *phityp;    /*!< "absolute", "differential". */
  int            nbchannel;  /*!< Number of spectral channels (equivalent to the corresponding OI_WAVELENGTH binary table). */
  int            nbvis;      /*!< Number of differential visibilities in this binary table (equivalent with the number of rows). */
  mat_viselem  **list_vis;   /*!< List containing all differential visibilities stored in mat_viselem data structures. */
} mat_oivis;

/*-----------------------------------------------------------------------------
  High Level Structures
  ----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/
mat_viselem *mat_viselem_new(int nbchannel);

cpl_error_code mat_viselem_delete(mat_viselem *el);

mat_oivis *mat_oivis_new(int nbvis, int nbchannel);

mat_viselem *mat_oivis_elem_new(mat_oivis *oivis);

cpl_error_code mat_oivis_delete(mat_oivis *oivis);

mat_oivis *mat_oivis_from_table(cpl_propertylist *plist,
				cpl_table *table);

cpl_table *mat_oivis_to_table(mat_oivis *oivis,
			      cpl_propertylist *plist);

mat_oivis *mat_oivis_load(cpl_frame *frame);

cpl_error_code  mat_oivis_append(char *filename, mat_oivis *oivis);

cpl_error_code mat_oivis_get_mjd_span(mat_oivis *oivis, double *min_mjd, double *max_mjd);

#endif
