/* $Id: mat_oiwavelength.c,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oiwavelength.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "mat_error.h"
#include "mat_utils.h"
#include "mat_oiwavelength.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/
/**
  @ingroup oiwav
  @brief This method creates a new mat_oiwavelength structure with all pointers set to NULL
  @param nbchannel Number of spectral channels
  @returns Pointer to new allocated mat_oiwavelength structure on success or NULL on failure
 
  This method creates a new mat_oiwavelength structure and sets all value to the default. For
  pointers the value is NULL for primitive data types the value is set to 0.
 */
mat_oiwavelength *mat_oiwavelength_new(int nbchannel)
{
  mat_oiwavelength *oiwavelength;

  oiwavelength = (mat_oiwavelength *)cpl_calloc(1, sizeof(mat_oiwavelength));
  if (oiwavelength == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oiwavelength");
      return NULL;
    }
  oiwavelength->effwave = (double *)cpl_calloc(nbchannel, sizeof(double));
  if (oiwavelength->effwave == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for oiwavelength->effwave");
      mat_oiwavelength_delete(oiwavelength);
      return NULL;
    }
  oiwavelength->effband = (double *)cpl_calloc(nbchannel, sizeof(double));
  if (oiwavelength->effband == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for oiwavelength->effband");
      mat_oiwavelength_delete(oiwavelength);
      return NULL;
    }
  oiwavelength->nbchannel = nbchannel;
  return oiwavelength;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oiwav
  @brief delete structure mat_oiwavelength
  @param  oiwavelength          current structure
  @return error code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_oiwavelength_delete(mat_oiwavelength *oiwavelength)
{
  mat_assert_value((oiwavelength!=NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no mat_oiwavelength (oiwavelength) argument given");
  if (oiwavelength->insname != NULL)
    {
      cpl_free(oiwavelength->insname);
      oiwavelength->insname = NULL;
    }
  if (oiwavelength->effwave != NULL)
    {
      cpl_free(oiwavelength->effwave);
      oiwavelength->effwave = NULL;
    }
  if (oiwavelength->effband != NULL)
    {
      cpl_free(oiwavelength->effband);
      oiwavelength->effband = NULL;
    }
  cpl_free(oiwavelength);
  return CPL_ERROR_NONE;
}

/**
  @ingroup oiwav
  @brief Load a  mat_oiwavelength data structure from a cpl table and a propertylist
  @param plist          propertylist containing the header 
  @param table          cpl table containing the data
  @return mat_oiwavelength
*/
mat_oiwavelength *mat_oiwavelength_from_table(cpl_propertylist *plist, cpl_table *table)
{
  mat_oiwavelength *oiwavelength;
  int             nbchannel;
  int             i;

  nbchannel = cpl_table_get_nrow(table);
  oiwavelength = mat_oiwavelength_new(nbchannel);
  if (oiwavelength == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oiwavelength");
      return NULL;
    }
  oiwavelength->extver    = mat_propertylist_get_int_default(plist, "EXTVER", 1);
  oiwavelength->insname   = mat_propertylist_get_string_default(NULL, 0, plist, "INSNAME", "empty");
  for (i = 0; i < nbchannel; i++)
    {
      oiwavelength->effwave[i] = (double)cpl_table_get_float(table, "EFF_WAVE", i, NULL);		  
      oiwavelength->effband[i] = (double)cpl_table_get_float(table, "EFF_BAND", i, NULL);
    }
  return oiwavelength;
}

/**
  @ingroup oiwav
  @brief copy a  mat_oiwavelength data structure into a cpl table and a propertylist
  @param oiwavelength     data structure to copy
  @param plist          propertylist containing the header 
  @return cpl table containing the data
*/
cpl_table *mat_oiwavelength_to_table(mat_oiwavelength *oiwavelength, cpl_propertylist *plist)
{
  cpl_table    *table;
  int           i;

  if (plist != NULL)
    {
      cpl_propertylist_append_string(plist, "EXTNAME", "OI_WAVELENGTH");
      cpl_propertylist_append_int(plist, "EXTVER", oiwavelength->extver);
      if ( cpl_propertylist_has(plist,"OI_REVN") )
	{
	  cpl_propertylist_erase(plist,"OI_REVN");
	}
      cpl_propertylist_append_int(plist, "OI_REVN", 2);
      cpl_propertylist_append_string(plist, "INSNAME", oiwavelength->insname);
    }
  table = cpl_table_new(oiwavelength->nbchannel);
  if (table == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_table");
      return NULL;
    }
  // Add columns
  cpl_table_new_column(table, "EFF_WAVE", CPL_TYPE_FLOAT);
  cpl_table_set_column_unit(table, "EFF_WAVE","m");
  cpl_table_new_column(table, "EFF_BAND", CPL_TYPE_FLOAT);
  cpl_table_set_column_unit(table, "EFF_BAND","m");
  // Fill the table
  for(i = 0; i < oiwavelength->nbchannel; i++)
    {
      cpl_table_set_float(table, "EFF_WAVE", i, oiwavelength->effwave[i]);
      cpl_table_set_float(table, "EFF_BAND", i, oiwavelength->effband[i]);
    }
  return table;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oiwav
  @brief Load a frame in mat_oiwavelength structure
  @param frame           current frame
  @return mat_oiwavelength
 */
/*-----------------------------------------------------------------------------*/
mat_oiwavelength * mat_oiwavelength_load(cpl_frame *frame)
{
  mat_oiwavelength *oiwavelength = NULL;
  int nbExtent = 0;
  int i = 0;
  char *keyExtname = NULL;
  cpl_table *table = NULL;                        
  cpl_propertylist *plist = NULL;                 
 
  mat_assert_value((frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_frame (frame) argument given");
    
  nbExtent = cpl_frame_get_nextensions(frame);
  for(i = 0; i < nbExtent; i++) 
    {
      plist = cpl_propertylist_load(cpl_frame_get_filename(frame), i+1); 
      keyExtname = (char *)cpl_propertylist_get_string(plist, "EXTNAME");
      if (keyExtname != NULL) 
	{
	  table = cpl_table_load(cpl_frame_get_filename(frame), i+1, 0);
	  if (!strcmp(keyExtname, "OI_WAVELENGTH"))
	    {
	      oiwavelength = mat_oiwavelength_from_table(plist, table);
	    } 
	  cpl_table_delete(table);           
	}
      cpl_propertylist_delete(plist);
    }
  return oiwavelength;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oiwav
  @brief Append an OI_WAVELENGTH binary table to an existing OIFITS file
  @param filename          name of the FITS file
  @param oiwave            the mat_oiwavelength structure containing the 
                           informations for filling the binary table
  @return cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code  mat_oiwavelength_append(char *filename,
					mat_oiwavelength *oiwave)
{
  cpl_table     *oiwaveTab = NULL;
  cpl_propertylist *keyTab = NULL;
  cpl_errorstate  prestate = cpl_errorstate_get();

  // Check input parameters
  mat_assert_value((oiwave!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oiwavelength (oiwave) argument given");
  mat_assert_value((filename!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no filename given");

  keyTab    = cpl_propertylist_new();
  oiwaveTab = mat_oiwavelength_to_table(oiwave, keyTab);
  cpl_table_save(oiwaveTab,NULL,keyTab,filename,CPL_IO_EXTEND);

  cpl_propertylist_delete(keyTab);
  cpl_table_delete(oiwaveTab);
  if (!cpl_errorstate_is_equal(prestate))
    {
      cpl_errorstate_set(prestate);
      return CPL_ERROR_UNSPECIFIED;
    }
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oiwav
  @brief compare two mat_oiwavelength structures
  @param oiwave1          mat_oiwavelength
  @param oiwave2          mat_oiwavelength
  @return cpl_boolean         CPL_TRUE if oiwave1 and oiwave2 are identical
 */
/*----------------------------------------------------------------------------*/

cpl_boolean mat_oiwavelength_compare(mat_oiwavelength *oiwave1, 
				     mat_oiwavelength *oiwave2){

    int i = 0;
    int cpt = 0;
   
    mat_assert_value((oiwave1 != NULL), CPL_ERROR_NULL_INPUT, CPL_FALSE,
    		     "no mat_oiwavelength (oiwave1) argument given");
    mat_assert_value((oiwave2 != NULL), CPL_ERROR_NULL_INPUT, CPL_FALSE,
    		     "no mat_oiwavelength (oiwave2) argument given");
    if (!strcmp(oiwave1->insname, oiwave2->insname) &&
    	(oiwave1->nbchannel == oiwave2->nbchannel))
    {
      for (i = 0; i< oiwave1->nbchannel; i++)
      {
    	  if ( fabs(oiwave1->effwave[i]-oiwave2->effwave[i]) < 1.E-9 &&
    	       fabs(oiwave1->effband[i]-oiwave2->effband[i]) < 1.E-9 )
    	  {
    	      cpt++;
    	  }
      }
      if (cpt == oiwave1->nbchannel)
      {
    	  return CPL_TRUE;
      }
      else
      {
    	  return CPL_FALSE;
      }
    }
    else
    {
    	return CPL_FALSE;
    }
    
}


