/* $Id: mat_statistics.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_statistics.c $
 */

#include <stdlib.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/

#include <string.h>
#include <complex.h>
#include <math.h>

#include "mat_statistics.h"
#include "mat_frame.h"
#include "mat_image.h"
#include "mat_utils.h"

/*-----------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/

#define MAT_MIN_GAIN_INTENSITY  1000.0
#define MAT_GRANULARITY         1024
#define MAT_NRUNS                  3

#define MAT_NO_FLAGS              0
#define MAT_POLY0_OUTLIER_FLAG    (1 << 0)
#define MAT_POLY1_OUTLIER_FLAG    (1 << 1)
#define MAT_POLY2_OUTLIER_FLAG    (1 << 2)
#define MAT_POLY3_OUTLIER_FLAG    (1 << 3)
#define MAT_OUTLIER_MASK          (MAT_POLY0_OUTLIER_FLAG | MAT_POLY1_OUTLIER_FLAG | MAT_POLY2_OUTLIER_FLAG | MAT_POLY3_OUTLIER_FLAG)
#define MAT_OUTLIER_FLAG          (1 << 4)


/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/

/**
   @brief Compares two double values, used for qsort (inreasing order)
   @param e1  The pointer to the first value
   @param e2  The pointer to the second value
   @returns -1 if e1 < e2, 0 if e1 == e2 and 1 if e1 > e2

   This function is used to sort an array of doubles in ascending order.
*/
static int mat_double_compare(const void *e1, const void *e2)
{
  double *x = (double*)e1;
  double *y = (double*)e2;
  
  if (*x > *y) return 1;
  if (*x == *y) return 0;
  return -1;
}

mat_statistics_info *mat_statistics_new(int size, double ofactor)
{
  mat_statistics_info *info;
  
  info = cpl_calloc(1, sizeof(mat_statistics_info));
  if (info == NULL)
    {
      return NULL;
    }
  info->use_average  = MAT_USE_AVERAGE_DEFAULT;
  info->use_variance = MAT_USE_VARIANCE_DEFAULT;
  info->size = size;
  if (size != 0)
    {
      info->values = cpl_calloc(size, sizeof(double));
      if (info->values == NULL)
	{
	  cpl_free(info);
	  return NULL;
	}
    }
  info->ofactor = ofactor;
  info->range_min = MAT_SMALLEST_VALUE;
  info->range_max = MAT_LARGEST_VALUE;
  return info;
}

mat_statistics_info *mat_statistics_realloc(mat_statistics_info *info, int size, double ofactor)
{
  // allocate a new structure if necessary
  if (info == NULL)
    {
      info = mat_statistics_new(size, ofactor);
    }
  // the allocated memory can only grow
  if (info->size < size)
    {
      int osize = info->size;
      int i;
      info->size = size;
      info->values = cpl_realloc(info->values, size*sizeof(double));
      if (info->values == NULL)
	{
	  cpl_free(info);
	  return NULL;
	}
      for (i = osize; i < size; i++) info->values[i] = 0.0;
    }
  info->ofactor = ofactor;
  return info;
}


void mat_statistics_delete(mat_statistics_info *info)
{
  if (info == NULL) return;
  info->size = 0;
  if (info->values != NULL)
    {
      cpl_free(info->values);
      info->values = NULL;
    }
  ///*
  info->hist_size = 0;
  if (info->hist_x != NULL)
    {
      cpl_free(info->hist_x);
      info->hist_x = NULL;
    }
  if (info->hist_y != NULL)
    {
      cpl_free(info->hist_y);
      info->hist_y = NULL;
    }
  if (info->hist_p != NULL)
    {
      cpl_free(info->hist_p);
      info->hist_p = NULL;
    }
  if (info->hist_q != NULL)
    {
      cpl_free(info->hist_q);
      info->hist_q = NULL;
    }
  if (info->hist_s != NULL)
    {
      cpl_free(info->hist_s);
      info->hist_s = NULL;
    }
  //*/
  cpl_free(info);
}

void mat_statistics_reset(mat_statistics_info *info)
{
  if (info == NULL) return;
  info->discarded_count = 0;
  info->nan_count = 0;
  info->inf_count = 0;
  info->whole_count = 0;
}

void mat_statistics_set_use(mat_statistics_info *info, int use_avg, int use_var)
{
  if (info == NULL) return;
  info->use_average  = use_avg;
  info->use_variance = use_var;
}

void mat_statistics_set_range(mat_statistics_info *info, double lower, double upper)
{
  if (info == NULL) return;
  info->range_min = lower;
  info->range_max = upper;
}
  
void mat_statistics_add_value(mat_statistics_info *info, double value)
{
  if (info == NULL) return;
  ///*
  if ((value < info->range_min) || (value > info->range_max))
    {
      info->discarded_count++;
      return;
    }
  if (isnan(value))
    {
      info->nan_count++;
      return;
    }
  if (isinf(value))
    {
      info->inf_count++;
      return;
    }
  //*/
  if (info->whole_count == info->size)
    {
      int osize = info->size;
      int i;
      info->values = (double*)cpl_realloc(info->values, (info->size + MAT_GRANULARITY)*sizeof(double));
      info->size += MAT_GRANULARITY;
      for (i = osize; i < info->size; i++) info->values[i] = 0.0;
    }
  if (info->values == NULL)
    {
      info->size = 0;
      return;
    }
  info->values[info->whole_count++] = value;
}

void mat_statistics_set_region(mat_statistics_info *info, cpl_image *image, cpl_mask *bpm, int x, int y, int nx, int ny)
{
  int i, j;

  if (info == NULL) return;
  mat_statistics_reset(info);
  for (j = y; j < y + ny; j++)
    {
      for (i = x; i < x + nx; i++)
	{
	  int      rejected;
	  double   v;
	  if (bpm != NULL)
	    {
	      if (cpl_mask_get(bpm, i, j) == CPL_BINARY_1) continue;
	    }
	  v = cpl_image_get(image, i, j, &rejected);
	  if (!rejected) mat_statistics_add_value(info, v);
	}
    }
}

void mat_statistics_set_sequence(mat_statistics_info *info, cpl_imagelist *list, cpl_mask *bpm, int x, int y)
{
  int    z, nz;

  if (info == NULL) return;
  mat_statistics_reset(info);
  if (bpm != NULL)
    {
      if (cpl_mask_get(bpm, x, y) == CPL_BINARY_1) return;
    }

  nz = cpl_imagelist_get_size(list);
  for (z = 0; z < nz; z++)
    {
      int         rejected;
      cpl_image  *image = cpl_imagelist_get(list, z);
      double      v = cpl_image_get(image, x, y, &rejected);
      if (!rejected) mat_statistics_add_value(info, v);
    }
}

static void mat_statistics_select_avg_var(mat_statistics_info *info)
{
  switch (info->use_average)
    {
    case MAT_USE_WHOLE_MEDIAN:
      info->average = info->whole_median;
      break;
    case MAT_USE_WHOLE_MEAN:
      info->average = info->whole_mean;
      break;
    case MAT_USE_WHOLE_MODE:
      info->average = info->whole_mode;
      break;
    case MAT_USE_REDUCED_MEDIAN:
      info->average = info->reduced_median;
      break;
    case MAT_USE_REDUCED_MEAN:
      info->average = info->reduced_mean;
      break;
    default:
      info->average = info->whole_mean;
    }
  switch (info->use_variance)
    {
    case MAT_USE_WHOLE_MEDVAR:
      info->variance = info->whole_medvar;
      break;
    case MAT_USE_WHOLE_VARIANCE:
      info->variance = info->whole_variance;
      break;
    case MAT_USE_REDUCED_MEDVAR:
      info->variance = info->reduced_medvar;
      break;
    case MAT_USE_REDUCED_VARIANCE:
      info->variance = info->reduced_variance;
      break;
    default:
      info->variance = info->whole_variance;
    }
}

static void mat_statistics_calc_avg_var(mat_statistics_info *info, int first, int last, double *median, double *medvar, double *mean, double *variance)
{
  double sum, sump;
  double err, errp;
  int    i, count;

  count = last - first + 1;
  // Calculate the median using only values inside the limits [limit_min .. limit_max] (no outliers)
  if (count%2 == 0)
    { // even count -> average of the two centers
      *median = 0.5*(info->values[(first + last)/2] + info->values[(first + last)/2+1]);
    }
  else
    {
      *median = info->values[(first + last)/2];
    }
  // Calculate the variance based on the median (no autocorrect via err) ignoring outliers
  // intermediate partial sum is used to avoid numerical errors due to summing up too many small values
  if (first == last)
    {
      *medvar = 0.0;
    }
  else
    {
      sum  = 0.0;
      sump = 0.0;
      for (i = first; i <= last; i++)
	{
	  double d = info->values[i] - *median;
	  if (i%1000 == 0)
	    {
	      sum += sump;
	      sump = 0.0;
	    }
	  sump += d*d;
	}
      sum += sump;
      *medvar = sum/(double)(last - first);
    }
  // Calculate the mean ignoring outliers
  // intermediate partial sum is used to avoid numerical errors due to summing up too many small values
  sum  = 0.0;
  sump = 0.0;
  for (i = first; i <= last; i++)
    {
      if (i%1000 == 0)
	{
	  sum += sump;
	  sump = 0.0;
	}
      sump += info->values[i];
    }
  sum += sump;
  *mean = sum/(double)(last - first + 1);
  // Calculate the variance ignoring outliers
  // intermediate partial sum and error is used to avoid numerical errors due to summing up too many small values
  if (first == last)
    {
      *variance = 0.0;
    }
  else
    {
      sum  = 0.0;
      sump = 0.0;
      err  = 0.0;
      errp = 0.0;
      for (i = first; i <= last; i++)
	{
	  double d = info->values[i] - *mean;
	  if (i%1000)
	    {
	      sum += sump;
	      sump = 0.0;
	      err += errp;
	      errp = 0.0;
	    }
	  sump += d*d;
	  errp += d;
	}
      sum += sump;
      err += errp;
      *variance = (sum - (err*err)/(double)(last - first + 1))/(double)(last - first);
    }
}

static void mat_statistics_calc_density(mat_statistics_info *info)
{
  int     i, j;
  int     fromIdx, toIdx;
  double  max_density = 0.0;
  double  peak_sum    = 0.0;
  double  peak_count  = 0.0;

  // the range is [fromIdx .. toIdx] (including)
  if (info->whole_count < 5*MAT_DENSITY_BIN_COUNT)
    { // not enough values to calculate a density distribution
      info->density_peak = info->whole_median;
      return;
    }
  toIdx = -1;
  // we have to deal with three cases:
  // 1. Only one value is present                   => peak is equal with the value
  // 2. Some successive bins contain the same value => merge these bins
  // 3. every bin contains a unique set of values   => standard case, leave all bins as they are
  info->density_count = 0;
  for (i = 0; i < MAT_DENSITY_BIN_COUNT; i++)
    {
      fromIdx = toIdx+1; // toIdx was included in the previous bin
      toIdx = ((i+1)*info->whole_count)/MAT_DENSITY_BIN_COUNT;
      toIdx = CPL_MIN(info->whole_count-1, toIdx);
      info->density_x[info->density_count] = 0.5*(info->values[fromIdx] + info->values[toIdx]);
      info->density_d[info->density_count] = (double)(toIdx - fromIdx + 1);
      if (info->pflag)
	{
	  cpl_msg_debug(cpl_func, "%s# density %f %f", info->prefix, info->density_x[info->density_count], info->density_d[info->density_count]);
	}
      info->density_count++;
    }
  // Merge two bins if they both represents the same value and therefore both have the same _x[]
  for (i = 0; i < info->density_count - 1; i++)
    {
      while ((i < info->density_count-1) && (info->density_x[i] == info->density_x[i+1]))
	{ // two successive bins have the same center -> merge them
	  info->density_d[i] += info->density_d[i+1];
	  for (j = i+1; j < info->density_count-1; j++)
	    {
	      info->density_x[j] += info->density_x[j+1];
	      info->density_d[j] += info->density_d[j+1];
	    }
	  info->density_count--;
	}
    }
  // Check for case (1)
  if (info->density_count == 1)
    {
      info->density_peak = info->density_x[0];
      if (info->pflag)
	{
	  cpl_msg_debug(cpl_func, "%s# density # single value peak %f", info->prefix, info->density_peak);
	}
      return;
    }
  // Calculate the width of each bin (starts in the middle between the previous bin and this bin and goes to ...)
  info->density_w[0] = info->density_x[1] - info->density_x[0];
  info->density_d[0] = info->density_d[0]/(double)info->whole_count/info->density_w[0];
  max_density = CPL_MAX(max_density, info->density_d[0]);
  for (i = 1; i < info->density_count-1; i++)
    {
      info->density_w[i] = 0.5*(info->density_x[i+1] - info->density_x[i-1]);
      info->density_d[i] = info->density_d[i]/(double)info->whole_count/info->density_w[i];
      max_density = CPL_MAX(max_density, info->density_d[i]);
    }
  info->density_w[info->density_count-1] = info->density_x[info->density_count-1] - info->density_x[info->density_count-2];
  info->density_d[info->density_count-1] = info->density_d[info->density_count-1]/(double)info->whole_count/info->density_w[info->density_count-1];
  max_density = CPL_MAX(max_density, info->density_d[info->density_count-1]);
  if (info->pflag)
    {
      for (i = 0; i < info->density_count; i++)
	{
	  cpl_msg_debug(cpl_func, "%s# density %f %f %f", info->prefix, info->density_x[i], info->density_w[i], info->density_d[i]);
	}
    }
  // Estimage the peak by using only bins whith a density above 80 percent of the largest density.
  for (i = 0; i < MAT_DENSITY_BIN_COUNT; i++)
    {
      if (info->density_d[i] < MAT_DENSITY_PEAK_LIMIT*max_density) continue;
      peak_sum   += info->density_x[i]*info->density_d[i];
      peak_count += info->density_d[i];
    }
  info->density_peak = peak_sum/peak_count;
  if (info->pflag)
    {
      cpl_msg_debug(cpl_func, "%s# density # peak limit %f peak %f", info->prefix, MAT_DENSITY_PEAK_LIMIT*max_density, info->density_peak);
    }
}

static void mat_statistics_calc_histogram(mat_statistics_info *info)
{
  double  valueMin  = info->values[( 5*info->whole_count)/100]; // 5 %
  double  valueMax  = info->values[(95*info->whole_count)/100]; // 95 %
  int     count, binIdx, valIdx;
  double  binLower, binUpper;
  double  max_count  = 0.0;
  double  peak_sum   = 0.0;
  double  peak_count = 0.0;

  if (info->whole_count < 5*MAT_DENSITY_BIN_COUNT)
    { // not enough values to calculate a density distribution
      info->hist_peak = info->whole_median;
      return;
    }
  // https://en.wikipedia.org/wiki/Histogram "Number of bins and width"/"Freedman–Diaconis' choice"
  // factor 2 replaced with 20, (10 times bigger bins than the rule)
  count = (int)(pow(info->whole_count, 1.0/3.0))/20; 
  count = CPL_MAX(20, count); // at least 5 bins!
  info->hist_bin = (valueMax - valueMin)/(double)(count - 1);
  // allocate and initialize memory for the histogram
  if (count > info->hist_size)
    { // not enough space
      info->hist_size = count;
      info->hist_x = cpl_realloc(info->hist_x, info->hist_size*sizeof(double));
      info->hist_y = cpl_realloc(info->hist_y, info->hist_size*sizeof(double));
      info->hist_p = cpl_realloc(info->hist_p, info->hist_size*sizeof(double));
      info->hist_q = cpl_realloc(info->hist_q, info->hist_size*sizeof(double));
      info->hist_s = cpl_realloc(info->hist_s, info->hist_size*sizeof(double));
    }
  info->hist_count = count;
  for (binIdx = 0; binIdx < count; binIdx++)
    {
      info->hist_x[binIdx] = 0.0;
      info->hist_y[binIdx] = 0.0;
    }
  // go to the first bin of the histogram and skip all values before this bin
  binUpper = valueMin - 0.5*info->hist_bin; // represents the upper limit of all ignored bins
  valIdx = 0;
  while (info->values[valIdx] < binUpper) valIdx++;
  // calculate the value histogram between [valueMin - 0.5*info->hist_bin ... valueMax + 0.5*info->hist_bin]
  if (info->pflag)
    {
      cpl_msg_debug(cpl_func, "%s# hist # info values = [%f .. %f] bins = [%f .. %f] (%d x %f)",
		    info->prefix, info->values[0], info->values[info->whole_count-1],
		    valueMin - 0.5*info->hist_bin, valueMax + 0.5*info->hist_bin,
		    count, info->hist_bin);
      cpl_msg_debug(cpl_func, "%s# hist # columns  1      2     3", info->prefix);
      cpl_msg_debug(cpl_func, "%s# hist # columns median value count", info->prefix);
    }
  for (binIdx = 0; binIdx < count; binIdx++)
    {
      // recalculate only the upper bin limit and reuse this value as lower limit for the next bin
      // -> avoids rounding errors
      binLower = binUpper;
      binUpper = valueMin + info->hist_bin*(double)binIdx + 0.5*info->hist_bin;
      info->hist_x[binIdx] = 0.5*(binLower + binUpper);
      while ((valIdx < info->whole_count) && (info->values[valIdx] < binUpper))
	{
	  info->hist_y[binIdx] += 1.0;
	  valIdx++;
	}
      max_count = CPL_MAX(max_count, info->hist_y[binIdx]);
      if (info->pflag)
	{
	  cpl_msg_debug(cpl_func, "%s# hist %f %f", info->prefix, info->hist_x[binIdx], info->hist_y[binIdx]);
	}
    }
  for (binIdx = 0; binIdx < count; binIdx++)
    {
      if (info->hist_y[binIdx] < MAT_HISTOGRAM_PEAK_LIMIT*max_count) continue;
      peak_sum   += info->hist_x[binIdx]*info->hist_y[binIdx];
      peak_count += info->hist_y[binIdx];
    }
  info->hist_peak = peak_sum/peak_count;
  if (info->pflag)
    {
      cpl_msg_debug(cpl_func, "%s# hist # peak limit %f peak %f", info->prefix, MAT_HISTOGRAM_PEAK_LIMIT*max_count, info->hist_peak);
    }
}

/*
  static void mat_statistics_find_peak(mat_statistics_info *info)
  {
  double  fp = 0.0, fq = 0.0, fs = 0.0; // y = q + s*(x-p)**2
  int     idx;
  int     found = 0;
  double  avg_y;

  // Try to calculate the mode of the distribution (assumes an unimodal distribution!)
  // Distinguish between real peaks and possible outliers is based on:
  //  - Both form factors (is and os) are negative.
  //  - For a real peak, both values should be similar (a small difference is allowed)
  //  - If the outer polynome is much wider than the inner polynome, the peak is a possible outlier
  //   .Y.................................................
  //   ...................................................
  //   .^.................................................
  //   .|.............C...................................
  //   .|.................................................
  //   .|.................................................
  //   .|.................................................
  //   .|.................................................
  //   .|........................O........................
  //   .|................I................................
  //   .|.................................................
  //   .|........I........................................
  //   .|.................................................
  //   .|.................................................
  //   .|..O..............................................
  //   .|.................................................
  //   -+----------------------------------------------> X
  //   ...................................................
  // The polynome I -> C -> I has a large negative form factor is
  // The polynome O -> C -> O has a small negative form factor os
  //    -is >> -os
  // In this case, the peak at C is a possible outlier
  //   .Y.................................................
  //   ...................................................
  //   .^.................................................
  //   .|.............C...................................
  //   .|................I................................
  //   .|.................................................
  //   .|.................................................
  //   .|........I........................................
  //   .|........................O........................
  //   .|................................................
  //   .|.................................................
  //   .|.................................................
  //   .|.................................................
  //   .|.................................................
  //   .|..O..............................................
  //   .|.................................................
  //   -+----------------------------------------------> X
  // The polynome I -> C -> I has a small negative form factor is
  // The polynome O -> C -> O has a small negative form factor os
  //    is ~ os
  // In this case, the peak at C is a possible maximum
  // we need at least a certain number in order to reliably detect a peak in the value distribution
  if (info->whole_count < 1000)
  {
  info->whole_mode = info->whole_median;
  return;
  }
  // Calculate a 2-order polynome using three adjacent values
  avg_y = 0.0;
  for (idx = 1; idx <= (info->hist_count - 2); idx++)
  {
  avg_y += info->hist_y[idx];
  mat_calc_poly2_peak(&(info->hist_x[idx-1]), &(info->hist_y[idx-1]),
  &(info->hist_p[idx]),
  &(info->hist_q[idx]),
  &(info->hist_s[idx]));
  }
  avg_y /= (double)(info->hist_count - 2);
  if (info->pflag)
  {
  cpl_msg_debug(cpl_func, "%s# poly # columns 1 2 3  4  5  6  7  8", info->prefix);
  cpl_msg_debug(cpl_func, "%s# poly # columns x y ip iq is op oq os", info->prefix);
  }
  for (idx = 2; idx <= (info->hist_count - 3); idx++)
  { // idx is the index of a possible peak (two values left and right are needed)
  double op, oq, os, ox[3], oy[3];
  ox[0] = info->hist_x[idx-2]; oy[0] = info->hist_y[idx-2];
  ox[1] = info->hist_x[idx];   oy[1] = info->hist_y[idx];
  ox[2] = info->hist_x[idx+2]; oy[2] = info->hist_y[idx+2];
  if (mat_calc_poly2_peak(ox, oy, &op, &oq, &os) != CPL_TRUE) continue; // cannot fit a 2-order polynome -> ignore this bin
  if (info->pflag)
  {
  cpl_msg_debug(cpl_func, "%s# poly %f %d %f %f %f %f %f %f", info->prefix, info->hist_x[idx], (int)(info->hist_y[idx]), info->hist_p[idx], info->hist_q[idx], info->hist_s[idx], op, oq, os);
  }
  if (info->hist_y[idx] < avg_y)
  { // not even the average y-value
  if (info->pflag) cpl_msg_debug(cpl_func, "%s# poly # value below %f (avg_y)", info->prefix, avg_y);
  continue;
  }
  if ((info->hist_y[idx-1] >= info->hist_y[idx]) || (info->hist_y[idx+1] >= info->hist_y[idx]))
  {
  // center value is not larger than left and right value
  if (info->pflag) cpl_msg_debug(cpl_func, "%s# poly # value below inner left or right value", info->prefix);
  continue;
  }
  if ((info->hist_y[idx-2] >= info->hist_y[idx]) || (info->hist_y[idx+2] >= info->hist_y[idx]))
  {
  // center value is not larger than left and right value
  if (info->pflag) cpl_msg_debug(cpl_func, "%s# poly # value below outer left or right value", info->prefix);
  continue;
  }
  if (info->pflag)
  {
  cpl_msg_debug(cpl_func, "%s# poly # candidate %f %f %f %f %f %f %f %f",
  info->prefix,
  info->hist_x[idx], info->hist_y[idx],
  info->hist_p[idx], info->hist_q[idx], info->hist_s[idx],
  op, oq, os);
  }
  if (!found)
  {
  found = 1;
  fp = info->hist_p[idx];
  fs = info->hist_s[idx];
  fq = info->hist_q[idx];
  if (info->pflag) cpl_msg_debug(cpl_func, "%s# poly # peak yi = %f + %f*(x - %f)**2 yo = %f + %f*(x - %f)**2", info->prefix, fq, fs, fp, oq, os, op);
  }
  else if (os > info->hist_s[idx])
  {
  fp = info->hist_p[idx];
  fs = info->hist_s[idx];
  fq = info->hist_q[idx];
  if (info->pflag) cpl_msg_debug(cpl_func, "%s# poly # peak yi = %f + %f*(x - %f)**2 yo = %f + %f*(x - %f)**2", info->prefix, fq, fs, fp, oq, os, op);
  }
  else
  {
  if (info->pflag)
  {
  cpl_msg_debug(cpl_func, "%s# poly # outlier %f %f %f %f %f -> yi = %f + %f*(x - %f)**2 yo = %f + %f*(x - %f)**2",
  info->prefix, info->hist_y[idx-2], info->hist_y[idx-1], info->hist_y[idx], info->hist_y[idx+1], info->hist_y[idx+2],
  info->hist_q[idx], info->hist_s[idx], info->hist_p[idx], oq, os, op);
  }
  }
  }
  if (found)
  {
  info->whole_mode = fp;
  if (info->pflag)
  {
  cpl_msg_debug(cpl_func, "%s# final f(x) = %f + %f*(x - %f)**2", info->prefix, fq, fs, fp);
  }
  }
  else
  {
  fq = 0.0;
  for (idx = 0; idx < info->hist_count; idx++)
  {
  if (info->hist_y[idx] > fq)
  {
  fp = info->hist_x[idx];
  fq = info->hist_y[idx];
  }
  }
  if (info->pflag)
  {
  cpl_msg_debug(cpl_func, "%s# peak %f at %f", info->prefix, fq, fp);
  }
  info->whole_mode = fp;
  }
  }
*/

/* Wikipedia, 'Kahan summation algorithm', section 'Further enhancements' */
static double mat_calc_neumaier_sum(double *d, int n)
{
  double sum = 0.0;
  double c   = 0.0;
  int    i;

  for (i = 0; i < n; i++)
    {
      double t = sum + d[i];
      if (fabs(sum) >= fabs(d[i]))
	{
	  c += (sum - t) + d[i];
	}
      else
	{
	  c += (d[i] - t) + sum;
	}
      sum = t;
    }
  return (sum + c);
}

void mat_statistics_calc(mat_statistics_info *info)
{
  int    first, last;

  if ((info->nan_count != 0) || (info->inf_count != 0))
    {
      cpl_msg_info(cpl_func, "%d nan and %d inf values ignored", info->nan_count, info->inf_count);
    }
  info->whole_total = mat_calc_neumaier_sum(info->values, info->whole_count);
  // ##### 0, 1 or 2 values #####
  // # 0 values
  if (info->whole_count == 0)
    {
      info->q25            = 0.0;
      info->q75            = 0.0;
      info->limit_min      = 0.0;
      info->limit_max      = 0.0;
      info->whole_median   = 0.0;
      info->whole_medvar   = 0.0;
      info->whole_mean     = 0.0;
      info->whole_variance = 0.0;
      info->whole_mode     = 0.0;
    }
  // # 1 value
  if (info->whole_count == 1)
    {
      info->q25            = info->values[0];
      info->q75            = info->values[0];
      info->limit_min      = info->values[0];
      info->limit_max      = info->values[0];
      info->whole_median   = info->values[0];
      info->whole_medvar   = 0.0;
      info->whole_mean     = info->values[0];
      info->whole_variance = 0.0;
      info->whole_mode     = info->values[0];
    }
  // # 2 values
  if (info->whole_count == 2)
    {
      info->q25            = CPL_MIN(info->values[0], info->values[1]);
      info->q75            = CPL_MAX(info->values[0], info->values[1]);
      info->limit_min      = CPL_MIN(info->values[0], info->values[1]);
      info->limit_max      = CPL_MAX(info->values[0], info->values[1]);
      info->whole_median   = 0.5*(info->values[0] + info->values[1]);
      info->whole_medvar   = 2.0*(info->values[0] - info->whole_median)*(info->values[0] - info->whole_median);
      info->whole_mean     = 0.5*(info->values[0] + info->values[1]);
      info->whole_variance = 2.0*(info->values[0] - info->whole_mean)*(info->values[0] - info->whole_mean);
      info->whole_mode     = 0.5*(info->values[0] + info->values[1]);
    }
  // # common for all special cases
  if (info->whole_count <= 2)
    {
      info->reduced_count    = info->whole_count;
      info->reduced_median   = info->whole_median;
      info->reduced_medvar   = info->whole_medvar;
      info->reduced_mean     = info->whole_mean;
      info->reduced_variance = info->whole_variance;
      mat_statistics_select_avg_var(info);
      return;
    }
  // ##### 3 and more values #####
  // # prepare calculation
  qsort(info->values, info->whole_count, sizeof(double), mat_double_compare);
  info->q25 = info->values[info->whole_count/4];
  info->q75 = info->values[(3*info->whole_count)/4];
  // # calculate the statistics using all values
  mat_statistics_calc_avg_var(info, 0, info->whole_count-1 , &(info->whole_median), &(info->whole_medvar), &(info->whole_mean), &(info->whole_variance));
  // Try to calculate the mode of the distribution (assumes an unimodal distribution!)
  mat_statistics_calc_density(info);
  mat_statistics_calc_histogram(info);
  info->whole_mode = info->density_peak;
  //info->whole_mode = info->hist_peak;
  //mat_statistics_find_peak(info);
  // # if requested calculate the statistics for values ignoring possible outliers
  if (info->ofactor != 0.0)
    { // Calculate the limit for the outlier detection including indices into info->values[]
      info->limit_min = info->q25 - info->ofactor*(info->q75 - info->q25);
      info->limit_max = info->q75 + info->ofactor*(info->q75 - info->q25);
      for (first = 0; info->values[first] < info->limit_min; first++);
      for (last = info->whole_count - 1; info->values[last] > info->limit_max; last--);
      // Calculate the statistics using only values inside the limits [limit_min .. limit_max] (no outliers)
      info->reduced_count = last - first + 1;
      mat_statistics_calc_avg_var(info, first, last, &(info->reduced_median), &(info->reduced_medvar), &(info->reduced_mean), &(info->reduced_variance));
    }
  else
    { // Copy the statistics calculated from all values
      info->limit_min        = info->values[0];
      info->limit_max        = info->values[info->whole_count - 1];
      info->reduced_count    = info->whole_count;
      info->reduced_median   = info->whole_median;
      info->reduced_medvar   = info->whole_medvar;
      info->reduced_mean     = info->whole_mean;
      info->reduced_variance = info->whole_variance;
    }
  // select and store the requested average and variance
  mat_statistics_select_avg_var(info);
}

/*----------------------------------------------------------------------------*/

void mat_poly_init(mat_poly_info *info, int first_data, int last_data)
{
  double x0 = (double)((first_data + last_data)/2);
  //double x0 = (double)first_data;
  double xs = 64.0;
  //double xs = ((double)(last_data - first_data))/10.0;
  double ys = 1.0;

  info->deg    = 0;
  info->xscale = xs;
  info->yscale = ys;
  info->x0u    = x0;
  info->a0n    = 0.0;
  info->a1n    = 0.0;
  info->a2n    = 0.0;
  info->a3n    = 0.0;
  info->c2t    = 0.0;
  info->c2g    = 0.0;
  info->c2c    = 0;
  info->K      = 0.0;
}

double mat_poly_eval(mat_poly_info *info, double xu)
{
  double xn = (xu - info->x0u)/info->xscale;
  double yn = info->a0n + xn*(info->a1n + xn*(info->a2n + xn*info->a3n));
  return (yn*info->yscale);
}

double mat_poly_derivative(mat_poly_info *info, double xu, int d)
{
  double xn = (xu - info->x0u)/info->xscale;
  double yn;

  switch (d)
    {
    case 0:
      yn = info->a0n + xn*(info->a1n + xn*(info->a2n + xn*info->a3n));
      break;
    case 1:
      yn = info->a1n + xn*(2.0*info->a2n + 3.0*xn*info->a3n);
      break;
    case 2:
      yn = 2.0*info->a2n + 6.0*xn*info->a3n;
      break;
    default:
      yn = 0.0;
    }
  return (yn*info->yscale);
}

/* Bronstein, 19. Auflage, Seite 642, Tabelle 4.1 */
double mat_poly_curvature(mat_poly_info *info, double xl, double xr)
{
  double x;
  double c = 0.0;

  for (x = xl; x <= xr; x += 0.1)
    {
      double d1 = mat_poly_derivative(info, x, 1);
      double d2 = mat_poly_derivative(info, x, 2);
      double hc = d2/pow(1 + d1*d1, 1.5);
      c = CPL_MAX(c, hc);
    }
  return c;
}

// xn = xu/xscale
// yn = yu/yscale
// fn(xn) = an + xn*(bn + xn*(cn + xn*dn))
char *mat_poly_get_func(mat_poly_info *info)
{
  double a0 = info->a0n*info->yscale;
  double a1 = info->a1n*info->yscale/info->xscale;
  double a2 = info->a2n*info->yscale/info->xscale/info->xscale;
  double a3 = info->a3n*info->yscale/info->xscale/info->xscale/info->xscale;

  /*
  switch(info->deg)
    {
    case 0: return cpl_sprintf("f(x) = %f", a0);
    case 1: return cpl_sprintf("f1(x) = %f + %f*(x - %f)", a0, a1, info->x0u);
    case 2: return cpl_sprintf("f2(x) = %f + (x - %f)*(%f + %f*(x - %f))", a0, info->x0u, a1, a2, info->x0u);
    default: return cpl_sprintf("f3(x) = %f + (x - %f)*(%f + (x - %f)*(%f + %f*(x - %f)))", a0, info->x0u, a1, info->x0u, a2, a3, info->x0u);
    }
  */
  switch(info->deg)
    {
    case 0: return cpl_sprintf("f(x) = %f", a0);
    case 1: return cpl_sprintf("f1(x) = %f + %f*(x - %f)", a0, a1, info->x0u);
    case 2: return cpl_sprintf("f2(x) = %f + %f*(x - %f) + %f*(x - %f)**2", a0, a1, info->x0u, a2, info->x0u);
    default: return cpl_sprintf("f3(x) = %f + %f*(x - %f) + %f*(x - %f)**2 + %f*(x - %f)**3", a0, a1, info->x0u, a2, info->x0u, a3, info->x0u);
    }
}

/*----------------------------------------------------------------------------*/

mat_fit_info *mat_fit_info_new(int(*f)(const double x[], const double a[], double *result), int(*dfda)(const double x[], const double a[], double result[]))
{
  mat_fit_info *info;
  int           i;

  info = cpl_calloc(1, sizeof(mat_fit_info));
  if (info == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate a mat_fit_info structure");
      return NULL;
    }
  // data used for function fit
  info->ndspace = 0;
  info->ndused  = 0;
  info->xu2f    = 1.0;
  info->yu2f    = 1.0;
  info->xu      = NULL;
  info->yu      = NULL;
  info->yeu     = NULL;
  // data used for LVM
  info->f = f;
  info->dfda = dfda;
  info->nfspace = 0;
  info->nfused = 0;
  info->fsize = NULL;
  info->mxf = NULL;
  info->vyf = NULL;
  //info->vyef = NULL;
  info->vaf = cpl_vector_new(MAX_COEFF);
  for (i = 0; i < MAX_COEFF; i++)
    {
      info->af[i] = 0;
      info->iaf[i] = 0;
    }
  //cpl_vector      *info->s;
  info->show_flag = 0;
  return info;
}

void mat_fit_info_delete(mat_fit_info *info)
{
  int           i;

  if (info == NULL) return;
  if (info->xu != NULL)
    {
      cpl_free(info->xu);
      info->xu = NULL;
    }
  if (info->yu != NULL)
    {
      cpl_free(info->yu);
      info->yu = NULL;
    }
  if (info->yeu != NULL)
    {
      cpl_free(info->yeu);
      info->yeu = NULL;
    }
  info->ndspace = 0;
  info->ndused = 0;
  for (i = 0; i < info->nfused; i++)
    {
      if (info->mxf[i] != NULL)
	{
	  cpl_matrix_delete(info->mxf[i]);
	  info->mxf[i] = NULL;
	}
      if (info->vyf[i] != NULL)
	{
	  cpl_vector_delete(info->vyf[i]);
	  info->vyf[i] = NULL;
	}
      /*
      if (info->vyef[i] != NULL)
	{
	  cpl_vector_delete(info->vyef[i]);
	  info->vyef[i] = NULL;
	}
      */
    }
  if (info->fsize != NULL)
    {
      cpl_free(info->fsize);
      info->fsize = NULL;
    }
  if (info->mxf != NULL)
    {
      cpl_free(info->mxf);
      info->mxf = NULL;
    }
  if (info->vyf != NULL)
    {
      cpl_free(info->vyf);
      info->vyf = NULL;
    }
  /*
  if (info->vyef != NULL)
    {
      cpl_free(info->vyef);
      info->vyef = NULL;
    }
  */
  info->nfspace = 0;
  info->nfused = 0;
  if (info->vaf != NULL)
    {
      cpl_vector_delete(info->vaf);
      info->vaf = NULL;
    }
  cpl_free(info);
}

void mat_fit_reset(mat_fit_info *info, int show_flag)
{
  info->ndused = 0;
  info->show_flag = show_flag;
}

void mat_fit_add_data(mat_fit_info *info, double x, double y, double ye)
{
  if (info == NULL) return;
  if (info->ndused == info->ndspace)
    {
      info->xu  = (double*)cpl_realloc(info->xu,  (info->ndspace + MAT_GRANULARITY)*sizeof(double));
      info->yu  = (double*)cpl_realloc(info->yu,  (info->ndspace + MAT_GRANULARITY)*sizeof(double));
      info->yeu = (double*)cpl_realloc(info->yeu, (info->ndspace + MAT_GRANULARITY)*sizeof(double));

      info->ndspace += MAT_GRANULARITY;
    }
  if (info->xu == NULL)
    {
      info->ndspace = 0;
      return;
    }
  info->xu[info->ndused]  = x;
  info->yu[info->ndused]  = y;
  info->yeu[info->ndused] = ye;
  info->ndused++;
}

void mat_fit_discard(mat_fit_info *info, int idx)
{
  int i;

  if (idx < 0) return;
  if (idx >= info->ndused) return;
  for (i = idx; i < info->ndused-1; i++)
    {
      info->xu[i]  = info->xu[i+1];
      info->yu[i]  = info->yu[i+1];
      info->yeu[i] = info->yeu[i+1];
    }
  info->ndused--;
}

void mat_fit_set_coeff(mat_fit_info *info, int idx, double value, int active)
{
  if ((idx < 0) || (idx >= MAX_COEFF)) return;
  info->af[idx]  = value;
  info->iaf[idx] = active;
}

void mat_fit_enable_coeff(mat_fit_info *info, int idx, int active)
{
  if ((idx < 0) || (idx >= MAX_COEFF)) return;
  info->iaf[idx] = active;
}

cpl_error_code mat_fit_calc_fit(mat_fit_info *info, int ndused)
{
  int             fi = -1, i;
  cpl_error_code  ec = CPL_ERROR_NONE;

  if (ndused == -1) ndused = info->ndused;
  // try to find the index to mxs, vys and vyes were the size is equal with ndused
  for (i = 0; i < info->nfused; i++)
    {
      if (info->fsize[i] == ndused)
	{
	  fi = i;
	  break;
	}
    }
  // allocate the necessary cpl_matrix and cpl_vector if they do not exist
  if (fi == -1)
    {
      fi = info->nfused;
      if (fi == info->nfspace)
	{ // we need even space to store the pointers
	  info->fsize =         (int*)cpl_realloc(info->fsize, (info->nfspace + 16)*sizeof(int));
	  info->mxf   = (cpl_matrix**)cpl_realloc(info->mxf,   (info->nfspace + 16)*sizeof(cpl_matrix*));
	  info->vyf   = (cpl_vector**)cpl_realloc(info->vyf,   (info->nfspace + 16)*sizeof(cpl_vector*));
	  //info->vyef  = (cpl_vector**)cpl_realloc(info->vyef,  (info->nfspace + 16)*sizeof(cpl_vector*));
	  info->nfspace += 16;
	}
      info->fsize[fi] = ndused;
      if (info->mxf[fi] == NULL)
	{
	  info->mxf[fi] = cpl_matrix_new(info->fsize[fi], 1);
	  if (info->mxf[fi] == NULL)
	    {
	      cpl_msg_error(cpl_func, "could not create the matrix for the x values");
	      return CPL_ERROR_UNSPECIFIED;
	    }
	}
      if (info->vyf[fi] == NULL)
	{
	  info->vyf[fi] = cpl_vector_new(info->fsize[fi]);
	  if (info->vyf[fi] == NULL)
	    {
	      cpl_msg_error(cpl_func, "could not create the vector for the y values");
	      return CPL_ERROR_UNSPECIFIED;
	    }
	}
      /*
      if (info->vyef[fi] == NULL)
	{
	  info->vyef[fi] = cpl_vector_new(info->fsize[fi]);
	  mat_assert_not_null(info->vyef[fi], "could not create the vector for the y error values");
	}
      */
    }
  // copy the specified coefficients (initial value) into the cpl_vector
  for (i = 0; i < MAX_COEFF; i++)
    {
      cpl_vector_set(info->vaf, i, info->af[i]);
    }
  // put the data into the cpl_matrix and cpl_vector
  for (i = 0; i < ndused; i++)
    {
      if (info->show_flag)
	{
	  cpl_msg_debug(cpl_func, "%g %g %g", info->xu[i], info->yu[i], info->yeu[i]);
	}
      cpl_matrix_set(info->mxf[fi],  i, 0, info->xu[i]*info->xu2f);
      cpl_vector_set(info->vyf[fi],  i,    info->yu[i]*info->yu2f);
      //cpl_vector_set(info->vyef[fi], i,    info->yeu[i]*info->xu2f); // ? xu2f ?
    }
  // do the real fit
  ec = cpl_fit_lvmq(info->mxf[fi], NULL,
		    info->vyf[fi], NULL,
		    info->vaf, info->iaf,
		    info->f, info->dfda,
		    CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
		    &(info->mse), NULL, NULL);
  if (ec != CPL_ERROR_NONE)
    {
      cpl_msg_info(cpl_func, "cannot fit a function");
      return ec;
    }
  // copy the fitted coefficients from the cpl_vector
  for (i = 0; i < MAX_COEFF; i++)
    {
      info->af[i] = cpl_vector_get(info->vaf, i);
    }
  return ec;
}

void mat_fit_calc_quality(mat_fit_info *info, int ndused, double xtest, double rdev)
{
  int i;
  int inside = 1;
  
  if (ndused == -1) ndused = info->ndused;
  info->rchi2 = 0.0;
  info->chi2  = 0.0;
  info->limit = 0.0;
  for (i = 1; i < ndused; i++)
    {
      double xm = info->xu[i]; // unscaled
      double ym = info->yu[i]; // unscaled
      double yf;               // unscaled
      double d, dr;
      double hx, hy;           // scaled
      hx = xm*info->xu2f;
      info->f(&hx, info->af, &hy);
      yf = hy/info->yu2f;
      d  = fabs(yf - ym);
      dr = d/ym;
      if (xm <= xtest)
	{
	  info->limit = xm;
	}
      else if (inside && (dr <= rdev))
	{
	  info->limit = xm;
	}
      else
	{
	  inside = 0;
	}
      if (info->show_flag)
	{
	  cpl_msg_debug(cpl_func, "measured %d %g %g -> %g chi2 %g rdist %g",
		       i, xm, ym, yf, d*d, (yf - ym)/ym);
	}
      info->chi2  += d*d;
      info->rchi2 += dr*dr;
    }
  info->stdev  = sqrt(info->chi2/(double)ndused);
  info->rstdev = sqrt(info->rchi2/(double)ndused);
}

/*----------------------------------------------------------------------------*/

mat_smooth_info *mat_smooth_info_new(int nspace,
				     int src_filter,
				     int dst_filter)
{
  int              with_src_filter = (src_filter != MAT_UNDEFINED_INTENSITY);
  int              with_dst_filter = (dst_filter != MAT_UNDEFINED_INTENSITY);
  mat_smooth_info *info;
  int              i;

  info = (mat_smooth_info*)cpl_calloc(1, sizeof(mat_smooth_info));
  if (info == NULL)
    {
      cpl_msg_error(cpl_func, "cannot create the data smoothing context");
      return NULL;
    }
  info->nspace = nspace;
  info->with_src_filter = with_src_filter;
  info->src_filter = src_filter;
  info->with_dst_filter = with_dst_filter;
  info->dst_filter = dst_filter;
  info->original = (double*)cpl_calloc(nspace, sizeof(double));
  if (info->original == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the original data");
      mat_smooth_info_delete(info);
      return NULL;
    }
  info->data = (double*)cpl_calloc(nspace, sizeof(double));
  if (info->data == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the original/modified data");
      mat_smooth_info_delete(info);
      return NULL;
    }
  info->range = (int*)cpl_calloc(nspace, sizeof(int));
  if (info->range == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the intensity range");
      mat_smooth_info_delete(info);
      return NULL;
    }
  info->loval = (double*)cpl_calloc(nspace, sizeof(double));
  if (info->loval == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the lower bounds");
      mat_smooth_info_delete(info);
      return NULL;
    }
  info->hival = (double*)cpl_calloc(nspace, sizeof(double));
  if (info->hival == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the upper bounds");
      mat_smooth_info_delete(info);
      return NULL;
    }
  info->work = (double*)cpl_calloc(nspace, sizeof(double));
  if (info->work == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the work space");
      mat_smooth_info_delete(info);
      return NULL;
    }
  info->smooth = (double*)cpl_calloc(nspace, sizeof(double));
  if (info->smooth == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the smooth data");
      mat_smooth_info_delete(info);
      return NULL;
    }
  info->flags = (int*)cpl_calloc(nspace, sizeof(int));
  if (info->flags == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the flags");
      mat_smooth_info_delete(info);
      return NULL;
    }
  for (i = 0; i < nspace; i++)
    {
      info->original[i] = 2e32;
      info->data[i]     = 2e32;
      info->range[i]    = -1;
      info->loval[i]    = 2e32;
      info->hival[i]    = 2e32;
      info->work[i]     = 2e32;
      info->smooth[i]   = 2e32;
      info->flags[i]    = 0;
    }
  info->show_flag = 0;
  info->chi2limit_flag = 1;
  info->poix = -1;
  info->poiy = -1;
  info->rx = 0;
  info->ry = 0;
  info->rnx = 0;
  info->rny = 0;
  info->nused = 0;
  info->first = 0;
  info->last = 0;
  info->count = 0;
  mat_poly_init(&(info->poly), 0, 64);
  return info;
}

void mat_smooth_info_delete(mat_smooth_info *info)
{
  if (info == NULL) return;
  if (info->original != NULL) cpl_free(info->original);
  if (info->data     != NULL) cpl_free(info->data);
  if (info->range    != NULL) cpl_free(info->range);
  if (info->loval    != NULL) cpl_free(info->loval);
  if (info->hival    != NULL) cpl_free(info->hival);
  if (info->work     != NULL) cpl_free(info->work);
  if (info->smooth   != NULL) cpl_free(info->smooth);
  if (info->flags    != NULL) cpl_free(info->flags);
  cpl_free(info);
}

void mat_smooth_reset(mat_smooth_info *info)
{
  info->nused = 0;
  info->rx = 0;
  info->ry = 0;
  info->rnx = 0;
  info->rny = 0;
  info->first = -1;
  info->last = -1;
}

void mat_smooth_add_value(mat_smooth_info *info, double value)
{
  if (info->nspace < info->nused)
    {
      cpl_msg_warning(cpl_func, "try to add value %f which is bejond the available space %d", value, info->nspace);
      return;
    }
  info->rnx                   += 1;
  info->rny                    = 1;
  info->original[info->nused]  = value;
  info->data[info->nused]      = info->original[info->nused];
  info->flags[info->nused]     = MAT_NO_FLAGS;
  info->range[info->nused]     = MAT_LINEAR_INTENSITY;
  info->first                  = 0;
  info->last                   = info->nused;
  info->nused++;
}


void mat_smooth_set(mat_smooth_info *info, cpl_image *src, cpl_image *range, int x, int y, int nx, int ny)
{
  int i, j;

  if (info->nspace < (nx*ny))
    {
      cpl_msg_warning(cpl_func, "try to add values %dx%d@%d,%d which is bejond the available space %d", nx, ny, x, y, info->nspace);
      return;
    }
  info->nused = 0;
  info->rx = x;
  info->ry = y;
  info->rnx = nx;
  info->rny = ny;
  for (j = y; j < y + ny; j++)
    {
      for (i = x; i < x + nx; i++)
	{
	  int rejected;
	  info->original[info->nused] = cpl_image_get(src, i, j, &rejected);
	  info->data[info->nused]     = info->original[info->nused];
	  info->flags[info->nused]    = MAT_NO_FLAGS;
	  if (rejected)
	    {
	      info->range[info->nused] = MAT_BADPIX_INTENSITY;
	    }
	  else if (range == NULL)
	    {
	      info->range[info->nused] = MAT_LINEAR_INTENSITY;
	    }
	  else
	    {
	      info->range[info->nused] = cpl_image_get(range, i, j, &rejected);
	    }
	  info->nused++;
	}
    }
  if (info->with_src_filter)
    {
      info->first = -1;
      for (i = 0; i < info->nused; i++)
	{
	  if (info->range[i] == info->src_filter)
	    {
	      info->first = i;
	      break;
	    }
	}
    }
  else
    {
      info->first = 0;
    }
  if (info->with_src_filter)
    {
      info->last = -1;
      for (i = info->nused - 1; i > 0; i--)
	{
	  if (info->range[i] == info->src_filter)
	    {
	      info->last = i;
	      break;
	    }
	}
    }
  else
    {
      info->last = info->nused - 1;
    }
}

void mat_smooth_apply(mat_smooth_info *info, cpl_image *dst, double scale)
{
  int i, j, k;

  if (dst == NULL) return;
  k = 0;
  if (info->with_dst_filter)
    {
      for (j = info->ry; j < info->ry + info->rny; j++)
	{
	  for (i = info->rx; i < info->rx + info->rnx; i++)
	    {
	      if (   (info->range[k] == info->dst_filter)
		     || (info->range[k] == MAT_BADPIX_INTENSITY)
		     || (info->range[k] == MAT_OUTLIER_INTENSITY)
		     )
		{
		  int rejected;
		  cpl_image_set(dst, i, j, cpl_image_get(dst, i, j, &rejected) + scale*info->smooth[k]);
		}
	      k++;
	    }
	}
    }
  else
    {
      for (j = info->ry; j < info->ry + info->rny; j++)
	{
	  for (i = info->rx; i < info->rx + info->rnx; i++)
	    {
	      int rejected;
	      cpl_image_set(dst, i, j, cpl_image_get(dst, i, j, &rejected) + scale*info->smooth[k]);
	      k++;
	    }
	}
    }
}

static int mat_check_data_point(mat_smooth_info *info, int i)
{
  if ((i < 0) || (i >= info->nused))
    {
      cpl_msg_warning(cpl_func, "data index out of bounds %d not in [0 .. %d]", i, info->nused-1);
      return 0;
    }
  if (!info->with_src_filter) {
    return 1;
  } else if (info->range[i] == info->src_filter) {
    return 1;
  } else {
    return 0;
  }
}

void mat_smooth_calc_result(mat_smooth_info *info, int first_smooth, int last_smooth)
{
  int     i;

  // calculate the smooth values
  for (i = first_smooth; i <= last_smooth; i++)
    {
      info->smooth[i] = mat_poly_eval(&(info->poly), i);
    }
}

static double mat_smooth_calc_chi2(mat_smooth_info *info,
				   mat_statistics_info *stat_info,
				   int first_data, int last_data,
				   int mask)
{
  int     i;
  double  chi2limit = 1e32;
  double  sum = 0.0;
  int     count = 0;

  // Use the current guess to calculate all chi2 and the chi2limit to distinguish between useful data points and garbage during the next iteration
  mat_statistics_reset(stat_info);
  for (i = first_data; i <= last_data; i++)
    {
      double du; // user space
      if (!mat_check_data_point(info, i)) continue;
      if ((info->flags[i] & MAT_OUTLIER_FLAG) != 0) continue;
      du = info->data[i] - mat_poly_eval(&(info->poly), (double)i);
      //mat_statistics_add_value(stat_info, fabs(du));
      mat_statistics_add_value(stat_info, du*du);
    }
  mat_statistics_calc(stat_info);
  info->poly.c2t = stat_info->whole_total;
  chi2limit = stat_info->limit_max;
  for (i = first_data; i <= last_data; i++)
    {
      double du; // user space
      if (!mat_check_data_point(info, i)) continue;
      du = info->data[i] - mat_poly_eval(&(info->poly), (double)i);
      //if (info->chi2limit_flag && (fabs(du) > chi2limit)) continue;
      if (info->chi2limit_flag && (du*du > chi2limit)) continue;
      if ((info->flags[i] & mask) != 0) continue;
      //sum   += fabs(du);
      sum   += du*du;
      count += 1;
    }
  info->poly.c2g = sum;
  info->poly.c2c = count;
  return chi2limit;
}

void mat_smooth_mark_outlier(mat_smooth_info *info, int first_data, int last_data, double chi2limit)
{
  int     i;

  for (i = first_data; i <= last_data; i++)
    {
      double du = info->data[i] - mat_poly_eval(&(info->poly), (double)i);
      //info->loval[i] = fabs(du);
      info->loval[i] = du*du;
      info->hival[i] = chi2limit;
      //if (fabs(du) > chi2limit) info->flags[i] |= MAT_OUTLIER_FLAG;
      if (du*du > chi2limit) info->flags[i] |= MAT_OUTLIER_FLAG;
    }
}

/*
  Chi2 fit or a 3rd order polynome (y = a + bx + cx^2 + dx^3)
  ===========================================================

  ---------------- Mathematica ----------------
  Clear[f]
  f[x_] := a + b*x + c*x^2 + d*x^3
  Expand[(y - f[x])^2]
  =>
  a^2 + 2 a b x + b^2 x^2 + 2 a c x^2 + 2 b c x^3 + 2 a d x^3 + c^2 x^4 + 2 b d x^4 + 2 c d x^5 + d^2 x^6 - 2 a y - 2 b x y - 2 c x^2 y - 2 d x^3 y + y^2
  ---------------- Mathematica ----------------
  Clear[f]
  f[x_, y_] := 
  N*a^2 + 2*a*b*sx + b^2*sxx + 2*a*c*sxx + 2*b*c*sxxx + 2*a*d*sxxx + c^2*sxxxx + 2*b*d*sxxxx + 2*c*d*sxxxxx + d^2*sxxxxxx - 2*a*sy - 2*b*sxy - 2*c*sxxy - 2*d*sxxxy + syy
  D[f[x, y], a]
  D[f[x, y], b]
  D[f[x, y], c]
  D[f[x, y], d]
  =>
  2 a N + 2 b sx + 2 c sxx + 2 d sxxx - 2 sy
  2 a sx + 2 b sxx + 2 c sxxx + 2 d sxxxx - 2 sxy
  2 a sxx + 2 b sxxx + 2 c sxxxx + 2 d sxxxxx - 2 sxxy
  2 a sxxx + 2 b sxxxx + 2 c sxxxxx + 2 d sxxxxxx - 2 sxxxy
  ----------------
  Deriving d from equation 1-4:
  ---------------- Mathematica ----------------
  Solve[a*N + b*sx + c*sxx + d*sxxx == sy && a*sx + b*sxx + c*sxxx + d*sxxxx == sxy && a*sxx + b*sxxx + c*sxxxx + d*sxxxxx == sxxy && a*sxxx + b*sxxxx + c*sxxxxx + d*sxxxxxx == sxxxy, {a, b, c, d}]
  =>
  d = -((sxx^3 sxxxy - 2 sx sxx sxxx sxxxy + N sxxx^2 sxxxy + sx^2 sxxxx sxxxy - N sxx sxxxx sxxxy - sxx^2 sxxx sxxy + sx sxxx^2 sxxy + sx sxx sxxxx sxxy - N sxxx sxxxx sxxy - sx^2 sxxxxx sxxy + N sxx sxxxxx sxxy + sxx sxxx^2 sxy - sxx^2 sxxxx sxy - sx sxxx sxxxx sxy + N sxxxx^2 sxy + sx sxx sxxxxx sxy - N sxxx sxxxxx sxy - sxxx^3 sy + 2 sxx sxxx sxxxx sy - sx sxxxx^2 sy - sxx^2 sxxxxx sy + sx sxxx sxxxxx sy)/(sxxx^4 - 3 sxx sxxx^2 sxxxx + sxx^2 sxxxx^2 + 2 sx sxxx sxxxx^2 - N sxxxx^3 + 2 sxx^2 sxxx sxxxxx - 2 sx sxxx^2 sxxxxx - 2 sx sxx sxxxx sxxxxx + 2 N sxxx sxxxx sxxxxx + sx^2 sxxxxx^2 - N sxx sxxxxx^2 - sxx^3 sxxxxxx + 2 sx sxx sxxx sxxxxxx - N sxxx^2 sxxxxxx - sx^2 sxxxx sxxxxxx + N sxx sxxxx sxxxxxx))
  ----------------
  Deriving c from equation 1-3, using d:
  ---------------- Mathematica ----------------
  Solve[a*N + b*sx + c*sxx + d*sxxx == sy && a*sx + b*sxx + c*sxxx + d*sxxxx == sxy && a*sxx + b*sxxx + c*sxxxx + d*sxxxxx == sxxy, {a, b, c}]
  =>
  c = -((d sxx^2 sxxx - d sx sxxx^2 - d sx sxx sxxxx + d N sxxx sxxxx + d sx^2 sxxxxx - d N sxx sxxxxx - sx^2 sxxy + N sxx sxxy + sx sxx sxy - N sxxx sxy - sxx^2 sy + sx sxxx sy)/(sxx^3 - 2 sx sxx sxxx + N sxxx^2 + sx^2 sxxxx - N sxx sxxxx))
  ----------------
  Deriving b from equation 1-2, using c, d:
  ---------------- Mathematica ----------------
  Solve[a*N + b*sx + c*sxx + d*sxxx == sy && a*sx + b*sxx + c*sxxx + d*sxxxx == sxy , {a, b}]
  =>
  b = -((c sx sxx - c N sxxx + d sx sxxx - d N sxxxx + N sxy - sx sy)/(sx^2 - N sxx))
  ----------------
  Deriving a from equation 1, using b, c, d:
  ---------------- Mathematica ----------------
  Solve[a*N + b*sx + c*sxx + d*sxxx == sy, {a}]
  =>
  a = (-b sx - c sxx - d sxxx + sy)/N
  ----------------
*/
static double mat_smooth_polyn_iter(mat_smooth_info *info, mat_statistics_info *stat_info, int n, int first_data, int last_data, double chi2limit, int mask)
{
  int      i;
  double   N       = 0.0;
  double   sx      = 0.0;
  double   sxx     = 0.0;
  double   sxxx    = 0.0;
  double   sxxxx   = 0.0;
  double   sxxxxx  = 0.0;
  double   sxxxxxx = 0.0;
  double   sy      = 0.0;
  double   sxy     = 0.0;
  double   sxxy    = 0.0;
  double   sxxxy   = 0.0;
  double   syy     = 0.0;
  double   a0 = 0.0, a1 = 0.0, a2 = 0.0, a3 = 0.0;

  // calculate all necessary sums ignoring data points with a chi2 larger than chi2limit using the existing polynome fit
  //cpl_msg_debug(cpl_func, "n = %d, data = [%d .. %d], chi2limit  = %f, mask = %d, chi2limit_flag = %d", n, first_data, last_data, chi2limit, mask, info->chi2limit_flag);
  if (n == 0) mat_statistics_reset(stat_info);
  for (i = first_data; i <= last_data; i++)
    {
      double du, xn, yn; // du user space, yn normalized space
      if (!mat_check_data_point(info, i)) continue;
      if ((info->flags[i] & mask) != 0) continue;
      if ((info->flags[i] & MAT_OUTLIER_FLAG) != 0) continue;
      if ((first_data == 16) && (last_data == 47))
	{
	  //cpl_msg_debug(cpl_func, "  data[%d] = %f, f(%d) = %f", i, info->data[i], i, mat_smooth_eval(info, i));
	  //cpl_msg_debug(cpl_func, "  data[%d] = %f", i, info->data[i]);
	}
      du = info->data[i] - mat_poly_eval(&(info->poly), i);
      if ((first_data == 16) && (last_data == 47))
	{
	  //cpl_msg_debug(cpl_func, "  data[%d] = %f, f(%d) = %f", i, info->data[i], i, mat_smooth_eval(info, i));
	  //cpl_msg_debug(cpl_func, "  data[%d] = %f du = %f", i, info->data[i], du);
	}
      //if (info->chi2limit_flag && (fabs(du) > chi2limit)) continue;
      if (info->chi2limit_flag && (du*du > chi2limit)) continue;
      xn = ((double)i - info->poly.x0u)/info->poly.xscale;
      yn = info->data[i]/info->poly.yscale;
      N       += 1.0;
      if (n == 0) mat_statistics_add_value(stat_info, yn);
      if (n >= 1)
	{
	  sx      += xn;
	  sxx     += xn*xn;
	  sy      += yn;
	  sxy     += xn*yn;
	}
      if (n >= 2)
	{
	  sxxx    += xn*xn*xn;
	  sxxxx   += xn*xn*xn*xn;
	  sxxy    += xn*xn*yn;
	}
      if (n >= 3)
	{
	  sxxxxx  += xn*xn*xn*xn*xn;
	  sxxxxxx += xn*xn*xn*xn*xn*xn;
	  sxxxy   += xn*xn*xn*yn;
	  syy     += yn*yn;
	}
    }
  info->poly.deg = n;
  if (N < 5.0)
    {
      char *str;
      cpl_msg_warning(cpl_func, "mat_smooth_polyn_iter(.., .., %d, %d, %d, %f, %d), not enough data points, got only %d data points",
		      n, first_data, last_data, chi2limit, mask, (int)N);
      str = mat_poly_get_func(&(info->poly));
      cpl_msg_warning(cpl_func, "  previous function: %s", str);
      cpl_free(str);
      info->poly.a0n = 0.0;
      info->poly.a1n = 0.0;
      info->poly.a2n = 0.0;
      info->poly.a3n = 0.0;
      return -1.0;
    }
  if (n == 0)
    {
      mat_statistics_calc(stat_info);
      a0 = stat_info->reduced_median;
    }
  else if (n == 1)
    {
      double xq = sx/N;
      double yq = sy/N;
      double ssxx = sxx - N*xq*xq;
      double ssxy = sxy - N*xq*yq;
      if (ssxx == 0.0)
	{ // degenerated straight line
	  return -1.0;
	}
      a1 = ssxy/ssxx;
      a0 = yq - a1*xq;
    }
  else if (n == 2)
    {
      double det = sxx*sxx*sxx - 2*sx*sxx*sxxx + N*sxxx*sxxx + sx*sx*sxxxx - N*sxx*sxxxx;
      a0 = -(-sxx*sxx*sxxy + sx*sxxx*sxxy + sxx*sxxx*sxy - sx*sxxxx*sxy - sxxx*sxxx*sy + sxx*sxxxx*sy)/det;
      a1 = -(sx*sxx*sxxy - N*sxxx*sxxy - sxx*sxx*sxy + N*sxxxx*sxy + sxx*sxxx*sy - sx*sxxxx*sy)/det;
      a2 = -(-sx*sx*sxxy + N*sxx*sxxy + sx*sxx*sxy - N*sxxx*sxy - sxx*sxx*sy + sx*sxxx*sy)/det;
    }
  else if (n == 3)
    {
      a3 =
	-(
	  (
	   sxx*sxx*sxx*sxxxy - 2*sx*sxx*sxxx*sxxxy + N*sxxx*sxxx*sxxxy + sx*sx*sxxxx*sxxxy
	   - N*sxx*sxxxx*sxxxy - sxx*sxx*sxxx*sxxy + sx*sxxx*sxxx*sxxy + sx*sxx*sxxxx*sxxy
	   - N*sxxx*sxxxx*sxxy - sx*sx*sxxxxx*sxxy + N*sxx*sxxxxx*sxxy + sxx*sxxx*sxxx*sxy
	   - sxx*sxx*sxxxx*sxy - sx*sxxx*sxxxx*sxy + N*sxxxx*sxxxx*sxy + sx*sxx*sxxxxx*sxy
	   - N*sxxx*sxxxxx*sxy - sxxx*sxxx*sxxx*sy + 2*sxx*sxxx*sxxxx*sy - sx*sxxxx*sxxxx*sy
	   - sxx*sxx*sxxxxx*sy + sx*sxxx*sxxxxx*sy
	   )
	  /
	  (
	   sxxx*sxxx*sxxx*sxxx - 3*sxx*sxxx*sxxx*sxxxx + sxx*sxx*sxxxx*sxxxx + 2*sx*sxxx*sxxxx*sxxxx
	   - N*sxxxx*sxxxx*sxxxx + 2*sxx*sxx*sxxx*sxxxxx - 2*sx*sxxx*sxxx*sxxxxx - 2*sx*sxx*sxxxx*sxxxxx
	   + 2*N*sxxx*sxxxx*sxxxxx + sx*sx*sxxxxx*sxxxxx - N*sxx*sxxxxx*sxxxxx - sxx*sxx*sxx*sxxxxxx
	   + 2*sx*sxx*sxxx*sxxxxxx - N*sxxx*sxxx*sxxxxxx - sx*sx*sxxxx*sxxxxxx + N*sxx*sxxxx*sxxxxxx
	   )
	  );
      a2 =
	-(
	  (
	   a3*sxx*sxx*sxxx - a3*sx*sxxx*sxxx - a3*sx*sxx*sxxxx + a3*N*sxxx*sxxxx + a3*sx*sx*sxxxxx - a3*N*sxx*sxxxxx
	   - sx*sx*sxxy + N*sxx*sxxy + sx*sxx*sxy - N*sxxx*sxy - sxx*sxx*sy + sx*sxxx*sy
	   )
	  /
	  (
	   sxx*sxx*sxx - 2*sx*sxx*sxxx + N*sxxx*sxxx + sx*sx*sxxxx - N*sxx*sxxxx
	   )
	  );
      a1 = -((a2*sx*sxx - a2*N*sxxx + a3*sx*sxxx - a3*N*sxxxx + N*sxy - sx*sy)/(sx*sx - N*sxx));
      a0 = (-a1*sx - a2*sxx - a3*sxxx + sy)/N;
    }
  info->poly.a0n = a0;
  info->poly.a1n = a1;
  info->poly.a2n = a2;
  info->poly.a3n = a3;
  info->poly.K = mat_poly_curvature(&(info->poly), (double)first_data, (double)last_data);
  return mat_smooth_calc_chi2(info, stat_info, first_data, last_data, mask);
}

double mat_smooth_polyn_window(mat_smooth_info *info, mat_statistics_info *stat_info, int n, int first_data, int last_data, int first_smooth, int last_smooth)
{
  double  chi2limit = 1e32;

  // Start without any guess
  mat_poly_init(&(info->poly), first_data, last_data);
  info->chi2limit_flag = 0;
  chi2limit = mat_smooth_polyn_iter(info, stat_info, n, first_data, last_data, chi2limit, MAT_NO_FLAGS);
  mat_smooth_calc_result(info, first_smooth, last_smooth);
  if (info->show_flag)
    {
      char *str = mat_poly_get_func(&(info->poly));
      cpl_msg_debug(cpl_func, "poly%d chi2 # %s -> %f %f %d %f", n, str, info->poly.c2t, info->poly.c2g, info->poly.c2c, info->poly.K);
      cpl_free(str);
    }
  return chi2limit;
}

double mat_smooth_polyx_window(mat_smooth_info *info, mat_statistics_info *stat_info,
			       int first_data, int last_data,
			       int first_smooth, int last_smooth)
{
  mat_poly_info   p1, p2, p3;
  double          p1c2l, p2c2l, p3c2l; // polynome specific chi2limit
  double          chi2limit;

  // try 1-order .. 3-order polynome and store the result
  p1c2l = mat_smooth_polyn_window(info, stat_info, 1, first_data, last_data, first_smooth, last_smooth);
  p1    = info->poly;
  p2c2l = mat_smooth_polyn_window(info, stat_info, 2, first_data, last_data, first_smooth, last_smooth);
  p2    = info->poly;
  p3c2l = mat_smooth_polyn_window(info, stat_info, 3, first_data, last_data, first_smooth, last_smooth);
  p3    = info->poly;
  // check which polynome is a better fit (chi2 and curvature) and restore this solution
  if ((p1.c2t < p2.c2t) && (p1.c2t < p3.c2t))
    { // poly-1 gives the best fit -> restore this solution
      info->poly = p1;
      chi2limit  = p1c2l;
    }
  else if (p2.c2t*p2.K < p3.c2t*p3.K)
    { // poly-2 gives the best fit including smaller curvature (larger radius) -> restore this solution
      info->poly = p2;
      chi2limit  = p2c2l;
    }
  else
    { // poly-3 gives the best fit -> restore this solution
      info->poly = p3;
      chi2limit  = p3c2l;
    }
  if (info->show_flag)
    {
      char *str = mat_poly_get_func(&(info->poly));
      cpl_msg_debug(cpl_func, "polyx final # %s -> %f %f %d %f", str, info->poly.c2t, info->poly.c2g, info->poly.c2c, info->poly.K);
      cpl_free(str);
    }
  mat_smooth_calc_result(info, first_smooth, last_smooth);
  return chi2limit;
}

/*
double mat_smooth_polyn_window_chi2(mat_smooth_info *info, mat_statistics_info *stat_info, int n, int first_data, int last_data, int first_smooth, int last_smooth)
{
  int     irun;
  double  chi2limit = 1e32;

  // Start without any guess
  mat_poly_init(&(info->poly), first_data, last_data);
  info->chi2limit_flag = 1;
  // The first guess is always based on a simple offset
  chi2limit = mat_smooth_polyn_iter(info, stat_info, 0, first_data, last_data, chi2limit, MAT_NO_FLAGS);
  if (chi2limit == -1.0) return -1.0;
  if (n == 0)
    { // we already have the best results, no iteration needed
      mat_smooth_calc_result(info, first_smooth, last_smooth);
      return (info->poly.c2g/info->poly.c2c);
    }
  // We will improve the result by fitting a straight line through the remaining data points at least once
  chi2limit = mat_smooth_polyn_iter(info, stat_info, 1, first_data, last_data, chi2limit, MAT_NO_FLAGS);
  if (chi2limit == -1.0) return -1.0;
  if (n == 1)
    { // we want to improve the straight line fit by further iterations
      for (irun = 1; irun <= MAT_NRUNS; irun++) // we already made one iteration
	{
	  chi2limit = mat_smooth_polyn_iter(info, stat_info, 1, first_data, last_data, chi2limit, MAT_NO_FLAGS);
	  if (chi2limit == -1.0) return -1.0;
	}
      mat_smooth_calc_result(info, first_smooth, last_smooth);
    }
  // We will improve the result by fitting a 2-order polynome through the remaining data points at least once
  chi2limit = mat_smooth_polyn_iter(info, stat_info, 2, first_data, last_data, chi2limit, MAT_NO_FLAGS);
  if (chi2limit == -1.0) return -1.0;
  if (n == 2)
    { // we want to improve the 2-order polynome fit by further iterations
      for (irun = 1; irun <= MAT_NRUNS; irun++) // we already made one iteration
	{
	  chi2limit = mat_smooth_polyn_iter(info, stat_info, 2, first_data, last_data, chi2limit, MAT_NO_FLAGS);
	  if (chi2limit == -1.0) return -1.0;
	}
      mat_smooth_calc_result(info, first_smooth, last_smooth);
    }
  // We will improve the result by fitting a 3-order polynome through the remaining data points at least once
  chi2limit = mat_smooth_polyn_iter(info, stat_info, 3, first_data, last_data, chi2limit, MAT_NO_FLAGS);
  if (chi2limit == -1.0) return -1.0;
  if (n == 3)
    { // we want to improve the 3-order polynome fit by further iterations
      for (irun = 1; irun <= MAT_NRUNS; irun++) // we already made one iteration
	{
	  chi2limit = mat_smooth_polyn_iter(info, stat_info, 3, first_data, last_data, chi2limit, MAT_NO_FLAGS);
	  if (chi2limit == -1.0) return -1.0;
	}
      mat_smooth_calc_result(info, first_smooth, last_smooth);
    }
  if (info->show_flag)
    {
      char *str = mat_smooth_get_func(info);
      cpl_msg_debug(cpl_func, "poly%d chi2 # %s -> %f %f %d", n, str, info->poly.c2t, info->poly.c2g, info->poly.c2c);
      cpl_free(str);
    }
  return chi2limit;
}
*/

double mat_smooth_polyn_window_chi2(mat_smooth_info *info, mat_statistics_info *stat_info, int n, int first_data, int last_data, int first_smooth, int last_smooth)
{
  int      irun;
  double   chi2limit = 1e32;

  // Start without any guess
  mat_poly_init(&(info->poly), first_data, last_data);
  info->chi2limit_flag = 1;
  // 1. fit a 0-order polynome
  if (n >= 0)
    {
      chi2limit = mat_smooth_polyn_iter(info, stat_info, 0, first_data, last_data, 1e32, MAT_NO_FLAGS);
      if (chi2limit == -1.0) return -1.0; // no fit possible
    }
  // 2. fit a 1-order polynome
  if (n >= 1)
    {
      chi2limit = 1e32;
      for (irun = 0; irun <= MAT_NRUNS; irun++)
	{
	  chi2limit = mat_smooth_polyn_iter(info, stat_info, 1, first_data, last_data, chi2limit, MAT_NO_FLAGS);
	  if (chi2limit == -1.0) return -1.0; // no fit possible
	}
    }
  // 3. fit a 2-order polynome
  if (n >= 2)
    {
      chi2limit = 1e32;
      for (irun = 0; irun <= MAT_NRUNS; irun++)
	{
	  chi2limit = mat_smooth_polyn_iter(info, stat_info, 2, first_data, last_data, chi2limit, MAT_NO_FLAGS);
	  if (chi2limit == -1.0) return -1.0; // no fit possible
	}
    }
  // 4. fit a 3-order polynome
  if (n >= 3)
    {
      chi2limit = 1e32;
      for (irun = 0; irun <= MAT_NRUNS; irun++)
	{
	  chi2limit = mat_smooth_polyn_iter(info, stat_info, 3, first_data, last_data, chi2limit, MAT_NO_FLAGS);
	  if (chi2limit == -1.0) return -1.0; // no fit possible
	}
    }
  mat_smooth_calc_result(info, first_smooth, last_smooth);
  if (info->show_flag)
    {
      char *str = mat_poly_get_func(&(info->poly));
      cpl_msg_debug(cpl_func, "poly%d chi2 # %s -> %f %f %d %f", n, str, info->poly.c2t, info->poly.c2g, info->poly.c2c, info->poly.K);
      cpl_free(str);
    }
  return chi2limit;
}

double mat_smooth_polyx_window_chi2(mat_smooth_info *info, mat_statistics_info *stat_info,
				    int first_data, int last_data,
				    int first_smooth, int last_smooth)
{
  int             i;
  mat_poly_info   p1, p2, p3;
  double          p1c2l = 1e32, p2c2l = 1e32, p3c2l = 1e32; // polynome specific chi2limit
  double          p1q, p2q, p3q;
  double          chi2limit;

  // Start without any guess
  mat_poly_init(&(info->poly), first_data, last_data);
  info->chi2limit_flag = 1;
  for (i = first_data; i <= last_data; i++) info->flags[i] &= ~MAT_OUTLIER_MASK; // use all data points
  // 1. fit a 0-order polynome
  chi2limit = mat_smooth_polyn_iter(info, stat_info, 0, first_data, last_data, 1e32, MAT_NO_FLAGS);
  if (chi2limit == -1.0) return -1.0; // no fit possible
  // 2. fit a 1-order polynome
  for (i = 0; i <= MAT_NRUNS; i++)
    {
      p1c2l = mat_smooth_polyn_iter(info, stat_info, 1, first_data, last_data, p1c2l, MAT_NO_FLAGS);
      if (p1c2l == -1.0) return -1.0; // no fit possible
    }
  if (info->show_flag)
    {
      char *str = mat_poly_get_func(&(info->poly));
      cpl_msg_debug(cpl_func, "polyx chi2 try # %s -> %f %f %d %f", str, info->poly.c2t, info->poly.c2g, info->poly.c2c, info->poly.K);
      cpl_free(str);
    }
  p1 = info->poly;
  // 3. fit a 2-order polynome
  for (i = 0; i <= MAT_NRUNS; i++)
    {
      p2c2l = mat_smooth_polyn_iter(info, stat_info, 2, first_data, last_data, p2c2l, MAT_NO_FLAGS);
      if (p2c2l == -1.0) return -1.0; // no fit possible
    }
  if (info->show_flag)
    {
      char *str = mat_poly_get_func(&(info->poly));
      cpl_msg_debug(cpl_func, "polyx chi2 try # %s -> %f %f %d %f", str, info->poly.c2t, info->poly.c2g, info->poly.c2c, info->poly.K);
      cpl_free(str);
    }
  p2 = info->poly;
  // 4. fit a 3-order polynome
  for (i = 0; i <= MAT_NRUNS; i++)
    {
      p3c2l = mat_smooth_polyn_iter(info, stat_info, 3, first_data, last_data, p3c2l, MAT_NO_FLAGS);
      if (p3c2l == -1.0) return -1.0; // no fit possible
    }
  if (info->show_flag)
    {
      char *str = mat_poly_get_func(&(info->poly));
      cpl_msg_debug(cpl_func, "polyx chi2 try # %s -> %f %f %d %f", str, info->poly.c2t, info->poly.c2g, info->poly.c2c, info->poly.K);
      cpl_free(str);
    }
  p3 = info->poly;
  /*
  p1q = p1.c2t;
  p2q = p2.c2t;
  p3q = p3.c2t;
  */
  p1q = p1.c2g/p1.c2c;
  p2q = p2.c2g/p2.c2c;
  p3q = p3.c2g/p3.c2c;
  // check which polynome is a better fit (chi2 only) and restore this solution
  if ((p1q < p2q) && (p1q < p3q))
    { // poly-1 gives the best fit -> restore this solution
      info->poly = p1;
      chi2limit  = p1c2l;
    }
  else if ((p2q < p1q) && (p2q < p3q))
    { // poly-2 gives the best fit -> restore this solution
      info->poly = p2;
      chi2limit  = p2c2l;
    }
  else
    { // poly-3 gives the best fit -> restore this solution
      info->poly = p3;
      chi2limit  = p3c2l;
    }
  if (info->show_flag)
    {
      char *str = mat_poly_get_func(&(info->poly));
      cpl_msg_debug(cpl_func, "polyx chi2 final # %s -> %f %f %d %f %f %f %f", str, info->poly.c2t, info->poly.c2g, info->poly.c2c, info->poly.K, p1q, p2q, p3q);
      cpl_free(str);
    }
  mat_smooth_calc_result(info, first_smooth, last_smooth);
  return chi2limit;
}

double mat_smooth_polyn_window_outlier(mat_smooth_info *info, mat_statistics_info *stat_info, int n,
				       int first_data, int last_data,
				       int first_smooth, int last_smooth,
				       double enhance_factor)
{
  int      i;
  int      noutlier = 0;
  int      flag;
  double   fchi2;

  // calculate the initial fit using all data points
  for (i = first_data; i <= last_data; i++) info->flags[i] &= ~MAT_OUTLIER_MASK; // use all data points
  mat_poly_init(&(info->poly), first_data, last_data);
  info->chi2limit_flag = 0;
  if (n == 0)
    flag = MAT_POLY0_OUTLIER_FLAG;
  else if (n == 1)
    flag = MAT_POLY1_OUTLIER_FLAG;
  else if (n == 2)
    flag = MAT_POLY2_OUTLIER_FLAG;
  else if (n == 3)
    flag = MAT_POLY3_OUTLIER_FLAG;
  else
    flag = MAT_NO_FLAGS;
  mat_smooth_polyn_iter(info, stat_info, n, first_data, last_data, 1e32, flag);
  ///*
  if (info->show_flag)
    {
      char *str = mat_poly_get_func(&(info->poly));
      cpl_msg_debug(cpl_func, "poly%d init # %s -> %f %f %d %f", n, str, info->poly.c2t, info->poly.c2g, info->poly.c2c, info->poly.K);
      cpl_free(str);
    }
  //*/
  //fchi2 = info->poly.c2t;
  fchi2 = info->poly.c2g;
  //fchi2 = info->poly.c2g/info->poly.c2c;
  // Try if the fit is much better if a data point is marked as outlier (flags[i] |= flag)
  while (((info->nused - noutlier) > 10) && (noutlier < 8))
    {
      int    found = 0;
      int    houtlier = -1;
      double hchi2 = enhance_factor*fchi2;
      for (i = first_data; i <= last_data; i++)
	{
	  double hhchi2;
	  if ((info->flags[i] & flag) != 0) continue; // already marked as outlier
	  if ((info->flags[i] & MAT_OUTLIER_FLAG) != 0) continue; // already marked as outlier
	  info->flags[i] |=  flag; // set outlier flag
	  mat_smooth_polyn_iter(info, stat_info, n, first_data, last_data, 1e32, flag);
	  info->flags[i] &= ~flag; // clear outlier flag
	  ///*
	  if (info->show_flag)
	    {
	      char *str = mat_poly_get_func(&(info->poly));
	      cpl_msg_debug(cpl_func, "poly%d iter # %s -> %f %f %d %f (ol %d)", n, str, info->poly.c2t, info->poly.c2g, info->poly.c2c, info->poly.K, i);
	      cpl_free(str);
	    }
	  //*/
	  //hhchi2 = info->poly.c2t;
	  hhchi2 = info->poly.c2g;
	  //hhchi2 = info->poly.c2g/info->poly.c2c;
	  if (hhchi2 < hchi2)
	    {
	      hchi2 = hhchi2;
	      houtlier = i;
	      found = 1;
	    }
	}
      if (!found) break;
      fchi2 = hchi2;
      info->flags[houtlier] |=  flag; // set outlier flag
      noutlier++;
      if (info->show_flag) cpl_msg_debug(cpl_func, "poly%d outlier # %d", n, houtlier);
    }
  // redo the best fit (it matches with the current info->flags[]!)
  mat_smooth_polyn_iter(info, stat_info, n, first_data, last_data, 1e32, flag);
  if (info->show_flag)
    {
      char *str = mat_poly_get_func(&(info->poly));
      cpl_msg_debug(cpl_func, "poly%d final # %s -> %f %f %d %f", n, str, info->poly.c2t, info->poly.c2g, info->poly.c2c, info->poly.K);
      cpl_free(str);
    }
  mat_smooth_calc_result(info, first_smooth, last_smooth);
  return info->poly.c2t;
}

double mat_smooth_polyx_window_outlier(mat_smooth_info *info, mat_statistics_info *stat_info,
				       int first_data, int last_data,
				       int first_smooth, int last_smooth,
				       double enhance_factor)
{
  mat_poly_info   p1, p2, p3;
  double          p1c2l, p2c2l, p3c2l; // polynome specific chi2limit
  double          chi2limit;

  // try 1-order .. 3-order polynome and store the result
  p1c2l = mat_smooth_polyn_window_outlier(info, stat_info, 1, first_data, last_data, first_smooth, last_smooth, enhance_factor);
  p1    = info->poly;
  p2c2l = mat_smooth_polyn_window_outlier(info, stat_info, 2, first_data, last_data, first_smooth, last_smooth, enhance_factor);
  p2    = info->poly;
  p3c2l = mat_smooth_polyn_window_outlier(info, stat_info, 3, first_data, last_data, first_smooth, last_smooth, enhance_factor);
  p3    = info->poly;
  // check which polynome is a better fit (chi2 only) and restore this solution
  if ((p1.c2t < p2.c2t) && (p1.c2t < p3.c2t))
    { // poly-1 gives the best fit -> restore this solution
      info->poly = p1;
      chi2limit  = p1c2l;
    }
  else if ((p2.c2t < p1.c2t) && (p2.c2t < p3.c2t))
    { // poly-2 gives the best fit -> restore this solution
      info->poly = p2;
      chi2limit  = p2c2l;
    }
  else
    { // poly-3 gives the best fit -> restore this solution
      info->poly = p3;
      chi2limit  = p3c2l;
    }
  if (info->show_flag)
    {
      char *str = mat_poly_get_func(&(info->poly));
      cpl_msg_debug(cpl_func, "polyx final # %s -> %f %f %d %f", str, info->poly.c2t, info->poly.c2g, info->poly.c2c, info->poly.K);
      cpl_free(str);
    }
  mat_smooth_calc_result(info, first_smooth, last_smooth);
  return chi2limit;
}

void mat_smooth_polyn_sequence(mat_smooth_info *info, int n, int plength)
{
  int                   iData, nData, nWin;
  mat_statistics_info  *stat_info;

  nData = info->nused;
  stat_info = mat_statistics_new(nData, 1.5);
  if (stat_info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "cannot allocate memory for mat_statistics_info");
      return;
    }
  mat_statistics_set_use(stat_info, MAT_USE_REDUCED_MEAN, MAT_USE_REDUCED_VARIANCE);
  if (plength == 0)
    {
      nWin = nData;
    }
  else if (plength > nData)
    {
      nWin = nData;
    }
  else
    {
      nWin = plength;
    }
  stat_info->pflag = info->show_flag;
  for (iData = 0; iData < nData; iData++) info->smooth[iData] = 0.0;
  // two cases:
  //  1. the window covers all data -> global polynom fit
  //     -> One segment data = [0 .. nData-1] smooth = [0 .. nData-1]
  //  2. the window is smaller than the number of data elements -> local polynom fit, split all data points into 2+ segments
  //     -> First Segment data = [0 .. nWin-1] smooth = [0 .. nWin/2]
  //     -> Intermediate Segments iDat = [nWin/2+1 .. nData-nWin/2-2] data = [iDat-nWin/2 .. iDat+nWin/2] smooth = [iDat .. iDat]
  //     -> Last Segment data = [nData-nWin .. nData-1] smooth = [nData-nWin/2-1 .. nData-1]
  if (nWin == nData)
    {
      mat_smooth_polyn_window_chi2(info, stat_info, n, 0, nData-1, 0, nData-1);
    }
  else
    {
      mat_smooth_polyn_window_chi2(info, stat_info, n, 0, nWin-1, 0, nWin/2);
      for (iData = nWin/2+1; iData <= nData-nWin/2-2; iData++)
	{
	  mat_smooth_polyn_window_chi2(info, stat_info, n, iData-nWin/2, iData+nWin/2, iData, iData);
	}
      mat_smooth_polyn_window_chi2(info, stat_info, n, nData-nWin, nData-1, nData-nWin/2-1, nData-1);
    }
  mat_statistics_delete(stat_info);
}

void mat_smooth_polyn_sequence_outlier(mat_smooth_info *info, int n, int plength, double enhance_factor)
{
  int                   iData, nData, nWin;
  mat_statistics_info  *stat_info;

  nData = info->nused;
  stat_info = mat_statistics_new(nData, 1.5);
  if (stat_info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "cannot allocate memory for mat_statistics_info");
      return;
    }
  mat_statistics_set_use(stat_info, MAT_USE_REDUCED_MEAN, MAT_USE_REDUCED_VARIANCE);
  if (plength == 0)
    {
      nWin = nData;
    }
  else if (plength > nData)
    {
      nWin = nData;
    }
  else
    {
      nWin = plength;
    }
  stat_info->pflag = info->show_flag;
  for (iData = 0; iData < nData; iData++) info->smooth[iData] = 0.0;
  // two cases:
  //  1. the window covers all data -> global polynom fit
  //     -> One segment data = [0 .. nData-1] smooth = [0 .. nData-1]
  //  2. the window is smaller than the number of data elements -> local polynom fit, split all data points into 2+ segments
  //     -> First Segment data = [0 .. nWin-1] smooth = [0 .. nWin/2]
  //     -> Intermediate Segments iDat = [nWin/2+1 .. nData-nWin/2-2] data = [iDat-nWin/2 .. iDat+nWin/2] smooth = [iDat .. iDat]
  //     -> Last Segment data = [nData-nWin .. nData-1] smooth = [nData-nWin/2-1 .. nData-1]
  if (nWin == nData)
    {
      mat_smooth_polyn_window_outlier(info, stat_info, n, 0, nData-1, 0, nData-1, enhance_factor);
    }
  else
    {
      mat_smooth_polyn_window_outlier(info, stat_info, n, 0, nWin-1, 0, nWin/2, enhance_factor);
      for (iData = nWin/2+1; iData <= nData-nWin/2-2; iData++)
	{
	  mat_smooth_polyn_window_outlier(info, stat_info, n, iData-nWin/2, iData+nWin/2, iData, iData, enhance_factor);
	}
      mat_smooth_polyn_window_outlier(info, stat_info, n, nData-nWin, nData-1, nData-nWin/2-1, nData-1, enhance_factor);
    }
  mat_statistics_delete(stat_info);
}

/*----------------------------------------------------------------------------*/

void mat_extract_info_set_data(mat_extract_info *info, cpl_image *dst, cpl_image *src, cpl_image *range)
{
  info->original = NULL;
  info->dst   = dst;
  info->src   = src;
  info->range = range;
  info->poix  = -1;
  info->poiy  = -1;
}

void mat_extract_info_set_original(mat_extract_info *info, cpl_image *original)
{
  info->original = original;
}

void mat_extract_info_set_filter(mat_extract_info *info, int src_filter, int dst_filter)
{
  info->src_filter = src_filter;
  info->dst_filter = dst_filter;
}

void mat_extract_info_set_window(mat_extract_info *info, mat_detector *det, int wintype, int x, int nx, int y, int ny)
{
  if (wintype == MAT_DETECTOR_WINDOW_TYPE)
    { // detector -> one row, one column, cell covers the whole detector
      info->x0 = 1;
      info->nx = det->nx;
      info->nc = 1;
      info->y0 = 1;
      info->ny = det->ny;
      info->nr = 1;
    }
  else if (wintype == MAT_CHANNEL_WINDOW_TYPE)
    { // channel -> nr rows, nc columns, cell covers one detector channel
      info->x0 = 1;
      info->nx = det->channel_nx;
      info->nc = det->channel_ncolumns;
      info->y0 = 1;
      info->ny = det->channel_ny;
      info->nr = det->channel_nrows;
    }
  else
    { // direct -> cell covers the given area
      info->x0 = x;
      info->nx = nx;
      info->nc = 1;
      info->y0 = y;
      info->ny = ny;
      info->nr = 1;
    }
}

void mat_extract_polyn(mat_extract_info *einfo, int dir, int n, int plength)
{
  int                  cc, cr, x, y;
  mat_smooth_info     *sinfo;

  switch (dir)
    {
    case MAT_BOTH_DIRECTION:
      // force the use of an offset, 1-d function does not make sense, since no surface fitting is implemented
      n = 0;
      plength = 0;
      sinfo = mat_smooth_info_new(einfo->nx*einfo->ny, einfo->src_filter, einfo->dst_filter);
      break;
    case MAT_HORIZONTAL_DIRECTION:
      sinfo = mat_smooth_info_new(einfo->nx, einfo->src_filter, einfo->dst_filter);
      break;
    case MAT_VERTICAL_DIRECTION:
      sinfo = mat_smooth_info_new(einfo->ny, einfo->src_filter, einfo->dst_filter);
      break;
    default:
      return;
    }
  if (sinfo == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot allocate space for smoothing operation.");
      return;
    }
  for (cr = 0; cr < einfo->nr; cr++)
    { // iterate over all rows
      int y0 = einfo->y0 + cr*einfo->ny;
      for (cc = 0; cc < einfo->nc; cc++)
	{ // iterate over all columns
	  int x0 = einfo->x0 + cc*einfo->nx;
	  switch (dir)
	    {
	    case MAT_BOTH_DIRECTION: // process a whole cell at once
	      mat_smooth_set(sinfo, einfo->src, einfo->range, x0, y0, einfo->nx, einfo->ny);
	      mat_smooth_polyn_sequence(sinfo, n, 0);
	      mat_smooth_apply(sinfo, einfo->src, -1.0);
	      mat_smooth_apply(sinfo, einfo->dst,  1.0);
	      break;
	    case MAT_HORIZONTAL_DIRECTION: // process each pixel row individually
	      for (y = 0; y < einfo->ny; y++)
		{
		  sinfo->show_flag = ((y0 + y) == einfo->poiy);
		  mat_smooth_set(sinfo, einfo->src, einfo->range, x0, y0 + y, einfo->nx, 1);
		  mat_smooth_polyn_sequence(sinfo, n, plength);
		  mat_smooth_apply(sinfo, einfo->src, -1.0);
		  mat_smooth_apply(sinfo, einfo->dst,  1.0);
		}
	      break;
	    case MAT_VERTICAL_DIRECTION: // process each pixel column individually
	      for (x = 0; x < einfo->nx; x++)
		{
		  if ((plength != 0) && (x%16 == 0)) cpl_msg_debug(cpl_func, "extract poly%d x %d", n, x);
		  sinfo->show_flag = ((x0 + x) == einfo->poix);
		  mat_smooth_set(sinfo, einfo->src, einfo->range, x0 + x, y0, 1, einfo->ny);
		  mat_smooth_polyn_sequence(sinfo, n, plength);
		  mat_smooth_apply(sinfo, einfo->src, -1.0);
		  mat_smooth_apply(sinfo, einfo->dst,  1.0);
		}
	    }
	}
    }
  mat_smooth_info_delete(sinfo);
}



/*----------------------------------------------------------------------------*/

/**
   @brief Calculates a robust median and variance by detecting and ignoring outliers.
   @param raw           This mat_gendata data structure contains a set of raw frames. 
   @param median        This mat_frame data structure will contain the median intensiy for each pixel.
   @param variance      This mat_frame data structure will contain the temporal variance for each pixel.
   @param cosmics       This flag determines if outliers should be ignored.
   @return cpl_error_code

   Since the pixel statistics, including the temporal variance, is used for the static
   bad pixel map (detect bad pixels), the flatfield map (calculate the conversion factor),
   and the detector monitoring (calculate important QC1 parameters), a contamination by
   cosmic rays (outliers) should be avoided. These cosmic rays influence the temporal variance,
   lead to extreme values, and therefore distort the detector characteristics.
*/
cpl_error_code mat_calc_statistics_cosmics(mat_gendata *raw,
					   mat_frame *median, 
					   mat_frame *variance,
					   int cosmics)
{
  int         r;
  int         nz;
  int         x, y, z;
  mat_statistics_info  *info;

  if (raw == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_gendata (raw) argument given");
    }
  if (raw->imgdata == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no valid mat_gendata (raw->imgdata) argument given, IMAGING_DATA part missing");
    }
  if (raw->imgdata->list_frame == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no valid mat_gendata (raw->imgdata->list_frame) argument given, IMAGING_DATA part empty");
    }
  if (raw->imgdata->list_frame[0] == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no valid mat_gendata (raw->imgdata->list_frame[0]) argument given, IMAGING_DATA part broken");
    }
  nz = raw->imgdata->nbframe;
  if (nz < 10)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "no valid mat_gendata (raw->imgdata) argument given, less than 10 raw frames");
    }
  if (median == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (median) argument given");
    }
  if (variance == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (variance) argument given");
    }
  if (cosmics)
    {
      info = mat_statistics_new(nz, 1.5);
    }
  else
    {
      info = mat_statistics_new(nz, 0.0);
    }
  if (info == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "cannot allocate memory for mat_statistics_info");
    }
  for (r = 0; r < median->nbsubwin; r++)
    {
      cpl_image     *image = median->list_subwin[r]->imgreg[0];
      int            nx    = cpl_image_get_size_x(image);
      int            ny    = cpl_image_get_size_y(image);
      cpl_imagelist *list  = cpl_imagelist_new();
      for (z = 0; z < nz; z++)
	{
	  cpl_imagelist_set(list, raw->imgdata->list_frame[z]->list_subwin[r]->imgreg[0], cpl_imagelist_get_size(list));
	}
      for (y = 1; y <= ny; y++)
	{
	  if ((y - 1)%16 == 0)
	    {
	      //cpl_msg_debug(cpl_func, "processing pixel line %d", y - 1);
	    }
	  for (x = 1; x <= nx; x++)
	    {
	      mat_statistics_set_sequence(info, list, NULL, x, y);
	      mat_statistics_calc(info);
	      /*
		if ((x == 1313) && (y == 1111))
		cpl_msg_debug(cpl_func, "statistics limits: %.1f %.1f whole: %d %.1f %.1f %.1f %.1f reduced: %d %.1f %.1f %.1f %.1f result: %.3f %.3f",
		info->limit_min, info->limit_max,
		info->whole_count,
		info->whole_median, info->whole_medvar,
		info->whole_mean, info->whole_variance,
		info->reduced_count,
		info->reduced_median, info->reduced_medvar,
		info->reduced_mean, info->reduced_variance,
		info->average, info->variance);
	      */
	      /*
		if ((x == 13) && (y == 13))
		{
		cpl_msg_debug(cpl_func, "  pixel(13,13): ovalue[13]=%f, intensity=%f, variance=%d",
		ovalues[13],
		intensity_median,
		variance_full);
		}
	      */
	      mat_frame_set(median, r, x, y, info->average);
	      mat_frame_set(variance, r, x, y, info->variance);
	      /*
		if ((x%13 == 0) && (y%7 == 0))
		cpl_msg_debug(cpl_func, "stat(%d,%d) %.1f %.1f (%d)", x, y, info->average, info->variance, info->reduced_count);
	      */
	    }
	}
      cpl_imagelist_unwrap(list);
    }
  mat_statistics_delete(info);
  return CPL_ERROR_NONE;
}

/**
   @brief This function calculates the pixel gain for each pixel inside a given region of an image.
   @param median    This image contains the median intensity for each pixel inside a sub-window on the detector.
   @param variance  This image contains the temporal variance for each pixel inside a sub-window on the detector.
   @param irange       This image contains the intensity range classification for each pixel.
   @param bpm       This is the optional bad pixel mask for the pixels.
   @param rx        This is the x-coordinate of the left side of the given region.
   @param ry        This is the y-coordinate of the bottom side of the given region.
   @param rnx       This is width of the region.
   @param rny       This is height of the region.
   @param gain      This floating point value will contain the median detector specific conversion factor in electrons per DU for the given region.
   @return cpl_error_code

   This function calculates the pixel gain for each pixel inside a given region of an image.
   The median of these pixel gains (without using the bad pixels specified by the optional
   bad pixel map) is returned. Only pixels which show an illumination level above a certain 
   limit are used to calculate the gain. This means that for the detector channels covered
   with a metal frame no conversion factor (gain) can be calculated.

   The median image is the temporal intensity median calculated from a series of flatfield
   images subtracted by the temporal intensity median calculated from a series of cold
   dark images. The same schema is used for the temporal variance. This means that ideally
   the input follows the photon statistics. Since the illumination of both detectors
   will be inhomogeneous, a simple threshold  will be used to filter (and use) only the
   bright pixels.

   Mathematical description:

   <pre>
   BEGIN
   Create a vector for the gain and set the count to 0;
   IF a bad pixel map is specified THEN
   Use the bad pixel mask for the median and the variance;
   ENDIF
   FOR EACH pixel inside the region DO
   IF the pixel is marked as bad THEN
   CONTINUE;
   ENDIF
   IF enough light on that pixel THEN
   gain := median[pixel]/variance[pixel];
   Add gain to the vector of gains;
   ENDIF
   ENDFOR
   IF a bad pixel map is specified THEN
   Remove the bad pixel mask from the median and the variance;
   ENDFOR
   Select the median value in the vector for the detector as global gain;
   END
   </pre>
*/
cpl_error_code mat_calc_gain_region(cpl_image *median,
				    cpl_image *variance,
				    cpl_image *irange,
				    cpl_mask *bpm,
				    int rx, int ry, int rnx, int rny,
				    double *gain,
				    int pflag)
{
  int                   x, y;
  mat_statistics_info  *info;

  if (median == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no cpl_image (median) argument given");
    }
  if (variance == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no cpl_image (variance) argument given");
    }
  if (gain == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no double* (gain) argument given");
    }
  info = mat_statistics_new(rnx*rny, 1.5);
  if (info == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "cannot allocate memory for mat_statistics_info");
    }
  mat_statistics_reset(info);
  for (x = rx; x < rx + rnx; x++)
    {
      for (y = ry; y < ry + rny; y++)
	{
	  double m, v;
	  int    rejected;
	  if ((bpm != NULL) && (cpl_mask_get(bpm, x, y) == CPL_BINARY_1)) continue; /* ignore bad pixels */
	  m = cpl_image_get(median, x, y, &rejected);
	  if (rejected) continue; /* skipped by badpixel map */
	  v = cpl_image_get(variance, x, y, &rejected);
	  if (rejected) continue; /* skipped by badpixel map */
	  if (v == 0.0) continue; /* skip pixel without valid variance */
	  if (irange != NULL)
	    {
	      if (cpl_image_get(irange, x, y, &rejected) != MAT_LINEAR_INTENSITY) continue;
	    }
	  else
	    {
	      if (m < MAT_MIN_GAIN_INTENSITY) continue;
	    }
	  /*
	    if ((x%13 == 0) && (y%7 == 0))
	    cpl_msg_debug(cpl_func, "stat(%d,%d) %.1f %.1f", x, y, m, v);
	  */
	  mat_statistics_add_value(info, m/v);
	}
    }
  if (info->whole_count < 10)
    {
      cpl_msg_debug(cpl_func, "could not calculate the detector region gain for less than 10 pixels");
      *gain = 1.0;
    }
  else
    {
      if (pflag)
	{
	  info->pflag = 1;
	  strncpy(info->prefix, "# det gain ", sizeof(info->prefix));
	}
      mat_statistics_calc(info);
      *gain = info->average;
      //*gain = info->whole_mode;
      /*
	cpl_msg_debug(cpl_func, "statistics limits: %.3f %.3f whole: %d %.3f %.3f %.3f %.3f reduced: %d %.3f %.3f %.3f %.3f",
	info->limit_min, info->limit_max,
	info->whole_count,
	info->whole_median, info->whole_medvar,
	info->whole_mean, info->whole_variance,
	info->reduced_count,
	info->reduced_median, info->reduced_medvar,
	info->reduced_mean, info->reduced_variance);
      */
      if (pflag)
	{
	  cpl_msg_debug(cpl_func, "# det gain # gain %lf electrons/DU (%d pixels used, peak %.4lf median %.4lf mean %.4lf)",
			*gain, info->reduced_count, info->whole_mode, info->reduced_median, info->reduced_mean);
	}
    }
  mat_statistics_delete(info);
  return CPL_ERROR_NONE;
}

/**
   @brief Calculates the conversion factor in electrons per DU.
   @param median       This image contains the intensity median for each pixel.
   @param variance     This image contains the temporal variance for each pixel.
   @param irange       This image contains the intensity range classification for each pixel.
   @param bpm          The optional badpixel mask.
   @param nchancols    The number of detector channel columns.
   @param nchanrows    The number of detector channel rows.
   @param detgain      The global (average) detector gain in electrons per DU.
   @param channelgain  Each detector channel will have its individual gain value.
   @return the cpl_error_code or CPL_ERROR_NONE

   This function calculates the conversion factor (gain) for the whole detector and
   for each detector channel.

   Mathematical description:

   BEGIN
   IF the detector gain is requested THEN
   CALL mat_calc_gain_region for the whole detector;
   ENDIF
   FOR EACH detector channel row DO
   FOR EACH detector channel column DO
   CALL mat_calc_gain_region for that detector channel;
   ENDFOR
   ENDFOR
   END

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
   - CPL_ERROR_UNSPECIFIED if the data cannot be used to calculate the global getector gain
*/
cpl_error_code mat_calc_gain(cpl_image *median,
			     cpl_image *variance,
			     cpl_image *irange,
			     cpl_mask *bpm,
			     int nchancols,
			     int nchanrows,
			     double *detgain,
			     cpl_vector *channelgain)
{
  int        nx, ny;
  int        dcnx, dcny;
  double     gain;

  if (median == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no cpl_image (median) argument given");
    }
  if (variance == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no cpl_image (variance) argument given");
    }
  nx = cpl_image_get_size_x(median);
  ny = cpl_image_get_size_y(median);
  dcnx = nx/nchancols;  /* detector channel width  */
  dcny = ny/nchanrows;  /* detector channel height */
  if (detgain != NULL)
    {
      mat_calc_gain_region(median, variance, irange, bpm, 1, 1, nx, ny, detgain, 1);
    }
  if (channelgain != NULL)
    {
      int dcr;
      cpl_vector_set_size(channelgain, nchanrows*nchancols);
      for (dcr = 0; dcr < nchanrows; dcr++)
	{
	  int dcy = dcr*dcny + 1;
	  int dcc;
	  for (dcc = 0; dcc < nchancols; dcc++)
	    {
	      int dcx = dcc*dcnx + 1;
	      int dci = dcr*nchancols + dcc;
	      mat_calc_gain_region(median, variance, irange, bpm, dcx, dcy, dcnx, dcny, &gain, 0);
	      cpl_vector_set(channelgain, dci, gain);
	    }
	}
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Calculates the spatial median and optional deviation for all detector channels on an image.
   @param data         This image contains the data.
   @param bpm          This is the optional bad pixel mask for the pixels.
   @param nchancols    The number of detector channel columns.
   @param nchanrows    The number of detector channel rows.
   @param list         This list will contain the extracted regions (one for each detector channel).
   @param median       The spatial median is stored in this parameter.
   @param dev          The optional spatial deviation is stored in this parameter.

   This function extracts from the image the region representing a detector channel and calculates
   the median and deviation for such a region. If a badpixel mask is given, these bad pixels are
   ignored. If the image has already a mask, it is ignored and restored at the end of the function.

   Mathematical description:

   <pre>
   BEGIN
   FOR EACH detector channel row DO
   FOR EACH detector channel column DO
   IF list != NULL THEN
   Extract the detector channel specific region from the image;
   Add the region to the list;
   ENDIF
   IF dev != NULL THEN
   CALL cpl_image_get_median_dev_window on that region;
   ELSE
   CALL cpl_image_get_median_window on that region;
   ENDIF
   ENDFOR
   ENDFOR
   END
   </pre>
*/
cpl_error_code mat_calc_median_dev(cpl_image *data,
				   cpl_mask *bpm,
				   int nchancols,
				   int nchanrows,
				   cpl_imagelist *list,
				   cpl_vector *median,
				   cpl_vector *dev)
{
  int                   nx, ny;
  int                   dcnx, dcny;
  int                   dcr, dcc, dcx, dcy, dci;
  mat_statistics_info  *info;
  
  if (data == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no cpl_image (data) argument given");
    }
  if (median == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no cpl_vector (median) argument given");
    }
  nx = cpl_image_get_size_x(data);
  ny = cpl_image_get_size_y(data);
  dcnx = nx/nchancols;    /* detector channel width  */
  dcny = ny/nchanrows;    /* detector channel height */
  info = mat_statistics_new(dcnx*dcny, 1.5);
  if (info == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "cannot allocate memory for mat_statistics_info");
    }
  /* calculate the offset for each detector channel */
  cpl_vector_set_size(median, nchanrows*nchancols);
  if (dev != NULL)
    {
      cpl_vector_set_size(dev, nchanrows*nchancols);
    }
  for (dcr = 0; dcr < nchanrows; dcr++)
    {
      dcy = dcr*dcny + 1;
      for (dcc = 0; dcc < nchancols; dcc++)
	{
	  dcx = dcc*dcnx + 1;
	  dci = dcr*nchancols + dcc;
	  if (list != NULL)
	    {
	      cpl_imagelist_set(list,
				cpl_image_extract(data, dcx, dcy, dcx + dcnx - 1, dcy + dcny - 1),
				cpl_imagelist_get_size(list));
	    }
	  mat_statistics_set_region(info, data, bpm, dcx, dcy, dcnx, dcny);
	  mat_statistics_calc(info);
	  cpl_vector_set(median, dci, info->average);
	  if (dev != NULL)
	    {
	      cpl_vector_set(dev, dci, sqrt(info->variance));
	    }
	}
    }
  mat_statistics_delete(info);
  return CPL_ERROR_NONE;
}

/**
   @brief Calculates the offset for all detector channels on an image.
   @param median         This image contains the intensity median.
   @param bpm            This is the optional bad pixel mask for the pixels.
   @param nchancols      The number of detector channel columns.
   @param nchanrows      The number of detector channel rows.
   @param channeloffset  This vector will contain all offsets.

   This function calculates for each detector channel (a specific region in the median image) the
   offset. If a badpixel mask is given, these bad pixels are ignored. If the image has already a mask,
   it is ignored and restored at the end of the function.

   Mathematical description:

   <pre>
   BEGIN
   FOR EACH detector channel row DO
   FOR EACH detector channel column DO
   CALL mat_calc_offset_region;
   ENDFOR
   ENDFOR
   END
   </pre>
*/
cpl_error_code mat_calc_offset(cpl_image *median,
			       cpl_mask *bpm,
			       int nchancols,
			       int nchanrows,
			       cpl_vector *channeloffset)
{
  int                   nx, ny;
  int                   dcnx, dcny;
  int                   dcr, dcc, dcx, dcy, dci;
  double                offset;
  mat_statistics_info  *info;
  
  if (median == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no cpl_image (median) argument given");
    }
  if (channeloffset == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no cpl_vector (channeloffset) argument given");
    }
  nx = cpl_image_get_size_x(median);
  ny = cpl_image_get_size_y(median);
  dcnx = nx/nchancols;    /* detector channel width  */
  dcny = ny/nchanrows;    /* detector channel height */
  info = mat_statistics_new(dcnx*dcny, 1.5); // ignore outlier
  if (info == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "cannot allocate memory for mat_statistics_info");
    }
  /* calculate the offset for each detector channel */
  cpl_vector_set_size(channeloffset, nchanrows*nchancols);
  for (dcr = 0; dcr < nchanrows; dcr++)
    {
      dcy = dcr*dcny + 1;
      for (dcc = 0; dcc < nchancols; dcc++)
	{
	  dcx = dcc*dcnx + 1;
	  dci = dcr*nchancols + dcc;
	  mat_statistics_set_region(info, median, bpm, dcx, dcy, dcnx, dcny);
	  mat_statistics_calc(info);
	  offset = info->average;
	  cpl_vector_set(channeloffset, dci, offset);
	  cpl_msg_debug(cpl_func, "offset[%d,%d] (%dx%d at %d,%d) = %.2f DU",
			dcc, dcr,
			dcnx, dcny, dcx, dcy,
			offset);
	}
    }
  mat_statistics_delete(info);
  return CPL_ERROR_NONE;
}

/**
   @brief Calculates the readout noise for a detector.
   @param variance      The variance is used for the noise calculation.
   @param bpm           The optional badpixel mask.
   @param nchancols     The number of detector channel columns.
   @param nchanrows     The number of detector channel rows.
   @param detgain       The global (average) detector gain in electrons per DU.
   @param detnoise      The global detector readout noise.
   @param channelgain   Each detector channel has its individual gain value.
   @param channelnoise  The readout noise for each detector channel.
   @return the cpl_error_code or CPL_ERROR_NONE

   This function calculates the readout noise for the whole detector and each individual
   detector channel. It is possible to use NULL for some arguments if the corresponding values
   are not requested.

   If the parameters detnoise is given, the global detector noise is calculated and
   stored in detnoise.

   If the parameters channelgain and channelnoise are given, the readout noise is
   calculated for each deector channel and stored in channelnoise.

   Mathematical description:

   <pre>
   BEGIN
   IF detnoise != NULL THEN
   CALL mat_calc_noise_region for the whole detector;
   ENDIF
   FOR EACH detector channel row DO
   FOR EACH detector channel column DO
   CALL mat_calc_noise_region for that detector channel;
   ENDFOR
   ENDFOR
   END
   </pre>

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
   - CPL_ERROR_INCOMPATIBLE_INPUT if the data does not span the whole detector width
*/

cpl_error_code mat_calc_noise(cpl_image *variance,
			      cpl_mask *bpm,
			      int nchancols,
			      int nchanrows,
			      double detgain,
			      double *detnoise,
			      cpl_vector *channelgain,
			      cpl_vector *channelnoise,
			      double dsgain)
{
  int                   nx, ny;
  int                   dcnx, dcny;
  double                noise, dsnoise, dunoise;
  mat_statistics_info  *info;
  
  if (variance == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no cpl_image (variance) argument given");
    }
  nx = cpl_image_get_size_x(variance);
  ny = cpl_image_get_size_y(variance);
  dcnx = nx/nchancols;    /* detector channel width  */
  dcny = ny/nchanrows;    /* detector channel height */
  info = mat_statistics_new(nx*ny, 1.5); // ignore outlier
  if (info == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "cannot allocate memory for mat_statistics_info");
    }
  /* calculate the noise for the whole detector */
  if (detnoise != NULL)
    {
      mat_statistics_set_region(info, variance, bpm, 1, 1, nx, ny);
      //info->pflag = 1;
      strncpy(info->prefix, "# det noise ", sizeof(info->prefix));
      mat_statistics_calc(info);
      //dunoise = sqrt(info->average);
      dunoise = sqrt(info->whole_mode);
      noise   = detgain*dunoise;
      dsnoise = dsgain*dunoise;
      *detnoise = noise;
      cpl_msg_debug(cpl_func, "# det noise # noise %.2f DU %.2f e- %.2f e- (ds) peak %.2f DU median %.2f DU mean %.2f DU",
		    dunoise, noise, dsnoise,
		    sqrt(info->whole_mode), sqrt(info->reduced_median), sqrt(info->reduced_mean));
      info->pflag = 0;
    }
  /* calculate the noise for each detector channel */
  if ((channelgain != NULL) && (channelnoise != NULL))
    {
      int dcr;
      cpl_vector_set_size(channelnoise, nchanrows*nchancols);
      for (dcr = 0; dcr < nchanrows; dcr++)
	{
	  int dcy = dcr*dcny + 1;
	  int dcc;
	  for (dcc = 0; dcc < nchancols; dcc++)
	    {
	      int dcx = dcc*dcnx + 1;
	      int dci = dcr*nchancols + dcc;
	      mat_statistics_set_region(info, variance, bpm, dcx, dcy, dcnx, dcny);
	      mat_statistics_calc(info);
	      dunoise = sqrt(info->average);
	      //dunoise = sqrt(info->whole_mode);
	      noise   = cpl_vector_get(channelgain, dci)*dunoise;
	      dsnoise = dsgain*dunoise;
	      cpl_vector_set(channelnoise, dci, noise);
	      cpl_msg_debug(cpl_func, "noise[%d,%d] (%dx%d at %d,%d) = %.2f DU %.2f e- %.2f e- (ds) peak %.2f DU median %.2f DU mean %.2f DU",
			    dcc, dcr,
			    dcnx, dcny, dcx, dcy,
			    dunoise, noise, dsnoise,
			    sqrt(info->whole_mode), sqrt(info->reduced_median), sqrt(info->reduced_mean));
	    }
	}
    }
  mat_statistics_delete(info);
  return CPL_ERROR_NONE;
}

static int binsizes[] = {
  1, 2, 3, 4, 5, 6, 7, 8, 9,
  10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60, 65, 70, 75, 80, 85, 90, 95,
  100, 150, 200, 250, 300, 350, 400, 450, 500, 550, 600, 650, 700, 750, 800, 850, 900, 950,
  1000, 1500, 2000, 2500, 3000, 3500, 4000, 4500, 5000, 5500, 6000, 6500, 7000, 7500, 8000, 8500, 9000, 9500,
  10000,
  0
};

/**
   @brief Calculates the Allan variance from an imagelist.
   @param frametime    The time between two consecutive frames (this is not DIT!).
   @param darks        This imagelist contains a list of cold dark frames.
   @param bpm          This is the optional badpixel map.

   This function calculates the average Allan variance using all good pixels of a sequence of
   cold dark frames. In order to save time, only a limited set of bin sizes is used. The time where
   the Allan variance has the lowest value is returned by this function and stored as QC1 parameter.

   Mathematical description:

   <pre>
   BEGIN
   Allocate an image which represtents the previous bin (pixel based);
   Allocate an image which represents the current bin (pixel based);
   IF a padpixel map is specified THEN
   Set the bad pixel masks for each images;
   ENDIF
   FOR EACH bin size from a list of sizes DO
   FOR EACH bin DO
   Calculate the average image of the current bin;
   IF It is not the first bin THEN
   Calculate the squared difference of the previous and current image;
   ENDIF
   Copy the current bin into the previous bin;
   ENDFOR
   Normalize the Allan variance;
   IF the current Allan variance is minimal THEN
   Store the current Allan variance as return value;
   ENDIF
   ENDFOR
   Free both allocated images;
   END
   </pre>
*/
double mat_calc_avar(double frametime,
		     cpl_imagelist *darks,
		     cpl_mask *bpm)
{
  cpl_image   *img  = NULL;
  cpl_image   *last = NULL;
  cpl_image   *curr = NULL;
  int          nx, ny, nz;
  int          i, k;
  double       avartime = 0.0;
  double       avar0    = 0.0;
  double       avarmin  = 0.0;

  img = cpl_imagelist_get(darks, 0);
  nx = cpl_image_get_size_x(img);
  ny = cpl_image_get_size_y(img);
  nz = cpl_imagelist_get_size(darks);
  /* allocate two images which represents the last and the current bin (pixel based) */
  last = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
  if (last == NULL)
    {
      cpl_error_code ecode = cpl_error_get_code();
      cpl_msg_error(cpl_func, "could not allocate the image for the last bin, code = %d, message = %s",
		    ecode,
		    cpl_error_get_message());
      return ecode;
    }
  curr = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
  if (curr == NULL)
    {
      cpl_error_code ecode = cpl_error_get_code();
      cpl_msg_error(cpl_func, "could not allocate the image for the current bin, code = %d, message = %s",
		    ecode,
		    cpl_error_get_message());
      cpl_image_delete(last);
      return ecode;
    }
  if (bpm != NULL)
    {
      cpl_image_reject_from_mask(last, bpm);
      cpl_image_reject_from_mask(curr, bpm);
    }
  /* iterate over all possible bin sizes */
  cpl_msg_debug(cpl_func, "Allan Variance:");
  for (i = 0; (binsizes[i] != 0) && (binsizes[i] < nz/7); i++)
    {
      int    binsize = binsizes[i];
      double avar  = 0.0;
      double count = 0.0;
      int    idx   = 0;
      int    nr;
      for (nr = 0; nr < nz/binsize; nr++)
	{
	  cpl_image_multiply_scalar(curr, 0.0);
	  for (k = 0; k < binsize; k++) {
	    cpl_image_add(curr, cpl_imagelist_get(darks, idx));
	    idx++;
	  }
	  cpl_image_divide_scalar(curr, (double)binsize);
	  if (nr != 0) {
	    cpl_image_subtract(last, curr);
	    cpl_image_multiply(last, last);
	    avar += cpl_image_get_flux(last);
	    count += (double)(nx*ny);
	  }
	  cpl_image_copy(last, curr, 1, 1);
	}
      avar /= 2.0*count;
      cpl_msg_debug(cpl_func, "   %lf %lf", frametime*(double)binsize, avar);
      if (i == 0)
	{
	  avar0 = avar;
	  avartime = frametime*(double)binsize;
	  avarmin = avar;
	}
      else if (avar < avarmin)
	{
	  avartime = frametime*(double)binsize;
	  avarmin = avar;
	}
    }
  if (bpm != NULL)
    {
      cpl_mask_delete(cpl_image_unset_bpm(last));
      cpl_mask_delete(cpl_image_unset_bpm(curr));
    }
  cpl_msg_debug(cpl_func, "Allan Variance: avar(%.3f) = %.3f, avar(%.3f) = %.3f",
	       frametime, avar0,
	       avartime, avarmin);
  cpl_image_delete(last);
  cpl_image_delete(curr);
  return avartime;
}

/**
   @brief Calculates the average horizontal and vertival correlation between pixels.
   @param median           This image contains the intensity median.
   @param flats            This list contains the list of flatfield images.
   @param bpm              This is the optional badpixel map.
   @param rx               This is the x-coordinate of the left side of the given region.
   @param ry               This is the y-coordinate of the bottom side of the given region.
   @param rnx              This is width of the region.
   @param rny              This is height of the region.
   @param acorrhorizontal  This parameter will contain the average horizontal pixelcorrelation.
   @param acorrvertical    This parameter will contain the average vertical pixelcorrelation.

   The pixel correlation is calculated using a FFT based algorithm. The returned values are
   an average calculated by using all images in the flatfield frame list.

   Mathematical description:

   <pre>
   BEGIN
   Create temporary images for the FFT;
   FOR EACH flatfield image DO
   IF a badpixel map is specified THEN
   Interpolate all bad pixels in the current image;
   ENDIF
   Extract the median compensated region;
   Calculate the autocorrelation using an FFT based algorithm;
   Add the horizontal and vertical correlation to the sum;
   ENDFOR
   Calculate the average of both correlations;
   Free the temporary images;
   END
   </pre>
*/
cpl_error_code mat_calc_autocorr_region(cpl_image *median,
					cpl_imagelist *flats,
					cpl_mask *bpm,
					int rx, int ry, int rnx, int rny,
					double *acorrhorizontal,
					double *acorrvertical)
{
  cpl_image *img   = NULL;
  int        nz;
  int        x, y, z;
  int        rejected;
  double     corr0 = 0.0;
  double     corrh = 0.0;
  double     corrv = 0.0;
  double     corro = 0.0;
  cpl_image *img1  = NULL;
  cpl_image *img2  = NULL;
  cpl_image *acimg = NULL;
  mat_statistics_info   *acinfo;

  //cpl_msg_debug(cpl_func, "mat_calc_autocorr_region(.., %d, %d, %d, %d, .., ..)", rx, ry, rnx, rny);
  *acorrhorizontal = 0.0;
  *acorrvertical   = 0.0;
  if (cpl_image_get_type(median) != CPL_TYPE_DOUBLE)
    {
      cpl_msg_error(cpl_func, "the median image is not a double precision image (CPL_TYPE_DOUBLE)");
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  img = cpl_imagelist_get(flats, 0);
  if (img == NULL)
    {
      cpl_msg_error(cpl_func, "cannot get the first image from the list of flats");
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (cpl_image_get_type(img) != CPL_TYPE_FLOAT)
    {
      cpl_msg_error(cpl_func, "the list of flats does not contain floating point images (CPL_TYPE_FLOAT)");
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  nz = cpl_imagelist_get_size(flats);
  img1 = cpl_image_new(rnx, rny, CPL_TYPE_DOUBLE_COMPLEX);
  if (img1 == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate a temporary double complex image of %dx%d pixels", rnx, rny);
      return CPL_ERROR_UNSPECIFIED;
    }
  img2 = cpl_image_new(rnx, rny, CPL_TYPE_DOUBLE_COMPLEX);
  if (img2 == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate a temporary double complex image of %dx%d pixels", rnx, rny);
      cpl_image_delete(img1);
      return CPL_ERROR_UNSPECIFIED;
    }
  acimg = cpl_image_new(rnx, rny, CPL_TYPE_DOUBLE);
  if (acimg == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate a temporary double image of %dx%d pixels", rnx, rny);
      cpl_image_delete(img1);
      cpl_image_delete(img2);
      return CPL_ERROR_UNSPECIFIED;
    }
  acinfo = mat_statistics_new(rnx*rny, 1.5);
  if (acinfo == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate a memory for mat_statistics_info (%dx%d pixels)", rnx, rny);
      cpl_image_delete(img1);
      cpl_image_delete(img2);
      cpl_image_delete(acimg);
      return CPL_ERROR_UNSPECIFIED;
    }
  mat_image_fill(acimg, 0.0);
  for (z = 0; z < nz; z++)
    {
      img = cpl_imagelist_get(flats, z);
      if (img == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot get image %d from the list of flats", z);
	  continue;
	}
      /* if the badpixel map is given, interpolate all bad pixels */
      if (bpm != NULL)
	{
	  cpl_image_reject_from_mask(img, bpm);
	  cpl_detector_interpolate_rejected(img);
	}
      /* calculate the autocorrelation */
      for (y = 0; y < rny; y++)
	{
	  for (x = 0; x < rnx; x++)
	    {
	      double v0 = cpl_image_get(img, rx + x, ry + y, &rejected) - cpl_image_get(median, rx + x, ry + y, &rejected);
	      double complex intensity = v0 + 0.0i;
	      // copy the intensity into the complex image
	      cpl_image_set_complex(img1, x + 1, y + 1, intensity);
	      /*
		if ((rx == 457) && (ry == 41) && (x == 0) && (y == 0))
		{
		cpl_msg_debug(cpl_func, "noise[%d] = %g", z, v0);
		}
	      */
	    }
	}
      cpl_fft_image(img2, img1, CPL_FFT_FORWARD);
      cpl_image_conjugate(img1, img2);
      cpl_image_multiply(img2, img1);
      cpl_fft_image(img1, img2, CPL_FFT_BACKWARD);
      for (y = 1; y <= rny; y++)
	for (x = 1; x <= rnx; x++)
	  {
	    double ac = creal(cpl_image_get_complex(img1, x, y, &rejected));
	    cpl_image_set(acimg, x, y, cpl_image_get(acimg, x, y, &rejected) + ac);
	  }
    }
  cpl_image_divide_scalar(acimg, (double)nz);
  /*
    if ((rx == 457) && (ry == 41))
    {
    for (y = rny; y != 0; y--)
    {
    cpl_msg_debug(cpl_func, "acimg[1-16,%d] = (%.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f)",
    y,
    cpl_image_get(acimg,  1, y, &rejected),
    cpl_image_get(acimg,  2, y, &rejected),
    cpl_image_get(acimg,  3, y, &rejected),
    cpl_image_get(acimg,  4, y, &rejected),
    cpl_image_get(acimg,  5, y, &rejected),
    cpl_image_get(acimg,  6, y, &rejected),
    cpl_image_get(acimg,  7, y, &rejected),
    cpl_image_get(acimg,  8, y, &rejected),
    cpl_image_get(acimg,  9, y, &rejected),
    cpl_image_get(acimg, 10, y, &rejected),
    cpl_image_get(acimg, 11, y, &rejected),
    cpl_image_get(acimg, 12, y, &rejected),
    cpl_image_get(acimg, 13, y, &rejected),
    cpl_image_get(acimg, 14, y, &rejected),
    cpl_image_get(acimg, 15, y, &rejected),
    cpl_image_get(acimg, 16, y, &rejected));
    }
    }
  */
  mat_statistics_reset(acinfo);
  for (y = -2; y <= 2; y++) //rny; y++)
    for (x = -2; x <= 2; x++) //rnx; x++)
      mat_statistics_add_value(acinfo, cpl_image_get(acimg, rnx/2 + x, rny/2 + y, &rejected));
  mat_statistics_calc(acinfo);
  corro = acinfo->average;
  corr0 = cpl_image_get(acimg, 1, 1, &rejected);
  corrh = cpl_image_get(acimg, 2, 1, &rejected);
  corrv = cpl_image_get(acimg, 1, 2, &rejected);
  /*
    cpl_msg_debug(cpl_func, "corr: corr0 %.3f corrh %.3f corrv %.3f corro %.3f h %.4f v %.4f",
    corr0, corrh, corrv, corro,
    (corrh - corro)/(corr0 - corro),
    (corrv - corro)/(corr0 - corro));
  */
  *acorrhorizontal = (corrh - corro)/(corr0 - corro);
  *acorrvertical   = (corrv - corro)/(corr0 - corro);
  //*acorrhorizontal = corrh/corr0;
  //*acorrvertical   = corrv/corr0;
  cpl_image_delete(img1);
  cpl_image_delete(img2);
  cpl_image_delete(acimg);
  mat_statistics_delete(acinfo);
  return CPL_ERROR_NONE;
}

/**
   @brief Calculates the average pixel correlation for all detector channels.
   @param median           This image contains the intensity median.
   @param flats            This list contains the list of flatfield images.
   @param bpm              This is the optional badpixel map.
   @param nchancols        The number of detector channel columns.
   @param nchanrows        The number of detector channel rows.
   @param acorrhorizontal  This parameter will contain the average horizontal pixelcorrelation.
   @param acorrvertical    This parameter will contain the average vertical pixelcorrelation.
   @param dahinfo          This parameter ...
   @param davinfo          this parameter

   This function calculates for each detector channel the pixel correlation and averages
   them.

   Mathematical description:

   <pre>
   BEGIN
   IF a badpixel map is specified THEN
   FOR EACH flatfield image DO
   Interpolate all bad pixels in the current image;
   ENDFOR
   ENDIF
   FOR EACH detector channel row DO
   FOR EACH detector channel column DO
   Determine the central 32x32 pixel area;
   CALL mat_calc_autocorr_region for that area;
   Sum up the correlation;
   ENDFOR
   ENDFOR
   Calculate the average correlation;
   END
   </pre>
*/
cpl_error_code mat_calc_autocorr(cpl_image *median,
				 cpl_imagelist *flats,
				 cpl_mask *bpm,
				 int nchancols,
				 int nchanrows,
				 double *acorrhorizontal,
				 double *acorrvertical,
				 mat_statistics_info *dahinfo,
				 mat_statistics_info *davinfo
				 )
{
  int                    nx, ny;
  int                    dcc, dcr; /* detector channel column and row */
  int                    dcx, dcy, dcnx, dcny;
  int                    rx, ry, rnx, rny;
  mat_statistics_info   *ahinfo;
  mat_statistics_info   *avinfo;

  nx = cpl_image_get_size_x(median);
  ny = cpl_image_get_size_y(median);
  dcnx = nx/nchancols;    /* detector channel width  */
  dcny = ny/nchanrows;    /* detector channel height */
  *acorrhorizontal = 0.0;
  *acorrvertical   = 0.0;
  ahinfo = mat_statistics_new(nchancols*nchanrows, 1.5);
  if (ahinfo == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate a memory for mat_statistics_info (%d horizontal correlations)", nchancols*nchanrows);
      return CPL_ERROR_UNSPECIFIED;
    }
  avinfo = mat_statistics_new(nchancols*nchanrows, 1.5);
  if (avinfo == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate a memory for mat_statistics_info (%d vertical correlations)", nchancols*nchanrows);
      mat_statistics_delete(ahinfo);
      return CPL_ERROR_UNSPECIFIED;
    }
  mat_statistics_reset(ahinfo);
  mat_statistics_reset(avinfo);
  /*
   * iterate over the detector channels in the sub-window and
   * calculate the horizontal and vertical correlation.
   */
  /* if a badpixel map is given, interpolate all bad pixels */
  if (bpm != NULL)
    {
      int z;
      int nz = cpl_imagelist_get_size(flats);
      for (z = 0; z < nz; z++)
	{
	  cpl_image *img = cpl_imagelist_get(flats, z);
	  if (img == NULL)
	    {
	      cpl_msg_error(cpl_func, "cannot get image %d from the list of flats", z);
	      continue;
	    }
	  cpl_image_reject_from_mask(img, bpm);
	  cpl_detector_interpolate_rejected(img);
	}
    }
  /* iterate over all detector channels and calculate the autocorrelation */
  for (dcr = 0; dcr < nchanrows; dcr++)
    {
      dcy = dcr*dcny + 1;
      for (dcc = 0; dcc < nchancols; dcc++)
	{
	  double ah, av;
	  int    isinterf = (dcc >= nchancols/2 - 5) && (dcc < nchancols/2 + 5);
	  dcx = dcc*dcnx + 1;
	  /* we use only the central 16x16 area of the available detector channel */
	  rnx = dcnx;
	  if (rnx > 16) rnx = 16;
	  rny = dcny;
	  if (rny > 16) rny = 16;
	  rx = dcx + (dcnx - rnx)/2;
	  ry = dcy + (dcny - rny)/2;
	  if (mat_calc_autocorr_region(median,
				       flats,
				       NULL,
				       rx, ry, rnx, rny,
				       &ah, &av)
	      == CPL_ERROR_NONE)
	    {
	      if (isinterf)
		{
		  mat_statistics_add_value(ahinfo, ah);
		  mat_statistics_add_value(avinfo, av);
		  if ((dahinfo != NULL) && (ah > 0.0)) mat_statistics_add_value(dahinfo, ah);
		  if ((davinfo != NULL) && (av > 0.0)) mat_statistics_add_value(davinfo, av);
		}
	      /*
		cpl_msg_debug(cpl_func,
		"channel[%d,%d]: hcorr = %.3f, vcorr = %.3f",
		dcr, dcc, ah, av);
	      */
	    }
	}
    }
  mat_statistics_calc(ahinfo);
  mat_statistics_calc(avinfo);
  if (ahinfo->reduced_count == 0)
    {
      cpl_msg_warning(cpl_func, "no detector channel has enough light to compute the correlation");
      mat_statistics_delete(ahinfo);
      mat_statistics_delete(avinfo);
      return CPL_ERROR_UNSPECIFIED;
    }
  else
    {
      *acorrhorizontal = ahinfo->average;
      *acorrvertical   = avinfo->average;
      //cpl_msg_debug(cpl_func, "autocorrelation horizontal %.3f vertival %.3f", *acorrhorizontal, *acorrvertical);
      mat_statistics_delete(ahinfo);
      mat_statistics_delete(avinfo);
      return CPL_ERROR_NONE;
    }
}

/**
   @brief Calculates for each detector channel the average fast and slow power spectrum.
   @param median       This image contains the intensity median.
   @param darks        This list contains the list of cold dark images.
   @param bpm          This is the optional badpixel map.
   @param nchancols    The number of detector channel columns.
   @param nchanrows    The number of detector channel rows.
   @param window       The windowing function (see the MAT_WINDOW_* constants).
   @param avg_fastps   This parameter will contain the average fast power spectrum for each detector channel.
   @param avg_slowps   This parameter will contain the average slow power spectrum for each detector channel.

   This function calculates the 1-d power spectra for the fast and slow direction for each
   detector channel. If specified a window function is applied before calculating the FFT.

   The fast power spectrum of a detector channel uses the horizontal pixel lines,
   the number of pixels is equal to the detector channel width (64 for the HAWAII-2,
   32 of the Aquarius detector). For each detector channel up to 2048 lines (for the HAWAII-2)
   or 512 lines (for the Aquarius) can be used for calculating the fast power spectrum.
   This means, that a total of 100 (number of raw frames) times 512 or 2048 fast power
   spectra are averaged for each detector channel.

   The slow power spectrum of a detector channel uses the vertical pixel lines,
   the maximum number of pixels is equal to 512 for the Aquarius and 2048 for
   the HAWAII-2 detector. This will result in up to 100 times 32 (for the Aquarius)
   or 64 (for the HAWAII-2) slow power spectra for each detector channel.

   The huge amount of calculated power spectra will lead to average power spectra
   where peaks induced by external noise can easily be identified.

   Mathematical description:

   <pre>
   BEGIN
   Initialize the average power spectra (size and content);
   FOR EACH cold dark image DO
   IF a median dark is specified THEN
   Subtract the median from that image;
   ENDIF
   IF a badpixel map is specified THEN
   Interpolate all bad pixels;
   ENDIF
   ENDFOR
   Allocate memory for the FFT calculation;
   Create a FFTW plan for the fast power spectrum;
   Create a FFTW plan for the slow power spectrum;
   FOR EACH cold dark image DO
   FOR EACH detector channel row DO
   FOR EACH detector channel column DO
   Copy the image data into the FFT input array;
   Apply an optional windowing function;
   Calculate the 1-d power spectra for the fast direction;
   Add all fast power spectra to the average;
   Calculate the 1-d power spectra for the slow direction;
   Add all slow power spectra to the average;
   ENDFOR
   ENDFOR
   ENDFOR
   Scale the average power spectra;
   Destroy the FFTW plans;
   Free all allocated memory;
   END
   </pre>
*/
cpl_error_code mat_calc_spectra1d(cpl_image *median,
				  cpl_imagelist *darks,
				  cpl_mask *bpm,
				  int nchancols,
				  int nchanrows,
				  int window,
				  cpl_vector **avg_fastps,
				  cpl_vector **avg_slowps)
{
  cpl_image      *dark = NULL;
  int             nx, ny, nz;
  int             x, y, z;
  int             dcx, dcy, dcnx, dcny;
  int             dcc, dcr; /* detector channel column and row */
  int             dci;
  cpl_image      *psFast = NULL;
  cpl_image      *psSlow = NULL;
  int             rejected;
  double          cx, cy, p;

  dark = cpl_imagelist_get(darks, 0);
  if (dark == NULL)
    {
      cpl_msg_error(cpl_func, "cannot get the first image from the list of darks");
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  nx = cpl_image_get_size_x(dark);
  ny = cpl_image_get_size_y(dark);
  nz = cpl_imagelist_get_size(darks);
  dcnx = nx/nchancols;    /* detector channel width  */
  dcny = ny/nchanrows;    /* detector channel height */
  /* initialize all average power spectra */
  for (dci = 0; dci < nchanrows*nchancols; dci++)
    {
      cpl_vector_set_size(avg_fastps[dci], dcnx/2 + 1);
      for (x = 0; x < dcnx/2 + 1; x++)
	{
	  cpl_vector_set(avg_fastps[dci], x, 0.0);
	}
      cpl_vector_set_size(avg_slowps[dci], dcny/2 + 1);
      for (y = 0; y < dcny/2 + 1; y++)
	{
	  cpl_vector_set(avg_slowps[dci], y, 0.0);
	}
    }
  /* if a badpixel map is given, interpolate all bad pixels */
  cpl_msg_debug(cpl_func, "compensating all frames using the dark median and an optional bad pixel map");
  for (z = 0; z < nz; z++)
    {
      dark = cpl_imagelist_get(darks, z);
      if (dark == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot get image %d from the list of darks", z);
	  continue;
	}
      if (median != NULL)
	{
	  cpl_image_subtract(dark, median);
	}
      if (bpm != NULL)
	{
	  cpl_image_reject_from_mask(dark, bpm);
	  cpl_detector_interpolate_rejected(dark);
	}
    }
  /* allocate storage for a detector channel image */
  psFast = cpl_image_new(dcnx, 1, CPL_TYPE_DOUBLE_COMPLEX);
  if (psFast == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate a temporary double complex image of %d pixels (fast ps)", dcnx);
      return CPL_ERROR_UNSPECIFIED;
    }
  psSlow = cpl_image_new(dcny, 1, CPL_TYPE_DOUBLE_COMPLEX);
  if (psSlow == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate a temporary double complex image of %d pixels (slow ps)", dcny);
      cpl_image_delete(psFast);
      return CPL_ERROR_UNSPECIFIED;
    }
  /* iterate over all images and calculate the 1-d power spectra for each detector channel */
  for (z = 0; z < nz; z++)
    {
      dark = cpl_imagelist_get(darks, z);
      if (dark == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot get image %d from the list of darks", z);
	  continue;
	}
      if (z%10 == 0)
	{
	  cpl_msg_debug(cpl_func, "fast and slow 1-d power spectra for frame %d", z);
	}
      /* iterate over all detector channels calculating the individual fast and slow power spectra */
      for (dcr = 0; dcr < nchanrows; dcr++)
	{
	  dcy = dcr*dcny + 1;
	  for (dcc = 0; dcc < nchancols; dcc++)
	    {
	      dcx = dcc*dcnx + 1;
	      dci = dcr*nchancols + dcc;
	      cx = ((double)(dcnx - 1))/2.0;
	      cy = ((double)(dcny - 1))/2.0;
	      /* calculate the fast power spectra for each individual pixel row */
	      for (y = 0; y < dcny; y++)
		{
		  /* copy the intensities of the row into the complex image including the windowing function */
		  for (x = 0; x < dcnx; x++)
		    {
		      double complex intensity = cpl_image_get(dark, dcx + x, dcy + y, &rejected) + 0.0i;
		      double complex weight    = 1.0 + 0.0i;
		      switch (window)
			{
			case MAT_WINDOW_GAUSS:
			  weight = exp(-0.5*((double)x - cx)*((double)x - cx)/(0.25*cx*cx)) + 0.0i;
			  break;
			case MAT_WINDOW_LANCZOS:
			  p = 2.0*((double)x)/((double)dcnx - 1.0) - 1.0;
			  if (p != 0.0)
			    {
			      weight = sin(p)/p + 0.0i;
			    }
			  break;
			case MAT_WINDOW_BLACKMAN:
			  p = 2.0*CPL_MATH_PI*((double)x)/((double)dcnx - 1.0);
			  weight = 0.3635819 - 0.4891775*cos(p) + 0.1365995*cos(2.0*p) - 0.0106411*cos(3.0*p) + 0.0i;
			  break;
			default:;
			}
		      cpl_image_set_complex(psFast, x + 1, 1, intensity*weight);
		    }
		  /* Calculate the 1-d power spectra for the fast (x) direction */
		  cpl_fft_image(psFast, psFast, CPL_FFT_FORWARD);
		  /* Add the (scaled) fast power spectra to the average fast power spectra for the current detector channel */
		  for (x = 1; x < dcnx/2 + 1; x++) /* only the first half and we ignore the constant part */
		    {
		      double v = cabs(cpl_image_get_complex(psFast, x + 1, 1, &rejected))/(double)dcny;
		      cpl_vector_set(avg_fastps[dci], x, cpl_vector_get(avg_fastps[dci], x) + v*v);
		    }
		}
	      /* calculate the slow power spectra for each individual pixel column */
	      for (x = 0; x < dcnx; x++)
		{
		  /* copy the intensities of the column into the complex image including the windowing function */
		  for (y = 0; y < dcny; y++)
		    {
		      double complex intensity = cpl_image_get(dark, dcx + x, dcy + y, &rejected) + 0.0i;
		      double complex weight    = 1.0 + 0.0i;
		      switch (window)
			{
			case MAT_WINDOW_GAUSS:
			  weight = exp(-0.5*((double)y - cy)*((double)y - cy)/(0.25*cy*cy)) + 0.0i;
			  break;
			case MAT_WINDOW_LANCZOS:
			  p = 2.0*((double)y)/((double)dcny - 1.0) - 1.0;
			  if (p != 0.0)
			    {
			      weight = sin(p)/p + 0.0i;
			    }
			  break;
			case MAT_WINDOW_BLACKMAN:
			  p = 2.0*CPL_MATH_PI*((double)y)/((double)dcny - 1.0);
			  weight = 0.3635819 - 0.4891775*cos(p) + 0.1365995*cos(2.0*p) - 0.0106411*cos(3.0*p) + 0.0i;
			  break;
			default:;
			}
		      cpl_image_set_complex(psSlow, y + 1, 1, intensity*weight);
		    }
		  /* Calculate the 1-d power spectra for the slow (y) direction */
		  cpl_fft_image(psSlow, psSlow, CPL_FFT_FORWARD);
		  /* Add the (scaled) slow power spectra to the average slow power spectra for the current detector channel */
		  for (y = 1; y < dcny/2 + 1; y++) /* only the first half and we ignore the constant part */
		    {
		      double v = cabs(cpl_image_get_complex(psSlow, y + 1, 1, &rejected))/(double)dcnx;
		      cpl_vector_set(avg_slowps[dci], y, cpl_vector_get(avg_slowps[dci], y) + v*v);
		    }
		}
	    }
	}
    }
  /* scale the average power spectra */
  for (dci = 0; dci < nchanrows*nchancols; dci++)
    {
      for (x = 0; x < dcnx/2 + 1; x++)
	{
	  cpl_vector_set(avg_fastps[dci], x, cpl_vector_get(avg_fastps[dci], x)/(double)nz);
	}
      for (y = 0; y < dcny/2 + 1; y++)
	{
	  cpl_vector_set(avg_slowps[dci], y, cpl_vector_get(avg_slowps[dci], y)/(double)nz);
	}
    }
  /* delete all allocated space */
  cpl_image_delete(psFast);
  cpl_image_delete(psSlow);
  return CPL_ERROR_NONE;
}

/**
   @brief This function searches for the five largest peaks in a 1-dimensional powerspectrum.
   @param frequency        The basic frequency for the powerspectra. This value is used to derive the frequency of a detected peak.
   @param powerspectrum    Vector containing the average 1-dimensional power spectrum.
   @param peaks            This vector contains the frequency (first component) and power (second component) of the five strongest powerspectra peaks.
   @return cpl_error_code

   This function searches for a given number of peaks in the power spectrum. The number
   of peaks is given as the number of elements in the bivector argument (peaks).

   Remark:

   If the difference between the index before and after the maximum where the
   values starts to increase is too large (a few elements), the peak is not
   sharp and could be ignored. This will possibly result in less than 5 peaks.
   This modification has to be checked when the first cold darks from the
   instrument are available (during integration in Heidelberg).

   Mathematical description:

   The use of the sum of hundreds of 1-dimensional power spectra will result in a smooth
   power spectrum and the following algorithm will be implemented.

   <pre>BEGIN
   Remove the noise underground;
   FOR iPeak := 1 TO 5 DO
   Find the largest value in the vector, index p;
   Fit a quadratic polynomial using ps[p-1], ps[p], ps[p+1];
   Calculate the frequency from the fit;
   Find the index before the maximum where the values starts to increase;
   Find the index after the maximum where the values starts to increase;
   Calculate the power inside the frequency range;
   Store the frequency and the power in the bivector;
   ENDFOR
   END</pre>
*/
cpl_error_code mat_find_peaks(double frequency,
			      cpl_vector *powerspectrum,
			      cpl_bivector *peaks)
{
  int    nx, x;
  int    np, p;
  int    peakposition;
  double peakvalue;
  double peakfrequency;
  double peakpower;
  double offset;
  double basefrequency;
  int    leftpos;
  int    rightpos;

  nx = cpl_vector_get_size(powerspectrum);
  np = cpl_vector_get_size(cpl_bivector_get_x(peaks));
  /*
   * The powerspectrum for N pixels is stored in a vcetor containing N/2 + 1 values (see mat_calc_spectra1d)!
   * This means that:
   *   x = 0            -> constant part
   *   x = 1            -> base frequency = (N/2)*frequency
   *   x = nx - 1 = N/2 -> frequency
   */
  basefrequency = frequency; /* (double)(nx - 1)*frequency; */
  /* initialize the peaks */
  for (p = 0; p < np; p++)
    {
      cpl_vector_set(cpl_bivector_get_x(peaks), p, 0.0);
      cpl_vector_set(cpl_bivector_get_y(peaks), p, 0.0);
    }
  /* try to remove the noise underground */
  offset = cpl_vector_get(powerspectrum, 1) + 1.0;
  for (x = 1; x < nx; x++)
    {
      double oldvalue = cpl_vector_get(powerspectrum, x);
      double newvalue = oldvalue;
      if (oldvalue < offset)
	{
	  offset = oldvalue;
	}
      newvalue = oldvalue - offset;
      /* cpl_msg_debug(cpl_func, "%d %.2f %.2f", x, oldvalue, newvalue); */
      cpl_vector_set(powerspectrum, x, newvalue);
    }
  /* FOR iPeak := 1 TO 5 DO */
  for (p = 0; p < np; p++)
    {
      peakvalue = 0.0;
      peakposition = -1;
      /* find the largest value in the vector, index p; */
      for (x = 0; x < nx; x++)
	{
	  if (cpl_vector_get(powerspectrum, x) > peakvalue)
	    {
	      peakvalue = cpl_vector_get(powerspectrum, x);
	      peakposition = x;
	    }
	}
      if (peakposition == -1)
	{
	  /* the given powerspectrum contains no further peaks */
	  if (p == 0)
	    { /* we did not found any peaks, show a message */
	      cpl_msg_debug(cpl_func, "cannot find any peak in an empty power spectrum");
	      return CPL_ERROR_NONE;
	    }
	  else
	    { /* we have at least found one peak */
	      return CPL_ERROR_NONE;
	    }
	}
      /* fit a quadratic polynomial using ps[p-1], ps[p], ps[p+1]; */
      if (peakposition == (nx - 1))
	{
	  /* we use the peakposition as an exact frequency */
	  peakfrequency = basefrequency*(double)peakposition;
	  cpl_msg_debug(cpl_func, "peakposition %d -> peakfrequency %.2f", peakposition, peakfrequency);
	}
      else
	{
	  double yl = cpl_vector_get(powerspectrum, peakposition - 1);
	  double ym = peakvalue;
	  double yr = cpl_vector_get(powerspectrum, peakposition + 1);
	  double dx = (yl - yr)/4/(ym - yr - (yl - yr)/2.0);
	  peakfrequency = basefrequency*((double)peakposition + dx);
	  cpl_msg_debug(cpl_func, "peakposition %d -> {%.2f, %.2f, %.2f} -> dx %.2f -> peakfrequency %.2f", peakposition, yl, ym, yr, dx, peakfrequency);
	}
      /* find the index before the maximum where the values starts to increase; */
      leftpos = peakposition;
      while ((leftpos > 0) && (cpl_vector_get(powerspectrum, leftpos - 1) < cpl_vector_get(powerspectrum, leftpos)))
	{
	  leftpos--;
	}
      /* find the index after the maximum where the values starts to increase; */
      rightpos = peakposition;
      while ((rightpos < (nx - 1)) && (cpl_vector_get(powerspectrum, rightpos + 1) < cpl_vector_get(powerspectrum, rightpos)))
	{
	  rightpos++;
	}
      /* calculate the power inside the frequency range from leftpos to rightpos; */
      peakpower = 0.0;
      for (x = leftpos; x <= rightpos; x++)
	{
	  peakpower += cpl_vector_get(powerspectrum, x);
	  cpl_vector_set(powerspectrum, x, 0.0);
	}
      /* store the frequency and the power in the bivector; */
      cpl_vector_set(cpl_bivector_get_x(peaks), p, peakfrequency);
      cpl_vector_set(cpl_bivector_get_y(peaks), p, peakpower);
      cpl_msg_debug(cpl_func, "peak at %.2f Hz with a power of %.2f", peakfrequency, peakpower);
    }
  /* we have found all requested peaks */
  return CPL_ERROR_NONE;
}
