/* $Id: mat_statistics.h,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_statistics.h $
 */

#ifndef MAT_STATISTICS_H
#define MAT_STATISTICS_H

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/

#include <cpl.h>
#include "mat_const.h"
#include "mat_drl.h"
#include "mat_detector.h"

/*-----------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/

#define MAT_OUTLIER_FACTOR         1.5
#define MAT_LARGEST_VALUE          +10000000000.0
#define MAT_SMALLEST_VALUE         -10000000000.0

#define MAT_DENSITY_BIN_COUNT     20
#define MAT_DENSITY_PEAK_LIMIT     0.8
#define MAT_HISTOGRAM_PEAK_LIMIT   0.8

#define MAT_USE_WHOLE_MEDIAN       1
#define MAT_USE_WHOLE_MEAN         2
#define MAT_USE_WHOLE_MODE         3
#define MAT_USE_REDUCED_MEDIAN     4
#define MAT_USE_REDUCED_MEAN       5

#define MAT_USE_WHOLE_MEDVAR       1
#define MAT_USE_WHOLE_VARIANCE     2
#define MAT_USE_REDUCED_MEDVAR     4
#define MAT_USE_REDUCED_VARIANCE   5

#define MAT_DETECTOR_WINDOW_TYPE    0
#define MAT_CHANNEL_WINDOW_TYPE     1
#define MAT_DIRECT_WINDOW_TYPE      2

#define MAT_BOTH_DIRECTION          0
#define MAT_HORIZONTAL_DIRECTION    1
#define MAT_VERTICAL_DIRECTION      2

#define MAT_USE_AVERAGE_DEFAULT    MAT_USE_REDUCED_MEDIAN
#define MAT_USE_VARIANCE_DEFAULT   MAT_USE_REDUCED_MEDVAR

#define MAX_COEFF                  10

typedef struct {
  int     use_average;      /*!< Specify which calculated value is used as average for easy access. */
  int     use_variance;     /*!< Specify which calculated value is used as variance for easy access. */
  int     size;             /*!< Size of the allocated memory for the double values. */
  double *values;           /*!< Allocated memory for the double values (order will be changed!). */
  int     hist_size;        /*!< Number of entries in the value histogram. */
  int     hist_count;       /*!< Number of used entries in the value histogram. */
  double  hist_bin;         /*!< Width of a histogram bin. */
  double *hist_x;           /*!< Allocated memory for the histogram x-coordinates (center of a bin). */
  double *hist_y;           /*!< Allocated memory for the histogram y-coordinates. */
  double *hist_p;           /*!< Allocated memory for the histogram y-coordinates. */
  double *hist_q;           /*!< Allocated memory for the histogram y-coordinates. */
  double *hist_s;           /*!< Allocated memory for the histogram y-coordinates. */
  double  hist_peak;
  int     density_count;    /*!< Number of used entries in the density data. */
  double  density_x[MAT_DENSITY_BIN_COUNT]; /*!< Centers of the density bins. */
  double  density_w[MAT_DENSITY_BIN_COUNT]; /*!< Width of the density bins. */
  double  density_d[MAT_DENSITY_BIN_COUNT]; /*!< Normalized density (sum of all densities is 1.0). */
  double  density_peak;
  int     pflag;            /*!< print flag, default 0 = silent. */
  char    prefix[32];       /*!< Prefix printed in case of pflag = 1. */
  double  q25;              /*!< 25 percent quartil. */
  double  q75;              /*!< 75 percent quartil. */
  double  ofactor;          /*!< Factor for detecting outliers (normally 1.5). */
  double  limit_min;        /*!< The lower limit for detecting outliers (calculated from the sorted values). */
  double  limit_max;        /*!< The upper limit for detecting outliers (calculated from the sorted values). */
  double  range_min;        /*!< Smallest allowed value, values below that value are counted as discarded. */
  double  range_max;        /*!< Largest allowed value, values above that value are counted as discarded. */
  int     discarded_count;  /*!< Number of values discarded because they are outside of [-MAT_LARGEST_VALUE .. +MAT_LARGEST_VALUE]. */
  int     nan_count;        /*!< Number of NAN values ignored. */
  int     inf_count;        /*!< Number of INF values ignored. */
  int     whole_count;      /*!< The number of data points in the values vector (includes outliers). */
  double  whole_total;      /*!< Sum of all data points. */
  double  whole_median;     /*!< The median using all values (including outliers). */
  double  whole_medvar;     /*!< The variance based on the median (no autocorrect like whole_variance). */
  double  whole_mean;       /*!< The mean using all values (including outliers). */
  double  whole_variance;   /*!< The variance using all values (including outliers). */
  double  whole_mode;       /*!< The mode (peak position) of the distribution (assuming an unimodal distribution). */
  int     reduced_count;    /*!< The number of data points without outliers (may be smaller than whole_count). */
  double  reduced_median;   /*!< The median ignoring outliers. */
  double  reduced_medvar;   /*!< The variance based on the median (no autocorrect like reduced_variance). */
  double  reduced_mean;     /*!< The mean ignoring outliers. */
  double  reduced_variance; /*!< The variance ignoring outliers. */
  double  average;          /*!< Depending which MAT_USE_* defines exists, the corresponding median/mean value is stored here. */
  double  variance;         /*!< Depending which MAT_USE_* defines exists, the corresponding medvar/variance value is stored here. */
} mat_statistics_info;

typedef struct {
  int              deg;
  int              c2c;
  double           x0u;      /*!< x-offset in user space (not scaled by xscale!). */
  double           xscale;   /*!< Scale factor for x-coordinate: xn = (xu - x0u)/xscale */
  double           yscale;   /*!< Scale factor for y-coordinate: yn = yu/yscale */
  double           a0n;      /*!< function coefficient a0 in normalized space (scaled by xscale, yscale). */
  double           a1n;      /*!< function coefficient a1 in normalized space (scaled by xscale, yscale). */
  double           a2n;      /*!< function coefficient a2 in normalized space (scaled by xscale, yscale). */
  double           a3n;      /*!< function coefficient a3 in normalized space (scaled by xscale, yscale). */
  double           c2t;      /*!< chi squared in user space (not scaled). */
  double           c2g;
  double           K;        /*!< Curvature (-> Bronstein) */
} mat_poly_info;

typedef struct {
  // data used for function fit
  int              ndspace;   /*!< The size of the allocated memory (data, ...). */
  int              ndused;    // number of useful data points (only linear and nonlinear intensity range)
  // scaling applied before fitting
  double           xu2f;      // xf := xu*xu2f
  double           yu2f;      // yf := yu*yu2f
  // unscaled original data
  double          *xu;         // original x values (unscaled exptime) including an artificial 0.0 [DU]
  double          *yu;         // original y values (unscaled median intensities) including an artificial 0.0 [s]
  double          *yeu;        // original y error (unscaled sqrt(variance)) [DU]
  // data used for LVM
  int            (*f)(const double x[], const double a[], double *result);
  int            (*dfda)(const double x[], const double a[], double result[]);
  int              nfspace;
  int              nfused;
  int             *fsize;     // list of number of data points used for a specific fit (size of mxs[i], vys[i] and vyes[i])
  cpl_matrix     **mxf;       // list of cpl_matrix for different numbers of useful data points (scaled!)
  cpl_vector     **vyf;       // list of cpl_vector for different numbers of useful data points (scaled!)
  //cpl_vector     **vyef;    // list of cpl_vector for different numbers of useful data points (scaled!)
  cpl_vector      *vaf;                   // cpl_vector for the function coefficients (scaled!)
  double           af[MAX_COEFF]; // (scaled!)
  int              iaf[MAX_COEFF];       // flag if a coefficient should be used for the fit
  double           mse;
  // unscaled results
  double           xtest;     // values below xtest are always used for chi2 calculation
  double           rdev;      // values above xtest must be better than rdev
  double           limit;     // highest x value were all fitted values are better than rdev
  double           stdev;
  double           rstdev;
  double           chi2;
  double           rchi2;
  //cpl_vector      *s;                   // cpl_vector for the polynomial evaluation (sample value)
  int              show_flag;
} mat_fit_info;

typedef struct {
  // parameters and input
  int              with_src_filter;  /*!< Flag if the data points at the source should be selected (intensity range). */
  int              src_filter;       /*!< The source intensity range filter. */
  int              with_dst_filter;  /*!< Flag if the data points at the destination should be selectd (intensity range). */
  int              dst_filter;       /*!< The destination intensity range filter. */
  int              show_flag;
  int              chi2limit_flag;
  int              poix;
  int              poiy;
  // original data, used data, smoothened data, ...
  int              rx;       /*!< The left side of the region (1-based). */
  int              ry;       /*!< The lower side of the region (1-based). */
  int              rnx;      /*!< The number of pixel columns (width). */
  int              rny;      /*!< The number of pixel rows (height). */
  int              nspace;   /*!< The size of the allocated memory (data, ...). */
  int              nused;    /*!< The number of used data points (nused <= nspace). */
  int              first;    /*!< The first usefull data point (only for 1-d applications). */
  int              last;     /*!< The last usefull data point (only for 1-d applications). */
  int              count;    /*!< The number of filtered data points (normally stored in work). */
  double          *original; /*!< The linearized original (_unmodified_) values. */
  double          *data;     /*!< The linearized original/modified values. */
  int             *range;    /*!< The intensity range for each pixel (MAT_OUTLIER_INTENSITY, MAT_BADPIX_INTENSITY, MAT_LOW_INTENSITY, MAT_LINEAR_INTENSITY). */
  double          *loval;    /*!< The lower bound of the data values (downward peaks are connected by straight lines). */
  double          *hival;    /*!< The upper bound of the data values (upward peaks are connected by straight lines). */
  double          *work;     /*!< The working space (can contain only count values due to filtering). */
  double          *smooth;   /*!< The smooth values (same order than the original values). */
  int             *flags;
  // fitted function
  mat_poly_info    poly;
} mat_smooth_info;

typedef struct {
  cpl_image    *original;
  cpl_image    *dst;
  cpl_image    *src;
  cpl_image    *range;
  int           src_filter;
  int           dst_filter;
  int           x0;
  int           y0;
  int           nx;
  int           ny;
  int           nc;
  int           nr;
  int           poix;
  int           poiy;
} mat_extract_info;

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/

mat_statistics_info *mat_statistics_new(int size, double ofactor);
mat_statistics_info *mat_statistics_realloc(mat_statistics_info *info, int size, double ofactor);
void mat_statistics_delete(mat_statistics_info *info);
void mat_statistics_reset(mat_statistics_info *info);
void mat_statistics_set_use(mat_statistics_info *info, int use_avg, int use_var);
void mat_statistics_set_range(mat_statistics_info *info, double lower, double upper);
void mat_statistics_add_value(mat_statistics_info *info, double value);
void mat_statistics_set_region(mat_statistics_info *info, cpl_image *image, cpl_mask *bpm, int x, int y, int nx, int ny);
void mat_statistics_set_sequence(mat_statistics_info *info, cpl_imagelist *list, cpl_mask *bpm, int x, int y);
void mat_statistics_calc(mat_statistics_info *info);

void mat_poly_init(mat_poly_info *info, int first_data, int last_data);
double mat_poly_eval(mat_poly_info *info, double x);
double mat_poly_derivative(mat_poly_info *info, double x, int d);
double mat_poly_curvature(mat_poly_info *info, double xl, double xr);
char *mat_poly_get_func(mat_poly_info *info);

mat_fit_info *mat_fit_info_new(int(*f)(const double x[], const double a[], double *result), int(*dfda)(const double x[], const double a[], double result[]));
void mat_fit_info_delete(mat_fit_info *info);
void mat_fit_reset(mat_fit_info *info, int show_flag);
void mat_fit_add_data(mat_fit_info *info, double x, double y, double ye);
void mat_fit_discard(mat_fit_info *info, int idx);
void mat_fit_set_coeff(mat_fit_info *info, int idx, double value, int active);
void mat_fit_enable_coeff(mat_fit_info *info, int idx, int active);
cpl_error_code mat_fit_calc_fit(mat_fit_info *info, int ndused);
void mat_fit_calc_quality(mat_fit_info *info, int ndused, double xtest, double rdev);

mat_smooth_info *mat_smooth_info_new(int nspace, int src_filter, int dst_filter);
void mat_smooth_info_delete(mat_smooth_info *info);
void mat_smooth_reset(mat_smooth_info *info);
void mat_smooth_add_value(mat_smooth_info *info, double value);
void mat_smooth_set(mat_smooth_info *info, cpl_image *src, cpl_image *range, int x, int y, int nx, int ny);
void mat_smooth_apply(mat_smooth_info *info, cpl_image *dst, double scale);
void mat_smooth_calc_result(mat_smooth_info *info, int first_smooth, int last_smooth);
void mat_smooth_mark_outlier(mat_smooth_info *info, int first_data, int last_data, double chi2limit);
//double mat_smooth_poly0_window(mat_smooth_info *info, mat_statistics_info *stat_info, int first_data, int last_data, int first_smooth, int last_smooth);
//double mat_smooth_poly1_window(mat_smooth_info *info, mat_statistics_info *stat_info, int first_data, int last_data, int first_smooth, int last_smooth);
//double mat_smooth_poly2_window(mat_smooth_info *info, mat_statistics_info *stat_info, int first_data, int last_data, int first_smooth, int last_smooth);
//double mat_smooth_poly3_window(mat_smooth_info *info, mat_statistics_info *stat_info, int first_data, int last_data, int first_smooth, int last_smooth);
//double mat_smooth_polyn_window(mat_smooth_info *info, mat_statistics_info *stat_info, int n, int first_data, int last_data, int first_smooth, int last_smooth);
double mat_smooth_polyn_window(mat_smooth_info *info, mat_statistics_info *stat_info, int n, int first_data, int last_data, int first_smooth, int last_smooth);
double mat_smooth_polyx_window(mat_smooth_info *info, mat_statistics_info *stat_info, int first_data, int last_data, int first_smooth, int last_smooth);
double mat_smooth_polyn_window_chi2(mat_smooth_info *info, mat_statistics_info *stat_info, int n, int first_data, int last_data, int first_smooth, int last_smooth);
double mat_smooth_polyx_window_chi2(mat_smooth_info *info, mat_statistics_info *stat_info, int first_data, int last_data, int first_smooth, int last_smooth);
double mat_smooth_polyn_window_outlier(mat_smooth_info *info, mat_statistics_info *stat_info, int n, int first_data, int last_data, int first_smooth, int last_smooth, double enhance_factor);
double mat_smooth_polyx_window_outlier(mat_smooth_info *info, mat_statistics_info *stat_info, int first_data, int last_data, int first_smooth, int last_smooth, double enhance_factor);
void mat_smooth_polyn_sequence(mat_smooth_info *info, int n, int plength);
void mat_smooth_polyn_sequence_outlier(mat_smooth_info *info, int n, int plength, double enhance_factor);

void mat_extract_info_set_data(mat_extract_info *info, cpl_image *dst, cpl_image *src, cpl_image *range);
void mat_extract_info_set_original(mat_extract_info *info, cpl_image *original);
void mat_extract_info_set_filter(mat_extract_info *info, int src_filter, int dst_filter);
void mat_extract_info_set_window(mat_extract_info *info, mat_detector *det, int wintype, int x, int nx, int y, int ny);
void mat_extract_polyn(mat_extract_info *einfo, int dir, int n, int plength);

cpl_error_code mat_calc_statistics_cosmics(mat_gendata *raw,
					   mat_frame *median, 
					   mat_frame *variance,
					   int cosmics);

cpl_error_code mat_calc_gain_region(cpl_image *median,
				    cpl_image *variance,
				    cpl_image *irange,
				    cpl_mask *bpm,
				    int rx, int ry, int rnx, int rny,
				    double *gain,
				    int pflag);

cpl_error_code mat_calc_gain(cpl_image *median,
			     cpl_image *variance,
			     cpl_image *irange,
			     cpl_mask *bpm,
			     int nchancols,
			     int nchanrows,
			     double *detgain,
			     cpl_vector *channelgain);

cpl_error_code mat_calc_median_dev(cpl_image *data,
				   cpl_mask *bpm,
				   int nchancols,
				   int nchanrows,
				   cpl_imagelist *list,
				   cpl_vector *median,
				   cpl_vector *dev);

cpl_error_code mat_calc_offset(cpl_image *median,
			       cpl_mask *bpm,
			       int nchancols,
			       int nchanrows,
			       cpl_vector *channeloffset);

cpl_error_code mat_calc_noise(cpl_image *variance,
			      cpl_mask *bpm,
			      int nchancols,
			      int nchanrows,
			      double detgain,
			      double *detnoise,
			      cpl_vector *channelgain,
			      cpl_vector *channelnoise,
			      double dsgain);

double mat_calc_avar(double frametime,
		     cpl_imagelist *darks,
		     cpl_mask *bpm);

cpl_error_code mat_calc_autocorr_region(cpl_image *median,
					cpl_imagelist *flats,
					cpl_mask *bpm,
					int rx, int ry, int rnx, int rny,
					double *acorrhorizontal,
					double *acorrvertical);

cpl_error_code mat_calc_autocorr(cpl_image *median,
				 cpl_imagelist *flats,
				 cpl_mask *bpm,
				 int nchancols,
				 int nchanrows,
				 double *acorrhorizontal,
				 double *acorrvertical,
				 mat_statistics_info *dahinfo,
				 mat_statistics_info *davinfo);

cpl_error_code mat_calc_spectra1d(cpl_image *median,
				  cpl_imagelist *darks,
				  cpl_mask *bpm,
				  int nchancols,
				  int nchanrows,
				  int window,
				  cpl_vector **avg_fastps,
				  cpl_vector **avg_slowps);

cpl_error_code mat_find_peaks(double frequency,
			      cpl_vector *powerspectra,
			      cpl_bivector *peaks);

#endif
