/* $Id: mat_kappamatrix_save-test.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_kappamatrix_save-test.c $
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <cpl.h>
#include <cpl_test.h>
#include <sys/stat.h>
#include "../mat_kappamatrix.h"
#include "../mat_apply_staticcalib.h"
#include "../mat_imagingdata.h"
#include "../mat_error.h"
#include "../mat_ext_beams_lib.h"
#include "../mat_est_corr_lib.h"

#define RESFILENAME "saveKappamatrixTest.fits"

mat_gendata * calibrate(cpl_frame *p_frame,mat_badpixel *bad, 
			mat_nonlinearity *nonlin,mat_obsflat *obsflat);

CPL_DIAG_PRAGMA_PUSH_IGN(-Wunused-function);
static cpl_error_code mat_kappamatrix_save_test_basics(void)
{
  char              *filename      = NULL;
  cpl_frame         *frame         = NULL;   
  cpl_frame         *sel_frame     = NULL;                                                           
  cpl_frameset      *usedframes    = NULL;
  mat_gendata       *pp_shutter    = NULL;
  mat_kappamatrix   *p_kappamatrix = NULL;                                        
  int                nbchannel;
  int                nb_extent;
  int                i;

  // Create the frame containing the FITS input file and load and calibrate this file
  frame      = cpl_frame_new();
  filename   = cpl_sprintf("%s/unit_06.fits", MAT_TESTDATADIR);
  cpl_frame_set_filename(frame, filename);
  cpl_free(filename);

  cpl_test(mat_check_photometry(frame) == CPL_FALSE);
  cpl_test(mat_check_interf(frame) == CPL_TRUE);

  pp_shutter = calibrate(frame, NULL, NULL, NULL);

  // Create the basic Kappa Matrix file (no table, only coefficients)
  nbchannel=pp_shutter->imgdet->list_region[0]->naxis[1];
  p_kappamatrix = cpl_calloc(1, sizeof(mat_kappamatrix)); // cpl_malloc(sizeof(mat_kappamatrix));
  p_kappamatrix->nbbeam = 4;
  p_kappamatrix->keywords = cpl_propertylist_duplicate(pp_shutter->keywords);
  p_kappamatrix->imgdet = mat_imagingdetector_duplicate(pp_shutter->imgdet);
  p_kappamatrix->list_kappacoef=cpl_calloc(p_kappamatrix->nbbeam,sizeof(mat_kappacoef *));
  for (i=0; i<p_kappamatrix->nbbeam; i++)
    {
      p_kappamatrix->list_kappacoef[i] = cpl_calloc(1, sizeof(mat_kappacoef)); // cpl_malloc(sizeof(mat_kappacoef));
      p_kappamatrix->list_kappacoef[i]->numdetector = pp_shutter->imgdet->list_region[i]->numdetector;
      p_kappamatrix->list_kappacoef[i]->numregion   = i+1;
      p_kappamatrix->list_kappacoef[i]->nbchannel   = nbchannel;
      p_kappamatrix->list_kappacoef[i]->coef        = cpl_matrix_new(3,nbchannel);
      p_kappamatrix->list_kappacoef[i]->errcoef     = cpl_matrix_new(3,nbchannel);
    }

  cpl_test(p_kappamatrix != NULL);

  // Put the 'KAPPA_MATRIX' extension into RESFILENAME
  usedframes = cpl_frameset_new();
  sel_frame  = cpl_frame_duplicate(frame);
  cpl_frame_set_group(sel_frame, CPL_FRAME_GROUP_RAW);
  cpl_frame_set_tag(sel_frame," RAW_VIS2");
  cpl_frameset_insert(usedframes, sel_frame);
  nb_extent  = cpl_frame_get_nextensions(frame);
  for (i = 0; i < nb_extent; i++)
    {
      cpl_propertylist *p_list        = cpl_propertylist_load(cpl_frame_get_filename(frame), i + 1);
      char             *p_key_extname = (char *)cpl_propertylist_get_string(p_list, "EXTNAME");
      //cpl_msg_info(cpl_func, "file %s extend %d EXTNAME %s", cpl_frame_get_filename(frame), i + 1, p_key_extname);
      if (!strcmp(p_key_extname, "IMAGING_DATA"))
	{
	  cpl_propertylist  *p_list_kappa = cpl_propertylist_duplicate(p_list);
	  cpl_parameterlist *parlist      = cpl_parameterlist_new();
	  cpl_propertylist_update_string(p_list_kappa, "EXTNAME", "KAPPA_MATRIX");
	  cpl_test( mat_kappamatrix_save(p_kappamatrix, p_list_kappa,
					 usedframes,
					 usedframes, parlist, "test", "CALIB",
					 "IMAGE", "SCIENCE",
					 (char *)RESFILENAME) ==
		    CPL_ERROR_NONE);
	  cpl_parameterlist_delete(parlist);
	  cpl_propertylist_delete(p_list_kappa);
    	}
      cpl_propertylist_delete(p_list);
    }

  // Copy the remaining extensions of the input file into the result file
  for (i = 0; i < nb_extent; i++)
    {
      cpl_propertylist *p_list        = cpl_propertylist_load(cpl_frame_get_filename(frame), i + 1);
      char             *p_key_extname = (char *)cpl_propertylist_get_string(p_list, "EXTNAME");
      if (p_key_extname != NULL)
	{
	  cpl_table * table = cpl_table_load(cpl_frame_get_filename(frame), i + 1, 0);
	  if (!strcmp(p_key_extname, "IMAGING_DETECTOR"))
	    { // IMAGING_DETECTOR -> Append the table without calibration regions
	      mat_imagingdetector_save(p_kappamatrix->imgdet, NULL, (char *)RESFILENAME);
	    }
	  else if (!strcmp(p_key_extname, "IMAGING_DATA"))
	    { // IMAGING_DATA -> Do nothing
	    }
	  else
	    { // ELSE -> append the table from the raw file
	      cpl_table_save(table, NULL, p_list, (char *)RESFILENAME, CPL_IO_EXTEND);
	    }
	  cpl_table_delete(table);
	}
      cpl_propertylist_delete(p_list);
    }

  //fprintf(stderr, "before mat_kappamatrix_free(p_kappamatrix)\n");
  mat_kappamatrix_free(p_kappamatrix);
  //fprintf(stderr, "before mat_gendata_delete(pp_sky_shutter)\n");
  mat_gendata_delete(pp_shutter);
  //fprintf(stderr, "before cpl_frame_delete(frame)\n");
  cpl_frame_delete(frame);
  //fprintf(stderr, "before cpl_frameset_delete(usedframes)\n");
  cpl_frameset_delete(usedframes);

  printf("coucou1\n");

  //remove(RESFILENAME);
  return cpl_error_get_code();
}
CPL_DIAG_PRAGMA_POP;
/*----------------------------------------------------------------------------*/
/**
   @brief Calibrate a raw mat_gendata (badpixel, nonlinearity and optical flat
   field) and Create a new mat_gendata structure with only one row in the 
   IMAGING_DATA binary table which is the sum of the rows contained in the 
   original raw data.
   @param p_frame : the frale containing the raw data
   @param bad     : mat_badpixel structure containing the BADPIX map
   @param nonlin  : mat_nonlinearity structure containing the NONLINEARITY map
   @param obsflat : mat_obsflat structure containing the OBS_FLATFIELD map
   @return mat_gendata
*/
/*-----------------------------------------------------------------------------*/
mat_gendata * calibrate(cpl_frame *p_frame,mat_badpixel*bad, 
			mat_nonlinearity *nonlin,mat_obsflat *obsflat)
{  
  mat_gendata  *raw             = NULL;
  mat_gendata  *cal             = NULL;
  mat_gendata  *final           = NULL;

  // Check input parameters
  mat_assert_value((p_frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,
		   "no p_frame argument given");
  // Load the FITS file
  raw = mat_gendata_load(p_frame, CPL_TYPE_FLOAT);
  if (raw == NULL) {
    cpl_msg_error(cpl_func,"could not load FITS file into rawdata");
    return NULL;
  }
  // Calibrate the raw data (do nothing, remove the reference regions only)
  //cal = mat_apply_staticcalib(raw, bad, nonlin, obsflat);
  cal = mat_calibration_detector(raw, bad, NULL, nonlin, obsflat, NULL, FULL_COMPENSATION, 1, 0);
  mat_gendata_delete(raw);
  if (cal == NULL) {
    cpl_msg_error(cpl_func,"could not calibrate rawdata");
    return NULL;
  }
  // Build the final result which should contain the ARRAY and a  summed up (total) frame
  final = (mat_gendata *)cpl_calloc(1, sizeof(mat_gendata)); // cpl_calloc() clears out the memory which is important!
  if (final == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for the final result");
    mat_gendata_delete(cal);
    return NULL;
  }
  final->keywords = cpl_propertylist_duplicate(cal->keywords);
  final->imgdet   = mat_imagingdetector_duplicate(cal->imgdet);
  if (final->imgdet == NULL) {
    cpl_msg_error(cpl_func,"could not duplicate cal->imgdet");
    mat_gendata_delete(cal);
    mat_gendata_delete(final);
    return NULL;
  }
  final->array = mat_array_duplicate(cal->array);
  if (final->array == NULL) {
    cpl_msg_error(cpl_func,"could not dupliacte cal->array");
    mat_gendata_delete(cal);
    mat_gendata_delete(final);
    return NULL;
  }
  final->imgdata = mat_imagingdata_sum(cal->imgdata, CPL_TYPE_FLOAT);
  if (final->imgdata == NULL) {
    cpl_msg_error(cpl_func,"could not sum cal->imgdata");
    mat_gendata_delete(cal);
    mat_gendata_delete(final);
    return NULL;
  }
  mat_gendata_delete(cal);
  return final;
}

/*----------------------------------------------------------------------------*/
/**
  @brief   Unit tests of KAPPAMATRIX SAVE module
 **/
/*----------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    mat_kappamatrix_save_test_basics();

    cpl_test_error(CPL_ERROR_NONE);
    return cpl_test_end(0);
}

