/* $Id: mat_cal_det.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_cal_det.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define MATISSE_LICENCE            "GPL"
#define MAX_NBDIT                  128

#define MATISSE_DO_DARK            "DARK"
#define MATISSE_DO_FLAT            "FLAT"

#define MATISSE_DO_BPM             "BADPIX"
#define MATISSE_DO_FFM             "FLATFIELD"

#define MATISSE_STATISTICS_PROCATG "STAT"

#define MATISSE_KW_EXPTIME         "EXPTIME"

#define MAT_OUTLIERS_LENGTH        31
#define MAT_OUTLIERS_FACTOR         1.5
#define MAT_POLY2_LENGTH           31
#define MAT_SMOOTH_NONLIN_PLENGTH  19

//#define DEVELOP 1


/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/
#include <strings.h>
#include <string.h>
#include <math.h>

#include <cpl.h>
#include "mat_drl.h"
#include "mat_const.h"
#include "mat_error.h"
#include "mat_utils.h"
#include "mat_frame.h"
#include "mat_image.h"
#include "mat_statistics.h"
#include "mat_badpixel.h"
#include "mat_flatfield.h"
#include "mat_nonlinearity.h"
#include "mat_detector.h"
#include "mat_imagingdetector.h"
#include "mat_apply_staticcalib.h"

/*-----------------------------------------------------------------------------
  Data types
  -----------------------------------------------------------------------------*/

#define STD_BP_TYPE       0
#define EMPTY_BP_TYPE     1
#define MERGE_BP_TYPE     2

#define STD_FF_TYPE       0
#define CONST_FF_TYPE     1
#define AVERAGE_FF_TYPE   2

#define STD_NL_TYPE       0
#define LINEAR_NL_TYPE    1
#define AVERAGE_NL_TYPE   2

#define STAT_OUTPUT_EXPERT_FLAG  (1 << 0)
#define RAW_EXPERT_FLAG          (1 << 1)
#define DARK_EXPERT_FLAG         (1 << 2)
#define FLAT_EXPERT_FLAG         (1 << 3)
#define NONLINEARITY_EXPERT_FLAG (1 << 4)
#define GAIN_EXPERT_FLAG         (1 << 5)
#define FLATFIELD_EXPERT_FLAG    (1 << 6)

//#define NOISE_EXPERT_FLAG        (1 << 5)
//#define BPM_EXPERT_FLAG          (1 << 7)
//#define BPM_DETAILS_EXPERT_FLAG  (1 << 9)
//#define FFM_DETAILS_EXPERT_FLAG  (1 << 10)
//#define NLM_DETAILS_EXPERT_FLAG  (1 << 11)

// S4 S3 S2 S1 =>
//  0  0  0  0  DARK
//  0  0  0  1  SH1
//  0  0  1  0  SH2
//  0  0  1  1  SH12
//  0  1  0  0  SH3
//  0  1  0  1  SH13
//  0  1  1  0  SH23
//  0  1  1  1  SH123
//  1  0  0  0  SH4
//  1  0  0  1  SH14   
//  1  0  1  0  SH24
//  1  0  1  1  SH124
//  1  1  0  0  SH34
//  1  1  0  1  SH134
//  1  1  1  0  SH234
//  1  1  1  1  SH1234
//              FLAT

typedef enum {
  DARK_IDX,
  SH1_IDX, SH2_IDX, SH12_IDX, SH3_IDX, SH13_IDX,
  SH23_IDX, SH123_IDX, SH4_IDX, SH14_IDX, SH24_IDX,
  SH124_IDX, SH34_IDX, SH134_IDX, SH234_IDX, SH1234_IDX,
  FLAT_IDX
} mat_psm_index;

typedef enum {
  DIT_NLMODE,   // nonlinearity is based on measurements at different DIT
  FLUX_NLMODE   // nonlinearity is based on measurements at different flux (shutter combinations)
} mat_nonlin_mode;

typedef struct {
  int      x;     // pixel x-coordinate (1-based, global)
  int      y;     // pixel y-coordinate (1-based, global)
  //int      dc;    // detector channel number (0-based)
  //int      wl;    // wavelength number (0-based)
  double   i1234; // intensity with several (2, 3 or 4) open shutters
  double   i1;    // intensity of shutter 1 open (or 0.0 when shutter 1 was closed)
  double   i2;    // intensity of shutter 2 open (or 0.0 when shutter 2 was closed)
  double   i3;    // intensity of shutter 3 open (or 0.0 when shutter 3 was closed)
  double   i4;    // intensity of shutter 4 open (or 0.0 when shutter 4 was closed)
} mat_intensity_info;

typedef struct {
  double           x[MAX_NBDIT + 1];    // original x values (unscaled exptime) including an artificial 0.0 [DU]
  double           y[MAX_NBDIT + 1];    // original y values (unscaled median intensities) including an artificial 0.0 [s]
  double           ye[MAX_NBDIT + 1];   // original y error (unscaled sqrt(variance)) [DU]
  int              r[MAX_NBDIT + 1];    // intensity range flag
  cpl_matrix      *mxs[MAX_NBDIT + 1];  // list of cpl_matrix for different numbers of useful data points
  cpl_vector      *vys[MAX_NBDIT + 1];  // list of cpl_vector for different numbers of useful data points
  cpl_vector      *vyes[MAX_NBDIT + 1]; // list of cpl_vector for different numbers of useful data points
  cpl_vector      *a;                   // cpl_vector for the function coefficients
  cpl_vector      *s;                   // cpl_vector for the polynomial evaluation (sample value)
  int              ia[MAX_COEFF];       // flag if a coefficient should be used for the fit
  int              nbtotal;             // number of data points (equal with number of files + 1)
  int              nbused;              // number of useful data points (only linear and nonlinear intensity range)
  double           xmin;
  double           xmax;
  double           ymin;
  double           ymax;
  double           xscale;              // scale factor [DU] -> scaled [DU]
  double           yscale;              // scale factor [s]  -> scaled [DU]
  double           mse;
  double           red_chisq;
  double           stdev;
  double           limit;
  int              isok;
  int              change_scale;
} mat_fit_info_ot;

typedef struct {
  double        dit;     // DIT of the pixel statistics
  char         *fname[FLAT_IDX + 1]; // frame/file names
  cpl_image    *rmed[FLAT_IDX + 1]; // temporal median for each pixel, indexed by PSMIDX
  cpl_image    *rvar[FLAT_IDX + 1]; // temporal variance for each pixel, indexed by PSMIDX
  cpl_image    *cmed[FLAT_IDX + 1]; // temporal median for each pixel, indexed by PSMIDX
  cpl_image    *cvar[FLAT_IDX + 1]; // temporal variance for each pixel, indexed by PSMIDX
  cpl_vector   *poidarks;
  cpl_vector   *poiflats;
  double        hacorr[FLAT_IDX + 1];
  double        vacorr[FLAT_IDX + 1];
} mat_psm_info;

typedef struct {
  // Parameters from command line of parameter file
  double               gain;
  int                  nditskip;
  int                  cosmics;
  double               darklimit;
  double               flatlimit;
  double               min_linear_range;
  double               max_linear_range;
  double               max_nonlinear_range;
  double               max_rel_deviation;
  double               max_abs_deviation;
  int                  use_defaults;
  int                  compensate;
  int                  bptype; /*!< bad pixel type (*_BP_TYPE constants) */
  int                  fftype; /*!< flatfield type (*_FF_TYPE constants) */
  int                  nltype; /*!< nonlinearity type (*_NL_TYPE constants) */
  int                  poix;
  int                  poiy;
  int                  expert;
  int                  ntflag;
  // Informations from the DARK and FLAT files
  mat_detector        *det;      // this is the detector specification determined from the first file
  cpl_propertylist    *keywords; // the keywords from the primary HDU of the first file.
  mat_imagingdetector *imgdet;   // IMAGING_DETECTOR binary table contents from the first file (used to check the sub-window setup)
  int                  hasphoto;
  int                  nx;  // image size NX
  int                  ny;         // image size NY
  double               timescale;
  int                  nbdit;          // number of pixel statistics (different DIT)
  mat_psm_info         psm[MAX_NBDIT]; // array with pixel statistics, each for one DIT
  // parameters and temporary data needed for the dark properties (average intensity and variance, dark current offset and slope)
  cpl_image           *darkmedraw; // summed up dark DU/s (all raw values)
  cpl_image           *darkmedref; // reference values for darkmedraw (from smoothing darkmedraw)
  cpl_image           *darkmedres; // residuals between reference and raw darkmed (res = raw - ref)
  cpl_image           *darkvarraw; // summed dark DU^2 (raw values)
  cpl_image           *darkrange;  // intensity range/mask (MAT_LOW_INTENSITY or MAT_LINEAR_INTENSITY)
  cpl_image           *dcoffset;   // dark current offset [DU]
  cpl_image           *dcslope;    // dark current slope [DU/s]
  // parameters and temporary data needed for the flat properties (photons [DU/s], noise [DU^2/s] and their reference values for bad pixel creation)
  cpl_image           *flatmedraw;   // summed up flat DU/ms (raw values, only up to 50 percent of full well, compensated for nonlinearity)
  cpl_image           *flatmedref;   // reference values for flatmedraw (from smoothing flatmedraw)
  cpl_image           *flatmedres;   // residuals between reference and raw flatmed (res = raw - ref)
  cpl_image           *flatvarraw;   // summed up flat DU^2 (raw values, only up to 50 percent of full well, compensated for nonlinearity)
  cpl_image           *flatrange;    // intensity range/mask (MAT_LOW_INTENSITY or MAT_LINEAR_INTENSITY)
  cpl_image           *flatbpm;      // relative difference to intensity (residual/reference)
  double               aqpcs[64];    // intensity scale for pixel columns (Aquarius, lower and upper detector channels, idx = dc*32 + c)
  double               aqdcr[64];
  double               rowrem[64*512]; // remanence offset for each row inside a specific detector column (therefore detector channels times detector channel height)
  // temporary data needed to create a conversion gain histogram (raw and nonlinearity compensated)
  double               gs_med;       // scaling: x = gs_med * median
  double               gs_var;       // scaling: y = gs_var * variance
  cpl_image           *gainhistraw;  // raw 2-dimensional histogram variance vs. median (see scaling gs_med, gs_var)
  cpl_image           *gainhistcal;  // calibrated 2-dimensional histogram variance vs. median (see scaling gs_med, gs_var)
  // parameters and temporary data needed for the nonlinearity characterization
  mat_nonlin_mode      nlmode;
  // global nonlinearity derived from exposures with different shutter combinations (_flux)
  int                  nlstart;   // first intensity, the nonlinearity law is calculated for
  int                  nlstep;    // step between successive intensities
  int                  nbintensities;  // number of used intensity infos (max 3*nx*ny)
  mat_intensity_info  *intensities;    // nx*ny allocated intensity infos (Aquarius nonlinearity)
  mat_statistics_info *stat1;          // statistics info (used for different purposes)
  mat_statistics_info *stat2;          // statistics info (used for different purposes)
  mat_statistics_info *stat3;          // statistics info (used for different purposes)
  int                  nl_nbentries;   // number of entries in each raw -> cal map (65536/nlstep + 1);
  double              *nl_map;         // global nonlinearity compensation polyline
  double               nl_limit;       // limit for the nonlinearity compensation (detector)
  int                  nbpoly;         // number of polyline samples (max 65536)
  double              *polyraw;        // nonlinearity polyline raw values (max 65536)
  double              *polycal;        // nonlinearity polyline calibrated values (max 65536)
  double              *polycalsmooth;  // smooth nonlinearity polyline calibrated values (max 65536)
  double              *nonlinmap;      /*!< mapping raw -> cal (nonlinearity for Aquarius detector only) */
  cpl_image           *ditscale;
  cpl_image           *nlhistraw;      // raw 2-dimensional histogram intensity vs. scaled DIT
  cpl_image           *nlhistcal;      // raw 2-dimensional histogram intensity vs. scaled DIT
  cpl_image           *nlcount;        // number of used intensities for each pixel
  double               da0;
  double               da1;
  double               da2;
  double               da3;
  double               dstdev;
  double               dlimit;
  // The final bad pixel map, flatfield map and nonlinearity map
  mat_badpixel        *bpm;
  mat_nonlinearity    *nlm;
  mat_flatfield       *ffm;
  mat_statistics_info *hacstat;          // statistics info (horizontal autocorrelation)
  mat_statistics_info *vacstat;          // statistics info (vertical autocorrelation)
} mat_cal_det_info;

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/
static int mat_cal_det_create(cpl_plugin *);
static int mat_cal_det_exec(cpl_plugin *);
static int mat_cal_det_destroy(cpl_plugin *);
static int mat_cal_det(cpl_parameterlist *, cpl_frameset *);

/*-----------------------------------------------------------------------------
  Static variables
  -----------------------------------------------------------------------------*/
static char mat_cal_det_description[] =
  "This plugin uses several series of cold dark and flatfield exposures to create "
  "the static calibration maps for both MATISSE detectors. Due to the two different "
  "detectors (HAWAII-2RG and Aquarius) the methods to derive the nonlinearity and "
  "flatfield maps are detector dependent. To be more precise, the plugin is able to "
  "calculate a flux dependent nonlinearity map (used for the Aquarius) and a intensity "
  "dependent nonlinearity map (used for the HAWAII-2RG). The content of the input files "
  "in the SOF determine which kind of nonlinearity map is created. OBs do exist for "
  "both detectors which take the necessary data automatically.\n"
  "\n"
  "For the intensity dependent nonlinearity, the SOF must contain:\n"
  "\n"
  " - At least five pairs of cold darks and flatfield exposures with different DIT.\n"
  " - The DO classification of the cold darks is DARK.\n"
  " - The DO classification of the flatfield exposures is FLAT.\n"
  "\n"
  "For the flux dependent nonlinearity, the SOF must contain (all with the same DIT):\n"
  "\n"
  " - The cold dark exposure, DO classification DARK.\n"
  " - A flatfield exposure with intensities going upt to 1/2 of full well capacity, DO classification FLAT. The template achieves this by using a polarizer.\n"
  " - A flatfield exposure with shutter 1 open and all other shutters closed, DO classification FLAT.\n"
  " - A flatfield exposure with shutter 2 open and all other shutters closed, DO classification FLAT.\n"
  " - A flatfield exposure with shutter 3 open and all other shutters closed, DO classification FLAT.\n"
  " - A flatfield exposure with shutter 4 open and all other shutters closed, DO classification FLAT.\n"
  " - A flatfield exposure with shutter 1 and 3 open and shutter 2 and 4 closed, DO classification FLAT.\n"
  " - A flatfield exposure with shutter 1, 2 and 3 open and shutter 4 closed, DO classification FLAT.\n"
  " - A flatfield exposure with all shutters open, DO classification FLAT.\n"
  "\n"
  "This plugin requires that the DO classification contains a substring like DARK or FLAT. This means "
  "that a DO classification like \"DARK\", \"CAL_DARK\" or \"DARK_SIDE_OF_THE_MOON\" is interpreted as DARK.\n"
  "\n"
  "The static calibration maps cover the whole detector, but due to the partial "
  "illumination of the detector, the flat field map contains the default (1.0) "
  "value for all not illuminated pixels. The nonlinearity coefficients (intensity "
  "dependent nonlinearity) are set to neutral values for these pixels. The global "
  "gain (conversion factor from electrons to DU) of a detector and the detector "
  "channel specific gain is calculated which are used later on for the scientific "
  "data reduction.\n"
  "\n"
  "Since the illumination of the pixels is crucial for the quality of the static "
  "calibration maps, the current of the infrared lamp must be selected with care:\n"
  "\n"
  " - For the HAWAII-2RG detector, the lamp current should be set to 4.5 A and the pinhole removed. The DIT values and instrument setup used by the MATISSE_gen_cal_det_L_SLOW and MATISSE_gen_cal_det_L_FAST OBs are adjusted to this lamp setup.\n"
  " - For the Aquarius detector, the nonlinearity map is valid for a specific DIT only. The infrared lamp current must be set manually following the procedure:\n"
  "   1. Set the Aquarius readout mode to SCI-HIGH-GAIN or SCI-LOW-GAIN.\n"
  "   2. Select the DIT.\n"
  "   3. Setup the instrument to:\n"
  "      INS SFN = DIAPH\n"
  "      INS PIN = PHOTO\n"
  "      INS FIN = OPEN\n"
  "      INS PON = DARK\n"
  "      INS DIN = HIGH\n"
  "   4. Wait until the image in the RTD is stable and take it as reference image (cold dark).\n"
  "   5. Open the polarizer wheel:\n"
  "      INS PON = OPEN\n"
  "   6. Subtract the reference from the image.\n"
  "   7. Adjust the infrared lamp current so that the brightest pixels have an intensity (dark subtracted!) of about 45000 DU.\n"
  "\n"
  "The process for creating the bad pixel map, flatfield map and nonlinearity map can be controlled to get different kinds "
  "of these static calibration maps. This is controlled using the three options --bptype, --fftype and --nltype:\n"
  " --bptype=0     This means the standard bad pixel map is created.\n"
  " --bptype=1     The bad pixel map is empty, no pixel is marked as bad.\n"
  " --bptype=2     A merged bad pixel map is created by using the standard map and\n"
  "                additionally given maps (DO classification BADPIX in the SOF file).\n"
  "                A pixel is a bad pixel if it is in about 25 percent of the maps marked as bad.\n"
  "                All added bad pixel maps should be created with --bptype=0.\n"
  "\n"
  " --fftype=0     This means the standard flatfield map is created.\n"
  " --fftype=1     The flatfield map contains a constant gain of 1.0.\n"
  " --fftype=2     An average flatfield map is created by using the standard map and\n"
  "                additionally given maps (DO classification FLATFIELD in the SOF file).\n"
  "                All added flatfield maps should be created with --fftype=0.\n"
  "\n"
  " --nltype=0     This means an individual nonlinearity response for each pixel is created (default for HAWAII-2RG).\n"
  " --ntlype=1     The nonlinearity map represents a strict linear behaviour.\n"
  " --nltype=2     An average nonlinearity behaviour is calculated based in the individual response (default for Aquarius).\n"
  "\n"
  "Using --bptype=2 and/or --fftype=2 needs to manually create a SOF in order to add the already existing badpixel and/or flatfield maps."
  "\n"
  "This plugin will create three static calibration files.\n"
  "\n"
  "The first FITS file (PRO.CATG = BADPIX) contains the bad pixel map "
  "for a specific detector. This detector is identified by the keywords "
  "in the primary header which are copied from the first used raw data file "
  "(the FITS file with the cold dark frames for the shortest exposure time).\n"
  "\n"
  " - Primary HDU : It contains the bad pixel map as an 8-bit image covering the whole detector. Good pixels are marked by a 0, bad pixels are identified by a 1 (binary image).\n"
  "\n"
  "The bad pixel map contains also the corresponding QC1 parameters.\n"
  "\n"
  "The second FITS file (PRO.CATG = FLATFIELD) contains the detector "
  "flatfield map for a specific detector. This flatfield is different "
  "from the observation specific flatfield, since the inhomogeneous "
  "illumination is compensated. The detector is identified by the keywords "
  "in the primary header which are copied from the first used raw data file "
  "(the FITS file with the cold dark frames for the shortest exposure time).\n"
  "\n"
  " - Primary HDU : It contains the flatfield map as an 32-bit floating point image covering the whole detector.\n"
  " - FFSTDERR : The standard error map (one value for each pixel) is stored in an image extension.\n"
  "\n"
  "For some pixels no valid flatfield value could be calculated due to "
  "a lack of sufficient illumination. For them, the flatfield value is set "
  "to 1.0 and the standard error to 0.0. The FITS file will "
  "contain the QC1 parameters describing the conversion factor between electrons "
  "and DU for the whole detector and for each detector channel. "
  "These values will be copied into the observation specific flatfield "
  "(OBS_FLATFIELD) by the mat_est_flat plugin. The mat_cal_image plugin "
  "will then use the conversion gain to convert the pixel values from DU "
  "into electrons.\n"
  "\n"
  "The third FITS file (PRO.CATG = NONLINEARITY) contains the detector "
  "specific nonlinearity map consisting of the function coefficients "
  "for each pixel or a global nonlinearity compensation polyline. "
  "The detector is identified by the keywords in the primary header "
  "which are copied from the first used raw data file (the FITS file "
  "with the cold dark frames for the shortest exposure time).\n"
  "\n"
  " - Primary HDU : It contains an image with the nonlinearity compensation limits for all pixels. They describe the intensity until for a given pixel the nonlinearity compensation works well due to the available data.\n"
  " - NLSTDERR : The quality of the compensation is specified as a standard error for each pixel. It describes the relative deviation of the function fit/global polyline and the individual pixel data.\n"
  " - NLHIST : This is a 2-d histogram showing the relation between measured intensity (x-axis) and calibrated intensity (y-axis) for all pixels.\n"
  " - NLCOEFF : This is an image cube containing the nonlinearity function coefficients for all pixels. If the illumination of a pixel prohibits the fitting of a function, the coefficients are set to neutral values. This extension does exist in intensity dependent nonlinearity maps (HAWAII-2RG).\n"
  " - NLRCMAP : This is a vector describing the global nonlinearity compensation polyline. Each value describes the calibrated intensity for a measured intensity which is 64 times the vector index (starting at 0). This extension does exist in flux dependent nonlinearity maps only (Aquarius).\n"
  "\n"
  "The FITS file will contain the same QC1 parameters describing the conversion "
  "factor between electrons and DU for the whole detector and for each detector "
  "channel as the flatfield map.\n"
  "\n"
  "Since the nonlinearity map for the HAWAII-II detector (LM-Band) describes the nonlinearity response for each individual pixel, "
  "the usable range depends on the maximum illumination on a pixel. It is possible to estimate an average nonlinearity response "
  "from all pixels and use this result for all pixels. This will lead to a nonlinearity compensation valid for all pixels even "
  "if they are not illuminated well. The plugin option --nltype allows selecting between the standard nonlinearity estimation "
  "for each pixel (default value 0) or estimating an average (global) nonlinearity response (value 1). The use of the global "
  "nonlinearity response will also lead to a very high usable intensity range, equal for all pixels.\n"
  "\n"
  "The following QC1 parameters will be calculated and stored in a cpl_propertylist and the primary HDU of the result FITS file:\n"
  "\n"
  " - QC DET<i> GAIN<p> : The global detector gain (in electrons per ADU, specific for a readout mode) will be used for scientific purposes.\n"
  " - QC DET<i> RON<p> : The global read-out noise (in electrons).\n"
  " - QC DET<i> NL LAW : The function used to describe the individual and average nonlinearity response.\n"
  " - QC DET<i> NL NCOEFF : The number of function coefficients (same as planes in the NLCOEFF image cube).\n"
  " - QC DET<i> NL A .. F : The coefficients describing the average nonlinearity response.\n"
  " - GC DET<i> HACORR<p> : The Autocorrelation coefficient between two horizontally adjacent pixels.\n"
  " - GC DET<i> VACORR<p> : The Autocorrelation coefficient between two vertically adjacent pixels.\n"
  " - QC DET<i> CHANNEL<j> GAIN<p> : The detector channel specific gain (in electrons per ADU, specific for a readout mode) will be used to detect detector and read-out electronics problems.\n"
  " - QC DET<i> CHANNEL<j> OFFSET<p> : The offset for the detector channels.\n"
  " - QC DET<i> CHANNEL<j> RON<p> : The read-out noise is given for each detector channel.\n"
  " - QC DET<i> FFM STDEV<p> : The standard deviation of the computed flatfield map (the mean is normalized to 1.0).\n"
  " - QC DET<i> GOODPIX : The number of good pixels flagged by a 1 in the bad pixel map.\n"
  " - QC DET<i> BADPIX : The number of bad pixels flagged by a 0 in the bad pixel map.\n"
  " - QC DET<i> GOODPIXELRATIO : The ratio good over total pixels.\n"
  "\n"
  "Where i is the detector number (1 = L/M-Band, 2 = N-Band), "
  "j is the detector channel number (1 .. 32 for the L/M-Band, 1 .. 64 for the N-Band), "
  "and p is the readout mode (HAWAII-2RG: 1 = slow readout, 2 = fast readout, "
  "Aquarius: 1 = low gain mode, 2 = high gain mode). The detector number "
  "(used for DET<i>) will be always 1 for the L/M-Band detector and 2 for "
  "the N-Band detector. In addition, the DET<i> CHIP1 ID keyword allows to "
  "distinguish between different detector chips. This is necessary, to deal "
  "with a detector change after, for example, an instrument repair. This will "
  "also result in a bad pixel and flatfield map which is only valid for "
  "a specific detector chip.\n"
  "\n"
  "Input files:\n"
  "\n"
  "  DO category:      Explanation:            Required:\n"
  "  DARK              Raw cold frames         Yes\n"
  "  FLAT              Raw exposed frames      Yes\n"
  "\n"
  "Output files:\n"
  "\n"
  "  DO category:      Explanation:\n"
  "  BADPIX            Static bad pixel map\n"
  "  FLATFIELD         Static flatfield map\n"
  "  NONLINEARITY      Static nonlinearity map\n"
  "\n"
  "It is possible to show some information during raw data processing using the --expert option. "
  "This is a set of flags each controls a certain type of additional information. This option "
  "does not change the behaviour of the plugin!:\n"
  "\n"
  "  1 a FITS file is created which contains the pixel statistics, the artificial darks and flats, 2-d histograms for the nonlinearity and conversion gain, ... , as an image cube\n"
  "  2 show details on the raw dark and flatfield exposures\n"
  "  4 show details on the artificial dark image and export details as STL files\n"
  "  8 show details on the artificial flatfield image and export details as STL files\n"
  " 16 show information needed to assess the nonlinearity estimation\n"
  " 32 show information needed to assess the conversion factor estimation\n"
  " 64 show information needed to assess the flatfield estimation\n"
;

/*-----------------------------------------------------------------------------
  Functions code
  -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
   @brief Build the list of available plugins, for this module.
   @param list the plugin list
   @return 0 if everything is ok, -1 otherwise
   Create the recipe instance and make it available to the application using the
   interface. This function is exported.
*/
/*----------------------------------------------------------------------------*/
int cpl_plugin_get_info(cpl_pluginlist * list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe );
  cpl_plugin *plugin = &recipe->interface;

  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_cal_det",
		  "Calculates a bad pixel map, a flatfield map and a non linearity map",
		  mat_cal_det_description,
		  "Matthias Heininger",
		  "mhein@mpifr-bonn.mpg.de",
		  MATISSE_LICENCE,
		  mat_cal_det_create,
		  mat_cal_det_exec,
		  mat_cal_det_destroy);
  cpl_pluginlist_append(list, plugin);
  return 0;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Setup the recipe options
   @param plugin the plugin
   @return 0 if everything is ok
   Defining the command-line/configuration parameters for the recipe.
*/
/*----------------------------------------------------------------------------*/
static int mat_cal_det_create(cpl_plugin * plugin)
{
  cpl_recipe    *recipe;
  cpl_parameter *p;

  /* Check that the plugin is part of a valid recipe */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  /* Create the parameters list in the cpl_recipe object */
  recipe->parameters = cpl_parameterlist_new();
  /* Fill the parameters list */

  /* --gain */
  p = cpl_parameter_new_range("matisse.mat_cal_det.gain",
			      CPL_TYPE_DOUBLE,
			      "Default conversion gain in [e-/DU].",
			      "matisse.mat_cal_det",
			      0.0, 0.0, 1e5);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "gain");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --nditskip */
  p = cpl_parameter_new_range("matisse.mat_cal_det.nditskip",
			      CPL_TYPE_INT,
			      "number of skipped frames",
			      "matisse.mat_cal_det",
			      0, 0, 20);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "nditskip");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --cosmics */
  p = cpl_parameter_new_range("matisse.mat_cal_det.cosmics",
			      CPL_TYPE_INT,
			      "flag if cosmics should be detected",
			      "matisse.mat_cal_det",
			      0, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cosmics");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --darklimit */
  p = cpl_parameter_new_range("matisse.mat_cal_det.darklimit",
			      CPL_TYPE_DOUBLE,
			      "Absolute limit used for darks: good = [-limit ... +limit]",
			      "matisse.mat_cal_det",
			      100.0, 0.0, 1e6);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "darklimit");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --flatlimit */
  p = cpl_parameter_new_range("matisse.mat_cal_det.flatlimit",
			      CPL_TYPE_DOUBLE,
			      "Relative limit used for flats: good = [ref*(1 - limit) ... ref*(1 + limit)]",
			      "matisse.mat_cal_det",
			      0.2, 0.0, 1.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "flatlimit");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --min_linear_range */
  p = cpl_parameter_new_range("matisse.mat_cal_det.min_linear_range",
			      CPL_TYPE_DOUBLE,
			      "Threshold for flatfield and non-linearity map calculation: LOW < threshold <= LINEAR",
			      "matisse.mat_cal_det",
			      100.0, 0.0, 1000000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "min_linear_range");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --max_linear_range */
  p = cpl_parameter_new_range("matisse.mat_cal_det.max_linear_range",
			      CPL_TYPE_DOUBLE,
			      "Threshold for flatfield and non-linearity map calculation: LINEAR < threshold <= NON_LINEAR",
			      "matisse.mat_cal_det",
			      15000.0, 0.0, 1000000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "max_linear_range");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --max_nonlinear_range */
  p = cpl_parameter_new_range("matisse.mat_cal_det.max_nonlinear_range",
			      CPL_TYPE_DOUBLE,
			      "Threshold for flatfield and non-linearity map calculation: NON_LINEAR < threshold <= SATURATED",
			      "matisse.mat_cal_det",
			      30000.0, 0.0, 1000000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "max_nonlinear_range");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --max_rel_deviation */
  p = cpl_parameter_new_range("matisse.mat_cal_det.max_rel_deviation",
			      CPL_TYPE_DOUBLE,
			      "Maximum relative deviation for the nonlinearity fit.",
			      "matisse.mat_cal_det",
			      0.01, 0.0, 1.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "max_rel_deviation");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --max_abs_deviation */
  p = cpl_parameter_new_range("matisse.mat_cal_det.max_abs_deviation",
			      CPL_TYPE_DOUBLE,
			      "Maximum absolute deviation for the nonlinearity fit [e-].",
			      "matisse.mat_cal_det",
			      10.0, 0.0, 1e5);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "max_abs_deviation");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --compensate */
  p = cpl_parameter_new_value("matisse.mat_cal_det.compensate",
			      CPL_TYPE_STRING,
			      "Defines which kind of compensation should be applied "
			      "(none = no compensation at all, all = all compensations possible, dd = detector specific defaults, "
			      "pb = subtract pixel bias, gb = subtract global bias, cb = subtract detector channel bias, rb = subtract row bias, "
			      "ct = subtract crosstalk)",
			      "matisse.mat_cal_det",
			      "pb,ct");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "compensate") ;
  cpl_parameterlist_append(recipe->parameters, p) ; 
 
  /* --nlstart */
  p = cpl_parameter_new_range("matisse.mat_cal_det.nlstart",
			      CPL_TYPE_INT,
			      "first nonlinearity mapping sampling point",
			      "matisse.mat_cal_det",
			      1024, 0, 8*1024);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "nlstart");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --nlstep */
  p = cpl_parameter_new_range("matisse.mat_cal_det.nlstep",
			      CPL_TYPE_INT,
			      "distance between two nonlinearity mapping sampling points",
			      "matisse.mat_cal_det",
			      64, 32, 1024);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "nlstep");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --bptype */
  p = cpl_parameter_new_range("matisse.mat_cal_det.bptype",
			      CPL_TYPE_INT,
			      "type of the bad pixel in the result file (0=standard, 1=empty, 2=merge)",
			      "matisse.mat_cal_det",
			      STD_BP_TYPE, STD_BP_TYPE, MERGE_BP_TYPE);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "bptype");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --fftype */
  p = cpl_parameter_new_range("matisse.mat_cal_det.fftype",
			      CPL_TYPE_INT,
			      "type of the flatfield in the result file (0=standard, 1=const, 2=average)",
			      "matisse.mat_cal_det",
			      STD_FF_TYPE, STD_FF_TYPE, AVERAGE_FF_TYPE);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fftype");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --nltype */
  p = cpl_parameter_new_range("matisse.mat_cal_det.nltype",
			      CPL_TYPE_INT,
			      "type of the nonlinearity in the result file (0=standard, 1=linear, 2=average)",
			      "matisse.mat_cal_det",
			      STD_NL_TYPE, STD_NL_TYPE, AVERAGE_NL_TYPE);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "nltype");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --poi */
  p = cpl_parameter_new_value("matisse.mat_cal_det.poi",
			      CPL_TYPE_STRING,
			      "pixel of interest: <x>,<y>",
			      "matisse.mat_cal_det",
			      "0,0");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "poi");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --expert */
  p = cpl_parameter_new_range("matisse.mat_cal_det.expert",
			      CPL_TYPE_INT,
			      "expert flag, shows additional information",
			      "matisse.mat_cal_det",
			      0, 0, 65535);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "expert");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --nt */
  p = cpl_parameter_new_value("matisse.mat_cal_det.nt",
			      CPL_TYPE_BOOL,
			      "Flag if a new method should be used for detector calibration.",
			      "matisse.mat_cal_det",
			      TRUE);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "nt");
  cpl_parameterlist_append(recipe->parameters, p);

  /* Return */
  return 0;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Execute the plugin instance given by the interface
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_cal_det_exec(cpl_plugin * plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  return mat_cal_det(recipe->parameters, recipe->frames);
}

/*----------------------------------------------------------------------------*/
/**
   @brief Destroy what has been created by the create function
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_cal_det_destroy(cpl_plugin * plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  cpl_parameterlist_delete(recipe->parameters);
  return 0;
}

static void mat_extract_image_poly0_local(cpl_image    *dst,
					  cpl_image    *src,
					  cpl_image    *range,
					  int           src_filter,
					  int           dst_filter,
					  int           lwidth,
					  int           lheight);

static void mat_extract_channel_poly0_global(mat_detector *det,
					     cpl_image    *dst,
					     cpl_image    *src,
					     cpl_image    *range,
					     int           src_filter,
					     int           dst_filter);

//static void mat_extract_channel_common_mode_h2(cpl_image    *dst,cpl_image    *src);
//static void mat_extract_channel_common_mode_aq(cpl_image    *dst, cpl_image    *src);

#ifdef DEVELOP
static int mat_compare_double(const void *e1, const void *e2)
{
  const double *d1 = (const double *)e1;
  const double *d2 = (const double *)e2;
  if (*d1 < *d2) return -1;
  if (*d1 > *d2) return +1;
  return 0;
}
#endif

static int mat_compare_intensity(const void *p1, const void *p2)
{
  const mat_intensity_info *i1 = (const mat_intensity_info *)p1;
  const mat_intensity_info *i2 = (const mat_intensity_info *)p2;
  if (i1->i1234 < i2->i1234) return -1;
  if (i1->i1234 > i2->i1234) return +1;
  return 0;
}

static int mat_compare_psm(const void *p1, const void *p2)
{
  const mat_psm_info *psm1 = (const mat_psm_info *)p1;
  const mat_psm_info *psm2 = (const mat_psm_info *)p2;
  if (psm1->dit < psm2->dit) return -1;
  if (psm1->dit > psm2->dit) return +1;
  return 0;
}

static double mat_nlm_flux_map_direct(mat_cal_det_info *info, double raw)
{
  // old implementation uses a resolution of 1 DU
  /*
  int r = (int)round(raw);

  if (r < 0) return raw;
  if (r >= 65536) return raw;
  return info->nonlinmap[r];
  */
  // new implementation includes linear interpolation, but is still faster than direct evaluation
  if ((raw < 0.0) || (raw > 65535.0))
    {
      return raw;
    }
  else
    {
      int    rcidx = (int)floor(raw);
      double frac  = raw - (double)rcidx;
      double lcal  = info->nonlinmap[rcidx];
      double rcal  = info->nonlinmap[rcidx+1];		  
      return (lcal*(1.0 - frac) + rcal*frac);
    }
}

/*
  f[x_] := a*x + b*x*Exp[(x - c)*d]
  =>
  D[f[x], x] = a + b E^(d (-c + x)) + b d E^(d (-c + x)) x
  D[f[x], a] = x
  D[f[x], b] =      E^(d (-c + x)) x
  D[f[x], c] = -b d E^(d (-c + x)) x
  D[f[x], d] =    b E^(d (-c + x)) x (-c + x)
  =>
  D[f[x], x] for x=0.0 -> a + b E^(-d c)
 */
static double mat_nlm_func(double x, double a0, double a1, double a2, double a3)
{
  return a0*x + a1*x*exp((x - a2)*a3);
}

static int mat_fit_nonlin_h2_f(const double x[], const double a[], double *result)
{
  // f[x_] := a*x + b*x*Exp[(x - c)*d]
  *result = a[0]*x[0] + a[1]*x[0]*exp((x[0] - a[2])*a[3]);
  return 0;
}
static int mat_fit_nonlin_h2_dfda(const double x[], const double a[], double result[])
{
  // D[f[x], a] = x
  // D[f[x], b] =      E^(d (-c + x)) x
  // D[f[x], c] = -b d E^(d (-c + x)) x
  // D[f[x], d] =    b E^(d (-c + x)) x (-c + x)
  double t1 = exp(a[3]*(-a[2] + x[0]))*x[0];  // E^(d (-c + x)) x
  result[0] =          x[0];
  result[1] =            t1;
  result[2] = -a[1]*a[3]*t1;
  result[3] =       a[1]*t1*(-a[2] + x[0]);
  return 0;
}


static void mat_info_init(mat_cal_det_info *info)
{
  int           i;
  mat_psm_index idx;

  info->use_defaults = 0;
  info->compensate = NO_COMPENSATION;
  info->hasphoto = 0;
  info->det = mat_detector_new();
  if (info->det == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for a mat_detector");
      return;
    }
  info->keywords = NULL;
  info->imgdet   = NULL;
  info->nbdit = 0;
  for (i = 0; i < MAX_NBDIT; i++)
    {
      info->psm[i].dit = 0.0;
      for (idx = DARK_IDX; idx <= FLAT_IDX; idx++)
	{
	  info->psm[i].fname[idx]  = NULL;
	  info->psm[i].rmed[idx]   = NULL;
	  info->psm[i].rvar[idx]   = NULL;
	  info->psm[i].cmed[idx]   = NULL;
	  info->psm[i].cvar[idx]   = NULL;
	  info->psm[i].hacorr[idx] = 0.0;
	  info->psm[i].vacorr[idx] = 0.0;
	}
      info->psm[i].poidarks = NULL;
      info->psm[i].poiflats = NULL;
    }
  info->darkmedraw    = NULL;
  info->darkmedref    = NULL;
  info->darkmedres    = NULL;
  info->darkvarraw    = NULL;
  info->darkrange     = NULL;
  info->dcoffset      = NULL;
  info->dcslope       = NULL;
  info->flatmedraw    = NULL;
  info->flatmedref    = NULL;
  info->flatmedres    = NULL;
  info->flatvarraw    = NULL;
  info->flatrange     = NULL;
  info->flatbpm       = NULL;
  info->gainhistraw   = NULL;
  info->gainhistcal   = NULL;
  info->nbintensities = 0;
  info->intensities   = NULL;
  info->stat1         = NULL;
  info->stat2         = NULL;
  info->stat3         = NULL;
  info->nl_nbentries  = 0;
  info->nl_map        = NULL;
  info->nbpoly        = 0;
  info->polyraw       = NULL;
  info->polycal       = NULL;
  info->polycalsmooth = NULL;
  info->nonlinmap     = NULL;
  info->ditscale      = NULL;
  info->nlhistraw     = NULL;
  info->nlhistcal     = NULL;
  info->nlcount       = NULL;
  info->bpm           = NULL;
  info->nlm           = NULL;
  info->ffm           = NULL;
  info->hacstat       = NULL;
  info->vacstat       = NULL;
  for (i = 0; i < 64; i++) info->aqpcs[i] = 1.0;
}

static void mat_info_delete(mat_cal_det_info *info)
{
  int   i;

  if (info->det != NULL)
    {
      mat_detector_delete(info->det);
      info->det = NULL;
    }
  if (info->keywords != NULL)
    {
      cpl_propertylist_delete(info->keywords);
      info->keywords = NULL;
    }
  if (info->imgdet != NULL)
    {
      mat_imagingdetector_delete(info->imgdet);
      info->imgdet = NULL;
    }
  info->nbdit = 0;
  for (i = 0; i < MAX_NBDIT; i++)
    {
      mat_psm_index idx;
      for (idx = DARK_IDX; idx <= FLAT_IDX; idx++)
	{
	  if (info->psm[i].fname[idx] != NULL)
	    {
	      cpl_free(info->psm[i].fname[idx]);
	      info->psm[i].fname[idx] = NULL;
	    }
	  if (info->psm[i].rmed[idx] != NULL)
	    {
	      cpl_image_delete(info->psm[i].rmed[idx]);
	      info->psm[i].rmed[idx] = NULL;
	    }
	  if (info->psm[i].rvar[idx] != NULL)
	    {
	      cpl_image_delete(info->psm[i].rvar[idx]);
	      info->psm[i].rvar[idx] = NULL;
	    }
	  if (info->psm[i].cmed[idx] != NULL)
	    {
	      cpl_image_delete(info->psm[i].cmed[idx]);
	      info->psm[i].cmed[idx] = NULL;
	    }
	  if (info->psm[i].cvar[idx] != NULL)
	    {
	      cpl_image_delete(info->psm[i].cvar[idx]);
	      info->psm[i].cvar[idx] = NULL;
	    }
	}
      if (info->psm[i].poidarks != NULL)
	{
	  cpl_vector_delete(info->psm[i].poidarks);
	  info->psm[i].poidarks = NULL;
	}
      if (info->psm[i].poiflats != NULL)
	{
	  cpl_vector_delete(info->psm[i].poiflats);
	  info->psm[i].poiflats = NULL;
	}
    }
  if (info->darkmedraw != NULL)
    {
      cpl_image_delete(info->darkmedraw);
      info->darkmedraw = NULL;
    }
  if (info->darkmedref != NULL)
    {
      cpl_image_delete(info->darkmedref);
      info->darkmedref = NULL;
    }
  if (info->darkmedres != NULL)
    {
      cpl_image_delete(info->darkmedres);
      info->darkmedres = NULL;
    }
  if (info->darkvarraw != NULL)
    {
      cpl_image_delete(info->darkvarraw);
      info->darkvarraw = NULL;
    }
  if (info->darkrange != NULL)
    {
      cpl_image_delete(info->darkrange);
      info->darkrange = NULL;
    }
  if (info->dcoffset != NULL)
    {
      cpl_image_delete(info->dcoffset);
      info->dcoffset = NULL;
    }
  if (info->dcslope != NULL)
    {
      cpl_image_delete(info->dcslope);
      info->dcslope = NULL;
    }
  if (info->flatmedraw != NULL)
    {
      cpl_image_delete(info->flatmedraw);
      info->flatmedraw = NULL;
    }
  if (info->flatmedref != NULL)
    {
      cpl_image_delete(info->flatmedref);
      info->flatmedref = NULL;
    }
  if (info->flatmedres != NULL)
    {
      cpl_image_delete(info->flatmedres);
      info->flatmedres = NULL;
    }
  if (info->flatvarraw != NULL)
    {
      cpl_image_delete(info->flatvarraw);
      info->flatvarraw = NULL;
    }
  if (info->flatrange != NULL)
    {
      cpl_image_delete(info->flatrange);
      info->flatrange = NULL;
    }
  if (info->flatbpm != NULL)
    {
      cpl_image_delete(info->flatbpm);
      info->flatbpm = NULL;
    }
  if (info->gainhistraw != NULL)
    {
      cpl_image_delete(info->gainhistraw);
      info->gainhistraw = NULL;
    }
  if (info->gainhistcal != NULL)
    {
      cpl_image_delete(info->gainhistcal);
      info->gainhistcal = NULL;
    }
  if (info->intensities != NULL)
    {
      cpl_free(info->intensities);
      info->intensities = NULL;
    }
  if (info->stat1 != NULL)
    {
      mat_statistics_delete(info->stat1);
      info->stat1 = NULL;
    }
  if (info->stat2 != NULL)
    {
      mat_statistics_delete(info->stat2);
      info->stat2 = NULL;
    }
  if (info->stat3 != NULL)
    {
      mat_statistics_delete(info->stat3);
      info->stat3 = NULL;
    }
  if (info->nl_map != NULL)
    {
      cpl_free(info->nl_map);
      info->nl_map = NULL;
    }
  if (info->polyraw != NULL)
    {
      cpl_free(info->polyraw);
      info->polyraw = NULL;
    }
  if (info->polycal != NULL)
    {
      cpl_free(info->polycal);
      info->polycal = NULL;
    }
  if (info->polycalsmooth != NULL)
    {
      cpl_free(info->polycalsmooth);
      info->polycalsmooth = NULL;
    }
  if (info->nonlinmap != NULL)
    {
      cpl_free(info->nonlinmap);
      info->nonlinmap = NULL;
    }
  if (info->ditscale != NULL)
    {
      cpl_image_delete(info->ditscale);
      info->ditscale = NULL;
    }
  if (info->nlhistraw != NULL)
    {
      cpl_image_delete(info->nlhistraw);
      info->nlhistraw = NULL;
    }
  if (info->nlhistcal != NULL)
    {
      cpl_image_delete(info->nlhistcal);
      info->nlhistcal = NULL;
    }
  if (info->nlcount != NULL)
    {
      cpl_image_delete(info->nlcount);
      info->nlcount = NULL;
    }
  if (info->bpm != NULL)
    {
      mat_badpixel_delete(info->bpm);
      info->bpm = NULL;
    }
  if (info->nlm != NULL)
    {
      mat_nonlinearity_delete(info->nlm);
      info->nlm = NULL;
    }
  if (info->ffm != NULL)
    {
      mat_flatfield_delete(info->ffm);
      info->ffm = NULL;
    }
  if (info->hacstat != NULL)
    {
      mat_statistics_delete(info->hacstat);
      info->hacstat = NULL;
    }
  if (info->vacstat != NULL)
    {
      mat_statistics_delete(info->vacstat);
      info->vacstat = NULL;
    }
}

static cpl_error_code mat_read_parameters(mat_cal_det_info *info, cpl_parameterlist *parlist)
{
  cpl_parameter    *param;

  /* --gain */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.gain");
  info->gain = cpl_parameter_get_double(param);
  /* --nditskip */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.nditskip");
  info->nditskip = cpl_parameter_get_int(param);
  /* --cosmics */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.cosmics");
  info->cosmics = cpl_parameter_get_int(param);
  /* --darklimit */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.darklimit");
  info->darklimit = cpl_parameter_get_double(param);
  /* --flatlimit */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.flatlimit");
  info->flatlimit = cpl_parameter_get_double(param);
  /* --min_linear_range */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.min_linear_range");
  info->min_linear_range = cpl_parameter_get_double(param);
  /* --max_linear_range */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.max_linear_range");
  info->max_linear_range = cpl_parameter_get_double(param);
  /* --max_nonlinear_range */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.max_nonlinear_range");
  info->max_nonlinear_range = cpl_parameter_get_double(param);
  /* --max_rel_deviation */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.max_rel_deviation");
  info->max_rel_deviation = cpl_parameter_get_double(param);
  /* --max_abs_deviation */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.max_abs_deviation");
  info->max_abs_deviation = cpl_parameter_get_double(param);
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.compensate");
  if (strcmp(cpl_parameter_get_string(param), "none") == 0) info->compensate = NO_COMPENSATION;
  if (strcmp(cpl_parameter_get_string(param), "all")  == 0) info->compensate = FULL_COMPENSATION;
  if (strcmp(cpl_parameter_get_string(param), "dd")  == 0) info->use_defaults = 1;
  if (strstr(cpl_parameter_get_string(param), "pb") != NULL) info->compensate |= PIXEL_BIAS_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "gb") != NULL) info->compensate |= GLOBAL_BIAS_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "cb") != NULL) info->compensate |= CHANNEL_BIAS_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "rb") != NULL) info->compensate |= HORIZONTAL_BIAS_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "ct") != NULL) info->compensate |= CROSSTALK_COMPENSATION;
  /* --nlstart */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.nlstart");
  info->nlstart = cpl_parameter_get_int(param);
  /* --nlstep */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.nlstep");
  info->nlstep = cpl_parameter_get_int(param);
  /* --bptype */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.bptype");
  info->bptype = cpl_parameter_get_int(param);
  /* --fftype */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.fftype");
  info->fftype = cpl_parameter_get_int(param);
  /* --nltype */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.nltype");
  info->nltype = cpl_parameter_get_int(param);
  /* --poi */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.poi");
  mat_parameter_get_int_double(param, "poi", &(info->poix), &(info->poiy));
  /* --expert */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.expert");
  info->expert = cpl_parameter_get_int(param);
  /* --nt */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_det.nt");
  info->ntflag = cpl_parameter_get_bool(param);
  return CPL_ERROR_NONE;
}

static void mat_select_detector_gain(mat_cal_det_info *info)
{
  // if no gain is given, use the detector specification from info->det to specify a suitable value
  if (info->gain != 0.0) return;
  switch (info->det->type)
    {
    case MAT_HAWAII2RG_DET:
      switch (info->det->read_id) {
      case 1:  // SCI-SLOW-SPEED
      case 3:  // TEC-CDS-RR-SO
      case 5:  // TEC-CDS-GR-SO
      case 7:  // TEC-SR-GR-SO
      case 9:  // TEC-RAMP-SO
	//info->gain = 2.73; // measured 20150427 in Nice
	info->gain = 2.85; // measured 20180130 at Paranal
	break;
      case 2:  // SCI-FAST-SPEED
      case 4:  // TEC-CDS-RR-FO
      case 6:  // TEC-CDS-GR-FO
      case 8:  // TEC-SR-GR-FO
      case 10: //TEC-RAMP-FO
	info->gain = 2.60; // measured 20150427 in Nice
	break;
      default:
	info->gain = 2.60;
      }
      break;
    case MAT_AQUARIUS_DET:
      switch (info->det->read_id) {
      case 1: //SCI-LOW-GAIN
      case 3: //TEC-SR-SH-LG
      case 5: //TEC-CDS-RR-LG
	info->gain = 190.0; // measured 20160523 in Nice
	break;
      case 2: //SCI-HIGH-GAIN
      case 4: //TEC-SR-SH-HG
      case 6: //TEC-CDS-RR-HG
	info->gain = 20.0;  // measured 20160523 in Nice
	break;
      default:
	info->gain = 20.0;
      }
      break;
    default:
      info->gain = 1.0;
    }
  cpl_msg_info(cpl_func, "Since no gain is given, %.2f e-/DU is selected (depends on detector and readout mode)", info->gain);
}

static cpl_error_code mat_store_statistics(mat_cal_det_info *info,
					   const char *fname,
					   cpl_parameterlist *parlist,
					   cpl_frameset *frameset)
{
  cpl_imagelist    *imglist = NULL;
  int               i;
  cpl_frame        *statframe = NULL;
  cpl_propertylist *plist = NULL;

  cpl_error_reset();
  cpl_ensure_code((info != NULL), CPL_ERROR_NULL_INPUT);
  /* create an empty image list and add all statistical data from the info */
  imglist = cpl_imagelist_new();
  for (i = 0; i < info->nbdit; i++)
    {
      mat_psm_index     idx;
      for (idx = DARK_IDX; idx <= FLAT_IDX; idx++)
	{
	  if (info->psm[i].rmed[idx] != NULL)
	    {
	      cpl_imagelist_set(imglist, info->psm[i].rmed[idx], cpl_imagelist_get_size(imglist));
	    }
	  if (info->psm[i].rvar[idx] != NULL)
	    {
	      cpl_imagelist_set(imglist, info->psm[i].rvar[idx], cpl_imagelist_get_size(imglist));
	    }
	}
    }
  if (info->darkmedraw != NULL)
    {
      cpl_imagelist_set(imglist, info->darkmedraw, cpl_imagelist_get_size(imglist));
    }
  if (info->darkmedref != NULL)
    {
      cpl_imagelist_set(imglist, info->darkmedref, cpl_imagelist_get_size(imglist));
    }
  if (info->darkmedres != NULL)
    {
      cpl_imagelist_set(imglist, info->darkmedres, cpl_imagelist_get_size(imglist));
    }
  if (info->darkvarraw != NULL)
    {
      cpl_imagelist_set(imglist, info->darkvarraw, cpl_imagelist_get_size(imglist));
    }
  if (info->darkrange != NULL)
    {
      cpl_imagelist_set(imglist, info->darkrange, cpl_imagelist_get_size(imglist));
    }
  if (info->dcoffset != NULL)
    {
      cpl_imagelist_set(imglist, info->dcoffset, cpl_imagelist_get_size(imglist));
    }
  if (info->dcslope != NULL)
    {
      cpl_imagelist_set(imglist, info->dcslope, cpl_imagelist_get_size(imglist));
    }
  if (info->flatmedraw != NULL)
    {
      cpl_imagelist_set(imglist, info->flatmedraw, cpl_imagelist_get_size(imglist));
    }
  if (info->flatmedref != NULL)
    {
      cpl_imagelist_set(imglist, info->flatmedref, cpl_imagelist_get_size(imglist));
    }
  if (info->flatmedres != NULL)
    {
      cpl_imagelist_set(imglist, info->flatmedres, cpl_imagelist_get_size(imglist));
    }
  if (info->flatvarraw != NULL)
    {
      cpl_imagelist_set(imglist, info->flatvarraw, cpl_imagelist_get_size(imglist));
    }
  if (info->flatrange != NULL)
    {
      cpl_imagelist_set(imglist, info->flatrange, cpl_imagelist_get_size(imglist));
    }
  if (info->flatbpm != NULL)
    {
      cpl_imagelist_set(imglist, info->flatbpm, cpl_imagelist_get_size(imglist));
    }
  if (info->ditscale != NULL)
    {
      cpl_imagelist_set(imglist, info->ditscale, cpl_imagelist_get_size(imglist));
    }
  if (info->nlhistraw != NULL)
    {
      cpl_imagelist_set(imglist, info->nlhistraw, cpl_imagelist_get_size(imglist));
    }
  if (info->nlhistcal != NULL)
    {
      cpl_imagelist_set(imglist, info->nlhistcal, cpl_imagelist_get_size(imglist));
    }
  if (info->nlcount != NULL)
    {
      cpl_imagelist_set(imglist, info->nlcount, cpl_imagelist_get_size(imglist));
    }
  if (info->gainhistraw != NULL)
    {
      cpl_imagelist_set(imglist, info->gainhistraw, cpl_imagelist_get_size(imglist));
    }
  if (info->gainhistcal != NULL)
    {
      cpl_imagelist_set(imglist, info->gainhistcal, cpl_imagelist_get_size(imglist));
    }
  /* Create product frame */
  statframe = cpl_frame_new();
  cpl_frame_set_filename(statframe, fname);
  cpl_frame_set_tag(statframe, MATISSE_STATISTICS_PROCATG);
  cpl_frame_set_type(statframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(statframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(statframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    /* remove (not delete!) all frames from the image list */
    cpl_imagelist_unwrap(imglist);
    cpl_frame_delete(statframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  if (cpl_dfs_setup_product_header(plist, statframe, frameset, parlist,
				   "mat_info_store_statistics", "MATISSE", "?Dictionary?", NULL) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_error_reset();
    cpl_propertylist_delete(plist);
    cpl_imagelist_unwrap(imglist);
    cpl_frame_delete(statframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  /* Save the file */
  if (cpl_imagelist_save(imglist, fname, CPL_TYPE_FLOAT, plist,
			 CPL_IO_CREATE) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_imagelist_unwrap(imglist);
    cpl_frame_delete(statframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  cpl_imagelist_unwrap(imglist);
  cpl_frame_delete(statframe);
  return CPL_ERROR_NONE;
}

static int mat_insert_statistics(mat_cal_det_info *info, double dit, mat_psm_index *idx, cpl_image *median, cpl_image *variance)
{
  int i;
  int found = -1;

  for (i = 0; i < info->nbdit; i++)
    {
      if (info->psm[i].dit != dit) continue;
      found = i;
      break;
    }
  if (found == -1)
    {
      if (info->nbdit == MAX_NBDIT)
	{
	  return -1;
	}
      else
	{
	  found = info->nbdit;
	  info->psm[found].dit = dit;
	  info->nbdit++;
	}
    }
  if (info->psm[found].rmed[*idx] != NULL)
    {
      cpl_image_delete(info->psm[found].rmed[*idx]);
      info->psm[found].rmed[*idx] = NULL;
    }
  info->psm[found].rmed[*idx] = cpl_image_duplicate(median);
  if (info->psm[found].cmed[*idx] != NULL)
    {
      cpl_image_delete(info->psm[found].cmed[*idx]);
      info->psm[found].cmed[*idx] = NULL;
    }
  info->psm[found].cmed[*idx] = cpl_image_duplicate(median);
  if (info->psm[found].rvar[*idx] != NULL)
    {
      cpl_image_delete(info->psm[found].rvar[*idx]);
      info->psm[found].rvar[*idx] = NULL;
    }
  info->psm[found].rvar[*idx] = cpl_image_duplicate(variance);
  if (info->psm[found].cvar[*idx] != NULL)
    {
      cpl_image_delete(info->psm[found].cvar[*idx]);
      info->psm[found].cvar[*idx] = NULL;
    }
  info->psm[found].cvar[*idx] = cpl_image_duplicate(variance);
  return found;
}

static cpl_error_code mat_calc_raw_statistics(mat_cal_det_info *info,
					      int isdark,
					      cpl_frame *frame)
{
  mat_gendata   *raw;
  mat_frame     *median;
  mat_frame     *variance;
  double         dit;
  int            rejected;
  mat_psm_index  pidx;
  int            didx, fidx;
  cpl_imagelist *imglist;
  double         hacorr;
  double         vacorr;

  // ***** Load the raw data into an internal data structure
  raw = mat_gendata_load_skip(frame, info->nditskip, CPL_TYPE_FLOAT);
  if (raw == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot read the FITS file");
      return CPL_ERROR_UNSPECIFIED;
    }
  /* 
     This plugin works only if the one and only region covers the whole detector!
     If the type of the detector is either a HAWAII-2RG or an Aquarius, the size
     of the sub-window must be 2048x2048 for the HAWAII-2RG or 1024x1024 for the Aquarius.
     If the detector type is unknown, the sub-window size of the first file is used.
  */
  // ***** If it is the first file (normally a DARK file), the detector specification is derived and it is checked that the data covers the whole detector.
  if (info->nbdit == 0)
    {
      // It is the first file -> get the detector specification and check it.
      mat_detector_decode_raw(info->det, raw);
      // Only on sub-window is allowed, if this is not the case, ignore this file.
      if (raw->imgdet->nbregion != 1)
	{
	  cpl_msg_error(cpl_func, "the file %s contains more than one region",
			cpl_frame_get_filename(frame));
	  mat_gendata_delete(raw);
	  return CPL_ERROR_INCOMPATIBLE_INPUT;
	}
      // The sub-window must cover the whole detector, if this is not the case, ignore this file.
      if ((info->det->nx != info->det->data_nx) || (info->det->ny != info->det->data_ny))
	{
	  cpl_msg_error(cpl_func,
			"the file %s does not cover the whole detector",
			cpl_frame_get_filename(frame));
	  mat_gendata_delete(raw);
	  return CPL_ERROR_INCOMPATIBLE_INPUT;
	}
      // it is the first file
      // -> load and keep the keywords from the primary header
      // -> load the imaging detector table for later use
      info->keywords = cpl_propertylist_load(cpl_frame_get_filename(frame), 0);
      info->imgdet = mat_imagingdetector_duplicate(raw->imgdet);
      info->nx = info->det->nx;
      info->ny = info->det->ny;
      if (info->det->type == MAT_AQUARIUS_DET)
	{
	  const char *str = cpl_propertylist_get_string(raw->keywords,"ESO INS PIN NAME");
	  info->timescale = 0.001;  // [ms]
	  // HIERARCH ESO INS PIN NAME = 'PHOTO   ' / OPTIi unique ID.
	  if (str != NULL)
	    {
	      if (strncmp(str, "PHOTO", 5) == 0) info->hasphoto = 1;
	    }
	  else
	    {
	      cpl_msg_info(cpl_func, "keyword HIERARCH ESO INS PIN NAME not found, assuming INTERF");
	      info->hasphoto = 0;
	    }
	}
      else
	{
	  const char *str = cpl_propertylist_get_string(raw->keywords,"ESO INS PIL NAME");
	  info->timescale = 1.0;    // [s]
	  // HIERARCH ESO INS PIL NAME = 'PHOTO   ' / OPTIi unique ID.
	  if (str != NULL)
	    {
	      if (strncmp(str, "PHOTO", 5) == 0) info->hasphoto = 1;
	    }
	  else
	    {
	      cpl_msg_info(cpl_func, "keyword HIERARCH ESO INS PIL NAME not found, assuming INTERF");
	      info->hasphoto = 0;
	    }
	}
    }
  // ***** Check that this file is compatible with the first file
  if (mat_detector_check_raw(info->det, raw) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func,
		    "the file %s contains data which is not compatible with the previously used data",
		    cpl_frame_get_filename(frame));
      mat_gendata_delete(raw);
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  // ***** Extract the DIT from the raw data (either from the primary header keyword or from IMAGING_DATA).
  dit = cpl_propertylist_get_double(raw->keywords, MATISSE_KW_EXPTIME);
  if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND)
    {
      // normally this FITS file is broken, but:
      // try to extract the exposure time from the IMAGING_DATA table (column EXPTIME)
      dit = raw->imgdata->list_frame[0]->exptime;
      cpl_error_reset();
    }
  // ***** Determine the pixel statistics (temporal median and variance)
  median = mat_frame_duplicate(raw->imgdata->list_frame[0], CPL_TYPE_DOUBLE);
  variance = mat_frame_duplicate(raw->imgdata->list_frame[0], CPL_TYPE_DOUBLE);
  if (mat_calc_statistics_cosmics(raw, median, variance, info->cosmics) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func, "Cannot calculate the pixel statistics for the file %s",
		    cpl_frame_get_filename(frame));
      mat_gendata_delete(raw);
      return CPL_ERROR_UNSPECIFIED;
    }
  if (info->hacstat == NULL)
    {
      info->hacstat = mat_statistics_realloc(info->hacstat, 1024, MAT_OUTLIER_FACTOR);
      if (info->hacstat == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the statistics (info->hacstat)");
	  return CPL_ERROR_UNSPECIFIED;
	}
      mat_statistics_reset(info->hacstat);
    }
  if (info->vacstat == NULL)
    {
      info->vacstat = mat_statistics_realloc(info->vacstat, 1024, MAT_OUTLIER_FACTOR);
      if (info->vacstat == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the statistics (info->vacstat)");
	  return CPL_ERROR_UNSPECIFIED;
	}
      mat_statistics_reset(info->vacstat);
    }
  imglist = cpl_imagelist_new();
  if (imglist != NULL)
    {
      int z, nz;
      nz = raw->imgdata->nbframe;
      for (z = 0; z < nz; z++)
	{
	  cpl_imagelist_set(imglist, raw->imgdata->list_frame[z]->list_subwin[0]->imgreg[0], z);
	}
      mat_calc_autocorr(median->list_subwin[0]->imgreg[0],
			imglist,
			NULL,
			info->det->channel_ncolumns,
			info->det->channel_nrows,
			&hacorr,
			&vacorr,
			info->hacstat,
			info->vacstat);
      cpl_imagelist_unwrap(imglist);
    }
  else
    {
      hacorr = 0.0;
      vacorr = 0.0;
    }
  if (isdark)
    { // pidx = PSM index, didx = DIT index, fidx = frame index
      pidx = DARK_IDX;
      didx = mat_insert_statistics(info, dit, &pidx, median->list_subwin[0]->imgreg[0], variance->list_subwin[0]->imgreg[0]);
      info->psm[didx].fname[pidx] = cpl_strdup(cpl_frame_get_filename(frame));
      if (info->psm[didx].poidarks == NULL)
	{
	  info->psm[didx].poidarks = cpl_vector_new(raw->imgdata->nbframe);
	}
      for (fidx = 0; fidx < raw->imgdata->nbframe; fidx++)
	{
	  cpl_image *img = raw->imgdata->list_frame[fidx]->list_subwin[0]->imgreg[0];
	  cpl_vector_set(info->psm[didx].poidarks, fidx, cpl_image_get(img, info->poix, info->poiy, &rejected));
	}
      info->psm[didx].hacorr[pidx] = hacorr;
      info->psm[didx].vacorr[pidx] = vacorr;
    }
  else
    {
      // detect the shutter combination of this exposure, default is SH1234_IDX (_not_ the start value of the variable!)
      if (info->det->type == MAT_AQUARIUS_DET)
	{
	  pidx = DARK_IDX;
	  if (cpl_propertylist_get_bool(raw->keywords,"ESO INS BSN1 ST")) pidx += SH1_IDX;
	  if (cpl_propertylist_get_bool(raw->keywords,"ESO INS BSN2 ST")) pidx += SH2_IDX;
	  if (cpl_propertylist_get_bool(raw->keywords,"ESO INS BSN3 ST")) pidx += SH3_IDX;
	  if (cpl_propertylist_get_bool(raw->keywords,"ESO INS BSN4 ST")) pidx += SH4_IDX;
	  if (strcmp(cpl_propertylist_get_string(raw->keywords, "ESO INS PON NAME"), "OPEN") != 0) pidx = FLAT_IDX;
	  if (pidx == DARK_IDX) pidx = SH1234_IDX; // if the keywords are not present, assume that all our shutters are open
	}
      else
	{
	  pidx = FLAT_IDX;
	}
      didx = mat_insert_statistics(info, dit, &pidx, median->list_subwin[0]->imgreg[0], variance->list_subwin[0]->imgreg[0]);
      info->psm[didx].fname[pidx] = cpl_strdup(cpl_frame_get_filename(frame));
      if (pidx == FLAT_IDX)
	{
	  if (info->psm[didx].poiflats == NULL)
	    {
	      info->psm[didx].poiflats = cpl_vector_new(raw->imgdata->nbframe);
	    }
	  for (fidx = 0; fidx < raw->imgdata->nbframe; fidx++)
	    {
	      cpl_image *img = raw->imgdata->list_frame[fidx]->list_subwin[0]->imgreg[0];
	      cpl_vector_set(info->psm[didx].poiflats, fidx, cpl_image_get(img, info->poix, info->poiy, &rejected));
	    }
	}
      info->psm[didx].hacorr[pidx] = hacorr;
      info->psm[didx].vacorr[pidx] = vacorr;
    }
  mat_frame_delete(median);   // the content was copied!
  mat_frame_delete(variance); // the content was copied!
  // ***** Free the memory for the raw data and return.
  mat_gendata_delete(raw);
  return CPL_ERROR_NONE;
}

static void mat_show_raw_info(mat_cal_det_info *info)
{
  int     didx, pidx;
  char   *mstr[] = {"DARK",
		    "SH1", "SH2", "SH12", "SH3",
		    "SH13", "SH23", "SH123", "SH4",
		    "SH14", "SH24", "SH124", "SH34",
		    "SH134", "SH234", "SH1234",
		    "FLAT"};

  if ((info->expert & RAW_EXPERT_FLAG) == 0) return;
  for (didx = 0; didx < info->nbdit; didx++)
    {
      for (pidx = DARK_IDX; pidx <= FLAT_IDX; pidx++)
	{
	  if (info->psm[didx].fname[pidx] == NULL) continue;
	  if ((info->poix != 0) && (info->poiy != 0))
	    {
	      int rejected;
	      double m = cpl_image_get(info->psm[didx].rmed[pidx], info->poix, info->poiy, &rejected);
	      double v = cpl_image_get(info->psm[didx].rvar[pidx], info->poix, info->poiy, &rejected);
	      cpl_msg_info(cpl_func, "Exposure %s dit %6.3f mode %s hacorr %.3f vacorr %.3f poi %7.1f %7.1f",
			   info->psm[didx].fname[pidx],
			   info->psm[didx].dit,
			   mstr[pidx],
			   info->psm[didx].hacorr[pidx],
			   info->psm[didx].vacorr[pidx],
			   m, v);
	    }
	  else
	    {
	      cpl_msg_info(cpl_func, "Exposure %s dit %.3f s mode %s hacorr %.3f vacorr %.3f",
			   info->psm[didx].fname[pidx],
			   info->psm[didx].dit,
			   mstr[pidx],
			   info->psm[didx].hacorr[pidx],
			   info->psm[didx].vacorr[pidx]);
	    }
	}
    }
}

static void mat_show_poi_statistics(mat_cal_det_info *info)
{
  int didx, rejected;

  if ((info->expert & RAW_EXPERT_FLAG) == 0) return;
  if ((info->poix == 0) || (info->poiy == 0)) return;
  for (didx = 0; didx < info->nbdit; didx++)
    {
      int fidx;
      cpl_msg_info(cpl_func, "# pixel at %d,%d DIT=%g avg %g %g",
		   info->poix, info->poiy, info->psm[didx].dit,
		   cpl_image_get(info->psm[didx].rmed[DARK_IDX], info->poix, info->poiy, &rejected),
		   cpl_image_get(info->psm[didx].rmed[FLAT_IDX], info->poix, info->poiy, &rejected));
      for (fidx = 0; fidx < CPL_MIN(cpl_vector_get_size(info->psm[didx].poidarks), cpl_vector_get_size(info->psm[didx].poiflats)); fidx++)
	{
	  cpl_msg_info(cpl_func, "%d %g %g",
		       fidx,
		       cpl_vector_get(info->psm[didx].poidarks, fidx),
		       cpl_vector_get(info->psm[didx].poiflats, fidx));
	}
    }
}

/*
static cpl_error_code mat_compensate_channel_crosstalk_aq(mat_cal_det_info *info, cpl_image *image)
{
  int      x, y, c;

  for (y = 1; y <= info->det->ny; y++)
    {
      for (x = 1; x <= info->det->channel_nx; x++)
	{
	  int    rejected;
	  int    l;
	  double v[32];
	  double b, o;
	  // get the 32 equivalent pixels (same position inside the 32 detector channels)
	  for (c = 0; c < 32; c++)
	    {
	      v[c] = cpl_image_get(image, x + c*info->det->channel_nx, y, &rejected);
	    }
	  // calculate the median of the background
	  // b = average value, ignoring the lowest 2 pixel intensities (bad pixels)
	  // 2.0*b = limit of the pixel intensity (used for crosstalk median)
	  qsort(v, 32, sizeof(double), mat_compare_double);
	  b = (v[2] + v[3] + v[4] + v[5])/4.0;
	  l = 0;
	  while ((l < 32) && (v[l] < 2.0*b)) l++;
	  if (l%2 == 0)
	    { // even number of intensities
	      o = 0.5*(v[l/2] + v[l/2 - 1]);
	    }
	  else
	    { // odd number of intensities
	      o = v[l/2];
	    }
	  // subtract the offset from all detector channels
	  for (c = 0; c < info->det->channel_ncolumns; c++)
	    {
	      cpl_image_set(image, x + c*32, y, cpl_image_get(image, x + c*32, y, &rejected) - o);
	    }
	}
    }
  return CPL_ERROR_NONE;
}
*/

static cpl_error_code mat_compensate_channel_crosstalk_aq(mat_cal_det_info *info, cpl_image *image)
{
  int         x, y, c;
  int         rejected;
  cpl_image  *ref_channel;
  cpl_image  *med_channel;

  ref_channel = cpl_image_new(info->det->channel_nx, info->det->channel_ny, CPL_TYPE_DOUBLE);
  if (ref_channel == NULL)
    {
      return CPL_ERROR_UNSPECIFIED;
    }
  med_channel = cpl_image_new(info->det->channel_nx, info->det->channel_ny, CPL_TYPE_DOUBLE);
  if (med_channel == NULL)
    {
      cpl_image_delete(ref_channel);
      return CPL_ERROR_UNSPECIFIED;
    }
  mat_image_fill(ref_channel, 0.0);
  mat_image_fill(med_channel, 0.0);
  // Set the _average_ of two reference pixels into the ref_channel image
  for (y = 1; y <= info->det->channel_ny; y++)
    {
      for (x = 1; x <= info->det->channel_nx; x++)
	{
	  double sum = 0.0;
	  sum += cpl_image_get(image, x, y,                     &rejected);
	  sum += cpl_image_get(image, x, info->det->ny - y + 1, &rejected);
	  cpl_image_set(ref_channel, x, y, sum/2.0);
	}
    }
  // estimate the median offset image
  mat_extract_image_poly0_local(med_channel, ref_channel, NULL, MAT_UNDEFINED_INTENSITY, MAT_UNDEFINED_INTENSITY, 3, 9);
  // subtract the crosstalk offset from all channels
  for (y = 1; y <= info->det->channel_ny; y++)
    {
      for (x = 1; x <= info->det->channel_nx; x++)
	{
	  double offset = cpl_image_get(med_channel, x, y, &rejected);
	  /*
	  if (info->expert >= 3)
	    {
	      cpl_msg_info(cpl_func, "crosstalk offset %d %d %g", x, y, offset);
	    }
	  */
	  for (c = 0; c < info->det->channel_ncolumns; c++)
	    {
	      int px = c*info->det->channel_nx + x;
	      int py = y;
	      cpl_image_set(image, px, py, cpl_image_get(image, px, py, &rejected) - offset);
	      py = info->det->ny - y + 1;
	      cpl_image_set(image, px, py, cpl_image_get(image, px, py, &rejected) - offset);
	    }
	}
    }
  cpl_image_delete(ref_channel);
  cpl_image_delete(med_channel);
  return CPL_ERROR_NONE;
}

/* Not used anymore since no 1/f noise must be compensated for the Aquarius detector
static cpl_error_code mat_compensate_horizontal_offset_aq(mat_cal_det_info *info, cpl_image *image)
{
  double                offset;
  int                   x, y;
  mat_statistics_info  *stat_info;

  stat_info = info->stat1 = mat_statistics_realloc(info->stat1, 2*info->det->channel_nx, 0.0);
  if (stat_info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "cannot allocate memory for mat_statistics_info");
      return CPL_ERROR_NULL_INPUT;
    }
  for (y = 1; y <= info->det->ny; y++)
    {
      mat_statistics_reset(stat_info);
      for (x = 1; x <= info->det->channel_nx; x++)
	{
	  int rejected;
	  double v = cpl_image_get(image, x, y, &rejected);
	  if (rejected) continue;
	  mat_statistics_add_value(stat_info, v);
	}
      for (x = info->det->nx - info->det->channel_nx + 1; x <= info->det->nx; x++)
	{
	  int rejected;
	  double v = cpl_image_get(image, x, y, &rejected);
	  if (rejected) continue;
	  mat_statistics_add_value(stat_info, v);
	}
      mat_statistics_calc(stat_info);
      offset = stat_info->average;
      for (x = 0; x <= info->det->nx; x++)
	{
	  int rejected;
	  cpl_image_set(image, x, y, cpl_image_get(image, x, y, &rejected) - offset);
	}
    }
  return CPL_ERROR_NONE;
}
*/

static cpl_error_code mat_read_and_process_data(mat_cal_det_info *info, cpl_frameset *frameset)
{
  cpl_frameset_iterator *it;
  cpl_frame             *rawframe;
  int                    i;

  cpl_msg_info(cpl_func, "Read all raw files (DO classification FLAT and DARK) and calculate the pixel statistics for each exposure.");
  // load all DARK frames and calculate the raw pixel statistics
  it = cpl_frameset_iterator_new(frameset);
  rawframe = cpl_frameset_iterator_get(it);
  while (rawframe != NULL)
    {
      // we use only calibrated target interferometric data
      if (strstr(cpl_frame_get_tag(rawframe), MATISSE_DO_DARK) != NULL)
	{ // we have found a *DARK* classified frame -> process it
	  cpl_error_code    ec;
	  cpl_msg_info(cpl_func, "loading and processing of %s", cpl_frame_get_filename(rawframe));
	  cpl_frame_set_group(rawframe, CPL_FRAME_GROUP_RAW);
	  ec = mat_calc_raw_statistics(info, 1, rawframe);
	  if (ec != CPL_ERROR_NONE)
	    {
	      cpl_frameset_iterator_delete(it);
	      return ec;
	    }
	}
      cpl_frameset_iterator_advance(it, 1);
      rawframe = cpl_frameset_iterator_get(it);
    }
  cpl_frameset_iterator_delete(it);
  if (info->nbdit == 0)
    {
      cpl_msg_error(cpl_func, "SOF does not contain valid dark files");
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  // load all FLAT frames and calculate the raw pixel statistics
  it = cpl_frameset_iterator_new(frameset);
  rawframe = cpl_frameset_iterator_get(it);
  while (rawframe != NULL)
    {
      // we use only calibrated target interferometric data
      if ((strstr(cpl_frame_get_tag(rawframe), MATISSE_DO_FLAT) != NULL) && (strcmp(cpl_frame_get_tag(rawframe), MATISSE_DO_FFM) != 0))
	{ // we have found a *FLAT* classified frame -> process it
	  cpl_error_code    ec;
	  cpl_msg_info(cpl_func, "loading and processing of %s", cpl_frame_get_filename(rawframe));
	  cpl_frame_set_group(rawframe, CPL_FRAME_GROUP_RAW);
	  ec = mat_calc_raw_statistics(info, 0, rawframe);
	  if (ec != CPL_ERROR_NONE)
	    {
	      cpl_frameset_iterator_delete(it);
	      return ec;
	    }
	}
      cpl_frameset_iterator_advance(it, 1);
      rawframe = cpl_frameset_iterator_get(it);
    }
  cpl_frameset_iterator_delete(it);
  // sort the pixel statistics with increasing DIT
  qsort(info->psm, info->nbdit, sizeof(mat_psm_info), mat_compare_psm);
  mat_show_raw_info(info);
  mat_statistics_calc(info->hacstat);
  mat_statistics_calc(info->vacstat);
  cpl_msg_info(cpl_func, "detector autocorrelation horizontal %.3f vertical %.3f", info->hacstat->average, info->vacstat->average);
  mat_show_poi_statistics(info);
  // compensate the flat statistics by subtracting the dark statistics
  // Apply a Aquarius specific channel crosstalk compensation
  if (info->use_defaults)
    {
      int compensate = PIXEL_BIAS_COMPENSATION;
      if (info->det->type == MAT_AQUARIUS_DET) compensate |= CROSSTALK_COMPENSATION;
      info->compensate = compensate;
      info->use_defaults = 0;
    }
  for (i = 0; i < info->nbdit; i++)
    {
      mat_psm_index  idx;
      for (idx = SH1_IDX; idx <= FLAT_IDX; idx++)
	{
	  if ((info->psm[i].rmed[idx] != NULL) && (info->psm[i].rmed[DARK_IDX] != NULL))
	    {
	      if (info->compensate & PIXEL_BIAS_COMPENSATION)
		{
		  cpl_image_subtract(info->psm[i].rmed[idx], info->psm[i].rmed[DARK_IDX]);
		}
	      if (info->compensate & CROSSTALK_COMPENSATION)
		{
		  if (info->det->type == MAT_AQUARIUS_DET)
		    {
		      mat_compensate_channel_crosstalk_aq(info, info->psm[i].rmed[idx]);
		      //mat_compensate_horizontal_offset_aq(info, info->psm[i].rmed[idx]);
		    }
		}
	    }
	  if ((info->psm[i].rvar[idx] != NULL) && (info->psm[i].rvar[DARK_IDX] != NULL))
	    {
	      if (info->compensate & PIXEL_BIAS_COMPENSATION)
		{
		  cpl_image_subtract(info->psm[i].rvar[idx], info->psm[i].rvar[DARK_IDX]);
		}
	    }
	}
    }
  return CPL_ERROR_NONE;
}

static void mat_extract_poly2_pip(mat_extract_info *einfo, int plength1, int plength2)
{
  int                  x, y;
  mat_smooth_info     *sinfo;

  sinfo = mat_smooth_info_new(einfo->nx, einfo->src_filter, einfo->dst_filter);
  if (sinfo == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot allocate space for smoothing operation.");
      return;
    }
  for (y = 0; y < einfo->ny; y++)
    { // iterate over all rows
      double peak             = 0.0;
      double valley           = 0.0;
      int    left_valley_idx  = 0;
      int    right_valley_idx = 0;
      int    nx;
      // Copy the original values and ranges to the temporary memory (MASKED, BADPIX, LINEAR)
      mat_smooth_set(sinfo, einfo->src, einfo->range, einfo->x0, einfo->y0 + y, einfo->nx, 1);
      if (sinfo->first == -1) continue;
      // Locally fit a second order polynome using up to plength1 values
      mat_smooth_polyn_sequence(sinfo, 2, plength1);
      // try to detect the borders between the photometric channels and the interferometric channel
      for (x = sinfo->first; x < sinfo->last; x++)
	{
	  peak = CPL_MAX(peak, sinfo->smooth[x]);
	}
      valley = peak;
      for (x = 240; x <= 290; x++)
	{
	  if (sinfo->smooth[x] < valley)
	    {
	      left_valley_idx = x;
	      valley = sinfo->smooth[x];
	    }
	}
      valley = peak;
      for (x = 700; x <= 800; x++)
	{
	  if (sinfo->smooth[x] < valley)
	    {
	      right_valley_idx = x;
	      valley = sinfo->smooth[x];
	    }
	}

      // smooth the left part with plength1
      nx = left_valley_idx - einfo->x0;
      mat_smooth_set(sinfo, einfo->src, einfo->range, einfo->x0, y, nx, 1);
      mat_smooth_polyn_sequence(sinfo, 2, plength1);
      mat_smooth_apply(sinfo, einfo->src, -1.0);
      mat_smooth_apply(sinfo, einfo->dst,  1.0);
      // smooth the central part with plength2
      nx = right_valley_idx - left_valley_idx + 1;
      mat_smooth_set(sinfo, einfo->src, einfo->range, left_valley_idx, y, nx, 1);
      mat_smooth_polyn_sequence(sinfo, 2, plength2);
      mat_smooth_apply(sinfo, einfo->src, -1.0);
      mat_smooth_apply(sinfo, einfo->dst,  1.0);
      // smooth the right part with plength1
      nx = einfo->x0 + einfo->nx - right_valley_idx;
      mat_smooth_set(sinfo, einfo->src, einfo->range, right_valley_idx + 1, y, nx, 1);
      mat_smooth_polyn_sequence(sinfo, 2, plength1);
      mat_smooth_apply(sinfo, einfo->src, -1.0);
      mat_smooth_apply(sinfo, einfo->dst,  1.0);
    }
  mat_smooth_info_delete(sinfo);
}

/*
static void mat_extract_channel(cpl_image *src, cpl_image *dst, int x, int y, int nx, int ny, mat_statistics_info *stat, double *spectrum, double *profile)
{
  int px, py;

  for (py = y; py < y + ny; py++)
    {
      mat_statistics_reset(stat);
      mat_statistics_set_region(stat, src, NULL, x, py, nx, 1);
      mat_statistics_calc(stat);
      spectrum[py] = stat->average;
      cpl_msg_info(cpl_func, "spectrum %d %f", py, spectrum[py]);
    }
  for (px = x; px < x + nx; px++)
    {
      mat_statistics_reset(stat);
      for (py = y; py < y + ny; py++)
	{
	  int rejected;
	  double value = cpl_image_get(src, px, py, &rejected);
	  mat_statistics_add_value(stat, value/spectrum[py]);
	}
      mat_statistics_calc(stat);
      profile[px] = stat->average;
      cpl_msg_info(cpl_func, "profile %d %f", px, profile[px]);
    }
  for (py = y; py < y + ny; py++)
    {
      for (px = x; px < x + nx; px++)
	{
	  int rejected;
	  cpl_image_set(src, px, py, cpl_image_get(src, px, py, &rejected) - spectrum[py]*profile[px]);
	  cpl_image_set(dst, px, py, cpl_image_get(dst, px, py, &rejected) + spectrum[py]*profile[px]);
	}
    }
}
*/

 /*
   It seems that for the Aquarius detector the first and last pixel of a 32-wide detector channel
   has a different gain (> 10 percent) than the other pixels. This function tries to estimate this
   by determining for each pixel row inside a specific detector column a smooth illumination and
   calculates the average scale between the original and smoothened values for the first and
   last pixel column.
   This scaling is then applied to compensate the artificial flatfield image. Later on this factor
   is multiplied into the flatfield for these pixel columns.
  */
static void mat_extract_aq_column_scale(mat_extract_info *einfo, double *aqpcs)
{
  mat_statistics_info *stat0  = NULL;
  mat_statistics_info *stat31 = NULL;
  mat_smooth_info     *sinfo;
  int                  x, y, dc, dr;

  // the first pixel column is about 5 percent darker than than normal
  stat0 = mat_statistics_new(64*512, MAT_OUTLIER_FACTOR);
  mat_statistics_set_use(stat0, MAT_USE_REDUCED_MEDIAN, MAT_USE_REDUCED_MEDVAR);
  // the last pixel column is about 2 percent brighter than normal
  stat31 = mat_statistics_new(64*512, MAT_OUTLIER_FACTOR);
  mat_statistics_set_use(stat31, MAT_USE_REDUCED_MEDIAN, MAT_USE_REDUCED_MEDVAR);
  sinfo = mat_smooth_info_new(32, MAT_UNDEFINED_INTENSITY, MAT_UNDEFINED_INTENSITY);
  if (sinfo == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot allocate space for smoothing operation.");
      mat_statistics_delete(stat0);
      mat_statistics_delete(stat31);
      return;
    }
  // Calculate a scale factor for each pixel column in each detector channel in order to estimate a global pixel scale factor (must go into the flatfield)
  mat_statistics_reset(stat0);
  mat_statistics_reset(stat31);
  for (dr = 0; dr < 2; dr++)
    {
      for (dc = 14; dc < 20; dc++)
	{
	  int x0   =           32*dc + 1;
	  int y0   =  512*dr         + 1;
	  for (y = 0; y < 512; y++)
	    {
	      mat_smooth_set(sinfo, einfo->src, einfo->range, x0, y0 + y, 32, 1);
	      mat_smooth_polyn_sequence(sinfo, 1, 0);
	      if (sinfo->original[0]  != 0.0) mat_statistics_add_value(stat0,  sinfo->original[0]/sinfo->smooth[0]);
	      if (sinfo->original[31] != 0.0) mat_statistics_add_value(stat31, sinfo->original[31]/sinfo->smooth[31]);
	    }
	}
    }
  mat_statistics_calc(stat0);
  aqpcs[0] = stat0->whole_mode;
  cpl_msg_info(cpl_func, " aqpcs 0 %f", aqpcs[0]);
  mat_statistics_calc(stat31);
  aqpcs[31] = stat31->whole_mode;
  cpl_msg_info(cpl_func, " aqpcs 31 %f", aqpcs[31]);
  mat_smooth_info_delete(sinfo);
  mat_statistics_delete(stat0);
  mat_statistics_delete(stat31);
  // correct each pixel column according to the calculated factor
  for (dc = 0; dc < 32; dc++)
    {
      int    x0 = 32*dc;
      for (y = 0; y < 1024; y++)
	{
	  int    rejected;
	  for (x = 0; x < 32; x++)
	    {
	      double cp, np;
	      if (einfo->original != NULL)
		{
		  cp = cpl_image_get(einfo->original, x0 + x + 1, y + 1, &rejected);
		  np = cp/aqpcs[x];
		  cpl_image_set(einfo->original, x0 + x + 1, y + 1, np);
		}
	      cp = cpl_image_get(einfo->src, x0 + x + 1, y + 1, &rejected);
	      np = cp/aqpcs[x];
	      cpl_image_set(einfo->src, x0 + x + 1, y + 1, np);
	    }
	}
    }
}

#ifdef DEVELOP
#define MAT_RAMP_COLUMNS 24 // total number of used pixels (1/2 covering the previous channel, 1/2 covering the next channel)

//#define MAT_DEBUG_Y      211
//#define MAT_DEBUG_DC      21

#define MAT_DEBUG_Y      371
#define MAT_DEBUG_DC      27

/*
static void mat_extract_single_ramp_iter(mat_smooth_info *sinfo, mat_statistics_info *stat,
					 double bias,
					 int *n, double *a0, double *a1, double *a2, double *a3, double *c2)
{
  int    x;
  
  for (x = 0; x < MAT_RAMP_COLUMNS/2; x++)
    {
      sinfo->data[MAT_RAMP_COLUMNS/2 + x] = sinfo->original[MAT_RAMP_COLUMNS/2 + x] + bias;
    }
  mat_smooth_polyx_window_outlier(sinfo, stat, 0, MAT_RAMP_COLUMNS-1, 0, MAT_RAMP_COLUMNS-1);
}
*/

/*
static void mat_extract_ramp(mat_extract_info *einfo)
{
  mat_statistics_info *stat;
  mat_smooth_info     *sinfo;
  int                  x, y, dc;

  stat = mat_statistics_new(64, MAT_OUTLIER_FACTOR);
  if (stat == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot allocate space for statistics.");
      return;
    }
  mat_statistics_set_use(stat, MAT_USE_REDUCED_MEDIAN, MAT_USE_REDUCED_MEDVAR);
  sinfo = mat_smooth_info_new(1024, MAT_UNDEFINED_INTENSITY, MAT_UNDEFINED_INTENSITY);
  if (sinfo == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot allocate space for smoothing operation.");
      mat_statistics_delete(stat);
      return;
    }
  sinfo->chi2limit_flag = 0;
  for (y = 0; y < 1024; y++)
    {
      double reco[32]; // column offset between detector channel columns due to a remanence effect
      reco[0] = 0.0;
      if ((y%8) == 0)
	{
	  cpl_msg_info(cpl_func, "ramp compensation for y %d", y);
	}
      if (y == MAT_DEBUG_Y)
	{
	  cpl_msg_info(cpl_func, "reco[%d] = %f", 0, reco[0]);
	}
      // 1. Estimate the offset between two detector channels due to a remanence effect
      for (dc = 1; dc < 32; dc++)
	{
	  int    x0   =           32*dc + 1 - MAT_RAMP_COLUMNS/2;
	  double di;
	  double bias, delta;
	  // The first guess uses no bias and calculate the reference chi2
	  sinfo->show_flag = ((y == MAT_DEBUG_Y) && (dc == MAT_DEBUG_DC));
	  bias  = 0.0;
	  mat_smooth_set(sinfo, einfo->src, einfo->range, x0, y+1, MAT_RAMP_COLUMNS, 1);
	  if (sinfo->show_flag)
	    {
	      for (x = 0; x < MAT_RAMP_COLUMNS; x++)
		{
		  cpl_msg_info(cpl_func, "ramp init %d %f", x - (MAT_RAMP_COLUMNS-1)/2, sinfo->original[x]);
		}
	    }
	  for (delta = 64.0; delta > 0.2; delta /= 4.0)
	    {
	      double fbias = 0.0;
	      double fchi2 = 1e32;
	      double pa0, pa1, pa2, pa3, pc2;
	      int    pn;
	      for (di = -10; di <= 10; di++)
		{
		  double hbias = bias + delta*(double)di;
		  mat_extract_single_ramp_iter(sinfo, stat, hbias, &pn, &pa0, &pa1, &pa2, &pa3, &pc2);
		  if (pc2 < fchi2)
		    {
		      if (sinfo->show_flag)
			{
			  if (pn == 1)
			    cpl_msg_info(cpl_func, "ramp found # bias %.2f f(x) = %f + x/64.0*%f", hbias, pa0, pa1);
			  else if (pn == 2)
			    cpl_msg_info(cpl_func, "ramp found # bias %.2f f(x) = %f + x/64.0*(%f + x/64.0*%f)", hbias, pa0, pa1, pa2);
			  else
			    cpl_msg_info(cpl_func, "ramp found # bias %.2f f(x) = %f + x/64.0*(%f + x/64.0*(%f + x/64.0*%f))", hbias, pa0, pa1, pa2, pa3);
			}
		      fbias = hbias;
		      fchi2 = pc2;
		    }
		}
	      if (fchi2 != 1e32)
		{
		  bias = fbias;
		}
	    }
	  reco[dc] = -bias;
	  if (y == MAT_DEBUG_Y)
	    {
	      cpl_msg_info(cpl_func, "reco[%d] = %f", dc, reco[dc]);
	    }
	}
      // 2. Remove the offset (as a linear ramp going from rcco[dc] down to 0.0 during 32 pixels)
      mat_smooth_set(sinfo, einfo->src, einfo->range, 1, y+1, 1024, 1);
      for (dc = 0; dc < 32; dc++)
	{
	  for (x = 0; x < 32; x++)
	    {
	      sinfo->smooth[32*dc + x] = ((double)(31 - x))/31.0*reco[dc];
	    }
	}
      if (einfo->original != NULL)
	{
	  //mat_smooth_apply(sinfo, einfo->original, -1.0);
	}
      mat_smooth_apply(sinfo, einfo->src, -1.0);
      mat_smooth_apply(sinfo, einfo->dst,  1.0);
    }
  mat_smooth_info_delete(sinfo);
  mat_statistics_delete(stat);
}
*/

static void mat_extract_ramp_guess(mat_smooth_info *info, mat_statistics_info *stat, int y, double *reco)
{
  int    x, dc;
  double loff[32]; // smooth offset at the left side of a detector channel (first pixel column - 0.5 pixel)
  double roff[32]; // smooth offset at the right side of a detector channel (last pixel column + 0.5 pixel)

  // The first guess uses a detector channel wide polynomial fit and an aproximate 3 perent of the intensity on the rightmost side.
  for (dc = 0; dc < 32; dc++)
    {
      info->show_flag = ((y == info->poiy) && (dc == (info->poix/32)));
      // fit the left/right side of the detector channel dc
      //mat_smooth_polyn_window_outlier(info, stat, 2, 32*dc, 32*dc + 15, 32*dc, 32*dc + 15, 0.5); // 0-based
      //mat_smooth_polyx_window_outlier(info, stat, 32*dc, 32*dc + 15, 32*dc, 32*dc + 15, 0.7); // 0-based
      //loff[dc] = mat_smooth_eval(info, 32*dc - 0.5);
      //mat_smooth_polyn_window_outlier(info, stat, 2, 32*dc + 16, 32*dc + 31, 32*dc + 16, 32*dc + 31, 0.5); // 0-based
      //mat_smooth_polyx_window_outlier(info, stat, 32*dc + 16, 32*dc + 31, 32*dc + 16, 32*dc + 31, 0.7); // 0-based
      //roff[dc] = mat_smooth_eval(info, 32*dc + 31 + 0.5);
      //mat_smooth_polyn_window_outlier(info, stat, 2, 32*dc, 32*dc + 31, 32*dc, 32*dc + 31, 0.7); // 0-based
      mat_smooth_polyx_window_outlier(info, stat, 32*dc, 32*dc + 31, 32*dc, 32*dc + 31, 0.7); // 0-based
      //mat_smooth_polyn_window_chi2(info, stat, 3, 32*dc, 32*dc + 31, 32*dc, 32*dc + 31); // 0-based
      loff[dc] = mat_smooth_eval(info, 32*dc      - 0.5);
      roff[dc] = mat_smooth_eval(info, 32*dc + 31 + 0.5);
    }
  for (x = 0; x < 1024; x++) info->work[x] = info->smooth[x];
  reco[0] = 0.0;
  if (y == info->poiy) cpl_msg_info(cpl_func, "reco[%d] = %f", 0, reco[0]);
  for (dc = 1; dc < 32; dc++)
    {
      //reco[dc] = loff[dc] - roff[dc-1];
      reco[dc] = 0.03*roff[dc];
      if (y == info->poiy) cpl_msg_info(cpl_func, "reco[%d] = %f", dc, reco[dc]);
    }
}

static void mat_extract_ramp_improve(mat_smooth_info *info, mat_statistics_info *stat, int y, double *reco)
{
  int    x, dc, di;

  for (dc = 1; dc < 32; dc++)
    { // indices for a MAT_RAMP_COLUMNS wide window spanning two detector channels
      int    lc_first = 32*dc - MAT_RAMP_COLUMNS/2;     // left channel first pixel
      int    rc_first = 32*dc;                          // right channel first pixel
      int    rc_last  = 32*dc + MAT_RAMP_COLUMNS/2 - 1; // right channel last pixel
      double fbias = reco[dc];  // use the first guess as a starting point
      double fchi2 = 1e32;
      double delta = 50.0;
      info->show_flag = ((y == info->poiy) && (dc == (info->poix/32)));
      while (delta > 0.2)
	{
	  double bias = fbias;
	  for (di = -10; di <= 10; di++)
	    {
	      double hbias = bias + delta*(double)di;
	      double hchi2;
	      // apply the previous channel bias as a ramp
	      ///*
	      for (x = 0; x < MAT_RAMP_COLUMNS/2; x++)
		{
		  double off = ((double)(MAT_RAMP_COLUMNS/2 - x - 1))/31.0*reco[dc-1];
		  info->data[lc_first + x] = info->original[lc_first + x] - off;
		}
	      //*/
	      // apply the new bias for the current channel as a ramp
	      for (x = 0; x < MAT_RAMP_COLUMNS/2; x++)
		{
		  double off = ((double)(31 - x))/31.0*hbias;
		  info->data[rc_first + x] = info->original[rc_first + x] - off;
		}
	      // try the best poly-n fit for that bias
	      //hchi2 = mat_smooth_polyx_window_outlier(info, stat, lc_first, rc_last, lc_first, rc_last, 0.7);
	      hchi2 = mat_smooth_polyn_window_outlier(info, stat, 2, lc_first, rc_last, lc_first, rc_last, 0.8);
	      if (info->show_flag)
		{
		  cpl_msg_info(cpl_func, "ramp bias %f %f %f %d", hbias, info->c2t, info->c2g, info->c2c);
		}
	      if (hchi2 < fchi2)
		{
		  fbias = hbias;
		  fchi2 = hchi2;
		}
	    }
	  delta /= 5.0;
	}
      reco[dc] = fbias;
    }
  if (y == info->poiy)
    {
      for (dc = 0; dc < 32; dc++) cpl_msg_info(cpl_func, "reco[%d] = %f", dc, reco[dc]);
    }
}

static void mat_extract_ramp(mat_extract_info *einfo)
{
  mat_statistics_info *stat;
  mat_smooth_info     *sinfo;
  int                  x, y, dc;

  // allocate structures for statistics and smoothing
  stat = mat_statistics_new(1024, MAT_OUTLIER_FACTOR);
  if (stat == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot allocate space for statistics.");
      return;
    }
  mat_statistics_set_use(stat, MAT_USE_REDUCED_MEDIAN, MAT_USE_REDUCED_MEDVAR);
  sinfo = mat_smooth_info_new(1024, MAT_UNDEFINED_INTENSITY, MAT_UNDEFINED_INTENSITY);
  if (sinfo == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot allocate space for smoothing operation.");
      mat_statistics_delete(stat);
      return;
    }
  sinfo->chi2limit_flag = 0;
  sinfo->poix = einfo->poix;
  sinfo->poiy = einfo->poiy;
  for (y = 0; y < 1024; y++)
    {
      double reco[32]; // column offset between detector channel columns due to a remanence effect
      if ((y%8) == 0)
	{
	  cpl_msg_info(cpl_func, "ramp compensation for y %d", y);
	}
      mat_smooth_set(sinfo, einfo->src, einfo->range, 1, y + 1, 1024, 1); // get a whole pixel row, 1-based
      // The first guess uses a detector channel wide polynomial fit and an aproximate 3 percent of the intensity on the rightmost side.
      mat_extract_ramp_guess(sinfo, stat, y, reco);
      // Now it is time to improve the first guess by a trial-and-error approach
      mat_extract_ramp_improve(sinfo, stat, y, reco);
      // Finally the ramp calculation
      for (dc = 0; dc < 32; dc++)
	{
	  for (x = 0; x < 32; x++)
	    {
	      double off = ((double)(31 - x))/31.0*reco[dc];
	      if (y == einfo->poiy) cpl_msg_info(cpl_func, "ramp %d %f %f %f %f", 32*dc + x, sinfo->original[32*dc + x], sinfo->work[32*dc + x], sinfo->smooth[32*dc + x], off);
	      sinfo->smooth[32*dc + x] = off;
	    }
	}
      if (einfo->original != NULL)
	{
	  //mat_smooth_apply(sinfo, einfo->original, -1.0);
	}
      mat_smooth_apply(sinfo, einfo->src, -1.0);
      mat_smooth_apply(sinfo, einfo->dst,  1.0);
    }
  mat_smooth_info_delete(sinfo);
  mat_statistics_delete(stat);
}

static void mat_extract_ramp_nt(mat_extract_info *einfo, double *rowrem)
{
  mat_statistics_info *stat;
  mat_smooth_info     *sinfo;
  int                  x, y, dr, dc;

  // allocate structures for statistics and smoothing
  stat = mat_statistics_new(1024, MAT_OUTLIER_FACTOR);
  if (stat == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot allocate space for statistics.");
      return;
    }
  mat_statistics_set_use(stat, MAT_USE_REDUCED_MEDIAN, MAT_USE_REDUCED_MEDVAR);
  sinfo = mat_smooth_info_new(1024, MAT_UNDEFINED_INTENSITY, MAT_UNDEFINED_INTENSITY);
  if (sinfo == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot allocate space for smoothing operation.");
      mat_statistics_delete(stat);
      return;
    }
  sinfo->chi2limit_flag = 0;
  sinfo->poix = einfo->poix;
  sinfo->poiy = einfo->poiy;
  // calculate a rough row remanence value, guess and improvement, horizontally
  for (dr = 0; dr < 2; dr++)
    {
      for (y = 0; y < 512; y++)
	{
	  double reco[32]; // column offset between detector channel columns due to a remanence effect
	  if ((y%8) == 0)
	    {
	      cpl_msg_info(cpl_func, "ramp compensation for y %d", 512*dr + y);
	    }
	  mat_smooth_set(sinfo, einfo->src, einfo->range, 1, 512*dr + y + 1, 1024, 1); // get a whole pixel row, 1-based
	  // The first guess uses a detector channel wide polynomial fit and an aproximate 3 perent of the intensity on the rightmost side.
	  mat_extract_ramp_guess(sinfo, stat, 512*dr + y, reco);
	  // Now it is time to improve the first guess by a trial-and-error approach
	  mat_extract_ramp_improve(sinfo, stat, 512*dr + y, reco);
	  if ((512*dr + y) == einfo->poiy)
	    {
	      for (dc = 0; dc < 32; dc++)
		{
		  for (x = 0; x < 32; x++)
		    {
		      double off = ((double)(31 - x))/31.0*reco[dc];
		      cpl_msg_info(cpl_func, "ramp prelim %d %f %f %f %f", 32*dc + x, sinfo->original[32*dc + x], sinfo->work[32*dc + x], sinfo->smooth[32*dc + x], off);
		    }
		}
	    }
	  for (dc = 0; dc < 32; dc++)
	    {
	      rowrem[512*(32*dr + dc) + y%512] = reco[dc];
	    }
	}
    }
  // Try to smooth the remanence for each detector channel, vertically
  /*
  for (dr = 0; dr < 2; dr++)
    {
      for (dc = 0; dc < 32; dc++)
	{
	  mat_smooth_reset(sinfo);
	  for (y = 0; y < 512; y++)
	    {
	      mat_smooth_add_value(sinfo, rowrem[512*(32*dr + dc) + y%512]);
	    }
	  // Locally fit a second order polynome using up to 19 values
	  mat_smooth_polyn_sequence_outlier(sinfo, 2, 29, 0.7);
	  for (y = 0; y < 512; y++)
	    {
	      rowrem[512*(32*dr + dc) + y%512] = sinfo->smooth[y];
	    }
	}
    }
  */
  // Apply the estimated row remanence
  for (dr = 0; dr < 2; dr++)
    {
      for (y = 0; y < 512; y++)
	{
	  // Get the current pixel row
	  mat_smooth_set(sinfo, einfo->src, einfo->range, 1, 512*dr + y + 1, 1024, 1); // get a whole pixel row, 1-based
	  // Calculate the smooth value directly from the row remanence information
	  for (dc = 0; dc < 32; dc++)
	    {
	      double reco = rowrem[512*(32*dr + dc) + y%512];
	      for (x = 0; x < 32; x++)
		{
		  double off = ((double)(31 - x))/31.0*reco;
		  if ((512*dr + y) == einfo->poiy) cpl_msg_info(cpl_func, "ramp final %d %f %f", 32*dc + x, sinfo->original[32*dc + x], off);
		  sinfo->smooth[32*dc + x] = off;
		}
	    }
	  // subtract/add the ramp from the data
	  if (einfo->original != NULL)
	    {
	      //mat_smooth_apply(sinfo, einfo->original, -1.0);
	    }
	  mat_smooth_apply(sinfo, einfo->src, -1.0);
	  mat_smooth_apply(sinfo, einfo->dst,  1.0);
	}
    }
  mat_smooth_info_delete(sinfo);
  mat_statistics_delete(stat);
}
#endif

#ifdef DEVELOP
#define MAT_AVG_COUNT  5  // MAT_AVG_COUNT must be odd!

static void mat_extract_poly2_pip_nt(mat_extract_info *einfo, int plength1, int plength2, double *aqpcs, double *aqdcr)
{
  int                  x, y, dc, dr;
  double              *spectrum = NULL;
  double              *profile = NULL;
  mat_statistics_info *stat = NULL;
  mat_smooth_info     *sinfo;

  // Try to extract the spectrum from the raw image by summing up each pixel row (median)
  spectrum = cpl_calloc(1024, sizeof(double));
  if (spectrum == NULL) return;
  profile = cpl_calloc(1024, sizeof(double));
  if (profile == NULL)
    {
      cpl_free(spectrum);
      return;
    }
  stat = mat_statistics_new(32*1024, MAT_OUTLIER_FACTOR);
  mat_statistics_set_use(stat, MAT_USE_REDUCED_MEDIAN, MAT_USE_REDUCED_MEDVAR);
  ///*
  sinfo = mat_smooth_info_new(1024, MAT_UNDEFINED_INTENSITY, MAT_UNDEFINED_INTENSITY);
  if (sinfo == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot allocate space for smoothing operation.");
      cpl_free(spectrum);
      cpl_free(profile);
      return;
    }
  // Calculate a pixel row remanence factor for each detector channel
  /*
  for (y = 0; y < 1024; y++)
    {
      double estimated[32]; // estimated value from previous detector channel (extrapolate)
      double measured[32];  // measured value on the leftmost side of a detector channel (first pixel to read inside a detector channel)
      double final[32];     // last measured value on the rightmost side of a detector channel (last pixel to read, an estimation on the measured value _before_ measured)
    }
  */
  /*
  for (dc = 0; dc < 32; dc++)
    {
      for (dr = 0; dr < 2; dr++)
	{
	  int x0   =           32*dc + 1;
	  int y0   =  512*dr         + 1;
	  mat_statistics_reset(stat);
	      mat_smooth_set(sinfo, einfo->src, einfo->range, x0, y0 + y, 32, 1);
	      mat_smooth_polyn_window(sinfo, 1, 0);
	      cil[dc] = sinfo->smooth[0];
	      cir[dc] = sinfo->smooth[31];
	    }
	  for (dc = 15; dc < 20; dc++)
	    {
	      cpl_msg_info(cpl_func, "dcdcf %d %d %f %f %f", dc, y, cir[dc-1], cil[dc], cir[dc]);
	    }
	}
      mat_statistics_calc(stat);
      aqpcs1[x] = stat->whole_mode;
      cpl_msg_info(cpl_func, " aqpcs %d %d %f", x, aqpcs1[x]);

      mat_statistics_reset(stat);
      for (y = 512; y < 1024; y++)
	{
	  for (dc = 14; dc < 20; dc++)
	    {
	      mat_smooth_set(sinfo, einfo->src, einfo->range, 32*dc + 1, y + 1, 32, 1);
	      mat_smooth_polyn_window(sinfo, 1, 0);
	      if (sinfo->original[x] != 0.0) mat_statistics_add_value(stat, sinfo->smooth[x]/sinfo->original[x]);
	      cil[dc] = sinfo->smooth[0];
	      cir[dc] = sinfo->smooth[31];
	    }
	  for (dc = 15; dc < 20; dc++)
	    {
	      cpl_msg_info(cpl_func, "dcdcf %d %d %f %f %f", dc+32, y, cir[dc-1], cil[dc], cir[dc]);
	    }
	}
      mat_statistics_calc(stat);
      aqpcs1[x] = stat->whole_mode;
      cpl_msg_info(cpl_func, " aqpcs %d %d %f", x, aqpcs1[x]);
    }
  */
  /*
  for (dc = 0; dc < 32; dc++)
    {
      int    x0 = 32*dc;
      for (y = 0; y < 512; y++)
	{
	  int    rejected;
	  for (x = 0; x < 32; x++)
	    {
	      double cp = cpl_image_get(einfo->src, x0 + x + 1, y + 1, &rejected);
	      //double np = 0.5*(1.0 + hrfactor[x])*cp;
	      double np = aqpcs[x]*cp;
	      cpl_image_set(einfo->src, x0 + x + 1, y + 1, np);
	    }
	}
      for (y = 512; y < 1024; y++)
	{
	  int    rejected;
	  for (x = 0; x < 32; x++)
	    {
	      double cp = cpl_image_get(einfo->src, x0 + x + 1, y + 1, &rejected);
	      //double np = 0.5*(1.0 + hrfactor[x])*cp;
	      double np = aqpcs[x]*cp;
	      cpl_image_set(einfo->src, x0 + x + 1, y + 1, np);
	    }
	}
    }
  */
  /*
  for (dc = 0; dc < 32; dc++)
    {
      int    x0 = 32*dc;
      for (y = 0; y < 1024; y++)
	{
	  int    rejected;
	  //double pv = 0.0313531*cpl_image_get(einfo->src, x0 + 32, y + 1, &rejected);
	  double pv = 0.020*cpl_image_get(einfo->src, x0 + 1, y + 1, &rejected); // 0.0313531
	  for (x = 0; x < 32; x++)
	    {
	      double cp = cpl_image_get(einfo->src, x0 + x + 1, y + 1, &rejected);
	      double np = cp - pv*(32.0 - (double)x)/32.0;
	      //pv *= 0.80; // 0.965
	      cpl_image_set(einfo->src, x0 + x + 1, y + 1, np);
	    }
	}
    }
  */
  /*
  mat_extract_channel(einfo->src, einfo->dst,
		      MAT_AQ_PHOTO1_LEFT, MAT_H2_BOTTOM, MAT_AQ_PHOTO1_RIGHT - MAT_AQ_PHOTO1_LEFT + 1, MAT_AQ_TOP - MAT_AQ_BOTTOM + 1,
		      stat, spectrum, profile);
  mat_extract_channel(einfo->src, einfo->dst,
		      MAT_AQ_PHOTO2_LEFT, MAT_AQ_BOTTOM, MAT_AQ_PHOTO2_RIGHT - MAT_AQ_PHOTO2_LEFT + 1, MAT_AQ_TOP - MAT_AQ_BOTTOM + 1,
		      stat, spectrum, profile);
  mat_extract_channel(einfo->src, einfo->dst,
		      MAT_AQ_INTERF_LEFT, MAT_AQ_BOTTOM, MAT_AQ_INTERF_RIGHT - MAT_AQ_INTERF_LEFT + 1, MAT_AQ_TOP - MAT_AQ_BOTTOM + 1,
		      stat, spectrum, profile);
  mat_extract_channel(einfo->src, einfo->dst,
		      MAT_AQ_PHOTO3_LEFT, MAT_AQ_BOTTOM, MAT_AQ_PHOTO3_RIGHT - MAT_AQ_PHOTO3_LEFT + 1, MAT_AQ_TOP - MAT_AQ_BOTTOM + 1,
		      stat, spectrum, profile);
  mat_extract_channel(einfo->src, einfo->dst,
		      MAT_AQ_PHOTO4_LEFT, MAT_AQ_BOTTOM, MAT_AQ_PHOTO4_RIGHT - MAT_AQ_PHOTO4_LEFT + 1, MAT_AQ_TOP - MAT_AQ_BOTTOM + 1,
		      stat, spectrum, profile);
  */
  /*
  for (dc = 0; dc < 32; dc++)
    {
      mat_extract_channel(einfo->src, einfo->dst, dc*32 + 1, 1, 32, 1024, stat, spectrum, profile);
    }
  */
  /*
  for (y = 0; y < einfo->ny; y++)
    { // iterate over all rows
      // estimate a horizontal slope inside each detector channel (32 pixel each)
      // goal is to remove the steps between successive detector channels
      double               ply[33];
      double               seq[10];
      ply[ 0] = 0.0;
      ply[32] = 0.0;
      for (dc = 1; dc < 32; dc++)
	{
	  int rejected;
	  seq[0] = cpl_image_get(einfo->src, 32*dc - 6 + 1, einfo->y0 + y, &rejected);
	  seq[1] = cpl_image_get(einfo->src, 32*dc - 5 + 1, einfo->y0 + y, &rejected);
	  seq[2] = cpl_image_get(einfo->src, 32*dc - 4 + 1, einfo->y0 + y, &rejected);
	  seq[3] = cpl_image_get(einfo->src, 32*dc - 3 + 1, einfo->y0 + y, &rejected);
	  seq[4] = cpl_image_get(einfo->src, 32*dc - 2 + 1, einfo->y0 + y, &rejected);
	  seq[5] = cpl_image_get(einfo->src, 32*dc + 1 + 1, einfo->y0 + y, &rejected);
	  seq[6] = cpl_image_get(einfo->src, 32*dc + 2 + 1, einfo->y0 + y, &rejected);
	  seq[7] = cpl_image_get(einfo->src, 32*dc + 3 + 1, einfo->y0 + y, &rejected);
	  seq[8] = cpl_image_get(einfo->src, 32*dc + 4 + 1, einfo->y0 + y, &rejected);
	  seq[9] = cpl_image_get(einfo->src, 32*dc + 5 + 1, einfo->y0 + y, &rejected);
	  qsort(seq, 10, sizeof(double), mat_compare_double);
	  ply[dc] = 0.5*(seq[4] + seq[5]);
	}
      if (einfo->poiy == (einfo->y0 + y))
	{
	  for (dc = 0; dc < 33; dc++)
	    cpl_msg_info(cpl_func, " dcply %d %f", dc, ply[dc]);
	}
    }
  */
  /*
  idx0 = (first_data + last_data)/2;
  for (i = first_smooth; i <= last_smooth; i++)
    {
      double x = ((double)(i - idx0))/64.0;
      info->smooth[i] = a + b*x + c*x*x;
    }
  */

  for (y = 0; y < einfo->ny; y++)
    { // iterate over all pixel rows
      sinfo->show_flag = (einfo->poiy == (einfo->y0 + y));
      // Copy the original values and ranges to the temporary memory (MASKED, BADPIX, LINEAR)
      mat_smooth_set(sinfo, einfo->src, einfo->range, einfo->x0, einfo->y0 + y, einfo->nx, 1);
      if (sinfo->show_flag)
	{
	  double   plx[32], ply[32];
	  for (dc = 0; dc < 32; dc++)
	    {
	      int last  = 32*dc + 32 - 2; // last pixel column (31, 63, ...)
	      int first = last - 11;
	      mat_smooth_polyn_window_chi2(sinfo, stat, 2, first, last, last, last);
	      plx[dc] = (double)last;
	      ply[dc] = sinfo->smooth[last];
	      cpl_msg_info(cpl_func, " dcply %f %f %f %f %f # f%d(x) = (%f + %f*(x - %f)/64.0 + %f*((x - %f)/64.0)**2)*%f",
			   plx[dc], ply[dc],
			   sinfo->a0n, sinfo->a1n, sinfo->a2n,
			   dc,
			   sinfo->a0n,
			   sinfo->a1n, sinfo->x0u,
			   sinfo->a2n, sinfo->x0u,
			   sinfo->yscale
			   );
	    }
	}
      for (dc = 0; dc < 32; dc++)
	{
	  //int x0 = 32*dc;
	  // Try to subtract an exponential add-on
	  /*
	  double is = 0.017*0.5*(sinfo->data[x0+1] + sinfo->data[x0+2]);
	  for (x = 0; x < 32; x++)
	    {
	      sinfo->data[x0+x] -= is*exp(-0.1*(double)x);
	    }
	  */
	  // Try to subtract an add-on
	  /*
	  double v = 0.0;
	  double o = 0.0;
	  for (x = 0; x < 32; x++)
	    {
	      double d = sinfo->data[x0+x] - v;
	      o = 0.9*o + 0.15*d;
	      v = sinfo->data[x0+x];
	      sinfo->data[x0+x] -= o;
	    }
	  */
	  // rescale according to hafactor[x]
	  /*
	  for (x = 0; x < 32; x++)
	    {
	      sinfo->data[x0+x] *= 0.5*(1.0 + hrfactor[x]);
	    }
	  */
	}
      //mat_smooth_polyn_window(sinfo, 2, plength1);
      //mat_smooth_apply(sinfo, einfo->src, -1.0);
      //mat_smooth_apply(sinfo, einfo->dst,  1.0);
    }
  mat_smooth_info_delete(sinfo);
  mat_statistics_delete(stat);
  cpl_free(spectrum);
  cpl_free(profile);
}
#endif

static void mat_smooth_poly0_global(mat_smooth_info *info)
{
  int                   i;
  double                offset;
  mat_statistics_info  *stat_info;

  stat_info = mat_statistics_new(info->nused, 1.5);
  if (stat_info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "cannot allocate memory for mat_statistics_info");
      return;
    }
  mat_statistics_reset(stat_info);
  // Sort the pixel intensities and estimate the median
  if (info->with_src_filter)
    {
      info->count = 0;
      for (i = info->first; i <= info->last; i++)
	{
	  if (info->range[i] == info->src_filter)
	    {
	      mat_statistics_add_value(stat_info, info->data[i]);
	    }
	}
    }
  else
    {
      for (i = 0; i < info->nused; i++)
	{
	  mat_statistics_add_value(stat_info, info->data[i]);
	}
    }
  mat_statistics_calc(stat_info);
  offset = stat_info->average;
  for (i = 0; i < info->nused; i++)
    {
      info->smooth[i] = offset;
    }
  mat_statistics_delete(stat_info);
}

/**
   @brief Try to smooth a vector of intensity values by locally fitting a 2. order polynome.
   @param info ...
   @param plength ...
   The function used is

   y -> a + b x + c x^2

   The least squared distance formula leads to the following equations:

   det -> sxx^3 - 2 sx sxx sxxx + N sxxx^2 + sx^2 sxxxx - N sxx sxxxx
   a -> -((-sxx^2 sxxy + sx sxxx sxxy + sxx sxxx sxy - sx sxxxx sxy - sxxx^2 sy + sxx sxxxx sy)/det)
   b -> -((sx sxx sxxy - N sxxx sxxy - sxx^2 sxy + N sxxxx sxy + sxx sxxx sy - sx sxxxx sy)/det)
   c -> -((-sx^2 sxxy + N sxx sxxy + sx sxx sxy - N sxxx sxy - sxx^2 sy + sx sxxx sy)/det)

   with
   
   sxx = Sum(x*x, 1 .. N)
   ...
 */
static void mat_extract_image_poly0_local(cpl_image    *dst,
					  cpl_image    *src,
					  cpl_image    *range,
					  int           src_filter,
					  int           dst_filter,
					  int           lwidth,
					  int           lheight)
{
  mat_smooth_info   *sinfo;
  mat_smooth_info   *oinfo;
  int                nx, ny;
  int                x, y;

  nx = cpl_image_get_size_x(src);
  ny = cpl_image_get_size_y(src);
  sinfo = mat_smooth_info_new(lwidth*lheight, src_filter, dst_filter);
  if (sinfo == NULL) return;
  oinfo = mat_smooth_info_new(nx*ny, src_filter, dst_filter);
  if (oinfo == NULL)
    {
      mat_smooth_info_delete(sinfo);
      cpl_msg_error(cpl_func, "Cannot allocate space for offset image.");
      return;
    }
  mat_smooth_set(oinfo, src, range, 1, 1, nx, ny);
  for (y = 0; y < ny; y++)
    {
      for (x = 0; x < nx; x++)
	{
	  int left   = CPL_MAX(x - lwidth/2, 0);
	  int right  = CPL_MIN(x + lwidth/2, nx - 1);
	  int bottom = CPL_MAX(y - lheight/2, 0);
	  int top    = CPL_MIN(y + lheight/2, ny - 1);
	  mat_smooth_set(sinfo, src, range, left + 1, bottom + 1, right - left + 1, top - bottom + 1);
	  mat_smooth_poly0_global(sinfo);
	  oinfo->smooth[y*nx + x] = sinfo->smooth[0];
	}
    }
  mat_smooth_apply(oinfo, src, -1.0);
  mat_smooth_apply(oinfo, dst,  1.0);
  mat_smooth_info_delete(sinfo);
  mat_smooth_info_delete(oinfo);
}

static void mat_extract_channel_poly0_global(mat_detector *det,
					     cpl_image    *dst,
					     cpl_image    *src,
					     cpl_image    *range,
					     int           src_filter,
					     int           dst_filter)
{
  mat_smooth_info   *sinfo;
  int                nx, ny;
  int                cr, cc;

  nx = det->channel_nx;
  ny = det->channel_ny;
  sinfo = mat_smooth_info_new(nx*ny, src_filter, dst_filter);
  if (sinfo == NULL) return;
  for (cr = 0; cr < det->channel_nrows; cr++)
    { // iterate over the detector channel rows
      for (cc = 0; cc < det->channel_ncolumns; cc++)
	{ // iterate over the detector channel columns
	  mat_smooth_set(sinfo, src, range, cc*nx + 1, cr*ny + 1, nx, ny);
	  mat_smooth_poly0_global(sinfo);
	  mat_smooth_apply(sinfo, src, -1.0);
	  mat_smooth_apply(sinfo, dst,  1.0);
	}
    }
  mat_smooth_info_delete(sinfo);
}

/*
static void mat_extract_channel_common_mode_h2(cpl_image *dst,cpl_image *src)
{
  int x, y;

  for (y = 0; y < 2048; y++)
    {
      for (x = 0; x < 64; x++)
	{
	  double    values[32];
	  double    m;
	  int       c;
	  for (c = 0; c < 16; c++)
	    {
	      int rejected;
	      // left to right
	      values[2*c]     = cpl_image_get(src, 128*c + x +   1, y + 1, &rejected);
	      // right to left
	      values[2*c + 1] = cpl_image_get(src, 128*c - x + 128, y + 1, &rejected);
	    }
	  qsort(values, 32, sizeof(double), mat_compare_double);
	  m = 0.5*(values[15] + values[16]);
	  for (c = 0; c < 16; c++)
	    {
	      int rejected;
	      // left to right
	      cpl_image_set(src, 128*c + x + 1, y + 1, cpl_image_get(src, 128*c + x + 1, y + 1, &rejected) - m);
	      cpl_image_set(dst, 128*c + x + 1, y + 1, cpl_image_get(dst, 128*c + x + 1, y + 1, &rejected) + m);
	      // right to left
	      cpl_image_set(src, 128*c - x + 128, y + 1, cpl_image_get(src, 128*c - x + 128, y + 1, &rejected) - m);
	      cpl_image_set(dst, 128*c - x + 128, y + 1, cpl_image_get(dst, 128*c - x + 128, y + 1, &rejected) + m);
	    }
	}
    }
}
*/

/*
static void mat_extract_channel_common_mode_aq(cpl_image *dst, cpl_image *src)
{
  int       r;

  for (r = 0; r < 2; r++)
    {
      int x, y;
      for (y = 0; y < 512; y++)
	{
	  for (x = 0; x < 32; x++)
	    {
	      double    values[32];
	      double    m;
	      int       c;
	      for (c = 0; c < 32; c++)
		{
		  int rejected;
		  values[c] = cpl_image_get(src, 32*c + x + 1, 512*r + y + 1, &rejected);
		}
	      qsort(values, 32, sizeof(double), mat_compare_double);
	      m = 0.5*(values[15] + values[16]);
	      for (c = 0; c < 32; c++)
		{
		  int rejected;
		  cpl_image_set(src, 32*c + x + 1, 512*r + y + 1, cpl_image_get(src, 32*c + x + 1, 512*r + y + 1, &rejected) - m);
		  cpl_image_set(dst, 32*c + x + 1, 512*r + y + 1, cpl_image_get(dst, 32*c + x + 1, 512*r + y + 1, &rejected) + m);
		}
	    }
	}
    }
}
*/

/*
  After calculating the raw statistics for all dark exposures, the dark current and an artificial
  dark statistics (median and variance) is estimated. The dark current is calculated for each pixel
  as a straight line based on several raw darks from exposures with different DIT. The artificial
  dark statistics is calculated by summing up the median intensity or temporal variance weigted by
  the DIT. The artificial median image is then separated into a regular (reference) dark and
  a spatial noise component. This spatial noise component is later used to determine if a pixel
  is good or bad (a range is used with absolute limits using the --darklimit option).
 */

static cpl_error_code mat_dark_prop_allocate(mat_cal_det_info *info)
{
  
  if (info->darkmedraw == NULL)
    {
      info->darkmedraw = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->darkmedraw == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the dark median (raw)");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->darkmedraw, 0.0);
  if (info->darkmedref == NULL)
    {
      info->darkmedref = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->darkmedref == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the dark median (reference)");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->darkmedref, 0.0);
  if (info->darkmedres == NULL)
    {
      info->darkmedres = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->darkmedres == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the dark median (residuals)");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->darkmedres, 0.0);
  if (info->darkvarraw == NULL)
    {
      info->darkvarraw = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->darkvarraw == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the dark variance (raw)");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->darkvarraw, 0.0);
  if (info->darkrange == NULL)
    {
      info->darkrange = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->darkrange == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the dark range map");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->darkrange, MAT_LOW_INTENSITY);
  if (info->dcoffset == NULL)
    {
      info->dcoffset = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->dcoffset == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the dark current offset");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->dcoffset, 0.0);
  if (info->dcslope == NULL)
    {
      info->dcslope = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->dcslope == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the dark current slope");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->dcslope, 0.0);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_dark_prop_calculate(mat_cal_det_info *info)
{
  int      i, x, y, rejected;
  double   dit;

  if (info->nbdit != 1)
    {
      cpl_msg_info(cpl_func, "Calculate the dark current for each pixel (dc = offset + slope*dit).");
      for (y = 1; y <= info->ny; y++)
	{
	  for (x = 1; x <= info->nx; x++)
	    {
	      double A = 0.0;
	      double B = 0.0;
	      double C = 0.0;
	      double E1 = 0.0;
	      double E2 = 0.0;
	      double det;
	      double cnt  = 0.0;
	      // calculate the 2x2 matrix
	      for (i = 0; i < info->nbdit; i++)
		{
		  double vx, vy, vye2;
		  if (info->psm[i].rmed[DARK_IDX] == NULL) continue; // no dark for that DIT available
		  vx    = info->psm[i].dit;
		  vy    = cpl_image_get(info->psm[i].rmed[DARK_IDX], x, y, &rejected);
		  vye2  = cpl_image_get(info->psm[i].rvar[DARK_IDX], x, y, &rejected);
		  A   += vx*vx/vye2;
		  B   += vx/vye2;
		  C   += 1.0/vye2;
		  E1  += vy*vx/vye2;
		  E2  += vy/vye2;
		  cnt += 1.0;
		}
	      if (cnt < 2) continue;
	      // calculate offset and slope
	      det = A*C - B*B;
	      // cpl_msg_info(cpl_func, "fit A = %g B = %g C = %g E1 = %g E2 = %g cnt = %g det = %g", A, B, C, E1, E2, cnt, det);
	      if (fabs(det) < 1e-6) continue;
	      cpl_image_set(info->dcoffset, x, y, (A*E2 - B*E1)/det);
	      cpl_image_set(info->dcslope,  x, y, (C*E1 - B*E2)/det);
	    }
	}
    }
  cpl_msg_info(cpl_func, "Calculate an average dark image using all available darks weighted by DIT.");
  dit = 0.0;
  if ((info->expert & DARK_EXPERT_FLAG) && (info->poix != 0) && (info->poiy != 0))
    {
      cpl_msg_info(cpl_func, "pixel at [%d,%d]", info->poix, info->poiy);
      cpl_msg_info(cpl_func, "dit median variance");
    }
  for (i = 0; i < info->nbdit; i++)
    {
      if (info->psm[i].rmed[DARK_IDX] == NULL) continue; // no dark for that DIT available
      // remove the detector channel offset (needed for uncorrelated readout modes)
      mat_extract_channel_poly0_global(info->det,
				       NULL,
				       info->psm[i].rmed[DARK_IDX],
				       info->darkrange,
				       MAT_LOW_INTENSITY, MAT_UNDEFINED_INTENSITY);
      if ((info->expert & DARK_EXPERT_FLAG) && (info->poix != 0) && (info->poiy != 0))
	{
	  cpl_msg_info(cpl_func, "%g %g %g",
		       info->psm[i].dit,
		       cpl_image_get(info->psm[i].rmed[DARK_IDX], info->poix, info->poiy, &rejected),
		       cpl_image_get(info->psm[i].rvar[DARK_IDX], info->poix, info->poiy, &rejected));
	}
      cpl_image_add(info->darkmedraw, info->psm[i].rmed[DARK_IDX]);
      cpl_image_add(info->darkvarraw, info->psm[i].rvar[DARK_IDX]);
      dit += info->psm[i].dit;
    }
  // Scale the dark current by the summed up DIT and the dark noise by the number of files
  cpl_image_multiply_scalar(info->darkmedraw, info->timescale/dit);
  cpl_image_divide_scalar(info->darkvarraw, (double)(info->nbdit));
  cpl_image_copy(info->darkmedres, info->darkmedraw, 1, 1);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_dark_prop_range(mat_cal_det_info *info)
{
  mat_image_fill(info->darkrange, MAT_LOW_INTENSITY);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_dark_prop_reference(mat_cal_det_info *info)
{
  cpl_image         *raw   = info->darkmedraw;
  cpl_image         *ref   = info->darkmedref;
  cpl_image         *res   = info->darkmedres;
  cpl_image         *range = info->darkrange;
  mat_extract_info   einfo;

  cpl_msg_info(cpl_func, "Separate the average dark image into a reference and residual component.");
  cpl_image_copy(res, raw, 1, 1);
  mat_image_fill(info->darkmedref, 0.0);
  mat_extract_info_set_data(&einfo, ref, res, range);
  if (info->expert & DARK_EXPERT_FLAG)
    {
      einfo.poix = info->poix;
      einfo.poiy = info->poiy;
    }
  cpl_msg_info(cpl_func, "Extract a pixel row offset spanning the whole detector.");
  // mat_extract_channel_offset(info->det, ref, res, range, MAT_LOW_INTENSITY, MAT_UNDEFINED_INTENSITY);
  // horizontal offsets for each pixel row
  mat_extract_info_set_filter(&einfo, MAT_LOW_INTENSITY, MAT_UNDEFINED_INTENSITY);
  mat_extract_info_set_window(&einfo, info->det, MAT_DETECTOR_WINDOW_TYPE, 0, 0, 0, 0);
  mat_extract_polyn(&einfo, MAT_HORIZONTAL_DIRECTION, 0, 0);
  if (info->expert & DARK_EXPERT_FLAG)
    {
      mat_image_validate(ref, NULL, -1000.0, 1000.0, "dark ref (h)");
      mat_image_validate(res, NULL, -1000.0, 1000.0, "dark res (h)");
    }
  cpl_msg_info(cpl_func, "Extracting a pixel column offset spanning detector channels only.");
  // vertical offsets for each pixel column inside a detector channel
  mat_extract_info_set_filter(&einfo, MAT_LOW_INTENSITY, MAT_UNDEFINED_INTENSITY);
  mat_extract_info_set_window(&einfo, info->det, MAT_CHANNEL_WINDOW_TYPE, 0, 0, 0, 0);
  mat_extract_polyn(&einfo, MAT_VERTICAL_DIRECTION, 0, 0);
  if (info->expert & DARK_EXPERT_FLAG)
    {
      mat_image_validate(ref, NULL, -1000.0, 1000.0, "dark ref (v)");
      mat_image_validate(res, NULL, -1000.0, 1000.0, "dark res (v)");
    }
  switch (info->det->type)
    {
    case MAT_HAWAII2RG_DET:
      //mat_extract_channel_common_mode_h2(ref, res);
      break;
    case MAT_AQUARIUS_DET:
      //mat_extract_channel_common_mode_aq(ref, res);
      break;
     default:;
    }
  if (info->expert & DARK_EXPERT_FLAG)
    {
      mat_image_validate(ref, NULL, -1000.0, 1000.0, "dark ref (cm)");
      mat_image_validate(res, NULL, -1000.0, 1000.0, "dark res (cm)");
    }
  // 1, 1, cpl_image_get_size_x(raw), cpl_image_get_size_y(raw)
  if (info->expert & DARK_EXPERT_FLAG)
    {
      mat_image_store_plain_stl("darkraw.stl", raw, 480, 450, 256, 256, 4.0);
      mat_image_store_plain_stl("darkref.stl", ref, 480, 450, 256, 256, 4.0);
      mat_image_store_plain_stl("darkres.stl", res, 480, 450, 256, 256, 4.0);
      //mat_image_store_column_stl("darkraw.stl", raw, 480, 450, 256, 256, 1.0);
      //mat_image_store_column_stl("darkref.stl", ref, 480, 450, 256, 256, 1.0);
      //mat_image_store_column_stl("darkres.stl", res, 480, 450, 256, 256, 1.0);
      //mat_image_store_pyramid_stl("darkraw.stl", raw, 480, 450, 256, 256, 1.0);
      //mat_image_store_pyramid_stl("darkref.stl", ref, 480, 450, 256, 256, 1.0);
      //mat_image_store_pyramid_stl("darkres.stl", res, 480, 450, 256, 256, 1.0);
    }
  return CPL_ERROR_NONE;
}

/*
  After calculating the pixel statistics for the flatfield exposures, an artificial flat statistics
  is estimated based on the calibrated pixel statistics using a global (detector) nonlinearity law.
 */

static cpl_error_code mat_flat_prop_allocate(mat_cal_det_info *info)
{
  // Allocate memory for the intensity [DU/s] and noise [DU^2] properties.
  if (info->flatmedraw == NULL)
    {
      info->flatmedraw = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->flatmedraw == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the flat median (raw)");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->flatmedraw, 0.0);
  if (info->flatmedref == NULL)
    {
      info->flatmedref = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->flatmedref == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the flat median (reference)");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->flatmedref, 0.0);
  if (info->flatmedres == NULL)
    {
      info->flatmedres = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->flatmedres == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the flat median (residuals)");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->flatmedres, 0.0);
  if (info->flatvarraw == NULL)
    {
      info->flatvarraw = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->flatvarraw == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the flat variance (raw)");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->flatvarraw, 0.0);
  if (info->flatrange == NULL)
    {
      info->flatrange = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->flatrange == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the flat range map");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->flatrange, MAT_LINEAR_INTENSITY);
  if (info->flatbpm == NULL)
    {
      info->flatbpm = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->flatbpm == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the relative bpm");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->flatbpm, 0.0);
  return CPL_ERROR_NONE;
}

static void mat_set_range(cpl_image *range, int flag, int x, int y, int nx, int ny)
{
  // use the given flag inside the whole area
  int i, j;
  for (j = y; j < y + ny; j++)
    {
      for (i = x; i < x + nx; i++)
	{
	  cpl_image_set(range, i, j, flag);
	}
    }
}

static void mat_estimate_range_h2(cpl_image *raw, cpl_image *range, double min_linear_range, int x, int y, int nx, int ny)
{
  int         px, py, rejected;

  mat_set_range(range, 0.0, x, y, nx, ny);
  for (px = x; px < x + nx; px++)
    {
      int b = y, t = y + ny - 1;
      while ((b <  y + ny) && (cpl_image_get(raw, px, b, &rejected) < min_linear_range)) b++;
      while ((t >= y)      && (cpl_image_get(raw, px, t, &rejected) < min_linear_range)) t--;
      if (b < t)
	{
	  for (py = b; py <= t; py++)
	    {
	      cpl_image_set(range, px, py, cpl_image_get(range, px, py, &rejected) + 1.0);
	    }
	}
    }
  for (py = y; py < y + ny; py++)
    {
      int l = x, r = x + nx - 1;
      while ((l <  x + nx) && (cpl_image_get(raw, l, py, &rejected) < min_linear_range)) l++;
      while ((r >= x)      && (cpl_image_get(raw, r, py, &rejected) < min_linear_range)) r--;
      if (l < r)
	{
	  for (px = l; px <= r; px++)
	    {
	      cpl_image_set(range, px, py, cpl_image_get(range, px, py, &rejected) + 1.0);
	    }
	}
    }
  // 0 -> MAT_LOW_INTENSITY
  // 1 -> MAT_LOW_INTENSITY
  // 2 -> MAT_LINEAR_INTENSITY
  for (py = y; py < y + ny; py++)
    {
      for (px = x; px < x + nx; px++)
	{
	  if (cpl_image_get(range, px, py, &rejected) == 2.0)
	    {
	      cpl_image_set(range, px, py, MAT_LINEAR_INTENSITY);
	    }
	  else
	    {
	      cpl_image_set(range, px, py, MAT_IGNORE_INTENSITY);
	    }
	}
    }
}

static void mat_estimate_range_aq(cpl_image *raw, cpl_image *range, double min_linear_range, int x, int y, int nx, int ny, mat_statistics_info *sinfo)
{
  int py;

  for (py = y; py < y + ny; py++)
    {
      double limit;
      int    l = x;
      int    r = x + nx - 1;
      int    px, rejected;
      mat_statistics_reset(sinfo);
      mat_statistics_set_region(sinfo, raw, NULL, x + nx/4, py, nx/2, 1);
      mat_statistics_calc(sinfo);
      limit = 0.2*sinfo->whole_median;
      while ((l <  x + nx) && ((cpl_image_get(raw, l, py, &rejected) < limit) || (cpl_image_get(raw, l, py, &rejected) < min_linear_range))) l++;
      while ((r >= x)      && ((cpl_image_get(raw, r, py, &rejected) < limit) || (cpl_image_get(raw, r, py, &rejected) < min_linear_range))) r--;
      for (px = x; px <= x + nx; px++)
	{
	  if ((px < l) || (px > r))
	    {
	      cpl_image_set(range, px, py, MAT_IGNORE_INTENSITY);
	    }
	  else
	    {
	      cpl_image_set(range, px, py, MAT_LINEAR_INTENSITY);
	    }
	}
    }
}

static cpl_error_code mat_flat_prop_range(mat_cal_det_info *info)
{
  cpl_image  *raw   = info->flatmedraw;
  cpl_image  *range = info->flatrange;

  cpl_msg_info(cpl_func, "Detect all pixels were the intensity is high enough to calculate a reliable flatfield.");
  // all -> MAT_IGNORE_INTENSITY
  mat_set_range(range, MAT_IGNORE_INTENSITY, 1, 1, info->nx, info->ny);
  if (info->det->type == MAT_HAWAII2RG_DET)
    {
      // frame to MAT_LOW_INTENSITY
      mat_set_range(range, MAT_LOW_INTENSITY,                 1,                1,       64, info->ny);
      mat_set_range(range, MAT_LOW_INTENSITY, info->nx - 64 + 1,                1,       64, info->ny);
      mat_set_range(range, MAT_LOW_INTENSITY,                 1,                1, info->nx,        8);
      mat_set_range(range, MAT_LOW_INTENSITY,                 1, info->ny - 8 + 1, info->nx,        8);
      // exposed -> MAT_LINEAR_INTENSITY
      mat_estimate_range_h2(raw, range, info->min_linear_range, MAT_H2_LEFT, MAT_H2_BOTTOM, MAT_H2_RIGHT - MAT_H2_LEFT + 1, MAT_H2_TOP - MAT_H2_BOTTOM + 1);
    }
  else if (info->det->type == MAT_AQUARIUS_DET)
    {
      // frame to MAT_LOW_INTENSITY
      mat_set_range(range, MAT_LOW_INTENSITY,                 1,                 1,       32, info->ny);
      mat_set_range(range, MAT_LOW_INTENSITY, info->nx - 32 + 1,                 1,       32, info->ny);
      mat_set_range(range, MAT_LOW_INTENSITY,                 1,                 1, info->nx,       16);
      mat_set_range(range, MAT_LOW_INTENSITY,                 1, info->ny - 16 + 1, info->nx,       16);
      // exposed -> MAT_LINEAR_INTENSITY
      mat_estimate_range_aq(raw, range, info->min_linear_range, MAT_AQ_PHOTO1_LEFT, MAT_AQ_BOTTOM, MAT_AQ_PHOTO1_RIGHT - MAT_AQ_PHOTO1_LEFT + 1, MAT_AQ_TOP - MAT_AQ_BOTTOM + 1, info->stat1);
      mat_estimate_range_aq(raw, range, info->min_linear_range, MAT_AQ_PHOTO2_LEFT, MAT_AQ_BOTTOM, MAT_AQ_PHOTO2_RIGHT - MAT_AQ_PHOTO2_LEFT + 1, MAT_AQ_TOP - MAT_AQ_BOTTOM + 1, info->stat1);
      mat_estimate_range_aq(raw, range, info->min_linear_range, MAT_AQ_INTERF_LEFT, MAT_AQ_BOTTOM, MAT_AQ_INTERF_RIGHT - MAT_AQ_INTERF_LEFT + 1, MAT_AQ_TOP - MAT_AQ_BOTTOM + 1, info->stat1);
      mat_estimate_range_aq(raw, range, info->min_linear_range, MAT_AQ_PHOTO3_LEFT, MAT_AQ_BOTTOM, MAT_AQ_PHOTO3_RIGHT - MAT_AQ_PHOTO3_LEFT + 1, MAT_AQ_TOP - MAT_AQ_BOTTOM + 1, info->stat1);
      mat_estimate_range_aq(raw, range, info->min_linear_range, MAT_AQ_PHOTO4_LEFT, MAT_AQ_BOTTOM, MAT_AQ_PHOTO4_RIGHT - MAT_AQ_PHOTO4_LEFT + 1, MAT_AQ_TOP - MAT_AQ_BOTTOM + 1, info->stat1);
    }
  else
    {
      int fl = info->det->frame_left;
      int fr = info->det->frame_right;
      int fb = info->det->frame_bottom;
      int ft = info->det->frame_top;
      // frame to MAT_LOW_INTENSITY
      mat_set_range(range, MAT_LOW_INTENSITY,                 1,                 1,       fl, info->ny);
      mat_set_range(range, MAT_LOW_INTENSITY, info->nx - fr + 1,                 1,       fr, info->ny);
      mat_set_range(range, MAT_LOW_INTENSITY,                 1,                 1, info->nx,       fb);
      mat_set_range(range, MAT_LOW_INTENSITY,                 1, info->ny - ft + 1, info->nx,       ft);
      mat_estimate_range_h2(raw, range, info->min_linear_range, fl+1, fb+1, info->nx - fr - fl, info->ny - fb - ft);
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_flat_prop_reference(mat_cal_det_info *info)
{
  cpl_image         *raw   = info->flatmedraw;
  cpl_image         *ref   = info->flatmedref;
  cpl_image         *res   = info->flatmedres;
  cpl_image         *range = info->flatrange;
  mat_extract_info   einfo;

  cpl_msg_info(cpl_func, "Separate the average flat image into a reference and residual component.");
  // ***** Initialise the reference and range images
  cpl_image_copy(res, raw, 1, 1);
  mat_image_fill(info->flatmedref, 0.0);
  //cpl_image_reject_from_mask(raw, info->bpm->list_badpixel[0]);
  //cpl_image_reject_from_mask(res, info->bpm->list_badpixel[0]);
  // Calculate the reference values for the flat intensity
  mat_extract_info_set_data(&einfo, ref, res, range);
  switch (info->det->type)
    {
    case MAT_HAWAII2RG_DET:
      // Selected method for HAWAII-2RG data taken with the correct configuration (pinhole removed and proper lamp current)
      // use the dark part to calculate a global offset applied to the whole detector
      cpl_msg_info(cpl_func, "Extract a global background.");
      mat_extract_info_set_filter(&einfo, MAT_LOW_INTENSITY, MAT_UNDEFINED_INTENSITY);
      mat_extract_info_set_window(&einfo, info->det, MAT_DETECTOR_WINDOW_TYPE, 0, 0, 0, 0);
      mat_extract_polyn(&einfo, MAT_BOTH_DIRECTION, 0, 0);
      if (info->expert & FLAT_EXPERT_FLAG)
	{
	  mat_image_validate(ref, NULL, -50000.0, 50000.0, "flat ref (h2 b)");
	  mat_image_validate(res, NULL, -50000.0, 50000.0, "flat res (h2 b)");
	}
      cpl_msg_info(cpl_func, "Extract a global 2-order polynome for each exposed pixel column.");
      // calculate for each exposed pixel column a global 2-order polynome applied to the exposed part
      mat_extract_info_set_filter(&einfo, MAT_LINEAR_INTENSITY, MAT_LINEAR_INTENSITY);
      mat_extract_info_set_window(&einfo, info->det, MAT_DIRECT_WINDOW_TYPE, MAT_H2_LEFT, MAT_H2_RIGHT - MAT_H2_LEFT + 1, MAT_H2_BOTTOM, MAT_H2_TOP - MAT_H2_BOTTOM + 1);
      mat_extract_polyn(&einfo, MAT_VERTICAL_DIRECTION, 2, 0);
      //mat_extract_polyn(&einfo, MAT_VERTICAL_DIRECTION, 2, 101);
      //mat_extract_polyn(&einfo, MAT_VERTICAL_DIRECTION, 2, 201);
      if (info->expert & FLAT_EXPERT_FLAG)
	{
	  mat_image_validate(ref, NULL, -50000.0, 50000.0, "flat ref (h2 v)");
	  mat_image_validate(res, NULL, -50000.0, 50000.0, "flat res (h2 v)");
	}
      break;
    case MAT_AQUARIUS_DET:
      /*
      mat_extract_channel_offset(info->det, ref, res, range,
				 MAT_LOW_INTENSITY, MAT_UNDEFINED_INTENSITY);
      */
      /*
      mat_extract_detector_poly0_local(info->det, ref, res, range,
				       MAT_UNDEFINED_INTENSITY, MAT_LINEAR_INTENSITY,
				       11, 11);
      */
      /*
      mat_extract_detector_horizontal_poly2_local1(ref, res, range,
						  MAT_LINEAR_INTENSITY, MAT_LINEAR_INTENSITY, 31,
						  56, 133,
						  info->det->frame_bottom, info->det->ny - info->det->frame_top + 1,
						  info->poiy);
      mat_extract_detector_horizontal_poly2_local1(ref, res, range,
						  MAT_LINEAR_INTENSITY, MAT_LINEAR_INTENSITY, 31,
						  168, 242,
						  info->det->frame_bottom, info->det->ny - info->det->frame_top + 1,
						  info->poiy);
      mat_extract_detector_horizontal_poly2_local1(ref, res, range,
						  MAT_LINEAR_INTENSITY, MAT_LINEAR_INTENSITY, 31,
						  276, 766,
						  info->det->frame_bottom, info->det->ny - info->det->frame_top + 1,
						  info->poiy);
      mat_extract_detector_horizontal_poly2_local1(ref, res, range,
						  MAT_LINEAR_INTENSITY, MAT_LINEAR_INTENSITY, 31,
						  776, 851,
						  info->det->frame_bottom, info->det->ny - info->det->frame_top + 1,
						  info->poiy);
      mat_extract_detector_horizontal_poly2_local1(ref, res, range,
						  MAT_LINEAR_INTENSITY, MAT_LINEAR_INTENSITY, 31,
						  885, 954,
						  info->det->frame_bottom, info->det->ny - info->det->frame_top + 1,
						  info->poiy);
      */
      if (info->hasphoto)
	{
	  // old implementation
	  /*
	  mat_extract_detector_horizontal_poly2_local2(ref, res, NULL,
						       MAT_UNDEFINED_INTENSITY, MAT_UNDEFINED_INTENSITY, 19, 79,
						       1, info->det->nx,
						       1, info->det->ny,
						       info->poiy);
	  */
	  // new implementation
	  //mat_calibration_compensate_ramp(info, res);
	  cpl_msg_info(cpl_func, "Extract a common pixel column scale (first and last column in a detector channel).");
	  mat_extract_info_set_filter(&einfo, MAT_UNDEFINED_INTENSITY, MAT_UNDEFINED_INTENSITY);
	  mat_extract_info_set_window(&einfo, info->det, MAT_DETECTOR_WINDOW_TYPE, 0, 0, 0, 0);
	  if (info->expert & FLAT_EXPERT_FLAG)
	    {
	      einfo.poix = info->poix;
	      einfo.poiy = info->poiy;
	    }
	  mat_extract_info_set_original(&einfo, raw);
	  mat_extract_aq_column_scale(&einfo, info->aqpcs);
#ifdef DEVELOP
	  //mat_extract_ramp(&einfo);
	  //mat_extract_ramp_nt(&einfo, info->rowrem);
#endif
	  cpl_msg_info(cpl_func, "Extract a local 2-order polynome for each pixel row (different window for photometric and interferometric channels).");
	  mat_extract_poly2_pip(&einfo, 19, 79);
#ifdef DEVELOP
	  //mat_extract_poly2_pip_nt(&einfo, 19, 79, info->aqpcs, info->aqdcr);
	  //mat_extract_polyn(&einfo, MAT_HORIZONTAL_DIRECTION, 2, 19);
#endif
	}
      else
	{
	  // old implementation
	  /*
	  mat_extract_detector_horizontal_poly2_local1(ref, res, NULL,
						       MAT_UNDEFINED_INTENSITY, MAT_UNDEFINED_INTENSITY, 79,
						       1, info->det->nx,
						       1, info->det->ny,
						       info->poiy);
	  */
	  // new implementation
	  mat_extract_info_set_filter(&einfo, MAT_UNDEFINED_INTENSITY, MAT_UNDEFINED_INTENSITY);
	  mat_extract_info_set_window(&einfo, info->det, MAT_DETECTOR_WINDOW_TYPE, 0, 0, 0, 0);
	  cpl_msg_info(cpl_func, "Extract a local 2-order polynome for each pixel row.");
	  mat_extract_polyn(&einfo, MAT_HORIZONTAL_DIRECTION, 2, 79);
	}
      /*
      mat_extract_detector_horizontal_poly0_local(ref, res, range,
						  MAT_UNDEFINED_INTENSITY, MAT_UNDEFINED_INTENSITY, 71,
						  1, info->det->nx,
						  1, info->det->ny,
						  info->poiy);
      */
      /*
      mat_extract_channel_horizontal_poly2_global(info->det, ref, res, range,
						  MAT_LINEAR_INTENSITY, MAT_LINEAR_INTENSITY);
      */
      break;
    default:;
    }
  if (info->expert & FLAT_EXPERT_FLAG)
    {
      mat_image_store_plain_stl("flatraw.stl", raw, 1, 1, cpl_image_get_size_x(raw), cpl_image_get_size_y(raw), 16.0);
      //mat_image_store_plain_stl("flatref.stl", ref, 1, 1, cpl_image_get_size_x(raw), cpl_image_get_size_y(raw), 16.0);
      //mat_image_store_plain_stl("flatres.stl", res, 1, 1, cpl_image_get_size_x(raw), cpl_image_get_size_y(raw), 16.0);
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_flat_prop_calibrate(mat_cal_det_info *info)
{
  cpl_size    nx, ny, x, y;
  //cpl_mask   *bpm = NULL;
  //cpl_image  *ffm = NULL;

  cpl_msg_info(cpl_func, "Calculate calibrated pixel median and variance for the flatfield exposures.");
  nx = info->nx;
  ny = info->ny;
  //if (info->bpm != NULL) bpm = info->bpm->list_badpixel[0];
  //if (info->ffm != NULL) ffm = info->ffm->list_flatfield[0];
  for (y = 1; y <= ny; y++)
    {
      for (x = 1; x <= nx; x++)
	{
	  int        f, rejected;
	  for (f = 0; f < info->nbdit; f++)
	    {
	      double mraw = cpl_image_get(info->psm[f].rmed[FLAT_IDX], x, y, &rejected);
	      double vraw = cpl_image_get(info->psm[f].rvar[FLAT_IDX], x, y, &rejected);
	      double mcal = 0.0;
	      double vcal = 0.0;
	      double nlf;
	      if (mraw == 0.0)
		{
		  mcal = mraw;
		  vcal = vraw;
		}
	      else
		{
		  mcal = mat_nlm_flux_map_direct(info, mraw);
		  nlf = mcal/mraw;
		  vcal = vraw*nlf*nlf;
		}
	      cpl_image_set(info->psm[f].cmed[FLAT_IDX], x, y, mcal);
	      cpl_image_set(info->psm[f].cvar[FLAT_IDX], x, y, vcal);
	    }
	}
    }
  if (info->expert & FLAT_EXPERT_FLAG)
    {
      int   f;
      for (f = 0; f < info->nbdit; f++)
	{
	  cpl_msg_info(cpl_func, "Check flatfield pixel statistics for DIT %.3f (after nonlinearity compensation)", info->psm[f].dit);
	  mat_image_validate(info->psm[f].rmed[FLAT_IDX], NULL, 0.0, 65536.0, "raw median");
	  mat_image_validate(info->psm[f].rvar[FLAT_IDX], NULL, 0.0, 65536.0, "raw variance");
	  mat_image_validate(info->psm[f].cmed[FLAT_IDX], NULL, 0.0, 65536.0, "calibrated median");
	  mat_image_validate(info->psm[f].cvar[FLAT_IDX], NULL, 0.0, 65536.0, "calibrated variance");
	}
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_flat_prop_flux_calculate(mat_cal_det_info *info)
{
  cpl_image  *medcal = NULL;
  cpl_image  *varcal = NULL;

  medcal = info->flatmedraw;
  varcal = info->flatvarraw;
  cpl_image_copy(medcal, info->psm[0].cmed[FLAT_IDX], 1, 1);
  cpl_image_copy(varcal, info->psm[0].cvar[FLAT_IDX], 1, 1);
  mat_flat_prop_range(info);
  mat_flat_prop_reference(info);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_flat_prop_dit_calculate(mat_cal_det_info *info)
{
  int         f, rejected;
  cpl_size    nx, ny, x, y;
  cpl_image  *med;
  cpl_image  *var;

  cpl_msg_info(cpl_func, "Calculate an average flat image (nonlinearity compensated) using several dit.");
  med = info->flatmedraw;
  var = info->flatvarraw;
  nx = cpl_image_get_size_x(med);
  ny = cpl_image_get_size_y(med);
  for (y = 1; y <= ny; y++)
    {
      for (x = 1; x <= nx; x++)
	{
	  double limit = 0.0;
	  double mcal  = 0.0;
	  double vcal  = 0.0;
	  double dit = 0.0;
	  limit = cpl_image_get(info->nlm->list_limit[0], x, y, &rejected);
	  if (limit == 0.0)
	    {
	      cpl_image_set(med, x, y, mcal);
	      cpl_image_set(var, x, y, vcal);
	      continue;
	    }
	  for (f = 0; f < info->nbdit; f++)
	    {
	      double      mraw  = cpl_image_get(info->psm[f].rmed[FLAT_IDX], x, y, &rejected);
	      double      hmcal = cpl_image_get(info->psm[f].cmed[FLAT_IDX], x, y, &rejected);
	      double      hvcal = cpl_image_get(info->psm[f].cvar[FLAT_IDX], x, y, &rejected);
	      if (mraw > limit)
		{
		  continue;
		}
	      if (mraw > 0.5*info->max_nonlinear_range) continue;
	      dit  += info->psm[f].dit;
	      mcal += hmcal;
	      vcal += hvcal;
	    }
	  if (dit != 0.0)
	    {
	      mcal *= info->timescale/dit;
	      vcal *= info->timescale/dit;
	    }
	  cpl_image_set(med, x, y, mcal);
	  cpl_image_set(var, x, y, vcal);
	}
    }
  if ((info->expert & FLAT_EXPERT_FLAG) && (info->poix != 0) && (info->poiy != 0))
    {
      for (f = 0; f < info->nbdit; f++)
	{
	  cpl_msg_info(cpl_func, "pixel[%d,%d] => median = %g, variance = %g",
		       info->poix, info->poiy,
		       cpl_image_get(info->psm[f].rmed[FLAT_IDX], info->poix, info->poiy, &rejected),
		       cpl_image_get(info->psm[f].rvar[FLAT_IDX], info->poix, info->poiy, &rejected));
	}
    }
  mat_flat_prop_range(info);
  mat_flat_prop_reference(info);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_detect_detector_absolute(cpl_mask   *bpm,
						   cpl_image  *data,
						   cpl_image  *reference,
						   cpl_image  *range,
						   double      range_filter,
						   double      limit)
{
  int         nx, ny;
  int         x, y;
  int         count = 0;

  nx = cpl_image_get_size_x(data);
  ny = cpl_image_get_size_y(data);
  for (y = 1; y <= ny; y++)
    {
      for (x = 1; x <= nx; x++)
	{
	  int    rejected;
	  double px_ref   = cpl_image_get(reference, x, y, &rejected);
	  double px_data  = cpl_image_get(data, x, y, &rejected);
	  double px_range = cpl_image_get(range, x, y, &rejected);
	  if (rejected) continue;
	  if (cpl_mask_get(bpm, x, y) == CPL_BINARY_1) continue;
	  if (px_range != range_filter) continue;
	  if ((px_data >= px_ref - limit) && (px_data <= px_ref + limit)) continue;
	  count++;
	  cpl_mask_set(bpm, x, y, CPL_BINARY_1);
	}
    }
  cpl_msg_info(cpl_func, "mat_detect_detector_absolute() %d bad pixel added", count);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_detect_detector_relative(cpl_mask   *bpm,
						   cpl_image  *data,
						   cpl_image  *reference,
						   cpl_image  *range,
						   double      range_filter,
						   double      limit,
						   cpl_image  *relval)
{
  int         nx, ny;
  int         x, y;
  int         count = 0;

  nx = cpl_image_get_size_x(data);
  ny = cpl_image_get_size_y(data);
  if (relval != NULL)
    {
      mat_image_fill(relval, 0.0);
    }
  for (y = 1; y <= ny; y++)
    {
      for (x = 1; x <= nx; x++)
	{
	  int    rejected;
	  double px_ref   = cpl_image_get(reference, x, y, &rejected);
	  double px_data  = cpl_image_get(data, x, y, &rejected);
	  double px_range = cpl_image_get(range, x, y, &rejected);
	  if (rejected) continue;
	  if (cpl_mask_get(bpm, x, y) == CPL_BINARY_1) continue;
	  if (px_range != range_filter) continue;
	  if ((relval != NULL) && (px_ref != 0.0))
	    {
	      cpl_image_set(relval, x, y, (px_data - px_ref)/px_ref);
	    }
	  if ((px_data >= px_ref - px_ref*limit) && (px_data <= px_ref + px_ref*limit)) continue;
	  count++;
	  cpl_mask_set(bpm, x, y, CPL_BINARY_1);
	}
    }
  cpl_msg_info(cpl_func, "mat_detect_detector_relative() %d bad pixel added", count);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_bpm_allocate(mat_cal_det_info *info)
{
  if (info->bpm != NULL) return CPL_ERROR_NONE;
  info->bpm = mat_badpixel_new(info->det, info->imgdet);
  mat_assert_not_null(info->bpm, "could not create the bad pixel map");
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_bpm_calculate_preliminary(mat_cal_det_info *info)
{
  mat_badpixel   *bpm = NULL;

  cpl_msg_info(cpl_func, "Calculate a preliminary bad pixel map based on the dark properties.");
  bpm = info->bpm;
  mat_badpixel_reset(bpm);
  mat_detect_detector_absolute(bpm->list_badpixel[0],
			       info->darkmedraw,
			       info->darkmedref,
			       info->darkrange,
			       MAT_LOW_INTENSITY,
			       info->darklimit);
  bpm->nbbad = 0;
  bpm->nbbad += cpl_mask_count(bpm->list_badpixel[0]);
  bpm->nbgood = bpm->nbtotal - bpm->nbbad;
  bpm->goodpixelratio = ((double)bpm->nbgood)/((double)bpm->nbtotal);
  cpl_msg_info(cpl_func, "  bad %d good %d ratio %g", bpm->nbbad, bpm->nbgood, bpm->goodpixelratio);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_bpm_calculate_final(mat_cal_det_info *info, cpl_frameset *frameset)
{
  mat_badpixel   *bpm = NULL;
  
  bpm = info->bpm;
  cpl_msg_info(cpl_func, "Calculate a final bad pixel map based on the dark and flat properties.");
  mat_badpixel_reset(bpm);
  if (info->bptype != EMPTY_BP_TYPE)
    { // A bad pixel map with marked pixels are requested -> first estimate the bad pixels based on the current measurement
      mat_detect_detector_absolute(bpm->list_badpixel[0],
				   info->darkmedraw,
				   info->darkmedref,
				   info->darkrange,
				   MAT_LOW_INTENSITY,
				   info->darklimit);
      mat_detect_detector_relative(bpm->list_badpixel[0],
				   info->flatmedraw,
				   info->flatmedref,
				   info->flatrange,
				   MAT_LINEAR_INTENSITY,
				   info->flatlimit,
				   info->flatbpm);
      if (info->bptype == MERGE_BP_TYPE)
	{
	  int bpmcount = cpl_frameset_count_tags(frameset, MATISSE_DO_BPM);
	  if (bpmcount != 0)
	    {
	      int        count = 1;
	      int        limit;
	      int        x, y, rejected;
	      // The start is the current mad pixel map
	      cpl_image *merged =  cpl_image_new_from_mask(bpm->list_badpixel[0]);
	      cpl_frame *bpmframe;
	      bpmframe = cpl_frameset_find(frameset, MATISSE_DO_BPM); // already checked previously!
	      do
		{
		  mat_badpixel     *hbpm;
		  cpl_msg_info(cpl_func, "Merge bad pixel map given in %s", cpl_frame_get_filename(bpmframe));
		  cpl_frame_set_group(bpmframe, CPL_FRAME_GROUP_RAW);
		  hbpm = mat_badpixel_load(bpmframe);
		  if (hbpm)
		    { // add the loaded bad pixel mat to the counts
		      cpl_image *hmap =  cpl_image_new_from_mask(hbpm->list_badpixel[0]);
		      count++;
		      cpl_image_add(merged, hmap);
		      cpl_image_delete(hmap);
		      mat_badpixel_delete(hbpm);
		    }
		} while ((bpmframe = cpl_frameset_find(frameset, NULL)) != NULL);
	      limit = count/4;
	      if (limit == 0) limit = 1;
	      for (y = 1; y <= info->ny; y++)
		{
		  for (x = 1; x <= info->nx; x++)
		    {
		      int c = (int)cpl_image_get(merged, x, y, &rejected);
		      cpl_mask_set(bpm->list_badpixel[0], x, y, (c > limit));
		    }
		}
	    }
	}
    }
  bpm->nbbad = 0;
  bpm->nbbad += cpl_mask_count(bpm->list_badpixel[0]);
  bpm->nbgood = bpm->nbtotal - bpm->nbbad;
  bpm->goodpixelratio = ((double)bpm->nbgood)/((double)bpm->nbtotal);
  cpl_msg_info(cpl_func, "  bad %d good %d ratio %g", bpm->nbbad, bpm->nbgood, bpm->goodpixelratio);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_fit_info_ot_delete(mat_fit_info_ot *info)
{
  int i;

  for (i = 0; i < MAX_NBDIT; i++)
    {
      if (info->mxs[i] != NULL)
	{
	  cpl_matrix_delete(info->mxs[i]);
	  info->mxs[i] = NULL;
	}
      if (info->vys[i] != NULL)
	{
	  cpl_vector_delete(info->vys[i]);
	  info->vys[i] = NULL;
	}
      if (info->vyes[i] != NULL)
	{
	  cpl_vector_delete(info->vyes[i]);
	  info->vyes[i] = NULL;
	}
    }
  if (info->a != NULL)
    {
      cpl_vector_delete(info->a);
      info->a = NULL;
    }
  if (info->s != NULL)
    {
      cpl_vector_delete(info->s);
      info->s = NULL;
    }
  cpl_free(info);
  return CPL_ERROR_NONE;
}

static mat_fit_info_ot *mat_fit_info_ot_new(void)
{
  mat_fit_info_ot  *info = NULL;

  info = cpl_calloc(1, sizeof(mat_fit_info_ot));
  if (info == NULL)
    {
      cpl_msg_error(cpl_func, "cannot create the function fitting context");
      return NULL;
    }
  info->a = cpl_vector_new(MAX_COEFF);
  if (info->a == NULL)
    {
      cpl_msg_error(cpl_func, "cannot create the cpl_vector for the function coefficients");
      mat_fit_info_ot_delete(info);
      return NULL;
    }
  info->s = cpl_vector_new(1);
  if (info->s == NULL)
    {
      cpl_msg_error(cpl_func, "cannot create the cpl_vector for the sample value");
      mat_fit_info_ot_delete(info);
      return NULL;
    }
  info->change_scale = 1;
  return info;
}

/*
  x  = xscale*median[DU]
  y  = yscale*DIT[s]
  ye = yscale*sqrt(variance[DU^2])
 */
static cpl_error_code mat_fit_set_pixel(mat_cal_det_info *pinfo, mat_fit_info_ot *finfo, int px, int py)
{
  int      f, nf;
  int      pflag = ((pinfo->expert & NONLINEARITY_EXPERT_FLAG) && (px == pinfo->poix) && (py == pinfo->poiy));

  finfo->isok = 1;
  nf = pinfo->nbdit;
  if (pflag)
    {
      cpl_msg_info(cpl_func, "# nonlinearity for pixel %d %d", px, py);
    }
  // Check if the current pixel is a bad one -> abort
  if (mat_badpixel_get(pinfo->bpm, 0, px, py) == CPL_BINARY_1)
    {
      if ((px == pinfo->poix) && (py == pinfo->poiy))
	{
	  cpl_msg_info(cpl_func, "# bad pixel");
	}
      finfo->isok = 0;
      return CPL_ERROR_NONE;
    }
  // Create an artificial data point for (0.0, 0.0)
  finfo->x[0]  = 0.0;
  finfo->y[0]  = 0.0;
  finfo->ye[0] = pinfo->max_abs_deviation/pinfo->gain;
  finfo->nbtotal = 1;
  finfo->nbused  = 1;
  // Copy all original values into the context
  for (f = 0; f < nf; f++)
    {
      int    rejected;
      double dx = cpl_image_get(pinfo->psm[f].rmed[FLAT_IDX], px, py, &rejected);
      double dy = pinfo->psm[f].dit;
      double dye_abs = pinfo->max_abs_deviation/pinfo->gain;   // max absolute deviation in DU
      double dye_rel = pinfo->max_rel_deviation*dx;            // max relative deviation in DU
      double dye_pn  = sqrt(fabs(dx*pinfo->gain))/pinfo->gain; // derived photon noise in DU
      double dye_m   = sqrt(fabs(cpl_image_get(pinfo->psm[f].rvar[FLAT_IDX], px, py, &rejected)));
      double dye = CPL_MAX(dye_m, CPL_MAX(dye_abs, CPL_MAX(dye_rel, dye_pn)));
      if (dx < pinfo->min_linear_range) continue;
      finfo->x[finfo->nbtotal]  = dx;  // measured intensity [DU]
      finfo->y[finfo->nbtotal]  = dy;  // DIT [s]
      finfo->ye[finfo->nbtotal] = dye; // measured noise [DU]
      finfo->nbtotal++;
    }
  finfo->nbused = finfo->nbtotal;
  if (pflag)
    {
      cpl_msg_info(cpl_func, "Try to remove the data points which show a full-well behaviour.");
    }
  // Remove all data points which do not reach the minimum flat intensit (from trail only!)
  for (; (finfo->nbused != 0) && (finfo->x[finfo->nbused-1] < pinfo->min_linear_range); finfo->nbused--)
    {
    }
  // Try to remove the data points which show a full-well behaviour
  for (f = 1; f <= finfo->nbused-2; f++)
    {
      // the measured intesity difference between the current and next data point
      double dxm = finfo->x[f + 1] - finfo->x[f];
      // the theoretical intensity difference calculated from the slope (local DU/t)
      double dxt = finfo->x[f]/finfo->y[f]*finfo->y[f + 1] - finfo->x[f];
      if (pflag)
	{
	  cpl_msg_info(cpl_func, " x[%d] = %g x[%d] = %g -> dxm = %g dxt = %g",
		       f, finfo->x[f], f + 1, finfo->x[f + 1], dxm, dxt);
	}
      if (finfo->x[f] < 0.5*pinfo->max_linear_range) continue;
      if (dxm <= 0.0)
	{ // The measured intensity difference is negative!
	  // -> a strong hint that we have reached full well
	  // -> we stop at the current data point
	  finfo->nbused = f + 1;
	  break;
	}
      if (dxm < 0.2*dxt)
	{ // The measured intensity difference is smaller than the theoretical one
	  // -> a hint that we have reached full well
	  // -> we stop at the current data point
	  finfo->nbused = f + 1;
	  break;
	}
    }
  if (pflag)
    {
      cpl_msg_info(cpl_func, "# DU DIT Error Flag");
      for (f = 0; f < finfo->nbtotal; f++)
	{
	  cpl_msg_info(cpl_func, "# %g %g %g %d", finfo->x[f], finfo->y[f], finfo->ye[f], (f < finfo->nbused));
	}
    }
  if (finfo->nbused < 5)
    {
      finfo->isok = 0;
      return CPL_ERROR_UNSPECIFIED;
    }
  else
    return CPL_ERROR_NONE;
}

/*
  uses the previously calculated nonlinearity to set an average nonlinearity response
 */
static cpl_error_code mat_fit_set_detector(mat_cal_det_info *pinfo, mat_fit_info_ot *finfo)
{
  int      nx, ny, px, py;
  double   raw   = 0.0;
  double   delta = 65536.0/(double)MAX_NBDIT;

  nx = pinfo->nx;
  ny = pinfo->ny;
  finfo->isok = 1;
  // Create an artificial data point for (0.0, 0.0)
  finfo->x[0]  = 0.0;
  finfo->y[0]  = 0.0;
  finfo->ye[0] = pinfo->max_abs_deviation/pinfo->gain;
  finfo->nbtotal = 1;
  finfo->nbused  = 1;
  if (pinfo->expert & NONLINEARITY_EXPERT_FLAG)
    {
      cpl_msg_info(cpl_func, "# det nonlin # data # title Detector nonlinearity response for the HAWAII-2RG detector.");
      cpl_msg_info(cpl_func, "# det nonlin # data # column 1 raw intensity [DU]");
      cpl_msg_info(cpl_func, "# det nonlin # data # column 2 calibrated intensity [DU]");
      cpl_msg_info(cpl_func, "# det nonlin # data # column 3 error of the calibrated intensity [DU]");
      cpl_msg_info(cpl_func, "# det nonlin # data # column 4 number of pixels which are used for the calibrated intensity");
      cpl_msg_info(cpl_func, "# det nonlin # data # column 5 25 percent quantil of the calibrated intensity [DU]");
      cpl_msg_info(cpl_func, "# det nonlin # data # column 6 75 percent quantil of the calibrated intensity [DU]");
      cpl_msg_info(cpl_func, "# det nonlin # data # column 7 mode of the calibrated intensity [DU]");
      cpl_msg_info(cpl_func, "# det nonlin # data # column 8 median calibrated intensity [DU]");
      cpl_msg_info(cpl_func, "# det nonlin # data # column 9 mean calibrated intensity [DU]");
      cpl_msg_info(cpl_func, "# det nonlin # data 0.0 0.0 0.0 1 0.0 0.0 0.0 0.0 0.0");
    }
  // incrementally calculate the average nonlinearity response by using the local response for all pixels with an limit above a given raw intensity
  for (raw = delta; raw < 65536.0; raw += delta)
    { // estimate the calibrated value for a given raw intensity
      pinfo->stat1 = mat_statistics_realloc(pinfo->stat1, nx*ny, MAT_OUTLIER_FACTOR);
      mat_statistics_reset(pinfo->stat1);
      for (px = pinfo->det->frame_left + 1; px <= nx - pinfo->det->frame_right; px++)
	{
	  for (py = pinfo->det->frame_bottom + 1; py <= ny - pinfo->det->frame_top - 1; py++)
	    {
	      int    rejected;
	      double limit, a0, a1, a2, a3;
	      limit = cpl_image_get(pinfo->nlm->list_limit[0], px, py, &rejected);
	      if (limit == 0.0) continue;
	      if (limit < (raw - 0.5*delta)) continue;
	      a0 = cpl_image_get(cpl_imagelist_get(pinfo->nlm->list_coeff[0], 0), px, py, &rejected);
	      a1 = cpl_image_get(cpl_imagelist_get(pinfo->nlm->list_coeff[0], 1), px, py, &rejected);
	      a2 = cpl_image_get(cpl_imagelist_get(pinfo->nlm->list_coeff[0], 2), px, py, &rejected);
	      a3 = cpl_image_get(cpl_imagelist_get(pinfo->nlm->list_coeff[0], 3), px, py, &rejected);
	      if ((a0 == 1.0) && (a1 == 0.0) && (a2 == 0.0) && (a3 == 0.0)) continue;
	      mat_statistics_add_value(pinfo->stat1, MAT_DU_SCALE*mat_nlm_func(raw/MAT_DU_SCALE, a0, a1, a2, a3));
	      /*
	      if (raw == 52224.0)
		{
		  cpl_msg_info(cpl_func, "# raw[%d,%d] %g limit %g a0-3 %g %g %g %g cal %f", px, py, raw, limit, a0, a1, a2, a3, MAT_DU_SCALE*mat_nlm_func(raw/MAT_DU_SCALE, a0, a1, a2, a3));
		}
	      */
	    }
	}
      if (pinfo->stat1->whole_count < 17) continue;
      pinfo->stat1->pflag = (pinfo->expert & NONLINEARITY_EXPERT_FLAG) && ((raw == 1024.0) || (raw == 6144.0) || (raw == 10240.0) || (raw == 17408.0) || (raw == 20480.0) || (raw == 52224.0));
      mat_statistics_calc(pinfo->stat1);
      if (pinfo->stat1->reduced_count < 11) continue;
      finfo->x[finfo->nbtotal]  = raw;                                  // measured intensity [DU]
      //finfo->y[finfo->nbtotal]  = pinfo->stat1->whole_mode;             // calibrated intensity [DU]
      finfo->y[finfo->nbtotal]  = pinfo->stat1->reduced_median;         // calibrated intensity [DU], median -> best results
      //finfo->y[finfo->nbtotal]  = pinfo->stat1->reduced_mean;           // calibrated intensity [DU]
      finfo->ye[finfo->nbtotal] = sqrt(pinfo->stat1->reduced_variance); // measured noise [DU]
      // <raw> <cal> <cal-error> <count> <q25> <q50> <q75> <peak> <mean>
      if (pinfo->expert & NONLINEARITY_EXPERT_FLAG)
	{
	  cpl_msg_info(cpl_func, "# det nonlin # data %.1f %.1f %.1f %d %.1f %.1f %.1f %.1f %.1f",
		       finfo->x[finfo->nbtotal],
		       finfo->y[finfo->nbtotal],
		       finfo->ye[finfo->nbtotal],
		       pinfo->stat1->reduced_count,
		       pinfo->stat1->q25,
		       pinfo->stat1->q75,
		       pinfo->stat1->whole_mode,
		       pinfo->stat1->reduced_median,
		       pinfo->stat1->reduced_mean);
	}
      finfo->nbtotal++;
      finfo->nbused++;
    }
  return CPL_ERROR_NONE;
}

/** Calculate the exposure time scale so that the linear part will be about (not exactly) 1. */
static cpl_error_code mat_fit_calc_scale(mat_fit_info_ot *finfo)
{
  int    f;
  double sumx = finfo->x[1];
  double sumy = finfo->y[1];

  if (finfo->nbused < 5)
    {
      finfo->isok = 0;
      finfo->xscale = 1.0/MAT_DU_SCALE;  // [DU] * xscale -> 0.0 .. 6.5535
      finfo->yscale = 1.0/MAT_DU_SCALE;  // [s]  * yscale -> 0.0 .. 6.????
      return CPL_ERROR_UNSPECIFIED;
    }
  for (f = 2; f < finfo->nbtotal; f++)
    {
      if (finfo->x[f] <= 10000.0)
	{
	  sumx += finfo->x[f];
	  sumy += finfo->y[f];
	}
    }
  finfo->xscale = 1.0/MAT_DU_SCALE;        // [DU] * xscale -> 0.0 .. 6.5535
  finfo->yscale = sumx/sumy/MAT_DU_SCALE;  // [s]  * yscale -> 0.0 .. 6.????
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_fit_use_values(mat_cal_det_info *pinfo, mat_fit_info_ot *finfo, int px, int py)
{
  int      f;

  // Allocate the cpl_matrix and cpl_vectors needed for the function fit
  if (finfo->mxs[finfo->nbused] == NULL)
    {
      finfo->mxs[finfo->nbused] = cpl_matrix_new(finfo->nbused, 1);
      mat_assert_not_null(finfo->mxs[finfo->nbused], "could not create the matrix for the x values");
    }
  if (finfo->vys[finfo->nbused] == NULL)
    {
      finfo->vys[finfo->nbused] = cpl_vector_new(finfo->nbused);
      mat_assert_not_null(finfo->vys[finfo->nbused], "could not create the vector for the y values");
    }
  if (finfo->vyes[finfo->nbused] == NULL)
    {
      finfo->vyes[finfo->nbused] = cpl_vector_new(finfo->nbused);
      mat_assert_not_null(finfo->vyes[finfo->nbused], "could not create the vector for the y error values");
    }
  // Fill the cpl_matrix and the cpl_vectors for the function fit
  for (f = 0; f < finfo->nbused; f++)
    {
      if ((px == pinfo->poix) && (py == pinfo->poiy) && (pinfo->expert & NONLINEARITY_EXPERT_FLAG))
	{
	  cpl_msg_debug(cpl_func, "%g %g %g", finfo->xscale*finfo->x[f], finfo->yscale*finfo->y[f], finfo->xscale*finfo->ye[f]);
	}
      cpl_matrix_set(finfo->mxs[finfo->nbused], f, 0, finfo->xscale*finfo->x[f]);
      cpl_vector_set(finfo->vys[finfo->nbused], f, finfo->yscale*finfo->y[f]);
      cpl_vector_set(finfo->vyes[finfo->nbused], f, finfo->xscale*finfo->ye[f]);
    }
  return CPL_ERROR_NONE;
}

static void mat_fit_show_poi(mat_cal_det_info *pinfo, mat_fit_info_ot *finfo, int px, int py, cpl_error_code ec)
{
  if ((pinfo->expert & NONLINEARITY_EXPERT_FLAG) == 0) return;
  if (px != pinfo->poix) return;
  if (py != pinfo->poiy) return;
  cpl_msg_info(cpl_func, "# fit ->");
  cpl_msg_info(cpl_func, "#    ec = %d nbtotal = %d nbused = %d",
	       (int)ec, finfo->nbtotal, finfo->nbused);
  cpl_msg_info(cpl_func, "#    stdev = %g limit %g",
	       finfo->stdev, finfo->limit);
  cpl_msg_info(cpl_func, "#    xscale = %g yscale = %g slope = %g",
	       finfo->xscale,
	       finfo->yscale,
	       cpl_vector_get(finfo->a, 0) + cpl_vector_get(finfo->a, 1)*exp(-cpl_vector_get(finfo->a, 2)*cpl_vector_get(finfo->a, 3)));
  cpl_msg_info(cpl_func, "#    f(x) = %g*x + %g*x*exp((x - %g)*%g)",
	       cpl_vector_get(finfo->a, 0),
	       cpl_vector_get(finfo->a, 1),
	       cpl_vector_get(finfo->a, 2),
	       cpl_vector_get(finfo->a, 3));
}

static cpl_error_code mat_fit_pixel_subset(mat_cal_det_info *pinfo, mat_fit_info_ot *finfo, int px, int py, int nbused)
{
  cpl_error_code  ec = CPL_ERROR_NONE;
  double          a0, a1, a2, a3;
  double          s0;
  int             f;

  // Set the start values for all three coefficients
  cpl_vector_set(finfo->a, 0, 1.0); // slope
  cpl_vector_set(finfo->a, 1, 0.6); //
  cpl_vector_set(finfo->a, 2, 7.6); //
  cpl_vector_set(finfo->a, 3, 0.4); //
  finfo->nbused = nbused;
  finfo->ia[0] = 1;
  finfo->ia[1] = 1;
  finfo->ia[2] = 1;
  finfo->ia[3] = 1;
  // rescale the original data points according to xscale and yscale and fill the vectors and matrices
  ec = mat_fit_use_values(pinfo, finfo, px, py);
  // do the function fit, this may fail!
  ec = cpl_fit_lvmq(finfo->mxs[finfo->nbused], NULL,
		    finfo->vys[finfo->nbused], NULL,
		    finfo->a, finfo->ia,
		    mat_fit_nonlin_h2_f, mat_fit_nonlin_h2_dfda,
		    CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
		    &(finfo->mse), NULL, NULL);
  if (ec != CPL_ERROR_NONE) finfo->isok = 0;
  // calculate the current slope and maybe change the scaling in order to hane a slope of nearly 1.0 (the first guess is maybe slightly wrong)
  a0 = cpl_vector_get(finfo->a, 0);
  a1 = cpl_vector_get(finfo->a, 1);
  a2 = cpl_vector_get(finfo->a, 2);
  a3 = cpl_vector_get(finfo->a, 3);
  s0 = a0 + a1*exp(-a2*a3);
  if ((a0 < 0.3) || (a0 >=  1.1)) finfo->isok = 0;
  if ((a1 < 0.0) || (a1 >=  2.0)) finfo->isok = 0;
  if ((a2 < 0.0) || (a2 >= 20.0)) finfo->isok = 0;
  if ((a3 < 0.0) || (a3 >= 40.0)) finfo->isok = 0;
  if ((pinfo->expert & NONLINEARITY_EXPERT_FLAG) && (px == pinfo->poix) && (py == pinfo->poiy))
    {
      cpl_msg_debug(cpl_func, "coeff = %g %g %g %g df/dx(0) = %g ok = %d", a0, a1, a2, a3, s0, finfo->isok);
    }
  if (!finfo->isok)
    {
      cpl_vector_set(finfo->a, 0, 1.0);
      cpl_vector_set(finfo->a, 1, 0.0);
      cpl_vector_set(finfo->a, 2, 0.0);
      cpl_vector_set(finfo->a, 3, 0.0);
      finfo->stdev = 0.0;
      finfo->limit = finfo->x[nbused-1];
      return ec;
    }
  if (finfo->change_scale)
    {
      finfo->yscale /= s0;
      ec = mat_fit_use_values(pinfo, finfo, px, py);
      ec = cpl_fit_lvmq(finfo->mxs[finfo->nbused], NULL,
			finfo->vys[finfo->nbused], NULL,
			finfo->a, finfo->ia,
			mat_fit_nonlin_h2_f, mat_fit_nonlin_h2_dfda,
			CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
			&(finfo->mse), NULL, NULL);
      if (ec != CPL_ERROR_NONE)
	{ // function fit does not work, revert to linear mapping
	  finfo->isok = 0;
	  cpl_vector_set(finfo->a, 0, 1.0);
	  cpl_vector_set(finfo->a, 1, 0.0);
	  cpl_vector_set(finfo->a, 2, 0.0);
	  cpl_vector_set(finfo->a, 3, 0.0);
	  finfo->stdev = 0.0;
	  finfo->limit = finfo->x[nbused-1];
	  return ec;
	}
    }
  a0 = cpl_vector_get(finfo->a, 0);
  a1 = cpl_vector_get(finfo->a, 1);
  a2 = cpl_vector_get(finfo->a, 2);
  a3 = cpl_vector_get(finfo->a, 3);
  // Calculate the standard deviation between the fit and the measured values
  finfo->stdev = 0.0;
  if (finfo->isok)
    {
      int inside_limit = 1;
      finfo->limit = 0.0;
      for (f = 1; f < finfo->nbused; f++)
	{
	  double bc, bi; // brightness calculated and ideal
	  double x, y;
	  bi = finfo->y[f]*finfo->yscale/finfo->xscale; // ideal brightness in [DU] (DIT -(yscale)-> y -(xscale)-> DU)
	  x = finfo->x[f]*finfo->xscale;
	  y = mat_nlm_func(x, a0, a1, a2, a3);
	  bc = y/finfo->xscale; // calculated brightness in [DU] (y -(xscale)-> DU)
	  if (x <= 1.0)
	    {
	      finfo->limit = finfo->x[f];
	    }
	  else if (inside_limit && (fabs((bi - bc)/bi) <= pinfo->max_rel_deviation))
	    {
	      finfo->limit = finfo->x[f];
	    }
	  else
	    {
	      inside_limit = 0;
	    }
	  if ((pinfo->expert & NONLINEARITY_EXPERT_FLAG) && (px == pinfo->poix) && (py == pinfo->poiy))
	    {
	      cpl_msg_debug(cpl_func, "original %d %g %g -> ideal %g calculated %g dist2 %g rel %g",
			    f, finfo->x[f], finfo->y[f], bi, bc, (bc - bi)*(bc - bi), (bi - bc)/bi);
	    }
	  finfo->stdev += (bc - bi)*(bc - bi)/(bi*bi);
	}
      finfo->stdev = sqrt(finfo->stdev/(double)finfo->nbused);
    }
  if (pinfo->expert & NONLINEARITY_EXPERT_FLAG)
    {
      mat_fit_show_poi(pinfo, finfo, px, py, ec);
    }
  return ec;
}


static cpl_error_code mat_nlm_allocate(mat_cal_det_info *info)
{
  int i;
  
  if (info->nlm != NULL) return CPL_ERROR_NONE;
  // allways allocate memory for a direct nonlinearity mapping (global nonlinearity only)
  info->nonlinmap = (double*)cpl_calloc(256*256, sizeof(double));
  if (info->nonlinmap == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for the raw -> cal mapping");
      return CPL_ERROR_UNSPECIFIED;
    }
  for (i = 0; i < 256*256; i++) info->nonlinmap[i] = (double)i;
  // determine the nonlinearity mode, because this influences the memory allocation, characterization and usage
  if (info->nbdit == 1)
    {
      info->nlmode = FLUX_NLMODE;
      info->nlm = mat_nonlinearity_new(info->det, info->imgdet, 0);
      strncpy(info->nlm->detnllaw, "a*x+b*(1-exp(-c*x**d))+e*(exp(f*x)-1)", sizeof(info->nlm->detnllaw));
      info->nlm->detnllaw[sizeof(info->nlm->detnllaw)-1] = '\0';
      info->nlm->detnlncoeff = 6;
      // allocate memory for the raw -> cal nonlinearity mapping
      info->polyraw = (double*)cpl_calloc(256*256, sizeof(double));
      if (info->polyraw == NULL)
	{
	  cpl_msg_error(cpl_func, "could not allocate memory for the nonlinearity polyline (raw)");
	  return CPL_ERROR_UNSPECIFIED;
	}
      info->polycal = (double*)cpl_calloc(256*256, sizeof(double));
      if (info->polycal == NULL)
	{
	  cpl_msg_error(cpl_func, "could not allocate memory for the nonlinearity polyline (cal)");
	  return CPL_ERROR_UNSPECIFIED;
	}
      info->polycalsmooth = (double*)cpl_calloc(256*256, sizeof(double));
      if (info->polycalsmooth == NULL)
	{
	  cpl_msg_error(cpl_func, "could not allocate memory for the nonlinearity polyline (calsmooth)");
	  return CPL_ERROR_UNSPECIFIED;
	}
      info->nl_nbentries = 65536/info->nlstep + 1;
      info->nl_map = cpl_calloc(info->nl_nbentries, sizeof(double));
      if (info->nl_map == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the global raw -> cal mapping");
	  return CPL_ERROR_UNSPECIFIED;
	}
      info->nl_limit = 0.0;
      if (info->nlm->rcmap != NULL)
	{ // we need to resize the raw -> cal mapping image
	  cpl_vector_delete(info->nlm->rcmap);
	  info->nlm->rcmap = cpl_vector_new(info->nl_nbentries);
	  if (info->nlm->rcmap == NULL)
	    {
	      cpl_msg_error(cpl_func, "could not reallocate the raw -> cal mapping (NLM)");
	      return CPL_ERROR_UNSPECIFIED;
	    }
	}
      if (info->nlcount == NULL)
	{
	  info->nlcount = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
	  if (info->nlcount == NULL)
	    {
	      cpl_msg_error(cpl_func, "cannot allocate memory for the nonlinearity counter (raw)");
	      return CPL_ERROR_UNSPECIFIED;
	    }
	}
      mat_image_fill(info->nlcount, 0.0);
    }
  else
    {
      info->nlmode = DIT_NLMODE;
      info->nlm = mat_nonlinearity_new(info->det, info->imgdet, 4);
      strncpy(info->nlm->detnllaw, "a*x+b*x*exp((x-c)*d)", sizeof(info->nlm->detnllaw));
      info->nlm->detnllaw[sizeof(info->nlm->detnllaw)-1] = '\0';
      info->nlm->detnlncoeff = 4;
      if (info->ditscale == NULL)
	{
	  info->ditscale = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
	  if (info->ditscale == NULL)
	    {
	      cpl_msg_error(cpl_func, "cannot allocate memory for the DIT scale values");
	      return CPL_ERROR_UNSPECIFIED;
	    }
	}
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_nlm_flux_extract_intensities(mat_cal_det_info *info)
{
  int x, y;
  int idx;
  cpl_image   *i1    = NULL;
  cpl_image   *i2    = NULL;
  cpl_image   *i3    = NULL;
  cpl_image   *i4    = NULL;
  cpl_image   *i13   = NULL;
  cpl_image   *i123  = NULL;
  cpl_image   *i1234 = NULL;
  double       limit;

  // Fill all intensity infos
  if (info->nbdit != 1)
    {
      cpl_msg_info(cpl_func, "Nonlinearity characterization is only for one DIT available.");
      return CPL_ERROR_UNSPECIFIED;
    }
  if (info->nlcount == NULL) return CPL_ERROR_UNSPECIFIED;
  i1    = info->psm[0].rmed[SH1_IDX];
  i2    = info->psm[0].rmed[SH2_IDX];
  i3    = info->psm[0].rmed[SH3_IDX];
  i4    = info->psm[0].rmed[SH4_IDX];
  i13   = info->psm[0].rmed[SH13_IDX];
  i123  = info->psm[0].rmed[SH123_IDX];
  i1234 = info->psm[0].rmed[SH1234_IDX];
  if (i1 == NULL)
    {
      cpl_msg_error(cpl_func, "no flatfield exposure with shutter 1 open");
      return CPL_ERROR_UNSPECIFIED;
    }
  if (i2 == NULL)
    {
      cpl_msg_error(cpl_func, "no flatfield exposure with shutter 2 open");
      return CPL_ERROR_UNSPECIFIED;
    }
  if (i3 == NULL)
    {
      cpl_msg_error(cpl_func, "no flatfield exposure with shutter 3 open");
      return CPL_ERROR_UNSPECIFIED;
    }
  if (i4 == NULL)
    {
      cpl_msg_error(cpl_func, "no flatfield exposure with shutter 4 open");
      return CPL_ERROR_UNSPECIFIED;
    }
  if ((i13 == NULL) && (i123 == NULL) && (i1234 == NULL))
    {
      cpl_msg_error(cpl_func, "no flatfield exposure with two, three or four shutters open");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->nbintensities = 0;
  limit = info->nlstart - info->nlstep;
  // Calculate the number of intensities
  for (y = 1; y <= info->ny; y++)
    {
      for (x = MAT_AQ_INTERF_LEFT; x <= MAT_AQ_INTERF_RIGHT; x++)
	{
	  int      rejected;
	  double   v1, v2, v3, v4;
	  v1 = cpl_image_get(i1, x, y, &rejected);
	  v2 = cpl_image_get(i2, x, y, &rejected);
	  v3 = cpl_image_get(i3, x, y, &rejected);
	  v4 = cpl_image_get(i4, x, y, &rejected);
	  if (i13   != NULL)
	    {
	      double v13     = cpl_image_get(i13, x, y, &rejected);
	      int    isphoto = 0; // if ((v1 ~ v13) || (v3 ~ v13)) -> ignore it
	      isphoto |= (fabs(v1 - v13) < 0.5*v13/2.0); // v1 should be about v13/2.0
	      isphoto |= (fabs(v3 - v13) < 0.5*v13/2.0); // v3 should be about v13/2.0
	      //isphoto |= (fabs(v1) < 0.5*v13/2.0);
	      //isphoto |= (fabs(v3) < 0.5*v13/2.0);
	      if ((v13 >= limit) && !isphoto) info->nbintensities++;
	    }
	  if (i123  != NULL)
	    {
	      double v123    = cpl_image_get(i123, x, y, &rejected);
	      int    isphoto = 0; // if ((v1 ~ v123) || (v2 ~ v123) || (v3 ~ v123)) -> ignore it
	      isphoto = 0;
	      isphoto |= (fabs(v1 - v123) < 0.5*v123/3.0); // v1 should be about v123/3.0
	      isphoto |= (fabs(v2 - v123) < 0.5*v123/3.0); // v2 should be about v123/3.0
	      isphoto |= (fabs(v3 - v123) < 0.5*v123/3.0); // v3 should be about v123/3.0
	      //isphoto |= (fabs(v1) < 0.5*v123/3.0);
	      //isphoto |= (fabs(v2) < 0.5*v123/3.0);
	      //isphoto |= (fabs(v3) < 0.5*v123/3.0);
	      if ((v123 >= limit) && !isphoto) info->nbintensities++;
	    }
	  if (i1234 != NULL)
	    {
	      double v1234   = cpl_image_get(i1234, x, y, &rejected);
	      int    isphoto = 0; // if ((v1 ~ v1234) || (v2 ~ v1234) || (v3 ~ v1234) || (v4 ~ v1234)) -> ignore it
	      isphoto |= (fabs(v1 - v1234) < 0.5*v1234/4.0); // v1 should be about v1234/4.0
	      isphoto |= (fabs(v2 - v1234) < 0.5*v1234/4.0); // v2 should be about v1234/4.0
	      isphoto |= (fabs(v3 - v1234) < 0.5*v1234/4.0); // v3 should be about v1234/4.0
	      isphoto |= (fabs(v4 - v1234) < 0.5*v1234/4.0); // v4 should be about v1234/4.0
	      //isphoto |= (fabs(v1) < 0.5*v1234/4.0);
	      //isphoto |= (fabs(v2) < 0.5*v1234/4.0);
	      //isphoto |= (fabs(v3) < 0.5*v1234/4.0);
	      //isphoto |= (fabs(v4) < 0.5*v1234/4.0);
	      if ((v1234 >= limit) && !isphoto) info->nbintensities++;
	    }
	}
    }
  if (info->nbintensities < 1000)
    {
      cpl_msg_error(cpl_func, "Not enough pixels with usefull intensities found");
      return CPL_ERROR_UNSPECIFIED;
    }
  // Allocate memory for all intensities
  info->intensities = (mat_intensity_info*)cpl_calloc(info->nbintensities, sizeof(mat_intensity_info));
  if (info->intensities == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for %d intensities", info->nbintensities);
      return CPL_ERROR_UNSPECIFIED;
    }
  idx = 0;
  for (y = 1; y <= info->ny; y++)
    {
      for (x = MAT_AQ_INTERF_LEFT; x <= MAT_AQ_INTERF_RIGHT; x++)
	{
	  int rejected;
	  double v1, v2, v3, v4;
	  v1 = cpl_image_get(i1, x, y, &rejected);
	  v2 = cpl_image_get(i2, x, y, &rejected);
	  v3 = cpl_image_get(i3, x, y, &rejected);
	  v4 = cpl_image_get(i4, x, y, &rejected);
	  if (i13 != NULL)
	    { // shutter 1 and 3 are open
	      double v13     = cpl_image_get(i13, x, y, &rejected);
	      int    isphoto = 0; // if ((v1 ~ v13) || (v3 ~ v13)) -> ignore it
	      isphoto |= (fabs(v1 - v13) < 0.5*v13/2.0); // v1 should be about v13/2.0
	      isphoto |= (fabs(v3 - v13) < 0.5*v13/2.0); // v3 should be about v13/2.0
	      //isphoto |= (fabs(v1) < 0.5*v13/2.0);
	      //isphoto |= (fabs(v3) < 0.5*v13/2.0);
	      if ((v13 >= limit) && !isphoto)
		{
		  info->intensities[idx].x = x;
		  info->intensities[idx].y = y;
		  info->intensities[idx].i1234 = v13;
		  info->intensities[idx].i1 = v1;
		  info->intensities[idx].i2 = 0.0;
		  info->intensities[idx].i3 = v3;
		  info->intensities[idx].i4 = 0.0;
		  if (info->expert & NONLINEARITY_EXPERT_FLAG)
		    {
		      cpl_msg_debug(cpl_func, "# intensity %d %d %d %.2f %.2f %.2f %.2f %.2f", idx, x, y, v13, v1, 0.0, v3, 0.0);
		    }
		  cpl_image_set(info->nlcount, x, y, cpl_image_get(info->nlcount, x, y, &rejected) + 1);
		  idx++;
		}
	    }
	  if (i123 != NULL)
	    { // shutter 1, 2 and 3 are open
	      double v123    = cpl_image_get(i123, x, y, &rejected);
	      int    isphoto = 0; // if ((v1 ~ v123) || (v2 ~ v123) || (v3 ~ v123)) -> ignore it
	      isphoto = 0;
	      isphoto |= (fabs(v1 - v123) < 0.5*v123/3.0); // v1 should be about v123/3.0
	      isphoto |= (fabs(v2 - v123) < 0.5*v123/3.0); // v2 should be about v123/3.0
	      isphoto |= (fabs(v3 - v123) < 0.5*v123/3.0); // v3 should be about v123/3.0
	      //isphoto |= (fabs(v1) < 0.5*v123/3.0);
	      //isphoto |= (fabs(v2) < 0.5*v123/3.0);
	      //isphoto |= (fabs(v3) < 0.5*v123/3.0);
	      if ((v123 >= limit) && !isphoto)
		{
		  info->intensities[idx].x = x;
		  info->intensities[idx].y = y;
		  info->intensities[idx].i1234 = v123;
		  info->intensities[idx].i1 = v1;
		  info->intensities[idx].i2 = v2;
		  info->intensities[idx].i3 = v3;
		  info->intensities[idx].i4 = 0.0;
		  if (info->expert & NONLINEARITY_EXPERT_FLAG)
		    {
		      cpl_msg_debug(cpl_func, "# intensity %d %d %d %.2f %.2f %.2f %.2f %.2f", idx, x, y, v123, v1, v2, v3, 0.0);
		    }
		  cpl_image_set(info->nlcount, x, y, cpl_image_get(info->nlcount, x, y, &rejected) + 1);
		  idx++;
		}
	    }
	  if (i1234 != NULL)
	    { // shutter 1, 2 and 3 are open
	      double v1234   = cpl_image_get(i1234, x, y, &rejected);
	      int    isphoto = 0; // if ((v1 ~ v1234) || (v2 ~ v1234) || (v3 ~ v1234) || (v4 ~ v1234)) -> ignore it
	      isphoto |= (fabs(v1 - v1234) < 0.5*v1234/4.0); // v1 should be about v1234/4.0
	      isphoto |= (fabs(v2 - v1234) < 0.5*v1234/4.0); // v2 should be about v1234/4.0
	      isphoto |= (fabs(v3 - v1234) < 0.5*v1234/4.0); // v3 should be about v1234/4.0
	      isphoto |= (fabs(v4 - v1234) < 0.5*v1234/4.0); // v4 should be about v1234/4.0
	      //isphoto |= (fabs(v1) < 0.5*v1234/4.0);
	      //isphoto |= (fabs(v2) < 0.5*v1234/4.0);
	      //isphoto |= (fabs(v3) < 0.5*v1234/4.0);
	      //isphoto |= (fabs(v4) < 0.5*v1234/4.0);
	      if ((v1234 >= limit) && !isphoto)
		{
		  info->intensities[idx].x = x;
		  info->intensities[idx].y = y;
		  info->intensities[idx].i1234 = v1234;
		  info->intensities[idx].i1 = v1;
		  info->intensities[idx].i2 = v2;
		  info->intensities[idx].i3 = v3;
		  info->intensities[idx].i4 = v4;
		  if (info->expert & NONLINEARITY_EXPERT_FLAG)
		    {
		      cpl_msg_debug(cpl_func, "# intensity %d %d %d %.2f %.2f %.2f %.2f %.2f", idx, x, y, v1234, v1, v2, v3, v4);
		    }
		  cpl_image_set(info->nlcount, x, y, cpl_image_get(info->nlcount, x, y, &rejected) + 1);
		  idx++;
		}
	    }
	}
    }
  // sort the pixel intensities
  qsort(info->intensities, info->nbintensities, sizeof(mat_intensity_info), mat_compare_intensity);
  return CPL_ERROR_NONE;
}

static void mat_nlm_flux_map_update(mat_cal_det_info *info, double raw, double cal)
{
  int    x_local;
  double x1_local, x2_local;
  double y1_local, y2_local;
  double m_local;

  if ((info->nbpoly != 0) && (raw <= info->polyraw[info->nbpoly - 1] + 0.5*(double)(info->nlstep))) return; // ignore this raw/cal pair
  info->polyraw[info->nbpoly] = raw;
  info->polycal[info->nbpoly] = cal;
  info->nbpoly++;
  if (info->nbpoly < 2) return;
  //cpl_msg_info(cpl_func, "updating nonlinearity map with %g -> %g", raw, cal);
  x1_local = info->polyraw[info->nbpoly - 2];
  y1_local = info->polycal[info->nbpoly - 2];
  x2_local = info->polyraw[info->nbpoly - 1];
  y2_local = info->polycal[info->nbpoly - 1];
  m_local = (y2_local - y1_local)/(x2_local - x1_local);
  x_local = (int)round(x1_local);
  while ((x_local < 65536) && ((double)x_local <= x2_local))
    {
      info->nonlinmap[x_local] = y1_local + m_local*((double)x_local - x1_local);
      x_local++;
    }
}

static cpl_error_code mat_nlm_flux_estimate_nonlinearity(mat_cal_det_info *info, double raw, double step)
{
  int      lower, upper, idx;
  double   llimit, ulimit;
  double   xmedian, ymedian, fmedian;

  info->stat1 = mat_statistics_realloc(info->stat1, 1024*1024, MAT_OUTLIER_FACTOR);
  if (info->stat1 == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the statistics (info->stat1)");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->stat2 = mat_statistics_realloc(info->stat2, 1024*1024, MAT_OUTLIER_FACTOR);
  if (info->stat2 == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the statistics (info->stat2)");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->stat3 = mat_statistics_realloc(info->stat3, 1024*1024, MAT_OUTLIER_FACTOR);
  if (info->stat3 == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the statistics (info->stat3)");
      return CPL_ERROR_UNSPECIFIED;
    }
  // find the intensity which is inside the range and near the lower limit
  lower = -1;
  for (idx = info->nbintensities - 1; idx >= 0; idx--)
    {
      if (info->intensities[idx].i1234 < (raw - step))  break;    // intensity < lower limit
      if (info->intensities[idx].i1234 > (raw + step))  continue; // intensity > upper limit
      lower = idx; // intensity inside range and the correct detector channel
    }
  // find the intensity which is inside the range and near the upper limit
  upper = -1;
  for (idx = 0; idx < info->nbintensities; idx++)
    {
      if (info->intensities[idx].i1234 < (raw - step))  continue; // intensity < lower limit
      if (info->intensities[idx].i1234 > (raw + step))  break;    // intensity > upper limit
      upper = idx; // intensity inside range and the correct detector channel
    }
  // check the indices
  if ((lower == -1) || (upper == -1))
    {
      // no data point inside the range found -> exit
      return CPL_ERROR_UNSPECIFIED;
    }
  // calculate the outlier limits
  mat_statistics_reset(info->stat1);
  for (idx = lower; idx <= upper; idx++)
    {
      double y = 0.0;
      y += mat_nlm_flux_map_direct(info, info->intensities[idx].i1);
      y += mat_nlm_flux_map_direct(info, info->intensities[idx].i2);
      y += mat_nlm_flux_map_direct(info, info->intensities[idx].i3);
      y += mat_nlm_flux_map_direct(info, info->intensities[idx].i4);
      //y /= info->nl_scale_dc[info->intensities[idx].dc];
      mat_statistics_add_value(info->stat1, y);
    }
  mat_statistics_calc(info->stat1);
  /*
  cpl_msg_info(cpl_func, "# stat %.2f %d : %d [%.1f .. %.1f] -> %d [%.2f .. %.2f]",
	       raw, c,
	       info->statistics->whole_count,
	       raw - step, raw + step,
	       info->statistics->reduced_count,
	       info->statistics->limit_min, info->statistics->limit_max);
  */
  // check if we have enough values, excluding the outliers, for a 2-degree polynome
  if (info->stat1->reduced_count < 13)
    {
      // not enough data point inside the range found -> exit
      return CPL_ERROR_UNSPECIFIED;
    }
  // info->stat1->limit_min and limit_min contains the outlier limits (for I1 + I2 + I3 + I4 !)
  llimit  = info->stat1->limit_min;
  ulimit  = info->stat1->limit_max;
  mat_statistics_reset(info->stat1);
  mat_statistics_reset(info->stat2);
  mat_statistics_reset(info->stat3);
  if (info->expert & NONLINEARITY_EXPERT_FLAG)
    {
      cpl_msg_debug(cpl_func, "# nlin outlier %.2f %.2f %.2f", raw, llimit, ulimit);
    }
  for (idx = lower; idx <= upper; idx++)
    {
      double x = info->intensities[idx].i1234;
      double y = 0.0;
      y += mat_nlm_flux_map_direct(info, info->intensities[idx].i1);
      y += mat_nlm_flux_map_direct(info, info->intensities[idx].i2);
      y += mat_nlm_flux_map_direct(info, info->intensities[idx].i3);
      y += mat_nlm_flux_map_direct(info, info->intensities[idx].i4);
      //y /= info->nl_scale_dc[info->intensities[idx].dc];
      if (y < llimit) continue; // outlier
      if (y > ulimit) continue; // outlier
      if (info->expert & NONLINEARITY_EXPERT_FLAG)
	{
	  cpl_msg_debug(cpl_func, "# nlin data %.2f %.2f %.2f", raw, x, y);
	}
      mat_statistics_add_value(info->stat1, x);
      mat_statistics_add_value(info->stat2, y);
      mat_statistics_add_value(info->stat3, y/x);
    }
  mat_statistics_calc(info->stat1);
  mat_statistics_calc(info->stat2);
  mat_statistics_calc(info->stat3);
  xmedian = info->stat1->whole_median;
  ymedian = info->stat2->whole_median;
  fmedian = info->stat3->whole_median;
  mat_nlm_flux_map_update(info, xmedian, ymedian);
  if (info->expert & NONLINEARITY_EXPERT_FLAG)
    {
      cpl_msg_info(cpl_func, "# nlin raw %.2f %d %.2f %.2f %.6f",
		   raw, (int)info->stat1->whole_count, xmedian, ymedian, fmedian);
    }
  return CPL_ERROR_NONE;
}

static void mat_nlm_flux_smooth_polyline(mat_cal_det_info *info, int plength)
{
  int                 i, j;
  
  if (info->expert & NONLINEARITY_EXPERT_FLAG)
    {
      cpl_msg_info(cpl_func, "# nlin smooth # title Smoothing of the detector nonlinearity for the Aquarius detector.");
      cpl_msg_info(cpl_func, "# nlin smooth # column 1 raw intensity [DU]");
      cpl_msg_info(cpl_func, "# nlin smooth # column 2 smoothed calibrated intensity (local 2-order polynome) [DU]");
    }
  for (i = plength/2; i < info->nbpoly - plength/2; i++)
    {
      double   det, a, b, c;
      double   count = 0.0;
      double   sx    = 0.0;
      double   sxx   = 0.0;
      double   sxxx  = 0.0;
      double   sxxxx = 0.0;
      double   sy    = 0.0;
      double   sxy   = 0.0;
      double   sxxy  = 0.0;
      for (j = -plength/2; j <= plength/2; j++)
	{
	  double x = info->polyraw[i + j] - info->polyraw[i];
	  double y = info->polycal[i + j];
	  count += 1.0;
	  sx    += x;
	  sxx   += x*x;
	  sxxx  += x*x*x;
	  sxxxx += x*x*x*x;
	  sy    += y;
	  sxy   += x*y;
	  sxxy  += x*x*y;
	}
      // det -> sxx^3 - 2 sx sxx sxxx + N sxxx^2 + sx^2 sxxxx - N sxx sxxxx
      // a -> -((-sxx^2 sxxy + sx sxxx sxxy + sxx sxxx sxy - sx sxxxx sxy - sxxx^2 sy + sxx sxxxx sy)/det)
      // b -> -((sx sxx sxxy - N sxxx sxxy - sxx^2 sxy + N sxxxx sxy + sxx sxxx sy - sx sxxxx sy)/det)
      // c -> -((-sx^2 sxxy + N sxx sxxy + sx sxx sxy - N sxxx sxy - sxx^2 sy + sx sxxx sy)/det)
      det = sxx*sxx*sxx - 2*sx*sxx*sxxx + count*sxxx*sxxx + sx*sx*sxxxx - count*sxx*sxxxx;
      a   = -(-sxx*sxx*sxxy + sx*sxxx*sxxy + sxx*sxxx*sxy - sx*sxxxx*sxy - sxxx*sxxx*sy + sxx*sxxxx*sy)/det;
      b   = -(sx*sxx*sxxy - count*sxxx*sxxy - sxx*sxx*sxy + count*sxxxx*sxy + sxx*sxxx*sy - sx*sxxxx*sy)/det;
      c   = -(-sx*sx*sxxy + count*sxx*sxxy + sx*sxx*sxy - count*sxxx*sxy - sxx*sxx*sy + sx*sxxx*sy)/det;
      if (i == plength/2)
	{ // interpolate the values between first and x-1
	  for (j = -plength/2; j <= -1; j++)
	    {
	      double x = info->polyraw[i + j] - info->polyraw[i];
	      info->polycalsmooth[i + j] = a + b*x + c*x*x;
	      if (info->expert & NONLINEARITY_EXPERT_FLAG)
		{
		  cpl_msg_info(cpl_func, "# nlin smooth %.2f %.2f", info->polyraw[i + j], info->polycalsmooth[i + j]);
		}
	    }
	}
      info->polycalsmooth[i] = a;
      if (info->expert & NONLINEARITY_EXPERT_FLAG)
	{
	  cpl_msg_info(cpl_func, "# nlin smooth %.2f %.2f", info->polyraw[i], info->polycalsmooth[i]);
	}
      if (i == info->nbpoly - plength/2 - 1)
	{ // interpolate between x+1 and last
	  for (j = 1; j <= plength/2; j++)
	    {
	      double x = info->polyraw[i + j] - info->polyraw[i];
	      info->polycalsmooth[i + j] = a + b*x + c*x*x;
	      if (info->expert & NONLINEARITY_EXPERT_FLAG)
		{
		  cpl_msg_info(cpl_func, "# nlin smooth %.2f %.2f", info->polyraw[i + j], info->polycalsmooth[i + j]);
		}
	    }
	}
    }
}

#define MAT_NONLIN_AQ_SCALE  0.001

/*
//#define MAT_NONLIN_AQ_NA     5
//  g[x_] := a*x + b*(1 - Exp[c*x]) + d*(Exp[e*x] - 1)
//  =>
//  D[g[x], x] = a - E^(c x) b c + E^(e x) d e
//  D[g[x], a] = x
//  D[g[x], b] = 1 - E^(c x)
//  D[g[x], c] = -E^(c x) b x
//  D[g[x], d] = -1 + E^(e x)
//  D[g[x], e] = E^(e x) d x
static int mat_fit_nonlin_aq_f(const double x[], const double a[], double *result)
{
  // g[x_] := a*x + b*(1 - Exp[c*x]) + d*(Exp[e*x] - 1)
  *result = a[0]*x[0] + a[1]*(1 - exp(a[2]*x[0])) + a[3]*(exp(a[4]*x[0]) - 1);
  return 0;
}
static int mat_fit_nonlin_aq_dfda(const double x[], const double a[], double result[])
{
  // D[g[x], a] = x
  // D[g[x], b] = 1 - E^(c x)
  // D[g[x], c] = -E^(c x) b x
  // D[g[x], d] = -1 + E^(e x)
  // D[g[x], e] = E^(e x) d x
  double t1 = exp(a[2]*x[0]); // E^(c x)
  double t2 = exp(a[4]*x[0]); // E^(e x)
  result[0] = x[0];
  result[1] = 1.0 - t1;
  result[2] = -t1*a[1]*x[0];
  result[3] = t2 - 1.0;
  result[4] = t2*a[4]*x[0];
  return 0;
}
*/

#define MAT_NONLIN_AQ_NA_T2 4
#define MAT_NONLIN_AQ_NA_T3 6
//  f[x_] := fa*x + fb*(1 - Exp[-fc*x^fd]) + fe*(Exp[ff*x] - 1)
//  =>
//  D[f[x],  x] = fa + E^(ff x) fe ff + E^(-fc x^fd) fb fc fd x^(-1 + fd)
//  D[f[x], fa] = x
//  D[f[x], fb] = 1 - E^(-fc x^fd)
//  D[f[x], fc] = E^(-fc x^fd) fb x^fd
//  D[f[x], fd] = E^(-fc x^fd) fb fc x^fd Log[x]
//  D[f[x], fe] = -1 + E^(ff x)
//  D[f[x], ff] = E^(ff x) fe x
static int mat_fit_nonlin_aq_t2_f(const double x[], const double a[], double *result)
{
  // f[x_] := fa*x + fb*(1 - Exp[-(fc*x)^fd])
  *result = a[0]*x[0] + a[1]*(1 - exp(-a[2]*pow(x[0], a[3])));
  return 0;
}
static int mat_fit_nonlin_aq_t2_dfda(const double x[], const double a[], double result[])
{
//  D[f[x], fa] = x
//  D[f[x], fb] = 1 - E^(-fc x^fd)
//  D[f[x], fc] = E^(-fc x^fd) fb x^fd
//  D[f[x], fd] = E^(-fc x^fd) fb fc x^fd Log[x]
  double t1 = exp(-a[2]*pow(x[0], a[3])); // E^(-fc x^fd)
  result[0] = x[0];
  result[1] = 1.0 - t1;
  result[2] = t1*a[1]*pow(x[0], a[3]);
  if (x[0] <= 0.0)
    {
      result[3] = 0.0;
    }
  else
    {
      result[3] = t1*a[1]*a[2]*pow(x[0], a[3])*log(x[0]);
    }
  return 0;
}

static int mat_fit_nonlin_aq_t3_f(const double x[], const double a[], double *result)
{
  // f[x_] := fa*x + fb*(1 - Exp[-fc*x^fd]) + fe*(Exp[ff*x] - 1)
  mat_fit_nonlin_aq_t2_f(x, a, result);
  *result += a[4]*(expm1(a[5]*x[0])); // expm1(x) == exp(x) - 1, but with better accuracy
  return 0;
}
static int mat_fit_nonlin_aq_t3_dfda(const double x[], const double a[], double result[])
{
//  D[f[x], fe] = -1 + E^(ff x)
//  D[f[x], ff] = E^(ff x) fe x
  double t2 = exp(a[5]*x[0]); // E^(ff x)
  mat_fit_nonlin_aq_t2_dfda(x, a, result);
  result[4] = t2 - 1.0;
  result[5] = t2*a[4]*x[0];
  return 0;
}

static cpl_error_code mat_nlm_flux_smooth_fit(mat_cal_det_info *info)
{
  int              ncount;
  //double           raw;
  int              i;
  cpl_error_code   ec = CPL_ERROR_NONE;
  cpl_matrix      *mx;
  cpl_vector      *vy  = NULL;
  cpl_vector      *a   = NULL;
  int              ia[MAT_NONLIN_AQ_NA_T3];
  double           mse;
  double           dit;
  //double           dy;
  double           fa, fb, fc, fd, fe, ff;

  // 1. Step: Make a first guess of the function parameters
  dit = 1000.0*info->psm[0].dit;
  fa = 0.25 + 0.01*dit; //(mat_nlm_flux_map_direct(info, 20000.0) - mat_nlm_flux_map_direct(info, 10000.0))/10000.0;
  fb = mat_nlm_flux_map_direct(info, 10000.0) - 10000.0*fa;
  fc = CPL_MIN(0.001, 0.02/dit/dit); // 0.0001
  fd = 0.95+0.009*dit; // 1.0
  ff = 0.000095 + 0.0000003*dit; // 0.0001
  //dy = mat_nlm_flux_map_direct(info, 40000.0) - (fb + 40000.0*fa);
  fe = 20.0 + 1400.0/dit; // dy/(exp(ff*40000.0) - 1.0);
  cpl_msg_info(cpl_func, "#    f(x) = %g*x + %g*(1 - exp(-%g*x**%g)) + %g*(exp(%g*x) - 1) (guess)", fa, fb, fc, fd, fe, ff);
  /*
  // 2. Step: use the first two terms and only data points up to 20480 (20480/nstep points)
  // allocate memory for ncount data pairs
  ncount = 10240/info->nlstep;
  mx = cpl_matrix_new(ncount, 1);
  mat_assert_not_null(mx, "could not create the matrix for the x values");
  vy = cpl_vector_new(ncount);
  mat_assert_not_null(vy, "could not create the vector for the y values");
  a = cpl_vector_new(MAT_NONLIN_AQ_NA_T2);
  mat_assert_not_null(a, "could not create the vector for the coefficients");
  // use only the first data pairs up to raw = 20480
  raw = (double)(info->nlstep);
  for (i = 0; i < ncount; i++)
    {
      double cal = mat_nlm_flux_map_direct(info, raw);
      cpl_matrix_set(mx,  i, 0, raw);
      cpl_vector_set(vy,  i,    cal);
      raw += (double)(info->nlstep);
    }
  // make the first function fit using only two terms (linear and 1. exponential)
  cpl_msg_info(cpl_func, "#    f(x) = %g*x + %g*(1 - exp(-%g*x**%g)) (init)", fa, fb, fc, fd);
  cpl_vector_set(a, 0, fa); ia[0] = 1;
  cpl_vector_set(a, 1, fb); ia[1] = 1;
  cpl_vector_set(a, 2, fc); ia[2] = 1;
  cpl_vector_set(a, 3, fd); ia[3] = 0;
  ec = cpl_fit_lvmq(mx, NULL,
		    vy, NULL,
		    a, ia,
		    mat_fit_nonlin_aq_t2_f, mat_fit_nonlin_aq_t2_dfda,
		    CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
		    &mse, NULL, NULL);
  fa = cpl_vector_get(a, 0);
  fb = cpl_vector_get(a, 1);
  fc = cpl_vector_get(a, 2);
  fd = cpl_vector_get(a, 3);
  cpl_msg_info(cpl_func, "#    f(x) = %g*x + %g*(1 - exp(-%g*x**%g)) (final)", fa, fb, fc, fd);
  // if the first fit does not work -> fallback to smoothing via local 2-order polynome and return
  if (ec != CPL_ERROR_NONE)
    {
      cpl_matrix_delete(mx);
      cpl_vector_delete(vy);
      cpl_vector_delete(a);
      cpl_msg_info(cpl_func, "#    function fit does not work, fallback to smoothing via local 2-order polynome (ec = %d)", ec);
      mat_nlm_flux_smooth_polyline(info, MAT_SMOOTH_NONLIN_PLENGTH);
      return CPL_ERROR_NONE;
    }
  */
  /*
  // 2. Step: Try to fit fd
  cpl_msg_info(cpl_func, "#    f(x) = %g*x + %g*(1 - exp(-%g*x**%g)) (init)", fa, fb, fc, fd);
  cpl_vector_set(a, 0, fa); ia[0] = 1;
  cpl_vector_set(a, 1, fb); ia[1] = 1;
  cpl_vector_set(a, 2, fc); ia[2] = 1;
  cpl_vector_set(a, 3, fd); ia[3] = 1;
  ec = cpl_fit_lvmq(mx, NULL,
		    vy, NULL,
		    a, ia,
		    mat_fit_nonlin_aq_t2_f, mat_fit_nonlin_aq_t2_dfda,
		    CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
		    &mse, NULL, NULL);
  fa = cpl_vector_get(a, 0);
  fb = cpl_vector_get(a, 1);
  fc = cpl_vector_get(a, 2);
  fd = cpl_vector_get(a, 3);
  cpl_msg_info(cpl_func, "#    f(x) = %g*x + %g*(1 - exp(-%g*x**%g)) (final)", fa, fb, fc, fd);
  // if the first fit does not work -> fallback to smoothing via local 2-order polynome and return
  if (ec != CPL_ERROR_NONE)
    {
      cpl_matrix_delete(mx);
      cpl_vector_delete(vy);
      cpl_vector_delete(a);
      cpl_msg_info(cpl_func, "#    function fit does not work, fallback to smoothing via local 2-order polynome (ec = %d)", ec);
      mat_nlm_flux_smooth_polyline(info, MAT_SMOOTH_NONLIN_PLENGTH);
      return CPL_ERROR_NONE;
    }
  */
  // 3. Step: use all terms and all data pairs
  // allocate memory for ncount data pairs
  ncount = info->nbpoly;
  //cpl_matrix_delete(mx);
  //cpl_vector_delete(vy);
  //cpl_vector_delete(a);
  mx = cpl_matrix_new(ncount, 1);
  if (mx == NULL)
    {
      cpl_msg_error(cpl_func, "could not create the matrix for the x values");
      return CPL_ERROR_NULL_INPUT;
    }
  vy = cpl_vector_new(ncount);
  if (vy == NULL)
    {
      cpl_matrix_delete(mx);
      cpl_msg_error(cpl_func, "could not create the vector for the y values");
      return CPL_ERROR_NULL_INPUT;
    }
  a = cpl_vector_new(MAT_NONLIN_AQ_NA_T3);
  if (a == NULL)
    {
      cpl_matrix_delete(mx);
      cpl_vector_delete(vy);
      cpl_msg_error(cpl_func, "could not create the vector for the coefficients");
      return CPL_ERROR_NULL_INPUT;
    }
  // Fill the cpl_matrix and the cpl_vectors for the function fit
  for (i = 0; i < ncount; i++)
    {
      cpl_matrix_set(mx,  i, 0, info->polyraw[i]);
      cpl_vector_set(vy,  i,    info->polycal[i]);
    }
  // make the second function fit using all terms (linear, 1. exponential and 2. exponential)
  cpl_msg_info(cpl_func, "#    f(x) = %g*x + %g*(1 - exp(-%g*x**%g)) + %g*(exp(%g*x) - 1) (init)", fa, fb, fc, fd, fe, ff);
  cpl_vector_set(a, 0, fa); ia[0] = 1;
  cpl_vector_set(a, 1, fb); ia[1] = 1;
  cpl_vector_set(a, 2, fc); ia[2] = 1;
  cpl_vector_set(a, 3, fd); ia[3] = 1;
  cpl_vector_set(a, 4, fe); ia[4] = 1;
  cpl_vector_set(a, 5, ff); ia[5] = 1;
  ec = cpl_fit_lvmq(mx, NULL,
		    vy, NULL,
		    a, ia,
		    mat_fit_nonlin_aq_t3_f, mat_fit_nonlin_aq_t3_dfda,
		    CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
		    &mse, NULL, NULL);
  fa = cpl_vector_get(a, 0);
  fb = cpl_vector_get(a, 1);
  fc = cpl_vector_get(a, 2);
  fd = cpl_vector_get(a, 3);
  fe = cpl_vector_get(a, 4);
  ff = cpl_vector_get(a, 5);
  cpl_msg_info(cpl_func, "#    f(x) = %g*x + %g*(1 - exp(-%g*x**%g)) + %g*(exp(%g*x) - 1) (final)", fa, fb, fc, fd, fe, ff);
  info->nlm->detnlcoeff[0] = fa;
  info->nlm->detnlcoeff[1] = fb;
  info->nlm->detnlcoeff[2] = fc;
  info->nlm->detnlcoeff[3] = fd;
  info->nlm->detnlcoeff[4] = fe;
  info->nlm->detnlcoeff[5] = ff;
  if (ec == CPL_ERROR_NONE)
    {
      for (i = 0; i < ncount; i++)
	{
	  double x[1];
	  double y;
	  x[0] = info->polyraw[i]; //*MAT_NONLIN_AQ_SCALE;
	  mat_fit_nonlin_aq_t3_f(x, cpl_vector_get_data(a), &y);
	  info->polycalsmooth[i] = y; ///MAT_NONLIN_AQ_SCALE;
	}
    }
  else
    {
      cpl_msg_info(cpl_func, "#    function fit does not work, fallback to smoothing via local 2-order polynome (ec = %d)", ec);
      mat_nlm_flux_smooth_polyline(info, MAT_SMOOTH_NONLIN_PLENGTH);
    }
  cpl_matrix_delete(mx);
  cpl_vector_delete(vy);
  cpl_vector_delete(a);
  return ec;
}

static void mat_nlm_flux_calc_stdev(mat_cal_det_info *info)
{
  cpl_image *stdev;
  cpl_image *i1_img, *i2_img, *i3_img, *i4_img, *i13_img, *i123_img, *i1234_img;
  int        x, y, nx, ny;

  if (info->nlm->list_stdev == NULL) return;
  if (info->nlm->list_stdev[0] == NULL) return;
  if (info->nlmode != FLUX_NLMODE) return;
  stdev = info->nlm->list_stdev[0];
  mat_image_fill(stdev, 0.0);
  nx = info->nx;
  ny = info->ny;
  i1_img    = info->psm[0].rmed[SH1_IDX];
  i2_img    = info->psm[0].rmed[SH2_IDX];
  i3_img    = info->psm[0].rmed[SH3_IDX];
  i4_img    = info->psm[0].rmed[SH4_IDX];
  i13_img   = info->psm[0].rmed[SH13_IDX];
  i123_img  = info->psm[0].rmed[SH123_IDX];
  i1234_img = info->psm[0].rmed[SH1234_IDX];
  for (y = 1; y <= ny; y++)
    {
      for (x = 1; x <= nx; x++)
	{
	  int rejected;
	  double chi2 = 0.0;
	  double count = 0.0;
	  if ((i13_img != NULL) && (i1_img != NULL) && (i3_img != NULL))
	    {
	      double i1, i3, i13;
	      i1  = cpl_image_get(i1_img, x, y, &rejected);
	      i3  = cpl_image_get(i3_img, x, y, &rejected);
	      i13 = cpl_image_get(i13_img, x, y, &rejected);
	      if (i13 > info->min_linear_range)
		{
		  i1  = mat_nlm_flux_map_direct(info, i1);
		  i3  = mat_nlm_flux_map_direct(info, i3);
		  i13 = mat_nlm_flux_map_direct(info, i13);
		  chi2 += (i1 + i3 - i13)*(i1 + i3 - i13)/(i13*i13);
		  //chi2 += (i1 + i3 - i13)/i13;
		  count += 1.0;
		}
	    }
	  if ((i123_img != NULL) && (i1_img != NULL) && (i2_img != NULL) && (i3_img != NULL))
	    {
	      double i1, i2, i3, i123;
	      i1   = cpl_image_get(i1_img, x, y, &rejected);
	      i2   = cpl_image_get(i2_img, x, y, &rejected);
	      i3   = cpl_image_get(i3_img, x, y, &rejected);
	      i123 = cpl_image_get(i123_img, x, y, &rejected);
	      if (i123 > info->min_linear_range)
		{
		  i1   = mat_nlm_flux_map_direct(info, i1);
		  i2   = mat_nlm_flux_map_direct(info, i2);
		  i3   = mat_nlm_flux_map_direct(info, i3);
		  i123 = mat_nlm_flux_map_direct(info, i123);
		  chi2 += (i1 + i2 + i3 - i123)*(i1 + i2 + i3 - i123)/(i123*i123);
		  //chi2 += (i1 + i2 + i3 - i123)/i123;
		  count += 1.0;
		}
	    }
	  if ((i1234_img != NULL) && (i1_img != NULL) && (i2_img != NULL) && (i3_img != NULL) && (i4_img != NULL))
	    {
	      double i1, i2, i3, i4, i1234;
	      i1    = cpl_image_get(i1_img, x, y, &rejected);
	      i2    = cpl_image_get(i2_img, x, y, &rejected);
	      i3    = cpl_image_get(i3_img, x, y, &rejected);
	      i4    = cpl_image_get(i4_img, x, y, &rejected);
	      i1234 = cpl_image_get(i1234_img, x, y, &rejected);
	      if (i1234 > info->min_linear_range)
		{
		  i1    = mat_nlm_flux_map_direct(info, i1);
		  i2    = mat_nlm_flux_map_direct(info, i2);
		  i3    = mat_nlm_flux_map_direct(info, i3);
		  i4    = mat_nlm_flux_map_direct(info, i4);
		  i1234 = mat_nlm_flux_map_direct(info, i1234);
		  chi2 += (i1 + i2 + i3 + i4 - i1234)*(i1 + i2 + i3 + i4 - i1234)/(i1234*i1234);
		  //chi2 += (i1 + i2 + i3 + i4 - i1234)/i1234;
		  count += 1.0;
		}
	    }
	  if (count != 0.0)
	    {
	      cpl_image_set(stdev, x, y, sqrt(chi2/count));
	      //cpl_image_set(stdev, x, y, chi2/count);
	    }
	}
    }
  info->nlm->has_stdev = 1;
}

static cpl_error_code mat_nlm_flux_calculate(mat_cal_det_info *info)
{
  cpl_error_code ec;
  int            i, j;
  double         raw, step;
  double         cal;
  double         a, b, c, d, e, f;
  double         x, y;
  double         slope;
  int            oexpert;

  cpl_msg_info(cpl_func, "Calculate a nonlinearity response for the whole detector based on flux levels (different open shutter combinations).");
  // extract the intensity information from the flatfield exposures with different shutter combinations
  oexpert = info->expert;
  ec = mat_nlm_flux_extract_intensities(info);
  // initialize the polyline as 1:1 mapping
  for (i = 0; i < info->nl_nbentries; i++)
    {
      cal = (double)(i*info->nlstep);
      info->nl_map[i] = cal;
    }
  // ################
  // 1. Step: Calculate a first guess by assuming a 1:1 mapping below nlstart
  // -> This will result in a step at nlstart where the nonlinearity is not smooth
  slope = 1.0;
  info->nbpoly = 0;
  step = (double)info->nlstep;
  for (raw = 0.0; raw < info->nlstart; raw += step)
    {
      mat_nlm_flux_map_update(info, raw, slope*raw);
    }
  if (info->expert & NONLINEARITY_EXPERT_FLAG)
    {
      cpl_msg_info(cpl_func, "# nlin raw # title Incrementally estimated detector nonlinearity for the Aquarius detector.");
      cpl_msg_info(cpl_func, "# nlin raw # column 1 artificial raw intensity [DU]");
      cpl_msg_info(cpl_func, "# nlin raw # column 2 number of pixels used to estimate the calibrated intensity");
      cpl_msg_info(cpl_func, "# nlin raw # column 3 median raw intensity of the used pixels [DU]");
      cpl_msg_info(cpl_func, "# nlin raw # column 4 median calibrated intensity of the used pixels [DU]");
      cpl_msg_info(cpl_func, "# nlin raw # column 5 median of the factor calibrated/raw intensity");
    }
  for (raw = info->nlstart; raw < info->intensities[info->nbintensities - 1].i1234 - 5.0*step; raw += step)
    {
      mat_nlm_flux_estimate_nonlinearity(info, raw, step);
    }
  // ################
  // 2. Step: Estimate a better start of the nonlinearity.
  // -> Since the first guess has a step at nlstart, it is tried to find a better assumption for
  //    the nonlinearity below nlstart. The previous solution was smoothing the nonlinearity with
  //    a local 2-order polynom and use a straight line from (0,0) to (nlstart,cal(nlstart)).
  //    If a global function is fitted to all nonlinearity points (raw,cal), every points between
  //    0 and nlstart are replaced by this function.
  // Therefore two approaches do exist:
  //   - Use the first part of a globally fitted function
  //   - Use a smoothened polyline and a slope calculated by cal(nlstart)/nlstart
  //if (info->ntflag)
  //{ // Use the first part of a globally fitted function
  //mat_nlm_flux_smooth_fit(info);
  //}
  //else
  //{ // Use a smoothened polyline and a slope calculated by cal(nlstart)/nlstart
  mat_nlm_flux_smooth_polyline(info, MAT_SMOOTH_NONLIN_PLENGTH);
  //}
  // exchange the old mapping vs. the smooth mapping
  j = info->nbpoly;
  info->nbpoly = 0;
  for (i = 0; i < j; i++) {
    mat_nlm_flux_map_update(info, info->polyraw[i], info->polycalsmooth[i]);
  }
  // update the first part of the nonlinearity mapping (below nlstart)
  info->nbpoly = 0;
  for (raw = 0.0; raw < info->nlstart; raw += step)
    {
      mat_nlm_flux_map_update(info, raw, mat_nlm_flux_map_direct(info, raw));
    }
  // ################
  // 3. Step: Calculate a second guess using the adapted beginning (see step 2)
  info->expert = 0;
  for (raw = info->nlstart; raw < info->intensities[info->nbintensities - 1].i1234 - 5.0*step; raw += step)
    {
      mat_nlm_flux_estimate_nonlinearity(info, raw, step);
    }
  info->expert = oexpert;
  // ################
  // 4. Step: Apply a final smoothing and use this result
  if (info->ntflag)
    { // Use the whole globally fitted function
      mat_nlm_flux_smooth_fit(info);
    }
  else
    { // Use the whole smoothened polyline
      mat_nlm_flux_smooth_polyline(info, MAT_SMOOTH_NONLIN_PLENGTH);
    }
  if (info->expert & NONLINEARITY_EXPERT_FLAG)
    {
      cpl_msg_info(cpl_func, "# nlin poly # title Polygon representation of the detector nonlinearity of the Aquarius detector.");
      cpl_msg_info(cpl_func, "# nlin poly # column 1 raw intensity [DU]");
      cpl_msg_info(cpl_func, "# nlin poly # column 2 raw photon flux [DU/ms]");
      cpl_msg_info(cpl_func, "# nlin poly # column 3 calibrated intensity [DU]");
      cpl_msg_info(cpl_func, "# nlin poly # column 4 calibrated photon flux [DU/ms]");
      cpl_msg_info(cpl_func, "# nlin poly # column 5 smooth calibrated intensity [DU]");
      cpl_msg_info(cpl_func, "# nlin poly # column 6 smooth calibrated photon flux [DU/ms]");
      for (i = 0; i < info->nbpoly; i++)
	{
	  cpl_msg_info(cpl_func, "# nlin poly %.2f %.2f %.2f %.2f %.2f %.2f",
		       info->polyraw[i],       info->polyraw[i]/info->psm[0].dit/1000.0,
		       info->polycal[i],       info->polycal[i]/info->psm[0].dit/1000.0,
		       info->polycalsmooth[i], info->polycalsmooth[i]/info->psm[0].dit/1000.0);
	}
    }
  // exchange the original (wiggly) polyline with the smooth version
  info->nl_limit = info->polyraw[info->nbpoly - 1];
  info->polycalsmooth[0] = 0.0;
  j = info->nbpoly;
  info->nbpoly = 0;
  for (i = 0; i < j; i++) {
    mat_nlm_flux_map_update(info, info->polyraw[i], info->polycalsmooth[i]);
  }
  // extend the last straight line segment until 65535 DU
  x = 65535.0;
  y = 65536.0*info->polycal[info->nbpoly - 1]/info->polyraw[info->nbpoly - 1];
  mat_nlm_flux_map_update(info, x, y);
  // ################
  // 5. Step: Create the final result (nl_limit, rcmap, ...)
  mat_image_fill(info->nlm->list_limit[0], info->nl_limit);
  info->dlimit = info->nl_limit;
  if (info->nlm->rcmap != NULL)
    {
      info->nlm->has_rcmap = 1;
      if (info->expert & NONLINEARITY_EXPERT_FLAG)
	{
	  cpl_msg_info(cpl_func, "# nlin map # title Final raw -> calibrated intensity mapping of the detector nonlinearity of the Aquarius detector.");
	  cpl_msg_info(cpl_func, "# nlin map # column 1 raw intensity [DU]");
	  cpl_msg_info(cpl_func, "# nlin map # column 2 calibrated intensity [DU]");
	}
      for (i = 0; i < info->nl_nbentries; i++)
	{
	  raw = (double)(i*info->nlstep);
	  cal = mat_nlm_flux_map_direct(info, raw);
	  if (info->expert & NONLINEARITY_EXPERT_FLAG)
	    {
	      cpl_msg_info(cpl_func, "# nlin map %.2f %.2f", raw, cal);
	    }
	  cpl_vector_set(info->nlm->rcmap, i, cal);
	}
    }
  mat_nlm_flux_calc_stdev(info);
  // fill the direct raw -> cal mapping with the detector nonlinearity
  // AQ: "a*x+b*(1-exp(-c*x**d))+e*(exp(f*x)-1)"
  a = info->nlm->detnlcoeff[0];
  b = info->nlm->detnlcoeff[1];
  c = info->nlm->detnlcoeff[2];
  d = info->nlm->detnlcoeff[3];
  e = info->nlm->detnlcoeff[4];
  f = info->nlm->detnlcoeff[5];
  cpl_msg_info(cpl_func, "nonlinearity AQ: %g*x+%g*(1-exp(-%g*x**%g))+%g*(exp(%g*x)-1)", a, b, c, d, e, f);
  for (i = 0; i < 65536; i++)
    {
      x = (double)i;
      y = a*x + b*(1.0 - exp(-c*pow(x, d))) + e*(exp(f*x) - 1.0);
      info->nonlinmap[i] = y;
    }
  return ec;
}

static cpl_error_code mat_nlm_dit_fit_pixel(mat_cal_det_info *pinfo, mat_fit_info_ot *finfo, int px, int py)
{
  cpl_error_code  ec;
  int             nbused;
  double          stdev;
  double          limit;
  int             i;
  int             not_valid;
  int             pflag = (((pinfo->expert & NONLINEARITY_EXPERT_FLAG) != 0) && (px == pinfo->poix) && (py == pinfo->poiy));

  // initialize the function coefficients (ideal/linear response)
  cpl_vector_set(finfo->a, 0, 1.0); // slope
  cpl_vector_set(finfo->a, 1, 0.0); //
  cpl_vector_set(finfo->a, 2, 0.0); //
  cpl_vector_set(finfo->a, 3, 0.0); //
  finfo->ia[0] = 1;
  finfo->ia[1] = 1;
  finfo->ia[2] = 1;
  finfo->ia[3] = 1;
  finfo->isok = 1;
  ec = mat_fit_set_pixel(pinfo, finfo, px, py);
  if (ec == CPL_ERROR_NONE) ec = mat_fit_calc_scale(finfo);
  if (pflag)
    {
      cpl_msg_info(cpl_func, "# scaling: xscale %g yscale %g", finfo->xscale, finfo->yscale);
    }
  // The first function fit uses all values already marked as ok
  nbused = finfo->nbused;
  if (ec == CPL_ERROR_NONE) ec = mat_fit_pixel_subset(pinfo, finfo, px, py, nbused);
  if (ec != CPL_ERROR_NONE)
    {
      cpl_vector_set(finfo->a, 0, 1.0); // slope
      cpl_vector_set(finfo->a, 1, 0.0); //
      cpl_vector_set(finfo->a, 2, 0.0); //
      cpl_vector_set(finfo->a, 3, 0.0); //
      finfo->stdev = 0.0;
      finfo->limit = finfo->x[nbused-1];
    }
  stdev = finfo->stdev;
  limit = finfo->limit;
  while ((nbused >= 5) && (ec == CPL_ERROR_NONE))
    {
      ec = mat_fit_pixel_subset(pinfo, finfo, px, py, nbused - 1);
      if ((finfo->stdev < stdev) && (finfo->limit >= limit) && (ec == CPL_ERROR_NONE))
	{ // the function fit works better if we ignore the last data point
	  // we permanently ignore the last data point
	  nbused--;
	  stdev = finfo->stdev;
	  limit = finfo->limit;
	}
      else
	{ // the function fit is not better when we ignore the last data point
	  // -> we have to recalculat the last function fit and leave the loop
	  ec = mat_fit_pixel_subset(pinfo, finfo, px, py, nbused);
	  break;
	}
    }
  if (finfo->isok && (pinfo->ditscale != NULL))
    {
      cpl_image_set(pinfo->ditscale, px, py, finfo->yscale);
    }
  if (ec != CPL_ERROR_NONE)
    {
      cpl_vector_set(finfo->a, 0, 1.0); // slope
      cpl_vector_set(finfo->a, 1, 0.0); //
      cpl_vector_set(finfo->a, 2, 0.0); //
      cpl_vector_set(finfo->a, 3, 0.0); //
      finfo->stdev = 0.0;
      finfo->limit = finfo->x[nbused-1];
    }
  // Check if the coefficients are inside the expected range
  not_valid = 0;
  for (i = 0; i < 4; i++)
    {
      if (!isfinite(cpl_vector_get(finfo->a, i)))
	{
	  not_valid = 1;
	  cpl_msg_info(cpl_func, "pixel at (%d,%d), a%d = %f", px, py, i, cpl_vector_get(finfo->a, i));
	}
    }
  if (!isfinite(finfo->stdev))
    {
      not_valid = 1;
      cpl_msg_info(cpl_func, "pixel at (%d,%d), stdev = %f", px, py, finfo->stdev);
    }
  if (!isfinite(finfo->limit))
    {
      not_valid = 1;
      cpl_msg_info(cpl_func, "pixel at (%d,%d), limit = %f", px, py, finfo->limit);
    }
  if (not_valid)
    {
      cpl_vector_set(finfo->a, 0, 1.0); // slope
      cpl_vector_set(finfo->a, 1, 0.0); //
      cpl_vector_set(finfo->a, 2, 0.0); //
      cpl_vector_set(finfo->a, 3, 0.0); //
      finfo->stdev = 0.0;
      finfo->limit = finfo->x[nbused-1];
    }
  mat_fit_show_poi(pinfo, finfo, px, py, ec);
  // Store the coefficients in the nonlinearity map
  cpl_image_set(cpl_imagelist_get(pinfo->nlm->list_coeff[0], 0), px, py, cpl_vector_get(finfo->a, 0));
  cpl_image_set(cpl_imagelist_get(pinfo->nlm->list_coeff[0], 1), px, py, cpl_vector_get(finfo->a, 1));
  cpl_image_set(cpl_imagelist_get(pinfo->nlm->list_coeff[0], 2), px, py, cpl_vector_get(finfo->a, 2));
  cpl_image_set(cpl_imagelist_get(pinfo->nlm->list_coeff[0], 3), px, py, cpl_vector_get(finfo->a, 3));
  cpl_image_set(pinfo->nlm->list_stdev[0], px, py, finfo->stdev);
  cpl_image_set(pinfo->nlm->list_limit[0], px, py, finfo->limit);
  return ec;
}

//#define INCREASING_FIT 1
#define DECREASING_FIT 1

static cpl_error_code mat_nlm_dit_fit_detector(mat_cal_det_info *pinfo, mat_fit_info_ot *finfo)
{
  cpl_error_code  ec;
  int             nbused;
  double          stdev;
  int             ochange_scale = finfo->change_scale;
  int             i;
  double          a, b, c, d;
  double          x, y;
#ifdef DECREASING_FIT
  double          limit;
#endif
#ifdef INCREASING_FIT
  int             onbused;
#endif

  cpl_msg_info(cpl_func, "Calculate a global nonlinearity response (detector nonlinearity).");
  // initialize the function coefficients (ideal/linear response)
  cpl_vector_set(finfo->a, 0, 1.0); // slope
  cpl_vector_set(finfo->a, 1, 0.6); //
  cpl_vector_set(finfo->a, 2, 7.5); //
  cpl_vector_set(finfo->a, 3, 0.4); //
  finfo->ia[0] = 1;
  finfo->ia[1] = 1;
  finfo->ia[2] = 1;
  finfo->ia[3] = 1;
  finfo->isok = 1;
  finfo->change_scale = 0;
  ec = mat_fit_set_detector(pinfo, finfo);
  if (ec == CPL_ERROR_NONE) ec = mat_fit_calc_scale(finfo);
  finfo->yscale = finfo->xscale;
  cpl_msg_info(cpl_func, "# det scaling: xscale %g yscale %g", finfo->xscale, finfo->yscale);
  // The first function fit uses all values already marked as ok
#ifdef DECREASING_FIT
  for (nbused = finfo->nbused/10; nbused < finfo->nbused-1; nbused++)
    {
      if (finfo->ye[nbused+1] < finfo->ye[nbused]) break;
    }
  //nbused = finfo->nbused;
  //cpl_msg_info(cpl_func, "decreasing, trying %d data points", nbused);
  if (ec == CPL_ERROR_NONE) ec = mat_fit_pixel_subset(pinfo, finfo, pinfo->poix, pinfo->poiy, nbused);
  if (ec != CPL_ERROR_NONE)
    {
      cpl_vector_set(finfo->a, 0, 1.0); // slope
      cpl_vector_set(finfo->a, 1, 0.0); //
      cpl_vector_set(finfo->a, 2, 0.0); //
      cpl_vector_set(finfo->a, 3, 0.0); //
      finfo->stdev = 0.0;
    }
  stdev = finfo->stdev;
  limit = finfo->limit;
  while ((nbused >= 5) && (ec == CPL_ERROR_NONE))
    {
      //cpl_msg_info(cpl_func, "decreasing, trying %d data points", nbused - 1);
      ec = mat_fit_pixel_subset(pinfo, finfo, pinfo->poix, pinfo->poiy, nbused - 1);
      if ((finfo->stdev < stdev) && (finfo->limit >= limit) && (ec == CPL_ERROR_NONE))
	{ // the function fit works better if we ignore the last data point
	  // we permanently ignore the last data point
	  nbused--;
	  stdev = finfo->stdev;
	  limit = finfo->limit;
	}
      else
	{ // the function fit is not better when we ignore the last data point -> leave the loop
	  break;
	}
    }
#endif
#ifdef INCREASING_FIT
  onbused = finfo->nbused;
  nbused  = onbused/2;
  //cpl_msg_info(cpl_func, "increasing, trying %d data points", nbused);
  if (ec == CPL_ERROR_NONE) ec = mat_fit_pixel_subset(pinfo, finfo, pinfo->poix, pinfo->poiy, nbused);
  if (ec != CPL_ERROR_NONE)
    {
      cpl_vector_set(finfo->a, 0, 1.0); // slope
      cpl_vector_set(finfo->a, 1, 0.0); //
      cpl_vector_set(finfo->a, 2, 0.0); //
      cpl_vector_set(finfo->a, 3, 0.0); //
      finfo->stdev = 0.0;
    }
  stdev = finfo->stdev;
  while ((nbused < onbused-1) && (ec == CPL_ERROR_NONE))
    {
      //cpl_msg_info(cpl_func, "increasing, trying %d data points", nbused + 1);
      ec = mat_fit_pixel_subset(pinfo, finfo, pinfo->poix, pinfo->poiy, nbused + 1);
      if ((finfo->stdev < 1.2*stdev) && (ec == CPL_ERROR_NONE))
	{ // the function fit works better if we add the next data point
	  // we permanently add the next data point
	  nbused++;
	  if (finfo->stdev > stdev)
	    {
	      stdev = finfo->stdev;
	    }
	}
      else
	{ // the function fit is not better when we include the next data point -> leave the loop
	  break;
	}
    }
#endif
  // -> we have to recalculate the last function fit
  //cpl_msg_info(cpl_func, "final, trying %d data points", nbused);
  ec = mat_fit_pixel_subset(pinfo, finfo, pinfo->poix, pinfo->poiy, nbused);
  if (ec != CPL_ERROR_NONE)
    {
      pinfo->da0 = 1.0; // slope
      pinfo->da1 = 0.0;
      pinfo->da2 = 0.0;
      pinfo->da3 = 0.0;
      pinfo->dstdev = 0.0;
      pinfo->dlimit = 65536.0;
    }
  // Store the coefficients in the nonlinearity map
  pinfo->da0 = cpl_vector_get(finfo->a, 0);
  pinfo->da1 = cpl_vector_get(finfo->a, 1);
  pinfo->da2 = cpl_vector_get(finfo->a, 2);
  pinfo->da3 = cpl_vector_get(finfo->a, 3);
  pinfo->dstdev = finfo->stdev;
  pinfo->dlimit = finfo->limit;
  pinfo->nlm->detnlcoeff[0] = pinfo->da0;
  pinfo->nlm->detnlcoeff[1] = pinfo->da1;
  pinfo->nlm->detnlcoeff[2] = pinfo->da2;
  pinfo->nlm->detnlcoeff[3] = pinfo->da3;
  cpl_msg_info(cpl_func, "# det nonlin # nonlin f(x) = %f*x + %f*x*exp((x - %f)*%f) [0.0 .. %f] stdev %f",
	       pinfo->da0, pinfo->da1, pinfo->da2, pinfo->da3, pinfo->dlimit, pinfo->dstdev);
  if (pinfo->nltype == AVERAGE_NL_TYPE)
    { // use the global nonlinearity law for all pixels
      mat_image_fill(cpl_imagelist_get(pinfo->nlm->list_coeff[0], 0), pinfo->da0);
      mat_image_fill(cpl_imagelist_get(pinfo->nlm->list_coeff[0], 1), pinfo->da1);
      mat_image_fill(cpl_imagelist_get(pinfo->nlm->list_coeff[0], 2), pinfo->da2);
      mat_image_fill(cpl_imagelist_get(pinfo->nlm->list_coeff[0], 3), pinfo->da3);
      // mat_image_fill(pinfo->ditscale, pinfo->yscale);
      mat_image_fill(pinfo->nlm->list_stdev[0], pinfo->dstdev);
      mat_image_fill(pinfo->nlm->list_limit[0], pinfo->dlimit);
    }
  finfo->change_scale = ochange_scale;
  // fill the direct raw -> cal mapping with the detector nonlinearity
  // H2: "a*x+b*x*exp((x-c)*d)"
  a = pinfo->da0;
  b = pinfo->da1;
  c = pinfo->da2;
  d = pinfo->da3;
  cpl_msg_info(cpl_func, "nonlinearity H2: %g*x+%g*x*exp((x-%g)*%g)", a, b, c, d);
  for (i = 0; i < 65536; i++)
    {
      x = (double)i/MAT_DU_SCALE;
      y = (a*x + b*x*exp((x - c)*d))*MAT_DU_SCALE;
      pinfo->nonlinmap[i] = y;
    }
  return ec;
}


static cpl_error_code mat_nlm_dit_calculate(mat_cal_det_info *info)
{
  mat_fit_info_ot      *finfo;
  int                x, nx, y, ny;
  int                count_good = 0;
  int                count_bad;

  nx = info->nx;
  ny = info->ny;
  cpl_msg_info(cpl_func, "Calculate a nonlinearity response for each pixel based on exposures with different dit.");
  mat_image_fill(info->ditscale, 0.0);
  // 1. Initialize the nonlinearity map and the function fit context depending on the detector type.
  mat_image_fill(cpl_imagelist_get(info->nlm->list_coeff[0], 0), 1.0);
  mat_image_fill(cpl_imagelist_get(info->nlm->list_coeff[0], 1), 0.0);
  mat_image_fill(cpl_imagelist_get(info->nlm->list_coeff[0], 2), 0.0);
  mat_image_fill(cpl_imagelist_get(info->nlm->list_coeff[0], 3), 0.0);
  mat_image_fill(info->nlm->list_stdev[0], 0.0);
  info->nlm->has_coeff = 1;
  info->nlm->has_stdev = 1;
  if (info->nltype == LINEAR_NL_TYPE)
    {
      mat_image_fill(info->nlm->list_limit[0], 65536.0);
    }
  else
    {
      mat_image_fill(info->nlm->list_limit[0], 0.0);
      finfo = mat_fit_info_ot_new();
      mat_assert_not_null(finfo, "could not create the fit context");
      for (x = info->det->frame_left + 1; x <= nx - info->det->frame_right; x++)
	{
	  int flag = 0;
	  if (info->det->type == MAT_AQUARIUS_DET)
	    {
	      if ((x >= 56) && (x <= 133)) flag = 1;
	      if ((x >= 168) && (x <= 242)) flag = 1;
	      if ((x >= 276) && (x <= 766)) flag = 1;
	      if ((x >= 776) && (x <= 851)) flag = 1;
	      if ((x >= 885) && (x <= 954)) flag = 1;
	    }
	  else
	    {
	      flag = 1;
	    }
	  if (!flag) continue;
	  for (y = info->det->frame_bottom + 1; y <= ny - info->det->frame_top - 1; y++)
	    {
	      mat_nlm_dit_fit_pixel(info, finfo, x, y);
	      count_good += finfo->isok;
	    }
	}
      count_bad = (nx - info->det->frame_left - info->det->frame_right)*(ny - info->det->frame_bottom - info->det->frame_top) - count_good;
      cpl_msg_info(cpl_func, "%d pixels are bad or illumination too low", count_bad);
      mat_nlm_dit_fit_detector(info, finfo);
      mat_fit_info_ot_delete(finfo);
    }
  if (info->expert & NONLINEARITY_EXPERT_FLAG)
    {
      mat_image_validate(cpl_imagelist_get(info->nlm->list_coeff[0], 0), NULL,    0.0,   2.0, "NLM A0");
      mat_image_validate(cpl_imagelist_get(info->nlm->list_coeff[0], 1), NULL, -200.0, 200.0, "NLM A1");
      mat_image_validate(cpl_imagelist_get(info->nlm->list_coeff[0], 2), NULL, -200.0, 200.0, "NLM A2");
      mat_image_validate(cpl_imagelist_get(info->nlm->list_coeff[0], 3), NULL, -200.0, 200.0, "NLM A3");
      mat_image_validate(info->nlm->list_stdev[0], NULL, 0.0,     10.0, "NLM STDEV");
      mat_image_validate(info->nlm->list_limit[0], NULL, 0.0, 100000.0, "NLM LIMIT");
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_nlm_finish(mat_cal_det_info *info)
{
  cpl_error_code retval;

  info->nlm->horizontalcorrelation = info->hacstat->average;
  info->nlm->verticalcorrelation   = info->vacstat->average;
  info->nlm->flatfieldstdev        = info->ffm->flatfieldstdev;
  // Estimate the detector gain.
  retval = mat_calc_gain(info->flatmedraw,
			 info->flatvarraw,
			 info->flatrange,
			 info->bpm->list_badpixel[0],
			 info->det->channel_ncolumns,
			 info->det->channel_nrows,
			 &(info->nlm->detectorgain),
			 info->nlm->channelgain);
  if (retval != CPL_ERROR_NONE) return retval;
  // Estimate the detector channel offset (includes the 4096 DU offset of MATISSE)
  retval = mat_calc_offset(info->psm[0].rmed[DARK_IDX],
			   info->bpm->list_badpixel[0],
			   info->det->channel_ncolumns,
			   info->det->channel_nrows,
			   info->nlm->channeloffset);
  if (retval != CPL_ERROR_NONE) return retval;
  // Estimate the detector noise.
  retval = mat_calc_noise(info->psm[0].rvar[DARK_IDX],
			  info->bpm->list_badpixel[0],
			  info->det->channel_ncolumns,
			  info->det->channel_nrows,
			  info->nlm->detectorgain,
			  &(info->nlm->detectornoise),
			  info->nlm->channelgain,
			  info->nlm->channelnoise,
			  info->gain);
  if (retval != CPL_ERROR_NONE) return retval;
  if (info->det->type == MAT_AQUARIUS_DET)
    { // the calculation of the detector gain does not work for the Aquarius
      // -> exchange the calculated RON and gain with the gain given as parameter (--gain=...)
      info->nlm->detectornoise = info->nlm->detectornoise/info->nlm->detectorgain*info->gain;
      info->nlm->detectorgain = info->gain;
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_nlm_hist_allocate(mat_cal_det_info *info)
{
  if (info->nlmode == FLUX_NLMODE)
    {
      if (info->nlhistraw == NULL)
	{
	  info->nlhistraw = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
	  if (info->nlhistraw == NULL)
	    {
	      cpl_msg_error(cpl_func, "cannot allocate memory for the nonlinearity histogram (raw)");
	      return CPL_ERROR_UNSPECIFIED;
	    }
	}
      mat_image_fill(info->nlhistraw, 0.0);
    }
  if (info->nlhistcal == NULL)
    {
      info->nlhistcal = cpl_image_new(info->nx, info->ny, CPL_TYPE_DOUBLE);
      if (info->nlhistcal == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the nonlinearity histogram (calibrated)");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->nlhistcal, 0.0);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_nlm_hist_calculate_flux(mat_cal_det_info *info)
{
  int f;
  int nx, ny;

  if (info->nlhistraw == NULL) return CPL_ERROR_UNSPECIFIED;
  if (info->nlhistcal == NULL) return CPL_ERROR_UNSPECIFIED;
  nx = info->nx;
  ny = info->ny;
  for (f = 0; f < info->nbdit; f++)
    {
      mat_psm_index idx;
      cpl_image  *i1_img = info->psm[f].rmed[SH1_IDX];
      cpl_image  *i2_img = info->psm[f].rmed[SH2_IDX];
      cpl_image  *i3_img = info->psm[f].rmed[SH3_IDX];
      cpl_image  *i4_img = info->psm[f].rmed[SH4_IDX];
      if ((i1_img == NULL) || (i2_img == NULL) || (i3_img == NULL) || (i4_img == NULL)) continue;
      //idx = SH1234_IDX;
      for (idx = SH1_IDX; idx < FLAT_IDX; idx++)
	{
	  int px, py;
	  cpl_image *is_img = info->psm[f].rmed[idx];
	  if (idx == SH1_IDX) continue;
	  if (idx == SH2_IDX) continue;
	  if (idx == SH3_IDX) continue;
	  if (idx == SH4_IDX) continue;
	  if (is_img == NULL) continue;
	  for (py = 1; py <= ny; py++)
	    {
	      for (px = 1; px <= nx; px++)
		{
		  int    rejected;
		  double v1 = 0.0, v2 = 0.0, v3 = 0.0, v4 = 0.0;
		  double mos, mmsr = 0.0, mmsc = 0.0, count = 0.0;
		  int    hx, hy;
		  int    isphoto = 0;
		  mos = cpl_image_get(is_img, px, py, &rejected);
		  if (rejected != 0) continue;
		  if (mos < info->min_linear_range) continue;
		  if ((idx & SH1_IDX) != 0)
		    {
		      v1 = cpl_image_get(i1_img, px, py, &rejected);
		      mmsr += v1;
		      mmsc += mat_nlm_flux_map_direct(info, v1);
		      count += 1.0;
		    }
		  if ((idx & SH2_IDX) != 0)
		    {
		      v2 = cpl_image_get(i2_img, px, py, &rejected);
		      mmsr += v2;
		      mmsc += mat_nlm_flux_map_direct(info, v2);
		      count += 1.0;
		    }
		  if ((idx & SH3_IDX) != 0)
		    {
		      v3 = cpl_image_get(i3_img, px, py, &rejected);
		      mmsr += v3;
		      mmsc += mat_nlm_flux_map_direct(info, v3);
		      count += 1.0;
		    }
		  if ((idx & SH4_IDX) != 0)
		    {
		      v4 = cpl_image_get(i4_img, px, py, &rejected);
		      mmsr += v4;
		      mmsc += mat_nlm_flux_map_direct(info, v4);
		      count += 1.0;
		    }
		  if ((idx & SH1_IDX) != 0) isphoto |= (fabs(v1 - mos) < 0.5*mos/count);
		  if ((idx & SH2_IDX) != 0) isphoto |= (fabs(v2 - mos) < 0.5*mos/count);
		  if ((idx & SH3_IDX) != 0) isphoto |= (fabs(v3 - mos) < 0.5*mos/count);
		  if ((idx & SH4_IDX) != 0) isphoto |= (fabs(v4 - mos) < 0.5*mos/count);
		  if (isphoto) continue;
		  hx = (int)round((double)nx*mos/256.0/256.0);
		  hy = (int)round((double)ny*mmsr/256.0/256.0);
		  if ((hx >= 1) && (hx <= nx) && (hy >= 1) && (hy <= ny))
		    {
		      cpl_image_set(info->nlhistraw, hx, hy, 1.0 + cpl_image_get(info->nlhistraw, hx, hy, &rejected));
		    }
		  hy = (int)round((double)ny*mmsc/256.0/256.0);
		  if ((hx >= 1) && (hx <= nx) && (hy >= 1) && (hy <= ny))
		    {
		      cpl_image_set(info->nlhistcal, hx, hy, 1.0 + cpl_image_get(info->nlhistcal, hx, hy, &rejected));
		    }
		}
	    }
	}
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_nlm_hist_calculate_dit(mat_cal_det_info *info)
{
  int         didx;
  int         nx, ny;
  cpl_image  *scale = info->ditscale;
  double      xscale = 1.0/MAT_DU_SCALE;
  double      yscale;

  if (info->nlhistcal == NULL) return CPL_ERROR_UNSPECIFIED;
  nx = info->nx;
  ny = info->ny;
  for (didx = 0; didx < info->nbdit; didx++)
    {
      int         px, py;
      cpl_image  *flat = info->psm[didx].rmed[FLAT_IDX];
      if (flat == NULL) continue;
      for (py = 1; py <= ny; py++)
	{
	  for (px = 1; px <= nx; px++)
	    {
	      int    rejected;
	      double x, y;
	      int    hx, hy;
	      yscale = cpl_image_get(scale, px, py, &rejected);
	      x = cpl_image_get(flat, px, py, &rejected)*xscale;
	      y = info->psm[didx].dit*yscale;
	      if (rejected != 0) continue;
	      hx = (int)round((double)nx*x/8);
	      hy = (int)round((double)ny*y/8);
	      if ((hx < 1) || (hx > ny) ) continue;
	      if ((hy < 1) || (hy > nx) ) continue;
	      cpl_image_set(info->nlhistcal, hx, hy, 1.0 + cpl_image_get(info->nlhistcal, hx, hy, &rejected));
	    }
	}
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_nlm_hist_finish(mat_cal_det_info *info)
{
  int        i, nx, ny;
  cpl_image *hists[2] = {info->nlhistraw, info->nlhistcal};

  nx = info->nx;
  ny = info->ny;
  for (i = 0; i < 2; i++)
    {
      int        x, y;
      double     maxcount = 0.0;
      cpl_image *hist = hists[i];
      if (hist == NULL) continue;
      maxcount = cpl_image_get_max(hist);
      for (x = 1; x <= nx; x++)
	{
	  cpl_image_set(hist, x,  1, maxcount);
	  cpl_image_set(hist, x, ny, maxcount);
	}
      for (y = 1; y <= ny; y++)
	{
	  cpl_image_set(hist,  1, y, maxcount);
	  cpl_image_set(hist, nx, y, maxcount);
	}
      for (x = 1; x <= nx; x++)
	{
	  cpl_image_set(hist, x, x, maxcount);
	}
    }
  if (info->nlm->hist != NULL)
    {
      if (info->nlhistcal != NULL)
	{
	  cpl_image_copy(info->nlm->hist, info->nlhistcal, 1, 1);
	  info->nlm->has_hist = 1;
	}
      else if (info->nlhistraw != NULL)
	{
	  cpl_image_copy(info->nlm->hist, info->nlhistraw, 1, 1);
	  info->nlm->has_hist = 1;
	}
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_ffm_allocate(mat_cal_det_info *info)
{
  if (info->ffm != NULL) return CPL_ERROR_NONE;
  info->ffm = mat_flatfield_new(info->det, info->imgdet);
  mat_assert_not_null(info->ffm, "could not create the flatfield map");
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_ffm_calculate(mat_cal_det_info *info, cpl_frameset *frameset)
{
  int f;

  cpl_msg_info(cpl_func, "Calculate the flatfield map (raw flat divided by reference flat).");
  mat_image_fill(info->ffm->list_flatfield[0], 1.0);
  if (info->fftype != CONST_FF_TYPE)
    {
      int             x, y, nx, ny, dcnx;
      cpl_mask       *bpmmask     = info->bpm->list_badpixel[0];
      cpl_image      *rawimg      = info->flatmedraw;
      cpl_image      *refimg      = info->flatmedref;
      cpl_image      *rangeimg    = info->flatrange;
      cpl_image      *flatimg     = info->ffm->list_flatfield[0];
      cpl_image      *stdevimg    = info->ffm->list_stdev[0];
      cpl_image      *nlmstdevimg = info->nlm->list_stdev[0];
      cpl_mask       *darkmask;
      cpl_stats      *stats;
      // The flatfield map is derived from the flatmedraw and flatmedref.
      // All pixels without enough flux, the flatfield is set to 1.0 (flatrange != MAT_LINEAR_INTENSITY).
      nx = info->nx;
      ny = info->nx;
      dcnx = info->det->channel_nx;
      darkmask = cpl_mask_new(nx, ny);
      if (darkmask == NULL)
	{
	  cpl_msg_error(cpl_func, "could not create the mask for bad/dark pixels");
	  return CPL_ERROR_NULL_INPUT;
	}
      for (y = 1; y <= ny; y++)
	{
	  for (x = 1; x <= nx; x++)
	    {
	      int         rejected;
	      cpl_binary  isbad = cpl_mask_get(bpmmask, x, y);
	      double      raw   = cpl_image_get(rawimg, x, y, &rejected);
	      double      ref   = cpl_image_get(refimg, x, y, &rejected);
	      double      range = cpl_image_get(rangeimg, x, y, &rejected);
	      if (isbad || (range != MAT_LINEAR_INTENSITY) || (raw == 0.0) || (ref == 0.0))
		{
		  cpl_image_set(flatimg, x, y, info->aqpcs[(x-1)%dcnx]);
		  cpl_mask_set(darkmask, x, y, CPL_BINARY_1);
		}
	      else
		{
		  cpl_image_set(flatimg, x, y, raw/ref*info->aqpcs[(x-1)%dcnx]);
		  //cpl_image_set(flatimg, x, y, info->aqpcs[(x-1)%dcnx]);
		  cpl_mask_set(darkmask, x, y, CPL_BINARY_0);
		}
	    }
	}
      if (info->expert & FLATFIELD_EXPERT_FLAG)
	{
	  mat_image_validate(flatimg, NULL, 0.0, 2.0, "FFM");
	}
      info->ffm->has_stdev = info->nlm->has_stdev;
      cpl_image_copy(stdevimg, nlmstdevimg, 1, 1);
      // Calculating the standard deviation of the flatfield must ignore all bad pixels and all
      // pixels with low intensity. The low intensity pixels are defined by all pixels with a 
      // weight less than info->min_linear_range. The bad pixels are already identified and stored in
      // the bad pixel map.
      cpl_image_reject_from_mask(flatimg, darkmask);
      stats = cpl_stats_new_from_image(flatimg, CPL_STATS_MEDIAN | CPL_STATS_MEDIAN_DEV);
      if (stats != NULL)
	{
	  double m = cpl_stats_get_median(stats);
	  double d = 1.4826*cpl_stats_get_median_dev(stats);
	  cpl_msg_info(cpl_func, "flatfield median = %f, stdev = %f", m, d);
	  info->ffm->flatfieldstdev = d;
	  cpl_stats_delete(stats);
	}
      else
	{
	  info->ffm->flatfieldstdev = 0.0;
	}
      cpl_mask_delete(cpl_image_unset_bpm(flatimg));
      cpl_mask_delete(darkmask);
      if (info->fftype == AVERAGE_FF_TYPE)
	{
	  int ffmcount = cpl_frameset_count_tags(frameset, MATISSE_DO_FFM);
	  if (ffmcount != 0)
	    {
	      double     count = 1.0;
	      cpl_frame *ffmframe = cpl_frameset_find(frameset, MATISSE_DO_FFM); // already checked previously!
	      do
		{
		  mat_flatfield     *hffm;
		  cpl_msg_info(cpl_func, "Load and merging the flatfield map in %s", cpl_frame_get_filename(ffmframe));
		  cpl_frame_set_group(ffmframe, CPL_FRAME_GROUP_RAW);
		  hffm = mat_flatfield_load(ffmframe);
		  if (hffm)
		    {
		      cpl_image_add(info->ffm->list_flatfield[0], hffm->list_flatfield[0]);
		      count += 1.0;
		      mat_flatfield_delete(hffm);
		    }
		} while ((ffmframe = cpl_frameset_find(frameset, NULL)) != NULL);
	      cpl_image_divide_scalar(info->ffm->list_flatfield[0], count);
	    }
	}
    }
  // Apply the flatfield compensation to the previously nonlinearity compensated median and variance images
  for (f = 0; f < info->nbdit; f++)
    {
      cpl_image *mcal = info->psm[f].cmed[FLAT_IDX];
      cpl_image *vcal = info->psm[f].cvar[FLAT_IDX];
      cpl_image_divide(mcal, info->ffm->list_flatfield[0]);
      cpl_image_divide(vcal, info->ffm->list_flatfield[0]);
      cpl_image_divide(vcal, info->ffm->list_flatfield[0]);
      if (info->expert & FLATFIELD_EXPERT_FLAG)
	{
	  cpl_msg_info(cpl_func, "Check flatfield pixel statistics for DIT %.3f (after flatfield compensation)", info->psm[f].dit);
	  mat_image_validate(mcal, NULL, 0.0, 2.0, "calibrated median");
	  mat_image_validate(vcal, NULL, 0.0, 2.0, "calibrated variance");
	}
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_ffm_finish(mat_cal_det_info *info)
{
  mat_smooth_info   *sinfo;
  int                nx, ny;
  cpl_image         *dcslope = info->dcslope;
  cpl_image         *range   = info->flatrange;
  double             dcmasked = 0.0;
  double             dcexposed = 0.0;

  nx = info->det->nx;
  ny = info->det->ny;
  sinfo = mat_smooth_info_new(nx*ny, MAT_LOW_INTENSITY, MAT_LOW_INTENSITY);
  if (sinfo != NULL)
    {
      // calculate the average dark current in the masked area
      mat_smooth_set(sinfo, dcslope, range, 1, 1, nx, ny);
      mat_smooth_poly0_global(sinfo);
      dcmasked = sinfo->smooth[0];
      mat_smooth_info_delete(sinfo);
    }
  sinfo = mat_smooth_info_new(nx*ny, MAT_LINEAR_INTENSITY, MAT_LINEAR_INTENSITY);
  if (sinfo != NULL)
    {
      // calculate the average dark current in the exposed area
      mat_smooth_set(sinfo, dcslope, range, 1, 1, nx, ny);
      mat_smooth_poly0_global(sinfo);
      dcexposed = sinfo->smooth[0];
      mat_smooth_info_delete(sinfo);
    }
  // Copy the already calculated values from the nonlinearity map
  info->ffm->detectorgain          = info->nlm->detectorgain;
  info->ffm->detectornoise         = info->nlm->detectornoise;
  info->ffm->horizontalcorrelation = info->nlm->horizontalcorrelation;
  info->ffm->verticalcorrelation   = info->nlm->verticalcorrelation;
  cpl_vector_copy(info->ffm->channelgain,   info->nlm->channelgain);
  cpl_vector_copy(info->ffm->channeloffset, info->nlm->channeloffset);
  cpl_vector_copy(info->ffm->channelnoise,  info->nlm->channelnoise);
  // Show some important data
  cpl_msg_info(cpl_func, "detector gain          %5.2f e-/DU          %5.2f e-/DU (ds)",
	       info->ffm->detectorgain, info->gain);
  cpl_msg_info(cpl_func, "detector noise         %5.2f DU    %5.2f e- %5.2f e- (ds)",
	       info->ffm->detectornoise/info->ffm->detectorgain,
	       info->ffm->detectornoise,
	       info->ffm->detectornoise/info->ffm->detectorgain*info->gain);
  cpl_msg_info(cpl_func, "dark current (masked)  %5.2f DU    %5.2f e- %5.2f e- (ds)",
	       dcmasked, dcmasked*info->ffm->detectorgain, dcmasked*info->gain);
  cpl_msg_info(cpl_func, "dark current (exposed) %5.2f DU    %5.2f e- %5.2f e- (ds)",
	       dcexposed, dcexposed*info->ffm->detectorgain, dcexposed*info->gain);
  if (info->det->type == MAT_AQUARIUS_DET)
    { // the calculation of the detector gain does not work for the Aquarius
      // -> exchange the calculated RON and gain with the gain given as parameter (--gain=...)
      info->ffm->detectornoise = info->ffm->detectornoise/info->ffm->detectorgain*info->gain;
      info->ffm->detectorgain = info->gain;
    }
  return CPL_ERROR_NONE;
}

/*
  Two conversion factor (gain) 2-d histograms are created. On is based on the raw pixel statistics,
  the second uses the calibrated pixel statistics. Each histogram uses the same axis scaling given
  in the log output as "conversion factor diagram scaling". The calculated conversion gain
  (which is also put into the result flatfield and nonlinearity map) is shown as a straigt line.
  Both histograms contains data from all pixels with a brightness below 95 percent of the maximum brightness.

  If the --expert option contains the GAIN_EXPERT_FLAG (32), the main features of the conversion gain
  histogram are put into the log output:
    - the median intensity (raw and calibrated), it is the center of a +-16 DU wide intensity window
    - number of raw pixels with an intensity inside the window
    - 25 and 75 percent quantils of the raw variance
    - mode (peak) and median of the raw variance
    - variance of the raw variance
    - number of calibrated pixels with an intensity inside the window
    - 25 and 75 percent quantils of the calibrated variance
    - mode (peak) and median of the calibrated variance
    - variance of the calibrated variance

  Both 2-d histograms are put into the statistics file (file name *_stat.fits) 
    - second to last image: raw conversion factor histogram
    - last image: calibrated conversion factor histogram
 */

static cpl_error_code mat_gain_hist_allocate(mat_cal_det_info *info)
{
  int       f;
  double    max_median   = 0.0;
  double    max_variance = 0.0;
  int       nx, ny;

  nx = info->nx;
  ny = info->ny;
  if (info->gainhistraw == NULL)
    {
      info->gainhistraw = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
      if (info->gainhistraw == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the raw conversion factor histogram");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->gainhistraw, 0.0);
  if (info->gainhistcal == NULL)
    {
      info->gainhistcal = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
      if (info->gainhistcal == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the calibrated conversion factor histogram");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  mat_image_fill(info->gainhistcal, 0.0);
  // Determine the largest intensity/variance values (95 percent) over all flat statistics
  for (f = 0; f < info->nbdit; f++)
    {
      mat_psm_index idx = FLAT_IDX;
      //for (idx = SH1_IDX; idx <= FLAT_IDX; idx++)
      //{
      double    quantils[3] = {0.05, 0.5, 0.95};
      double    values[3];
      if (info->psm[f].rmed[idx] != NULL)
	{
	  mat_image_get_quantils(info->psm[f].rmed[idx], 3, quantils, values);
	  max_median   = CPL_MAX(max_median, values[2]);
	}
      if (info->psm[f].rvar[idx] != NULL)
	{
	  mat_image_get_quantils(info->psm[f].rvar[idx], 3, quantils, values);
	  max_variance = CPL_MAX(max_variance, values[2]);
	}
      //}
    }
  // Calculate the horizontal/median scaling factor for the 2-d conversion factor histograms
  switch (nx)
    {
    case 512 : info->gs_med =  500.0; break;
    case 1024: info->gs_med = 1000.0; break;
    case 2048: info->gs_med = 2000.0; break;
    default: info->gs_med = (double)nx;
    }
  if (max_median > 60000.0) {
    info->gs_med /= 70000.0;
  } else if (max_median > 50000.0) {
    info->gs_med /= 60000.0;
  } else if (max_median > 40000.0) {
    info->gs_med /= 50000.0;
  } else if (max_median > 30000.0) {
    info->gs_med /= 40000.0;
  } else if (max_median > 20000.0) {
    info->gs_med /= 30000.0;
  } else if (max_median > 10000.0) {
    info->gs_med /= 20000.0;
  } else {
    info->gs_med /= 10000.0;
  }
  // Calculate the vertical/variance scaling factor for the 2-d conversion factor histograms
  switch (ny)
    {
    case 512 : info->gs_var =  500.0; break;
    case 1024: info->gs_var = 1000.0; break;
    case 2048: info->gs_var = 2000.0; break;
    default: info->gs_var = (double)ny;
    }
  if (max_variance > 60000.0) {
    info->gs_var /= 70000.0;
  } else if (max_variance > 50000.0) {
    info->gs_var /= 60000.0;
  } else if (max_variance > 40000.0) {
    info->gs_var /= 50000.0;
  } else if (max_variance > 30000.0) {
    info->gs_var /= 40000.0;
  } else if (max_variance > 20000.0) {
    info->gs_var /= 30000.0;
  } else if (max_variance > 10000.0) {
    info->gs_var /= 20000.0;
  } else {
    info->gs_var /= 10000.0;
  }
  cpl_msg_info(cpl_func, "conversion factor diagram scaling:");
  cpl_msg_info(cpl_func, "  x = %g * median, max median = %g", info->gs_med, max_median);
  cpl_msg_info(cpl_func, "  y = %g * variance, max variance = %g", info->gs_var, max_variance);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_gain_hist_calculate(mat_cal_det_info *info)
{
  int       f;
  int        nx, ny;
  int        x, y, rejected;
  cpl_image *rhist = info->gainhistraw;
  cpl_image *chist = info->gainhistcal;

  nx = info->nx;
  ny = info->ny;
  // create images representing a raw and a calibrated 2-d histogram of variance vs median for each pixel
  mat_image_fill(rhist, 0.0);
  mat_image_fill(chist, 0.0);
  for (f = 0; f < info->nbdit; f++)
    {
      cpl_image *med, *var;
      med = info->psm[f].rmed[FLAT_IDX];
      var = info->psm[f].rvar[FLAT_IDX];
      if ((med != NULL) && (var != NULL))
	{
	  for (y = 1; y <= ny; y++)
	    {
	      for (x = 1; x <= nx; x++)
		{
		  double m, v;
		  int    px, py;
		  m = cpl_image_get(med, x, y, &rejected);
		  v = cpl_image_get(var, x, y, &rejected);
		  if (v == -1.0) continue;
		  px = 1 + (int)round(m*info->gs_med);
		  py = 1 + (int)round(v*info->gs_var);
		  if ((px < 1) || (px > nx)) continue;   // median outside of the diagram
		  if ((py < 1) || (py > ny)) continue;   // variance outside of the diagram
		  cpl_image_set(rhist, px, py, 1.0 + cpl_image_get(rhist, px, py, &rejected));
		}
	    }
	}
      med = info->psm[f].cmed[FLAT_IDX];
      var = info->psm[f].cvar[FLAT_IDX];
      if ((med != NULL) && (var != NULL))
	{
	  for (y = 1; y <= ny; y++)
	    {
	      for (x = 1; x <= nx; x++)
		{
		  double m, v;
		  int    px, py;
		  m = cpl_image_get(med, x, y, &rejected);
		  v = cpl_image_get(var, x, y, &rejected);
		  if (v == -1.0) continue;
		  px = 1 + (int)round(m*info->gs_med);
		  py = 1 + (int)round(v*info->gs_var);
		  if ((px < 1) || (px > nx)) continue;   // median outside of the diagram
		  if ((py < 1) || (py > ny)) continue;   // variance outside of the diagram
		  cpl_image_set(chist, px, py, 1.0 + cpl_image_get(chist, px, py, &rejected));
		}
	    }
	}
    }
  return CPL_ERROR_NONE;
}


static cpl_error_code mat_gain_hist_plot(mat_cal_det_info *info)
{
  int        i, f;
  int        nx, ny;
  int        x, y, rejected;

  if ((info->expert & GAIN_EXPERT_FLAG) == 0) return CPL_ERROR_NONE;
  nx = info->nx;
  ny = info->ny;
  // calculate a 2-d plot of the conversion gain (raw and calibrated)
  cpl_msg_info(cpl_func, "# det gain # data # title Representation of the 2-d conversion factor histogram (raw and calibrated).");
  cpl_msg_info(cpl_func, "# det gain # data # column 1 median intensity");
  cpl_msg_info(cpl_func, "# det gain # data # column 2 number of pixels with a raw intensity in [median-16 .. median+16]");
  cpl_msg_info(cpl_func, "# det gain # data # column 3 25 percent quantil of the raw variance");
  cpl_msg_info(cpl_func, "# det gain # data # column 4 75 percent quantil of the raw variance");
  cpl_msg_info(cpl_func, "# det gain # data # column 5 mode (peak) of the raw variance");
  cpl_msg_info(cpl_func, "# det gain # data # column 6 median of the raw variance");
  cpl_msg_info(cpl_func, "# det gain # data # column 7 variance of the raw variance");
  cpl_msg_info(cpl_func, "# det gain # data # column 8 number of pixels with a calibrated intensity in [median-16 .. median+16]");
  cpl_msg_info(cpl_func, "# det gain # data # column 9 25 percent quantil of the calibrated variance");
  cpl_msg_info(cpl_func, "# det gain # data # column 10 75 percent quantil of the calibrated variance");
  cpl_msg_info(cpl_func, "# det gain # data # column 11 mode (peak) of the calibrated variance");
  cpl_msg_info(cpl_func, "# det gain # data # column 12 median of the calibrated variance");
  cpl_msg_info(cpl_func, "# det gain # data # column 13 variance of the calibrated variance");
  info->stat1 = mat_statistics_realloc(info->stat1, nx*ny, MAT_OUTLIER_FACTOR);
  info->stat2 = mat_statistics_realloc(info->stat2, nx*ny, MAT_OUTLIER_FACTOR);
  for (i = 0; i < 65536/64; i++)
    {
      double raw_min = (double)(i*64 - 16);
      double raw_max = (double)(i*64 + 16);
      mat_statistics_reset(info->stat1);
      mat_statistics_reset(info->stat2);
      for (f = 0; f < info->nbdit; f++)
	{
	  cpl_image *med, *var;
	  med = info->psm[f].rmed[FLAT_IDX];
	  var = info->psm[f].rvar[FLAT_IDX];
	  if ((med != NULL) && (var != NULL))
	    {
	      for (y = 1; y <= ny; y++)
		{
		  for (x = 1; x <= nx; x++)
		    {
		      double m, v;
		      m = cpl_image_get(med, x, y, &rejected);
		      v = cpl_image_get(var, x, y, &rejected);
		      if (v == -1.0) continue;
		      if ((m >= raw_min) && (m < raw_max)) mat_statistics_add_value(info->stat1, v);
		    }
		}
	    }
	  med = info->psm[f].cmed[FLAT_IDX];
	  var = info->psm[f].cvar[FLAT_IDX];
	  if ((med != NULL) && (var != NULL))
	    {
	      for (y = 1; y <= ny; y++)
		{
		  for (x = 1; x <= nx; x++)
		    {
		      double m, v;
		      m = cpl_image_get(med, x, y, &rejected);
		      v = cpl_image_get(var, x, y, &rejected);
		      if (v == -1.0) continue;
		      if ((m >= raw_min) && (m < raw_max)) mat_statistics_add_value(info->stat2, v);
		    }
		}
	    }
	}
      info->stat1->pflag = ((i*64 == 44288) || (i*64 == 30464) || (i*64 == 32000) || (i*64 == 33664) || (i*64 == 35392) || (i*64 == 45824) || (i*64 == 45888));
      info->stat2->pflag = info->stat1->pflag;
      mat_statistics_calc(info->stat1);
      mat_statistics_calc(info->stat2);
      if ((info->stat1->reduced_count > 13) || (info->stat2->reduced_count > 13))
	{
	  cpl_msg_info(cpl_func, "# det gain # data %d %d %.1f %.1f %.1f %.1f %.1f %d %.1f %.1f %.1f %.1f %.1f",
		       i*64,
		       info->stat1->reduced_count, info->stat1->q25, info->stat1->q75, info->stat1->whole_mode, info->stat1->reduced_median, info->stat1->reduced_variance,
		       info->stat2->reduced_count, info->stat2->q25, info->stat2->q75, info->stat2->whole_mode, info->stat2->reduced_median, info->stat2->reduced_variance
		       );
	}
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_gain_hist_finish(mat_cal_det_info *info)
{
  cpl_image   *hists[2];
  int          i;
  int       nx, ny;

  hists[0] = info->gainhistraw;
  hists[1] = info->gainhistcal;
  nx = info->nx;
  ny = info->ny;
  for (i = 0; i < 2; i++)
    {
      cpl_image *hist = hists[i];
      int        x, y;
      double     maxcount = 0;
      maxcount = cpl_image_get_max(hist);
      for (x = 1; x <= nx; x++)
	{
	  cpl_image_set(hist, x,  1, maxcount);
	  cpl_image_set(hist, x, ny, maxcount);
	}
      for (y = 1; y <= ny; y++)
	{
	  cpl_image_set(hist,  1, y, maxcount);
	  cpl_image_set(hist, nx, y, maxcount);
	}
      for (x = 1; x <= nx; x++)
	{
	  double m = (double)(x - 1)/info->gs_med;
	  double v = m/info->ffm->detectorgain;
	  int py = 1 + (int)round(v*info->gs_var);
	  cpl_image_set(hist, x, py, maxcount);
	}
    }
  return CPL_ERROR_NONE;
}

static int mat_count_frames(cpl_frameset *frameset, const char *tag)
{
  cpl_frameset_iterator *it;
  cpl_frame             *frame;
  int                    count = 0;
  
  it = cpl_frameset_iterator_new(frameset);
  frame = cpl_frameset_iterator_get(it);
  while (frame != NULL)
    {
      // we use only calibrated target interferometric data, we have to ignore possible flatfield files in the SOF !!!
      if ((strstr(cpl_frame_get_tag(frame), tag) != NULL) && (strcmp(cpl_frame_get_tag(frame), MATISSE_DO_FFM) != 0)) count++;
      cpl_frameset_iterator_advance(it, 1);
      frame = cpl_frameset_iterator_get(it);
    }
  cpl_frameset_iterator_delete(it);
  return count;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Interpret the command line options and execute the data processing
   @param parlist the parameters list
   @param frameset the frames list
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_cal_det(cpl_parameterlist *parlist,
		       cpl_frameset *frameset)
{
  mat_cal_det_info  info;
  int               darkcount;
  int               flatcount;
  //int               irun;
  cpl_error_code    ec = CPL_ERROR_NONE;

  mat_info_init(&info);
  /***** Get all parameters needed. */
  if (mat_read_parameters(&info, parlist) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Failed to retrieve the input parameters");
    mat_info_delete(&info);
    return -1;
  }
  /***** Check the number of DARK and FLAT files given in the SOF. */
  darkcount = mat_count_frames(frameset, MATISSE_DO_DARK);
  flatcount = mat_count_frames(frameset, MATISSE_DO_FLAT);
  if (darkcount == 0)
    {
      cpl_msg_error(cpl_func, "SOF does not have any dark files");
      mat_info_delete(&info);
      return -1;
    }
  if (flatcount == 0)
    {
      cpl_msg_error(cpl_func, "SOF does not have any flat field files ");
      mat_info_delete(&info);
      return -1;
    }
  if (darkcount > flatcount)
    {
      cpl_msg_error(cpl_func, "SOF does not have the same number of dark and flat files");
      mat_info_delete(&info);
      return -1;
    }
  // ******************** pixel statistics
  // - Read all files with dark exposures and calculate their pixel statistics (temporal median and variance).
  // - Read all files with flatfield exposures and calculate their pixel statistics (temporal median and variance).
  // - Compensate the flat statistics by subtracting the dark statistics (median and variance).
  // - Use a detector related default value if nothing specific is given as parameter
  // - Show a specific pixel statistics (needs --poi=<r>,<x>,<y> and --expert=2)
  if (ec == CPL_ERROR_NONE) ec = mat_read_and_process_data(&info, frameset);
  if (ec == CPL_ERROR_NONE) mat_select_detector_gain(&info);
  // ******************** dark properties
  // - allocate temporay memory for the dark properties
  // - calculate the dark current (offset and slope)
  // - calculate the average dark intensity and variance
  if (ec == CPL_ERROR_NONE) ec = mat_dark_prop_allocate(&info);
  if (ec == CPL_ERROR_NONE) ec = mat_dark_prop_calculate(&info);
  if (ec == CPL_ERROR_NONE) ec = mat_dark_prop_range(&info);
  if (ec == CPL_ERROR_NONE) ec = mat_dark_prop_reference(&info);
  // ******************** preliminary bad pixel map
  // - allocate temporary memory and the final bad pixel map
  // - determine bad pixels using only the dark properties (absolut intensity limit: info->darkmedres)
  if (ec == CPL_ERROR_NONE) ec = mat_bpm_allocate(&info);
  if (ec == CPL_ERROR_NONE) ec = mat_bpm_calculate_preliminary(&info);
  // ******************** nonlinearity characterization
  if (ec == CPL_ERROR_NONE) ec = mat_nlm_allocate(&info);
  if (ec == CPL_ERROR_NONE)
    {
      if (info.nlmode == FLUX_NLMODE) ec = mat_nlm_flux_calculate(&info);
      if (info.nlmode == DIT_NLMODE)  ec = mat_nlm_dit_calculate(&info);
    }
  // ******************** flat properties
  // - allocate temporay memory for the flat properties
  // - create an artificial average flat median and variance
  if (ec == CPL_ERROR_NONE) ec = mat_flat_prop_allocate(&info);
  if (ec == CPL_ERROR_NONE) ec = mat_flat_prop_calibrate(&info);
  if (ec == CPL_ERROR_NONE)
    {
      if (info.nlmode == FLUX_NLMODE) ec = mat_flat_prop_flux_calculate(&info);
      if (info.nlmode == DIT_NLMODE)  ec = mat_flat_prop_dit_calculate(&info);
    }
  // ******************** final bad pixel map
  // - determine bad pixels using the dark properties (absolut intensity limit: info->darkmedres)
  // - determine bad pixels using the flat properties (relative intensity limit: info->flatmedres)
  if (ec == CPL_ERROR_NONE) ec = mat_bpm_calculate_final(&info, frameset);
  // ******************** flatfield map
  if (ec == CPL_ERROR_NONE) ec = mat_ffm_allocate(&info);
  if (ec == CPL_ERROR_NONE) ec = mat_ffm_calculate(&info, frameset);
  // ******************** finalize all detector characterization map calculation
  if (ec == CPL_ERROR_NONE) ec = mat_nlm_finish(&info);
  if (ec == CPL_ERROR_NONE) ec = mat_ffm_finish(&info);
  // ******************** conversion gain histograms
  // - allocate temporal memory for both histograms (raw + calibrated)
  if (ec == CPL_ERROR_NONE) ec = mat_gain_hist_allocate(&info);
  if (ec == CPL_ERROR_NONE) ec = mat_gain_hist_calculate(&info);
  if (ec == CPL_ERROR_NONE) ec = mat_gain_hist_plot(&info);
  if (ec == CPL_ERROR_NONE) ec = mat_gain_hist_finish(&info);
  // ******************** nonlinearity histogram
  if (ec == CPL_ERROR_NONE) ec = mat_nlm_hist_allocate(&info);
  if (ec == CPL_ERROR_NONE)
    {
      if (info.nlmode == FLUX_NLMODE) ec = mat_nlm_hist_calculate_flux(&info);
      if (info.nlmode == DIT_NLMODE)  ec = mat_nlm_hist_calculate_dit(&info);
    }
  if (ec == CPL_ERROR_NONE) ec = mat_nlm_hist_finish(&info);

  if (ec == CPL_ERROR_NONE)
    {
      /***** If requested, store some calculated frames in a FITS file. */
      if (info.expert & STAT_OUTPUT_EXPERT_FLAG)
	{
	  mat_store_statistics(&info, "mat_cal_det_stat.fits",
			       parlist,
			       frameset);
	}
      /***** Store the final detector calibration maps as FITS files. */
      mat_badpixel_save(info.bpm,
			mat_detector_get_bpm_name(info.det),
			"mat_cal_det",
			parlist,
			frameset);
      mat_nonlinearity_save(info.nlm,
			    mat_detector_get_nlm_name(info.det),
			    "mat_cal_det",
			    parlist,
			    frameset);
      mat_flatfield_save(info.ffm,
			 mat_detector_get_ffm_name(info.det),
			 "mat_cal_det",
			 parlist,
			 frameset);
    }
  mat_info_delete(&info);
  /* Return */
  if (cpl_error_get_code())
    return -1;
  else
    return 0;
}

