/* $Id: mat_cal_image.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_cal_image.c $
 */
/*------------------------------------------------------------------------------
                                   Includes
 ------------------------------------------------------------------------------*/
#include <cpl.h>
#include "mat_drl.h"
#include "mat_cal_image_lib.h"
/* #include "mat_check_tag.h" */
/*------------------------------------------------------------------------------
                                   Define
 ------------------------------------------------------------------------------*/
#define TEL_NUMBER 4
#define RECIPE_NAME "mat_cal_image"



/* Plugin detailed description */
static const char *mat_cal_image_help =
"This plugin uses static calibration maps and the observation flatfield to convert raw interferograms into calibrated interferograms. The algorithm for this calibration is able to compensate for detector and instrument effects like individual pixel gain, nonlinearity, defect pixels and optical distortion.\n"
"\n"
"The provided calibration maps specifies which part of the calibration can be performed. In addition, a parameters provides a finer control about the applied calibration:\n"
"\n"
" - If the static bad pixel map (DO classification BADPIX) is provided, the intensity of the defect pixels can be replaced with an interpolated value derived from the neighbour pixels.\n"
"\n"
" - If the static flatfield map (DO classification FLATFIELD) is provided, the individual pixel gain can be compensated for.\n"
"\n"
" - If the static nonlinearity map (DO classification NONLINEARITY) is provided, the individual pixel nonlinearity can be compensated for.\n"
"\n"
" - If the observation (setup) specific flatfield map (DO classification OBS_FLATFIELD) is provided, the individual pixel offset and the instrument flatfield can be compensated for.\n"
"\n"
" - If the optical distortion map (DO classification SHIFT_MAP) is provided, the warping of the optical mapping can be compensated for.\n"
"\n"
"Each calibration map provided specified only the available compensation method. The parameter --compensate is used to define which method should be applied if the necessary calibration map is provided. This parameter is a comma separated list of keywords:\n"
"\n"
"  none = no compensation at all\n"
"  all  = all compensations possible (depends on the provided maps)\n"
"  pb   = subtract pixel bias (OBS_FLATFIELD is needed)\n"
"  gb   = subtract global bias (OBS_FLATFIELD is needed)\n"
"  cb   = subtract detector channel bias (OBS_FLATFIELD is needed)\n"
"  rb   = subtract row bias (OBS_FLATFIELD is needed)\n"
"  ct   = subtract detector channel crosstalk (OBS_FLATFIELD is needed)\n"
"  nl   = nonlinearity compensation (NONLINEARITY is needed)\n"
"  if   = divide by instrument flat (OBS_FLATFIELD is needed)\n"
"  df   = divide by detector flat (FLATFIELD is needed)\n"
"  bp   = bad pixel interpolation (BADPIX is needed)\n"
"  el   = convert to electrons (conversion factor from parameter --gain, FLATFIELD or OBS_FLATFIELD)\n"
"  od   = remove optical distortion (SHIFT_MAP is needed)\n"
"\n"
"The default value for this parameter is \"pb,gb,nl,if,bp\".\n"
"The conversion factor (needed for the \"el\" method) is provided with the parameter --gain. If this value is 0.0 (the default), the conversion factor in the static flatfield map (FLATFIELD) or the observation flatfield (OBS_FLATFIELD) is used.\n"
"\n"
"The parameter --reduce allows that the result file does not contain the reference regions (--reduce=TRUE, the default value) or they should be kept (--reduce=FALSE).\n"
  "\n"
  "Some raw files exist where, due to a programming error, the relation between frame and timetable entry (columns TIME, LOCALOPD, STEPPING_PHASE and TARTYP) is broken. It is possible to correct this with the option --tartyp. This option is a set of flags controlling the behaviour for this correction step:\n"
  "\n"
  "   1 = The TARTYP in the timetable is N*S+U+N*T+U (trailing U). If the TARTYP is U+N*S+U+N*T (leading U), this flag must be cleared\n"
  "   2 = show the total flux in the interferrometric channel (some offset is subtracted)\n"
  "   4 = show the correlation between successive frames using interferometric channel\n"
  "   8 = estimate the TARTYP using three algorithms\n"
  "  16 = change the values in the column TARTYP by correcting excess frames\n"
  "  32 = exchange all unknown TARTYP (U) either with S (sky) or T (target)\n"
  "  64 = change the values in the column TIME by correcting excess frames\n"
  " 128 = change the values in the columns LOCALOPD and STEPPING_PHASE by correcting excess frames\n"
  " 256 = show the extracted exposure setup\n"
  " 512 = show the first chopping cycle\n"
  "1024 = show the extracted timetable\n"
  "2048 = show the modified frame info\n"
  "4096 = use shift calculated from square wave fit\n"
  "8192 = use shift calculated from timing\n"
  "\n"
  "The default value of --tartyp does nothing related to the timetable."
  "\n"
  "Due to a programming error sometimes frames were created between starting the sequencer and setting the TRIGGER flag. This number can be estimated by analyzing the first LM- and N-Band exposure after starting the TIM-Board trigger, DATE-OBS is closest to INS TIMDL START or INS TIMDN START, with the command:\n"
  "\n"
  "   esorex mat_cal_image --compensate=none --tartyp=9 <sof-file containg the first exposure>\n"
  "\n"
  "In the message (for example):\n"
  "\n"
  "  setup timing: obs 17225.944 s start 17198.000 s period 0.273 s start frame 17226.392 s start exp 17226.392 possible shift 1\n"
  "  setup timing: maybe 1 excess frame(s) -> TIME, TARTYP, STEPPING_PHASE and LOCALOPD may need adjustment!\n"
  "  raw data: entries 271 roi: 2 = 625x110 used x = [156 .. 468] y = [27 .. 82]\n"
  "  chopping: entries 8 first 0 cycles 33 phase 0 shift 1\n"
  "  timetable: entries 80 first 0 second 80 cycles 3 phase 0 shift 1\n"
  "\n"
  "Analyzing the timing, chopping and timetable gives a shift of 1 frame for the TARTYP, this is equal to 1 excess frame which should be corrected.\n"
  "The number after \"shift\" is the number of excess frames and for applying the timetable correction, this number can be given via:\n"
  "\n"
  "  --excess_count_lm=<count>   for LM-Band observations\n"
  "or\n"
  "  --excess_count_n=<count>    for N-Band observations\n"
  "\n"
  "If a change of the TARTYP is requested (flag 16), an excess count other than 0 is preferred before using the calculated shift (flag 8). It is possible to specify both numbers for all raw files, the appropriate number is selected according to the detector of a specific raw file. It is also possible to specify the number of excess frames of each specific exposure identified by ESO TPL EXPNO:\n"
  "\n"
  "  --excess_count_list=<expno>,<count>{,<expno>,<count>}\n"
  "\n"
  "Input Files:\n"
  "\n"
  "    DO category:           Explanation:                Required:\n"
  "    CALIB_SRC_RAW          Lab Lamp data               Yes (if lab data)\n"
  "    HOT_DARK               Lab Background data         Yes (if lab data)\n"
  "    SKY_RAW                Sky data                    No \n"
  "    CALIB_RAW              Calibrator data             Yes (if calibrator)\n"
  "    TARGET_RAW             Target data                 Yes (if target)\n"
  "    SHIFT_MAP              Distortion Map              Yes\n"
  "    OBS_FLATFIELD          Observing Flat Field        Yes\n"
  "    BADPIX                 Bad Pixel Map               Yes\n"
  "    NONLINEARITY           Nonlinearity Map            Yes\n"
  "\n"
  "Output Files:\n"
  "\n"
  "    DO category:           Explanation:\n"
  "    TARGET_CAL             Calibrated data (if target)\n"
  "    CALIB_CAL              Calibrated data (if calibrator or lab data)\n";

static int mat_cal_image_create(cpl_plugin *);
static int mat_cal_image_exec(cpl_plugin *);
static int mat_cal_image_destroy(cpl_plugin *);

int cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = (cpl_plugin *)recipe;
    cpl_plugin_init(plugin,
                     CPL_PLUGIN_API,
                     MATISSE_BINARY_VERSION,
                     CPL_PLUGIN_TYPE_RECIPE,
                     "mat_cal_image",
                     "Calibration of the raw images",
                     mat_cal_image_help,
                     "Philippe Berio",
                     PACKAGE_BUGREPORT,
                     "GPL",
                     mat_cal_image_create,
                     mat_cal_image_exec,
                     mat_cal_image_destroy);
    cpl_pluginlist_append(list, plugin);
    return 0;
    }

static int mat_cal_image_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe = (cpl_recipe *)plugin;
    cpl_parameter *p;


    recipe->parameters = cpl_parameterlist_new();
    /* --compensate */
    p = cpl_parameter_new_value("matisse.mat_cal_image.compensate",
				CPL_TYPE_STRING,
				"Defines which kind of compensation should be applied "
				"(none = no compensation at all, all = all compensations possible, dd = detector specific defaults, "
				"pb = subtract pixel bias, gb = subtract global bias, cb = subtract detector channel bias, rb = subtract row bias, "
				"ct = subtract crosstalk, "
				"nl = nonlinearity compensation, if = divide by instrument flat, df = divide by detector flat, "
				"bp = bad pixel interpolation, el = convert to electrons, od = remove optical distortion)",
				"matisse.mat_cal_image",
				"pb,cb,rb,nl,if,bp,od");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "compensate") ;
    cpl_parameterlist_append(recipe->parameters, p) ; 
 
    /* --gain */
    p = cpl_parameter_new_range("matisse.mat_cal_image.gain",
				CPL_TYPE_DOUBLE,
				"Default conversion gain in [e-/DU].",
				"matisse.mat_cal_image",
				0.0, 0.0, 1e5);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "gain");
    cpl_parameterlist_append(recipe->parameters, p);

    /* --reduce */
    p = cpl_parameter_new_value("matisse.mat_cal_image.reduce",
				CPL_TYPE_BOOL,
				"Flag if the reference sub-windows should be removed from the result.",
				"matisse.mat_cal_image",
				TRUE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "reduce");
    cpl_parameterlist_append(recipe->parameters, p);

    /* --ioi */
    p = cpl_parameter_new_value("matisse.mat_cal_image.ioi",
				CPL_TYPE_STRING,
				"images of interest: <first>,<count>",
				"matisse.mat_cal_image",
				"0,0");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "ioi");
    cpl_parameterlist_append(recipe->parameters, p);
    
    /* --tartyp */
    p = cpl_parameter_new_value("matisse.mat_cal_image.tartyp",
				CPL_TYPE_INT,
				"TARTYP estimation (flag, please add the following values for the desired effect: 0 = none, 1 = N*S+U+N*T+U, 2 = show total flux, 4 = show correlation, 8 = estimate TARTYP, 16 = change TARTYP, 32 = exchange U with S or T, 64 = change TIME, 128 = change LOCALOPD and STEPPING_PHASE, 256 = show the extracted exposure setup, 512 = show the first chopping cycle, 1024 = show the extracted timetable, 2048 = show the modified frame info, 4096 = use shift calculated from square wave fit, 8192 = use shift calculated from timing)",
				"matisse.mat_cal_image",
				57);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "tartyp");
    cpl_parameterlist_append(recipe->parameters, p);

    /* --excess_count_lm */
    p = cpl_parameter_new_value("matisse.mat_cal_image.excess_count_lm",
				CPL_TYPE_INT,
				"Excess frames mistakenly produced before first TIM-Board trigger (LM-Band)",
				"matisse.mat_cal_image",
				MAT_EXCESS_COUNT_UNSET);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "excess_count_lm");
    cpl_parameterlist_append(recipe->parameters, p);

    /* --excess_count_n */
    p = cpl_parameter_new_value("matisse.mat_cal_image.excess_count_n",
				CPL_TYPE_INT,
				"Excess frames mistakenly produced before first TIM-Board trigger (N-Band)",
				"matisse.mat_cal_image",
				MAT_EXCESS_COUNT_UNSET);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "excess_count_n");
    cpl_parameterlist_append(recipe->parameters, p);

    /* --excess_count_list */
    p = cpl_parameter_new_value("matisse.mat_cal_image.excess_count_list",
				CPL_TYPE_STRING,
				"Specification of an excess count for each exposure number (ESO TPL EXPNO keyword, excess count pair) as a list of numbers separate by comma.",
				"matisse.mat_cal_image",
				"none");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "excess_count_list");
    cpl_parameterlist_append(recipe->parameters, p);

    /* --binning */
    /* p = cpl_parameter_new_value("matisse.mat_cal_image.binning", */
    /* 				CPL_TYPE_INT, */
    /* 				"Number of columns to bin (odd number only)", */
    /* 				"matisse.mat_cal_image", */
    /* 				1); */
    /* cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "binning"); */
    /* cpl_parameterlist_append(recipe->parameters, p); */
    
    cpl_msg_info("mat_cal_image_create", "Parameters created.");

    return 0;
}
static int mat_cal_image_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe = (cpl_recipe *)plugin;
    cpl_parameterlist_delete(recipe->parameters);
    return 0;
}


static int mat_cal_image_exec(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
   
  if (mat_cal_image_lib(recipe->frames,recipe->parameters,RECIPE_NAME)) {
    cpl_msg_warning(cpl_func,"Error in mat_cal_image_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }

  return 0;
}

