/* $Id: mat_cal_imarec.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_cal_imarec.c $
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define MATISSE_LICENCE         "GPL"

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <complex.h>

#include <cpl.h>

#include "mat_error.h"
#include "mat_oifits.h"
#include "mat_utils.h"
#include "mat_image.h"
#include "asa_user.h"
#include "lbfgsb.h"

#define WITH_PRECISION
//#define WITH_GRAD_SCALING
#define WITH_BESSELJ

/*-----------------------------------------------------------------------------
  Defines
  -----------------------------------------------------------------------------*/
#define MAT_DO_TARGET_CAL_INT "TARGET_CAL_INT"
#define MAT_DO_START_IMAGE    "START_IMAGE"
#define MAT_DO_PRIOR_IMAGE    "PRIOR_IMAGE"
#define MAT_DO_MODEL_IMAGE    "MODEL_IMAGE"
#define MAT_DO_OBJECT_MASK    "OBJECT_MASK"
#define MATISSE_REC_PROCATG   "TARGET_REC"
#define MAT_QC_PREC 4

#define MAT_MODE_IMAGE_OPTIONAL      -1
#define MAT_MODE_IMAGE_MANDATORY      0
#define MAT_MODE_STAR                 1
#define MAT_MODE_GAUSSIAN_DISC        2
#define MAT_MODE_UNIFORM_DISC         3
#define MAT_MODE_FULLY_DARKENED_DISC  4
#define MAT_MODE_LORENTZ_DISC         5

#define MAT_NB_TERM_IT                20

#define MAT_NB_LAMBDA_WINDOW          32

#define MAT_DELTA_START               4.0
#define MAT_DELTA_SCALE               0.5
#define MAT_DELTA_STEPS               2
#define MAT_DELTA_MIN                 0.1

#define MAT_CLAMP(_l, _u, _v) (((_v) < (_l)) ? (_l): ((_v) > (_u)) ? (_u):(_v))

#define MAT_CHYPOT2(_c) (creal(_c)*creal(_c) + cimag(_c)*cimag(_c))

#define INFO_NONE           0
#define INFO_DEBUG          (1 << 0)
#define INFO_PARAMETER      (1 << 1)
#define INFO_INPUT_IMAGES   (1 << 2)
#define INFO_INPUT_VIS2     (1 << 3)
#define INFO_INPUT_T3       (1 << 4)
#define INFO_INPUT_VIS      (1 << 5)
#define INFO_PREPARE        (1 << 6)
#define INFO_ITER_STD       (1 << 7)
#define INFO_ITER_DETAILS   (1 << 8)
#define INFO_FITS           (1 << 9)
#define INFO_RESULT         (1 << 10)
#define INFO_ALL            ((1 << 11) - 1)

#define INFO_FITS_MAX_ITER  20

#define MODE_BISPECTRUM     1
#define MODE_COMPLEX_VIS    2

#define ENGINE_ASA_CG       1
#define ENGINE_L_BFGS_B     2

#define USE_BIS_FLAG        (1 << 0)
#define USE_CP_FLAG         (1 << 1)
#define USE_VIS2_FLAG       (1 << 2)
#define USE_PHI_FLAG        (1 << 3)
#define USE_T3_TABLE        (1 << 4)
#define USE_VIS2_TABLE      (1 << 5)
#define USE_VIS_TABLE       (1 << 6)

/*-----------------------------------------------------------------------------
  Data types
  -----------------------------------------------------------------------------*/
/**
   @brief This data structure contains one loaded and remapped squared visibility.

   Each squared visibility (OI_VIS2 binary table) is stored in such a data structure.
   The uv coordinates are remapped, depending on the wavelength and pixel scaling,
   to array coordinates for the FFT.
*/
typedef struct {
  double          lambda;  /*!< Wavelength of this squared visibility measurement. */
  double          mvis2;   /*!< Copy of one OI_VIS2.VIS2DATA column element (wavelength dependend). */
  double          mvar;    /*!< Squared value of one OI_VIS2.VIS2ERR column element (wavelength dependend). */
  double          u;       /*!< Remapped u coordinate, depends on pixel scaling and wavelength. */
  double          v;       /*!< Remapped v coordinate, depends on pixel scaling and wavelength. */
  double          weight;  /*!< The weight is derived from the density of the measurements after uv remapping. */
  double          wuvvis2; /*!< Weight factor:  weight/fmax(fabs(el->mvar), 1e-30) */
  double          bl;      /*!< Original baseline length in [m]. */
  double complex  ftuk;    /*!< Interpolated complex visibiliy at u,v (from current reconstruction). */
  double          rvis2;   /*!< Reconstructed squared visibility. */
} mat_vis2;

/**
   @brief This data structure contains one loaded and remapped bispectrum entry.

   Each bispectrum entry (OI_T3 binary table) is stored in such a data structure.
   The uv coordinates are remapped, depending on the wavelength and pixel scaling,
   to array coordinates for the FFT. If the binary table contains no valid T3AMP
   and T3MAPERR values, they can be derived from the associated OI_VIS2 entries.
*/
typedef struct {
  double          lambda;  /*!< Wavelength of this closure phase measurement. */
  double          mamp;    /*!< Measured amplitude. */
  double          mamperr; /*!< Copy of one OI_T3.T3AMPERR column entry (wavelength dependend). */
  double          mcp;     /*!< Measured closure phase. */
  double          mcperr;  /*!< Copy of one OI_T3.T3PHIERR column entry (wavelength dependend). */
  double complex  mbis;    /*!< The complex bispectrum value (wavelength dependend), derived from OI_T3.T3AMP and OI_T3.T3PHI. */
  double complex  mphasor; /*!< Normalized bispectrum value (amplitude normalized to 1.0). */
  double          mvar;    /*!< The bispectrum variance (wavelength dependend), derived from OI_T3.T3AMP, OI_T3.T3AMPERR and OI_T3.T3PHIERR. */
  double          u1;      /*!< Remapped u coordinate (OI_T3.U1COORD column), depends on pixel scaling and wavelength. */
  double          v1;      /*!< Remapped v coordinate (OI_T3.V1COORD column), depends on pixel scaling and wavelength. */
  double          u2;      /*!< Remapped u coordinate (OI_T3.U2COORD column), depends on pixel scaling and wavelength. */
  double          v2;      /*!< Remapped v coordinate (OI_T3.V2COORD column), depends on pixel scaling and wavelength. */
  double          u3;      /*!< u1 + u2 */
  double          v3;      /*!< v1 + v2 */
  double          weight;  /*!< The weight is derived from the density of the measurements after uv remapping. */
  double          wuvbis;  /*!< Weight factor for the bispectrum: weight/fmax(fabs(el->mvar), 1e-30). */
  double          wuvcp;   /*!< Weight factor for the closure phase only. */
  double          wuvph;   /*!< Weight factor for the bispectrum phase only. */
  double          wuvmod;  /*!< Weight factor for the bispectrum amplitude only. */
  double complex  ftuk;    /*!< Interpolated complex visibiliy at u1,v1 (from current reconstruction). */
  double complex  ftvk;    /*!< Interpolated complex visibiliy at u2,v2 (from current reconstruction). */
  double complex  ftwk;    /*!< Interpolated complex visibiliy at u3,v3 (from current reconstruction). */
  double complex  rbis;    /*!< Reconstructed closure phase. */
  double complex  rphasor; /*!< Normalized reconstructed bispectrum value (amplitude normalized to 1.0). */
  double          ramp;    /*!< Reconstructed amplitude. */
} mat_bis;

/**
   @brief This data structure contains one loaded and remapped complex visibility.

   Each complex visibility (OI_VIS binary table) is stored in such a data structure.
   The uv coordinates are remapped, depending on the wavelength and pixel scaling,
   to array coordinates for the FFT.
*/
typedef struct {
  double          lambda;   /*!< Wavelength of this squared visibility measurement. */
  double          mamp;     /*!< Copy of one OI_VIS.VISAMP column element (wavelength dependend). */
  double          mamperr;  /*!< Copy of one OI_VIS_VISAMPERR column element (wavelength dependend). */
  double          mphi;     /*!< Copy of one OI_VIS.VISPHI column element (wavelength dependend). */
  double          mphierr;  /*!< Copy of one OI_VIS.VISPHIERR column element (wavelength dependend). */
  double complex  mvis;     /*!< Copy of one OI_VIS.VISAMP/OI_VIS.VISPHI column element (wavelength dependend). */
  double complex  mphasor;  /*!< Normalized measured complex visibility (amplitude normalizeds to 1.0). */
  double          mvar;     /*!< Calculated visibility variance (wavelength dependend). */
  double          mvis2;    /*!< Measured squared visibility derived from the complex visibility. */
  double          mvis2err; /*!< Derived squared visibility error. */
  double          u;        /*!< Remapped u coordinate, depends on pixel scaling and wavelength. */
  double          v;        /*!< Remapped v coordinate, depends on pixel scaling and wavelength. */
  double          weight;   /*!< The weight is derived from the density of the measurements after uv remapping. */
  double          wuvvis;   /*!< Weight factor for the complex visibility:  weight/fmax(fabs(el->mvar), 1e-30) */
  double          wuvph;    /*!< Weight factor for the fourier phase only */
  double          wuvmod;   /*!< Weight factor for the fourier amplitude only */
  double          wuvvis2;  /*!< Weight factor for the squared visibility derived from the complex visibility. */
  double          wuvphi;   /*!< Weight factor for the fourier angle (weight/(mphierr*mphierr)) */
  double          bl;       /*!< Original baseline length in [m]. */
  double complex  ftuk;     /*!< Interpolated complex visibiliy at u,v (from current reconstruction). */
  double complex  rvis;     /*!< Reconstructed visibility. */
  double complex  rphasor;  /*!< Normalized reconstructed complex visibility (amplitude normalizeds to 1.0). */
  double          ramp;     /*!< Reconstructed amplitude. */
  double          rvis2;    /*!< Reconstructed squared visibility. */
} mat_vis;

/**
   @brief Data structure for chi2 and residual calculation.

   During an iteration, each used bispectrum (T3), squared visibility (VIS2) and complex visibility (VIS)
   is used updating sum, weight, respos and resneg. At the end of an iteration, the chi2 and residual is
   calculated from these values.
*/
typedef struct {
  double chi2;             /*!< Final chi2 value for a specific cost element (see data structure mat_rec) */
  double residual;         /*!< Final residual ratio for a specific cost element */
  double sum;              /*!< Current sum for chi2 calculation */
  double weight;           /*!< Current weight for chi2 calculation */
  double respos;           /*!< Current sum of positive residuals */
  double resneg;           /*!< Current sum of negative residuals */
} mat_chi2_info;

/**
   @brief This data structure contains the result of one image reconstruction run.

   Each image reconstruction result contains the images (reconstruction and convolved reconstruction),
   the object mask radius, the regularization hyperparameter, the reconstruction quality including the
   derived errors and lists of reconstructed squared visibilities and closure phases.
*/
typedef struct {
  int             om_idx;     /*!< The loop counter for the object mask radius loop. */
  double          om;         /*!< The radius of the object mask [mas] for this reconstruction. */
  int             mu_idx;     /*!< The loop counter for the regularization parameter loop. */
  double          mu;         /*!< The regularization parameter for this reconstruction. */
  int             nbit;       /*!< The number of iterations for this reconstruction. */
  cpl_image      *rec_image;  /*!< The reconstructed image. */
  cpl_image      *conv_image; /*!< The reconstructed image convolved according to the longest baseline and a scale factor. */
  mat_chi2_info   c2cost;     /*!< The global cost value for ther current iteration */
  mat_chi2_info   c2bis;      /*!< The chi squared and residual ratios for the bispectrum */
  mat_chi2_info   c2cp;       /*!< The chi squared and residual ratios for the closure phase */
  mat_chi2_info   c2vis2;     /*!< The chi squared and residual ratios for the squared visibilities */
  mat_chi2_info   c2vis;      /*!< The chi squared and residual ratios for the complex visibilities */
  mat_chi2_info   c2amp;      /*!< The chi squared and residual ratios for the absolute amplitude */
  mat_chi2_info   c2phi;      /*!< The chi squared and residual ratios for the absolute phase */
  double          cost;       /*!< The cost function value used for ASA-CG. */
  double          qrec;       /*!< The reconstruction quality, derived from chi2vis2, rresvis2, chi2cp and rrescp or from chi2amp, rresamp, chi2phi and rresphi. */
  double          distbc;     /*!< The distance (beauty contest method) between the convolved reconstruction and model image (if a model image is provided). */
  double          dist;       /*!< The distance between the convolved reconstruction and model image (if a model image is provided). */
  double         *vis2_list;  /*!< The reconstructed squared visibilities from the reconstructed image derived at the uv coordinates from the measurements. */
  double complex *bis_list;   /*!< The reconstruced bispectrum from the reconstructed image derived at the uv coordinates from the measurements. */
  double complex *vis_list;   /*!< The reconstruced complex visibility from the reconstructed image derived at the uv coordinates from the measurements. */
} mat_rec;

/**
   @brief This data strucure is a replacement for global variables needed for the image reconstruction.

   This data structure contain all plugin parameters for easy access. Also temporary information needed for loading the calibrated
   interferometric data (e.g. factors for lambda scaling) is stored in such a data structure. All images and arrays needed for
   one image reconstruction run and the surrounding loops (e.g. current regularization hyperparameter) are stored. This data structure
   contains all image reconstruction results as a vector of mat_rec data structures.
*/
typedef struct {
  cpl_parameterlist *parlist;            /*!< Easy access to the plugin parameter list. */
  cpl_frameset      *frameset;           /*!< Easy access to the plugin frameset. */
  // ***** parameters for the ASA-CG algorithm
  asa_parm        asa;                   /*!< This data structure contains the parameter for ASA. */
  asacg_parm      asacg;                 /*!< This data structure contains the parameters for ASA-CG. */
  double          grad_tol;              /*!< The gradient tolerance limit for ASA-CG (parameter --grad_tol, default 1.0e-6). */
  // ***** size of the reconstruction (number of pixels and field of view)
  double          FOV;                   /*!< Field of view for the reconstructed image in [mas]. [40.0] */
  double          flux;                  /*!< Total Flux [1.0] */
  int             npix;                  /*!< Size of the reconstructed image in pixels. Powers of 2 should be used (speeds up the FFT), but this is not enforced. [256] */
  int             nbresult;              /*!< Number of reconstructions written to the result file. The best result is always stored. If 0 is given, all created reconstructions are stored in the result file. [0] */
  double          dxrek;                 /*!< pixel size of the reconstruction [mas], calculated as FOV/npix. */
  // ***** wavelength filter and uv-scaling
  int             lambda_count;                      /*!< Number of wavelength windows. */
  double          lambda_from[MAT_NB_LAMBDA_WINDOW]; /*!< Shortest wavelength for the input data in [um]. [1.0] */
  double          lambda_to[MAT_NB_LAMBDA_WINDOW];   /*!< Longest wavelength for the input data in [um]. [15.0] */
  cpl_vector     *lambda_list;           /*!< This vector contains a list of wavelength from the current OI_WAVELENGTH binary table. */
  cpl_vector     *lambda_scale;          /*!< Scale factors for baseline -> pixel. */
  // ***** engine speficies which optimizer is used (ASA-CG or L-BFGS-B)
  int             engine;                /*!< Specifies which optimizer should be used. [ASA-CG] */
  // ***** mode specifying the data used by this algorithm (bispectrum or complex visibilities) */
  int             algo_mode;             /*!< Specifies if bispectrum or complex visibilities are used for reconstruction. */
  // ***** flag for calculating the T3 amplitude and amplitude error (see paper)
  int             calc_t3amp;            /*!< Flag if the T3 amplitude and error should be calculated. [0] */
  int             calc_vis2f0;           /*!< Flag if an artificial squared visibility and error for f=0 should be calculated. [1] */
  int             calc_visamp;           /*!< Flag if the VIS amplitude and error should be calculated. [0] */
  int             calc_visf0;            /*!< Flag if an artificial complex visibility and error for f=0 should be calculated. [1] */
  // ***** specification of the start image and the image itself
  int             start_mode;            /*!< The mode for reading/creating the start image. 0 = read from file, 1 = point source, 2 = gaussian disc, 3 = uniform disc, 4 = fully darkened disc, 5 = Lorentz disc. [0] */
  double          start_param;           /*!< Additional parameter for the start image creation (mode=0 -> scale [mas/px], mode=2 -> FWHM [mas], mode=3 -> diameter [mas], mode=4 -> diameter [mas], mode=5 -> FWHM [mas]). */
  int             start_select_mode;     /*!< Mode selection for the start image. [4] */
  cpl_image      *start_image_loaded;    /*!< The loaded or created primary start image, always the first start image used. */
  // ***** specification of the prior image and the image itself
  int             prior_mode;            /*!< The mode for reading/creating the prior image. 0 = read from file, 1 = point source, 2 = gaussian disc, 3 = uniform disc, 4 = fully darkened disc, 5 = Lorentz disc. [0] */
  double          prior_param;           /*!< Additional parameter for the prior image creation (mode=0 -> scale [mas/px], mode=2 -> FWHM [mas], mode=3 -> diameter [mas], mode=4 -> diameter [mas], mode=5 -> FWHM [mas]). */
  int             prior_select_mode;     /*!< Mode selection for the prior image. [4] */
  cpl_image      *prior_image_loaded;    /*!< The loaded or created primary prior image, always the first prior image used. */
  // ***** an optional original/model image (only for testing purposes!)
  double          model_scale;           /*!< Pixel scale of the optional model image [mas/px]. */
  cpl_image      *model_image_loaded;    /*!< If a model image is specified, this cpl image contains the loaded and rescaled model image. */
  cpl_image      *model_image_convolved; /*!< If a model image is specified, this cpl image contains the convolved model image. */
  // ***** loaded and mapped interferometric data (squared visibilities, VIS2)
  int             nbvis2;                /*!< The number of loaded squared visibilities. */
  int             nbvis2_allocated;      /*!< The number of squared visibilities for which memory is already allocated. */
  mat_vis2       *vis2_list;             /*!< The list of data structures for squared visibilities. */
  // ***** loaded and mapped interferometric data (bispectra, T3)
  int             nbbis;                 /*!< The number of loaded bispectrum elements. */
  int             nbbis_allocated;       /*!< The number of bispectrum elements for which memory is already allocated. */
  mat_bis        *bis_list;              /*!< The list of data structures for bispectrum elements. */
  // ***** loaded and mapped interferometric data (complex visibilities, VIS)
  int             nbvis;                 /*!< The number of loaded visibilities. */
  int             nbvis_allocated;       /*!< The number of visibilities for which memory is already allocated. */
  mat_vis        *vis_list;              /*!< The list of data structures for squared visibilities. */
  // ***** calculated values from the loaded interferrometric data
  double          minbaseline_px;        /*!< The shortest baseline mapped to the FFT array. */
  double          maxbaseline_px;        /*!< The longest baseline mapped to the FFT array. */
  double          minbaseline_m;         /*!< The shortest baseline in [m]. */
  double          maxbaseline_m;         /*!< The longest baseline in [m].  */
  double          maxfov;                /*!< The biggest calculated FOV [mas]. */
  double          mindxrek;              /*!< The smallest calculated pixel size [mas]. */
  // ***** object mask definition, the mask itself, the bounding box, the pixel coordinates list, ...
  double          om_start;              /*!< Start radius of the object mask [mas]. [1.0] */
  double          om_step;               /*!< Step size for the object mask radius scan [mas]. [1.0] */
  int             om_count;              /*!< Number of object mask radius scans. [6] */
  double          om;                    /*!< The current object mask radius [mas]. */
  double          om_scale;              /*!< Pixel scale of the optional object mask image list. */
  cpl_imagelist  *om_image_list;         /*!< Loaded or created object mask image list. */
  cpl_image      *om_image;              /*!< This image contains the current object mask. */
  int             om_xa;                 /*!< The left side of the object mask (reduced: om_xa > 1). */
  int             om_xe;                 /*!< The right side of the object mask (reduced: om_xe < npix). */
  int             om_ya;                 /*!< The bottom side of the object mask (reduced: om_ya > 1). */
  int             om_ye;                 /*!< The top side of the object mask (reduced: om_ye < npix). */
  int             nbom;                  /*!< The number of pixels (equal to 1.0) inside the object mask. */
  int            *om_list_x;             /*!< A look-up table for the x-coordinate of the pixels (equal to 1.0) inside the object mask. */
  int            *om_list_y;             /*!< A look-up table for the y-coordinate of the pixels (equal to 1.0) inside the object mask. */
  // ***** regularisation parameter and data
  double          mu_start;              /*!< Start value for the regularization parameter mu. [1.0] */
  double          mu_factor;             /*!< Factor between two consecutive regularization parameter values. [0.1] */
  int             mu_count;              /*!< Number of regularization parameter scans. [6] */
  double          mu;                    /*!< The current regularization parameter. */
  int             reg_func;              /*!< Regularisation function (0 = no regularization). [0] */
  double          reg_eps;               /*!< epsilon for regularization function #4 (edge preserving). */
  // ***** additional input parameters
  int             wiener_filter;         /*!< Flag if a Wiener filter is applied to the gradient. */
  double          filter_fwhm;           /*!< FWHM of a gaussian filter for the gradient image. */
  double          filter_factor;         /*!< Factor which is used to calculate the FWHM for the next iteration. */
  double          filter;                /*!< Current filter FWHM, modified after each iteration. */
#ifdef WITH_GRAD_SCALING
  double          grad_scale_start;
  double          grad_scale_factor;
  double          grad_scale;
#endif
  double          weight_power;          /*!< Weight power (uv weight calculation). [1.0] */
  double          conv_scale;            /*!< Scale factor for the convolution (1.0 means max baseline is used). [1.0] */
  int             cost_func;             /*!< Cost function (1 or 2). [1] */
  double          cost_weight;           /*!< Weight for the cost function 2 (weight between closure phase and modulus term). [0.0] */
  int             ncorr;                 /*!< Number of corrections used in L-BFGS-B. [5] */
  double          factr;                 /*!< L-BFGS-B tolerance for termination test. [1e-6] */
  double          pgtol;                 /*!< L-BFGS-B projected gradient tolerance for termination test. [1e1] */
  int             asa_count;             /*!< Number of successive ASA-CG runs for the same parameters. */
  double          cc_threshold;          /*!< Threshold for cross correlation. [0.05] */
#ifdef WITH_PRECISION
  int             precision;             /*!< Number of digits after decimal point used for gradient and cost value. */
#endif
  double          mjd_tol;               /*!< Maximum allowed MJD difference for finding a VIS2 element for a T3 element. */
  double          bl_tol;                /*!< Maximum allowed baseline difference for finding a VIS2 element for a T3 element. */
  double          wl_tol;                /*!< Wavelength tolerance (filter) in [um]. */
  int             info_flags;            /*!< Flags controlling informations during reconstruction. */
  int             guess;                 /*!< Flag if only a model fit is requested. */
  double          fit_fwhm;              /*!< Start value for the model fit (diameter). */
  unsigned int    noise_seed;            /*!< Seed value for random generator (random()) usde for addint noise to mesurements. */
  double          noise_factor;          /*!< Factor to calculate a noise value: noise = factor*error*gaussian_random(). */
  char            vis2_name[256];        /*!< ASCII file for measured and reconstructed squared visibilities. [] */
  char            cp_name[256];          /*!< ASCII file for measured and reconstructed closure phases. [] */
  char            vis_name[256];         /*!< ASCII file for measured and reconstructed complex visibilities. [] */
  // ***** reconstructed images
  char            tname[256];            /*!< Name of the target, extracted from OI_TARGET binary table. */
  int             nbrec;                 /*!< Number of already finished reconstructions. */
  mat_rec        *rec_list;              /*!< List of reconstructions (same order than nested loops). */
  mat_rec       **rec_sorted;            /*!< Sorted list of reconstructions (list of pointers!), needed to select start and/or prior image and fill the result file. */
  mat_rec        *rec_curr;              /*!< Pointer to the currect image reconstruction data structure. */
  cpl_image      *tent_image;            /*!< This cpl image contains the tent function used for the convolution (according to maxbaseline_px). */
  // ***** data needed for one image reconstruction run (one ASA-CG run), may be changed during reconstruction
  int             nbit;                  /*!< The number of iterations for the current image reconstruction run (updated inside ASA-CG). */
  cpl_vector     *lower_bounds;          /*!< A vector containing the smallest intensity values (0.0), needed for ASA-CG. */
  cpl_vector     *upper_bounds;          /*!< A vector containing the largest intensity values (1.0), needed for ASA-CG. */
  cpl_image      *rec_image;             /*!< The current reconstructed image. */
  cpl_image      *nrec_image;            /*!< The normalized current reconstructed image. */
  cpl_vector     *rec_vector;            /*!< The linearized reconstructed image (input for ASA-CG). */
  cpl_image      *frec_image;            /*!< The FFT of the current reconstruction (double complex). */
  cpl_image      *prior_image;           /*!< The currently used prior image, needed for the regularization. */
  cpl_image      *dcost_image;           /*!< Gradient of the cost function. */
  cpl_image      *fdcost_image;          /*!< FFT of the gradient (double complex) of the cost function. */
  cpl_image      *fdcostvar_image;       /*!< Variance of the FFT of the gradient (double). */
  cpl_image      *dreg_image;            /*!< Gradient of the regularization function. */
  cpl_image      *uv_image;              /*!< The uv density image, calculated by mapping the squared visibilities and bispectrum elements onto an array. */
  cpl_image      *filter_image;          /*!< Gaussian for filtering the gradient image. */
  cpl_image      *tdbl_image;            /*!< A simple temporary image with the same size as the reconstruction (double). */
  cpl_vector     *tdbl_vector;           /*!< A simple temporary vector with the same size as the reconstruction (double). */
  cpl_image      *tcpl_image;            /*!< A simple temporary image with the same size as the reconstruction (double complex). */
  double          fit_gd_fwhm;           /*!< The FWHM for a fitted gaussian disc. */
  double          fit_ud_diameter;       /*!< The diameter for a fitted uniform disc. */
  double          fit_fdd_diameter;      /*!< The diameter for a fitted fully darkened disc. */
  double          fit_ld_fwhm;           /*!< The FWHM for a fitted Lorentz disc. */
  double          FOV_guess;             /*!< This is the FOV as a guess according to the loaded interferometric data. */
  int             npix_guess;            /*!< This is the number of pixels derived from the shortest and longest baseline. */
  int             start_mode_guess;      /*!< This the suggested mode for the start image and the prior image, derived from the model fit. */
  double          start_param_guess;     /*!< This is the suggested parameter for the start image and the prior image, derived from the model fit. */
  int             use_flags;             /*!< This contains the USE_?_FLAG specifications. */
} mat_cal_imarec_info;

typedef struct {
  integer  *bound_types; // 0 = unbounded, 1 = lower, 2 = lower and upper 3 = upper
  double   *g; // gradient at x[]
  double   *wa; // Working array for L-BFGS-B, double, (2*nbcorr + 5)*nbvar * 11*nbcorr*nbcorr + 8*nbcorr).
  integer  *iwa; // Working array for L-BFGS-B, integer, 3*nbvar
  integer   csave;
  logical   lsave[4];
  integer   isave[44];
  double    dsave[29];
} lbfgsb_context;

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/
static int mat_cal_imarec_create(cpl_plugin *);
static int mat_cal_imarec_exec(cpl_plugin *);
static int mat_cal_imarec_destroy(cpl_plugin *);
static int mat_cal_imarec(cpl_parameterlist *, cpl_frameset *);

/*-----------------------------------------------------------------------------
  Static variables
  -----------------------------------------------------------------------------*/
static char mat_cal_imarec_description[] =
  "This plugin uses one or several OI-FITS files with calibrated interferometric "
  "measurements to reconstruct an object. This process is controlled by two "
  "additional (optional) inputs specifying a start image and a so called prior "
  "image and an extensive set of recipe parameters.\n"
  "Each OI-FITS file (DO classification TARGET_CAL_INT) must comply to the OI-FITS standard. "
  "This means they contain exactly one OI_TARGET table plus one or more of the data "
  "tables OI_VIS2, OI_T3 or OI_VIS. Each data table must refer to an OI_WAVELENGTH table "
  "present in the file. All other tables defined in the standard are not used by "
  "this plugin.\n"
  "\n"
  "The image reconstruction method is able to use two different kind of data:\n"
  "\n"
  "  1. The bispectrum data is used for the image reconstruction. This method part "
  "is described in details in an A&A paper (Hofmann, K.-H., Weigelt, G., & Schertl, "
  "D. 2014, A&A, 565, A48). It uses the measured calibrated squared visibilities "
  "(OI_VIS2 table) and closure phases (OI_T3 table) at several positions and "
  "wavelength in the uv plane. From these observations the bispectrum elements "
  "and their errors are computed.\n"
  "\n"
  "  2. The complex visibilities are used for the image reconstruction. This "
  "method part uses the measured complex visibilities (OI_VIS table) at several "
  "positions and wavelength in the uv plane.\n"
  "\n"
  "The iterative algorithm searches for the two-dimensional image whose bispectrum "
  "and/or complex visibilities best agrees with the observed bispectrum and/or "
  "complex visibilities.\n"
  "\n"
  "The --algo_mode option is used to select the image reconstruction algorithm:\n"
  "\n"
  "  --algo_mode=1  Bispectrum method (default)\n"
  "  --algo_mode=2  Complex visibilities method\n"
  "  --algo_mode=3  Bispectrum and complex visibilities are used\n"
  "\n"
  "Because of the sparse uv coverage, a regularization term can be "
  "incorporated into the cost function.\n"
  "\n"
  "Minimization of cost function is performed with two different algorithms:\n"
  "\n"
  " 1. By the large-scale, bound-constrained nonlinear optimization algorithm ASA_CG\n"
  "    1) Hager, W. W., & Zhang, H. 2005, SIAM J. Optim., 16, 170\n"
  "    2) Hager, W. W., & Zhang, H. 2006, SIAM J. Optim., 17, 526)\n"
  "\n"
  " 2. By a Limited Memory Algorithm for Bound Constrained Optimization:\n"
  "    1) R. H. Byrd, P. Lu and J. Nocedal. A Limited Memory Algorithm\n"
  "       for Bound Constrained Optimization, (1995), SIAM Journal on Scientific\n"
  "       and Statistical Computing , 16, 5, pp. 1190-1208.\n"
  "    2) C. Zhu, R. H. Byrd and J. Nocedal. L-BFGS-B: Algorithm 778: L-BFGS-B,\n"
  "       FORTRAN routines for large scale bound constrained optimization (1997),\n"
  "       ACM Transactions on Mathematical Software, Vol 23, Num. 4, pp. 550 - 560.\n"
  "    3) J.L. Morales and J. Nocedal. L-BFGS-B: Remark on Algorithm 778: L-BFGS-B,\n"
  "       FORTRAN routines for large scale bound constrained optimization (2011),\n"
  "       to appear in ACM Transactions on Mathematical Software.\n"
  "\n"
  "Before each reconstruction the start and prior image is set to the loaded images "
  "or to a previous reconstruction. This implies a feedback from previous reconstruction "
  "runs and therefore the whole reconstruction chain is needed and no shortcut does exist.\n"
  "\n"
  "The result is defined by the following parameters:\n"
  "\n"
  "  --fov      : Field of view for the reconstructed image in [mas].\n"
  "  --npix     : Size of the reconstructed image in pixels (power of 2).\n"
  "  --nbresult : Number of reconstructions written to the result file.\n"
  "\n"
  "The calibrated interferometric data is filtered using a specific wavelength "
  "intervall. It is possible to specify several different wavelength intervalls:\n"
  "\n"
  "  --lambda_from : Shortest wavelength for the input data in [um].\n"
  "  --lambda_to   : Longest wavelength for the input data in [um].\n"
  "  --lambda_list : A list of lambda ranges (pairs of lower and upper wavelength [um]).\n"
  "\n"
  "Each reconstruction is made for a specific combination of object mask radius "
  "and regularization parameter. Both are controlled with the following parameters:\n"
  "\n"
  "  --om_start  : Start radius of the object mask [mas].\n"
  "  --om_step   : Step size for the object mask radius scan [mas].\n"
  "  --om_count  : Number of object mask radius scans.\n"
  "  --mu_start  : Start value for the regularization parameter mu.\n"
  "  --mu_factor : Factor between two consecutive regularization parameter values.\n"
  "  --mu_count  : Number of regularization parameter scans.\n"
  "\n"
  "It is possible to load one or several objects masks from FITS images "
  "(DO classification OBJECT_MASK). In that case a scaling factor must "
  "be given with the --om_scale parameter. All FITS images in the primary header "
  "unit of all given OBJECT_MASK files are used in the same order as in the SOF.\n"
  "\n"
  "The start image for the iterative algorithm and the prior (target estimate, "
  "needed for the regularization) can be provided using two parameters for each "
  "image:\n"
  "\n"
  "  --start_mode  : The mode for reading/creating the start image.\n"
  "  --start_param : Additional parameter for the start image creation.\n"
  "  --prior_mode  : The mode for reading/creating the prior image.\n"
  "  --prior_param : Additional parameter for the prior image creation.\n"
  "\n"
  "The mode parameter specifies which kind of image should be used and the param "
  "parameter gives an additional value:\n"
  "\n"
  "  mode=0 param=<scale>    : An image is read from a FITS file (DO classification START_IMAGE or PRIOR_IMAGE). The parameter specifies the image scale in [mas/px].\n"
  "  mode=1                  : A point source is put in the image center, no parameter is needed.\n"
  "  mode=2 param=<fwhm>     : A disc with a gaussian profile is put into the image center. The parameter gives the FWHM in [mas].\n"
  "  mode=3 param=<diameter> : A uniform disc is put into the image center. The parameter gives the diameter in [mas].\n"
  "  mode=4 param=<diameter> : A disc with limb darkening: I(r) = I0*sqrt(r0^2 - r^2)/r0\n"
  "                            is put into the center of the image. The parameter gives the diameter (2*r0) in [mas].\n"
  "  mode=5 param=<fwhm>     : A disc with a Lorentz profile: I(r) = 1/(1 + a^2*r^2)^(3/2) is put into the image center.\n"
  "                            The parameter gives the FWHM in [mas] (a = 2/FWHM*sqrt(2^(2/3) - 1)).\n"
  "\n"
  "It is possible to feed a model image of the target into the plugin. In the SOF file "
  "this image is classified as MODEL_IMAGE and the pixel scale must be given using "
  "the --model_scale parameter in [mas/px]. If such a model image is specified, the "
  "distance between this model image and the reconstruction is calculated after each "
  "reconstruction (inner loop):\n"
  "\n"
  "  dist    = Sqrt(Sum{ (m(x,y) - ok(x,y))^2 }/Sum{ m(x,y)^2 })       (see A&A paper, equation 14)\n"
  "  distbc  = Sqrt(Sum{ m(x,y)*(m(x,y) - ok(x,y))^2 }/Sum{ m(x,y) })  (see Lawson, P. R., Cotton, W. D., Hummel, C. A., et al. 2004, in New Frontiers in Stellar Interferometry, ed. W. A. Traub, SPIE Conf. Ser., 5491, 886)\n"

  "  m(x,y)  = model image at (x,y)\n"
  "  ok(x,y) = reconstructed image at (x,y)\n"
  "\n"
  "For each reconstruction run, the start image and the prior (estimated target) "
  "can be selected due to two given criteria (--start_select and --prior_select):\n"
  "\n"
  "  select=0 : The default image is selected.\n"
  "  select=1 : Use the previous reconstruction.\n"
  "  select=2 : The best reconstruction with the same or the previous object mask radius is selected.\n"
  "  select=3 : The best reconstruction up to now.\n"
  "  select=4 : The previous reconstruction or the best reconstruction with the previous object mask radius is selected.\n"
  "  select=5 : The default image or the previous reconstructed image with the same object mask radius is selected.\n"
  "\n"
  "The IRBis algorithm supports two different cost functions (see A&A paper, section 2.2.1). "
  "They can be selected using the --cost_func parameter:\n"
  "\n"
  "  cost_func=1 : chi squared of the bispectrum (equation 4)\n"
  //"                Q[ok(x,y)] = Sum{ weight(u,v)/var(u,v)*|gamma0*ibis(u,v) - mbis(u,v)|^2 }\n"
  "  cost_func=2 : chi squared of the bispectrum phasors (equation 5)\n"
  //"                Q[ok(x,y)] = Sum{ weight(u,v)/varph(u,v)*|gamma0*exp(i iph(u,v)) - exp(i mph(u,v)|^2\n"
  //"                                  + weight(u,v)/varmod(u,v)*f0*|gamma0*imod(u,v) - mmod(u,v)|^2\n"
  //"  cost_func=3 : chi squared of bispectrum and fourier phasors only\n"
  //"  ok(x,y)     = reconstructed image at (x,y)\n"
  //"  weight(u,v) = density weight for a specific bispectrum element\n"
  //"  var(u,v)    = squared bispectrum error\n"
  //"  ibis(u,v)   = iterated bispectrum element\n"
  //"  mbis(u,v)   = measured bispectrum element\n"
  //"  varph(u,v)  = squared error of the bispectrum phase\n"
  //"  iph(u,v)    = iterated bispectrum phase\n"
  //"  mph(u,v)    = measured bispectrum phase\n"
  //"  varmod(u,v) = squared error of the bispectrum modulus\n"
  //"  imod(u,v)   = iterated bispectrum modulus\n"
  //"  mmod(u,v)   = measured bispectrum modulus\n"
  //"  f0          = weight for the closure phase \n"
  "\n"
  "For the regularization term one of the following methods (--reg-func) can be "
  "selected:\n"
  "\n"
  "  reg_func=0 : no regularization\n"

  "  reg_func=1 : pixel intensity quadratic\n"
  //"      H(x,y) = |ok(x,y)|^2/prior(x,y)\n"

  "  reg_func=2 : maximum entropy\n"
  //"      H(x,y) = ok(x,y)*alog(ok(x,y)/prior(x,y)) - ok(x,y) + prior(x,y)\n"

  "  reg_func=3 : pixel difference quadratic\n"
  //"      H(x,y) = [|ok(x,y)-ok(x+dx,y)|^2 + |ok(x,y)-ok(x,y+dy)|^2] / prior(x,y)\n"

  "  reg_func=4 : edge preserving\n"
  //"      H(x,y) = [sqrt[|ok(x+dx,y)-ok(x,y)|^2 + |ok(x,y+dy)-ok(x,y)|^2 + eps^2]-eps] / prior(x,y)\n"

  "  reg_func=5 : smoothness\n"
  //"      H(x,y) = Sum{  |ok(x,y)-ok(x+dx,y+dy)|^2 / prior(x,y), dx={0,1}, dy={0,1}}\n"

  "  reg_func=6 : quadratic Tikhonov\n"
  //"      H(x,y) = |ok(x,y)-prior(x,y)|^2\n"

  //"  ok(x,y)    = reconstructed image at (x,y)\n"
  //"  prior(x,y) = rough estimate of the target at (x,y)\n"
  //"  eps        = additional parameter for edge preserving regularization\n"
  //"  dx         = 1\n"
  //"  dy         = 1\n"
  "\n"
  "The numbering of the regularization functions is the same as in the A&A paper, section 2.2.2. "
  "If a negative number (for example -4) is used, the prior image is set to a "
  "constant image (no estimate of the target).\n"
  "\n"
  "It is possible to call the ASA-CG algorithm several times (parameter --asa_count) "
  "for one reconstruction. This can give better results, because during one run certain "
  "limits decrease. This can lead to a situation where the optimization algorithm get "
  "stuck in a local optimum.\n"
  "\n"
  "During the image reconstruction a parameter (--info_flags) can be used to control "
  "the output of the plugin. The parameter value is a list of comma separated keywords:"
  "\n"
  "  none     : No additional output of the plugin.\n"
  "  debug    : Very specific debugging messages.\n"
  "  param    : Some parameters and the results of the fit process.\n"
  "  images   : Start, prior and model image.\n"
  "  vis2     : Used squared visibilities.\n"
  "  t3       : Used closure phases.\n"
  "  cvis     : Used complex visibilities.\n"
  "  prepare  : Some additional information during preparing the image reconstruction.\n"
  "  istd     : For each iteration the chi2, residuals, quality of the reconstruction and cost value.\n"
  "  idetails : Detailed values for calculating chi2 and residuals.\n"
  "  fits     : Some images (double and complex) are stored in FITS files.\n"
  "  result   : Reconstruction results for each reconstruction run (one combination of om and mu)\n"
  "  all      : Enable all messages.\n"
  "\n"
  "The results from an image reconstruction run are stored in a FITS file "
  "using the following structure:\n"
  "\n"
  "  - The primary header unit contains the best reconstructed image and some "
  "QC values for that image.\n"

  "  - The image extension REC_CONV contains the convolved reconstructed image. "
  "The PSF is based on the longest baseline of the input data and may be scaled "
  "using --conv_scale (negative value -> gaussian PSF, positive value -> tent PSF).\n"

  "  - The binary table extension REC_LIST contains nbresult reconstructions "
  "sorted by the reconstruction quality (QREC).\n"

  "  - The image extension UV_COVERAGE contains an image showing the uv coverage "
  "(mapped to the FFT array).\n"

  "  - The image extension START_IMAGE contains the scaled and shifted start image.\n"

  "  - The image extension PRIOR_IMAGE contains the scaled and shifted prior image.\n"

  "  - The image extension OBJECT_MASK contains all scaled or created object masks.\n"

  "  - The optional image extension MODEL_IMAGE contains the scaled and shifted model image "
  "(DO classification MODEL_IMAGE).\n"

  "  - The optional image extension MODEL_CONV contains the convolved model image "
  "(DO classification MODEL_IMAGE).\n"

  "  - The binary table extension REC_VIS2 contains the measured and the reconstructed "
  "squared visibilities. This table exists when the bispectrum method is used.\n"

  "  - The binary table extension REC_T3 contains the measured and the reconstructed "
  "closure phases. This table exists when the bispectrum method is used.\n"

  "  - The binary table extension REC_VIS contains the measured and the reconstructed "
  "complex visibilities. This table exists when the complex visibilities method is used.\n"
  "\n"
  "A detailed description of the format (binary tables, QC values, ...) can be found in the pipeline manual.\n"
  "\n"
  "This recipe is performing a fit of some simple shapes (gaussian disc, fully darkened disc, uniform disk and Lorentz disc) "
  "to the squared visibilities and complex visibilities (after converting them into squared visibilities). It is possible "
  "to terminate the recipe execution after this step using the option --guess=1. Sometimes the initial value used for the "
  "shape fit is not ok, a better value can be provided using the option --fit_fwhm (value specified in [mas]).\n"
  "\n"
  "This plugin is able to use calibrated interferometric input data scaled with a specific flux. It that case, "
  "this flux must be specified with the --flux option. This will rescale the input data to a standard flux of 1.0. "
  "The result is not rescaled to the specified flux!.\n"
  ;

/*-----------------------------------------------------------------------------
  Functions code
  -----------------------------------------------------------------------------*/

/**
   @brief Build the list of available plugins, for this module.
   @param list the plugin list
   @return 0 if everything is ok, -1 otherwise
   Create the recipe instance and make it available to the application using the
   interface. This function is exported.
*/
int cpl_plugin_get_info(cpl_pluginlist * list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe );
  cpl_plugin *plugin = &recipe->interface;

  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_cal_imarec",
		  "Reconstructs an image based on calibrated interferometric measurements",
		  mat_cal_imarec_description,
		  "Matthias Heininger",
		  "mhein@mpifr-bonn.mpg.de",
		  MATISSE_LICENCE,
		  mat_cal_imarec_create,
		  mat_cal_imarec_exec,
		  mat_cal_imarec_destroy);
  cpl_pluginlist_append(list, plugin);
  return 0;
}

/**
   @brief Setup the recipe options
   @param plugin the plugin
   @return 0 if everything is ok
   Defining the command-line/configuration parameters for the recipe.
*/
static int mat_cal_imarec_create(cpl_plugin * plugin)
{
  cpl_recipe    *recipe;
  cpl_parameter *p;

  /* Check that the plugin is part of a valid recipe */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  /* Create the parameters list in the cpl_recipe object */
  recipe->parameters = cpl_parameterlist_new();
  /* Fill the parameters list */

  /* --fov */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.fov",
			      CPL_TYPE_DOUBLE,
			      "Field of view for the reconstructed image in [mas].",
			      "matisse.mat_cal_imarec",
			      40.0, 0.001, 10000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fov") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --flux */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.flux",
			      CPL_TYPE_DOUBLE,
			      "Total flux of the input data (amplitude scale in T3, VIS2 and VIS) in [Jy].",
			      "matisse.mat_cal_imarec",
			      1.0, 0.001, 10000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "flux") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --npix */
  p = cpl_parameter_new_value("matisse.mat_cal_imarec.npix",
			      CPL_TYPE_INT,
			      "Size of the reconstructed image in pixels. "
			      "Powers of 2 should be used (speeds up the FFT), "
			      "but this is not enforced.",
			      "matisse.mat_cal_imarec",
			      256);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "npix") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --nbresult */
  p = cpl_parameter_new_value("matisse.mat_cal_imarec.nbresult",
			      CPL_TYPE_INT,
			      "Number of reconstructions written to the result file. "
			      "The best result is always stored. "
			      "If 0 is given, all created reconstructions are "
			      "stored in the result file.",
			      "matisse.mat_cal_imarec",
			      0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "nbresult") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --lambda_from */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.lambda_from",
			      CPL_TYPE_DOUBLE,
			      "Shortest wavelength for the input data in [um].",
			      "matisse.mat_cal_imarec",
			      1.0, 1e-3, 1e6);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "lambda_from") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --lambda_to */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.lambda_to",
			      CPL_TYPE_DOUBLE,
			      "Longest wavelength for the input data in [um].",
			      "matisse.mat_cal_imarec",
			      15.0, 1e-3, 1e6);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "lambda_to") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --lambda_list */
  p = cpl_parameter_new_value("matisse.mat_cal_imarec.lambda_list",
			      CPL_TYPE_STRING,
			      "A list of lambda ranges (pairs of lower and upper wavelength). "
			      "It is a sequence of comma separated floating point numbers or \'none\'. "
			      "This list overwrites the --lambda_from and --lambda_to parameters.",
			      "matisse.mat_cal_imarec",
			      "none");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "lambda_list");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --engine */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.engine",
			      CPL_TYPE_INT,
			      "Specifies the optimization engine used for the image reconstruction. "
			      "1 = ASA-CG, 2 = L-BFGS-B",
			      "matisse.mat_cal_imarec",
			      1, 1, 2);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "engine") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --algo_mode */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.algo_mode",
			      CPL_TYPE_INT,
			      "Specifies if bispectrum and/or complex visibilities are used for reconstruction. "
			      "1 = use bispectrum, 2 = use complex visibilities, 3 = use bispectrum and complex visibilities",
			      "matisse.mat_cal_imarec",
			      1, 1, 3);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "algo_mode") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --calc_t3amp */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.calc_t3amp",
			      CPL_TYPE_INT,
			      "Flag if the T3 amplitude and error should be calculated.",
			      "matisse.mat_cal_imarec",
			      1, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "calc_t3amp") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --calc_vis2f0 */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.calc_vis2f0",
			      CPL_TYPE_INT,
			      "Flag if an artificial squared visibility and error for f=0 should be calculated.",
			      "matisse.mat_cal_imarec",
			      1, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "calc_vis2f0") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --calc_visamp */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.calc_visamp",
			      CPL_TYPE_INT,
			      "Flag if the VIS amplitude and error should be calculated.",
			      "matisse.mat_cal_imarec",
			      1, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "calc_visamp") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --calc_visf0 */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.calc_visf0",
			      CPL_TYPE_INT,
			      "Flag if an artificial complex visibility and error for f=0 should be calculated.",
			      "matisse.mat_cal_imarec",
			      1, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "calc_visf0") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --start_mode */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.start_mode",
			      CPL_TYPE_INT,
			      "The mode for reading/creating the start image. "
			      "0 = read from file, 1 = point source, 2 = gaussian disc, "
			      "3 = uniform disc, 4 = fully darkened disc, "
			      "5 = Lorentz disc.",
			      "matisse.mat_cal_imarec",
			      0, 0, 5);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "start_mode") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --start_param */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.start_param",
			      CPL_TYPE_DOUBLE,
			      "Additional parameter for the start image creation "
			      "(mode=0 -> scale [mas/px], "
			      "mode=2 -> FWHM [mas], "
			      "mode=3 -> diameter [mas], "
			      "mode=4 -> diameter [mas], "
			      "mode=5 -> FWHM [mas]).",
			      "matisse.mat_cal_imarec",
			      0.0, 0.0, 1000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "start_param") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --start_select */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.start_select",
			      CPL_TYPE_INT,
			      "Mode selection for the start image",
			      "matisse.mat_cal_imarec",
			      4, 0, 5);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "start_select") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --prior_mode */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.prior_mode",
			      CPL_TYPE_INT,
			      "The mode for reading/creating the prior image. "
			      "0 = read from file, 1 = point source, 2 = gaussian disc, "
			      "3 = uniform disc, 4 = fully darkened disc, "
			      "5 = Lorentz disc.",
			      "matisse.mat_cal_imarec",
			      0, 0, 5);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "prior_mode") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --prior_param */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.prior_param",
			      CPL_TYPE_DOUBLE,
			      "Additional parameter for the prior image creation "
			      "(mode=0 -> scale [mas/px], "
			      "mode=2 -> FWHM [mas], "
			      "mode=3 -> diameter [mas], "
			      "mode=4 -> diameter [mas], "
			      "mode=5 -> FWHM [mas]).",
			      "matisse.mat_cal_imarec",
			      0.0, 0.0, 1000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "prior_param") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --prior_select */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.prior_select",
			      CPL_TYPE_INT,
			      "Mode selection for the prior image",
			      "matisse.mat_cal_imarec",
			      4, 0, 5);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "prior_select") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --model_scale */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.model_scale",
			      CPL_TYPE_DOUBLE,
			      "Pixel scale of the optional model image [mas/px].",
			      "matisse.mat_cal_imarec",
			      0.1, 0.0, 1000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "model_scale") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --weight_power */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.weight_power",
			      CPL_TYPE_DOUBLE,
			      "Weight power (uv weight calculation).",
			      "matisse.mat_cal_imarec",
			      1.0, 0.0, 10.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "weight_power") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --om_start */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.om_start",
			      CPL_TYPE_DOUBLE,
			      "Start radius of the object mask [mas].",
			      "matisse.mat_cal_imarec",
			      1.0, 1e-3, 1e3);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "om_start") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --om_step */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.om_step",
			      CPL_TYPE_DOUBLE,
			      "Step size for the object mask radius scan [mas].",
			      "matisse.mat_cal_imarec",
			      1.0, 0.0, 1e3);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "om_step") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --om_count */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.om_count",
			      CPL_TYPE_INT,
			      "Number of object mask radius scans.",
			      "matisse.mat_cal_imarec",
			      6, 1, 100);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "om_count") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --om_scale */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.om_scale",
			      CPL_TYPE_DOUBLE,
			      "Pixel scale of the optional object mask image [mas/px].",
			      "matisse.mat_cal_imarec",
			      0.1, 0.0, 1e3);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "om_scale") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --mu_start */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.mu_start",
			      CPL_TYPE_DOUBLE,
			      "Start value for the regularization parameter mu.",
			      "matisse.mat_cal_imarec",
			      1.0, 0.0, 1e6);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "mu_start") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --mu_factor */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.mu_factor",
			      CPL_TYPE_DOUBLE,
			      "Factor between two consecutive regularization parameter values.",
			      "matisse.mat_cal_imarec",
			      0.1, 0.0, 10.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "mu_factor") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --mu_count */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.mu_count",
			      CPL_TYPE_INT,
			      "Number of regularization parameter scans.",
			      "matisse.mat_cal_imarec",
			      6, 1, 100);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "mu_count") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --reg_func */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.reg_func",
			      CPL_TYPE_INT,
			      "Regularisation function (0 = no regularization).",
			      "matisse.mat_cal_imarec",
			      0, -6, 6);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "reg_func") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --reg_eps */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.reg_eps",
			      CPL_TYPE_DOUBLE,
			      "Epsilon for regularisation function 4 (edge preserving).",
			      "matisse.mat_cal_imarec",
			      0.0, 0.0, 1000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "reg_eps") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --grad_tol */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.grad_tol",
			      CPL_TYPE_DOUBLE,
			      "Tolerance value for ASA_CG.",
			      "matisse.mat_cal_imarec",
			      0.00000000001, 0.0, 1.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "grad_tol") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --conv_scale */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.conv_scale",
			      CPL_TYPE_DOUBLE,
			      "Scale factor for the convolution (1.0 means max baseline is used, negative value -> gaussian PSF, positive value -> tent PSF).",
			      "matisse.mat_cal_imarec",
			      1.0, -10.0, 10.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "conv_scale") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --cost_func */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.cost_func",
			      CPL_TYPE_INT,
			      "Cost function (1, 2 or 3).",
			      "matisse.mat_cal_imarec",
			      1, -1, 3);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cost_func") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --cost_weight */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.cost_weight",
			      CPL_TYPE_DOUBLE,
			      "Weight for the cost function 2 (weight between closure phase and modulus term).",
			      "matisse.mat_cal_imarec",
			      0.0, 0.0, 1000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cost_weight") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --ncorr */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.ncorr",
			      CPL_TYPE_INT,
			      "Number of  corrections for L-BFGS-B.",
			      "matisse.mat_cal_imarec",
			      5, 3, 20);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "ncorr") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --factr */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.factr",
			      CPL_TYPE_DOUBLE,
			      "L-BFGS-B tolerance for termination test.",
			      "matisse.mat_cal_imarec",
			      10.0, 0.1, 1e12);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "factr") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --pg_tol */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.pg_tol",
			      CPL_TYPE_DOUBLE,
			      "L-BFGS-B projected gradient tolerance for termination test.",
			      "matisse.mat_cal_imarec",
			      1e-6, 0.0, 1.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "pg_tol") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --asa_count */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.asa_count",
			      CPL_TYPE_INT,
			      "Number of ASA-CG iterations per reconstruction",
			      "matisse.mat_cal_imarec",
			      1, 1, 5);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "asa_count") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --cc_threshold */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.cc_threshold",
			      CPL_TYPE_DOUBLE,
			      "Threshold for cross correlation.",
			      "matisse.mat_cal_imarec",
			      0.05, 0.0, 1.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cc_threshold") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --mjd_tol */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.mjd_tol",
			      CPL_TYPE_DOUBLE,
			      "Maximum allowed MJD difference for finding a VIS2 element for a T3 element [d].",
			      "matisse.mat_cal_imarec",
			      0.0001, 0.0, 1.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "mjd_tol") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --bl_tol */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.bl_tol",
			      CPL_TYPE_DOUBLE,
			      "Maximum allowed baseline difference for finding a VIS2 element for a T3 element [m].",
			      "matisse.mat_cal_imarec",
			      0.05, 0.0, 1.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "bl_tol") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --wl_tol */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.wl_tol",
			      CPL_TYPE_DOUBLE,
			      "Maximum allowed wavelength difference wavelength filter [um].",
			      "matisse.mat_cal_imarec",
			      0.0, 0.0, 1e6);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "wl_tol") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

#ifdef WITH_PRECISION
  /* --precision */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.precision",
			      CPL_TYPE_INT,
			      "Number of digits after decimal point for gradient and cost value (precision < 0 : round relative, precision == 0 : no round, precision > 0 : round absolute)",
			      "matisse.mat_cal_imarec",
			      0, -25, 25);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "precision") ;
  cpl_parameterlist_append(recipe->parameters, p) ;
#endif

  /* --wiener_filter */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.wiener_filter",
			      CPL_TYPE_INT,
			      "Flag if a Wiener filter is applied to the gradient.",
			      "matisse.mat_cal_imarec",
			      0, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "wiener_filter") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --filter_fwhm */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.filter_fwhm",
			      CPL_TYPE_DOUBLE,
			      "Start value for a gaussian gradient filter(FWHM) in [mas].",
			      "matisse.mat_cal_imarec",
			      0.0, 0.0, 100000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "filter_fwhm") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --filter_factor */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.filter_factor",
			      CPL_TYPE_DOUBLE,
			      "Factor between two consecutive gradient filter sizes.",
			      "matisse.mat_cal_imarec",
			      0.99, 0.0, 1.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "filter_factor") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

#ifdef WITH_GRAD_SCALING
  /* --scale_start */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.scale_start",
			      CPL_TYPE_DOUBLE,
			      "Start value for a gradient scale.",
			      "matisse.mat_cal_imarec",
			      1.0, 1.0, 100000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "scale_start") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --scale_factor */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.scale_factor",
			      CPL_TYPE_DOUBLE,
			      "Factor between two consecutive gradient scales.",
			      "matisse.mat_cal_imarec",
			      1.0, 0.0, 1.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "scale_factor") ;
  cpl_parameterlist_append(recipe->parameters, p) ;
#endif

  /* --guess */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.guess",
			      CPL_TYPE_INT,
			      "Flag if only a model fit is requested.",
			      "matisse.mat_cal_imarec",
			      0, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "guess") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --fit_fwhm */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.fit_fwhm",
			      CPL_TYPE_DOUBLE,
			      "Start FWHM for the model fit [mas].",
			      "matisse.mat_cal_imarec",
			      2.0, 0.0, 1.0e3);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fit_fwhm") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --noise_seed */
  p = cpl_parameter_new_value("matisse.mat_cal_imarec.noise_seed",
			      CPL_TYPE_INT,
			      "Seed value for the noise random generator.",
			      "matisse.mat_cal_imarec",
			      42);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "noise_seed") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --noise_factor */
  p = cpl_parameter_new_range("matisse.mat_cal_imarec.noise_factor",
			      CPL_TYPE_DOUBLE,
			      "Noise factor (noise_sigma = error*factor) for the noise random generator.",
			      "matisse.mat_cal_imarec",
			      0.0, 0.0, 1.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "noise_factor") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --info_flags */
  p = cpl_parameter_new_value("matisse.mat_cal_imarec.info_flags",
			      CPL_TYPE_STRING,
			      "Flags controlling the information printed during reconstruction",
			      "matisse.mat_cal_imarec",
			      "param");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "info_flags") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --vis2_name */
  p = cpl_parameter_new_value("matisse.mat_cal_imarec.vis2_name",
			      CPL_TYPE_STRING,
			      "ASCII file for measured and reconstructed squared visibilities",
			      "matisse.mat_cal_imarec",
			      "");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "vis2_name");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --cp_name */
  p = cpl_parameter_new_value("matisse.mat_cal_imarec.cp_name",
			      CPL_TYPE_STRING,
			      "ASCII file for measured and reconstructed closure phases",
			      "matisse.mat_cal_imarec",
			      "");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cp_name");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --vis_name */
  p = cpl_parameter_new_value("matisse.mat_cal_imarec.vis_name",
			      CPL_TYPE_STRING,
			      "ASCII file for measured and reconstructed complex visibilities",
			      "matisse.mat_cal_imarec",
			      "");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "vis_name");
  cpl_parameterlist_append(recipe->parameters, p);

  /* Return */
  return 0;
}

/**
   @brief Execute the plugin instance given by the interface
   @param plugin the plugin
   @return 0 if everything is ok
*/
static int mat_cal_imarec_exec(cpl_plugin * plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  return mat_cal_imarec(recipe->parameters, recipe->frames);
}

/**
   @brief Destroy what has been created by the create function
   @param plugin the plugin
   @return 0 if everything is ok
*/
static int mat_cal_imarec_destroy(cpl_plugin * plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  cpl_parameterlist_delete(recipe->parameters);
  return 0;
}

/*--------------------------------------------------------------------*/
/* mat_cal_imarec_info related functions (init, destroy, ...)         */
/*--------------------------------------------------------------------*/
/**
   @brief Initializes the data structure holding all global variables.
   @param info   This data structure contains the image reconstruction context (a replacement for global variables).
*/
static void mat_info_init(mat_cal_imarec_info *info)
{
  asa_default(&(info->asa));
  asa_cg_default(&(info->asacg));
  info->vis2_name[0] = '\0';
  info->cp_name[0] = '\0';
  info->vis_name[0] = '\0';
  info->lambda_list = NULL;
  info->lambda_scale = NULL;
  info->start_image_loaded = NULL;
  info->prior_image_loaded = NULL;
  info->model_image_loaded = NULL;
  info->model_image_convolved = NULL;
  info->nbvis2 = 0;
  info->nbvis2_allocated = 0;
  info->vis2_list = NULL;
  info->nbbis = 0;
  info->nbbis_allocated = 0;
  info->bis_list = NULL;
  info->nbvis = 0;
  info->nbvis_allocated = 0;
  info->vis_list = NULL;
  info->minbaseline_px = 1e30;
  info->maxbaseline_px = 0.0;
  info->minbaseline_m = 1e30;
  info->maxbaseline_m = 0.0;
  info->maxfov = 0.0;
  info->mindxrek = 1e30;
  info->om_image_list = NULL;
  info->om_image = NULL;
  info->om_list_x = NULL;
  info->om_list_y = NULL;
  info->nbrec = 0;
  info->rec_list = NULL;
  info->rec_sorted = NULL;
  info->rec_curr = NULL;
  info->tent_image = NULL;
  info->lower_bounds = NULL;
  info->upper_bounds = NULL;
  info->rec_image = NULL;
  info->nrec_image = NULL;
  info->rec_vector = NULL;
  info->frec_image = NULL;
  info->prior_image = NULL;
  info->dcost_image = NULL;
  info->fdcost_image = NULL;
  info->fdcostvar_image = NULL;
  info->dreg_image = NULL;
  info->uv_image = NULL;
  info->filter_image = NULL;
  info->tdbl_image = NULL;
  info->tcpl_image = NULL;
  info->tdbl_vector = NULL;
  info->use_flags = 0;
}

/**
   @brief This function frees all allocated memory from an mat_cal_imarec_info data structure.
   @param info   This data structure contains the image reconstruction context (a replacement for global variables).
*/
static void mat_info_delete(mat_cal_imarec_info *info)
{
  int i;

  if (info->lambda_list != NULL)
    {
      cpl_vector_delete(info->lambda_list);
      info->lambda_list = NULL;
    }
  if (info->lambda_scale != NULL)
    {
      cpl_vector_delete(info->lambda_scale);
      info->lambda_scale = NULL;
    }
  if (info->start_image_loaded != NULL)
    {
      cpl_image_delete(info->start_image_loaded);
      info->start_image_loaded = NULL;
    }
  if (info->prior_image_loaded != NULL)
    {
      cpl_image_delete(info->prior_image_loaded);
      info->prior_image_loaded = NULL;
    }
  if (info->model_image_loaded != NULL)
    {
      cpl_image_delete(info->model_image_loaded);
      info->model_image_loaded = NULL;
    }
  if (info->model_image_convolved != NULL)
    {
      cpl_image_delete(info->model_image_convolved);
      info->model_image_convolved = NULL;
    }
  if (info->vis2_list != NULL)
    {
      cpl_free(info->vis2_list);
      info->nbvis2 = 0;
      info->nbvis2_allocated = 0;
      info->vis2_list = NULL;
    }
  if (info->bis_list != NULL)
    {
      cpl_free(info->bis_list);
      info->nbbis = 0;
      info->nbbis_allocated = 0;
      info->bis_list = NULL;
    }
  if (info->vis_list != NULL)
    {
      cpl_free(info->vis_list);
      info->nbvis = 0;
      info->nbvis_allocated = 0;
      info->vis_list = NULL;
    }
  if (info->om_image_list != NULL)
    {
      cpl_imagelist_delete(info->om_image_list);
      info->om_image_list = NULL;
    }
  info->om_image = NULL;
  if (info->om_list_x != NULL)
    {
      cpl_free(info->om_list_x);
      info->om_list_x = NULL;
    }
  if (info->om_list_y != NULL)
    {
      cpl_free(info->om_list_y);
      info->om_list_y = NULL;
    }
  if (info->rec_list != NULL)
    {
      for (i = 0; i < info->nbrec; i++)
	{
	  if (info->rec_list[i].rec_image != NULL)
	    {
	      cpl_image_delete(info->rec_list[i].rec_image);
	      info->rec_list[i].rec_image = NULL;
	    }
	  if (info->rec_list[i].conv_image != NULL)
	    {
	      cpl_image_delete(info->rec_list[i].conv_image);
	      info->rec_list[i].conv_image = NULL;
	    }
	  if (info->rec_list[i].vis2_list != NULL)
	    {
	      cpl_free(info->rec_list[i].vis2_list);
	      info->rec_list[i].vis2_list = NULL;
	    }
	  if (info->rec_list[i].bis_list != NULL)
	    {
	      cpl_free(info->rec_list[i].bis_list);
	      info->rec_list[i].bis_list = NULL;
	    }
	  if (info->rec_list[i].vis_list != NULL)
	    {
	      cpl_free(info->rec_list[i].vis_list);
	      info->rec_list[i].vis_list = NULL;
	    }
	}
      cpl_free(info->rec_list);
      info->rec_list = NULL;
    }
  if (info->rec_sorted != NULL)
    { // the pointers in this list are already freed!
      cpl_free(info->rec_sorted);
      info->rec_sorted = NULL;
    }
  info->rec_curr = NULL;
  if (info->tent_image != NULL)
    {
      cpl_image_delete(info->tent_image);
      info->tent_image = NULL;
    }
  if (info->lower_bounds != NULL)
    {
      cpl_vector_delete(info->lower_bounds);
      info->lower_bounds = NULL;
    }
  if (info->upper_bounds != NULL)
    {
      cpl_vector_delete(info->upper_bounds);
      info->upper_bounds = NULL;
    }
  if (info->rec_image != NULL)
    {
      cpl_image_delete(info->rec_image);
      info->rec_image = NULL;
    }
  if (info->nrec_image != NULL)
    {
      cpl_image_delete(info->nrec_image);
      info->nrec_image = NULL;
    }
  if (info->rec_vector != NULL)
    {
      cpl_vector_delete(info->rec_vector);
      info->rec_vector = NULL;
    }
  if (info->frec_image != NULL)
    {
      cpl_image_delete(info->frec_image);
      info->frec_image = NULL;
    }
  if (info->prior_image != NULL)
    {
      cpl_image_delete(info->prior_image);
      info->prior_image = NULL;
    }
  if (info->dcost_image != NULL)
    {
      cpl_image_delete(info->dcost_image);
      info->dcost_image = NULL;
    }
  if (info->fdcost_image != NULL)
    {
      cpl_image_delete(info->fdcost_image);
      info->fdcost_image = NULL;
    }
  if (info->fdcostvar_image != NULL)
    {
      cpl_image_delete(info->fdcostvar_image);
      info->fdcostvar_image = NULL;
    }
  if (info->dreg_image != NULL)
    {
      cpl_image_delete(info->dreg_image);
      info->dreg_image = NULL;
    }
  if (info->uv_image != NULL)
    {
      cpl_image_delete(info->uv_image);
      info->uv_image = NULL;
    }
  if (info->filter_image != NULL)
    {
      cpl_image_delete(info->filter_image);
      info->filter_image = NULL;
    }
  if (info->tdbl_image != NULL)
    {
      cpl_image_delete(info->tdbl_image);
      info->tdbl_image = NULL;
    }
  if (info->tcpl_image != NULL)
    {
      cpl_image_delete(info->tcpl_image);
      info->tcpl_image = NULL;
    }
  if (info->tdbl_vector != NULL)
    {
      cpl_vector_delete(info->tdbl_vector);
      info->tdbl_vector = NULL;
    }
}

/**
   @brief Allocates all memory needed for an image reconstruction run.
   @param info   This data structure contains the image reconstruction context (a replacement for global variables).
*/
static cpl_error_code mat_init_reconstruction(mat_cal_imarec_info *info)
{
  int i;

  // info->lambda_list and info->lambda_scale was previously allocated
  // info->start_image_loaded is created in the mat_read_image function
  // info->prior_image_loaded is created in the mat_read_image function
  // info->vis2_list is (re)allocated in the mat_add_vis2 function
  // info->bis_list is (re)allocated in the mat_add_t3 function
  // info->vis_list is (re)allocated in the mat_add_vis function
  // info->om_image_list is allocated in the mat_create_object_masks function
  info->om_list_x = (int *)cpl_calloc(info->npix*info->npix, sizeof(int));
  if (info->om_list_x == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for object mask x-coordinates");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->om_list_y = (int *)cpl_calloc(info->npix*info->npix, sizeof(int));
  if (info->om_list_y == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for object mask y-coordinates");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->rec_list = (mat_rec *)cpl_calloc(info->om_count*info->mu_count, sizeof(mat_rec));
  if (info->rec_list == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for reconstructions");
      return CPL_ERROR_UNSPECIFIED;
    }
  for (i = 0; i < info->om_count*info->mu_count; i++)
    {
      if (info->nbvis2 != 0)
	{
	  info->rec_list[i].vis2_list = (double *)cpl_calloc(info->nbvis2, sizeof(double));
	  if (info->rec_list[i].vis2_list == NULL)
	    {
	      cpl_msg_error(cpl_func, "can't allocate memory for derived squared visibilities");
	      return CPL_ERROR_UNSPECIFIED;
	    }
	}
      if (info->nbbis != 0)
	{
	  info->rec_list[i].bis_list = (double complex*)cpl_calloc(info->nbbis, sizeof(double complex));
	  if (info->rec_list[i].bis_list == NULL)
	    {
	      cpl_msg_error(cpl_func, "can't allocate memory for derived bispectrum");
	      return CPL_ERROR_UNSPECIFIED;
	    }
	}
      if (info->nbvis != 0)
	{
	  info->rec_list[i].vis_list = (double complex *)cpl_calloc(info->nbvis, sizeof(double complex));
	  if (info->rec_list[i].vis_list == NULL)
	    {
	      cpl_msg_error(cpl_func, "can't allocate memory for derived complex visibilities");
	      return CPL_ERROR_UNSPECIFIED;
	    }
	}
    }
  info->rec_sorted = (mat_rec **)cpl_calloc(info->om_count*info->mu_count, sizeof(mat_rec *));
  if (info->rec_sorted == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for sorted reconstruction indices");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->tent_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
  if (info->tent_image == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for tent image");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->lower_bounds = cpl_vector_new(info->npix*info->npix);
  if (info->lower_bounds == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for lower pixel intensities");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->upper_bounds = cpl_vector_new(info->npix*info->npix);
  if (info->upper_bounds == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for upper pixel intensities");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->rec_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
  if (info->rec_image == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for reconstructed object");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->nrec_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
  if (info->nrec_image == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for normalized reconstructed object");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->rec_vector = cpl_vector_new(info->npix*info->npix);
  if (info->rec_vector == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for linearized reconstruction");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->frec_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE_COMPLEX);
  if (info->frec_image == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for reconstruction (FFT)");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->prior_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
  if (info->prior_image == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for prior");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->dcost_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
  if (info->dcost_image == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for gradient");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->fdcost_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE_COMPLEX);
  if (info->fdcost_image == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for gradient (FFT)");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->fdcostvar_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
  if (info->fdcostvar_image == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for gradient (FFT) variance");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->dreg_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
  if (info->dreg_image == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for regularization gradient");
      return CPL_ERROR_UNSPECIFIED;
    }
  if (info->filter_fwhm != 0.0)
    {
      info->filter_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
      if (info->filter_image == NULL)
	{
	  cpl_msg_error(cpl_func, "can't allocate memory for filter image");
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  info->tdbl_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
  if (info->tdbl_image == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for temporary double array");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->tcpl_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE_COMPLEX);
  if (info->tcpl_image == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for temporary complex array");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->tdbl_vector = cpl_vector_new(info->npix*info->npix);
  if (info->tdbl_vector == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for temporary double vector");
      return CPL_ERROR_UNSPECIFIED;
    }
  return CPL_ERROR_NONE;
}

/*--------------------------------------------------------------------*/
/* parameter handling (plugin and ASA-CG)                             */
/*--------------------------------------------------------------------*/
/**
   @brief Gets the lambda list from a parameter string
   @param info  A mat_cal_imarec_info data structure as a substitute for global variables.
   @param parlist A cpl_parameterlist containing the plugin parameters.
   @return 1 if at least ine lambda filter range was read from the --lambda_list parameter.
*/
static int mat_get_lambda_list(mat_cal_imarec_info  *info,
			       cpl_parameterlist *parlist)
{
  cpl_parameter    *param = NULL;
  const char       *arg = NULL;
  int               i, j;
  char              str[32];

  info->lambda_count = 0;
  if (parlist == NULL)
    {
      return 0;
    }
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.lambda_list");
  if (param == NULL)
    {
      return 0;
    }
  arg = cpl_parameter_get_string(param);
  if (strcmp(arg, "none") == 0)
    {
      // no lambda list available, use the normal --lambda_from and --lambda_to parameters
      return 1;
    }
  i = 0;
  while (arg[i] != '\0')
    {
      /* 1. Extract the lower lambda value from the range list */
      j = 0;
      while ((arg[i] != '\0') && (arg[i] != ','))
	{
	  if (j < 30)
	    {
	      str[j++] = arg[i];
	    }
	  i++;
	}
      str[j] = '\0';
      if (str[0] == '\0')
	{
	  // no floating point number read
	  break;
	}
      info->lambda_from[info->lambda_count] = atof(str);
      /* 2. Skip the comma character between two lambda values */
      if (arg[i] == ',')
	{
	  i++;
	}
      /* 3. Extract the upper lambda value from the range list */
      j = 0;
      while ((arg[i] != '\0') && (arg[i] != ','))
	{
	  if (j < 30)
	    {
	      str[j++] = arg[i];
	    }
	  i++;
	}
      str[j] = '\0';
      if (str[0] == '\0')
	{
	  // no floating point number read
	  break;
	}
      info->lambda_to[info->lambda_count] = atof(str);
      cpl_msg_info(cpl_func, "lambda_range[%d] =%f .. %f",
		   info->lambda_count,
		   info->lambda_from[info->lambda_count],
		   info->lambda_to[info->lambda_count]);
      /* 4. Skip a comma if present and go to the next wavelength range. */
      if (arg[i] == ',')
	{
	  i++;
	}
      info->lambda_count++;
      if (info->lambda_count == MAT_NB_LAMBDA_WINDOW) break;
    }
  return (info->lambda_count != 0);
}

/**
   @brief Gets all plugin parameters from a cpl_parameterlist and stores them in a mat_cal_imarec_info data structure.
   @param info  A mat_cal_imarec_info data structure as a substitute for global variables.
   @param parlist A cpl_parameterlist containing the plugin parameters.
   @return 0 if everything is ok
*/
static int mat_get_parameters(mat_cal_imarec_info  *info,
			      cpl_parameterlist *parlist)
{
  cpl_parameter       *param = NULL;
  int                  i;
  double               v;
  const char          *str;

  /* retrieve all parameters */
  /* --fov */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.fov");
  info->FOV = cpl_parameter_get_double(param);
  /* --flux */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.flux");
  info->flux = cpl_parameter_get_double(param);
  /* --npix */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.npix");
  info->npix = cpl_parameter_get_int(param);
  /* --nbresult */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.nbresult");
  info->nbresult = cpl_parameter_get_int(param);
  mat_get_lambda_list(info, parlist);
  if (info->lambda_count == 0)
    {
      info->lambda_count = 1;
      /* --lambda_from */
      param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.lambda_from");
      info->lambda_from[0] = cpl_parameter_get_double(param);
      /* --lambda_to */
      param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.lambda_to");
      info->lambda_to[0] = cpl_parameter_get_double(param);
    }
  /* --engine */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.engine");
  info->engine = cpl_parameter_get_int(param);
  /* --algo_mode */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.algo_mode");
  info->algo_mode = cpl_parameter_get_int(param);
  /* --calc_t3amp */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.calc_t3amp");
  info->calc_t3amp = cpl_parameter_get_int(param);
  /* --calc_vis2f0 */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.calc_vis2f0");
  info->calc_vis2f0 = cpl_parameter_get_int(param);
  /* --calc_visamp */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.calc_visamp");
  info->calc_visamp = cpl_parameter_get_int(param);
  /* --calc_visf0 */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.calc_visf0");
  info->calc_visf0 = cpl_parameter_get_int(param);
  /* --start_mode */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.start_mode");
  info->start_mode = cpl_parameter_get_int(param);
  /* --start_param */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.start_param");
  info->start_param = cpl_parameter_get_double(param);
  /* --start_select */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.start_select");
  info->start_select_mode = cpl_parameter_get_int(param);
  /* --prior_mode */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.prior_mode");
  info->prior_mode = cpl_parameter_get_int(param);
  /* --prior_param */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.prior_param");
  info->prior_param = cpl_parameter_get_double(param);
  /* --prior_select */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.prior_select");
  info->prior_select_mode = cpl_parameter_get_int(param);
  /* --model_scale */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.model_scale");
  info->model_scale = cpl_parameter_get_double(param);
  /* --weight_power */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.weight_power");
  info->weight_power = cpl_parameter_get_double(param);
  /* --om_start */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.om_start");
  info->om_start = cpl_parameter_get_double(param);
  /* --om_step */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.om_step");
  info->om_step = cpl_parameter_get_double(param);
  /* --om_count */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.om_count");
  info->om_count = cpl_parameter_get_int(param);
  /* --om_scale */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.om_scale");
  info->om_scale = cpl_parameter_get_double(param);
  /* --mu_start */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.mu_start");
  info->mu_start = cpl_parameter_get_double(param);
  /* --mu_factor */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.mu_factor");
  info->mu_factor = cpl_parameter_get_double(param);
  /* --mu_count */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.mu_count");
  info->mu_count = cpl_parameter_get_int(param);
  /* --reg_func */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.reg_func");
  info->reg_func = cpl_parameter_get_int(param);
  /* --reg_eps */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.reg_eps");
  info->reg_eps = cpl_parameter_get_double(param);
  /* --grad_tol */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.grad_tol");
  info->grad_tol = cpl_parameter_get_double(param);
  /* --conv_scale */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.conv_scale");
  info->conv_scale = cpl_parameter_get_double(param);
  /* --cost_func */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.cost_func");
  info->cost_func = cpl_parameter_get_int(param);
  if (info->cost_func == 0)
    {
      info->cost_func = 1;
    }
  /* --cost_weight */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.cost_weight");
  info->cost_weight = cpl_parameter_get_double(param);
  /* --ncorr */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.ncorr");
  info->ncorr = cpl_parameter_get_int(param);
  /* --factr */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.factr");
  info->factr = cpl_parameter_get_double(param);
  /* --pg_tol */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.pg_tol");
  info->pgtol = cpl_parameter_get_double(param);
  /* --asa_count */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.asa_count");
  info->asa_count = cpl_parameter_get_int(param);
  /* --cc_threshold */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.cc_threshold");
  info->cc_threshold = cpl_parameter_get_double(param);
  /* --mjd_tol */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.mjd_tol");
  info->mjd_tol = cpl_parameter_get_double(param);
  /* --bl_tol */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.bl_tol");
  info->bl_tol = cpl_parameter_get_double(param);
  /* --wl_tol */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.wl_tol");
  info->wl_tol = cpl_parameter_get_double(param);
#ifdef WITH_PRECISION
  /* --precision */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.precision");
  info->precision = cpl_parameter_get_int(param);
#endif
  /* --wiener_filter */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.wiener_filter");
  info->wiener_filter = cpl_parameter_get_int(param);
  /* --filter_fwhm */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.filter_fwhm");
  info->filter_fwhm = cpl_parameter_get_double(param);
  /* --filter_factor */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.filter_factor");
  info->filter_factor = cpl_parameter_get_double(param);
#ifdef WITH_GRAD_SCALING
  /* --scale_start */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.scale_start");
  info->grad_scale_start = cpl_parameter_get_double(param);
  /* --scale_factor */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.scale_factor");
  info->grad_scale_factor = cpl_parameter_get_double(param);
#endif
  /* --noise_seed */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.noise_seed");
  info->noise_seed = cpl_parameter_get_int(param);
  /* --noise_factor */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.noise_factor");
  info->noise_factor = cpl_parameter_get_double(param);
  if (info->noise_factor != 0.0)
    {
      srand(info->noise_seed);
    }
  /* --guess */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.guess");
  info->guess = cpl_parameter_get_int(param);
  /* --fit_fwhm */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.fit_fwhm");
  info->fit_fwhm = cpl_parameter_get_double(param);
  /* --info_flags */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.info_flags");
  str = cpl_parameter_get_string(param);
  info->info_flags = INFO_NONE;
  if (strstr(str, "none") != NULL)     info->info_flags  = INFO_NONE;
  if (strstr(str, "debug") != NULL)    info->info_flags |= INFO_DEBUG;
  if (strstr(str, "param") != NULL)    info->info_flags |= INFO_PARAMETER;
  if (strstr(str, "images") != NULL)   info->info_flags |= INFO_INPUT_IMAGES;
  if (strstr(str, "vis2") != NULL)     info->info_flags |= INFO_INPUT_VIS2;
  if (strstr(str, "t3") != NULL)       info->info_flags |= INFO_INPUT_T3;
  if (strstr(str, "cvis") != NULL)     info->info_flags |= INFO_INPUT_VIS;
  if (strstr(str, "prepare") != NULL)  info->info_flags |= INFO_PREPARE;
  if (strstr(str, "istd") != NULL)     info->info_flags |= INFO_ITER_STD;
  if (strstr(str, "idetails") != NULL) info->info_flags |= INFO_ITER_DETAILS;
  if (strstr(str, "fits") != NULL)     info->info_flags |= INFO_FITS;
  if (strstr(str, "result") != NULL)   info->info_flags |= INFO_RESULT;
  if (strstr(str, "all") != NULL)      info->info_flags |= INFO_ALL;
  if (info->info_flags == INFO_NONE)
    { // if the flags != INFO_NONE means that at least one kyword was found in the argument -> no number
      int flags;
      if (sscanf(str, "%8d", &flags) == 1)
	{
	  info->info_flags = flags;
	}
    }
  /* --vis2_name */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.vis2_name");
  strncpy(info->vis2_name, cpl_parameter_get_string(param), 256);
  info->vis2_name[255] = '\0';
  /* --cp_name */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.cp_name");
  strncpy(info->cp_name, cpl_parameter_get_string(param), 256);
  info->cp_name[255] = '\0';
  /* --vis_name */
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_imarec.vis_name");
  strncpy(info->vis_name, cpl_parameter_get_string(param), 256);
  info->vis_name[255] = '\0';
  /* calculate some derived values */
  info->dxrek = info->FOV/(double)(info->npix);
  if (info->nbresult == 0) info->nbresult = info->om_count*info->mu_count;
  if (info->nbresult > (info->om_count*info->mu_count)) info->nbresult = info->om_count*info->mu_count;
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "lambda filter: %f um .. %f um",
		   info->lambda_from[0], info->lambda_to[0]);
      cpl_msg_info(cpl_func, "start image: mode=%d, param=%f", info->start_mode, info->start_param);
      cpl_msg_info(cpl_func, "prior image: mode=%d, param=%f", info->prior_mode, info->prior_param);
      cpl_msg_info(cpl_func, "object mask: %f + %f*[0 .. %d]",
		   info->om_start, info->om_step, info->om_count - 1);
      v = info->mu_start;
      for (i = 0; i < info->mu_count; i++) v *= info->mu_factor;
      cpl_msg_info(cpl_func, "reg parameter: %f , %f, ..., %f",
		   info->mu_start, info->mu_start*info->mu_factor, v);
    }
  /* mapping of algo_mode and cost_func into use_flags (USE_?_TABLE) */
  info->use_flags = 0;
  if (info->algo_mode & MODE_BISPECTRUM)
    { // bispectrum mode is always using T3 and VIS2 tables
      info->use_flags |= (USE_T3_TABLE | USE_VIS2_TABLE);
    }
  if (info->algo_mode & MODE_COMPLEX_VIS)
    { // complex visibilities mode is always using VIS table
      info->use_flags |= USE_VIS_TABLE;
    }
  if (info->cost_func == 3)
    { // cost function 3 means always using VIS2 table
      info->use_flags   |= USE_VIS2_TABLE;
      info->calc_t3amp   = 0; // It makes no sense to calculate the T3 amplitude when only the phasor is used!
      info->calc_visamp  = 0; // It makes no sense to calculate the VIS amplitude when only the phasor is used!
    }
  return 0;
}

/**
   @brief Set the ASA-CG parameters in the info data structure to value used during the image reconstruction
   @param info the data structure collecting the 'static' image reconstruction variables
   @return void
   The data structure elements asaparmint, asaparmdouble, asacgparmint and asacgparmdouble contains all parameters
   needed by the ASA-CG algorithm. This function sets all these parameters to values which are adapted to the
   image reconstruction algorithm. Some of these parameters are modified in the main reconstruction loop.

   The asaparmint contains the following elements (T == 1; F == 0, <em>parameters</em> values that the user may wish to modify):
   <table>
   <tr><th>No</th><th>Name</th><th>Default</th><th>Description</th></tr>
   <tr><td>0</td><td><em>PrintFinal</em></td><td>1</td><td>T => print final statistics;<br/> F => no printout of statistics</td></tr>
   <tr><td>1</td><td><em>PrintLevel</em></td><td>0</td><td>Level 0  = no printing,<br/> ... ,<br/> Level 4 = maximum printing</td></tr>
   <tr><td>2</td><td><em>PrintParms</em></td><td>0</td><td>T => print parameters values;<br/>F =>do not display parmeter values</td></tr>
   <tr><td>3</td><td><em>AArmijo</em></td><td>0</td><td>T => use approximate nonmonotone Armijo line search;<br/>F => use ordinary nonmonotone Armijo line search, switch to approximate Armijo when |f_r-f| < AArmijoFac*|min (f_r, f_{max})|</td></tr>
   <tr><td>4</td><td><em>StopRule</em></td><td>1</td><td>T => ||proj_grad||_infty <= max(grad_tol,initial ||grad||_infty*StopFac);<br/>F => ||proj_grad||_infty <= grad_tol*(1 + |f_k|)</td></tr>
   <tr><td>5</td><td><em>PertRule</em></td><td>1</td><td>T => estimated error in function value = eps*|min (f_r, f_{max});<br/>F => estimated error in function value = eps</td></tr>
   <tr><td>6</td><td><em>GradProjOnly</em></td><td>0</td><td>T => only use gradient projection algorithm;<br/>F => let algorithm decide between grad_proj and cg_descent</td></tr>
   <tr><td>7</td><td><em>max_backsteps</em></td><td>50</td><td>abort cbb when Armijo line search backtracks at least max_backstep times</td></tr>
   <tr><td>8</td><td><em>nshrink</em></td><td>50</td><td>search for non nan function value by shrinking search interval at most nshrink times</td></tr>
   <tr><td>9</td><td>L</td><td>&nbsp;</td><td>update fr if fmin was not improved after</td></tr>
   <tr><td>10</td><td>m</td><td>&nbsp;</td><td>fmax = max (f_{k-i}, i = 0, 1, ...,</td></tr>
   <tr><td>11</td><td>P</td><td>&nbsp;</td><td>update fr if P previous initial stepsize was</td></tr>
   <tr><td>12</td><td>nm</td><td>&nbsp;</td><td>CBB cycle length</td></tr>
   <tr><td>13</td><td>parm4</td><td>NUM_INT_ASA</td><td>maximum previous BB steps used</td></tr>
   </table>

   The asaparmdouble contains the following elements (<em>parameters</em> values that the user may wish to modify):
   <table>
   <tr><th>No</th><th>Name</th><th>Default</th><th>Description</th></tr>
   <tr><td>0</td><td><em>AArmijoFac</em></td><td>1e-8</td><td>&nbsp;</td></tr>
   <tr><td>1</td><td><em>StopFac</em></td><td>0.0</td><td>&nbsp;</td></tr>
   <tr><td>2</td><td><em>eps</em></td><td>1e-6</td><td>&nbsp;</td></tr>
   <tr><td>3</td><td><em>maxit_fac</em></td><td>1.797693134862316E+308</td><td>abort cbb after maxit_fac*n iterations in one pass through cbb</td></tr>
   <tr><td>4</td><td><em>totit_fac</em></td><td>1.797693134862316E+308</td><td>abort cbb after totit_fac*n iterations in all passes through cbb</td></tr>
   <tr><td>5</td><td><em>maxfunc_fac</em></td><td>1.797693134862316E+308</td><td>abort cbb iteration after maxfunc_fac*n function evaluations</td></tr>
   <tr><td>6</td><td><em>pert_lo</em></td><td>2.220446049250313E-013</td><td>loj + pert_lo &lt; xj &lt; hij - pert_hi => xj free</td></tr>
   <tr><td>7</td><td><em>pert_hi</em></td><td>2.220446049250313E-013</td><td></td></tr>
   <tr><td>8</td><td><em>nan_fac</em></td><td>0.2</td><td>factor by which interval shrinks when searching for non nan value</td></tr>
   <tr><td>9</td><td>gamma0</td><td>&nbsp;</td><td>criterion for reinitializing BB stepsize</td></tr>
   <tr><td>10</td><td>gamma1</td><td>&nbsp;</td><td>criterion for updating reference value fr</td></tr>
   <tr><td>11</td><td>gamma2</td><td>&nbsp;</td><td>criterion for updating reference value fr</td></tr>
   <tr><td>12</td><td>delta</td><td>&nbsp;</td><td>Armijo line search parameter</td></tr>
   <tr><td>13</td><td>lmin</td><td>&nbsp;</td><td>Lower bound for initial stepsize</td></tr>
   <tr><td>14</td><td>lmax</td><td>&nbsp;</td><td>Upper bound for initial stepsize</td></tr>
   <tr><td>15</td><td>parm1</td><td>&nbsp;</td><td>used when attempting a quadratic</td></tr>
   <tr><td>16</td><td>parm2</td><td>&nbsp;</td><td>used when attempting a quadratic</td></tr>
   <tr><td>17</td><td>parm3</td><td>&nbsp;</td><td>criterion for reinitializing the BB stepsize</td></tr>
   <tr><td>18</td><td>tau1</td><td>&nbsp;</td><td>if ginorm &lt; tau1*pgnorm, continue gradient</td></tr>
   <tr><td>19</td><td>tau1_decay</td><td>&nbsp;</td><td>decay factor for tau1</td></tr>
   <tr><td>20</td><td>tau2</td><td>&nbsp;</td><td>ginorm &lt; tau2*pgnorm implies subproblem</td></tr>
   <tr><td>21</td><td>tau2_decay</td><td>&nbsp;</td><td>decay factor for tau2</td></tr>
   <tr><td>22</td><td>pgdecay</td><td>&nbsp;</td><td>criterion for checking undecided index set</td></tr>
   <tr><td>23</td><td>armijo_decay</td><td>&nbsp;</td><td>decay factor in Armijo line search</td></tr>
   <tr><td>24</td><td>armijo0</td><td>&nbsp;</td><td>criterion for quadratic interpolation in cbb line search</td></tr>
   <tr><td>25</td><td>armijo1</td><td>&nbsp;</td><td>criterion for quadratic interpolation in cbb line search</td></tr>
   </table>

   Inside the main loop asaparmdouble[3], asaparmdouble[4] and asaparmdouble[5] are modified.

   The asacgparmint contains the following elements (T == 1; F == 0, <em>parameters</em> values that the user may wish to modify):
   <table>
   <tr><th>No</th><th>Name</th><th>Default</th><th>Description</th></tr>
   <tr><td>0</td><td><em>PrintLevel</em></td><td>0</td><td>Level 0  = no printing,<br/> ... ,<br/> Level 4 = maximum printing</td></tr>
   <tr><td>1</td><td><em>PrintParmscg</em></td><td>0</td><td>T => print parameters values;<br/>F => do not display parmeter values</td></tr>
   <tr><td>2</td><td><em>AWolfe</em></td><td>0</td><td>T => use approximate Wolfe line search;<br/>F => use ordinary Wolfe line search, switch to approximate Wolfe when |f_k+1-f_k| &lt; AWolfeFac*C_k, C_k = average size of cost</td></tr>
   <tr><td>3</td><td><em>PertRulecg</em></td><td>1</td><td>T => estimated error in function value is eps*Ck,<br/>F => estimated error in function value is eps</td></tr>
   <tr><td>4</td><td><em>QuadStep</em></td><td>1</td><td>T => attempt quadratic interpolation in line search when |f_k+1 - f_k|/f_k <= QuadCutOff;<br/>F => no quadratic interpolation step</td></tr>
   <tr><td>5</td><td><em>debug</em></td><td>0</td><td>T => check that f_k+1 - f_k <= debugtol*C_k;<br/>F => no checking of function values</td></tr>
   <tr><td>6</td><td><em>nexpand</em></td><td>50</td><td>maximum number of times the bracketing interval grows</td></tr>
   <tr><td>7</td><td><em>nshrinkcg</em></td><td>6</td><td>maximum number of times the bracketing interval shrinks</td></tr>
   <tr><td>8</td><td><em>nsecant</em></td><td>50</td><td>maximum number of secant iterations in line search is nsecant</td></tr>
   <tr><td>9</td><td>AdaptiveBeta</td><td>&nbsp;</td><td>T => choose beta adaptively, F => use theta</td></tr>
   <tr><td>10</td><td>qrestart</td><td>&nbsp;</td><td>number of iterations the function is nearly quadratic before a restart</td></tr>
   </table>

   The asacgparmdouble contains the following elements (<em>parameters</em> values that the user may wish to modify):
   <table>
   <tr><th>No</th><th>Name</th><th>Default</th><th>Description</th></tr>
   <tr><td>0</td><td><em>AWolfeFac</em></td><td>1e-3</td><td>&nbsp;</td></tr>
   <tr><td>1</td><td><em>Qdecay</em></td><td>0.7</td><td>factor in [0, 1] used to compute average cost magnitude C_k as follows:<br/>Q_k = 1 + (Qdecay)Q_k-1, Q_0 = 0,  C_k = C_k-1 + (|f_k| - C_k-1)/Q_k</td></tr>
   <tr><td>2</td><td><em>epscg</em></td><td>1e-6</td><td>&nbsp;</td></tr>
   <tr><td>3</td><td><em>egrow</em></td><td>10.0</td><td>factor by which eps grows when line search fails during contraction</td></tr>
   <tr><td>4</td><td><em>QuadCutOff</em></td><td>1e-12</td><td>&nbsp;</td></tr>
   <tr><td>5</td><td><em>debugtol</em></td><td>1e-10</td><td>&nbsp;</td></tr>
   <tr><td>6</td><td><em>step</em></td><td>0.0</td><td>if step is nonzero, it is the initial step of the initial line search</td></tr>
   <tr><td>7</td><td><em>maxit_faccg</em></td><td>1.797693134862316E+308</td><td>abort cg after maxit_fac*n iterations in one pass</td></tr>
   <tr><td>8</td><td><em>totit_faccg</em></td><td>1.797693134862316E+308</td><td>abort cg after totit_fac*n iterations in all passes</td></tr>
   <tr><td>9</td><td><em>restart_fac</em></td><td>6.0</td><td>conjugate gradient method restarts after (n*restart_fac) iterations</td></tr>
   <tr><td>10</td><td><em>feps</em></td><td>0.0</td><td>stop when -alpha*dphi0 (estimated change in function value) <= feps*|f|</td></tr>
   <tr><td>11</td><td><em>nan_rho</em></td><td>1.3</td><td>after encountering nan, growth factor when searching for a bracketing interval</td></tr>
   <tr><td>12</td><td><em>nan_decay</em></td><td>0.1</td><td>after encountering nan, decay factor for stepsize</td></tr>
   <tr><td>13</td><td>deltacg</td><td>&nbsp;</td><td>Wolfe line search parameter</td></tr>
   <tr><td>14</td><td>sigma</td><td>&nbsp;</td><td>Wolfe line search parameter</td></tr>
   <tr><td>15</td><td>gammacg</td><td>&nbsp;</td><td>decay factor for bracket interval width</td></tr>
   <tr><td>16</td><td>rho</td><td>&nbsp;</td><td>growth factor when searching for initial bracketing interval</td></tr>
   <tr><td>17</td><td>psi0</td><td>&nbsp;</td><td>factor used in starting guess for iteration 1</td></tr>
   <tr><td>18</td><td>psi1</td><td>&nbsp;</td><td>in performing a QuadStep, we evaluate the function at psi1*previous step</td></tr>
   <tr><td>19</td><td>psi2</td><td>&nbsp;</td><td>when starting a new cg iteration, our initial guess for the line search stepsize is psi2*previous step</td></tr>
   <tr><td>20</td><td>BetaLower</td><td>&nbsp;</td><td>lower bound factor for beta</td></tr>
   <tr><td>21</td><td>theta</td><td>&nbsp;</td><td>parameter describing the cg_descent family</td></tr>
   <tr><td>22</td><td>qeps</td><td>&nbsp;</td><td>parameter in cost error for quadratic restart criterion</td></tr>
   <tr><td>23</td><td>qrule</td><td>&nbsp;</td><td>parameter used to decide is cost is quadratic</td></tr>
   </table>

   Inside the main loop asacgparmdouble[7], asacgparmdouble[8], asacgparmdouble[9] are modified.
*/
static void mat_init_asacg_parameters(mat_cal_imarec_info *info)
{
  // set the MATISSE specific asa parameters
  asa_default(&(info->asa));  // init all parameters with their default values
  // parameters values that the user may wish to modify
  // 
  // T => print final statistics
  // F => no printout of statistics */
  info->asa.PrintFinal = 0;

  // Level 0  = no printing, ... , Level 4 = maximum printing
  info->asa.PrintLevel = 0;

  // T => print parameters values
  // F => do not display parmeter values
  info->asa.PrintParms = 0;

  // T => use approximate nonmonotone Armijo line search
  // F => use ordinary nonmonotone Armijo line search, switch to
  // approximate Armijo when |f_r-f| < AArmijoFac*|min (f_r, f_{max})|
  info->asa.AArmijo = 1;
  // double info->asa.AArmijoFac ;

  // Stop Rules:
  // T => ||proj_grad||_infty <= max(grad_tol,initial ||grad||_infty*StopFac)
  // F => ||proj_grad||_infty <= grad_tol*(1 + |f_k|)
  info->asa.StopRule = 1;
  // double info->asa.StopFac ;

  // T => estimated error in function value = eps*|min (f_r, f_{max}) |
  // F => estimated error in function value = eps
  // int    info->asa.PertRule ;
  // double info->asa.eps ;

  // T => only use gradient projection algorithm
  // F => let algorithm decide between grad_proj and cg_descent
  info->asa.GradProjOnly = 0;

  // abort cbb when Armijo line search backtracks at least max_backstep times
  // int info->asa.max_backsteps ;

  // abort cbb after maxit_fac*n iterations in one pass through cbb
  info->asa.maxit_fac = 0.01;

  // abort cbb after totit_fac*n iterations in all passes through cbb
  info->asa.totit_fac = 0.01;

  // abort cbb iteration after maxfunc_fac*n function evaluations
  info->asa.maxfunc_fac = 0.01;

  // loj + pert_lo < xj < hij - pert_hi => xj free
  // double info->asa.pert_lo ;
  // double info->asa.pert_hi ;

  // search for non nan function value by shrinking search interval
  // at most nshrink times
  // int info->asa.nshrink ;

  // factor by which interval shrinks when searching for non nan value
  // double info->asa.nan_fac ;

  asa_cg_default(&(info->asacg));  // init all parameters with their default values
  // parameters values that the user may wish to modify
  // 
  // Level 0  = no printing), ... , Level 4 = maximum printing
  info->asacg.PrintLevel = 0;

  // T => print parameters values
  // F => do not display parmeter values
  info->asacg.PrintParms = 0;

  // T => use approximate Wolfe line search
  // F => use ordinary Wolfe line search, switch to approximate Wolfe when
  //      |f_k+1-f_k| < AWolfeFac*C_k, C_k = average size of cost
  // int    info->asacg.AWolfe ;
  // double info->asacg.AWolfeFac ;

  // factor in [0, 1] used to compute average cost magnitude C_k as follows:
  //   Q_k = 1 + (Qdecay)Q_k-1, Q_0 = 0,  C_k = C_k-1 + (|f_k| - C_k-1)/Q_k
  // double info->asacg.Qdecay ;

  // T => estimated error in function value is eps*Ck,
  // F => estimated error in function value is eps
  // int    info->asacg.PertRule ;
  // double info->asacg.eps ;

  // factor by which eps grows when line search fails during contraction
  // double info->asacg.egrow ;

  // T => attempt quadratic interpolation in line search when
  //      |f_k+1 - f_k|/f_k <= QuadCutOff
  // F => no quadratic interpolation step
  // int    info->asacg.QuadStep ;
  // double info->asacg.QuadCutOff ;

  // T => check that f_k+1 - f_k <= debugtol*C_k
  // F => no checking of function values
  info->asacg.debug = 1;
  // double info->asacg.debugtol ;

  // if step is nonzero, it is the initial step of the initial line search
  // double info->asacg.step ;

  // abort cg after maxit_fac*n iterations in one pass
  info->asacg.maxit_fac = 0.01;

  // abort cg after totit_fac*n iterations in all passes
  info->asacg.totit_fac = 0.01;

  // maximum number of times the bracketing interval grows
  // int info->asacg.nexpand ;

  // maximum number of times the bracketing interval shrinks
  // int info->asacg.nshrink ;

  // maximum number of secant iterations in line search is nsecant
  // int info->asacg.nsecant ;

  // conjugate gradient method restarts after (n*restart_fac) iterations
  // double info->asacg.restart_fac ;

  // stop when -alpha*dphi0 (estimated change in function value) <= feps*|f|
  info->asacg.feps = 1.0e-12;

  // after encountering nan, growth factor when searching for
  // a bracketing interval
  // double info->asacg.nan_rho ;

  // after encountering nan, decay factor for stepsize
  // double info->asacg.nan_decay ;

  /*
  //get_asa_parm(info->asaparmdouble, info->asaparmint);
  info->asaparmint[1] = 0;
  // IMPORTANT: use only nonmonotone Armijo line search, code hangs if ordinary nonmonotone Armijo is used!
  info->asaparmint[3] = 1;
  info->asaparmint[4] = 1;
  info->asaparmint[6] = 0;
  //set_asa_parm(info->asaparmdouble, info->asaparmint);

  //get_asacg_parm(info->asacgparmdouble, info->asacgparmint);
  info->asacgparmint[0] = 0;
  //  bisher erfolgreich angewandt: info->asacgparmint[5] = 1
  info->asacgparmint[5] = 1;
  //  info->asacgparmint[5] = 0;
  //set_asacg_parm(info->asacgparmdouble, info->asacgparmint);
  */
}

/**
   @brief Initializes a mat_chi2_info data structure at the beginning of an iteration.
   @param info Pointer to a chi2_info data structure
*/
static void mat_chi2_init(mat_chi2_info *info)
{
  info->chi2 = 0.0;
  info->residual = 0.0;
  info->sum = 0.0;
  info->weight = 0.0;
  info->respos = 0.0;
  info->resneg = 0.0;
}

/**
   @brief Updates the chi2 components (sum and weight)
   @param info Pointer to a chi2_info data structure
   @param value Added to sum
   @param weight Added to weight
*/
static void mat_chi2_add_value(mat_chi2_info *info, double value, double weight)
{
  info->sum    += value;
  info->weight += weight;
}

/**
   @brief Updates th eresidual components (respos and resneg)
   @param info Pointer to a chi2_info data structure
   @param residual Double signed residual
*/
static void mat_chi2_add_residual_double(mat_chi2_info *info, double residual)
{
  if (residual >= 0.0)
    {
      info->respos +=  residual;
    }
  else
    {
      info->resneg += -residual;
    }
}

/*
  static void mat_chi2_add_residual_complex(mat_chi2_info *info, double complex residual)
  {
  mat_chi2_add_residual_double(info, creal(residual));
  mat_chi2_add_residual_double(info, cimag(residual));
  }
*/

/**
   @brief Updates the chi2 and residual components (sum, weight, respos and resneg)
   @param info Pointer to a chi2_info data structure
   @param value Added to sum
   @param weight Added to weight
   @param residual Double signed residual
*/
static void mat_chi2_add_value_residual_double(mat_chi2_info *info, double value, double weight, double residual)
{
  mat_chi2_add_value(info, value, weight);
  mat_chi2_add_residual_double(info, residual);
}

/**
   @brief Updates the chi2 and residual components (sum, weight, respos and resneg)
   @param info Pointer to a chi2_info data structure
   @param value Added to sum
   @param weight Added to weight
   @param residual Double complex signed residual
*/
static void mat_chi2_add_value_residual_complex(mat_chi2_info *info, double value, double weight, double complex residual)
{
  mat_chi2_add_value(info, value, weight);
  mat_chi2_add_residual_double(info, creal(residual));
  mat_chi2_add_residual_double(info, cimag(residual));
}

/**
   @brief Calculates the chi2 and residual ratio using sun, weight, respos and resneg
   @param info Pointer to a chi2_info data structure
*/
static void mat_chi2_update(mat_chi2_info *info)
{
  if (info->weight != 0.0)
    {
      info->chi2 = info->sum/info->weight;
    }
  else
    {
      info->chi2 = 0.0;
    }
  if ((info->respos > 0.0) && (info->resneg > 0.0))
    info->residual = fmax(info->resneg/info->respos, info->respos/info->resneg);
  else
    info->residual = 10000.0;
}

/**
   @brief Initialize all cost components (mat_chi2_info data structures inside a mat_rec data structure)
   @param rec A mat_rec data structure representing the current reconstruction and iteration
*/
static void mat_cost_info_init(mat_rec *rec)
{
  mat_chi2_init(&(rec->c2cost));
  mat_chi2_init(&(rec->c2bis));
  mat_chi2_init(&(rec->c2cp));
  mat_chi2_init(&(rec->c2vis2));
  mat_chi2_init(&(rec->c2vis));
  mat_chi2_init(&(rec->c2amp));
  mat_chi2_init(&(rec->c2phi));
}

/**
   @brief Calculate all cost components (mat_chi2_info data structures inside a mat_rec data structure)
   @param rec A mat_rec data structure representing the current reconstruction and iteration
*/
static void mat_cost_info_update_rec(mat_rec *rec)
{
  mat_chi2_update(&(rec->c2cost));
  mat_chi2_update(&(rec->c2bis));
  mat_chi2_update(&(rec->c2cp));
  mat_chi2_update(&(rec->c2vis2));
  mat_chi2_update(&(rec->c2vis));
  mat_chi2_update(&(rec->c2amp));
  mat_chi2_update(&(rec->c2phi));
}

#ifdef WITH_BESSELJ
/*----------------------------------------------------------------------*/
/* implementation of the Bessel function fo the first kind (not in C99) */
/*----------------------------------------------------------------------*/

/* bessel.c
                      Copyright (c) 1998 
                  Kapteyn Institute Groningen
                     All Rights Reserved.
*/

/*
  #>            bessel.dc2

  Function:     BESSEL

  Purpose:      Evaluate Bessel function J, Y, I, K of integer order.

  Category:     MATH

  File:         bessel.c

  Author:       M.G.R. Vogelaar

  Use:          See bessj.dc2, bessy.dc2, bessi.dc2 or bessk.dc2
                      
  Description:  The differential equation 

                       2
                   2  d w       dw      2   2
                  x . --- + x . --- + (x - v ).w = 0
                        2       dx
                      dx
                      
                has two solutions called Bessel functions of the first kind
                Jv(x) and Bessel functions of the second kind Yv(x).
                The routines bessj and bessy return the J and Y for 
                integer v and therefore are called Bessel functions 
                of integer order.
              
                The differential equation 

                       2
                   2  d w       dw      2   2
                  x . --- + x . --- - (x + v ).w = 0
                        2       dx
                      dx
                      
                has two solutions called modified Bessel functions
                Iv(x) and Kv(x).
                The routines bessi and bessk return the I and K for 
                integer v and therefore are called Modified Bessel 
                functions of integer order.
                (Abramowitz & Stegun, Handbook of mathematical 
                functions, ch. 9, pages 358,- and 374,- )
                            
                The implementation is based on the ideas from 
                Numerical Recipes, Press et. al. 
                This routine is NOT callable in FORTRAN.

  Updates:      Jun 29, 1998: VOG, Document created.
  #<
*/

#define ACC 40.0
#define BIGNO 1.0e10
#define BIGNI 1.0e-10

static double bessj0( double x )
/*------------------------------------------------------------*/
/* PURPOSE: Evaluate Bessel function of first kind and order  */
/*          0 at input x                                      */
/*------------------------------------------------------------*/
{
  double ax,z;
  double xx,y,ans,ans1,ans2;

  if ((ax=fabs(x)) < 8.0) {
    y=x*x;
    ans1=57568490574.0+y*(-13362590354.0+y*(651619640.7+y*(-11214424.18+y*(77392.33017+y*(-184.9052456)))));
    ans2=57568490411.0+y*(1029532985.0+y*(9494680.718+y*(59272.64853+y*(267.8532712+y*1.0))));
    ans=ans1/ans2;
  } else {
    z=8.0/ax;
    y=z*z;
    xx=ax-0.785398164;
    ans1=1.0+y*(-0.1098628627e-2+y*(0.2734510407e-4+y*(-0.2073370639e-5+y*0.2093887211e-6)));
    ans2=-0.1562499995e-1+y*(0.1430488765e-3+y*(-0.6911147651e-5+y*(0.7621095161e-6-y*0.934935152e-7)));
    ans=sqrt(0.636619772/ax)*(cos(xx)*ans1-z*sin(xx)*ans2);
  }
  return ans;
}

static double bessj1( double x )
/*------------------------------------------------------------*/
/* PURPOSE: Evaluate Bessel function of first kind and order  */
/*          1 at input x                                      */
/*------------------------------------------------------------*/
{
  double ax,z;
  double xx,y,ans,ans1,ans2;

  if ((ax=fabs(x)) < 8.0) {
    y=x*x;
    ans1=x*(72362614232.0+y*(-7895059235.0+y*(242396853.1+y*(-2972611.439+y*(15704.48260+y*(-30.16036606))))));
    ans2=144725228442.0+y*(2300535178.0+y*(18583304.74+y*(99447.43394+y*(376.9991397+y*1.0))));
    ans=ans1/ans2;
  } else {
    z=8.0/ax;
    y=z*z;
    xx=ax-2.356194491;
    ans1=1.0+y*(0.183105e-2+y*(-0.3516396496e-4+y*(0.2457520174e-5+y*(-0.240337019e-6))));
    ans2=0.04687499995+y*(-0.2002690873e-3+y*(0.8449199096e-5+y*(-0.88228987e-6+y*0.105787412e-6)));
    ans=sqrt(0.636619772/ax)*(cos(xx)*ans1-z*sin(xx)*ans2);
    if (x < 0.0) ans = -ans;
  }
  return ans;
}

/*
#>            bessj.dc2

Function:     bessj

Purpose:      Evaluate Bessel function of first kind of integer order.

Category:     MATH

File:         bessel.c

Author:       M.G.R. Vogelaar

Use:          #include "bessel.h"
              double   result; 
              result = bessj( int n,
                              double x )


              bessj    Return the Bessel function of integer order
                       for input value x.
              n        Integer order of Bessel function.
              x        Double at which the function is evaluated.

                      
Description:  bessj evaluates at x the Bessel function of the first kind 
              and of integer order n. 
              This routine is NOT callable in FORTRAN.

Updates:      Jun 29, 1998: VOG, Document created.
#<
*/

static double bessj( int n, double x )
/*------------------------------------------------------------*/
/* PURPOSE: Evaluate Bessel function of first kind and order  */
/*          n at input x                                      */
/* The function can also be called for n = 0 and n = 1.       */
/*------------------------------------------------------------*/
{
  int    j, jsum, m;
  double ax, bj, bjm, bjp, sum, tox, ans;


  if (n < 0)
    {
      return 0.0;
    }
  ax=fabs(x);
  if (n == 0)
    return( bessj0(ax) );
  if (n == 1)
    return( bessj1(ax) );
      

  if (ax == 0.0)
    return 0.0;
  else if (ax > (double) n) {
    tox=2.0/ax;
    bjm=bessj0(ax);
    bj=bessj1(ax);
    for (j=1;j<n;j++) {
      bjp=j*tox*bj-bjm;
      bjm=bj;
      bj=bjp;
    }
    ans=bj;
  } else {
    tox=2.0/ax;
    m=2*((n+(int) sqrt(ACC*n))/2);
    jsum=0;
    bjp=ans=sum=0.0;
    bj=1.0;
    for (j=m;j>0;j--) {
      bjm=j*tox*bj-bjp;
      bjp=bj;
      bj=bjm;
      if (fabs(bj) > BIGNO) {
	bj *= BIGNI;
	bjp *= BIGNI;
	ans *= BIGNI;
	sum *= BIGNI;
      }
      if (jsum) sum += bj;
      jsum=!jsum;
      if (j == n) ans=bjp;
    }
    sum=2.0*sum-bj;
    ans /= sum;
  }
  return  x < 0.0 && n%2 == 1 ? -ans : ans;
}
#endif

/*--------------------------------------------------------------------*/
/* cpl_vector related functions                                       */
/*--------------------------------------------------------------------*/
/**
   @brief Sets all elements of a vector to a specific value.
   @param vec  The cpl vector as destination.
   @param v    The double float value used to initialize the vector.
*/
static void mat_vector_fill(cpl_vector *vec, double v)
{
  int             i;
  int             n = cpl_vector_get_size(vec);
  double         *dp;

  dp = cpl_vector_get_data(vec);
  for (i = 0; i < n; i++)
    {
      *dp++ = v;
    }
}

/*--------------------------------------------------------------------*/
/* cpl_image related functions                                        */
/* The images used in this plugin have coordinates                    */
/* between -N/2 and N/2-1 or between 0 and N-1.                       */
/*--------------------------------------------------------------------*/
/*--------------------------------------------------------------------*/
/* Functions needed to prepare the image reconstruction               */
/*--------------------------------------------------------------------*/
/**
   @brief Reads an image from a file or creates one using command line parameters.
   @param info      A mat_cal_imarec_info data structure as a substitute for global variables.
   @param frameset  A cpl frameset containing a FITS file with an image.
   @param tag       A tag used to specify a FITS file in the frame set.
   @param mode      A mode for reading or creating an image.
   @param param     An additional parameter for rescaling or creating an image.

   This function reads an image from a FITS file and rescales it or creates an image
   using an additional parameter:

   <table>
   <tr>
   <th>mode</th>
   <th>param</th>
   <th>description</th>
   </tr>
   <tr>
   <td>0</td>
   <td>scale</td>
   <td>An image is read from a FITS file (DO classification as parameter <em>tag</em>). The parameter specifies the image scale in [mas/px].</td>
   </tr>
   <tr>
   <td>1</td>
   <td>&nbsp;</td>
   <td>A point source is put in the image center, no parameter is needed.</td>
   </tr>
   <tr>
   <td>2</td>
   <td>FWHM</td>
   <td>A disc with a gaussian profile is put into the image center. The parameter gives the FWHM in [mas].</td>
   </tr>
   <tr>
   <td>3</td>
   <td>diameter</td>
   <td>A uniform disc is put into the image center. The parameter gives the diameter in [mas].</td>
   </tr>
   <tr>
   <td>4</td>
   <td>diameter</td>
   <td>A disc with limb darkening:<br/>I(r) = I0(1 - w(1 - cos(theta)))<br/>w = 1.0<br/>cos(theta) = sqrt(1 - r^2/r0)<br/>is put into the center of the image. The parameter gives the diameter (2*r0) in [mas].</td>
   </tr>
   </table>
*/
static cpl_image *mat_read_image(mat_cal_imarec_info *info,
				 cpl_frameset *frameset,
				 const char *tag,
				 int mode,
				 double param)
{
  int        fcount;
  int        x, y;
  double     dsize;
  cpl_image *img = NULL;

  if (info->info_flags & INFO_INPUT_IMAGES)
    {
      cpl_msg_info(cpl_func, "create an image (%s), mode=%d, param=%f, dxrek=%f",
		   tag, mode, param, info->dxrek);
    }
  img = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
  if (img == 0)
    {
      cpl_msg_error(cpl_func, "can't allocate image");
      return NULL;
    }
  mat_image_fill(img, 0.0);
  switch (mode)
    {
    case MAT_MODE_IMAGE_OPTIONAL: // read an (optional) image from a file given by the tag
    case MAT_MODE_IMAGE_MANDATORY:  // read an (mandatory) image from a file given by the tag
      fcount = cpl_frameset_count_tags(frameset, tag);
      if (fcount == 1)
	{ // we found an image, try to load it
	  cpl_frame *imgframe = NULL;
	  cpl_image *oimg = NULL;
	  imgframe = cpl_frameset_find(frameset, tag); // already checked previously!
	  cpl_frame_set_group(imgframe, CPL_FRAME_GROUP_RAW);
	  if (info->info_flags & INFO_INPUT_IMAGES)
	    {
	      cpl_msg_info(cpl_func, "  reading image from %s", cpl_frame_get_filename(imgframe));
	    }
	  oimg = cpl_image_load(cpl_frame_get_filename(imgframe), CPL_TYPE_DOUBLE,0,0);
	  if(oimg != NULL)
	    {
	      //scale startimage
	      if (info->info_flags & INFO_INPUT_IMAGES)
		{
		  cpl_msg_info(cpl_func, "  scale image with dxrek=%f, scale=%f", info->dxrek, param);
		}
	      mat_image_swap(oimg);
	      mat_image_scale(img, info->dxrek, oimg, param);
	      cpl_image_delete(oimg);
	    }
	  else if (mode == MAT_MODE_IMAGE_OPTIONAL)
	    { // the optional image could not be loaded-> return NULL
	      cpl_image_delete(img);
	      img = NULL;
	    }
	  else
	    { // the mandatory image could not be loaded  -> return an empty image
	      cpl_msg_error(cpl_func, "can't load image from %s", cpl_frame_get_filename(imgframe));
	    }
	}
      else if (fcount == 0)
	{
	  if (mode == MAT_MODE_IMAGE_OPTIONAL)
	    { // the optional image does not exist -> return NULL
	      cpl_image_delete(img);
	      img = NULL;
	    }
	  else
	    { // the mandatory image does not exist -> return an empty image
	      cpl_msg_error(cpl_func, "the mandatory image does not exist");
	    }
	}
      else
	{
	  if (mode == MAT_MODE_IMAGE_OPTIONAL)
	    {
	      cpl_msg_error(cpl_func, "only zero or one optional image (tag=%s) is allowed in the SOF", tag);
	      cpl_image_delete(img);
	      img = NULL;
	    }
	  else
	    {
	      cpl_msg_error(cpl_func, "only zero or one mandatory image (tag=%s) is allowed in the SOF", tag);
	    }
	}
      break;
    case MAT_MODE_STAR: // create a single star
      if (info->info_flags & INFO_INPUT_IMAGES)
	{
	  cpl_msg_info(cpl_func, "  create a single star");
	}
      mat_image_set_double(img, 0, 0, 1.0);
      break;
    case MAT_MODE_GAUSSIAN_DISC: // create a gaussian disc with a given FWHM
      param /= info->dxrek; // convert into pixels
      if (info->info_flags & INFO_INPUT_IMAGES)
	{
	  cpl_msg_info(cpl_func, "  create a gaussian disc with FWHM=%g [px]", param);
	}
      mat_image_create_gaussian(img, param);
      break;
    case MAT_MODE_UNIFORM_DISC: // create a uniform disc with a given diameter
      param /= info->dxrek; // convert into pixels
      if (info->info_flags & INFO_INPUT_IMAGES)
	{
	  cpl_msg_info(cpl_func, "  create a uniform disc with diameter=%g [px]", param);
	}
      dsize = 0.25*param*param; // calculate the scaled radius (squared)
      for (x = -info->npix/2; x < info->npix/2; x++)
	{
	  for (y = -info->npix/2; y < info->npix/2; y++)
	    {
	      double r2 = (double)(x*x + y*y);
	      if (r2 <= dsize)
		{
		  mat_image_set_double(img, x, y, 1.0);
		}
	    }
	}
      break;
    case MAT_MODE_FULLY_DARKENED_DISC: // create a fully darkened disc with a given diameter
      param /= info->dxrek; // convert into pixels
      if (info->info_flags & INFO_INPUT_IMAGES)
	{
	  cpl_msg_info(cpl_func, "  create a fully darkened disc with diameter=%g [px]", param);
	}
      dsize = 0.25*param*param; // calculate the scaled radius (squared)
      for (x = -info->npix/2; x < info->npix/2; x++)
	{
	  for (y = -info->npix/2; y < info->npix/2; y++)
	    {
	      double r2 = (double)(x*x + y*y);
	      if (r2 <= dsize)
		{
		  mat_image_set_double(img, x, y, sqrt(1.0 - r2/dsize));
		}
	    }
	}
      break;
    case MAT_MODE_LORENTZ_DISC: // create a Lorentz disc with a given FWHM
      param /= info->dxrek; // convert into pixels
      if (info->info_flags & INFO_INPUT_IMAGES)
	{
	  cpl_msg_info(cpl_func, "  create a Lorentz disc with FWHM=%g [px]", param);
	}
      mat_image_create_lorentzian(img, param);
      break;
    default:;
    }
  if (info->info_flags & INFO_INPUT_IMAGES)
    {
      double total = mat_image_get_total(img);
      cpl_msg_info(cpl_func, "  total flux %g", total);
    }
  mat_image_normalize(img);
  return img;
}

static cpl_imagelist *mat_read_imagelist(mat_cal_imarec_info *info,
					 cpl_frameset *frameset,
					 const char *tag,
					 double param)
{
  cpl_frameset_iterator *it;
  cpl_frame             *frame;
  cpl_imagelist         *img_list = NULL;

  if (info->info_flags & INFO_INPUT_IMAGES)
    {
      cpl_msg_info(cpl_func, "create an image list (%s), param=%f, dxrek=%f",
		   tag, param, info->dxrek);
    }
  img_list = cpl_imagelist_new();
  if (img_list == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate image list");
      return NULL;
    }

  // load all FITS files images tagged as tag
  it = cpl_frameset_iterator_new(frameset);
  frame = cpl_frameset_iterator_get(it);
  while (frame != NULL)
    {
      // we use only frames with the given tag
      if (strcmp(cpl_frame_get_tag(frame), tag) == 0)
	{
	  cpl_imagelist   *hlist;
	  cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
	  if (info->info_flags & INFO_PARAMETER)
	    {
	      cpl_msg_info(cpl_func, "loading and processing of %s", cpl_frame_get_filename(frame));
	    }
	  hlist = cpl_imagelist_load(cpl_frame_get_filename(frame), CPL_TYPE_DOUBLE, 0);
	  if (hlist == NULL)
	    {
	      cpl_msg_error(cpl_func, "can't load image from file %s", cpl_frame_get_filename(frame));
	    }
	  else
	    {
	      cpl_image   *himg;
	      cpl_image   *img = NULL;
	      cpl_size     i;
	      if (info->info_flags & INFO_INPUT_IMAGES)
		{
		  cpl_msg_info(cpl_func, "  scale %d images with dxrek=%f, scale=%f", (int)cpl_imagelist_get_size(hlist), info->dxrek, param);
		}
	      for (i = 0; i < cpl_imagelist_get_size(hlist); i++)
		{
		  himg = cpl_imagelist_get(hlist, i);
		  img = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
		  if (img != NULL)
		    {
		      mat_image_swap(himg);
		      mat_image_scale(img, info->dxrek, himg, param);
		      cpl_imagelist_set(img_list, img, cpl_imagelist_get_size(img_list));
		    }
		}
	      cpl_imagelist_delete(hlist);
	    }
	}
      cpl_frameset_iterator_advance(it, 1);
      frame = cpl_frameset_iterator_get(it);
    }
  cpl_frameset_iterator_delete(it);
  if (cpl_imagelist_get_size(img_list) == 0)
    {
      cpl_imagelist_delete(img_list);
      return NULL;
    }
  else
    {
      return img_list;
    }
}

/**
   @brief Selects a start or prior image using a criteria specified with a parameter.
   @param info    The image reconstruction context.
   @param def     The default image (loaded start or prior image).
   @param mode    The selection mode.
   @param iom     Index number of the object mask radius (outer loop index).
   @param imu     Index number of the regularization hyper parameter (inner loop).
   @returns A CPL image.

   For each image reconstruction run inside the two loops (object mask radius
   and regularization hyper parameter) a start and prior image is selected. For
   both images a mode parameter (-start_select and -prior_select) defines the selection
   schema:

   <ol start="0">
   <li>The default image is selected.</li>
   <li>Use the previous reconstruction.</li>
   <li>The best reconstruction with the same or the previous object mask radius is selected.</li>
   <li>The best reconstruction up to now.</li>
   <li>The previous reconstruction or the best reconstruction with the previous object mask radius is selected.</li>
   <li>The default image or the previous reconstructed image with the same object mask radius is selected.</li>
   </ol>
*/
static cpl_image *mat_select_image(mat_cal_imarec_info *info, cpl_image *def, int mode, int iom, int imu)
{
  int         i;

  if (info->nbrec == 0)
    { // no reconstruction yet -> use the default image
      return def;
    }
  if (mode == 0)
    { // the default image is selected
      return def;
    }
  if (mode == 1)
    { // use the previous reconstruction (info->nbrec - 1)
      return info->rec_list[info->nbrec - 1].rec_image;
    }
  if (mode == 2)
    { // the best reconstruction with the same or the previous object mask radius is selected
      if (imu == 0)
	{ // it is the first reconstruction for the current object mask radius
	  // -> use the previous mask radius (because we already have reconstructions (see first test!) iom is at least 1!)
	  for (i = 0; i < info->nbrec; i++)
	    {
	      if (info->rec_sorted[i]->om_idx == (iom - 1))
		{ // we use the first reconstructed image with the previous object mask radius
		  return info->rec_sorted[i]->rec_image;
		}
	    }
	}
      else
	{ // we already have at least one reconstruction with the same object mask radius
	  for (i = 0; i < info->nbrec; i++)
	    {
	      if (info->rec_sorted[i]->om_idx == iom)
		{ // we use the first reconstructed image with the previous object mask radius
		  return info->rec_sorted[i]->rec_image;
		}
	    }
	}
    }
  if (mode == 3)
    { // the best reconstruction up to now
      return info->rec_sorted[0]->rec_image;
    }
  if (mode == 4)
    { // the best reconstruction with the previous object mask radius is selected
      if (imu == 0)
	{ // it is the first reconstruction for the current object mask radius
	  // -> use the previous mask radius (because we already have reconstructions (see first test!) iom is at least 1!)
	  for (i = 0; i < info->nbrec; i++)
	    {
	      if (info->rec_sorted[i]->om_idx == (iom - 1))
		{ // we use the first reconstructed image with the previous object mask radius
		  return info->rec_sorted[i]->rec_image;
		}
	    }
	}
      else
	{ // we already have at least one reconstruction with the same object mask radius
	  // -> use the last reconstruction (info->nbrec - 1)
	  return info->rec_list[info->nbrec - 1].rec_image;
	}
    }
  if (mode == 5)
    { // the default image or the previous reconstructed image with the same object mask radius is selected
      if (imu == 0)
	{ // it is the first reconstruction for the current object mask radius
	  // -> use the default image
	  return def;
	}
      else
	{ // we already have at least one reconstruction with the same object mask radius
	  for (i = 0; i < info->nbrec; i++)
	    {
	      if (info->rec_sorted[i]->om_idx == iom)
		{ // we use the first reconstructed image with the previous object mask radius
		  return info->rec_sorted[i]->rec_image;
		}
	    }
	}
    }
  // undefined selection mode -> use the default image
  return def;
}

/*--------------------------------------------------------------------*/
/* Function for loading and preprocessing the interferometric data    */
/*--------------------------------------------------------------------*/
/**
   @brief Calculates scale factors for mapping uv coordinates into FFT array indices.
   @param info      A mat_cal_imarec_info data structure as a substitute for global variables.
   @param oiwave    A data structure containing one OI_WAVELENGTH binary table.
   @returns An error code

   This function calculates fr each wavelength in an OI_WAVELENGTH binary table a scale
   factor for the uv coordinate mapping. If a wavelength is outside of the lambda filter,
   the scale factor for this wavelength is set to 0.0. The scale factor is calculated using
   f := pi/180*FOV/lambda (beware of the units for the field of view and the wavelength).
*/
static cpl_error_code mat_calc_baseline_scale(mat_cal_imarec_info *info, mat_oiwavelength *oiwave)
{
  int i, j;

  if ((info == NULL) || (oiwave == NULL))
    {
      return CPL_ERROR_NULL_INPUT;
    }
  if (info->lambda_list == NULL)
    {
      info->lambda_list = cpl_vector_new(oiwave->nbchannel);
    }
  else
    {
      cpl_vector_set_size(info->lambda_list, oiwave->nbchannel);
    }
  if (info->lambda_scale == NULL)
    {
      info->lambda_scale = cpl_vector_new(oiwave->nbchannel);
    }
  else
    {
      cpl_vector_set_size(info->lambda_scale, oiwave->nbchannel);
    }
  if (info->lambda_scale == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the baseline scaling");
      return CPL_ERROR_NULL_INPUT;
    }
  for (i = 0; i < oiwave->nbchannel; i++)
    {
      int  inside = 0;
      cpl_vector_set(info->lambda_list, i, oiwave->effwave[i]);
      for (j = 0; j < info->lambda_count; j++)
	{
	  double ll, ul;
	  ll = (info->lambda_from[j] - info->wl_tol)/1.0e6; // um -> m
	  ul = (info->lambda_to[j] + info->wl_tol)/1.0e6;   // um -> m
	  if ((oiwave->effwave[i] >= ll) && (oiwave->effwave[i] <= ul))
	    {
	      inside = 1;
	    }
	  if (info->info_flags & (INFO_INPUT_VIS2 | INFO_INPUT_T3 | INFO_PREPARE))
	    {
	      cpl_msg_info(cpl_func, "  check %.3g with [%.3g .. %.3g] -> %d",
			   oiwave->effwave[i], ll, ul, inside);
			   
	    }
	}
      if (inside)
	{
	  cpl_vector_set(info->lambda_scale, i, info->FOV*CPL_MATH_PI/(oiwave->effwave[i]*18.0*6.0*6.0*1.0e6));
	}
      else
	{
	  cpl_vector_set(info->lambda_scale, i, 0.0);
	}
      if (info->info_flags & (INFO_INPUT_VIS2 | INFO_INPUT_T3 | INFO_PREPARE))
	{
	  cpl_msg_info(cpl_func, "scale m -> pixel = %f, lambda = %.3E",
		       cpl_vector_get(info->lambda_scale, i), oiwave->effwave[i]);
	}
    }
  return CPL_ERROR_NONE;
}

/* Gaussian noise with mean m and variance s, uses the Box-Muller transformation */
static double gaussian_noise(double m, double s)
{
  double r1, r2, val;
  
  r1=((double)rand())/((double)RAND_MAX);
  r2=((double)rand())/((double)RAND_MAX);
  val=sqrt(-2.0*log(r1))*cos(2.0*M_PI*r2);
  val=s*val+m;
  return val;
}

/**
   @brief Filters and adds all squared visibilities to the internal data structure.
   @param info      A mat_cal_imarec_info data structure as a substitute for global variables.
   @param oivis2    A data structure containing a OI_VIS2 binary table.
   @returns An error code

   This function adds all entries of a loaded OI_VIS2 binary table to the internal
   data structures after filtering (use of the lambda filter and the FLAG column in
   the binary table) and remapping of the uv coordinates.
*/
static cpl_error_code mat_add_vis2(mat_cal_imarec_info *info, mat_oivis2 *oivis2)
{
  int     ivis2, ichan;
  int     nb = 0;
  double  blm, blpx;
  double  flux2;

  if ((info == NULL) || (oivis2 == NULL))
    {
      return CPL_ERROR_NULL_INPUT;
    }
  flux2 = info->flux*info->flux;
  for (ivis2 = 0; ivis2 < oivis2->nbvis2; ivis2++)
    {
      mat_vis2elem  *el = oivis2->list_vis2[ivis2];
      blm = sqrt(el->ucoord*el->ucoord + el->vcoord*el->vcoord);
      if (blm == 0.0) continue; // ignore data points with a baseline of 0.0
      info->minbaseline_m = fmin(blm, info->minbaseline_m);
      info->maxbaseline_m = fmax(blm, info->maxbaseline_m);
      for (ichan = 0; ichan < oivis2->nbchannel; ichan++)
	{
	  mat_vis2 *vis2;
	  double    scale;
	  if (el->bandflag[ichan]) continue; // Ignore if flag == true (OI-FITS paper from 2005, September 28)
	  if (el->vis2err[ichan] < 0.0) continue; // Request from Harl-Heinz Hofmann
	  if (cpl_vector_get(info->lambda_scale, ichan) == 0.0) continue; // Ignore data outside the given range
	  // ensure that we have enough space for at least two additional entries (due to adding the zero frequency later)
	  if ((info->nbvis2 + 1) >= info->nbvis2_allocated)
	    { // we allocate a fixed amount of entries
	      info->nbvis2_allocated += 1000;
	      info->vis2_list = cpl_realloc(info->vis2_list, info->nbvis2_allocated*sizeof(mat_vis2));
	      if (info->vis2_list == NULL)
		{
		  cpl_msg_error(cpl_func, "cannot allocate memory for the squared visibilities");
		  return CPL_ERROR_NULL_INPUT;
		}
	    }
	  vis2 = &(info->vis2_list[info->nbvis2]);
	  vis2->lambda = cpl_vector_get(info->lambda_list, ichan);
	  if (info->noise_factor == 0.0)
	    {
	      vis2->mvis2  = fabs(el->vis2[ichan]/flux2);
	      vis2->mvar   = el->vis2err[ichan]*el->vis2err[ichan]/flux2/flux2;
	    }
	  else
	    {
	      vis2->mvis2  = fabs(el->vis2[ichan]/flux2);
	      vis2->mvar   = el->vis2err[ichan]*el->vis2err[ichan]/flux2/flux2;
	      vis2->mvis2 += gaussian_noise(0.0, el->vis2err[ichan]/flux2*info->noise_factor);
	    }
	  vis2->u      = cpl_vector_get(info->lambda_scale, ichan)*el->ucoord;
	  vis2->v      = cpl_vector_get(info->lambda_scale, ichan)*el->vcoord;
	  vis2->weight = 0.0;
	  vis2->bl     = blm;
	  blpx = sqrt(vis2->u*vis2->u + vis2->v*vis2->v);
	  info->minbaseline_px = fmin(blpx, info->minbaseline_px);
	  info->maxbaseline_px = fmax(blpx, info->maxbaseline_px);
	  // update the calculated baseline, FOV and pixel scale
	  scale = vis2->lambda/blm/CPL_MATH_PI*180.0*60.0*60.0*1000.0;
	  info->maxfov   = fmax(scale, info->maxfov);
	  info->mindxrek = fmin(scale, info->mindxrek);
	  info->nbvis2++;
	  nb++;
	}
    }
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "  %d squared visibilities added", nb);
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Filters and adds all closure phases to the internal data structure.
   @param info      A mat_cal_imarec_info data structure as a substitute for global variables.
   @param oit3      A data structure containing a OI_T3 binary table.
   @returns An error code

   This function adds all entries of a loaded OI_T3 binary table to the internal
   data structures after filtering (use of the lambda filter and the FLAG column in
   the binary table) and remapping of the uv coordinates.
*/
static cpl_error_code mat_add_t3(mat_cal_imarec_info *info, mat_oit3 *oit3)
{
  int     it3, ichan;
  int     onb;
  double  bl;
  double  flux3;

  if ((info == NULL) || (oit3 == NULL))
    {
      return CPL_ERROR_NULL_INPUT;
    }
  flux3 = info->flux*info->flux*info->flux;
  onb = info->nbbis;
  for (it3 = 0; it3 < oit3->nbt3; it3++)
    {
      mat_t3elem *el = oit3->list_t3[it3];
      if ((el->u1coord == 0.0) && (el->v1coord == 0.0)) continue;
      if ((el->u2coord == 0.0) && (el->v2coord == 0.0)) continue;
      for (ichan = 0; ichan < oit3->nbchannel; ichan++)
	{
	  mat_bis *bis;
	  double   phi, phierr;
	  double   amp, amperr;
	  if (el->bandflag[ichan]) continue; // Ignore if flag == true (OI-FITS paper from 2005, September 28)
	  if (el->t3phierr[ichan] < 0.0) continue;  // Request from Harl-Heinz Hofmann
	  if (el->t3amperr[ichan] < 0.0) continue;  // Request from Harl-Heinz Hofmann
	  if (cpl_vector_get(info->lambda_scale, ichan) == 0.0) continue; // Ignore data outside the given range
	  // ensure that we have enough space for at least on additional entry
	  if (info->nbbis >= info->nbbis_allocated)
	    { // we allocate a fixed amount of entries
	      info->nbbis_allocated += 1000;
	      info->bis_list = cpl_realloc(info->bis_list, info->nbbis_allocated*sizeof(mat_bis));
	      if (info->bis_list == NULL)
		{
		  cpl_msg_error(cpl_func, "cannot allocate memory for the bispectrum");
		  return CPL_ERROR_NULL_INPUT;
		}
	    }
	  phi    = CPL_MATH_PI*el->t3phi[ichan]/180.0;     // degree -> radian
	  phierr = CPL_MATH_PI*el->t3phierr[ichan]/180.0;  // degree -> radian
	  amp    = fabs(el->t3amp[ichan]/flux3);
	  amperr = fabs(el->t3amperr[ichan]/flux3);
	  if ((info->cost_func != 3) && (amp == 0.0) && (amperr == 0.0)) continue;
	  //if (phierr > CPL_MATH_PI/2.0) continue;
	  bis = &(info->bis_list[info->nbbis]);
	  if (info->noise_factor != 0.0)
	    {
	      amp += gaussian_noise(0.0, amperr*info->noise_factor);
	      phi += gaussian_noise(0.0, phierr*info->noise_factor);
	    }
	  bis->lambda = cpl_vector_get(info->lambda_list, ichan);
	  bis->mamp    = amp;
	  bis->mamperr = amperr;
	  bis->mcp     = phi;
	  bis->mcperr  = phierr;
	  bis->mphasor = (cos(phi) + sin(phi)*I);
	  bis->mbis    = bis->mphasor*(amp + 0.0*I);
	  bis->mvar    = amperr*amperr + amp*amp*(1 - exp(-phierr*phierr));
	  bis->u1    = cpl_vector_get(info->lambda_scale, ichan)*el->u1coord;
	  bis->v1    = cpl_vector_get(info->lambda_scale, ichan)*el->v1coord;
	  bis->u2    = cpl_vector_get(info->lambda_scale, ichan)*el->u2coord;
	  bis->v2    = cpl_vector_get(info->lambda_scale, ichan)*el->v2coord;
	  bis->u3    = bis->u1 + bis->u2;
	  bis->v3    = bis->v1 + bis->v2;
	  bis->weight = 0.0;
	  bl = sqrt(bis->u1*bis->u1 + bis->v1*bis->v1);
	  info->minbaseline_px = fmin(bl, info->minbaseline_px);
	  info->maxbaseline_px = fmax(bl, info->maxbaseline_px);
	  bl = sqrt(bis->u2*bis->u2 + bis->v2*bis->v2);
	  info->minbaseline_px = fmin(bl, info->minbaseline_px);
	  info->maxbaseline_px = fmax(bl, info->maxbaseline_px);
	  bl = sqrt(bis->u3*bis->u3 + bis->v3*bis->v3);
	  info->minbaseline_px = fmin(bl, info->minbaseline_px);
	  info->maxbaseline_px = fmax(bl, info->maxbaseline_px);
	  info->nbbis++;
	}
    }
  if (info->nbbis == onb)
    {
      if (info->calc_t3amp)
	{
	  cpl_msg_warning(cpl_func, "OI_T3 binary table contains no valid T3AMP and T3AMPERR values, please correct this table or use a different file");
	}
      else
	{
	  cpl_msg_warning(cpl_func, "OI_T3 binary table contains no valid T3AMP and T3AMPERR values, please use --calc_t3amp=1 or a different file");
	}
      return CPL_ERROR_UNSPECIFIED;
    }
  else
    {
      if (info->info_flags & INFO_PARAMETER)
	{
	  cpl_msg_info(cpl_func, "  %d bispectra added", info->nbbis - onb);
	}
    }
  return CPL_ERROR_NONE;
}

static mat_vis2elem *mat_find_vis2elem(mat_cal_imarec_info *info, mat_oivis2 *oivis2, double mjd, double u, double v)
{
  int           i;
  mat_vis2elem *found;
  double        found_diff_mjd;
  double        found_diff_bl;

  found = oivis2->list_vis2[0];
  found_diff_mjd = fabs(found->dateobsmjd - mjd);
  found_diff_bl  = sqrt((found->ucoord - u)*(found->ucoord - u) + (found->vcoord - v)*(found->vcoord - v));
  for (i = 0; i < oivis2->nbvis2; i++)
    {
      mat_vis2elem *el;
      double        diff_mjd;
      double        diff_bl;
      el = oivis2->list_vis2[i];
      diff_mjd = fabs(el->dateobsmjd - mjd);
      diff_bl  = sqrt((el->ucoord - u)*(el->ucoord - u) + (el->vcoord - v)*(el->vcoord - v));
      if ((diff_mjd <= found_diff_mjd) && (diff_bl <= found_diff_bl))
	{
	  found = el;
	  found_diff_mjd = diff_mjd;
	  found_diff_bl  = diff_bl ;
	}
    }
  if ((found_diff_mjd > info->mjd_tol) || (found_diff_bl  > info->bl_tol))
    {
      if (info->info_flags & INFO_DEBUG)
	{
	  cpl_msg_info(cpl_func, "VIS2 not found for uv %g %g mjd %g  best match: uv %g %g mjd %g diff uv %g mjd %g",
		       u, v, mjd,
		       found->ucoord, found->ucoord, found->dateobsmjd,
		       found_diff_bl, found_diff_mjd);
	}
      return NULL;
    }
  return found;
}

/**
   @brief Calculates the T3 amplitude out of the VIS2 data.
   @param info    The image reconstruction context.
   @param oit3    This data structure contains one OI_T3 binary table.
   @param oivis2  This data structure contains one OI_VIS2 binary table.
   @returns A cpl_error_code.

   For each T3 table entry the associated VIS2 entries are searched
   and used to calculate the T3 amplitude and error. These values are
   stored in the T3 table entry and later used from the mat_add_t3()
   function.
*/
static cpl_error_code mat_calculate_t3amp(mat_cal_imarec_info *info, mat_oit3 *oit3, mat_oivis2 *oivis2)
{
  int           it3, ivis2, ichan;

  if ((info == NULL) || (oit3 == NULL) || (oivis2 == NULL))
    {
      return CPL_ERROR_NULL_INPUT;
    }
  // we sort the OI_VIS2 array with increasing u/v coordinate (1. u-, 2. v-coordinate)
  if (info->info_flags & INFO_INPUT_T3)
    {
      cpl_msg_info(cpl_func, "  VIS2: %dx%d, T3: %dx%d",
		   oivis2->nbvis2, oivis2->nbchannel,
		   oit3->nbt3, oit3->nbchannel);
      for (ivis2 = 0; ivis2 < oivis2->nbvis2; ivis2++)
	{
	  mat_vis2elem *el = oivis2->list_vis2[ivis2];
	  cpl_msg_info(cpl_func, "  VIS2[%d] uv=(%g, %g)", ivis2, el->ucoord, el->vcoord);
	}
    }

  for (it3 = 0; it3 < oit3->nbt3; it3++)
    {
      mat_t3elem    *t3 = oit3->list_t3[it3];
      mat_vis2elem  *v2a  = NULL;
      mat_vis2elem  *v2b  = NULL;
      mat_vis2elem  *v2c  = NULL;
      // search and (hopefully find) the VIS2 elements for the current T3 element
      v2a = mat_find_vis2elem(info, oivis2, t3->dateobsmjd, t3->u1coord, t3->v1coord);
      v2b = mat_find_vis2elem(info, oivis2, t3->dateobsmjd, t3->u2coord, t3->v2coord);
      v2c = mat_find_vis2elem(info, oivis2, t3->dateobsmjd, t3->u1coord + t3->u2coord, t3->v1coord + t3->v2coord);
      if ((v2a != NULL) && (v2b != NULL) && (v2c != NULL))
	{
	  for (ichan = 0; ichan < oit3->nbchannel; ichan++)
	    {
	      double v2u   = v2a->vis2[ichan];
	      double v2v   = v2b->vis2[ichan];
	      double v2uv  = v2c->vis2[ichan];
	      double v2ue  = v2a->vis2err[ichan];
	      double v2ve  = v2b->vis2err[ichan];
	      double v2uve = v2c->vis2err[ichan];
	      double namp  = sqrt(fabs(v2u*v2v*v2uv)); // namp >= 0.0
	      double nerr;
	      if ((v2a->bandflag[ichan] == 1) || (v2b->bandflag[ichan] == 1) || (v2c->bandflag[ichan] == 1))
		{
		  t3->bandflag[ichan] = 1;
		  continue;
		}
	      if ((v2ue <= 0.0) || (v2ve <= 0.0) || (v2uve <= 0.0) || (namp == 0.0))
		{
		  t3->bandflag[ichan] = 1;
		  continue;
		}
	      nerr = 0.5*namp*sqrt(v2ue*v2ue/v2u/v2u + v2ve*v2ve/v2v/v2v + v2uve*v2uve/v2uv/v2uv); // nerr >= 0.0
	      if (info->info_flags & INFO_INPUT_T3)
		{
		  cpl_msg_info(cpl_func,
			       "  T3: uv1=(%g, %g), uv2=(%g, %g), old=%g (%g), new=%g (%g)",
			       t3->u1coord, t3->v1coord,
			       t3->u2coord, t3->v2coord,
			       t3->t3amp[ichan], t3->t3amperr[ichan],
			       namp, nerr);
		}
	      t3->t3amp[ichan]    = namp;
	      t3->t3amperr[ichan] = nerr;
	    }
	}
      else
	{ // we did not find all three VIS2 entries => warning and ignore this T3 entry
	  cpl_msg_warning(cpl_func,
			  "  can not find VIS2 data for T3 with (%g, %g)=%d, (%g, %g)=%d and (%g, %g)=%d",
			  t3->u1coord, t3->v1coord, (v2a != NULL),
			  t3->u2coord, t3->v2coord, (v2b != NULL),
			  t3->u1coord + t3->u2coord, t3->v1coord + t3->v2coord, (v2c != NULL));
	  for (ichan = 0; ichan < oit3->nbchannel; ichan++)
	    {
	      t3->bandflag[ichan] = 1;
	    }
	}
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Filters and adds all visibilities to the internal data structure.
   @param info      A mat_cal_imarec_info data structure as a substitute for global variables.
   @param oivis     A data structure containing a OI_VIS binary table.
   @returns An error code

   This function adds all entries of a loaded OI_VIS binary table to the internal
   data structures after filtering (use of the lambda filter and the FLAG column in
   the binary table) and remapping of the uv coordinates.
*/
static cpl_error_code mat_add_vis(mat_cal_imarec_info *info, mat_oivis *oivis)
{
  int     ivis, ichan;
  int     nb = 0;
  double  blm, blpx;
  double  flux;

  if ((info == NULL) || (oivis == NULL))
    {
      return CPL_ERROR_NULL_INPUT;
    }
  flux = info->flux;
  for (ivis = 0; ivis < oivis->nbvis; ivis++)
    {
      mat_viselem  *el = oivis->list_vis[ivis];
      blm = sqrt(el->ucoord*el->ucoord + el->vcoord*el->vcoord);
      if (blm == 0.0) continue; // ignore data points with a baseline of 0.0
      info->minbaseline_m = fmin(blm, info->minbaseline_m);
      info->maxbaseline_m = fmax(blm, info->maxbaseline_m);
      for (ichan = 0; ichan < oivis->nbchannel; ichan++)
	{
	  mat_vis  *vis;
	  double    scale;
	  double    phi, phierr;
	  double    amp, amperr;
	  if (el->bandflag[ichan]) continue; // Ignore if flag == true (OI-FITS paper from 2005, September 28)
	  if (el->visamperr[ichan] < 0.0) continue; // Request from Harl-Heinz Hofmann
	  if (el->visphierr[ichan] < 0.0) continue; // Request from Harl-Heinz Hofmann
	  if (cpl_vector_get(info->lambda_scale, ichan) == 0.0) continue; // Ignore data outside the given range
	  // ensure that we have enough space for at least two additional entries (due to adding the zero frequency later)
	  if ((info->nbvis + 1) >= info->nbvis_allocated)
	    { // we allocate a fixed amount of entries
	      info->nbvis_allocated += 1000;
	      info->vis_list = cpl_realloc(info->vis_list, info->nbvis_allocated*sizeof(mat_vis));
	      if (info->vis_list == NULL)
		{
		  cpl_msg_error(cpl_func, "cannot allocate memory for the visibilities");
		  return CPL_ERROR_NULL_INPUT;
		}
	    }
	  phi    = CPL_MATH_PI*el->visphi[ichan]/180.0;     // degree -> radian
	  phierr = CPL_MATH_PI*el->visphierr[ichan]/180.0;  // degree -> radian
	  amp    = fabs(el->visamp[ichan]/flux);
	  amperr = fabs(el->visamperr[ichan]/flux);
	  if (isnan(amperr)) amperr = amp/100; // HACK!
	  if ((info->cost_func != 3) && (amp == 0.0) && (amperr == 0.0)) continue;
	  cpl_msg_info(cpl_func, "VIS %d amp %f %f phi %f %f", info->nbvis, amp, amperr, phi, phierr);
	  vis = &(info->vis_list[info->nbvis]);
	  //if (phierr > CPL_MATH_PI/2.0) continue;
	  if (info->noise_factor != 0.0)
	    {
	      amp += gaussian_noise(0.0, amperr*info->noise_factor);
	      phi += gaussian_noise(0.0, phierr*info->noise_factor);
	    }
	  vis->lambda = cpl_vector_get(info->lambda_list, ichan);
	  vis->mphasor  = (cos(phi) + sin(phi)*I);
	  vis->mamp     = amp;
	  vis->mamperr  = amperr;
	  vis->mphi     = phi;
	  vis->mphierr  = phierr;
	  vis->mvis     = vis->mphasor*(amp + 0.0*I);
	  vis->mvar     = amperr*amperr + amp*amp*(1 - exp(-phierr*phierr));
	  vis->mvis2    = amp*amp;
	  vis->mvis2err = 2.0*amperr*amp; // TODO: pruefen mit Karl-Heinz (V2err = 2*Verr*V, was ist Verr?)
	  vis->u      = cpl_vector_get(info->lambda_scale, ichan)*el->ucoord;
	  vis->v      = cpl_vector_get(info->lambda_scale, ichan)*el->vcoord;
	  vis->weight = 0.0;
	  vis->bl     = blm;
	  blpx = sqrt(vis->u*vis->u + vis->v*vis->v);
	  info->minbaseline_px = fmin(blpx, info->minbaseline_px);
	  info->maxbaseline_px = fmax(blpx, info->maxbaseline_px);
	  // update the calculated baseline, FOV and pixel scale
	  scale = vis->lambda/blm/CPL_MATH_PI*180.0*60.0*60.0*1000.0;
	  info->maxfov   = fmax(scale, info->maxfov);
	  info->mindxrek = fmin(scale, info->mindxrek);
	  info->nbvis++;
	  nb++;
	}
    }
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "  %d visibilities added", nb);
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Calculates the VIS amplitude out of the VIS2 data.
   @param info    The image reconstruction context.
   @param oivis   This data structure contains one OI_VIS binary table.
   @param oivis2  This data structure contains one OI_VIS2 binary table.
   @returns A cpl_error_code.

   For each VIS table entry the associated VIS2 entries are searched
   and used to calculate the VIS amplitude and error. These values are
   stored in the VIS table entry and later used from the mat_add_vis()
   function.
*/
static cpl_error_code mat_calculate_visamp(mat_cal_imarec_info *info, mat_oivis *oivis, mat_oivis2 *oivis2)
{
  int           ivis, ivis2, ichan;

  if ((info == NULL) || (oivis == NULL) || (oivis2 == NULL))
    {
      return CPL_ERROR_NULL_INPUT;
    }
  // we sort the OI_VIS2 array with increasing u/v coordinate (1. u-, 2. v-coordinate)
  if (info->info_flags & INFO_INPUT_VIS)
    {
      cpl_msg_info(cpl_func, "  VIS2: %dx%d, VIS: %dx%d",
		   oivis2->nbvis2, oivis2->nbchannel,
		   oivis->nbvis, oivis->nbchannel);
      for (ivis2 = 0; ivis2 < oivis2->nbvis2; ivis2++)
	{
	  mat_vis2elem *el = oivis2->list_vis2[ivis2];
	  cpl_msg_info(cpl_func, "  VIS2[%d] uv=(%g, %g)", ivis2, el->ucoord, el->vcoord);
	}
    }

  for (ivis = 0; ivis < oivis->nbvis; ivis++)
    {
      mat_viselem   *vis = oivis->list_vis[ivis];
      mat_vis2elem  *v2  = NULL;
      // search and (hopefully find) the VIS2 elements for the current VIS element
      v2 = mat_find_vis2elem(info, oivis2, vis->dateobsmjd, vis->ucoord, vis->vcoord);
      if (v2 != NULL)
	{
	  for (ichan = 0; ichan < oivis->nbchannel; ichan++)
	    {
	      double v2u  = fabs(v2->vis2[ichan]);
	      double v2ue = fabs(v2->vis2err[ichan]);
	      double namp = sqrt(v2u);
	      double nerr;
	      if (v2->bandflag[ichan] == 1)
		{
		  vis->bandflag[ichan] = 1;
		  continue;
		}
	      if ((v2ue <= 0.0) || (namp == 0.0))
		{
		  vis->bandflag[ichan] = 1;
		  continue;
		}
	      nerr = 0.5*namp*v2ue/v2u;
	      if (info->info_flags & INFO_INPUT_VIS)
		{
		  cpl_msg_info(cpl_func,
			       "  VIS: uv=(%g, %g), old=%g (%g), new=%g (%g)",
			       vis->ucoord, vis->vcoord,
			       vis->visamp[ichan], vis->visamperr[ichan],
			       namp, nerr);
		}
	      vis->visamp[ichan]    = namp;
	      vis->visamperr[ichan] = nerr;
	    }
	}
      else
	{ // we did not find the VIS2 entry => warning and ignore this VIS entry
	  cpl_msg_warning(cpl_func,
			  "  can not find VIS2 data for VIS with (%g, %g)=%d",
			  vis->ucoord, vis->vcoord, (v2 != NULL));
	  for (ichan = 0; ichan < oivis->nbchannel; ichan++)
	    {
	      vis->bandflag[ichan] = 1;
	    }
	}
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Loads all interferometric data from the set of files.
   @param info    The image reconstruction context.
   @returns -1 n case of an error (no interferometric data in the SOF).
*/
static int mat_load_interferometric_data(mat_cal_imarec_info *info)
{
  cpl_frameset_iterator *it;
  cpl_frame             *frame;
  mat_oifits            *oifits;
  int                    i, j;

  // load all OI-FITS files (squared visibilities and bispectra)
  it = cpl_frameset_iterator_new(info->frameset);
  frame = cpl_frameset_iterator_get(it);
  while (frame != NULL)
    {
      // we use only calibrated target interferometric data
      if (strcmp(cpl_frame_get_tag(frame), MAT_DO_TARGET_CAL_INT) == 0)
	{
	  cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
	  if (info->info_flags & INFO_PARAMETER)
	    {
	      cpl_msg_info(cpl_func, "loading and processing of %s", cpl_frame_get_filename(frame));
	    }
	  oifits = mat_oifits_load(frame);
	  if (oifits != NULL)
	    {
	      // copy the target name (only lower case letters and digits!) into info.tname
	      i = 0;
	      j = 0;
	      while ((oifits->oitarget->targetname[i] != '\0') && (j < 255))
		{
		  if (isalnum(oifits->oitarget->targetname[i]))
		    {
		      info->tname[j++] = tolower(oifits->oitarget->targetname[i]);
		    }
		  if (oifits->oitarget->targetname[i] == '-')
		    {
		      info->tname[j++] = '-';
		    }
		  if (oifits->oitarget->targetname[i] == '.')
		    {
		      info->tname[j++] = '.';
		    }
		  if (oifits->oitarget->targetname[i] == '_')
		    {
		      info->tname[j++] = '_';
		    }
		  i++;
		}
	      info->tname[j] = '\0';
	      // iterate over all OI-FITS tupels
	      for (i = 0; i < oifits->nbtupel; i++)
		{
		  mat_oifits_select_tupel(oifits, i);
		  if (info->info_flags & INFO_PARAMETER)
		    {
		      cpl_msg_info(cpl_func, "  processing data for ARRNAME=%s, INSNAME=%s",
				   oifits->tupel_list[oifits->sel_tupel]->arrname,
				   oifits->tupel_list[oifits->sel_tupel]->insname);
		    }
		  // calculate the baseline to pixel scale for the selected wavelengths
		  mat_calc_baseline_scale(info, oifits->oiwave);
		  /* we have to satisfy the following dependencies from the cost function:
		            || CF == 1   || CF == 2   || CF == 3
		     table  || BIS | VIS || BIS | VIS || BIS | VIS
		     -------++-----+-----++-----+-----++-----+-----
		      T3    ||  X  |  -  ||  X  |  -  ||  X  |  -
		     -------++-----+-----++-----+-----++-----+-----
		      VIS2  ||  X  |  -  ||  X  |  -  ||  X  |  X
		     -------++-----+-----++-----+-----++-----+-----
		      VIS   ||  -  |  X  ||  -  |  X  ||  -  |  X
		     -------++-----+-----++-----+-----++-----+-----
		     In function mat_get_parameters() info->algo_mode and info->cost_func
		     is used to derive the info->use_flags USE_VIS_TABLE, USE_T3_TABLE and USE_VIS_TABLE.
		  */
		  if (info->use_flags & USE_VIS_TABLE)
		    { // add the filtered complex visibilities to the internal data
		      if (info->calc_visamp)
			{ // calculate the VIS amplitude and error from the OI-FITS binary table OI_VIS2
			  mat_calculate_visamp(info, oifits->oivis, oifits->oivis2);
			}
		      mat_add_vis(info, oifits->oivis);
		    }
		  if (info->use_flags & USE_T3_TABLE)
		    { // add the filtered bispectra to the internal data
		      if (info->calc_t3amp)
			{ // calculate the T3 amplitude and error from the OI-FITS binary table OI_VIS2
			  mat_calculate_t3amp(info, oifits->oit3, oifits->oivis2);
			}
		      mat_add_t3(info, oifits->oit3);
		    }
		  if (info->use_flags & USE_VIS2_TABLE)
		    { // add the filtered squared visibilities to the internal data
		      mat_add_vis2(info, oifits->oivis2);
		    }
		}
	      mat_oifits_delete(oifits);
	    }
	}
      cpl_frameset_iterator_advance(it, 1);
      frame = cpl_frameset_iterator_get(it);
    }
  cpl_frameset_iterator_delete(it);
  if (info->algo_mode & MODE_COMPLEX_VIS)
    {
      if (info->nbvis == 0)
	{
	  cpl_msg_error(cpl_func, "NO complex visibilities loaded, exiting");
	  return -1;
	}
      if (info->calc_visf0)
	{ // add an artificial complex visibility for the 0-th frequency
	  mat_vis *nvis = NULL;
	  mat_vis *fvis = &(info->vis_list[0]);
	  double   fbl  = fvis->u*fvis->u + fvis->v*fvis->v;
	  for (i = 1; i < info->nbvis; i++)
	    {
	      mat_vis *vis = &(info->vis_list[i]);
	      double   bl  = vis->u*vis->u + vis->v*vis->v;
	      if (bl < fbl)
		{
		  fvis = vis;
		  fbl = bl;
		}
	    }
	  nvis = &(info->vis_list[info->nbvis]); // we have already allocated enough space in function mat_add_vis
	  nvis->mvis     = 1.0 + 0.0*I;
	  nvis->mphasor  = 1.0 + 0.0*I;
	  nvis->mamp     = 1.0;
	  nvis->mvar     = 0.25*fvis->mvar;
	  nvis->mamperr  = 0.5*fvis->mamperr;
	  nvis->mphierr  = 0.5*fvis->mphierr;
	  nvis->mvis2    = 1.0;
	  nvis->mvis2err = 2.0*nvis->mamperr*nvis->mamp;
	  nvis->lambda   = fvis->lambda;
	  nvis->u        = 0.0;
	  nvis->v        = 0.0;
	  nvis->weight   = 0.0;
	  nvis->bl       = 0.0;
	  info->nbvis++;
	}
    }
  if (info->algo_mode & MODE_BISPECTRUM)
    {
      if (info->nbbis == 0)
	{
	  cpl_msg_error(cpl_func, "NO bispectrum loaded, exiting");
	  return -1;
	}
      if ((info->nbvis2 > 0) && info->calc_vis2f0)
	{ // add an artificial squared visibility for the 0-th frequency
	  mat_vis2 *nvis2 = NULL;
	  mat_vis2 *fvis2 = &(info->vis2_list[0]);
	  double    fbl   = fvis2->u*fvis2->u + fvis2->v*fvis2->v;
	  for (i = 1; i < info->nbvis2; i++)
	    {
	      mat_vis2 *vis2 = &(info->vis2_list[i]);
	      double    bl   = vis2->u*vis2->u + vis2->v*vis2->v;
	      if (bl < fbl)
		{
		  fvis2 = vis2;
		  fbl = bl;
		}
	    }
	  nvis2 = &(info->vis2_list[info->nbvis2]); // we have already allocated enough space in function mat_add_vis2
	  nvis2->mvis2  = 1.0;
	  nvis2->mvar   = 0.25*fvis2->mvar;
	  nvis2->lambda = fvis2->lambda;
	  nvis2->u      = 0.0;
	  nvis2->v      = 0.0;
	  nvis2->weight = 0.0;
	  nvis2->bl     = 0.0;
	  info->nbvis2++;
	}
    }
  return 0;
}

static cpl_error_code mat_calc_uv_coverage(mat_cal_imarec_info *info)
{
  int ivis2, ibis, ivis;

  info->uv_image = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
  if (info->uv_image == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for uv image");
      return CPL_ERROR_NULL_INPUT;
    }
  mat_image_fill(info->uv_image, 0.0);
  for (ivis2 = 0; ivis2 < info->nbvis2; ivis2++)
    {
      mat_vis2 *ela = &(info->vis2_list[ivis2]);
      mat_image_add_double_interpolated(info->uv_image,  ela->u,  ela->v, 1.0);
      mat_image_add_double_interpolated(info->uv_image, -ela->u, -ela->v, 1.0);
    }
  for (ibis = 0; ibis < info->nbbis; ibis++)
    {
      mat_bis *ela = &(info->bis_list[ibis]);
      mat_image_add_double_interpolated(info->uv_image,  ela->u1,  ela->v1, 1.0);
      mat_image_add_double_interpolated(info->uv_image, -ela->u1, -ela->v1, 1.0);
      mat_image_add_double_interpolated(info->uv_image,  ela->u2,  ela->v2, 1.0);
      mat_image_add_double_interpolated(info->uv_image, -ela->u2, -ela->v2, 1.0);
      mat_image_add_double_interpolated(info->uv_image,  ela->u3,  ela->v3, 1.0);
      mat_image_add_double_interpolated(info->uv_image, -ela->u3, -ela->v3, 1.0);
    }
  for (ivis = 0; ivis < info->nbvis; ivis++)
    {
      mat_vis *ela = &(info->vis_list[ivis]);
      mat_image_add_double_interpolated(info->uv_image,  ela->u,  ela->v, 1.0);
      mat_image_add_double_interpolated(info->uv_image, -ela->u, -ela->v, 1.0);
    }
  return CPL_ERROR_NONE;
}

static void mat_fill_uv_arrays(double *au1, double *av1,
			       double *au2, double *av2,
			       double u1, double v1,
			       double u2, double v2,
			       int ef)
{
  // build the uv arrays for a candidate
  //     u1,     v1,     u2,     v2
  //     u2,     v2,     u1,     v1
  //     u1,     v1, -u1-u2, -v1-v2
  //     u2,     v2, -u1-u2, -v1-v2
  // -u1-u2, -v1-v2,     u1,     v1
  // -u1-u2, -v1-v2,     u2,     v2
  au1[0] =       u1; av1[0] =       v1; au2[0] =       u2; av2[0] =       v2;
  au1[1] =       u2; av1[1] =       v2; au2[1] =       u1; av2[1] =       v1;
  au1[2] =       u1; av1[2] =       v1; au2[2] = -u1 - u2; av2[2] = -v1 - v2;
  au1[3] =       u2; av1[3] =       v2; au2[3] = -u1 - u2; av2[3] = -v1 - v2;
  au1[4] = -u1 - u2; av1[4] = -v1 - v2; au2[4] =       u1; av2[4] =       v1;
  au1[5] = -u1 - u2; av1[5] = -v1 - v2; au2[5] =       u2; av2[5] =       v2;
  if (ef)
    {
      au1[6]  = -au1[0]; av1[6]  = -av1[0]; au2[6]  = -au2[0]; av2[6]  = -av2[0];
      au1[7]  = -au1[1]; av1[7]  = -av1[1]; au2[7]  = -au2[1]; av2[7]  = -av2[1];
      au1[8]  = -au1[2]; av1[8]  = -av1[2]; au2[8]  = -au2[2]; av2[8]  = -av2[2];
      au1[9]  = -au1[3]; av1[9]  = -av1[3]; au2[9]  = -au2[3]; av2[9]  = -av2[3];
      au1[10] = -au1[4]; av1[10] = -av1[4]; au2[10] = -au2[4]; av2[10] = -av2[4];
      au1[11] = -au1[5]; av1[11] = -av1[5]; au2[11] = -au2[5]; av2[11] = -av2[5];
    }
}

static double mat_test_uv_distance(double *au1, double *av1,
				   double *au2, double *av2,
				   double *bu1, double *bv1,
				   double *bu2, double *bv2,
				   int na, int nb, double r2)
{
  int    k, l;
  double inside = 0.0;

  for (k = 0; k < na; k++)
    {
      for (l = 0; l < nb; l++)
	{
	  double d2 = 0.0;
	  d2 += (au1[k] - bu1[l])*(au1[k] - bu1[l]);
	  d2 += (av1[k] - bv1[l])*(av1[k] - bv1[l]);
	  d2 += (au2[k] - bu2[l])*(au2[k] - bu2[l]);
	  d2 += (av2[k] - bv2[l])*(av2[k] - bv2[l]);
	  if (d2 <= r2) inside += 1.0;
	}
    }
  return (inside/((double)na));
}

/**
   @brief Calculates the weight for all input data.
   @param info Contains the image reconstruction context.

   This method calculates for each input data point (squared visibility and closutre phase)
   a weight which specifies its influence for the chi squared and gradient calculation.
   This weight is calculated using the number of data points inside a specific radius of
   the current data point. This radius is calculated by:

   r = baseline_max/sqrt(sqrt(2/3*(nbvis2 + 2*nbbis)))

   baseline_max : longest baseline mapped to the pixel coordinates

   nbvis2       : number of squared visibilities

   nbbis        : number os closure phases

   The weight is calculated by:

   weight = pow(1.0/count, weight_power)

   count        : number of data points inside r

   weight_power : plugin parameter --weight_power, default 1.0
*/
static cpl_error_code mat_calc_weights_bis(mat_cal_imarec_info *info)
{ // gbis
  double  au1[12], av1[12], au2[12], av2[12];
  double  bu1[12], bv1[12], bu2[12], bv2[12];
  double  r2;
  int     i, j;
  double  weightsum = 0.0;
  double  wuvbissum = 0.0;
  double  wuvphsum = 0.0;
  double  wuvmodsum = 0.0;

  //r2 = 4.0*info->maxbaseline_px*info->maxbaseline_px/(double)(2*info->nbvis2);
  //r2 = info->maxbaseline_px*info->maxbaseline_px/sqrt(4.0/3.0*(double)(info->nbbis));
  if (info->cost_func != 3)
    {
      r2 = info->maxbaseline_px*info->maxbaseline_px/sqrt(2.0/3.0*(double)(info->nbvis2 + 2*info->nbbis)); // <- auch fuer Kostenfunktion 3?
    }
  else
    {
      r2 = info->maxbaseline_px*info->maxbaseline_px/sqrt(2.0/3.0*(double)(2*info->nbbis)); // <- auch fuer Kostenfunktion 3?
    }
  if (info->info_flags & INFO_PREPARE)
    {
      cpl_msg_info(cpl_func, "weight calculation: r2 = %f", r2);
    }
  // calculate the weight for each bispectrum
  for (i = 0; i < info->nbbis; i++)
    {
      mat_bis *ela = &(info->bis_list[i]);
      double   count  = 0.0;
      mat_fill_uv_arrays(au1, av1, au2, av2, ela->u1, ela->v1, ela->u2, ela->v2, 1);
      if (info->cost_func != 3)
	{
	  for (j = 0; j < info->nbvis2; j++)
	    {
	      mat_vis2 *elb = &(info->vis2_list[j]);
	      mat_fill_uv_arrays(bu1, bv1, bu2, bv2, elb->u, elb->v, 0.0, 0.0, 0);
	      count += mat_test_uv_distance(au1, av1, au2, av2, bu1, bv1, bu2, bv2, 12, 6, r2);
	    }
	}
      for (j = 0; j < info->nbbis; j++)
	{
	  mat_bis *elb = &(info->bis_list[j]);
	  mat_fill_uv_arrays(bu1, bv1, bu2, bv2, elb->u1, elb->v1, elb->u2, elb->v2, 1);
	  count += mat_test_uv_distance(au1, av1, au2, av2, bu1, bv1, bu2, bv2, 12, 12, r2);
	}
      ela->weight  = pow(1.0/count, info->weight_power);
      ela->wuvbis  = ela->weight/(fabs(ela->mvar) + 1e-30);
      ela->wuvcp   = ela->weight/(ela->mcperr*ela->mcperr + 1e-30);
      ela->wuvph  = ela->weight/(1.0 - exp(-ela->mcperr*ela->mcperr) + 1e-30);
      if (info->cost_func != 3)
	{
	  ela->wuvmod  = ela->weight/(ela->mamperr*ela->mamperr + 1e-30);
	}
      else
	{ // cost function 3 uses the bispectrum phasor only, artificial amplitide 1.0 -> no wuvmod available
	  ela->wuvmod  = 0.0;
	}
      weightsum += ela->weight;
      wuvbissum += ela->wuvbis;
      wuvphsum  += ela->wuvph;
      wuvmodsum += ela->wuvmod;
      if (info->info_flags & (INFO_INPUT_T3 | INFO_PREPARE))
	{
	  cpl_msg_info(cpl_func, "bis[%d]: bis = (%f, %f), var = %g, cperr = %f, uv1 = (%f, %f), uv2 = (%f, %f), weight = %f, count = %f, wuv = %f, %f, %f",
		       i,
		       creal(ela->mbis), cimag(ela->mbis), ela->mvar, ela->mcperr,
		       ela->u1, ela->v1, ela->u2, ela->v2,
		       ela->weight, count, ela->wuvbis, ela->wuvph, ela->wuvmod);
	}
    }
  if (info->info_flags & INFO_PREPARE)
    {
      cpl_msg_info(cpl_func, "average bispectrum weights: weight %f wuvbis %f wuvph %f wuvmod %f count %d",
		   weightsum/((double)info->nbbis),
		   wuvbissum/((double)info->nbbis),
		   wuvphsum/((double)info->nbbis),
		   wuvmodsum/((double)info->nbbis),
		   info->nbbis);
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_calc_weights_vis2(mat_cal_imarec_info *info)
{ // gpow
  double  au1[12], av1[12], au2[12], av2[12];
  double  bu1[12], bv1[12], bu2[12], bv2[12];
  double  r2;
  int     i, j;
  double  weightsum  = 0.0;
  double  wuvvis2sum = 0.0;

  //r2 = 4.0*info->maxbaseline_px*info->maxbaseline_px/(double)(2*info->nbvis2);
  //r2 = info->maxbaseline_px*info->maxbaseline_px/sqrt(4.0/3.0*(double)(info->nbbis));
  if (info->cost_func != 3)
    {
      r2 = info->maxbaseline_px*info->maxbaseline_px/sqrt(2.0/3.0*(double)(info->nbvis2 + 2*info->nbbis));
    }
  else
    {
      r2 = info->maxbaseline_px*info->maxbaseline_px/sqrt(2.0/3.0*(double)(info->nbvis2));
    }
  if (info->info_flags & INFO_PREPARE)
    {
      cpl_msg_info(cpl_func, "weight calculation: r2 = %f", r2);
    }
  // calculate the weight for each squared visibility
  for (i = 0; i < info->nbvis2; i++)
    {
      mat_vis2 *ela = &(info->vis2_list[i]);
      double    count = 0.0;
      mat_fill_uv_arrays(au1, av1, au2, av2, ela->u, ela->v, 0.0, 0.0, 0);
      for (j = 0; j < info->nbvis2; j++)
	{
	  mat_vis2 *elb = &(info->vis2_list[j]);
	  mat_fill_uv_arrays(bu1, bv1, bu2, bv2, elb->u, elb->v, 0.0, 0.0, 0);
	  count += mat_test_uv_distance(au1, av1, au2, av2, bu1, bv1, bu2, bv2, 6, 6, r2);
	}
      if (info->cost_func != 3)
	{
	  for (j = 0; j < info->nbbis; j++)
	    {
	      mat_bis *elb = &(info->bis_list[j]);
	      mat_fill_uv_arrays(bu1, bv1, bu2, bv2, elb->u1, elb->v1, elb->u2, elb->v2, 1);
	      count += mat_test_uv_distance(au1, av1, au2, av2, bu1, bv1, bu2, bv2, 6, 12, r2);
	    }
	}
      ela->weight  = pow(1.0/count, info->weight_power);
      ela->wuvvis2 = ela->weight/fmax(fabs(ela->mvar), 1e-30);
      weightsum  += ela->weight;
      wuvvis2sum += ela->wuvvis2;
      if (info->info_flags & (INFO_INPUT_VIS2 | INFO_PREPARE))
	{
	  cpl_msg_info(cpl_func, "vis2[%d]: vis2 = %f, var = %g, uv = (%f, %f), weight = %f, count = %f, wuv = %f",
		       i, ela->mvis2, ela->mvar, ela->u, ela->v, ela->weight, count, ela->wuvvis2);
	}
    }
  if (info->info_flags & INFO_PREPARE)
    {
      cpl_msg_info(cpl_func, "average squared visibility weights: weight %f wuvvis2 %f count %d",
		   weightsum/((double)info->nbvis2),
		   wuvvis2sum/((double)info->nbvis2),
		   info->nbvis2);
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_calc_weights_vis(mat_cal_imarec_info *info)
{ // gft
  double  au[2], av[2];
  double  bu[2], bv[2];
  double  r2;
  int     i, j, k, l;
  double  weightsum  = 0.0;
  double  wuvvissum  = 0.0;
  double  wuvphsum   = 0.0;
  double  wuvmodsum  = 0.0;
  double  wuvvis2sum = 0.0;
  double  wuvphisum  = 0.0;

  // radius0pow = 2. * maxbaseline / sqrt(float(2*anzahlvis))
  r2 = 4.0*info->maxbaseline_px*info->maxbaseline_px/(double)(2*info->nbvis);
  if (info->info_flags & INFO_PREPARE)
    {
      cpl_msg_info(cpl_func, "weight calculation: r2 = %f", r2);
    }
  // calculate the weight for each visibility
  for (i = 0; i < info->nbvis; i++)
    {
      mat_vis  *ela = &(info->vis_list[i]);
      double    count = 0.0;
      au[0] =  ela->u; av[0] =  ela->v;
      au[1] = -ela->u; av[1] = -ela->v;
      for (j = 0; j < info->nbvis; j++)
	{
	  double inside = 0.0;
	  double rdiv   = 1.0;
	  mat_vis  *elb = &(info->vis_list[j]);
	  bu[0] =  elb->u; bv[0] =  elb->v;
	  bu[1] = -elb->u; bv[1] = -elb->v;
	  if ((elb->u == 0.0) && (elb->v == 0.0)) rdiv = 2.0;
	  for (k = 0; k < 2; k++)
	    {
	      for (l = 0; l < 2; l++)
		{
		  double d2 = 0.0;
		  d2 += (au[k] - bu[l])*(au[k] - bu[l]);
		  d2 += (av[k] - bv[l])*(av[k] - bv[l]);
		  if (d2 <= r2) inside += 1.0;
		}
	    }
	  count += inside/(2.0*rdiv);
	}
      ela->weight  = pow(1.0/count, info->weight_power);
      ela->wuvvis  = ela->weight/fmax(fabs(ela->mvar), 1e-30);
      ela->wuvph  = ela->weight/(1.0 - exp(-ela->mphierr*ela->mphierr) + 1e-30);
      if (info->cost_func != 3)
	{
	  ela->wuvmod  = ela->weight/(ela->mamperr*ela->mamperr + 1e-30);
	}
      else
	{ // cost function 3 uses the fourier phasor only, artificial amplitide 1.0 -> no wuvmod available
	  ela->wuvmod  = 0.0;
	}
      ela->wuvvis2 = ela->weight/(ela->mvis2err*ela->mvis2err + 1e-30);
      ela->wuvphi  = ela->weight/(ela->mphierr*ela->mphierr + 1e-30);
      weightsum  += ela->weight;
      wuvvissum  += ela->wuvvis;
      wuvphsum   += ela->wuvph;
      wuvmodsum  += ela->wuvmod;
      wuvvis2sum += ela->wuvvis2;
      wuvphisum  += ela->wuvphi;
      if (info->info_flags & (INFO_INPUT_VIS | INFO_PREPARE))
	{
	  cpl_msg_info(cpl_func, "vis[%d]: vis = (%f %f), var = %g, uv = (%f, %f), weight = %f, count = %f, wuv = %f, %f, %f, %f, %f",
		       i, creal(ela->mvis), cimag(ela->mvis), ela->mvar, ela->u, ela->v, ela->weight, count, ela->wuvvis, ela->wuvph, ela->wuvmod, ela->wuvvis2, ela->wuvphi);
	}
    }
  if (info->info_flags & INFO_PREPARE)
    {
      cpl_msg_info(cpl_func, "average complex visibility weights: weight %f wuvvis %f wuvph %f wuvmod %f wuvvis2 %f wuvphi %f count %d",
		   weightsum/((double)info->nbvis),
		   wuvvissum/((double)info->nbvis),
		   wuvphsum/((double)info->nbvis),
		   wuvmodsum/((double)info->nbvis),
		   wuvvis2sum/((double)info->nbvis),
		   wuvphisum/((double)info->nbvis),
		   info->nbvis);
    }
  return CPL_ERROR_NONE;
}



/*--------------------------------------------------------------------*/
/* Function for making an educated guess for some parameters          */
/*--------------------------------------------------------------------*/
/**
   @brief Function giving the squared visibilities of a gaussian disc.
   @param x       Independent parameters (only one).
   @param a       Function parameters (only one, sigma)
   @param result  Function value
*/
static int mat_fit_gauss_f(const double x[], const double a[], double *result)
{
  double hx = x[0]*CPL_MATH_PI;
  *result = exp(-4.0*hx*hx*a[0]*a[0]);
  return 0;
}

/**
   @brief Calculates the partial derivation for a gaussian disc.
   @param x       Independent parameters (only one).
   @param a       Function parameters (only one, sigma)
   @param result  Partial derivations (only one partial derivation)
*/
static int mat_fit_gauss_dfda(const double x[], const double a[], double result[])
{
  double hx = x[0]*CPL_MATH_PI;
  result[0] = -8.0*hx*hx*a[0]*exp(-4.0*hx*hx*a[0]*a[0]);
  return 0;
}

/**
   @brief Function giving the squared visibilities of a fully darkened disc.
   @param x       Independent parameters (only one).
   @param a       Function parameters (only one, radius)
   @param result  Function value
*/
static int mat_fit_fdd_f(const double x[], const double a[], double *result)
{
  if (x[0] == 0.0)
    {
      *result = 1.0;
    }
  else
    {
      double p = 2.0*CPL_MATH_PI*x[0]*a[0];
      double w = sin(p)/p - cos(p);
      *result = 3.0*w/(p*p)*3.0*w/(p*p);
    }
  return 0;
}

/**
   @brief Calculates the partial derivation for a fully darkened disc.
   @param x       Independent parameters (only one).
   @param a       Function parameters (only one, radius)
   @param result  Partial derivations (only one partial derivation)
*/
static int mat_fit_fdd_dfda(const double x[], const double a[], double result[])
{
  double p = 2.0*CPL_MATH_PI*x[0]*a[0];
  double w = sin(p)/p - cos(p);
  //result[0] = 2.0*(3.0*w)*(3.0*CPL_MATH_PI*x[0]*(p*sin(p) - w) - 6.0*w)/(p*p*p*p*p);
  // p = 2 fda x
  // w = -Cos[2 fda x] + Sin[2 fda x]/(2 fda x)
  // -9 w^2/(4 fda^5 x^4) + 9 w (-w + 2 x Sin[p])/(8 fda^4 x^4)
  result[0] = 2.0*3.0/p/p*3.0/p/p*w/(2.0*a[0])*(p*sin(p) - 3.0*w);
  return 0;
}

/**
   @brief Function giving the squared visibilities of a uniform disc.
   @param x       Independent parameters (only one).
   @param a       Function parameters (only one, radius)
   @param result  Function value
*/
static int mat_fit_ud_f(const double x[], const double a[], double *result)
{
  if (x[0] == 0.0)
    {
      *result = 1.0;
    }
  else
    {
      double p = 2.0*CPL_MATH_PI*x[0]*a[0];
#ifdef WITH_BESSELJ
      *result = 4.0*(bessj(1, p)/p)*(bessj(1, p)/p);
#else
      *result = 4.0*(jn(1, p)/p)*(jn(1, p)/p); // not C99
#endif
    }
  return 0;
}

/**
   @brief Calculates the partial derivation for a uniform disc.
   @param x       Independent parameters (only one).
   @param a       Function parameters (only one, radius)
   @param result  Partial derivations (only one partial derivation)
*/
static int mat_fit_ud_dfda(const double x[], const double a[], double result[])
{
  double p  = 2.0*CPL_MATH_PI*a[0]*x[0];
  double f1 = CPL_MATH_PI*CPL_MATH_PI*a[0]*a[0]*a[0]*x[0]*x[0];
  double f2 = CPL_MATH_PI*a[0]*a[0]*x[0];
#ifdef WITH_BESSELJ
  result[0] = -2.0*bessj(1, p)*bessj(1, p)/f1 + 2.0*bessj(1, p)*(bessj(0, p) - bessj(2, p))/f2;
#else
  result[0] = -2.0*jn(1, p)*jn(1, p)/f1 + 2.0*jn(1, p)*(jn(0, p) - jn(2, p))/f2; // not C99
#endif
  return 0;
}

/**
   @brief Function giving the squared visibilities of a Lorentz disc.
   @param x       Independent parameters (only one).
   @param a       Function parameters (only one, a)
   @param result  Function value
*/
static int mat_fit_ld_f(const double x[], const double a[], double *result)
{
  *result = exp(-4.0*CPL_MATH_PI*x[0]/a[0]);
  return 0;
}

/**
   @brief Calculates the partial derivation for a Lorentz disc.
   @param x       Independent parameters (only one).
   @param a       Function parameters (only one, sigma)
   @param result  Partial derivations (only one partial derivation)
*/
static int mat_fit_ld_dfda(const double x[], const double a[], double result[])
{
  result[0] = 4.0*CPL_MATH_PI*x[0]/a[0]/a[0]*exp(-4.0*CPL_MATH_PI*x[0]/a[0]);
  return 0;
}

/**
   @brief Fit the squared visibilities to a gaussian disc, a uniform disc and a fully darkened disc.
   @param info  Contains the image reconstruction context.

   A nonlinear fit routine is used to derive the radius (FWHM) for a gaussian disc, a uniform disc and a fully darkened disc
   from the loaded squared visibilities. After a first fit, another fit is computed using only the squared visibilities
   including the first lobe of the fitted function. This should give more accurate radii. At the end the modes and parameter
   values for the start and the prior image are printed out.
*/
static void mat_fit_start_image_vis2(mat_cal_imarec_info *info)
{
  int            i, j;
  cpl_matrix    *x;
  cpl_vector    *y;
  cpl_vector    *ye;
  cpl_vector    *a;
  double         mse;
  double         red_chisq;
  int            mode;
  double         param;
  double         chi2;
  double         gd_f;
  double         ud_f;
  double         fdd_f;
  double         ld_f;
  int            gd_count = 0;
  int            ud_count = 0;
  int            fdd_count = 0;
  int            ld_count = 0;
  int            flag = 1;
  int            nbvis2;
  double         fwhm;
  cpl_error_code ec;

  if (info->calc_vis2f0)
    {
      nbvis2 = info->nbvis2 - 1; // ignore the zero frequency
    }
  else
    {
      nbvis2 = info->nbvis2;
    }
  // The first fit uses all visibility data (no spatial frequency filter active)
  // allocate the temporary memory for the fit
  x  = cpl_matrix_new(nbvis2, 1);
  y  = cpl_vector_new(nbvis2);
  ye = cpl_vector_new(nbvis2);
  a  = cpl_vector_new(1);
  // fill the input for all fit routines
  for (i = 0; i < nbvis2; i++)
    {
      mat_vis2 *el = &(info->vis2_list[i]);
      double    r = sqrt(el->u*el->u + el->v*el->v);
      double    f = r/info->FOV;
      double    v = fabs(el->mvis2);
      double    e = sqrt(el->mvar);
      cpl_matrix_set(x, i, 0, f);
      cpl_vector_set(y, i, v);
      cpl_vector_set(ye, i, e);
      //cpl_msg_info(cpl_func, "V2[%d]: %.6f %.6f %.6f", i, f, v, e);
    }
  // do the fit
  cpl_vector_set(a, 0, info->fit_fwhm*FWHM2SIGMA);
  ec = cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
		    mat_fit_gauss_f, mat_fit_gauss_dfda,
		    CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
		    &mse, &red_chisq, NULL);
  fwhm = cpl_vector_get(a, 0)/FWHM2SIGMA;
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "gaussian disc, first try => sigma = %g fwhm = %g ec = %d", cpl_vector_get(a, 0), fwhm, (int)ec);
    }
  gd_f = 0.4/cpl_vector_get(a, 0); // spatial frequency limit for the fit of a gaussian disc
  //cpl_vector_set(a, 0, 1.8*fwhm);
  //cpl_vector_set(a, 0, 1.0);
  ec = cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
		    mat_fit_fdd_f, mat_fit_fdd_dfda,
		    CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
		    &mse, &red_chisq, NULL);
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "fully darkened disc, first try => radius = %g ec = %d", cpl_vector_get(a, 0), (int)ec);
    }
  if (ec == CPL_ERROR_NONE)
    {
      fdd_f = 1.23/cpl_vector_get(a, 0); // spatial frequency limit for the fit of a fully darkened disc
    }
  else
    {
      fdd_f = gd_f;
    }
  //cpl_vector_set(a, 0, 1.6*fwhm);
  //cpl_vector_set(a, 0, 1.0);
  ec = cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
		    mat_fit_ud_f, mat_fit_ud_dfda,
		    CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
		    &mse, &red_chisq, NULL);
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "uniform disc, first try => radius = %g ec = %d", cpl_vector_get(a, 0), (int)ec);
    }
  if (ec == CPL_ERROR_NONE)
    {
      ud_f = 1.11/cpl_vector_get(a, 0); // spatial frequency limit for the fit of a uniform disc
    }
  else
    {
      ud_f = gd_f;
    }
  ld_f = 0.0;
  // Now we try to get a better fit by using a filter for spatial frequencies
  // 1. calculate the number of remaining data points for each of the fitted functions
  for (i = 0; i < nbvis2; i++)
    {
      mat_vis2 *el = &(info->vis2_list[i]);
      double    r = sqrt(el->u*el->u + el->v*el->v);
      double    f = r/info->FOV;
      if (f <= gd_f) gd_count++;
      if (f <= fdd_f) fdd_count++;
      if (f <= ud_f) ud_count++;
      if (f > ld_f) ld_f = f;
      ld_count++;
    }
  // 2. Try a better fit for the gaussian disc
  flag = flag && (cpl_matrix_set_size(x, gd_count, 1) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(y, gd_count) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(ye, gd_count) == CPL_ERROR_NONE);
  j = 0;
  for (i = 0; i < nbvis2; i++)
    {
      mat_vis2 *el = &(info->vis2_list[i]);
      double    r = sqrt(el->u*el->u + el->v*el->v);
      double    f = r/info->FOV;
      double    v = fabs(el->mvis2);
      double    e = sqrt(el->mvar);
      if (f <= gd_f)
	{
	  cpl_matrix_set(x, j, 0, f);
	  cpl_vector_set(y, j, v);
	  cpl_vector_set(ye, j, e);
	  //cpl_msg_info(cpl_func, "V2[%d]: %.6f %.6f %.6f", j, f, v, e);
	  j++;
	}
    }
  cpl_vector_set(a, 0, info->fit_fwhm*FWHM2SIGMA);
  cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
	       mat_fit_gauss_f, mat_fit_gauss_dfda,
	       CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
	       &mse, &red_chisq, NULL);
  info->fit_gd_fwhm = cpl_vector_get(a, 0)/FWHM2SIGMA;
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "fit gaussian disk (mode=%d):           FWHM = %.3f mas, mse = %5.3e, red_chisq = %7.3f, spatfreq = 0 .. %7.3f 1/mas, nbvis2 = %d",
		   MAT_MODE_GAUSSIAN_DISC, info->fit_gd_fwhm, mse, red_chisq, gd_f, gd_count);
    }
  mode = MAT_MODE_GAUSSIAN_DISC;
  param = info->fit_gd_fwhm;
  chi2 = red_chisq;
  // 3. Try a better fit for the uniform darkened disc
  flag = flag && (cpl_matrix_set_size(x, ud_count, 1) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(y, ud_count) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(ye, ud_count) == CPL_ERROR_NONE);
  j = 0;
  for (i = 0; i < nbvis2; i++)
    {
      mat_vis2 *el = &(info->vis2_list[i]);
      double    r = sqrt(el->u*el->u + el->v*el->v);
      double    f = r/info->FOV;
      double    v = fabs(el->mvis2);
      double    e = sqrt(el->mvar);
      if (f <= ud_f)
	{
	  cpl_matrix_set(x, j, 0, f);
	  cpl_vector_set(y, j, v);
	  cpl_vector_set(ye, j, e);
	  //cpl_msg_info(cpl_func, "V2[%d]: %.6f %.6f %.6f", j, f, v, e);
	  j++;
	}
    }
  //cpl_vector_set(a, 0, 1.6*info->fit_gd_fwhm);
  cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
	       mat_fit_ud_f, mat_fit_ud_dfda,
	       CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
	       &mse, &red_chisq, NULL);
  info->fit_ud_diameter = 2.0*cpl_vector_get(a, 0);
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "fit uniform disc (mode=%d):        diameter = %.3f mas, mse = %5.3e, red_chisq = %7.3f, spatfreq = 0 .. %7.3f 1/mas, nbvis2 = %d",
		   MAT_MODE_UNIFORM_DISC, info->fit_ud_diameter, mse, red_chisq, ud_f, ud_count);
    }
  if (red_chisq < chi2)
    {
      mode = MAT_MODE_UNIFORM_DISC;
      param = info->fit_ud_diameter;
      chi2 = red_chisq;
    }
  // 4. Try a better fit for the fully darkened disc
  flag = flag && (cpl_matrix_set_size(x, fdd_count, 1) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(y, fdd_count) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(ye, fdd_count) == CPL_ERROR_NONE);
  j = 0;
  for (i = 0; i < nbvis2; i++)
    {
      mat_vis2 *el = &(info->vis2_list[i]);
      double    r = sqrt(el->u*el->u + el->v*el->v);
      double    f = r/info->FOV;
      double    v = fabs(el->mvis2);
      double    e = sqrt(el->mvar);
      if (f <= fdd_f)
	{
	  cpl_matrix_set(x, j, 0, f);
	  cpl_vector_set(y, j, v);
	  cpl_vector_set(ye, j, e);
	  //cpl_msg_info(cpl_func, "V2[%d]: %.6f %.6f %.6f", j, f, v, e);
	  j++;
	}
    }
  //cpl_vector_set(a, 0, 1.8*info->fit_gd_fwhm);
  cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
	       mat_fit_fdd_f, mat_fit_fdd_dfda,
	       CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
	       &mse, &red_chisq, NULL);
  info->fit_fdd_diameter = 2.0*cpl_vector_get(a, 0);
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "fit fully darkened disc (mode=%d): diameter = %.3f mas, mse = %5.3e, red_chisq = %7.3f, spatfreq = 0 .. %7.3f 1/mas, nbvis2 = %d",
		   MAT_MODE_FULLY_DARKENED_DISC, info->fit_fdd_diameter, mse, red_chisq, fdd_f, fdd_count);
    }
  if (red_chisq < chi2)
    {
      mode = MAT_MODE_FULLY_DARKENED_DISC;
      param = info->fit_fdd_diameter;
      chi2 = red_chisq;
    }
  // 5. Try a better fit for the Lorentz disc
  flag = flag && (cpl_matrix_set_size(x, ld_count, 1) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(y, ld_count) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(ye, ld_count) == CPL_ERROR_NONE);
  j = 0;
  for (i = 0; i < nbvis2; i++)
    {
      mat_vis2 *el = &(info->vis2_list[i]);
      double    r = sqrt(el->u*el->u + el->v*el->v);
      double    f = r/info->FOV;
      double    v = fabs(el->mvis2);
      double    e = sqrt(el->mvar);
      if (f <= ld_f)
	{
	  cpl_matrix_set(x, j, 0, f);
	  cpl_vector_set(y, j, v);
	  cpl_vector_set(ye, j, e);
	  //cpl_msg_info(cpl_func, "V2[%d]: %.6f %.6f %.6f", j, f, v, e);
	  j++;
	}
    }
  //cpl_vector_set(a, 0, info->fit_gd_fwhm);
  cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
	       mat_fit_ld_f, mat_fit_ld_dfda,
	       CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
	       &mse, &red_chisq, NULL);
  info->fit_ld_fwhm = FWHM2LDA/cpl_vector_get(a, 0);
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "fit Lorentz disk (mode=%d):            FWHM = %.3f mas, mse = %5.3e, red_chisq = %7.3f, spatfreq = 0 .. %7.3f 1/mas, nbvis2 = %d",
		   MAT_MODE_LORENTZ_DISC, info->fit_ld_fwhm, mse, red_chisq, ld_f, ld_count);
    }
  if (red_chisq < chi2)
    {
      mode = MAT_MODE_LORENTZ_DISC;
      param = info->fit_ld_fwhm;
    }

  info->start_mode_guess  = mode;
  info->start_param_guess = param;
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "suggested parameters: (ok=%d)", flag);
      cpl_msg_info(cpl_func, "--start_select=4 --start_mode=%d --start_param=%.3f", mode, param);
      cpl_msg_info(cpl_func, "--prior_select=4 --prior_mode=%d --prior_param=%.3f", mode, param);
    }
  // free all temporary memory
  cpl_matrix_delete(x);
  cpl_vector_delete(y);
  cpl_vector_delete(ye);
  cpl_vector_delete(a);
}

/**
   @brief Fit the complex visibilities to a gaussian disc, a uniform disc and a fully darkened disc.
   @param info  Contains the image reconstruction context.

   A nonlinear fit routine is used to derive the radius (FWHM) for a gaussian disc, a uniform disc and a fully darkened disc
   from the loaded complex visibilities. After a first fit, another fit is computed using only the complex visibilities
   including the first lobe of the fitted function. This should give more accurate radii. At the end the modes and parameter
   values for the start and the prior image are printed out.
*/
static void mat_fit_start_image_vis(mat_cal_imarec_info *info)
{
  int            i, j;
  cpl_matrix    *x;
  cpl_vector    *y;
  cpl_vector    *ye;
  cpl_vector    *a;
  double         mse;
  double         red_chisq;
  int            mode;
  double         param;
  double         chi2;
  double         gd_f;
  double         ud_f;
  double         fdd_f;
  double         ld_f;
  int            gd_count = 0;
  int            ud_count = 0;
  int            fdd_count = 0;
  int            ld_count = 0;
  int            flag = 1;
  int            nbvis;
  cpl_error_code ec;

  if (info->calc_visf0)
    {
      nbvis = info->nbvis - 1; // ignore the zero frequency
    }
  else
    {
      nbvis = info->nbvis;
    }
  // The first fit uses all visibility data (no spatial frequency filter active)
  // allocate the temporary memory for the fit
  x  = cpl_matrix_new(nbvis, 1);
  y  = cpl_vector_new(nbvis);
  ye = cpl_vector_new(nbvis);
  a  = cpl_vector_new(1);
  // fill the input for all fit routines
  for (i = 0; i < nbvis; i++)
    {
      mat_vis  *el = &(info->vis_list[i]);
      double    r = sqrt(el->u*el->u + el->v*el->v);
      double    f = r/info->FOV;
      double    v = MAT_CHYPOT2(el->mvis);
      double    e = 2.0*el->mamp*el->mamperr;
      cpl_matrix_set(x, i, 0, f);
      cpl_vector_set(y, i, v);
      cpl_vector_set(ye, i, e);
      //cpl_msg_info(cpl_func, "V2[%d]: %.6f %.6f %.6f", i, f, v, e);
    }
  // do the fit
  cpl_vector_set(a, 0, info->fit_fwhm*FWHM2SIGMA);
  ec = cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
		    mat_fit_gauss_f, mat_fit_gauss_dfda,
		    CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
		    &mse, &red_chisq, NULL);
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "gaussian disc, first try => sigma = %g ec = %d", cpl_vector_get(a, 0), (int)ec);
    }
  gd_f = 0.4/cpl_vector_get(a, 0); // spatial frequency limit for the fit of a gaussian disc
  //cpl_vector_set(a, 0, 1.0);
  cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
	       mat_fit_fdd_f, mat_fit_fdd_dfda,
	       CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
	       &mse, &red_chisq, NULL);
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "fully darkened disc, first try => radius = %g ec = %d", cpl_vector_get(a, 0), (int)ec);
    }
  fdd_f = 1.23/cpl_vector_get(a, 0); // spatial frequency limit for the fit of a fully darkened disc
  //cpl_vector_set(a, 0, 1.0);
  cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
	       mat_fit_ud_f, mat_fit_ud_dfda,
	       CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
	       &mse, &red_chisq, NULL);
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "uniform disc, first try => radius = %g ec = %d", cpl_vector_get(a, 0), (int)ec);
    }
  ud_f = 1.11/cpl_vector_get(a, 0); // spatial frequency limit for the fit of a uniform disc
  ld_f = 0.0;
  // Now we try to get a better fit by using a filter for spatial frequencies
  // 1. calculate the number of remaining data points for each of the fitted functions
  for (i = 0; i < nbvis; i++)
    {
      mat_vis  *el = &(info->vis_list[i]);
      double    r = sqrt(el->u*el->u + el->v*el->v);
      double    f = r/info->FOV;
      if (f <= gd_f) gd_count++;
      if (f <= fdd_f) fdd_count++;
      if (f <= ud_f) ud_count++;
      if (f > ld_f) ld_f = f;
      ld_count++;
    }
  // 2. Try a better fit for the gaussian disc
  flag = flag && (cpl_matrix_set_size(x, gd_count, 1) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(y, gd_count) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(ye, gd_count) == CPL_ERROR_NONE);
  j = 0;
  for (i = 0; i < nbvis; i++)
    {
      mat_vis  *el = &(info->vis_list[i]);
      double    r = sqrt(el->u*el->u + el->v*el->v);
      double    f = r/info->FOV;
      double    v = MAT_CHYPOT2(el->mvis);
      double    e = 2.0*el->mamp*el->mamperr;
      if (f <= gd_f)
	{
	  cpl_matrix_set(x, j, 0, f);
	  cpl_vector_set(y, j, v);
	  cpl_vector_set(ye, j, e);
	  //cpl_msg_info(cpl_func, "V2[%d]: %.6f %.6f %.6f", j, f, v, e);
	  j++;
	}
    }
  cpl_vector_set(a, 0, info->fit_fwhm*FWHM2SIGMA);
  cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
	       mat_fit_gauss_f, mat_fit_gauss_dfda,
	       CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
	       &mse, &red_chisq, NULL);
  info->fit_gd_fwhm = cpl_vector_get(a, 0)/FWHM2SIGMA;
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "fit gaussian disk (mode=%d):           FWHM = %.3f mas, mse = %5.3e, red_chisq = %7.3f, spatfreq = 0 .. %7.3f 1/mas, nbvis2 = %d",
		   MAT_MODE_GAUSSIAN_DISC, info->fit_gd_fwhm, mse, red_chisq, gd_f, gd_count);
    }
  mode = MAT_MODE_GAUSSIAN_DISC;
  param = info->fit_gd_fwhm;
  chi2 = red_chisq;
  // 3. Try a better fit for the uniform darkened disc
  flag = flag && (cpl_matrix_set_size(x, ud_count, 1) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(y, ud_count) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(ye, ud_count) == CPL_ERROR_NONE);
  j = 0;
  for (i = 0; i < nbvis; i++)
    {
      mat_vis  *el = &(info->vis_list[i]);
      double    r = sqrt(el->u*el->u + el->v*el->v);
      double    f = r/info->FOV;
      double    v = MAT_CHYPOT2(el->mvis);
      double    e = 2.0*el->mamp*el->mamperr;
      if (f <= ud_f)
	{
	  cpl_matrix_set(x, j, 0, f);
	  cpl_vector_set(y, j, v);
	  cpl_vector_set(ye, j, e);
	  //cpl_msg_info(cpl_func, "V2[%d]: %.6f %.6f %.6f", j, f, v, e);
	  j++;
	}
    }
  //cpl_vector_set(a, 0, 1.0);
  cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
	       mat_fit_ud_f, mat_fit_ud_dfda,
	       CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
	       &mse, &red_chisq, NULL);
  info->fit_ud_diameter = 2.0*cpl_vector_get(a, 0);
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "fit uniform disc (mode=%d):        diameter = %.3f mas, mse = %5.3e, red_chisq = %7.3f, spatfreq = 0 .. %7.3f 1/mas, nbvis2 = %d",
		   MAT_MODE_UNIFORM_DISC, info->fit_ud_diameter, mse, red_chisq, ud_f, ud_count);
    }
  if (red_chisq < chi2)
    {
      mode = MAT_MODE_UNIFORM_DISC;
      param = info->fit_ud_diameter;
      chi2 = red_chisq;
    }
  // 4. Try a better fit for the fully darkened disc
  flag = flag && (cpl_matrix_set_size(x, fdd_count, 1) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(y, fdd_count) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(ye, fdd_count) == CPL_ERROR_NONE);
  j = 0;
  for (i = 0; i < nbvis; i++)
    {
      mat_vis  *el = &(info->vis_list[i]);
      double    r = sqrt(el->u*el->u + el->v*el->v);
      double    f = r/info->FOV;
      double    v = MAT_CHYPOT2(el->mvis);
      double    e = 2.0*el->mamp*el->mamperr;
      if (f <= fdd_f)
	{
	  cpl_matrix_set(x, j, 0, f);
	  cpl_vector_set(y, j, v);
	  cpl_vector_set(ye, j, e);
	  //cpl_msg_info(cpl_func, "V2[%d]: %.6f %.6f %.6f", j, f, v, e);
	  j++;
	}
    }
  //cpl_vector_set(a, 0, 1.0);
  cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
	       mat_fit_fdd_f, mat_fit_fdd_dfda,
	       CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
	       &mse, &red_chisq, NULL);
  info->fit_fdd_diameter = 2.0*cpl_vector_get(a, 0);
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "fit fully darkened disc (mode=%d): diameter = %.3f mas, mse = %5.3e, red_chisq = %7.3f, spatfreq = 0 .. %7.3f 1/mas, nbvis2 = %d",
		   MAT_MODE_FULLY_DARKENED_DISC, info->fit_fdd_diameter, mse, red_chisq, fdd_f, fdd_count);
    }
  if (red_chisq < chi2)
    {
      mode = MAT_MODE_FULLY_DARKENED_DISC;
      param = info->fit_fdd_diameter;
      chi2 = red_chisq;
    }
  // 5. Try a better fit for the Lorentz disc
  flag = flag && (cpl_matrix_set_size(x, gd_count, 1) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(y, gd_count) == CPL_ERROR_NONE);
  flag = flag && (cpl_vector_set_size(ye, gd_count) == CPL_ERROR_NONE);
  j = 0;
  for (i = 0; i < nbvis; i++)
    {
      mat_vis  *el = &(info->vis_list[i]);
      double    r = sqrt(el->u*el->u + el->v*el->v);
      double    f = r/info->FOV;
      double    v = MAT_CHYPOT2(el->mvis);
      double    e = 2.0*el->mamp*el->mamperr;
      if (f <= ld_f)
	{
	  cpl_matrix_set(x, j, 0, f);
	  cpl_vector_set(y, j, v);
	  cpl_vector_set(ye, j, e);
	  //cpl_msg_info(cpl_func, "V2[%d]: %.6f %.6f %.6f", j, f, v, e);
	  j++;
	}
    }
  //cpl_vector_set(a, 0, 1.0);
  cpl_fit_lvmq(x, NULL, y, ye, a, NULL,
	       mat_fit_ld_f, mat_fit_ld_dfda,
	       CPL_FIT_LVMQ_TOLERANCE, CPL_FIT_LVMQ_COUNT, CPL_FIT_LVMQ_MAXITER,
	       &mse, &red_chisq, NULL);
  info->fit_ld_fwhm = FWHM2LDA/cpl_vector_get(a, 0);
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "fit Lorentz disk (mode=%d):            FWHM = %.3f mas, mse = %5.3e, red_chisq = %7.3f, spatfreq = 0 .. %7.3f 1/mas, nbvis2 = %d",
		   MAT_MODE_LORENTZ_DISC, info->fit_ld_fwhm, mse, red_chisq, ld_f, ld_count);
    }
  if (red_chisq < chi2)
    {
      mode = MAT_MODE_LORENTZ_DISC;
      param = info->fit_ld_fwhm;
    }

  info->start_mode_guess  = mode;
  info->start_param_guess = param;
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "suggested parameters: (ok=%d)", flag);
      cpl_msg_info(cpl_func, "--start_select=4 --start_mode=%d --start_param=%.3f", mode, param);
      cpl_msg_info(cpl_func, "--prior_select=4 --prior_mode=%d --prior_param=%.3f", mode, param);
    }
  // free all temporary memory
  cpl_matrix_delete(x);
  cpl_vector_delete(y);
  cpl_vector_delete(ye);
  cpl_vector_delete(a);
}

/**
   @brief Calculate some parameter suggestions from the input data and parameters.
   @param info  Contains the image reconstruction context.

   After filtering and loading the interferrometric data some values are derived from the
   baselines and wavelength window. Some tests show that the suggested parameters are a good
   starting point for reconstructions. The calculated parameter values are only printed out
   but they are _not_ used by the plugin!
*/
static void mat_guess_parameters(mat_cal_imarec_info *info)
{
  double  fov;
  int     hnpix;
  int     npix = 32;

  if (info->cost_func != 3)
    {
      if (info->algo_mode & MODE_COMPLEX_VIS)
	{
	  mat_fit_start_image_vis(info);
	}
      if (info->algo_mode & MODE_BISPECTRUM)
	{
	  mat_fit_start_image_vis2(info);
	}
    }
  else
    {
      mat_fit_start_image_vis2(info);
    }
  // calculate some rough values (limits)
  hnpix = (int)ceil(4*info->maxfov/info->mindxrek);
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "fov   = %g mas", info->maxfov);
      cpl_msg_info(cpl_func, "psize = %g mas", info->mindxrek);
      cpl_msg_info(cpl_func, "npix  = %d (4*fov/psize)", hnpix);
    }
  // calculate some suggested parameter values
  while (npix < hnpix) npix *= 2; // we need a certain kind of oversampling
  fov = info->maxfov*((double)npix)/((double)hnpix); // the real field of view should be greater than calculated
  info->FOV_guess  = fov;
  info->npix_guess = npix;
  if (info->info_flags & INFO_PARAMETER)
    {
      cpl_msg_info(cpl_func, "suggested parameters:");
      cpl_msg_info(cpl_func, "--fov=%.2f", fov);
      cpl_msg_info(cpl_func, "--npix=%d", npix);
      cpl_msg_info(cpl_func, "-> psize = %g mas", fov/(double)npix);
    }
}

static cpl_error_code mat_create_object_masks(mat_cal_imarec_info *info)
{
  double       omr;
  int          i, x, y;

  // load the optional object mask list
  info->om_image_list = mat_read_imagelist(info,
					   info->frameset,
					   MAT_DO_OBJECT_MASK,
					   info->om_scale);
  if (info->om_image_list != NULL)
    { // we have read at least one object mask from a FITS file
      // -> convert values != 0.0 into 1.0
      // -> adapt the parameters
      for (i = 0; i < cpl_imagelist_get_size(info->om_image_list); i++)
	{
	  cpl_image *om = cpl_imagelist_get(info->om_image_list, i);
	  // it is not clear if I can use cpl_image_threshold due to the insuficcient documentation
	  for (y = 1; y <= info->npix; y++)
	    {
	      for (x = 1; x <= info->npix; x++)
		{
		  int rejected;
		  if (cpl_image_get(om, x, y, &rejected) != 0.0)
		    {
		      cpl_image_set(om, x, y, 1.0);
		    }
		}
	    }
	}
      info->om_count = cpl_imagelist_get_size(info->om_image_list);
      info->om_start = 0.0; // set a vrtual starting point
      info->om_step  = 1.0; // set a virtual step size
      return CPL_ERROR_NONE;
    }
  // we have to create the list ob object masks according to the parameters
  info->om_image_list = cpl_imagelist_new();
  if (info->om_image_list == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for object mask list");
      return CPL_ERROR_UNSPECIFIED;
    }
  omr = info->om_start;
  for (i = 0; i < info->om_count; i++)
    {
      int        omr2 = (int)round(omr*omr/info->dxrek/info->dxrek);
      int        dd = 1; // due to regularization
      // create the object mask
      cpl_image *om = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
      if (om == NULL)
	{
	  cpl_msg_error(cpl_func, "can't allocate memory for object mask");
	  return CPL_ERROR_UNSPECIFIED;
	}
      // set the mask according to the specified size
      for (x = -info->npix/2 + dd; x < info->npix/2 - dd; x++)
	{
	  for (y = -info->npix/2 + dd; y < info->npix/2 - dd; y++)
	    {
	      int r2 = x*x + y*y;
	      if (r2 <= omr2)
		{
		  mat_image_set_double(om, x, y, 1.0);
		}
	      else
		{
		  mat_image_set_double(om, x, y, 0.0);
		}
	    }
	}
      // append the mask to the list
      cpl_imagelist_set(info->om_image_list, om, cpl_imagelist_get_size(info->om_image_list));
      // increase the object mask radius
      omr += info->om_step;
    }
  return CPL_ERROR_NONE;
}

/*--------------------------------------------------------------------*/
/* Functions needed directly for the reconstruction                   */
/*--------------------------------------------------------------------*/

static cpl_error_code mat_prepare_reconstruction(mat_cal_imarec_info *info, mat_rec *rec)
{
  int          x, y;
  int          dd;
  char         fname[256];

  // select and prepare the start image
  cpl_image_copy(info->rec_image, mat_select_image(info, info->start_image_loaded, info->start_select_mode, rec->om_idx, rec->mu_idx), 1, 1);
  // select and prepare the prior image
  switch (info->reg_func)
    {
    case -1: // quadratic, with prior = 1.0
    case -2: // max entropy, with prior = 1.0
    case -3: // smoothness, prior = 1.0
    case -4: // total variation (edge preserving smoothness), prior = 1.0
    case -5: // smoothness, with prior = 1.0
      mat_image_fill(info->prior_image, 1.0);
      break;
    case -6: // quadr. Tikhonov, with prior = 1.0/npix^2
      mat_image_fill(info->prior_image, 1.0/(double)(info->npix*info->npix));
      break;
    default: // positive regularization function: prior = selected image depends on mode
      cpl_image_copy(info->prior_image, mat_select_image(info, info->prior_image_loaded, info->prior_select_mode, rec->om_idx, rec->mu_idx), 1, 1);
    }
  info->om_xa = 0;
  info->om_xe = 0;
  info->om_ya = 0;
  info->om_ye = 0;
  dd = 1; // due to regularization
  for (x = -info->npix/2 + dd; x < info->npix/2 - dd; x++)
    {
      for (y = -info->npix/2 + dd; y < info->npix/2 - dd; y++)
	{
	  if (mat_image_get_double(info->om_image, x, y) != 0.0)
	    {
	      if (x < info->om_xa) info->om_xa = x;
	      if (x > info->om_xe) info->om_xe = x;
	      if (y < info->om_ya) info->om_ya = y;
	      if (y > info->om_ye) info->om_ye = y;
	    }
	}
    }
  // create the look-up table for the coordinates (inside the bounding box!)
  info->nbom = 0;
  for (y = info->om_ya; y <= info->om_ye; y++)
    {
      for (x = info->om_xa; x <= info->om_xe; x++)
	{
	  if (mat_image_get_double(info->om_image, x, y) != 0.0)
	    {
	      info->om_list_x[info->nbom] = x;
	      info->om_list_y[info->nbom] = y;
	      info->nbom++;
	    }
	}
    }
  if (info->info_flags & INFO_PREPARE)
    {
      //cpl_msg_info(cpl_func, "object mask: [%d .. %d],[%d .. %d]: %d elements", info->om_xa, info->om_xe, info->om_ya, info->om_ye, info->nbom);
    }
  if (info->info_flags & INFO_FITS)
    {
      snprintf(fname, 256, "om_%d_%d.fits", rec->om_idx, rec->mu_idx);
      mat_image_store_double(info->om_image, fname, "OMASK", info->parlist, info->frameset);
    }
  // set the smallest and largest pixel intensities (lower_bounds and upper_bounds)
  mat_vector_fill(info->lower_bounds, 0.0);
  mat_vector_fill(info->upper_bounds, 1.0);
  // initialize the rest of the mat_rec data structure
  rec->c2bis.chi2 = 1.0e32;
  return CPL_ERROR_NONE;
}

static void mat_calc_sort_criteria(mat_rec *rec)
{
  double sum = 0.0;
  double count = 0.0;

  // Kriterium:
  //   |Cv-1|+|RR{V^2}-1|
  // + |Cb-1|+|RR{CP}-1|
  // + |Cp-1|+|RR{PHI}-1|
  if (rec->c2vis2.chi2 != 0.0)
    { // Cv=Chi^2{V^2} oder Cv=1/Chi^2{V^2} bei Chi^2{V^2}<1,
      double cv;
      if (rec->c2vis2.chi2 >= 1.0)
	cv = rec->c2vis2.chi2;
      else
	cv = 1.0/rec->c2vis2.chi2;
      sum   += fabs(cv - 1.0) + fabs(rec->c2vis2.residual - 1.0);
      count += 2.0;
    }
  if (rec->c2cp.chi2 != 0.0)
    { // Cb=Chi^2{CP} oder Cb=1/Chi^2{CP} bei Chi^2{CP}<1
      double cb;
      if (rec->c2cp.chi2 >= 1.0)
	cb = rec->c2cp.chi2;
      else
	cb = 1.0/rec->c2cp.chi2;
      sum   += fabs(cb - 1.0) + fabs(rec->c2cp.residual - 1.0);
      count += 2.0;
    }
  if (rec->c2phi.chi2 != 0.0)
    { // Cp=Chi^2{PHI} oder Cp=1/Chi^2{PHI} bei Chi^2{PHI}<1
      double cp;
      if (rec->c2phi.chi2 >= 1.0)
	cp = rec->c2phi.chi2;
      else
	cp = 1.0/rec->c2phi.chi2;
      sum   += fabs(cp - 1.0) + fabs(rec->c2phi.residual - 1.0);
      count += 2.0;
    }
  rec->qrec = sum/count;
}

/**
   @brief Derives the squared visibilities and closure phases on the measured uc coordinates.
   @param info  Contains the image reconstruction context.
   @returns void

   The current reconstruction is used to calculate visibilities and closure phases for the uv coordinates
   given in the input interferometric data. This allows a comparison between measured and reconstructed
   squared visibilities and closure phases. These values are stored in the result file.
*/
static void mat_derive_bis_and_vis(mat_cal_imarec_info *info)
{
  int i;
 
  // derive the squared visibilities from the reconstruction
  for (i = 0; i < info->nbvis2; i++)
    {
      mat_vis2 *el = &(info->vis2_list[i]);
      el->ftuk    = mat_image_get_complex_interpolated(info->frec_image, el->u, el->v);
      el->rvis2   = MAT_CHYPOT2(el->ftuk);
    }
  // derive the closure phase from the reconstruction
  for (i = 0; i < info->nbbis; i++)
    {
      mat_bis *el = &(info->bis_list[i]);
      el->ftuk    = mat_image_get_complex_interpolated(info->frec_image, el->u1, el->v1);
      el->ftvk    = mat_image_get_complex_interpolated(info->frec_image, el->u2, el->v2);
      el->ftwk    = mat_image_get_complex_interpolated(info->frec_image, el->u3, el->v3);
      el->rbis    = el->ftuk*el->ftvk*conj(el->ftwk);
      el->ramp    = cabs(el->rbis);
      el->rphasor = el->rbis*(1.0/(el->ramp + 1e-30) + 0.0*I);
    }
  // derive the visibilities from the reconstruction
  for (i = 0; i < info->nbvis; i++)
    {
      mat_vis *el = &(info->vis_list[i]);
      el->ftuk    = mat_image_get_complex_interpolated(info->frec_image, el->u, el->v);
      el->rvis    = el->ftuk;
      el->rvis2   = MAT_CHYPOT2(el->ftuk);
      el->ramp    = cabs(el->rvis);
      el->rphasor = el->rvis*(1.0/(el->ramp + 1e-30) + 0.0*I);
    }
}

// ******************************************************************************************************
// ***** Cost function 1 using bispectrum and/or fourier spectrum (BIS + VIS2 and/or VIS)
// ******************************************************************************************************

/**
   @brief Calculate a weight factor such as that the chi2 is minimal for the bispectrum.
   @param info    Contains the image reconstruction context.
   @returns The weight factor.

   The measured and reconstructed closure phases and squared visibilities are used
   to calculate a weight factur such as that the chi squared between reconstruction
   and measurement is minimal. This function is used by the first cost function.
*/
static double mat_calc_cf1_cost_weight(mat_cal_imarec_info *info, double total)
{ // gamma0bisftQ1, transfer FORTRAN -> C is correct!
  int    i;
  double sum0 = 0.0;
  double sum1 = 0.0;
  
  if (info->use_flags & USE_T3_TABLE)
    {
      for (i = 0; i < info->nbbis; i++)
	{
	  /* FORTRAN -> C
	     bis                             -> el->mbis
	     bisk                            -> el->rbis
	     gbis(j)/(abs(cbivar(j))+level)  -> el->wuvbis
	  */
	  mat_bis *el = &(info->bis_list[i]);
	  /* FORTRAN (slightly modified):
	     wuv  = gbis(j)/(abs(cbivar(j))+level)
	     sum0 = sum0 + 2. * real( bis*conjg(bisk) ) * wuv
	     sum1 = sum1 + 2. * cabs( bisk )**2 * wuv
	  */
	  sum0 += 2.0*creal(el->mbis*conj(el->rbis))*el->wuvbis;
	  sum1 += 2.0*MAT_CHYPOT2(el->rbis)         *el->wuvbis;
	}
    }
  if (info->use_flags & USE_VIS2_TABLE)
    {
      for (i = 0; i < info->nbvis2; i++)
	{
	  /* FORTRAN -> C
	     v2                             -> el->mvis2
	     v2k                            -> el->rvis2
	     gpow(j)/(abs(v2var(j))+level)  -> el->wuvvis2
	     cabs(ftima00(0,0))             -> total
	  */
	  mat_vis2 *el = &(info->vis2_list[i]);
	  /* FORTRAN (slightly modified):
	     wuv  = gpow(j)/(abs(v2var(j))+level)
	     sum0 = sum0 + cabs(ftima00(0,0))   *v2 *v2k*wuv
	     sum1 = sum1 + cabs(ftima00(0,0))**2*v2k*v2k*wuv
	  */
	  sum0 +=       total*el->mvis2*el->rvis2*el->wuvvis2;
	  sum1 += total*total*el->rvis2*el->rvis2*el->wuvvis2;
	}
    }
  if (info->use_flags & USE_VIS_TABLE)
    {
      for (i = 0; i < info->nbvis; i++)
	{
	  /* FORTRAN -> C
	     cft                           -> el->mvis
	     cftk                          -> el->rvis
	     gft(j)/(abs(ftvar(j))+level)  -> el->wuvvis
	  */
	  mat_vis *el = &(info->vis_list[i]);
	  /* FORTRAN (slightly modified):
	     cft = ft(j)
	     wuv = gft(j)/(abs(ftvar(j))+level)
	     sum0 = sum0 + real( cft*conjg(cftk) ) * wuv
	     sum1 = sum1 + cabs( cftk )**2         * wuv
	  */
	  sum0 += creal(el->mvis*conj(el->rvis))*el->wuvvis;
	  sum1 += MAT_CHYPOT2(el->rvis)         *el->wuvvis;
	}
    }
  return fmax(sum0/(fabs(sum1) + 1e-30), 1e-30);
}

/**
   @brief Updates the gradient using one closure phase element.
   @param info    Contains the image reconstruction context.
   @param el      One closure phase element.
   @param weight  Weight for this closure phase element from the reconstruction.
   @param wuv     Weight for the closure phase.
   @returns The reconstructed closure phase.

   The reconstructed closure phase (interpolated from the FFT of the previous
   reconstruction at the same (fractional uv coordinates) is calculated.
   This value and the measured closure phase (parameter el) is used to update
   the gradient image. The reconstructed closure phase is returned as function
   value. This function is used by the first cost function.
*/
static void mat_calc_cf1_fdcost_bis(mat_cal_imarec_info *info, mat_bis *el, double weight, double wuv)
{ // dkbisQ1, transfer FORTRAN -> C is correct!
  double complex A, fd;
  double         fdv;

  /* FORTRAN -> C
     gamma0  -> weight
     bis     -> el->mbis
     bisk    -> el->rbis*weight
  */
  /* FORTRAN:
     bisk = ftuk*ftvk*conjg(ftwk) * cmplx(gamma0,0.)
     A = bisk - bis
  */
  A = el->rbis*(weight + 0.0*I) - el->mbis;
  /* FORTRAN (slightly modified):
     ftC(ux,uy)    += A * ftwk*conjg(ftvk) * cmplx(wuv,0.)
     ftCvar(ux,uy) += var * cabs(ftwk*ftvk)**2 * wuv**2
  */
  fd  = A*el->ftwk*conj(el->ftvk)*(wuv + 0.0*I);
  fdv = el->mvar*MAT_CHYPOT2(el->ftwk*el->ftvk)*wuv*wuv;
  mat_image_add_complex_interpolated(info->fdcost_image,    el->u1,  el->v1, fd);
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u1,  el->v1, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u1, -el->v1, fdv);
  /* FORTRAN (slightly modified):
     ftC(vx,vy)    += A * ftwk*conjg(ftuk) * cmplx(wuv,0.)
     ftCvar(vx,vy) += var * cabs(ftwk*ftuk)**2 * wuv**2
  */
  fd  = A*el->ftwk*conj(el->ftuk)*(wuv + 0.0*I);
  fdv = el->mvar*MAT_CHYPOT2(el->ftwk*el->ftuk)*wuv*wuv;
  mat_image_add_complex_interpolated(info->fdcost_image,    el->u2,  el->v2, fd);
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u2,  el->v2, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u2, -el->v2, fdv);
  /* FORTRAN (slightly modified):
     ftC(wx,wy)    += conjg(A) * ftuk*ftvk * cmplx(wuv,0.)
     ftCvar(wx,wy) += var * cabs(ftuk*ftvk)**2 * wuv**2
  */
  fd  = conj(A)*el->ftuk*el->ftvk*(wuv + 0.0*I);
  fdv = el->mvar*MAT_CHYPOT2(el->ftuk*el->ftvk)*wuv*wuv;
  mat_image_add_complex_interpolated(info->fdcost_image,    el->u3,  el->v3, fd);
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u3,  el->v3, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u3, -el->v3, fdv);
}

/**
   @brief Updates the gradient using one squared visibility element.
   @param info    Contains the image reconstruction context.
   @param el      One squared visibility element.
   @param weight  Weight for this squared visibility element from the reconstruction.
   @param wuv     Weight for this squared visibility element.
   @param total   Integral of the current reconstructed image
   @returns The reconstructed squared visibility.

   The reconstructed squared visibility (interpolated from the FFT of the previous
   reconstruction at the same (fractional uv coordinates) is calculated.
   This value and the measured squared visibility (parameter el) is used to update
   the gradient image. The reconstructed squared visibility is returned as function
   value.
*/
static void mat_calc_cf1_fdcost_vis2(mat_cal_imarec_info *info, mat_vis2 *el, double weight, double wuv, double total)
{ // dkpowQ1, transfer FORTRAN -> C is correct!
  double         A;
  double complex fd;
  double         fdv;

  /* FORTRAN -> C
     gamma0           -> weight
     cabs(ftok(0,0))  -> total
     cabs(ftuk)**2    -> el->rvis2
  */
  /* FORTRAN:
     powk = cabs(ftok(0,0)) * cabs(ftuk)**2 * gamma0
     A = powk - pow
  */
  A = total*el->rvis2*weight - el->mvis2;
  /* FORTRAN (slightly modified):
     ft0 = ftok(0,0)
     ftC(ux,uy)      += cmplx(A,0.) * ft0 * ftuk * cmplx(2.*wuv,0.)
     ftCvar(ux,uy)   += + var * cabs(ft0*ftuk)**2 * wuv**2 * 4.
  */
  fd  = (2.0*A*total*wuv + 0.0*I)*el->ftuk;
  fdv = 4.0*el->mvar*el->rvis2*total*total*wuv*wuv;
  mat_image_add_complex_interpolated(info->fdcost_image,    el->u,  el->v, fd);
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u,  el->v, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u, -el->v, fdv);
  /* FORTRAN (slightly modified):
     ftC(0,0)    += cmplx(A,0.) * ftuk*conjg(ftuk) * cmplx(2.*wuv,0.)
     ftCvar(0,0) += var * cabs(ftuk*conjg(ftuk))**2 * wuv**2 * 4.
  */
  fd  = (2.0*A*wuv + 0.0*I)*el->ftuk*conj(el->ftuk);
  fdv = 4.0*el->mvar*MAT_CHYPOT2(el->ftuk*conj(el->ftuk))*wuv*wuv;
  mat_image_add_complex(info->fdcost_image,   0, 0, fd);
  mat_image_add_double(info->fdcostvar_image, 0, 0, fdv);
}

static void mat_calc_cf1_fdcost_vis(mat_cal_imarec_info *info, mat_vis *el, double weight, double wuv)
{ // dkftQ1, transfer FORTRAN -> C is correct!
  double complex A;
  double complex fd;
  double         fdv;

  /* FORTRAN (slightly modified):
     ftwertk = ftwertk + cmplx(gewu(mu,nu)*gamma0,0.) * ftu
     A = ftwertk - ftwert
  */
  A = el->rvis*(weight + 0.0*I) - el->mvis;
  /* FORTRAN (slightly modified):
     ftC(ux,uy)    += A * cmplx(wuv,0.)
     ftCvar(ux,uy) += var * wuv**2
  */
  fd = A*(wuv + 0.0*I);
  mat_image_add_complex_interpolated(info->fdcost_image,  el->u,  el->v, fd);
  fdv = el->mvar*wuv*wuv;
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u,  el->v, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u, -el->v, fdv);
}

static void mat_calc_cf1_costgrad(mat_cal_imarec_info *info, mat_rec *rec, double total)
{ // costgradbispowftq1, transfer Fortran -> C is correct!
  int    i;
  double weight;

  /* FORTRAN -> C
     gamma0  -> weight
     gamma1  -> total
  */
  weight = mat_calc_cf1_cost_weight(info, total);
  if (info->info_flags & INFO_ITER_DETAILS)
    {
      cpl_msg_info(cpl_func, "   total = %.6f, weight = %.6e", total, weight);
    }
  if (info->use_flags & USE_T3_TABLE)
    { // costgradbispowq1 (part)
      info->use_flags |= (USE_BIS_FLAG | USE_CP_FLAG);
      for (i = 0; i < info->nbbis; i++)
	{
	  /* FORTRAN -> C
	     bis                              -> elbis->mbis
	     ftuk*ftvk*conjg(ftwk)            -> elbis->rbis
	     gbis0(j)                         -> elbis->weight
	     gbis0(j)/(abs(cbivar(j))+level)  -> elbis->wuvbis
	     gbis0(j)/(cpperror(j)**2+level)  -> elbis->wuvcp
	  */
	  mat_bis       *elbis = &(info->bis_list[i]);
	  double         ddcp;
	  double complex wdbis, ddbis;
	  /* FORTRAN (slightly modified):
	     wuv = 12.*gamma0*gbis(j)/(abs(cbivar(j))+level)
	     call dkbisQ1( rux,ruy,rvx,rvy, bis,var, gamma0, wuv, ftima00, bisk, ftC,ftCvar, id )
	     bisk1 = bisk * cmplx(1./(gamma0*gamma1*gamma1*gamma1),0.)
	     dcp = phase0(bis*conjg(bisk1))
	     with
	     bisk = ftuk*ftvk*conjg(ftwk) * cmplx(gamma0,0.)
	  */
	  mat_calc_cf1_fdcost_bis(info, elbis, weight, 12.0*weight*elbis->wuvbis);
	  wdbis  = elbis->rbis*(weight + 0.0*I)                  - elbis->mbis; // weighted bispectrum difference
	  ddbis  = elbis->rbis*(1.0/(total*total*total) + 0.0*I) - elbis->mbis; // direct bispectrum difference
	  ddcp   = carg(elbis->mphasor*conj(elbis->rphasor));                   // direct closure phase difference
	  // update the global cost value (using the weighted difference)
	  /* FORTRAN (slightly modified):
	     cost    += cabs( bisk-bis )**2 * wuv/gamma0
	     sumcost += 12.* gbis(j)
	  */
	  mat_chi2_add_value(&(rec->c2cost),
			     12.0*MAT_CHYPOT2(wdbis)*elbis->wuvbis,
			     12.0*elbis->weight);
	  // update the bispectra specific values (using the direct difference)
	  /* FORTRAN (slightly modified):
	     costbisft += 12.*cabs( bisk1-bis )**2 *   gbis0(j)/(abs(cbivar(j))+level)
	     sumbisft  += 12.*                         gbis0(j)
	     residual   = 12.*(bisk1-bis) * sqrt( 2. * gbis0(j)/(abs(cbivar(j))+level) )
	  */
	  mat_chi2_add_value_residual_complex(&(rec->c2bis),
					      12.0*MAT_CHYPOT2(ddbis)*elbis->wuvbis,
					      12.0*elbis->weight,
					      ddbis*(12.0*sqrt(2.0*elbis->wuvbis) + 0.0*I));
	  // update the closure phase specific values (using the direct difference)
	  /* FORTRAN (slightly modified):
	     costcp   += abs(dcp*180./pi)**2 * gbis0(j)/(cpperror(j)**2+level)
	     sumcp    +=                       gbis0(j)
	     residual  = (dcp*180./pi) * s qrt(gbis0(j))/(abs(cpperror(j))+level)
	  */
	  mat_chi2_add_value_residual_double(&(rec->c2cp),
					     ddcp*ddcp*elbis->wuvcp,
					     elbis->weight,
					     ddcp*sqrt(elbis->wuvcp));
	}
      if ((info->info_flags & INFO_FITS) && (info->nbit  <= INFO_FITS_MAX_ITER))
	{
	  char   fname[256];
	  snprintf(fname, 256, "fdcost_bis_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
	  mat_image_store_complex(info->fdcost_image, fname, "FREC", info->parlist, info->frameset);
	}
    }
  if (info->use_flags & USE_VIS2_TABLE)
    { // costgradbispowq1 (part)
      info->use_flags |= (USE_BIS_FLAG | USE_VIS2_FLAG);
      for (i = 0; i < info->nbvis2; i++)
	{
	  /* FORTRAN -> C
	     v2                               -> elvis2->mvis2
	     cabs(ftuk)**2                    -> elvis2->rvis2
	     gpow(j)                          -> elvis2->weight
	     gpow(j)/(abs(v2var(j))+level)    -> elvis2->wuvvis2
	  */
	  mat_vis2 *elvis2 = &(info->vis2_list[i]);
	  double    wdvis2, ddvis2;
	  /* FORTRAN (slightly modified):
	     wuv = 6.*gamma0*gpow(j)/(abs(v2var(j))+level)
	     call dkpowQ1( rux,ruy, v2,var, gamma0, wuv, ftima00, v2k, ftC,ftCvar, id )
	     v2k1 = v2k * 1./(gamma0*gamma1*gamma1*gamma1)
	     with
	     v2k = cabs(ftok(0,0)) * cabs(ftuk)**2 * gamma0
	  */
	  mat_calc_cf1_fdcost_vis2(info, elvis2, weight, 6.0*weight*elvis2->wuvvis2, total);
	  wdvis2  = total*weight*elvis2->rvis2               - elvis2->mvis2; // weighted squared visibility difference
	  ddvis2  =              elvis2->rvis2/(total*total) - elvis2->mvis2; // direct squared visibility difference
	  // update the global cost value (using the weighted difference)
	  /* FORTRAN (slightly modified):
	     cost    += abs( v2k-v2 )**2 * wuv/gamma0
	     sumcost += 6.* gpow(j)
	  */
	  mat_chi2_add_value(&(rec->c2cost),
			     6.0*wdvis2*wdvis2*elvis2->wuvvis2,
			     6.0*elvis2->weight);
	  // update the bispectra specific values (using the direct difference)
	  /* FORTRAN (slightly modified):
	     costbisft += 6.*abs(v2k1-v2)**2 *    gpow0(j)/(abs(v2var(j))+level)
	     sumbisft  += 6.                     *gpow0(j)
	     residual   = 6.*   (v2k1-v2) * sqrt( gpow0(j)/(abs(v2var(j))+level) )
	  */
	  mat_chi2_add_value_residual_double(&(rec->c2bis),
					     6.0*ddvis2*ddvis2*elvis2->wuvvis2,
					     6.0*elvis2->weight,
					     6.0*ddvis2*sqrt(elvis2->wuvvis2));
	  // update the squared visibility specific values (using the direct difference)
	  /* FORTRAN (slightly modified):
	     costvis2 += abs(v2k1-v2)**2 *    gpow0(j)/(abs(v2var(j))+level)
	     sumvis2  +=                      gpow0(j)
	     residual  = 6.*(v2k1-v2) * sqrt( gpow0(j)/(abs(v2var(j))+level) )
	  */
	  mat_chi2_add_value_residual_double(&(rec->c2vis2),
					     ddvis2*ddvis2*elvis2->wuvvis2,
					     elvis2->weight,
					     6.0*ddvis2*sqrt(elvis2->wuvvis2));
	}
      if ((info->info_flags & INFO_FITS) && (info->nbit  <= INFO_FITS_MAX_ITER))
	{
	  char   fname[256];
	  snprintf(fname, 256, "fdcost_vis2_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
	  mat_image_store_complex(info->fdcost_image, fname, "FREC", info->parlist, info->frameset);
	}
    }
  if (info->use_flags & USE_VIS_TABLE)
    { // costgradftq1 (part)
      info->use_flags |= (USE_BIS_FLAG | USE_VIS2_FLAG | USE_PHI_FLAG);
      for (i = 0; i < info->nbvis; i++)
	{
	  /* FORTRAN -> C
	     cft                           -> elvis->mvis
	     ftuk                          -> elvis->rvis
	     gft(j)                        -> elvis->weight
	     gft(j)/(abs(ftvar(j))+level)  -> elvis->wuvvis
	     gft0(j)/(ftperror**2+level)   -> elvis->wuvphi
	  */
	  mat_vis        *elvis = &(info->vis_list[i]);
	  double          ddphi, ddvis2;
	  double complex  wdvis, ddvis;
	  /* FORTRAN (slightly modified):
	     wuv = 6.*gamma0*gft(j)/(abs(ftvar(j))+level)
	     call dkftQ1( rux,ruy, cft,var,gamma0, wuv, ftima00, cftk, ftC,ftCvar, id )
	     cftk1 = cftk * cmplx(1./(gamma0*gamma1),0.)
	     dcp   = phase0(cft*conjg(cftk1))
	     with
	     cftk = ftuk*gamma0
	  */
	  mat_calc_cf1_fdcost_vis(info, elvis, weight, 6.0*weight*elvis->wuvvis);
	  wdvis  = elvis->rvis*(weight + 0.0*I)    - elvis->mvis;  // weighted visibility difference
	  ddvis  = elvis->rvis*(1.0/total + 0.0*I) - elvis->mvis;  // direct visibility difference
	  ddphi  = carg(elvis->mphasor*conj(elvis->rphasor));      // direct phase difference
	  ddvis2 = elvis->rvis2/(total*total)      - elvis->mvis2; // direct squared visibility difference
	  // update the global cost value (using the weighted difference)
	  /* FORTRAN (slightly modified):
	     cost    += cabs( cftk-cft )**2 * wuv/gamma0
	     sumcost += 6.*gft(j)
	  */
	  mat_chi2_add_value(&(rec->c2cost),
			     6.0*MAT_CHYPOT2(wdvis)*elvis->wuvvis,
			     6.0*elvis->weight);
	  // update the bispectra specificvalues (using the direct difference)
	  /* FORTRAN (slightly modified):
	     costbisft += 6.*cabs(cftk1-cft)**2 *gft0(j)/(abs(ftvar(j))+level)
	     sumbisft  += 6.*                    gft0(j)
	     residual   = real(cftk1-cft) * sqrt( gft0(j)/( 0.5*(abs(ftvar(j))+level) ) )
	  */
	  mat_chi2_add_value_residual_complex(&(rec->c2bis),
					      6.0*MAT_CHYPOT2(ddvis)*elvis->wuvvis,
					      6.0*elvis->weight,
					      ddvis*(sqrt(2.0*elvis->wuvvis) + 0.0*I));
	  // update the squared visibility specific values, currently NOT in FORTRAN, derived from VIS2 part (using the direct difference)
	  mat_chi2_add_value_residual_double(&(rec->c2vis2),
					     ddvis2*ddvis2*elvis->wuvvis2,
					     elvis->weight,
					     6.0*ddvis2*sqrt(elvis->wuvvis2));
	  // update the phase specific values (using the direct difference)
	  /* FORTRAN (slightly modified):
	     costftph += abs(dcp*180./pi)**2 * gft0(j)/(ftperror**2+level)
	     sumftph  +=                       gft0(j)
	     residual  = (dcp*180./pi)  * sqrt(gft0(j))/(abs(ftperror)+level)
	  */
	  mat_chi2_add_value_residual_double(&(rec->c2phi),
					     ddphi*ddphi*elvis->wuvphi,
					     elvis->weight,
					     ddphi*sqrt(elvis->wuvphi));
	}
      if ((info->info_flags & INFO_FITS) && (info->nbit  <= INFO_FITS_MAX_ITER))
	{
	  char   fname[256];
	  snprintf(fname, 256, "fdcost_vis_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
	  mat_image_store_complex(info->fdcost_image, fname, "FREC", info->parlist, info->frameset);
	}
    }
}


// ******************************************************************************************************
// ***** Cost function 2 using bispectrum and/or fourier spectrum (BIS + VIS2 and/or VIS)
// ******************************************************************************************************

/**
   @brief Calculate a weight factor such as that the chi2 is minimal for the bispectrum.
   @param info    Contains the image reconstruction context.
   @returns The weight factor.

   The measured and reconstructed closure phases and squared visibilities are used
   to calculate a weight factur such as that the chi squared between reconstruction
   and measurement is minimal. This function is used by the first cost function.
*/
static double mat_calc_cf2_cost_weight(mat_cal_imarec_info *info)
{ // gamma0bisftQ2, transfer Fortran -> C is correct!
  int    i;
  double sum0 = 0.0;
  double sum1 = 0.0;

  if (info->use_flags & USE_T3_TABLE)
    { // gamma0Q2/gamma0bisftQ2
      for (i = 0; i < info->nbbis; i++)
	{
	  /* FORTRAN -> C
	     f0                                    -> info->cost_weight
	     bis                                   -> el->mbis
	     bisk                                  -> el->rbis
	     bis *cmplx(1./(cabs(bis)+level),0.)   -> el->mphasor
	     bisk*cmplx(1./(cabs(bisk)+level),0.)  -> el->rphasor
	     gbis(j)/(1.-exp(-cpvar)+level)        -> el->wuvph
	     gbis(j)/(abs(modvar)+level)           -> el->wuvmod
	  */
	  mat_bis        *el     = &(info->bis_list[i]);
	  /* FORTRAN (slightly modified):
	     cpvar  = (cpperror(j)*pi/180.) * (cpperror(j)*pi/180.)
	     wuvph  = gbis(j)/(1.-exp(-cpvar)+level)
	     modvar = cbivar(j) - cabs(bis)**2 * (1.-exp(-cpvar))
	     wuvmod = gbis(j)/(abs(modvar)+level)
	     C      = bis *cmplx(1./(cabs(bis)+level),0.)
	     Ck     = bisk*cmplx(1./(cabs(bisk)+level),0.)
	     sum0   = sum0 + 2.*real( C*conjg(Ck) )*wuvph + 2.*f0*cabs(bis)*cabs(bisk)*wuvmod
	     sum1   = sum1 + 2.                    *wuvph + 2.*f0*cabs(bisk)**2       *wuvmod
	  */
	  sum0 += 2.0*creal(el->mphasor*conj(el->rphasor))*el->wuvph;
	  sum1 += 2.0                                     *el->wuvph;
	  sum0 += 2.0*info->cost_weight*el->mamp*el->ramp*el->wuvmod;
	  sum1 += 2.0*info->cost_weight*el->ramp*el->ramp*el->wuvmod;
	}
    }
  if (info->use_flags & USE_VIS2_TABLE)
    {
      double ft00 = cabs(mat_image_get_complex(info->frec_image, 0, 0));
      for (i = 0; i < info->nbvis2; i++)
	{
	  /* FORTRAN -> C
	     f0                             -> info->cost_weight
	     v2                             -> el->mvis2
	     v2k                            -> el->rvis2
	     gpow(j)/(abs(v2var(j))+level)  -> el->wuvvis2
	   */
	  mat_vis2 *el = &(info->vis2_list[i]);
	  /* FORTRAN (slightly modified):
	     wuv = gpow(j)/(abs(v2var(j))+level)
	     sum0 = sum0 + f0*cabs(ftima00(0,0))*v2*v2k*wuv
	     sum1 = sum1 + f0*cabs(ftima00(0,0))**2*v2k*v2k*wuv
	  */
	  sum0 += info->cost_weight*ft00     *el->mvis2*el->rvis2*el->wuvvis2;
	  sum1 += info->cost_weight*ft00*ft00*el->rvis2*el->rvis2*el->wuvvis2;
	}
    }
  if (info->use_flags & USE_VIS_TABLE)
    {
      for (i = 0; i < info->nbvis; i++)
	{
	  /* FORTRAN -> C
	     f0                                        -> info->cost_weight
	     cft                                       -> el->mvis
	     cftk                                      -> el->rvis
	     cft  * cmplx( 1./(cabs(cft)+level),0. )   -> el->mphasor
	     cftk * cmplx( 1./(cabs(cftk)+level),0. )  -> el->rphasor
	     gft(j)/(1.-exp(-phvar)+level)             -> el->wuvph
	     gft(j)/(abs(modvar)+level)                -> el->wuvmod
	  */
	  mat_vis        *el     = &(info->vis_list[i]);
	  /* FORTRAN (slightly modified):
	     phvar  = ftpherr(j) * ftpherr(j)
	     var    = ftvar(j)
	     modvar = var-cabs(cft)**2*(1.-exp(-phvar))
	     wuvph  = gft(j)/(1.-exp(-phvar)+level)
	     wuvmod = gft(j)/(abs(modvar)+level)
	     Ck     = cftk * cmplx( 1./(cabs(cftk)+level),0. )
	     C      = cft  * cmplx( 1./(cabs(cft)+level),0. )
	     sum0   = sum0 + real( Ck*conjg(C) ) * wuvph + f0*cabs(cftk)*cabs(cft) * wuvmod
	     sum1   = sum1 +                       wuvph + f0*cabs(cftk)*cabs(cftk)* wuvmod
	  */
	  sum0 += creal(el->mphasor*conj(el->rphasor))*el->wuvph;
	  sum1 +=                                      el->wuvph;
	  sum0 += info->cost_weight*el->mamp*el->ramp*el->wuvmod;
	  sum1 += info->cost_weight*el->ramp*el->ramp*el->wuvmod;
	}
    }
  return fmax(sum0/(fabs(sum1) + 1e-30), 1e-30);
}

/**
   @brief Updates the gradient using one closure phase element.
   @param info    Contains the image reconstruction context.
   @param el      One closure phase element.
   @param weight  Weight for this closure phase element from the reconstruction.
   @param wuvph  Weight for the phase.
   @param wuvmod  Weight for the modulus.
   @returns The reconstructed closure phase.

   The reconstructed closure phase (interpolated from the FFT of the previous
   reconstruction at the same (fractional uv coordinates) is calculated.
   This value and the measured closure phase (parameter el) is used to update
   the gradient image. The reconstructed closure phase is returned as function
   value. This function is used by the second cost function.
*/
static void mat_calc_cf2_fdcost_bis(mat_cal_imarec_info *info, mat_bis *el, double weight, double wuvph, double wuvmod)
{ // dkbisQ2, transfer Fortran -> C is correct!
  double complex A, C0, fd;
  double         B, D, E, fdv;
  double         wbisk, cpvar, modvar;
  double         F1, F2;

  /* FORTRAN -> C
     gamma0                                -> weight
     level                                 -> 1e-30
     bis                                   -> el->mbis
     bisk                                  -> el->rbis
     bis*cmplx(1./(cabs(bis)+level),0.)    -> el->mphasor
     bisk*cmplx(1./(cabs(bisk)+level),0.)  -> el->rphasor
   */
  /* FORTRAN (slightly modified):
     C     = bis*cmplx(1./(cabs(bis)+level),0.)
     Ck    = bisk*cmplx(1./(cabs(bisk)+level),0.)
     A     = cmplx(wuvph,0.) * ( cmplx(gamma0,0.)*Ck - C ) * cmplx(1./(cabs(bisk)+level),0.)
     B     = real( bisk * conjg(A) )
     E     = wuvmod * ( gamma0*cabs(bisk) - cabs(bis) ) * (1./(cabs(bisk)+level))
     C0    = Ck * conjg(C)
     D     = real( C0*C0 )
     with
     wbisk  := 1./(cabs(el->rbis)+level)
     cpvar  := el->mcperr*el->mcperr
     modvar := el->mamperr*el->mamperr
  */
  wbisk  = 1.0/(el->ramp + 1e-30);
  cpvar  = el->mcperr*el->mcperr;
  modvar = el->mamperr*el->mamperr;
  A  = (wuvph*wbisk + 0.0*I)*((weight + 0.0*I)*el->rphasor - el->mphasor);
  B  = creal(el->rbis*conj(A));
  E  = wuvmod*(weight*el->ramp - el->mamp)*wbisk;
  C0 = el->rphasor*conj(el->mphasor);
  D  = creal(C0*C0);
  /* FORTRAN (slightly modified):
     ftC(ux,uy)    += A *ftwk*conjg(ftvk) - cmplx( B/(cabs(ftuk)**2),0. )*ftuk + cmplx( f0*E*cabs(ftvk*ftwk)**2,0. )*ftuk
     ftCvar(ux,uy) += 0.5*wuvph**2*(1.-exp(-cpvar)) * ( 1. + exp(-cpvar) * D ) * (1./(cabs(ftuk)**2+level)) + f0**2*wuvmod**2*modvar*cabs(ftvk*ftwk)**2 
     with
     F1 := 0.5*wuvph**2*(1.-exp(-cpvar)) * ( 1. + exp(-cpvar) * D )
     F2 := f0**2*wuvmod**2*modvar
  */
  F1 = 0.5*wuvph*wuvph*(1.0 - exp(-cpvar))*(1.0 + exp(-cpvar)*D);
  F2 = info->cost_weight*info->cost_weight*wuvmod*wuvmod*modvar;
  fd = A*el->ftwk*conj(el->ftvk)
    - (B/MAT_CHYPOT2(el->ftuk) + 0.0*I)*el->ftuk
    + (info->cost_weight*E*MAT_CHYPOT2(el->ftvk*el->ftwk) + 0.0*I)*el->ftuk;
  fdv = F1*(1./(MAT_CHYPOT2(el->ftuk) + 1e-30)) + F2*MAT_CHYPOT2(el->ftvk*el->ftwk);
  mat_image_add_complex_interpolated(info->fdcost_image,    el->u1,  el->v1, fd);
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u1,  el->v1, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u1, -el->v1, fdv);
  /* FORTRAN (slightly modified):
     ftC(vx,vy)    +=  A *ftwk*conjg(ftuk) - cmplx( B/(cabs(ftvk)**2),0. )*ftvk + cmplx( f0*E*cabs(ftuk*ftwk)**2,0. )*ftvk
     ftCvar(vx,vy) +=  0.5*wuvph**2*(1.-exp(-cpvar)) * ( 1. + exp(-cpvar) * D ) * (1./(cabs(ftvk)**2+level)) + f0**2*wuvmod**2*modvar*cabs(ftuk*ftwk)**2
  */
  fd = A*el->ftwk*conj(el->ftuk)
    - (B/MAT_CHYPOT2(el->ftvk) + 0.0*I)*el->ftvk
    + (info->cost_weight*E*MAT_CHYPOT2(el->ftuk*el->ftwk) + 0.0*I)*el->ftvk;
  fdv = F1*(1./(MAT_CHYPOT2(el->ftvk) + 1e-30)) + F2*MAT_CHYPOT2(el->ftuk*el->ftwk);
  mat_image_add_complex_interpolated(info->fdcost_image,    el->u2,  el->v2, fd);
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u2,  el->v2, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u2, -el->v2, fdv);
  /* FORTRAN (slightly modified):
     ftC(wx,wy)    += conjg(A) *ftuk*ftvk - cmplx( B/(cabs(ftwk)**2),0. )*ftwk +  cmplx( f0*E*cabs(ftuk*ftvk)**2,0. )*ftwk
     ftCvar(wx,wy) += 0.5*wuvph**2*(1.-exp(-cpvar)) * ( 1. + exp(-cpvar) * D ) * (1./(cabs(ftwk)**2+level)) +  f0**2*wuvmod**2*modvar*cabs(ftuk*ftvk)**2
  */
  fd = conj(A)*el->ftuk*el->ftvk
    - (B/MAT_CHYPOT2(el->ftwk) + 0.0*I)*el->ftwk
    + (info->cost_weight*E*MAT_CHYPOT2(el->ftuk*el->ftvk) + 0.0*I)*el->ftwk;
  fdv = F1*(1./(MAT_CHYPOT2(el->ftwk) + 1e-30)) + F2*MAT_CHYPOT2(el->ftuk*el->ftvk);
  mat_image_add_complex_interpolated(info->fdcost_image,    el->u3,  el->v3, fd);
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u3,  el->v3, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u3, -el->v3, fdv);
}

/**
   @brief Updates the gradient using one squared visibility element.
   @param info    Contains the image reconstruction context.
   @param el      One squared visibility element.
   @param weight  Weight for this squared visibility element from the reconstruction.
   @param wuv     Weight for this squared visibility element.
   @returns The reconstructed squared visibility.

   The reconstructed squared visibility (interpolated from the FFT of the previous
   reconstruction at the same (fractional uv coordinates) is calculated.
   This value and the measured squared visibility (parameter el) is used to update
   the gradient image. The reconstructed squared visibility is returned as function
   value.
*/
static void mat_calc_cf2_fdcost_vis2(mat_cal_imarec_info *info, mat_vis2 *el, double weight, double wuv, double total)
{ // dkpowQ2, transfer Fortran -> C is correct!
  double         A;
  double complex fd;
  double         fdv;

  /* FORTRAN -> C
     gamma0           -> weight
     cabs(ftok(0,0))  -> total
     pow              -> el->mvis2
     cabs(ftuk)**2    -> el->rvis2
     var              -> el->mvar
     cabs(ft0*ftuk)**2  -> el->rvis2*total*total
   */
  /* FORTRAN:
     powk = cabs(ftok(0,0)) * cabs(ftuk)**2 * gamma0
     A = powk - pow
  */
  A = total*el->rvis2*weight - el->mvis2;
  /* FORTRAN:
     ft0 = ftok(0,0)
     ftC(ux,uy)      += cmplx(A,0.) * ft0 * ftuk * cmplx(2.*wuv,0.)
     ftCvar(ux,uy)   += var * cabs(ft0*ftuk)**2 * wuv**2 * 4.
  */
  fd  = (2.0*A*total*wuv + 0.0*I)*el->ftuk;
  fdv = 4.0*el->mvar*el->mvis2*total*total*wuv*wuv;
  mat_image_add_complex_interpolated(info->fdcost_image,    el->u,  el->v, fd);
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u,  el->v, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u, -el->v, fdv);
  /* FORTRAN:
     ftC(0,0)    += cmplx(A,0.) * ftuk*conjg(ftuk) * cmplx(2.*wuv,0.)
     ftCvar(0,0) += var * cabs(ftuk*conjg(ftuk))**2 * wuv**2 * 4.
  */
  fd  = (2.0*A*wuv + 0.0*I)*el->ftuk*conj(el->ftuk);
  fdv = 4.0*el->mvar*MAT_CHYPOT2(el->ftuk*conj(el->ftuk))*wuv*wuv;
  mat_image_add_complex(info->fdcost_image,    0, 0, fd);
  mat_image_add_double(info->fdcostvar_image,  0, 0, fdv);
}

static void mat_calc_cf2_fdcost_vis(mat_cal_imarec_info *info, mat_vis *el, double weight, double wuvph, double wuvmod)
{ // dkftQ2, transfer Fortran -> C is correct!
  double complex A, B, D;
  double         phivar, modvar;
  double complex fd;
  double         fdv;

  /* FORTRAN -> C
     gamma0                                      -> weight
     ftwert                                      -> el->mvis
     ftwertk                                     -> el->rvis
     ftwert*cmplx(1./(cabs(ftwert)+level),0.)    -> el->mphasor
     ftwertk*cmplx(1./(cabs(ftwertk)+level),0.)  -> el->rphasor
  */
  /* FORTRAN:
     Ck = ftwertk*cmplx(1./(cabs(ftwertk)+level),0.)
     C  = ftwert*cmplx(1./(cabs(ftwert)+level),0.)
     A = cmplx(wuvph/(2.*cabs(ftwertk)+level),0.) * ( cmplx(gamma0,0.)*Ck - C )
     B = conjg(A) * Ck*Ck
     D = cmplx(f0*wuvmod/(cabs(ftwertk)+level),0.) * ( cmplx(gamma0*cabs(ftwertk),0.) - cmplx(cabs(ftwert),0.) ) * ftwertk
     modvar = var - cabs(ftwert)**2*(1.-exp(-phvar))
     with
     phivar := el->mphierr*el->mphierr
     modvar := el->mamperr*el->mamperr
  */
  phivar = el->mphierr*el->mphierr;
  modvar = el->mamperr*el->mamperr;
  A  = (wuvph/(2.0*el->ramp + 1e-30) + 0.0*I)*((weight + 0.0*I)*el->rphasor - el->mphasor);
  B  = conj(A)*el->rphasor*el->rphasor;
  D  = (info->cost_weight*wuvmod/(el->ramp + 1e-30) + 0.0*I)*((weight*el->ramp + 0.0*I) - (el->mamp + 0.0*I))*el->rvis;
  /* FORTRAN:
     ftC(ux,uy)    += A - B + D
     ftCvar(ux,uy) += 0.5*wuvph**2 * (1.-exp(-phvar)) * ( 1. + real(conjg(Ck*Ck)*C*C)*exp(-phvar) ) * (1./(cabs(ftwertk)**2+level)) + f0**2*wuvmod**2*modvar
  */
  fd = A - B + D;
  fdv = 0.5*wuvph*wuvph*(1.0 - exp(-phivar))*(1.0 + creal(conj(el->rphasor*el->rphasor)*el->mphasor*el->mphasor)*exp(-phivar))*(1.0/(el->rvis2 + 1e-30))
    + info->cost_weight*info->cost_weight*wuvmod*wuvmod*modvar;
  mat_image_add_complex_interpolated(info->fdcost_image,    el->u,  el->v, fd);
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u,  el->v, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u, -el->v, fdv);
}

static void mat_calc_cf2_costgrad(mat_cal_imarec_info *info, mat_rec *rec, double total)
{ // costgradbispowftq2, transfer Fortran -> C is correct!
  int    i;
  double weight;

  /* FORTRAN -> C
     gamma0  -> weight
     gamma1  -> total
     f0      -> info->cost_weight
  */
  weight = mat_calc_cf2_cost_weight(info);
  if (info->info_flags & INFO_ITER_DETAILS)
    {
      cpl_msg_info(cpl_func, "   total = %.6f, weight = %.6e", total, weight);
    }
  if (info->use_flags & USE_T3_TABLE)
    {
      info->use_flags |= (USE_BIS_FLAG | USE_CP_FLAG);
      for (i = 0; i < info->nbbis; i++)
	{
	  /* FORTRAN -> C
	     bis                                       -> elbis->mbis
	     ftuk*ftvk*conjg(ftwk)                     -> elbis->rbis
	     bis*cmplx(1./(cabs(bis)+level),0.)        -> elbis->mphasor
	     bisk*cmplx(1./(cabs(bisk)+level),0.)      -> elbis->rphasor
	     gbis0(j)/(abs(cbivar(j))+level)           -> elbis->wuvbis
	     gbis0(j)/(cpperror(j)**2+level)           -> elbis->wuvcp
	     gbis(j)/(1.-exp(-cpvar)+level)            -> elbis->wuvph
	     gbis(j)/(abs(modvar)+level)               -> elbis->wuvmod
	  */
	  mat_bis       *elbis = &(info->bis_list[i]);
	  double         ddcp, wdmod;
	  double complex ddbis, wdph;
	  /* FORTRAN (modified):
	     cpvar = (cpperror(j)*pi/180.) * (cpperror(j)*pi/180.)
	     modvar = cbivar(j) - cabs(bis)**2 * (1.-exp(-cpvar))
	     wuvph  = 12.*gamma0*gbis(j)/(1.-exp(-cpvar)+level)
	     wuvmod = 12.*gamma0*gbis(j)/(abs(modvar)+level)
	     call dkbisQ2( rux,ruy,rvx,rvy, bis,cpvar,modvar, gamma0, wuvph,wuvmod,f0, ftima00, bisk, ftC,ftCvar, id )
	     bisk1 = bisk * cmplx(1./(gamma0*gamma1*gamma1*gamma1),0.)
	     dcp   = phase0(bis*conjg(bisk1))
	     with
	     bisk = ftuk*ftvk*conjg(ftwk) * cmplx(gamma0,0.)
	  */
	  mat_calc_cf2_fdcost_bis(info, elbis, weight, 12.0*weight*elbis->wuvph, 12.0*weight*elbis->wuvmod);
	  wdph    = elbis->rphasor*(weight + 0.0I)                - elbis->mphasor; // weighted phasor difference
	  wdmod   = elbis->ramp*weight                            - elbis->mamp;    // weighted amplitude difference
	  ddbis   = elbis->rbis*(1.0/(total*total*total) + 0.0*I) - elbis->mbis;    // direct bispectrum difference
	  ddcp    = carg(elbis->mphasor*conj(elbis->rphasor));                      // direct closure phase difference
	  // update the global cost value (using the weighted difference)
	  /* FORTRAN (modified):
	     cost    += cabs( bisk*cmplx(gamma0/(cabs(bisk)+level),0.)-bis*cmplx(1./(cabs(bis)+level),0.) )**2*wuvph/gamma0 + f0*abs(cabs(bisk)-cabs(bis))**2*wuvmod/gamma0	
	     sumcost += (1.+f0)*12.*gbis(j)
	  */
	  mat_chi2_add_value(&(rec->c2cost),
			     12.0*MAT_CHYPOT2(wdph)*elbis->wuvph,
			     12.0*elbis->weight);
	  mat_chi2_add_value(&(rec->c2cost),
			     12.0*info->cost_weight*wdmod*wdmod*elbis->wuvmod,
			     12.0*info->cost_weight*elbis->weight);
	  // update the bispectra specific values (using the direct difference)
	  /* FORTRAN (modified):
	     costbisft += 12.* cabs(bisk1-bis)**2    * gbis0(j)/(abs(cbivar(j))+level)
	     sumbisft  += 12.                        * gbis0(j)
	     residual   = 12.*     (bisk1-bis) * sqrt( gbis0(j)/( 0.5*(abs(cbivar(j))+level) ) )
	  */
	  mat_chi2_add_value_residual_complex(&(rec->c2bis),
					      12.0*MAT_CHYPOT2(ddbis)*elbis->wuvbis,
					      12.0*elbis->weight,
					      ddbis*(12.0*sqrt(2.0*elbis->wuvbis) + 0.0*I));
	  // update the closure phase specific values (using the direct difference)
	  /* FORTRAN:
	     costcp   += abs(dcp*180./pi)**2   * gbis0(j)/(cpperror(j)**2+level)
	     sumcp    += gbis0(j)
	     residual  =    (dcp*180./pi) * sqrt(gbis0(j))/(abs(cpperror(j))+level)
	  */
	  mat_chi2_add_value_residual_double(&(rec->c2cp),
					     ddcp*ddcp*elbis->wuvcp,
					     elbis->weight,
					     ddcp*sqrt(elbis->wuvcp));
	}
      if ((info->info_flags & INFO_FITS) && (info->nbit  <= INFO_FITS_MAX_ITER))
	{
	  char   fname[256];
	  snprintf(fname, 256, "fdcost_bis_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
	  mat_image_store_complex(info->fdcost_image, fname, "FREC", info->parlist, info->frameset);
	}
    }
  if (info->use_flags & USE_VIS2_TABLE)
    {
      info->use_flags |= (USE_BIS_FLAG | USE_VIS2_FLAG);
      for (i = 0; i < info->nbvis2; i++)
	{
	  /* FORTRAN -> C
	     v2                             -> elvis2->mvis2
	     cabs(ftuk)**2                  -> elvis2->rvis2
	     gpow(j)/(abs(v2var(j))+level)  -> elvis2->wuvvis2
	   */
	  mat_vis2 *elvis2 = &(info->vis2_list[i]);
	  double    wdvis2, ddvis2;
	  /* FORTRAN (modified):
	     v2 = vis2(j)
	     wuv = f0*6.*gamma0*gpow(j)/(abs(v2var(j))+level)
	     var = v2var(j)
	     call dkpowQ2( rux,ruy, v2,var, gamma0, wuv, ftima00, v2k,  ftC,ftCvar, id )
	     v2k1 = v2k * 1./(gamma0*gamma1*gamma1*gamma1)
	     with
	     v2k = cabs(ftok(0,0)) * cabs(ftuk)**2 * gamma0
	  */
	  mat_calc_cf2_fdcost_vis2(info, elvis2, weight, 6.0*info->cost_weight*weight*elvis2->wuvvis2, total);
	  wdvis2  = weight*total*elvis2->rvis2               - elvis2->mvis2; // weighted squared visibility difference
	  ddvis2  =              elvis2->rvis2/(total*total) - elvis2->mvis2; // direct squared visibility difference
	  // update the global cost value (using the weighted difference)
	  /* FORTRAN (modified):
	     cost    += abs( v2k-v2 )**2 * wuv/gamma0
	     sumcost += 6.*f0*gpow(j)
	  */
	  mat_chi2_add_value(&(rec->c2cost),
			     6.0*wdvis2*wdvis2*info->cost_weight*elvis2->wuvvis2,
			     6.0              *info->cost_weight*elvis2->weight);
	  // update the bispectra specific values (using the direct difference)
	  /* FORTRAN (modified):
	     costbisft += 6.* abs(v2k1-v2)**2    * gpow0(j)/(abs(v2var(j))+level)
	     sumbisft  += 6.                     * gpow0(j)
	     residual   = 6.*    (v2k1-v2) * sqrt( gpow0(j)/(abs(v2var(j))+level) )
	  */
	  mat_chi2_add_value_residual_double(&(rec->c2bis),
					     6.0*ddvis2*ddvis2*elvis2->wuvvis2,
					     6.0*elvis2->weight,
					     6.0*ddvis2*sqrt(elvis2->wuvvis2));
	  // update the squared visibility specific values (using the direct difference)
	  /* FORTRAN (modified):
	     costvis2 += abs(v2k1-v2)**2    * gpow0(j)/(abs(v2var(j))+level)
	     sumvis2  +=                      gpow0(j)
	     residual  = 6.*(v2k1-v2) * sqrt( gpow0(j)/(abs(v2var(j))+level) )
	  */
	  mat_chi2_add_value_residual_double(&(rec->c2vis2),
					     ddvis2*ddvis2*elvis2->wuvvis2,
					     elvis2->weight,
					     6.0*ddvis2*sqrt(elvis2->wuvvis2));
	}
      if ((info->info_flags & INFO_FITS) && (info->nbit  <= INFO_FITS_MAX_ITER))
	{
	  char   fname[256];
	  snprintf(fname, 256, "fdcost_vis2_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
	  mat_image_store_complex(info->fdcost_image, fname, "FREC", info->parlist, info->frameset);
	}
    }
  if (info->use_flags & USE_VIS_TABLE)
    {
      info->use_flags |= (USE_BIS_FLAG | USE_VIS2_FLAG | USE_PHI_FLAG);
      for (i = 0; i < info->nbvis; i++)
	{
	  /* FORTRAN -> C
	    cft                                          -> elvis->mvis
	    ftuk                                         -> elvis->rvis
	    gft(j)/(1.-exp(-phvar)+level)                -> elvis->wuvph
	    gft(j)/(abs(modvar)+level)                   -> elvis->wuvmod
	    gft0(j)/(ftperror**2+level)                  -> elvis->wuvphi
	    cft*cmplx(1./(cabs(cft)+level),0.)           -> elvis->mphasor
	    cftk*cmplx(1./(cabs(cftk)+level),0.)         -> elvis->rphasor
	    cabs(cft)                                    -> elvis->mamp
	    cabs(cftk)                                   -> elvis->ramp*weight
	   */
	  mat_vis        *elvis = &(info->vis_list[i]);
	  double          wdamp, ddphi, ddvis2;
	  double complex  wdph, ddvis;
	  /* FORTRAN (modified):
	    phvar = ftpherr(j) * ftpherr(j)
	    modvar = var-cabs(cft)**2*(1.-exp(-phvar))
	    wuvph  = 6.*gamma0*gft(j)/(1.-exp(-phvar)+level)
	    wuvmod = 6.*gamma0*gft(j)/(abs(modvar)+level)
	    call dkftQ2( rux,ruy, cft,    phvar,var,gamma0, wuvph,wuvmod,f0, ftima00, cftk,    ftC,ftCvar, id )
	    cftk1 = cftk * cmplx(1./(gamma0*gamma1),0.)
	    dcp       = phase0(cft*conjg(cftk1))
	    with
	    cftk = ftuk*gamma0
	  */
	  mat_calc_cf2_fdcost_vis(info, elvis, weight, 6.0*weight*elvis->wuvph, 6.0*weight*elvis->wuvmod);
	  wdph   = elvis->rphasor*(weight + 0.0*I) - elvis->mphasor;  // weighted phasor difference
	  wdamp  = elvis->ramp*weight - elvis->mamp;                  // weighted amplitude difference
	  ddvis  = elvis->rvis*(1.0/total + 0.0*I) - elvis->mvis;     // direct complex visibility difference
	  ddphi  = carg(elvis->mphasor*conj(elvis->rphasor));         // direct phase difference
	  ddvis2 = elvis->rvis2/(total*total)      - elvis->mvis2;    // direct squared visibility difference
	  // update the global cost value (using the weighted difference)
	  /* FORTRAN (modified):
	    cost    = cost + cabs( cftk*cmplx(gamma0/(cabs(cftk)+level),0.)-cft*cmplx(1./(cabs(cft)+level),0.) )**2*wuvph/gamma0 + f0*(cabs(cftk)-cabs(cft))**2*wuvmod/gamma0
	    sumcost = sumcost  + 6.*gft(j)*(1.+f0)
	  */
	  mat_chi2_add_value(&(rec->c2cost),
			     6.0*MAT_CHYPOT2(wdph)*elvis->wuvph,
			     6.0*elvis->weight);
	  mat_chi2_add_value(&(rec->c2cost),
			     6.0*info->cost_weight*wdamp*wdamp*elvis->wuvmod,
			     6.0*elvis->weight*info->cost_weight);
	  // update the bispectra specific values (using the direct difference)
	  /*
	    costbisft += 6.*cabs(cftk1-cft)**2    * gft0(j)/(abs(ftvar(j))+level)
	    sumbisft  += 6.*gft0(j)
	    residual   =        (cftk1-cft) * sqrt( gft0(j)/( 0.5*(abs(ftvar(j))+level) ) )
	  */
	  mat_chi2_add_value_residual_complex(&(rec->c2bis),
					      6.0*MAT_CHYPOT2(ddvis)*elvis->wuvvis,
					      6.0*elvis->weight,
					      ddvis*(sqrt(2.0*elvis->wuvvis) + 0.0*I));
	  // update the squared visibility specific values, currently NOT in FORTRAN, derived from VIS2 part (using the direct difference)
	  mat_chi2_add_value_residual_double(&(rec->c2vis2),
					     ddvis2*ddvis2*elvis->wuvvis2,
					     elvis->weight,
					     6.0*ddvis2*sqrt(elvis->wuvvis2));
	  // update the absolute phase specific values (using the direct difference)
	  /* FORTRAN (slightly modified):
	    costftph += abs(dcp*180./pi)**2 *   gft0(j)/(ftperror**2+level)
	    sumftph  +=                         gft0(j)
	    residual  =    (dcp*180./pi) * sqrt(gft0(j))/(abs(ftperror)+level)
	  */
	  mat_chi2_add_value_residual_double(&(rec->c2phi),
					     ddphi*ddphi*elvis->wuvphi,
					     elvis->weight,
					     ddphi*sqrt(elvis->wuvphi));
	}
      if ((info->info_flags & INFO_FITS) && (info->nbit  <= INFO_FITS_MAX_ITER))
	{
	  char   fname[256];
	  snprintf(fname, 256, "fdcost_vis_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
	  mat_image_store_complex(info->fdcost_image, fname, "FREC", info->parlist, info->frameset);
	}
    }
}

// ******************************************************************************************************
// ***** Cost function 3 using bispectrum and/or fourier spectrum (BIS + VIS2 and/or VIS) phasors only
// ******************************************************************************************************

/**
   @brief Calculate a weight factor such as that the chi2 is minimal for the bispectrum.
   @param info    Contains the image reconstruction context.
   @returns The weight factor.

   The measured and reconstructed closure phases and squared visibilities are used
   to calculate a weight factur such as that the chi squared between reconstruction
   and measurement is minimal. This function is used by the first cost function.

   The weight is calculated using the original T3, VIS2 and VIS weights based on the uv density
   and the variance of the phase or squared visibility.
*/
static double mat_calc_cf3_cost_weight(mat_cal_imarec_info *info)
{ // gamma0bisftQ3, transfer Fortran -> C is correct!
  int    i;
  double sum0 = 0.0;
  double sum1 = 0.0;

  if (info->use_flags & USE_T3_TABLE)
    {
      for (i = 0; i < info->nbbis; i++)
	{
	  /* FORTRAN:
	     bis = cbi(j)
	     bisk = ftuk*ftvk*conjg(ftwk)
	     cpvar = (cpperror(j)*pi/180.) * (cpperror(j)*pi/180.)
	     wuvph = gbis(j)/(1.-exp(-cpvar)+level)
	     C  = bis *cmplx(1./(cabs(bis)+level),0.)
	     Ck = bisk*cmplx(1./(cabs(bisk)+level),0.)
	     sum0 = sum0 + real( C*conjg(Ck) ) * wuvph
	     sum1 = sum1 + wuvph
	  */
	  mat_bis        *el = &(info->bis_list[i]);
	  sum0 += creal(el->rphasor*conj(el->mphasor))*el->wuvph;
	  sum1 +=                                      el->wuvph;
	}
    }
  if (info->use_flags & USE_VIS2_TABLE)
    {
      for (i = 0; i < info->nbvis2; i++)
	{
	  /* FORTRAN:
	     v2 = vis2(j)
	     v2k = cabs(ftuk)**2
	     wuv = gpow(j)/(abs(v2var(j))+level)
	     sum0 = sum0 + f0*v2*v2k*wuv
	     sum1 = sum1 + f0*v2k*v2k*wuv
	  */
	  mat_vis2 *el = &(info->vis2_list[i]);
	  sum0 += info->cost_weight*el->mvis2*el->rvis2*el->wuvvis2;
	  sum1 += info->cost_weight*el->rvis2*el->rvis2*el->wuvvis2;
	}
    }
  if (info->use_flags & USE_VIS_TABLE)
    {
      for (i = 0; i < info->nbvis; i++)
	{
	  /* FORTRAN:
	     cft = ft(j)
	     phvar = ftpherr(j) * ftpherr(j)
	     wuvph = gft(j)/(1.-exp(-phvar)+level)
	     Ck    = cftk * cmplx( 1./(cabs(cftk)+level),0. )
	     C     = cft  * cmplx( 1./(cabs(cft)+level),0. )
	     sum0 = sum0 + real( Ck*conjg(C) ) * wuvph
	     sum1 = sum1 + wuvph
	  */
	  mat_vis *el = &(info->vis_list[i]);
	  sum0 += creal(el->rphasor*conj(el->mphasor))*el->wuvph;
	  sum1 +=                                      el->wuvph;
	}
    }
  return fmax(sum0/(fabs(sum1) + 1e-30), 1e-30);
}

/**
   @brief Updates the gradient using one closure phase element (phasor only).
   @param info    Contains the image reconstruction context.
   @param el      One closure phase element.
   @param weight  Weight for this closure phase element from the reconstruction.
   @param wuvph  Weight for the phase.
   @returns The reconstructed closure phase.

   The reconstructed closure phase (interpolated from the FFT of the previous
   reconstruction at the same (fractional uv coordinates) is calculated.
   This value and the measured closure phase (parameter el) is used to update
   the gradient image. The reconstructed closure phase is returned as function
   value. This function is used by the third cost function.
*/
static void mat_calc_cf3_fdcost_bis(mat_cal_imarec_info *info, mat_bis *el, double weight, double wuvph)
{ // dkbisQ3, transfer Fortran -> C is correct!
  double complex A, fd, C0;
  double         B, D, V, fdv;
  double         cpvar;

  /* FORTRAN:
     bisk = ftuk*ftvk*conjg(ftwk)
     C  = bis*cmplx(1./(cabs(bis)+level),0.)
     Ck = bisk*cmplx(1./(cabs(bisk)+level),0.)
     A  = cmplx(wuvph,0.) * ( cmplx(gamma0,0.)*Ck - C ) * cmplx(1./(cabs(bisk)+level),0.)
     B  = real( bisk * conjg(A) )
     C0 = Ck * conjg(C)
     D  = real( C0*C0 )
     with
     el->mbis    := bis
     el->rbis    := bisk
     el->rphasor := bisk*cmplx(1./(cabs(bisk)+level),0.)
     el->mphasor := bis *cmplx(1./(cabs(bis )+level),0.)
  */
  A  = ((weight + 0.0*I)*el->rphasor - el->mphasor)*(wuvph/(cabs(el->rbis) + 1e-30) + 0.0*I);
  B  = creal(el->rbis*conj(A));
  C0 = el->rphasor*conj(el->mphasor);
  D  = creal(C0*C0);
  /* FORTRAN:
     ftC(ux,uy)    += A *ftwk*conjg(ftvk)  - cmplx( B/(cabs(ftuk)**2+level),0. )*ftuk
     ftCvar(ux,uy) += 0.5*wuvph**2*(1.-exp(-cpvar)) * ( 1. + exp(-cpvar) * D ) * (1./(cabs(ftuk)**2+level))
     with
     cpvar := el->mcperr*el->mcperr
     V     := 0.5*wuvph*wuvph*(1.0 - exp(-cpvar))*(1.0 + exp(-cpvar)*D);
  */
  cpvar = el->mcperr*el->mcperr; // cperr != el->mvar !!!
  V  = 0.5*wuvph*wuvph*(1.0 - exp(-cpvar))*(1.0 + exp(-cpvar)*D);

  fd = A*el->ftwk*conj(el->ftvk) - (B/MAT_CHYPOT2(el->ftuk) + 0.0*I)*el->ftuk;
  mat_image_add_complex_interpolated(info->fdcost_image,  el->u1,  el->v1, fd);
  fdv = V/(MAT_CHYPOT2(el->ftuk) + 1e-30);
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u1,  el->v1, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u1, -el->v1, fdv);

  fd = A*el->ftwk*conj(el->ftuk) - (B/MAT_CHYPOT2(el->ftvk) + 0.0*I)*el->ftvk;
  mat_image_add_complex_interpolated(info->fdcost_image,  el->u2,  el->v2, fd);
  fdv = V/(MAT_CHYPOT2(el->ftvk) + 1e-30);
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u2,  el->v2, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u2, -el->v2, fdv);

  fd = conj(A)*el->ftuk*el->ftvk - (B/MAT_CHYPOT2(el->ftwk) + 0.0*I)*el->ftwk;
  mat_image_add_complex_interpolated(info->fdcost_image,  el->u3,  el->v3, fd);
  fdv = V/(MAT_CHYPOT2(el->ftwk) + 1e-30);
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u3,  el->v3, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u3, -el->v3, fdv);
}

/**
   @brief Updates the gradient using one squared visibility element (phasor only).
   @param info    Contains the image reconstruction context.
   @param el      One squared visibility element.
   @param weight  Weight for this squared visibility element from the reconstruction.
   @param wuv     Weight for this squared visibility element.
   @returns The reconstructed squared visibility.

   The reconstructed squared visibility (interpolated from the FFT of the previous
   reconstruction at the same (fractional uv coordinates) is calculated.
   This value and the measured squared visibility (parameter el) is used to update
   the gradient image. The reconstructed squared visibility is returned as function
   value.
*/
static void mat_calc_cf3_fdcost_vis2(mat_cal_imarec_info *info, mat_vis2 *el, double weight, double wuv)
{ // dkpowQ3, transfer Fortran -> C is correct!
  double         A;
  double complex fd;
  double         fdv;

  /* FORTRAN:
     powk = cabs(ftuk)**2 * gamma0
     A = powk - pow
     with
     pow   := el->mvis2
     powk  := el->rvis2*weight
  */
  A = el->rvis2*weight - el->mvis2;
  /* FORTRAN:
     ftC(ux,uy)    = ftC(ux,uy)      + cmplx(A,0.) * ftuk * cmplx(2.*gew(mu,nu)*wuv,0.)
     ftCvar(ux,uy) = ftCvar(ux,uy)   + var * cabs(ftuk)**2 * wuv**2 * 4. * gew(mu,nu)
     with
     var   := el->mvar
     cabs(ftuk)**2   ->  el->rvis2
     ftuk  := el->ftuk
  */

  fd = el->ftuk*(2.0*A*wuv + 0.0*I);
  mat_image_add_complex_interpolated(info->fdcost_image,  el->u,  el->v, fd);
  fdv = 4.0*el->mvar*el->rvis2*wuv*wuv;
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u,  el->v, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u, -el->v, fdv);
}

static void mat_calc_cf3_fdcost_vis(mat_cal_imarec_info *info, mat_vis *el, double weight, double wuvph)
{ // dkftQ3, transfer Fortran -> C is correct!
  double complex A, B, C, Ck;
  double complex fd;
  double         fdv;
  double         phvar;

  phvar = el->mphierr*el->mphierr;
  /* FORTRAN:
     Ck = ftwertk*cmplx(1./(cabs(ftwertk)+level),0.)
     C  = ftwert *cmplx(1./(cabs(ftwert )+level),0.)
     A  = cmplx(wuvph/(2.*cabs(ftwertk)+level),0.) * ( cmplx(gamma0,0.)*Ck - C )
     B  = conjg(A) * Ck*Ck
     with
     ftwert   := el->mvis;
     ftwertk  := el->rvis;
     ftwert*cmplx(1./(cabs(ftwert)+level),0.)    -> el->mphasor
     ftwertk*cmplx(1./(cabs(ftwertk)+level),0.)  -> el->rphasor
  */
  Ck = el->rphasor;
  C  = el->mphasor;
  A  = (wuvph/(2.0*el->ramp + 1e-30) + 0.0*I)*((weight + 0.0*I)*Ck - C);
  B  = conj(A)*Ck*Ck;
  /* FORTRAN:
     ftC(ux,uy)    = ftC(ux,uy)    + (A - B)
     ftCvar(ux,uy) = ftCvar(ux,uy) + 0.5*wuvph**2 * (1.-exp(-phvar)) * ( 1. + real(conjg(Ck*Ck)*C*C)*exp(-phvar) ) * (1./(cabs(ftwertk)**2+level))
  */

  fd = A - B;
  mat_image_add_complex_interpolated(info->fdcost_image,  el->u,  el->v, fd);
  fdv = 0.5*wuvph*wuvph
    *(1.0 - exp(-phvar))
    *(1.0 + creal(conj(Ck*Ck)*C*C)*exp(-phvar))
    *(1.0/(MAT_CHYPOT2(el->rvis) + 1e-30));
  mat_image_add_double_interpolated(info->fdcostvar_image,  el->u,  el->v, fdv);
  mat_image_add_double_interpolated(info->fdcostvar_image, -el->u, -el->v, fdv);
}

static void mat_calc_cf3_costgrad(mat_cal_imarec_info *info, mat_rec *rec, double total)
{ // costgradbispowftq3, transfer Fortran -> C is correct!
  int    i;
  double weight;

  /* FORTRAN -> C
     gamma0  -> weight
     gamma1  -> total
  */
  weight = mat_calc_cf3_cost_weight(info);
  if (info->info_flags & INFO_ITER_DETAILS)
    {
      cpl_msg_info(cpl_func, "   total = %.6f, weight = %.6e", total, weight);
    }
  if (info->use_flags & USE_T3_TABLE)
    {
      info->use_flags |= (USE_BIS_FLAG | USE_CP_FLAG);
      for (i = 0; i < info->nbbis; i++)
	{
	  /* FORTRAN -> C
	     bis                                   -> elbis->mbis
	     ftuk*ftvk*conjg(ftwk)                 -> elbis->rbis
	     bis*cmplx(1./(cabs(bis)+level),0.)    -> elbis->mphasor
	     bisk*cmplx(1./(cabs(bisk)+level),0.)  -> elbis->rphasor
	     gbis(j)/(1.-exp(-cpvar)+level)        -> elbis->wuvph
	   */
	  mat_bis       *elbis = &(info->bis_list[i]);
	  double         ddcp;
	  double complex wdph, ddph;
	  /* FORTRAN:
	     cpvar = (cpperror(j)*pi/180.) * (cpperror(j)*pi/180.)
	     wuvph = gamma0*gbis(j)/(1.-exp(-cpvar)+level)
	     call dkbisQ3( rux,ruy,rvx,rvy, bis, cpvar, gamma0, wuvph, ftima00, bisk, ftC,ftCvar, id )
	     C   = bis*cmplx(1./(cabs(bis)+level),0.)
	     Ck  = bisk*cmplx(1./(cabs(bisk)+level),0.)
	     dcp = phase0(bis*conjg(bisk))
	     with
	     bisk = ftuk*ftvk*conjg(ftwk) * cmplx(gamma0,0.)
	  */
	  mat_calc_cf3_fdcost_bis(info, elbis, weight, weight*elbis->wuvph);
	  wdph = elbis->rphasor*(weight + 0.0*I) - elbis->mphasor;  // weighted phasor difference
	  ddph = elbis->rphasor                  - elbis->mphasor;  // direct phasor difference
	  ddcp = carg(elbis->mphasor*conj(elbis->rphasor));         // direct closure phase difference
	  // update the global cost value (using the weighted difference)
	  /* FORTRAN (modified):
	     cost    = cost + cabs( cmplx(gamma0,0.)*Ck - C )**2*wuvph/gamma0
	     sumcost = sumcost  + gbis(j)
	  */
	  mat_chi2_add_value(&(rec->c2cost),
			     MAT_CHYPOT2(wdph)*elbis->wuvph,
			     elbis->weight);
	  // update the bispectra specific values from bispectrum phasor (using the direct difference)
	  /* FORTRAN (modified):
	     costbisft += cabs( Ck - C )**2 * gbis0(j)/(1.-exp(-cpvar)+level)
	     sumbisft  += gbis0(j)
	     residual   = [real/aimag]( Ck - C ) * sqrt( 2.0 * gbis0(j)/(1.-exp(-cpvar)+level))
	  */
	  mat_chi2_add_value_residual_complex(&(rec->c2bis),
					      MAT_CHYPOT2(ddph)*elbis->wuvph,
					      elbis->weight,
					      ddph*(sqrt(2.0*elbis->wuvph) + 0.0*I));
	  // update the closure phase specific values (using the direct difference)
	  /* FORTRAN (modified):
	     costcp   += abs(dcp*180./pi)**2   * gbis0(j)/(cpperror(j)**2+level)
	     sumcp    += gbis0(j)
	     residual  =    (dcp*180./pi) * sqrt(gbis0(j))/(abs(cpperror(j))+level)
	  */
	  mat_chi2_add_value_residual_double(&(rec->c2cp),
					     ddcp*ddcp*elbis->wuvcp,
					     elbis->weight,
					     ddcp*sqrt(elbis->wuvcp));
	}
      if ((info->info_flags & INFO_FITS) && (info->nbit  <= INFO_FITS_MAX_ITER))
	{
	  char   fname[256];
	  snprintf(fname, 256, "fdcost_bis_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
	  mat_image_store_complex(info->fdcost_image, fname, "FREC", info->parlist, info->frameset);
	}
    }
  if (info->use_flags & USE_VIS2_TABLE)
    {
      info->use_flags |= (USE_BIS_FLAG | USE_VIS2_FLAG);
      for (i = 0; i < info->nbvis2; i++)
	{
	  /* FORTRAN -> C
	     v2                             -> elvis2->mvis2
	     cabs(ftuk)**2                  -> elvis2->rvis2
	     gpow(j)/(abs(v2var(j))+level)  -> elvis2->wuvvis2
	   */
	  mat_vis2 *elvis2 = &(info->vis2_list[i]);
	  double    wdvis2, ddvis2;
	  /* FORTRAN:
	     wuv = f0*gamma0*gpow(j)/(abs(v2var(j))+level)
	     call dkpowQ3( rux,ruy, v2,var, gamma0, wuv, ftima00, v2k, ftC,ftCvar, id )
	     v2k1 = v2k * 1./(gamma0*gamma1*gamma1)
	     with
	     v2k = cabs(ftok(0,0)) * cabs(ftuk)**2 * gamma0
	  */
	  mat_calc_cf3_fdcost_vis2(info, elvis2, weight, info->cost_weight*weight*elvis2->wuvvis2);
	  wdvis2 = elvis2->rvis2*weight        - elvis2->mvis2; // weighted squared visibility difference
	  ddvis2 = elvis2->rvis2/(total*total) - elvis2->mvis2; // direct squared visibility difference
	  // update the global cost value (using the weighted difference)
	  /* FORTRAN (modified):
	     cost    += abs( v2k-v2 )**2 * wuv/gamma0
	     sumcost += f0*gpow(j)
	  */
	  mat_chi2_add_value(&(rec->c2cost),
			     info->cost_weight*wdvis2*wdvis2*elvis2->wuvvis2,
			     info->cost_weight*elvis2->weight);
	  // update the bispectra values (using the direct difference)
	  /* FORTRAN (modified):
	     costbisft += abs(v2k1-v2)**2 *   gpow0(j)/(abs(v2var(j))+level)
	     sumbisft  +=                     gpow0(j)
	     residual   =    (v2k1-v2)* sqrt( gpow0(j)/(abs(v2var(j))+level) )
	  */
	  mat_chi2_add_value_residual_double(&(rec->c2bis),
					     ddvis2*ddvis2*elvis2->wuvvis2,
					     elvis2->weight,
					     ddvis2*sqrt(elvis2->wuvvis2));
	  // update the squared visibility specific values (using the direct difference)
	  /* FORTRAN (modified):
	     costvis2 += abs(v2k1-v2)**2    * gpow0(j)/(abs(v2var(j))+level)
	     sumvis2  +=                      gpow0(j)
	     residual  =    (v2k1-v2) * sqrt( gpow0(j)/(abs(v2var(j))+level) )
	  */
	  mat_chi2_add_value_residual_double(&(rec->c2vis2),
					     ddvis2*ddvis2*elvis2->wuvvis2,
					     elvis2->weight,
					     ddvis2*sqrt(elvis2->wuvvis2));
	}
      if ((info->info_flags & INFO_FITS) && (info->nbit  <= INFO_FITS_MAX_ITER))
	{
	  char   fname[256];
	  snprintf(fname, 256, "fdcost_vis2_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
	  mat_image_store_complex(info->fdcost_image, fname, "FREC", info->parlist, info->frameset);
	}
    }
  if (info->use_flags & USE_VIS_TABLE)
    {
      info->use_flags |= (USE_BIS_FLAG | USE_PHI_FLAG);
      for (i = 0; i < info->nbvis; i++)
	{
	  /* FORTRAN -> C
	     cft                                  -> elvis->mvis
	     ftuk                                 -> elvis->rvis
	     cft*cmplx(1./(cabs(cft)+level),0.)   -> elvis->mphasor
	     cftk*cmplx(1./(cabs(cftk)+level),0.) -> elvis->rphasor
	     gft(j)/(1.-exp(-phvar)+level)        -> elvis->wuvph
	     gft0(j)/(ftperror**2+level)          -> elvis->wuvphi
	  */
	  mat_vis        *elvis = &(info->vis_list[i]);
	  double          ddphi;
	  double complex  wdph, ddph;
	  /* FORTRAN:
	     phvar = ftpherr(j) * ftpherr(j)
	     wuvph = gamma0*gft(j)/(1.-exp(-phvar)+level)
	     call dkftQ3( rux,ruy, cft,    phvar,gamma0, wuvph, ftima00, cftk,    ftC,ftCvar, id )
	     C   = cft*cmplx(1./(cabs(cft)+level),0.)
	     Ck  = cftk*cmplx(1./(cabs(cftk)+level),0.)
	     dcp = phase0(cft*conjg(cftk))
	     with
	     cftk = ftuk*gamma0
	  */
	  mat_calc_cf3_fdcost_vis(info, elvis, weight, weight*elvis->wuvph);
	  wdph  = elvis->rphasor*(weight + 0.0*I) - elvis->mphasor; // weighted phasor difference
	  ddph  = elvis->rphasor                  - elvis->mphasor; // direct phasor difference
	  ddphi = carg(elvis->mphasor*conj(elvis->rphasor));        // direct phase difference
	  // update the global cost value (using the weighted difference)
	  /* FORTRAN (modified):
	     cost    += cabs( cmplx(gamma0,0.)*Ck - C )**2*wuvph/gamma0
	     sumcost += gft(j)
	  */
	  mat_chi2_add_value(&(rec->c2cost),
			     MAT_CHYPOT2(wdph)*elvis->wuvph,
			     elvis->weight);
	  // update the bispectra values from complex visibility phasor (using the direct difference)
	  /* FORTRAN (modified):
	     costbisft +=         cabs(Ck - C)**2    * gft0(j)/(1.-exp(-phvar)+level)
	     sumbisft  += gft0(j)
	     residual   = [real/aimag](Ck - C) * sqrt( gft0(j)/( 0.5*(1.-exp(-phvar)+level) ) )
	  */
	  mat_chi2_add_value_residual_complex(&(rec->c2bis),
					      MAT_CHYPOT2(ddph)*elvis->wuvph,
					      elvis->weight,
					      ddph*(sqrt(2.0*elvis->wuvph) + 0.0*I));
	  // update the phasor related values (using the direct difference)
	  /* FORTRAN (modified):
	     costftph += abs(dcp*180./pi)**2 * gft0(j)/(ftperror**2+level)
	     sumftph  += gft0(j)
	     residual  = (dcp*180./pi) * sqrt(gft0(j))/(abs(ftperror)+level)
	  */
	  mat_chi2_add_value_residual_double(&(rec->c2phi),
					     ddphi*ddphi*elvis->wuvphi,
					     elvis->weight,
					     ddphi*sqrt(elvis->wuvphi));
	}
      if ((info->info_flags & INFO_FITS) && (info->nbit  <= INFO_FITS_MAX_ITER))
	{
	  char   fname[256];
	  snprintf(fname, 256, "fdcost_vis_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
	  mat_image_store_complex(info->fdcost_image, fname, "FREC", info->parlist, info->frameset);
	}
    }
}

/**
   @brief Calculates value and gradient of a regularization function.
   @param info  Data structure with the image reconstruction context.

   This function uses one of several regularization methods to calculate
   the cost value and gradient of that method. An optional prior can be
   used for all methods except Tikhonov. The numbering of the methods
   follow the schema in tha A&A paper. If a negative number is given,
   a constant prior image is used. The regularization hyperparameter is
   later applied.
   <pre>
   reg_func=0 : no regularization

   reg_func=1 : pixel intensity quadratic:
   H1(x,y)  = |ok(x,y)|^2/prior(x,y)
   dH1(x,y) = 2.*ok(x,y)/prior(x,y)

   reg_func=2 : maximum entropy:
   H2(x,y)  = ok(x,y)*alog(ok(x,y)/prior(x,y)) - ok(x,y) + prior(x,y)
   dH2(x,y) = alog(ok(x,y)) - alog(prior(x,y))

   reg_func=3 : pixel difference quadratic:
   H3(x,y)  = [|ok(x,y)-ok(x+dx,y)|^2 + |ok(x,y)-ok(x,y+dy)|^2] / prior(x,y)
   dH3(x,y) = 2*(ok(x,y) - ok(x+dx,y))/prior(x,y)
   + 2*(ok(x,y) - ok(x,y+dy))/prior(x,y)
   + 2*(ok(x,y) - ok(x-dx,y))/prior(x-dx,y)
   + 2*(ok(x,y) - ok(x,y-dy))/prior(x,y-dy)

   reg_func=4 : edge preserving:
   H4(x,y)  = [sqrt[|ok(x+dx,y)-ok(x,y)|^2 + |ok(x,y+dy)-ok(x,y)|^2 + eps^2]-eps] / prior(x,y)
   dH4(x,y) = 1/[sqrt{|Nabla{ok(x,y)|^2   +eps^2}*prior(x,y)   ] * {-[ok(x+dx,y)-ok(x,y)]-[ok(x,y+dy)-ok(x,y)]}
   + 1/[sqrt{|Nabla{ok(x-dx,y)|^2+eps^2}*prior(x-dx,y)] * [ok(x,y)-ok(x-dx,y)]
   + 1/[sqrt{|Nabla{ok(x,y-dy)|^2+eps^2}*prior(x,y-dy)] * [ok(x,y)-ok(x,y-dy)]
   with Nabla{ok(x,y)|^2 = |ok(x+dx,y)-ok(x,y)|^2 + |ok(x,y+dy)-ok(x,y)|^2

   reg_func=5 : smoothness
   H5(x,y)  = Sum{  |ok(x,y)-ok(x+dx,y+dy)|^2 / prior(x,y), dx={0,1}, dy={0,1}}
   dH5(x,y) = Sum{2*{ok(x,y)-ok(x+dx,y+dy)}   / prior(x,y), dx={0,1}, dy={0,1}}

   reg_func=6 : quadratic Tikhonov:
   H6(x,y)  =   |ok(x,y)-prior(x,y)|^2
   dH6(x,y) = 2*[ok(x,y)-prior(x,y)]

   ok(x,y)    = reconstructed image at (x,y)
   prior(x,y) = rough estimate of the target at (x,y)
   eps        = additional parameter for edge preserving regularization
   dx         = 1
   dy         = 1
   cost       = Sum{H(x,y)}
   </pre>
*/
static double mat_regularization(mat_cal_imarec_info *info)
{
  int     x, y;
  int     dx, dy;
  double  level = 0.0000454; // log(level) = -10;
  double  eps;
  cpl_image *ok0;
  cpl_image *prior;
  cpl_image *dh0;
  double  cost = 0.0;

  // normalize the current reconstruction
  cpl_image_copy(info->nrec_image, info->rec_image, 1, 1);
  mat_image_normalize(info->nrec_image);
  // set the result to 0.0
  mat_image_fill(info->dreg_image, 0.0);
  eps   = info->reg_eps;
  ok0   = info->nrec_image;
  prior = info->prior_image;
  dh0   = info->dreg_image;
  switch (info->reg_func)
    {
    case -1: // quadratic, with prior = 1.0
    case  1: // quadratic
      // H1(x,y)  = |ok(x,y)|^2/prior(x,y)
      // dH1(x,y) = 2.*ok(x,y)/prior(x,y)
      for (y = info->om_ya; y <= info->om_ye; y++)
	{
	  for (x = info->om_xa; x <= info->om_xe; x++)
	    {
	      double o = mat_image_get_double(ok0, x, y);
	      double p = fmax(level, mat_image_get_double(prior, x, y));
	      mat_image_set_double(dh0, x, y, 2.0*o/p);
	      cost += o*o/p;
	    }
	}
      break;
    case -2: // max entropy, with prior = 1.0/npix^2
    case  2: // max entropy 
      // H2(x,y)  = ok(x,y)*alog(ok(x,y)/prior(x,y)) - ok(x,y) + prior(x,y)
      // dH2(x,y) = alog(ok(x,y)) - alog(prior(x,y))
      for (y = info->om_ya; y <= info->om_ye; y++)
	{
	  for (x = info->om_xa; x <= info->om_xe; x++)
	    {
	      // TODO: Check if this is ok!
	      double o = mat_image_get_double(ok0, x, y);
	      double p = mat_image_get_double(prior, x, y);
	      double lo = log(fmax(level, o));
	      double lp = log(fmax(level, p));
              // ok(x,y)=level == alog(level) = -10; mit level = 0.0000454
              // fuer ok(x,y), prior(x,y)<=level wird alog(ok(x,y))=alog(prior(x,y))=-10 gesetzt
              // fuer ok(x,y), prior(x,y)> level wird alog(ok(x,y)) und alog(prior(x,y)) berechnet
	      mat_image_set_double(dh0, x, y, lo - lp);
	      cost += o*(lo - lp) - o + p;
	    }
	}
      break;
    case -3: // smoothness, prior = 1.0
    case  3: // smoothness
      // H3(x,y)  = [|ok(x,y)-ok(x+dx,y)|^2 + |ok(x,y)-ok(x,y+dy)|^2] / prior(x,y)
      // dH3(x,y) = 2*(ok(x,y) - ok(x+dx,y))/prior(x,y)
      //          + 2*(ok(x,y) - ok(x,y+dy))/prior(x,y)
      //          + 2*(ok(x,y) - ok(x-dx,y))/prior(x-dx,y)
      //          + 2*(ok(x,y) - ok(x,y-dy))/prior(x,y-dy)
      for (y = info->om_ya; y <= info->om_ye; y++)
	{
	  for (x = info->om_xa; x <= info->om_xe; x++)
	    {
	      double o  = mat_image_get_double(ok0, x, y);
	      double or = mat_image_get_double(ok0, x + 1, y);
	      double ol = mat_image_get_double(ok0, x - 1, y);
	      double ot = mat_image_get_double(ok0, x, y + 1);
	      double ob = mat_image_get_double(ok0, x, y - 1);
	      double p  = fmax(level, mat_image_get_double(prior, x, y));
	      double pl = fmax(level, mat_image_get_double(prior, x - 1, y));
	      double pb = fmax(level, mat_image_get_double(prior, x, y - 1));
	      mat_image_set_double(dh0, x, y, 2.0*((o - or)/p + (o - ot)/p + (o - ol)/pl + (o - ob)/pb));
	      cost += ((o - or)*(o - or) + (o - ot)*(o - ot))/p;
	    }
	}
      break;
    case -4: // total variation (edge preserving smoothness), prior = 1.0
    case  4: // total variation (edge preserving smoothness)
      // H4(x,y)  = [sqrt[|ok(x+dx,y)-ok(x,y)|^2 + |ok(x,y+dy)-ok(x,y)|^2 + eps^2]-eps] / prior(x,y)
      // dH4(x,y) = 1/[sqrt{|Nabla{ok(x,y)|^2   +eps^2}*prior(x,y)   ] * {-[ok(x+dx,y)-ok(x,y)]-[ok(x,y+dy)-ok(x,y)]}
      //          + 1/[sqrt{|Nabla{ok(x-dx,y)|^2+eps^2}*prior(x-dx,y)] * [ok(x,y)-ok(x-dx,y)]
      //          + 1/[sqrt{|Nabla{ok(x,y-dy)|^2+eps^2}*prior(x,y-dy)] * [ok(x,y)-ok(x,y-dy)]
      // mit |Nabla{ok(x,y)|^2:= |ok(x+dx,y)-ok(x,y)|^2 + |ok(x,y+dy)-ok(x,y)|^2
      for (y = info->om_ya; y <= info->om_ye; y++)
	{
	  for (x = info->om_xa; x <= info->om_xe; x++)
	    {
	      double o   = mat_image_get_double(ok0, x, y);
	      double or  = mat_image_get_double(ok0, x + 1, y);
	      double ol  = mat_image_get_double(ok0, x - 1, y);
	      double ot  = mat_image_get_double(ok0, x, y + 1);
	      double ob  = mat_image_get_double(ok0, x, y - 1);
	      double olt = mat_image_get_double(ok0, x - 1, y + 1);
	      double orb = mat_image_get_double(ok0, x + 1, y - 1);
	      double p   = fmax(level, mat_image_get_double(prior, x, y));
	      double pl  = fmax(level, mat_image_get_double(prior, x - 1, y));
	      double pb  = fmax(level, mat_image_get_double(prior, x, y - 1));
	      double nabla21 = (or - o)*(or - o) + (ot - o)*(ot - o);
	      double nabla22 = (o - ol)*(o - ol) + (olt - ol)*(olt - ol);
	      double nabla23 = (orb - ob)*(orb - ob) + (o - ob)*(o - ob);
	      double f1 = 1./(sqrt(nabla21 + eps*eps)*p);
	      double f2 = 1./(sqrt(nabla22 + eps*eps)*pl);
	      double f3 = 1./(sqrt(nabla23 + eps*eps)*pb);
	      mat_image_set_double(dh0, x, y, f1*( -(or - o)-(ot - o) ) + f2*(o - ol) + f3*(o - ob));
	      cost += (sqrt(nabla21 + eps*eps) - eps)/p;
	    }
	}
      break;
    case -5: // smoothness, with prior = 1.0
    case  5: // smoothness
      // H14(ok(x,y)) = Sum{|ok(x,y)-ok(x+dx,y+dy)|^2} / prior(x,y)
      // dH14 = 4.*{ok(x,y)-ok(x+dx,y+dy)} / prior(x,y)
      for (y = info->om_ya; y <= info->om_ye; y++)
	{
	  for (x = info->om_xa; x <= info->om_xe; x++)
	    {
	      double o = mat_image_get_double(ok0, x, y);
	      double p = fmax(level, mat_image_get_double(prior, x, y));
	      for (dy = -1; dy <= 1; dy++)
		{
		  for (dx = -1; dx <= 1; dx++)
		    {
		      double d = o - mat_image_get_double(ok0, x + dx, y + dy);
		      mat_image_add_double(dh0, x, y, 2.0*d/p);
		      cost += d*d/p;
		    }
		}
	    }
	}
      break;
    case -6: // quadr. Tikhonov, with prior = 1.0/npix^2
    case  6: // quadr. Tikhonov
      // H6(ok(x,z)) = Sum{[|ok(x,y,z)-prior(x,y,z)|^2}
      // dH6:=2*[ok(x,y,z)-prior(x,y,z)]
      for (y = info->om_ya; y <= info->om_ye; y++)
	{
	  for (x = info->om_xa; x <= info->om_xe; x++)
	    {
	      double o  = mat_image_get_double(ok0, x, y);
	      double p  = fmax(level, mat_image_get_double(prior, x, y));
	      mat_image_set_double(dh0, x, y, 2.0*(o - p));
	      cost += (o - p)*(o - p);
	    }
	}
      break;
    default:;
    }
  cpl_image_multiply_scalar(info->dreg_image, info->mu);
  return info->mu*cost;
}

/**
   @brief Calculates the gradient and cost value for the current reconstructed image.
   @param info  Contains the image reconstruction context.
   @param rec   Contains the current reconstruction.

   This function uses the previous reconstruction, the prior image and the regularization method
   to calculate the gradient and cost value needed for ASA-CG to calculate a new reconstruction:

   <ol>
   <li>Calculate a FFT for the previous reconstruction (the start image for the first iteration step).</li>
   <li>Calculate a weight factor such as that the chi2 is minimal for the bispectrum.</li>
   <li>Calculate the gradient (FFT of the gradient) and cost value based on the closure phases (depends on the selected cost function).</li>
   <li>Calculate the gradient (FFT of the gradient) and cost value based on the squared visibilities.</li>
   <li>Calculate the final cost value, sort criteria (qrec) and QC parameters.</li>
   <li>Apply the regularization method.</li>
   <li>Calculate the final gradient image (including regularization).</li>
   </ol>
*/
static void mat_calc_costgrad(mat_cal_imarec_info *info, mat_rec *rec)
{
  double    total;

  mat_cost_info_init(rec);
  // 1. Calculate a FFT for the previous reconstruction (the start image for the first iteration step)
  if ((info->info_flags & INFO_FITS) && (info->nbit <= INFO_FITS_MAX_ITER))
    {
      char   fname[256];
      snprintf(fname, 256, "rec_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
      mat_image_store_double(info->rec_image, fname, "REC", info->parlist, info->frameset);
    }
  mat_image_fft_forward(info->frec_image, info->rec_image);
  total = cabs(mat_image_get_complex(info->frec_image, 0, 0));
  if (total < 1e-10)
    {
      //cpl_msg_info(cpl_func, "warning: total[%d] == %g", info->nbit, total);
      total += 1e-10;
    }
  // 2. Derive the bispectrum and Fourier phases from the current reconstruction
  mat_derive_bis_and_vis(info);
  if ((info->info_flags & INFO_FITS) && (info->nbit <= INFO_FITS_MAX_ITER))
    {
      char   fname[256];
      snprintf(fname, 256, "frec_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
      mat_image_store_complex(info->frec_image, fname, "FREC", info->parlist, info->frameset);
    }
  mat_image_fill(info->fdcost_image, 0.0);
  mat_image_fill(info->fdcostvar_image, 0.0);
  // 3. Calculate a weight factor such as that the chi2 is minimal for the bispectrum and/or Fourier phase (depends on the selected cost function)
  // done inside mat_calc_cf?_costgrad()
  // 4. Calculate the gradient (FFT of the gradient) and cost value based on the closure phases and/or fourier phases (depends on the selected cost function)
  switch (info->cost_func)
    {
    case -1:
    case 1:
      mat_calc_cf1_costgrad(info, rec, total);
      break;
    case 2:
      mat_calc_cf2_costgrad(info, rec, total);
      break;
    case 3:
      mat_calc_cf3_costgrad(info, rec, total);
      break;
    default:;
    }
  // 4d. Calculate the final cost value, sort criteria (qrec) and QC parameters
  mat_cost_info_update_rec(rec);
  rec->cost = rec->c2cost.chi2;
  mat_calc_sort_criteria(rec);
  if (info->info_flags & INFO_ITER_DETAILS)
    {
      cpl_msg_info(cpl_func, "    bis(%g, %g, %g, %g) => chi2 = %f, rres = %f",
		   rec->c2bis.sum, rec->c2bis.weight, rec->c2bis.respos, rec->c2bis.resneg, rec->c2bis.chi2, rec->c2bis.residual);
      cpl_msg_info(cpl_func, "    cp(%g, %g, %g, %g) => chi2 = %f, rres = %f",
		   rec->c2cp.sum, rec->c2cp.weight, rec->c2cp.respos, rec->c2cp.resneg, rec->c2cp.chi2, rec->c2cp.residual);
      cpl_msg_info(cpl_func, "    vis2(%g, %g, %g, %g) => chi2 = %f, rres = %f",
		   rec->c2vis2.sum, rec->c2vis2.weight, rec->c2vis2.respos, rec->c2vis2.resneg, rec->c2vis2.chi2, rec->c2vis2.residual);
      cpl_msg_info(cpl_func, "    vis(%g, %g, %g, %g) => chi2 = %f, rres = %f",
		   rec->c2vis.sum, rec->c2vis.weight, rec->c2vis.respos, rec->c2vis.resneg, rec->c2vis.chi2, rec->c2vis.residual);
      cpl_msg_info(cpl_func, "    amp(%g, %g, %g, %g) => chi2 = %f, rres = %f",
		   rec->c2amp.sum, rec->c2amp.weight, rec->c2amp.respos, rec->c2amp.resneg, rec->c2amp.chi2, rec->c2amp.residual);
      cpl_msg_info(cpl_func, "    phi(%g, %g, %g, %g) => chi2 = %f, rres = %f",
		   rec->c2phi.sum, rec->c2phi.weight, rec->c2phi.respos, rec->c2phi.resneg, rec->c2phi.chi2, rec->c2phi.residual);
      cpl_msg_info(cpl_func, "    cost(%g, %g) => %g",
		   rec->c2cost.sum, rec->c2cost.weight, rec->cost);
      cpl_msg_info(cpl_func, "    qrec = %g", rec->qrec);
    }
  if ((info->info_flags & INFO_FITS) && (info->nbit  <= INFO_FITS_MAX_ITER))
    {
      char   fname[256];
      snprintf(fname, 256, "fdcost_%d_%d_%d.fits",
	       rec->om_idx, rec->mu_idx, info->nbit);
      mat_image_store_complex(info->fdcost_image, fname, "FREC", info->parlist, info->frameset);
    }
  // 7. Wienerfilter (optional)
  if (info->wiener_filter == 1)
    {
      int u, v;
      for (u = -info->npix/2; u < info->npix/2; u++)
	{
	  for (v = -info->npix/2; v < info->npix/2; v++)
	    {
	      double complex fd  = mat_image_get_complex(info->fdcost_image, u, v);
	      double         fdv = mat_image_get_double(info->fdcostvar_image, u, v);
	      if (fdv > 0.0)
		{
		  double snr2 = MAT_CHYPOT2(fd)/fdv;
		  double wf   = snr2/(1.0 + snr2);
		  mat_image_set_complex(info->fdcost_image, u, v, (wf + 0.0*I)*fd);
		}
	    }
	}
    }
  // 8. Calculate the real gradient image
  mat_image_fft_backward(info->dcost_image, info->fdcost_image, 1./rec->c2cost.weight + 0.0*I);
  if ((info->info_flags & INFO_FITS) && (info->nbit  <= INFO_FITS_MAX_ITER))
    {
      char   fname[256];
      snprintf(fname, 256, "dcost_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
      mat_image_store_double(info->dcost_image, fname, "DCOST", info->parlist, info->frameset);
    }
  if ((info->algo_mode & MODE_BISPECTRUM) && (info->cost_func == -1))
    {
      if (rec->cost < 1.0)
	{
	  cpl_image_multiply_scalar(info->dcost_image, -1.0);
	}
      rec->cost = fabs(rec->cost - 1.0);
    }
  // 9. Apply the regularization method
  rec->cost += mat_regularization(info);
  if ((info->info_flags & INFO_FITS) && (info->nbit  <= INFO_FITS_MAX_ITER))
    {
      char  fname[256];
      snprintf(fname, 256, "dreg_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
      mat_image_store_double(info->dreg_image, fname, "DREG", info->parlist, info->frameset);
    }
  // 10. Calculate the final gradient image (including regularization)
  cpl_image_add(info->dcost_image, info->dreg_image);
  cpl_image_multiply(info->dcost_image, info->om_image);
  // 10. Apply an optional filter to the gradient image
  if (info->filter >= info->dxrek )
    {
      mat_image_create_gaussian(info->filter_image, info->filter/info->dxrek);
      mat_image_normalize(info->filter_image);
      mat_image_fft_forward(info->fdcost_image, info->dcost_image);
      mat_image_fft_forward(info->tcpl_image, info->filter_image);
      cpl_image_multiply(info->fdcost_image, info->tcpl_image);
      mat_image_fft_backward(info->dcost_image, info->fdcost_image, 1.0); //1.0/(double)(info->npix*info->npix));
      info->filter *= info->filter_factor;
    }
#ifdef WITH_GRAD_SCALING
  if (info->grad_scale > 1.0)
    {
      cpl_image_multiply_scalar(info->dcost_image, info->grad_scale);
      info->grad_scale *= info->grad_scale_factor;
    }
#endif
  if ((info->info_flags & INFO_FITS) && ((info->nbit  <= INFO_FITS_MAX_ITER) || (info->nbit == 31)))
    {
      char   fname[256];
      snprintf(fname, 256, "grad_%d_%d_%d.fits", rec->om_idx, rec->mu_idx, info->nbit);
      mat_image_swap(info->dcost_image);
      mat_image_store_double(info->dcost_image, fname, "GRAD", info->parlist, info->frameset);
      mat_image_swap(info->dcost_image);
    }
}

/**
   @brief Linearize an image using only the elements inside the object mask.
*/
static cpl_error_code mat_image_to_vector(mat_cal_imarec_info *info, double *dst, cpl_image *src)
{
  int i;

  mat_assert_not_null(info, "no image reconstruction context (info) given)");
  mat_assert_not_null(dst, "no destination vector (dst) given)");
  mat_assert_not_null(src, "no source image (dst) given)");
  for (i = 0; i < info->nbom; i++)
    {
      dst[i] = mat_image_get_double(src, info->om_list_x[i], info->om_list_y[i]);
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Delinearize a vector using only the elements inside the object mask.
*/
static cpl_error_code mat_vector_to_image(mat_cal_imarec_info *info, cpl_image *dst, double *src)
{
  int i;

  mat_assert_not_null(info, "no image reconstruction context (info) given)");
  mat_assert_not_null(dst, "no destination image (dst) given)");
  mat_assert_not_null(src, "no source vector (dst) given)");
  mat_image_fill(dst, 0.0); // clear the image
  for (i = 0; i < info->nbom; i++)
    {
      mat_image_set_double(dst, info->om_list_x[i], info->om_list_y[i], src[i]);
    }
  return CPL_ERROR_NONE;
}

static void mat_show_iter_info(mat_cal_imarec_info *info, mat_rec *rec, char *tag)
{
  char *bis_str;
  char *cp_str;
  char *vis2_str;
  char *phi_str;

  if ((info->info_flags & INFO_ITER_STD) == 0) return;
  if (info->use_flags & USE_BIS_FLAG)
    {
      bis_str = cpl_sprintf(", bis: %10.3f, %10.3f", rec->c2bis.chi2, rec->c2bis.residual);
    }
  else
    {
      bis_str = cpl_strdup("");
    }
  if (info->use_flags & USE_CP_FLAG)
    {
      cp_str = cpl_sprintf(", cp: %10.3f, %10.3f", rec->c2cp.chi2, rec->c2cp.residual);
    }
  else
    {
      cp_str = cpl_strdup("");
    }
  if (info->use_flags & USE_VIS2_FLAG)
    {
      vis2_str = cpl_sprintf(", vis2: %10.3f, %10.3f", rec->c2vis2.chi2, rec->c2vis2.residual);
    }
  else
    {
      vis2_str = cpl_strdup("");
    }
  if (info->use_flags & USE_PHI_FLAG)
    {
      phi_str = cpl_sprintf(", phi: %10.3f, %10.3f", rec->c2phi.chi2, rec->c2phi.residual);
    }
  else
    {
      phi_str = cpl_strdup("");
    }
  cpl_msg_info(cpl_func, "it %4d (%s)  cost=%10.3f qrec=%10.3f%s%s%s%s", info->nbit, tag, rec->cost, rec->qrec, bis_str, cp_str, vis2_str, phi_str);
  cpl_free(bis_str);
  cpl_free(cp_str);
  cpl_free(vis2_str);
  cpl_free(phi_str);

}

/**
   @brief Calculate the cost value for the current reconstruction.
   @param asa  Argument provided from ASA-CG
   @returns Cost value

   This function is the first function provided to ASA-CG.
   It converts the linearized image into a 2-dimensional image,
   calls the mat_calc_costgrad function and returns the cost value.
*/
static double mat_calc_value(asa_objective *asa)
{
  mat_cal_imarec_info  *info = (mat_cal_imarec_info *)(asa->user);
  mat_rec              *rec = info->rec_curr;

  info->nbit++;
  // delinearize the current reconstruction into an image
  mat_vector_to_image(info, info->rec_image, asa->x);
  // calculate the gradient and cost value
  mat_calc_costgrad(info, rec);
  mat_show_iter_info(info, rec, "v");
  // return the cost value
#ifdef WITH_PRECISION
  if (info->precision > 0)
    {
      return mat_round(rec->cost, info->precision);
    }
  else if (info->precision < 0)
    {
      return mat_round_relative(rec->cost, -info->precision);
    }
  else
    {
      return rec->cost;
    }
#else
  return rec->cost;
#endif
}

/**
   @brief Calculate the gradient for the current reconstruction.
   @param asa  Argument provided from ASA-CG

   This function is the second function provided to ASA-CG.
   It converts the linearized image into a 2-dimensional image,
   calls the mat_calc_costgrad function,
   linearize the 2-dimensional gradient and returns the cost value.
*/
static void mat_calc_grad(asa_objective *asa)
{
  mat_cal_imarec_info  *info = (mat_cal_imarec_info *)(asa->user);
  mat_rec              *rec = info->rec_curr;

  info->nbit++;
  // delinearize the current reconstruction into an image
  mat_vector_to_image(info, info->rec_image, asa->x);
  // calculate the gradient and cost value
  mat_calc_costgrad(info, rec);
  mat_show_iter_info(info, rec, "g");
#ifdef WITH_PRECISION
  if (info->precision > 0)
    {
      mat_image_round(info->dcost_image, info->precision);
    }
  else if (info->precision < 0)
    {
      mat_image_round_relative(info->dcost_image, -info->precision);
    }
#endif
  // linearize the gradient image
  mat_image_to_vector(info, asa->g, info->dcost_image);
  return;
}

/**
   @brief Calculate gradient and cost value for the current reconstruction.
   @param asa  Argument provided from ASA-CG
   @returns Cost value

   This function is the third function provided to ASA-CG.
   It converts the linearized image into a 2-dimensional image,
   calls the mat_calc_costgrad function,
   linearize the 2-dimensional gradient and returns the cost value.
*/
static double mat_calc_value_grad_basic(mat_cal_imarec_info *info, mat_rec *rec, double *x, double *g)
{
  info->nbit++;
  // delinearize the current reconstruction into an image
  mat_vector_to_image(info, info->rec_image, x);
  // calculate the gradient and cost value
  mat_calc_costgrad(info, rec);
  mat_show_iter_info(info, rec, "vg");
#ifdef WITH_PRECISION
  if (info->precision > 0)
    {
      mat_image_round(info->dcost_image, info->precision);
    }
  else if (info->precision < 0)
    {
      mat_image_round_relative(info->dcost_image, -info->precision);
    }
#endif
  // linearize the gradient image
  mat_image_to_vector(info, g, info->dcost_image);
  // return the cost value
#ifdef WITH_PRECISION
  if (info->precision > 0)
    {
      return mat_round(rec->cost, info->precision);
    }
  else if (info->precision < 0)
    {
      return mat_round_relative(rec->cost, -info->precision);
    }
  else
    {
      return rec->cost;
    }
#else
  return rec->cost;
#endif
}

static double mat_calc_value_grad(asa_objective *asa)
{
  mat_cal_imarec_info  *info = (mat_cal_imarec_info *)(asa->user);
  mat_rec              *rec = info->rec_curr;

  return mat_calc_value_grad_basic(info, rec, asa->x, asa->g);
}



/*--------------------------------------------------------------------*/
/* Functions for postprocessing and storing the result                */
/*--------------------------------------------------------------------*/

/*
  Beauty Contest Image Distance:

  distbc = Sqrt(Sum(m(i)*(m(i) - r(i))^2,i=1..N)/Sum(m(i),i=1..N))

  m(i) := pixel intensity for the model (reference) image
  r(i) := pixel intensity for the reconstructed image

  The distance formula can be rewritten as:

  distbc = Sqrt(Sum(m(m - r)^2)/Sum(m))
  = Sqrt((Sum(mmm) - 2*Sum(mmr) + Sum(mrr))/Sum(m))

  It is not clear if both images are scaled such that this error measurement is minimal.
  Therefore a scale factor for the reconstruction is introduced and the error minimized for
  a certain value for this scale factor.

  f = scale factor for the reconstruction

  distbc(f) = Sqrt((Sum(mmm) - 2*f*Sum(mmr) + ff*Sum(mrr))/Sum(m))
  distbc(f) != min
  -> Sum(mmm) - 2*f*Sum(mmr) + ff*Sum(mrr) != min
  -> -2*f*Sum(mmr) + ff*Sum(mrr) != min
  -> 2f*Sum(mrr) - 2*Sum(mmr) = 0
  -> f = Sum(mmr)/Sum(mrr)

  introduce f into distbc(f) gets:

  distbc_min = Sqrt((Sum(mmm) - 2*Sum(mmr)/Sum(mrr)*Sum(mmr) + Sum(mmr)/Sum(mrr)*Sum(mmr)/Sum(mrr)*Sum(mrr))/Sum(m))
  = Sqrt((Sum(mmm) - 2*Sum(mmr)*Sum(mmr)/Sum(mrr) + Sum(mmr)*Sum(mmr)/Sum(mrr))/Sum(m))
  = Sqrt((Sum(mmm) - Sum(mmr)*Sum(mmr)/Sum(mrr))/Sum(m))

  Bonn Image Distance:

  dist = Sqrt(Sum((m(i) - r(i))^2,i=1..N)/Sum(m(i)^2,i=1..N))

  The distance formula can be rewritten as:

  dist = Sqrt(Sum((m - r)^2)/Sum(m^2))
  = Sqrt(Sum(mm) - 2*Sum(mr) + Sum(rr))/Sum(mm))

  It is not clear if both images are scaled such that this error measurement is minimal.
  Therefore a scale factor for the reconstruction is introduced and the error minimized for
  a certain value for this scale factor.

  f = scale factor for the reconstruction

  dist(f) = Sqrt(Sum(mm) - 2*f*Sum(mr) + ff*Sum(rr))/Sum(mm))
  dist(f) != min
  -> Sum(mm) - 2*f*Sum(mr) - ff*Sum(rr) != min
  -> -2*f*Sum(mr) + ff*Sum(rr) != min
  -> 2*f*Sum(rr) - 2*Sum(mr) = 0
  -> f = Sum(mr)/Sum(rr)

  introduce f into dist(f) gets:

  dist_min = Sqrt((Sum(mm) - 2*Sum(mr)/Sum(rr)*Sum(mr) + Sum(mr)/Sum(rr)*Sum(mr)/Sum(rr)*Sum(rr))/Sum(mm))
  = Sqrt((Sum(mm) - 2*Sum(mr)*Sum(mr)/Sum(rr) + Sum(mr)*Sum(mr)/Sum(rr))/Sum(mm))
  = Sqrt((Sum(mm) - Sum(mr)*Sum(mr)/Sum(rr))/Sum(mm))
*/

/**
   @brief The reconsructed image is shifted and both distance values are calculated.
   @param info    The image reconstruction context.
   @param rec     The current reconstruction.
   @param dx      The shift in x-direction
   @param dy      The shift in y-direction
   @param distbc  The calculated distance (Beauty contest formula)
   @param dist    The calculated distance (KHH formula)
*/
static void mat_test_rec_model_dist(mat_cal_imarec_info *info, mat_rec *rec, double dx, double dy, double *distbc, double *dist)
{
  int    x, y, rejected;
  // distbc  = Sqrt((Sum(mmm) - Sum(mmr)*Sum(mmr)/Sum(mrr))/Sum(m))
  // dist = Sqrt((Sum(mm) - Sum(mr)*Sum(mr)/Sum(rr))/Sum(mm))
  double smmm = 0.0, smmr = 0.0, smrr = 0.0, sm = 0.0; // needed for distbc
  double smm = 0.0, smr = 0.0, srr = 0.0;              // needed for dist

  mat_image_shift(info->tdbl_image, dx, dy, rec->conv_image);
  for (x = 1; x <= info->npix; x++)
    {
      for (y = 1; y <= info->npix; y++)
	{
	  double   mv = cpl_image_get(info->model_image_convolved, x, y, &rejected);
	  double   rv = cpl_image_get(info->tdbl_image, x, y, &rejected);
	  smmm += mv*mv*mv;
	  smmr += mv*mv*rv;
	  smrr += mv*rv*rv;
	  sm   += mv;
	  smm  += mv*mv;
	  smr  += mv*rv;
	  srr  += rv*rv;
	}
    }
  *distbc  = sqrt((smmm - smmr*smmr/smrr)/sm);
  *dist = sqrt((smm - smr*smr/srr)/smm);
}

/**
   @brief Calculate the smallest distance between a reconstruction and a model by finding an optimal shifted position.
   @param info   The image reconstruction context.
   @param rec    The current reconstruction.

   The method for finding an optimal correlation between a reconstructed
   image and a model image uses a simple schema. First a shift
   delta value is defined and several possible shifts (-2*delta, -2*delta)
   .. (+2*delta,+2*delta) tried. The shift with the smallest
   distance between the shifted reconstruction and the model image
   is then used as a new starting point and the delta value is
   reduced (divided by 2). This method is repeated until the delta
   value is below 0.1 pixel.
*/
static void mat_calc_rec_model_dist(mat_cal_imarec_info *info, mat_rec *rec)
{
  double d = MAT_DELTA_START;
  double dx = 0.0;
  double dy = 0.0;
  double min_dx, min_dy, min_distbc, min_dist;
  int    i, j;

  min_dx = 0.0;
  min_dy = 0.0;
  mat_test_rec_model_dist(info, rec, dx, dy, &min_distbc, &min_dist);
  if (info->info_flags & INFO_DEBUG)
    {
      cpl_msg_info(cpl_func, "reference distance (0,0), distbc = %f, dist = %f", min_distbc, min_dist);
    }
  while (d >= MAT_DELTA_MIN)
    {
      for (i = -MAT_DELTA_STEPS; i <= MAT_DELTA_STEPS; i++)
	{
	  for (j = -MAT_DELTA_STEPS; j <= MAT_DELTA_STEPS; j++)
	    {
	      double distbc, dist;
	      if ((i == 0) && (j == 0)) continue; // We already have this value as reference/minimum value.
	      mat_test_rec_model_dist(info, rec, dx + d*(double)i, dy + d*(double)j, &distbc, &dist);
	      if (dist < min_dist)
		{
		  min_dx = d*(double)i;
		  min_dy = d*(double)j;
		  min_distbc = distbc;
		  min_dist = dist;
		}
	    }
	}
      dx += min_dx;
      dy += min_dy;
      if (info->info_flags & INFO_DEBUG)
	{
	  cpl_msg_info(cpl_func, "   best distance dx = %f, dy = %f, distbc = %f, dist = %f", dx, dy, min_distbc, min_dist);
	}
      // the new shifted position is now the center
      min_dx = 0.0;
      min_dy = 0.0;
      d *= MAT_DELTA_SCALE;
    }
  // store the smallest distance in the reconstruction
  rec->distbc = min_distbc;
  rec->dist = min_dist;
  if (info->info_flags & INFO_DEBUG)
    {
      cpl_msg_info(cpl_func, "model - reconstruction distance : distbc=%.4f, dist=%.4f, shift=%f,%f",
		   rec->distbc, rec->dist, dx, dy);
    }
}

/**
   @brief Shows the cost function results for a specific reconstruction.
   @param info   The image reconstruction context.
   @param rec    The current reconstruction.
*/
static void mat_show_rec_info(mat_cal_imarec_info *info, mat_rec *rec)
{
  char *prefix_str;
  char *bis_str;
  char *cp_str;
  char *vis2_str;
  char *phi_str;
  char *dist_str;

  /* the output string has the following format:
     rec om = %6.3f, mu = %6.4e, qrec = %9.4f, cost = %9.3f, nbit = %4d         <= for all cost functions/algo modes combinations
     , chi2bis = %9.4f, rresbis = %8.4f                                         <= MODE_BISPECTRUM
     , chi2cp = %9.4f, rrescp = %8.4f                                           <= MODE_BISPECTRUM
     , chi2vis2 = %9.4f, rresvis2 = %8.4f                                       <= MODE_BISPECTRUM or cost function == 3
     , chi2phi = %9.4f, rresphi = %8.4f                                         <= MODE_COMPLEX_VIS
     , distbc = %.6f, dist = %.4f                                               <= if a model image is given, the distance is calculated
  */
  if ((info->info_flags & INFO_RESULT) == 0) return;
  prefix_str = cpl_sprintf("rec om = %6.3f, mu = %6.4e, qrec = %9.4f, cost = %9.3f, nbit = %4d",
			   info->om, info->mu, rec->qrec, rec->cost, info->nbit);
  if (info->use_flags & USE_BIS_FLAG)
    {
      bis_str = cpl_sprintf(", chi2bis = %9.4f, rresbis = %8.4f",
			    rec->c2bis.chi2, rec->c2bis.residual);
    }
  else
    {
      bis_str = cpl_strdup("");
    }
  if (info->use_flags & USE_CP_FLAG)
    {
      cp_str = cpl_sprintf(", chi2cp = %9.4f, rrescp = %8.4f", rec->c2cp.chi2, rec->c2cp.residual);
    }
  else
    {
      cp_str = cpl_strdup("");
    }
  if (info->use_flags & USE_VIS2_FLAG)
    {
      vis2_str = cpl_sprintf(", chi2vis2 = %9.4f, rresvis2 = %8.4f", rec->c2vis2.chi2, rec->c2vis2.residual);
    }
  else
    {
      vis2_str = cpl_strdup("");
    }
  if (info->use_flags & USE_PHI_FLAG)
    {
      phi_str = cpl_sprintf(", chi2phi = %9.4f, rresphi = %8.4f", rec->c2phi.chi2, rec->c2phi.residual);
    }
  else
    {
      phi_str = cpl_strdup("");
    }
  if (info->model_image_convolved != NULL)
    {
      dist_str = cpl_sprintf(", distbc = %.6f, dist = %.4f", rec->distbc, rec->dist);
    }
  else
    {
      dist_str = cpl_strdup("");
    }
  cpl_msg_info(cpl_func, "%s%s%s%s%s%s", prefix_str, bis_str, cp_str, vis2_str, phi_str, dist_str);
  cpl_free(prefix_str);
  cpl_free(bis_str);
  cpl_free(cp_str);
  cpl_free(vis2_str);
  cpl_free(phi_str);
  cpl_free(dist_str);
}

/**
   @brief Update the internal data structure of one reconstruction.
   @param info   The image reconstruction context.
   @param rec    The current reconstruction.

   After converting the linearized mage into a 2-dimensional image,
   the reconstructed squared visibilities and closure phases are
   derived and saved. If a model image is given to the plugin, the
   image distance between this model and the reconstruction (convolved)
   is calculated.
*/
static void mat_update_reconstruction(mat_cal_imarec_info *info, mat_rec *rec)
{
  int i;

  rec->nbit = info->nbit;
  mat_image_normalize(info->rec_image);
  // calculate the gradient and cost value
  mat_calc_costgrad(info, rec);
  if (info->use_flags & USE_T3_TABLE)
    {// Copy the reconstructed T3 into reconstruction specific arrays.
      for (i = 0; i < info->nbbis; i++)
	{
	  rec->bis_list[i]  = info->bis_list[i].rbis;
	}
    }
  if (info->use_flags & USE_VIS2_TABLE)
    {// Copy the reconstructed VIS2 into reconstruction specific arrays.
      for (i = 0; i < info->nbvis2; i++)
	{
	  rec->vis2_list[i] = info->vis2_list[i].rvis2;
	}
    }
  if (info->use_flags & USE_VIS_TABLE)
    {// Copy the reconstructed VIS into reconstruction specific arrays.
      for (i = 0; i < info->nbvis; i++)
	{
	  rec->vis_list[i] = info->vis_list[i].rvis;
	}
    }
  //mat_image_normalize(info->rec_image);
  // copy the reconstruction into the list of reconstructions
  rec->rec_image = cpl_image_duplicate(info->rec_image);
  mat_image_convolve(info->tdbl_image, rec->rec_image, info->tent_image, info->frec_image);
  rec->conv_image = cpl_image_duplicate(info->tdbl_image);
  if (info->info_flags & INFO_FITS)
    {
      char fname[256];
      snprintf(fname, 256, "rec_%d_%d.fits", rec->om_idx, rec->mu_idx);
      mat_image_store_double(info->rec_image, fname, "RECONSTR", info->parlist, info->frameset);
      snprintf(fname, 256, "dcost_%d_%d.fits", rec->om_idx, rec->mu_idx);
      mat_image_store_double(info->dcost_image, fname, "GRADIENT", info->parlist, info->frameset);
    }
  mat_calc_rec_model_dist(info, rec);
  mat_show_rec_info(info, rec);
}

/**
   @brief Add WCS keywords to an image for converting pixels into mas.
*/
static cpl_error_code mat_add_wcs_image(mat_cal_imarec_info *info, cpl_propertylist *plist)
{
  cpl_ensure_code((info != NULL), CPL_ERROR_NULL_INPUT);
  cpl_ensure_code((plist != NULL), CPL_ERROR_NULL_INPUT);
  cpl_propertylist_append_string(plist, "CTYPE1", " ");
  cpl_propertylist_append_string(plist, "CUNIT1", "mas");
  cpl_propertylist_append_double(plist, "CRPIX1", 0.5*(double)(info->npix + 1));
  cpl_propertylist_append_double(plist, "CRVAL1", 0.0);
  cpl_propertylist_append_double(plist, "CDELT1", info->dxrek);
  cpl_propertylist_append_string(plist, "CTYPE2", " ");
  cpl_propertylist_append_string(plist, "CUNIT2", "mas");
  cpl_propertylist_append_double(plist, "CRPIX2", 0.5*(double)(info->npix + 1));
  cpl_propertylist_append_double(plist, "CRVAL2", 0.0);
  cpl_propertylist_append_double(plist, "CDELT2", info->dxrek);
  return CPL_ERROR_NONE;
}

/**
   @brief Add WCS keywords to a binary table column for converting pixels into mas.
*/
static cpl_error_code mat_add_wcs_column(mat_cal_imarec_info *info, cpl_propertylist *plist, int col)
{
  
  cpl_ensure_code((info != NULL), CPL_ERROR_NULL_INPUT);
  cpl_ensure_code((plist != NULL), CPL_ERROR_NULL_INPUT);

  char *kname = NULL;
  kname = cpl_sprintf("1CTYP%d", col + 1); cpl_propertylist_append_string(plist, kname, " ");                           ; cpl_free(kname);
  kname = cpl_sprintf("1CUNI%d", col + 1); cpl_propertylist_append_string(plist, kname, "mas");                         ; cpl_free(kname);
  kname = cpl_sprintf("1CRPX%d", col + 1); cpl_propertylist_append_double(plist, kname, 0.5*(double)(info->npix + 1));  ; cpl_free(kname);
  kname = cpl_sprintf("1CRVL%d", col + 1); cpl_propertylist_append_double(plist, kname, 0.0);                           ; cpl_free(kname);
  kname = cpl_sprintf("1CDLT%d", col + 1); cpl_propertylist_append_double(plist, kname, info->dxrek);                   ; cpl_free(kname);
  kname = cpl_sprintf("2CTYP%d", col + 1); cpl_propertylist_append_string(plist, kname, " ");                           ; cpl_free(kname);
  kname = cpl_sprintf("2CUNI%d", col + 1); cpl_propertylist_append_string(plist, kname, "mas");                         ; cpl_free(kname);
  kname = cpl_sprintf("2CRPX%d", col + 1); cpl_propertylist_append_double(plist, kname, 0.5*(double)(info->npix + 1));  ; cpl_free(kname);
  kname = cpl_sprintf("2CRVL%d", col + 1); cpl_propertylist_append_double(plist, kname, 0.0);                           ; cpl_free(kname);
  kname = cpl_sprintf("2CDLT%d", col + 1); cpl_propertylist_append_double(plist, kname, info->dxrek);                   ; cpl_free(kname);

  return CPL_ERROR_NONE;
}


static cpl_error_code mat_store_suggested_parameters(mat_cal_imarec_info *info,
						     const char *rname,
						     cpl_parameterlist *parlist,
						     cpl_frameset *frameset,
						     cpl_frame *recframe)
{
  cpl_propertylist *plist = NULL;

  /* the primary header contains only the best reconstruction and the QC keywords */
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  if (cpl_dfs_setup_product_header(plist, recframe, frameset, parlist,
				   rname, PACKAGE "/" PACKAGE_VERSION, "?Dictionary?", NULL) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_error_reset();
    cpl_propertylist_delete(plist);
    return CPL_ERROR_UNSPECIFIED;
  }
  // Add the values from the model fit and the suggested parameter values
  cpl_propertylist_append_double(plist, "ESO PARAM FIT GDFWHM",  mat_round(info->fit_gd_fwhm, MAT_QC_PREC));
  cpl_propertylist_append_double(plist, "ESO PARAM FIT UDDIAM",  mat_round(info->fit_ud_diameter, MAT_QC_PREC));
  cpl_propertylist_append_double(plist, "ESO PARAM FIT FDDDIAM", mat_round(info->fit_fdd_diameter, MAT_QC_PREC));
  cpl_propertylist_append_double(plist, "ESO PARAM FIT LDFWHM",  mat_round(info->fit_ld_fwhm, MAT_QC_PREC));
  cpl_propertylist_append_double(plist, "ESO PARAM GUESS FOV",   mat_round(info->FOV_guess, MAT_QC_PREC));
  cpl_propertylist_append_double(plist, "ESO PARAM GUESS NPIX",  mat_round(info->npix_guess, MAT_QC_PREC));
  cpl_propertylist_append_double(plist, "ESO PARAM GUESS MODE",  mat_round(info->start_mode_guess, MAT_QC_PREC));
  cpl_propertylist_append_double(plist, "ESO PARAM GUESS PARAM", mat_round(info->start_param_guess, MAT_QC_PREC));
  // Save the property list to a FITS file
  char *fname=cpl_sprintf("rec_%s.fits", info->tname);
  if (cpl_propertylist_save(plist, fname, CPL_IO_CREATE) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_propertylist_delete(plist);
      cpl_free(fname); fname = NULL;
      return CPL_ERROR_UNSPECIFIED;
    }
  cpl_free(fname); fname = NULL;
  cpl_propertylist_delete(plist);
  return CPL_ERROR_NONE;
}

/**
   @brief Stores the best reconstruction (direct and convolved) and the QC parameters ina FITS file.
*/
static cpl_error_code mat_store_best_reconstruction(mat_cal_imarec_info *info,
						    const char *rname,
						    cpl_parameterlist *parlist,
						    cpl_frameset *frameset,
						    cpl_frame *recframe)
{

  cpl_propertylist *plist = NULL;
  mat_rec          *rec;

  /* the primary header contains only the best reconstruction and the QC keywords */
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  if (cpl_dfs_setup_product_header(plist, recframe, frameset, parlist,
				   rname, PACKAGE "/" PACKAGE_VERSION, "?Dictionary?", NULL) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_error_reset();
    cpl_propertylist_delete(plist);
    return CPL_ERROR_UNSPECIFIED;
  }
  /* add the QC values for the best reconstruction */
  rec = info->rec_sorted[0];
  //cpl_propertylist_append_double(plist, "ESO QC REC OM", rec->om);
  //cpl_propertylist_append_double(plist, "ESO QC REC MU", rec->mu);
  cpl_propertylist_append_double(plist, "ESO QC REC QREC", mat_round(rec->qrec, MAT_QC_PREC));
  cpl_propertylist_append_double(plist, "ESO QC REC COST", mat_round(rec->cost, MAT_QC_PREC));
  if (info->use_flags & USE_BIS_FLAG)
    {
      cpl_propertylist_append_double(plist, "ESO QC REC CHI2BIS", mat_round(rec->c2bis.chi2, MAT_QC_PREC));
      cpl_propertylist_append_double(plist, "ESO QC REC RRESBIS", mat_round(rec->c2bis.residual, MAT_QC_PREC));
    }
  if (info->use_flags & USE_CP_FLAG)
    {
      cpl_propertylist_append_double(plist, "ESO QC REC CHI2CP", mat_round(rec->c2cp.chi2, MAT_QC_PREC));
      cpl_propertylist_append_double(plist, "ESO QC REC RRESCP", mat_round(rec->c2cp.residual, MAT_QC_PREC));
    }
  if (info->use_flags & USE_VIS2_FLAG)
    {
      cpl_propertylist_append_double(plist, "ESO QC REC CHI2VIS2", mat_round(rec->c2vis2.chi2, MAT_QC_PREC));
      cpl_propertylist_append_double(plist, "ESO QC REC RRESVIS2", mat_round(rec->c2vis2.residual, MAT_QC_PREC));
    }
  if (info->use_flags & USE_PHI_FLAG)
    {
      cpl_propertylist_append_double(plist, "ESO QC REC CHI2PHI", mat_round(rec->c2phi.chi2, MAT_QC_PREC));
      cpl_propertylist_append_double(plist, "ESO QC REC RRESPHI", mat_round(rec->c2phi.residual, MAT_QC_PREC));
    }
  mat_add_wcs_image(info, plist);
  /* Save the primary header unit to the file (best unconvolved reconstruction)*/
  char *fname=cpl_sprintf("rec_%s.fits", info->tname);
  if (cpl_image_save(rec->rec_image, fname, CPL_TYPE_FLOAT, plist,
		     CPL_IO_CREATE) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_free(fname); fname = NULL;
    cpl_propertylist_delete(plist);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  /* this HDU contains the best reconstruction (convolved) */
  plist = cpl_propertylist_new();
  cpl_propertylist_append_string(plist, "EXTNAME", "REC_CONV");
  mat_add_wcs_image(info, plist);
  if (cpl_image_save(rec->conv_image, fname, CPL_TYPE_FLOAT, plist,
  		     CPL_IO_EXTEND) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
  		  cpl_error_get_code(),
  		  cpl_error_get_message());
    cpl_free(fname); fname = NULL;
    cpl_propertylist_delete(plist);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_free(fname); fname = NULL;
  cpl_propertylist_delete(plist);
  return CPL_ERROR_NONE;
}

/**
   @brief Appends a binary table with the sorted reconstructions to a FITS file.
*/
static cpl_error_code mat_store_reconstruction_list(mat_cal_imarec_info *info)
{
  cpl_propertylist *plist = NULL;
  cpl_table        *table = NULL;
  cpl_array        *arr = NULL;
  int               i, nr, nx, ny, x, y;
  int               rec_col, conv_col;
  mat_rec          *rec;

  if (info->nbresult < 0)
    {
      nr = info->om_count*info->mu_count;
    }
  else
    {
      nr = info->nbresult;
    }
  /* this HDU contains the sorted/unsorted list of some/all reconstructions */
  /* create the binary table containing the QC values and the reconstructions */
  nx = info->npix;
  ny = info->npix;
  table = cpl_table_new(nr);
  if (table == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the table for the reconstructions, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      return ec;
    }
  if (cpl_table_new_column(table, "OM", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the object mask radius column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "MU", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the regularization factor column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "NBIT", CPL_TYPE_INT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the number of iterations column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "QREC", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the reconstruction quality column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "COST", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the cost value column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (info->use_flags & USE_BIS_FLAG)
    {
      if (cpl_table_new_column(table, "CHI2BIS", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the chi2 (bispectrum) column, code = %d, message = %s",
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
      if (cpl_table_new_column(table, "RRESBIS", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the residual ratio (bispectrum) column, code = %d, message = %s",
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
    }
  if (info->use_flags & USE_CP_FLAG)
    {
      if (cpl_table_new_column(table, "CHI2CP", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the chi2 (closure phase) column, code = %d, message = %s",
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
      if (cpl_table_new_column(table, "RRESCP", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the residual ratio (closure phase) column, code = %d, message = %s",
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
    }
  if (info->use_flags & USE_VIS2_FLAG)
    {
      if (cpl_table_new_column(table, "CHI2VIS2", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the chi2 (visibility) column, code = %d, message = %s",
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
      if (cpl_table_new_column(table, "RRESVIS2", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the residual ratio (visibility) column, code = %d, message = %s",
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
    }
  if (info->use_flags & USE_PHI_FLAG)
    {
      if (cpl_table_new_column(table, "CHI2PHI", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the chi2 (complex visibility phase) column, code = %d, message = %s",
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
      if (cpl_table_new_column(table, "RRESPHI", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the residual ratio (complex visibility phase) column, code = %d, message = %s",
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
    }
  rec_col = cpl_table_get_ncol(table);
  if (cpl_table_new_column_array(table, "REC", CPL_TYPE_FLOAT, nx*ny) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the reconstruction image column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  conv_col = cpl_table_get_ncol(table);
  if (cpl_table_new_column_array(table, "CONV", CPL_TYPE_FLOAT, nx*ny) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the convolved reconstruction image column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  arr = cpl_array_new(2, CPL_TYPE_INT);
  if (arr != NULL)
    {
      cpl_array_set(arr, 0, nx);
      cpl_array_set(arr, 1, ny);
      cpl_table_set_column_dimensions(table, "REC", arr);
      cpl_table_set_column_dimensions(table, "CONV", arr);
      cpl_array_delete(arr);
      arr = NULL;
    }
  else
    {
      cpl_msg_error(cpl_func, "cannot set the dimension of both reconstruction images column");
    }
  // Set the dimensions for both images
  arr = cpl_array_new(nx*ny, CPL_TYPE_FLOAT);
  if (arr == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the image column array, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  for (i = 0; i < nr; i++)
    {
      if (info->nbresult < 0)
	{
	  rec = &(info->rec_list[i]);
	}
      else
	{
	  rec = info->rec_sorted[i];
	}
      cpl_table_set_float(table, "OM", i, (float)rec->om);
      cpl_table_set_float(table, "MU", i, (float)rec->mu);
      cpl_table_set_int(table, "NBIT", i, rec->nbit);
      cpl_table_set_float(table, "QREC", i, (float)mat_round(rec->qrec, MAT_QC_PREC));
      cpl_table_set_float(table, "COST", i, (float)mat_round(rec->cost, MAT_QC_PREC));
      if (info->use_flags & USE_BIS_FLAG)
	{
	  cpl_table_set_float(table, "CHI2BIS",  i, (float)mat_round(rec->c2bis.chi2, MAT_QC_PREC));
	  cpl_table_set_float(table, "RRESBIS",  i, (float)mat_round(rec->c2bis.residual, MAT_QC_PREC));
	}
      if (info->use_flags & USE_CP_FLAG)
	{
	  cpl_table_set_float(table, "CHI2CP",   i, (float)mat_round(rec->c2cp.chi2, MAT_QC_PREC));
	  cpl_table_set_float(table, "RRESCP",   i, (float)mat_round(rec->c2cp.residual, MAT_QC_PREC));
	}
      if (info->use_flags & USE_VIS2_FLAG)
	{
	  cpl_table_set_float(table, "CHI2VIS2", i, (float)mat_round(rec->c2vis2.chi2, MAT_QC_PREC));
	  cpl_table_set_float(table, "RRESVIS2", i, (float)mat_round(rec->c2vis2.residual, MAT_QC_PREC));
	}
      if (info->use_flags & USE_PHI_FLAG)
	{
	  cpl_table_set_float(table, "CHI2PHI",  i, (float)mat_round(rec->c2phi.chi2, MAT_QC_PREC));
	  cpl_table_set_float(table, "RRESPHI",  i, (float)mat_round(rec->c2phi.residual, MAT_QC_PREC));
	}
      for (x = 0; x < nx; x++)
	{
	  for (y = 0; y < ny; y++)
	    {
	      cpl_array_set(arr, y*nx + x, mat_image_get_double(rec->rec_image, x, y));
	    }
	}
      cpl_table_set_array(table, "REC", i, arr);
      for (x = 0; x < nx; x++)
	{
	  for (y = 0; y < ny; y++)
	    {
	      cpl_array_set(arr, y*nx + x, mat_image_get_double(rec->conv_image, x, y));
	    }
	}
      cpl_table_set_array(table, "CONV", i, arr);
    }
  cpl_array_delete(arr);
  /* store the binary table extension containing the statistical data */
  plist = cpl_propertylist_new();
  mat_add_wcs_column(info, plist, rec_col);
  mat_add_wcs_column(info, plist, conv_col);
  cpl_propertylist_append_string(plist, "EXTNAME", "REC_LIST");
  char *fname=cpl_sprintf("rec_%s.fits", info->tname);
  if (cpl_table_save(table, NULL, plist, fname, CPL_IO_EXTEND) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_free(fname); fname = NULL;
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_free(fname); fname = NULL;
  cpl_propertylist_delete(plist);
  cpl_table_delete(table);
  return CPL_ERROR_NONE;
}

/**
   @brief Appends all additional images (for example the uv coverage) to a FITS file.
*/
static cpl_error_code mat_store_additional_images(mat_cal_imarec_info *info)
{
  cpl_propertylist *plist = NULL;
  //int               i, j, n, nr, nx, ny, x, y;
  char *fname=cpl_sprintf("rec_%s.fits", info->tname);
  /* this HDU contains the uv coverage */
  if (info->uv_image != NULL)
    {
      plist = cpl_propertylist_new();
      cpl_propertylist_append_string(plist, "EXTNAME", "UV_COVERAGE");
      if (cpl_image_save(info->uv_image, fname, CPL_TYPE_FLOAT, plist,
			 CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_propertylist_delete(plist);
	cpl_free(fname); fname = NULL;
	return CPL_ERROR_UNSPECIFIED;
      }
      cpl_propertylist_delete(plist);
    }
  /* this HDU contains the loaded/created start image */
  if (info->start_image_loaded != NULL)
    {
      plist = cpl_propertylist_new();
      cpl_propertylist_append_string(plist, "EXTNAME", "START_IMAGE");
      mat_add_wcs_image(info, plist);
      if (cpl_image_save(info->start_image_loaded, fname, CPL_TYPE_FLOAT, plist,
			 CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_propertylist_delete(plist);
	cpl_free(fname); fname = NULL;
	return CPL_ERROR_UNSPECIFIED;
      }
      cpl_propertylist_delete(plist);
    }
  /* this HDU contains the loaded/created prior image */
  if (info->prior_image_loaded != NULL)
    {
      plist = cpl_propertylist_new();
      cpl_propertylist_append_string(plist, "EXTNAME", "PRIOR_IMAGE");
      mat_add_wcs_image(info, plist);
      if (cpl_image_save(info->prior_image_loaded, fname, CPL_TYPE_FLOAT, plist,
			 CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_propertylist_delete(plist);
	cpl_free(fname); fname = NULL;
	return CPL_ERROR_UNSPECIFIED;
      }
      cpl_propertylist_delete(plist);
    }
  /* this HDU contains the object masks */
  if (info->om_image_list != NULL)
    {
      plist = cpl_propertylist_new();
      cpl_propertylist_append_string(plist, "EXTNAME", "OBJECT_MASK");
      mat_add_wcs_image(info, plist);
      if (cpl_imagelist_save(info->om_image_list, fname, CPL_TYPE_FLOAT, plist,
			     CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_propertylist_delete(plist);
	cpl_free(fname); fname = NULL;
	return CPL_ERROR_UNSPECIFIED;
      }
      cpl_propertylist_delete(plist);
    }
  /* this HDU contains the optional scaled model image */
  if (info->model_image_loaded != NULL)
    {
      plist = cpl_propertylist_new();
      cpl_propertylist_append_string(plist, "EXTNAME", "MODEL_IMAGE");
      mat_add_wcs_image(info, plist);
      if (cpl_image_save(info->model_image_loaded, fname, CPL_TYPE_FLOAT, plist,
			 CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_propertylist_delete(plist);
	cpl_free(fname); fname = NULL;
	return CPL_ERROR_UNSPECIFIED;
      }
      cpl_propertylist_delete(plist);
    }
  /* this HDU contains the optional scaled model image (convolved) */
  if (info->model_image_convolved != NULL)
    {
      plist = cpl_propertylist_new();
      cpl_propertylist_append_string(plist, "EXTNAME", "MODEL_CONV");
      mat_add_wcs_image(info, plist);
      if (cpl_image_save(info->model_image_convolved, fname, CPL_TYPE_FLOAT, plist,
			 CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_propertylist_delete(plist);
	cpl_free(fname); fname = NULL;
	return CPL_ERROR_UNSPECIFIED;
      }
      cpl_propertylist_delete(plist);

    }
  cpl_free(fname); fname = NULL;
  return CPL_ERROR_NONE;
}

/**
   @brief Append the measured and reconstructed squared visibilities to a FITS file.
*/
static cpl_error_code mat_store_reconstructed_vis2_table(mat_cal_imarec_info *info)
{
  cpl_propertylist *plist = NULL;
  cpl_table        *table = NULL;
  int               i, j, nr;
  mat_rec          *rec;

  if (info->nbresult < 0)
    {
      nr = info->om_count*info->mu_count;
    }
  else
    {
      nr = info->nbresult;
    }
  /* this HDU contains the measured squared visibilities and the squared visibilities derived from the reconstructions */
  table = cpl_table_new(info->nbvis2);
  if (table == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the table for the derived squared visibilities, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      return ec;
    }
  if (cpl_table_new_column(table, "LAMBDA", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the wavelength column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "UCOORD", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the u-coordinate column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "VCOORD", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the v-coordinate column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "BL", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the baseline column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "VIS2DATA", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the squared visibility column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "VIS2ERR", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the squared visibility error column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "GDF", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the gaussian disc fit column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "UDF", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the uniform disc fit column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "FDDF", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the fully darkened disc fit column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "LDF", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the Lorentz disc fit column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  cpl_table_set_column_unit(table, "LAMBDA", "m");
  cpl_table_set_column_unit(table, "UCOORD", "1/mas");
  cpl_table_set_column_unit(table, "VCOORD", "1/mas");
  cpl_table_set_column_unit(table, "BL",     "m");
  for (i = 0; i < nr; i++)
    { // create a column for each reconstruction (same order as in RECLIST)
      char cname[256];
      snprintf(cname, 256, "VIS2DATA%d", i + 1);
      if (cpl_table_new_column(table, cname, CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the %d derived squared visibility column, code = %d, message = %s",
			i + 1,
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
    }
  for (i = 0; i < info->nbvis2; i++)
    {
      mat_vis2 *el     = &(info->vis2_list[i]);
      double    r      = sqrt(el->u*el->u + el->v*el->v);
      double    f      = r/info->FOV;
      double    gd_v2  = 0.0;
      double    ud_v2  = 0.0;
      double    fdd_v2 = 0.0;
      double    ld_v2  = 0.0;
      double    a;
      a = info->fit_gd_fwhm*FWHM2SIGMA;
      mat_fit_gauss_f(&f, &a, &gd_v2);
      a = 0.5*info->fit_ud_diameter;
      mat_fit_ud_f(&f, &a, &ud_v2);
      a = 0.5*info->fit_fdd_diameter;
      mat_fit_fdd_f(&f, &a, &fdd_v2);
      a = FWHM2LDA/info->fit_ld_fwhm;
      mat_fit_ld_f(&f, &a, &ld_v2);
      cpl_table_set_double(table, "LAMBDA",   i, el->lambda);
      cpl_table_set_double(table, "UCOORD",   i, el->u/info->FOV);
      cpl_table_set_double(table, "VCOORD",   i, el->v/info->FOV);
      cpl_table_set_double(table, "BL",       i, el->bl);
      cpl_table_set_double(table, "VIS2DATA", i, el->mvis2);
      cpl_table_set_double(table, "VIS2ERR",  i, sqrt(el->mvar));
      cpl_table_set_double(table, "GDF",      i, gd_v2);
      cpl_table_set_double(table, "UDF",      i, ud_v2);
      cpl_table_set_double(table, "FDDF",     i, fdd_v2);
      cpl_table_set_double(table, "LDF",      i, ld_v2);
    }
  for (i = 0; i < nr; i++)
    {
      char cname[256];
      snprintf(cname, 256, "VIS2DATA%d", i + 1);
      if (info->nbresult < 0)
	{
	  rec = &(info->rec_list[i]);
	}
      else
	{
	  rec = info->rec_sorted[i];
	}
      for (j = 0; j < info->nbvis2; j++)
	{
	  cpl_table_set_double(table, cname, j, rec->vis2_list[j]);
	}
    }
  plist = cpl_propertylist_new();
  cpl_propertylist_append_string(plist, "EXTNAME", "REC_VIS2");
  char *fname=cpl_sprintf("rec_%s.fits", info->tname);
  if (cpl_table_save(table, NULL, plist, fname, CPL_IO_EXTEND) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_table_delete(table);
    cpl_free(fname); fname = NULL;
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_free(fname); fname = NULL;
  cpl_propertylist_delete(plist);
  cpl_table_delete(table);
  return CPL_ERROR_NONE;
}

/**
   @brief Stores the measured and reconstructed squared visibilities in an ASCII-file.
*/
static cpl_error_code mat_store_reconstructed_vis2_ascii(mat_cal_imarec_info *info)
{
  int               i, j, nr;
  mat_rec          *rec;
  FILE             *dat;

  if (info->vis2_name[0] == '\0') return CPL_ERROR_NONE;
  if (info->nbresult < 0)
    {
      nr = info->om_count*info->mu_count;
    }
  else
    {
      nr = info->nbresult;
    }
  // write the measured and reconstructed squared visibilities into a file
  dat = fopen(info->vis2_name, "w");
  if (dat != NULL)
    {
      fprintf(dat, "# measured squared visibilities vs reconstructed squared visibilities\n");
      fprintf(dat, "# nr lambda u v bl vis2 err gd ud fdd ld");
      for (j = 0; j < nr; j++)
	{
	  fprintf(dat, " rec%d", j + 1);
	}
      fprintf(dat, "\n");
      for (j = 0; j < nr; j++)
	{
	  if (info->nbresult < 0)
	    {
	      rec = &(info->rec_list[j]);
	    }
	  else
	    {
	      rec = info->rec_sorted[j];
	    }
	  fprintf(dat, "# qrec%d %.6f\n", j+1, rec->qrec);
	}
      for (i = 0; i < info->nbvis2; i++)
	{
	  mat_vis2 *el     = &(info->vis2_list[i]);
	  double    r      = sqrt(el->u*el->u + el->v*el->v);
	  double    f      = r/info->FOV;
	  double    gd_v2  = 0.0;
	  double    ud_v2  = 0.0;
	  double    fdd_v2 = 0.0;
	  double    ld_v2  = 0.0;
	  double    a;
	  a = info->fit_gd_fwhm*FWHM2SIGMA;
	  mat_fit_gauss_f(&f, &a, &gd_v2);
	  a = 0.5*info->fit_ud_diameter;
	  mat_fit_ud_f(&f, &a, &ud_v2);
	  a = 0.5*info->fit_fdd_diameter;
	  mat_fit_fdd_f(&f, &a, &fdd_v2);
	  a = FWHM2LDA/info->fit_ld_fwhm;
	  mat_fit_ld_f(&f, &a, &ld_v2);
	  fprintf(dat, "%d %g %.6f %.6f %.6f %.6f %.6f %.6f %.6f %.6f %.6f",
		  i,
		  el->lambda,
		  el->u/info->FOV,
		  el->v/info->FOV,
		  el->bl,
		  el->mvis2,
		  sqrt(el->mvar),
		  gd_v2,
		  ud_v2,
		  fdd_v2,
		  ld_v2);
	  for (j = 0; j < nr; j++)
	    {
	      if (info->nbresult < 0)
		{
		  rec = &(info->rec_list[j]);
		}
	      else
		{
		  rec = info->rec_sorted[j];
		}
	      fprintf(dat, " %.6f", rec->vis2_list[i]);
	    }
	  fprintf(dat, "\n");
	}
      fclose(dat);
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Append the measured and reconstructed closure phases to a  FITS file.
*/
static cpl_error_code mat_store_reconstructed_t3_table(mat_cal_imarec_info *info)
{
  cpl_propertylist *plist = NULL;
  cpl_table        *table = NULL;
  int               i, j, nr;
  mat_rec          *rec;

  if (info->nbresult < 0)
    {
      nr = info->om_count*info->mu_count;
    }
  else
    {
      nr = info->nbresult;
    }
  /* this HDU contains the measured closure phases and the closure phases derived from the reconstructions */
  table = cpl_table_new(info->nbbis);
  if (table == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the table for the derived closure phases, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      return ec;
    }
  if (cpl_table_new_column(table, "LAMBDA", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the wavelength column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "U1COORD", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the u-coordinate column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "V1COORD", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the v-coordinate column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "U2COORD", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the u-coordinate column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "V2COORD", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the v-coordinate column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "T3AMP", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the amplitude column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "T3AMPERR", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the amplitude error column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "T3PHI", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the closure phase column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "T3PHIERR", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the closure phase error column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "BISERR", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the bispectrum error column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  cpl_table_set_column_unit(table, "LAMBDA", "m");
  cpl_table_set_column_unit(table, "U1COORD", "1/mas");
  cpl_table_set_column_unit(table, "V1COORD", "1/mas");
  cpl_table_set_column_unit(table, "U2COORD", "1/mas");
  cpl_table_set_column_unit(table, "V2COORD", "1/mas");
  for (i = 0; i < nr; i++)
    { // create a column for each reconstruction (same order as in RECLIST)
      char cname[256];
      snprintf(cname, 256, "T3AMP%d", i + 1);
      if (cpl_table_new_column(table, cname, CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the %d derived amplitude column, code = %d, message = %s",
			i + 1,
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
      snprintf(cname, 256, "T3PHI%d", i + 1);
      if (cpl_table_new_column(table, cname, CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the %d derived closure phase column, code = %d, message = %s",
			i + 1,
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
    }
  for (i = 0; i < info->nbbis; i++)
    {
      mat_bis       *el   = &(info->bis_list[i]);
      cpl_table_set_double(table, "LAMBDA",   i, el->lambda);
      cpl_table_set_double(table, "U1COORD",  i, el->u1/info->FOV);
      cpl_table_set_double(table, "V1COORD",  i, el->v1/info->FOV);
      cpl_table_set_double(table, "U2COORD",  i, el->u2/info->FOV);
      cpl_table_set_double(table, "V2COORD",  i, el->v2/info->FOV);
      cpl_table_set_double(table, "T3AMP",    i, el->mamp);
      cpl_table_set_double(table, "T3AMPERR", i, el->mamperr);
      cpl_table_set_double(table, "T3PHI",    i, el->mcp);
      cpl_table_set_double(table, "T3PHIERR", i, el->mcperr);
      cpl_table_set_double(table, "BISERR",   i, sqrt(el->mvar));
    }
  for (i = 0; i < nr; i++)
    {
      char cname[256];
      snprintf(cname, 256, "T3AMP%d", i + 1);
      if (info->nbresult < 0)
	{
	  rec = &(info->rec_list[i]);
	}
      else
	{
	  rec = info->rec_sorted[i];
	}
      for (j = 0; j < info->nbbis; j++)
	{
	  cpl_table_set_double(table, cname, j, cabs(rec->bis_list[j]));
	}
      snprintf(cname, 256, "T3PHI%d", i + 1);
      if (info->nbresult < 0)
	{
	  rec = &(info->rec_list[i]);
	}
      else
	{
	  rec = info->rec_sorted[i];
	}
      for (j = 0; j < info->nbbis; j++)
	{
	  cpl_table_set_double(table, cname, j, carg(rec->bis_list[j]));
	}
    }
  plist = cpl_propertylist_new();
  cpl_propertylist_append_string(plist, "EXTNAME", "REC_T3");
  char *fname=cpl_sprintf("rec_%s.fits", info->tname);
  if (cpl_table_save(table, NULL, plist, fname, CPL_IO_EXTEND) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_free(fname); fname = NULL;
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_free(fname); fname = NULL;
  cpl_propertylist_delete(plist);
  cpl_table_delete(table);
  return CPL_ERROR_NONE;
}

/**
   @brief Stores the measured and reconstructed closure phases in an ASCII-file.
*/
static cpl_error_code mat_store_reconstructed_t3_ascii(mat_cal_imarec_info *info)
{
  int               i, j, nr;
  mat_rec          *rec;
  FILE             *dat;

  if (info->cp_name[0] == '\0') return CPL_ERROR_NONE;
  if (info->nbresult < 0)
    {
      nr = info->om_count*info->mu_count;
    }
  else
    {
      nr = info->nbresult;
    }
  // write the measured and reconstructed closure phases into a file
  dat = fopen(info->cp_name, "w");
  if (dat != NULL)
    {
      fprintf(dat, "# measured closure phases vs reconstructed closure phases\n");
      fprintf(dat, "# nr lambda u1 v1 u2 v2 amp amperr cp cperr biserr");
      for (j = 0; j < nr; j++)
	{
	  fprintf(dat, " rec%d_amp rec%d_cp", j + 1, j + 1);
	}
      fprintf(dat, "\n");
      for (j = 0; j < nr; j++)
	{
	  if (info->nbresult < 0)
	    {
	      rec = &(info->rec_list[j]);
	    }
	  else
	    {
	      rec = info->rec_sorted[j];
	    }
	  fprintf(dat, "# qrec%d %.6f\n", j+1, rec->qrec);
	}
      for (i = 0; i < info->nbbis; i++)
	{
	  mat_bis *el   = &(info->bis_list[i]);
	  fprintf(dat, "%d %g %.6f %.6f %.6f %.6f %.6f %.6f %.6f %.6f %.6f",
		  i,
		  el->lambda,
		  el->u1/info->FOV,
		  el->v1/info->FOV,
		  el->u2/info->FOV,
		  el->v2/info->FOV,
		  el->mamp,
		  el->mamperr,
		  el->mcp,
		  el->mcperr,
		  sqrt(el->mvar));
	  for (j = 0; j < nr; j++)
	    {
	      if (info->nbresult < 0)
		{
		  rec = &(info->rec_list[j]);
		}
	      else
		{
		  rec = info->rec_sorted[j];
		}
	      fprintf(dat, " %.6f %.6f", cabs(rec->bis_list[i]), carg(rec->bis_list[i]));
	    }
	  fprintf(dat, "\n");
	}
      fclose(dat);
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Append the measured and reconstructed complex visibilities to a FITS file.
*/
static cpl_error_code mat_store_reconstructed_vis_table(mat_cal_imarec_info *info)
{
  cpl_propertylist *plist = NULL;
  cpl_table        *table = NULL;
  int               i, j, nr;
  mat_rec          *rec;

  if (info->nbresult < 0)
    {
      nr = info->om_count*info->mu_count;
    }
  else
    {
      nr = info->nbresult;
    }
  /* this HDU contains the measured complex visibilities and the complex visibilities derived from the reconstructions */
  table = cpl_table_new(info->nbvis);
  if (table == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the table for the derived complex visibilities, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      return ec;
    }
  if (cpl_table_new_column(table, "LAMBDA", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the wavelength column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "UCOORD", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the u-coordinate column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "VCOORD", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the v-coordinate column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "BL", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the baseline column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "VISAMP", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the amplitude column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "VISAMPERR", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the amplitude error column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "VISPHI", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the closure phase column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "VISPHIERR", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the closure phase error column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "GDF", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the gaussian disc fit column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "UDF", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the uniform disc fit column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "FDDF", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the fully darkened disc fit column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "LDF", CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the Lorentz disc fit column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  cpl_table_set_column_unit(table, "LAMBDA", "m");
  cpl_table_set_column_unit(table, "UCOORD", "1/mas");
  cpl_table_set_column_unit(table, "VCOORD", "1/mas");
  cpl_table_set_column_unit(table, "BL",     "m");
  for (i = 0; i < nr; i++)
    { // create two columns for each reconstruction (same order as in RECLIST)
      char cname[256];
      snprintf(cname, 256, "VISAMP%d", i + 1);
      if (cpl_table_new_column(table, cname, CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the %d derived amplitude column, code = %d, message = %s",
			i + 1,
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
      snprintf(cname, 256, "VISPHI%d", i + 1);
      if (cpl_table_new_column(table, cname, CPL_TYPE_DOUBLE) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_msg_error(cpl_func,
			"cannot allocate the %d derived phase column, code = %d, message = %s",
			i + 1,
			ec,
			cpl_error_get_message());
	  cpl_table_delete(table);
	  return ec;
	}
    }
  for (i = 0; i < info->nbvis; i++)
    {
      mat_vis  *el     = &(info->vis_list[i]);
      double    r      = sqrt(el->u*el->u + el->v*el->v);
      double    f      = r/info->FOV;
      double    gd_v2  = 0.0;
      double    ud_v2  = 0.0;
      double    fdd_v2 = 0.0;
      double    ld_v2  = 0.0;
      double    a;
      a = info->fit_gd_fwhm*FWHM2SIGMA;
      mat_fit_gauss_f(&f, &a, &gd_v2);
      a = 0.5*info->fit_ud_diameter;
      mat_fit_ud_f(&f, &a, &ud_v2);
      a = 0.5*info->fit_fdd_diameter;
      mat_fit_fdd_f(&f, &a, &fdd_v2);
      a = FWHM2LDA/info->fit_ld_fwhm;
      mat_fit_ld_f(&f, &a, &ld_v2);
      cpl_table_set_double(table, "LAMBDA",    i, el->lambda);
      cpl_table_set_double(table, "UCOORD",    i, el->u/info->FOV);
      cpl_table_set_double(table, "VCOORD",    i, el->v/info->FOV);
      cpl_table_set_double(table, "BL",        i, el->bl);
      cpl_table_set_double(table, "VISAMP",    i, el->mamp);
      cpl_table_set_double(table, "VISAMPERR", i, el->mamperr);
      cpl_table_set_double(table, "VISPHI",    i, el->mphi);
      cpl_table_set_double(table, "VISPHIERR", i, el->mphierr);
      cpl_table_set_double(table, "GDF",       i, sqrt(gd_v2));
      cpl_table_set_double(table, "UDF",       i, sqrt(ud_v2));
      cpl_table_set_double(table, "FDDF",      i, sqrt(fdd_v2));
      cpl_table_set_double(table, "LDF",       i, sqrt(ld_v2));
    }
  for (i = 0; i < nr; i++)
    {
      char cname[256];
      snprintf(cname, 256, "VISAMP%d", i + 1);
      if (info->nbresult < 0)
	{
	  rec = &(info->rec_list[i]);
	}
      else
	{
	  rec = info->rec_sorted[i];
	}
      for (j = 0; j < info->nbvis; j++)
	{
	  cpl_table_set_double(table, cname, j, cabs(rec->vis_list[j]));
	}
      snprintf(cname, 256, "VISPHI%d", i + 1);
      if (info->nbresult < 0)
	{
	  rec = &(info->rec_list[i]);
	}
      else
	{
	  rec = info->rec_sorted[i];
	}
      for (j = 0; j < info->nbvis; j++)
	{
	  cpl_table_set_double(table, cname, j, carg(rec->vis_list[j]));
	}
    }
  plist = cpl_propertylist_new();
  cpl_propertylist_append_string(plist, "EXTNAME", "REC_VIS");
  char *fname=cpl_sprintf("rec_%s.fits", info->tname);
  if (cpl_table_save(table, NULL, plist, fname, CPL_IO_EXTEND) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_free(fname); fname = NULL;
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_free(fname); fname = NULL;
  cpl_propertylist_delete(plist);
  cpl_table_delete(table);
  return CPL_ERROR_NONE;
}

/**
   @brief Stores the measured and reconstructed complex visibilities in an ASCII-file.
*/
static cpl_error_code mat_store_reconstructed_vis_ascii(mat_cal_imarec_info *info)
{
  int               i, j, nr;
  mat_rec          *rec;
  FILE             *dat;

  if (info->vis_name[0] == '\0') return CPL_ERROR_NONE;
  if (info->nbresult < 0)
    {
      nr = info->om_count*info->mu_count;
    }
  else
    {
      nr = info->nbresult;
    }
  // write the measured and reconstructed closure phases into a file
  dat = fopen(info->vis_name, "w");
  if (dat != NULL)
    {
      fprintf(dat, "# measured complex visibilities vs reconstructed complex visibilities\n");
      fprintf(dat, "# nr lambda u v bl amp amperr phi phierr gd ud fdd ld");
      for (j = 0; j < nr; j++)
	{
	  fprintf(dat, " rec%d_amp rec%d_phi", j + 1, j + 1);
	}
      fprintf(dat, "\n");
      for (j = 0; j < nr; j++)
	{
	  if (info->nbresult < 0)
	    {
	      rec = &(info->rec_list[j]);
	    }
	  else
	    {
	      rec = info->rec_sorted[j];
	    }
	  fprintf(dat, "# qrec%d %.6f\n", j+1, rec->qrec);
	}
      for (i = 0; i < info->nbvis; i++)
	{
	  mat_vis  *el     = &(info->vis_list[i]);
	  double    r      = sqrt(el->u*el->u + el->v*el->v);
	  double    f      = r/info->FOV;
	  double    gd_v2  = 0.0;
	  double    ud_v2  = 0.0;
	  double    fdd_v2 = 0.0;
	  double    ld_v2  = 0.0;
	  double    a;
	  a = info->fit_gd_fwhm*FWHM2SIGMA;
	  mat_fit_gauss_f(&f, &a, &gd_v2);
	  a = 0.5*info->fit_ud_diameter;
	  mat_fit_ud_f(&f, &a, &ud_v2);
	  a = 0.5*info->fit_fdd_diameter;
	  mat_fit_fdd_f(&f, &a, &fdd_v2);
	  a = FWHM2LDA/info->fit_ld_fwhm;
	  mat_fit_ld_f(&f, &a, &ld_v2);
	  fprintf(dat, "%d %g %.6f %.6f %.6f %.6f %.6f %.6f %.6f %.6f %.6f %.6f %.6f",
		  i,
		  el->lambda,
		  el->u/info->FOV,
		  el->v/info->FOV,
		  el->bl,
		  el->mamp,
		  el->mamperr,
		  el->mphi,
		  el->mphierr,
		  sqrt(gd_v2),
		  sqrt(ud_v2),
		  sqrt(fdd_v2),
		  sqrt(ld_v2));
	  for (j = 0; j < nr; j++)
	    {
	      if (info->nbresult < 0)
		{
		  rec = &(info->rec_list[j]);
		}
	      else
		{
		  rec = info->rec_sorted[j];
		}
	      fprintf(dat, " %.6f %.6f", cabs(rec->vis_list[i]), carg(rec->vis_list[i]));
	    }
	  fprintf(dat, "\n");
	}
      fclose(dat);
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Creates and fills the reconsruction result file.
   @param info      Contains the image reconstruction context.
   @param rname     The name of the plugin (mat_cal_imarec).
   @param parlist   The parameters list of the plugin.
   @param frameset  The frameset of the plugin (input frames and result frame afterwards).
*/
static cpl_error_code mat_store_reconstructions(mat_cal_imarec_info *info,
						const char *rname,
						cpl_parameterlist *parlist,
						cpl_frameset *frameset)
{
  cpl_frame        *recframe = NULL;
  int               i, n;
  mat_rec          *rec;

  cpl_error_reset();
  cpl_ensure_code((info != NULL), CPL_ERROR_NULL_INPUT);
  char *fname=cpl_sprintf("rec_%s.fits", info->tname);

  if (info->info_flags & INFO_RESULT)
    {
      cpl_msg_info(cpl_func, "storing the reconstructed images in file %s", fname);
    }
  cpl_free(fname); fname = NULL;
  n = info->om_count*info->mu_count;
  // swap the quadrants for all images
  if (info->guess == 0)
    {
      for (i = 0; i < n; i++)
	{
	  rec = &(info->rec_list[i]);
	  mat_image_swap(rec->rec_image);
	  mat_image_swap(rec->conv_image);
	}
      mat_image_swap(info->uv_image);
      if (info->start_image_loaded != NULL)
	{
	  mat_image_swap(info->start_image_loaded);
	}
      if (info->prior_image_loaded != NULL)
	{
	  mat_image_swap(info->prior_image_loaded);
	}
      for (i = 0; i < cpl_imagelist_get_size(info->om_image_list); i++)
	{
	  cpl_image *om = cpl_imagelist_get(info->om_image_list, i);
	  mat_image_swap(om);
	}
      if (info->model_image_loaded != NULL)
	{
	  mat_image_swap(info->model_image_loaded);
	}
      if (info->model_image_convolved != NULL)
	{
	  mat_image_swap(info->model_image_convolved);
	}
    }
  /* Create product frame */
  recframe = cpl_frame_new();
  fname=cpl_sprintf("rec_%s.fits", info->tname);
  cpl_frame_set_filename(recframe, fname);
  cpl_free(fname); fname = NULL;
  cpl_frame_set_tag(recframe, MATISSE_REC_PROCATG);
  cpl_frame_set_type(recframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(recframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(recframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_frame_delete(recframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  if (info->guess == 1)
    {
      if (mat_store_suggested_parameters(info, rname, parlist, frameset, recframe) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_frame_delete(recframe);
	  return ec;
	}
    }
  else
    {
      if (mat_store_best_reconstruction(info, rname, parlist, frameset, recframe) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_frame_delete(recframe);
	  return ec;
	}
      if (mat_store_reconstruction_list(info) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_frame_delete(recframe);
	  return ec;
	}
      if (mat_store_additional_images(info) != CPL_ERROR_NONE)
	{
	  cpl_error_code ec = cpl_error_get_code();
	  cpl_frame_delete(recframe);
	  return ec;
	}
      if (info->use_flags & USE_T3_TABLE)
	{
	  if (mat_store_reconstructed_t3_table(info) != CPL_ERROR_NONE)
	    {
	      cpl_error_code ec = cpl_error_get_code();
	      cpl_frame_delete(recframe);
	      return ec;
	    }
	  if (mat_store_reconstructed_t3_ascii(info) != CPL_ERROR_NONE)
	    {
	      cpl_error_code ec = cpl_error_get_code();
	      cpl_frame_delete(recframe);
	      return ec;
	    }
	}
      if (info->use_flags & USE_VIS2_TABLE)
	{
	  if (mat_store_reconstructed_vis2_table(info) != CPL_ERROR_NONE)
	    {
	      cpl_error_code ec = cpl_error_get_code();
	      cpl_frame_delete(recframe);
	      return ec;
	    }
	  if (mat_store_reconstructed_vis2_ascii(info) != CPL_ERROR_NONE)
	    {
	      cpl_error_code ec = cpl_error_get_code();
	      cpl_frame_delete(recframe);
	      return ec;
	    }
	}
      if (info->use_flags & USE_VIS_TABLE)
	{
	  if (mat_store_reconstructed_vis_table(info) != CPL_ERROR_NONE)
	    {
	      cpl_error_code ec = cpl_error_get_code();
	      cpl_frame_delete(recframe);
	      return ec;
	    }
	  if (mat_store_reconstructed_vis_ascii(info) != CPL_ERROR_NONE)
	    {
	      cpl_error_code ec = cpl_error_get_code();
	      cpl_frame_delete(recframe);
	      return ec;
	    }
	}
    }
  /* Log the saved file in the input frameset */
  cpl_frameset_insert(frameset, recframe);
  return CPL_ERROR_NONE;
}

static void mat_dump_rec(mat_cal_imarec_info *info)
{
  int i;

  if (info->info_flags & INFO_DEBUG)
    {
      for (i = 0; i < info->nbrec; i++)
	{
	  mat_rec *rec = info->rec_sorted[i];
	  cpl_msg_info(cpl_func, "rec[%d] om = %g, mu = %g, qrec = %g", i, rec->om, rec->mu, rec->qrec);
	}
    }
}

/**
   @brief Calculates the distance between the complex visibilities and a shifted image
   @param info    Contains the image reconstruction context.
   @param image   The image which is shifted to a new position.
   @param dx      Shift in x-direction.
   @param dy      Shift in y-direction.
   @returns The distance as Chi2.
*/
static double mat_test_shift_image_to_vis(mat_cal_imarec_info *info, cpl_image *image, double dx, double dy)
{
  int    i;
  double sum0 = 0.0;
  double sum1 = 0.0;

  mat_image_shift(info->tdbl_image, dx, dy, image);
  mat_image_fft_forward(info->frec_image, info->tdbl_image);
  mat_derive_bis_and_vis(info);
  for (i = 0; i < info->nbvis; i++)
    {
      mat_vis *el = &(info->vis_list[i]);
      sum0 += MAT_CHYPOT2(el->mvis - el->rvis)*el->wuvvis2;
      sum1 += el->weight;
    }
  return (sum0/(fabs(sum1) + 1e-30));
}

/**
   @brief This function tests several shift verctors in order to find an optimal correlation between the image and the complex visibilities.
   @param info    Contains the image reconstruction context.
   @param image   The image which is shifted to a new position.

   The method for finding an optimal correlation between an image
   and the complex visibilities uses a simple schema. First a shift
   delta value is defined and several possible shifts (-2*delta, -2*delta)
   .. (+2*delta,+2*delta) tried. The shift with the smallest chi squared
   distance between the shifted image and the complex visibilities
   is then used as a new starting point and the delta value is
   reduced (divided by 2). This method is repeated until the delta
   value is below 0.1 pixel. Finally the image is shifted to the
   optimal position.
*/
static void mat_shift_image_to_vis(mat_cal_imarec_info *info, cpl_image *image)
{
  double d = MAT_DELTA_START;
  double dx = 0.0;
  double dy = 0.0;
  double min_dx, min_dy, min_chi2;
  int    i, j;

  // check if we need to shift an image at all
  if (info->algo_mode != MODE_COMPLEX_VIS) return;
  /* Calculate the Chi2 for a (0,0) shift as reference. */
  min_dx = 0.0;
  min_dy = 0.0;
  min_chi2 = mat_test_shift_image_to_vis(info, image, dx, dy);
  if (info->info_flags & INFO_DEBUG)
    {
      cpl_msg_info(cpl_func, "reference shift (0,0), chi2 = %f", min_chi2);
    }
  while (d >= MAT_DELTA_MIN)
    {
      for (i = -MAT_DELTA_STEPS; i <= MAT_DELTA_STEPS; i++)
	{
	  for (j = -MAT_DELTA_STEPS; j <= MAT_DELTA_STEPS; j++)
	    {
	      double chi2;
	      if ((i == 0) && (j == 0)) continue; /* We already have this value as reference/minimum value. */
	      chi2 = mat_test_shift_image_to_vis(info, image, dx + d*(double)i, dy + d*(double)j);
	      if (chi2 < min_chi2)
		{
		  min_dx = d*(double)i;
		  min_dy = d*(double)j;
		  min_chi2 = chi2;
		}
	    }
	}
      dx += min_dx;
      dy += min_dy;
      if (info->info_flags & INFO_DEBUG)
	{
	  cpl_msg_info(cpl_func, "   best shift dx = %f, dy = %f, chi2 = %f", dx, dy, min_chi2);
	}
      /* the new shifted position is now the center */
      min_dx = 0.0;
      min_dy = 0.0;
      d *= MAT_DELTA_SCALE;
    }
  if (info->info_flags & INFO_PREPARE)
    {
      cpl_msg_info(cpl_func, "   final shift dx = %f, dy = %f, chi2 = %f", dx, dy, min_chi2);
    }
  /* do the final shift and copy the shifted image back */
  mat_image_shift(info->tdbl_image, dx, dy, image);
  cpl_image_copy(image, info->tdbl_image, 1, 1);
}

//static void mat_use_asa_cg(mat_cal_imarec_info *info, mat_rec  *rec)
static void mat_use_asa_cg(mat_cal_imarec_info *info)
{
  int k;

  for (k = 0; k < info->asa_count; k++)
    {
      // call asa cg
      mat_init_asacg_parameters(info);
      asa_cg(cpl_vector_get_data(info->rec_vector),
	     cpl_vector_get_data(info->lower_bounds),
	     cpl_vector_get_data(info->upper_bounds),
	     info->nbom,
	     NULL,
	     &(info->asacg),
	     &(info->asa),
	     info->grad_tol,
	     mat_calc_value,
	     mat_calc_grad,
	     mat_calc_value_grad,
	     NULL,
	     info);
    }
}

static cpl_error_code lbfgsb_context_init(lbfgsb_context *context, int nbvar, int nbcorr)
{
  int i;

  // Initialize
  context->bound_types = NULL;
  context->g = NULL;
  context->wa = NULL;
  context->iwa = NULL;
  // allocate
  context->bound_types = (integer*)cpl_calloc(nbvar, sizeof(integer));
  if (context->bound_types == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for L-BFGS-B variable bound type");
      return CPL_ERROR_UNSPECIFIED;
    }
  for (i = 0; i < nbvar; i++)
    {
      context->bound_types[i] = 2;
    }
  context->g = (double*)cpl_calloc(nbvar, sizeof(double));
  if (context->g == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for L-BFGS-B gradient");
      return CPL_ERROR_UNSPECIFIED;
    }
  context->wa = (double*)cpl_calloc(2*nbcorr*nbvar + 11*nbcorr*nbcorr + 5*nbvar + 8*nbcorr, sizeof(double));
  if (context->wa == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for L-BFGS-B double working area");
      return CPL_ERROR_UNSPECIFIED;
    }
  context->iwa = (integer*)cpl_calloc(3*nbvar, sizeof(integer));
  if (context->iwa == NULL)
    {
      cpl_msg_error(cpl_func, "can't allocate memory for L-BFGS-B integer working area");
      return CPL_ERROR_UNSPECIFIED;
    }
  return CPL_ERROR_NONE;
}

static void lbfgsb_context_delete(lbfgsb_context *context)
{
  if (context->bound_types != NULL)
    {
      cpl_free(context->bound_types);
      context->bound_types = NULL;
    }
  if (context->g != NULL)
    {
      cpl_free(context->g);
      context->g = NULL;
    }
  if (context->wa != NULL)
    {
      cpl_free(context->wa);
      context->wa = NULL;
    }
  if (context->iwa != NULL)
    {
      cpl_free(context->iwa);
      context->iwa = NULL;
    }
}
/*
  int lbfgsb_setulb(integer n, integer m, double *x, 
  double *l, double *u, integer *nbd, double *f, double *g,
  double factr, double pgtol, double *wa, integer *iwa,
  integer *task, integer iprint, integer *csave, logical *lsave, 
  integer *isave, double *dsave);
  typedef struct {
  integer  *bound_types; // 0 = unbounded, 1 = lower, 2 = lower and upper 3 = upper
  double   *g; // gradient at x[]
  double   *wa; // Working array for L-BFGS-B, double, (2*nbcorr + 5)*nbvar * 11*nbcorr*nbcorr + 8*nbcorr).
  integer  *iwa; // Working array for L-BFGS-B, integer, 3*nbvar
  integer   csave;
  logical   lsave[4];
  integer   isave[44];
  double    dsave[29];
  } lbfgsb_context;
*/
static void mat_use_lbfgsb(mat_cal_imarec_info *info, mat_rec *rec, lbfgsb_context *context)
{
  integer  n = (integer)(info->nbom);
  integer  m = (integer)(info->ncorr);
  int      i;
  integer  task = START;
  double   f = 0.0;
  integer  iprint = -1;

  for (i = 0; i < 2*m*n + 11*m*m + 5*n + 8*m; i++) context->wa[i] = 0.0;
  for (i = 0; i < 3*n; i++) context->iwa[i] = 0;
  context->csave = 0;
  for (i = 0; i < 4; i++) context->lsave[i] = 1;
  for (i = 0; i < 44; i++) context->isave[i] = 0;
  for (i = 0; i < 29; i++) context->dsave[i] = 0.0;
  do {
    lbfgsb_setulb(n, m,
		  cpl_vector_get_data(info->rec_vector), 
		  cpl_vector_get_data(info->lower_bounds),
		  cpl_vector_get_data(info->upper_bounds),
		  context->bound_types,
		  &f, context->g,
		  info->factr, info->pgtol,
		  context->wa, context->iwa,
		  &task, iprint,
		  &(context->csave), context->lsave, context->isave, context->dsave);
    if (info->info_flags & INFO_ITER_STD)
      {
	cpl_msg_info(cpl_func, "it %4d : L-BFGS-B %d",
		     info->nbit, (int)task);
      }
    if (task == NEW_X)
      {
	if (context->dsave[12] <= info->pgtol*(1.0 + fabs(f)))
	  {
	    task = STOP_GRAD;
	  }
	else
	  {
	    info->nbit++;
	    continue;
	  }
      }
    if (IS_FG(task))
      {
	f = mat_calc_value_grad_basic(info, rec,
				      cpl_vector_get_data(info->rec_vector),
				      context->g);
	continue;
      }
    if (IS_CONVERGED(task) || (task == STOP_GRAD))
      {
	return;
      }
    if (IS_STOP(task))
      {
	cpl_msg_info(cpl_func, "L-BFGS-B STOP: task = %d", (int)task);
	return;
      }
    else if (IS_WARNING(task))
      {
	cpl_msg_info(cpl_func, "L-BFGS-B WARNING: task = %d", (int)task);
	return;
      }
    else if (IS_ERROR(task))
      {
	cpl_msg_info(cpl_func, "L-BFGS-B ERROR: task = %d", (int)task);
	return;
      }
    else
      {
	cpl_msg_info(cpl_func, "L-BFGS-B ???: task = %d", (int)task);
	return;
      }
  } while(1);
}

/*--------------------------------------------------------------------*/
/* Main plugin function                                               */
/*--------------------------------------------------------------------*/

/**
   @brief Compares two reconstructions using the reconstruction quality (qrec).
   @param r1   First reconstruction (mat_rec **).
   @param r2   Second reconstruction (mat_rec **).
   @returns -1, 0 or +1
*/
static int mat_compare_rec(const void *r1, const void *r2)
{
  const mat_rec  *hr1 = *(mat_rec **)r1;
  const mat_rec  *hr2 = *(mat_rec **)r2;

  if (hr1->qrec < hr2->qrec) return -1;
  if (hr1->qrec > hr2->qrec) return +1;
  return 0;
}

/**
   @brief Interpret the command line options and execute the data processing
   @param parlist the parameters list
   @param frameset the frames list
   @return 0 if everything is ok

   This is the main image reconstruction function. After getting the plugin parameters (and storing them in a mat_cal_imarec_info data structure), up to three
   images (DO classification START_IMAGE, PRIOR_IMAGE and MODEL_IMAGE) are read. When the calibrated interferrometric data is loaded, the uv-coordinates are remapped
   using a wavelength specific factor to FFT array coordinates. The image reconstructions are done inside two nested loops. The outer loop iterates over different object mask radii and the inner loop iterates over different regularization hyperparameter values. At the end some or all reconstruction results are stored in a FITS file.

   The following pseudocode illustrates the overall schema:

   <pre>BEGIN
   Read the calibrated interferometric data (with wavelength filter);
   FOR EACH object mask radius DO
   FOR EACH regularization parameter value DO
   Select start image and prior;
   Do a reconstruction using ASA_CG;
   Manage a sorted list of reconstructions;
   ENDFOR
   ENDFOR
   Store the results in a FITS file;
   END</pre>
*/
static int mat_cal_imarec(cpl_parameterlist *parlist,
			  cpl_frameset *frameset)
{
  mat_cal_imarec_info    info;
  lbfgsb_context         context;
  int                    i, j;

  // initialize and get all command line parameters
  mat_info_init(&info);
  info.parlist = parlist;
  info.frameset = frameset;
  //mat_init_asacg_parameters(&info);
  mat_get_parameters(&info, parlist);
  if (info.engine == ENGINE_L_BFGS_B)
    {
      lbfgsb_context_init(&context, info.npix*info.npix, info.ncorr);
    }
  // load the specified start image
  info.start_image_loaded = mat_read_image(&info,
					   frameset,
					   MAT_DO_START_IMAGE,
					   info.start_mode,
					   info.start_param);
  // load the specified prior image
  info.prior_image_loaded = mat_read_image(&info,
					   frameset,
					   MAT_DO_PRIOR_IMAGE,
					   info.prior_mode,
					   info.prior_param);
  // load the specified model image
  info.model_image_loaded = mat_read_image(&info,
					   frameset,
					   MAT_DO_MODEL_IMAGE,
					   -1,
					   info.model_scale);
  mat_create_object_masks(&info);
  if (mat_load_interferometric_data(&info) != 0)
    {
      mat_info_delete(&info);
      return -1;
    }
  if (info.info_flags & (INFO_INPUT_VIS2 | INFO_INPUT_T3 | INFO_INPUT_VIS | INFO_PREPARE))
    {
      cpl_msg_info(cpl_func, "  baseline: %f .. %f [px]", info.minbaseline_px, info.maxbaseline_px);
      cpl_msg_info(cpl_func, "  baseline: %f .. %f [m]", info.minbaseline_m, info.maxbaseline_m);
    }
  // calculate the data weights as inverse of the density
  if (info.use_flags & USE_T3_TABLE)
    {
      mat_calc_weights_bis(&info);
    }
  if (info.use_flags & USE_VIS2_TABLE)
    {
      mat_calc_weights_vis2(&info);
    }
  if (info.use_flags & USE_VIS_TABLE)
    {
      mat_calc_weights_vis(&info);
    }
  mat_calc_uv_coverage(&info);
  // allocate all necessary memory
  if (mat_init_reconstruction(&info) != CPL_ERROR_NONE)
    {
      mat_info_delete(&info);
      /* Return */
      if (cpl_error_get_code())
	return -1;
      else
	return 0;
    }
  if (info.algo_mode & MODE_COMPLEX_VIS)
    {
      if (info.start_image_loaded != NULL)
	{
	  if (info.info_flags & INFO_DEBUG)
	    {
	      cpl_msg_info(cpl_func, "mapping start image to complex visibilities");
	    }
	  mat_shift_image_to_vis(&info, info.start_image_loaded);
	}
      if (info.prior_image_loaded != NULL)
	{
	  if (info.info_flags & INFO_DEBUG)
	    {
	      cpl_msg_info(cpl_func, "mapping prior image to complex visibilities");
	    }
	  mat_shift_image_to_vis(&info, info.prior_image_loaded);
	}
      if (info.model_image_loaded != NULL)
	{
	  if (info.info_flags & INFO_DEBUG)
	    {
	      cpl_msg_info(cpl_func, "mapping model image to complex visibilities");
	    }
	  mat_shift_image_to_vis(&info, info.model_image_loaded);
	}
    }
  mat_guess_parameters(&info);
  if (info.guess == 0)
    {
      if (info.conv_scale < 0.0)
	{
	  mat_image_calc_gaussian(info.tent_image, -info.maxbaseline_px*info.conv_scale);
	}
      else
	{
	  mat_image_calc_tent(info.tent_image, info.maxbaseline_px*info.conv_scale);
	}
      if (info.info_flags & INFO_FITS)
	{
	  mat_image_store_double(info.tent_image, "tent.fits", "TENT", parlist, frameset);
	}
      // convolve the optional model image with the tent
      if (info.model_image_loaded != NULL)
	{
	  mat_image_convolve(info.tdbl_image, info.model_image_loaded, info.tent_image, info.frec_image);
	  info.model_image_convolved = cpl_image_duplicate(info.tdbl_image);
	}
      info.nbrec = 0; // we do not have any reconstructed image yet
      // loop over all object mask radius
      info.om = info.om_start;
      for (i = 0; i < info.om_count; i++)
	{
	  info.om_image = cpl_imagelist_get(info.om_image_list, i);
	  // loop over all regularization factors
	  info.mu = info.mu_start;
	  for (j = 0; j < info.mu_count; j++)
	    {
	      mat_rec  *rec = &(info.rec_list[i*info.mu_count + j]);
	      info.rec_curr = rec;
	      info.nbit = 0;
	      // store the current object mask radius (index and value) and regularization parameter (value and index) in the reconstruction
	      rec->om_idx = i;
	      rec->om = info.om;
	      rec->mu_idx = j;
	      rec->mu = info.mu;
	      // prepare the reconstruction
	      mat_prepare_reconstruction(&info, rec);
	      if (info.info_flags & INFO_FITS)
		{
		  char      fname[256];
		  snprintf(fname, 256, "start_%d_%d.fits", i, j);
		  mat_image_store_double(info.rec_image, fname, "START", parlist, frameset);
		}
	      // linearize the start image
	      mat_image_to_vector(&info, cpl_vector_get_data(info.rec_vector), info.rec_image);
	      if ((i == 0) && (j == 0))
		{
		  info.filter = info.filter_fwhm;
		}
	      else
		{
		  info.filter = 0.0;
		}
#ifdef WITH_GRAD_SCALING
	      info.grad_scale = info.grad_scale_start;
#endif
	      switch (info.engine)
		{
		case ENGINE_ASA_CG:
		  //mat_use_asa_cg(&info, rec);
		  mat_use_asa_cg(&info);
		  break;
		case ENGINE_L_BFGS_B:
		  mat_use_lbfgsb(&info, rec, &context);
		  break;
		default:
		  cpl_msg_error(cpl_func, "unknown optimazation engine %d", info.engine);
		}
	      // calculate the criteria for sorting the reconstructions
	      mat_calc_sort_criteria(rec);
	      // delinearize the reconstructed image
	      mat_vector_to_image(&info, info.rec_image, cpl_vector_get_data(info.rec_vector));
	      mat_update_reconstruction(&info, rec);
	      info.rec_sorted[info.nbrec] = rec;
	      info.nbrec++;
	      // update the sorted list of reconstruction indices
	      qsort(info.rec_sorted, info.nbrec, sizeof(mat_rec *), mat_compare_rec);
	      // increase the regularization factor
	      info.mu *= info.mu_factor;
	    }
	  // increase the object mask radius
	  info.om += info.om_step;
	}
      mat_dump_rec(&info);
    }
  mat_store_reconstructions(&info, "mat_cal_imarec", parlist, frameset);
  if (info.engine == ENGINE_L_BFGS_B)
    {
      lbfgsb_context_delete(&context);
    }
  mat_info_delete(&info);
  /* Return */
  if (cpl_error_get_code())
    return -1;
  else
    return 0;
}
