/* $Id: mat_est_flat.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_est_flat.c $
 */
/*------------------------------------------------------------------------------
  Includes
  ------------------------------------------------------------------------------*/
#include <string.h>
#include <cpl.h>
#include "mat_drl.h"
#include "mat_utils.h"
#include "mat_const.h"
#include "mat_error.h"
#include "mat_frame.h"
#include "mat_image.h"
#include "mat_statistics.h"
#include "mat_badpixel.h"
#include "mat_nonlinearity.h"
#include "mat_flatfield.h"
#include "mat_obsflat.h"
#include "mat_apply_staticcalib.h"

/*------------------------------------------------------------------------------
  Define
  ------------------------------------------------------------------------------*/
#define TEL_NUMBER 4
#define MATISSE_DO_OBSDARK         "OBSDARK"
#define MATISSE_DO_OBSFLAT         "OBSFLAT"
#define MATISSE_DO_BPM             "BADPIX"
#define MATISSE_DO_NLM             "NONLINEARITY"
#define MATISSE_DO_FFM             "FLATFIELD"

#define MAT_CONSTANT_FLATFIELD     0
#define MAT_DETECTOR_FLATFIELD     1
#define MAT_INSTRUMENT_FLATFIELD   2

#define MATISSE_MIN_FRAMES         32

/*-----------------------------------------------------------------------------
  Data types
  -----------------------------------------------------------------------------*/
typedef struct {
  int                  obsflat_type;
  int                  recalc_flat;
  double               gain;
  mat_gendata         *rawdark;
  mat_gendata         *rawflat;
  mat_detector        *det;
  mat_frame           *dmedian;
  mat_frame           *dvariance;
  mat_frame           *fmedian;
  mat_frame           *fvariance;
  mat_badpixel        *bpm;
  mat_nonlinearity    *nlm;
  mat_flatfield       *ffm;
  mat_obsflat         *ofm;
} mat_est_flat_info;



/* Plugin detailed description */
static const char mat_est_flat_help[] = 
  "This plugin calculates for a specific observation a setup specific optical flat and bias map.\n"
  "Both maps use the same setup (geometry and exposure time) as the science observation.\n"
  "This plugin uses a raw file containing 50 cold dark frames (DO classification OBSDARK) and\n"
  "a raw file containing 50 flatfield frame (DO classification OBSFLAT). They are produced by\n"
  "the MATISSE_gen_cal_obsflat template. In addition, the static bad pixel map, the static\n"
  "nonlinearity map and the static flatfield map are needed. The static bad pixel map (PRO.CATG = BADPIX)\n"
  "is needed to ignore the bad pixels during normalization of the optical flat. The static\n"
  "nonlinearity map is needed to separate the optical (large scale) variations from the fix pattern noise\n"
  "and the nonlinearity (small scale) variations. The static flatfield map (PRO.CATG = FLATFIELD)\n"
  "is used to get the conversion factor for converting ADU into electrons and copy it into\n"
  "the optical flatfield.\n"
  "\n"
  "This plugin produces a FITS file which contains the following data:\n"
  "\n"
  " - An empty primary HDU which contains the keywords from the first raw data file\n"
  "   and the QC1 parameters.\n"
  " - A binary table extension (OBS_FLAT_BIAS) containing the optical flat and\n"
  "   the bias map (median cold dark).\n"
  "\n"
  "The following QC1 parameters will be calculated (or copied) and stored in a cpl_propertylist\n"
  "and the primary HDU of the result FITS file:\n"
  "\n"
  " - QC DET<i> GAIN<p> : The global detector gain (in electrons per ADU, specific for a readout mode)\n"
  "   will be used for scientific purposes. This is a copy from the static flatfield map.\n"
  " - QC DET<i> CHANNEL<j> GAIN<p> : The detector gain (in electrons per ADU) for each channel\n"
  "   is also a copy from the static flatfield map\n"
  " - QC DET<i> CHANNEL<j> OFFSET<p> : The offset for the detector channels and each readout mode.\n"
  "\n"
  "Where\n"
  " i is the detector number (1 = L/M-Band, 2 = N-Band),\n"
  " j is the detector channel number (1 .. 32 for the L/M-Band, 1 .. 64 for the N-Band), and\n"
  " p is the readout mode (HAWAII-2RG: 1 = slow readout, 2 = fast readout,\n"
  "   Aquarius: 1 = low gain mode, 2 = high gain mode).\n"
  "\n"
  "The detector number (used for DET<i>) will be always 1 for the L/M-Band detector and\n"
  "2 for the N-Band detector. In addition, the DET<i> CHIP1 ID keyword allows to distinguish\n"
  "between different detector chips. This is necessary, to deal with a detector change after,\n"
  "for example, an instrument repair."
  "\n"
  "This plugin usually creates a scientific observation specific flatfield using\n"
  "exactly one file containing dark frames (OBSDARK), one file containing flatfield frames (OBSFLAT)\n"
  "the optional static badpixel map (BADPIXEL), the optional static nonlinearity map (NONLINEARITY)\n"
  "and the optional static flatfield map (FLATFIELD).\n"
  "\n"
  "In addition, an engineering setup specific flatfield can be created by using\n"
  "exactly one file containing dark frames (OBSDARK), no flatfield frames, the optional\n"
  "static badpixel map (BADPIXEL), the optional static nonlinearity map (NONLINEARITY)\n"
  "and the mandatory static flatfield map (FLATFIELD).\n"
  "\n"
  "The engineering setup specific flatfield contains the pixel bias map computed from the\n"
  "dark frames (OBSDARK) and the flatfield extracted from the static detector flatfield (FLATFIELD).\n"
  "\n"
  "If all files (OBSDARK, OBSFLAT, BADPIX, FLATFIELD and NONLINEARITY) are given in the SOF,\n"
  "the parameter --obsflat_type can be used to determine the type of OBS_FLATFIELD produced:\n"
  "\n"
  " --obsflat_type=const   the flatfield is set to 1.0 for all pixels\n"
  " --obsflat_type=det     the static flatfield is copied (detector flatfield)\n"
  " --obsflat_type=instr   the instrument flatfield is created\n"
  "\n"
  "The --recalc_flat parameter can be used to recalculate the detector flatfield by\n"
  "creating the reference image from a function fit of the OBSFLAT frames."
  "\n"
  "Input Files:\n"
  "\n"
  "    DO category:           Explanation:                Required:\n"
  "    OBSDARK                dark data                   Yes\n"
  "    OBSFLAT                flat data                   Yes\n"
  "    FLATFIELD              Flat Field                  Yes\n"
  "    BADPIX                 Bad Pixel Map               Yes\n"
  "    NONLINEARITY           Nonlinearity Map            Yes\n"
  "\n"
  "Output Files:\n"
  "\n"
  "    DO category:           Explanation:\n"
  "    OBS_FLATFIELD          Observing Flat Field\n";

static int mat_est_flat_create(cpl_plugin *);
static int mat_est_flat_exec(cpl_plugin *);
static int mat_est_flat_destroy(cpl_plugin *);
static int mat_est_flat(cpl_parameterlist *, cpl_frameset *);

static void mat_info_init(mat_est_flat_info *info);
static void mat_info_delete(mat_est_flat_info *info);

int cpl_plugin_get_info(cpl_pluginlist *list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
  cpl_plugin *plugin = (cpl_plugin *)recipe;
  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_est_flat",
		  "Estimation of the ObsFlatField Map",
		  mat_est_flat_help,
		  "Tim Kroener, Matthias Heininger",
		  "mhein@mpifr-bonn.mpg.de",
		  "GPL",
		  mat_est_flat_create,
		  mat_est_flat_exec,
		  mat_est_flat_destroy);
  cpl_pluginlist_append(list, plugin);
  return 0;
}

static int mat_est_flat_create(cpl_plugin *plugin)
{
  cpl_recipe      *recipe;
  cpl_parameter   *p;

  /* Check that the plugin is part of a valid recipe */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  /* Create the parameters list in the cpl_recipe object */
  recipe->parameters = cpl_parameterlist_new();
  /* Fill the parameters list */

  /* --obsflat_type */
  p = cpl_parameter_new_value("matisse.mat_est_flat.obsflat_type",
			      CPL_TYPE_STRING,
			      "Defines which kind of OBS_FLATFIELD is created "
			      "(const = 1.0 for all pixels, "
			      "det = detector flat, "
			      "instr = instrument flat)",
			      "matisse.mat_est_flat",
			      "det");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "obsflat_type");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --recalc_flat */
  p = cpl_parameter_new_range("matisse.mat_est_flat.recalc_flat",
			      CPL_TYPE_INT,
			      "Flag if the detector flatfield should be recalculated.",
			      "matisse.mat_est_flat",
			      0, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "recalc_flat") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --gain */
  p = cpl_parameter_new_range("matisse.mat_est_flat.gain",
			      CPL_TYPE_DOUBLE,
			      "Default conversion gain in [e-/DU].",
			      "matisse.mat_est_flat",
			      0.0, 0.0, 1e5);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "gain");
  cpl_parameterlist_append(recipe->parameters, p);

  return 0;
}
static int mat_est_flat_destroy(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  cpl_parameterlist_delete(recipe->parameters);
  //cpl_memory_dump();
  return 0;
}

static int mat_est_flat_exec(cpl_plugin *plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  return mat_est_flat(recipe->parameters, recipe->frames);
}

static cpl_vector *mat_vec_running_average(cpl_vector *vec,int period)
{
  cpl_vector *output=NULL;
  int i=0;
  int j=0;
  double ma=0.;
  int sz=0;
  
  sz=cpl_vector_get_size(vec);

  output=cpl_vector_new(sz);
  for(i=0;i<sz;i++) {
    if (i < period/2) {
      ma=0.;
      for(j=0;j<=2*i;j++) {
	ma+=cpl_vector_get(vec,j);
      }
      ma/=(2*i+1);
      cpl_vector_set(output,i,ma);
    } else if (i>sz-1-period/2) {
      ma=0.;
      for(j=i+1-(sz-i);j<=sz-1;j++) {
	ma+=cpl_vector_get(vec,j);
      }
      ma/=(2*sz-2*i-1);
      cpl_vector_set(output,i,ma);
    } else {
      ma=0.;
      for(j=i-period/2;j<=i+period/2;j++) {
	ma+=cpl_vector_get(vec,j);
      }
      ma/=(period+1);
      cpl_vector_set(output,i,ma);
    }
  }
  return output;
}

static void mat_info_init(mat_est_flat_info *info)
{
  /* initialize the info data structure */
  info->rawdark = NULL;
  info->rawflat = NULL;
  info->det = mat_detector_new();
  if (info->det == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for a mat_detector");
      return;
    }
  info->dmedian   = NULL;
  info->dvariance = NULL;
  info->fmedian   = NULL;
  info->fvariance = NULL;
  info->bpm = NULL;
  info->nlm = NULL;
  info->ffm = NULL;
  info->ofm = NULL;
}

static void mat_info_delete(mat_est_flat_info *info){
  /* delete the info data structure */
  if (info->rawdark != NULL)
    {
      mat_gendata_delete(info->rawdark);
      info->rawdark = NULL;
    }
  if (info->rawflat != NULL)
    {
      mat_gendata_delete(info->rawflat);
      info->rawflat = NULL;
    }
  if (info->det != NULL)
    {
      mat_detector_delete(info->det);
      info->det = NULL;
    }
  if (info->dmedian != NULL)
    {
      mat_frame_delete(info->dmedian);
      info->dmedian = NULL;
    }
  if (info->dvariance != NULL)
    {
      mat_frame_delete(info->dvariance);
      info->dvariance = NULL;
    }
  if (info->fmedian != NULL)
    {
      mat_frame_delete(info->fmedian);
      info->fmedian = NULL;
    }
  if (info->fvariance != NULL)
    {
      mat_frame_delete(info->fvariance);
      info->fvariance = NULL;
    }
  if (info->bpm != NULL)
    {
      mat_badpixel_delete(info->bpm);
      info->bpm = NULL;
    }
  if (info->nlm != NULL)
    {
      mat_nonlinearity_delete(info->nlm);
      info->nlm = NULL;
    }
  if (info->ffm != NULL)
    {
      mat_flatfield_delete(info->ffm);
      info->ffm = NULL;
    }
  if(info->ofm != NULL){
    mat_obsflat_delete(info->ofm);
    info->ofm = NULL;
  }

}

static cpl_error_code mat_load_static_maps(mat_est_flat_info *info, cpl_frameset *frameset)
{
  /* load the optional static badpixel map */
  if (cpl_frameset_count_tags(frameset, MATISSE_DO_BPM) == 1)
    {
      cpl_frame *frame = cpl_frameset_find(frameset, MATISSE_DO_BPM);
      cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
      info->bpm = mat_badpixel_load(frame);
    }
  /* load the optional static nonlinearity map */
  if (cpl_frameset_count_tags(frameset, MATISSE_DO_NLM) == 1)
    {
      cpl_frame *frame = cpl_frameset_find(frameset, MATISSE_DO_NLM);
      cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
      info->nlm = mat_nonlinearity_load(frame);
    }
  /* load the optional static flatfield map */
  if (cpl_frameset_count_tags(frameset, MATISSE_DO_FFM) == 1)
    {
      cpl_frame *frame = cpl_frameset_find(frameset, MATISSE_DO_FFM);
      cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
      info->ffm = mat_flatfield_load(frame);
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_load_darks(mat_est_flat_info *info, cpl_frameset *frameset)
{
  cpl_frame *frame = cpl_frameset_find(frameset, MATISSE_DO_OBSDARK);
  cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
  cpl_msg_info(cpl_func, "Loading the raw OBSDARK frames.");
  info->rawdark = mat_gendata_load(frame, CPL_TYPE_FLOAT);

  if(info->rawdark == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func, "Cannot read the FITS file containing the dark frames");
      return ec;
    }
  else
    {
      cpl_msg_info(cpl_func, "raw data file successfully loaded");
    }
  if (info->rawdark->imgdata->nbframe < MATISSE_MIN_FRAMES)
    {
      printf("%d\n",info->rawdark->imgdata->nbframe);
      cpl_msg_error(cpl_func, "The OBSDARK does not contain at least %d frames", MATISSE_MIN_FRAMES);
      return CPL_ERROR_UNSPECIFIED;
    }
  // NO_COMPENSATION               0
  // PIXEL_BIAS_COMPENSATION       (1 << 0)
  // FLATFIELD_COMPENSATION        (1 << 1)
  // NONLINEARITY_COMPENSATION     (1 << 2)
  // BADPIXEL_COMPENSATION         (1 << 3)
  // GLOBAL_BIAS_COMPENSATION      (1 << 4)
  // CHANNEL_BIAS_COMPENSATION     (1 << 5)
  // HORIZONTAL_BIAS_COMPENSATION  (1 << 6)
  // VERTICAL_BIAS_COMPENSATION    (1 << 7)
  // CORRELATION_COMPENSATION      (1 << 8)
  // FULL_COMPENSATION             ((1 << 9) - 1)
  // compensate the raw dark frames (no reduce, inplace)
  // TODO: the compensation methods needs to be checked
  /* currently no compensation is applied
  info->rawdark = mat_calibration_detector(info->rawdark,
				     info->bpm, info->ffm, info->nlm, NULL, NULL,
				     NO_COMPENSATION, 0, 1);
  */
  cpl_msg_info(cpl_func, "Calculate the pixel statistics for the dark frames.");
  info->dmedian = mat_frame_duplicate(info->rawdark->imgdata->list_frame[0], CPL_TYPE_DOUBLE);
  info->dvariance = mat_frame_duplicate(info->rawdark->imgdata->list_frame[0], CPL_TYPE_DOUBLE);
  mat_calc_statistics_cosmics(info->rawdark, info->dmedian, info->dvariance, 0);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_load_flats(mat_est_flat_info *info, cpl_frameset *frameset)
{
  cpl_frame *frame = cpl_frameset_find(frameset, MATISSE_DO_OBSFLAT);
  cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
  cpl_msg_info(cpl_func, "Loading the raw OBSFLAT frames.");
  info->rawflat = mat_gendata_load(frame, CPL_TYPE_FLOAT);
  if(info->rawflat == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func, "Cannot read the FITS file containing the flat frames");
      return ec;
    }
  else
    {
      cpl_msg_info(cpl_func, "raw data file successfully loaded");
    }
  if (info->rawflat->imgdata->nbframe < MATISSE_MIN_FRAMES)
    {
      cpl_msg_error(cpl_func, "The OBSFLAT does not contain at least %d frames", MATISSE_MIN_FRAMES);
      return CPL_ERROR_UNSPECIFIED;
    }
  cpl_msg_info(cpl_func, "Calculate the pixel statistics for the flatfield frames.");
  // NO_COMPENSATION               0
  // PIXEL_BIAS_COMPENSATION       (1 << 0)
  // FLATFIELD_COMPENSATION        (1 << 1)
  // NONLINEARITY_COMPENSATION     (1 << 2)
  // BADPIXEL_COMPENSATION         (1 << 3)
  // GLOBAL_BIAS_COMPENSATION      (1 << 4)
  // CHANNEL_BIAS_COMPENSATION     (1 << 5)
  // HORIZONTAL_BIAS_COMPENSATION  (1 << 6)
  // VERTICAL_BIAS_COMPENSATION    (1 << 7)
  // CORRELATION_COMPENSATION      (1 << 8)
  // FULL_COMPENSATION             ((1 << 9) - 1)
  // compensate the raw flat frames (no reduce, inplace)
  // TODO: the compensation methods needs to be checked
  info->rawflat = mat_calibration_detector(info->rawflat,
					   info->bpm, info->ffm, info->nlm, NULL, info->dmedian,
					   FULL_COMPENSATION & ~(DETECTOR_FLATFIELD_COMPENSATION | INSTRUMENT_FLATFIELD_COMPENSATION),
					   0,
					   1);
  // calculate the pixel statistics for the flatfield frames
  info->fmedian = mat_frame_duplicate(info->rawflat->imgdata->list_frame[0], CPL_TYPE_DOUBLE);
  info->fvariance = mat_frame_duplicate(info->rawflat->imgdata->list_frame[0], CPL_TYPE_DOUBLE);
  mat_calc_statistics_cosmics(info->rawflat, info->fmedian, info->fvariance, 0);
  mat_frame_subtract(info->fvariance, info->dvariance);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_create_obsflat(mat_est_flat_info *info)
{
  cpl_msg_info(cpl_func, "Create the new observation specific flatfield map data structure.");
  mat_detector_decode_raw(info->det, info->rawdark);
  info->ofm = mat_obsflat_new(info->det, info->rawdark->imgdet);
  if(info->ofm == NULL)
    {
      return CPL_ERROR_UNSPECIFIED;
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_estimate_obs_flat_constant(mat_est_flat_info *info)
{
  int   r;

  for (r = 0; r < info->ofm->imgdet->nbregion; r++)
    {
      mat_image_fill(info->ofm->list_obsflat[r], 1.0);
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_estimate_obs_flat_detector_static(mat_est_flat_info *info)
{
  int   r;

  cpl_msg_info(cpl_func, "We have an engineering obsflat -> extract from static flatfield map.");
  for (r = 0; r < info->ofm->imgdet->nbregion; r++)
    {
      mat_region *reg = info->ofm->imgdet->list_region[r];
      int         llx = reg->corner[0];
      int         lly = reg->corner[1];
      int         urx = reg->corner[0] + reg->naxis[0] - 1;
      int         ury = reg->corner[1] + reg->naxis[1] - 1;
      if (info->ofm->list_obsflat[r] != NULL)
	{
	  cpl_image_delete(info->ofm->list_obsflat[r]);
	  info->ofm->list_obsflat[r] = NULL;
	}
      info->ofm->list_obsflat[r] = cpl_image_extract(info->ffm->list_flatfield[0], llx, lly, urx, ury);
    }
  return CPL_ERROR_NONE;
}

/* static cpl_error_code mat_estimate_obs_flat_detector_recalc(mat_est_flat_info *info) */
/* { */
/*   return CPL_ERROR_NONE; */
/* } */

static cpl_error_code mat_estimate_obs_flat_instrument(mat_est_flat_info *info)
{
  int          r, x, y, nx, ny;
  cpl_image   *imflat    = NULL;
  cpl_image   **imOsc    = NULL;
  cpl_vector *vec1=NULL;
  cpl_vector *vec2=NULL;
  int kernelFilterSize=0;
  int sz;
  cpl_vector *w;
  cpl_vector *values;
  cpl_vector *fit;
	    
  if (info->det->nr == 2) {
    if (!strcmp((char *)cpl_propertylist_get_string(info->rawdark->keywords,
						    "ESO INS DIN ID"),"HIGH")) {
      kernelFilterSize=160;
    } else {
      kernelFilterSize=4;
    }
  } else {
    kernelFilterSize=1;
  }

  imOsc=cpl_malloc(info->fmedian->nbsubwin*sizeof(cpl_image *));

  // Estimate Perrot-Fabry Oscillation
  cpl_msg_info(cpl_func, "Estimate the observation flatfield.");
  for (r = 0; r < info->fmedian->nbsubwin; r++)
    {
      imflat = info->fmedian->list_subwin[r]->imgreg[0];
      if (info->rawflat->imgdet->list_region[r]->correlation == 0) {
	nx = cpl_image_get_size_x(imflat);
	ny = cpl_image_get_size_y(imflat);
	imOsc[r]=cpl_image_new(nx,ny,CPL_TYPE_FLOAT);	
      } else {
	nx = cpl_image_get_size_x(imflat);
	ny = cpl_image_get_size_y(imflat);
	imOsc[r]=cpl_image_new(nx,ny,CPL_TYPE_FLOAT);	  
	vec1=cpl_vector_new(ny);
	for (x = 1; x <= nx; ++x) {
	  for (y = 1; y <= ny; ++y) {
	    int rejected;
	    cpl_vector_set(vec1,y-1,cpl_image_get(imflat, x, y, &rejected));
	  }

	  // remove the absorption line at 9 micron before filtering
	  // only in HIGH resolution with Aquarius
	  // remove the central spike
	  if (info->det->nr == 2 && 
	      !strcmp((char *)cpl_propertylist_get_string(info->rawdark->keywords,
							  "ESO INS DIN ID"),"HIGH")) {
	    sz=40;
	    w=cpl_vector_new(sz);
	    values=cpl_vector_new(sz);
	    fit=cpl_vector_new(sz);
	    for(int i=0;i<sz;++i) {
	      cpl_vector_set(values,i,cpl_vector_get(vec1,160+i));
	      if (i<10 || i>=30) {
		cpl_vector_set(w,i,1.E5);
	      } else {
		cpl_vector_set(w,i,1.);
	      }
	    }
	    mat_polyfit_1d(values,w,3,fit,0);
	    for(int i=10;i<30;++i) {
	      cpl_vector_set(vec1,160+i,cpl_vector_get(fit,i));
	    }
	    cpl_vector_delete(w);
	    cpl_vector_delete(values);
	    cpl_vector_delete(fit);


	    sz=30;
	    w=cpl_vector_new(sz);
	    values=cpl_vector_new(sz);
	    fit=cpl_vector_new(sz);
	    for(int i=0;i<sz;++i) {
	      cpl_vector_set(values,i,cpl_vector_get(vec1,445+i));
	      if (i<10 || i>=20) {
		cpl_vector_set(w,i,1.E5);
	      } else {
		cpl_vector_set(w,i,1.);
	      }
	    }
	    mat_polyfit_1d(values,w,3,fit,0);
	    for(int i=10;i<20;++i) {
	      cpl_vector_set(vec1,445+i,cpl_vector_get(fit,i));
	    }
	    cpl_vector_delete(w);
	    cpl_vector_delete(values);
	    cpl_vector_delete(fit);



	  }
	  vec2=mat_vec_running_average(vec1,kernelFilterSize);


	  /* if (r==10 && x==100) { */
	  /*   for (y = 1; y <= ny; ++y) { */
	  /*     printf("%f %f\n",cpl_vector_get(vec1,y-1),cpl_vector_get(vec2,y-1)); */
	  /*   } */
	  /* } */

	  for (y = 1; y <= ny; ++y) {
	    cpl_image_set(imOsc[r], x, y, cpl_vector_get(vec1,y-1)/cpl_vector_get(vec2,y-1));
	  }	    
	  cpl_vector_delete(vec2);
	}   
	cpl_vector_delete(vec1);
      }
    }
  

  // calculate the observation flatfield
  cpl_msg_info(cpl_func, "Estimate the observation flatfield.");
  for (r = 0; r < info->fmedian->nbsubwin; r++) {
    imflat = info->fmedian->list_subwin[r]->imgreg[0];
    if (info->rawflat->imgdet->list_region[r]->correlation == 0) { 
      // we have a reference region -> set the gain to 1.0
      mat_image_fill(imflat, 1.0);
    } else {
      nx = cpl_image_get_size_x(imflat);
      ny = cpl_image_get_size_y(imflat);
      // we have a potometric or interferometric region -> calculate the flatfield
      for (y = 1; y <= ny; ++y) {
	vec1=cpl_vector_new(nx);
	for (x = 1; x <= nx; ++x) {
	  int rejected;
	  cpl_vector_set(vec1,x-1,cpl_image_get(imflat, x, y, &rejected));
	}
	vec2=mat_vec_running_average(vec1,32);
	for (x = 1; x <= nx; ++x) {
	  int rejected;
	  cpl_image_set(imflat, x, y, cpl_image_get(imflat, x, y, &rejected)/cpl_vector_get(vec2,x-1));
	}
	cpl_vector_delete(vec1);
	cpl_vector_delete(vec2);
      }
      cpl_image_multiply(imflat,imOsc[r]);
    }
    if (cpl_image_copy(info->ofm->list_obsflat[r], imflat, 1, 1) != CPL_ERROR_NONE) {
      cpl_msg_error(cpl_func, "could not copy obsflat, code = %d, message = %s ",
		    cpl_error_get_code(), cpl_error_get_message());
    }
    cpl_image_delete(imOsc[r]);
  }
  cpl_free(imOsc);
  


  return CPL_ERROR_NONE;
}

static cpl_error_code mat_estimate_obs_bias(mat_est_flat_info *info)
{
  int    r;

  cpl_msg_info(cpl_func, "Copy the pixel bias calculated from the dark statistics.");
  for (r = 0; r < info->dmedian->nbsubwin; r++)
    {
      cpl_image *imdark = info->dmedian->list_subwin[r]->imgreg[0];
      if (cpl_image_copy(info->ofm->list_bias[r],imdark, 1, 1) != CPL_ERROR_NONE)
	{
	  cpl_msg_error(cpl_func, "could not copy obsbias, code = %d, message = %s ", cpl_error_get_code(), cpl_error_get_message());
	}
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_copy_keywords(mat_est_flat_info *info)
{
  int   i;

  if (info->ffm != NULL)
    { // Copy the values from the static flatfield to obsflat
      if(cpl_vector_copy(info->ofm->channelgain, info->ffm->channelgain))
	{
	  cpl_msg_error(cpl_func, "could not copy channelgain vector, code = %d, message = %s ",cpl_error_get_code(),cpl_error_get_message());
	}
      if(cpl_vector_copy(info->ofm->channelnoise, info->ffm->channelnoise))
	{
	  cpl_msg_error(cpl_func, "could not copy channelnoise vector, code = %d, message = %s ",cpl_error_get_code(),cpl_error_get_message());
	}
      if(cpl_vector_copy(info->ofm->channeloffset, info->ffm->channeloffset))
	{
	  cpl_msg_error(cpl_func, "could not copy channeloffset vector, code = %d, message = %s ",cpl_error_get_code(),cpl_error_get_message());
	}
      if (info->gain == 0.0)
	{
	  info->ofm->detectorgain = info->ffm->detectorgain;
	  info->ofm->detectornoise = info->ffm->detectornoise;
	}
      else
	{
	  info->ofm->detectorgain = info->gain;
	  info->ofm->detectornoise = info->ffm->detectornoise/info->ffm->detectorgain*info->gain;
	}
    }
  else
    {
      for (i = 0; i < cpl_vector_get_size(info->ofm->channelgain); i++)
	{
	  cpl_vector_set(info->ofm->channelgain, i, 1.0);
	  cpl_vector_set(info->ofm->channelnoise, i, 0.0);
	  cpl_vector_set(info->ofm->channeloffset, i, 0.0);
	}
      info->ofm->detectorgain = info->gain;
      info->ofm->detectornoise = 0.0;
    }
  return CPL_ERROR_NONE;
}

static int mat_est_flat(cpl_parameterlist * parlist, cpl_frameset* frameset)
{
  mat_est_flat_info  info;
  cpl_parameter     *param;
  int                i = 0;
  int                darkcount;
  int                flatcount;
  int                ffmcount;
  /* int                nrs; */
  int                nbExtent = 0;
  cpl_propertylist  *plist = NULL;
  char              *keyExtname = NULL;
//  const char        *grism = NULL;
  cpl_table         *table = NULL;
  cpl_frame         *darkrawframe;

  mat_info_init(&info);
  // retrieve the input parameters
  // --obsflat_type
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_flat.obsflat_type");
  if (strcmp(cpl_parameter_get_string(param), "const") == 0)
    {
      info.obsflat_type = MAT_CONSTANT_FLATFIELD;
    }
  else if (strcmp(cpl_parameter_get_string(param), "det") == 0)
    {
      info.obsflat_type = MAT_DETECTOR_FLATFIELD;
    }
  else if (strcmp(cpl_parameter_get_string(param), "instr") == 0)
    {
      info.obsflat_type = MAT_INSTRUMENT_FLATFIELD;
    }
  else
    {
      info.obsflat_type = MAT_INSTRUMENT_FLATFIELD;
    }
  // --recalc_flat
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_flat.recalc_flat");
  info.recalc_flat = cpl_parameter_get_int(param);
  /* --gain */
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_flat.gain");
  info.gain = cpl_parameter_get_double(param);

  // Count the number of files for each mandatory/optional DO class.
  darkcount = cpl_frameset_count_tags(frameset, MATISSE_DO_OBSDARK);
  if (darkcount == 0)
    {
      cpl_msg_error(cpl_func, "SOF does not contain the mandatory file with the dark exposures.");
      mat_info_delete(&info);
      return -1;
    }
  flatcount = cpl_frameset_count_tags(frameset, MATISSE_DO_OBSFLAT);
  ffmcount  = cpl_frameset_count_tags(frameset, MATISSE_DO_FFM);
  if ((flatcount == 0) && (info.obsflat_type == MAT_INSTRUMENT_FLATFIELD))
    {
      info.obsflat_type = MAT_DETECTOR_FLATFIELD;
    }
  if ((flatcount == 0) && (ffmcount == 0) && (info.obsflat_type == MAT_DETECTOR_FLATFIELD))
    {
      info.obsflat_type = MAT_CONSTANT_FLATFIELD;
    }
  mat_load_static_maps(&info, frameset);
  /* load the mandatory dark frames */
  if (darkcount == 1)
    {
      if (mat_load_darks(&info, frameset) != CPL_ERROR_NONE)
	{
	  cpl_msg_error(cpl_func, "Error during reading the OBSDARK file.");
	  mat_info_delete(&info);
	  return -1;
	}
    }
  /* load the optional flat frames */
  if (flatcount == 1)
    {
      if (mat_load_flats(&info, frameset) != CPL_ERROR_NONE)
	{
	  cpl_msg_error(cpl_func, "Error during reading the OBSFLAT file.");
	  mat_info_delete(&info);
	  return -1;
	}
    }
  mat_create_obsflat(&info);
  switch (info.obsflat_type)
    {
    case MAT_DETECTOR_FLATFIELD:
      if (info.recalc_flat == 0)
	{
	  mat_estimate_obs_flat_detector_static(&info);
	}
      /* else */
      /* 	{ */
      /* 	  mat_estimate_obs_flat_detector_recalc(&info); */
      /* 	} */
      break;
    case MAT_INSTRUMENT_FLATFIELD:
      mat_estimate_obs_flat_instrument(&info);
      break;
    case MAT_CONSTANT_FLATFIELD:
    default:
      mat_estimate_obs_flat_constant(&info);
    }
  mat_estimate_obs_bias(&info);
  mat_copy_keywords(&info);

  /* switch (info.det->nr) { */
  /*   case 1: */
  /*     grism = cpl_propertylist_get_string(info.rawdark->keywords,"ESO INS DIL ID"); */
  /*     if (grism == NULL) */
  /* 	{ */
  /* 	  nrs = 0; */
  /* 	} */
  /*     else if (!strcmp(grism, "LOW")) */
  /* 	{ */
  /* 	  nrs=1; */
  /* 	} */
  /*     else if (!strcmp(grism, "MED")) */
  /* 	{ */
  /* 	  nrs=2; */
  /* 	} */
  /*     else if (!strcmp(grism, "HIGH")) */
  /* 	{ */
  /* 	  nrs=3; */
  /* 	} */
  /*     else if (!strcmp(grism, "HIGH+")) */
  /* 	{ */
  /* 	  nrs=4;       */
  /* 	} */
  /*     else */
  /* 	{ */
  /* 	  nrs=0; */
  /* 	} */
  /*     break; */
  /*   case 2: */
  /*     grism = cpl_propertylist_get_string(info.rawdark->keywords,"ESO INS DIN ID"); */
  /*     if (grism == NULL) */
  /* 	{ */
  /* 	  nrs = 0; */
  /* 	} */
  /*     else if (!strcmp(grism, "LOW")) */
  /* 	{ */
  /* 	  nrs=1; */
  /* 	} */
  /*     else if (!strcmp(grism, "HIGH")) */
  /* 	{ */
  /* 	  nrs=2; */
  /* 	} */
  /*     else */
  /* 	{ */
  /* 	  nrs=0; */
  /* 	} */
  /*     break; */
  /*   default: */
  /*     nrs=0; */
  /* } */

  mat_obsflat_save(info.ofm,mat_detector_get_offm_name(info.det),
  		   "mat_est_flat",parlist,frameset);

  //Append the others tables from the raw file
  darkrawframe = cpl_frameset_find(frameset, MATISSE_DO_OBSDARK);
  nbExtent = cpl_frame_get_nextensions(darkrawframe);
  for(i = 0; i < nbExtent; ++i) {
    plist=cpl_propertylist_load(cpl_frame_get_filename(darkrawframe),i+1);
    keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
    if (keyExtname != NULL) {
      // Other cases : append the table from the raw file
      if (!strcmp(keyExtname,"ARRAY_DESCRIPTION") ||
  	  !strcmp(keyExtname,"ARRAY_GEOMETRY") ||
  	  !strcmp(keyExtname,"OPTICAL_TRAIN")) {
  	table=cpl_table_load(cpl_frame_get_filename(darkrawframe),i+1,0);
  	cpl_table_save(table, NULL, plist, mat_detector_get_offm_name(info.det), CPL_IO_EXTEND);
  	cpl_table_delete(table);
      }
    }
    cpl_propertylist_delete(plist);
  }


  mat_info_delete(&info);

  /* Return */
  if (cpl_error_get_code())
    return -1;
  else
    return 0;
}

