/* $Id: mat_est_kappa.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_est_kappa.c $
 */
/*------------------------------------------------------------------------------
  Includes
  ------------------------------------------------------------------------------*/
#include <cpl.h>
#include <stdio.h>
#include "mat_drl.h"
#include "mat_kappamatrix.h"
#include "mat_apply_staticcalib.h"
#include "mat_imagingdata.h"
#include "mat_shift.h"
#include "mat_imagingdetector.h"
#include "mat_error.h"
#include "mat_const.h"
/*------------------------------------------------------------------------------
  Define
  ------------------------------------------------------------------------------*/
#define TEL_NUMBER 4
#define NB_FILE_MAX 10
#define RECIPE_NAME "mat_est_kappa"
#define MATISSE_DO_OBSFLAT         "OBS_FLATFIELD"
#define MATISSE_DO_BADPIXEL         "BADPIX"
#define MATISSE_DO_NONLIN         "NONLINEARITY"
/*------------------------------------------------------------------------------
  Prototype
  ------------------------------------------------------------------------------*/
mat_gendata * calibrate(cpl_frame *p_frame,mat_badpixel *bad, 
			mat_nonlinearity *nonlin,mat_obsflat *obsflat);
int classification(cpl_propertylist *p_list, 
		   int nb_file[], char *p_shutter_name0[],
		   char *p_shutter_name1[], char *p_shutter_name2[],
		   char *p_shutter_name3[], char *p_sky_name0[], 
		   char *p_sky_name1[], char *p_sky_name2[], 
		   char *p_sky_name3[],
		   char *p_pszfilename, int flagHotDark);


/* Plugin detailed description */
static const char *
mat_est_kappa_help = 
  "This plugin calculates from calibration measurements the intensity ratio, \n"
  "the relative shift and zoom between each photometric channel and the \n"
  "interferometric one for each spectral channel. The input data are: a \n"
  "series of images containing the laboratory background or the sky \n"
  "(KAPPA_HOTDARK or KAPPA_SKY), a series of images for each shutter \n"
  "with the internal source or the target (KAPPA_SRC or KAPPA_OBJ), the \n"
  "static shift map (SHIFT_MAP), the static bad pixel map (BADPIX), \n"
  "the static nonlinearity map (NONLINEARITY) and the observation specific \n"
  "flatfield map (OBS_FLATFIELD). This plugin produces the KAPPA_MATRIX \n"
  "and computes the QC DET<i> SHIFT<l> MEAN, QC DET<i> SHIFT<l> STDEV, \n"
  "QC DET<i> ZOOM<l> MEAN, QC DET<i> ZOOM<l> STDEV, QC DET<i> RATIO<l> MEAN, \n"
  "QC DET<i> RATIO<l> STDEV and QC DET<i> FLUX<l>."
  "\n"
  "Input Files:\n"
  "\n"
  "    DO category:           Explanation:                Required:\n"
  "    KAPPA_HOTDARK          Lab Background data         Yes\n"
  "    KAPPA_SRC              Lab Lamp data               Yes\n"
  "    SHIFT_MAP              Distortion Map              Yes\n"
  "    OBS_FLATFIELD          Observing Flat Field        Yes\n"
  "    BADPIX                 Bad Pixel Map               Yes\n"
  "    NONLINEARITY           Nonlinearity Map            Yes\n"
  "\n"
  "Output Files:\n"
  "\n"
  "    DO category:           Explanation:\n"
  "    KAPPA_MATRIX           Kappa Matrix Map\n";

static int mat_est_kappa_create(cpl_plugin *);
static int mat_est_kappa_exec(cpl_plugin *);
static int mat_est_kappa_destroy(cpl_plugin *);
static int mat_est_kappa(cpl_parameterlist *parlist, cpl_frameset *frames);



/*-----------------------------------------------------------------------------
  Functions code
  -----------------------------------------------------------------------------*/
/**
   @brief Build the list of available plugins, for this module.
   @param list the plugin list
   @return 0 if everything is ok, -1 otherwise
   Create the recipe instance and make it available to the application using the
   interface. This function is exported.
*/
/*----------------------------------------------------------------------------*/
int cpl_plugin_get_info(cpl_pluginlist *list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
  cpl_plugin *plugin = (cpl_plugin *)recipe;
  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_est_kappa",
		  "Estimation of the Kappa Matrix",
		  mat_est_kappa_help,
		  "Philippe Berio",
		  PACKAGE_BUGREPORT,
		  "GPL",
		  mat_est_kappa_create,
		  mat_est_kappa_exec,
		  mat_est_kappa_destroy);
  cpl_pluginlist_append(list, plugin);
  return 0;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Setup the recipe options
   @param plugin the plugin
   @return 0 if everything is ok
   Defining the command-line/configuration parameters for the recipe.
*/
/*----------------------------------------------------------------------------*/
static int mat_est_kappa_create(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  recipe->parameters = cpl_parameterlist_new();
    
  return 0;
}


/*----------------------------------------------------------------------------*/
/**
   @brief Destroy what has been created by the ’create’ function
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_est_kappa_destroy(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  cpl_parameterlist_delete(recipe->parameters);
  return 0;
}


/*----------------------------------------------------------------------------*/
/**
   @brief Calibrate a raw mat_gendata (badpixel, nonlinearity and optical flat
   field) and Create a new mat_gendata structure with only one row in the 
   IMAGING_DATA binary table which is the sum of the rows contained in the 
   original raw data.
   @param p_frame : the frale containing the raw data
   @param bad     : mat_badpixel structure containing the BADPIX map
   @param nonlin  : mat_nonlinearity structure containing the NONLINEARITY map
   @param obsflat : mat_obsflat structure containing the OBS_FLATFIELD map
   @return mat_gendata
*/
/*-----------------------------------------------------------------------------*/
mat_gendata * calibrate(cpl_frame *p_frame,mat_badpixel*bad, 
			mat_nonlinearity *nonlin,mat_obsflat *obsflat)
{  
  mat_gendata *p_gendata_tmp=NULL;
  mat_gendata  *p_gendata_tmp_calib=NULL;
  mat_gendata  *p_gendata=NULL;
  // Check input parameters
  mat_assert_value((p_frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no p_frame argument given");

  p_gendata=cpl_malloc(sizeof(mat_gendata));
  if (p_gendata == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for p_gendata");
    return NULL;
  }
  p_gendata_tmp = mat_gendata_load(p_frame, CPL_TYPE_FLOAT);
  if (p_gendata_tmp == NULL) {
    cpl_msg_error(cpl_func,"could not load p_gendata_tmp");
    mat_gendata_delete(p_gendata);
    return NULL;
  }
  p_gendata_tmp_calib = mat_calibration_detector(p_gendata_tmp, bad, NULL, nonlin, obsflat, NULL,
  						 /* NO_COMPENSATION, 1, 0); */
  						 FULL_COMPENSATION & ~CHANNEL_BIAS_COMPENSATION, 1, 0);
  /* p_gendata_tmp_calib = mat_apply_staticcalib(p_gendata_tmp, */
  /* 					      bad, nonlin, obsflat); */
  if (p_gendata_tmp_calib == NULL) {
    cpl_msg_error(cpl_func,"could not calibrate p_gendata_tmp_calib");
    mat_gendata_delete(p_gendata_tmp);
    mat_gendata_delete(p_gendata);
    return NULL;
  }
  mat_gendata_delete(p_gendata_tmp);
  (*p_gendata).keywords =
    cpl_propertylist_duplicate((*p_gendata_tmp_calib).keywords);
  (*p_gendata).imgdet =
    mat_imagingdetector_duplicate((*p_gendata_tmp_calib).imgdet);
  if (p_gendata->imgdet == NULL) {
    cpl_msg_error(cpl_func,"could not duplicate p_gendata->imgdet");
    mat_gendata_delete(p_gendata);
    return NULL;
  }
  (*p_gendata).array =
    mat_array_duplicate((*p_gendata_tmp_calib).array);
  if (p_gendata->array == NULL) {
    cpl_msg_warning(cpl_func,"could not duplicate p_gendata->array");
    /* return NULL; */
  }
  (*p_gendata).imgdata =
    mat_imagingdata_sum((*p_gendata_tmp_calib).imgdata, CPL_TYPE_FLOAT);
  if (p_gendata->imgdata == NULL) {
    cpl_msg_error(cpl_func,"could not sum p_gendata->imgdata");
    mat_gendata_delete(p_gendata);
    return NULL;
  }
  mat_gendata_delete(p_gendata_tmp_calib);


  return p_gendata;
}


/*----------------------------------------------------------------------------*/
/**
   @brief Classify the frames with respect to the shutter (open or close)
   @param p_list : primary header of the frame
   @param nb_file : array of int containing the number of frames for each 
   kind of frames (Dimension = 5, 1 for each sutter and 
   1 for the Sky)
   @param p_shutter_name0 : array of char containing the filename for shutter 1 
   @param p_shutter_name1 : array of char containing the filename for shutter 2
   @param p_shutter_name2 : array of char containing the filename for shutter 3
   @param p_shutter_name3 : array of char containing the filename for shutter 4
   @param p_sky_name0 : array of char containing the filename for the sky for shutter 1 
   @param p_sky_name1 : array of char containing the filename for the sky for shutter 2 
   @param p_sky_name2 : array of char containing the filename for the sky for shutter 3 
   @param p_sky_name3 : array of char containing the filename for the sky for shutter 4 
   @param p_pszfilename : filename of the frame
   @param flagHotDark : 1 if HotDark, 0 otherwise
   @return 0 (success) or -1 (failure)
*/
/*-----------------------------------------------------------------------------*/
int classification(cpl_propertylist *p_list, int nb_file[],  
		   char *p_shutter_name0[],
		   char *p_shutter_name1[], char *p_shutter_name2[],
		   char *p_shutter_name3[], char *p_sky_name0[],
		   char *p_sky_name1[], char *p_sky_name2[],
		   char *p_sky_name3[],
		   char *p_pszfilename, int flagHotDark)
{
  char *p_shutter1 = NULL;
  char *p_shutter2 = NULL;
  char *p_shutter3 = NULL;
  char *p_shutter4 = NULL;
  /* int nb_shutter_dark=0; */
  int flagBool=0;
  char *p_chipname=NULL;

  // Check input parameters
  mat_assert_value((p_list!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no p_list argument given");
  mat_assert_value((p_pszfilename!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no p_pszfilename argument given");
  mat_assert_value((p_shutter_name0!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no p_shutter_name0 argument given");
  mat_assert_value((p_shutter_name1!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no p_shutter_name1 argument given");
  mat_assert_value((p_shutter_name2!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no p_shutter_name2 argument given");
  mat_assert_value((p_shutter_name3!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no p_shutter_name3 argument given");
  mat_assert_value((p_sky_name0!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no p_sky_name argument given");
  mat_assert_value((p_sky_name1!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no p_sky_name argument given");
  mat_assert_value((p_sky_name2!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no p_sky_name argument given");
  mat_assert_value((p_sky_name3!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no p_sky_name argument given");
  

  p_chipname =
    (char *)cpl_propertylist_get_string(p_list,"ESO DET CHIP NAME");

  if(strcmp(p_chipname, "AQUARIUS") == 0) 
    {
     if (cpl_propertylist_get_type(p_list,"ESO INS BSN1 ST") == CPL_TYPE_STRING) {
	p_shutter1 = 
	  (char *)cpl_propertylist_get_string(p_list,"ESO INS BSN1 ST");
	p_shutter2 = 
	  (char *)cpl_propertylist_get_string(p_list,"ESO INS BSN2 ST");
	p_shutter3 = 
	  (char *)cpl_propertylist_get_string(p_list,"ESO INS BSN3 ST");
	p_shutter4 = 
	  (char *)cpl_propertylist_get_string(p_list,"ESO INS BSN4 ST");
      } else {
       flagBool=1;
	if (cpl_propertylist_get_bool(p_list,"ESO INS BSN1 ST")) {
	  p_shutter1=cpl_sprintf("T");
	} else {
	  p_shutter1=cpl_sprintf("F");
	}
	if (cpl_propertylist_get_bool(p_list,"ESO INS BSN2 ST")) {
	  p_shutter2=cpl_sprintf("T");
	} else {
	  p_shutter2=cpl_sprintf("F");
	}
	if (cpl_propertylist_get_bool(p_list,"ESO INS BSN3 ST")) {
	  p_shutter3=cpl_sprintf("T");
	} else {
	  p_shutter3=cpl_sprintf("F");
	}
	if (cpl_propertylist_get_bool(p_list,"ESO INS BSN4 ST")) {
	  p_shutter4=cpl_sprintf("T");
	} else {
	  p_shutter4=cpl_sprintf("F");
	}
      }
    }
  else if(strcmp(p_chipname, "HAWAII-2RG") == 0 || strcmp(p_chipname, "HAWAII2RG") == 0) 
    {
     if (cpl_propertylist_get_type(p_list,"ESO INS BSL1 ST") == CPL_TYPE_STRING) {
       p_shutter1 = 
	 (char *)cpl_propertylist_get_string(p_list,"ESO INS BSL1 ST");
       p_shutter2 = 
	 (char *)cpl_propertylist_get_string(p_list,"ESO INS BSL2 ST");
       p_shutter3 = 
	 (char *)cpl_propertylist_get_string(p_list,"ESO INS BSL3 ST");
       p_shutter4 = 
	 (char *)cpl_propertylist_get_string(p_list,"ESO INS BSL4 ST");
     } else {
       flagBool=1;
	if (cpl_propertylist_get_bool(p_list,"ESO INS BSL1 ST")) {
	  p_shutter1=cpl_sprintf("T");
	} else {
	  p_shutter1=cpl_sprintf("F");
	}
	if (cpl_propertylist_get_bool(p_list,"ESO INS BSL2 ST")) {
	  p_shutter2=cpl_sprintf("T");
	} else {
	  p_shutter2=cpl_sprintf("F");
	}
	if (cpl_propertylist_get_bool(p_list,"ESO INS BSL3 ST")) {
	  p_shutter3=cpl_sprintf("T");
	} else {
	  p_shutter3=cpl_sprintf("F");
	}
	if (cpl_propertylist_get_bool(p_list,"ESO INS BSL4 ST")) {
	  p_shutter4=cpl_sprintf("T");
	} else {
	  p_shutter4=cpl_sprintf("F");
	}
      }       
    }
  else{
    cpl_msg_info(cpl_func,"erreur");
  }
  if(p_shutter1 == NULL && p_shutter2 == NULL && p_shutter3 == NULL 
     && p_shutter4 == NULL) 
    {
      cpl_msg_warning(cpl_func, "No shutter found");
      cpl_free(p_shutter2);
      cpl_free(p_shutter3);
      cpl_free(p_shutter4);
      return -1;
    }
  /* if (strcmp(p_shutter1, "T") == 0 && */
  /*     strcmp(p_shutter2, "T") == 0 && */
  /*     strcmp(p_shutter3, "T") == 0 && */
  /*     strcmp(p_shutter4, "T") == 0) { */
  if (flagHotDark) {
    if(strcmp(p_shutter1, "T") == 0)
      { 
	p_sky_name0[nb_file[0]] = p_pszfilename;
	nb_file[0] = nb_file[0]+1;
      }
    else if(strcmp(p_shutter2, "T") == 0) 
      { 
	p_sky_name1[nb_file[1]] = p_pszfilename;
	nb_file[1] = nb_file[1]+1;
      }
    else if(strcmp(p_shutter3, "T") == 0) 
      {
	p_sky_name2[nb_file[2]] = p_pszfilename;
	nb_file[2] = nb_file[2]+1;
      }
    else if(strcmp(p_shutter4, "T") == 0) 
      {
	p_sky_name3[nb_file[3]] = p_pszfilename;
	nb_file[3] = nb_file[3]+1;
      }
  } else {
    if(strcmp(p_shutter1, "T") == 0)
      { 
	p_shutter_name0[nb_file[4]] = p_pszfilename;
	nb_file[4] = nb_file[4]+1;
      }
    else if(strcmp(p_shutter2, "T") == 0) 
      { 
	p_shutter_name1[nb_file[5]] = p_pszfilename;
	nb_file[5] = nb_file[5]+1;
      }
    else if(strcmp(p_shutter3, "T") == 0) 
      {
	p_shutter_name2[nb_file[6]] = p_pszfilename;
	nb_file[6] = nb_file[6]+1;
      }
    else if(strcmp(p_shutter4, "T") == 0) 
      {
	p_shutter_name3[nb_file[7]] = p_pszfilename;
	nb_file[7] = nb_file[7]+1;
      }
    /* else */
    /*   { */
    /* 	if(strcmp(p_shutter1, "F") == 0)  */
    /* 	  nb_shutter_dark++; */
    /* 	if(strcmp(p_shutter2, "F") == 0)  */
    /* 	  nb_shutter_dark++; */
    /* 	if(strcmp(p_shutter3, "F") == 0)  */
    /* 	  nb_shutter_dark++; */
    /* 	if(strcmp(p_shutter4, "F") == 0) 	     */
    /* 	  nb_shutter_dark++; */
    /* 	if (nb_shutter_dark>=4)  */
    /* 	  { */
    /* 	    p_sky_name[nb_file[0]] = p_pszfilename; */
    /* 	    nb_file[0] = nb_file[0]+1; */
    /* 	  } */
    /*   } */
  }
  if (flagBool) {
    cpl_free(p_shutter1);
    cpl_free(p_shutter2);
    cpl_free(p_shutter3);
    cpl_free(p_shutter4);
  }

  return 0;
}



/*----------------------------------------------------------------------------*/
/**
   @brief Execute the plugin instance given by the interface. This plugin
   calculates from calibration measurements the intensity ratio, the relative 
   shift 
   and zoom between each photometric channel and the interferometric one for 
   each 
   spectral channel. The input data are: a series of images containing the 
   laboratory background or the sky (KAPPA_HOTDARK or KAPPA_SKY), a series of 
   images for each shutter with the internal source or the target (KAPPA_SRC or 
   KAPPA_OBJ), the static shift map (SHIFT_MAP), the static bad pixel map 
   (BADPIX), 
   the static nonlinearity map (NONLINEARITY) and the observation specific 
   flatfield map (OBS_FLATFIELD). This plugin produces the 
   KAPPA_MATRIX and computes the QC DETi SHIFTl MEAN, QC DETi SHIFTl STDEV, 
   QC DETi ZOOMl MEAN, QC DETi ZOOMl STDEV, QC DETi RATIOl MEAN, 
   QC DETi RATIOl STDEV and QC DETi FLUXl.
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_est_kappa_exec(cpl_plugin *plugin)
{ 
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) {
    recipe=(cpl_recipe *)plugin;
  } else {
    return -1;
  }
  return mat_est_kappa(recipe->parameters,recipe->frames);
}


int mat_est_kappa(cpl_parameterlist *parlist, cpl_frameset *frames) {
  cpl_frame     *cur_frame=NULL;
  cpl_frame     *first_frame=NULL;
  cpl_frameset *usedframes=NULL;
  cpl_frame *sel_frame=NULL;
  cpl_table *table=NULL;
  cpl_propertylist *p_list=NULL;
  cpl_propertylist *p_list_kappa=NULL;
  mat_gendata **pp_shutter1=NULL;
  mat_gendata **pp_shutter2=NULL;
  mat_gendata **pp_shutter3=NULL;
  mat_gendata **pp_shutter4=NULL;
  mat_gendata **pp_sky_shutter1=NULL;
  mat_gendata **pp_sky_shutter2=NULL;
  mat_gendata **pp_sky_shutter3=NULL;
  mat_gendata **pp_sky_shutter4=NULL;
  mat_gendata  *p_gendata_tmp=NULL;
  mat_shiftmap *p_shiftmap=NULL;
  /* int background=0; */
  int positionSky1=0;
  int positionSky2=0;
  int positionSky3=0;
  int positionSky4=0;
  int position1 = 0;
  int position2 = 0;
  int position3 = 0;
  int position4 = 0;
  int i=0;
  int  hotdark=0;
  int flag_sky=0;
  int ksrc=0;
  int kobj=0;
  int nb_extent=0;
  int nb_file[8]={0,0,0,0,0,0,0,0};
  char *p_filename=NULL;
  char *p_filetag=NULL;
  char *p_shutter_name1[NB_FILE_MAX];
  char *p_shutter_name2[NB_FILE_MAX];
  char *p_shutter_name3[NB_FILE_MAX];
  char *p_shutter_name4[NB_FILE_MAX];
  char *p_sky_name1[NB_FILE_MAX];
  char *p_sky_name2[NB_FILE_MAX];
  char *p_sky_name3[NB_FILE_MAX];
  char *p_sky_name4[NB_FILE_MAX];
  char *p_key_extname=NULL;
  char filenameFits[]="KAPPA_MATRIX.fits";
  mat_kappamatrix *p_kappamatrix=NULL;

  mat_badpixel *bpm=NULL;
  mat_nonlinearity *nonlin=NULL;
  mat_obsflat *obsflat=NULL;
  cpl_frame *bpmframe = NULL;
  int bpmcount=0;
  cpl_frame *nonlinframe = NULL;
  int nonlincount=0;
  cpl_frame *obsflatframe = NULL;
  int obsflatcount=0;
  cpl_frameset_iterator *it=NULL;
  cpl_errorstate prestate = cpl_errorstate_get();
  
  
  usedframes=cpl_frameset_new(); 
  // LOAD STATIC CALIB MAPS
  bpmcount = cpl_frameset_count_tags(frames, MATISSE_DO_BADPIXEL);
  if (bpmcount > 1) {
    cpl_msg_error(cpl_func, "only zero or one badpixel map is allowed in the SOF");
    cpl_frameset_delete(usedframes);    
    return -1;
  }
  /* load the optional badpixel map */
  if (bpmcount == 1) {
    bpmframe = cpl_frameset_find(frames, MATISSE_DO_BADPIXEL); // already checked previously!
    cpl_frame_set_group(bpmframe, CPL_FRAME_GROUP_CALIB);
    sel_frame=cpl_frame_duplicate(bpmframe);
    cpl_frameset_insert(usedframes,sel_frame);
    bpm = mat_badpixel_load(bpmframe);
  }
  nonlincount = cpl_frameset_count_tags(frames, MATISSE_DO_NONLIN);
  if (nonlincount > 1) {
    cpl_msg_error(cpl_func, "only zero or one nonlin map is allowed in the SOF");
    cpl_frameset_delete(usedframes);    
    return -1;
  }
  /* load the optional nonlin map */
  if (nonlincount == 1) {
    nonlinframe = cpl_frameset_find(frames, MATISSE_DO_NONLIN); // already checked previously!
    cpl_frame_set_group(nonlinframe, CPL_FRAME_GROUP_CALIB);
    sel_frame=cpl_frame_duplicate(nonlinframe);
    cpl_frameset_insert(usedframes,sel_frame);
    nonlin = mat_nonlinearity_load(nonlinframe);
  }
  obsflatcount = cpl_frameset_count_tags(frames, MATISSE_DO_OBSFLAT);
  if (obsflatcount > 1) {
    cpl_msg_error(cpl_func, "only one flatfield map is allowed in the SOF");
    cpl_frameset_delete(usedframes);    
    return -1;
  }
  /* load the optional flatfield map */
  if (obsflatcount == 1) {
    obsflatframe = cpl_frameset_find(frames, MATISSE_DO_OBSFLAT); // already checked previously!
    cpl_frame_set_group(obsflatframe, CPL_FRAME_GROUP_CALIB);
    sel_frame=cpl_frame_duplicate(obsflatframe);
    cpl_frameset_insert(usedframes,sel_frame);
    obsflat = mat_obsflat_load(obsflatframe);
  }

  // 1. Check the data type of the input frames. 
  // Retrieve frames files
  p_list = NULL;
  p_filename = NULL;
  p_filetag = NULL;
  p_shiftmap = NULL;
  
  flag_sky = 0;   
  hotdark = 0;
  ksrc = 0;
  kobj = 0;
  
  

  prestate=cpl_errorstate_get();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      // Get Classification Tag
      p_filename = (char *)cpl_frame_get_filename(cur_frame);
      p_filetag  = (char *)cpl_frame_get_tag(cur_frame);
      p_list = cpl_propertylist_load(p_filename,0);
      if (strcmp(p_filetag,"KAPPA_HOTDARK") == 0) {
	if (classification(p_list, nb_file,
			   p_shutter_name1, p_shutter_name2,
			   p_shutter_name3, p_shutter_name4,
			   p_sky_name1, p_sky_name2,
			   p_sky_name3, p_sky_name4,
			   p_filename, 1) == 0) {
	  hotdark++;
	}
	sel_frame=cpl_frame_duplicate(cur_frame);
	if (first_frame == NULL)
	  {
	    first_frame=cpl_frame_duplicate(cur_frame);
	  }
	cpl_frame_set_group(sel_frame,CPL_FRAME_GROUP_RAW);
	cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
	cpl_frameset_insert(usedframes,sel_frame);
      }
      if (strcmp(p_filetag,"KAPPA_SKY") == 0 ) {
	if (classification(p_list, nb_file, 
			   p_shutter_name1, p_shutter_name2,
			   p_shutter_name3, p_shutter_name4,
			   p_sky_name1, p_sky_name2,
			   p_sky_name3, p_sky_name4,
			   p_filename, 1) == 0) {
	  flag_sky++;
	}
	sel_frame=cpl_frame_duplicate(cur_frame);
	cpl_frame_set_group(sel_frame,CPL_FRAME_GROUP_RAW);
	cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
	cpl_frameset_insert(usedframes,sel_frame);
      }
      if (strcmp(p_filetag,"KAPPA_SRC") == 0) {
	if (classification(p_list, nb_file, 
			   p_shutter_name1, p_shutter_name2,
			   p_shutter_name3, p_shutter_name4,
			   p_sky_name1, p_sky_name2,
			   p_sky_name3, p_sky_name4,
			   p_filename, 0) == 0) {
	  ksrc++;
	}
	sel_frame=cpl_frame_duplicate(cur_frame);
	cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
	cpl_frame_set_group(sel_frame,CPL_FRAME_GROUP_RAW);
	cpl_frameset_insert(usedframes,sel_frame);
      }
      if (strcmp(p_filetag,"KAPPA_OBJ") == 0) {
	if (classification(p_list, nb_file, 
			   p_shutter_name1, p_shutter_name2,
			   p_shutter_name3, p_shutter_name4,
			   p_sky_name1, p_sky_name2,
			   p_sky_name3, p_sky_name4,
			   p_filename, 0) == 0) {
	  kobj++;
	}
	sel_frame=cpl_frame_duplicate(cur_frame);
	cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
	cpl_frame_set_group(sel_frame,CPL_FRAME_GROUP_RAW);
	cpl_frameset_insert(usedframes,sel_frame);
      }
      if (strcmp(p_filetag,"SHIFT_MAP") == 0) {
	p_shiftmap = mat_shiftmap_load(cur_frame);
	sel_frame=cpl_frame_duplicate(cur_frame);
	cpl_frame_set_group(sel_frame, CPL_FRAME_GROUP_CALIB);
	cpl_frameset_insert(usedframes,sel_frame);
      }
      cpl_propertylist_delete(p_list);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }

  if (p_shiftmap == NULL) {
    cpl_msg_warning(cpl_func, "SHIFT MAP not loaded");
    return -1;
  }

  // Verify files type
  if ((ksrc>0) && (kobj>0)) {
    cpl_msg_warning(cpl_func,"bad picture type");
    return -1;
  }
  if ((hotdark>0) && (flag_sky>0)) {
    cpl_msg_warning(cpl_func,"bad file type");
    return -1;
  }
  if ((ksrc>0) && (hotdark==0)) {
    cpl_msg_warning(cpl_func,"missing dark file");
    return -1;
  }
  if ((kobj>1) && (flag_sky==0)) {
    cpl_msg_warning(cpl_func,"missing dark file");
    return -1;
  }
  if ((nb_file[1]==0) ||(nb_file[2]==0) || (nb_file[3]==0) ||(nb_file[4]==0)) {
    cpl_msg_warning(cpl_func, "missing sky shutter %d %d %d %d",
		    nb_file[0], nb_file[1], nb_file[2], nb_file[3]);
    return -1;
  }
  if ((nb_file[4]==0) ||(nb_file[5]==0) || (nb_file[6]==0) ||(nb_file[7]==0)) {
    cpl_msg_warning(cpl_func, "missing shutter %d %d %d %d",
		    nb_file[4], nb_file[5], nb_file[6], nb_file[7]);
    return -1;
  }



  // 2. Load input frames in DRL structures. 
  // Allocate mat_gendata structures according to the number of file
  
  /* if(hotdark>0) */
  /*   background=hotdark; */
  /* else */
  /*   background=flag_sky; */
  pp_shutter1 = cpl_calloc( nb_file[4] , sizeof(mat_gendata*));
  if (pp_shutter1 == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for pp_shutter1");
    cpl_frame_delete(first_frame);
    return -1;
  }
  pp_shutter2 = cpl_calloc( nb_file[5] , sizeof(mat_gendata*));
  if (pp_shutter2 == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for pp_shutter2");
    cpl_free(pp_shutter1);
    cpl_frame_delete(first_frame);
    return -1;
  }
  pp_shutter3 = cpl_calloc( nb_file[6] , sizeof(mat_gendata*));
  if (pp_shutter3 == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for pp_shutter3");
    cpl_free(pp_shutter1);
    cpl_free(pp_shutter2);
    cpl_frame_delete(first_frame);
    return -1;
  }
  pp_shutter4 = cpl_calloc( nb_file[7] , sizeof(mat_gendata*));
  if (pp_shutter4 == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for pp_shutter4");
    cpl_free(pp_shutter1);
    cpl_free(pp_shutter2);
    cpl_free(pp_shutter3);
    cpl_frame_delete(first_frame);
    return -1;
  }
  pp_sky_shutter1 = cpl_calloc( nb_file[0], sizeof(mat_gendata*));
  if (pp_sky_shutter1 == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for pp_sky_shutter1");
    cpl_free(pp_shutter1);
    cpl_free(pp_shutter2);
    cpl_free(pp_shutter3);
    cpl_free(pp_shutter4);
    return -1;
  }
  pp_sky_shutter2 = cpl_calloc( nb_file[1], sizeof(mat_gendata*));
  if (pp_sky_shutter2 == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for pp_sky_shutter2");
    cpl_free(pp_shutter1);
    cpl_free(pp_shutter2);
    cpl_free(pp_shutter3);
    cpl_free(pp_shutter4);
    cpl_free(pp_sky_shutter1);
    cpl_frame_delete(first_frame);
    return -1;
  }
  pp_sky_shutter3 = cpl_calloc( nb_file[2], sizeof(mat_gendata*));
  if (pp_sky_shutter3 == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for pp_sky_shutter3");
    cpl_free(pp_shutter1);
    cpl_free(pp_shutter2);
    cpl_free(pp_shutter3);
    cpl_free(pp_shutter4);
    cpl_free(pp_sky_shutter1);
    cpl_free(pp_sky_shutter2);
    return -1;
  }
  pp_sky_shutter4 = cpl_calloc( nb_file[3], sizeof(mat_gendata*));
  if (pp_sky_shutter4 == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for pp_sky_shutter4");
    cpl_free(pp_shutter1);
    cpl_free(pp_shutter2);
    cpl_free(pp_shutter3);
    cpl_free(pp_shutter4);
    cpl_free(pp_sky_shutter1);
    cpl_free(pp_sky_shutter2);
    cpl_free(pp_sky_shutter3);
    cpl_frame_delete(first_frame);
    return -1;
  }
  p_filename = NULL;
  p_filetag = NULL;
  
  prestate=cpl_errorstate_get();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      p_filename = (char *)cpl_frame_get_filename(cur_frame);
      p_filetag  = (char *)cpl_frame_get_tag(cur_frame);
      //calibrate mat_gendata
      if (strcmp(p_filetag,"KAPPA_HOTDARK") == 0 ||
	  strcmp(p_filetag, "KAPPA_SKY") == 0) {
	i=0;
	while(i<nb_file[0]) {
	  if(strcmp(p_filename, p_sky_name1[i]) == 0) {
	    if (p_shiftmap == NULL) {
	      pp_sky_shutter1[positionSky1] = calibrate(cur_frame, bpm, nonlin, obsflat);
	    } else {
	      p_gendata_tmp=calibrate(cur_frame, bpm, nonlin, obsflat);
	      pp_sky_shutter1[positionSky1] =mat_apply_shift(p_gendata_tmp,p_shiftmap,0);
	      if (pp_sky_shutter1[positionSky1] == NULL) {
		return CPL_ERROR_INCOMPATIBLE_INPUT;
	      }
	      mat_gendata_delete(p_gendata_tmp);
	    }
	    positionSky1++;
	  }
	  i++;
	}
	i=0;
	while(i<nb_file[1]) {
	  if(strcmp(p_filename, p_sky_name2[i]) == 0) {
	    if (p_shiftmap == NULL) {
	      pp_sky_shutter2[positionSky2] = calibrate(cur_frame, bpm, nonlin, obsflat);
	    } else {
	      p_gendata_tmp=calibrate(cur_frame, bpm, nonlin, obsflat);
	      pp_sky_shutter2[positionSky2] =mat_apply_shift(p_gendata_tmp,p_shiftmap,0);
	      if (pp_sky_shutter2[positionSky2] == NULL) {
		return CPL_ERROR_INCOMPATIBLE_INPUT;
	      }
	      mat_gendata_delete(p_gendata_tmp);
	    }
	    positionSky2++;
	  }
	  i++;
	}
	i=0;
	while(i<nb_file[2]) {
	  if(strcmp(p_filename, p_sky_name3[i]) == 0) {
	    if (p_shiftmap == NULL) {
	      pp_sky_shutter3[positionSky3] = calibrate(cur_frame, bpm, nonlin, obsflat);
	    } else {
	      p_gendata_tmp=calibrate(cur_frame, bpm, nonlin, obsflat);
	      pp_sky_shutter3[positionSky3] = mat_apply_shift(p_gendata_tmp,p_shiftmap,0);
	      if (pp_sky_shutter3[positionSky3] == NULL) {
		return CPL_ERROR_INCOMPATIBLE_INPUT;
	      }
	      mat_gendata_delete(p_gendata_tmp);
	    }
	    positionSky3++;
	  }
	  i++;
	}
	i=0;
	while(i<nb_file[3]) {
	  if(strcmp(p_filename, p_sky_name4[i]) == 0) {
	    if (p_shiftmap == NULL) {
	      pp_sky_shutter4[positionSky4] = calibrate(cur_frame, bpm, nonlin, obsflat);
	    } else {
	      p_gendata_tmp=calibrate(cur_frame, bpm, nonlin, obsflat);
	      pp_sky_shutter4[positionSky4] = mat_apply_shift(p_gendata_tmp,p_shiftmap,0);
	      if (pp_sky_shutter4[positionSky4] == NULL) {
		return CPL_ERROR_INCOMPATIBLE_INPUT;
	      }
	      mat_gendata_delete(p_gendata_tmp);
	    }
	    positionSky4++;
	  }
	  i++;
	}

      } else if  (strcmp(p_filetag,"KAPPA_OBJ") == 0 ||
		  strcmp(p_filetag, "KAPPA_SRC") == 0 ) {
	/* p_list = cpl_propertylist_load(p_filename,0); */
	i=0;
	while(i<nb_file[4]) {
	  if(strcmp(p_filename, p_shutter_name1[i]) == 0) {
	    if (p_shiftmap == NULL) {
	      pp_shutter1[position1] = calibrate(cur_frame, bpm, nonlin, obsflat);
	    } else {
	      p_gendata_tmp=calibrate(cur_frame, bpm, nonlin, obsflat);
	      pp_shutter1[position1] = mat_apply_shift(p_gendata_tmp,p_shiftmap,0);
	      if (pp_shutter1[position1] == NULL) {
		return CPL_ERROR_INCOMPATIBLE_INPUT;
	      }
	      mat_gendata_delete(p_gendata_tmp);
	    }
	    position1++;
	  }
	  i++;
	}
	i=0;
	while(i<nb_file[5]) {
	  if(strcmp(p_filename, p_shutter_name2[i]) == 0) {
	    if (p_shiftmap == NULL) {
	      pp_shutter2[position2] =calibrate(cur_frame, bpm, nonlin, obsflat);
	    } else {
	      p_gendata_tmp=calibrate(cur_frame, bpm, nonlin, obsflat);
	      pp_shutter2[position2] = mat_apply_shift(p_gendata_tmp,p_shiftmap,0);
	      if (pp_shutter2[position2] == NULL) {
		return CPL_ERROR_INCOMPATIBLE_INPUT;
	      }
	      mat_gendata_delete(p_gendata_tmp);
	    }
	    position2++;
	  }
	  i++;
	}
	i=0;
	while(i<nb_file[6]) {
	  if(strcmp(p_filename, p_shutter_name3[i]) == 0) {
	    if (p_shiftmap == NULL) {
	      pp_shutter3[position3] = calibrate(cur_frame, bpm, nonlin, obsflat);
	    } else {
	      p_gendata_tmp = calibrate(cur_frame, bpm, nonlin, obsflat);
	      pp_shutter3[position3] = mat_apply_shift(p_gendata_tmp,p_shiftmap,0);
	      if (pp_shutter3[position3] == NULL) {
		return CPL_ERROR_INCOMPATIBLE_INPUT;
	      }
	      mat_gendata_delete(p_gendata_tmp);
	    }
	    position3++;
	  }
	  i++;
	}
	i=0;
	while(i<nb_file[7]) {
	  if(strcmp(p_filename, p_shutter_name4[i]) == 0) {
	    if (p_shiftmap == NULL) {
	      pp_shutter4[position4] = calibrate(cur_frame, bpm, nonlin, obsflat);
	    } else {
	      p_gendata_tmp = calibrate(cur_frame, bpm, nonlin, obsflat);
	      pp_shutter4[position4] = mat_apply_shift(p_gendata_tmp,p_shiftmap,0);
	      if (pp_shutter4[position4] == NULL) {
		return CPL_ERROR_INCOMPATIBLE_INPUT;
	      }
	      mat_gendata_delete(p_gendata_tmp);
	    }
	    position4++;
	  }
	  i++;
	}
	/* cpl_propertylist_delete(p_list); */
      
      } 
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }


  p_kappamatrix = mat_compute_kappamatrix(pp_shutter1, pp_shutter2,
					  pp_shutter3, pp_shutter4, 
					  pp_sky_shutter1,
					  pp_sky_shutter2,
					  pp_sky_shutter3,
					  pp_sky_shutter4,
					  nb_file);
  if(p_kappamatrix == NULL) {
    cpl_msg_warning(cpl_func, "NULL pointer returned by mat_compute_kappa.");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }
  // 5. Save the mat_shiftmap structure in a FITS file
  // 5.1 Create the file with primary header and KAPPA_TABLE
  prestate=cpl_errorstate_get();
  /* it= cpl_frameset_iterator_new(usedframes); */
  /* cur_frame = cpl_frameset_iterator_get(it); */
  /* cpl_frameset_iterator_delete(it); */
  /* nb_extent = cpl_frame_get_nextensions(cur_frame); */
  nb_extent = cpl_frame_get_nextensions(first_frame);
  // Find the table IMAGING_DATA
  for(i=0; i<nb_extent; i++) {
    /* p_list = cpl_propertylist_load(cpl_frame_get_filename(cur_frame),i+1); */
    p_list = cpl_propertylist_load(cpl_frame_get_filename(first_frame),i+1);
    p_key_extname = (char *)cpl_propertylist_get_string(p_list,"EXTNAME");
    if (!strcmp(p_key_extname,"IMAGING_DATA")) {
      p_list_kappa = cpl_propertylist_duplicate(p_list);
      // Rename the table IMAGING_DATA to KAPPA_MATRIX
      cpl_propertylist_update_string(p_list_kappa, "EXTNAME","KAPPA_MATRIX");
      // Substract the number of calibration regions to the number of regions
      cpl_propertylist_update_int(p_list_kappa, "NREGION",
				  p_kappamatrix->imgdet->nbregion);

      cpl_propertylist_append_double(p_kappamatrix->keywords,"PRO DISP COEF0",p_shiftmap->dispcoef[0]);
      cpl_propertylist_append_double(p_kappamatrix->keywords,"PRO DISP COEF1",p_shiftmap->dispcoef[1]);
      cpl_propertylist_append_double(p_kappamatrix->keywords,"PRO DISP COEF2",p_shiftmap->dispcoef[2]);
      mat_kappamatrix_save(p_kappamatrix,p_list_kappa,frames,usedframes,
			   parlist,RECIPE_NAME,
			   "KAPPA_MATRIX","IMAGE","F",filenameFits);
      cpl_propertylist_delete(p_list_kappa);
    }
    cpl_propertylist_delete(p_list);
  }
  
  // 5.2 Append the others tables from the raw file except IMAGING_DATA
  for(i=0;i<nb_extent;i++) {
    /* p_list=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),i+1); */
    p_list=cpl_propertylist_load(cpl_frame_get_filename(first_frame),i+1);
    p_key_extname=(char *)cpl_propertylist_get_string(p_list,"EXTNAME");
    if (p_key_extname != NULL) {
      /* table=cpl_table_load(cpl_frame_get_filename(cur_frame),i+1,0); */
      table=cpl_table_load(cpl_frame_get_filename(first_frame),i+1,0);
      // Case IMAGING_DETECTOR : Append the table without calibration regions
      if (!strcmp(p_key_extname,"IMAGING_DETECTOR")) {
	mat_imagingdetector_save(p_kappamatrix->imgdet, NULL, filenameFits);
      }
      // Other cases : append the table from the raw file
      if (!strcmp(p_key_extname,"ARRAY_DESCRIPTION") ||
	  !strcmp(p_key_extname,"ARRAY_GEOMETRY") ||
	  !strcmp(p_key_extname,"OPTICAL_TRAIN")) {
	cpl_table_save(table, NULL, p_list, filenameFits, CPL_IO_EXTEND);
      }
      cpl_table_delete(table);
    }
    cpl_propertylist_delete(p_list);
  }

  

  mat_shiftmap_free(p_shiftmap);
  mat_badpixel_delete(bpm);
  mat_nonlinearity_delete(nonlin);
  mat_obsflat_delete(obsflat);

  cpl_frame_delete(first_frame);

  if ((*p_kappamatrix).keywords != NULL) cpl_propertylist_delete((*p_kappamatrix).keywords);
  if ((*p_kappamatrix).imgdet != NULL) mat_imagingdetector_delete((*p_kappamatrix).imgdet);
  for(i=0; i<p_kappamatrix->nbbeam; i++) {
    if (p_kappamatrix->list_kappacoef[i]->coef != NULL) cpl_matrix_delete(p_kappamatrix->list_kappacoef[i]->coef);
    if (p_kappamatrix->list_kappacoef[i]->errcoef != NULL) cpl_matrix_delete(p_kappamatrix->list_kappacoef[i]->errcoef);
    if ((*p_kappamatrix).list_kappacoef[i] != NULL) cpl_free((*p_kappamatrix).list_kappacoef[i]);
  }
  if ((*p_kappamatrix).list_kappacoef != NULL) cpl_free((*p_kappamatrix).list_kappacoef);
  if (p_kappamatrix != NULL) cpl_free(p_kappamatrix);


  if (usedframes != NULL) cpl_frameset_delete(usedframes);

  for(i=0; i<nb_file[0]; i++) {
    if (pp_sky_shutter1[i] != NULL) mat_gendata_delete(pp_sky_shutter1[i]);
  }
  if (pp_sky_shutter1 != NULL) cpl_free(pp_sky_shutter1);

  for(i=0; i<nb_file[1]; i++) {
    if (pp_sky_shutter2[i] != NULL) mat_gendata_delete(pp_sky_shutter2[i]);
  }
  if (pp_sky_shutter2 != NULL) cpl_free(pp_sky_shutter2);

  for(i=0; i<nb_file[2]; i++) {
    if (pp_sky_shutter3[i] != NULL) mat_gendata_delete(pp_sky_shutter3[i]);
  }
  if (pp_sky_shutter3 != NULL) cpl_free(pp_sky_shutter3);

  for(i=0; i<nb_file[3]; i++) {
    if (pp_sky_shutter4[i] != NULL) mat_gendata_delete(pp_sky_shutter4[i]);
  }
  if (pp_sky_shutter4 != NULL) cpl_free(pp_sky_shutter4);

  for(i=0; i<nb_file[4]; i++) {
    if (pp_shutter1[i] != NULL) mat_gendata_delete(pp_shutter1[i]);
  }
  if (pp_shutter1 != NULL) cpl_free(pp_shutter1);

  for(i=0; i<nb_file[5]; i++) {
    if (pp_shutter2[i] != NULL) mat_gendata_delete(pp_shutter2[i]);
  }
  if (pp_shutter2 != NULL) cpl_free(pp_shutter2);

  for(i=0; i<nb_file[6]; i++) {
    if (pp_shutter3[i] != NULL) mat_gendata_delete(pp_shutter3[i]);
  }
  if (pp_shutter3 != NULL) cpl_free(pp_shutter3);

  for(i=0; i<nb_file[7]; i++) {
    if (pp_shutter4[i] != NULL) mat_gendata_delete(pp_shutter4[i]);
  }
  if (pp_shutter4 != NULL) cpl_free(pp_shutter4);






  /* for(i=0; i<background; i++) { */
  /*   if (pp_sky[i] != NULL) mat_gendata_delete(pp_sky[i]); */
  /* } */
  /* if (pp_sky != NULL) { */
  /*   cpl_free(pp_sky); */
  /* } */
	

  return 0;
}
