/* $Id: mat_ext_beams.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_ext_beams.c $
 */
/*------------------------------------------------------------------------------
                                   Includes
 ------------------------------------------------------------------------------*/
#include <cpl.h>
#include "mat_drl.h"
#include "mat_ext_beams_lib.h"
/*------------------------------------------------------------------------------
                                   Define
 ------------------------------------------------------------------------------*/
#define TEL_NUMBER 4
#define RECIPE_NAME "mat_ext_beams"



/* Plugin detailed description */
static const char *
mat_ext_beams_help = "In the case of SiPhot mode, this recipe transforms the photometric beams \n"
  "of each frame from the photmetric channel into the interferometric one using the Kappa Matrix.\n"
  "In the case of HighSens raw data, the mapping is not necessary. Furthermore, the average sky, measured via chopping,\n"
  " is subtracted from the photometric beam intensity to get the photometric intensity produced by the target alone\n"
  "\n"
  "Input Files:\n"
  "\n"
  "    DO category:           Explanation:                Required:\n"
  "    TARGET_CAL             Cal data (if target)        Yes (if target)\n"
  "    CALIB_CAL              Cal data (if calibrator     Yes (if calibrator or lab data \n"
  "    KAPPA_MATRIX           Kappa Matrix Map            Yes (if SiPhot)\n"
  "\n"
  "Output Files:\n"
  "\n"
  "    DO category:           Explanation:\n"
  "    PHOT_BEAMS             Photometric Beams in interferometric channel\n"
;
static int mat_ext_beams_create(cpl_plugin *);
static int mat_ext_beams_exec(cpl_plugin *);
static int mat_ext_beams_destroy(cpl_plugin *);

int cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = (cpl_plugin *)recipe;
    cpl_plugin_init(plugin,
                     CPL_PLUGIN_API,
                     MATISSE_BINARY_VERSION,
                     CPL_PLUGIN_TYPE_RECIPE,
                     "mat_ext_beams",
                     "Extraction of the photometry",
                     mat_ext_beams_help,
                     "Philippe Berio",
                     PACKAGE_BUGREPORT,
                     "GPL",
                     mat_ext_beams_create,
                     mat_ext_beams_exec,
                     mat_ext_beams_destroy);
    cpl_pluginlist_append(list, plugin);
    return 0;
    }

static int mat_ext_beams_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe = (cpl_recipe *)plugin;
    cpl_parameter *p=NULL;
    cpl_parameter *p1=NULL;
    cpl_parameter *p2=NULL;
    /* cpl_parameter *p3=NULL; */
    recipe->parameters = cpl_parameterlist_new();

    p = cpl_parameter_new_value("matisse.mat_ext_beams.hampelFilterKernel",
    				CPL_TYPE_INT, "Only for L/M band. Apply atemporal Hampel filter to all pixels before deriving the photometry. This filter improves the photometric estimation in case of faint stars (< 5Jy in L band with AT). The parameter fixes the size fo the kernel of the filter (kernel=10 is recommended).","matisse.mat_ext_beams",10);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "hampelFilterKernel") ;
    cpl_parameterlist_append(recipe->parameters, p) ;

    p1 = cpl_parameter_new_value("matisse.mat_raw_estimates.spectralBinning",
    				CPL_TYPE_INT,
    				 "Spectral Binning in pixels. Should be an odd integer number. If aven even integer is given, the pipeline will select the next odd integer.",
    				"matisse.mat_raw_estimates",1);
    cpl_parameter_set_alias(p1, CPL_PARAMETER_MODE_CLI, "spectralBinning") ;
    cpl_parameterlist_append(recipe->parameters, p1) ;

    p2 = cpl_parameter_new_value("matisse.mat_ext_beams.replaceTel",
				 CPL_TYPE_INT, "Replace Photometry of one telescope by the mean of the 3 others. An interger with 2 digits is expected. The first digit is used fot UTs and the second for ATs. 0: none, 1: AT1/UT1, 2: AT2/UT2, 3: AT3/UT3, 4: AT4/UT4. For example: replaceTel=01 means ==> replace AT1, replaceTel=32 means ==> replace UT3 and AT2.","matisse.mat_ext_beams",3);
    cpl_parameter_set_alias(p2, CPL_PARAMETER_MODE_CLI, "replaceTel") ;
    cpl_parameterlist_append(recipe->parameters, p2) ;

   cpl_msg_info("mat_ext_beams_create", "Parameters created.");

    return 0;
}
static int mat_ext_beams_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe = (cpl_recipe *)plugin;
    cpl_parameterlist_delete(recipe->parameters);
    return 0;
}


static int mat_ext_beams_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe = (cpl_recipe *)plugin;
     
    if (mat_ext_beams_lib(recipe->frames,recipe->parameters,RECIPE_NAME)) {
      cpl_msg_warning(cpl_func,"Error in mat_ext_beams_lib");
      return CPL_ERROR_ILLEGAL_OUTPUT;
    }
    return 0;
}

