/* $Id: mat_proc_incoher.c ,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
/*------------------------------------------------------------------------------
                                   Includes
 ------------------------------------------------------------------------------*/
#include <cpl.h>
#include "mat_drl.h"
#include "mat_proc_incoher_lib.h"
/*------------------------------------------------------------------------------
                                   Define
 ------------------------------------------------------------------------------*/
#define TEL_NUMBER 4
#define RECIPE_NAME "mat_proc_incoher"



/* Plugin detailed description */
static const char *
mat_proc_incoher_help = "This recipe calculates the average power and bi spectrum from all selected frames.\n"
  "It derives raw squared visibility and raw closure phase. As a by-product, this recieps computes the spectrum\n"
  "It is also possible to compute correlated flux insted of the squared visibility (corrFlux=TRUE). In this case,  \n"
  "In this case, PHOT_BEAMS data are not required.\n"
  "\n"
  "Input Files:\n"
  "\n"
  "    DO category:           Explanation:                Required:\n"
  "    OBJ_CORR_FLUX          Correlated Flux             Yes \n"
  "    PHOT_BEAMS             Photometric Beam            No (corrFlux parameter)\n"
  "\n"
  "Output Files:\n"
  "\n"
  "    DO category:           Explanation:\n"
  "    RAW_VIS2               raw squared visibility\n"
  "    RAW_CPHASE             raw closure phase\n"
  "    RAW_SPECTRUM           spectrum\n";
static int mat_proc_incoher_create(cpl_plugin *);
static int mat_proc_incoher_exec(cpl_plugin *);
static int mat_proc_incoher_destroy(cpl_plugin *);

/*----------------------------------------------------------------------------*/
/**
   @brief Build the list of available plugins, for this module.
   @param list the plugin list
   @return 0 if everything is ok, -1 otherwise
   Create the recipe instance and make it available to the application using the
   interface. This function is exported.
*/
/*----------------------------------------------------------------------------*/
int cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = (cpl_plugin *)recipe;
    cpl_plugin_init(plugin,
                     CPL_PLUGIN_API,
                     MATISSE_BINARY_VERSION,
                     CPL_PLUGIN_TYPE_RECIPE,
                     "mat_proc_incoher",
                     "Incoherent processing",
                     mat_proc_incoher_help,
                     "Philippe Berio",
                     PACKAGE_BUGREPORT,
                     "GPL",
                     mat_proc_incoher_create,
                     mat_proc_incoher_exec,
                     mat_proc_incoher_destroy);
    cpl_pluginlist_append(list, plugin);
    return 0;
    }

/*----------------------------------------------------------------------------*/
/**
   @brief Setup the recipe options
   @param plugin the plugin
   @return 0 if everything is ok
   Defining the command-line/configuration parameters for the recipe.
*/
/*----------------------------------------------------------------------------*/
static int mat_proc_incoher_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe = (cpl_recipe *)plugin;
    cpl_parameter *p2;
    recipe->parameters = cpl_parameterlist_new();

    p2 = cpl_parameter_new_value("matisse.mat_proc_incoher.corrFlux",
				CPL_TYPE_BOOL, 
				 "corrFlux option, If TRUE, correlated flux are computed instead of quiared visibility.", 
				"matisse.mat_proc_incoher",FALSE);
    cpl_parameter_set_alias(p2, CPL_PARAMETER_MODE_CLI, "corrFlux") ;
    cpl_parameterlist_append(recipe->parameters, p2) ;

    cpl_msg_info("mat_proc_incoher_create", "Parameters created.");

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Destroy what has been created by the ’create’ function
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_proc_incoher_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe = (cpl_recipe *)plugin;
    cpl_parameterlist_delete(recipe->parameters);
    return 0;
}


/*----------------------------------------------------------------------------*/
/**
   @brief Execute the plugin instance given by the interface. This plugin
   calculates the average power and bi spectrum from all selected frames.
   From these averages the photometrically calibrated squared raw visibility 
   and the raw closure phase are derived for each spectral channel. Since 
   in low resolution mode the coherence length and the amplitude of the 
   atmospheric OPD variations are similar, the fringe contrast depends on 
   the actual atmospheric OPD. With the help of the OPD measures in 
   mat_est_opd the OPD dependent fring contrast-loss can be compensated. 
   As a by-product, this recipes computes the spectrum.

   The input data are:the observation specific flatfield map (OBS_FLATFIELD), 
   the extracted photometry beams without sky background (PHOT_BEAMS), 
   the complex correlated fluxes(OBJ_CORR_FLUX) and the atmospheric 
   OPD and water vapor phase offset (OPD_WVPO) (optional). 
   This plugin produces the raw suared visibility (RAW_VIS2), the raw closure 
   phase (RAW_CPHASE) and the raw spectrum (RAW_SPECTRUM).It computes also
   the QC DETi VISSQRm, QC DETi VISSQRm SNR, QC DETi CPHASEn and
   QC DETi CPHASEn STDEV.
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_proc_incoher_exec(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
   
  if (mat_proc_incoher_lib(recipe->frames,recipe->parameters,RECIPE_NAME)) {
    cpl_msg_warning(cpl_func,"Error in mat_proc_incoher_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }

  return 0;
}

