/* $Id: mat_raw_estimates.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_raw_estimates.c $
 */
/*------------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "mat_drl.h"
#include "mat_cal_image_lib.h"
#include "mat_ext_beams_lib.h"
#include "mat_est_corr_lib.h"
#include "mat_proc_incoher_lib.h"
#include "mat_proc_coher_lib.h"
#include "mat_merge_results_lib.h"
#include "mat_est_tf_lib.h"
#include "mat_est_opd_lib.h"
#include "mat_utils.h"
/*------------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/
#define TEL_NUMBER 4
#define RECIPE_NAME "mat_raw_estimates"


/* Plugin detailed description */
static const char *
mat_raw_estimates_help = 
  "This plugin calls all data reduction steps which \n"
  "transform the measured raw frames to the average raw interferometric \n"
  "products like raw squared visibilities, raw closure phases and raw \n"
  "differential phases. This recipes could be applied to target or calibrator frames.\n"
  "\n"
  "Input Files:\n"
  "\n"
  "    DO category:           Explanation:                Required:\n"
  "    CALIB_SRC_RAW          Lab Lamp data               Yes (if lab data)\n"
  "    HOT_DARK               Lab Background data         Yes (if lab data)\n"
  "    SKY_RAW                Sky data                    No \n"
  "    CALIB_RAW              Calibrator data             Yes (if calibrator)\n"
  "    TARGET_RAW             Target data                 Yes (if target)\n"
  "    OBS_FLATFIELD          Observing Flat Field        Yes\n"
  "    BADPIX                 Bad Pixel Map               Yes\n"
  "    NONLINEARITY           Nonlinearity Map            Yes\n"
  "    SHIFT_MAP              Distorsion Map              Yes\n"
  "    KAPPA_mATRIX           Kappa Matrix                No\n"
  "\n"
  "Output Files:\n"
  "\n"
  "    DO category:           Explanation:\n"
  "    TARGET_RAW_INT         OIFITS file with raw data on target\n"
  "    CALIB_RAW_INT          OIFITS file with raw data on calibrator\n"
  "    OI_OPDWVPO             OPD estimation\n";
static int mat_raw_estimates_create(cpl_plugin *);
static int mat_raw_estimates_exec(cpl_plugin *);
static int mat_raw_estimates_destroy(cpl_plugin *);
static int mat_raw_estimates(cpl_parameterlist *parlist, cpl_frameset *frames);

int cpl_plugin_get_info(cpl_pluginlist *list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
  cpl_plugin *plugin = (cpl_plugin *)recipe;
  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_raw_estimates",
		  "All data reduction steps (target)",
		  mat_raw_estimates_help,
		  "Philippe Berio",
		  PACKAGE_BUGREPORT,
		  "GPL",
		  mat_raw_estimates_create,
		  mat_raw_estimates_exec,
		  mat_raw_estimates_destroy);
  cpl_pluginlist_append(list, plugin);
  return 0;
}

static int mat_raw_estimates_create(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  cpl_parameter *p;
  recipe->parameters = cpl_parameterlist_new();

  /* --compensate */
  p = cpl_parameter_new_value("matisse.mat_cal_image.compensate",
			      CPL_TYPE_STRING,
			      "Defines which kind of compensation should be applied "
			      "(none = no compensation at all, all = all compensations possible, dd = detector specific defaults, "
			      "pb = subtract pixel bias, gb = subtract global bias, cb = subtract detector channel bias, rb = subtract row bias, "
			      "ct = subtract crosstalk, "
			      "nl = nonlinearity compensation, if = divide by instrument flat, df = divide by detector flat, "
			      "bp = bad pixel interpolation, el = convert to electrons, od = remove optical distortion)",
			      "matisse.mat_cal_image",
			      "pb,cb,rb,nl,if,bp,od");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "compensate") ;
  cpl_parameterlist_append(recipe->parameters, p) ; 
 
  /* --gain */
  p = cpl_parameter_new_range("matisse.mat_cal_image.gain",
			      CPL_TYPE_DOUBLE,
			      "Default conversion gain in [e-/DU].",
			      "matisse.mat_cal_image",
			      0.0, 0.0, 1e5);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "gain");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --reduce */
  p = cpl_parameter_new_value("matisse.mat_cal_image.reduce",
			      CPL_TYPE_BOOL,
			      "Flag if the reference sub-windows should be removed from the result.",
			      "matisse.mat_cal_image",
			      TRUE);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "reduce");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --ioi */
  p = cpl_parameter_new_value("matisse.mat_cal_image.ioi",
			      CPL_TYPE_STRING,
			      "images of interest: <first>,<count>",
			      "matisse.mat_cal_image",
			      "0,0");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "ioi");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --tartyp */
  p = cpl_parameter_new_value("matisse.mat_cal_image.tartyp",
			      CPL_TYPE_INT,
			      "TARTYP estimation (flag, please add the following values for the desired effect: 0 = none, 1 = N*S+U+N*T+U, 2 = show total flux, 4 = show correlation, 8 = estimate TARTYP, 16 = change TARTYP, 32 = exchange U with S or T, 64 = change TIME, 128 = change LOCALOPD and STEPPING_PHASE, 256 = show the extracted exposure setup, 512 = show the first chopping cycle, 1024 = show the extracted timetable, 2048 = show the modified frame info, 4096 = use shift calculated from square wave fit, 8192 = use shift calculated from timing)",
			      "matisse.mat_cal_image",
			      57);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "tartyp");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --excess_count_lm */
  p = cpl_parameter_new_value("matisse.mat_cal_image.excess_count_lm",
			      CPL_TYPE_INT,
			      "Excess frames mistakenly produced before first TIM-Board trigger (LM-Band)",
			      "matisse.mat_cal_image",
			      MAT_EXCESS_COUNT_UNSET);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "excess_count_lm");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --excess_count_n */
  p = cpl_parameter_new_value("matisse.mat_cal_image.excess_count_n",
			      CPL_TYPE_INT,
			      "Excess frames mistakenly produced before first TIM-Board trigger (N-Band)",
			      "matisse.mat_cal_image",
			      MAT_EXCESS_COUNT_UNSET);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "excess_count_n");
  cpl_parameterlist_append(recipe->parameters, p);
  
  /* --excess_count_list */
  p = cpl_parameter_new_value("matisse.mat_cal_image.excess_count_list",
			      CPL_TYPE_STRING,
			      "Specification of an excess count for each exposure number (ESO TPL EXPNO keyword, excess count pair) as a list of numbers separate by comma.",
			      "matisse.mat_cal_image",
			      "none");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "excess_count_list");
  cpl_parameterlist_append(recipe->parameters, p);

  /* /\* --binning *\/ */
  /* p = cpl_parameter_new_value("matisse.mat_cal_image.binning", */
  /* 			      CPL_TYPE_INT, */
  /* 			      "Number of columns to bin (odd number only)", */
  /* 			      "matisse.mat_cal_image", */
  /* 			      1); */
  /* cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "binning"); */
  /* cpl_parameterlist_append(recipe->parameters, p); */

  /* p = cpl_parameter_new_value("matisse.mat_ext_beams.useAvgSky", */
  /* 			       CPL_TYPE_BOOL, "useAvgSky option",  */
  /* 			       "matisse.mat_raw_estimates",FALSE); */
  /* cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "useAvgSky") ; */
  /* cpl_parameterlist_append(recipe->parameters, p); */

  /* p = cpl_parameter_new_value("matisse.mat_ext_beams.useKappaMatrix", */
  /* 			       CPL_TYPE_BOOL, "useKappaMatrix option",  */
  /* 			       "matisse.mat_raw_estimates",TRUE); */
  /* cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "useKappaMatrix") ; */
  /* cpl_parameterlist_append(recipe->parameters, p) ; */

    p = cpl_parameter_new_value("matisse.mat_ext_beams.hampelFilterKernel",
    				CPL_TYPE_INT, "Only for L/M band. Apply atemporal Hampel filter to all pixels before deriving the photometry. This filter improves the photometric estimation in case of faint stars (< 5Jy in L band with AT). The parameter fixes the size fo the kernel of the filter (kernel=10 is recommended).","matisse.mat_ext_beams",10);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "hampelFilterKernel") ;
    cpl_parameterlist_append(recipe->parameters, p) ;

    p = cpl_parameter_new_value("matisse.mat_ext_beams.replaceTel",
			      CPL_TYPE_INT, "Replace Photometry of one telescope by the mean of the 3 others. An interger with 2 digits is expected. The first digit is used fot UTs and the second for ATs. 0: none, 1: AT1/UT1, 2: AT2/UT2, 3: AT3/UT3, 4: AT4/UT4. For example: replaceTel=01 means ==> replace AT1, replaceTel=32 means ==> replace UT3 and AT2.","matisse.mat_raw_estimates",3);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "replaceTel") ;
  cpl_parameterlist_append(recipe->parameters, p) ;
  
  /* p = cpl_parameter_new_value("matisse.mat_ext_beams.useAvgShape", */
  /* 			       CPL_TYPE_BOOL, "Only for L/M band. Compute the total flux of the photometry per frame and the average photometric shape for computing the photometry per frame. Could imporve the photometric estimation in case of faint stars (< 5Jy in L band).",  */
  /* 			       "matisse.mat_raw_estimates",FALSE); */
  /* cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "useAvgShape") ; */
  /* cpl_parameterlist_append(recipe->parameters, p); */

  
  p = cpl_parameter_new_value("matisse.mat_est_corr.useOpdMod",
			       CPL_TYPE_BOOL, "useOpdMod option", 
			       "matisse.mat_raw_estimates",FALSE);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "useOpdMod") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  p = cpl_parameter_new_value("matisse.mat_est_corr.coherentIntegTime",
			      CPL_TYPE_DOUBLE, "Specify a coherent integration time (in s)", 
			      "matisse.mat_raw_estimates",0.);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "coherentIntegTime") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* p = cpl_parameter_new_value("matisse.mat_est_opd.ChromaticOpdFit", */
  /*   				CPL_TYPE_BOOL, "Chromatic OPD Fit Option ", */
  /*   				"matisse.mat_est_opd",FALSE); */
  /* cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "ChromaticOpdFit") ; */
  /* cpl_parameterlist_append(recipe->parameters, p) ; */

  p = cpl_parameter_new_value("matisse.mat_proc_incoher.corrFlux",
			       CPL_TYPE_BOOL, "corrFlux option", 
			       "matisse.mat_raw_estimates",FALSE);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "corrFlux") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* p = cpl_parameter_new_value("matisse.mat_est_tf.catalog", */
  /* 			       CPL_TYPE_INT, */
  /* 			       "calibrator catalog", */
  /* 			       "matisse.mat_raw_estimates", */
  /* 			       1,0,1); */
  /* cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "catalog"); */
  /* cpl_parameterlist_append(recipe->parameters, p); */

  /* p = cpl_parameter_new_value("matisse.mat_est_tf.diamStar", */
  /* 			       CPL_TYPE_DOUBLE, */
  /* 			       "calibrator angular diameter", */
  /* 			       "matisse.mat_raw_estimates", */
  /* 			       1.0,0.0001,10.0); */
  /* cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "diamStar"); */
  /* cpl_parameterlist_append(recipe->parameters, p); */
    
  /* p = cpl_parameter_new_value("matisse.mat_est_tf.diamErr", */
  /* 			       CPL_TYPE_DOUBLE, */
  /* 			       "calibrator angular diameter error", */
  /* 			       "matisse.mat_raw_estimates", */
  /* 			       0.01, 0.00001, 10.0); */
  /* cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "diamErr"); */
  /* cpl_parameterlist_append(recipe->parameters, p); */

  p = cpl_parameter_new_value("matisse.mat_merge_results.cumulBlock",
				CPL_TYPE_BOOL, "cumul all blocks of an OB", 
				"matisse.mat_raw_estimates",FALSE);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cumulBlock") ;
  cpl_parameterlist_append(recipe->parameters, p) ; 

  p = cpl_parameter_new_value("matisse.mat_proc_coher.coherentAlgo",
			      CPL_TYPE_INT,
			      "Estimation Algorithm (1: AMBER like Method, 2: CRAL Cohrent Integration Method)",
			      "matisse.mat_raw_estimates",
			      2, 1, 2);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "coherentAlgo");
  cpl_parameterlist_append(recipe->parameters, p);

  p = cpl_parameter_new_value("matisse.mat_raw_estimates.spectralBinning",
			       CPL_TYPE_INT,
			       "Spectral Binning in pixels. Should be an odd integer number. If aven even integer is given, the pipeline will select the next odd integer.",
			       "matisse.mat_raw_estimates",1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "spectralBinning") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

    cpl_msg_info("mat_raw_estimates_create", "Parameters created.");

  return 0;
}
static int mat_raw_estimates_destroy(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  cpl_parameterlist_delete(recipe->parameters);
  return 0;
}


static int mat_raw_estimates_exec(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) {
    recipe=(cpl_recipe *)plugin;
  } else {
    return -1;
  }
  return mat_raw_estimates(recipe->parameters,recipe->frames);
  
}

int mat_raw_estimates(cpl_parameterlist *parlist, cpl_frameset *frames) {
  cpl_frame *cur_frame=NULL;
  char *pszFileName=NULL;
  char *pszFileTag=NULL;
  char *detName=NULL;
  int flagCalib=0;
  int flagTarget=0;
  cpl_parameter *p;
  cpl_frameset_iterator *it=NULL;
  cpl_errorstate prestate = cpl_errorstate_get(); 
  int cpt=0;
  int nbPhot=0;
  cpl_propertylist *plist=NULL;
  cpl_parameter *correlatedFlux=NULL;


  
  correlatedFlux=cpl_parameterlist_find(parlist,"matisse.mat_proc_incoher.corrFlux");
  /* prestate=cpl_errorstate_get(); */
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
    //Get Classification Tag
      pszFileName = (char*)cpl_frame_get_filename( cur_frame );
      pszFileTag  = (char *)cpl_frame_get_tag(cur_frame);
      if (cpt == 0) {
	plist=cpl_propertylist_load(pszFileName,0);
	detName=(char *)cpl_propertylist_get_string(plist,"ESO DET CHIP NAME");
	if (detName == NULL) {
	  cpl_msg_error(cpl_func,"no ESO DET CHIP NAME keyword in frame");
	  cpl_propertylist_delete(plist);
	  return -1;
	}
      }
      if (strcmp(pszFileTag,"CALIB_RAW") == 0 ||
	  strcmp(pszFileTag,"CALIB_SRC_RAW") == 0 ||
	  strcmp(pszFileTag,"TARGET_RAW") == 0 ) {
	if (!strcmp(detName, "AQUARIUS")) {
	  if (mat_check_photometry(cur_frame)) {
	    nbPhot++;
	  }	  
	}
      }
     
      if (strcmp(pszFileTag,"CALIB_RAW") == 0) {
	flagCalib=1;
      }
      if (strcmp(pszFileTag,"TARGET_RAW") == 0) {
	flagTarget=1;
      }
      cpt++;
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }
  if (flagCalib && flagTarget) {
    cpl_msg_error(cpl_func,
		  "the sof could not contains TARGET_RAW \
                       and CALIB_RAW simultanesouly.");
    return 1;
  } else {
    if (flagCalib) {
      cpl_msg_info(cpl_func,"Calibrator data");
    }	
    if (flagTarget) {
      cpl_msg_info(cpl_func,"Target data");
    }
    if (!strcmp(detName, "AQUARIUS") && nbPhot!=8 ) {
      cpl_msg_info(cpl_func,"Photometry frames missing. Correlated Flux estimation only");
      cpl_parameter_set_bool(correlatedFlux,1);
    } else {
     cpl_msg_info(cpl_func,"No Photometry frame missing.");
    }
  }
  cpl_propertylist_delete(plist);



  cpl_msg_info(cpl_func,"****************************************************");
  cpl_msg_info(cpl_func,"Starting mat_cal_image");
  cpl_msg_info(cpl_func,"****************************************************");
  if (mat_cal_image_lib(frames,parlist,RECIPE_NAME)) {
    cpl_msg_error(cpl_func,"Error in mat_cal_image_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }



  
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      pszFileTag  = (char *)cpl_frame_get_tag(cur_frame);
      if ( (strcmp(pszFileTag,"SKY_RAW") == 0 ||
  	    strcmp(pszFileTag,"CALIB_RAW") == 0 ||
  	    strcmp(pszFileTag,"TARGET_RAW") == 0 ||
  	    strcmp(pszFileTag,"CALIB_SRC_RAW") == 0 ||
  	    strcmp(pszFileTag,"HOT_DARK") == 0 ||
	    strcmp(pszFileTag,"RMNREC") == 0 )
  	   ) {
  	cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
      }
      if ( (strcmp(pszFileTag,"BADPIX") == 0 ||
  	    strcmp(pszFileTag,"OBS_FLATLIED") == 0 ||
  	    strcmp(pszFileTag,"NONLINEARITY") == 0 ||
  	    strcmp(pszFileTag,"KAPPA_MATRIX") == 0 ||
  	    strcmp(pszFileTag,"SHIFT_MAP") == 0 ||
  	    strcmp(pszFileTag,"JSDC_CAT") == 0 )
  	   ) {
  	cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_CALIB);
      }
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);



  if (cpl_parameter_get_bool(correlatedFlux) == 0){
    cpl_msg_info(cpl_func,"****************************************************");
    cpl_msg_info(cpl_func,"Starting mat_ext_beams");
    cpl_msg_info(cpl_func,"****************************************************");
    if (mat_ext_beams_lib(frames,parlist,RECIPE_NAME)) {
      cpl_msg_warning(cpl_func,"Error in mat_ext_beams_lib");
      return CPL_ERROR_ILLEGAL_OUTPUT;
    }
  }

  cpl_msg_info(cpl_func,"****************************************************");
  cpl_msg_info(cpl_func,"Starting mat_est_corr");
  cpl_msg_info(cpl_func,"****************************************************");
  if (mat_est_corr_lib(frames,parlist,RECIPE_NAME)) {
    cpl_msg_error(cpl_func,"Error in mat_est_corr_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }

  cpl_msg_info(cpl_func,"****************************************************");
  cpl_msg_info(cpl_func,"Starting mat_est_opd");
  cpl_msg_info(cpl_func,"****************************************************");
  if (mat_est_opd_lib(frames,parlist,RECIPE_NAME)) {
    cpl_msg_error(cpl_func,"Error in mat_est_opd_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }

  cpl_msg_info(cpl_func,"****************************************************");
  cpl_msg_info(cpl_func,"Starting mat_proc_coher");
  cpl_msg_info(cpl_func,"****************************************************");
  if (mat_proc_coher_lib(frames,parlist,RECIPE_NAME)) {
    cpl_msg_error(cpl_func,"Error in mat_proc_coher_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }
  cpl_msg_info(cpl_func,"****************************************************");
  cpl_msg_info(cpl_func,"Starting mat_proc_incoher");
  cpl_msg_info(cpl_func,"****************************************************");
  if (mat_proc_incoher_lib(frames,parlist,RECIPE_NAME)) {
    cpl_msg_error(cpl_func,"Error in mat_proc_incoher_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }


  cpl_msg_info(cpl_func,"****************************************************");
  cpl_msg_info(cpl_func,"Starting mat_est_tf");
  cpl_msg_info(cpl_func,"****************************************************");
  if (mat_est_tf_lib(parlist,frames,RECIPE_NAME)) {
    cpl_msg_error(cpl_func,"Error in mat_est_tf_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }

  cpl_msg_info(cpl_func,"****************************************************");
  cpl_msg_info(cpl_func,"Starting mat_merge_results");
  cpl_msg_info(cpl_func,"****************************************************");

  // Add bcdIn parameter
  // Put bcdMode=OUT-OUT No Chop
  p = cpl_parameter_new_value("matisse.mat_merge_results.bcdMode",
  			      CPL_TYPE_INT, "Select frames with BCD OUT-OUT (0) \
or frames with BCD IN-IN (1) or frames with BCD IN-OUT (2) or frames with BCD OUT-IN (3)",
  			      "matisse.mat_raw_estimates",0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "bcdMode") ;
  cpl_parameterlist_append(parlist, p) ;

  if (mat_merge_results_lib(frames, parlist,RECIPE_NAME,NULL)) {
    cpl_msg_error(cpl_func,"Error in mat_merge_results_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }
  // Put bcdMode=IN-IN No Chop
  cpl_parameter_set_int(p,1);
  if (mat_merge_results_lib(frames, parlist,RECIPE_NAME,NULL)) {
    cpl_msg_error(cpl_func,"Error in mat_merge_results_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }
  // Put bcdMode=IN-OUT No Chop
  cpl_parameter_set_int(p,2);
  if (mat_merge_results_lib(frames, parlist,RECIPE_NAME,NULL)) {
    cpl_msg_error(cpl_func,"Error in mat_merge_results_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }
  // Put bcdMode=OUT-IN No Chop
  cpl_parameter_set_int(p,3);
  if (mat_merge_results_lib(frames, parlist,RECIPE_NAME,NULL)) {
    cpl_msg_error(cpl_func,"Error in mat_merge_results_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }
  // Put bcdMode=OUT-OUT Chop
  cpl_parameter_set_int(p,4);
  if (mat_merge_results_lib(frames, parlist,RECIPE_NAME,NULL)) {
    cpl_msg_error(cpl_func,"Error in mat_merge_results_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }
  // Put bcdMode=IN-IN Chop
  cpl_parameter_set_int(p,5);
  if (mat_merge_results_lib(frames, parlist,RECIPE_NAME,NULL)) {
    cpl_msg_error(cpl_func,"Error in mat_merge_results_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }
  // Put bcdMode=IN-OUT Chop
  cpl_parameter_set_int(p,6);
  if (mat_merge_results_lib(frames, parlist,RECIPE_NAME,NULL)) {
    cpl_msg_error(cpl_func,"Error in mat_merge_results_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }
  // Put bcdMode=OUT-IN Chop
  cpl_parameter_set_int(p,7);
  if (mat_merge_results_lib(frames, parlist,RECIPE_NAME,NULL)) {
    cpl_msg_error(cpl_func,"Error in mat_merge_results_lib");
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }
    
  prestate=cpl_errorstate_get();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      pszFileTag  = (char *)cpl_frame_get_tag(cur_frame);
      if ( (strcmp(pszFileTag,"RAW_VIS2") == 0 ||
  	    strcmp(pszFileTag,"RAW_CPHASE") == 0 ||
  	    strcmp(pszFileTag,"RAW_DPHASE") == 0 ||
  	    strcmp(pszFileTag,"RAW_SPECTRUM") == 0 ||
  	    strcmp(pszFileTag,"RAW_TF2") == 0 ||
  	    strcmp(pszFileTag,"CAL_VIS2") == 0 ||
  	    strcmp(pszFileTag,"CAL_CPHASE") == 0 ||
  	    strcmp(pszFileTag,"CAL_DPHASE") == 0 ||
  	    strcmp(pszFileTag,"PHOT_BEAMS") == 0 ||
  	    strcmp(pszFileTag,"OBJ_CORR_FLUX") == 0 ||
  	    strcmp(pszFileTag,"CALIB_CAL") == 0 ||
  	    strcmp(pszFileTag,"TARGET_CAL") == 0 ||
  	    strcmp(pszFileTag,"OI_OPDWVPO") == 0)
  	   ) {
  	cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_PRODUCT);
      }
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  /* if (!cpl_errorstate_is_equal(prestate)) { */
  /*   cpl_errorstate_set(prestate); */
  /* } */


  return 0;
}
