/* $Id: mat_wave_cal.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_wave_cal.c $
 */
/*------------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/
#include <cpl.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include "mat_drl.h"
#include "mat_shift.h"
#include "mat_imagingdetector.h"
#include "mat_utils.h"
/*------------------------------------------------------------------------------
  Define
  ----------------------------------------------------------------------------*/
#define TEL_NUMBER 4
#define RECIPE_NAME "mat_wave_cal"
#define NREGION 5
// Lines 4.0287, 4.05226, 4.65378, 4.67251
#define LAMBDA_BRACKET_ALPHA 4.05226
#define LAMBDA_PFUND_BETA 4.65378


/* Plugin detailed description */
static const char *
mat_wave_cal_help = 
  "This plugin estimates the dispersion law by comparing a sky spectrum recorded with MATISSE with \n"
  "one or several theoretical sky spectra provided by the user. These theoretical sky spectra \n"
  "are provided in the sof files (with the tag SPEC_ATM). For each theoretical spectra, the recipes \n"
  "determines the best second order polynomial law which minimizes the difference betwen the measured and \n"
  "the theoretical spectra in a least squared sense. The coefficients of the polynome are computed \n"
  "with the simulated annealing method. Once each second order polynome has been determined for each theoretical \n"
  "spectra, the recipes selects the law which minimizes the difference betwen the measured and \n"
  "the theoretical spectra. Finaly, the recipes modified the SHIT_MAP provided in input by replacing the \n"
  "3 dispersion law coefficients by the the 3 coefficients thus determined.\n"
  "\n"
  "Input Files:\n"
  "\n"
  "    DO category:           Explanation:                Required:\n"
  "    TARGET_CAL             Cal data (if target)        Yes (if target)\n"
  "    CALIB_CAL              Cal data (if calibrator)    Yes (if calibrator or lab data \n"
  "    SHIFT_MAP              Shift Map                   Yes \n"
  "\n"
  "Output Files:\n"
  "\n"
  "    DO category:           Explanation:\n"
  "    SHIFT_MAP              Updated dispersion law in Shift Map\n"
  ;
static int mat_wave_cal_create(cpl_plugin *);
static int mat_wave_cal_exec(cpl_plugin *);
static int mat_wave_cal_destroy(cpl_plugin *);
static int mat_wave_cal(cpl_parameterlist *parlist, cpl_frameset *frames);

int cpl_plugin_get_info(cpl_pluginlist *list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
  cpl_plugin *plugin = (cpl_plugin *)recipe;
  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_wave_cal",
		  "Dispersion law determination",
		  mat_wave_cal_help,
		  "Philippe Berio",
		  PACKAGE_BUGREPORT,
		  "GPL",
		  mat_wave_cal_create,
		  mat_wave_cal_exec,
		  mat_wave_cal_destroy);
  cpl_pluginlist_append(list, plugin);
  return 0;
}

static int mat_wave_cal_create(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  recipe->parameters = cpl_parameterlist_new();

  return 0;
}
static int mat_wave_cal_destroy(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  cpl_parameterlist_delete(recipe->parameters);
  return 0;
}


static int mat_wave_cal_exec(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) {
    recipe=(cpl_recipe *)plugin;
  } else {
    return -1;
  }
  return mat_wave_cal(recipe->parameters,recipe->frames);
}

int mat_wave_cal(cpl_parameterlist *parlist, cpl_frameset *frames) {
  cpl_frame *cur_frame=NULL;
  char *pszFileName=NULL;
  char *pszFileTag=NULL;
  char *keyExtname=NULL;
  cpl_propertylist *plist=NULL;
  cpl_propertylist *plist_shift=NULL;
  mat_shiftmap *shiftMap=NULL;
  int nbExtent=0;
  /* int n=0; */
  int i=0;
  int j=0;
  int k=0;
  int pis=0;
  int pxShift=0;
  mat_gendata *gendata=NULL;
  cpl_image **imgsum=NULL;
  cpl_vector **spectrum=NULL;
  cpl_vector **spectrumfiltered=NULL;
  cpl_vector *coefGlobal=NULL;
  char filenameFits[]="SHIFT_MAP.fits";
  cpl_frameset *usedframes=NULL;
  cpl_frame *first_frame=NULL;
  cpl_table *table=NULL;

  cpl_frameset_iterator *it=NULL;
  char *detName=NULL;
  char *resolution=NULL;
  char *filter=NULL;
  cpl_vector *vec;
  int limSmin=0;
  int limSmax=0;
  int sizeW=0.;
  int sizeS=0.;
  double val=0.;
  cpl_vector *offsetwave=NULL;
  cpl_vector *offsetpx=NULL;
  cpl_vector *vecX=NULL;
  cpl_vector **vecY=NULL;
  double x0,sigma,area,offset;
  int npx=80;
  int px=160;
  double delta=0.;
  cpl_vector **vecFit=NULL;

  usedframes=cpl_frameset_duplicate(frames);
   

  // 2.Load Calibration MAP
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      pszFileName = (char*)cpl_frame_get_filename( cur_frame );
      pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
    
      if (strcmp(pszFileTag,"SHIFT_MAP") == 0) {
	if (shiftMap==NULL) {
	  shiftMap = mat_shiftmap_load(cur_frame);
	  cpl_msg_info(cpl_func, "SHIFT_MAP loaded");

	  first_frame=cpl_frame_duplicate(cur_frame);
	  nbExtent=cpl_frame_get_nextensions(cur_frame);
	  for(i=0;i<nbExtent;i++) {
	    plist=cpl_propertylist_load(pszFileName,i+1);
	    keyExtname=(char *)
	      cpl_propertylist_get_string(plist,"EXTNAME");
	    if (!strcmp(keyExtname,"SHIFT_MAP")) {
	      plist_shift= cpl_propertylist_duplicate(plist);
	    }
	    cpl_propertylist_delete(plist);
	  }


	} else {
	  cpl_msg_warning(cpl_func, "Several SHIFT_MAP frames found");
	  cpl_frameset_delete(usedframes);
	  return CPL_ERROR_INCOMPATIBLE_INPUT;
	}
      }
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);

  if ( shiftMap == NULL ) {
    cpl_msg_warning(cpl_func, "SHIF_MAP missing");
    cpl_frameset_delete(usedframes);
    return CPL_ERROR_INCOMPATIBLE_INPUT;
  }

  imgsum=cpl_malloc(NREGION*sizeof(cpl_image *));
  for(j=0;j<NREGION;j++)
    {
      imgsum[j]=NULL;
    }

  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      pszFileName= (char *)cpl_frame_get_filename( cur_frame );
      pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
      
      if (strcmp(pszFileTag,"TARGET_CAL") == 0 || strcmp(pszFileTag,"CALIB_CAL") == 0 ) {
	if (detName == NULL)
	  {
	    plist=cpl_propertylist_load(pszFileName,0);
	    detName=(char *)mat_propertylist_get_string_default(NULL,0,plist,"ESO DET CHIP NAME","");
	    if (!strcmp(detName, "")) {
	      cpl_msg_error(cpl_func,"no ESO DET CHIP NAME keyword in frame");
	      return -1;
	    }
	    if (!strcmp(detName, "AQUARIUS")) {
	      cpl_msg_error(cpl_func,"This recipes does not work with N band data");
	    } else {
	      resolution=(char *)mat_propertylist_get_string_default(NULL,0,plist,"ESO INS DIL ID","");
	      if ( strcmp(resolution,"HIGH+") != 0)
		{
		  cpl_msg_error(cpl_func,"This recipes work for Very High Resolution only.");
		}
	      filter=(char *)mat_propertylist_get_string_default(NULL,0,plist,"ESO INS FIL ID","");
	    }
	    cpl_propertylist_delete(plist);
	  }

  	if (gendata != NULL)
	  {
  	    mat_gendata_delete(gendata);
  	  }
  	gendata = mat_gendata_load(cur_frame,CPL_TYPE_FLOAT);
  	if (gendata == NULL) {
  	  cpl_msg_error(cpl_func,"no gendata loaded");
  	  return -1;
  	}
	
	
  	// Summ all images corresponding to Target
  	pxShift=gendata->imgdet->list_region[0]->corner[1];
  	for(j=0;j<gendata->imgdet->nbregion;j++)
  	  {
  	    for(k=0;k<gendata->imgdata->nbframe;k++)
  	      {
  		if ( !strcmp(gendata->imgdata->list_frame[k]->tartype,"T"))
  		  {
  		    if (imgsum[j]==NULL)
  		      {
  			imgsum[j]=cpl_image_duplicate(gendata->imgdata->list_frame[k]->list_subwin[j]->imgreg[0]);
  		      }
  		    else
  		      {
  			cpl_image_add(imgsum[j],gendata->imgdata->list_frame[k]->list_subwin[j]->imgreg[0]);
  		      }
  		  }
  	      }
  	  }
      }
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  if(gendata != NULL) mat_gendata_delete(gendata);
  // Compute the spectrum by collapsing in the spatial direction
  spectrum=cpl_malloc(NREGION*sizeof(cpl_vector *));
  spectrumfiltered=cpl_malloc(NREGION*sizeof(cpl_vector *));
  offsetwave=cpl_vector_new(NREGION);
  offsetpx=cpl_vector_new(NREGION);
  for(j=0;j<NREGION;j++)
    {
      if (j==0)
  	{
  	  limSmin=50;
  	  limSmax=91;
  	}
      else if (j==1)
  	{
  	  limSmin=65;
  	  limSmax=106;
  	}
      else if (j==2)
  	{
  	  limSmin=230;
  	  limSmax=330;
  	}
      else if (j==3)
  	{
  	  limSmin=65;
  	  limSmax=106;
  	}
       else if (j==4)
  	{
  	  limSmin=65;
  	  limSmax=106;
  	}
       if (sizeW==0)
  	{
  	  sizeW=cpl_image_get_size_y(imgsum[j]);
  	  sizeS=cpl_image_get_size_x(imgsum[j]);
  	}
      spectrum[j]=cpl_vector_new(sizeW);
      for(i=0;i<sizeW;i++)
  	{
  	  val=0.;
  	  for(int ix=limSmin;ix<limSmax;ix++)
  	    {
  	      val+=cpl_image_get(imgsum[j],ix+1,i+1,&pis);
  	    }
  	  cpl_vector_set(spectrum[j],i,val);
  	}
      vec=cpl_vector_filter_lowpass_create(spectrum[j],CPL_LOWPASS_GAUSSIAN,15);
      for(i=0;i<sizeW;i++)
  	{
  	  cpl_vector_set(spectrum[j],i,cpl_vector_get(vec,i));
  	}
      cpl_vector_delete(vec);
      spectrumfiltered[j]=cpl_vector_filter_lowpass_create(spectrum[j],CPL_LOWPASS_LINEAR,101);
      cpl_vector_divide(spectrum[j],spectrumfiltered[j]);
   }


  
  /* for(i=0;i<sizeW;i++) */
  /*   { */
  /*     printf("%f %f\n",shiftMap->dispcoef[2]*pow(i+pxShift,2.0)+shiftMap->dispcoef[1]* */
  /* 	     (i+pxShift)+shiftMap->dispcoef[0],cpl_vector_get(spectrum[2],i)); */
  /*   } */

  

  // Select window around spectral lines
  vecY=cpl_malloc(NREGION*sizeof(cpl_vector *));
  vecFit=cpl_malloc(NREGION*sizeof(cpl_vector *));
  if (strcmp(filter,"L")==0)
    {
      npx=80;
      px=190;
      for(j=0;j<NREGION;j++)
  	{
  	  vecFit[j]=cpl_vector_new(npx);
  	  if (j==0)
  	    {
  	      vecX= cpl_vector_new(npx);
  	    }
  	  vecY[j]= cpl_vector_new(npx);
  	  for(i=0;i<npx;i++)
  	    {
  	      if(j==0)
  		{
  		  cpl_vector_set(vecX,i,shiftMap->dispcoef[2]*pow(i+px-npx/2+pxShift,2.0)+shiftMap->dispcoef[1]*
  				 (i+px-npx/2+pxShift)+shiftMap->dispcoef[0]);
  		}
  	      cpl_vector_set(vecY[j],i,cpl_vector_get(spectrum[j],i+px-npx/2));
  	    }
  	  cpl_vector_fit_gaussian(vecX,NULL,vecY[j],NULL,CPL_FIT_ALL,&x0,&sigma,&area,&offset,NULL,NULL,NULL);
  	  cpl_vector_set(offsetwave,j,x0);
  	  delta=pow(shiftMap->dispcoef[1],2.)-4.*(shiftMap->dispcoef[0]-x0)*shiftMap->dispcoef[2];
  	  cpl_vector_set(offsetpx,j,(-shiftMap->dispcoef[1]-sqrt(delta))/(2*shiftMap->dispcoef[2]));
	}
      
      for(j=0;j<NREGION;j++)
  	{
  	  cpl_vector_delete(vecY[j]);
  	  cpl_vector_delete(vecFit[j]);
  	}
      cpl_vector_delete(vecX);
   }
  else if (strcmp(filter,"M")==0)
    {
      npx=80;
      px=1690;
      for(j=0;j<NREGION;j++)
  	{
  	  vecFit[j]=cpl_vector_new(npx);
  	  if (j==0)
  	    {
  	      vecX= cpl_vector_new(npx);
  	    }
  	  vecY[j]= cpl_vector_new(npx);
  	  for(i=0;i<npx;i++)
  	    {
  	      if(j==0)
  		{
  		  cpl_vector_set(vecX,i,shiftMap->dispcoef[2]*pow(i+px-npx/2+pxShift,2.0)+shiftMap->dispcoef[1]*
  				 (i+px-npx/2+pxShift)+shiftMap->dispcoef[0]);
  		}
  	      cpl_vector_set(vecY[j],i,cpl_vector_get(spectrum[j],i+px-npx/2));
  	    }
  	  cpl_vector_fit_gaussian(vecX,NULL,vecY[j],NULL,CPL_FIT_ALL,&x0,&sigma,&area,&offset,NULL,NULL,NULL);
  	  cpl_vector_set(offsetwave,j,x0);
  	  delta=pow(shiftMap->dispcoef[1],2.)-4.*(shiftMap->dispcoef[0]-x0)*shiftMap->dispcoef[2];
  	  cpl_vector_set(offsetpx,j,(-shiftMap->dispcoef[1]-sqrt(delta))/(2*shiftMap->dispcoef[2]));
	}
      /* for(i=0;i<npx;i++) */
      /* 	printf("%f %f %f\n",cpl_vector_get(vecX,i),cpl_vector_get(vecY[2],i),cpl_vector_get(spectrum[2],i+px-npx/2)); */
      for(j=0;j<NREGION;j++)
  	{
  	  cpl_vector_delete(vecY[j]);
  	  cpl_vector_delete(vecFit[j]);
  	}
      cpl_vector_delete(vecX);
    }

  cpl_free(vecY);
  cpl_free(vecFit);





  /* for(i=0;i<sizeW;i++) */
  /*   { */
  /*     printf("%f %f\n",shiftMap->dispcoef[2]*pow(i+pxShift,2.0)+shiftMap->dispcoef[1]* */
  /* 	     (i+pxShift)+shiftMap->dispcoef[0],cpl_vector_get(spectrum[2],i)) ; */
  /*   } */


  cpl_vector_dump(offsetpx,NULL);
  cpl_msg_info(cpl_func,"Old dispersion law : %f + %f px + %e px.px",
  	       shiftMap->dispcoef[0],shiftMap->dispcoef[1],
  	       shiftMap->dispcoef[2]);
  if (strcmp(filter,"L")==0)
    {
      shiftMap->dispcoef[0]+=LAMBDA_BRACKET_ALPHA-cpl_vector_get(offsetwave,2);
    }
  else
    {
       shiftMap->dispcoef[0]+=LAMBDA_PFUND_BETA-cpl_vector_get(offsetwave,2);
    }
  cpl_msg_info(cpl_func,"New dispersion law : %f + %f px + %e px.px",
  	       shiftMap->dispcoef[0],shiftMap->dispcoef[1],
  	       shiftMap->dispcoef[2]);

  for(j=0;j<NREGION;j++)
    {
      if (j != 2)
  	{
  	  for(i=0;i<sizeS;i++)
  	    {
  	      delta=cpl_matrix_get(shiftMap->list_shiftcoefy[j]->coefdistor,0,i);
  	      delta-=cpl_vector_get(offsetpx,2)-cpl_vector_get(offsetpx,j);
  	      cpl_matrix_set(shiftMap->list_shiftcoefy[j]->coefdistor,0,i,delta);
  	    }
  	}
    }
  cpl_propertylist_erase_regexp(shiftMap->keywords,"ESO PRO REC",0);

  
  it= cpl_frameset_iterator_new(usedframes);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  mat_shiftmap_save(shiftMap,plist_shift,frames,usedframes,
  		    parlist,
  		    RECIPE_NAME,"SHIFT_MAP","IMAGE","F",filenameFits);
  cpl_frameset_delete(usedframes);
  cpl_propertylist_delete(plist_shift);
  // Append the others tables from the raw file except IMAGING_DATA
  nbExtent=cpl_frame_get_nextensions(first_frame);
  for(i=0;i<nbExtent;i++) {
    plist=cpl_propertylist_load(cpl_frame_get_filename(first_frame),i+1);
    keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
    if (keyExtname != NULL) {
      table=cpl_table_load(cpl_frame_get_filename(first_frame),i+1,0);
      if (!strcmp(keyExtname,"IMAGING_DETECTOR")) {
  	mat_imagingdetector_save(shiftMap->imgdet, NULL, filenameFits);
      }
      // Other cases : append the table from the raw file
      if (!strcmp(keyExtname,"ARRAY_DESCRIPTION") ||
  	  !strcmp(keyExtname,"ARRAY_GEOMETRY") ||
  	  !strcmp(keyExtname,"OPTICAL_TRAIN")) {
  	cpl_table_save(table, NULL, plist, filenameFits, CPL_IO_EXTEND);
      }
      cpl_table_delete(table);
    }
    cpl_propertylist_delete(plist);
  }
  cpl_frame_delete(first_frame);
    
  if (coefGlobal != NULL) {
    cpl_vector_delete(coefGlobal);
  }
  cpl_free(detName);
  cpl_free(resolution);
  cpl_free(filter);

  if (shiftMap != NULL) mat_shiftmap_free(shiftMap);

  cpl_vector_delete(offsetwave);
  cpl_vector_delete(offsetpx);
  if (imgsum != NULL)
    {
      for(j=0;j<NREGION;j++)
  	{
  	  cpl_image_delete(imgsum[j]);
  	}
    }
  cpl_free(imgsum);
  if (spectrum != NULL)
    {
      for(j=0;j<NREGION;j++)
  	{
  	  cpl_vector_delete(spectrum[j]);
  	}
    }
  cpl_free(spectrum);
  if (spectrumfiltered != NULL)
    {
      for(j=0;j<NREGION;j++)
  	{
  	  cpl_vector_delete(spectrumfiltered[j]);
  	}
    }
  cpl_free(spectrumfiltered);
  return 0;
}
