import os
import numpy as np
from adari_core.plots.text import TextPlot
from adari_core.report import AdariReportBase
from adari_core.plots.images import ImagePlot
from adari_core.plots.histogram import HistogramPlot
from adari_core.plots.panel import Panel

from .matisse_utils import MatisseSetupInfo, MatisseReportMixin

class MatisseFlatReport(MatisseReportMixin, AdariReportBase):
    def __init__(self):
        super().__init__("matisse_flat")

    def parse_sof(self):
        raw_flat = None
        master_flat = None

        for filename, catg in self.inputs:
            if catg == "OBSDARK" and raw_flat is None:
                raw_flat = filename
            elif catg == "OBS_FLATFIELD" and master_flat is None:
                master_flat = filename

        # Build and return the file name list
        file_lists = []
        if raw_flat is not None:
            if master_flat is not None:
                file_lists.append(
                    {
                        "master_im": master_flat,
                        "raw_im": raw_flat,
                    }
                )

        return file_lists

    def get_data(self, data, ext):
        for i in range(len(data)):
            if str(data[i].header.get("EXTNAME")) == ext:
                out_data = data[i].data
        return out_data

    def generate_panels(self, **kwargs):

        panels = {}

        master_im = self.hdus[0]["master_im"]
        ext_data = "OBS_FLAT_BIAS"
        ext_info = "IMAGING_DETECTOR"
        master_data = self.get_data(master_im, ext_data)
        info_data = self.get_data(master_im, ext_info)
        data_f = master_data[0][1:]
        data_b = master_data[1][1:]

        row_index = np.where(info_data["REGNAME"] == "INTERF_1")[0]

        hdr = master_im[0].header
        procatg = hdr.get("HIERARCH ESO PRO CATG")
        p = Panel(3, 3, height_ratios=[1, 4, 4])

        py = 0
        px = 0
        vspace = 0.2

        fname = master_im.filename()

        t1 = TextPlot(columns=1, v_space=vspace)
        col1 = (
            str(hdr.get("INSTRUME")),
            "EXTNAME: " + ext_data,
            "PRO CATG: " + str(hdr.get("HIERARCH ESO PRO CATG")),
            "FILE NAME: " + os.path.basename(fname),
            "RAW1 NAME: " + str(hdr.get("HIERARCH ESO PRO REC1 RAW1 NAME")),
        )
        t1.add_data(col1)
        p.assign_plot(t1, px, py, xext=2)

        px = px + 2
        t2 = TextPlot(columns=1, v_space=vspace, xext=1)
        self.metadata = MatisseSetupInfo.flat(master_im)
        col2 = self.metadata
        t2.add_data(col2)
        p.assign_plot(t2, px, py, xext=1)

        master_plot_f = ImagePlot(data_f[row_index[0]], title=f"{procatg} F")
        p.assign_plot(master_plot_f, 0, 1, xext=2)

        master_plot_b = ImagePlot(data_b[row_index[0]], title=f"{procatg} B")
        p.assign_plot(master_plot_b, 0, 2, xext=2)

        histogram = HistogramPlot(
            master_data=data_f[row_index[0]],
            raw_data=None,
            title="Counts histogram F",
            v_min=0.8,
            v_max=1.2,
        )
        p.assign_plot(histogram, 2, 1)

        addme = {
            "report_name": f"matisse_{procatg}",
            "report_description": "MATISSE master flat panel",
            "report_tags": [],
        }

        panels[p] = addme

        return panels


rep = MatisseFlatReport()
