from edps import data_source, match_rules
from edps.generator.time_range import *

from .matisse_classification import *

# Convention for Data sources Association rule levels:
# Each data source can have several match function which correspond to different
# quality levels for the selected data. The level is specified as a number that
# follows this convention:
#   level < 0: more restrictive than the calibration plan.
#   level = 0 follows the calibration plan.
#   level = 1 quality sufficient for scientific purposes.
#   level = 2 probably still acceptable quality.
#   level = 3 significant risk of bad quality results.

ONE_YEAR = RelativeTimeRange(-365, 365)
FOUR_MONTHS = RelativeTimeRange(-120, 120)

instrument = [kwd.instrume]
det_name_and_tpl_start = [kwd.det_name, kwd.tpl_start]
detector_kwd = [kwd.det_name, kwd.det_read_curname]
very_high_resolution = [kwd.det_name, kwd.ins_dil_id, kwd.ins_fil_id]
ins_kwd = [kwd.ins_dil_id, kwd.ins_din_id]
group_observations = det_name_and_tpl_start + ins_kwd + [kwd.ins_mode]
group_target = [kwd.det_chip_name, kwd.ins_dil_name, kwd.ins_dil_name, kwd.ins_pil_name, kwd.ins_pin_name, kwd.tpl_id,
                kwd.tpl_start, kwd.cfg_bcd_mode]
cophasing_setup = [kwd.det_name, kwd.ins_mode] + ins_kwd
rmnrec_setup = [kwd.ins_mode, kwd.ins_dil_id, kwd.ins_din_id]
bcd = [kwd.ins_bcd1_id, kwd.ins_bcd2_id]
# --- Raw files data sources -----------------------------------------------------------------------

# --- DataSources for detector_kwd monitoring ---
raw_dark = (data_source()
            .with_classification_rule(dark_class)
            .with_setup_keywords(detector_kwd + [kwd.det_seq1_dit])
            .with_grouping_keywords(detector_kwd + [kwd.tpl_start])
            .with_match_function(rules.is_assoc_dark, time_range=UNLIMITED, level=0)
            .build())

# Task basic_imaging requires different dark association rules
assoc_dark_detector_map = (match_rules()
                           .with_match_keywords([kwd.det_chip_name, kwd.det_read_curname]))

raw_flat = (data_source()
            .with_classification_rule(flat_class)
            .with_setup_keywords(detector_kwd)
            .with_grouping_keywords(detector_kwd + [kwd.tpl_start])
            .with_match_keywords(detector_kwd + [kwd.tpl_start], time_range=UNLIMITED, level=0)
            .build())
# ---

# --- DataSource for flat fielding ---
raw_flat_on = (data_source("LAMP_FLAT")
               .with_classification_rule(flat_on_class)
               .with_classification_rule(flat_off_class)
               .with_setup_keywords(detector_kwd + ins_kwd)
               .with_grouping_keywords(detector_kwd + ins_kwd + [kwd.tpl_start])
               .with_match_function(rules.is_assoc_flat, time_range=ONE_MONTH, level=0)
               .with_match_function(rules.is_assoc_flat, time_range=UNLIMITED, level=3)
               .build())

assoc_flats_to_science = (match_rules()
                          .with_match_function(rules.is_assoc_flat_science, time_range=ONE_MONTH, level=0)
                          .with_match_function(rules.is_assoc_flat_science, time_range=FOUR_MONTHS, level=3))

assoc_flats_to_calibrator = (match_rules()
                             .with_match_function(rules.is_assoc_flat_calibrator, time_range=RelativeTimeRange(-30, 0),
                                                  level=0)
                             .with_match_function(rules.is_assoc_flat_calibrator, time_range=ONE_MONTH, level=1)
                             .with_match_function(rules.is_assoc_flat_calibrator, time_range=FOUR_MONTHS, level=3))

assoc_flats_to_lamp = (match_rules()
                       .with_match_function(rules.is_assoc_flat_lamp, time_range=UNLIMITED, level=0))

assoc_flats_to_kappa = (match_rules()
                        .with_match_function(rules.is_assoc_flat_kappa, time_range=UNLIMITED, level=0))

# --- DataSources for distortion ---
raw_distortion_very_high_resolution = (data_source("DISTORTION_VERY_HIGH_RESOLUTION")
                                       .with_classification_rule(distortion_calibrator)
                                       .with_classification_rule(distortion_sky)
                                       .with_grouping_keywords(very_high_resolution + [kwd.tpl_start])
                                       .with_setup_keywords(very_high_resolution)
                                       .with_match_function(rules.is_assoc_distortion)
                                       .build())

shift_map = (data_source("SHIFT_MAP_VERY_HIGH_RESOLUTION")
             .with_classification_rule(SHIFT_MAP)
             .with_match_keywords(very_high_resolution, time_range=ONE_YEAR, level=0)
             .with_match_keywords(very_high_resolution, time_range=UNLIMITED, level=3)
             .build())

kappa_matrix_static = (data_source("KAPPA_MATRIX_VERY_HIGH_RESOLUTION")
                       .with_classification_rule(KAPPA_MATRIX)
                       .with_match_keywords(very_high_resolution + [kwd.ins_pol_name], time_range=ONE_YEAR, level=0)
                       .with_match_keywords(very_high_resolution + [kwd.ins_pol_name], time_range=UNLIMITED, level=3)
                       .build())

same_raw_inputs = (match_rules()
                   .with_match_keywords([kwd.mjd_obs]))

raw_distortion = (data_source("DISTORTION")
                  .with_classification_rule(distor_hotdark_class)
                  .with_classification_rule(distor_images_class)
                  .with_classification_rule(spectra_hotdark_class)
                  .with_classification_rule(spectra_images_class)
                  .with_setup_keywords(ins_kwd + [kwd.det_name])
                  .with_grouping_keywords(ins_kwd + [kwd.det_name, kwd.tpl_start])
                  .with_match_function(rules.is_assoc_distortion, time_range=UNLIMITED, level=0)
                  .build())
# ---

# - Intensity ratio,  the relative shift and zoom between each photometric channel and the
#  interferometric one for each spectral channel.
raw_kappa = (data_source("KAPPA_MATRIX")
             .with_classification_rule(kappa_hotdark_class)
             .with_classification_rule(kappa_src_class)
             .with_classification_rule(kappa_object_class)
             .with_setup_keywords(ins_kwd + [kwd.det_name, kwd.ins_fil_id])
             .with_grouping_keywords(ins_kwd + [kwd.det_name, kwd.tpl_start])
             .with_match_function(rules.is_assoc_kappa, time_range=ONE_MONTH, level=0)
             .with_match_function(rules.is_assoc_kappa, time_range=UNLIMITED, level=3)
             .build())

# Datasources collecting raw calibrations
raw_monitoring_shutter_open = (data_source("TEST_AND_HOT_CALIBRATIONS")
                               .with_classification_rule(hot_dark_shutter_open_class)
                               .with_classification_rule(hot_lamp_shutter_open_class)
                               .with_grouping_keywords(group_observations + bcd)
                               .with_setup_keywords(ins_kwd + bcd + [kwd.det_name, kwd.ins_mode])
                               .with_match_keywords([kwd.mjd_obs])
                               .build())

raw_monitoring_shutter_closed = (data_source("TEST_AND_HOT_CALIBRATIONS_CLOSED")
                                 .with_classification_rule(hot_dark_shutter_closed_class)
                                 .with_classification_rule(hot_lamp_shutter_closed_class)
                                 .with_match_keywords(group_observations)
                                 .build())

raw_calibrator_shutter_open = (data_source("CALIBRATOR")
                               .with_classification_rule(raw_target_calib_shutter_open_class)
                               .with_classification_rule(raw_source_calib_shutter_open_class)
                               .with_grouping_keywords(group_observations + bcd)
                               .with_setup_keywords(ins_kwd + bcd + [kwd.det_name, kwd.ins_mode])
                               .with_match_keywords([kwd.tpl_start], time_range=SAME_NIGHT, level=0)
                               .build())

raw_calibrator_shutter_closed = (data_source("CALIBRATOR_CLOSED")
                                 .with_classification_rule(raw_target_calib_shutter_closed_class)
                                 .with_classification_rule(raw_source_calib_shutter_closed_class)
                                 .with_match_keywords(group_observations)
                                 .build())

raw_science_shutter_open = (data_source("SCIENCE")
                            .with_classification_rule(raw_target_sci_shutter_open_class)
                            .with_grouping_keywords(group_observations + bcd)
                            .with_setup_keywords(ins_kwd + bcd + [kwd.det_name, kwd.ins_mode])
                            .with_match_keywords(group_observations)
                            .build())

raw_science_shutter_closed = (data_source("SCIENCE_CLOSED")
                              .with_classification_rule(raw_target_sci_shutter_closed_class)
                              .with_match_keywords(group_observations)
                              .build())

raw_sky = (data_source("SKY")
           .with_classification_rule(raw_sky_class)
           .with_grouping_keywords(group_observations)
           .with_match_keywords(group_observations)
           .build())

jsdc_cat = (data_source()
            .with_classification_rule(jsdc_cat_class)
            .with_match_keywords(instrument, time_range=UNLIMITED, level=0)
            .build())

# Datasources for processed data

reduced_target = (data_source("UNCALIBRATED_TARGET_VISIBILITIES")
                  .with_classification_rule(reduced_target_outout_class)
                  .with_classification_rule(reduced_target_inin_class)
                  .with_classification_rule(reduced_target_outin_class)
                  .with_classification_rule(reduced_target_inout_class)
                  .with_grouping_keywords(group_target)
                  .build())

reduced_calibrator = (data_source("UNCALIBRATED_CALIBRATOR_VISIBILITIES")
                      .with_classification_rule(reduced_calibrator_outout_class)
                      .with_classification_rule(reduced_calibrator_inin_class)
                      .with_classification_rule(reduced_calibrator_outin_class)
                      .with_classification_rule(reduced_calibrator_inout_class)
                      .with_grouping_keywords([kwd.arcfile])
                      .build())

# Other datasources: test and Reflective Memory Network Recorder
raw_rmnrec = (data_source("RMNrec")
              .with_classification_rule(calib_raw_rmnrec_class)
              .with_classification_rule(target_raw_rmnrec_class)
              .with_setup_keywords(rmnrec_setup)
              .with_grouping_keywords(rmnrec_setup + [kwd.tpl_start, kwd.dpr_type])
              .with_match_keywords([kwd.tpl_start, kwd.obs_id], time_range=ONE_DAY, level=0)
              .build())

raw_test_calib = (data_source("TEST_CALIBRATIONS")
                  .with_classification_rule(test_calib_raw_class)
                  .with_classification_rule(raw_std_image_class)
                  .with_classification_rule(test_calib_raw_class)
                  .with_classification_rule(raw_ali_refpix_class)
                  .with_setup_keywords(detector_kwd)
                  .with_grouping_keywords(detector_kwd + [kwd.tpl_start])
                  .with_match_function(rules.is_assoc_dark, time_range=UNLIMITED, level=0)
                  .build())

raw_acquisition = (data_source("ACQUISITION")
                   .with_classification_rule(raw_acq_pupil_calib_class)
                   .with_classification_rule(raw_acq_pupil_object_class)
                   .with_classification_rule(raw_acq_pupil_sky_class)
                   .with_classification_rule(raw_acq_image_calib_class)
                   .with_classification_rule(raw_acq_image_object_class)
                   .with_classification_rule(raw_acq_image_sky_class)
                   .with_classification_rule(raw_acq_search_calib_class)
                   .with_classification_rule(raw_acq_search_object_class)
                   .with_classification_rule(raw_acq_search_sky_class)
                   .with_setup_keywords([kwd.det_name])
                   .with_grouping_keywords(detector_kwd + [kwd.tpl_start, kwd.dpr_type])
                   .with_match_function(rules.is_assoc_dark, time_range=UNLIMITED, level=0)
                   .build())

raw_im_cold = (data_source("BASIC_DETECTOR_CALIBRATION")
               .with_classification_rule(im_cold_class)
               .with_setup_keywords(detector_kwd)
               .with_grouping_keywords(detector_kwd + [kwd.tpl_start])
               .with_match_function(rules.is_assoc_dark, time_range=UNLIMITED, level=0)
               .build())

raw_cophasing = (data_source("COPHASING")
                 .with_classification_rule(cophasing_lab_dark_class)
                 .with_classification_rule(cophasing_lab_lamp_class)
                 .with_classification_rule(cophasing_lab_dark_class)
                 .with_classification_rule(cophasing_lab_lamp_class)
                 .with_classification_rule(cophasing_image_hotdark_class)
                 .with_classification_rule(cophasing_image_lamp_class)
                 .with_classification_rule(cophasing_pupil_hotdark_class)
                 .with_classification_rule(cophasing_pupil_lamp_class)
                 .with_setup_keywords(cophasing_setup)
                 .with_grouping_keywords(cophasing_setup + [kwd.tpl_start])
                 .build())

raw_im_extended = (data_source("EXTENDED_DETECTOR_CALIBRATION")
                   .with_classification_rule(im_extended_class)
                   .with_setup_keywords(detector_kwd)
                   .with_grouping_keywords(detector_kwd + [kwd.tpl_start])
                   .with_match_function(rules.is_assoc_dark, time_range=UNLIMITED, level=0)
                   .build())

raw_im_remanence = (data_source("REMANENCE_DETECTOR_CALIBRATION")
                    .with_classification_rule(im_remanence1_class)
                    .with_classification_rule(im_remanence2_class)
                    .with_setup_keywords(detector_kwd)
                    .with_grouping_keywords(detector_kwd + [kwd.tpl_start, kwd.dpr_type])
                    .with_match_function(rules.is_assoc_dark, time_range=UNLIMITED, level=0)
                    .build())
