from edps import Job, RecipeInvocationArguments, RecipeInvocationResult
from edps import RecipeInputs, ClassifiedFitsFile, List, get_parameter, JobParameters

from . import matisse_keywords as kwd


# --- Generic function to run a recipe
def run_recipe(input_file, associated_files, parameters, recipe_name, args, invoker, renamer) -> RecipeInvocationResult:
    # input_file: main input and category. Format: List[files], where files have the format
    #             File(string_with_full_path, string_with_category, "")
    # associated_files: calibrations. Format List[files], where files have the format
    #             File(string_with_full_path, string_with_category, "")
    # parameters: non default recipe parameters. Format {'parameter_name1': value1, 'parameter_name2': value2}
    # recipe_name: recipe name  Format: string
    # args, invoker: extra stuff provided by the task that calls the function calling run_recipe()

    inputs = RecipeInputs(main_upstream_inputs=input_file, associated_upstream_inputs=associated_files)
    arguments = RecipeInvocationArguments(inputs=inputs, parameters=parameters,
                                          job_dir=args.job_dir, input_map={},
                                          logging_prefix=args.logging_prefix)

    return invoker.invoke(recipe_name, arguments, renamer, create_subdir=True)


# --- Functions to set the parameters for the recipe mat_cal_det ----------------------------------
def set_mat_cal_det_params(job: Job, gain, darklimit, flatlimit, max_rel_deviation):
    job.parameters.recipe_parameters["matisse.mat_cal_det.gain"] = gain
    job.parameters.recipe_parameters["matisse.mat_cal_det.darklimit"] = darklimit
    job.parameters.recipe_parameters["matisse.mat_cal_det.flatlimit"] = flatlimit
    job.parameters.recipe_parameters["matisse.mat_cal_det.max_rel_deviation"] = max_rel_deviation


def setup_dark(job: Job):
    # ---- Change parameter setting depending on the input data type ----------
    # If a parameter is not changed, the value specified in matisse_paramters.yaml or the recipe default
    # is used (in this preferental order).

    f = job.input_files
    det_speed = f[0].get_keyword_value(kwd.det_read_curname, None)

    if det_speed == "SCI-SLOW-SPEED":
        set_mat_cal_det_params(job, 2.73, 100, 0.3, 0.01)

    elif det_speed == "SCI-FAST-SPEED":
        set_mat_cal_det_params(job, 2.60, 100, 0.3, 0.01)

    elif det_speed == "SCI-LOW-GAIN":
        set_mat_cal_det_params(job, 190, 100, 0.2, 0.02)

    elif det_speed == "SCI-HIGH-GAIN":
        set_mat_cal_det_params(job, 20, 200, 0.2, 0.01)


# -------------------------------------------------------------------------------------------------

# --- Functions to set the parameters for the recipe mat_ext_beams --------------------------------
def set_ext_beams_parameters(job: Job, hampel=10, binning=1, replace=3):
    job.parameters.recipe_parameters["matisse.mat_ext_beams.hampelFilterKernel"] = hampel
    job.parameters.recipe_parameters["matisse.mat_raw_estimates.spectralBinning"] = binning
    job.parameters.recipe_parameters["matisse.mat_ext_beams.replaceTel"] = replace


def set_ext_beams(job: Job):
    # ---- Change parameter setting depending on the input data type ----------
    # If a parameter is not changed, the value specified in matisse_paramters.yaml or the recipe default
    # is used (in this preferental order).
    # The recipe input is a job, therefore I have to look into job.input_jobs[0].product
    det_name = job.input_jobs[0].product.get_keyword_value(kwd.det_name, None)
    if det_name == "MATISSE-LM":
        set_ext_beams_parameters(job, hampel=10, binning=5, replace=0)
    elif det_name == "MATISSE-N":
        set_ext_beams_parameters(job, hampel=0, binning=7, replace=3)


# -------------------------------------------------------------------------------------------------
# --- Functions to set the parameters for the recipe mat_est_corr --------------------------------
def set_est_corr_parameters(job: Job, mode='false', time=0.0, binning=5):
    job.parameters.recipe_parameters["matisse.mat_est_corr.useOpdMod"] = mode
    job.parameters.recipe_parameters["matisse.mat_est_corr.coherentIntegTime"] = time
    job.parameters.recipe_parameters["matisse.mat_raw_estimates.spectralBinning"] = binning


def set_est_corr(job: Job):
    # ---- Change parameter setting depending on the input data type ----------
    # If a parameter is not changed, the value specified in matisse_paramters.yaml or the recipe default
    # is used (in this preferental order).
    # The recipe input is a job, therefore I have to look into job.input_jobs[0].product
    det_name = job.input_jobs[0].product.get_keyword_value(kwd.det_name, None)
    if det_name == "MATISSE-LM":
        set_est_corr_parameters(job, mode="false", time=0, binning=5)
    elif det_name == "MATISSE-N":
        set_est_corr_parameters(job, mode="true", time=0, binning=7)


# -------------------------------------------------------------------------------------------------

# -- Function to define the (dynamic) parameter that specifies the resolution of the main input
def which_resolution(files: List[ClassifiedFitsFile]):
    res = files[0].get_keyword_value(kwd.ins_dil_id, None)
    return res


def which_band(files: List[ClassifiedFitsFile]):
    res = files[0].get_keyword_value(kwd.det_name, None)
    return res


def which_bcd_mode(files: List[ClassifiedFitsFile]):
    res1 = files[0].get_keyword_value(kwd.ins_bcd1_id, "None")
    res2 = files[0].get_keyword_value(kwd.ins_bcd2_id, "None")
    return res1 + '-' + res2


# Functions that define conditions depending on the values of the dynamic parameters
def is_standard_resolutions(params: JobParameters) -> bool:
    return get_parameter(params, "which_resolution") != "HIGH+"


def is_LM(params: JobParameters) -> bool:
    return get_parameter(params, "which_band") == "MATISSE-LM"


def is_N(params: JobParameters) -> bool:
    return get_parameter(params, "which_band") == "MATISSE-N"


def is_very_high_resolution(params: JobParameters) -> bool:
    return get_parameter(params, "which_resolution") == "HIGH+"


def set_bcd_mode(job: Job):
    bcdmode = job.parameters.get_workflow_param("which_bcd_mode")
    job.parameters.recipe_parameters["matisse.mat_merge_results.cumulBlock"] = "TRUE"

    if bcdmode == "OUT-OUT":
        job.parameters.recipe_parameters['matisse.mat_merge_results.bcdMode'] = 0
    elif bcdmode == "IN-IN":
        job.parameters.recipe_parameters['matisse.mat_merge_results.bcdMode'] = 1
    elif bcdmode == "IN-OUT":
        job.parameters.recipe_parameters['matisse.mat_merge_results.bcdMode'] = 2
    elif bcdmode == "OUT-IN":
        job.parameters.recipe_parameters['matisse.mat_merge_results.bcdMode'] = 3
    else:
        job.parameters.recipe_parameters['matisse.mat_merge_results.bcdMode'] = -1
