/* $Id: mat_apply_opdmod.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/09/26 16:50:00 $
 * $Revision: 0.4 $
 * $Name: mat_apply_opdmod.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/
#include <math.h>

#include "mat_apply_opdmod.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/

double factorN[6]={1,1,1,1,1,1};
double factorL[6]={1.,1.,1.,1.,1.,1.};
//double factorL[6]={0.,0.,0.,0.,0.,0.};

//N bcd IN
/* double gainUpInN[4]={1.050,1.07564,1.01262,1.0301}; */
/* double gainDownInN[4]={1.043,1.0776,1.0088,1.02336}; */
/* double offsetInN[4]={-0.56517,0.08895,0.30267,0.32602}; */
double gainUpInN[4]={1.,1.,1.,1.};
double gainDownInN[4]={1.,1.,1.,1.};
double offsetInN[4]={0.,0.0,0.,0.};

//N bcd OUT
/* double gainUpOutN[4]={1.0465,1.0943,1.011,1.0241}; */
/* double gainDownOutN[4]={1.0460,1.0955,1.008,1.0212}; */
/* double offsetOutN[4]={-0.49775,0.02015,0.2148,0.35388}; */
double gainUpOutN[4]={1.,1.,1.,1.};
double gainDownOutN[4]={1.,1.,1.,1.};
double offsetOutN[4]={0.,0.0,0.,0.};

//L bcd IN
/* double gainUpInL[4]={-1.07445592,-1.08594216,-1.07040287,-1.06000657}; */
/* double gainDownInL[4]={-1.0684817,-1.07316373,-1.07045833,-1.05623881}; */
/* double offsetInL[4]={0.01613656,0.16462196,-0.12370515,-0.04182857}; */
double gainUpInL[4]={-1.,-1.,-1.,-1.};
double gainDownInL[4]={-1.,-1.,-1.,-1.};
double offsetInL[4]={0.,0.0,0.,0.};

// L bcd OUT
/* double gainUpOutL[4]={-1.07403707,-1.08589892,-1.07121155,-1.06020638}; */
/* double gainDownOutL[4]={-1.06817308,-1.07223093,-1.06978822,-1.05708504}; */
/* double offsetOutL[4]={0.00466866,0.17744104,-0.10229686,-0.06476895}; */
double gainUpOutL[4]={-1.,-1.,-1.,-1.};
double gainDownOutL[4]={-1.,-1.,-1.,-1.};
double offsetOutL[4]={0.,0.0,0.,0.};

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*/
/**
   @ingroup signal
   @brief remove the residual background by applying the opd modulation
   @param maxstep           number of opd modulation steps
   @param real              real part of the fourirer transform
   @param imag              imaginary part of the fourier transform
   @param filledCycle       vector containing the status of each step 
   ( 1 = step provided)
   @param localOpd          vector containg 4 values of OPD for each step
   @param lambda            wavelength of the spectral channel
   @param result            cpl_vector containing the real and imaginary part 
   of the fourier transform of one spectral channle 
   corrected from teh residual background
   @param resolution        Spectral resolution
   @param nbdet             Detector number (1:HAWAII, 2:AQUARIUS)
   @param flagBcdIN         =1 BCD IN; =0 BCD OUT
   @return 1 in case of error
*/
/*----------------------------------------------------------------------------*/
int mat_apply_opdmod(int maxstep, 
		     cpl_vector *real,
		     cpl_vector *imag,
		     int filledCycle[],
		     cpl_vector *localOpd,
		     double lambda,
		     cpl_vector *result,
		     int resolution,
		     int nbdet,
                     int flagBcdIN) {
    
     
  int i=0;
  int j=0;
  int option=1; //(option=0 LS method option=1 AVG method)
  int cpt=0;
  int dimSpatial=0;
  int nbStep=0;
  double opd=0.;
  /* double bid=0; */
  double a=0.;
  double b=0.;
  cpl_matrix *coeff=NULL;
  cpl_matrix *rhs=NULL;
  cpl_matrix *sol=NULL;
  double posFringePeak[6];
  double dOverLambda[6];
  int iPeak=0;
  int flagUp=0;
  double gain[4];
  double offset[4];

  // Check input parameters
  mat_assert_value((maxstep>0),CPL_ERROR_NULL_INPUT, 1,
		   "maxstep sould be greater than 0");
  mat_assert_value((real!=NULL),CPL_ERROR_NULL_INPUT, 1, 
		   "no cpl_vector (real) argument given");
  mat_assert_value((imag!=NULL),CPL_ERROR_NULL_INPUT, 1, 
		   "no cpl_vector (imag) argument given");
  mat_assert_value((localOpd!=NULL),CPL_ERROR_NULL_INPUT, 1, 
		   "no cpl_vector (localOpd) argument given");
  mat_assert_value((lambda>0),CPL_ERROR_NULL_INPUT, 1, 
		   "lambda should be greater than 0");

  for(j=0;j<maxstep-1;j++) {
    if (filledCycle[j]) {
      if (cpl_vector_get(localOpd,j)-cpl_vector_get(localOpd,j+1)>0) {
        flagUp=0;
	break;
      } else {
        flagUp=1;
	break;
      }
    }
  }
  
  for(int iPiezo=0;iPiezo<4;iPiezo++) {
    if (flagUp==1) {
      if (flagBcdIN == 1) {
        if (nbdet == 2) {
          gain[iPiezo]=gainUpInN[iPiezo];
          offset[iPiezo]=offsetInN[iPiezo]/2.;
	} else {
	  gain[iPiezo]=gainUpInL[iPiezo];
          offset[iPiezo]=offsetInL[iPiezo]/2.;
	}
      } else {
        if (nbdet == 2) {
          gain[iPiezo]=gainUpOutN[iPiezo];
          offset[iPiezo]=offsetOutN[iPiezo]/2.;
	} else {
          gain[iPiezo]=gainUpOutL[iPiezo];
          offset[iPiezo]=offsetOutL[iPiezo]/2.;
	}
      }
    } else {
      if (flagBcdIN == 1) {
        if (nbdet == 2) {
          gain[iPiezo]=gainDownInN[iPiezo];
          offset[iPiezo]=-1.*offsetInN[i]/2.;
  	} else {
          gain[iPiezo]=gainDownInL[iPiezo];
          offset[iPiezo]=-1.*offsetInL[iPiezo]/2.;
  	}
      } else {
        if (nbdet == 2) {
          gain[iPiezo]=gainDownOutN[iPiezo];
          offset[iPiezo]=-1.*offsetOutN[iPiezo]/2.;
  	} else {
          gain[iPiezo]=gainDownOutL[iPiezo];
          offset[iPiezo]=-1.*offsetOutL[iPiezo]/2.;
  	}
      }
    }
  }

  dimSpatial=cpl_vector_get_size(real)/maxstep;


  for(iPeak=0;iPeak<6;iPeak++) {
    posFringePeak[iPeak]=mat_get_pos_fringe_peak(lambda,nbdet,resolution,iPeak);
    dOverLambda[iPeak]=(posFringePeak[iPeak]-dimSpatial/2)/(3*(iPeak+1));
  }


  //Determine the number of valid steps
  nbStep=0;
  for(j=0;j<maxstep;j++) {
    if (filledCycle[j]) {
      nbStep++;
    }
  }
  if (nbStep < 2) {
    cpl_msg_warning(cpl_func,"Not enough valid steps - nbStep=%d",nbStep);
    return -1;
  }
  
  for(i=0;i<2*dimSpatial;i++) {
    cpl_vector_set(result,i,0.);
  }
  
  for(i=0;i<dimSpatial;i++) {
    cpl_vector_set(result,i,cpl_vector_get(real,i));
    cpl_vector_set(result,i+dimSpatial,cpl_vector_get(imag,i));
  }
  for(i=dimSpatial/2;i<dimSpatial;i++) {
    rhs=cpl_matrix_new(2*nbStep,1);
    if (rhs == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
      return CPL_ERROR_UNSPECIFIED;
    }
    coeff=cpl_matrix_new(2*nbStep,4);
    if (coeff == NULL) {
      cpl_matrix_delete(rhs);
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
      return CPL_ERROR_UNSPECIFIED;
    }
    

    // 1st peak : T3T4 3D
    // 2nd peak : T1T2 6D
    // 3rd peak : T2T3 9D
    // 4th peak : T2T4 12D
    // 5th peak : T1T3 15D
    // 6th peak : T1T4 19D

    // 1st fringe peak
    if ( (i-dimSpatial/2) <= 4.5*dOverLambda[0] ) {
      if (option == 0 ) {
	cpt=0;
	for(j=0;j<maxstep;j++) {
	  if (filledCycle[j]) {
	    if (nbdet==2)
	      opd=factorN[0]*(gain[3]*cpl_vector_get(localOpd,3*maxstep+j)-gain[2]*cpl_vector_get(localOpd,2*maxstep+j)+offset[3]-offset[2]);
	    if (nbdet==1)
	      opd=factorL[0]*(gain[3]*cpl_vector_get(localOpd,3*maxstep+j)-gain[2]*cpl_vector_get(localOpd,2*maxstep+j)+offset[3]-offset[2]);
	    cpl_matrix_set(rhs,cpt,0,cpl_vector_get(real,j*dimSpatial+i));
	    cpl_matrix_set(rhs,cpt+nbStep,0,cpl_vector_get(imag,j*dimSpatial+i));
	    cpl_matrix_set(coeff,cpt,0,cos(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt,1,sin(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt,2,1.);
	    cpl_matrix_set(coeff,cpt,3,0.);
	    cpl_matrix_set(coeff,cpt+nbStep,0,-1.*sin(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt+nbStep,1,cos(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt+nbStep,2,0.);
	    cpl_matrix_set(coeff,cpt+nbStep,3,1.);
	    cpt++;
	  }
	}
	sol=cpl_matrix_solve_normal(coeff,rhs);
	if (sol == NULL) {
	  cpl_msg_error(cpl_func,
			"could not solve normal system (1st fringe peak)");
	  return CPL_ERROR_UNSPECIFIED;
	}
	cpl_vector_set(result,i,cpl_matrix_get(sol,0,0));
	cpl_vector_set(result,i+dimSpatial,cpl_matrix_get(sol,1,0));
	if ( i > dimSpatial/2) {
	  cpl_vector_set(result,2*(dimSpatial/2)-i,cpl_matrix_get(sol,0,0));
	  cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,
			 -1.*cpl_matrix_get(sol,1,0));
	}
	cpl_matrix_delete(sol);
      } else {
	cpt=0;
	a=0.;
	b=0.;
	for(j=0;j<maxstep;j++) {
	  if (filledCycle[j]) {
	    if (nbdet==2)
	      opd=factorN[0]*(gain[3]*cpl_vector_get(localOpd,3*maxstep+j)-gain[2]*cpl_vector_get(localOpd,2*maxstep+j)+offset[3]-offset[2]);
	    if (nbdet==1)
	      opd=factorL[0]*(gain[3]*cpl_vector_get(localOpd,3*maxstep+j)-gain[2]*cpl_vector_get(localOpd,2*maxstep+j)+offset[3]-offset[2]);
	    b+=cpl_vector_get(imag,j*dimSpatial+i)*cos(2*CPL_MATH_PI*opd/lambda)-
	      cpl_vector_get(real,j*dimSpatial+i)*sin(2*CPL_MATH_PI*opd/lambda);
	    a+=cpl_vector_get(imag,j*dimSpatial+i)*sin(2*CPL_MATH_PI*opd/lambda)+
	      cpl_vector_get(real,j*dimSpatial+i)*cos(2*CPL_MATH_PI*opd/lambda);
	    cpt++;
	  }
	}
	a/=cpt;
	b/=cpt;
	cpl_vector_set(result,i,a);
	cpl_vector_set(result,i+dimSpatial,b);
	if ( i > dimSpatial/2) {
	  cpl_vector_set(result,2*(dimSpatial/2)-i,a);
	  cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*b);
	}
      }
      // 2nd fringe peak
    } else if ( fabs(i-posFringePeak[1]) <= 1.6*dOverLambda[1] ) {
      if (option == 0 ) {
	cpt=0;
	for(j=0;j<maxstep;j++) {
	  if (filledCycle[j]) {
	    if (nbdet==2)
	      opd=factorN[1]*(gain[1]*cpl_vector_get(localOpd,1*maxstep+j)-gain[0]*cpl_vector_get(localOpd,0*maxstep+j)+offset[1]-offset[0]);
	    if (nbdet==1)
	      opd=factorL[1]*(gain[1]*cpl_vector_get(localOpd,1*maxstep+j)-gain[0]*cpl_vector_get(localOpd,0*maxstep+j)+offset[1]-offset[0]);
	    cpl_matrix_set(rhs,cpt,0,
			   cpl_vector_get(real,j*dimSpatial+i));
	    cpl_matrix_set(rhs,cpt+nbStep,0,
			   cpl_vector_get(imag,j*dimSpatial+i));
	    cpl_matrix_set(coeff,cpt,0,cos(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt,1,sin(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt,2,1.);
	    cpl_matrix_set(coeff,cpt,3,0.);
	    cpl_matrix_set(coeff,cpt+nbStep,0,-1.*sin(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt+nbStep,1,cos(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt+nbStep,2,0.);
	    cpl_matrix_set(coeff,cpt+nbStep,3,1.);
	    cpt++;
	  }
	}
	sol=cpl_matrix_solve_normal(coeff,rhs);
	if (sol == NULL) {
	  cpl_msg_error(cpl_func,
			"could not solve normal system (2nd fringe peak)");
	  return CPL_ERROR_UNSPECIFIED;
	}
	cpl_vector_set(result,i,cpl_matrix_get(sol,0,0));
	cpl_vector_set(result,i+dimSpatial,cpl_matrix_get(sol,1,0));
	if ( i > dimSpatial/2) {
	  cpl_vector_set(result,2*(dimSpatial/2)-i,cpl_matrix_get(sol,0,0));
	  cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.
			 *cpl_matrix_get(sol,1,0));
	}
	cpl_matrix_delete(sol);
      } else {
	cpt=0;
	a=0.;
	b=0.;
	for(j=0;j<maxstep;j++) {
	  if (filledCycle[j]) {
	    if (nbdet==2)
	      opd=factorN[1]*(gain[1]*cpl_vector_get(localOpd,1*maxstep+j)-gain[0]*cpl_vector_get(localOpd,0*maxstep+j)+offset[1]-offset[0]);
	    if (nbdet==1)
	      opd=factorL[1]*(gain[1]*cpl_vector_get(localOpd,1*maxstep+j)-gain[0]*cpl_vector_get(localOpd,0*maxstep+j)+offset[1]-offset[0]);
	    b+=cpl_vector_get(imag,j*dimSpatial+i)*cos(2*CPL_MATH_PI*opd/lambda)-
	      cpl_vector_get(real,j*dimSpatial+i)*sin(2*CPL_MATH_PI*opd/lambda);
	    a+=cpl_vector_get(imag,j*dimSpatial+i)*sin(2*CPL_MATH_PI*opd/lambda)+
	      cpl_vector_get(real,j*dimSpatial+i)*cos(2*CPL_MATH_PI*opd/lambda);
	    cpt++;
	  }
	}
	a/=cpt;
	b/=cpt;
	cpl_vector_set(result,i,a);
	cpl_vector_set(result,i+dimSpatial,b);
	if ( i > dimSpatial/2) {
	  cpl_vector_set(result,2*(dimSpatial/2)-i,a);
	  cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*b);
	}
      }
	    
      // 3rd fringe peak
    } else if ( fabs(i-posFringePeak[2]) <= 1.6*dOverLambda[2] ) {
      if (option == 0 ) {
	cpt=0;
	for(j=0;j<maxstep;j++) {
	  if (filledCycle[j]) {
	    if (nbdet==2)
	      opd=factorN[2]*(gain[2]*cpl_vector_get(localOpd,2*maxstep+j)-gain[1]*cpl_vector_get(localOpd,1*maxstep+j)+offset[2]-offset[1]);
	    if (nbdet==1)
	      opd=factorL[2]*(gain[2]*cpl_vector_get(localOpd,2*maxstep+j)-gain[1]*cpl_vector_get(localOpd,1*maxstep+j)+offset[2]-offset[1]);
	    cpl_matrix_set(rhs,cpt,0,
			   cpl_vector_get(real,j*dimSpatial+i));
	    cpl_matrix_set(rhs,cpt+nbStep,0,
			   cpl_vector_get(imag,j*dimSpatial+i));
	    cpl_matrix_set(coeff,cpt,0,cos(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt,1,sin(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt,2,1.);
	    cpl_matrix_set(coeff,cpt,3,0.);
	    cpl_matrix_set(coeff,cpt+nbStep,0,-1.*sin(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt+nbStep,1,cos(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt+nbStep,2,0.);
	    cpl_matrix_set(coeff,cpt+nbStep,3,1.);
	    cpt++;
	  }
	}
	sol=cpl_matrix_solve_normal(coeff,rhs);
	if (sol == NULL) {
	  cpl_msg_error(cpl_func,
			"could not solve normal system (3rd fringe peak)");
	  return CPL_ERROR_UNSPECIFIED;
	}
	cpl_vector_set(result,i,cpl_matrix_get(sol,0,0));
	cpl_vector_set(result,i+dimSpatial,cpl_matrix_get(sol,1,0));
	if ( i > dimSpatial/2) {
	  cpl_vector_set(result,2*(dimSpatial/2)-i,cpl_matrix_get(sol,0,0));
	  cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,
			 -1.*cpl_matrix_get(sol,1,0));
	}
	cpl_matrix_delete(sol);
      } else {
	cpt=0;
	a=0.;
	b=0.;
	for(j=0;j<maxstep;j++) {
	  if (filledCycle[j]) {
	    if (nbdet==2)
	      opd=factorN[2]*(gain[2]*cpl_vector_get(localOpd,2*maxstep+j)-gain[1]*cpl_vector_get(localOpd,1*maxstep+j)+offset[2]-offset[1]);
	    if (nbdet==1)
	      opd=factorL[2]*(gain[2]*cpl_vector_get(localOpd,2*maxstep+j)-gain[1]*cpl_vector_get(localOpd,1*maxstep+j)+offset[2]-offset[1]);
	    b+=cpl_vector_get(imag,j*dimSpatial+i)*cos(2*CPL_MATH_PI*opd/lambda)-
	      cpl_vector_get(real,j*dimSpatial+i)*sin(2*CPL_MATH_PI*opd/lambda);
	    a+=cpl_vector_get(imag,j*dimSpatial+i)*sin(2*CPL_MATH_PI*opd/lambda)+
	      cpl_vector_get(real,j*dimSpatial+i)*cos(2*CPL_MATH_PI*opd/lambda);
	    cpt++;
	  }
	}
	a/=cpt;
	b/=cpt;
	cpl_vector_set(result,i,a);
	cpl_vector_set(result,i+dimSpatial,b);
	if ( i > dimSpatial/2) {
	  cpl_vector_set(result,2*(dimSpatial/2)-i,a);
	  cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*b);
	}
      }

      // 4th fringe peak
    } else if ( fabs(i-posFringePeak[3]) <= 1.6*dOverLambda[3] ) {
      if (option == 0 ) {
	cpt=0;
	for(j=0;j<maxstep;j++) {
	  if (filledCycle[j]) {
	    if (nbdet==2) 
	      opd=factorN[3]*(gain[3]*cpl_vector_get(localOpd,3*maxstep+j)-gain[1]*cpl_vector_get(localOpd,1*maxstep+j)+offset[3]-offset[1]);
	    if (nbdet==1)
	      opd=factorL[3]*(gain[3]*cpl_vector_get(localOpd,3*maxstep+j)-gain[1]*cpl_vector_get(localOpd,1*maxstep+j)+offset[3]-offset[1]);
	    cpl_matrix_set(rhs,cpt,0,
			   cpl_vector_get(real,j*dimSpatial+i));
	    cpl_matrix_set(rhs,cpt+nbStep,0,
			   cpl_vector_get(imag,j*dimSpatial+i));
	    cpl_matrix_set(coeff,cpt,0,cos(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt,1,sin(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt,2,1.);
	    cpl_matrix_set(coeff,cpt,3,0.);
	    cpl_matrix_set(coeff,cpt+nbStep,0,-1.*sin(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt+nbStep,1,cos(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt+nbStep,2,0.);
	    cpl_matrix_set(coeff,cpt+nbStep,3,1.);
	    cpt++;
	  }
	}
	sol=cpl_matrix_solve_normal(coeff,rhs);
	if (sol == NULL) {
	  cpl_msg_error(cpl_func,
			"could not solve normal system (4th fringe peak)");
	  return CPL_ERROR_UNSPECIFIED;
	}
	cpl_vector_set(result,i,cpl_matrix_get(sol,0,0));
	cpl_vector_set(result,i+dimSpatial,cpl_matrix_get(sol,1,0));
	if ( i > dimSpatial/2) {
	  cpl_vector_set(result,2*(dimSpatial/2)-i,cpl_matrix_get(sol,0,0));
	  cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,
			 -1.*cpl_matrix_get(sol,1,0));
	}
	cpl_matrix_delete(sol);
      } else {
	cpt=0;
	a=0.;
	b=0.;
	for(j=0;j<maxstep;j++) {
	  if (filledCycle[j]) {
	    if (nbdet==2) 
	      opd=factorN[3]*(gain[3]*cpl_vector_get(localOpd,3*maxstep+j)-gain[1]*cpl_vector_get(localOpd,1*maxstep+j)+offset[3]-offset[1]);
	    if (nbdet==1)
	      opd=factorL[3]*(gain[3]*cpl_vector_get(localOpd,3*maxstep+j)-gain[1]*cpl_vector_get(localOpd,1*maxstep+j)+offset[3]-offset[1]);
	    b+=cpl_vector_get(imag,j*dimSpatial+i)*cos(2*CPL_MATH_PI*opd/lambda)-
	      cpl_vector_get(real,j*dimSpatial+i)*sin(2*CPL_MATH_PI*opd/lambda);
	    a+=cpl_vector_get(imag,j*dimSpatial+i)*sin(2*CPL_MATH_PI*opd/lambda)+
	      cpl_vector_get(real,j*dimSpatial+i)*cos(2*CPL_MATH_PI*opd/lambda);
	    cpt++;
	  }
	}
	a/=cpt;
	b/=cpt;
	cpl_vector_set(result,i,a);
	cpl_vector_set(result,i+dimSpatial,b);
	if ( i > dimSpatial/2) {
	  cpl_vector_set(result,2*(dimSpatial/2)-i,a);
	  cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*b);
	}
      }

      // 5th fringe peak
    } else if ( fabs(i-posFringePeak[4]) <= 1.6*dOverLambda[4] ) {
      if (option == 0 ) {
	cpt=0;
	for(j=0;j<maxstep;j++) {
	  if (filledCycle[j]) {
	    if (nbdet==2)
	      opd=factorN[4]*(gain[2]*cpl_vector_get(localOpd,2*maxstep+j)-gain[0]*cpl_vector_get(localOpd,0*maxstep+j)+offset[2]-offset[0]);
	    if (nbdet==1)
	      opd=factorL[4]*(gain[2]*cpl_vector_get(localOpd,2*maxstep+j)-gain[0]*cpl_vector_get(localOpd,0*maxstep+j)+offset[2]-offset[0]);
	    cpl_matrix_set(rhs,cpt,0,
			   cpl_vector_get(real,j*dimSpatial+i));
	    cpl_matrix_set(rhs,cpt+nbStep,0,
			   cpl_vector_get(imag,j*dimSpatial+i));
	    cpl_matrix_set(coeff,cpt,0,cos(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt,1,sin(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt,2,1.);
	    cpl_matrix_set(coeff,cpt,3,0.);
	    cpl_matrix_set(coeff,cpt+nbStep,0,-1.*sin(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt+nbStep,1,cos(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt+nbStep,2,0.);
	    cpl_matrix_set(coeff,cpt+nbStep,3,1.);
	    cpt++;
	  }
	}
	sol=cpl_matrix_solve_normal(coeff,rhs);
	if (sol == NULL) {
	  cpl_msg_error(cpl_func,
			"could not solve normal system (5th fringe peak)");
	  return CPL_ERROR_UNSPECIFIED;
	}
	cpl_vector_set(result,i,cpl_matrix_get(sol,0,0));
	cpl_vector_set(result,i+dimSpatial,cpl_matrix_get(sol,1,0));
	if ( i > dimSpatial/2) {
	  cpl_vector_set(result,2*(dimSpatial/2)-i,cpl_matrix_get(sol,0,0));
	  cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,
			 -1.*cpl_matrix_get(sol,1,0));
	}
	cpl_matrix_delete(sol);
      } else {
	cpt=0;
	a=0.;
	b=0.;
	for(j=0;j<maxstep;j++) {
	  if (filledCycle[j]) {
	    if (nbdet==2)
	      opd=factorN[4]*(gain[2]*cpl_vector_get(localOpd,2*maxstep+j)-gain[0]*cpl_vector_get(localOpd,0*maxstep+j)+offset[2]-offset[0]);
	    if (nbdet==1)
	      opd=factorL[4]*(gain[2]*cpl_vector_get(localOpd,2*maxstep+j)-gain[0]*cpl_vector_get(localOpd,0*maxstep+j)+offset[2]-offset[0]);
	    b+=cpl_vector_get(imag,j*dimSpatial+i)*cos(2*CPL_MATH_PI*opd/lambda)-
	      cpl_vector_get(real,j*dimSpatial+i)*sin(2*CPL_MATH_PI*opd/lambda);
	    a+=cpl_vector_get(imag,j*dimSpatial+i)*sin(2*CPL_MATH_PI*opd/lambda)+
	      cpl_vector_get(real,j*dimSpatial+i)*cos(2*CPL_MATH_PI*opd/lambda);
	    cpt++;
	  }
	}
	a/=cpt;
	b/=cpt;
	cpl_vector_set(result,i,a);
	cpl_vector_set(result,i+dimSpatial,b);
	if ( i > dimSpatial/2) {
	  cpl_vector_set(result,2*(dimSpatial/2)-i,a);
	  cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*b);
	}
      }

      // 6th fringe peak
    } else if ( fabs(i-posFringePeak[5]) <= 1.6*dOverLambda[5] || (i > posFringePeak[5]+1.6*dOverLambda[5]) ) {
      if (option == 0 ) {
	cpt=0;
	for(j=0;j<maxstep;j++) {
	  if (filledCycle[j]) {
	    if (nbdet==2)
	      opd=factorN[5]*(gain[3]*cpl_vector_get(localOpd,3*maxstep+j)-gain[0]*cpl_vector_get(localOpd,0*maxstep+j)+offset[3]-offset[0]);
	    if (nbdet==1)
	      opd=factorL[5]*(gain[3]*cpl_vector_get(localOpd,3*maxstep+j)-gain[0]*cpl_vector_get(localOpd,0*maxstep+j)+offset[3]-offset[0]);
	    cpl_matrix_set(rhs,cpt,0,cpl_vector_get(real,j*dimSpatial+i));
	    cpl_matrix_set(rhs,cpt+nbStep,0,cpl_vector_get(imag,j*dimSpatial+i));
	    cpl_matrix_set(coeff,cpt,0,cos(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt,1,sin(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt,2,1.);
	    cpl_matrix_set(coeff,cpt,3,0.);
	    cpl_matrix_set(coeff,cpt+nbStep,0,-1.*sin(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt+nbStep,1,cos(2*CPL_MATH_PI*opd/lambda));
	    cpl_matrix_set(coeff,cpt+nbStep,2,0.);
	    cpl_matrix_set(coeff,cpt+nbStep,3,1.);
	    cpt++;
	  }
	}
	sol=cpl_matrix_solve_normal(coeff,rhs);
	if (sol == NULL) {
	  cpl_msg_error(cpl_func,"could not solve normal system (6th fringe peak)");
	  return CPL_ERROR_UNSPECIFIED;
	}
	cpl_vector_set(result,i,cpl_matrix_get(sol,0,0));
	cpl_vector_set(result,i+dimSpatial,cpl_matrix_get(sol,1,0));
	if ( i > dimSpatial/2) {
	  cpl_vector_set(result,2*(dimSpatial/2)-i,cpl_matrix_get(sol,0,0));
	  cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*cpl_matrix_get(sol,1,0));
	}
	cpl_matrix_delete(sol);
      } else {
	cpt=0;
	a=0.;
	b=0.;
	for(j=0;j<maxstep;j++) {
	  if (filledCycle[j]) {
	    if (nbdet==2)
	      opd=factorN[5]*(gain[3]*cpl_vector_get(localOpd,3*maxstep+j)-gain[0]*cpl_vector_get(localOpd,0*maxstep+j)+offset[3]-offset[0]);
	    if (nbdet==1)
	      opd=factorL[5]*(gain[3]*cpl_vector_get(localOpd,3*maxstep+j)-gain[0]*cpl_vector_get(localOpd,0*maxstep+j)+offset[3]-offset[0]);
	    b+=cpl_vector_get(imag,j*dimSpatial+i)*cos(2*CPL_MATH_PI*opd/lambda)-
	      cpl_vector_get(real,j*dimSpatial+i)*sin(2*CPL_MATH_PI*opd/lambda);
	    a+=cpl_vector_get(imag,j*dimSpatial+i)*sin(2*CPL_MATH_PI*opd/lambda)+
	      cpl_vector_get(real,j*dimSpatial+i)*cos(2*CPL_MATH_PI*opd/lambda);
	    cpt++;
	  }
	}
	a/=cpt;
	b/=cpt;
	cpl_vector_set(result,i,a);
	cpl_vector_set(result,i+dimSpatial,b);
	if ( i > dimSpatial/2) {
	  cpl_vector_set(result,2*(dimSpatial/2)-i,a);
	  cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*b);
	}
      }

    }

    /* else if ( i > posFringePeak[5]+1.5*dOverLambda[5] ) { */
    /*   cpt=0; */
    /*   for(j=0;j<maxstep;j++) { */
    /* 	if (filledCycle[j]) { */
    /* 	  bid=cpl_vector_get(result,i)+ */
    /* 	    cpl_vector_get(real,j*dimSpatial+i); */
    /* 	  cpl_vector_set(result,i,bid); */
    /* 	  bid=cpl_vector_get(result,i+dimSpatial)+ */
    /* 	    cpl_vector_get(imag,j*dimSpatial+i); */
    /* 	  cpl_vector_set(result,i+dimSpatial,bid); */
    /* 	  cpt++; */
    /* 	} */
    /*   } */
    /*   bid=cpl_vector_get(result,i)/cpt; */
    /*   cpl_vector_set(result,i,bid); */
    /*   bid=cpl_vector_get(result,i+dimSpatial)/cpt; */
    /*   cpl_vector_set(result,i+dimSpatial,bid); */
    /*   if ( i > dimSpatial/2) { */
    /* 	cpl_vector_set(result,2*(dimSpatial/2)-i,cpl_vector_get(result,i)); */
    /* 	cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i, */
    /* 		       -1.*cpl_vector_get(result,i+dimSpatial)); */
    /*   } */
    /* } */
    cpl_matrix_delete(rhs);
    cpl_matrix_delete(coeff);
  }
  
  return 0;
}






/*-----------------------------------------------------------------------------*/
/**
   @ingroup signal
   @brief Demodulate the fringes peak 
   @param real              real part of the fourirer transform
   @param imag              imaginary part of the fourier transform
   @param localOpd          vector containg 4 values of OPD for each step
   @param lambda            wavelength of the spectral channel
   @param result            cpl_vector containing the real and imaginary part 
   of the fourier transform of one spectral channle 
   corrected from teh residual background
   @param resolution        spectral resolution
   @param nbdet             Detector number (1:HAWAII, 2:AQUARIUS)
   @param flagUp            =1 OPD ramp UP, =0 OPD ramp down
   @param flagBcdIn         =1 BCD IN =0 BCD OUT
   @return 1 in case of error
*/
/*-----------------------------------------------------------------------------*/
int mat_apply_demod(cpl_vector *real,cpl_vector *imag,double *localOpd,
		    double lambda,cpl_vector *result,int resolution,int nbdet,
                    int flagUp, int flagBcdIN) {

  int i=0;
  int dimSpatial=0;
  double opd=0.;
  double a=0.;
  double b=0.;
  double posFringePeak[6];
  double dOverLambda[6];
  double offset[4];
  double gain[4];
  int iPeak=0;

  // Check input parameters
  mat_assert_value((real!=NULL),CPL_ERROR_NULL_INPUT, 1,"no cpl_vector (real) argument given");
  mat_assert_value((imag!=NULL),CPL_ERROR_NULL_INPUT, 1,"no cpl_vector (imag) argument given");
  mat_assert_value((localOpd!=NULL),CPL_ERROR_NULL_INPUT, 1,"no cpl_vector (localOpd) argument given");
  mat_assert_value((lambda>0),CPL_ERROR_NULL_INPUT, 1,"lambda should be greater than 0");

  dimSpatial=cpl_vector_get_size(real);
  
  for(iPeak=0;iPeak<6;iPeak++) {
    posFringePeak[iPeak]=mat_get_pos_fringe_peak(lambda,nbdet,resolution,iPeak);
    dOverLambda[iPeak]=(posFringePeak[iPeak]-dimSpatial/2)/(3*(iPeak+1));
  }

  for(int iPiezo=0;iPiezo<4;iPiezo++) {
    if (flagUp==1) {
      if (flagBcdIN == 1) {
        if (nbdet == 2) {
          gain[iPiezo]=gainUpInN[iPiezo];
          offset[iPiezo]=offsetInN[iPiezo]/2.;
	} else {
	  gain[iPiezo]=gainUpInL[iPiezo];
          offset[iPiezo]=offsetInL[iPiezo]/2.;
	}
      } else {
        if (nbdet == 2) {
          gain[iPiezo]=gainUpOutN[iPiezo];
          offset[iPiezo]=offsetOutN[iPiezo]/2.;
	} else {
          gain[iPiezo]=gainUpOutL[iPiezo];
          offset[iPiezo]=offsetOutL[iPiezo]/2.;
	}
      }
    } else {
      if (flagBcdIN == 1) {
        if (nbdet == 2) {
          gain[iPiezo]=gainDownInN[iPiezo];
          offset[iPiezo]=-1.*offsetInN[iPiezo]/2.;
  	} else {
          gain[iPiezo]=gainDownInL[iPiezo];
          offset[iPiezo]=-1.*offsetInL[iPiezo]/2.;
  	}
      } else {
        if (nbdet == 2) {
          gain[iPiezo]=gainDownOutN[iPiezo];
          offset[iPiezo]=-1.*offsetOutN[iPiezo]/2.;
  	} else {
          gain[iPiezo]=gainDownOutL[iPiezo];
          offset[iPiezo]=-1.*offsetOutL[iPiezo]/2.;
  	}
      }
    }
  }
  
  for(i=0;i<dimSpatial;i++) {
    // copy frequency outside fringe peaks
    /* if ( (abs(i-dimSpatial/2) <= 1.5*dOverLambda[0]) || (i > posFringePeak[5]+1.5*dOverLambda[5]) ) { */
      cpl_vector_set(result,i,cpl_vector_get(real,i));
      cpl_vector_set(result,i+dimSpatial,cpl_vector_get(imag,i));
    /* } */
    /* if ( (i < dimSpatial-posFringePeak[5]-1.5*dOverLambda[5]) ) { */
    /*   cpl_vector_set(result,i,cpl_vector_get(real,i)); */
    /*   cpl_vector_set(result,i+dimSpatial,cpl_vector_get(imag,i)); */
    /* } */
  }

  for(i=dimSpatial/2;i<dimSpatial;i++) {
  /* for(i=0;i<dimSpatial;i++) { */
    // 1st fringe peak
    if ( fabs(i-posFringePeak[0]) <= 1.5*dOverLambda[0] ) {
    /* if ( (abs(i-dimSpatial/2) > 1.5*dOverLambda && abs(i-dimSpatial/2) <= 4.5*dOverLambda) ) { */
      if (nbdet==2)
	opd=factorN[0]*(gain[3]*localOpd[3]-gain[2]*localOpd[2]+offset[3]-offset[2]);
      if (nbdet==1)
	opd=factorL[0]*(gain[3]*localOpd[3]-gain[2]*localOpd[2]+offset[3]-offset[2]);
      if (i < dimSpatial/2) {
	opd*=(-1.);
      }
      b=cpl_vector_get(imag,i)*cos(2*CPL_MATH_PI*opd/lambda)-
	cpl_vector_get(real,i)*sin(2*CPL_MATH_PI*opd/lambda);
      a=cpl_vector_get(imag,i)*sin(2*CPL_MATH_PI*opd/lambda)+
	cpl_vector_get(real,i)*cos(2*CPL_MATH_PI*opd/lambda);


      /* a=cpl_vector_get(real,i)*cos(dephasage+2*CPL_MATH_PI*opd/lambda)+ */
      /*   cpl_vector_get(imag,i)*sin(dephasage+2*CPL_MATH_PI*opd/lambda); */
      /* b=cpl_vector_get(imag,i)*cos(dephasage+2*CPL_MATH_PI*opd/lambda)- */
      /*   cpl_vector_get(real,i)*sin(dephasage+2*CPL_MATH_PI*opd/lambda); */
      cpl_vector_set(result,i,a);
      cpl_vector_set(result,i+dimSpatial,b);
      cpl_vector_set(result,2*(dimSpatial/2)-i,a);
      cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*b);
      // 2nd fringe peak
    } else if ( fabs(i-posFringePeak[1]) <= 1.5*dOverLambda[1] ) {
    /* } else if ( (abs(i-dimSpatial/2) > 4.5*dOverLambda && abs(i-dimSpatial/2) <= 7.5*dOverLambda) ) { */
      if (nbdet==2)
	opd=factorN[1]*(gain[1]*localOpd[1]-gain[0]*localOpd[0]+offset[1]-offset[0]);
      if (nbdet==1)
	opd=factorL[1]*(gain[1]*localOpd[1]-gain[0]*localOpd[0]+offset[1]-offset[0]);
      if (i < dimSpatial/2) {
	opd*=(-1.);
      }
      b=cpl_vector_get(imag,i)*cos(2*CPL_MATH_PI*opd/lambda)-
	cpl_vector_get(real,i)*sin(2*CPL_MATH_PI*opd/lambda);
      a=cpl_vector_get(imag,i)*sin(2*CPL_MATH_PI*opd/lambda)+
	cpl_vector_get(real,i)*cos(2*CPL_MATH_PI*opd/lambda);
      cpl_vector_set(result,i,a);
      cpl_vector_set(result,i+dimSpatial,b);
      // 3rd fringe peak
      cpl_vector_set(result,2*(dimSpatial/2)-i,a);
      cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*b);
    } else if ( fabs(i-posFringePeak[2]) <= 1.5*dOverLambda[2] ) {
    /* } else if ( (abs(i-dimSpatial/2) > 7.5*dOverLambda &&  */
    /* 		 abs(i-dimSpatial/2) <= 10.5*dOverLambda) ) { */
      if (nbdet==2)
	opd=factorN[2]*(gain[2]*localOpd[2]-gain[1]*localOpd[1]+offset[2]-offset[1]);
      if (nbdet==1)
	opd=factorL[2]*(gain[2]*localOpd[2]-gain[1]*localOpd[1]+offset[2]-offset[1]);
      if (i < dimSpatial/2) {
	opd*=(-1.);
      }
      b=cpl_vector_get(imag,i)*cos(2*CPL_MATH_PI*opd/lambda)-
	cpl_vector_get(real,i)*sin(2*CPL_MATH_PI*opd/lambda);
      a=cpl_vector_get(imag,i)*sin(2*CPL_MATH_PI*opd/lambda)+
	cpl_vector_get(real,i)*cos(2*CPL_MATH_PI*opd/lambda);
      cpl_vector_set(result,i,a);
      cpl_vector_set(result,i+dimSpatial,b);
      cpl_vector_set(result,2*(dimSpatial/2)-i,a);
      cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*b);
      // 4th fringe peak
    } else if ( fabs(i-posFringePeak[3]) <= 1.5*dOverLambda[3] ) {
    /* } else if ( (abs(i-dimSpatial/2) > 10.5*dOverLambda &&  */
    /* 		 abs(i-dimSpatial/2) <= 13.5*dOverLambda) ) { */
      if (nbdet==2)
	opd=factorN[3]*(gain[3]*localOpd[3]-gain[1]*localOpd[1]+offset[3]-offset[1]);
      if (nbdet==1)
	opd=factorL[3]*(gain[3]*localOpd[3]-gain[1]*localOpd[1]+offset[3]-offset[1]);
      if (i < dimSpatial/2) {
	opd*=(-1.);
      }
      b=cpl_vector_get(imag,i)*cos(2*CPL_MATH_PI*opd/lambda)-
	cpl_vector_get(real,i)*sin(2*CPL_MATH_PI*opd/lambda);
      a=cpl_vector_get(imag,i)*sin(2*CPL_MATH_PI*opd/lambda)+
	cpl_vector_get(real,i)*cos(2*CPL_MATH_PI*opd/lambda);
      cpl_vector_set(result,i,a);
      cpl_vector_set(result,i+dimSpatial,b);
      cpl_vector_set(result,2*(dimSpatial/2)-i,a);
      cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*b);
      // 5th fringe peak
    } else if ( fabs(i-posFringePeak[4]) <= 1.5*dOverLambda[4] ) {
    /* } else if ( (abs(i-dimSpatial/2) > 13.5*dOverLambda &&  */
    /* 		 abs(i-dimSpatial/2) <= 16.5*dOverLambda) ) { */
      if (nbdet==2)
	opd=factorN[4]*(gain[2]*localOpd[2]-gain[0]*localOpd[0]+offset[2]-offset[0]);
      if (nbdet==1)
	opd=factorL[4]*(gain[2]*localOpd[2]-gain[0]*localOpd[0]+offset[2]-offset[0]);
      if (i < dimSpatial/2) {
	opd*=(-1.);
      }
      b=cpl_vector_get(imag,i)*cos(2*CPL_MATH_PI*opd/lambda)-
	cpl_vector_get(real,i)*sin(2*CPL_MATH_PI*opd/lambda);
      a=cpl_vector_get(imag,i)*sin(2*CPL_MATH_PI*opd/lambda)+
	cpl_vector_get(real,i)*cos(2*CPL_MATH_PI*opd/lambda);
      cpl_vector_set(result,i,a);
      cpl_vector_set(result,i+dimSpatial,b);
      cpl_vector_set(result,2*(dimSpatial/2)-i,a);
      cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*b);
      // 6th fringe peak
    } else if ( fabs(i-posFringePeak[5]) <= 1.5*dOverLambda[5] ) {
    /* } else if ( (abs(i-dimSpatial/2) > 16.5*dOverLambda &&  */
    /* 		 abs(i-dimSpatial/2) <= 19.5*dOverLambda) ) { */
      if (nbdet==2)
	opd=factorN[5]*(gain[3]*localOpd[3]-gain[0]*localOpd[0]+offset[3]-offset[0]);
      if (nbdet==1)
	opd=factorL[5]*(gain[3]*localOpd[3]-gain[0]*localOpd[0]+offset[3]-offset[0]);
      if (i < dimSpatial/2) {
	opd*=(-1.);
      }
      b=cpl_vector_get(imag,i)*cos(2*CPL_MATH_PI*opd/lambda)-
	cpl_vector_get(real,i)*sin(2*CPL_MATH_PI*opd/lambda);
      a=cpl_vector_get(imag,i)*sin(2*CPL_MATH_PI*opd/lambda)+
	cpl_vector_get(real,i)*cos(2*CPL_MATH_PI*opd/lambda);
      cpl_vector_set(result,i,a);
      cpl_vector_set(result,i+dimSpatial,b);
      cpl_vector_set(result,2*(dimSpatial/2)-i,a);
      cpl_vector_set(result,dimSpatial+2*(dimSpatial/2)-i,-1.*b);
    }
  }

  return 0;
}
