/* $Id: mat_apply_staticcalib.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/03/28 11:03:00 $
 * $Revision: 0.5 $
 * $Name: mat_apply_staticcalib.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "mat_drl.h"
#include "mat_const.h"
#include "mat_apply_staticcalib.h"
#include "mat_error.h"
#include "mat_detector.h"
#include "mat_flatfield.h"
#include "mat_badpixel.h"
#include "mat_nonlinearity.h"
#include "mat_obsflat.h"
#include "mat_statistics.h"
#include "mat_utils.h"

/*-----------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/
#define UNUSED(x) (void)(x)

#if defined (_OPENMP) && !defined( __APPLE__)
#define PARALLEL 1 // needed to activate special memory management code for parallel version (CPL memory management is _not_ really thread safe)
#include "omp.h"
#endif

#include <sys/time.h>

//#define WITH_CORRELATION 1
#define MAT_SKY_TARGET_INT_RANGE 2.5

typedef struct {
  int           raw_idx; /*!< index of this region in the original raw data */
  int           raw_col; /*!< the column of the sub-window, starting at 0 */
  int           raw_row; /*!< the row of the sub-window, starting at 0 */
  int           cal_idx; /*!< index of this region in the calibrated data or -1 */
  mat_region   *region;  /*!< the pointer to the original region data structure (raw data) */
} mat_calreg;

typedef struct {
  mat_detector          det;                  /*!< this the detector specification from the raw data */
  double                exptime;              /*!< EXPTIME from the raw data (needed for the Aquarius nonlinearity compensation) */
  mat_imagingdetector  *raw_imgdet;           /*!< describes the detector configuration of the raw data */
  int                   raw_ncols;            /*!< number of sub-window columns in the raw data */
  int                   raw_nrows;            /*!< number of sub-window rows in the raw data */
  mat_imagingdetector  *cal_imgdet;           /*!< describes the detector configuration of the calibrated data */
  int                   compensation;         /*!< describes which compensation is requested and possible */
  mat_calreg           *list_region;          /*!< contains the ordered list of regions */
  int                  *map_region;           /*!< the mapping of the calibrated sub window index to the index into list_region */
  mat_badpixel         *bpm;                  /*!< the bad pixel map mapped to the raw data, created */
  mat_flatfield        *ffm;                  /*!< the flatfield map mapped to the raw data, created */
  mat_nonlinearity     *nlm;                  /*!< the nonlinearity map mapped to the raw data, created */
  mat_obsflat          *ofm;                  /*!< the observation specific flatfield, compatible to the raw data */
  mat_frame            *pbm;                  /*!< the observation specific pixel bias map, compatible to the raw data */
  mat_frame            *content;              /*!< contains for each reference pixel the detector channel number */
  int                  *list_hasref;          /*!< contains for each sub-window a flag if it contains reference pixels */
  int                  *list_hassci;          /*!< contains for each sub-window a flag if it contains normal pixels */
  double                rclimit;              /*!< intensity limit [DU] up to which the nonlinearity can be compensated for */
  double               *rcmap;                /*!< mapping raw -> cal (nonlinearity for Aquarius detector only) */
  int                   nbthreads;            /*!< Number of entries in the following lists (equal with the number of threads used to compensate all frames) */
  mat_frame           **frame_list;           /*!< List of pointers to mat_frame needed to compensate detector effects (OpenMP compliant) */
  mat_statistics_info **stat_list;            /*!< List of pointers to mat_statistics_infoneeded to compensate detector effects (OpenMP compliant) */
  double              **list_dc_list;         /*!< List of pointers to sum/mean intensity (bias) for each detector channel OpenMP compliant) */
} mat_calibration_info;

typedef struct {
  mat_detector_type     type;                 /*!< detector type as an enumeration (MAT_*_DET) */
  int                   nx;                   /*!< total detector width in pixel */
  int                   ny;                   /*!< total detector height in pixel */
  int                   expno;                /*!< Value of ESO TPL EXPNO */
  double                obsstart;             /*!< Start of the observation (MJD-OBS, only the fractional part [s]) */
  double                timstart;             /*!< Start of the detector specific TIM-Board trigger (ESO INS TIMDL/N START, only the fractional part [s]) */
  double                timperiod;            /*!< Trigger period (ESO INS TIMDL/N PERIOD [s]) */
  double                framestart;           /*!< Time of the first frame (only the fractional part [s]) */
  double                chopfreq;             /*!< Frequency of the chopper (ESO ISS CHOP FREQ, in [Hz]) */
  double                chopperiod;           /*!< Period length of the chopper [s] */
  char                  sdch;                 /*!< TARTYP for a sky 'S' or dark 'D' frame, used to create the raw/estimated TARTYP */
  char                  tbch;                 /*!< TARTYP for a target 'T' od bright 'B' frame, used to create the raw/estimated TARTYP */
  int                   nbsky;                /*!< Largest number of consecutive sky frames including dead frames */
  int                   nbtarget;             /*!< Largest number of consecutive target frames including dead frames */
  int                   nbdead;               /*!< Number of dead frames */
  int                   stepphase_min;        /*!< Smallest stepping phase in the raw data */
  int                   stepphase_max;        /*!< Largest stepping phase in the raw data */
  int                   nstepphases;          /*!< Number of different stepping phases */
  int                   tot_target;           /*!< Number of target frames needed to have an even number of stepping phases in the full timetable */
  int                   tot_chop;             /*!< Number of chopping cycles in the timetable */
  int                   tot_entries;          /*!< Number of entries/frames in the timetable */
  int                   raw_first_index;      /*!< Frame index for the first frame in the raw data (framestart-timstart)/timperiod */
  int                   est_time_shift;       /*!< Estimated shift from timing data */
} mat_exposure_setup_info;

typedef struct {
  int                   nbentries;            /*!< Number of frames in the rad data file after leading it (excluding skipped frames) */
  double                intensity_offset;     /*!< Smallest intensity, subtracted from all frame intensities (smaller numbers in logs) */
  double               *intensity;            /*!< Total intensity in the central 1/4 of the central region for each frame [DU] */
  double               *correlation;          /*!< Correlation of a frame with the next frame in the central 1/4 of the central region */
  double               *V;                    /*!< Calculated triangle (Allouche + Lagarde) */
  int                   cr_idx;               /*!< index of the central region (normally the interferometric channel) */
  int                   cr_nx;                /*!< Width of the central region (full width) */
  int                   cr_ny;                /*!< Height of the central region (full height) */
  int                   roi_left;             /*!< Left side of the central 1/4 of the central region (local column) */
  int                   roi_right;            /*!< Right side of the central 1/4 of the central region (inclusive) */
  int                   roi_bottom;           /*!< Bottom side of the central 1/4 of the central region (local row) */
  int                   roi_top;              /*!< Top side of the central 1/4 of the central region (inclusive) */
} mat_raw_data_info;

typedef struct {
  int                   nbentries;            /*!< Number of frames in one chopping cycle */
  // pattern of the first chopping cycle in the timetable (SSSUTTTU or USSSUTTT)
  char                 *raw_tartyp;           /*!< Raw TARTYP for each frame in one chopping cycle */
  char                 *opt_tartyp;           /*!< Optimal TARTYP including replacing (optional) U by S or T. This TARTYP is later used as a replacement. */
  int                  *stepping_phase;       /*!< STEPPING_PHASE for each frame of the first chopping cycle in a timetable */
  int                   first;                /*!< First raw frame which starts a full chopping cycle */
  int                   cycles;               /*!< Number of full chopping cycles in the rad data */
  int                   phase;                /*!< Phase between raw data and chopping cycle: entry := (frame index + phase) MOD nbentries (0-based) */
  double               *intensity;            /*!< Average total intensity in the central 1/4 of the central region [DU] */
  double               *relint;               /*!< Relative intensity */
  double               *V;                    /*!< Calculated triangle (Allouche + Lagarde) */
  double               *S;
  double               *skyS;
  double               *targetS;
  int                   est_V_shift;          /*!< Estimated shift between raw TARTYP and TARTYP calculated from V */
  int                   est_S_shift;          /*!< Frame shift derived from fitting a square wave to the intensity */
  char                 *est_tartyp;           /*!< Estimated TARTYP based on a shifted opt_tartyp. */
} mat_chopping_cycle_info;

typedef struct {
  int                   nbentries;            /*!< Number of frames in one chopping cycle */
  // pattern of the first chopping cycle in the timetable (SSSUTTTU or USSSUTTT)
  char                 *raw_tartyp;           /*!< Raw TARTYP for each frame in one chopping cycle */
  int                   cycles;               /*!< Number of full chopping cycles in the rad data */
  double               *intensity;            /*!< Average total intensity in the central 1/4 of the central region [DU] */
  double               *V;                    /*!< Calculated triangle (Allouche + Lagarde) */
  int                   est_V_shift;          /*!< Estimated shift between raw TARTYP and TARTYP calculated from V */
  char                 *est_tartyp;           /*!< Estimated TARTYP based on a shifted opt_tartyp. */
} mat_allouche_info;

typedef struct {
  int                   nbentries;            /*!< contains the number of timetable entries (extracted from TARTYP) */
  char                 *raw_tartyp;           /*!< the TARTYP flags for the reconstructed timetable (starts with S or D) */
  char                 *opt_tartyp;           /*!< Optimal TARTYP including replacing (optional) U by S or T. This TARTYP is later used as a replacement. */
  double               *localopd1;            /*!< LOCALOPD[1] */
  double               *localopd2;            /*!< LOCALOPD[2] */
  double               *localopd3;            /*!< LOCALOPD[3] */
  double               *localopd4;            /*!< LOCALOPD[4] */
  int                  *stepping_phase;       /*!< STEPPING_PHASE */
  int                   first;                /*!< The raw frame index for the first frame of the first chopping cycle of the first timetable */
  int                   second;               /*!< The raw frame index for the first frame of the first chopping cycle of the second timetable */
  int                   cycles;               /*!< Number of complete timetable sequences in the raw data */
  int                   phase;                /*!< timetable index equals to (frame number[0 based] + phase) modulo nbentries */
  double               *intensity;            /*!< median intensity from the center of the interferrometric channel of all frames with the same timetable index */
  double               *V;                    /*!< Calculated triangle (Allouche + Lagarde) */
#ifdef WITH_CORRELATION
  double               *correlation;          /*!< for each timetable entry a median image is calculated and the correlation between two successive median images calculated */
#endif
  int                   est_V_shift;          /*!< shift between TARTYP(file) and TARTYP(intensity) */
  char                 *est_tartyp;           /*!< Estimated TARTYP based on a shifted opt_tartyp in chopping cycle */
} mat_timetable_info;

typedef struct {
  mat_statistics_info      *stat_info;        /*!< contains the data structure for statistic calculations ignoring outliers */
  mat_exposure_setup_info   es;
  mat_raw_data_info         rd;
  mat_chopping_cycle_info   cc;
  mat_allouche_info         al;
  mat_timetable_info        tt;
} mat_tartyp_info;

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/
static void mat_calibration_info_init(mat_calibration_info *info)
{
  /* initialize the info data structure */
  info->det.type = MAT_UNKNOWN_DET;
  info->raw_imgdet = NULL;
  info->cal_imgdet = NULL;
  info->compensation = NO_COMPENSATION;
  info->list_region = NULL;
  info->map_region = NULL;
  info->bpm = NULL;
  info->ffm = NULL;
  info->nlm = NULL;
  info->ofm = NULL;
  info->pbm = NULL;
  info->content = NULL;
  info->list_hasref = NULL;
  info->list_hassci = NULL;
  info->rclimit  = 65556.0;
  info->rcmap    = NULL;
  info->nbthreads = 0;
  info->frame_list = NULL;
  info->stat_list = NULL;
  info->list_dc_list = NULL;
}

static void mat_calibration_info_delete(mat_calibration_info *info)
{
  int i;
  
  /* delete the info data structure */
  if (info->raw_imgdet != NULL)
    {
      mat_imagingdetector_delete(info->raw_imgdet);
      info->raw_imgdet = NULL;
    }
  if (info->cal_imgdet != NULL)
    {
      mat_imagingdetector_delete(info->cal_imgdet);
      info->cal_imgdet = NULL;
    }
  mat_free(info->list_region);
  mat_free(info->map_region);
  if (info->bpm != NULL)
    {
      mat_badpixel_delete(info->bpm);
      info->bpm = NULL;
    }
  if (info->ffm != NULL)
    {
      mat_flatfield_delete(info->ffm);
      info->ffm = NULL;
    }
  if (info->nlm != NULL)
    {
      mat_nonlinearity_delete(info->nlm);
      info->nlm = NULL;
    }
  if (info->ofm != NULL)
    {
      mat_obsflat_delete(info->ofm);
      info->ofm = NULL;
    }
  if (info->pbm != NULL)
    {
      info->pbm = NULL;
    }
  if (info->content != NULL)
    {
      mat_frame_delete(info->content);
      info->content = NULL;
    }
  mat_free(info->list_hasref);
  mat_free(info->list_hassci);
  mat_free(info->rcmap);
  if (info->nbthreads != 0)
    {
      for (i = 0; i < info->nbthreads; i++)
	{
	  if (info->frame_list[i] != NULL)
	    {
	      mat_frame_delete(info->frame_list[i]);
	      info->frame_list[i] = NULL;
	    }
	  if (info->stat_list[i] != NULL)
	    {
	      mat_statistics_delete(info->stat_list[i]);
	      info->stat_list[i] = NULL;
	    }
	  mat_free(info->list_dc_list[i]);
	}
      mat_free(info->frame_list);
      mat_free(info->stat_list);
      mat_free(info->list_dc_list);
      info->nbthreads = 0;
    }
}

static cpl_error_code mat_calibration_alloc_workspace(mat_calibration_info *info, int nbthreads)
{
  int i;

  cpl_msg_info(cpl_func, "allocating workspace (frame, statistics and dc_bias) for %d threads", nbthreads);
  if (info->nbthreads != 0)
    {
      cpl_msg_error(cpl_func, "mat_calibration_alloc_workspace() was called twice!");
    }

  info->nbthreads = nbthreads;

  info->frame_list = (mat_frame**)cpl_calloc(nbthreads, sizeof(mat_frame*));
  if (info->frame_list == NULL) return CPL_ERROR_UNSPECIFIED;

  info->stat_list  = (mat_statistics_info**)cpl_calloc(nbthreads, sizeof(mat_statistics_info*));
  if (info->stat_list == NULL) return CPL_ERROR_UNSPECIFIED;

  info->list_dc_list = (double**)cpl_calloc(nbthreads, sizeof(double*));
  if (info->list_dc_list == NULL) return CPL_ERROR_UNSPECIFIED;

  for (i = 0; i < nbthreads; i++)
    {
      info->frame_list[i] = mat_frame_new(info->raw_imgdet, CPL_TYPE_DOUBLE);
      if (info->frame_list[i] == NULL)
	{
	  cpl_msg_error(cpl_func,"could not create temporary frame info->frame_list[%d]", i);
	  return CPL_ERROR_UNSPECIFIED;
	}
      info->stat_list[i] = mat_statistics_new(info->det.nx*info->det.ny, 1.5); // enough space for all pixels, ignoring outliers
      if (info->stat_list[i] == NULL)
	{
	  cpl_msg_error(cpl_func,"could not create mat_statistics_info info->stat_list[%d]", i);
	  return CPL_ERROR_UNSPECIFIED;
	}
      info->list_dc_list[i] = (double *)cpl_calloc(info->det.channel_nrows*info->det.channel_ncolumns, sizeof(double));
      if (info->list_dc_list[i] == NULL)
	{
	  cpl_msg_error(cpl_func,"could not create double vector info->list_dc_list[%d]", i);
	  return CPL_ERROR_UNSPECIFIED;
	}
    }
  return CPL_ERROR_NONE;
}

static int mat_calreg_sort(const void *r1, const void *r2)
{
  mat_region   *rr1 = ((mat_calreg *)r1)->region;
  mat_region   *rr2 = ((mat_calreg *)r2)->region;

  if (rr1->corner[1] < rr2->corner[1]) return -1;
  if (rr1->corner[1] > rr2->corner[1]) return +1;
  if (rr1->corner[0] < rr2->corner[0]) return -1;
  if (rr1->corner[0] > rr2->corner[0]) return +1;
  return 0; // should not happen!
}

/**
   @brief Derives the calibration information from a raw data file
   @param info    Contains the calibration information and temporary data storage.
   @param det     Contains the detector description
   @param imgdet  Contains the sub-window setup.
   @returns an error code

   This function uses a detector specification (det) and a sub-window setup (imgdet), to determine
   all information needed to calibrate raw rames. This include sorting the sub-windows in the raw data,
   deriving the sub-window column and row structure. In addition all temporary memory is allocated which
   is needed during the calibration process.
*/
static cpl_error_code mat_calibration_info_set_setup(mat_calibration_info *info,
						     mat_detector *det,
						     mat_imagingdetector *imgdet)
{
  int  r;
  int  x, y;
  int  col, row;
  int  count;

  info->det = *det;
  info->raw_imgdet = mat_imagingdetector_duplicate(imgdet);
  cpl_msg_info(cpl_func, "mat_calibration_info_set_raw(...): mat_imagingdetector duplicated");
  /* build a vector of region information and sort it in reading order (1. criteria vertical, 2. criteria horizontal */
  info->list_region = (mat_calreg *)cpl_calloc(info->raw_imgdet->nbregion, sizeof(mat_calreg));
  if (info->list_region == NULL)
    {
      cpl_msg_error(cpl_func,"could not create mat_calreg vector (info->list_region)");
      return CPL_ERROR_UNSPECIFIED;
    }
  info->map_region = (int *)cpl_calloc(info->raw_imgdet->nbregion, sizeof(int));
  if (info->map_region == NULL)
    {
      cpl_msg_error(cpl_func,"could not create int vector (info->map_region)");
      return CPL_ERROR_UNSPECIFIED;
    }
  cpl_msg_info(cpl_func, "mat_calibration_info_set_raw(...): list_region and map_region allocated");
  for (r = 0; r < info->raw_imgdet->nbregion; r++)
    {
      info->list_region[r].raw_idx = r;
      info->list_region[r].cal_idx = -1;
      info->list_region[r].region = info->raw_imgdet->list_region[r];
      info->map_region[r] = -1;
    }
  qsort(info->list_region, info->raw_imgdet->nbregion, sizeof(mat_calreg), mat_calreg_sort);
  cpl_msg_info(cpl_func, "mat_calibration_info_set_raw(...): list_region sorted");
  /* determine the sub-window row and column for each sub-window in the raw data */
  info->raw_ncols = 0;
  info->raw_nrows = 0;
  x = 0;
  col = 0;
  row = 0;
  for (r = 0; r < info->raw_imgdet->nbregion; r++)
    {
      if (info->list_region[r].region->corner[0] < x)
	{
	  x = 0;
	  col = 0;
	  row++;
	}
      info->list_region[r].raw_col = col;
      info->list_region[r].raw_row = row;
      if (row > info->raw_nrows) info->raw_nrows = row;
      if (col > info->raw_ncols) info->raw_ncols = col;
      col++;
      x = info->list_region[r].region->corner[0];
    }
  cpl_msg_info(cpl_func, "mat_calibration_info_set_raw(...): row and column number assigned to sub-windows (%d, %d)", info->raw_ncols, info->raw_nrows);
  /* info->raw_nrows and info->raw_ncols contains the highest column and row number! */
  info->raw_ncols++;
  info->raw_nrows++;
  /* create a temporary storage for a raw data frame */
  /*
  // we need a temporary storage for the calibration process
  info->frame = mat_frame_new(info->raw_imgdet, CPL_TYPE_DOUBLE);
  if (info->frame == NULL)
    {
      cpl_msg_error(cpl_func,"could not create temporary frame (info->frame)");
      return CPL_ERROR_UNSPECIFIED;
    }
  */
  // we need a frame which contains for each pixel the detector channel number
  info->content = mat_frame_new(info->raw_imgdet, CPL_TYPE_INT);
  if (info->content == NULL)
    {
      cpl_msg_error(cpl_func,"could not create temporary content (info->content)");
      return CPL_ERROR_UNSPECIFIED;
    }
  // we need a vector containing for each sub-window a flag if this sub-window contains at least one reference pixel
  info->list_hasref = (int *)cpl_calloc(info->raw_imgdet->nbregion, sizeof(int));
  if (info->list_hasref == NULL)
    {
      cpl_msg_error(cpl_func,"could not create int vector (info->list_hasref)");
      return CPL_ERROR_UNSPECIFIED;
    }
  // we need a vector containing for each sub-window a flag if this sub-window contains at least one science pixel
  info->list_hassci = (int *)cpl_calloc(info->raw_imgdet->nbregion, sizeof(int));
  if (info->list_hassci == NULL)
    {
      cpl_msg_error(cpl_func,"could not create int vector (info->list_hassci)");
      return CPL_ERROR_UNSPECIFIED;
    }

  // assign the detector channel to all reference pixels
  count = 0;
  for (r = 0; r < info->raw_imgdet->nbregion; r++)
    {
      cpl_image *sw = info->content->list_subwin[r]->imgreg[0];
      int        nx = cpl_image_get_size_x(sw);
      int        ny = cpl_image_get_size_y(sw);
      int        x0 = info->raw_imgdet->list_region[r]->corner[0];
      int        y0 = info->raw_imgdet->list_region[r]->corner[1];
      count += nx*ny;
      for (y = 1; y <= ny; y++)
	{
	  int dy = y0 + y - 2; // detector y-coordinate (0-based!)
	  for (x = 1; x <= nx; x++)
	    {
	      int dx = x0 + x - 2; // detector x-coordinate (0-based!)
	      // calculate the detector channel number (1-based)
	      int dcc = dx/info->det.channel_nx; // 0-based!
	      int dcr = dy/info->det.channel_ny; // 0-based!
	      int dc  = dcc + dcr*info->det.channel_ncolumns + 1;
	      if (mat_detector_is_reference(&(info->det), dx, dy))
		{ // we have a reference pixel => use the detector channel number directly
		  cpl_image_set(sw, x, y, dc);
		}
	      else
		{ // we are outside of the frame => use the negative detector channel number
		  cpl_image_set(sw, x, y, -dc);
		}
	    }
	}
      info->list_hasref[r] = mat_detector_contains_reference(&(info->det), x0, y0, nx, ny);
      info->list_hassci[r] = mat_detector_contains_nonreference(&(info->det), x0, y0, nx, ny);
    }
  cpl_msg_info(cpl_func, "mat_calibration_info_set_raw(...): temporary storage created");
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_calibration_info_set_badpixel(mat_calibration_info *info, mat_badpixel *bpm)
{
  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (info->raw_imgdet == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_imagingdetector (info->raw_imgdet) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (bpm == NULL)
    { /* we will remove a badpixel map (disabling the bad pixel interpolation) */
      info->bpm = NULL;
      return CPL_ERROR_NONE;
    }
  if (mat_detector_check(&(info->det), bpm->det) != CPL_ERROR_NONE)
    { /* the bad pixel map is not compatible with the raw data -> error and ignore */
      info->bpm = NULL;
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  info->bpm = mat_badpixel_map(bpm, info->raw_imgdet);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_calibration_info_set_flatfield(mat_calibration_info *info, mat_flatfield *ffm)
{
  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (info->raw_imgdet == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_imagingdetector (info->raw_imgdet) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (ffm == NULL)
    { // we will remove a flatfield map (disabling the flatfield compensation)
      info->ffm = NULL;
      return CPL_ERROR_NONE;
    }
  if (mat_detector_check(&(info->det), ffm->det) != CPL_ERROR_NONE)
    { // the flatfield map is not compatible with the raw data -> error and ignore
      info->ffm = NULL;
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  info->ffm = mat_flatfield_map(ffm, info->raw_imgdet);
  return CPL_ERROR_NONE;
}

/*
  For applying the nonlinearity compensation, three use cases do exist:

  1. Each pixel has an individual response (the coefficients for different pixels differ: (has_coeff && (coeff[0] == {1,0,0,0})))

  2. A global response is defined by a law (has_nllaw)

  3. A global response is defined only by a map (has_rcmap)
 */

static void mat_calibration_create_nonlinearity_mapping(mat_calibration_info *info)
{
  if (info->nlm == NULL)
    { // paranoia
      cpl_msg_error(cpl_func, "no nonlinearity map provided");
    }
  else if (info->nlm->nltype == MAT_UNKNOWN_NL)
    { // the NLM contains an unknown method (wrong DO classification?)
      cpl_msg_error(cpl_func, "unknown nonlinearity compensation, wrong DO classification?");
    }
  else if (info->nlm->nltype == MAT_INDIVIDUAL_NL)
    { // individual nonlinearity compensation is used
      cpl_msg_info(cpl_func, "each pixel uses an individual nonlinearity compensation");
    }
  else if (info->nlm->nltype == MAT_GLOBAL_MAP_NL)
    { // global nonlinearity compensation defined as a map (old NLM files contain no explicite global function coefficients)
      int      i, n;
      double   step;
      cpl_msg_info(cpl_func, "global nonlinearity compensation using a map");
      info->rclimit = info->nlm->rclimit;
      info->rcmap = (double*)cpl_calloc(65536, sizeof(double)); // 1 DU resolution
      if (info->rcmap == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the cal -> raw map (%d entries)", 65536);
	  return;
	}
      n = cpl_vector_get_size(info->nlm->rcmap);
      step = round(65536.0/(double)(n - 1));
      for (i = 0; i < (n - 1); i++)
	{
	  double x1 = step*(double)i;
	  double x2 = x1 + step;
	  double y1 = cpl_vector_get(info->nlm->rcmap, i + 0);
	  double y2 = cpl_vector_get(info->nlm->rcmap, i + 1);
	  double m = (y2 - y1)/step;
	  int    x = (int)round(x1);
	  while (((double)x <= x2) && (x < 65536))
	    {
	      info->rcmap[x] = y1 + m*((double)x - x1);
	      x++;
	    }
	}
    }
  else if (info->nlm->nltype == MAT_GLOBAL_LAW_NL)
    { // global nonlinearity compensation defined as a function (4 or 6 coefficients)
      int    i;
      info->rclimit = info->nlm->rclimit;
      info->rcmap = (double*)cpl_calloc(65536, sizeof(double)); // 1 DU resolution
      if (info->rcmap == NULL)
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the cal -> raw map (%d entries)", 65536);
	  return;
	}
      if (info->nlm->detnlncoeff == 4)
	{ // H2: "a*x+b*x*exp((x-c)*d)"
	  double a = info->nlm->detnlcoeff[0];
	  double b = info->nlm->detnlcoeff[1];
	  double c = info->nlm->detnlcoeff[2];
	  double d = info->nlm->detnlcoeff[3];
	  cpl_msg_info(cpl_func, "global nonlinearity compensation using a function f(x) = %g*x + %g*x*exp((x - %g)*%g)", a, b, c, d);
	  for (i = 0; i < 65536; i++)
	    {
	      double x = (double)i/MAT_DU_SCALE;
	      double y = (a*x + b*x*exp((x - c)*d))*MAT_DU_SCALE;
	      info->rcmap[i] = y;
	    }
	}
      else if (info->nlm->detnlncoeff == 6)
	{ // AQ: "a*x+b*(1-exp(-c*x**d))+e*(exp(f*x)-1)"
	  double a = info->nlm->detnlcoeff[0];
	  double b = info->nlm->detnlcoeff[1];
	  double c = info->nlm->detnlcoeff[2];
	  double d = info->nlm->detnlcoeff[3];
	  double e = info->nlm->detnlcoeff[4];
	  double f = info->nlm->detnlcoeff[5];
	  cpl_msg_info(cpl_func, "global nonlinearity compensation using a function f(x) = %g*x + %g*(1.0 - exp(-%g*x**%g)) + %g*(exp(%g*x) - 1.0)", a, b, c, d, e, f);
	  for (i = 0; i < 65536; i++)
	    {
	      double x = (double)i;
	      double y = a*x + b*(1.0 - exp(-c*pow(x, d))) + e*(exp(f*x) - 1.0);
	      info->rcmap[i] = y;
	    }
	}
      else
	{ // unknown: 1:1 mapping
	  cpl_msg_info(cpl_func, "global nonlinearity compensation using a function f(x) = ??? with %d coefficients", info->nlm->detnlncoeff);
	  for (i = 0; i < 65536; i++)
	    {
	      double x = (double)i;
	      info->rcmap[i] = x;
	    }
	}
    }
  cpl_msg_info(cpl_func,"Nonlinearity compensation prepared");
}

static cpl_error_code mat_calibration_info_set_nonlinearity(mat_calibration_info *info, mat_nonlinearity *nlm)
{
  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (info->raw_imgdet == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_imagingdetector (info->raw_imgdet) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (nlm == NULL)
    { /* we will remove a nonlinearity map (disabling the nonlinearity compensation) */
      info->nlm = NULL;
      return CPL_ERROR_NONE;
    }
  if (mat_detector_check(&(info->det), nlm->det) != CPL_ERROR_NONE)
    { /* the nonlinearity map is not compatible with the raw data -> error and ignore */
      info->nlm = NULL;
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  info->nlm = mat_nonlinearity_map(nlm, info->raw_imgdet);
  mat_calibration_create_nonlinearity_mapping(info);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_calibration_info_set_obsflat(mat_calibration_info *info, mat_obsflat *ofm)
{
  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (info->raw_imgdet == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_imagingdetector (info->raw_imgdet) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (ofm == NULL)
    { /* we will remove a obsflat map (disabling the obsflat compensation) */
      info->ofm = NULL;
      return CPL_ERROR_NONE;
    }
  if (mat_detector_check(&(info->det), ofm->det) != CPL_ERROR_NONE)
    { /* the obsflat map is not compatible with the raw data -> error and ignore */
      info->ofm = NULL;
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  info->ofm = mat_obsflat_map(ofm, info->raw_imgdet);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_calibration_info_set_pixelbias(mat_calibration_info *info, mat_frame *pbm)
{
  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (info->raw_imgdet == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_imagingdetector (info->raw_imgdet) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (pbm == NULL)
    { /* we will remove a pixel bias map (disabling the pixel bias compensation) */
      info->pbm = NULL;
      return CPL_ERROR_NONE;
    }
  if (mat_frame_check_layout(pbm, info->raw_imgdet) != CPL_ERROR_NONE)
    { /* the pixel bias map is not compatible with the raw data -> error and ignore */
      info->pbm = NULL;
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  info->pbm = pbm;
  return CPL_ERROR_NONE;
}

/**
   @brief Selects the compensation method depending on a parameter and the available calibration maps.
   @param info          Contains the calibration context.
   @param compensation  A set of flags describing the requested compensation methods.
   @param reduce        A flag describing if the exclusive reference sub-windows should be removed. 
   @param reduce_original  Original flag describing if the exclusive reference sub-windows should be removed. 
   @returns a cpl_error_code

   The used compensation method depends on:

   - the available calibration maps

   - the detector type

   - the requested methods

   The available calibration maps together with the detector type determine which compensation methods are
   in available. The subset with the requested compensation methods give the used methods.
*/
static cpl_error_code mat_calibration_info_select_method(mat_calibration_info *info, int compensation, int reduce, int reduce_original)
{
  int available = NO_COMPENSATION;
  int ref = MAT_NO_REF;
  int r, col, row;
  int rr, cr;

  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  cpl_msg_info(cpl_func, "mat_calibration_info_select_method(...):");
  /* determine which reference regions are available */
  // keep in mind that the corner coordinates are 1-based and the detector frame is specified as number of pixels!
  for (col = 0; col < info->raw_ncols; col++)
    {
      r = col; /* the region index is simply the column number, because the first sub-window row is ok */
      ref |= mat_detector_decode_reference(&(info->det),
					   info->list_region[r].region->corner[0],
					   info->list_region[r].region->corner[1],
					   info->list_region[r].region->naxis[0],
					   info->list_region[r].region->corner[1]);
    }
  for (row = 0; row < info->raw_nrows; row++)
    {
      r = row*info->raw_ncols;
      ref |= mat_detector_decode_reference(&(info->det),
					   info->list_region[r].region->corner[0],
					   info->list_region[r].region->corner[1],
					   info->list_region[r].region->naxis[0],
					   info->list_region[r].region->corner[1]);
    }
  cpl_msg_info(cpl_func, "   reference regions determined");
  /* and now derive all compensation methods which are allowed due to the available maps */
  if (info->bpm != NULL)
    { // add the bad pixel interpolation
      available |= BADPIXEL_COMPENSATION;
    }
  if (info->ffm != NULL)
    { // add the detector flatfield compensation
      available |= DETECTOR_FLATFIELD_COMPENSATION;
    }
  if (info->ofm != NULL)
    { // add the instrument flatfield compensation
      available |= INSTRUMENT_FLATFIELD_COMPENSATION;
    }
  if (info->nlm != NULL)
    { // add the nonlinearity compensation
      available |= NONLINEARITY_COMPENSATION;
    }
  // temporarily only!!!
  if (ref != MAT_NO_REF)
    {
      available |= GLOBAL_BIAS_COMPENSATION;
    }
  if (ref & (MAT_BOTTOM_REF | MAT_TOP_REF))
    {
      available |= CHANNEL_BIAS_COMPENSATION;
    }
  if (ref & (MAT_LEFT_REF | MAT_RIGHT_REF))
    {
      available |= HORIZONTAL_BIAS_COMPENSATION;
    }
  if (ref & (MAT_BOTTOM_REF | MAT_TOP_REF))
    {
      available |= VERTICAL_BIAS_COMPENSATION;
    }
  if ((info->ofm != NULL) || (info->pbm != NULL))
    { /* this is currently the only source for a pixel bias which is needed for all offset compensations */
      available |= PIXEL_BIAS_COMPENSATION;
      if (ref != MAT_NO_REF)
	{
	  available |= GLOBAL_BIAS_COMPENSATION;
	}
      if (ref & (MAT_BOTTOM_REF | MAT_TOP_REF))
	{
	  available |= CHANNEL_BIAS_COMPENSATION;
	}
      if (ref & (MAT_LEFT_REF | MAT_RIGHT_REF))
	{
	  available |= HORIZONTAL_BIAS_COMPENSATION;
	}
      if (ref & (MAT_BOTTOM_REF | MAT_TOP_REF))
	{
	  available |= VERTICAL_BIAS_COMPENSATION;
	}
      if ((ref & MAT_DC_REF) && ((info->det.type == MAT_HAWAII2RG_DET) || (info->det.type == MAT_AQUARIUS_DET)))
	{ // the crosstalk between detector channels is only activated for the HAWAII-2RG/AQUARIUS detector!
	  available |= CROSSTALK_COMPENSATION;
	}
      available |= RAMP_COMPENSATION;
    }
  cpl_msg_info(cpl_func, "   available compensation methods derived(%d)", available);
  /* and now use the subset of the requested and the available compensation methods */
  cpl_msg_info(cpl_func, " -> pixel bias (pb):         sel %d avail %d -> %d",
	       ((compensation & PIXEL_BIAS_COMPENSATION) != 0),
	       ((available & PIXEL_BIAS_COMPENSATION) != 0),
	       (((compensation & available) & PIXEL_BIAS_COMPENSATION) != 0));
  cpl_msg_info(cpl_func, " -> global bias (gb):        sel %d avail %d -> %d",
	       ((compensation & GLOBAL_BIAS_COMPENSATION) != 0),
	       ((available & GLOBAL_BIAS_COMPENSATION) != 0),
	       (((compensation & available) & GLOBAL_BIAS_COMPENSATION) != 0));
  cpl_msg_info(cpl_func, " -> channel bias (cb):       sel %d avail %d -> %d",
	       ((compensation & CHANNEL_BIAS_COMPENSATION) != 0),
	       ((available & CHANNEL_BIAS_COMPENSATION) != 0),
	       (((compensation & available) & CHANNEL_BIAS_COMPENSATION) != 0));
  cpl_msg_info(cpl_func, " -> row bias (rb):           sel %d avail %d -> %d",
	       ((compensation & HORIZONTAL_BIAS_COMPENSATION) != 0),
	       ((available & HORIZONTAL_BIAS_COMPENSATION) != 0),
	       (((compensation & available) & HORIZONTAL_BIAS_COMPENSATION) != 0));
  cpl_msg_info(cpl_func, " -> channel crosstalk (ct):  sel %d avail %d -> %d",
	       ((compensation & CROSSTALK_COMPENSATION) != 0),
	       ((available & CROSSTALK_COMPENSATION) != 0),
	       (((compensation & available) & CROSSTALK_COMPENSATION) != 0));
  cpl_msg_info(cpl_func, " -> nonlinearity (nl):       sel %d avail %d -> %d",
	       ((compensation & NONLINEARITY_COMPENSATION) != 0),
	       ((available & NONLINEARITY_COMPENSATION) != 0),
	       (((compensation & available) & NONLINEARITY_COMPENSATION) != 0));
  cpl_msg_info(cpl_func, " -> detector flat (df):      sel %d avail %d -> %d",
	       ((compensation & DETECTOR_FLATFIELD_COMPENSATION) != 0),
	       ((available & DETECTOR_FLATFIELD_COMPENSATION) != 0),
	       (((compensation & available) & DETECTOR_FLATFIELD_COMPENSATION) != 0));
  cpl_msg_info(cpl_func, " -> instrument flat (if):    sel %d avail %d -> %d",
	       ((compensation & INSTRUMENT_FLATFIELD_COMPENSATION) != 0),
	       ((available & INSTRUMENT_FLATFIELD_COMPENSATION) != 0),
	       (((compensation & available) & INSTRUMENT_FLATFIELD_COMPENSATION) != 0));
  cpl_msg_info(cpl_func, " -> bad pixel (bp):          sel %d avail %d -> %d",
	       ((compensation & BADPIXEL_COMPENSATION) != 0),
	       ((available & BADPIXEL_COMPENSATION) != 0),
	       (((compensation & available) & BADPIXEL_COMPENSATION) != 0));
  cpl_msg_info(cpl_func, " -> optical distortion (od): sel %d avail %d -> %d",
	       ((compensation & DISTORTION_COMPENSATION) != 0),
	       ((available & DISTORTION_COMPENSATION) != 0),
	       (((compensation & available) & DISTORTION_COMPENSATION) != 0));
  cpl_msg_info(cpl_func, " -> electrons (el):          sel %d avail %d -> %d",
	       ((compensation & CONVERSION_COMPENSATION) != 0),
	       ((available & CONVERSION_COMPENSATION) != 0),
	       (((compensation & available) & CONVERSION_COMPENSATION) != 0));
  info->compensation = compensation & available;
  /* set the sub-window mapping between raw and calibrated data, this includes the reduce flag */
  cr = 0;
  if (!reduce_original)
    { /* all sub-windows in the raw data will go into the calibrated data (reorganized or course) */
      for (rr = 0; rr < info->raw_imgdet->nbregion; rr++)
	{
	  info->list_region[rr].cal_idx = cr;
	  cr++;
	}
    }
  else
    { /* remove all exclusive reference regions */
      for (rr = 0; rr < info->raw_imgdet->nbregion; rr++)
	{
	  cpl_msg_info(cpl_func, "   region %dx%d at %d,%d", 
		       info->list_region[rr].region->naxis[0], info->list_region[rr].region->naxis[1],
		       info->list_region[rr].region->corner[0], info->list_region[rr].region->corner[1]);
	  if (!mat_detector_contains_nonreference(&(info->det), info->list_region[rr].region->corner[0], info->list_region[rr].region->corner[1], info->list_region[rr].region->naxis[0], info->list_region[rr].region->naxis[1])) continue;
	  cpl_msg_info(cpl_func, "      added as nr=%d", cr);
	  info->list_region[rr].cal_idx = cr;
	  cr++;
	}
    }
  cpl_msg_info(cpl_func, "   calibrated sub-windows determined: reduce=%d, cr=%d", reduce_original, cr);
  cr = 0;
  if (!reduce)
    { /* all sub-windows in the raw data will go into the calibrated data (reorganized or course) */
      for (rr = 0; rr < info->raw_imgdet->nbregion; rr++)
	{
	  info->map_region[cr] = rr;
	  cr++;
	}
    }
  else
    { /* remove all exclusive reference regions */
      for (rr = 0; rr < info->raw_imgdet->nbregion; rr++)
	{
	  cpl_msg_info(cpl_func, "   region %dx%d at %d,%d", 
		       info->list_region[rr].region->naxis[0], info->list_region[rr].region->naxis[1],
		       info->list_region[rr].region->corner[0], info->list_region[rr].region->corner[1]);
	  if (!mat_detector_contains_nonreference(&(info->det), info->list_region[rr].region->corner[0], info->list_region[rr].region->corner[1], info->list_region[rr].region->naxis[0], info->list_region[rr].region->naxis[1])) continue;
	  cpl_msg_info(cpl_func, "      added as nr=%d", cr);
	  info->map_region[cr] = rr;
	  cr++;
	}
    }
  cpl_msg_info(cpl_func, "   calibrated sub-windows determined (cr=%d)", cr);
  /* cr contains now the number of calibrated sub-windows, we need this for the new mat_imagingdetector */
  info->cal_imgdet = mat_imagingdetector_new(NULL, cr);
  if (info->cal_imgdet == NULL)
    {
      cpl_msg_error(cpl_func,"could not create imaging detector for the calibrated data (info->cal_imgdet)");
      return CPL_ERROR_UNSPECIFIED;
    }
  cpl_msg_info(cpl_func, "   calibrated imagingdetector created");
  /* and now copy the imaging detector from the raw data */
  //strncpy(info->cal_imgdet->origin, info->raw_imgdet->origin, SIZE_MAX_KEYWORDS);


  /* strncpy(info->cal_imgdet->instrument, info->raw_imgdet->instrument, SIZE_MAX_KEYWORDS); */
  if (info->cal_imgdet->instrument != NULL)
    {
      cpl_free(info->cal_imgdet->instrument);
      info->cal_imgdet->instrument = NULL;
    }
  info->cal_imgdet->instrument=cpl_strdup(info->raw_imgdet->instrument);


  info->cal_imgdet->dateobsmjd = info->raw_imgdet->dateobsmjd;

  /* strncpy(info->cal_imgdet->dateobs, info->raw_imgdet->dateobs, SIZE_MAX_KEYWORDS); */
  if (info->cal_imgdet->dateobs != NULL)
    {
      cpl_free(info->cal_imgdet->dateobs);
      info->cal_imgdet->dateobs = NULL;
    }
  info->cal_imgdet->dateobs=cpl_strdup(info->raw_imgdet->dateobs);

  //strncpy(info->cal_imgdet->date, info->raw_imgdet->date, SIZE_MAX_KEYWORDS);


  /* strncpy(info->cal_imgdet->dcsdictionaryid, info->raw_imgdet->dcsdictionaryid, SIZE_MAX_KEYWORDS); */
  if (info->cal_imgdet->dcsdictionaryid != NULL)
    {
      cpl_free(info->cal_imgdet->dcsdictionaryid);
      info->cal_imgdet->dcsdictionaryid = NULL;
    }
  info->cal_imgdet->dcsdictionaryid=cpl_strdup(info->raw_imgdet->dcsdictionaryid);

  /* strncpy(info->cal_imgdet->dcsid, info->raw_imgdet->dcsid, SIZE_MAX_KEYWORDS); */
  if (info->cal_imgdet->dcsid != NULL)
    {
      cpl_free(info->cal_imgdet->dcsid);
      info->cal_imgdet->dcsid = NULL;
    }
  info->cal_imgdet->dcsid=cpl_strdup(info->raw_imgdet->dcsid);

  info->cal_imgdet->nbdetector = info->raw_imgdet->nbdetector;
  info->cal_imgdet->nbtel = info->raw_imgdet->nbtel;
  for (cr = 0; cr < info->cal_imgdet->nbregion; cr++)
    {
      rr = info->list_region[info->map_region[cr]].raw_idx;
      /* mat_region_copy(info->cal_imgdet->list_region[cr], info->raw_imgdet->list_region[rr], MAT_MAXTEL); */
      mat_region_copy(info->cal_imgdet->list_region[cr], info->raw_imgdet->list_region[rr], info->raw_imgdet->nbtel);
      info->cal_imgdet->list_region[cr]->numregion = cr + 1; /* was overwritten by mat_region_copy */
    }
  cpl_msg_info(cpl_func, "   calibrated imagingdetector filled");
  return CPL_ERROR_NONE;
}

/**
   @brief Compensates for the pixel bias by using a cold dark.
   @param info   Contains the calibration context.
   @returns a cpl_error_code

   This method uses a map calculated as average from a series of cold darks with the same
   readout mode, sub-window setup and exposure time as the raw images. This map is contained
   in the observation specific flatfield map (info->ofm).
*/
static cpl_error_code mat_calibration_compensate_pixel_bias(mat_calibration_info *info, mat_frame *frame)
{
  int    r;
  //int    rejected;

  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (info->pbm != NULL)
    {
      /* the ordering of the sub-windows in frame and info->pbm is the same! */
      for (r = 0; r < frame->nbsubwin; r++)
	{
	  cpl_image_subtract(frame->list_subwin[r]->imgreg[0], info->pbm->list_subwin[r]->imgreg[0]);
	}
    }
  else if (info->ofm != NULL)
    {
      /* the ordering of the sub-windows in frame and info->ofm is the same!*/
      for (r = 0; r < frame->nbsubwin; r++)
      	{
	  cpl_image *raw  = frame->list_subwin[r]->imgreg[0];
	  cpl_image *bias = info->ofm->list_bias[r];
	  /*
	  cpl_msg_info(cpl_func, "  raw[%d,%d] = %g",
		       (int)cpl_image_get_size_x(raw)/2,
		       (int)cpl_image_get_size_y(raw)/2,
		       cpl_image_get(raw, cpl_image_get_size_x(raw)/2, cpl_image_get_size_y(raw)/2, &rejected));
	  cpl_msg_info(cpl_func, "  bias[%d,%d] = %g",
		       (int)cpl_image_get_size_x(bias)/2,
		       (int)cpl_image_get_size_y(bias)/2,
		       cpl_image_get(bias, cpl_image_get_size_x(bias)/2, cpl_image_get_size_y(bias)/2, &rejected));
	  */
      	  cpl_image_subtract(raw, bias);
	  /*
	  cpl_msg_info(cpl_func, "  cal[%d,%d] = %g",
		       (int)cpl_image_get_size_x(raw)/2,
		       (int)cpl_image_get_size_y(raw)/2,
		       cpl_image_get(raw, cpl_image_get_size_x(raw)/2, cpl_image_get_size_y(raw)/2, &rejected));
	  */
      	}
    }
  else
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_obsflat (info->ofm) or mat_frame (info->pbm) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Compensates for the individual pixel gain.
   @param info   Contains the calibration context.
   @returns a cpl_error_code

   This method uses a flatfield map of a observation specific flatfield map to compensate
   for the individual pixel gain. If both maps are given, the flatfield map (info->ffm)
   takes precedence.
*/
static cpl_error_code mat_calibration_compensate_flatfield(mat_calibration_info *info, mat_frame *frame)
{
  int    r;

  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if ((info->compensation & DETECTOR_FLATFIELD_COMPENSATION) && (info->ffm != NULL))
    {
      /* the ordering of the sub-windows in frame and info->ffm is the same! */
      for (r = 0; r < frame->nbsubwin; r++)
	{
	  cpl_image_divide(frame->list_subwin[r]->imgreg[0], info->ffm->list_flatfield[r]);
	}
    }
  else if ((info->compensation & INSTRUMENT_FLATFIELD_COMPENSATION) && (info->ofm != NULL))
    {
      /* the ordering of the sub-windows in frame and info->ofm is the same! */
      for (r = 0; r < frame->nbsubwin; r++)
      	{
      	  cpl_image_divide(frame->list_subwin[r]->imgreg[0], info->ofm->list_obsflat[r]);
      	}
    }
  else
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_flatfield (info->ffm) or mat_obsflat (info->ofm) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Compensates for the nonlinearity of each pixel of the Aquarius detector
   @param info   Contains the calibration context.
   @returns a cpl_error_code

   This method uses a nonlinearity map to compensate for the individual pixel nonlinearity.
   This compensation does not include the individual pixel gain!
*/
static cpl_error_code mat_calibration_compensate_nonlinearity(mat_calibration_info *info, mat_frame *frame, int framenr)
{
  int    r;
  int    limitcount = 0;

  mat_assert_not_null(info, "no valid mat_calibration_info (info) given as argument");
  mat_assert_not_null(info->nlm, "no mat_nonlinearity (info->nlm) argument given");
  /* the ordering of the sub-windows in frame and info->nlm is the same! */
  for (r = 0; r < frame->nbsubwin; r++)
    {
      cpl_image  *img = frame->list_subwin[r]->imgreg[0];
      cpl_size    i, n = cpl_image_get_size_x(img)*cpl_image_get_size_y(img); // number of pixels
      double     *iptr = cpl_image_get_data_double(img);                      // intensity
      double     *lptr = cpl_image_get_data_double(info->nlm->list_limit[r]); // limit
      
      if ((info->nlm->nltype == MAT_GLOBAL_MAP_NL) || (info->nlm->nltype == MAT_GLOBAL_LAW_NL))
	{ // The resolution is 1 DU for rcmap
	  for (i = 0; i < n; i++)
	    {
	      double raw = *iptr;
	      if ((raw >= 0.0) && (raw < 65535.0)) // rcmap contains 65536 entries, there is no [65536]!
		{ // use linear interpolation between data points (difference in the rcmap is only 1 DU!)
		  int    rcidx = (int)floor(raw);
		  double frac  = raw - (double)rcidx;
		  double lcal  = info->rcmap[rcidx];
		  double rcal  = info->rcmap[rcidx+1];		  
		  *iptr = lcal*(1.0 - frac) + rcal*frac;
		}
	      if (raw > *lptr) limitcount++;
	      iptr++;
	      lptr++;
	    }
	}
      else if ((info->nlm->nltype == MAT_INDIVIDUAL_NL) && (info->nlm->nbcoeff == 4))
	{ // f(x) = a*x + b*x*exp((x - c)*d);
	  double *aptr = cpl_image_get_data_double(cpl_imagelist_get(info->nlm->list_coeff[r], 0));
	  double *bptr = cpl_image_get_data_double(cpl_imagelist_get(info->nlm->list_coeff[r], 1));
	  double *cptr = cpl_image_get_data_double(cpl_imagelist_get(info->nlm->list_coeff[r], 2));
	  double *dptr = cpl_image_get_data_double(cpl_imagelist_get(info->nlm->list_coeff[r], 3));
	  for (i = 0; i < n; i++)
	    {
	      double raw = *iptr;
	      if ((raw >= 0.0) && (raw < 65536.0))
		{
		  double x = raw/MAT_DU_SCALE;
		  double y = (*aptr)*x + (*bptr)*x*exp((x - (*cptr))*(*dptr));
		  *iptr = y*MAT_DU_SCALE;
		}
	      if (raw > *lptr) limitcount++;
	      iptr++;
	      lptr++;
	      aptr++;
	      bptr++;
	      cptr++;
	      dptr++;
	    }
	}
    }
  if (limitcount > 31)
    {
      cpl_msg_warning(cpl_func, "%d pixels in frame %d have an intensity which is beyond their nonlinearity compensation limit.",limitcount, framenr);
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Compensates for bad pixels.
   @param info   Contains the calibration context.
   @returns a cpl_error_code

   This method uses a CPL method (cpl_detector_interpolate_rejected) for replacing
   the intensities of bad pixels by an interpolated value.
*/
static cpl_error_code mat_calibration_compensate_badpixel(mat_calibration_info *info, mat_frame *frame)
{
  int    r;
  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (info->bpm == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_badpixel (info->bpm) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  /* the ordering of the sub-windows in frame and info->bpm is the same! */
  for (r = 0; r < frame->nbsubwin; r++)
    {
      cpl_image_reject_from_mask(frame->list_subwin[r]->imgreg[0], info->bpm->list_badpixel[r]);
      cpl_detector_interpolate_rejected(frame->list_subwin[r]->imgreg[0]);
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Compensates for a global bias by using the reference pixels under the metal frame.
   @param info   Contains the calibration context.
   @returns a cpl_error_code

   The content mat_frame in the calibration context contains for each pixel the detector channel
   number. The reference pixels use a positive number, whereas the other pixels use a negative
   number. The gobal offset is calculated as intensity median of all pixels with a positive
   detector channel number.
*/
static cpl_error_code mat_calibration_compensate_global_bias(mat_calibration_info *info, mat_frame *frame, mat_statistics_info *sinfo)
{
  int     r;
  int     oa, ov;
  double  bias;

  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (info->content == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (info->content) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  // copy the intensities of all pixels under the metal frame into info->stat_info
  mat_statistics_reset(sinfo);
  oa = sinfo->use_average;
  ov = sinfo->use_variance;
  mat_statistics_set_use(sinfo, MAT_USE_REDUCED_MEAN, MAT_USE_REDUCED_VARIANCE);
  for (r = 0; r < frame->nbsubwin; r++)
    {
      cpl_image  *content = info->content->list_subwin[r]->imgreg[0];
      cpl_image  *img     = frame->list_subwin[r]->imgreg[0];
      cpl_mask   *bad     = NULL;
      cpl_size    nx, ny, x, y;
      int         rejected;
      nx = cpl_image_get_size_x(img);
      ny = cpl_image_get_size_y(img);
      // if a bad pixel map is given, get the mask for the current sub-window
      if (info->bpm != NULL) bad = info->bpm->list_badpixel[r];
      for (y = 1; y <= ny; y++)
	{
	  for (x = 1; x <= nx; x++)
	    {
	      int dc;
	      // if a bad pixel map is given, ignore all bad pixels
	      if ((bad != NULL) && (cpl_mask_get(bad, x, y) == CPL_BINARY_1)) continue;
	      dc = (int)cpl_image_get(content, x, y, &rejected);
	      if (dc > 0)
		{
		  mat_statistics_add_value(sinfo, cpl_image_get(img, x, y, &rejected));
		}
	    }
	}
    }
  // calculate the global bias as median
  if (sinfo->whole_count == 0)
    {
      // no reference pixels => return without an error
      return CPL_ERROR_NONE;
    }
  mat_statistics_calc(sinfo);
  bias = sinfo->reduced_mean; //average;
  //bias = sinfo->reduced_mean;
  // subtract the global bias from all pixels, including the reference pixels
  for (r = 0; r < frame->nbsubwin; r++)
    {
      cpl_image_subtract_scalar(frame->list_subwin[r]->imgreg[0], bias);
    }
  mat_statistics_set_use(sinfo, oa, ov);
  return CPL_ERROR_NONE;
}

/**
   @brief Compensates for a detector channel specific bias by using the reference pixels under the metal frame.
   @param info   Contains the calibration context.
   @returns a cpl_error_code

   The content mat_frame in the calibration context contains for each pixel the detector channel
   number. The reference pixels use a positive number, whereas the other pixels use a negative
   number. The detector channel specific offset is calculated as intensity mean of all pixels of that
   specific detector channel. This offset is then applied to all pixels of that channel.

   This function seems to be a little complicated, but is must be kept in mind that all scientific and the
   lower and upper reference sub-windows cover more than detector channel.
*/
static cpl_error_code mat_calibration_compensate_channel_bias(mat_calibration_info *info, mat_frame *frame, mat_statistics_info *sinfo, double *list_mean)
{
  int   dc, r;
  int   ndc;
  int   oa, ov;

  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  ndc = info->det.channel_nrows*info->det.channel_ncolumns;
  if (info->content == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (info->content) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (info->list_hasref == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no int vector (info->list_hasref) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  // initialize the detector channel mean
  for (dc = 0; dc < ndc; dc++)
    {
      list_mean[dc] = 0.0;
    }
  // calculate the detector channel mean, ignoring the bad pixels
  oa = sinfo->use_average;
  ov = sinfo->use_variance;
  mat_statistics_set_use(sinfo, MAT_USE_REDUCED_MEAN, MAT_USE_REDUCED_VARIANCE);
  for (dc = 0; dc < ndc; dc++)
    {
      //double m;
      mat_statistics_reset(sinfo);
      for (r = 0; r < frame->nbsubwin; r++)
	{
	  cpl_image  *content = info->content->list_subwin[r]->imgreg[0];
	  cpl_image  *img     = frame->list_subwin[r]->imgreg[0];
	  cpl_mask   *bad     = NULL;
	  cpl_size    nx, ny, x, y;
	  int         rejected;
	  nx = cpl_image_get_size_x(img);
	  ny = cpl_image_get_size_y(img);
	  // if a bad pixel map is given, get the mask for the current sub-window
	  if (info->bpm != NULL) bad = info->bpm->list_badpixel[r];
	  // we want to ignore all sub-windows which do not have any reference pixels at all
	  // if we have a scientific sub-window setup, all pixels of a sub-window
	  // are reference pixels or not. Only for calibration reasons a sub-window may
	  // contain reference and non-reference pixels. In that case, the lack of speed
	  // is acceptable.
	  if (info->list_hasref[r] == 0) continue;

	  for (y = 1; y <= ny; y++)
	    {
	      for (x = 1; x <= nx; x++)
		{
		  // if a bad pixel map is given, ignore all bad pixels
		  if ((bad != NULL) && (cpl_mask_get(bad, x, y) == CPL_BINARY_1)) continue;
		  if ((dc + 1) != (int)cpl_image_get(content, x, y, &rejected)) continue; // 1-based in content!
		  mat_statistics_add_value(sinfo, cpl_image_get(img, x, y, &rejected));
		}
	    }
	}
      mat_statistics_calc(sinfo);
      //m = sinfo->reduced_mean;
      list_mean[dc] = sinfo->average;
      /*
      if (sinfo->reduced_count != 0)
	{
	  cpl_msg_info(cpl_func, "dco[%d] = %.2f %.2f %d", dc, sinfo->average, sinfo[dc]->variance, sinfo->reduced_count);
	}
      */
    }
  mat_statistics_set_use(sinfo, oa, ov);
  // and now subtract the detector channel specific bias from all pixels
  for (r = 0; r < frame->nbsubwin; r++)
    {
      cpl_image  *content = info->content->list_subwin[r]->imgreg[0];
      cpl_image  *img     = frame->list_subwin[r]->imgreg[0];
      cpl_size    nx, ny, x, y;
      int         rejected;
      nx = cpl_image_get_size_x(img);
      ny = cpl_image_get_size_y(img);
      for (y = 1; y <= ny; y++)
	{
	  for (x = 1; x <= nx; x++)
	    {
	      dc = (int)cpl_image_get(content, x, y, &rejected); // 1-based!
	      if (dc < 0) dc = -dc;
	      cpl_image_set(img, x, y, cpl_image_get(img, x, y, &rejected) - list_mean[dc - 1]);
	    }
	}
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Compensate for a horizontal bias (1/f noise) by using the reference pixels on the left and/or right side.
   @param info   Contains the calibration context.
   @returns a cpl_error_code

   This compensation method is usefull for HAWAII-1 detectors, because the masked
   part of the detector shows clearly a 1/f noise which spreads horizontally over the whole detector quadrant.
   If this method is also useful for the HAWAII-2RG detector must be evaluated.
*/
static cpl_error_code mat_calibration_compensate_horizontal_bias(mat_calibration_info *info, mat_frame *frame, mat_statistics_info *sinfo)
{
  int            row, col;
  int            r;

  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  for (row = 0; row < info->raw_nrows; row++)
    {
      int          y, ny;
      cpl_image   *left_data     = NULL;
      cpl_mask    *left_bad      = NULL;
      cpl_image   *left_content  = NULL;
      cpl_image   *right_data    = NULL;
      cpl_mask    *right_bad     = NULL;
      cpl_image   *right_content = NULL;
      ny = info->raw_imgdet->list_region[info->list_region[row*info->raw_ncols].raw_idx]->naxis[1];
      r = info->list_region[row*info->raw_ncols].raw_idx;
      if (info->list_hasref[r] != 0)
	{
	  left_data    = frame->list_subwin[r]->imgreg[0];
	  if (info->bpm != NULL) left_bad = info->bpm->list_badpixel[r];
	  left_content = info->content->list_subwin[r]->imgreg[0];
	}
      r = info->list_region[row*info->raw_ncols + info->raw_ncols - 1].raw_idx;
      if (info->list_hasref[r] != 0)
	{
	  right_data    = frame->list_subwin[r]->imgreg[0];
	  if (info->bpm != NULL) right_bad = info->bpm->list_badpixel[r];
	  right_content = info->content->list_subwin[r]->imgreg[0];
	}
      for (y = 1; y <= ny; y++)
	{
	  double bias;
	  // calculate the row bias for the current pixel row
	  mat_statistics_reset(sinfo);
	  if (left_data != NULL)
	    {
	      int x, nx;
	      nx = cpl_image_get_size_x(left_data);
	      for (x = 1; x <= nx; x++)
		{
		  int rejected;
		  if ((left_bad != NULL) && (cpl_mask_get(left_bad, x, y) == CPL_BINARY_1)) continue;
		  if ((int)cpl_image_get(left_content, x, y, &rejected) < 0) continue;
		  mat_statistics_add_value(sinfo, cpl_image_get(left_data, x, y, &rejected));
		}
	    }
	  if (right_data != NULL)
	    {
	      int x, nx;
	      nx = cpl_image_get_size_x(right_data);
	      for (x = 1; x <= nx; x++)
		{
		  int rejected;
		  if ((right_bad != NULL) && (cpl_mask_get(right_bad, x, y) == CPL_BINARY_1)) continue;
		  if ((int)cpl_image_get(right_content, x, y, &rejected) < 0) continue;
		  mat_statistics_add_value(sinfo, cpl_image_get(right_data, x, y, &rejected));
		}
	    }
	  mat_statistics_calc(sinfo);
	  bias  = sinfo->reduced_mean; //average;
	  for (col = 0; col < info->raw_ncols; col++)
	    {
	      cpl_image  *data;
	      int         x, nx;
	      r = info->list_region[row*info->raw_ncols + col].raw_idx;
	      data = frame->list_subwin[r]->imgreg[0]; // the current sub-window
	      nx = cpl_image_get_size_x(data);
	      for (x = 1; x <= nx; x++)
		{
		  int rejected;
		  cpl_image_set(data, x, y, cpl_image_get(data, x, y, &rejected) - bias);
		}
	    }
	}
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_calibration_compensate_vertical_bias(mat_calibration_info *info, mat_frame *frame)
{
  UNUSED(frame);
  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Compensates the crosstalk between detector channels by using the reference pixels on the left and/or right side.
   @param info   Contains the calibration context.
   @returns a cpl_error_code

   For the crosstalk compensation on the left and/or right side of the detector a whole detector channel is needed.
   It was previously checked that we have a complete detector channel as reference on the left and/or right side.
*/
static cpl_error_code mat_calibration_compensate_crosstalk(mat_calibration_info *info, mat_frame *frame)
{
  int            ref_left = 0;
  int            ref_right = 0;
  int            row, col;
  mat_region    *reg = NULL;
  int            r;
  int            dcw;
  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  // check on which side we have a whole detector channel
  reg = info->list_region[0].region; // this is the leftmost sub-window
  if ((info->det.frame_left >= info->det.channel_nx) && (reg->corner[0] == 1) && (reg->naxis[0] >= info->det.channel_nx))
    { // the metal frame covers a whole detector channel on the left side of the detector and the sub-window covers a whole masked detector channel at the left side of the detector
      ref_left = 1;
    }
  reg = info->list_region[info->raw_ncols - 1].region; // this the rightmost sub-window
  if ((info->det.frame_right >= info->det.channel_nx) && ((reg->corner[0] + reg->naxis[0] - 1) == info->det.nx) && (reg->naxis[0] >= info->det.channel_nx))
    { // the metal frame covers a whole detector channel on the right side of the detector and the sub-window covers a whole masked detector channel at the right side of the detector
      ref_right = 1;
    }
  dcw = info->det.channel_nx;
  for (row = 0; row < info->raw_nrows; row++)
    {
      cpl_image  *leftsw;
      cpl_image  *rightsw;
      r = info->list_region[row*info->raw_ncols].raw_idx;
      leftsw = frame->list_subwin[r]->imgreg[0]; // the left sub-window
      r = info->list_region[row*info->raw_ncols + info->raw_ncols - 1].raw_idx;
      rightsw = frame->list_subwin[r]->imgreg[0]; // the right sub-window
      for (col = 0; col < info->raw_ncols; col++)
	{
	  cpl_image  *img;
	  int         x0, nx, ny;
	  int         x, y;
	  r = info->list_region[row*info->raw_ncols + col].raw_idx;
	  img = frame->list_subwin[r]->imgreg[0]; // the current sub-window
	  nx = info->raw_imgdet->list_region[r]->naxis[0];
	  ny = info->raw_imgdet->list_region[r]->naxis[1];
	  x0 = info->raw_imgdet->list_region[r]->corner[0]; // 1-based!
	  for (y = 1; y <= ny; y++)
	    {
	      for (x = 1; x <= nx; x++)
		{
		  int dx = x0 + x - 2; // detector x-coordinate (0-based!)
		  // calculate the detector channel row and column (0-based)
		  int dcc = dx/info->det.channel_nx;
		  int rejected;
		  double count = 0.0;
		  double bias = 0.0;
		  if ((dcc == 0) || (dcc == (info->det.channel_ncolumns - 1))) continue; // ignore the reference pixels, we need them!
		  if (ref_left && (dx%dcw + 1 != 1))
		    { // we can use pixel intensities from the left side of the detector
		      count += 1.0;
		      bias  += cpl_image_get(leftsw, dx%dcw + 1, y, &rejected);
		    }
		  if (ref_right && (dx%dcw + 1 != dcw))
		    { // we can use pixel intensities from the right side of the detector
		      count += 1.0;
		      bias  += cpl_image_get(rightsw, cpl_image_get_size_x(rightsw) - dcw + dx%dcw + 1, y, &rejected);
		    }
		  bias /= count;
		  // and finally subtract the bias
		  cpl_image_set(img, x, y, cpl_image_get(img, x, y, &rejected) - bias);
		}
	    }
	}
    }
  // we have to set all used reference pixels to zero
  if (ref_left)
    { // update the sub-windows on the left side
      for (row = 0; row < info->raw_nrows; row++)
	{
	  cpl_image  *content;
	  cpl_image  *img;
	  cpl_size    nx, ny, x, y;
	  int         rejected;
	  r = info->list_region[row*info->raw_ncols].raw_idx;
	  content = info->content->list_subwin[r]->imgreg[0];
	  img     = frame->list_subwin[r]->imgreg[0];
	  nx = cpl_image_get_size_x(img);
	  ny = cpl_image_get_size_y(img);
	  // the following loop order depends on the fact that the detector channels run in vertical order
	  for (x = 1; x <= nx; x++)
	    {
	      int dcc = ((int)cpl_image_get(content, x, 1, &rejected) - 1)%info->det.channel_ncolumns; // channel column, 0-based!
	      if (dcc == 0)
		{
		  for (y = 1; y <= ny; y++)
		    {
		      cpl_image_set(img, x, y, 0.0);
		    }
		}
	    }
	}
    }
  if (ref_right)
    { // update the sub-windows on the right side
      for (row = 0; row < info->raw_nrows; row++)
	{
	  cpl_image  *content;
	  cpl_image  *img;
	  cpl_size    nx, ny, x, y;
	  int         rejected;
	  r = info->list_region[row*info->raw_ncols + info->raw_ncols - 1].raw_idx;
	  content = info->content->list_subwin[r]->imgreg[0];
	  img     = frame->list_subwin[r]->imgreg[0];
	  nx = cpl_image_get_size_x(img);
	  ny = cpl_image_get_size_y(img);
	  // the following loop order depends on the fact that the detector channels run in vertical order
	  for (x = 1; x <= nx; x++)
	    {
	      int dcc = ((int)cpl_image_get(content, x, 1, &rejected) - 1)%info->det.channel_ncolumns; // channel column, 0-based!
	      if (dcc == (info->det.channel_ncolumns - 1))
		{
		  for (y = 1; y <= ny; y++)
		    {
		      {
			cpl_image_set(img, x, y, 0.0);
		      }
		    }
		}
	    }
	}
    }
  return CPL_ERROR_NONE;
}

#define MAT_RAMP_FIT_COUNT_MIN 6
#define MAT_RAMP_FIT_COUNT_MAX 10

static double mat_calibration_fit_poly2(cpl_image *cal, int dcx, int y, double bias, double *a0, double *a1, double *a2)
{
  int      x;
  double   det, a, b, c;
  double   count = 0.0;
  double   sx    = 0.0;
  double   sxx   = 0.0;
  double   sxxx  = 0.0;
  double   sxxxx = 0.0;
  double   sy    = 0.0;
  double   sxy   = 0.0;
  double   sxxy  = 0.0;
  double   sum0  = 0.0;
  int      outlier;
  int      left, right;
  
  left = -CPL_MIN(MAT_RAMP_FIT_COUNT_MAX,     dcx - 1);
  right = CPL_MIN(MAT_RAMP_FIT_COUNT_MAX - 1, cpl_image_get_size_x(cal) - dcx);
  for (x = left; x <= right; x++)
    {
      int    rejected;
      double z    = cpl_image_get(cal, dcx + x, y, &rejected);
      if (x >= 0) z += bias;
      sx    += (double)x;
      sxx   += (double)(x*x);
      sxxx  += (double)(x*x*x);
      sxxxx += (double)(x*x*x*x);
      sy    += z;
      sxy   += (double)(x)*z;
      sxxy  += (double)(x*x)*z;
    }
  count = right - left + 1;
  // det -> sxx^3 - 2 sx sxx sxxx + N sxxx^2 + sx^2 sxxxx - N sxx sxxxx
  // a -> -((-sxx^2 sxxy + sx sxxx sxxy + sxx sxxx sxy - sx sxxxx sxy - sxxx^2 sy + sxx sxxxx sy)/det)
  // b -> -((sx sxx sxxy - N sxxx sxxy - sxx^2 sxy + N sxxxx sxy + sxx sxxx sy - sx sxxxx sy)/det)
  // c -> -((-sx^2 sxxy + N sxx sxxy + sx sxx sxy - N sxxx sxy - sxx^2 sy + sx sxxx sy)/det)
  det = sxx*sxx*sxx - 2*sx*sxx*sxxx + count*sxxx*sxxx + sx*sx*sxxxx - count*sxx*sxxxx;
  a   = -(-sxx*sxx*sxxy + sx*sxxx*sxxy + sxx*sxxx*sxy - sx*sxxxx*sxy - sxxx*sxxx*sy + sxx*sxxxx*sy)/det;
  b   = -(sx*sxx*sxxy - count*sxxx*sxxy - sxx*sxx*sxy + count*sxxxx*sxy + sxx*sxxx*sy - sx*sxxxx*sy)/det;
  c   = -(-sx*sx*sxxy + count*sxx*sxxy + sx*sxx*sxy - count*sxxx*sxy - sxx*sxx*sy + sx*sxxx*sy)/det;
  // calculate chi squared of the fit
  sum0    = 0.0;
  outlier = 0;
  for (x = left; x <= right; x++)
    {
      int    rejected;
      double z    = cpl_image_get(cal, dcx + x, y, &rejected);
      double zf   = a + b*(double)x + c*(double)(x*x);
      if (x >= 0) z += bias;
      if ((z - zf)*(z - zf) > sum0)
	{
	  outlier = x;
	  sum0    = (z - zf)*(z - zf);
	}
    }
  sum0    = 0.0;
  for (x = left; x <= right; x++)
    {
      int    rejected;
      double z    = cpl_image_get(cal, dcx + x, y, &rejected);
      double zf   = a + b*(double)x + c*(double)(x*x);
      if (x >= 0) z += bias;
      if (x == outlier) continue;
      sum0 += (z - zf)*(z - zf);
    }
  *a0 = a;
  *a1 = b;
  *a2 = c;
  return sum0/(double)(count - 1);
}

#define PIXEL_Y 82

static cpl_error_code mat_calibration_compensate_single_ramp(cpl_image *cal, int dcx, int dcw)
{
  int   x, y, nx, ny;

  nx = cpl_image_get_size_x(cal);
  ny = cpl_image_get_size_y(cal);
  for (y = 1; y <= ny; y++)
    {
      double delta, lbias, rbias;
      double la0, la1, la2, lchi2;
      double ra0, ra1, ra2, rchi2;
      lbias = 0.0;
      rbias = 0.0;
      lchi2 = mat_calibration_fit_poly2(cal, dcx,       y,  lbias, &la0, &la1, &la2);
      rchi2 = mat_calibration_fit_poly2(cal, dcx + dcw, y, -rbias, &ra0, &ra1, &ra2);
      if (y == PIXEL_Y)
	{
	  for (x = 1; x <= nx; x++)
	    {
	      int    rejected;
	      cpl_msg_info(cpl_func, "ramp: z[%d] = %.1f", x, cpl_image_get(cal, x, y, &rejected));
	    }
	  cpl_msg_info(cpl_func, "ramp: init: lbias %.1f lchi2 %.1f (%.1f %.1f %.1f) rbias %.f1 rchi2 %.1f (%.1f %.1f %.1f)",
		       lbias, lchi2, la0, la1, la2, rbias, rchi2, ra0, ra1, ra2);
	}
      delta = 100;
      do {
	double hchi2;
	hchi2 = mat_calibration_fit_poly2(cal, dcx, y, lbias + delta, &la0, &la1, &la2);
	if (y == PIXEL_Y)
	  cpl_msg_info(cpl_func, "ramp: iter: lbias %.1f lchi2 %.1f (%.1f %.1f %.1f)", lbias + delta, hchi2, la0, la1, la2);
	if (hchi2 < lchi2)
	  {
	    lbias += delta;
	    lchi2 = hchi2;
	    continue;
	  }
	hchi2 = mat_calibration_fit_poly2(cal, dcx, y, lbias - delta, &la0, &la1, &la2);
	if (y == PIXEL_Y)
	  cpl_msg_info(cpl_func, "ramp: iter: lbias %.1f lchi2 %.1f (%.1f %.1f %.1f)", lbias - delta, hchi2, la0, la1, la2);
	if (hchi2 < lchi2)
	  {
	    lbias -= delta;
	    lchi2 = hchi2;
	    continue;
	  }
	if (delta < 0.25) break;
	delta *= 0.5;
      } while (1);
      delta = 100.0;
      do {
	double hchi2;
	hchi2 = mat_calibration_fit_poly2(cal, dcx + dcw, y, -(rbias + delta), &ra0, &ra1, &ra2);
	if (y == PIXEL_Y)
	    cpl_msg_info(cpl_func, "ramp: iter: rbias %.1f rchi2 %.1f (%.1f %.1f %.1f)", -(rbias + delta), hchi2, ra0, ra1, ra2);
	if (hchi2 < rchi2)
	  {
	    rbias += delta;
	    rchi2 = hchi2;
	    continue;
	  }
	hchi2 = mat_calibration_fit_poly2(cal, dcx + dcw, y, -(rbias - delta), &ra0, &ra1, &ra2);
	if (y == PIXEL_Y)
	  cpl_msg_info(cpl_func, "ramp: iter: rbias %.1f rchi2 %.1f (%.1f %.1f %.1f)", -(rbias - delta), hchi2, ra0, ra1, ra2);
	if (hchi2 < rchi2)
	  {
	    rbias -= delta;
	    rchi2 = hchi2;
	    continue;
	  }
	if (delta < 0.25) break;
	delta *= 0.5;
      } while (1);
      if (y == PIXEL_Y)
	{
	  cpl_msg_info(cpl_func, "ramp: final: lbias %.1f lchi2 %.1f (%.1f %.1f %.1f) rbias %.1f rchi2 %.1f (%.1f %.1f %.1f)",
		       lbias, lchi2, la0, la1, la2, -rbias, rchi2, ra0, ra1, ra2);
	}
      for (x = dcx; x < dcx + dcw; x++)
	{
	  int rejected;
	  cpl_image_set(cal, x, y, cpl_image_get(cal, x, y, &rejected) + (lbias + (rbias - lbias)/(double)dcw*(double)(x - dcx)));
	}
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_calibration_compensate_ramp(mat_calibration_info *info, mat_frame *frame)
{
  int            row, col;
  int            dcw;

  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  dcw = info->det.channel_nx;
  for (row = 0; row < info->raw_nrows; row++)
    {
      for (col = 0; col < info->raw_ncols; col++)
	{
	  int         r   = info->list_region[row*info->raw_ncols + col].raw_idx;
	  int         nx  = info->raw_imgdet->list_region[r]->naxis[0];
	  //int         ny  = info->raw_imgdet->list_region[r]->naxis[1];
	  int         x0  = info->raw_imgdet->list_region[r]->corner[0];  // 1-based
	  //int         y0  = info->raw_imgdet->list_region[r]->corner[1];  // 1-based
	  cpl_image  *img = frame->list_subwin[r]->imgreg[0];       // the current sub-window;
	  if (info->list_hassci[r] == 0) continue; // ignore all non scientific regions!
	  if ((897 - MAT_RAMP_FIT_COUNT_MIN > x0) && (897 + dcw + MAT_RAMP_FIT_COUNT_MIN < x0 + nx))
	    {
	      mat_calibration_compensate_single_ramp(img, 897 - x0 + 1, dcw);
	    }
	}
    }

  return CPL_ERROR_NONE;
}

static cpl_error_code mat_calibration_compensate_frame(mat_calibration_info *info, mat_frame *cal, mat_frame *raw, int framenr)
{
  int                   cr, rr;
  mat_imgreg           *rreg = NULL, *creg = NULL;
  int                   i;
  mat_frame            *frame     = NULL;
  mat_statistics_info  *stat_info = NULL;            /*!< contains the data structure for statistic calculations ignoring outliers */
  double               *list_mean = NULL;            /*!< contains the sum/mean intensity (bias) for each detector channel */

  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (raw == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (raw) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (cal == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (cal) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
#ifdef PARALLEL
  // With OpenMP -> get the working space exclusively for this thread
  frame     = info->frame_list[omp_get_thread_num()];
  stat_info = info->stat_list[omp_get_thread_num()];
  list_mean = info->list_dc_list[omp_get_thread_num()];
  //#pragma omp critical
  //cpl_msg_info(cpl_func, "thread %d of %d calibrates frame %d", omp_get_thread_num(), omp_get_num_threads(), framenr);
#else
  // No OpenMP -> only one available working frame, statistics and dc bias list
  frame     = info->frame_list[0];
  stat_info = info->stat_list[0];
  list_mean = info->list_dc_list[0];
  //cpl_msg_info(cpl_func, "single thread calibrates frame %d", framenr);
#endif
  if (cal != raw)
    {
      // copy the frame specific information from the raw into the calibrated frame (not for inplace)
      cal->time = raw->time;
      cal->exptime = raw->exptime;
      for (i = 0; i < MAT_MAXTEL; i++)
	{
	  cal->opd[i] = raw->opd[i];
	  cal->localopd[i] = raw->localopd[i];
	}
      cal->stepphase = raw->stepphase;
      cal->tartype[0] = raw->tartype[0];
      cal->tartype[1] = raw->tartype[1];
      for (cr = 0; cr < info->cal_imgdet->nbregion; cr++)
	{
	  rr = info->list_region[info->map_region[cr]].raw_idx;
	  rreg = raw->list_subwin[rr];
	  creg = cal->list_subwin[cr];
	  /* and now copy the region specific information from the raw into the calibrated region */
	  creg->numregion = cr + 1;
	  creg->numdetector = rreg->numdetector;
	  creg->nbimgreg = rreg->nbimgreg;
	}
    }
  // copy the raw data (float) into the temporary frame (double)
  for (rr = 0; rr < info->raw_imgdet->nbregion; rr++)
    {
      cpl_size    nx, ny, x, y;
      int         rejected;
      double      value;
      nx = cpl_image_get_size_x(raw->list_subwin[rr]->imgreg[0]);
      ny = cpl_image_get_size_y(raw->list_subwin[rr]->imgreg[0]);
      for (y = 1; y <= ny; y++)
	{
	  for (x = 1; x <= nx; x++)
	    {
	      value = cpl_image_get(raw->list_subwin[rr]->imgreg[0], x, y, &rejected);
	      cpl_image_set(frame->list_subwin[rr]->imgreg[0], x, y, value);
	    }
	}
    }
  /* do the compensation, the ordering is important! */      
  if (info->compensation & PIXEL_BIAS_COMPENSATION) mat_calibration_compensate_pixel_bias(info, frame);
  if (info->compensation & GLOBAL_BIAS_COMPENSATION) mat_calibration_compensate_global_bias(info, frame, stat_info);
  if (info->compensation & CHANNEL_BIAS_COMPENSATION) mat_calibration_compensate_channel_bias(info, frame, stat_info, list_mean);
  if (info->compensation & HORIZONTAL_BIAS_COMPENSATION) mat_calibration_compensate_horizontal_bias(info, frame, stat_info);
  if (info->compensation & VERTICAL_BIAS_COMPENSATION) mat_calibration_compensate_vertical_bias(info, frame);
  if (info->compensation & CROSSTALK_COMPENSATION)
    { /* this compensation requires a previous bad pixel interpolation */
      if (info->compensation & BADPIXEL_COMPENSATION) mat_calibration_compensate_badpixel(info, frame);
      mat_calibration_compensate_crosstalk(info, frame);
    }
  if (info->compensation & NONLINEARITY_COMPENSATION) mat_calibration_compensate_nonlinearity(info, frame, framenr);
  if (info->compensation & (DETECTOR_FLATFIELD_COMPENSATION | INSTRUMENT_FLATFIELD_COMPENSATION)) mat_calibration_compensate_flatfield(info, frame);
  if (info->compensation & BADPIXEL_COMPENSATION) mat_calibration_compensate_badpixel(info, frame);
  if (info->compensation & RAMP_COMPENSATION) mat_calibration_compensate_ramp(info, frame);
  // copy the calibrated images from the temporary frame (double) into the calibrated frame (float)
  for (cr = 0; cr < info->cal_imgdet->nbregion; cr++)
    {
      cpl_size    nx, ny, x, y;
      int         rejected;
      double      value;
      rr = info->list_region[info->map_region[cr]].raw_idx;
      nx = cpl_image_get_size_x(cal->list_subwin[cr]->imgreg[0]);
      ny = cpl_image_get_size_y(cal->list_subwin[cr]->imgreg[0]);
      for (y = 1; y <= ny; y++)
	{
	  for (x = 1; x <= nx; x++)
	    {
	      value = cpl_image_get(frame->list_subwin[rr]->imgreg[0], x, y, &rejected);
	      cpl_image_set(cal->list_subwin[cr]->imgreg[0], x, y, value);
	    }
	}
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Removed the exclusive reference sub-windows.
   @param info   Contains the calibration context.
   @param cal    Contains a series of calibrated frames.
   @returns a cpl_error_code

   This method modifies a mat_gendata data structure in order to remove the sub-windows
   which contains only reference pixels. All sub-windows which contain at least one normal
   pixels are kept.

   The implementation of this method is complicated, because it must be ensured that even if an error
   occurs, the data structure for the calibrated data is consistent at all time.
*/
static cpl_error_code mat_calibration_reduce(mat_calibration_info  *info, mat_gendata *cal)
{
  // we need some temporary space to reorganize the regions (sub-windows)
  // inside a mat_imagingdetector and a mat_frame data structure
  mat_region      **list_region = NULL;
  mat_imgreg      **list_subwin = NULL;
  int               rr, cr, ncr, i;

  if (info == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_calibration (info) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (cal == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (cal) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  // we allocate he temporary space
  list_region = (mat_region **)cpl_calloc(info->raw_imgdet->nbregion, sizeof(mat_region *));
  if (list_region == NULL)
    {
      cpl_msg_error(cpl_func,"could not create mat_region * vector (list_region)");
      return CPL_ERROR_UNSPECIFIED;
    }
  list_subwin = (mat_imgreg **)cpl_calloc(info->raw_imgdet->nbregion, sizeof(mat_imgreg *));
  if (list_subwin == NULL)
    {
      cpl_msg_error(cpl_func,"could not create mat_imgreg * vector (list_subwin)");
      cpl_free(list_region);
      return CPL_ERROR_UNSPECIFIED;
    }
  // we will reorganize the region definition inside a mat_imagingdetector data structure
  // this is done by:
  // 1. moving the science regions into the temporary data structure (list_region)
  // 2. deleting the pure reference regions
  // 3. moving the science regions back into the original vector (cal->imgdet->list_region) which is now longer than needed
  ncr = 0;
  for (rr = 0; rr < info->raw_imgdet->nbregion; rr++)
    {
      cr = info->list_region[rr].cal_idx;
      if (cr == -1)
	{ // this is a pure reference region => delete it
	  mat_region_delete(cal->imgdet->list_region[rr]);
	}
      else
	{ // this region contains science data => use it
	  list_region[cr] = cal->imgdet->list_region[rr];
	  ncr++;
	}
      cal->imgdet->list_region[rr] = NULL;
    }
  for (cr = 0; cr < ncr; cr++)
    {
      cal->imgdet->list_region[cr] = list_region[cr];
      cal->imgdet->list_region[cr]->numregion = (cr + 1); // renumber the region
    }
  cal->imgdet->nbregion = ncr;
  // we will reorganize the regions inside each frame
  // this will be done by:
  // 1. moving the science regions into the temporary data structure (list_subwin)
  // 2. deleting the pure reference regions
  // 3. moving the science regions back into the original vector (cal->imgdata->list_frame[i]->list_subwin) which is now longer than needed
  for (i = 0; i < cal->imgdata->nbframe; i++)
    {
      mat_frame *frame = cal->imgdata->list_frame[i];
      for (rr = 0; rr < info->raw_imgdet->nbregion; rr++)
	{
	  cr = info->list_region[rr].cal_idx;
	  if (cr == -1)
	    { // this is a pure reference region => delete it
	      mat_imgreg_delete(frame->list_subwin[rr]);
	    }
	  else
	    { // this region contains science data => use it
	      list_subwin[cr] = frame->list_subwin[rr];
	    }
	  frame->list_subwin[rr] = NULL;
	}
      for (cr = 0; cr < ncr; cr++)
	{
	  frame->list_subwin[cr] = list_subwin[cr];
	  frame->list_subwin[cr]->numregion = (cr + 1); // renumber the region
	}
      frame->nbsubwin = ncr;
    }
  cpl_free(list_region);
  cpl_free(list_subwin);
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup cal
   @brief Compensate for detector effects in the raw images and optionally removes the reference sub-windows.
   @param raw           Contains a series of raw images.
   @param bpm           Contains a bad pixel map (optional).
   @param ffm           Contains the flafield map (optional).
   @param nlm           Contains a nonlinearity map (optional).
   @param ofm           Contains the observation specific flatfield map (optional).
   @param pbm           Contains a pixel bias map (optional).
   @param compensation  A set of flags describing the requested compensation methods.
   @param reduce        A flag if the reference sub-windows should be removed.
   @param inplace       A flag that the compensation should modify the input frames.
   @returns A structure containing the calibrated images or NULL.

   This function uses up to three calibration maps to calibrate a series of raw frames
   and produce a series of calibrated frames. This calibration covers the following
   effects:

   - intensity offset
   - gain variations or nonlinearity
   - observation specific illumination
   - bad pixels

   It depends on the provided calibration maps and a parameter (compensation) to decide
   which effect is compensated. In addition to the compensation, the reference regions
   could be removed from the result (controlled by reduce).

   The static calibration map parameters (bpm, ffm, nlm and ofm) determine which kind of
   detector effect compensation is available. The compensation of each frame has to follow
   exactly the following order:

   1. If a observation specific flatfield map (ofm) is given, the dark from this map is subtracted
   from a raw frame. If a pixel bias map (pbm) is given, it is subtracted from a raw frame.
   2. If reference pixels are available, a global detector bias is calculated and subtracted
   from each pixel of a raw frame.
   3. If reference pixels for detector channels are available, an individual bias is calculated and
   subtracted from the pixels of that detector channel.
   4. If reference pixels on the left and/or right side of a detector are part of a raw frame,
   a pixel row bias is calculated and subtracted from all pixels of a row.
   5. If reference pixels on the top and/or bottom side of a detector are part of a raw frame,
   a pixel column bias is calculated and subtracted from all pixels of a column.
   6. If reference pixels on the left and/or right side of a detector are part of a raw frame and they
   cover a whole detector channel, the bad pixels of this channel are compensated and the whole
   detector channel is pixel by pixel subtracted from all other detector channels.
   7. If a static flatfield map or a observation specific flatfield map are provided, all pixels
   are divided by the corresponding value in the map.
   8. If a static nonlinearity map is provided, all pixels are compensated using the corresponding
   nonlinearity coefficients in the map. This compensation does not compensate the individual
   pixel gain!
   9. If a bad pixel map is given, the bad pixel values are replaced by interpolated values.

*/
/*-----------------------------------------------------------------------------*/
mat_gendata *mat_calibration_detector(mat_gendata *raw,
				      mat_badpixel *bpm,
				      mat_flatfield *ffm,
				      mat_nonlinearity *nlm,
				      mat_obsflat *ofm,
				      mat_frame *pbm,
				      int compensation,
				      int reduce,
				      int inplace)
{
  mat_calibration_info  info;
  mat_detector          det;
  mat_gendata          *cal = NULL;
  int                   i;
  int                   reduce_original = reduce;


  /* cpl_errorstate prestate = cpl_errorstate_get(); */
  /* initialize the calibration context and fill it according to the parameters */
  // modify the reduce flag if an inplace compensation is requested
  if (inplace) reduce = 0;
  cpl_msg_info(cpl_func, "mat_apply_staticcalib(...): prepare info");
  mat_calibration_info_init(&info);
  mat_detector_decode_raw(&det, raw);
  info.exptime = cpl_propertylist_get_double(raw->keywords, "EXPTIME");
  mat_calibration_info_set_setup(&info, &det, raw->imgdet);
  mat_calibration_info_set_badpixel(&info, bpm);
  mat_calibration_info_set_flatfield(&info, ffm);
  mat_calibration_info_set_nonlinearity(&info, nlm);
  mat_calibration_info_set_obsflat(&info, ofm);
  mat_calibration_info_set_pixelbias(&info, pbm);
  mat_calibration_info_select_method(&info, compensation, reduce, reduce_original);
  if (inplace)
    {
      cal = raw; // we use an inplace compensation
    }
  else
    {
      /* create the data structure for the result (maybe reduced sub-windows) */
      cal = (mat_gendata *)cpl_calloc(1, sizeof(mat_gendata));
      if (cal == NULL) {
  	cpl_msg_error(cpl_func, "could not allocate memory for the calibrated data");
  	mat_calibration_info_delete(&info);
  	return NULL;
      }
      cal->keywords = cpl_propertylist_duplicate(raw->keywords);
      cal->imgdet = mat_imagingdetector_duplicate(info.cal_imgdet);
      cal->imgdata = mat_imagingdata_new(cal->imgdet, raw->imgdata->nbframe, CPL_TYPE_FLOAT);
      for (i = 0; i < cal->imgdata->nbtel; i++)
  	{
  	  cal->imgdata->staindex[i] = raw->imgdata->staindex[i];
  	  cal->imgdata->armindex[i] = raw->imgdata->armindex[i];
  	}
      cal->imgdata->maxstep = raw->imgdata->maxstep;
      cal->imgdata->maxins = raw->imgdata->maxins;
      cal->array = mat_array_duplicate(raw->array);
    }
  /* calibrate each frame */
  struct timeval tv1, tv2; // To measure timing of parallel section
  cpl_msg_info(cpl_func, "Starting parallel loop of %d", raw->imgdata->nbframe);
  gettimeofday(&tv1, NULL);
#ifdef PARALLEL
#pragma omp parallel
  {
#pragma omp single
    mat_calibration_alloc_workspace(&info, omp_get_num_threads());
#pragma omp for
    for (i = 0; i < raw->imgdata->nbframe; i++)
      {
	mat_calibration_compensate_frame(&info, cal->imgdata->list_frame[i], raw->imgdata->list_frame[i], i+1);
      }
  }
#else
  {
    mat_calibration_alloc_workspace(&info, 1);
    for (i = 0; i < raw->imgdata->nbframe; i++)
      {
	mat_calibration_compensate_frame(&info, cal->imgdata->list_frame[i], raw->imgdata->list_frame[i], i+1);
      }
  }
#endif

  gettimeofday(&tv2, NULL);
  cpl_msg_info(cpl_func, "Wall time for parallel loop = %f seconds\n",
               (double) (tv2.tv_usec - tv1.tv_usec) / 1000000 +
               (double) (tv2.tv_sec - tv1.tv_sec));

  if (reduce_original && inplace)
    {
      // We calibrated the date without removing the calibration sub-windows,
      // but they should be removed: ->  We have to remove them afterwards
      mat_calibration_reduce(&info, cal);
    }
  /* if (!cpl_errorstate_is_equal(prestate)) { */
  /*   cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one); */
  /*   cpl_errorstate_set(prestate); */
  /*   mat_gendata_delete(cal); */
  /*   mat_calibration_info_delete(&info); */
  /*   return NULL; */
  /* } */
  mat_calibration_info_delete(&info);
  return cal;
}

/**
   @ingroup cal
   @brief Compensate for all detector effects in the raw images and removes the reference sub-windows.
   @param raw           Contains a series of raw images.
   @param bpm           Contains a bad pixel map (optional).
   @param nlm           Contains a nonlinearity map (optional).
   @param ofm           Contains the observation specific flatfield map (optional).
   @returns A structure containing the calibrated images or NULL.

   This function uses up to three calibration maps to calibrate a series of raw frames
   and produce a series of calibrated frames. This calibration covers the following
   effects:

   - intensity offset
   - gain variations or nonlinearity
   - observation specific illumination
   - bad pixels

   It depends on the provided calibration maps to decide which effect is compensated.
   In addition to the compensation, the reference regions are removed from the result.
*/
mat_gendata *mat_apply_staticcalib(mat_gendata *raw,
                                   mat_badpixel *bpm,
                                   mat_nonlinearity *nlm,
                                   mat_obsflat *ofm)
{
  mat_gendata *cal;
  char *detNameData=NULL;
  char *detNameOfm=NULL;
  char *specResData=NULL;
  char *specResOfm=NULL;
  if (ofm != NULL) {
    cpl_msg_info(cpl_func, "checking if the OBS_FLATFIELD is compatible");
    detNameData=(char *)cpl_propertylist_get_string(raw->keywords,"ESO DET CHIP NAME");
    detNameOfm=(char *)cpl_propertylist_get_string(ofm->keywords,"ESO DET CHIP NAME");
    if ((detNameData == NULL) || (detNameOfm == NULL))
      {
	cpl_msg_info(cpl_func, "RAW or OBS_FLATFIELD does not contain the ESO DET CHIP NAME keyword");
      }
    else if ( strcmp(detNameData,detNameOfm) ) {
      cpl_msg_info(cpl_func,"The OBS_FALTFIELD is not compatible with the data (Detector)");
      return NULL;
    }
    cpl_msg_info(cpl_func, "   OBS_FLATFIELD and RAW data are from the same detector (%s)", detNameData);
    
    // Check detector and spectral resolution
    if(!strcmp(detNameData, "AQUARIUS")) {
      specResData=(char *)cpl_propertylist_get_string(raw->keywords,"ESO INS DIN POS");
      specResOfm=(char *)cpl_propertylist_get_string(ofm->keywords,"ESO INS DIN POS");
    } else {
      specResData=(char *)cpl_propertylist_get_string(raw->keywords,"ESO INS DIL POS");
      specResOfm=(char *)cpl_propertylist_get_string(ofm->keywords,"ESO INS DIL POS");
    }
    if ((specResData == NULL) || (specResOfm == NULL))
      {
	cpl_msg_info(cpl_func, "RAW or OBS_FLATFIELD does not contain the ESO INS DI[N|L] POS keyword");
      }
    else if ( strcmp(specResData,specResOfm) ) {
      cpl_msg_info(cpl_func,"The OBS_FLATFIELD is not compatible with the data (spectral resolution)");
      return NULL;
    }
  }
  cpl_msg_info(cpl_func, "executing mat_calibration_detector()");
  cal = mat_calibration_detector(raw, bpm, NULL, nlm, ofm, NULL, FULL_COMPENSATION, 1, 0);
  return cal;
}


static int mat_exposure_setup_info_init(mat_exposure_setup_info *info)
{
  info->type            = MAT_UNKNOWN_DET;
  info->nx              = 0;
  info->ny              = 0;
  info->expno           = 0;
  info->obsstart        = 0.0;
  info->timstart        = 0.0;
  info->timperiod       = 0.0;
  info->framestart      = 0.0;
  info->chopfreq        = 0.0;
  info->chopperiod      = 0.0;
  info->sdch            = 'S';
  info->tbch            = 'T';
  info->nbsky           = 0;
  info->nbtarget        = 0;
  info->nbdead          = 0;
  info->stepphase_min   = 0;
  info->stepphase_max   = 0;
  info->nstepphases     = 0;
  info->tot_target      = 0;
  info->tot_chop        = 0;
  info->tot_entries     = 0;
  info->raw_first_index = 0;
  info->est_time_shift  = 0;
  return 1;
}

static int mat_raw_data_info_init(mat_raw_data_info *info)
{
  info->nbentries        = 0;
  info->intensity_offset = 1e64;
  info->intensity        = NULL;
  info->correlation      = NULL;
  info->V                = NULL;
  info->cr_idx           = -1;
  info->cr_nx            = 0;
  info->cr_ny            = 0;
  info->roi_left         = 0;
  info->roi_right        = 0;
  info->roi_bottom       = 0;
  info->roi_top          = 0;
  return 1;
}

static int mat_raw_data_info_alloc(mat_raw_data_info *info, int nbentries)
{
  info->nbentries = nbentries;
  info->intensity = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->intensity == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the raw data median intensity.");
      return 0;
    }
  info->correlation = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->correlation == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the raw data correlation.");
      return 0;
    }
  info->V = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->V == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the raw data V.");
      return 0;
    }
  return 1;
}

static int mat_raw_data_info_free(mat_raw_data_info *info)
{
  if (info->intensity != NULL)
    {
      cpl_free(info->intensity);
      info->intensity = NULL;
    }
  if (info->correlation != NULL)
    {
      cpl_free(info->correlation);
      info->correlation = NULL;
    }
  if (info->V != NULL)
    {
      cpl_free(info->V);
      info->V = NULL;
    }
  return 1;
}

static void mat_chopping_cycle_info_init(mat_chopping_cycle_info *info)
{
  info->nbentries           = 0;
  info->raw_tartyp          = NULL;
  info->opt_tartyp          = NULL;
  info->stepping_phase      = NULL;
  info->first               = 0;
  info->cycles              = 0;
  info->phase               = 0;
  info->intensity           = NULL;
  info->relint              = NULL;
  info->V                   = NULL;
  info->S                   = NULL;
  info->skyS                = NULL;
  info->targetS             = NULL;
  info->est_V_shift         = 0;
  info->est_S_shift         = 0;
  info->est_tartyp          = NULL;
}

static int mat_chopping_cycle_info_alloc(mat_chopping_cycle_info *info, int nbentries)
{
  info->nbentries = nbentries;
  info->raw_tartyp = (char*)cpl_calloc(info->nbentries, sizeof(char));
  if (info->raw_tartyp == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the raw chopping cycle TARTYP.");
      return 0;
    }
  info->opt_tartyp = (char*)cpl_calloc(info->nbentries, sizeof(char));
  if (info->opt_tartyp == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the optimal chopping cycle TARTYP.");
      return 0;
    }
  info->stepping_phase = (int*)cpl_calloc(info->nbentries, sizeof(int));
  if (info->stepping_phase == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the first chopping cycle STEPPING PHASE.");
      return 0;
    }
  info->intensity = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->intensity == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the chopping cycle raw data median intensity.");
      return 0;
    }
  info->relint = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->relint == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the chopping cycle relative intensity.");
      return 0;
    }
  info->V = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->V == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the first chopping cycle V.");
      return 0;
    }
  info->S = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->S == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the first chopping cycle S.");
      return 0;
    }
  info->skyS = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->skyS == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the first chopping cycle S sky.");
      return 0;
    }
  info->targetS = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->targetS == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the first chopping cycle S target.");
      return 0;
    }
  info->est_tartyp = (char*)cpl_calloc(info->nbentries, sizeof(char));
  if (info->est_tartyp == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the estimated chopping cycle TARTYP.");
      return 0;
    }
  return 1;
}

static void mat_chopping_cycle_info_free(mat_chopping_cycle_info *info)
{
  if (info->raw_tartyp != NULL)
    {
      cpl_free(info->raw_tartyp);
      info->raw_tartyp = NULL;
    }
  if (info->opt_tartyp != NULL)
    {
      cpl_free(info->opt_tartyp);
      info->opt_tartyp = NULL;
    }
  if (info->stepping_phase != NULL)
    {
      cpl_free(info->stepping_phase);
      info->stepping_phase = NULL;
    }
  if (info->intensity != NULL)
    {
      cpl_free(info->intensity);
      info->intensity = NULL;
    }
  if (info->relint != NULL)
    {
      cpl_free(info->relint);
      info->relint = NULL;
    }
  if (info->V != NULL)
    {
      cpl_free(info->V);
      info->V = NULL;
    }
  if (info->S != NULL)
    {
      cpl_free(info->S);
      info->S = NULL;
    }
  if (info->skyS != NULL)
    {
      cpl_free(info->skyS);
      info->skyS = NULL;
    }
  if (info->targetS != NULL)
    {
      cpl_free(info->targetS);
      info->targetS = NULL;
    }
  if (info->est_tartyp != NULL)
    {
      cpl_free(info->est_tartyp);
      info->est_tartyp = NULL;
    }
}

static void mat_allouche_info_init(mat_allouche_info *info)
{
  info->nbentries           = 0;
  info->raw_tartyp          = NULL;
  info->cycles              = 0;
  info->intensity           = NULL;
  info->V                   = NULL;
  info->est_V_shift         = 0;
  info->est_tartyp          = NULL;
}

static int mat_allouche_info_alloc(mat_allouche_info *info, int nbentries)
{
  info->nbentries = nbentries;
  info->raw_tartyp = (char*)cpl_calloc(info->nbentries, sizeof(char));
  if (info->raw_tartyp == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the raw chopping cycle TARTYP.");
      return 0;
    }
  info->intensity = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->intensity == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the chopping cycle raw data median intensity.");
      return 0;
    }
  info->V = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->V == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the first chopping cycle V.");
      return 0;
    }
  info->est_tartyp = (char*)cpl_calloc(info->nbentries, sizeof(char));
  if (info->est_tartyp == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the estimated chopping cycle TARTYP.");
      return 0;
    }
  return 1;
}

static void mat_allouche_info_free(mat_allouche_info *info)
{
  if (info->raw_tartyp != NULL)
    {
      cpl_free(info->raw_tartyp);
      info->raw_tartyp = NULL;
    }
  if (info->intensity != NULL)
    {
      cpl_free(info->intensity);
      info->intensity = NULL;
    }
  if (info->V != NULL)
    {
      cpl_free(info->V);
      info->V = NULL;
    }
  if (info->est_tartyp != NULL)
    {
      cpl_free(info->est_tartyp);
      info->est_tartyp = NULL;
    }
}

static void mat_timetable_info_init(mat_timetable_info *info)
{
  info->nbentries      = 0;
  info->raw_tartyp     = NULL;
  info->opt_tartyp     = NULL;
  info->localopd1      = NULL;
  info->localopd2      = NULL;
  info->localopd3      = NULL;
  info->localopd4      = NULL;
  info->stepping_phase = NULL;
  info->first          = -1;
  info->second         = -1;
  info->cycles         = 0;
  info->phase          = 0;
  info->intensity      = NULL;
  info->V              = NULL;
#ifdef WITH_CORRELATION
  info->correlation    = NULL;
#endif
  info->est_V_shift    = 0;
  info->est_tartyp     = NULL;
}

static int mat_timetable_info_alloc(mat_timetable_info *info, int nbentries)
{
  info->nbentries = nbentries;
  info->raw_tartyp = (char*)cpl_calloc(info->nbentries, sizeof(char));
  if (info->raw_tartyp == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the timetable TARTYP.");
      return 0;
    }
  info->opt_tartyp = (char*)cpl_calloc(info->nbentries, sizeof(char));
  if (info->opt_tartyp == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the timetable TARTYP (without chopping deadtime).");
      return 0;
    }
  info->localopd1 = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->localopd1 == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the timetable LOCAL OPD[1].");
      return 0;
    }
  info->localopd2 = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->localopd2 == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the timetable LOCAL OPD[2].");
      return 0;
    }
  info->localopd3 = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->localopd3 == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the timetable LOCAL OPD[3].");
      return 0;
    }
  info->localopd4 = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->localopd4 == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the timetable LOCAL OPD[4].");
      return 0;
    }
  info->stepping_phase = (int*)cpl_calloc(info->nbentries, sizeof(int));
  if (info->stepping_phase == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the timetable STEPPING PHASE.");
      return 0;
    }
  info->intensity = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->intensity == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the timetable intensity.");
      return 0;
    }
  info->V = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->V == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the timetable V.");
      return 0;
    }
#ifdef WITH_CORRELATION
  info->correlation = (double*)cpl_calloc(info->nbentries, sizeof(double));
  if (info->correlation == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the timetable correlation.");
      return 0;
    }
#endif
  info->est_tartyp = (char*)cpl_calloc(info->nbentries, sizeof(char));
  if (info->est_tartyp == NULL)
    {
      cpl_msg_info(cpl_func,"Cannot allocate memory for the estimated TARTYP.");
      return 0;
    }
  return 1;
}

static void mat_timetable_info_free(mat_timetable_info *info)
{
  if (info->raw_tartyp != NULL)
    {
      cpl_free(info->raw_tartyp);
      info->raw_tartyp = NULL;
    }
  if (info->opt_tartyp != NULL)
    {
      cpl_free(info->opt_tartyp);
      info->opt_tartyp = NULL;
    }
  if (info->localopd1 != NULL)
    {
      cpl_free(info->localopd1);
      info->localopd1 = NULL;
    }
  if (info->localopd2 != NULL)
    {
      cpl_free(info->localopd2);
      info->localopd2 = NULL;
    }
  if (info->localopd3 != NULL)
    {
      cpl_free(info->localopd3);
      info->localopd3 = NULL;
    }
  if (info->localopd4 != NULL)
    {
      cpl_free(info->localopd4);
      info->localopd4 = NULL;
    }
  if (info->stepping_phase != NULL)
    {
      cpl_free(info->stepping_phase);
      info->stepping_phase = NULL;
    }
  if (info->intensity != NULL)
    {
      cpl_free(info->intensity);
      info->intensity = NULL;
    }
  if (info->V != NULL)
    {
      cpl_free(info->V);
      info->V = NULL;
    }
#ifdef WITH_CORRELATION
  if (info->correlation != NULL)
    {
      cpl_free(info->correlation);
      info->correlation = NULL;
    }
#endif
  if (info->est_tartyp != NULL)
    {
      cpl_free(info->est_tartyp);
      info->est_tartyp = NULL;
    }
}

static void mat_tartyp_info_init(mat_tartyp_info *info)
{
  info->stat_info = NULL;
  mat_exposure_setup_info_init(&(info->es));
  mat_raw_data_info_init(&(info->rd));
  mat_chopping_cycle_info_init(&(info->cc));
  mat_allouche_info_init(&(info->al));
  mat_timetable_info_init(&(info->tt));
}

static void mat_tartyp_info_free(mat_tartyp_info *info)
{
  if (info->stat_info != NULL)
    {
      mat_statistics_delete(info->stat_info);
      info->stat_info = NULL;
    }
  mat_raw_data_info_free(&(info->rd));
  mat_chopping_cycle_info_free(&(info->cc));
  mat_allouche_info_free(&(info->al));
  mat_timetable_info_free(&(info->tt));
}

static int mat_gcd(int a, int b)
{
  while (b != 0) {
    int t = b;
    b = a % b;
    a = t;
  }
  return a;
}

static int mat_lcm(int a, int b)
{
  int val = (a / mat_gcd(a,b)) * b;
  return val;
}

/*
  Try to extract the exposure setup (timing, stepping phases, number of sky, target and dead frames, ... from the keywords and raw data
*/
static int mat_tartyp_analyze_setup(mat_tartyp_info *info, mat_gendata *data, int flags, int excess_count, int skip_count)
{
  mat_detector          det;
  mat_imagingdata      *imgdata = data->imgdata;
  const char           *hstr = NULL;
  char                 *start_utc = NULL;
  char                 *strptr = NULL;
  long                  hour, minute;
  double                second;
  int                   iFrame;
  char                  last_tt  = 'X';
  int                   count_tt = 0;

  mat_detector_decode_raw(&det, data);
  info->es.type  = det.type;
  info->es.nx    = det.nx;
  info->es.ny    = det.ny;
  info->es.expno = mat_propertylist_get_int_default(data->keywords, "ESO TPL EXPNO", 0);
  // MJD-OBS (raw file creation date) in seconds (without day number)
  info->es.obsstart = cpl_propertylist_get_double(data->keywords, "MJD-OBS");
  info->es.obsstart -= floor(info->es.obsstart);
  info->es.obsstart *= 24.0*60.0*60.0;
  //cpl_msg_info(cpl_func, "MJD-OBS %f", info->es.obsstart);
  // get TIM-Board trigger setup, depends on the detector
  switch (info->es.type)
    {
    case MAT_HAWAII2RG_DET:
      hstr      = cpl_propertylist_get_string(data->keywords, "ESO INS TIMDL START");
      info->es.timperiod = mat_propertylist_get_double_default(data->keywords, "ESO INS TIMDL PERIOD", 0.0);
      break;
    case MAT_AQUARIUS_DET:
      hstr      = cpl_propertylist_get_string(data->keywords, "ESO INS TIMDN START");
      info->es.timperiod = mat_propertylist_get_double_default(data->keywords, "ESO INS TIMDN PERIOD", 0.0);
      break;
    default: // wrong detector
      return 0;
    }
  if (hstr == NULL)
    {
      cpl_msg_info(cpl_func, "timing: no TIM-Board used for taking the data");
      return 0;
    }
  start_utc = cpl_strdup(hstr);
  // decode TIM-Board start into MJD in seconds (without day number)
  strptr = start_utc;
  while (strptr[0] != 'T') strptr++;
  strptr++;
  hour = strtol(strptr, &strptr, 10);
  strptr++; // skip ':'
  minute = strtol(strptr, &strptr, 10);
  strptr++; // skip ':'
  second = strtod(strptr, &strptr);
  cpl_free(start_utc);
  info->es.timstart = ((double)hour)*3600.0 + ((double)minute)*60.0 + second;
  //cpl_msg_info(cpl_func, "ESO INS TIMDL/N START = %f ESO INS TIMDL/N PERIOD = %f", info->es.timstart, info->es.timperiod);
  info->es.framestart = 24.0*60.0*60.0*(imgdata->list_frame[0]->time - floor(imgdata->list_frame[0]->time));
  if (info->es.framestart < info->es.timstart) info->es.framestart += 24.0*3600.0;
  info->es.chopfreq =  mat_propertylist_get_double_default(data->keywords, "ESO ISS CHOP FREQ", 0.0);
  if (info->es.chopfreq == 0.0) return 1;  // No Chopping -> return
  info->es.chopperiod = 1.0/info->es.chopfreq;
  info->es.raw_first_index = (int)round((info->es.framestart - info->es.timstart)/info->es.timperiod);
  info->es.nbsky    = 0;
  info->es.nbtarget = 0;
  info->es.nbdead   = 0;
  info->es.stepphase_min = imgdata->list_frame[0]->stepphase;
  info->es.stepphase_max = imgdata->list_frame[0]->stepphase;
  for (iFrame = 0; iFrame < imgdata->nbframe; iFrame++)
    {
      mat_frame *frm;
      frm = imgdata->list_frame[iFrame];
      // update stepping phase
      info->es.stepphase_min = CPL_MIN(info->es.stepphase_min, frm->stepphase);
      info->es.stepphase_max = CPL_MAX(info->es.stepphase_max, frm->stepphase);
      // determine the TARTYP S/D and T/B case and the largest sky/target sequence length
      // the following code allows for nbDeadFrames == 0 (see mtoSERVER::CreateClassicTimeTable() )
      if ((frm->tartype[0] == 'S') || (frm->tartype[0] == 'D'))
	{
	  info->es.nbsky = CPL_MAX(info->es.nbsky, 1);
	  info->es.sdch = frm->tartype[0];
	}
      else if ((frm->tartype[0] == 'T') || (frm->tartype[0] == 'B'))
	{
	  info->es.nbtarget = CPL_MAX(info->es.nbtarget, 1);
	  info->es.tbch = frm->tartype[0];
	}
      else
	{
	  info->es.nbdead = CPL_MAX(info->es.nbdead, 1);
	}
      // Estimate the longest sequence of a specific target type
      if (last_tt != frm->tartype[0])
	{
	  last_tt  = frm->tartype[0];
	  count_tt = 1;
	}
      else
	{
	  count_tt++;
	  if ((last_tt == 'S') || (last_tt == 'D'))
	    {
	      info->es.nbsky = CPL_MAX(info->es.nbsky, count_tt);
	    }
	  else if ((last_tt == 'T') || (last_tt == 'B'))
	    {
	      info->es.nbtarget = CPL_MAX(info->es.nbtarget, count_tt);
	    }
	  else
	    {
	      info->es.nbdead = CPL_MAX(info->es.nbdead, count_tt);
	    }
	}
    }
  info->es.nstepphases = info->es.stepphase_max - info->es.stepphase_min + 1;
  info->es.tot_target = mat_lcm(info->es.nbtarget, 2*info->es.nstepphases);
  info->es.tot_chop = info->es.tot_target/(info->es.nbtarget);
  info->es.tot_entries = info->es.tot_chop*(info->es.nbsky + info->es.nbtarget + 2*info->es.nbdead);
  info->es.est_time_shift = (int)floor((info->es.framestart - info->es.obsstart)/info->es.timperiod) - skip_count;
  info->es.nbsky     += info->es.nbdead;
  info->es.nbtarget  += info->es.nbdead;
  // setup timing:    obs <> s start <> s period <> s start frame <> s start exp <> s possible shift <> 
  // setup chopping:  freq <> Hz period <> s frames <> nbsky <> nbtarget <> nbdead <>
  // setup timetable: tot_target <> tot_chop <> tot_entries <> stepphases <>
  // setup frames:    first <> before <> excess <> skipped <>
  cpl_msg_info(cpl_func, "setup timing: obs %.3f s start %.3f s period %.3f s start frame %.3f s start exp %.3f possible shift %d",
	       info->es.obsstart, info->es.timstart, info->es.timperiod,
	       info->es.framestart, info->es.framestart - info->es.timperiod*(double)skip_count,
	       info->es.est_time_shift);
  if (info->es.est_time_shift != 0)
    {
      cpl_msg_info(cpl_func, "setup timing: maybe %d excess frame(s) -> TIME, TARTYP, STEPPING_PHASE and LOCALOPD may need adjustment!", info->es.est_time_shift);
    }
  if (flags & TARTYP_SHOW_SETUP)
    {
      cpl_msg_info(cpl_func, "setup chopping: freq %.6f Hz period %.3f s frames %.1f nbsky %d nbtarget %d nbdead %d",
		   info->es.chopfreq, info->es.chopperiod, info->es.chopperiod/info->es.timperiod,
		   info->es.nbsky, info->es.nbtarget, info->es.nbdead);
      cpl_msg_info(cpl_func, "setup timetable: tot_target %d tot_chop %d tot_entries %d stepphases %d",
		   info->es.tot_target, info->es.tot_chop, info->es.tot_entries, info->es.nstepphases);
      cpl_msg_info(cpl_func, "setup frames: first %d before %d excess %d skipped %d",
		   info->es.raw_first_index, info->es.raw_first_index - skip_count, excess_count, skip_count);
    }
  return 1;
}

static int mat_tartyp_analyze_rawdata(mat_tartyp_info *info, mat_gendata *data, int flags)
{
  mat_imagingdetector  *imgdet  = data->imgdet;
  mat_imagingdata      *imgdata = data->imgdata;
  int                   cdist;
  int                   iRegion;
  int                   iFrame;
  int                   iOffset;
  int                   cclen;

  // find the region which is nearest to the center of the detector
  cdist = info->es.nx;
  info->rd.cr_idx = -1;
  for (iRegion = 0; iRegion < imgdet->nbregion; iRegion++)
    {
      int          dist;
      mat_region  *reg = imgdet->list_region[iRegion];
      cpl_msg_info(cpl_func, "region %d %d x %d at (%d, %d)", iRegion, reg->naxis[0], reg->naxis[1], reg->corner[0], reg->corner[1]);
      if (reg->corner[0] > info->es.nx/2) continue;
      if (reg->corner[0] + reg->naxis[0] < info->es.nx/2) continue;
      dist = info->es.ny/2 - (reg->corner[1] + reg->naxis[1]/2);
      if (dist < 0) dist = -dist;
      if (dist < cdist)
	{
	  cpl_msg_info(cpl_func, "iRegion = %d info->rd.cr_idx = %d dist = %d cdist = %d", iRegion, info->rd.cr_idx, dist, cdist);
	  info->rd.cr_idx = iRegion;
	  info->rd.cr_nx = reg->naxis[0];
	  info->rd.cr_ny = reg->naxis[1];
	  cdist   = dist;
	}
    }
  if (info->rd.cr_idx == -1)
    {
      cpl_msg_info(cpl_func, "timetable no sub-window covers the interferrometric channel");
      return 0;
    }
  if (flags & TARTYP_USE_CENTER_INTERF)
    {
      // Select the center of the interferrometric region
      info->rd.roi_left   = info->rd.cr_nx/4;
      info->rd.roi_right  = (3*info->rd.cr_nx)/4;
      info->rd.roi_bottom = info->rd.cr_ny/4;
      info->rd.roi_top    = (3*info->rd.cr_ny)/4;
    }
  else
    {
      // Select the whole interferometric region
      info->rd.roi_left   = 1;
      info->rd.roi_right  = info->rd.cr_nx;
      info->rd.roi_bottom = 1;
      info->rd.roi_top    = info->rd.cr_ny;
    }
  cpl_msg_info(cpl_func,"raw data: entries %d roi: %d = %dx%d used x = [%d .. %d] y = [%d .. %d]",
	       imgdata->nbframe,
	       info->rd.cr_idx, info->rd.cr_nx, info->rd.cr_ny,
	       info->rd.roi_left, info->rd.roi_right,
	       info->rd.roi_bottom, info->rd.roi_top);

  if (!mat_raw_data_info_alloc(&(info->rd), imgdata->nbframe)) return 0;
  // show intensity and correlation if wanted (flags)
  for (iFrame = 0; iFrame < imgdata->nbframe; iFrame++)
    {
      mat_frame *frm_curr = imgdata->list_frame[iFrame];
      cpl_image *img_curr = frm_curr->list_subwin[info->rd.cr_idx]->imgreg[0];
      int        x, y;
      // calculate the total intensity (central part of the central subwindow)
      // old implementation
      /*
      mat_statistics_reset(info->stat_info);
      for (y = info->rd.roi_bottom; y <= info->rd.roi_top; y++)
	{
	  for (x = info->rd.roi_left; x <= info->rd.roi_right; x++)
	    {
	      int    rejected;
	      double v = cpl_image_get(img_curr, x, y, &rejected);
	      if (!rejected) mat_statistics_add_value(info->stat_info, v);
	    }
	}
      mat_statistics_calc(info->stat_info);
      //info->rd.intensity[iFrame] = info->stat_info->reduced_mean;
      info->rd.intensity[iFrame] = info->stat_info->whole_total;
      */
      // new implementation
      info->rd.intensity[iFrame] = cpl_image_get_flux_window(img_curr, info->rd.roi_left, info->rd.roi_bottom, info->rd.roi_right, info->rd.roi_top);
      info->rd.intensity_offset = CPL_MIN(info->rd.intensity_offset, info->rd.intensity[iFrame]);
      // calculate the correlation of the current frame with the next frame (central part of the central subwindow)
      info->rd.correlation[iFrame] = 0.0;
      if (iFrame != (imgdata->nbframe - 1))
	{
	  cpl_image *img_next = imgdata->list_frame[iFrame + 1]->list_subwin[info->rd.cr_idx]->imgreg[0];
	  for (y = info->rd.roi_bottom; y <= info->rd.roi_top; y++)
	    {
	      double hcorr = 0.0;
	      for (x = info->rd.roi_left; x <= info->rd.roi_right; x++)
		{
		  int    rejected;
		  double vc = cpl_image_get(img_curr, x, y, &rejected);
		  double vn = cpl_image_get(img_next, x, y, &rejected);
		  hcorr += vc*vn;
		}
	      info->rd.correlation[iFrame] += hcorr;
	    }
	  info->rd.correlation[iFrame] /= (double)((info->rd.roi_top - info->rd.roi_bottom + 1)*(info->rd.roi_right - info->rd.roi_left + 1));
	}
    }
  cclen = info->es.nbsky + info->es.nbtarget;
  for (iFrame = 0; iFrame < imgdata->nbframe; iFrame++)
    {
      info->rd.intensity[iFrame] -= info->rd.intensity_offset;
    }
  for (iFrame = 0; iFrame < imgdata->nbframe; iFrame++)
    {
      if (iFrame < cclen/2)
	{
	  info->rd.V[iFrame] = 0.0;
	}
      else if (iFrame > imgdata->nbframe - cclen/2 - 1)
	{
	  info->rd.V[iFrame] = 0.0;
	}
      else
	{
	  info->rd.V[iFrame] = 0.0;
	  for (iOffset = -cclen/2; iOffset < 0; iOffset++)
	    { // intensities before the current frame are negative
	      info->rd.V[iFrame] -= info->rd.intensity[iFrame + iOffset];
	    }
	  for (iOffset = 0; iOffset < cclen/2; iOffset++)
	    { // intensities after the current frame (inclusive) are positive
	      info->rd.V[iFrame] += info->rd.intensity[iFrame + iOffset];
	    }
	}
    }
  return 1;
}

static void mat_tartyp_show_rawdata(mat_tartyp_info *info, mat_gendata *data, int flags)
{
  mat_imagingdata      *imgdata = data->imgdata;
  int                   iFrame;

  for (iFrame = 0; iFrame < imgdata->nbframe; iFrame++)
    {
      mat_frame *frm_curr = imgdata->list_frame[iFrame];
      // show the values if requested
      if ((flags & TARTYP_SHOW_INTENSITY) != 0)
	{
	  if ((flags & TARTYP_SHOW_CORRELATION) != 0)
	    {
	      cpl_msg_info(cpl_func, "frame %d time %.12f TARTYP %c SP %d I %.6E V %+.6E C with %d %g",
			   iFrame, frm_curr->time, frm_curr->tartype[0], frm_curr->stepphase,
			   info->rd.intensity[iFrame],
			   info->rd.V[iFrame],
			   iFrame + 1, info->rd.correlation[iFrame]);
	    }
	  else
	    {
	      cpl_msg_info(cpl_func, "frame %d time %.12f TARTYP %c SP %d I %.6E V %+.6E",
			   iFrame, frm_curr->time, frm_curr->tartype[0], frm_curr->stepphase,
			   info->rd.intensity[iFrame],
			   info->rd.V[iFrame]);
	    }
	}
      else
	{
	  if ((flags & TARTYP_SHOW_CORRELATION) != 0)
	    {
	      cpl_msg_info(cpl_func, "frame %d time %.12f TARTYP %c SP %d C with %d %g",
			   iFrame, frm_curr->time, frm_curr->tartype[0], frm_curr->stepphase,
			   iFrame + 1, info->rd.correlation[iFrame]);
	    }
	  else
	    { // nothing to show
	    }
	}
    }
}

static int mat_tartyp_analyze_chopping(mat_tartyp_info *info, mat_gendata *data, int flags)
{
  mat_imagingdata      *imgdata;
  int                   iCycle;
  int                   iFrame;
  int                   iEntry;
  int                   iOffset;
  double                fV   = 0.0;
  double                fS   = 0.0;
  int                   fIdx = -1;
  char                  sdch, sddch, tbch, tbdch;

  imgdata = data->imgdata;
  if (!mat_chopping_cycle_info_alloc(&(info->cc), info->es.nbsky + info->es.nbtarget))
    {
      return 0;
    }
  if (!mat_allouche_info_alloc(&(info->al), info->es.nbsky + info->es.nbtarget))
    {
      return 0;
    }
  // Three kinds of TARTYP sequences do exist:
  //   periodic shutter:   N*B, M*D
  //   minimal setup:      <space>, <space>
  //   normal observation: N*S+U+M*T+U or U+N*S+U+M*T
  // According to the docuentation, the hopper starts moving at a certain time and needs about 20 ms to reach a stable position.
  // This would lead to a sequence like U+N*S+U+M*T which is currently (20191113) _not_ used in the MATISSE OS!
  // 1. create the pattern for the first chopping cycle (original -> raw_tartyp[], 'U' changed with 'S' or 'T' -> opt_tartyp[])
  iEntry = 0;
  sdch = info->es.sdch;
  tbch = info->es.tbch;
  if (flags & TARTYP_REMOVE_TARTYP_U)
    { // replace 'U' by 'S' or 'T'
      sddch = sdch;
      tbdch = tbch;
    }
  else
    { // keep 'U'
      sddch = 'U';
      tbdch = 'U';
    }
  if (flags & TARTYP_CHOP_SSSUTTTU)
    { // TARTYP sequence: SSSUTTTU
      int currstep = 1;
      for (iFrame = 0; iFrame < info->es.nbsky - info->es.nbdead; iFrame++)
	{
	  info->cc.raw_tartyp[iEntry] = sdch;
	  info->cc.opt_tartyp[iEntry] = sdch;
	  info->cc.stepping_phase[iEntry] = currstep;
	  iEntry++;
	}
      for (iFrame = 0; iFrame < info->es.nbdead; iFrame++)
	{
	  info->cc.raw_tartyp[iEntry] = 'U';
	  info->cc.opt_tartyp[iEntry] = sddch;
	  info->cc.stepping_phase[iEntry] = currstep;
	  iEntry++;
	}
      for (iFrame = 0; iFrame < info->es.nbtarget - info->es.nbdead; iFrame++)
	{
	  info->cc.raw_tartyp[iEntry] = tbch;
	  info->cc.opt_tartyp[iEntry] = tbch;
	  info->cc.stepping_phase[iEntry] = currstep;
	  if (info->es.nstepphases != 1)
	    {
	      currstep ++;
	      if (currstep > info->es.nstepphases)
		{
		  currstep = 1;
		}
	    }
	  iEntry++;
	}
      for (iFrame = 0; iFrame < info->es.nbdead; iFrame++)
	{
	  info->cc.raw_tartyp[iEntry] = 'U';
	  info->cc.opt_tartyp[iEntry] = tbdch;
	  info->cc.stepping_phase[iEntry] = currstep;
	  iEntry++;
	}
    }
  else
    { // TARTYP sequence: USSSUTTT
      for (iFrame = 0; iFrame < info->es.nbdead; iFrame++)
	{
	  info->cc.raw_tartyp[iEntry] = 'U';
	  info->cc.opt_tartyp[iEntry] = sddch;
	  info->cc.stepping_phase[iEntry] = 1;
	  iEntry++;
	}
      for (iFrame = 0; iFrame < info->es.nbsky - info->es.nbdead; iFrame++)
	{
	  info->cc.raw_tartyp[iEntry] = sdch;
	  info->cc.opt_tartyp[iEntry] = sdch;
	  info->cc.stepping_phase[iEntry] = 1;
	  iEntry++;
	}
      for (iFrame = 0; iFrame < info->es.nbdead; iFrame++)
	{
	  info->cc.raw_tartyp[iEntry] = 'U';
	  info->cc.opt_tartyp[iEntry] = tbdch;
	  info->cc.stepping_phase[iEntry] = 1;
	  iEntry++;
	}
      for (iFrame = 0; iFrame < info->es.nbtarget - info->es.nbdead; iFrame++)
	{
	  info->cc.raw_tartyp[iEntry] = tbch;
	  info->cc.opt_tartyp[iEntry] = tbch;
	  if (info->es.nstepphases == 1)
	    {
	      info->cc.stepping_phase[iEntry] = 1;
	    }
	  else
	    {
	      info->cc.stepping_phase[iEntry] = 1 + iFrame;
	    }
	  iEntry++;
	}
    }
  // 2. Find the first occurrence of the chopping cycle pattern in the raw data
  info->cc.first = -1;
  for (iFrame = 0; iFrame < (2*info->cc.nbentries + 1); iFrame++)
    {
      cpl_boolean found = CPL_TRUE;
      for (iEntry = 0; iEntry < info->cc.nbentries; iEntry++)
	{
	  found = found && (imgdata->list_frame[iFrame + iEntry]->tartype[0] == info->cc.raw_tartyp[iEntry]);
	  //found = found && (imgdata->list_frame[iFrame + iEntry]->stepphase  == info->cc.stepping_phase[iEntry]); // needed to find the first timetable entry
	  if (!found) break;
	}
      if (found)
	{
	  info->cc.first = iFrame;
	  break;
	}
    }
  if (info->cc.first == -1)
    {
      cpl_msg_info(cpl_func, "cannot find the first occurence of the chopping cycle template");
      for (iEntry = 0; iEntry < info->cc.nbentries; iEntry++)
	{
	  cpl_msg_info(cpl_func, "chopping table: %d TARTYP %c STEPPING PHASE %d", iEntry, info->cc.raw_tartyp[iEntry], info->cc.stepping_phase[iEntry]);
	}
      return 0;
    }
  info->cc.cycles = (info->rd.nbentries - info->cc.first)/info->cc.nbentries;
  // the following line of code is weird but correct!
  info->cc.phase = (info->cc.nbentries - info->cc.first%info->cc.nbentries)%info->cc.nbentries;
  // calculate an average image intensity for each frame inside a chopping cycle
  for (iEntry = 0; iEntry < info->cc.nbentries; iEntry++)
    {
      info->cc.intensity[iEntry] = 0.0;
      for (iCycle = 0; iCycle < info->cc.cycles; iCycle++)
	{
	  info->cc.intensity[iEntry] += info->rd.intensity[info->cc.first + iCycle*info->cc.nbentries + iEntry];
	}
      info->cc.intensity[iEntry] /= (double)info->cc.cycles;
    }
  info->al.cycles = info->rd.nbentries/info->al.nbentries;
  for (iEntry = 0; iEntry < info->al.nbentries; iEntry++)
    {
      info->al.raw_tartyp[iEntry] = imgdata->list_frame[iEntry]->tartype[0];
      info->al.intensity[iEntry] = 0.0;
      for (iCycle = 0; iCycle < info->al.cycles; iCycle++)
	{
	  info->al.intensity[iEntry] += info->rd.intensity[iCycle*info->al.nbentries + iEntry];
	}
      info->al.intensity[iEntry] /= (double)info->al.cycles;
    }
  // Calculate triangle (Allouche + Lagarde)
  for (iEntry = 0; iEntry < info->cc.nbentries; iEntry++)
    {
      info->cc.V[iEntry] = 0.0;
      for (iOffset = 0; iOffset < info->cc.nbentries; iOffset++)
	{
	  if (iOffset < info->cc.nbentries/2)
	    {
	      info->cc.V[iEntry] += info->cc.intensity[(iEntry + iOffset)%info->cc.nbentries];
	    }
	  else
	    {
	      info->cc.V[iEntry] -= info->cc.intensity[(iEntry + iOffset)%info->cc.nbentries];
	    }
	}
    }
  // Estimate a possible shift based on cc.V only (Allouche + Lagarde)
  for (iEntry = 0; iEntry < info->cc.nbentries; iEntry++)
    {
      if (info->cc.V[iEntry] > fV)
	{
	  fV = info->cc.V[iEntry];
	  fIdx = iEntry;
	}
    }
  if (fIdx == -1)
    {
      cpl_msg_info(cpl_func, "cannot detect a V peak");
      flags &= ~TARTYP_CHANGE_TARTYP;
    }
  info->cc.est_V_shift = fIdx - info->cc.nbentries/2;
  for (iEntry = 0; iEntry < info->al.nbentries; iEntry++)
    {
      info->al.V[iEntry] = 0.0;
      for (iOffset = 0; iOffset < info->al.nbentries; iOffset++)
	{
	  if (iOffset < info->al.nbentries/2)
	    {
	      info->al.V[iEntry] += info->al.intensity[(iEntry + iOffset)%info->al.nbentries];
	    }
	  else
	    {
	      info->al.V[iEntry] -= info->al.intensity[(iEntry + iOffset)%info->al.nbentries];
	    }
	}
    }
  fV = 0.0;
  for (iEntry = 0; iEntry < info->al.nbentries; iEntry++)
    {
      if ((info->al.raw_tartyp[iEntry] == 'U') && (info->al.raw_tartyp[(iEntry + 1)%info->al.nbentries] == tbch))
	{
	  info->al.est_V_shift = (iEntry + 1)%info->al.nbentries;
	  break;
	}
    }
  fIdx = -1;
  for (iEntry = 0; iEntry < info->al.nbentries; iEntry++)
    {
      if (info->al.V[iEntry] > fV)
	{
	  fV = info->al.V[iEntry];
	  fIdx = iEntry;
	}
    }
  if (fIdx == -1)
    {
      cpl_msg_info(cpl_func, "cannot detect a V peak");
      flags &= ~TARTYP_CHANGE_TARTYP;
    }
  info->al.est_V_shift = fIdx - info->al.est_V_shift;
  // Fit a square wave with different phase (sky - target transition) of the intensities
  for (iEntry = 0; iEntry < info->cc.nbentries; iEntry++)
    {
      double chi2 = 0.0;
      // Estimate the average target (info->cc.nbentries/2 frames after iEntry (including)) intensity and the chi2
      mat_statistics_reset(info->stat_info);
      for (iOffset = 0; iOffset < info->cc.nbentries/2; iOffset++)
	{
	  mat_statistics_add_value(info->stat_info, info->cc.intensity[(iEntry + iOffset)%info->cc.nbentries]);
	}
      mat_statistics_calc(info->stat_info);
      info->cc.targetS[iEntry] = info->stat_info->whole_mean;
      chi2 += info->stat_info->whole_variance;
      mat_statistics_reset(info->stat_info);
      for (iOffset = info->cc.nbentries/2; iOffset < info->cc.nbentries; iOffset++)
	{
	  mat_statistics_add_value(info->stat_info, info->cc.intensity[(iEntry + iOffset)%info->cc.nbentries]);
	}
      mat_statistics_calc(info->stat_info);
      info->cc.skyS[iEntry] = info->stat_info->whole_mean;
      chi2 += info->stat_info->whole_variance;
      info->cc.S[iEntry] = chi2;
    }
  // Estimate a possible shift based on cc.S only
  fIdx = -1;
  for (iEntry = 0; iEntry < info->cc.nbentries; iEntry++)
    {
      if (info->cc.skyS[iEntry] > info->cc.targetS[iEntry]) continue; // mean sky intensity > mean target intensity
      if (fIdx == -1)
	{
	  fS   = info->cc.S[iEntry];
	  fIdx = iEntry;
	}
      else if (info->cc.S[iEntry] < fS)
	{
	  fS   = info->cc.S[iEntry];
	  fIdx = iEntry;
	}
    }
  if (fIdx == -1)
    {
      cpl_msg_info(cpl_func, "cannot detect a S peak");
    }
  else
    {
      for (iEntry = 0; iEntry < info->cc.nbentries; iEntry++)
	{
	  info->cc.relint[iEntry] = (info->cc.intensity[iEntry] - info->cc.skyS[fIdx])/(info->cc.targetS[fIdx] - info->cc.skyS[fIdx]);
	}
    }
  info->cc.est_S_shift = fIdx - info->cc.nbentries/2;
  // estimate a new TARTYP based on the calculate shift
  for (iFrame = 0; iFrame < info->cc.nbentries; iFrame++)
    {
      info->cc.est_tartyp[iFrame] = info->cc.opt_tartyp[(iFrame + info->cc.nbentries - info->cc.est_V_shift)%info->cc.nbentries];
      info->al.est_tartyp[iFrame] = info->al.raw_tartyp[(iFrame + info->al.nbentries - info->al.est_V_shift)%info->al.nbentries];
    }
  return 1;
}

static void mat_tartyp_show_chopping(mat_tartyp_info *info, int flags)
{
  int  iEntry;

  cpl_msg_info(cpl_func, "chopping: entries %d first %d cycles %d %d phase %d shift %d %d %d",
	       info->cc.nbentries, info->cc.first, info->cc.cycles, info->al.cycles, info->cc.phase,
	       info->cc.est_V_shift, info->cc.est_S_shift, info->al.est_V_shift);
  if (flags & TARTYP_SHOW_CHOPPING)
    {
      for (iEntry = 0; iEntry < info->cc.nbentries; iEntry++)
	{
	  /*
	  cpl_msg_info(cpl_func, "chopping table: %2d TARTYP RAW %c %c OPT %c INT %c EST %c %c SP %2d I %.6E %+.6f %.6E V %+.6E %+.6E S %.6E %.6E %.6E",
		       iEntry,
		       info->cc.raw_tartyp[iEntry], info->al.raw_tartyp[iEntry],
		       info->cc.opt_tartyp[iEntry], info->cc.int_tartyp[iEntry],
		       info->cc.est_tartyp[iEntry], info->al.est_tartyp[iEntry],
		       info->cc.stepping_phase[iEntry],
		       info->cc.intensity[iEntry] - info->cc.sky_tar_sep, info->cc.relint[iEntry], info->al.intensity[iEntry],
		       info->cc.V[iEntry], info->al.V[iEntry],
		       info->cc.S[iEntry], info->cc.skyS[iEntry], info->cc.targetS[iEntry]);
	  */
	  cpl_msg_info(cpl_func, "chopping table: %2d TARTYP %c %c %c %.6E %+7.4f V %+.6E S %.6E %.6E %.6E V %c %c %.6E %+.6E",
		       iEntry,
		       info->cc.raw_tartyp[iEntry], info->cc.opt_tartyp[iEntry], info->cc.est_tartyp[iEntry],
		       info->cc.intensity[iEntry], info->cc.relint[iEntry],
		       info->cc.V[iEntry],
		       info->cc.S[iEntry], info->cc.skyS[iEntry], info->cc.targetS[iEntry],
		       info->al.raw_tartyp[iEntry], info->al.est_tartyp[iEntry], info->al.intensity[iEntry], info->al.V[iEntry]
		       );
	}
    }
}


static int mat_tartyp_analyze_timetable(mat_tartyp_info *info, mat_gendata *data, int flags)
{
  mat_imagingdata      *imgdata;
  int                   iCycle;
  int                   iFrame;
  int                   iEntry;
  int                   nEntries;
  int                   iOffset;
#ifdef WITH_CORRELATION
  cpl_imagelist        *list_img = NULL;
  cpl_image            *img_curr = NULL;
  cpl_image            *img_next = NULL;
#endif
  double                fV   = 0.0;
  int                   fIdx = -1;

  imgdata = data->imgdata;
  // 1. Try to find the first chopping cycle in the raw data (chopping template given in info->cc)
  info->tt.first  = -1;
  info->tt.second = -1;
  for (iFrame = 0; iFrame < (info->rd.nbentries - info->cc.nbentries); iFrame++)
    {
      cpl_boolean found = CPL_TRUE;
      for (iEntry = 0; iEntry < info->cc.nbentries; iEntry++)
	{
	  found = found && (imgdata->list_frame[iFrame + iEntry]->tartype[0] == info->cc.raw_tartyp[iEntry]);
	  found = found && (imgdata->list_frame[iFrame + iEntry]->stepphase  == info->cc.stepping_phase[iEntry]);
	  if (!found) break;
	}
      if (found)
	{
	  info->tt.first = iFrame;
	  break;
	}
    }
  if (info->tt.first == -1)
    {
      cpl_msg_info(cpl_func, "cannot find the first occurence of the chopping cycle template for the timetable");
      return 0;
    }
  // 2. Try to find another first chopping cycle after the first one, index difference gives timetable size
  for (iFrame = info->tt.first + info->cc.nbentries; iFrame < (info->rd.nbentries - info->cc.nbentries); iFrame += info->cc.nbentries)
    {
      cpl_boolean found = CPL_TRUE;
      for (iEntry = 0; iEntry < info->cc.nbentries; iEntry++)
	{
	  found = found && (imgdata->list_frame[iFrame + iEntry]->tartype[0] == info->cc.raw_tartyp[iEntry]);
	  found = found && (imgdata->list_frame[iFrame + iEntry]->stepphase  == info->cc.stepping_phase[iEntry]);
	  if (!found) break;
	}
      if (found)
	{
	  info->tt.second = iFrame;
	  break;
	}
    }
  if (info->tt.second == -1)
    {
      cpl_msg_info(cpl_func, "cannot find the second occurence of the chopping cycle template for the timetable");
      return 0;
    }
  nEntries = info->tt.second - info->tt.first;
  // 3. Step: allocate memory
  if (!mat_timetable_info_alloc(&(info->tt), nEntries))
    {
      return 0;
    }
  info->tt.cycles = (info->rd.nbentries - info->tt.first)/info->tt.nbentries;
  // the following line of code is weird but correct!
  info->tt.phase = (info->tt.nbentries - info->tt.first%info->tt.nbentries)%info->tt.nbentries;
  // 4. Extract the timetable from the raw data
  for (iEntry = 0; iEntry < info->tt.nbentries; iEntry++)
    {
      info->tt.raw_tartyp[iEntry]     = imgdata->list_frame[info->tt.first + iEntry]->tartype[0];
      info->tt.opt_tartyp[iEntry]     = info->cc.opt_tartyp[iEntry%info->cc.nbentries];
      info->tt.localopd1[iEntry]      = imgdata->list_frame[info->tt.first + iEntry]->localopd[0];
      info->tt.localopd2[iEntry]      = imgdata->list_frame[info->tt.first + iEntry]->localopd[1];
      info->tt.localopd3[iEntry]      = imgdata->list_frame[info->tt.first + iEntry]->localopd[2];
      info->tt.localopd4[iEntry]      = imgdata->list_frame[info->tt.first + iEntry]->localopd[3];
      info->tt.stepping_phase[iEntry] = imgdata->list_frame[info->tt.first + iEntry]->stepphase;
    }
  // 5. Step: get for each timetable entry a median intensity from the center of the interferrometric channel
  // calculate the average image intensity for each timetable entry
  for (iEntry = 0; iEntry < info->tt.nbentries; iEntry++)
    {
      info->tt.intensity[iEntry] = 0.0;
      for (iCycle = 0; iCycle < info->tt.cycles; iCycle++)
	{
	  info->tt.intensity[iEntry] += info->rd.intensity[info->tt.first + iCycle*info->tt.nbentries + iEntry];
	}
      info->tt.intensity[iEntry] /= (double)info->tt.cycles;
    }
  // Calculate triangle (Allouche + Lagarde)
  for (iEntry = 0; iEntry < info->tt.nbentries; iEntry++) // tt.nbentries is correct (fill the whole timetable)
    {
      info->tt.V[iEntry] = 0.0;
      for (iOffset = -info->cc.nbentries/2; iOffset < info->cc.nbentries/2; iOffset++) // cc.nbentries is correct (span one chopping cycle)
	{
	  if (iOffset >= 0)
	    {
	      info->tt.V[iEntry] += info->tt.intensity[(iEntry + iOffset)%info->tt.nbentries];
	    }
	  else
	    {
	      info->tt.V[iEntry] -= info->tt.intensity[(iEntry + iOffset + info->tt.nbentries)%info->tt.nbentries]; // '+ info->tt.nbentries' ensures positive index for iOffset < 0
	    }
	}
    }
  // Estimate a possible shift based on cc.V only (Allouche + Lagarde)
  for (iEntry = 0; iEntry < info->cc.nbentries; iEntry++) // we look for the V maximum in the first chopping cycle only
    {
      if (info->tt.V[iEntry] > fV)
	{
	  fV = info->tt.V[iEntry];
	  fIdx = iEntry;
	}
    }
  if (fIdx == -1)
    {
      cpl_msg_info(cpl_func, "cannot detect a V peak");
      flags &= ~TARTYP_CHANGE_TARTYP;
    }
  info->tt.est_V_shift = fIdx - info->cc.nbentries/2;
  cpl_msg_info(cpl_func,"timetable: entries %d first %d second %d cycles %d phase %d shift %d",
	       info->tt.nbentries, info->tt.first, info->tt.second, info->tt.cycles, info->tt.phase, info->tt.est_V_shift);
  // estimate a new TARTYP based on the calculate shift
  for (iFrame = 0; iFrame < info->tt.nbentries; iFrame++)
    {
      info->tt.est_tartyp[iFrame] = info->cc.opt_tartyp[(iFrame + info->tt.nbentries - info->tt.est_V_shift)%info->cc.nbentries];
    }
#ifdef WITH_CORRELATION
  // create for each timetable entry a median image and calculate the correlation between two successive median images
  img_curr = NULL;
  img_next = NULL;
  for (iEntry = 0; iEntry < info->tt.nbentries; iEntry++)
    {
      int x, y;
      double correlation = 0.0;
      // Create the median image for the current timetable entry if it does not exist (normally only for the first timetable entry)
      if (img_curr == NULL)
	{
	  list_img = cpl_imagelist_new();
	  if (list_img == NULL)
	    {
	      cpl_msg_error(cpl_func, "Cannot allocate image list (list_img) for median image for a timetable entry.");
	      break;
	    }
	  for (iFrame = 0; iFrame < imgdata->nbframe; iFrame++)
	    {
	      cpl_image *img;
	      if ((iFrame + info->tt.phase)%info->tt.nbentries != iEntry) continue;
	      img = imgdata->list_frame[iFrame]->list_subwin[info->rd.cr_idx]->imgreg[0];
	      cpl_imagelist_set(list_img, img, cpl_imagelist_get_size(list_img));
	    }
	  img_curr = cpl_imagelist_collapse_median_create(list_img);
	  cpl_imagelist_unwrap(list_img);
	  if (img_curr == NULL)
	    {
	      cpl_msg_error(cpl_func, "Cannot create the median image (img_curr) from list_img");
	      break;
	    }
	}
      // Create the median image for the next timetable entry (needed for each iEntry)
      if (img_next == NULL)
	{
	  list_img = cpl_imagelist_new();
	  if (list_img == NULL)
	    {
	      cpl_msg_error(cpl_func, "Cannot allocate image list (list_img) for median image for a timetable entry.");
	      break;
	    }
	  for (iFrame = 0; iFrame < imgdata->nbframe; iFrame++)
	    {
	      cpl_image *img;
	      if ((iFrame + info->tt.phase)%info->tt.nbentries != (iEntry + 1)%info->tt.nbentries) continue;
	      img = imgdata->list_frame[iFrame]->list_subwin[info->rd.cr_idx]->imgreg[0];
	      cpl_imagelist_set(list_img, img, cpl_imagelist_get_size(list_img));
	    }
	  img_next = cpl_imagelist_collapse_median_create(list_img);
	  cpl_imagelist_unwrap(list_img);
	  if (img_next == NULL)
	    {
	      cpl_msg_error(cpl_func, "Cannot create the median image (img_next) from list_img");
	      break;
	    }
	}
      // Calculate the correlation between two median images
      for (y = 1; y <= info->rd.cr_ny; y++)
	{
	  double hcorr = 0.0;
	  for (x = 1; x <= info->rd.cr_nx; x++)
	    {
	      int    rejected;
	      double vc = cpl_image_get(img_curr, x, y, &rejected);
	      double vn = cpl_image_get(img_next, x, y, &rejected);
	      hcorr += vc*vn;
	    }
	  correlation += hcorr;
	}
      correlation /= (double)(info->rd.cr_nx*info->rd.cr_ny);
      info->tt.correlation[iEntry] = correlation;
      if (img_curr != NULL)
	{
	  cpl_image_delete(img_curr);
	  img_curr = NULL;
	}
      if (img_next != NULL)
	{
	  img_curr = img_next;
	  img_next = NULL;
	}
    }
  if (img_curr != NULL)
    {
      cpl_image_delete(img_curr);
      img_curr = NULL;
    }
  if (img_next != NULL)
    {
      cpl_image_delete(img_next);
      img_next = NULL;
    }
  if (flags & TARTYP_SHOW_TIMETABLE)
    {
      for (iEntry = 0; iEntry < info->tt.nbentries; iEntry++)
	{
	  cpl_msg_info(cpl_func, "timetable: %3d TARTYP RAW %c OPT %c EST %c SP %2d OPD %6.3f %6.3f %6.3f %6.3f I %.6E C %.6E V %+6E",
		       iEntry,
		       info->tt.raw_tartyp[iEntry], info->tt.opt_tartyp[iEntry], info->tt.est_tartyp[iEntry],
		       info->tt.stepping_phase[iEntry],
		       info->tt.localopd1[iEntry], info->tt.localopd2[iEntry], info->tt.localopd3[iEntry], info->tt.localopd4[iEntry],
		       info->tt.intensity[iEntry], info->tt.correlation[iEntry], info->tt.V[iEntry]);
	}
    }
#else
  if (flags & TARTYP_SHOW_TIMETABLE)
    {
      for (iEntry = 0; iEntry < info->tt.nbentries; iEntry++)
	{
	  cpl_msg_info(cpl_func, "timetable: %3d TARTYP RAW %c OPT %c EST %c SP %2d OPD %6.3f %6.3f %6.3f %6.3f I %.6E V %+6E",
		       iEntry,
		       info->tt.raw_tartyp[iEntry], info->tt.opt_tartyp[iEntry], info->tt.est_tartyp[iEntry],
		       info->tt.stepping_phase[iEntry],
		       info->tt.localopd1[iEntry], info->tt.localopd2[iEntry], info->tt.localopd3[iEntry], info->tt.localopd4[iEntry],
		       info->tt.intensity[iEntry],
		       info->tt.V[iEntry]);
	}
    }
#endif
  return 1;
}

static int mat_tartyp_change(mat_tartyp_info *info, mat_gendata *data, int flags, int excess_count)
{
  mat_imagingdata      *imgdata;
  int                   iFrame;
  int                   corr_first_index;
  double                timstart_mjd;

  // raw_first_index  : frame index for the first frame in the raw data (framestart - timstart)/timperiod
  // corr_first_index : correct frame index for the first frame in the raw data (raw_first_index - est_shift)
  imgdata = data->imgdata;
  if (excess_count == MAT_EXCESS_COUNT_UNSET)
    { // Use the calculated shift as number of excess frames 
      if (flags & TARTYP_USE_WAVE_FIT_SHIFT)
	{ // use shift calculated from rectangular wave fit
	  excess_count = info->cc.est_S_shift;
	}
      else if (flags & TARTYP_USE_TIME_SHIFT)
	{ // use shift calculated from timing
	  excess_count = info->es.est_time_shift;
	}
      else
	{ // use the shift calculated from V
	  excess_count = info->cc.est_V_shift;
	}
    }
  if ((excess_count == 0) && ((flags & TARTYP_REMOVE_TARTYP_U) == 0)) return 1; // do nothing, since no shift detected or provided as plugin option
  corr_first_index = info->es.raw_first_index - excess_count; // correct frame index of the first raw frame
  // list_frame[0]->time : MJD [d]
  // info->es.timperiod  : [s]
  timstart_mjd     = round(24.0*3600.0*imgdata->list_frame[0]->time - info->es.raw_first_index*info->es.timperiod)/(24.0*3600.0); // MJD of the TIM-Board trigger start, granularity 1 s
  cpl_msg_info(cpl_func, "changing TIME, TARTYP, ... based on raw_first_index %d excess_count %d -> corr_first_index %d TIM-MJD %.12f",
	       info->es.raw_first_index, excess_count, corr_first_index, timstart_mjd);
  // change for each raw frame the requested values (TIME, TARTYP, LOCALOPD, STEPPING_PHASE)
  for (iFrame = 0; iFrame < imgdata->nbframe; iFrame++)
    {
      mat_frame *frame = imgdata->list_frame[iFrame];
      int fidx  = corr_first_index + iFrame;          // correct frame index for the current frame
      int ccidx = fidx%info->cc.nbentries;            // index of the current raw frame into the chopping cycle
      int ttidx = fidx%info->tt.nbentries;            // index of the current raw frame into the timetable
      if (flags & TARTYP_CHANGE_TARTYP) // paranoia
	{
	  frame->tartype[0] = info->cc.opt_tartyp[ccidx]; // cc.opt_tartyp contains optional 'U' -> 'S'/'D'/'T'/'B'
	}
      if (flags & TARTYP_CHANGE_TIME)
	{
	  frame->time = timstart_mjd + info->es.timperiod/24.0/3600.0*(double)fidx;
	}
      if (flags & TARTYP_CHANGE_OPD)
	{
	  frame->localopd[0] = info->tt.localopd1[ttidx];
	  frame->localopd[1] = info->tt.localopd2[ttidx];
	  frame->localopd[2] = info->tt.localopd3[ttidx];
	  frame->localopd[3] = info->tt.localopd4[ttidx];
	  frame->stepphase   = info->tt.stepping_phase[ttidx];
	}
      if (flags & TARTYP_SHOW_RESULT)
	{
	  cpl_msg_info(cpl_func, "frame %4d time %.12f TARTYP %c SP %2d OPD %6.3f %6.3f %6.3f %6.3f I %.6E",
		       iFrame, frame->time, frame->tartype[0], frame->stepphase,
		       frame->localopd[0], frame->localopd[1], frame->localopd[2], frame->localopd[3],
		       info->rd.intensity[iFrame]);
	}
    }
  return 1;
}

static void mat_tartyp_result_init(mat_tartyp_result *result)
{
  if (result == NULL) return;
  result->expno               = 0;
  result->timstart            = 0.0;
  result->timperiod           = 0.0;
  result->framestart          = 0.0;
  result->chopfreq            = 0.0;
  result->delay               = 0.0;
  result->nbsky               = 0;
  result->nbtarget            = 0;
  result->nbdead              = 0;
  result->est_time_shift      = 0;
  result->est_V_shift         = 0;
  result->est_S_shift         = 0;
  result->est_N_shift         = 0;
  result->sky_relint[0]       = 0.0;
  result->sky_relint[1]       = 0.0;
  result->sky_relint[2]       = 0.0;
  result->sky_relint[3]       = 0.0;
  result->tar_relint[0]       = 1.0;
  result->tar_relint[1]       = 1.0;
  result->tar_relint[2]       = 1.0;
  result->tar_relint[3]       = 1.0;
}

static void mat_tartyp_result_set(mat_tartyp_result *result, mat_tartyp_info *info)
{
  if (result == NULL) return;
  // copy the exposure properties into the result
  result->expno               = info->es.expno;
  result->timstart            = info->es.timstart;
  result->timperiod           = info->es.timperiod;
  result->framestart          = info->es.framestart;
  result->chopfreq            = info->es.chopfreq;
  result->delay               = info->es.framestart - info->es.timstart;
  result->nbsky               = info->es.nbsky;
  result->nbtarget            = info->es.nbtarget;
  result->nbdead              = info->es.nbdead;
  result->est_time_shift      = info->es.est_time_shift;
  // copy the chopping properties into the result
  result->est_V_shift         = info->cc.est_V_shift;
  result->est_S_shift         = info->cc.est_S_shift;
  result->est_N_shift         = info->al.est_V_shift;
  if ((info->cc.relint != NULL) && (info->cc.nbentries >= 4))
    {
      // First two sky frames (after moving the chopper from target to sky)
      result->sky_relint[0] = info->cc.relint[0];
      result->sky_relint[1] = info->cc.relint[1];
      // last two sky frames (before moving the chopper from sky to target)
      result->sky_relint[2] = info->cc.relint[info->cc.nbentries/2 - 2];
      result->sky_relint[3] = info->cc.relint[info->cc.nbentries/2 - 1];
      // first two target frames (after moving the chopper from sky to target)
      result->tar_relint[0] = info->cc.relint[info->cc.nbentries/2 + 0];
      result->tar_relint[1] = info->cc.relint[info->cc.nbentries/2 + 1];
      // last two target frames (before moving the chopper from target to sky)
      result->tar_relint[2] = info->cc.relint[info->cc.nbentries - 2];
      result->tar_relint[3] = info->cc.relint[info->cc.nbentries - 1];
    }
}

void mat_tartyp_check(mat_gendata *data, int flags, int excess_count, int skip_count, mat_tartyp_result *result)
{
  mat_tartyp_info       info;
  mat_imagingdata      *imgdata;
  int                   iFrame;

  imgdata = data->imgdata;
  // initialize info structure
  mat_tartyp_result_init(result);
  mat_tartyp_info_init(&info);
  // ===========================================================
  // analyze the raw file setup (timing, ...) and check if chopping was used at all, this is done even if no flag is set at all
  if (!mat_tartyp_analyze_setup(&info, data, flags, excess_count, skip_count))
    {
      mat_tartyp_result_set(result, &info);
      mat_tartyp_info_free(&info);
      return;
    }
  if (info.es.chopfreq == 0.0)
    {
      cpl_msg_info(cpl_func, "no chopping, therefore no TARTYP change necessary");
      mat_tartyp_result_set(result, &info);
      mat_tartyp_info_free(&info);
      return;
    }
  // Check if anything should be done at all (showing timing information is always done), do not eat up precious runtime if nothing is requested
  if ((flags & (TARTYP_SHOW_INTENSITY | TARTYP_SHOW_CORRELATION | TARTYP_ESTIMATE_TARTYP)) == 0)
    {
      mat_tartyp_result_set(result, &info);
      mat_tartyp_info_free(&info);
      return;
    }
  info.stat_info = mat_statistics_realloc(info.stat_info, 1024*1024, 1.5);
  // ===========================================================
  // analyze the raw data (calculating intensity per frame and correlation between successive frames and show the result (optional), check chopping again
  if (!mat_tartyp_analyze_rawdata(&info, data, flags))
    {
      mat_tartyp_result_set(result, &info);
      mat_tartyp_info_free(&info);
      return;
    }
  if (imgdata->nbframe == 1)
    { // Only one frame -> nothing can be made
      cpl_msg_info(cpl_func, "only one frame, therefore nothing to do");
      mat_tartyp_result_set(result, &info);
      mat_tartyp_info_free(&info);
      return;
    }
  if ((info.es.nbsky == 0) || (info.es.nbtarget == 0))
    {
      cpl_msg_info(cpl_func, "No chopping detected: nsky %d ntarget %d ndead %d", info.es.nbsky, info.es.nbtarget, info.es.nbdead);
      mat_tartyp_result_set(result, &info);
      mat_tartyp_info_free(&info);
      return;
    }
  for (iFrame = 1; iFrame < imgdata->nbframe; iFrame++)
    {
      if (imgdata->list_frame[0]->tartype[0] != imgdata->list_frame[iFrame]->tartype[0]) break;
    }
  if (iFrame == imgdata->nbframe)
    { // All frames have the same TARTYP
      cpl_msg_info(cpl_func, "only one TARTYP used, therefore nothing to do");
      mat_tartyp_result_set(result, &info);
      mat_tartyp_info_free(&info);
      return;
    }
  if ((flags & TARTYP_ESTIMATE_TARTYP) == 0)
    { // Without estimating the TARTYP any modification makes no sense
      mat_tartyp_result_set(result, &info);
      mat_tartyp_info_free(&info);
      return;
    }
  // ===========================================================
  // analyze chopping and create a template for the first chopping cycle
  if (!mat_tartyp_analyze_chopping(&info, data, flags))
    {
      mat_tartyp_result_set(result, &info);
      mat_tartyp_info_free(&info);
      return;
    }
  mat_tartyp_show_rawdata(&info, data, flags);
  mat_tartyp_show_chopping(&info, flags);
  // ===========================================================
  // analyze the timetable
  if (!mat_tartyp_analyze_timetable(&info, data, flags))
    {
      mat_tartyp_result_set(result, &info);
      mat_tartyp_info_free(&info);
      return;
    }
  if ((flags & TARTYP_CHANGE_TARTYP) == 0)
    { // Without changing the TARTYP any modification makes no sense
      mat_tartyp_result_set(result, &info);
      mat_tartyp_info_free(&info);
      return;
    }
  // ===========================================================
  // change TIME, TRATYP, LOCALOPD and/or STEPPING_PHASE according to the estimated shift
  mat_tartyp_result_set(result, &info);
  mat_tartyp_change(&info, data, flags, excess_count);
  mat_tartyp_info_free(&info);
}
