/* $Id: mat_compute_cphase.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_compute_cphase.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#include <string.h>
#endif
#include <stdio.h>
/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/
#include <math.h>

#include "mat_compute_cphase.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
  Define
  ----------------------------------------------------------------------------*/
/*-----------------------------------------------------------------------------
  Functions prototypes
  ----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
   @ingroup signal
   @brief Compute the bispectrum and the closure phase
   @brief from a mat_corrflux structure. 
   @brief This function fills a mat_oifits with the computed closure phases.
   @param *corrFlux           contains the mat_corrflux structure
   @param *oifits             contains the mat_oifits structure in which the 
   @param *nofringe           contains the NOFRINGE mat_corrflux structure
   closure phases are stored
   @return 0 if no error 
*/
int mat_compute_cphase(mat_corrflux *corrFlux,mat_oifits *oifits,mat_corrflux *nofringe,
		       mat_oifits *oivis, mat_oiopdwvpo *oiopd) {
  int k=0;
  int l=0;
  /* int n=0; */
  int m=0;
  int cptWave=0;
  int uPixelMin=0;
  int uPixelMax=0;
  int vPixelMin=0;
  int vPixelMax=0;
  int uPixelMin0=0;
  int vPixelMin0=0;
  int pis=0;
  int numDet=0;
  cpl_image *foore=NULL;
  cpl_image *fooim=NULL;
  cpl_vector *dispCoef=NULL;
  double lambda=0.;
  double dOverLambda=0.;
  double aVal=0.;
  double bVal=0.;
  double cVal=0.;
  double dVal=0.;
  double eVal=0.;
  double fVal=0.;
  /* double cphi=0.; */
  double val=0.;
  double val2=0.;
  cpl_image **biSpecReal=NULL;
  cpl_image **biSpecImag=NULL;

  cpl_image **biSpecRealNoFringe=NULL;
  cpl_image **biSpecImagNoFringe=NULL;

  cpl_vector *fit=NULL;
  cpl_vector *vecw=NULL;
  cpl_vector *vecy=NULL;
  cpl_vector *vecz=NULL;
  char *detName=NULL;
  double num=0.;
  char kwd[64];
  int resolution=0;
  char *specResData=NULL;
  char    *fil  =NULL;
  double posFringePeak=0.;
  int iReg=0;
  int iFrame=0;
  int iWlen=0;
  int iTrip=0;
  int dimSpatial=0;
  int dimSpectral=0;
  int corner=0;
  int nbFrameOverPhot=0;
  int nbFrameOverPhotNoFringe=0;
  cpl_image *ftReal=NULL;
  cpl_image *ftImag=NULL;
  double avgz=0.;
  double avgy=0.;
  double limit=0.;
  int ix=0;
  int iy=0;
  int uPixelSize=0;
  int vPixelSize=0;
  int nbFrameTarget=0;
  int nbFrameTargetNoFringe=0;
  int      expno = 0;
  char     *output = NULL;
    

    
  // Check input parameters
  mat_assert((corrFlux!=NULL),CPL_ERROR_NULL_INPUT, "no mat_corrflux argument given");
  mat_assert((oifits!=NULL),CPL_ERROR_NULL_INPUT,"no mat_oifits argument given");


  if(oivis == NULL || oiopd == NULL)
    {
  
      expno = cpl_propertylist_get_int(corrFlux->keywords,"ESO TPL EXPNO");

      // Detector determination
      detName=(char *)
	cpl_propertylist_get_string(corrFlux->keywords,"ESO DET CHIP NAME");
      if (detName == NULL) {
	cpl_msg_error(cpl_func,
		      "no ESO DET CHIP NAME keyword in frame");
	return -1;
      }
      specResData=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS DIN ID");
      if(!strcmp(detName, "AQUARIUS")) {
	numDet=2;
	if (!strcmp(specResData,"LOW")) {
	  resolution=1;
	}
	if (!strcmp(specResData,"HIGH")) {
	  resolution=2;
	}
      }
      else {
	numDet=1;
	specResData=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS DIL ID");
	fil=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS FIL NAME");
	if (!strcmp(specResData,"LOW")) {
	  resolution=1;
	}
	if (!strcmp(specResData,"MED")) {
	  resolution=2;
	}
	if (!strcmp(specResData,"HIGH")) {
	  resolution=3;
	}
	if (!strcmp(specResData,"HIGH+") && !strcmp(fil,"L"))
	  {
	    resolution=4;
	  }
	if (!strcmp(specResData,"HIGH+") && !strcmp(fil,"M"))
	  {
	    resolution=5;
	  }
      }
      // Loading of the dispersion coefficients
      dispCoef=cpl_vector_new(N_DEG_DISPERSION_LAW+1);
      cpl_vector_set(dispCoef,0,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF0"));
      cpl_vector_set(dispCoef,1,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF1"));
      cpl_vector_set(dispCoef,2,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF2"));
      cpl_vector_set(dispCoef,3,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF3"));
      cpl_vector_set(dispCoef,4,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF4"));

      // BISPECTRUM
      // 1st peak : T2T3T4 u=3D v=9D
      // 2nd peak : T1T2T3 u=6D v=9D
      // 3rd peak : T1T2T4 u=6D v=12D
      // 4th peak : T1T3T4 u=3D v=15D
      cptWave=0;
      nbFrameOverPhot=corrFlux->nbframe/corrFlux->imgdet->nbregion;
      nbFrameTarget=nbFrameOverPhot;
      if (cpl_propertylist_has(corrFlux->keywords,"PRO FRAME TARGET")) {
	nbFrameTarget=cpl_propertylist_get_int(corrFlux->keywords,"PRO FRAME TARGET");
      }
  
      if (nofringe != NULL)
	{
	  nbFrameOverPhotNoFringe=nofringe->nbframe/nofringe->imgdet->nbregion;
	  nbFrameTargetNoFringe=nbFrameOverPhotNoFringe;
	  if (cpl_propertylist_has(nofringe->keywords,"PRO FRAME TARGET")) {
	    nbFrameTargetNoFringe=cpl_propertylist_get_int(nofringe->keywords,"PRO FRAME TARGET");
	  }
	}
      
      for(iReg=0;iReg<corrFlux->imgdet->nbregion;iReg++) {
	dimSpatial=corrFlux->imgdet->list_region[iReg]->naxis[0];
	dimSpectral=corrFlux->imgdet->list_region[iReg]->naxis[1];
	corner=corrFlux->imgdet->list_region[iReg]->corner[1];

	if (nofringe != NULL)
	  {
	    biSpecRealNoFringe=cpl_calloc(nofringe->imgdet->list_region[iReg]->naxis[1],sizeof(cpl_image *));
	    biSpecImagNoFringe=cpl_calloc(nofringe->imgdet->list_region[iReg]->naxis[1],sizeof(cpl_image *));
	    for(iWlen=0;iWlen<dimSpectral;iWlen++) {
	      lambda=mat_get_lambda(dispCoef,iWlen+corner);
	      posFringePeak=mat_get_pos_fringe_peak(lambda,numDet,resolution,2);
	      dOverLambda=(posFringePeak-dimSpatial/2)/9.;
	      uPixelMin=(int)(dOverLambda*1.5)-1;
	      uPixelMax=(int)(dOverLambda*7.5)+1;
	      vPixelMin=(int)(dOverLambda*7.5)-1;
	      vPixelMax=(int)(dOverLambda*16.5)+1;
	      uPixelSize=uPixelMax-uPixelMin+1;
	      vPixelSize=vPixelMax-vPixelMin+1;
	      biSpecRealNoFringe[iWlen]=cpl_image_new(uPixelSize,vPixelSize,CPL_TYPE_DOUBLE);
	      biSpecImagNoFringe[iWlen]=cpl_image_new(uPixelSize,vPixelSize,CPL_TYPE_DOUBLE);

	      for(iFrame=5;iFrame<nbFrameTargetNoFringe;iFrame++) {
		if (iWlen==0 && iFrame % 50 == 0) {
		  cpl_msg_info(cpl_func,"NoFringe Frame %d/%d: processing in progress", iFrame, nbFrameTargetNoFringe);
		}
		foore=cpl_image_new(uPixelSize,vPixelSize,CPL_TYPE_DOUBLE);
		fooim=cpl_image_new(uPixelSize,vPixelSize,CPL_TYPE_DOUBLE);

		ftReal=nofringe->list_corr[iReg*nbFrameOverPhotNoFringe+iFrame]->imgreg[0];
		ftImag=nofringe->list_corr[iReg*nbFrameOverPhotNoFringe+iFrame]->imgreg[1];

		/* define the pointers to the images */
		cpl_size naxis1 = cpl_image_get_size_x(ftReal);
		const float * pftReal = cpl_image_get_data_float(ftReal);
		const float * pftImag = cpl_image_get_data_float(ftImag);
		double * pfoore = cpl_image_get_data_double(foore);
		double * pfooim = cpl_image_get_data_double(fooim);

		/* #pragma omp parallel for private(l,m,aVal,bVal,cVal,dVal,eVal,fVal) */
		for(l=uPixelMin;l<=uPixelMax;l++) {
		  for(m=vPixelMin;m<=vPixelMax;m++) {
		    cpl_size index1 = naxis1     * (iWlen)       + (l +     dimSpatial/2);
		    cpl_size index2 = naxis1     * (iWlen)       + (m +     dimSpatial/2);
		    cpl_size index3 = naxis1     * (iWlen)       + (l + m + dimSpatial/2);
		    cpl_size index4 = uPixelSize * (m-vPixelMin) + (l-uPixelMin);

		    aVal=pftReal[index1];
		    bVal=pftImag[index1];
		    cVal=pftReal[index2];
		    dVal=pftImag[index2];
		    eVal=pftReal[index3];
		    fVal=pftImag[index3];

		    pfoore[index4] = aVal*cVal*eVal-bVal*dVal*eVal+
		      bVal*cVal*fVal+aVal*dVal*fVal;

		    pfooim[index4]= bVal*cVal*eVal+aVal*dVal*eVal-
		      aVal*cVal*fVal+bVal*dVal*fVal;
		  }
		}
		cpl_image_add(biSpecRealNoFringe[iWlen],foore);
		cpl_image_add(biSpecImagNoFringe[iWlen],fooim);
		cpl_image_delete(foore);
		cpl_image_delete(fooim);
	      }

	      cpl_image_divide_scalar(biSpecRealNoFringe[iWlen],nbFrameTargetNoFringe-5);
	      cpl_image_divide_scalar(biSpecImagNoFringe[iWlen],nbFrameTargetNoFringe-5);
	    }
	    output=cpl_sprintf("BSrealNoFringe_%d.fits",expno);
	    cpl_image_save(biSpecRealNoFringe[20],output, CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE);
	    cpl_free(output);
	    output=cpl_sprintf("BSimagNoFringe_%d.fits",expno);
	    cpl_image_save(biSpecImagNoFringe[20],output, CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE);
	    cpl_free(output);
	  }
    

    
  

	cpl_msg_info(cpl_func,"Computation of the closure phases");
	biSpecReal=cpl_calloc(corrFlux->imgdet->list_region[iReg]->naxis[1],sizeof(cpl_image *));
	biSpecImag=cpl_calloc(corrFlux->imgdet->list_region[iReg]->naxis[1],sizeof(cpl_image *));
	for(iWlen=0;iWlen<dimSpectral;iWlen++)
	  {
	    lambda=mat_get_lambda(dispCoef,iWlen+corner);
	    posFringePeak=mat_get_pos_fringe_peak(lambda,numDet,resolution,2);
	    dOverLambda=(posFringePeak-dimSpatial/2)/9.;
	    uPixelMin=(int)(dOverLambda*1.5)-1;
	    uPixelMax=(int)(dOverLambda*7.5)+1;
	    vPixelMin=(int)(dOverLambda*7.5)-1;
	    vPixelMax=(int)(dOverLambda*16.5)+1;
	    uPixelSize=uPixelMax-uPixelMin+1;
	    vPixelSize=vPixelMax-vPixelMin+1;
	    biSpecReal[iWlen]=cpl_image_new(uPixelSize,vPixelSize,CPL_TYPE_DOUBLE);
	    biSpecImag[iWlen]=cpl_image_new(uPixelSize,vPixelSize,CPL_TYPE_DOUBLE);

	    for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
	      {
		if (iWlen==0 && iFrame % 50 == 0)
		  {
		    cpl_msg_info(cpl_func,"Frame %d/%d: processing in progress", iFrame, nbFrameTarget);
		  }
		foore=cpl_image_new(uPixelSize,vPixelSize,CPL_TYPE_DOUBLE);
		fooim=cpl_image_new(uPixelSize,vPixelSize,CPL_TYPE_DOUBLE);

		ftReal=corrFlux->list_corr[iReg*nbFrameOverPhot+iFrame]->imgreg[0];
		ftImag=corrFlux->list_corr[iReg*nbFrameOverPhot+iFrame]->imgreg[1];

		/* define the pointers to the images */
		cpl_size naxis1 = cpl_image_get_size_x(ftReal);
		const float * pftReal = cpl_image_get_data_float(ftReal);
		const float * pftImag = cpl_image_get_data_float(ftImag);
		double * pfoore = cpl_image_get_data_double(foore);
		double * pfooim = cpl_image_get_data_double(fooim);

		/* #pragma omp parallel for private(l,m,aVal,bVal,cVal,dVal,eVal,fVal) */
		for(l=uPixelMin;l<=uPixelMax;l++)
		  {
		    for(m=vPixelMin;m<=vPixelMax;m++)
		      {
			cpl_size index1 = naxis1     * (iWlen)       + (l +     dimSpatial/2);
			cpl_size index2 = naxis1     * (iWlen)       + (m +     dimSpatial/2);
			cpl_size index3 = naxis1     * (iWlen)       + (l + m + dimSpatial/2);
			cpl_size index4 = uPixelSize * (m-vPixelMin) + (l-uPixelMin);

			aVal=pftReal[index1];
			bVal=pftImag[index1];
			cVal=pftReal[index2];
			dVal=pftImag[index2];
			eVal=pftReal[index3];
			fVal=pftImag[index3];

			pfoore[index4] = aVal*cVal*eVal-bVal*dVal*eVal+
			  bVal*cVal*fVal+aVal*dVal*fVal;

			pfooim[index4]= bVal*cVal*eVal+aVal*dVal*eVal-
			  aVal*cVal*fVal+bVal*dVal*fVal;
		      }
		  }
		cpl_image_add(biSpecReal[iWlen],foore);
		cpl_image_add(biSpecImag[iWlen],fooim);
		cpl_image_delete(foore);
		cpl_image_delete(fooim);
	      }

	    cpl_image_divide_scalar(biSpecReal[iWlen],nbFrameTarget);
	    cpl_image_divide_scalar(biSpecImag[iWlen],nbFrameTarget);
	  }
	output=cpl_sprintf("BSreal_%d.fits",expno);
	cpl_image_save(biSpecReal[20],output, CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE);
	cpl_free(output);
	output=cpl_sprintf("BSimag_%d.fits",expno);
	cpl_image_save(biSpecImag[20],output, CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE);
	cpl_free(output);

	// Closure phase computation
	for(iWlen=0;iWlen<dimSpectral;iWlen++)
	  {
	    lambda=mat_get_lambda(dispCoef,iWlen+corner);
	    posFringePeak=mat_get_pos_fringe_peak(lambda,numDet,resolution,2);
	    dOverLambda=(posFringePeak-dimSpatial/2)/9.;
	    for(iTrip=4;iTrip>=0;--iTrip)
	      {
		uPixelMin0=(int)(1.5*dOverLambda)-1;
		vPixelMin0=(int)(7.5*dOverLambda)-1;
		if (iTrip == 0)
		  {
		    uPixelMin=(int)(1.5*dOverLambda)-1;
		    uPixelMax=(int)(4.5*dOverLambda)+1;
		    vPixelMin=(int)(7.5*dOverLambda)-1;
		    vPixelMax=(int)(10.5*dOverLambda)+1;
		  }
		else if (iTrip==1)
		  {
		    uPixelMin=(int)(4.5*dOverLambda)-1;
		    uPixelMax=(int)(7.5*dOverLambda)+1;
		    vPixelMin=(int)(7.5*dOverLambda)-1;
		    vPixelMax=(int)(10.5*dOverLambda)+1;
		  }
		else if (iTrip==2)
		  {
		    uPixelMin=(int)(4.5*dOverLambda)-1;
		    uPixelMax=(int)(7.5*dOverLambda)+1;
		    vPixelMin=(int)(10.5*dOverLambda)-1;
		    vPixelMax=(int)(13.5*dOverLambda)+1;
		  }
		else if (iTrip==3)
		  {
		    uPixelMin=(int)(1.5*dOverLambda)-1;
		    uPixelMax=(int)(4.5*dOverLambda)+1;
		    vPixelMin=(int)(13.5*dOverLambda)-1;
		    vPixelMax=(int)(16.5*dOverLambda)+1;
		  }
		else if (iTrip==4)
		  {
		    uPixelMin=(int)(4.5*dOverLambda)-1;
		    uPixelMax=(int)(7.5*dOverLambda)+1;
		    vPixelMin=(int)(13.5*dOverLambda)-1;
		    vPixelMax=(int)(16.5*dOverLambda)+1;
		  }		    
		uPixelSize=uPixelMax-uPixelMin+1;
		vPixelSize=vPixelMax-vPixelMin+1;
		vecw=cpl_vector_new(uPixelSize);
		vecy=cpl_vector_new(uPixelSize);
		vecz=cpl_vector_new(uPixelSize);
		limit=uPixelMax-uPixelMin-1-dOverLambda/2.;
		for(m=0;m<uPixelSize;m++)
		  {
		    ix=uPixelMin-uPixelMin0+m+1;
		    aVal=0.;
		    bVal=0.;
		    for(k=0;k<vPixelSize;k++)
		      {
			iy=vPixelMin-vPixelMin0+k+1;
			aVal+=cpl_image_get(biSpecReal[iWlen],ix,iy,&pis);
			bVal+=cpl_image_get(biSpecImag[iWlen],ix,iy,&pis);
		      }
		    cpl_vector_set(vecy,m,aVal);
		    cpl_vector_set(vecz,m,bVal);
		    if ( m>dOverLambda/2. && m<limit )
		      {
			cpl_vector_set(vecw,m,1.E-20);
		      }
		    else
		      {
			cpl_vector_set(vecw,m,1.0);
		      }
		  }

		fit=cpl_vector_new(uPixelSize);
		mat_polyfit_1d(vecy,vecw,1,fit,1);
		if (nofringe != NULL)
		  {
		    cpl_vector_subtract(vecy,fit);
		  }
		cpl_vector_delete(fit);

		fit=cpl_vector_new(uPixelSize);
		mat_polyfit_1d(vecz,vecw,1,fit,1);
		if (nofringe != NULL)
		  {
		    cpl_vector_subtract(vecz,fit);
		  }
		cpl_vector_delete(fit);

		if ( iTrip==4 ){
		  cpl_vector_multiply(vecz,vecz);
		  num=cpl_vector_get_mean(vecz);
		}

		avgz=cpl_vector_get_mean(vecz);
		avgy=cpl_vector_get_mean(vecy);

		// FIll oifits->oit3 with computed closure phases
		if (iTrip < 4)
		  {
		    if (oifits->oit3->nbt3 == 4)
		      {
			val=atan2(avgz,avgy)*CPL_MATH_DEG_RAD;
			if (isnan(val))
			  {
			    val=0.;
			  }
			oifits->oit3->list_t3[iTrip]->t3phi[cptWave]=val;
			val=sqrt(num/(pow(avgz,2.0)+pow(avgy,2.0)))*CPL_MATH_DEG_RAD;
			if (isnan(val))
			  {
			    val=1.E3;
			  }
			oifits->oit3->list_t3[iTrip]->t3phierr[cptWave]=fabs(val);
			if (numDet == 1)
			  {
			    if (fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 3.5) <= 0.7 || fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 4.84) <= 0.36)
			      {
				oifits->oit3->list_t3[iTrip]->bandflag[cptWave]=CPL_FALSE;
			      }
			    else
			      {
				oifits->oit3->list_t3[iTrip]->bandflag[cptWave]=CPL_TRUE;
			      }
			  }
			else
			  {
			    if (fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 8.59) <= 0.59 || fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 11.28) <= 1.72)
			      {
				oifits->oit3->list_t3[iTrip]->bandflag[cptWave]=CPL_FALSE;
			      }
			    else
			      {
				oifits->oit3->list_t3[iTrip]->bandflag[cptWave]=CPL_TRUE;
			      }
			  }
		      }
		    else if (oifits->oit3->nbt3 == 1)
		      {
			if (iTrip == 1)
			  {
			    val=atan2(avgz,avgy)*CPL_MATH_DEG_RAD;
			    if (isnan(val))
			      {
				val=0.;
			      }
			    oifits->oit3->list_t3[iTrip]->t3phi[cptWave]=val;
			    val=sqrt(num/(pow(avgz,2.0)+pow(avgy,2.0)))*CPL_MATH_DEG_RAD;
			    if (isnan(val))
			      {
				val=1.E3;
			      }
			    oifits->oit3->list_t3[iTrip]->t3phierr[cptWave]=fabs(val);
			    oifits->oit3->list_t3[iTrip]->bandflag[cptWave]=CPL_FALSE;
			  }
		      } 
		  }
		cpl_vector_delete(vecy);
		cpl_vector_delete(vecz);
		cpl_vector_delete(vecw);
	      }
	    cptWave++;
	  }




	for(iWlen=0;iWlen<dimSpectral;iWlen++) {
	  cpl_image_delete(biSpecReal[iWlen]);
	  cpl_image_delete(biSpecImag[iWlen]);
	}
	cpl_free(biSpecReal);
	cpl_free(biSpecImag);
	if (nofringe != NULL)
	  {
	    for(iWlen=0;iWlen<dimSpectral;iWlen++) {
	      cpl_image_delete(biSpecRealNoFringe[iWlen]);
	      cpl_image_delete(biSpecImagNoFringe[iWlen]);
	    }
	    cpl_free(biSpecRealNoFringe);
	    cpl_free(biSpecImagNoFringe);
	  }


      }
      cpl_vector_delete(dispCoef);
    }
  else
    {
      int indexClosure[4][3]={{0,2,3},{2,1,4},{3,1,5},{0,4,5}};
      double a1,a2,a3,b1,b2,b3;
      double phi1,phi2,phi3;
      cpl_vector *bsre=NULL;
      cpl_vector *bsim=NULL;
  
      detName=(char *)
	cpl_propertylist_get_string(corrFlux->keywords,"ESO DET CHIP NAME");
      if (detName == NULL) {
	cpl_msg_error(cpl_func,
		      "no ESO DET CHIP NAME keyword in frame");
	return -1;
      }
      if(!strcmp(detName, "AQUARIUS")) {
	numDet=2;
      }
      else {
	numDet=1;
      }
      dimSpectral=oivis->oiwave->nbchannel;
      bsre=cpl_vector_new(dimSpectral);
      bsim=cpl_vector_new(dimSpectral);
      for(iTrip=0;iTrip<4;iTrip++)
	{
	  for(iWlen=0;iWlen<dimSpectral;iWlen++)
	    {
	      a1=cos(oivis->oivis->list_vis[indexClosure[iTrip][0]]->visphi[iWlen]/CPL_MATH_DEG_RAD);
	      a2=cos(oivis->oivis->list_vis[indexClosure[iTrip][1]]->visphi[iWlen]/CPL_MATH_DEG_RAD);
	      a3=cos(-oivis->oivis->list_vis[indexClosure[iTrip][2]]->visphi[iWlen]/CPL_MATH_DEG_RAD);
	      b1=sin(oivis->oivis->list_vis[indexClosure[iTrip][0]]->visphi[iWlen]/CPL_MATH_DEG_RAD);
	      b2=sin(oivis->oivis->list_vis[indexClosure[iTrip][1]]->visphi[iWlen]/CPL_MATH_DEG_RAD);
	      b3=sin(-oivis->oivis->list_vis[indexClosure[iTrip][2]]->visphi[iWlen]/CPL_MATH_DEG_RAD);
	      cpl_vector_set(bsre,iWlen,a1*a2*a3-b1*b2*a3-a1*b2*b3-a2*b1*b3);
	      cpl_vector_set(bsim,iWlen,a1*b2*a3+a2*b1*a3+a1*a2*b3-b1*b2*b3);
	      aVal=0.;
	      bVal=0.;
	      for(iFrame=0;iFrame<oiopd->oiopd->nbopd;iFrame+=6)
	      	{
	      	  phi1=oiopd->oiopd->list_opd[iFrame+indexClosure[iTrip][0]]->humoff+2*CPL_MATH_PI*
	      	    oiopd->oiopd->list_opd[iFrame+indexClosure[iTrip][0]]->opd/(oivis->oiwave->effwave[iWlen]*1.E6);
	      	  phi2=oiopd->oiopd->list_opd[iFrame+indexClosure[iTrip][1]]->humoff+2*CPL_MATH_PI*
	      	    oiopd->oiopd->list_opd[iFrame+indexClosure[iTrip][1]]->opd/(oivis->oiwave->effwave[iWlen]*1.E6);
	      	  phi3=oiopd->oiopd->list_opd[iFrame+indexClosure[iTrip][2]]->humoff+2*CPL_MATH_PI*
	      	    oiopd->oiopd->list_opd[iFrame+indexClosure[iTrip][2]]->opd/(oivis->oiwave->effwave[iWlen]*1.E6);
	      	  a1=cos(phi1);
	      	  a2=cos(phi2);
	      	  a3=cos(-phi3);
	      	  b1=sin(phi1);
	      	  b2=sin(phi2);
	      	  b3=sin(-phi3);
	      	  aVal+=a1*a2*a3-b1*b2*a3-a1*b2*b3-a2*b1*b3;
	      	  bVal+=a1*b2*a3+a2*b1*a3+a1*a2*b3-b1*b2*b3;
	      	}
	      oifits->oit3->list_t3[iTrip]->t3phi[iWlen]=(atan2(cpl_vector_get(bsim,iWlen),cpl_vector_get(bsre,iWlen))+atan2(bVal,aVal))*CPL_MATH_DEG_RAD;
	      oifits->oit3->list_t3[iTrip]->t3phierr[iWlen]=sqrt(pow(oivis->oivis->list_vis[indexClosure[iTrip][0]]->visphierr[iWlen],2.0)+
								 pow(oivis->oivis->list_vis[indexClosure[iTrip][1]]->visphierr[iWlen],2.0)+
								 pow(oivis->oivis->list_vis[indexClosure[iTrip][2]]->visphierr[iWlen],2.0));
	      oifits->oit3->list_t3[iTrip]->bandflag[iWlen]=CPL_FALSE;
	    }
	  
	}
      cpl_vector_delete(bsre);
      cpl_vector_delete(bsim);

    }


  // QC parameter Computation
  for(iTrip=0;iTrip<oifits->oit3->nbt3;iTrip++) {
    val=0.;
    val2=0.;
    cptWave=0.;
    for(iWlen=0;iWlen<oifits->oit3->nbchannel;iWlen++) {
      if (numDet == 1) {
	if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 3.5) <= 0.55 || fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 4.75) <= 0.25) {
	  val+=oifits->oit3->list_t3[iTrip]->t3phi[iWlen];
	  val2+=pow(oifits->oit3->list_t3[iTrip]->t3phierr[iWlen],2.0);
	  cptWave++;
	}
      } else if (numDet == 2) {
	if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 9.0) <= 1.0) {
	  val+=oifits->oit3->list_t3[iTrip]->t3phi[iWlen];
	  val2+=pow(oifits->oit3->list_t3[iTrip]->t3phierr[iWlen],2.0);
	  cptWave++;
	}
      } else {
	cptWave=1;
	val=0.;
	val2=0.;
      }

    }
    val/=cptWave;
    val2/=cptWave;
    if (val2 < 0) val2=fabs(val2);
    snprintf(kwd, 64, "ESO QC DET%d CPHASE%d",numDet,iTrip+1);
    cpl_propertylist_append_double(oifits->keywords, kwd, val);
    snprintf(kwd, 64, "ESO QC DET%d CPHASE%d STDEV",numDet,iTrip+1);
    cpl_propertylist_append_double(oifits->keywords, kwd, sqrt(val2));
  }

  // Add Generic QC parameters
  cpl_propertylist *qclist=NULL;
  qclist = cpl_propertylist_new();
  mat_add_generic_qc(oifits->keywords,qclist);
  cpl_propertylist_append(oifits->keywords,qclist);
  cpl_propertylist_delete(qclist);

  return(0);
	
}
