/* $Id: mat_compute_spectrum.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_compute_spectrum.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#include <string.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/

#include "mat_compute_spectrum.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
  Define
  ----------------------------------------------------------------------------*/
/*-----------------------------------------------------------------------------
  Functions prototypes
  ----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
   @ingroup signal
   @brief Compute the spectrum from a mat_photbeams structure. 
   This function fills a mat_oifits with the computed spectrum.
   @param phot               contains the mat_photbeams structure
   @param oifits             contains the mat_oifits strucuture in which 
   the spectrum is stored
   @return 0 if no error 
*/
int mat_compute_spectrum(mat_photbeams *phot, mat_oifits *oifits)
{
    int i=0;
    int j=0;
    int ix=0;
    int k=0;
    int cpt=0;
    /* int iReg=0; */
    int cptWave=0;
    int pis=0;
    int nbBand=0;
    /* char *charFoo=NULL; */
    /* char *insMode=NULL; */
    char *bcd1Status=NULL;
    char *bcd2Status=NULL;
    cpl_image **imgPhot=NULL;
    cpl_image **imgPhotErr=NULL;
    cpl_image **imgCol=NULL;
    cpl_image **imgColErr=NULL;
    cpl_image *imgFoo=NULL;
    char *sliderPhot=NULL;
    char *detName=NULL;
    char mode[3]=" ";
    int nbdetector=0;
   //    cpl_vector *vec;


    // Check input parameters
    mat_assert((phot!=NULL),CPL_ERROR_NULL_INPUT, "no mat_phot argument given");
    mat_assert((oifits!=NULL),CPL_ERROR_NULL_INPUT, "no mat_oifits argument given");
    // Mode determination
    /* insMode=(char *)cpl_propertylist_get_string(oifits->keywords,  */
    /* 						"ESO INS MODE"); */
    /* charFoo=strndup(insMode,1); */
    /* nbTel=atoi(charFoo); */
    /* free(charFoo); */

    bcd1Status = (char *)cpl_propertylist_get_string(
	oifits->keywords,"ESO INS BCD1 ID");
    bcd2Status = (char *)cpl_propertylist_get_string(
	oifits->keywords,"ESO INS BCD2 ID");
    detName=(char *)
	cpl_propertylist_get_string(phot->keywords,"ESO DET CHIP NAME");
    if (detName == NULL)
	{
	cpl_msg_error(cpl_func,
		      "no ESO DET CHIP NAME keyword in frame");
	return -1;
	}


    if(!strcmp(detName, "AQUARIUS"))
	{
	  sliderPhot=
	    (char *)cpl_propertylist_get_string(phot->keywords, 
						"ESO INS PIN NAME");	
	  nbdetector=2;
	}
    else
      {
	sliderPhot=
	  (char *)cpl_propertylist_get_string(phot->keywords, 
					      "ESO INS PIL NAME");	
	  nbdetector=1;
      }
    if (!strcmp(sliderPhot, "PHOTO"))
	{
	sprintf(mode,"s");
	}
    else
	{
	sprintf(mode,"h");
	}


    
    nbBand=phot->imgdet->nbregion/4;

    imgPhot    = cpl_malloc(4*sizeof(cpl_image *));
    imgPhotErr = cpl_malloc(4*sizeof(cpl_image *));
    imgCol     = cpl_malloc(4*sizeof(cpl_image *));
    imgColErr  = cpl_malloc(4*sizeof(cpl_image *));
    cpt=0;
    cptWave=0;
    for(i=0;i<nbBand;i++)
	{
	for(k=0;k<4;k++)
	    {
	    imgPhot[k]=cpl_image_duplicate(phot->list_phot[cpt]->imgreg[0]);
	    if (strstr(mode,"s") != NULL )
		{
		imgFoo=cpl_image_duplicate(phot->list_phot[cpt]->imgreg[0]);
		cpl_image_multiply(imgFoo,imgFoo);
		imgPhotErr[k]=cpl_image_duplicate(imgFoo);
		cpl_image_delete(imgFoo);
		}
	    cpt++;
	    if (strstr(mode,"s") != NULL )
		{
		// SiPhot Mode
		for(j=1;j<phot->nbphotxframe/phot->imgdet->nbregion;j++)
		    {
		    cpl_image_add(imgPhot[k],phot->list_phot[cpt]->imgreg[0]);
		    imgFoo=cpl_image_duplicate(phot->list_phot[cpt]->imgreg[0]);
		    cpl_image_multiply(imgFoo,imgFoo);
		    cpl_image_add(imgPhotErr[k],imgFoo);
		    cpl_image_delete(imgFoo);
		    cpt++;
		    }
		cpl_image_divide_scalar(imgPhot[k],
					phot->nbphotxframe/phot->imgdet->nbregion);
		cpl_image_divide_scalar(imgPhotErr[k],
					phot->nbphotxframe/phot->imgdet->nbregion);
		imgFoo=cpl_image_duplicate(imgPhot[k]);
		cpl_image_multiply(imgFoo,imgFoo);
		cpl_image_subtract(imgPhotErr[k],imgFoo);
		cpl_image_delete(imgFoo);
	      
		}
	    else
		{
		imgPhotErr[k]=cpl_image_duplicate(phot->list_phot[cpt]->imgreg[0]);
		cpt++;
		// HighSens Mode
		cpl_image_multiply(imgPhotErr[k],imgPhotErr[k]);
		}
	    imgColErr[k]=cpl_image_collapse_create(imgPhotErr[k],1);
	    for(ix=1;ix<=cpl_image_get_size_y(imgColErr[k]);ix++)
		{
		cpl_image_set(imgColErr[k],1,ix,sqrt(cpl_image_get(imgColErr[k],1,ix,&pis)));
		}
	    imgCol[k]=cpl_image_collapse_create(imgPhot[k],1);
	    }
	for(k=1;k<=cpl_image_get_size_y(imgCol[0]);k++)
	    {
	    // Compute the averaged flux over the photometric channels
	    // and the flux error
	    if (oifits->array->nbstation==4)
		{
		oifits->oispect->list_flux[0]->fluxdata[cptWave]=
		    cpl_image_get(imgCol[0],1,k,&pis);
		oifits->oispect->list_flux[1]->fluxdata[cptWave]=
		    cpl_image_get(imgCol[1],1,k,&pis);
		oifits->oispect->list_flux[2]->fluxdata[cptWave]=
		    cpl_image_get(imgCol[2],1,k,&pis);
		oifits->oispect->list_flux[3]->fluxdata[cptWave]=
		    cpl_image_get(imgCol[3],1,k,&pis);
	    
		oifits->oispect->list_flux[0]->fluxerr[cptWave]=
		  fabs(cpl_image_get(imgColErr[0],1,k,&pis));
		oifits->oispect->list_flux[1]->fluxerr[cptWave]=
		  fabs(cpl_image_get(imgColErr[1],1,k,&pis));
		oifits->oispect->list_flux[2]->fluxerr[cptWave]=
		  fabs(cpl_image_get(imgColErr[2],1,k,&pis));
		oifits->oispect->list_flux[3]->fluxerr[cptWave]=
		  fabs(cpl_image_get(imgColErr[3],1,k,&pis));

		if (nbdetector == 1)
			{
		                if (fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 3.5) <= 0.55 || fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 4.75) <= 0.25)
				  //if (fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 3.5) <= 0.7 || fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 4.84) <= 0.36)
					{
						oifits->oispect->list_flux[0]->bandflag[cptWave]=CPL_FALSE;
						oifits->oispect->list_flux[1]->bandflag[cptWave]=CPL_FALSE;
						oifits->oispect->list_flux[2]->bandflag[cptWave]=CPL_FALSE;
						oifits->oispect->list_flux[3]->bandflag[cptWave]=CPL_FALSE;
					}
				else
					{
						oifits->oispect->list_flux[0]->bandflag[cptWave]=CPL_TRUE;
						oifits->oispect->list_flux[1]->bandflag[cptWave]=CPL_TRUE;
						oifits->oispect->list_flux[2]->bandflag[cptWave]=CPL_TRUE;
						oifits->oispect->list_flux[3]->bandflag[cptWave]=CPL_TRUE;
					}
			}
		else
			{
			        if (fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 9.0) <= 1.0)
				  //if (fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 8.59) <= 0.59 || fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 11.28) <= 1.72)
					{
						oifits->oispect->list_flux[0]->bandflag[cptWave]=CPL_FALSE;
						oifits->oispect->list_flux[1]->bandflag[cptWave]=CPL_FALSE;
						oifits->oispect->list_flux[2]->bandflag[cptWave]=CPL_FALSE;
						oifits->oispect->list_flux[3]->bandflag[cptWave]=CPL_FALSE;
					}
				else
					{
						oifits->oispect->list_flux[0]->bandflag[cptWave]=CPL_TRUE;
						oifits->oispect->list_flux[1]->bandflag[cptWave]=CPL_TRUE;
						oifits->oispect->list_flux[2]->bandflag[cptWave]=CPL_TRUE;
						oifits->oispect->list_flux[3]->bandflag[cptWave]=CPL_TRUE;
					}
			}
		}
	    else if (oifits->array->nbstation==3)
		{
		if (strcmp(bcd2Status,"OUT") )
		    {
		    // IN or OUT-IN
		    oifits->oispect->list_flux[0]->fluxdata[cptWave]=
			cpl_image_get(imgCol[0],1,k,&pis);
		    oifits->oispect->list_flux[1]->fluxdata[cptWave]=
			cpl_image_get(imgCol[2],1,k,&pis);
		    oifits->oispect->list_flux[2]->fluxdata[cptWave]=
			cpl_image_get(imgCol[3],1,k,&pis);
	    
		    oifits->oispect->list_flux[0]->fluxerr[cptWave]=
		      fabs(cpl_image_get(imgColErr[0],1,k,&pis));
		    oifits->oispect->list_flux[1]->fluxerr[cptWave]=
		      fabs(cpl_image_get(imgColErr[2],1,k,&pis));
		    oifits->oispect->list_flux[2]->fluxerr[cptWave]=
		      fabs(cpl_image_get(imgColErr[3],1,k,&pis));
		    }
		else
		    {
		    //IN or OUT - OUT
		    oifits->oispect->list_flux[0]->fluxdata[cptWave]=
			cpl_image_get(imgCol[0],1,k,&pis);
		    oifits->oispect->list_flux[1]->fluxdata[cptWave]=
			cpl_image_get(imgCol[1],1,k,&pis);
		    oifits->oispect->list_flux[2]->fluxdata[cptWave]=
			cpl_image_get(imgCol[2],1,k,&pis);
	    
		    oifits->oispect->list_flux[0]->fluxerr[cptWave]=
		      fabs(cpl_image_get(imgColErr[0],1,k,&pis));
		    oifits->oispect->list_flux[1]->fluxerr[cptWave]=
		      fabs(cpl_image_get(imgColErr[1],1,k,&pis));
		    oifits->oispect->list_flux[2]->fluxerr[cptWave]=
		      fabs(cpl_image_get(imgColErr[2],1,k,&pis));
		    }
		oifits->oispect->list_flux[0]->bandflag[cptWave]=CPL_FALSE;
		oifits->oispect->list_flux[1]->bandflag[cptWave]=CPL_FALSE;
		oifits->oispect->list_flux[2]->bandflag[cptWave]=CPL_FALSE;
		}
	    else
		{
		if (strcmp(bcd2Status,"OUT") )
		    {
		    if (strcmp(bcd1Status,"OUT") )
			{
			// IN-IN
			oifits->oispect->list_flux[0]->fluxdata[cptWave]=
			    cpl_image_get(imgCol[2],1,k,&pis);
			oifits->oispect->list_flux[1]->fluxdata[cptWave]=
			    cpl_image_get(imgCol[3],1,k,&pis);
		
			oifits->oispect->list_flux[0]->fluxerr[cptWave]=
			  fabs(cpl_image_get(imgColErr[2],1,k,&pis));
			oifits->oispect->list_flux[1]->fluxerr[cptWave]=
			  fabs(cpl_image_get(imgColErr[3],1,k,&pis));
			}
		    else
			{
			//OUT-IN
			oifits->oispect->list_flux[0]->fluxdata[cptWave]=
			    cpl_image_get(imgCol[0],1,k,&pis);
			oifits->oispect->list_flux[1]->fluxdata[cptWave]=
			    cpl_image_get(imgCol[3],1,k,&pis);
		
			oifits->oispect->list_flux[0]->fluxerr[cptWave]=
			  fabs(cpl_image_get(imgColErr[0],1,k,&pis));
			oifits->oispect->list_flux[1]->fluxerr[cptWave]=
			  fabs(cpl_image_get(imgColErr[3],1,k,&pis));
			}
		    }
		else
		    {
		    if (strcmp(bcd1Status,"OUT") )
			{
			// IN-OUT
			oifits->oispect->list_flux[0]->fluxdata[cptWave]=
			    cpl_image_get(imgCol[1],1,k,&pis);
			oifits->oispect->list_flux[1]->fluxdata[cptWave]=
			    cpl_image_get(imgCol[3],1,k,&pis);
		
			oifits->oispect->list_flux[0]->fluxerr[cptWave]=
			  fabs(cpl_image_get(imgColErr[1],1,k,&pis));
			oifits->oispect->list_flux[1]->fluxerr[cptWave]=
			  fabs(cpl_image_get(imgColErr[3],1,k,&pis));
			}
		    else
			{
			//OUT-OUT
			oifits->oispect->list_flux[0]->fluxdata[cptWave]=
			    cpl_image_get(imgCol[0],1,k,&pis);
			oifits->oispect->list_flux[1]->fluxdata[cptWave]=
			    cpl_image_get(imgCol[1],1,k,&pis);
		
			oifits->oispect->list_flux[0]->fluxerr[cptWave]=
			  fabs(cpl_image_get(imgColErr[0],1,k,&pis));
			oifits->oispect->list_flux[1]->fluxerr[cptWave]=
			  fabs(cpl_image_get(imgColErr[1],1,k,&pis));
			}
		    }
		oifits->oispect->list_flux[0]->bandflag[cptWave]=CPL_FALSE;
		oifits->oispect->list_flux[1]->bandflag[cptWave]=CPL_FALSE;
		}
	    cptWave++;
	    }
	for(k=0;k<4;k++)
	    {
	    cpl_image_delete(imgPhot[k]);
	    cpl_image_delete(imgPhotErr[k]);
	    cpl_image_delete(imgCol[k]);
	    cpl_image_delete(imgColErr[k]);
	    }
	}
    cpl_free(imgPhot);
    cpl_free(imgCol);
    cpl_free(imgPhotErr);
    cpl_free(imgColErr);
    if (cptWave != oifits->oiwave->nbchannel)
	{
	printf("%d %d\n",cptWave,oifits->oiwave->nbchannel);
	cpl_msg_warning(cpl_func,"The size of the spectrum does not \
correspond to the number of spectral channel");
	}

    // QC parameter Computation
    double val2=0.;
    double val[4]={0.,0.,0.,0.};
    double valtrans[4]={0.,0.,0.,0.};
    double conv_factor=0.,dit=0.,stel=0.;
    char kwd1[64],kwd2[64],kwd3[64];
    char *bcdMode=NULL;
    double h=6.62607004e-34;

    stel=CPL_MATH_PI*pow(oifits->array->list_station[0]->diameter,2.0)/4.;
    dit=(double) cpl_propertylist_get_double(oifits->keywords,"ESO DET SEQ1 DIT");
    if (nbdetector == 1)
      {
    	conv_factor=dit*2.12E-6*0.9*stel*0.3;
      }
    else
      {
    	conv_factor=15.*dit*4.62E-6*0.5*stel*1.0;
      }
    printf("%e\n",conv_factor);
    
    bcdMode = (char *) cpl_propertylist_get_string(oifits->keywords,"ESO CFG BCD MODE");
    for(i=0;i<oifits->array->nbstation;i++) {
      cptWave=0;
      for(j=0;j<oifits->oivis2->nbchannel;j++) {
	if (nbdetector == 1) {
	  if (fabs(oifits->oiwave->effwave[j]*1.e6 - 3.5) <= 0.55 || fabs(oifits->oiwave->effwave[j]*1.e6 - 4.75) <= 0.25) {
	    //if (fabs(oifits->oiwave->effwave[j]*1.e6 - 3.5) <= 0.7 || fabs(oifits->oiwave->effwave[j]*1.e6 - 4.84) <= 0.36) {
	    val[i]+=oifits->oispect->list_flux[i]->fluxdata[j];
	    cptWave++;
	    valtrans[i]+=oifits->oispect->list_flux[i]->fluxdata[j];
	  }
	} else if (nbdetector == 2) {
	  if (fabs(oifits->oiwave->effwave[j]*1.e6 - 9.0) <= 1.0) {
	    //if (fabs(oifits->oiwave->effwave[j]*1.e6 - 8.59) <= 0.59 || fabs(oifits->oiwave->effwave[j]*1.e6 - 11.28) <= 1.72) {
	    val[i]+=oifits->oispect->list_flux[i]->fluxdata[j];
	    cptWave++;
	    valtrans[i]+=oifits->oispect->list_flux[i]->fluxdata[j];
	  }
	} else {
	  cptWave=1;
	  val[i]=0.;
	}
      }
      val[i]/=cptWave;
      val2+=val[i];

    if (nbdetector == 1)
      {
	valtrans[i]*=(h*3.5E-6*1.E26)/conv_factor;
      }
    else
      {
	valtrans[i]*=(h*10.5E-6*1.E26)/conv_factor;
      }
    }
    for(i=0;i<oifits->array->nbstation;i++) {
      if (strcmp(bcdMode, "OUT-OUT") == 0 )
	{
	  if (i == 0)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP5");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP5");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP5");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP1");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP1");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP1");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);

	    }
	  else if (i == 1)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP7");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP7");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP7");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP3");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP3");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP3");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	  else if (i == 2)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP3");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP3");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP3");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP7");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP7");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP7");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	  else if (i == 3)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP1");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP1");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP1");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP5");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP5");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP5");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	}
      else if (strcmp(bcdMode, "IN-OUT") == 0 )
	{
	  if (i == 0)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP7");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP7");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP7");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP1");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP1");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP1");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	  else if (i == 1)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP5");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP5");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP5");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP3");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP3");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP3");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	  else if (i == 2)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP3");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP3");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP3");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP5");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP5");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP5");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	  else if (i == 3)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP1");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP1");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP1");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP7");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP7");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP7");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	}
      else if (strcmp(bcdMode, "OUT-IN") == 0 )
	{
	  if (i == 0)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP5");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP5");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP5");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP3");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP3");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP3");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	  else if (i == 1)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP7");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP7");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP7");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP1");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP1");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP1");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	  else if (i == 2)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP1");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP1");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP1");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP7");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP7");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP7");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	  else if (i == 3)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP3");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP3");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP3");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP5");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP5");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP5");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	}
      else if (strcmp(bcdMode, "IN-IN") == 0 )
	{
	  if (i == 0)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP7");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP7");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP7");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP3");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP3");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP3");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	  else if (i == 1)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP5");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP5");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP5");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP1");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP1");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP1");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	  else if (i == 2)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP1");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP1");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP1");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP5");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP5");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP5");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	  else if (i == 3)
	    {
	      if (nbdetector == 2)
		{
		  snprintf(kwd1, 64, "ESO QC DET2 FLUX IP3");
		  snprintf(kwd2, 64, "ESO QC DET2 RATIO IP3");
		  snprintf(kwd3, 64, "ESO QC DET2 FLUX JY IP3");
		}
	      else if (nbdetector == 1)
		{
		  snprintf(kwd1, 64, "ESO QC DET1 FLUX IP7");
		  snprintf(kwd2, 64, "ESO QC DET1 RATIO IP7");
		  snprintf(kwd3, 64, "ESO QC DET1 FLUX JY IP7");
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd1, val[i]);
	      if (val2 < 1.E-3)
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, 0.);
		}
	      else
		{
		  cpl_propertylist_append_double(oifits->keywords, kwd2, val[i]/val2);
		}
	      cpl_propertylist_append_double(oifits->keywords, kwd3, valtrans[i]);
	    }
	}
    }
    
    // Add Second version of QC parameters
    cpl_vector *valip=NULL;
    valip=cpl_vector_new(4);
    if (nbdetector==1)
      {
	cpl_vector_set(valip,0,cpl_propertylist_get_double(oifits->keywords,"ESO QC DET1 FLUX IP1"));
	cpl_vector_set(valip,1,cpl_propertylist_get_double(oifits->keywords,"ESO QC DET1 FLUX IP3"));
	cpl_vector_set(valip,2,cpl_propertylist_get_double(oifits->keywords,"ESO QC DET1 FLUX IP5"));
	cpl_vector_set(valip,3,cpl_propertylist_get_double(oifits->keywords,"ESO QC DET1 FLUX IP7"));
      }
    else
      {
	cpl_vector_set(valip,0,cpl_propertylist_get_double(oifits->keywords,"ESO QC DET2 FLUX IP1"));
	cpl_vector_set(valip,1,cpl_propertylist_get_double(oifits->keywords,"ESO QC DET2 FLUX IP3"));
	cpl_vector_set(valip,2,cpl_propertylist_get_double(oifits->keywords,"ESO QC DET2 FLUX IP5"));
	cpl_vector_set(valip,3,cpl_propertylist_get_double(oifits->keywords,"ESO QC DET2 FLUX IP7"));
      }
    cpl_propertylist_append_double(oifits->keywords, "ESO QC TOT FLUX", cpl_vector_get_sum(valip));
    cpl_propertylist_append_double(oifits->keywords, "ESO QC FLUX AVG", cpl_vector_get_mean(valip));
    cpl_propertylist_append_double(oifits->keywords, "ESO QC FLUX STDEV", cpl_vector_get_stdev(valip));
    if (cpl_vector_get_stdev(valip) != 0.)
      {
	cpl_propertylist_append_double(oifits->keywords, "ESO QC FLUX SNR", cpl_vector_get_mean(valip)/cpl_vector_get_stdev(valip));
      }
    else
      {
	cpl_propertylist_append_double(oifits->keywords, "ESO QC FLUX SNR",1.E5);
      }
    cpl_vector_delete(valip);
    
    // Add Generic QC parameters
    cpl_propertylist *qclist=NULL;
    qclist = cpl_propertylist_new();
    mat_add_generic_qc(oifits->keywords,qclist);
    cpl_propertylist_append(oifits->keywords,qclist);
    cpl_propertylist_delete(qclist);
    return(0);



    return CPL_ERROR_NONE;	
}
