/* $Id: mat_compute_tf2.c,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_compute_tf2.c $
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#include <memory.h>
#endif


/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "mat_compute_tf2.h"
#define NB_BASE_MAX 6
/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/






/*----------------------------------------------------------------------------*/
/**
  @ingroup signal
  @brief This function estimate the squared transfer function
  @param oifits         ...
  @param oifits_vis     ...
  @param diameter       the angular diameter in mas and 
  @param diameterErr    error on the angular diameter
  @param diameterName   diameterName
  @param Lmag           L magnitude
  @param Mmag           M magnitude
  @param Nmag           N magnitude
  @return mat_oifits * (Structure containing INTERP_TF2 and RAW_TF2)
 */
/*----------------------------------------------------------------------------*/



mat_oifits *mat_compute_tf2(mat_oifits *oifits, mat_oifits *oifits_vis, double diameter,double diameterErr,
			    char *diameterName, double diameterLmag,
			    double diameterMmag, double diameterNmag)
{
  
    mat_oifits *result = NULL;
    mat_oivis2 *oivis2 = NULL;
    mat_oivis  *oivis  = NULL;
    mat_oitf2  *oitf2  = NULL;
    cpl_propertylist *qclist = NULL;
    double base = 0.0;
    double aux_vcoord = 0.0;
    double aux_ucoord = 0.0;
    double Z = 0.0;
    double V2star = 0.0;
    double V2starErr = 0.0;
    double TF = 0.0;
    double TFErr = 0.0;
    double TFErr1 = 0.0;
    double TFErr2 = 0.0;
    double SNR = 0.0;
    double SNRcal = 0.0;
    double *mean_tf = NULL;  
    int listBaseline[NB_BASE_MAX][2];
    int flag=0;
    int i = 0;
    int j = 0;
    int k = 0;
    int indexChipName=0;
    int aux = 0;
    int nbbase = 0;
    char *qcname_mean = NULL;
    char *pszChipName=NULL;
    cpl_errorstate prestate = cpl_errorstate_get();
    double foo=0.;
    
  /*
- Compute the theoretical squared visibility from the angular diameter
double gsl_sf_bessel_J1 (double x)
– Divide the raw squared visibility by the theoretical one
– Compute the error bar of the estimate of transfer function
Store the results in a mat_oitranfunc structures
Compute the QC1 parameters????
Store the results in a cpl_propertylist: qclist
   */

    qclist = cpl_propertylist_new();
    oivis2 = oifits->oivis2;
    if (oifits_vis != NULL)
      {
	oivis  = oifits_vis->oivis;
      }
    /*Allocate the structure and initialize its attributes*/
    result = mat_oifits_new();
    oitf2 = mat_oitf2_new(oivis2->nbvis2, oivis2->nbchannel);
    oitf2->dateobs   = cpl_strdup(oivis2->dateobs);
    oitf2->arrayname = cpl_strdup(oivis2->arrayname);
    oitf2->insname   = cpl_strdup(oivis2->insname);
    mat_oifits_add_tf2(result, oitf2); // adds TF2 to result, needs valid INSNAME and ARRNAME (set before)
    /*Load data */
    for(i=0; i<oitf2->nbtf2; i++)
      {
	mat_tf2elem  *tf2el  = oitf2->list_tf2[i];
	mat_vis2elem *vis2el = oivis2->list_vis2[i];
	aux_ucoord = vis2el->ucoord;
	aux_vcoord = vis2el->vcoord;
	base = sqrt(aux_ucoord*aux_ucoord+aux_vcoord*aux_vcoord);
	tf2el->time            = vis2el->time;
	tf2el->dateobsmjd      = vis2el->dateobsmjd;
	tf2el->exptime         = vis2el->exptime;
	tf2el->stationindex[0] = vis2el->stationindex[0];
	tf2el->stationindex[1] = vis2el->stationindex[1];
	/*compute TF and TF error*/
	for(j=0; j<oitf2->nbchannel; j++)
	  {
	    Z = CPL_MATH_PI*(CPL_MATH_PI*diameter/(180.0*3600.0*1000.0))*base/(oifits->oiwave->effwave[j]);
	    V2star = pow(2*gsl_sf_bessel_J1(Z)/Z,2);
	    V2starErr = 8*fabs(gsl_sf_bessel_Jn(2,Z)*gsl_sf_bessel_J1(Z)/Z)*(diameterErr/diameter);
	    SNR    = vis2el->vis2[j]/vis2el->vis2err[j];
	    SNRcal = V2star/V2starErr;
	    //!!!!!!!!!!!!
	    // ERROR ON THE CALIBRATOR DIAMETER NOT TAKEN INTO ACCOUNT
	    //!!!!!!!!!!!!
	    //SNRcal=1.E5;

	    if (SNRcal > 0)
	      {
		TF     = (vis2el->vis2[j]/V2star) * (1.+1./pow(SNRcal,2.0));
		foo = pow(TF,2.0)*((1./pow(SNR,2.0))+(1./pow(SNRcal,2.0))-(1./pow(SNRcal,4.0))+(3./(pow(SNR,2.0)*pow(SNRcal,2.0))));
		if (foo > 0)
		  {
		    TFErr  = sqrt(foo);
		  }
		else
		  {
		    TFErr   = TF * ( (1./SNRcal) + (1./SNR) );
		  }
		
	      }
	    else if (SNRcal > 2 && SNRcal<=4)
	      {
		TF = (vis2el->vis2[j]/V2star);
		foo = pow(TF,2.0)*((1./pow(SNR,2.0))+(1./pow(SNRcal,2.0))-(1./pow(SNRcal,4.0))+(3./(pow(SNR,2.0)*pow(SNRcal,2.0))));
		if (foo > 0)
		  {
		    TFErr1  = sqrt(foo);
		    TFErr2   = TF * ( (1./SNRcal) + (1./SNR) );
		    TFErr   = ((4-SNRcal) * TFErr2 + (SNRcal -2) * TFErr1) / 2.;
		  }
		else
		  {
		    TFErr   = TF * ( (1./SNRcal) + (1./SNR) );
		  }
		
	      }
	    else
	      {
		TF = (vis2el->vis2[j]/V2star);
		TFErr   = TF * ( (1./SNRcal) + (1./SNR) );
	      }
	    tf2el->tf2[j]    = TF;
	    tf2el->tf2err[j] = TFErr;
	    tf2el->tf[j]    = -1.;
	    tf2el->tferr[j] = 1.E5;
	    if ( oivis != NULL )
	      {
		if (strstr(oivis->amptyp,"differential") == NULL)
		  {
		    mat_viselem  *visel  = oivis->list_vis[i];
		    SNR    = visel->visamp[j]/visel->visamperr[j];
		    SNRcal = 2*SNRcal; // because SNR_V = 2 x SNR_V2
		    if (SNRcal > 0)
		      {
			TF     = (visel->visamp[j]/sqrt(V2star)) * (1.+1./pow(SNRcal,2.0));
			foo = pow(TF,2.0)*((1./pow(SNR,2.0))+(1./pow(SNRcal,2.0))-(1./pow(SNRcal,4.0))+(3./(pow(SNR,2.0)*pow(SNRcal,2.0))));
			if (foo > 0)
			  {
			    TFErr  = sqrt(foo);
			  }
			else
			  {
			    TFErr   = TF * ( (1./SNRcal) + (1./SNR) );
			  }
		      }
		    else if (SNRcal > 2 && SNRcal<=4)
		      {
			TF = (visel->visamp[j]/sqrt(V2star));
			foo = pow(TF,2.0)*((1./pow(SNR,2.0))+(1./pow(SNRcal,2.0))-(1./pow(SNRcal,4.0))+(3./(pow(SNR,2.0)*pow(SNRcal,2.0))));
			if (foo > 0)
			  {
			    TFErr1  = sqrt(foo);
			    TFErr2   = TF * ( (1./SNRcal) + (1./SNR) );
			    TFErr   = ((4-SNRcal) * TFErr2 + (SNRcal -2) * TFErr1) / 2.;
			  }
			else
			  {
			    TFErr   = TF * ( (1./SNRcal) + (1./SNR) );
			  }
		      }
		    else
		      {
			TF = (visel->visamp[j]/sqrt(V2star));
			TFErr   = TF * ( (1./SNRcal) + (1./SNR) );
		      }
		    tf2el->tf[j]    = TF;
		    tf2el->tferr[j] = TFErr;
		  }
	      }
	  }
      }
    result->keywords = cpl_propertylist_new();
    /* Error handling */
    if (!cpl_errorstate_is_equal(prestate)) {     	
	cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);   
	cpl_errorstate_set(prestate);
	cpl_msg_info(cpl_func, "mat_compute_tf2 error");
	cpl_propertylist_delete(qclist);
	return NULL;
    }
    

    // Determine the list of baselines
    nbbase=0;
    for(i=0;i<oifits->oivis2->nbvis2;i++)
      {
	mat_vis2elem *vis2el = oivis2->list_vis2[i];
	flag=1;
	for(j=0;j<nbbase;j++) {
	  if ( vis2el->stationindex[0] == listBaseline[j][0] && 
	       vis2el->stationindex[1] == listBaseline[j][1] )
	    {
	      flag=0;
	      break;
	    }
	}
	if (flag)
	  {
	    listBaseline[nbbase][0] = vis2el->stationindex[0];
	    listBaseline[nbbase][1] = vis2el->stationindex[1];
	    nbbase++;
	  }
      }
    // Detector Index Determination
    pszChipName= 
	(char *)cpl_propertylist_get_string(oifits->keywords,
					    "ESO DET CHIP NAME");
    if (pszChipName == NULL) {
	cpl_msg_error(cpl_func,"No ESO DET CHIP NAME keyword");
	cpl_propertylist_delete(qclist);

	return NULL;
    }
    if(strcmp(pszChipName, "AQUARIUS") == 0) {
	indexChipName=2;
    }
    else if(strcmp(pszChipName, "HAWAII-2RG") == 0) {
	indexChipName=1;
    } else {
	cpl_msg_warning(cpl_func,
			"Wrong chip name found");
	indexChipName=3;
    }



    /*compute QC parameters*/
    mean_tf = cpl_malloc(nbbase*sizeof(double));
    cpl_vector *valqc=NULL;
    valqc=cpl_vector_new(nbbase);
    for(k=0;k<nbbase;k++)
      {
 	mean_tf[k]=0;
	aux=0;
	for(i=0;i<oivis2->nbvis2;i++)
	  {
	    mat_tf2elem  *tf2el  = oitf2->list_tf2[i];
	    mat_vis2elem *vis2el = oivis2->list_vis2[i];
	    if ( vis2el->stationindex[0] == listBaseline[k][0] && 
		 vis2el->stationindex[1] == listBaseline[k][1] )
	      {
		for(j=0;j<oitf2->nbchannel;j++) {
		  if (indexChipName == 1) {
		    if (fabs(oifits->oiwave->effwave[j]*1.e6 - 3.5) <= 0.55 || fabs(oifits->oiwave->effwave[j]*1.e6 - 4.75) <= 0.25) {
		      //if (fabs(oifits->oiwave->effwave[j]*1.e6 - 3.5) <= 0.7 || fabs(oifits->oiwave->effwave[j]*1.e6 - 4.84) <= 0.36) {
		      mean_tf[k] += tf2el->tf2[j];
		      aux++;
		    }
		  } else if (indexChipName == 2) {
		    if (fabs(oifits->oiwave->effwave[j]*1.e6 - 9.0) <= 1.0) {
		      //if (fabs(oifits->oiwave->effwave[j]*1.e6 - 8.59) <= 0.59 || fabs(oifits->oiwave->effwave[j]*1.e6 - 11.28) <= 1.72) {
		      mean_tf[k] += tf2el->tf2[j];
		      aux++;
		    }
		  }
		}
	      }
	  }
	if (aux == 0.)
	  {
	    mean_tf[k]=0.;
	  }
	else
	  {
	    mean_tf[k]/=aux;
	  }
	qcname_mean=cpl_sprintf("ESO QC DET%d TFSQR%d MEAN",indexChipName,k+1);
	//printf("MEAN_TF(%d)= %f\n",k,mean_tf[k]);
	cpl_propertylist_append_double(qclist, qcname_mean, mean_tf[k]);
	cpl_vector_set(valqc,k,mean_tf[k]);
	cpl_free(qcname_mean);
    }    
    cpl_propertylist_append_double(qclist, "ESO QC TF AVG", cpl_vector_get_mean(valqc));
    cpl_propertylist_append_double(qclist, "ESO QC TF STDEV", cpl_vector_get_stdev(valqc));
    if ( cpl_vector_get_stdev(valqc) != 0.)
      {
	cpl_propertylist_append_double(qclist, "ESO QC TF SNR", cpl_vector_get_mean(valqc)/cpl_vector_get_stdev(valqc));
      }
    else
      {
	cpl_propertylist_append_double(qclist, "ESO QC TF SNR", 1.E5);
      }
    cpl_vector_delete(valqc);
    mat_add_generic_qc(oifits->keywords,qclist);
    cpl_propertylist_append(result->keywords, qclist);

    
    cpl_propertylist_append_double(result->keywords,"ESO PRO JSDC DIAMETER",diameter);
    cpl_propertylist_append_double(result->keywords,"ESO PRO JSDC DIAMETER ERROR",diameterErr);
    cpl_propertylist_append_double(result->keywords,"ESO PRO JSDC LMAG",diameterLmag);
    cpl_propertylist_append_double(result->keywords,"ESO PRO JSDC MMAG",diameterMmag);
    cpl_propertylist_append_double(result->keywords,"ESO PRO JSDC NMAG",diameterNmag);
    cpl_propertylist_append_string(result->keywords,"ESO PRO JSDC NAME",diameterName);

    double fluxjy=0.;
    double fluxjsdc=0.;
    
    if (indexChipName == 1 && cpl_propertylist_has(oifits->keywords,"ESO QC DET1 FLUX JY IP1"))
      {
	if (diameterLmag < 1.E5)
	  {
	    // reference : alf Ori
	    fluxjsdc=996.*pow(10.,(-1.29-diameterLmag)/2.5);
	    if (fluxjsdc > 0.)
	      {
		fluxjy=cpl_propertylist_get_double(oifits->keywords,"ESO QC DET1 FLUX JY IP1");
		cpl_propertylist_append_double(result->keywords,"ESO QC DET1 TRANS IP1",fluxjy/fluxjsdc);
		fluxjy=cpl_propertylist_get_double(oifits->keywords,"ESO QC DET1 FLUX JY IP3");
		cpl_propertylist_append_double(result->keywords,"ESO QC DET1 TRANS IP3",fluxjy/fluxjsdc);
		fluxjy=cpl_propertylist_get_double(oifits->keywords,"ESO QC DET1 FLUX JY IP5");
		cpl_propertylist_append_double(result->keywords,"ESO QC DET1 TRANS IP5",fluxjy/fluxjsdc);
		fluxjy=cpl_propertylist_get_double(oifits->keywords,"ESO QC DET1 FLUX JY IP7");
		cpl_propertylist_append_double(result->keywords,"ESO QC DET1 TRANS IP7",fluxjy/fluxjsdc);
	      }
	  }
      }
    else
      {
	if (diameterNmag < 1.E5 && cpl_propertylist_has(oifits->keywords,"ESO QC DET2 FLUX JY IP1"))
	  {
	    // reference : alf Ori
	    fluxjsdc=118.*pow(10.,(-0.99-diameterNmag)/2.5);
	    if (fluxjsdc > 0.)
	      {
		fluxjy=cpl_propertylist_get_double(oifits->keywords,"ESO QC DET2 FLUX JY IP1");
		cpl_propertylist_append_double(result->keywords,"ESO QC DET2 TRANS IP1",fluxjy/fluxjsdc);
		fluxjy=cpl_propertylist_get_double(oifits->keywords,"ESO QC DET2 FLUX JY IP3");
		cpl_propertylist_append_double(result->keywords,"ESO QC DET2 TRANS IP3",fluxjy/fluxjsdc);
		fluxjy=cpl_propertylist_get_double(oifits->keywords,"ESO QC DET2 FLUX JY IP5");
		cpl_propertylist_append_double(result->keywords,"ESO QC DET2 TRANS IP5",fluxjy/fluxjsdc);
		fluxjy=cpl_propertylist_get_double(oifits->keywords,"ESO QC DET2 FLUX JY IP7");
		cpl_propertylist_append_double(result->keywords,"ESO QC DET2 TRANS IP7",fluxjy/fluxjsdc);
	      }
	  }
      }
	
    
    /* nbbase = (oifits->array->nbstation)*(oifits->array->nbstation-1)/2; */
    /* for(k = 0;k < nbbase; k++){ */
    /* 	mean_tf[k ]= 0; */
    /* 	j = k; */
    /* 	aux = 0; */
    /* 	while(j<=length-oifits->array->nbstation) { */
    /* 	    for(i = 0; i<oifits->oitf2->nbchannel; i++) { */
    /* 		mean_tf[k] = mean_tf[k]+(*result).oitf2->list_tf2[j]->tf2[i]; */
    /* 		aux = aux+1; */
    /* 	    } */
    /* 	    j = j+oifits->array->nbstation; */
    /* 	} */
    /* 	mean_tf[k] = mean_tf[k]/aux; */
    /* 	qcname_mean = cpl_sprintf("ESO QC DET TF2%d MEAN", k+1); */
    /* 	printf("MEAN_TF(%d)= %f\n",k,mean_tf[k]); */
    /* 	cpl_propertylist_append_double(qclist, qcname_mean, mean_tf[k]); */
    /* 	cpl_propertylist_append(result->keywords, qclist); */
    /* 	cpl_free(qcname_mean); */
    /* } */
    cpl_free(mean_tf);
    cpl_propertylist_delete(qclist);

    return result;
}

 
