/* $Id: mat_detector.h,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_detector.h $
 */

#ifndef MAT_DETECTOR_H
#define MAT_DETECTOR_H


/*-----------------------------------------------------------------------------
                      Includes
------------------------------------------------------------------------------*/
#include <complex.h>
#include "cpl.h"
#include <complex.h>

#include "mat_const.h"
#include "mat_drl.h"
/*-----------------------------------------------------------------------------
                      Defines
------------------------------------------------------------------------------*/

#define MAT_IGNORE_INTENSITY     -4
#define MAT_OUTLIER_INTENSITY    -3
#define MAT_BADPIX_INTENSITY     -2
#define MAT_MASKED_INTENSITY     -1
#define MAT_UNDEFINED_INTENSITY   0
#define MAT_LOW_INTENSITY         1
#define MAT_LINEAR_INTENSITY      2
#define MAT_NONLINEAR_INTENSITY   3
#define MAT_SATURATED_INTENSITY   4

#define MAT_H2_LEFT           130
#define MAT_H2_RIGHT         1910

#define MAT_H2_BOTTOM          50
#define MAT_H2_TOP           1980

#define MAT_AQ_PHOTO1_LEFT     52
#define MAT_AQ_PHOTO1_RIGHT   143

#define MAT_AQ_PHOTO2_LEFT    158
#define MAT_AQ_PHOTO2_RIGHT   249

#define MAT_AQ_INTERF_LEFT    273
#define MAT_AQ_INTERF_RIGHT   763

//#define MAT_AQ_INTERF_LEFT   260
//#define MAT_AQ_INTERF_RIGHT  760

#define MAT_AQ_PHOTO3_LEFT    774
#define MAT_AQ_PHOTO3_RIGHT   855

#define MAT_AQ_PHOTO4_LEFT    878
#define MAT_AQ_PHOTO4_RIGHT   970

#define MAT_AQ_BOTTOM          40
#define MAT_AQ_TOP            980

/**
   @ingroup det
*/
typedef enum {
  MAT_UNKNOWN_DET,
  MAT_HAWAII2RG_DET,
  MAT_AQUARIUS_DET
} mat_detector_type;

/*-----------------------------------------------------------------------------
                      Data Structures
------------------------------------------------------------------------------*/

/**
   @ingroup det
   @brief This structure contains all information which describes a detector used for a specific raw data file.

   This information contains the physical properties of a detector, the read out mode,
   the size of the raw data and the properties of the metal frame before a detector
   (which blocks the light for the pixels on the dtector border). For calculating the
   freqeuency of the power spectra peaks, the pixel timing is also specified.
*/
typedef struct {
  mat_detector_type   type;             /*!< detector type as an enumeration (MAT_*_DET) */
  int                 nr;               /*!< the detector number HAWAII2RG: 1, Aquarius: 2, other:1 */
  char                id[80];           /*!< DET CHIP ID */
  char                name[80];         /*!< DET CHIP NAME */
  int                 nx;               /*!< total detector width in pixel */
  int                 ny;               /*!< total detector height in pixel */
  int                 read_id;          /*!< DET READ CURID */
  char                read_name[80];    /*!< DET READ CURNAME */
  int                 channel_nrows;    /*!< number of channel rows (HAWAII-2RG: 1, Aquarius: 2, other: 1) */
  int                 channel_ncolumns; /*!< number of channel columns (HAWAII-2RG: 32, Aquarius: 32, other: 1) */
  int                 channel_nx;       /*!< width of a detector channel (calculated) */
  int                 channel_ny;       /*!< height of a detector channel (calculated) */
  int                 frame_left;       /*!< left frame width (HAWAII-2RG: 128, Aquarius: 32, other: 0) */
  int                 frame_right;      /*!< right frame width (HAWAII-2RG: 128, Aquarius: 32, other: 0) */
  int                 frame_bottom;     /*!< bottom frame height (HAWAII-2RG: 32, Aquarius: 32, other: 0) */
  int                 frame_top;        /*!< top frame height (HAWAII-2RG: 32, Aquarius: 32, other: 0) */
  int                 data_nregions;    /*!< number of data sub-windows */
  int                 data_nx;          /*!< width of the one data sub-window */
  int                 data_ny;          /*!< height of the one data sub-window */
  double              horizontal_timing[4]; /*!< horizontal pixel timing: [0] + nx*[1] + ny*[2] + nx*ny*[3] */
  double              vertical_timing[4];   /*!< vertical pixel timing: [0] + nx*[1] + ny*[2] + nx*ny*[3] */
  double              linear_range;     /*!< top of the linear range for a detector in DU */
  char                fname[256];       /*!< memory for a file name */
} mat_detector;

mat_detector *mat_detector_new(void);
mat_detector *mat_detector_duplicate(mat_detector *det);
cpl_error_code mat_detector_delete(mat_detector *det);
cpl_error_code mat_detector_decode(mat_detector *det, cpl_propertylist *plist);
cpl_error_code mat_detector_decode_raw(mat_detector *det, mat_gendata *raw);
cpl_error_code mat_detector_check(mat_detector *det, mat_detector *rdet);
cpl_error_code mat_detector_check_raw(mat_detector *det, mat_gendata *raw);
cpl_error_code mat_detector_check_raw_full(mat_detector *det, mat_gendata *raw);
cpl_error_code mat_detector_check_frame(mat_detector *det, mat_frame *data);
cpl_error_code mat_detector_check_frame_full(mat_detector *det, mat_frame *data);
int mat_detector_is_reference(mat_detector *det, int x, int y);
int mat_detector_contains_reference(mat_detector *det, int x, int y, int nx, int ny);
int mat_detector_decode_reference(mat_detector *det, int x, int y, int nx, int ny);
int mat_detector_contains_nonreference(mat_detector *det, int x, int y, int nx, int ny);
char *mat_detector_get_bpm_name(mat_detector *det);
char *mat_detector_get_ffm_name(mat_detector *det);
char *mat_detector_get_nlm_name(mat_detector *det);
char *mat_detector_get_imb_name(mat_detector *det);
char *mat_detector_get_ime_name(mat_detector *det);
char *mat_detector_get_imr_name(mat_detector *det);
char *mat_detector_get_offm_name(mat_detector *det);

cpl_error_code mat_detector_calc_timing(mat_detector *det,
					cpl_propertylist *plist,
					double *mindit,
					double *tpixel,
					double *tline,
					double *tframe);

#endif
