/* $Id: mat_shift.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_eop.c $
 */

#ifndef _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 200809L
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <string.h>

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "mat_eop.h"
#include "mat_error.h"
/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/
#define LINE_SIZE 188
#define BUFFER_LENGTH 32768

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

cpl_error_code mat_eop_interpolate (cpl_size n, double *mjd,
				      double *pmx, double *pmy,
				      double *dut,
                                      cpl_table * eop_table);

void eraAtcoq (double rc, double dc, double pmr, double pmd, double px,
	       double rv, eraASTROM *astrom, double enuob[3]);
void dtp2s (double xi, double eta, double raz,
	    double decz, double *ra, double *dec);

void difference (double x[3], double y[3], double z[3]);
void multiply (double xyz[3], double factor);
void normalize (double xyz[3]);
void cross (double x[3], double y[3], double z[3]);
double mat_get_decep (const cpl_propertylist * plist, double coef);
double mat_get_raep(const cpl_propertylist * plist, double coef);
double mat_get_robj_raep(const cpl_propertylist * plist);
double mat_get_robj_decep(const cpl_propertylist * plist);
double mat_get_sobj_x (const cpl_propertylist * plist);
double mat_get_sobj_y (const cpl_propertylist * plist);
const char * mat_get_string_default (const cpl_propertylist * plist,
				     const char *name,
				     const char *def);
void eraAtboq (double rc, double dc, eraASTROM *astrom, double enuob[3]);
void rotate_vector (double in[3], double angle, double axis[3], double out[3]);
/*-----------------------------------------------------------------------------
                             Functions code
 -----------------------------------------------------------------------------*/
/* 
 * Private function to interpolate EOP parameter from EOP_PARAM
 * and manipulate coordinates and 3D vectors
 */

cpl_error_code mat_eop_interpolate (cpl_size n, double *mjd,
                                      double *pmx, double *pmy,
                                      double *dut,
                                      cpl_table * eop_table)
{  
  cpl_ensure_code (eop_table, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code (n>0, CPL_ERROR_ILLEGAL_INPUT);
  cpl_ensure_code (mjd, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code (pmx, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code (pmy, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code (dut, CPL_ERROR_NULL_INPUT);

  /* Wrap inputs in vectors */
  cpl_vector *vmjd = cpl_vector_wrap (n, mjd);
  cpl_vector *vpmx = cpl_vector_wrap (n, pmx);
  cpl_vector *vpmy = cpl_vector_wrap (n, pmy);
  cpl_vector *vdut = cpl_vector_wrap (n, dut);

  /* Check validity of input mjd with table */
  if (cpl_vector_get_min (vmjd) < cpl_table_get_column_min (eop_table, "MJD") ||
      cpl_vector_get_max (vmjd) > cpl_table_get_column_max (eop_table, "MJD"))
  {
    cpl_msg_warning (cpl_func, "Some MJD are outside the EOP_PARAM range (MJD=%.2f, %.2f..%.2f). Use EOP and DUT=0.0s",
					 cpl_vector_get_mean (vmjd),
					 cpl_table_get_column_min (eop_table, "MJD"),
					 cpl_table_get_column_max (eop_table, "MJD"));
    cpl_vector_unwrap (vmjd);
    cpl_vector_unwrap (vpmx);
    cpl_vector_unwrap (vpmy);
    cpl_vector_unwrap (vdut);
    return CPL_ERROR_NONE;
  }

  /* /\* Check if within the prediction range *\/ */
  /* if(cpl_vector_get_max (vmjd) > cpl_propertylist_get_double (header, "ESO QC EOP MJD LAST FINAL")) */
  /* { */
  /*   cpl_msg_warning (cpl_func, "Some MJD are inside the EOP_PARAM prediction range..."); */
  /* } */

  /* Wrap output vectors */
  cpl_size nrow = cpl_table_get_nrow (eop_table);
  cpl_vector * eop_mjd = cpl_vector_wrap (nrow, cpl_table_get_data_double (eop_table, "MJD"));
  cpl_vector * eop_pmx = cpl_vector_wrap (nrow, cpl_table_get_data_double (eop_table, "PMX"));
  cpl_vector * eop_pmy = cpl_vector_wrap (nrow, cpl_table_get_data_double (eop_table, "PMY"));
  cpl_vector * eop_dut = cpl_vector_wrap (nrow, cpl_table_get_data_double (eop_table, "DUT"));
  
  /* Perform interpolations */
  cpl_bivector *mjd_pmx = cpl_bivector_wrap_vectors (vmjd, vpmx);
  cpl_bivector *eop_mjd_pmx = cpl_bivector_wrap_vectors (eop_mjd, eop_pmx);
  cpl_bivector_interpolate_linear (mjd_pmx, eop_mjd_pmx);
  cpl_bivector_unwrap_vectors (mjd_pmx);
  cpl_bivector_unwrap_vectors (eop_mjd_pmx);

  cpl_bivector *mjd_pmy = cpl_bivector_wrap_vectors (vmjd, vpmy);
  cpl_bivector *eop_mjd_pmy = cpl_bivector_wrap_vectors (eop_mjd, eop_pmy);
  cpl_bivector_interpolate_linear (mjd_pmy, eop_mjd_pmy);
  cpl_bivector_unwrap_vectors (mjd_pmy);
  cpl_bivector_unwrap_vectors (eop_mjd_pmy);

  cpl_bivector *mjd_dut = cpl_bivector_wrap_vectors (vmjd, vdut);
  cpl_bivector *eop_mjd_dut = cpl_bivector_wrap_vectors (eop_mjd, eop_dut);
  cpl_bivector_interpolate_linear (mjd_dut, eop_mjd_dut);
  cpl_bivector_unwrap_vectors (mjd_dut);
  cpl_bivector_unwrap_vectors (eop_mjd_dut);
  
  /* Print diagnostics */
  cpl_msg_info(cpl_func, "EOP averages: MJD=%.2f DUT1=%.3f PMX=%.3farcsec PMY=%.3farcsec",
			   cpl_vector_get_mean (vmjd), cpl_vector_get_mean (vdut),
			   cpl_vector_get_mean (vpmx), cpl_vector_get_mean (vpmy));

  /* Unwrap vectors */
  cpl_vector_unwrap (vmjd);
  cpl_vector_unwrap (vpmx);
  cpl_vector_unwrap (vpmy);
  cpl_vector_unwrap (vdut);
  cpl_vector_unwrap (eop_mjd);
  cpl_vector_unwrap (eop_pmx);
  cpl_vector_unwrap (eop_pmy);
  cpl_vector_unwrap (eop_dut);
  

  return CPL_ERROR_NONE;
}


/*
 * Quick transform from Celestial to Observed
 *     rc,dc  double   ICRS right ascension at J2000.0 (radians)
 *     pr     double   RA proper motion dRA/dt, not cos(Dec) x dRA/dt (radians/year)
 *     pd     double   Dec proper motion dDec/dt (radians/year)
 *     px     double   parallax (arcsec)
 *     rv     double   radial velocity (km/s, +ve if receding)
 */
void eraAtcoq(double rc, double dc, double pmr, double pmd, double px, double rv, eraASTROM *astrom, double enuob[3])
{
    double ri, di;
    double aob, zob, hob, dob, rob;

    /* Transform from Celestial to Intermediate */
    eraAtciq (rc, dc, pmr, pmd, px, rv, astrom, &ri, &di);

    /* Transform Intermediate to Observed. */
    eraAtioq (ri, di, astrom, &aob, &zob, &hob, &dob, &rob);

    /* Convert from equatorial to cartesian */
    enuob[0] = sin(zob) * sin(aob);
    enuob[1] = sin(zob) * cos(aob);
    enuob[2] = cos(zob);
}


void dtp2s(double xi, double eta, double raz, double decz, double *ra, double *dec)
{
    double sdecz = sin(decz);
    double cdecz = cos(decz);
    double denom = cdecz - eta * sdecz;
    double d = atan2(xi, denom) + raz;
    *ra = fmod(d, 2.0 * CPL_MATH_PI);
    *dec = atan2(sdecz + eta * cdecz, sqrt(xi * xi + denom * denom));
}

void difference(double x[3], double y[3], double z[3])
{
    for (int i=0; i<3; i++) {
	  z[i] = x[i] - y[i];
    }
}

void multiply(double xyz[3], double factor)
{
    xyz[0] *= factor;
    xyz[1] *= factor;
    xyz[2] *= factor;
}

void normalize(double xyz[3])
{
    double norm = sqrt(xyz[0]*xyz[0] + xyz[1]*xyz[1] + xyz[2]*xyz[2]);
    multiply (xyz, 1.0/norm);
}

void cross(double x[3], double y[3], double z[3])
{
    z[0] = x[1] * y[2] - x[2] * y[1];
    z[1] = x[2] * y[0] - x[0] * y[2];
    z[2] = x[0] * y[1] - x[1] * y[0];
}

void rotate_vector (double in[3], double angle, double axis[3], double out[3])
{
	double rv[3];
	double rm[3][3];
	eraSxp(angle, axis, rv);
	eraRv2m(rv, rm);
	eraRxp(rm, in, out);
}

void eraAtboq (double rc, double dc, eraASTROM *astrom, double enuob[3])
{
    double ri, di;
    double aob, zob, hob, dob, rob;

    /* Transform from BCRS to Intermediate */
    eraAtciq (rc, dc, 0.0, 0.0, 0.0, 0.0, astrom, &ri, &di);

    /* Transform from Intermediate to Observed. */
    eraAtioq (ri, di, astrom, &aob, &zob, &hob, &dob, &rob);

    /* Convert from equatorial to cartesian */
	eraS2c(CPL_MATH_PI/2.0-aob, CPL_MATH_PI/2.0-zob, enuob);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Compute the UCOORD and VCOORD (uv)
 * 
 * @param input_table   input/output data
 * @param oi_array      input OI_ARRAY table
 * @param header        input header
 * @param eop_table:    table containing the EOP pameters
 * 
 * This function re-computes the UCOORD and VCOORD of the OI_VIS
 * tables using the HEADER coordinates, the OI_ARRAY baseline and
 * the MJD time of the OI_VIS table. It is based on ERFA library.
 * The SC and FT OI_VIS tables are updated.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code mat_eop_uv_oivis(mat_oivis * oivis,
				mat_array * oiarray,
				cpl_propertylist * hdr_data,
				cpl_table * eop_table)
{

  cpl_ensure_code (oivis,   CPL_ERROR_NULL_INPUT);
  cpl_ensure_code (oiarray, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code (hdr_data, CPL_ERROR_NULL_INPUT);
  

  cpl_table * oi_vis = mat_oivis_to_table(oivis,NULL);
  cpl_table * oi_array = mat_array_to_table(oiarray,NULL);



  double t_skip = 2./24/3600, mjd0 = -1.0, mjd1 = -1.0;
  cpl_msg_info (cpl_func, "Compute pointing with full ERFA every %.2f s",t_skip*24*3600.0);

  /* Read UTC as MJD */
  double *mjd = cpl_table_get_data_double (oi_vis, "MJD");
  double mean_mjd = cpl_table_get_column_mean (oi_vis, "MJD");

  /* Checked by J. Woillez */
  double elev=0.;
  double lon=0.;
  double lat=0.;
  if (cpl_propertylist_has(hdr_data,"ESO ISS GEOELEV")) {
    elev = cpl_propertylist_get_double(hdr_data,"ESO ISS GEOELEV"); // Height in [m]
  }
  if (cpl_propertylist_has(hdr_data,"ESO ISS GEOLON")) {
    lon = cpl_propertylist_get_double(hdr_data,"ESO ISS GEOLON") * CPL_MATH_RAD_DEG; // Lat in [rad]
  }
  if (cpl_propertylist_has(hdr_data,"ESO ISS GEOLAT")) {
    lat = cpl_propertylist_get_double(hdr_data,"ESO ISS GEOLAT") * CPL_MATH_RAD_DEG; // Lon in [rad], East positive
  }
  
  /* Compute Earth Orientation Parameters for the mean MJD */
  double dut1 = 0, pmx = 0, pmy = 0;
  if (eop_table != NULL) {
      mat_eop_interpolate (1, &mean_mjd, &pmx, &pmy, &dut1, eop_table);
  } else {
      cpl_msg_warning (cpl_func, "No EOP_PARAM. Use EOP and DUT=0.0s");
  }
    
  /* We use the FT coordinate for all uv coordinates */
  double raep=0.;
  double decp=0.;
  double pmra=0.;
  double pmdec=0.;
  double parallax = 0.;
  double sysvel = 0.0;
  /* if (strstr(cpl_propertylist_get_string(hdr_data,"INSTRUME"),"GRAVITY") != NULL) { */
  /*   raep = mat_get_raep (hdr_data,0.5); // [rad] */
  /*   decp = mat_get_decep (hdr_data,0.5); // [rad] */
  /*   if  (cpl_propertylist_has(hdr_data,"ESO FT ROBJ PMA")) { */
  /*     const char * str=cpl_propertylist_get_string(hdr_data,"ESO FT ROBJ PMA"); */
  /*     sscanf(str,"%lf",&pmra); */
  /*     pmra *= CPL_MATH_RAD_DEG / 3600.0 / cos(decp); // dRA/dt, not cos(Dec)xdRA/dt [rad/year] */
  /*   } */
  /*   if  (cpl_propertylist_has(hdr_data,"ESO FT ROBJ PMD")) { */
  /*     const char * str=cpl_propertylist_get_string(hdr_data,"ESO FT ROBJ PMD"); */
  /*     sscanf(str,"%lf",&pmdec); */
  /*     pmdec *= CPL_MATH_RAD_DEG / 3600.0; // dDec/dt [rad/year] */
  /*   } */
  /*   if  (cpl_propertylist_has(hdr_data,"ESO FT ROBJ PARALLAX")) { */
  /*     parallax=cpl_propertylist_get_double(hdr_data,"ESO FT ROBJ PARALLAX"); // [as] */
  /*   } */
  /* } else { */
    if  (cpl_propertylist_has(hdr_data,"RA")) {
      raep = cpl_propertylist_get_double(hdr_data,"RA");
    /* if  (cpl_propertylist_has(hdr_data,"ESO COU GUID RA")) { */
    /*   raep = cpl_propertylist_get_double(hdr_data,"ESO COU GUID RA"); */
      raep *= 2.* CPL_MATH_PI / 360.; 
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO COU GUID RA not found");
    }
    if  (cpl_propertylist_has(hdr_data,"DEC")) {
      decp = cpl_propertylist_get_double(hdr_data,"DEC");
    /* if  (cpl_propertylist_has(hdr_data,"ESO COU GUID DEC")) { */
    /*   decp = cpl_propertylist_get_double(hdr_data,"ESO COU GUID DEC"); */
      decp *= 2.* CPL_MATH_PI / 360.;
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO COU GUID DEC not found");
    }
    if  (cpl_propertylist_has(hdr_data,"ESO TEL ARG PMA")) {
      pmra= cpl_propertylist_get_double(hdr_data,"ESO TEL ARG PMA");
      pmra *= CPL_MATH_RAD_DEG / 3600.0 / cos(decp); 
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO TEL ARG PMA not found");
    }
    if  (cpl_propertylist_has(hdr_data,"ESO TEL ARG PMD")) {
      pmdec= cpl_propertylist_get_double(hdr_data,"ESO TEL ARG PMD");
      pmdec *= CPL_MATH_RAD_DEG / 3600.0; 
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO TEL ARG PMD not found");
    }
    if  (cpl_propertylist_has(hdr_data,"ESO TEL ARG PARALLAX")) {
      parallax = cpl_propertylist_get_double(hdr_data,"ESO TEL ARG PARALLAX");
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO TEL ARG PARALLAX not found");
    }

  /* } */

  /* Allocate memory for the tmp computations */
  eraASTROM astrom;
  /* double eo, rcUp, dcUp, rcUm, dcUm, rcVp, dcVp, rcVm, dcVm; */
  /* double enuobUp[3], enuobUm[3], enuobVp[3], enuobVm[3], eu[3], ev[3]; */
  
  
  /* Step for finite difference, 10 arcsec chosen for optimal accuracy */
  double eps = 10.0 / 3600.0 * CPL_MATH_RAD_DEG; // [rad]

  double eo;
  double rb, db;
  double eUb[3], eVb[3], eWb[3], eZb[3];
  double eWo_up[3], eWo_um[3], eWo_vp[3], eWo_vm[3];
  double eWb_up[3], eWb_um[3], eWb_vp[3], eWb_vm[3];
  double rb_up, db_up, rb_um, db_um, rb_vp, db_vp, rb_vm, db_vm;
  double eUo[3], eVo[3], eWo[3];
  double norm;
  double pressure = 0.0; // Pressure at zero to disable atmospheric refraction
  double temperature = 0.0;
  double humidity = 0.0;
  double wavelength = 0.0;

  /* Prepare centered finite differences
   * eU corresponds to +RA
   * eV corresponds to +DEC */
  /* dtp2s (+eps, 0.0, raep, decp, &rcUp, &dcUp); */
  /* dtp2s (-eps, 0.0, raep, decp, &rcUm, &dcUm); */
  /* dtp2s (0.0, +eps, raep, decp, &rcVp, &dcVp); */
  /* dtp2s (0.0, -eps, raep, decp, &rcVm, &dcVm); */

  /* Loop on bases to compute the physical 3D baseline as T2-T1 [m]
   * Note: The telescope locations STAXYZ are given in the [West,South,Up]
   * frame (ESO convention), whereas the UVW, calculated later on, are in
   * the [East,North,Up] frame. Hence the sign changes below on X and Y. */
  double baseline[6][3];
  for ( int base = 0; base < 6; base ++) {
	  int tel1=0; while ( cpl_table_get (oi_array, "STA_INDEX", tel1, NULL) != mat_table_get_value (oi_vis, "STA_INDEX", base, 0) ) tel1++;
	  int tel2=0; while ( cpl_table_get (oi_array, "STA_INDEX", tel2, NULL) != mat_table_get_value (oi_vis, "STA_INDEX", base, 1) ) tel2++;
	  baseline[base][0] = -(mat_table_get_value (oi_array, "STAXYZ", tel2, 0) - mat_table_get_value (oi_array, "STAXYZ", tel1, 0));
	  baseline[base][1] = -(mat_table_get_value (oi_array, "STAXYZ", tel2, 1) - mat_table_get_value (oi_array, "STAXYZ", tel1, 1));
	  baseline[base][2] = +(mat_table_get_value (oi_array, "STAXYZ", tel2, 2) - mat_table_get_value (oi_array, "STAXYZ", tel1, 2));
  }
  
  
  /* Loop on rows. */
  cpl_size n_row = cpl_table_get_nrow (oi_vis);
  for (cpl_size row = 0 ; row < n_row ; row++) {
    /* Transformation from ICRS to Observed (neglect refraction, all at zero)
     * Update precession/nudation every t_skip, otherwise rotate earth only
     * If the time is strickly the same as previous row, we skip this computation */
      
    if (mjd[row] != mjd1 ) {
          
      if ( fabs (mjd[row]-mjd0) > t_skip ) {
	eraApco13 (2400000.5, mjd[row], dut1, lon, lat, elev, pmx/3600.0*CPL_MATH_RAD_DEG,
		   pmy/3600.0*CPL_MATH_RAD_DEG, pressure, temperature, humidity, wavelength, &astrom, &eo);
	mjd0 = mjd[row]; 
      } else {
	eraAper13 (2400000.5, mjd[row] + dut1/(24.0*3600.0), &astrom);
      }




            /* Apply proper motion (ICRS->BCRS) */
            eraPmpx(raep, decp, pmra, pmdec, parallax, sysvel, astrom.pmt, astrom.eb, eWb);
            eraC2s(eWb, &rb, &db);
            
            /* Create (eU,eV,eW) in the BCRS */
            eraS2c(0.0, CPL_MATH_PI/2.0, eZb);  // eZc is the unit vector to the ICRS pole
            eraPxp(eZb, eWb, eUb);  // eUc is the cross product of eZc and eWc
            eraPn(eUb, &norm, eUb);  // eUc is normalized to a unit vector
            eraPxp(eWb, eUb, eVb);  // eWc is the cross product of eWc and eUc
            
            /* Create a 10 arcsec cardinal asterism around eWb in the BCRS */
            rotate_vector(eWb, -eps, eVb, eWb_up);
            rotate_vector(eWb, +eps, eVb, eWb_um);
            rotate_vector(eWb, +eps, eUb, eWb_vp);
            rotate_vector(eWb, -eps, eUb, eWb_vm);
            eraC2s(eWb_up, &rb_up, &db_up);
            eraC2s(eWb_um, &rb_um, &db_um);
            eraC2s(eWb_vp, &rb_vp, &db_vp);
            eraC2s(eWb_vm, &rb_vm, &db_vm);
            
            /* Transform the pointing direction and the cardinal asterism from BCRS to observed */
            eraAtboq (rb   , db   , &astrom, eWo   );
            eraAtboq (rb_up, db_up, &astrom, eWo_up);
            eraAtboq (rb_um, db_um, &astrom, eWo_um);
            eraAtboq (rb_vp, db_vp, &astrom, eWo_vp);
            eraAtboq (rb_vm, db_vm, &astrom, eWo_vm);
            
            /* Compute the observed (eUo,eVo,eWo) reference frame */
            eraPxp(eWo_up, eWo_um, eUo);
            eraPxp(eWo, eUo, eUo);
            eraSxp(1./(2.0*eps), eUo, eUo);
            eraPxp(eWo_vp, eWo_vm, eVo);
            eraPxp(eWo, eVo, eVo);
            eraSxp(1./(2.0*eps), eVo, eVo);



      /* /\* Transform from celestial to intermediate, compute eU *\/ */
      /* eraAtcoq (rcUp, dcUp, pmra, pmdec, parallax, sysvel, &astrom, enuobUp); */
      /* eraAtcoq (rcUm, dcUm, pmra, pmdec, parallax, sysvel, &astrom, enuobUm); */
      /* difference (enuobUp, enuobUm, eu); */
      /* normalize (eu); */
          
      /* /\* Transform from celestial to intermediate, compute eV *\/ */
      /* eraAtcoq (rcVp, dcVp, pmra, pmdec, parallax, sysvel, &astrom, enuobVp); */
      /* eraAtcoq (rcVm, dcVm, pmra, pmdec, parallax, sysvel, &astrom, enuobVm); */
      /* difference (enuobVp, enuobVm, ev); */
      /* normalize (ev); */
    }
      
    /* Project physical baseline into u,v */
    int base = row % 6;
    /* oivis->list_vis[row]->ucoord = eu[0] * baseline[base][0] + eu[1] * baseline[base][1] + eu[2] * baseline[base][2]; */
    /* oivis->list_vis[row]->vcoord = ev[0] * baseline[base][0] + ev[1] * baseline[base][1] + ev[2] * baseline[base][2]; */
    double n_air = 1.0002028;
    oivis->list_vis[row]->ucoord = (eUo[0] * baseline[base][0] + eUo[1] * baseline[base][1] + eUo[2] * baseline[base][2])/n_air;
    oivis->list_vis[row]->vcoord = (eVo[0] * baseline[base][0] + eVo[1] * baseline[base][1] + eVo[2] * baseline[base][2])/n_air;
      
    mjd1 = mjd[row];
  } /* End loop on rows */
  

  cpl_table_delete(oi_array);
  cpl_table_delete(oi_vis);
  

  return CPL_ERROR_NONE;
}


cpl_error_code mat_eop_uv_oivis2(mat_oivis2 * oivis2,
				 mat_array * oiarray,
				 cpl_propertylist * hdr_data,
				 cpl_table * eop_table)
{

  cpl_ensure_code (oivis2,   CPL_ERROR_NULL_INPUT);
  cpl_ensure_code (oiarray, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code (hdr_data, CPL_ERROR_NULL_INPUT);
  

  cpl_table * oi_vis2 = mat_oivis2_to_table(oivis2,NULL);
  cpl_table * oi_array = mat_array_to_table(oiarray,NULL);



  double t_skip = 2./24/3600, mjd0 = -1.0, mjd1 = -1.0;
  cpl_msg_info (cpl_func, "Compute pointing with full ERFA every %.2f s",t_skip*24*3600.0);

  /* Read UTC as MJD */
  double *mjd = cpl_table_get_data_double (oi_vis2, "MJD");
  double mean_mjd = cpl_table_get_column_mean (oi_vis2, "MJD");

  /* Checked by J. Woillez */
  double elev=0.;
  double lon=0.;
  double lat=0.;
  if (cpl_propertylist_has(hdr_data,"ESO ISS GEOELEV")) {
    elev = cpl_propertylist_get_double(hdr_data,"ESO ISS GEOELEV"); // Height in [m]
  }
  if (cpl_propertylist_has(hdr_data,"ESO ISS GEOLON")) {
    lon = cpl_propertylist_get_double(hdr_data,"ESO ISS GEOLON") * CPL_MATH_RAD_DEG; // Lat in [rad]
  }
  if (cpl_propertylist_has(hdr_data,"ESO ISS GEOLAT")) {
    lat = cpl_propertylist_get_double(hdr_data,"ESO ISS GEOLAT") * CPL_MATH_RAD_DEG; // Lon in [rad], East positive
  }
  
  /* Compute Earth Orientation Parameters for the mean MJD */
  double dut1 = 0, pmx = 0, pmy = 0;
  if (eop_table != NULL) {
      mat_eop_interpolate (1, &mean_mjd, &pmx, &pmy, &dut1, eop_table);
  } else {
      cpl_msg_warning (cpl_func, "No EOP_PARAM. Use EOP and DUT=0.0s");
  }
    
  double raep=0.;
  double decp=0.;
  double pmra=0.;
  double pmdec=0.;
  double parallax = 0.;
  double sysvel = 0.0;
  /* if (strstr(cpl_propertylist_get_string(hdr_data,"INSTRUME"),"GRAVITY") != NULL) { */
  /*   raep = mat_get_raep (hdr_data,0.5); // [rad] */
  /*   decp = mat_get_decep (hdr_data,0.5); // [rad] */
  /*   if  (cpl_propertylist_has(hdr_data,"ESO FT ROBJ PMA")) { */
  /*     const char * str=cpl_propertylist_get_string(hdr_data,"ESO FT ROBJ PMA"); */
  /*     sscanf(str,"%lf",&pmra); */
  /*     pmra *= CPL_MATH_RAD_DEG / 3600.0 / cos(decp); // dRA/dt, not cos(Dec)xdRA/dt [rad/year] */
  /*   } */
  /*   if  (cpl_propertylist_has(hdr_data,"ESO FT ROBJ PMD")) { */
  /*     const char * str=cpl_propertylist_get_string(hdr_data,"ESO FT ROBJ PMD"); */
  /*     sscanf(str,"%lf",&pmdec); */
  /*     pmdec *= CPL_MATH_RAD_DEG / 3600.0; // dDec/dt [rad/year] */
  /*   } */
  /*   if  (cpl_propertylist_has(hdr_data,"ESO FT ROBJ PARALLAX")) { */
  /*     parallax=cpl_propertylist_get_double(hdr_data,"ESO FT ROBJ PARALLAX"); // [as] */
  /*   } */
  /* } else { */

  
    /* if  (cpl_propertylist_has(hdr_data,"ESO COU GUID RA")) { */
    /*   raep = cpl_propertylist_get_double(hdr_data,"ESO COU GUID RA"); */
    if  (cpl_propertylist_has(hdr_data,"RA")) {
      raep = cpl_propertylist_get_double(hdr_data,"RA");
      raep *= 2.* CPL_MATH_PI / 360.; 
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO COU GUID RA not found");
    }
    if  (cpl_propertylist_has(hdr_data,"DEC")) {
      decp = cpl_propertylist_get_double(hdr_data,"DEC");
    /* if  (cpl_propertylist_has(hdr_data,"ESO COU GUID DEC")) { */
    /*   decp = cpl_propertylist_get_double(hdr_data,"ESO COU GUID DEC"); */
      decp *= 2.* CPL_MATH_PI / 360.;
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO COU GUID DEC not found");
    }
    if  (cpl_propertylist_has(hdr_data,"ESO TEL TARG PMA")) {
      pmra= cpl_propertylist_get_double(hdr_data,"ESO TEL TARG PMA");
      pmra *= CPL_MATH_RAD_DEG / 3600.0 / cos(decp); 
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO TEL TARG PMA not found");
    }
    if  (cpl_propertylist_has(hdr_data,"ESO TEL TARG PMD")) {
      pmdec= cpl_propertylist_get_double(hdr_data,"ESO TEL TARG PMD");
      pmdec *= CPL_MATH_RAD_DEG / 3600.0; 
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO TEL TARG PMD not found");
    }
    if  (cpl_propertylist_has(hdr_data,"ESO TEL TARG PARALLAX")) {
      parallax = cpl_propertylist_get_double(hdr_data,"ESO TEL TARG PARALLAX");
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO TEL TARG PARALLAX not found");
    }

  /* } */

  /* Allocate memory for the tmp computations */
  eraASTROM astrom;
  /* double eo, rcUp, dcUp, rcUm, dcUm, rcVp, dcVp, rcVm, dcVm; */
  /* double enuobUp[3], enuobUm[3], enuobVp[3], enuobVm[3], eu[3], ev[3]; */
  
  
  /* Step for finite difference, 10 arcsec chosen for optimal accuracy */
  double eps = 10.0 / 3600.0 * CPL_MATH_RAD_DEG; // [rad]

  double eo;
  double rb, db;
  double eUb[3], eVb[3], eWb[3], eZb[3];
  double eWo_up[3], eWo_um[3], eWo_vp[3], eWo_vm[3];
  double eWb_up[3], eWb_um[3], eWb_vp[3], eWb_vm[3];
  double rb_up, db_up, rb_um, db_um, rb_vp, db_vp, rb_vm, db_vm;
  double eUo[3], eVo[3], eWo[3];
  double norm;
  double pressure = 0.0; // Pressure at zero to disable atmospheric refraction
  double temperature = 0.0;
  double humidity = 0.0;
  double wavelength = 0.0;
  


  /* Prepare centered finite differences
   * eU corresponds to +RA
   * eV corresponds to +DEC */
  /* dtp2s (+eps, 0.0, raep, decp, &rcUp, &dcUp); */
  /* dtp2s (-eps, 0.0, raep, decp, &rcUm, &dcUm); */
  /* dtp2s (0.0, +eps, raep, decp, &rcVp, &dcVp); */
  /* dtp2s (0.0, -eps, raep, decp, &rcVm, &dcVm); */

  /* Loop on bases to compute the physical 3D baseline as T2-T1 [m]
   * Note: The telescope locations STAXYZ are given in the [West,South,Up]
   * frame (ESO convention), whereas the UVW, calculated later on, are in
   * the [East,North,Up] frame. Hence the sign changes below on X and Y. */
  double baseline[6][3];
  for ( int base = 0; base < 6; base ++) {
	  int tel1=0; while ( cpl_table_get (oi_array, "STA_INDEX", tel1, NULL) != mat_table_get_value (oi_vis2, "STA_INDEX", base, 0) ) tel1++;
	  int tel2=0; while ( cpl_table_get (oi_array, "STA_INDEX", tel2, NULL) != mat_table_get_value (oi_vis2, "STA_INDEX", base, 1) ) tel2++;
	  baseline[base][0] = -(mat_table_get_value (oi_array, "STAXYZ", tel2, 0) - mat_table_get_value (oi_array, "STAXYZ", tel1, 0));
	  baseline[base][1] = -(mat_table_get_value (oi_array, "STAXYZ", tel2, 1) - mat_table_get_value (oi_array, "STAXYZ", tel1, 1));
	  baseline[base][2] = +(mat_table_get_value (oi_array, "STAXYZ", tel2, 2) - mat_table_get_value (oi_array, "STAXYZ", tel1, 2));
  }
  
  
  /* Loop on rows. */
  cpl_size n_row = cpl_table_get_nrow (oi_vis2);
  for (cpl_size row = 0 ; row < n_row ; row++) {
    /* Transformation from ICRS to Observed (neglect refraction, all at zero)
     * Update precession/nudation every t_skip, otherwise rotate earth only
     * If the time is strickly the same as previous row, we skip this computation */
    if (mjd[row] != mjd1 ) {
      if ( fabs (mjd[row]-mjd0) > t_skip ) {
	eraApco13 (2400000.5, mjd[row], dut1, lon, lat, elev, pmx/3600.0*CPL_MATH_RAD_DEG,
		   pmy/3600.0*CPL_MATH_RAD_DEG, pressure, temperature, humidity, wavelength, &astrom, &eo);
	mjd0 = mjd[row]; 
      } else {
	eraAper13 (2400000.5, mjd[row] + dut1/(24.0*3600.0), &astrom);
      }


            /* Apply proper motion (ICRS->BCRS) */
            eraPmpx(raep, decp, pmra, pmdec, parallax, sysvel, astrom.pmt, astrom.eb, eWb);
            eraC2s(eWb, &rb, &db);
            
            /* Create (eU,eV,eW) in the BCRS */
            eraS2c(0.0, CPL_MATH_PI/2.0, eZb);  // eZc is the unit vector to the ICRS pole
            eraPxp(eZb, eWb, eUb);  // eUc is the cross product of eZc and eWc
            eraPn(eUb, &norm, eUb);  // eUc is normalized to a unit vector
            eraPxp(eWb, eUb, eVb);  // eWc is the cross product of eWc and eUc
            
            /* Create a 10 arcsec cardinal asterism around eWb in the BCRS */
            rotate_vector(eWb, -eps, eVb, eWb_up);
            rotate_vector(eWb, +eps, eVb, eWb_um);
            rotate_vector(eWb, +eps, eUb, eWb_vp);
            rotate_vector(eWb, -eps, eUb, eWb_vm);
            eraC2s(eWb_up, &rb_up, &db_up);
            eraC2s(eWb_um, &rb_um, &db_um);
            eraC2s(eWb_vp, &rb_vp, &db_vp);
            eraC2s(eWb_vm, &rb_vm, &db_vm);
            
            /* Transform the pointing direction and the cardinal asterism from BCRS to observed */
            eraAtboq (rb   , db   , &astrom, eWo   );
            eraAtboq (rb_up, db_up, &astrom, eWo_up);
            eraAtboq (rb_um, db_um, &astrom, eWo_um);
            eraAtboq (rb_vp, db_vp, &astrom, eWo_vp);
            eraAtboq (rb_vm, db_vm, &astrom, eWo_vm);
            
            /* Compute the observed (eUo,eVo,eWo) reference frame */
            eraPxp(eWo_up, eWo_um, eUo);
            eraPxp(eWo, eUo, eUo);
            eraSxp(1./(2.0*eps), eUo, eUo);
            eraPxp(eWo_vp, eWo_vm, eVo);
            eraPxp(eWo, eVo, eVo);
            eraSxp(1./(2.0*eps), eVo, eVo);

      /* /\* Transform from celestial to intermediate, compute eU *\/ */
      /* eraAtcoq (rcUp, dcUp, pmra, pmdec, parallax, sysvel, &astrom, enuobUp); */
      /* eraAtcoq (rcUm, dcUm, pmra, pmdec, parallax, sysvel, &astrom, enuobUm); */
      /* difference (enuobUp, enuobUm, eu); */
      /* normalize (eu); */
          
      /* /\* Transform from celestial to intermediate, compute eV *\/ */
      /* eraAtcoq (rcVp, dcVp, pmra, pmdec, parallax, sysvel, &astrom, enuobVp); */
      /* eraAtcoq (rcVm, dcVm, pmra, pmdec, parallax, sysvel, &astrom, enuobVm); */
      /* difference (enuobVp, enuobVm, ev); */
      /* normalize (ev); */
    }
      
    /* Project physical baseline into u,v */
    int base = row % 6;
    /* oivis2->list_vis2[row]->ucoord = eu[0] * baseline[base][0] + eu[1] * baseline[base][1] + eu[2] * baseline[base][2]; */
    /* oivis2->list_vis2[row]->vcoord = ev[0] * baseline[base][0] + ev[1] * baseline[base][1] + ev[2] * baseline[base][2]; */
    double n_air = 1.0002028;   
    oivis2->list_vis2[row]->ucoord = (eUo[0] * baseline[base][0] + eUo[1] * baseline[base][1] + eUo[2] * baseline[base][2])/n_air;
    oivis2->list_vis2[row]->vcoord = (eVo[0] * baseline[base][0] + eVo[1] * baseline[base][1] + eVo[2] * baseline[base][2])/n_air;
      
    mjd1 = mjd[row];
  } /* End loop on rows */
  

  cpl_table_delete(oi_array);
  cpl_table_delete(oi_vis2);
  

  return CPL_ERROR_NONE;
}

cpl_error_code mat_eop_uv_oit3(mat_oit3 * oit3,
			       mat_array * oiarray,
			       cpl_propertylist * hdr_data,
			       cpl_table * eop_table)
{

  cpl_ensure_code (oit3,   CPL_ERROR_NULL_INPUT);
  cpl_ensure_code (oiarray, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code (hdr_data, CPL_ERROR_NULL_INPUT);
  

  cpl_table * oi_t3 = mat_oit3_to_table(oit3,NULL);
  cpl_table * oi_array = mat_array_to_table(oiarray,NULL);



  double t_skip = 2./24/3600, mjd0 = -1.0, mjd1 = -1.0;
  cpl_msg_info (cpl_func, "Compute pointing with full ERFA every %.2f s",t_skip*24*3600.0);

  /* Read UTC as MJD */
  double *mjd = cpl_table_get_data_double (oi_t3, "MJD");
  double mean_mjd = cpl_table_get_column_mean (oi_t3, "MJD");

  /* Checked by J. Woillez */
  double elev=0.;
  double lon=0.;
  double lat=0.;
  if (cpl_propertylist_has(hdr_data,"ESO ISS GEOELEV")) {
    elev = cpl_propertylist_get_double(hdr_data,"ESO ISS GEOELEV"); // Height in [m]
  }
  if (cpl_propertylist_has(hdr_data,"ESO ISS GEOLON")) {
    lon = cpl_propertylist_get_double(hdr_data,"ESO ISS GEOLON") * CPL_MATH_RAD_DEG; // Lat in [rad]
  }
  if (cpl_propertylist_has(hdr_data,"ESO ISS GEOLAT")) {
    lat = cpl_propertylist_get_double(hdr_data,"ESO ISS GEOLAT") * CPL_MATH_RAD_DEG; // Lon in [rad], East positive
  }
  
  /* Compute Earth Orientation Parameters for the mean MJD */
  double dut1 = 0, pmx = 0, pmy = 0;
  if (eop_table != NULL) {
      mat_eop_interpolate (1, &mean_mjd, &pmx, &pmy, &dut1, eop_table);
  } else {
      cpl_msg_warning (cpl_func, "No EOP_PARAM. Use EOP and DUT=0.0s");
  }
    
  double raep=0.;
  double decp=0.;
  double pmra=0.;
  double pmdec=0.;
  double parallax = 0.;
  double sysvel = 0.0;
  /* if (strstr(cpl_propertylist_get_string(hdr_data,"INSTRUME"),"GRAVITY") != NULL) { */
  /*   raep = mat_get_raep (hdr_data,0.5); // [rad] */
  /*   decp = mat_get_decep (hdr_data,0.5); // [rad] */
  /*   if  (cpl_propertylist_has(hdr_data,"ESO FT ROBJ PMA")) { */
  /*     const char * str=cpl_propertylist_get_string(hdr_data,"ESO FT ROBJ PMA"); */
  /*     sscanf(str,"%lf",&pmra); */
  /*     pmra *= CPL_MATH_RAD_DEG / 3600.0 / cos(decp); // dRA/dt, not cos(Dec)xdRA/dt [rad/year] */
  /*   } */
  /*   if  (cpl_propertylist_has(hdr_data,"ESO FT ROBJ PMD")) { */
  /*     const char * str=cpl_propertylist_get_string(hdr_data,"ESO FT ROBJ PMD"); */
  /*     sscanf(str,"%lf",&pmdec); */
  /*     pmdec *= CPL_MATH_RAD_DEG / 3600.0; // dDec/dt [rad/year] */
  /*   } */
  /*   if  (cpl_propertylist_has(hdr_data,"ESO FT ROBJ PARALLAX")) { */
  /*     parallax=cpl_propertylist_get_double(hdr_data,"ESO FT ROBJ PARALLAX"); // [as] */
  /*   } */
  /* } else { */
    if  (cpl_propertylist_has(hdr_data,"RA")) {
      raep = cpl_propertylist_get_double(hdr_data,"RA");
    /* if  (cpl_propertylist_has(hdr_data,"ESO COU GUID RA")) { */
    /*   raep = cpl_propertylist_get_double(hdr_data,"ESO COU GUID RA"); */
      raep *= 2.* CPL_MATH_PI / 360.; 
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO COU GUID RA not found");
    }
    if  (cpl_propertylist_has(hdr_data,"DEC")) {
      decp = cpl_propertylist_get_double(hdr_data,"DEC");
    /* if  (cpl_propertylist_has(hdr_data,"ESO COU GUID DEC")) { */
    /*   decp = cpl_propertylist_get_double(hdr_data,"ESO COU GUID DEC"); */
      decp *= 2.* CPL_MATH_PI / 360.;
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO COU GUID DEC not found");
    }
    if  (cpl_propertylist_has(hdr_data,"ESO TEL ARG PMA")) {
      pmra= cpl_propertylist_get_double(hdr_data,"ESO TEL ARG PMA");
      pmra *= CPL_MATH_RAD_DEG / 3600.0 / cos(decp); 
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO TEL ARG PMA not found");
    }
    if  (cpl_propertylist_has(hdr_data,"ESO TEL ARG PMD")) {
      pmdec= cpl_propertylist_get_double(hdr_data,"ESO TEL ARG PMD");
      pmdec *= CPL_MATH_RAD_DEG / 3600.0; 
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO TEL ARG PMD not found");
    }
    if  (cpl_propertylist_has(hdr_data,"ESO TEL ARG PARALLAX")) {
      parallax = cpl_propertylist_get_double(hdr_data,"ESO TEL ARG PARALLAX");
    } else {
      cpl_msg_warning(cpl_func,"Keyword ESO TEL ARG PARALLAX not found");
    }

  /* } */

  /* Allocate memory for the tmp computations */
  eraASTROM astrom;
  /* double eo, rcUp, dcUp, rcUm, dcUm, rcVp, dcVp, rcVm, dcVm; */
  /* double enuobUp[3], enuobUm[3], enuobVp[3], enuobVm[3], eu[3], ev[3]; */
  
  
  /* Step for finite difference, 10 arcsec chosen for optimal accuracy */
  double eps = 10.0 / 3600.0 * CPL_MATH_RAD_DEG; // [rad]

  double eo;
  double rb, db;
  double eUb[3], eVb[3], eWb[3], eZb[3];
  double eWo_up[3], eWo_um[3], eWo_vp[3], eWo_vm[3];
  double eWb_up[3], eWb_um[3], eWb_vp[3], eWb_vm[3];
  double rb_up, db_up, rb_um, db_um, rb_vp, db_vp, rb_vm, db_vm;
  double eUo[3], eVo[3], eWo[3];
  double norm;
  double pressure = 0.0; // Pressure at zero to disable atmospheric refraction
  double temperature = 0.0;
  double humidity = 0.0;
  double wavelength = 0.0;

  /* Prepare centered finite differences
   * eU corresponds to +RA
   * eV corresponds to +DEC */
  /* dtp2s (+eps, 0.0, raep, decp, &rcUp, &dcUp); */
  /* dtp2s (-eps, 0.0, raep, decp, &rcUm, &dcUm); */
  /* dtp2s (0.0, +eps, raep, decp, &rcVp, &dcVp); */
  /* dtp2s (0.0, -eps, raep, decp, &rcVm, &dcVm); */

  /* Loop on bases to compute the physical 3D baseline as T2-T1 [m]
   * Note: The telescope locations STAXYZ are given in the [West,South,Up]
   * frame (ESO convention), whereas the UVW, calculated later on, are in
   * the [East,North,Up] frame. Hence the sign changes below on X and Y. */
  double baseline1[4][3];
  double baseline2[4][3];
  for ( int base = 0; base < 4; base ++) {
  	  int tel1=0; while ( cpl_table_get (oi_array, "STA_INDEX", tel1, NULL) != mat_table_get_value (oi_t3, "STA_INDEX", base, 0) ) tel1++;
  	  int tel2=0; while ( cpl_table_get (oi_array, "STA_INDEX", tel2, NULL) != mat_table_get_value (oi_t3, "STA_INDEX", base, 1) ) tel2++;
  	  int tel3=0; while ( cpl_table_get (oi_array, "STA_INDEX", tel3, NULL) != mat_table_get_value (oi_t3, "STA_INDEX", base, 2) ) tel3++;
  	  baseline1[base][0] = -(mat_table_get_value (oi_array, "STAXYZ", tel2, 0) - mat_table_get_value (oi_array, "STAXYZ", tel1, 0));
  	  baseline1[base][1] = -(mat_table_get_value (oi_array, "STAXYZ", tel2, 1) - mat_table_get_value (oi_array, "STAXYZ", tel1, 1));
  	  baseline1[base][2] = +(mat_table_get_value (oi_array, "STAXYZ", tel2, 2) - mat_table_get_value (oi_array, "STAXYZ", tel1, 2));
  	  baseline2[base][0] = -(mat_table_get_value (oi_array, "STAXYZ", tel3, 0) - mat_table_get_value (oi_array, "STAXYZ", tel2, 0));
  	  baseline2[base][1] = -(mat_table_get_value (oi_array, "STAXYZ", tel3, 1) - mat_table_get_value (oi_array, "STAXYZ", tel2, 1));
  	  baseline2[base][2] = +(mat_table_get_value (oi_array, "STAXYZ", tel3, 2) - mat_table_get_value (oi_array, "STAXYZ", tel2, 2));
  }
  
  
  /* Loop on rows. */
  cpl_size n_row = cpl_table_get_nrow (oi_t3);
  for (cpl_size row = 0 ; row < n_row ; row++) {
    /* Transformation from ICRS to Observed (neglect refraction, all at zero)
     * Update precession/nudation every t_skip, otherwise rotate earth only
     * If the time is strickly the same as previous row, we skip this computation */
      
    if (mjd[row] != mjd1 ) {
          
      if ( fabs (mjd[row]-mjd0) > t_skip ) {
	eraApco13 (2400000.5, mjd[row], dut1, lon, lat, elev, pmx/3600.0*CPL_MATH_RAD_DEG,
		   pmy/3600.0*CPL_MATH_RAD_DEG, pressure, temperature, humidity, wavelength, &astrom, &eo);
	mjd0 = mjd[row]; 
      } else {
	eraAper13 (2400000.5, mjd[row] + dut1/(24.0*3600.0), &astrom);
      }


            /* Apply proper motion (ICRS->BCRS) */
            eraPmpx(raep, decp, pmra, pmdec, parallax, sysvel, astrom.pmt, astrom.eb, eWb);
            eraC2s(eWb, &rb, &db);
            
            /* Create (eU,eV,eW) in the BCRS */
            eraS2c(0.0, CPL_MATH_PI/2.0, eZb);  // eZc is the unit vector to the ICRS pole
            eraPxp(eZb, eWb, eUb);  // eUc is the cross product of eZc and eWc
            eraPn(eUb, &norm, eUb);  // eUc is normalized to a unit vector
            eraPxp(eWb, eUb, eVb);  // eWc is the cross product of eWc and eUc
            
            /* Create a 10 arcsec cardinal asterism around eWb in the BCRS */
            rotate_vector(eWb, -eps, eVb, eWb_up);
            rotate_vector(eWb, +eps, eVb, eWb_um);
            rotate_vector(eWb, +eps, eUb, eWb_vp);
            rotate_vector(eWb, -eps, eUb, eWb_vm);
            eraC2s(eWb_up, &rb_up, &db_up);
            eraC2s(eWb_um, &rb_um, &db_um);
            eraC2s(eWb_vp, &rb_vp, &db_vp);
            eraC2s(eWb_vm, &rb_vm, &db_vm);
            
            /* Transform the pointing direction and the cardinal asterism from BCRS to observed */
            eraAtboq (rb   , db   , &astrom, eWo   );
            eraAtboq (rb_up, db_up, &astrom, eWo_up);
            eraAtboq (rb_um, db_um, &astrom, eWo_um);
            eraAtboq (rb_vp, db_vp, &astrom, eWo_vp);
            eraAtboq (rb_vm, db_vm, &astrom, eWo_vm);
            
            /* Compute the observed (eUo,eVo,eWo) reference frame */
            eraPxp(eWo_up, eWo_um, eUo);
            eraPxp(eWo, eUo, eUo);
            eraSxp(1./(2.0*eps), eUo, eUo);
            eraPxp(eWo_vp, eWo_vm, eVo);
            eraPxp(eWo, eVo, eVo);
            eraSxp(1./(2.0*eps), eVo, eVo);

      /* /\* Transform from celestial to intermediate, compute eU *\/ */
      /* eraAtcoq (rcUp, dcUp, pmra, pmdec, parallax, sysvel, &astrom, enuobUp); */
      /* eraAtcoq (rcUm, dcUm, pmra, pmdec, parallax, sysvel, &astrom, enuobUm); */
      /* difference (enuobUp, enuobUm, eu); */
      /* normalize (eu); */
          
      /* /\* Transform from celestial to intermediate, compute eV *\/ */
      /* eraAtcoq (rcVp, dcVp, pmra, pmdec, parallax, sysvel, &astrom, enuobVp); */
      /* eraAtcoq (rcVm, dcVm, pmra, pmdec, parallax, sysvel, &astrom, enuobVm); */
      /* difference (enuobVp, enuobVm, ev); */
      /* normalize (ev); */
    }
      
    /* Project physical baseline into u,v */
    int base = row % 4;
    /* oit3->list_t3[row]->u1coord = eu[0] * baseline1[base][0] + eu[1] * baseline1[base][1] + eu[2] * baseline1[base][2]; */
    /* oit3->list_t3[row]->v1coord = ev[0] * baseline1[base][0] + ev[1] * baseline1[base][1] + ev[2] * baseline1[base][2]; */
    /* oit3->list_t3[row]->u2coord = eu[0] * baseline2[base][0] + eu[1] * baseline2[base][1] + eu[2] * baseline2[base][2]; */
    /* oit3->list_t3[row]->v2coord = ev[0] * baseline2[base][0] + ev[1] * baseline2[base][1] + ev[2] * baseline2[base][2]; */
    double n_air = 1.0002028;
    oit3->list_t3[row]->u1coord = (eUo[0] * baseline1[base][0] + eUo[1] * baseline1[base][1] + eUo[2] * baseline1[base][2])/n_air;
    oit3->list_t3[row]->v1coord = (eVo[0] * baseline1[base][0] + eVo[1] * baseline1[base][1] + eVo[2] * baseline1[base][2])/n_air;
    oit3->list_t3[row]->u2coord = (eUo[0] * baseline2[base][0] + eUo[1] * baseline2[base][1] + eUo[2] * baseline2[base][2])/n_air;
    oit3->list_t3[row]->v2coord = (eVo[0] * baseline2[base][0] + eVo[1] * baseline2[base][1] + eVo[2] * baseline2[base][2])/n_air;

    mjd1 = mjd[row];
  } /* End loop on rows */
  

  cpl_table_delete(oi_array);
  cpl_table_delete(oi_t3);
  

  return CPL_ERROR_NONE;
}




/*-----------------------------------------------------------------------------
                    Function to read the target coordinates
 -----------------------------------------------------------------------------*/
double mat_get_decep (const cpl_propertylist * plist, double coef)
{
  cpl_ensure (plist, CPL_ERROR_NULL_INPUT, 0.0);
  cpl_ensure (coef >= 0 && coef <= 1, CPL_ERROR_ILLEGAL_INPUT, 0.0);

  /* Get RA and DEC in [rad] */
  // double raz  = gravi_pfits_get_robj_raep (plist);
  double decz = mat_get_robj_decep (plist);

  /* Get and convert the offsets from [mas] to [rad] */
  double xi  = mat_get_sobj_x (plist) / 3600000. * CPL_MATH_RAD_DEG * coef;
  double eta = mat_get_sobj_y (plist) / 3600000. * CPL_MATH_RAD_DEG * coef;

  double sdecz = sin(decz);
  double cdecz = cos(decz);
  double denom = cdecz - eta * sdecz;

  /* Compute new coordinates in [rad] */
  double dec = atan2 (sdecz+eta*cdecz, sqrt(xi*xi + denom*denom));

  /* Return in [rad] */
  return dec;
}

double mat_get_raep(const cpl_propertylist * plist, double coef)
{
  cpl_ensure (plist, CPL_ERROR_NULL_INPUT, 0.0);
  cpl_ensure (coef >= 0 && coef <= 1, CPL_ERROR_ILLEGAL_INPUT, 0.0);

  /* Get RA and DEC in [rad] */
  double raz  = mat_get_robj_raep (plist);
  double decz = mat_get_robj_decep (plist);

  /* Get and convert the offsets from [mas] to [rad] */
  double xi  = mat_get_sobj_x (plist) / 3600000. * CPL_MATH_RAD_DEG * coef;
  double eta = mat_get_sobj_y (plist) / 3600000. * CPL_MATH_RAD_DEG * coef;

  double sdecz = sin(decz);
  double cdecz = cos(decz);
  double denom = cdecz - eta * sdecz;

  /* Compute new coordinates in [rad] */
  double ra  = atan2 (xi,denom) + raz;

  /* Make ra within 0-2pi */
  ra = fmod (ra, CPL_MATH_2PI);
  if ( ra < 0.0 ) ra += CPL_MATH_2PI;

  /* Return in [rad] */
  return ra;
}


double mat_get_robj_raep(const cpl_propertylist * plist)
{
  cpl_ensure (plist, CPL_ERROR_NULL_INPUT, 99.);
  cpl_errorstate prestate = cpl_errorstate_get();
  char tmp[15];
  
  /* Read the parameter as string */
  const char * str_value;
  str_value = mat_get_string_default (plist, "ESO FT ROBJ ALPHA", "000000.00");
  
  double value = 99.;
  
  /* Read of the form '043555.24' */
  cpl_msg_debug( cpl_func, "Found '%s'", str_value );
  
  /* Read 2 first chars */
  strncpy(tmp, str_value, 2);
  tmp[2] = '\0';
  cpl_msg_debug( cpl_func, "Found tmp '%s' -> %f", tmp, atof(tmp) );
  value = atof(tmp);
  
  /* Read 2 more chars */
  strncpy(tmp, str_value+2, 4);
  tmp[2] = '\0';
  cpl_msg_debug( cpl_func, "Found tmp '%s' -> %f", tmp, atof(tmp) );
  value += atof(tmp) / 60.0;
  
  /* Read remaining data */
  strcpy(tmp, str_value+4);
  cpl_msg_debug( cpl_func, "Found tmp '%s' -> %f", tmp, atof(tmp) );
  value += atof(tmp) / 3600.0;
  
  /* Convert hours to deg */
  value *= 360. / 24;
  
  /* Verbose */
  cpl_msg_debug( cpl_func, "Convert RA='%s' into RA=%fdeg", str_value, value);
  
  if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND){
    cpl_errorstate_set (prestate);
    cpl_msg_warning(cpl_func, "rarp doesn't exist in this file.");
    value=0;
  }
  
  /* Check for a change in the CPL error state */
  /* - if it did change then propagate the error and return */
  cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);
  
  /* Convert in [rad] */
  return value * CPL_MATH_RAD_DEG;
}

double mat_get_robj_decep (const cpl_propertylist * plist)
{
  cpl_ensure (plist, CPL_ERROR_NULL_INPUT, 99.);
  
  cpl_errorstate prestate = cpl_errorstate_get();
  double value = 99.;
  char tmp[15];
  double sign;
  
  const char * str_value;
  str_value = mat_get_string_default (plist, "ESO FT ROBJ DELTA", "+000000.00");
  
  /* Read of the form '163033.49' -- issue with +/- */
  cpl_msg_debug( cpl_func, "Found '%s'", str_value );
  
  /* Check the first digit is a sign + or -
     and discard it */
  if ( str_value[0] == '-' ) {
    sign = -1.0;
    str_value = str_value + 1;
  }
  else if ( str_value[0] == '+' ) {
    sign = +1.0;
    str_value = str_value + 1;
  }
  else {
    sign = +1.0;
  }
  
  /* Read 2 first chars */
  strncpy(tmp, str_value, 2);
  tmp[2] = '\0';
  cpl_msg_debug( cpl_func, "Found tmp '%s' -> %f", tmp, atof(tmp) );
  value = atof(tmp);
  
  /* Read 2 more chars */
  strncpy(tmp, str_value+2, 4);
  tmp[2] = '\0';
  cpl_msg_debug( cpl_func, "Found tmp '%s' -> %f", tmp, atof(tmp) );
  value += atof(tmp) / 60.0;
  
  /* Read remaining data */
  strcpy(tmp, str_value+4);
  cpl_msg_debug( cpl_func, "Found tmp '%s' -> %f", tmp, atof(tmp) );
  value += atof(tmp) / 3600.0;
  
  /* Apply sign */
  value *= sign;
  
  /* Verbose */
  cpl_msg_debug( cpl_func, "Convert DEC='%s' into DEC=%fdeg", str_value, value);
  
  if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND){
    cpl_errorstate_set (prestate);
    cpl_msg_warning(cpl_func, "decep doesn't exist in this file.");
    
    value=0;
  }
  
  /* Check for a change in the CPL error state */
  /* - if it did change then propagate the error and return */
  cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);
  
  /* Convert in [rad] */
  return value * CPL_MATH_RAD_DEG;
}


double mat_get_sobj_x (const cpl_propertylist * plist)
{
  double value = 0.;
  if (cpl_propertylist_has(plist,"ESO INS SOBJ X")) {
    cpl_propertylist_get_double(plist,"ESO INS SOBJ X");
  }
  return value;
}

double mat_get_sobj_y (const cpl_propertylist * plist)
{
  double value = 0.;
  if (cpl_propertylist_has(plist,"ESO INS SOBJ Y")) {
    cpl_propertylist_get_double(plist,"ESO INS SOBJ Y");
  }
  return value;
}

const char * mat_get_string_default (const cpl_propertylist * plist,
				     const char *name,
				     const char *def)
{
  cpl_ensure (plist, CPL_ERROR_NULL_INPUT, def);
  cpl_ensure (name,  CPL_ERROR_NULL_INPUT, def);
  
  const char *output;
  
  if (cpl_propertylist_has(plist, name)) {
    /* Try to read this keyword */
    output = cpl_propertylist_get_string(plist, name);
  } else {
    /* Get the default and add warning only if not CALIB */
    output = def;
  }
  
  return output;
}
