/* $Id: mat_frame.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_frame.c $
 */

#include <stdlib.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <string.h>

#include "mat_error.h"
#include "mat_frame.h"
#include "mat_imagingdata.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

/**
   @brief Creates a new mat_frame using the imaging detector setup
   @param imgdet  The imaging detectur setup (sub-window setup)
   @param type    The CPL data type which is used for the enclosed CPL images.
   @returns a newly created frame or NULL

   This function creates a mat_frame data structure. For each sub-window given
   in the imgdet parameters a CPL image is created.
 */
mat_frame *mat_frame_new(const mat_imagingdetector *imgdet, cpl_type type)
{
  mat_frame *frame = NULL;
  int        r;

  frame = (mat_frame *)cpl_calloc(1, sizeof(mat_frame));
  if (frame == NULL) return NULL;
  frame->nbsubwin = imgdet->nbregion;
  frame->list_subwin = (mat_imgreg **)cpl_calloc(imgdet->nbregion, sizeof(mat_imgreg*));
  if (frame->list_subwin == NULL)
    {
      mat_frame_delete(frame);
      return NULL;
    }
  for (r = 0; r < imgdet->nbregion; r++)
    {
      frame->list_subwin[r] = (mat_imgreg *)cpl_calloc(1, sizeof(mat_imgreg));
      if (frame->list_subwin[r] == NULL)
	{
	  mat_frame_delete(frame);
	  return NULL;
	}
      frame->list_subwin[r]->nbimgreg = 1;
      frame->list_subwin[r]->numregion = imgdet->list_region[r]->numregion;
      frame->list_subwin[r]->numdetector = imgdet->list_region[r]->numdetector;
      frame->list_subwin[r]->imgreg = (cpl_image **) cpl_calloc(1, sizeof(cpl_image *));
      if (frame->list_subwin[r]->imgreg == NULL)
	{
	  mat_frame_delete(frame);
	  return NULL;
	}
      frame->list_subwin[r]->imgreg[0] = cpl_image_new(imgdet->list_region[r]->naxis[0], imgdet->list_region[r]->naxis[1], type);
      if (frame->list_subwin[r]->imgreg[0] == NULL)
	{
	  mat_frame_delete(frame);
	  return NULL;
	}
    }
  return frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Duplicate a mat_frame structure
  @param frame          mat_frame structure to duplicate
  @param type    The cpl_type used for the new CPL images.
  @return framedup
 */
/*----------------------------------------------------------------------------*/
mat_frame *mat_frame_duplicate(const mat_frame *frame, cpl_type type)
{
  mat_frame *frame_dup = NULL;
  int j;
  
  frame_dup = cpl_calloc(1, sizeof(mat_frame));
  frame_dup->time= frame->time;
  frame_dup->exptime= frame->exptime;
  for (j = 0; j < MAT_MAXTEL; j++)
    {
      frame_dup->opd[j]      = frame->opd[j];
      frame_dup->localopd[j] = frame->localopd[j];
    }
  frame_dup->stepphase  = frame->stepphase;
  frame_dup->tartype[0] = frame->tartype[0];
  frame_dup->tartype[1] = frame->tartype[1];
  frame_dup->nbsubwin   = frame->nbsubwin;
  // Copy sub windows
  frame_dup->list_subwin = cpl_calloc(frame_dup->nbsubwin, sizeof(mat_imgreg*));
  for (j = 0; j < frame_dup->nbsubwin; j++)
    {
      frame_dup->list_subwin[j]= mat_imgreg_duplicate(frame->list_subwin[j], type); 
    }
  return frame_dup;
}

/**
   @brief Deletes a mat_frame data structude and the enclosing mat_imgreg data structures.
   @param frame   A mat_frame data structure.
   @returns a cpl_error_code

   This function deletes a mat_frame data structure. This includes the enclosed mat_imgreg
   data structures and their enclosed cpl_image data structures.
 */
cpl_error_code mat_frame_delete(mat_frame *frame)
{
  cpl_ensure_code((frame != NULL), CPL_ERROR_NULL_INPUT);
  if (frame->list_subwin != NULL)
    {
      int r;
      for (r = 0; r < frame->nbsubwin; r++)
	{
	  if (frame->list_subwin[r] != NULL)
	    {
	      mat_imgreg_delete(frame->list_subwin[r]);
	      frame->list_subwin[r] = NULL;
	    }
	}
      cpl_free(frame->list_subwin);
      frame->nbsubwin = 0;
      frame->list_subwin = NULL;
    }
  cpl_free(frame);
  return CPL_ERROR_NONE;
}

/**
   @brief Sets all CPL images of a mat_frame to a given double value.
   @param frame   A mat_frame data structure.
   @param value   A double precision value.
   @returns A cpl_error_code

   This function sets all pixels in all images of the given frame to the same value.
   The current implementation supports only CPL images with a CPL_DOUBLE_TYPE.
 */
cpl_error_code mat_frame_fill(mat_frame *frame,
			      double value)
{
  mat_imgreg *region = NULL;
  cpl_image  *image  = NULL;
  double     *pixels = NULL;
  int r, nx, ny, n, i;

  cpl_ensure_code((frame != NULL), CPL_ERROR_NULL_INPUT);
  for (r = 0; r < frame->nbsubwin; r++)
    {
      region = frame->list_subwin[r];
      image = region->imgreg[0]; // exactly one cpl_image for each region!
      cpl_ensure_code((cpl_image_get_type(image) == CPL_TYPE_DOUBLE), CPL_ERROR_INVALID_TYPE);
      nx = cpl_image_get_size_x(image);
      ny = cpl_image_get_size_y(image);
      n = nx*ny;
      pixels = cpl_image_get_data_double(image);
      if (pixels == NULL)
	{
	  return CPL_ERROR_TYPE_MISMATCH;
	}
      for (i = 0; i < n; i++)
	{
	  *pixels++ = value;
	}
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Multiplies all CPL images of a mat_frame with a given double value.
   @param frame   A mat_frame data structure.
   @param value   A double precision value.
   @returns A cpl_error_code

   This function multiplies all pixels in all images of the given frame with the same value.
   The current implementation supports only CPL images with a CPL_DOUBLE_TYPE.
 */
cpl_error_code mat_frame_multiply(mat_frame *frame,
				  double value)
{
  mat_imgreg *region = NULL;
  cpl_image  *image  = NULL;
  int         r;
  int         nx, ny, n, i;

  cpl_ensure_code((frame != NULL), CPL_ERROR_NULL_INPUT);
  for (r = 0; r < frame->nbsubwin; r++)
    {
      region = frame->list_subwin[r];
      image = region->imgreg[0]; // exactly one cpl_image for each region!
      cpl_ensure_code(((cpl_image_get_type(image) == CPL_TYPE_DOUBLE)
		       || (cpl_image_get_type(image) == CPL_TYPE_FLOAT)),
		      CPL_ERROR_INVALID_TYPE);
      if (cpl_image_get_type(image) == CPL_TYPE_DOUBLE)
	{
	  double     *pixels = NULL;
	  nx = cpl_image_get_size_x(image);
	  ny = cpl_image_get_size_y(image);
	  n = nx*ny;
	  pixels = cpl_image_get_data_double(image);
	  if (pixels == NULL)
	    {
	      return CPL_ERROR_TYPE_MISMATCH;
	    }
	  for (i = 0; i < n; i++)
	    {
	      *pixels++ *= value;
	    }
	}
      else if (cpl_image_get_type(image) == CPL_TYPE_FLOAT)
	{
	  float     *pixels = NULL;
	  nx = cpl_image_get_size_x(image);
	  ny = cpl_image_get_size_y(image);
	  n = nx*ny;
	  pixels = cpl_image_get_data_float(image);
	  if (pixels == NULL)
	    {
	      return CPL_ERROR_TYPE_MISMATCH;
	    }
	  for (i = 0; i < n; i++)
	    {
	      *pixels++ *= (float)value;
	    }
	}
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Gets a pixel value of a mat_frame.
   @param frame  A mat_frame data strcuture.
   @param r      The region (sub-window) number (0-based).
   @param x      The x-coordinate inside the region (1-based).
   @param y      The y-coordinate inside the region (1-based).
   @returns The pixel value or 0.0

   This function returns the pixel value inside a mat_frame data structure.
   It is a simple convenience function.
 */
double mat_frame_get(const mat_frame *frame,
		     int r,
		     int x,
		     int y)
{
  mat_imgreg *region = NULL;
  cpl_image  *image  = NULL;
  int pis_rejected;

  if (frame == NULL) return 0.0;
  if (frame->list_subwin == NULL) return 0.0;
  if ((r < 0) || (r >= frame->nbsubwin)) return 0.0;
  region = frame->list_subwin[r];
  image = region->imgreg[0]; // exactly one cpl_image for each region!
  return cpl_image_get(image, x, y, &pis_rejected);
}

/**
   @brief Sets a pixel value of a mat_frame to a specific value
   @param frame  A mat_frame data strcuture.
   @param r      The region (sub-window) number (0-based).
   @param x      The x-coordinate inside the region (1-based).
   @param y      The y-coordinate inside the region (1-based).
   @param value  The given pixel value.
   @returns A cpl_error_code.

   This function sets the pixel value inside a mat_frame data structure to agiven value.
   It is a simple convenience function.
 */
cpl_error_code mat_frame_set(mat_frame *frame,
			     int r,
			     int x,
			     int y,
			     double value)
{
  mat_imgreg *region = NULL;
  cpl_image  *image  = NULL;

  cpl_ensure_code((frame != NULL), CPL_ERROR_NULL_INPUT);
  cpl_ensure_code((frame->list_subwin != NULL), CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(((r < frame->nbsubwin) && (r >= 0)), CPL_ERROR_ACCESS_OUT_OF_RANGE);
  region = frame->list_subwin[r];
  image = region->imgreg[0]; // exactly one cpl_image for each region!
  cpl_image_set(image, x, y, value);
  return CPL_ERROR_NONE;
}

/* /\** */
/*    @brief Applies cpl_image_filter_mask to all sub-windows of a mat_frame data structure. */
/*    @param dst   The filtered images as a mat_frame data structure. */
/*    @param src   The original images as a mat_frame data structure. */
/*    @param kernel  The filter kernel as a cpl_mask data structure. */
/*    @param filter  The filter mode. */
/*    @param border  The border mode. */
/*    @returns A cpl_error_code. */

/*    This function simply applies the cpl_image_filter_mask function to all sub-windows */
/*    in a given mat_frame data structure. The last three parameters for the */
/*    cpl_image_filter_mask function are given as arguments to this function. */
/*  *\/ */
/* cpl_error_code mat_frame_filter_mask(mat_frame *dst, */
/* 				     const mat_frame *src, */
/* 				     const cpl_mask *kernel, */
/* 				     cpl_filter_mode filter, */
/* 				     cpl_border_mode border) */
/* { */
/*   int r; */

/*   cpl_ensure_code((dst != NULL), CPL_ERROR_NULL_INPUT); */
/*   cpl_ensure_code((dst->list_subwin != NULL), CPL_ERROR_NULL_INPUT); */
/*   cpl_ensure_code((src != NULL), CPL_ERROR_NULL_INPUT); */
/*   cpl_ensure_code((src->list_subwin != NULL), CPL_ERROR_NULL_INPUT); */
/*   for (r = 0; r < dst->nbsubwin; r++) */
/*     { */
/*       cpl_image  *i1 = dst->list_subwin[r]->imgreg[0]; // exactly one cpl_image for each region! */
/*       cpl_image  *i2 = src->list_subwin[r]->imgreg[0]; // exactly one cpl_image for each region! */
/*       cpl_image_filter_mask(i1, i2, kernel, filter, border); */
/*     } */
/*   return CPL_ERROR_NONE; */
/* } */

/**
   @brief Subtracts a frame from another frame (f1 -= f2).
   @brief f1    A mat_frame data structure.
   @brief f2    A mat_frame data structure.
   @returns A cpl_error_code.

   This function subtracts a mat_frame data structure from another one (inplace).
   It simply applies cpl_image_subtract to all sub-windows.
 */
cpl_error_code mat_frame_subtract(mat_frame *f1,
				  mat_frame *f2)
{
  int r;

  cpl_ensure_code((f1 != NULL), CPL_ERROR_NULL_INPUT);
  cpl_ensure_code((f1->list_subwin != NULL), CPL_ERROR_NULL_INPUT);
  cpl_ensure_code((f2 != NULL), CPL_ERROR_NULL_INPUT);
  cpl_ensure_code((f2->list_subwin != NULL), CPL_ERROR_NULL_INPUT);
  for (r = 0; r < f1->nbsubwin; r++)
    {
      cpl_image  *i1 = f1->list_subwin[r]->imgreg[0]; // exactly one cpl_image for each region!
      cpl_image  *i2 = f2->list_subwin[r]->imgreg[0]; // exactly one cpl_image for each region!
      cpl_image_subtract(i1, i2);
    }
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Checks if a MATISSE raw frame is compatible with a IMAGING_DETECTOR sub-window setup.
  @param frame           Contains a MATISSE raw frame.
  @param imgdet          mat_imagingdetector structure as reference
  @return cpl_error_code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_frame_check_layout(const mat_frame *frame, const mat_imagingdetector *imgdet)
{
  int i;

  if (imgdet == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
				   "no mat_imagingdetector (imgdet) argument given");
    }
  if (frame == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
				   "no mat_frame (frame) argument given");
    }
  if (imgdet->nbregion != frame->nbsubwin)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
				   "incompatible data, number of sub-windows not match %d != %d",
				   imgdet->nbregion, frame->nbsubwin);
    }
  for (i = 0; i < imgdet->nbregion; i++)
    {
      mat_region *rreg = imgdet->list_region[i];
      mat_imgreg *freg = frame->list_subwin[i];
      cpl_image  *fimg = freg->imgreg[0];
      if (rreg->numregion != freg->numregion)
	{
	  return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
				       "incompatible data, sub-windows numbers differ rreg[%d] %d != freg[%d] %d",
				       i, rreg->numregion, i, freg->numregion);
	}
      if (rreg->numdetector != freg->numdetector)
	{
	  return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
				       "incompatible data, sub-windows detectors differ rreg[%d] %d != freg[%d] %d",
				       i, rreg->numdetector, i, freg->numdetector);
	}
      if (rreg->naxis[0] != cpl_image_get_size_x(fimg))
	{
	  return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
				       "incompatible data, sub-windows x-dimension differ rreg[%d] %d != freg[%d] %d",
				       i, rreg->naxis[0], i, (int)cpl_image_get_size_x(fimg));
	}
      if (rreg->naxis[1] != cpl_image_get_size_y(fimg))
	{
	  return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
				       "incompatible data, sub-windows y-dimension differ rreg[%d] %d != freg[%d] %d",
				       i, rreg->naxis[1], i, (int)cpl_image_get_size_y(fimg));
	}
    }
  return CPL_ERROR_NONE;
}
