/* $Id: mat_get_diameter.c,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_get_diameter.c $
 */

#ifndef _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 200809L
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "mat_get_diameter.h"


/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/


#define RECIPE_NAME "mat_est_tf"



/*----------------------------------------------------------------------------*/
/**
   @ingroup signal
  @brief This function is called by the mat_est_tf recipe
         for one object, find the calibrator
  @param catalog : catalog filename
  @param ra : degree
  @param dec : degree
  @param diam : diameters and error 
  @param name : diameter name
  @param mag : magnitudes
  @return 0, -1 if error
 */
/*----------------------------------------------------------------------------*/

int mat_get_diameter(char *catalog, double  ra, double  dec, 
		     double* diam, char *name, double *mag) {
    
    cpl_table *table = NULL;
    int i = 0;
    int j = 0;
    char *aux1 = NULL;
    char *aux2 = NULL;
    char *saveaux1 = NULL;
    char *saveaux2 = NULL;
    const char *str = NULL;
    const char *columnName = NULL;
    const char *delimiter=" ";
    int hourRA = 0;
    int minuteRA = 0.0;
    double secondRA = 0.0;
    double searchRA;
    double searchDEC;
    int degreeDEC=0;
    int minuteDEC=0;
    int signDEC=0;
    double secondDEC=0;
    double diffAngle=1.;
    double auxDiffAngle=0.;   
    cpl_errorstate prestate = cpl_errorstate_get();

    if (diam == NULL) {
	return -1;
    }

    table=cpl_table_load(catalog,1,0);
    if (table == NULL) {
	cpl_msg_error(cpl_func,"Problem when loading %s",catalog);
	return -1;
    }
    for(i=0;i<cpl_table_get_nrow(table);i++){

      columnName =  "RAJ2000";
	/* aux1 = strtok((char *)cpl_table_get_string(table,columnName, i), delimiter); */
	aux1 = strtok_r((char *)cpl_table_get_string(table,columnName, i), delimiter, &saveaux1);
	j=0;
	/*find coordinates*/
	while (aux1 != NULL)
	{
	    if(j == 0)
		hourRA = atoi(aux1);
	     else if (j == 1)
		 minuteRA =  atoi(aux1);
	     else
		 secondRA = atof(aux1);
	     /* aux1= strtok (NULL, delimiter); */
	     aux1= strtok_r (NULL, delimiter, &saveaux1);
	     j++;
	 }

	j=0;
	columnName =  "DEJ2000";
	/* aux2 = strtok((char *)cpl_table_get_string(table, columnName, i), delimiter); */
	aux2 = strtok_r ((char *)cpl_table_get_string(table, columnName, i), delimiter, &saveaux2);
	 while (aux2 != NULL)
	 {
	   if(j == 0) {
	     degreeDEC = atoi(aux2);
	     if ( strstr(aux2,"-00") != NULL || degreeDEC <0 )
	       signDEC=-1;
	     else
	       signDEC=1;
	   } else if (j == 1)
		 minuteDEC =  atoi(aux2);
	     else
		 secondDEC = atof(aux2);
	     /* aux2= strtok (NULL, delimiter); */
	     aux2= strtok_r (NULL, delimiter, &saveaux2);
	     j++;
	 }

	 /*convert coordinates */
	 searchRA= HOUR_TO_RAD *  ((double) hourRA + minuteRA/60.0 +
			      secondRA/3600.0);
	 searchDEC=  DEG_TO_RAD * (((double) degreeDEC + signDEC*minuteDEC/60.0 +
					signDEC*secondDEC/3600.0) );
         /* if (degreeDEC >= 0.) { */
	 /*     searchDEC=  DEG_TO_RAD * (((double) degreeDEC + minuteDEC/60.0 + */
	 /* 				secondDEC/3600.0) ); */
	 /* } else { */
	 /*     searchDEC=  DEG_TO_RAD * (((double) degreeDEC - minuteDEC/60.0 - */
	 /* 				secondDEC/3600.0) ); */
	 /* } */
	 
	 /*distance*/
	 auxDiffAngle = acos ( sin(searchDEC) * sin( DEG_TO_RAD *dec) +
	 		    cos(searchDEC) * cos( DEG_TO_RAD *dec) *
	 		    cos (DEG_TO_RAD *ra - searchRA) );	

	 if (auxDiffAngle < diffAngle) {
	     diffAngle = auxDiffAngle;
	     columnName = "UDDK";
	     diam[0] = cpl_table_get_float(table, columnName, i, NULL);
	     columnName = "UDDL";
	     diam[1] = cpl_table_get_float(table, columnName, i, NULL);
	     columnName = "UDDM";
	     diam[2] = cpl_table_get_float(table, columnName, i, NULL);
	     columnName = "UDDN";
	     diam[3] = cpl_table_get_float(table, columnName, i, NULL);

	     columnName = "E_LDD";
	     if (cpl_table_has_column(table,columnName) == 0) {
	       columnName = "e_LDD";
	     }
	     diam[4] = cpl_table_get_float(table, columnName, i, NULL);

	     columnName = "NAME";
	     if (cpl_table_has_column(table,columnName) == 0) {
	       columnName = "Name";
	     }
	     str = cpl_table_get_string(table, columnName, i);

	     sprintf(name,"%s\n",str);
	     
	     columnName = "LMAG";
	     if (cpl_table_has_column(table,columnName) == 0) {
	       columnName = "Lmag";
	     }
	     mag[0] = cpl_table_get_float(table, columnName, i, NULL);
	     if (isnan(mag[0]) != 0)
	       {
		 mag[0]=99.;
	       }
	     columnName = "MMAG";
	     if (cpl_table_has_column(table,columnName) == 0) {
	       columnName = "Mmag";
	     }
	     mag[1] = cpl_table_get_float(table, columnName, i, NULL);
	     if (isnan(mag[1]) != 0)
	       {
		 mag[1]=99.;
	       }
	     columnName = "NMAG";
	     if (cpl_table_has_column(table,columnName) == 0) {
	       columnName = "Nmag";
	     }
	     mag[2] = cpl_table_get_float(table, columnName, i, NULL);   
	     if (isnan(mag[2]) != 0)
	       {
		 mag[2]=99.;
	       }
	 }

	 
    }
    
    if (!cpl_errorstate_is_equal(prestate)) {
	cpl_msg_info(cpl_func,"mat_get_diameter error: ");
	cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
	cpl_errorstate_set(prestate);
	return -1;
    } 
    if (diffAngle > 2.E-4)
      {
	cpl_msg_warning(cpl_func,"The distance between the calibrator found in");
	cpl_msg_warning(cpl_func,"the catalog and the coordinates found in ");
	cpl_msg_warning(cpl_func,"the FITS file seems too important (%f)",diffAngle);
	cpl_table_delete(table);
	return(-1);
      }
    cpl_msg_info(cpl_func, 
		 "The calibrator found in the catalog is %s (distance %f)", 
		 name, diffAngle);
    cpl_table_delete(table);
    return 0;
}
  
