/* $Id: mat_image.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2016/04/11 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_image.c $
 */

#include <stdlib.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/
#include "mat_error.h"
#include "mat_utils.h"
#include "mat_image.h"

/**
   @brief Swaps the four quadrants of an image.
   @param img A CPL image where al fou quadrants are swapped.

   Inside this plugin all images use a coordinate system where 0,0 represents
   the image center and negative coordinates are allowed (between -N/2 and N/2-1).
   Methods like mat_image_get_double() are used to access and manipulates these
   images. The FFT methods are compliant with this coordinate system.

   If such images are read from a file (for example the start image) or written
   into a file (for examples the best reconstruction), the quadrants must be
   swapped in order to map the image center 0,0 (internal) to the
   N/2+1,N/2+1 coordinates of the CPL coordinate system.
*/
void mat_image_swap(cpl_image *img)
{
  int             i, j, nx, ny;
  float          *fp = NULL;
  double         *dp = NULL;
  float complex  *fcp = NULL;
  double complex *dcp = NULL;

  nx = cpl_image_get_size_x(img);
  ny = cpl_image_get_size_y(img);
  switch (cpl_image_get_type(img))
    {
    case CPL_TYPE_FLOAT:
      fp = (float *)cpl_image_get_data(img);
      // umsortieren der vier Quadranten
      for (j = 0; j < ny/2; j++) {
	int js = j;
	int jd = j + ny/2;
	float   *a1 = fp + js*nx;
	float   *a2 = fp + jd*nx + nx/2;
	float    d;
	for (i = 0; i < nx/2; i++) {
	  d = *a1; *a1 = *a2; *a2 = d;
	  a1++; a2++;
	}
      }
      for (j = 0; j < ny/2; j++) {
	int js = j + ny/2;
	int jd = j;
	float   *a1 = fp + js*nx;
	float   *a2 = fp + jd*nx + nx/2;
	float    d;
	for (i = 0; i < nx/2; i++) {
	  d = *a1; *a1 = *a2; *a2 = d;
	  a1++; a2++;
	}
      }
      break;
    case CPL_TYPE_DOUBLE:
      dp = (double *)cpl_image_get_data(img);
      // umsortieren der vier Quadranten
      for (j = 0; j < ny/2; j++) {
	int js = j;
	int jd = j + ny/2;
	double   *a1 = dp + js*nx;
	double   *a2 = dp + jd*nx + nx/2;
	double    d;
	for (i = 0; i < nx/2; i++) {
	  d = *a1; *a1 = *a2; *a2 = d;
	  a1++; a2++;
	}
      }
      for (j = 0; j < ny/2; j++) {
	int js = j + ny/2;
	int jd = j;
	double   *a1 = dp + js*nx;
	double   *a2 = dp + jd*nx + nx/2;
	double    d;
	for (i = 0; i < nx/2; i++) {
	  d = *a1; *a1 = *a2; *a2 = d;
	  a1++; a2++;
	}
      }
      break;
    case CPL_TYPE_FLOAT_COMPLEX:
      fcp = (float complex *)cpl_image_get_data(img);
      // umsortieren der vier Quadranten
      for (j = 0; j < ny/2; j++) {
	int js = j;
	int jd = j + ny/2;
	float complex   *a1 = fcp + js*nx;
	float complex   *a2 = fcp + jd*nx + nx/2;
	float complex    d;
	for (i = 0; i < nx/2; i++) {
	  d = *a1; *a1 = *a2; *a2 = d;
	  a1++; a2++;
	}
      }
      for (j = 0; j < ny/2; j++) {
	int js = j + ny/2;
	int jd = j;
	float complex   *a1 = fcp + js*nx;
	float complex   *a2 = fcp + jd*nx + nx/2;
	float complex    d;
	for (i = 0; i < nx/2; i++) {
	  d = *a1; *a1 = *a2; *a2 = d;
	  a1++; a2++;
	}
      }
      break;
    case CPL_TYPE_DOUBLE_COMPLEX:
      dcp = (double complex *)cpl_image_get_data(img);
      // umsortieren der vier Quadranten
      for (j = 0; j < ny/2; j++) {
	int js = j;
	int jd = j + ny/2;
	double complex   *a1 = dcp + js*nx;
	double complex   *a2 = dcp + jd*nx + nx/2;
	double complex    d;
	for (i = 0; i < nx/2; i++) {
	  d = *a1; *a1 = *a2; *a2 = d;
	  a1++; a2++;
	}
      }
      for (j = 0; j < ny/2; j++) {
	int js = j + ny/2;
	int jd = j;
	double complex   *a1 = dcp + js*nx;
	double complex   *a2 = dcp + jd*nx + nx/2;
	double complex    d;
	for (i = 0; i < nx/2; i++) {
	  d = *a1; *a1 = *a2; *a2 = d;
	  a1++; a2++;
	}
      }
      break;
    default:;
    }
}

/**
   @brief Calculates the total flux on an image with less numerical problems.
   @param img An CPL image.
   @returns The total flux of an image.

   In order to calculate the total flux of an image, the flux of a pixel row
   is calculated and these sums are added up to the total flux. This gives
   a higher numerical accuracy, because if an image contains pixel values
   with a high dynamic range, a simple sum over all pixels depends on the pixel order.
 */
double mat_image_get_total(cpl_image *img)
{
  if (cpl_image_get_type(img) != CPL_TYPE_DOUBLE)
    {
      return cpl_image_get_flux(img);
    }
  else
    {
      double   *p = cpl_image_get_data(img);
      double    sum = 0.0;
      int       nx = cpl_image_get_size_x(img);
      int       ny = cpl_image_get_size_y(img);
      int       x, y;

      for (y = 1; y <= ny; y++)
	{
	  double hsum = 0.0;
	  for (x = 1; x <= nx; x++)
	    {
	      hsum += *p++;
	    }
	  sum += hsum;
	}
      return sum;
    }
}

/**
   @brief Normalizes the integral flux of an image.
   @param img A CPL image.
 */
void mat_image_normalize(cpl_image *img)
{
  if (cpl_image_get_type(img) != CPL_TYPE_DOUBLE)
    {
      cpl_image_normalise(img, CPL_NORM_FLUX);
      return;
    }
  else
    {
      double total = mat_image_get_total(img);
      if (total != 0.0)
	{
	  cpl_image_divide_scalar(img, total);
	}
    }
}

/**
   @brief Sets all elements of a cpl image to a specific value.
   @param img   The cpl image as destination.
   @param v     A complex double value used to initialize the image.

   If the image is a simple floating point image (CPL_TYPE_FLOAT or
   CPL_TYPE_DOUBLE), the real component of the complex parameter is used.
   If the image is a complex image, the complex parameter is directly used.
   If the image is an integer image, all pixels are set by using a combination
   of cpl_image_multiply_scalar and cpl_image_add_scalar.
*/
void mat_image_fill(cpl_image *img, double complex v)
{
  int             i;
  int             n = cpl_image_get_size_x(img)*cpl_image_get_size_y(img);
  float          *fp;
  double         *dp;
  float complex  *fcp;
  double complex *dcp;

  switch (cpl_image_get_type(img))
    {
    case CPL_TYPE_FLOAT:
      fp = (float *)cpl_image_get_data(img);
      for (i = 0; i < n; i++)
	{
	  *fp++ = (float)creal(v);
	}
      break;
    case CPL_TYPE_DOUBLE:
      dp = cpl_image_get_data(img);
      for (i = 0; i < n; i++)
	{
	  *dp++ = creal(v);
	}
      break;
    case CPL_TYPE_FLOAT_COMPLEX:
      fcp = (float complex *)cpl_image_get_data(img);
      for (i = 0; i < n; i++)
	{
	  *fcp++ = (float complex)v;
	}
      break;
    case CPL_TYPE_DOUBLE_COMPLEX:
      dcp = cpl_image_get_data(img);
      for (i = 0; i < n; i++)
	{
	  *dcp++ = v;
	}
      break;
    default:
      cpl_image_multiply_scalar(img, 0.0);
      cpl_image_add_scalar(img, creal(v));
    }
}

/**
   @brief Gets a pixel value at coordinates -N/2 .. N/2-1, -N/2 .. N/2-1.
   @param img  An image with the center at 0,0.
   @param x    The x-coordinate (-N/2 .. N/2-1).
   @param y    The y-coordinate (-N/2 .. N/2-1).
   @returns The pixel value at x,y.

   The (internal) coordinates between -N/2 and N/2-1 are mapped to
   CPL coordinates between 1 and N. The coordinate (0,0) (internal) is
   equivalent with (1,1) (CPL), (-1,0) (internal) is equivalent
   with (N,1) (CPL).
 */
double mat_image_get_double(cpl_image *img, int x, int y)
{
  int nx, ny, rejected;

  nx = cpl_image_get_size_x(img);
  ny = cpl_image_get_size_y(img);
  return cpl_image_get(img, 1 + (x + nx)%nx, 1 + (y + ny)%ny, &rejected);
}

/**
   @brief Sets a pixel at coordinates -N/2 .. N/2-1, -N/2 .. N/2-1.
   @param img  An image with the center at 0,0.
   @param x    The x-coordinate (-N/2 .. N/2-1).
   @param y    The y-coordinate (-N/2 .. N/2-1).
   @param v    The new pixel value at x,y.
 */
void mat_image_set_double(cpl_image *img, int x, int y, double v)
{
  int nx, ny;

  nx = cpl_image_get_size_x(img);
  ny = cpl_image_get_size_y(img);
  cpl_image_set(img, 1 + (x + nx)%nx, 1 + (y + ny)%ny, v);
}

/**
   @brief Adds a value to the pixel at coordinates -N/2 .. N/2-1, -N/2 .. N/2-1.
   @param img  An image with the center at 0,0.
   @param x    The x-coordinate (-N/2 .. N/2-1).
   @param y    The y-coordinate (-N/2 .. N/2-1).
   @param v    The increment at x,y.
 */
void mat_image_add_double(cpl_image *img, int x, int y, double v)
{
  int nx, ny, rejected;

  nx = cpl_image_get_size_x(img);
  ny = cpl_image_get_size_y(img);
  cpl_image_set(img, 1 + (x + nx)%nx, 1 + (y + ny)%ny, v + cpl_image_get(img, 1 + (x + nx)%nx, 1 + (y + ny)%ny, &rejected));
}

/**
   @brief Gets a complex pixel value at coordinates -N/2 .. N/2-1, -N/2 .. N/2-1.
   @param img  An image with the center at 0,0.
   @param x    The x-coordinate (-N/2 .. N/2-1).
   @param y    The y-coordinate (-N/2 .. N/2-1).
   @returns The complex pixel value at x,y.
 */
double complex mat_image_get_complex(cpl_image *img, int x, int y)
{
  int nx, ny, rejected;

  nx = cpl_image_get_size_x(img);
  ny = cpl_image_get_size_y(img);
  return cpl_image_get_complex(img, 1 + (x + nx)%nx, 1 + (y + ny)%ny, &rejected);
}

/**
   @brief Sets a complex pixel value at coordinates -N/2 .. N/2-1, -N/2 .. N/2-1.
   @param img  An image with the center at 0,0.
   @param x    The x-coordinate (-N/2 .. N/2-1).
   @param y    The y-coordinate (-N/2 .. N/2-1).
   @param v    The new pixel value at x,y.
 */
void mat_image_set_complex(cpl_image *img, int x, int y, double complex v)
{
  int nx, ny;

  nx = cpl_image_get_size_x(img);
  ny = cpl_image_get_size_y(img);
  cpl_image_set_complex(img, 1 + (x + nx)%nx, 1 + (y + ny)%ny, v);
}

/**
   @brief Adds a complex value to the pixel at coordinates -N/2 .. N/2-1, -N/2 .. N/2-1.
   @param img  An image with the center at 0,0.
   @param x    The x-coordinate (-N/2 .. N/2-1).
   @param y    The y-coordinate (-N/2 .. N/2-1).
   @param v    The complex increment at x,y.
 */
void mat_image_add_complex(cpl_image *img, int x, int y, double complex v)
{
  int nx, ny, rejected;

  nx = cpl_image_get_size_x(img);
  ny = cpl_image_get_size_y(img);
  cpl_image_set_complex(img, 1 + (x + nx)%nx, 1 + (y + ny)%ny, v + cpl_image_get_complex(img, 1 + (x + nx)%nx, 1 + (y + ny)%ny, &rejected));
}

/**
   @brief Calculates the four coefficients for a bilinear interpolation.
   @param x  The floating point x-coordinate [-N/2 .. N/2[.
   @param y  The floating point y-coordinate [-N/2 .. N/2[.
   @param w  An array with the four coefficients.
   @param ix An array with the integer four x-coordinates.
   @param iy An array with the integer four y-coordinates.

   From the floating point x- and y-coordinates the fraction is translated
   to the coefficients used for bilinear interpolation. In addition the
   correct four integer coordinates are given.
*/
static void mat_calc_bilinear_weights(double x, double y, double *w, int *ix, int *iy)
{
  int     x0, y0;
  double  dh, dv;

  x0    = (int)floor(x);
  y0    = (int)floor(y);
  dh    = x - floor(x);
  dv    = y - floor(y);
  w[0]  = (1.0 - dh)*(1.0 - dv); //  0    0
  w[1]  =         dh*(1.0 - dv); //  1    0
  w[2]  = (1.0 - dh)*dv;         //  0    1
  w[3]  =         dh*dv;         //  1    1
  ix[0] = (x0);
  ix[1] = (x0 + 1);
  ix[2] = (x0);
  ix[3] = (x0 + 1);
  iy[0] = (y0);
  iy[1] = (y0);
  iy[2] = (y0 + 1);
  iy[3] = (y0 + 1);
}

/**
   @brief Gets a double value from an image using a bilinear interpolation.
   @param src   The cpl image used as source.
   @param x     The floating point x-coordinate inside the image.
   @param y     The floating point y-coordinate inside the image.
   @returns The interpolated double value.

   After calculating the weights for a bilinear interpolation, the return
   value is calculated by a weighted sum of four pixel values. The image
   coordinates are 0-based including negative values [-N/2 .. N/2[ used for
   the image space.
*/
double mat_image_get_double_interpolated(cpl_image *src, double x, double y)
{
  double  w[4];
  int     ix[4];
  int     iy[4];

  mat_calc_bilinear_weights(x, y, w, ix, iy);
  return
      w[0]*mat_image_get_double(src, ix[0], iy[0])
    + w[1]*mat_image_get_double(src, ix[1], iy[1])
    + w[2]*mat_image_get_double(src, ix[2], iy[2])
    + w[3]*mat_image_get_double(src, ix[3], iy[3]);
}

/**
   @brief Adds a double value to an image using a bilinear interpolation.
   @param dst   The cpl image used as destination.
   @param x     The floating point x-coordinate inside the image.
   @param y     The floating point y-coordinate inside the image.
   @param val   The double value.

   After calculating the weights for a bilinear interpolation, the parameter val
   is added to four pixel values using the weights. The image
   coordinates are 0-based with negative values [-N/2 .. N/2-1] used for
   the image space.
*/
void mat_image_add_double_interpolated(cpl_image *dst, double x, double y, double val)
{
  double  w[4];
  int     ix[4];
  int     iy[4];

  mat_calc_bilinear_weights(x, y, w, ix, iy);
  mat_image_add_double(dst, ix[0], iy[0], val*w[0]);
  mat_image_add_double(dst, ix[1], iy[1], val*w[1]);
  mat_image_add_double(dst, ix[2], iy[2], val*w[2]);
  mat_image_add_double(dst, ix[3], iy[3], val*w[3]);
}

/**
   @brief Gets a double complex value from an image using a bilinear interpolation.
   @param src   The cpl image used as source.
   @param x     The floating point x-coordinate inside the image.
   @param y     The floating point y-coordinate inside the image.
   @returns The interpolated double complex value.

   After calculating the weights for a bilinear interpolation, the return
   value is calculated by a weighted sum of four pixel values. The image
   coordinates are 0-based with negative values [-N/2 .. N/2-1] used for
   the Fourier space.
*/
double complex mat_image_get_complex_interpolated(cpl_image *src, double x, double y)
{
  double  w[4];
  int     ix[4];
  int     iy[4];

  mat_calc_bilinear_weights(x, y, w, ix, iy);
  return
    (w[0] + 0.0*I)*mat_image_get_complex(src, ix[0], iy[0])
    + (w[1] + 0.0*I)*mat_image_get_complex(src, ix[1], iy[1])
    + (w[2] + 0.0*I)*mat_image_get_complex(src, ix[2], iy[2])
    + (w[3] + 0.0*I)*mat_image_get_complex(src, ix[3], iy[3]);
}

/**
   @brief Adds a double complex value to an image using a bilinear interpolation.
   @param dst   The cpl image used as destination.
   @param x     The floating point x-coordinate inside the image.
   @param y     The floating point y-coordinate inside the image.
   @param val   The double complex value.

   After calculating the weights for a bilinear interpolation, the parameter val
   is added to four pixel values using the weights. In addition the conjugated value
   is added to the mirrored position. The image
   coordinates are 0-based with negative values [-N/2 .. N/2-1] used for
   the Fourier space.
*/
void mat_image_add_complex_interpolated(cpl_image *dst, double x, double y, double complex val)
{
  double  w[4];
  int     ix[4];
  int     iy[4];

  mat_calc_bilinear_weights(x, y, w, ix, iy);
  mat_image_add_complex(dst,  ix[0],  iy[0], val*(w[0] + 0.0*I));
  mat_image_add_complex(dst, -ix[0], -iy[0], conj(val)*(w[0] + 0.0*I));
  mat_image_add_complex(dst,  ix[1],  iy[1], val*(w[1] + 0.0*I));
  mat_image_add_complex(dst, -ix[1], -iy[1], conj(val)*(w[1] + 0.0*I));
  mat_image_add_complex(dst,  ix[2],  iy[2], val*(w[2] + 0.0*I));
  mat_image_add_complex(dst, -ix[2], -iy[2], conj(val)*(w[2] + 0.0*I));
  mat_image_add_complex(dst,  ix[3],  iy[3], val*(w[3] + 0.0*I));
  mat_image_add_complex(dst, -ix[3], -iy[3], conj(val)*(w[3] + 0.0*I));
}

/**
   @brief Calculates the FFT (double -> double complex).
   @param dst  Contains the double complex image for the FFT.
   @param src  Contains a double image as source.

   The double image is converted into a double complex image and stored in the
   destination double complex image. After that, an inplace FFT is applied to the
   destination double complex image.
 */
void mat_image_fft_forward(cpl_image *dst, cpl_image *src)
{
  int x, y, rejected;
  int n = cpl_image_get_size_x(src);

  for (x = 1; x <= n; x++)
    {
      for (y = 1; y <= n; y++)
	{
	  double v = cpl_image_get(src, x, y, &rejected);
	  cpl_image_set_complex(dst, x, y, v + 0.0*I);
	}
    }
  // we use a complex -> complex FFT
  cpl_fft_image(dst, dst, CPL_FFT_FORWARD);
}

/**
   @brief Calculates the inverse FFT (double complex -> double) including scaling
   @param dst    Contains the double image as result.
   @param src    Contains the double complex image.
   @param scale  Contains a double complex as scale (applid before the FFT).

   The double complex image is scaled and an inplace inverse FFT applied. The
   real part of the result is copied into the destination double image.
 */
void mat_image_fft_backward(cpl_image *dst, cpl_image *src, double complex scale)
{
  int x, y, rejected;
  int n = cpl_image_get_size_x(src);

  // scale the input (complex) array
  for (x = 1; x <= n; x++)
    {
      for (y = 1; y <= n; y++)
	{
	  double complex v = scale*cpl_image_get_complex(src, x, y, &rejected);
	  cpl_image_set_complex(src, x, y, v);
	}
    }
  // do the inverse FFT (inplace, complex -> complex)
  cpl_fft_image(src, src, CPL_FFT_BACKWARD);
  // convert the complex result into a double image (using only the real part)
  for (x = 1; x <= n; x++)
    {
      for (y = 1; y <= n; y++)
	{
	  double v = creal(cpl_image_get_complex(src, x, y, &rejected));
	  cpl_image_set(dst, x, y, v);
	}
    }
}

/*
   @brief Calculates the image centroid [-N/2 .. N/2[, [-N/2 .. N/2[.
   @param img  Contains the input image (coordinates [-N/2 .. N/2[, [-N/2 .. N/2[).
   @param cx   Will contain the x-coordinate of the image centroid as double.
   @param cy   Will contain the y-coordinate of the image centroid as double.

   This method calculates the image centroid for an image using coordinates
   between -N/2 and N/2-1. The calculation gives better results for images
   with a high dynamic range.

   This method is only implemented for images of type CPL_TYPE_DOUBLE. For all
   other image types, the CPL method cpl_stats_new_from_image(img, CPL_STATS_CENTROID)
   is used.
 */
void mat_image_get_centroid(cpl_image *img, double *cx, double *cy)
{
  if (cpl_image_get_type(img) != CPL_TYPE_DOUBLE)
    {
      cpl_stats *stats;
      mat_image_swap(img);
      stats = cpl_stats_new_from_image(img, CPL_STATS_CENTROID);
      mat_image_swap(img);
      if (stats != NULL)
	{
	  *cx = cpl_stats_get_centroid_x(stats);
	  *cy = cpl_stats_get_centroid_y(stats);
	  cpl_stats_delete(stats);
	}
      else
	{
	  cpl_msg_error(cpl_func, "can't calculate the centroid for the start image");
	}
    }
  else
    {
      int  nx = cpl_image_get_size_x(img);
      int  ny = cpl_image_get_size_y(img);
      int  x, y;
      double xsum = 0.0, xcount = 0.0;
      double ysum = 0.0, ycount = 0.0;
      for (y = -ny/2; y < ny/2; y++)
	{
	  double hsum = 0.0;
	  for (x = -nx/2; x < nx/2; x++)
	    {
	      hsum += mat_image_get_double(img, x, y);
	    }
	  ysum   += hsum*(double)y;
	  ycount += hsum;
	}
      if (ycount != 0.0)
	{
	  *cy = ysum/ycount;
	}
      else
	{
	  *cy = 0.0;
	}
      for (x = -nx/2; x < nx/2; x++)
	{
	  double hsum = 0.0;
	  for (y = -ny/2; y < ny/2; y++)
	    {
	      hsum += mat_image_get_double(img, x, y);
	    }
	  xsum   += hsum*(double)x;
	  xcount += hsum;
	}
      if (xcount != 0.0)
	{
	  *cx = xsum/xcount;
	}
      else
	{
	  *cx = 0.0;
	}
    }
}

/**
   @brief Rounds all values of an image to a certain number of digits.
   @param img        An image.
   @param precision  Number of digits after the decimal point.
   @returns void

   All values of an image are rounded to a certain number of digits after the decimal point.
   This method is used to limit the precision of the gradient image which goes into
   the ASA-CG algorithm.
 */
void mat_image_round(cpl_image *img, int precision)
{
  int             i;
  int             n = cpl_image_get_size_x(img)*cpl_image_get_size_y(img);
  float          *fp;
  double         *dp;
  float complex  *fcp;
  double complex *dcp;
  float           fscale = 1.0;
  double          dscale = 1.0;

  while (precision != 0) {
    fscale *= 10.0;
    dscale *= 10.0;
    precision--;
  }
  switch (cpl_image_get_type(img))
    {
    case CPL_TYPE_FLOAT:
      fp = (float *)cpl_image_get_data(img);
      for (i = 0; i < n; i++)
	{
	  *fp = roundf(*fp * fscale)/fscale;
	  fp++;
	}
      break;
    case CPL_TYPE_DOUBLE:
      dp = cpl_image_get_data(img);
      for (i = 0; i < n; i++)
	{
	  *dp = round(*dp * dscale)/dscale;
	  dp++;
	}
      break;
    case CPL_TYPE_FLOAT_COMPLEX:
      fcp = (float complex *)cpl_image_get_data(img);
      for (i = 0; i < n; i++)
	{
	  *fcp = roundf(creal(*fcp) * fscale)/fscale + roundf(cimag(*fcp) * fscale)/fscale*I;
	  fcp++;
	}
      break;
    case CPL_TYPE_DOUBLE_COMPLEX:
      dcp = cpl_image_get_data(img);
      for (i = 0; i < n; i++)
	{
	  *dcp = round(creal(*dcp) * dscale)/dscale + round(cimag(*dcp) * dscale)/dscale*I;
	  dcp++;
	}
      break;
    default:;
    }
}

void mat_image_round_relative(cpl_image *img, int precision)
{
  int             i;
  int             n = cpl_image_get_size_x(img)*cpl_image_get_size_y(img);
  float          *fp;
  double         *dp;
  float complex  *fcp;
  double complex *dcp;
  switch (cpl_image_get_type(img))
    {
    case CPL_TYPE_FLOAT:
      fp = (float *)cpl_image_get_data(img);
      for (i = 0; i < n; i++)
	{
	  *fp = (float)mat_round_relative((double)*fp, precision);
	  fp++;
	}
      break;
    case CPL_TYPE_DOUBLE:
      dp = cpl_image_get_data(img);
      for (i = 0; i < n; i++)
	{
	  *dp = mat_round_relative(*dp, precision);
	  dp++;
	}
      break;
    case CPL_TYPE_FLOAT_COMPLEX:
      fcp = (float complex *)cpl_image_get_data(img);
      for (i = 0; i < n; i++)
	{
	  *fcp = (float)mat_round_relative((double)creal(*fcp), precision) + (float)mat_round_relative((double)cimag(*fcp), precision)*I;
	  fcp++;
	}
      break;
    case CPL_TYPE_DOUBLE_COMPLEX:
      dcp = cpl_image_get_data(img);
      for (i = 0; i < n; i++)
	{
	  *dcp = mat_round_relative(creal(*dcp), precision) + mat_round_relative(cimag(*dcp), precision)*I;
	  dcp++;
	}
      break;
    default:;
    }
}

/**
   @brief Rescale an image from a given pixel scane to a destination pixel scale.
   @param dst       The cpl image used as destination.
   @param dstscale  The pixel scale of the destination imape [mas/pixel].
   @param src       The cpl image used as source.
   @param srcscale  The pixel scale of the source imape [mas/pixel].
   @returns An error

   The source image (src) with a given pixel scale (srcscale) is rescaled
   with a different pixel scale (dstscale) and stored in another image (dst).
   A simple bilinear interpolation is used. This means only small scale differences
   are allowed.
*/
cpl_error_code mat_image_scale(cpl_image *dst, double dstscale, cpl_image *src, double srcscale)
{
  cpl_size dnx = cpl_image_get_size_x(dst);
  cpl_size dny = cpl_image_get_size_y(dst);
  cpl_size snx = cpl_image_get_size_x(src);
  cpl_size sny = cpl_image_get_size_y(src);
  double   zoom = dstscale/srcscale;

  mat_image_fill(dst, 0.0);
  if (zoom < 1.0)
    {
      cpl_size dx, dy;
      for (dy = -dny/2; dy < dny/2; dy++)
	{
	  double sy = zoom*(double)dy;
	  if (sy < -sny/2) continue;  // we are outside of the source image
	  if (sy >= sny/2) continue;  // we are outside of the source image
	  for (dx = -dnx/2; dx < dnx/2; dx++)
	    {
	      double sx = zoom*(double)dx;
	      if (sx < -snx/2) continue;  // we are outside of the source image
	      if (sx >= snx/2) continue;  // we are outside of the source image
	      mat_image_set_double(dst, dx, dy, mat_image_get_double_interpolated(src, sx, sy));
	    }
	}
    }
  else
    {
      cpl_size sx, sy;
      cpl_image *weight = cpl_image_new(dnx, dny, CPL_TYPE_DOUBLE);
      if (weight == NULL)
	{
	  cpl_msg_error(cpl_func, "can't allocate memory for weight image");
	}
      zoom = 1.0/zoom;
      for (sy = -snx/2; sy < sny/2; sy++)
	{
	  double dy = zoom*(double)sy;
	  if (dy < -dny/2) continue;
	  if (dy >= dny/2) continue;
	  for (sx = -snx/2; sx < snx/2; sx++)
	    {
	      double dx = zoom*(double)sx;
	      if (dx < -dnx/2) continue;
	      if (dx >= dnx/2) continue;
	      mat_image_add_double_interpolated(dst, dx, dy, mat_image_get_double(src, sx, sy));
	      if (weight != NULL)
		{
		  mat_image_add_double_interpolated(weight, dx, dy, 1.0);
		}
	    }
	}
      if (weight != NULL)
	{
	  cpl_size dx, dy;
	  for (dy = 0; dy < dny; dy++)
	    {
	      for (dx = 0; dx < dnx; dx++)
		{
		  double w = mat_image_get_double(weight, dx, dy);
		  if (w > 0.0)
		    {
		      mat_image_set_double(dst, dx, dy, mat_image_get_double(dst, dx, dy)/w);
		    }
		}
	    }
	  cpl_image_delete(weight);
	}
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Rescale an image from a given pixel scane to a destination pixel scale.
   @param dst       The cpl image used as destination.
   @param xshift    The pixel shift in horizontal direction (dx = sx + xshift).
   @param yshift    The pixel shift in vertical direction (dy = sy + yshift).
   @param src       The cpl image used as source.
   @returns An error

   The source image (src) is shifted to a new position. A positive xshift shifts
   the source image to the right, a positive yshift shifts the source image up.
   A simple bilinear interpolation is used.
*/
cpl_error_code mat_image_shift(cpl_image *dst, double xshift, double yshift, cpl_image *src)
{
  cpl_size dnx = cpl_image_get_size_x(dst);
  cpl_size dny = cpl_image_get_size_y(dst);
  cpl_size snx = cpl_image_get_size_x(src);
  cpl_size sny = cpl_image_get_size_y(src);
  cpl_size dx, dy;

  mat_image_fill(dst, 0.0);
  mat_image_swap(src);
  for (dy = 0; dy < dny; dy++)
    {
      double sy = (double)dy - yshift;
      if (sy < 0.0) continue;    // we are outside of the source image
      if (sy > (double)sny) continue;  // we are outside of the source image
      for (dx = 0; dx < dnx; dx++)
	{
	  double sx = (double)dx - xshift;
	  if (sx < 0.0) continue;    // we are outside of the source image
	  if (sx > (double)snx) continue;  // we are outside of the source image
	  mat_image_set_double(dst, dx, dy, mat_image_get_double_interpolated(src, sx, sy));
	}
    }
  mat_image_swap(src);
  mat_image_swap(dst);
  return CPL_ERROR_NONE;
}

/**
   @brief Calculate an artificial PSF.
   @param tent   Will contain the scale factors in Fourier space.
   @param r0     Baseline for specifying a circular shaped PSF.

   Calculates an artificial PSF for a circular telescope with a given diameter.
   Before calling this function, the real baseline may be scaled by a factor to give r0.
*/
void mat_image_calc_tent(cpl_image *tent, double r0)
{
  int x, y, n;
  double rk0 = acos(0.0);

  n = cpl_image_get_size_x(tent);
  for (x = -n/2; x < n/2; x++)
    {
      for (y = -n/2; y < n/2; y++)
	{
	  double r = sqrt((double)(x*x + y*y));
	  if (r < r0)
	    {
	      double ratio = r/r0;
	      double rk = acos(ratio) - ratio*sqrt(1 - ratio*ratio);
	      mat_image_set_double(tent, x, y, rk/rk0);
	    }
	  else
	    {
	      mat_image_set_double(tent, x, y, 0.0);
	    }
	}
    }
}
/**
   @brief Calculate an artificial Gaussian PSF.
   @param tent   Will contain the scale factors in Fourier space.
   @param r0     Baseline for specifying a circular shaped PSF.

   Calculates a Gaussian as an artificial PSF of a circular telescope. The advantage of a Gaussian PSF
   is the absence of disturbing rings which occure in the correct PSF of a circular telescope.
   The 1/e decay of the Gaussian is r0/2, where r0 is the maximum baseline of the telescope array
   (may be scaled by a factor). The 2-dimensional integrals of the Gaussian PSF and the correct PSF are identical.
*/
void mat_image_calc_gaussian(cpl_image *gaussian, double r0)
{
  int x, y, n;

  n = cpl_image_get_size_x(gaussian);
  for (x = -n/2; x < n/2; x++)
    {
      for (y = -n/2; y < n/2; y++)
	{
	  double r = sqrt((double)(x*x + y*y));
	  double g = exp(-pow(r/(r0/2.0), 2.0));
	  mat_image_set_double(gaussian, x, y, g);
	}
    }
}


/**
   @brief Convolve an image with an artificial PSF.
   @param dst   Will contain the convolved image.
   @param src   Contains the image (e.g. reconstruction).
   @param tent  Contain a PSF as (FFT) scale factor.
   @param hci   A temporary complex double image.

   Calculates a simple convolution using an artificial PSF given as scale factors in Fourier space.
*/
void mat_image_convolve(cpl_image *dst, cpl_image *src, cpl_image *tent, cpl_image *hci)
{
  int x, y, n, rejected;
  double  r0;

  n = cpl_image_get_size_x(dst);
  mat_image_fft_forward(hci, src);
  r0 = cabs(mat_image_get_complex(hci, 0, 0))*mat_image_get_double(tent, 0, 0);
  for (x = 1; x <= n; x++)
    {
      for (y = 1; y <= n; y++)
	{
	  cpl_image_set_complex(hci, x, y, cpl_image_get_complex(hci, x, y, &rejected)*(cpl_image_get(tent, x, y, &rejected)/r0 + 0.0*I));
	}
    }
  mat_image_fft_backward(dst, hci, 1.0);
}

/**
   @brief Stores a cpl image into a FITS file, used for debugging purposes.
   @param img   A CPL image.
   @param fname   The FITS file name
   @param catg    The PRO.CATG keyword value.
   @param parlist  The parameterlist of the plugin.
   @param frameset The frameset of the plugin.
   @returns An cpl_error_code

   This is a simple convinience method to store a CPL image in a FITS file.
*/
cpl_error_code mat_image_store_double(cpl_image *img,
				      const char *fname,
				      const char *catg,
				      cpl_parameterlist *parlist,
				      cpl_frameset *frameset)
{
  cpl_frame        *statframe = NULL;
  cpl_propertylist *plist = NULL;

  cpl_error_reset();
  /* Create product frame */
  statframe = cpl_frame_new();
  cpl_frame_set_filename(statframe, fname);
  cpl_frame_set_tag(statframe, catg);
  cpl_frame_set_type(statframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(statframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(statframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_frame_delete(statframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  if (cpl_dfs_setup_product_header(plist, statframe, frameset, parlist,
				   "mat_info_store_statistics", "MATISSE", "?Dictionary?", NULL) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_error_reset();
    cpl_propertylist_delete(plist);
    cpl_frame_delete(statframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  /* Add QC parameters in the header */
  /*
    cpl_propertylist_append_double(plist, "ESO QC QCPARAM", qc_param);
  */
  /* Save the file */
  if (cpl_image_save(img, fname, CPL_TYPE_FLOAT, plist,
		     CPL_IO_CREATE) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_frame_delete(statframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  cpl_frame_delete(statframe);
  return CPL_ERROR_NONE;
}

/**
   @brief Stores a cpl imagelist into a FITS file, used for debugging purposes.
   @param img_list   A CPL image list.
   @param fname   The FITS file name
   @param catg    The PRO.CATG keyword value.
   @param parlist  The parameterlist of the plugin.
   @param frameset The frameset of the plugin.
   @returns An cpl_error_code

   This is a simple convinience method to store a CPL image in a FITS file.
*/
cpl_error_code mat_imagelist_store_double(cpl_imagelist *img_list,
					  const char *fname,
					  const char *catg,
					  cpl_parameterlist *parlist,
					  cpl_frameset *frameset)
{
  cpl_frame        *statframe = NULL;
  cpl_propertylist *plist = NULL;

  cpl_error_reset();
  /* Create product frame */
  statframe = cpl_frame_new();
  cpl_frame_set_filename(statframe, fname);
  cpl_frame_set_tag(statframe, catg);
  cpl_frame_set_type(statframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(statframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(statframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code())
    {
      cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_frame_delete(statframe);
      return CPL_ERROR_UNSPECIFIED;
    }
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  if (cpl_dfs_setup_product_header(plist, statframe, frameset, parlist,
				   "mat_info_store_statistics", "MATISSE", "?Dictionary?", NULL) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_error_reset();
      cpl_propertylist_delete(plist);
      cpl_frame_delete(statframe);
      return CPL_ERROR_UNSPECIFIED;
    }
  /* Add QC parameters in the header */
  /*
    cpl_propertylist_append_double(plist, "ESO QC QCPARAM", qc_param);
  */
  /* Save the file */
  if (cpl_imagelist_save(img_list, fname, CPL_TYPE_FLOAT, plist,
			 CPL_IO_CREATE) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_propertylist_delete(plist);
      cpl_frame_delete(statframe);
      return CPL_ERROR_UNSPECIFIED;
    }
  cpl_propertylist_delete(plist);
  cpl_frame_delete(statframe);
  return CPL_ERROR_NONE;
}

/**
   @brief Stores a cpl image (double complex) into a FITS file, used for debugging purposes.
   @param cimg   A CPL image.
   @param fname   The FITS file name
   @param catg    The PRO.CATG keyword value.
   @param parlist  The parameterlist of the plugin.
   @param frameset The frameset of the plugin.
   @returns An cpl_error_code

   This is a simple convinience method to store a CPL image in a FITS file.
   The double complex image is stored as four images. The first image contains
   the real part, the second contains the imaginary part of a complex image.
   The third image contains the amplitude and the fourth image the phase of
   the complex image.
*/
cpl_error_code mat_image_store_complex(cpl_image *cimg,
				       const char *fname,
				       const char *catg,
				       cpl_parameterlist *parlist,
				       cpl_frameset *frameset)
{
  int             x, y, n, rejected;
  cpl_imagelist  *img_list = NULL;
  cpl_image      *rimg = NULL;
  cpl_image      *iimg = NULL;
  cpl_image      *aimg = NULL;
  cpl_image      *pimg = NULL;
  cpl_error_code  rc;

  img_list = cpl_imagelist_new();
  if (img_list == NULL)
    {
      cpl_msg_error(cpl_func, "Error while creating the imagelist, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      return CPL_ERROR_UNSPECIFIED;
    }
  n = cpl_image_get_size_x(cimg);
  rimg = cpl_image_new(n, n, CPL_TYPE_DOUBLE);
  if (rimg == NULL)
    {
      cpl_msg_error(cpl_func, "Error while creating the image for the real part, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_imagelist_delete(img_list);
      return CPL_ERROR_UNSPECIFIED;
    }
  iimg = cpl_image_new(n, n, CPL_TYPE_DOUBLE);
  if (iimg == NULL)
    {
      cpl_msg_error(cpl_func, "Error while creating the image for the imaginary part, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_imagelist_delete(img_list);
      cpl_image_delete(rimg);
      return CPL_ERROR_UNSPECIFIED;
    }
  aimg = cpl_image_new(n, n, CPL_TYPE_DOUBLE);
  if (aimg == NULL)
    {
      cpl_msg_error(cpl_func, "Error while creating the image for the modulus, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_imagelist_delete(img_list);
      cpl_image_delete(rimg);
      cpl_image_delete(iimg);
      return CPL_ERROR_UNSPECIFIED;
    }
  pimg = cpl_image_new(n, n, CPL_TYPE_DOUBLE);
  if (pimg == NULL)
    {
      cpl_msg_error(cpl_func, "Error while creating the image for the phase angle, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_imagelist_delete(img_list);
      cpl_image_delete(rimg);
      cpl_image_delete(iimg);
      cpl_image_delete(aimg);
      return CPL_ERROR_UNSPECIFIED;
    }
  cpl_imagelist_set(img_list, rimg, 0);
  cpl_imagelist_set(img_list, iimg, 1);
  cpl_imagelist_set(img_list, aimg, 2);
  cpl_imagelist_set(img_list, pimg, 3);
  for (y = 1; y <= n; y++)
    {
      for (x = 1; x <= n; x++)
	{
	  double complex v = cpl_image_get_complex(cimg, x, y, &rejected);
	  cpl_image_set(rimg, x, y, creal(v));
	  cpl_image_set(iimg, x, y, cimag(v));
	  cpl_image_set(aimg, x, y, cabs(v));
	  cpl_image_set(pimg, x, y, carg(v));
	}
    }
  rc = mat_imagelist_store_double(img_list, fname, catg, parlist, frameset);
  cpl_imagelist_delete(img_list);
  return rc;
}

cpl_error_code mat_image_create_gaussian(cpl_image *img, double fwhm)
{
  double dsize = fwhm*fwhm*FWHM2SIGMA2;
  int    nx = cpl_image_get_size_x(img);
  int    ny = cpl_image_get_size_y(img);
  int    x, y;

  for (x = -nx/2; x < nx/2; x++)
    {
      for (y = -ny/2; y < ny/2; y++)
	{
	  double r2 = (double)(x*x + y*y);
	  mat_image_set_double(img, x, y, exp(-0.5*r2/dsize));
	}
    }
  return CPL_ERROR_NONE;
}

cpl_error_code mat_image_create_lorentzian(cpl_image *img, double fwhm)
{
  double a2 = FWHM2LDA*FWHM2LDA/fwhm/fwhm;
  int    nx = cpl_image_get_size_x(img);
  int    ny = cpl_image_get_size_y(img);
  int    x, y;

  for (x = -nx/2; x < nx/2; x++)
    {
      for (y = -ny/2; y < ny/2; y++)
	{
	  double r2 = (double)(x*x + y*y);
	  mat_image_set_double(img, x, y, 1.0/pow(1.0 + a2*r2, 1.5));
	}
    }
  return CPL_ERROR_NONE;
}

cpl_error_code mat_image_get_quantils(cpl_image *img, int nquantils, double *quantils, double *values)
{
  cpl_vector  *vec;
  cpl_vector  *hvec;
  int          i, n;

  mat_assert_not_null(img, "no valid cpl_image (img) given as argument");
  mat_assert_not_null(quantils, "no valid double* (quantils) given as argument");
  mat_assert_not_null(values, "no valid double* (values) given as argument");
  n = cpl_image_get_size_x(img)*cpl_image_get_size_y(img);
  vec = cpl_vector_new(n);
  mat_assert_not_null(vec, "could not allocate memory for temporary cpl_vector for sorting");
  hvec = cpl_vector_wrap(n, (double*)cpl_image_get_data(img));
  cpl_vector_copy(vec, hvec);
  cpl_vector_unwrap(hvec);
  cpl_vector_sort(vec, CPL_SORT_ASCENDING);
  for (i = 0; i < nquantils; i++)
    {
      int index = (int)round(quantils[i]*(double)n);
      values[i] = cpl_vector_get(vec, index);
    }
  cpl_vector_delete(vec);
  return CPL_ERROR_NONE;
}

/*
#define MAT_IS_LESS       -1
#define MAT_IS_INSIDE      0
#define MAT_IS_GREATER    +1
#define MAT_IS_INFINITE    2
#define MAT_IS_NAN         3

there is a sequence of precedence:

INSIDE -> LESS/GREATER -> INFINITE/NAN
 */
cpl_error_code mat_image_validate(cpl_image *img, cpl_image *mask, double lower, double upper, const char *label)
{
  int    x, nx, y, ny;
  int    count_less    = 0;
  int    count_inside  = 0;
  int    count_greater = 0;
  int    count_inf     = 0;
  int    count_nan     = 0;
  int    count_zero    = 0;
  int    count_one     = 0;

  mat_assert_not_null(img, "no valid cpl_image (img) given as argument");
  nx = cpl_image_get_size_x(img);
  ny = cpl_image_get_size_y(img);
  for (y = 1; y <= ny; y++)
    {
      for (x = 1; x <= nx; x++)
	{
	  int rejected;
	  double v = cpl_image_get(img, x, y, &rejected);
	  double m = MAT_IS_INSIDE;
	  if (isnan(v))
	    {
	      count_nan++;
	      m = MAT_IS_NAN;
	      if ((label != NULL) && (count_nan == 1))
		{
		  cpl_msg_info(cpl_func, "image %s contains a NAN at (%d,%d)", label, x, y);
		}
	    }
	  else if (isinf(v))
	    {
	      count_inf++;
	      m = MAT_IS_INFINITE;
	      if ((label != NULL) && (count_inf == 1))
		{
		  cpl_msg_info(cpl_func, "image %s contains a INF at (%d,%d)", label, x, y);
		}
	    }
	  else
	    {
	      if (v == 0.0) count_zero++;
	      if (v == 1.0) count_one++;
	      if (v < lower)
		{
		  count_less++;
		  m = MAT_IS_LESS;
		}
	      else if (v > upper)
		{
		  count_greater++;
		  m = MAT_IS_GREATER;
		}
	      else
		{
		  count_inside++;
		  m = MAT_IS_INSIDE;
		}
	    }
	  if (mask != NULL)
	    {
	      double om = cpl_image_get(mask, x, y, &rejected);
	      if (m == MAT_IS_LESS)
		{
		  if (om == MAT_IS_INSIDE) cpl_image_set(mask, x, y, MAT_IS_LESS);
		}
	      else if (m == MAT_IS_GREATER)
		{
		  if (om == MAT_IS_INSIDE) cpl_image_set(mask, x, y, MAT_IS_GREATER);
		}
	      else if ((m == MAT_IS_INFINITE) || (m == MAT_IS_NAN))
		{
		  cpl_image_set(mask, x, y, m);
		}
	    }
	}
    }
  if (label != NULL)
    {
      if ((count_inf + count_nan) != 0)
	{
	  cpl_msg_warning(cpl_func, "image %s contains less %d inside %d greater %d infinite %d nan %d zero %d one %d",
			  label,
			  count_less, count_inside, count_greater,
			  count_inf, count_nan,
			  count_zero, count_one);
	}
      else
	{
	  cpl_msg_info(cpl_func, "image %s contains less %d inside %d greater %d infinite %d nan %d zero %d one %d",
		       label,
		       count_less, count_inside, count_greater,
		       count_inf, count_nan,
		       count_zero, count_one);
	}
    }
  return CPL_ERROR_NONE;
}

typedef struct {
  double x;
  double y;
  double z;
} mat_vector3d;

static void mat_vector3d_set(mat_vector3d *v, double x, double y, double z)
{
  v->x = x;
  v->y = y;
  v->z = z;
}

static void mat_vector3d_normalize(mat_vector3d *v)
{
  double f = 1.0/sqrt(v->x*v->x + v->y*v->y + v->z*v->z);
  v->x *= f;
  v->y *= f;
  v->z *= f;
}

static void mat_vector3d_subtract(mat_vector3d *vr, const mat_vector3d *v1, const mat_vector3d *v2)
{
  vr->x = v1->x - v2->x;
  vr->y = v1->y - v2->y;
  vr->z = v1->z - v2->z;
}

static void mat_vector3d_cross(mat_vector3d *vr, const mat_vector3d *v1, const mat_vector3d *v2)
{
  vr->x = v1->y*v2->z - v1->z*v2->y;
  vr->y = v2->x*v1->z - v2->z*v1->x;
  vr->z = v1->x*v2->y - v1->y*v2->x;
}

static void mat_write_stl_header(FILE *fp, int nfacets)
{
  char           header[80];
  uint32_t       hnfacets;
  size_t         i;

  hnfacets = (uint32_t)nfacets;
  for (i = 0; i < sizeof(header); i++) header[i] = '\0';
  strcpy(header, "MATISSE DRS plain STL\n");
  fwrite(header, sizeof(header), 1, fp);
  fwrite(&hnfacets, sizeof(hnfacets), 1, fp);
}

// the three points are counter clockwise seen from the outside
static void mat_write_stl_triangle(FILE *fp, const mat_vector3d *p1, const mat_vector3d *p2, const mat_vector3d *p3)
{
  mat_vector3d    normal;
  mat_vector3d    d12, d13;
  float           facet[12];
  uint16_t        attr = 0;

  mat_vector3d_subtract(&d13, p3, p1);
  mat_vector3d_subtract(&d12, p2, p1);
  mat_vector3d_cross(&normal, &d12, &d13);
  mat_vector3d_normalize(&normal);
  facet[ 0] = normal.x;
  facet[ 1] = normal.y;
  facet[ 2] = normal.z;
  facet[ 3] = p1->x;
  facet[ 4] = p1->y;
  facet[ 5] = p1->z;
  facet[ 6] = p2->x;
  facet[ 7] = p2->y;
  facet[ 8] = p2->z;
  facet[ 9] = p3->x;
  facet[10] = p3->y;
  facet[11] = p3->z;
  fwrite(facet, sizeof(facet), 1, fp);
  fwrite(&attr, sizeof(attr), 1, fp);
}

// the four points are counter clockwise seen from the outside
static  void mat_write_stl_quad(FILE *fp, const mat_vector3d *p1, const mat_vector3d *p2, const mat_vector3d *p3, const mat_vector3d *p4)
{
  mat_vector3d    normal;
  mat_vector3d    d12, d13;
  float           facet[12];
  uint16_t        attr = 0;

  mat_vector3d_subtract(&d13, p3, p1);
  mat_vector3d_subtract(&d12, p2, p1);
  mat_vector3d_cross(&normal, &d12, &d13);
  mat_vector3d_normalize(&normal);
  facet[ 0] = normal.x;
  facet[ 1] = normal.y;
  facet[ 2] = normal.z;
  facet[ 3] = p1->x;
  facet[ 4] = p1->y;
  facet[ 5] = p1->z;
  facet[ 6] = p2->x;
  facet[ 7] = p2->y;
  facet[ 8] = p2->z;
  facet[ 9] = p3->x;
  facet[10] = p3->y;
  facet[11] = p3->z;
  fwrite(facet, sizeof(facet), 1, fp);
  fwrite(&attr, sizeof(attr), 1, fp);
  facet[ 3] = p1->x;
  facet[ 4] = p1->y;
  facet[ 5] = p1->z;
  facet[ 6] = p3->x;
  facet[ 7] = p3->y;
  facet[ 8] = p3->z;
  facet[ 9] = p4->x;
  facet[10] = p4->y;
  facet[11] = p4->z;
  fwrite(facet, sizeof(facet), 1, fp);
  fwrite(&attr, sizeof(attr), 1, fp);
}

cpl_error_code mat_image_store_plain_stl(const char *fname, const cpl_image *img, int x, int y, int nx, int ny, double zscale)
{
  FILE          *fp;
  int            nfacets;
  int            i, j;
  double         imin, imax;
  int            rejected;
  double         zoffset, zf;
  double         v1, v2, v3, v4;
  double         z1, z2, z3, z4;
  mat_vector3d   p1t, p2t, p3t, p4t, p1b, p2b, p3b, p4b;

  // determine the number of triangles (facets)
  nfacets  = 2*(nx-1)*(ny-1);     // between four pixel 2 triangles forming a quad
  nfacets += 4*(ny-1);            // pixel walls at left/right side of the image
  nfacets += 4*(nx-1);            // pixel walls at bottom/top side of the image
  nfacets += 2*(nx-1) + 2*(ny-1); // triangles at the bottom of the 3-d mountain
  imin = cpl_image_get(img, x, y, &rejected);
  imax = imin;
  for (i = 0; i < nx; i++)
    {
      for (j = 0; j < ny; j++)
	{
	  double v = cpl_image_get(img, x + i, y + j, &rejected);
	  imin = CPL_MIN(imin, v);
	  imax = CPL_MAX(imax, v);
	}
    }
  // check if the image contains only one value (horizontal plane)
  if (imin == imax)
    {
      cpl_msg_info(cpl_func, "Image is a horizontal plane, no STL file (%s) created", fname);
      return CPL_ERROR_UNSPECIFIED;
    }
  zoffset = imin;
  zf = -1.0;
  // Create a new STL file for a plain image view
  if ((fp = fopen(fname, "w")) == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot create plain STL file %s", fname);
      return CPL_ERROR_FILE_NOT_CREATED;
    }
  // write STL header
  mat_write_stl_header(fp, nfacets);
  // Assignment of the image intensities, calculated z height and points
  //  v4---------v3
  //  |          |
  //  |          |
  //  |     Q    |
  //  |          |
  //  |          |
  //  v1---------v2
  //
  for (i = 0; i < nx-1; i++)
    {
      for (j = 0; j < ny-1; j++)
	{
	  v1 = cpl_image_get(img, x+i,   y+j,   &rejected);
	  v2 = cpl_image_get(img, x+i+1, y+j,   &rejected);
	  v3 = cpl_image_get(img, x+i+1, y+j+1, &rejected);
	  v4 = cpl_image_get(img, x+i,   y+j+1, &rejected);
	  z1 = (v1 - zoffset)/zscale;
	  z2 = (v2 - zoffset)/zscale;
	  z3 = (v3 - zoffset)/zscale;
	  z4 = (v4 - zoffset)/zscale;
	  // create the 3D points
	  mat_vector3d_set(&p1t, i,     j,     z1);
	  mat_vector3d_set(&p2t, i+1,   j,     z2);
	  mat_vector3d_set(&p3t, i+1,   j+1,   z3);
	  mat_vector3d_set(&p4t, i,     j+1,   z4);
	  mat_vector3d_set(&p1b, i,     j,     zf);
	  mat_vector3d_set(&p2b, i+1,   j,     zf);
	  mat_vector3d_set(&p3b, i+1,   j+1,   zf);
	  mat_vector3d_set(&p4b, i,     j+1,   zf);
	  // Create the Quad between 4 pixels
	  if (fabs(z3-z1) < fabs(z3-z2))
	    {
	      mat_write_stl_triangle(fp, &p1t, &p2t, &p3t);
	      mat_write_stl_triangle(fp, &p3t, &p4t, &p1t);
	    }
	  else
	    {
	      mat_write_stl_triangle(fp, &p2t, &p3t, &p4t);
	      mat_write_stl_triangle(fp, &p4t, &p1t, &p2t);
	    }
	  // create the left side if (i == 0)
	  if (i == 0)
	    {
	      mat_write_stl_quad(fp, &p4b, &p1b, &p1t, &p4t);
	    }
	  // create the right side if (i == (nx-1))
	  if (i == (nx-2))
	    {
	      mat_write_stl_quad(fp, &p2b, &p3b, &p3t, &p2t);
	    }
	  // create the bottom side if (j == 0)
	  if (j == 0)
	    {
	      mat_write_stl_quad(fp, &p1b, &p2b, &p2t, &p1t);
	    }
	  // create the top side if (j == (ny-1))
	  if (j == (ny-2))
	    {
	      mat_write_stl_quad(fp, &p3b, &p4b, &p4t, &p3t);
	    }
	}
    }
  // Create all triangles representing the bottom side of the image
  mat_vector3d_set(&p1b, nx/2, ny/2, zf); // fixed center
  for (i = 0; i < nx-2; i++) // bottom sector
    { // counter clockwise looking upwards
      mat_vector3d_set(&p2b, i+1, 0, zf);
      mat_vector3d_set(&p3b, i,   0, zf);
      mat_write_stl_triangle(fp, &p1b, &p2b, &p3b);
    }
  for (j = 0; j < ny-2; j++) // right  sector
    { // counter clockwise looking upwards
      mat_vector3d_set(&p2b, nx-1, j+1, zf);
      mat_vector3d_set(&p3b, nx-1, j,   zf);
      mat_write_stl_triangle(fp, &p1b, &p2b, &p3b);
    }
  for (i = 0; i < nx-2; i++) // top  sector
    { // counter clockwise looking upwards
      mat_vector3d_set(&p2b, i,   ny-1, zf);
      mat_vector3d_set(&p3b, i+1, ny-1, zf);
      mat_write_stl_triangle(fp, &p1b, &p2b, &p3b);
    }
  for (j = 0; j < ny-2; j++) // left  sector
    { // counter clockwise looking upwards
      mat_vector3d_set(&p2b, 0, j,   zf);
      mat_vector3d_set(&p3b, 0, j+1, zf);
      mat_write_stl_triangle(fp, &p1b, &p2b, &p3b);
    }
  fclose(fp);
  return CPL_ERROR_NONE;
}

cpl_error_code mat_image_store_column_stl(const char *fname, const cpl_image *img, int x, int y, int nx, int ny, double zscale)
{
  FILE          *fp;
  int            nfacets;
  int            i, j;
  double         imin, imax;
  int            rejected;
  double         zoffset;
  double         zf; // z value of the floor (1% below the smallest image intensity)
  mat_vector3d   p1, p2, p3, p4;

  // determine the number of triangles (facets)
  nfacets  = 2*nx*ny;     // top of each pixel consists of 2 triangles forming a square
  nfacets += 2*(nx+1)*ny; // pixel walls at left/right side of each pixel
  nfacets += 2*nx*(ny+1); // pixel walls at bottom/top side of each pixel
  nfacets += 2*nx + 2*ny; // triangles at the bottom of the 3-d mountain
  // neighbour pixels with the same height do not have a vertical wall between -> remove the 2 triangles
  imin = cpl_image_get(img, x, y, &rejected);
  imax = imin;
  for (i = 0; i < nx; i++)
    {
      for (j = 0; j < ny; j++)
	{
	  double v = cpl_image_get(img, x + i, y + j, &rejected);
	  imax = CPL_MAX(imax, v);
	  imin = CPL_MIN(imin, v);
	}
    }
  for (i = x; i < x + nx - 1; i++)
    {
      for (j = y; j < y + ny; j++)
	{
	  double vc = cpl_image_get(img, i,   j, &rejected);
	  double vn = cpl_image_get(img, i+1, j, &rejected);
	  if (vc == vn) nfacets -= 2; // same height -> no intermediate wall (left - right)
	}
    }
  for (i = x; i < x + nx; i++)
    {
      for (j = y; j < y + ny - 1; j++)
	{
	  double vc = cpl_image_get(img, i, j,   &rejected);
	  double vn = cpl_image_get(img, i, j+1, &rejected);
	  if (vc == vn) nfacets -= 2; // same height -> no intermediate wall (bottom - top)
	}
    }
  // check if the image contains only one value (horizontal plane)
  if (imin == imax)
    {
      cpl_msg_info(cpl_func, "Image is a horizontal plane, no STL file (%s) created", fname);
      return CPL_ERROR_UNSPECIFIED;
    }
  zoffset = imin;
  zf = -1.0;
  // Create a new STL file for a plain image view
  if ((fp = fopen(fname, "w")) == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot create plain STL file %s", fname);
      return CPL_ERROR_FILE_NOT_CREATED;
    }
  // write STL header
  mat_write_stl_header(fp, nfacets);
  // Create all squares representing the top of all pixels, unit normal (0, 0, 1)
  //  (i, j+1) ----- (i+1, j+1)
  //     |                |
  //     |                |
  //     |                |
  //     |                |
  //     |                |
  //  (i, j) -------- (i+1, j)
  // pij
  // p1 = (i, j)
  // p2 = (i+1, j)
  // p3 = (i+1, j+1)
  // p4 = (i, j+1)
  for (i = 0; i < nx; i++)
    {
      for (j = 0; j < ny; j++)
	{
	  double vc = cpl_image_get(img, x + i, y + j, &rejected);
	  double z = (vc - zoffset)/zscale;
	  mat_vector3d_set(&p1, i,   j,   z);
	  mat_vector3d_set(&p2, i,   j+1, z);
	  mat_vector3d_set(&p3, i+1, j+1, z);
	  mat_vector3d_set(&p4, i+1, j,   z);
	  mat_write_stl_quad(fp, &p1, &p2, &p3, &p4);
	}
    }
  // Create all squares representing the wall on the left/right side of the pixels
  for (i = 0; i < nx-1; i++)
    {
      for (j = 0; j < ny; j++)
	{
	  double vl = cpl_image_get(img, x + i,     y + j, &rejected);
	  double vr = cpl_image_get(img, x + i + 1, y + j, &rejected);
	  if (vl == vr) continue; // same intensity -> no wall
	  if (vl > vr)
	    { // transition from high to low intensity
	      double zt = (vl - zoffset)/zscale;
	      double zb = (vr - zoffset)/zscale;
	      mat_vector3d_set(&p1, i+1, j,   zb);
	      mat_vector3d_set(&p2, i+1, j+1, zb);
	      mat_vector3d_set(&p3, i+1, j+1, zt);
	      mat_vector3d_set(&p4, i+1, j,   zt);
	    }
	  else
	    { // transition from low to high intensity
	      double zt = (vr - zoffset)/zscale;
	      double zb = (vl - zoffset)/zscale;
	      mat_vector3d_set(&p1, i+1, j+1, zb);
	      mat_vector3d_set(&p2, i+1, j,   zb);
	      mat_vector3d_set(&p3, i+1, j,   zt);
	      mat_vector3d_set(&p4, i+1, j+1, zt);
	    }
	  mat_write_stl_quad(fp, &p1, &p2, &p3, &p4);
	}
    }
  // Create all squares representing the wall on the top/bottom side of the pixels
  for (i = 0; i < nx; i++)
    {
      for (j = 0; j < ny-1; j++)
	{
	  double vb = cpl_image_get(img, x + i, y + j,     &rejected);
	  double vt = cpl_image_get(img, x + i, y + j + 1, &rejected);
	  if (vb == vt) continue; // same intensity -> no wall
	  if (vb > vt)
	    { // transition from high to low intensity
	      double zt = (vb - zoffset)/zscale;
	      double zb = (vt - zoffset)/zscale;
	      mat_vector3d_set(&p1, i+1, j+1,   zb);
	      mat_vector3d_set(&p2, i,   j+1,   zb);
	      mat_vector3d_set(&p3, i,   j+1,   zt);
	      mat_vector3d_set(&p4, i+1, j+1,   zt);
	    }
	  else
	    { // transition from low to high intensity
	      double zt = (vt - zoffset)/zscale;
	      double zb = (vb - zoffset)/zscale;
	      mat_vector3d_set(&p1, i,   j+1,   zb);
	      mat_vector3d_set(&p2, i+1, j+1,   zb);
	      mat_vector3d_set(&p3, i+1, j+1,   zt);
	      mat_vector3d_set(&p4, i,   j+1,   zt);
	    }
	  mat_write_stl_quad(fp, &p1, &p2, &p3, &p4);
	}
    }
  // Create all squares representing the left border of the image
  for (j = 0; j < ny; j++)
    {
      double v  = cpl_image_get(img, x, y + j, &rejected);
      double zt = (v - zoffset)/zscale;
      mat_vector3d_set(&p1, 0, j+1,   zf);
      mat_vector3d_set(&p2, 0, j,     zf);
      mat_vector3d_set(&p3, 0, j,     zt);
      mat_vector3d_set(&p4, 0, j+1,   zt);
      mat_write_stl_quad(fp, &p1, &p2, &p3, &p4);
    }
  // Create all squares representing the right border of the image
  for (j = 0; j < ny; j++)
    {
      double v  = cpl_image_get(img, x + nx - 1, y + j, &rejected);
      double zt = (v - zoffset)/zscale;
      mat_vector3d_set(&p1, nx, j,   zf);
      mat_vector3d_set(&p2, nx, j+1, zf);
      mat_vector3d_set(&p3, nx, j+1, zt);
      mat_vector3d_set(&p4, nx, j,   zt);
      mat_write_stl_quad(fp, &p1, &p2, &p3, &p4);
    }
  // Create all squares representing the bottom border of the image
  for (i = 0; i < nx; i++)
    {
      double v  = cpl_image_get(img, x + i, y, &rejected);
      double zt = (v - zoffset)/zscale;
      mat_vector3d_set(&p1, i,   0,   zf);
      mat_vector3d_set(&p2, i+1, 0,   zf);
      mat_vector3d_set(&p3, i+1, 0,   zt);
      mat_vector3d_set(&p4, i,   0,   zt);
      mat_write_stl_quad(fp, &p1, &p2, &p3, &p4);
    }
  // Create all squares representing the top border of the image
  for (i = 0; i < nx; i++)
    {
      double v  = cpl_image_get(img, x + i, y + ny - 1, &rejected);
      double zt = (v - zoffset)/zscale;
      mat_vector3d_set(&p1, i+1, ny,   zf);
      mat_vector3d_set(&p2, i,   ny,   zf);
      mat_vector3d_set(&p3, i,   ny,   zt);
      mat_vector3d_set(&p4, i+1, ny,   zt);
      mat_write_stl_quad(fp, &p1, &p2, &p3, &p4);
    }
  // Create all triangles representing the bottom side of the image
  mat_vector3d_set(&p1, nx/2, ny/2, zf); // fixed center
  for (i = 0; i < nx-1; i++) // bottom sector
    { // counter clockwise looking upwards
      mat_vector3d_set(&p2, i+1, 0, zf);
      mat_vector3d_set(&p3, i,   0, zf);
      mat_write_stl_triangle(fp, &p1, &p2, &p3);
    }
  for (j = 0; j < ny-1; j++) // right  sector
    { // counter clockwise looking upwards
      mat_vector3d_set(&p2, nx, j+1, zf);
      mat_vector3d_set(&p3, nx, j,   zf);
      mat_write_stl_triangle(fp, &p1, &p2, &p3);
    }
  for (i = 0; i < nx-1; i++) // top  sector
    { // counter clockwise looking upwards
      mat_vector3d_set(&p2, i,   ny, zf);
      mat_vector3d_set(&p3, i+1, ny, zf);
      mat_write_stl_triangle(fp, &p1, &p2, &p3);
    }
  for (j = 0; j < ny-1; j++) // left  sector
    { // counter clockwise looking upwards
      mat_vector3d_set(&p2, 0, j,   zf);
      mat_vector3d_set(&p3, 0, j+1, zf);
      mat_write_stl_triangle(fp, &p1, &p2, &p3);
    }
  fclose(fp);
  return CPL_ERROR_NONE;
}

cpl_error_code mat_image_store_pyramid_stl(const char *fname, const cpl_image *img, int x, int y, int nx, int ny, double zscale)
{
  FILE          *fp;
  int            nfacets;
  int            i, j;
  double         imin, imax;
  int            rejected;
  double         zoffset, zf;
  double         v00, v10, v20, v01, v11, v21, v02, v12, v22;
  mat_vector3d   p0, p1t, p2t, p3t, p4t, p1b, p2b, p3b, p4b;

  // determine the number of triangles (facets)
  nfacets  = 4*nx*ny;     // top of each pixel consists of 4 triangles forming a pyramid
  nfacets += 4*(ny+1);    // pixel walls at left/right side of the image
  nfacets += 4*(nx+1);    // pixel walls at bottom/top side of the image
  nfacets += 2*nx + 2*ny; // triangles at the bottom of the 3-d mountain
  imin = cpl_image_get(img, x, y, &rejected);
  imax = imin;
  for (i = 0; i < nx; i++)
    {
      for (j = 0; j < ny; j++)
	{
	  double v = cpl_image_get(img, x + i, y + j, &rejected);
	  imin = CPL_MIN(imin, v);
	  imax = CPL_MAX(imax, v);
	}
    }
  // check if the image contains only one value (horizontal plane)
  if (imin == imax)
    {
      cpl_msg_info(cpl_func, "Image is a horizontal plane, no STL file (%s) created", fname);
      return CPL_ERROR_UNSPECIFIED;
    }
  zoffset = imin;
  zf = -1.0;
  // Create a new STL file for a plain image view
  if ((fp = fopen(fname, "w")) == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot create plain STL file %s", fname);
      return CPL_ERROR_FILE_NOT_CREATED;
    }
  // write STL header
  mat_write_stl_header(fp, nfacets);
  // Assignment of the image intensities, calculated z height and points (active pixel is the central pixel [11] of the diagram)
  //  +----------+----------+----------+
  //  |          |          |          |
  //  |          |          |          |
  //  |    v02   |    v12   |   v22    |
  //  |          |          |          |
  //  |          |          |          |
  //  +----------+----------+----------+
  //  |          |          |          |
  //  |          |          |          |
  //  |    v01   |    v11   |   v21    |
  //  |          |          |          |
  //  |          |          |          |
  //  +----------+----------+----------+
  //  |          |          |          |
  //  |          |          |          |
  //  |    v00   |    v10   |   v20    |
  //  |          |          |          |
  //  |          |          |          |
  //  +----------+----------+----------+
  // image intensities -> z height
  //  z0 = v11;
  //  z1 = 1/4*(v00 + v10 + v01 + v11)
  //  z2 = 1/4*(v10 + v20 + v11 + v21)
  //  z3 = 1/4*(v11 + v21 + v12 + v22)
  //  z4 = 1/4*(v01 + v11 + v02 + v12)
  // corners of the pyramid and the footprint on the floor
  //  p0 = central peak of a pixel
  //  p1t, p2t, p3t, p4t = position of the pyramid base (p1t lower left, p2t, lower right, ... counter clockwise)
  //  p1b, p2b, p3b, p4b = position of the pyramid base with a height of the image floor (zf)
  // triangles (pyramid)
  //  T1 = {p0, p1t, p2t}
  //  T2 = {p0, p2t, p3t}
  //  T3 = {p0, p3t, p4t}
  //  T4 = {p0, p4t, p1t}
  // quad bottom side (optional)
  //  Q1 = {p1b, p2b, p2t, p1t}
  // quad right side (optional)
  //  Q2 = {p1b, p3b, p3t, p2t}
  // quad top side (optional)
  //  Q3 = {p3b, p4b, p4t, p3t}
  // quad left side (optional)
  //  Q4 = {p4b, p1b, p1t, p4t}
  //
  for (i = 0; i < nx; i++)
    {
      for (j = 0; j < ny; j++)
	{
	  double         z0  = 0.0, z1  = 0.0, z2  = 0.0, z3  = 0.0, z4  = 0.0;
	  double         z0c = 0.0, z1c = 0.0, z2c = 0.0, z3c = 0.0, z4c = 0.0;
	  if (i != 0)
	    { // there are pixels left from the current pixel, x+i-1 is save to use
	      if (j != 0)
		{ // there are pixels below the current pixel, y+j-1 is save
		  v00 = cpl_image_get(img, x+i-1, y+j-1, &rejected);
		  z1 += v00; z1c += 1.0; // use this intensity for p1t
		}
	      v01 = cpl_image_get(img, x+i-1, y+j, &rejected);
	      z1 += v01; z1c += 1.0; // use this intensity for p1t
	      z4 += v01; z4c += 1.0; // use this intensity for p4t
	      if (j != (ny-1))
		{ // there are pixels above the current pixel, y+j+1 is save
		  v02 = cpl_image_get(img, x+i-1, y+j+1, &rejected);
		  z4 += v02; z4c += 1.0; // use this intensity for p4t
		}
	    }
	  // x+i is save
	  if (j != 0)
	    { // there are pixels below the current pixel, y+j-1 is save
	      v10 = cpl_image_get(img, x+i,   y+j-1, &rejected);
	      z1 += v10; z1c += 1.0; // use this intensity for p1t
	      z2 += v10; z2c += 1.0; // use this intensity for p2t
	    }
	  v11 = cpl_image_get(img, x+i,   y+j,   &rejected);
	  z0 += v11; z0c += 1.0; // use this intensity for p0
	  z1 += v11; z1c += 1.0; // use this intensity for p1t
	  z2 += v11; z2c += 1.0; // use this intensity for p2t
	  z3 += v11; z3c += 1.0; // use this intensity for p3t
	  z4 += v11; z4c += 1.0; // use this intensity for p4t
	  if (j != (ny-1))
	    { // there are pixels above the current pixel, y+j+1 is save
	      v12 = cpl_image_get(img, x+i,   y+j+1, &rejected);
	      z3 += v12; z3c += 1.0; // use this intensity for p3t
	      z4 += v12; z4c += 1.0; // use this intensity for p4t
	    }
	  if (i != (nx-1))
	    { // there are pixels right from the current pixel, x+i+1 is save to use
	      if (j != 0)
		{ // there are pixels below the current pixel, y+j-1 is save
		  v20 = cpl_image_get(img, x+i+1, y+j-1, &rejected);
		  z2 += v20; z2c += 1.0; // use this intensity for p2t
		}
	      v21 = cpl_image_get(img, x+i+1, y+j,   &rejected);
	      z2 += v21; z2c += 1.0; // use this intensity for p2t
	      z3 += v21; z3c += 1.0; // use this intensity for p3t
	      if (j != (ny-1))
		{ // there are pixels above the current pixel, y+j+1 is save
		  v22 = cpl_image_get(img, x+i+1, y+j+1, &rejected);
		  z3 += v22; z3c += 1.0; // use this intensity for p3t
		}
	    }
	  z0 = (z0/z0c - zoffset)/zscale;
	  z1 = (z1/z1c - zoffset)/zscale;
	  z2 = (z2/z2c - zoffset)/zscale;
	  z3 = (z3/z3c - zoffset)/zscale;
	  z4 = (z4/z4c - zoffset)/zscale;
	  // create the 3D points
	  mat_vector3d_set(&p0,  i+0.5, j+0.5, z0);
	  mat_vector3d_set(&p1t, i,     j,     z1);
	  mat_vector3d_set(&p2t, i+1,   j,     z2);
	  mat_vector3d_set(&p3t, i+1,   j+1,   z3);
	  mat_vector3d_set(&p4t, i,     j+1,   z4);
	  mat_vector3d_set(&p1b, i,     j,     zf);
	  mat_vector3d_set(&p2b, i+1,   j,     zf);
	  mat_vector3d_set(&p3b, i+1,   j+1,   zf);
	  mat_vector3d_set(&p4b, i,     j+1,   zf);
	  // create the pyramid
	  mat_write_stl_triangle(fp, &p0, &p1t, &p2t);
	  mat_write_stl_triangle(fp, &p0, &p2t, &p3t);
	  mat_write_stl_triangle(fp, &p0, &p3t, &p4t);
	  mat_write_stl_triangle(fp, &p0, &p4t, &p1t);
	  // create the left side if (i == 0)
	  if (i == 0)
	    {
	      mat_write_stl_quad(fp, &p4b, &p1b, &p1t, &p4t);
	    }
	  // create the right side if (i == (nx-1))
	  if (i == (nx-1))
	    {
	      mat_write_stl_quad(fp, &p2b, &p3b, &p3t, &p2t);
	    }
	  // create the bottom side if (j == 0)
	  if (j == 0)
	    {
	      mat_write_stl_quad(fp, &p1b, &p2b, &p2t, &p1t);
	    }
	  // create the top side if (j == (ny-1))
	  if (j == (ny-1))
	    {
	      mat_write_stl_quad(fp, &p3b, &p4b, &p4t, &p3t);
	    }
	}
    }
  // Create all triangles representing the bottom side of the image
  mat_vector3d_set(&p1b, nx/2, ny/2, zf); // fixed center
  for (i = 0; i < nx-1; i++) // bottom sector
    { // counter clockwise looking upwards
      mat_vector3d_set(&p2b, i+1, 0, zf);
      mat_vector3d_set(&p3b, i,   0, zf);
      mat_write_stl_triangle(fp, &p1b, &p2b, &p3b);
    }
  for (j = 0; j < ny-1; j++) // right  sector
    { // counter clockwise looking upwards
      mat_vector3d_set(&p2b, nx, j+1, zf);
      mat_vector3d_set(&p3b, nx, j,   zf);
      mat_write_stl_triangle(fp, &p1b, &p2b, &p3b);
    }
  for (i = 0; i < nx-1; i++) // top  sector
    { // counter clockwise looking upwards
      mat_vector3d_set(&p2b, i,   ny, zf);
      mat_vector3d_set(&p3b, i+1, ny, zf);
      mat_write_stl_triangle(fp, &p1b, &p2b, &p3b);
    }
  for (j = 0; j < ny-1; j++) // left  sector
    { // counter clockwise looking upwards
      mat_vector3d_set(&p2b, 0, j,   zf);
      mat_vector3d_set(&p3b, 0, j+1, zf);
      mat_write_stl_triangle(fp, &p1b, &p2b, &p3b);
    }
  fclose(fp);
  return CPL_ERROR_NONE;
}

