/* $Id: mat_imagingdetector.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_imagingdetector.c $
 */

#include <stdlib.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>

#include "mat_error.h"
#include "mat_utils.h"
#include "mat_frame.h"
#include "mat_imagingdetector.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
                                   Functions
 -----------------------------------------------------------------------------*/
/**
   @ingroup imgdet
   @brief Creates a new mat_imagingdetector data structure according to a property list and the number of sub-windows.
   @param plist     The optional property list containing important keywords.
   @param nbregion  The number of sub-windows.
   @returns A new mat_imagingdetector data structure of NULL.

   A new data structure is allocated and filled with some keyword values
   from the (optional) property list. The nbregion parameter determines
   the number of sub-windows of the new data structure. All values are
   initialized with default values except the keywords provided in the
   property list.
 */
mat_imagingdetector *mat_imagingdetector_new(cpl_propertylist *plist, int nbregion)
{
  mat_imagingdetector *imgdet = NULL;
  int                  r;

  imgdet = (mat_imagingdetector*)cpl_calloc(1, sizeof(mat_imagingdetector));
  if (imgdet == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_imagingdetector");
      return NULL;
    }
  if (plist != NULL)
    {
      /* try to copy the value from a few keywords from the propertylist */
      //imgdet->origin          = mat_propertylist_copy_string(plist, "ORIGIN");
      imgdet->instrument      = mat_propertylist_copy_string(plist, "INSTRUME");
      imgdet->dateobsmjd      = cpl_propertylist_get_double(plist,"MJD-OBS");
      imgdet->dateobs         = mat_propertylist_copy_string(plist, "DATE-OBS");
      //imgdet->date            = mat_propertylist_copy_string(plist, "DATE");
      imgdet->dcsdictionaryid = mat_propertylist_copy_string(plist, "ESO DET DID");
      imgdet->dcsid           = mat_propertylist_copy_string(plist, "ESO DET ID");
    }
  else
    {
      /* no propertylist given, at least allocate the memory for the strings */
      //imgdet->origin          = cpl_calloc(SIZE_MAX_KEYWORDS, sizeof(char));
      imgdet->instrument      = cpl_calloc(SIZE_MAX_KEYWORDS, sizeof(char));
      imgdet->dateobsmjd      = 0.0;
      imgdet->dateobs         = cpl_calloc(SIZE_MAX_KEYWORDS, sizeof(char));
      //imgdet->date            = cpl_calloc(SIZE_MAX_KEYWORDS, sizeof(char));
      imgdet->dcsdictionaryid = cpl_calloc(SIZE_MAX_KEYWORDS, sizeof(char));
      imgdet->dcsid           = cpl_calloc(SIZE_MAX_KEYWORDS, sizeof(char));
    }
  imgdet->nbdetector      = 1;
  imgdet->nbregion        = nbregion;
  imgdet->nbtel           = 1;
  /* create the array of regions */
  imgdet->list_region = cpl_calloc(imgdet->nbregion, sizeof(mat_region*));
  if (imgdet->list_region == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for imgdet->list_region");
      mat_imagingdetector_delete(imgdet);
      return NULL;
    }
  /* create all regions and fill them with provisional values */
  for (r = 0; r < imgdet->nbregion; r++)
    {
      mat_region *region = mat_region_new(r + 1, MAT_MAXTEL, 1 + r, 1, 1, 1);
      if (region == NULL)
	{
	  cpl_msg_error(cpl_func,"could not allocate memory for imgdet->list_region[%d]", r);
	  mat_imagingdetector_delete(imgdet);
	  return NULL;
	}
      imgdet->list_region[r] = region;
      region->numregion = r + 1;
    }
  return imgdet;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup imgdet
  @brief delete structure mat_imagingdetector
  @param imgdet           current structure
  @return error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mat_imagingdetector_delete(mat_imagingdetector *imgdet) {
  int i=0;

  // Check input parameters
  mat_assert_value((imgdet!=NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no mat_imagingdetector (imgdet) argument given");
  // mat_free(imgdet->origin);
  mat_free(imgdet->instrument);
  mat_free(imgdet->dateobs);
  // mat_free(imgdet->date);
  mat_free(imgdet->dcsdictionaryid);
  mat_free(imgdet->dcsid);
  if (imgdet->list_region != NULL)
    {
      for (i=0;i<imgdet->nbregion;i++)
	{
	  if (imgdet->list_region[i] != NULL)
	    {
	      mat_region_delete(imgdet->list_region[i]);
	      imgdet->list_region[i] = NULL;
	    }
	}
      mat_free(imgdet->list_region);
      imgdet->nbregion = 0;
    }
  cpl_free(imgdet);
  return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup imgdet
  @brief Duplicate a mat_imagingdetector structure
  @param imgdet           mat_imagingdetector structure to duplicate
  @return mat_imagingdetector
 */
/*-----------------------------------------------------------------------------*/
mat_imagingdetector * mat_imagingdetector_duplicate(const mat_imagingdetector * imgdet)
{
  mat_imagingdetector * imagingDet=NULL;
  int i=0;
  cpl_errorstate prestate = cpl_errorstate_get();

  if(imgdet == NULL){
    return NULL;
  }

// Pointers allocation
  imagingDet=cpl_calloc(1, sizeof(mat_imagingdetector));
  if (imagingDet ==NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for imagingDet");
    return NULL;
  }

  //imagingDet->origin=cpl_strdup(imgdet->origin);
  imagingDet->instrument=cpl_strdup(imgdet->instrument);
  imagingDet->dateobs=cpl_strdup(imgdet->dateobs);
  //imagingDet->date=cpl_strdup(imgdet->date);
  imagingDet->dcsdictionaryid=cpl_strdup(imgdet->dcsdictionaryid);
  imagingDet->dcsid=cpl_strdup(imgdet->dcsid);
// Set data from IMAGING_DETECTOR header
  /* strcpy(imagingDet->origin, imgdet->origin); */
  /* strcpy(imagingDet->instrument, imgdet->instrument); */
  imagingDet->dateobsmjd = imgdet->dateobsmjd;
  /* strcpy(imagingDet->dateobs, imgdet->dateobs); */
  /* strcpy(imagingDet->date, imgdet->date); */
  /* strcpy(imagingDet->dcsdictionaryid, imgdet->dcsdictionaryid); */
  /* strcpy(imagingDet->dcsid, imgdet->dcsid); */
  imagingDet->nbdetector = imgdet->nbdetector;
  imagingDet->nbregion = imgdet->nbregion;
  imagingDet->nbtel = imgdet->nbtel;
// Set data from IMAGING_DETECTOR table
// Copy regions
  imagingDet->list_region=cpl_calloc(imagingDet->nbregion, sizeof(mat_region *));
  for(i=0;i<imagingDet->nbregion;i++) {
    imagingDet->list_region[i]= 
         mat_region_duplicate(imgdet->list_region[i], imgdet->nbtel);
  }
  
  /* Error handling */
  if (!cpl_errorstate_is_equal(prestate)) { 
    cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
    mat_imagingdetector_delete(imagingDet);
    return NULL;
  }

  return imagingDet;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup imgdet
  @brief Load raw data in mat_imagingdetector structure
  @param plist           current cpl_propertylist
  @param table           current cpl_table
  @return mat_imagingdetector
 */
/*-----------------------------------------------------------------------------*/
mat_imagingdetector *mat_imagingdetector_from_table(cpl_propertylist * plist,
						    cpl_table *table)
{
  int i=0;
  int j=0;
  mat_imagingdetector * imagingDet=NULL;
  cpl_errorstate prestate = cpl_errorstate_get();

  if(plist == NULL){
    return NULL;
  } 


  if(table == NULL){
    return NULL;
  } 

// Pointers allocation
  imagingDet=cpl_calloc(1, sizeof(mat_imagingdetector));
  if (imagingDet == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for mat_imagingdetector");
    return NULL;
  }
  // Set data from IMAGING_DETECTOR header
  //imagingDet->origin = mat_propertylist_copy_string(plist, "ORIGIN");
  imagingDet->instrument = mat_propertylist_copy_string(plist, "INSTRUME");
  imagingDet->dateobsmjd=cpl_propertylist_get_double(plist,"MJD-OBS");
  imagingDet->dateobs = mat_propertylist_copy_string(plist, "DATE-OBS");
  //imagingDet->date = mat_propertylist_copy_string(plist, "DATE");
  imagingDet->dcsdictionaryid = mat_propertylist_copy_string(plist, "ESO DET DID");
  imagingDet->dcsid = mat_propertylist_copy_string(plist, "ESO DET ID");
  imagingDet->nbdetector=cpl_propertylist_get_int(plist,"NDETECT");
  imagingDet->nbregion=cpl_propertylist_get_int(plist,"NREGION");
  imagingDet->nbtel=cpl_propertylist_get_int(plist,"MAXTEL");
// Set data from IMAGING_DETECTOR table
// Copy regions
  imagingDet->list_region=cpl_calloc(imagingDet->nbregion, sizeof(mat_region*));
  if (imagingDet->list_region == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for imagingDet->list_region");
      mat_imagingdetector_delete(imagingDet);
      return NULL;
    }
  
  for(i=0;i<imagingDet->nbregion;i++) {
    imagingDet->list_region[i]=cpl_calloc(1, sizeof(mat_region)); 
    if (imagingDet->list_region[i] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for imagingDet->list_region[i]");
      mat_imagingdetector_delete(imagingDet);
      return NULL;
    }
    imagingDet->list_region[i]->numregion=
      cpl_table_get_int(table,"REGION",i,NULL);
    imagingDet->list_region[i]->regionname=cpl_calloc(10, sizeof(char));
    strncpy(imagingDet->list_region[i]->regionname,
	   (char *)cpl_table_get_string(table,"REGNAME",i),10);
    imagingDet->list_region[i]->numdetector=
      (int)cpl_table_get_int(table,"DETECTOR",i,NULL);
    imagingDet->list_region[i]->numbeam=
      cpl_calloc(imagingDet->nbtel,sizeof(int));
    /* heininger:
       modification for AMBER raw data:
       if MAXTEL = 1 -> PORTS = I and not an array!
    */
    if (cpl_table_get_column_depth(table, "PORTS") == 0)
      {
	imagingDet->list_region[i]->numbeam[0] = cpl_table_get_int(table,"PORTS",i, NULL);
      }
    else
      {
	for(j=0;j<imagingDet->nbtel;j++)
	  {
	    imagingDet->list_region[i]->numbeam[j]=
	      cpl_array_get_int(cpl_table_get_array(table,"PORTS",i),j,NULL);
	  }
      }
    imagingDet->list_region[i]->correlation=
      cpl_table_get_int(table,"CORRELATION",i,NULL);
    for(j=0;j<2;j++)
      {
	imagingDet->list_region[i]->corner[j] =
	  cpl_array_get_int(cpl_table_get_array(table,"CORNER",i),j,NULL);
	if (imagingDet->list_region[i]->corner[j] == 0)
	  { // heininger: correct for an error in some IMAGING_DETECTOR binary tables
	    imagingDet->list_region[i]->corner[j] = 1;
	  }
      }
    /* heininger:
       modification for AMBER raw data:
       the GAIN column is E for AMBER and D for MATISSE
    */
    if (cpl_table_get_column_type(table, "GAIN") == CPL_TYPE_FLOAT)
      {
	imagingDet->list_region[i]->gain = (double)cpl_table_get_float(table, "GAIN", i, NULL);
      }
    else
      {
	imagingDet->list_region[i]->gain = cpl_table_get_double(table, "GAIN", i, NULL);
      }
    for(j=0;j<2;j++) {
      imagingDet->list_region[i]->crval[j]=
	cpl_array_get_double(cpl_table_get_array(table,"CRVAL",i),j,NULL);
    }
    for(j=0;j<2;j++) {
      imagingDet->list_region[i]->crpix[j]=
	cpl_array_get_float(cpl_table_get_array(table,"CRPIX",i),j,NULL);
    }
    /* heininger:
       modification for AMBER raw data:
       this column is simply 8A and not 16A with (8,2)
    */
    if (cpl_table_get_column_depth(table, "CTYPE") == 0)
      {
	for(j=0;j<2;j++) {
	  imagingDet->list_region[i]->ctype[j]=cpl_strdup(cpl_table_get_string(table,"CTYPE",i));
	}
      }
    else
      {
	for(j=0;j<2;j++) {
	  imagingDet->list_region[i]->ctype[j]=
	    cpl_strdup(cpl_array_get_string(cpl_table_get_array(table,"CTYPE",i),j));

	  /* imagingDet->list_region[i]->ctype[j]=cpl_calloc(10, sizeof(char)); */
	  /* strncpy(imagingDet->list_region[i]->ctype[j], */
	  /* 	  cpl_array_get_string(cpl_table_get_array(table,"CTYPE",i),j),10); */
	}
      }
    for(j=0;j<2;j++) {
      imagingDet->list_region[i]->naxis[j]=
	cpl_array_get_int(cpl_table_get_array(table,"NAXIS",i),j,NULL);
    }
  }

/* Error handling */
  if (!cpl_errorstate_is_equal(prestate)) { 
    cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
    // TODO MHE: remove the following two comment tokens
    //cpl_free(imagingDet);
    //return NULL;
  }
  return imagingDet;
}

/**
   @ingroup imgdet
   @brief Stores a mat_imagingdetector data structure into a property list and a table.
   @param imgdet   The IMAGING_DETECTOR extension as mat_imagingdetector data structure.
   @param plist    An optional property list for storing the keywords of the extension.

   This function creates a cpl_table data structure and fills it with the sub-window setup
   given as a mat_imagingdetector data structure. If a property list is given, the keywords
   of the mat_imagingdetector data structure are stored in this list.
*/
cpl_table *mat_imagingdetector_to_table(mat_imagingdetector *imgdet,
					cpl_propertylist *plist)
{
  cpl_table  *table;
  int         i, j;
  cpl_array  *array = NULL;

  if (imgdet == NULL)
    {
      return NULL;
    }
  // put all properties of a IMAGING_DETECTOR binary table into the property list (if one is given)
  if (plist != NULL)
    {
      cpl_propertylist_append_string(plist,"EXTNAME","IMAGING_DETECTOR");
      //cpl_propertylist_append_string(plist,"ORIGIN",imgdet->origin);
      cpl_propertylist_append_string(plist,"INSTRUME",imgdet->instrument);
      cpl_propertylist_append_double(plist,"MJD-OBS",imgdet->dateobsmjd);
      cpl_propertylist_append_string(plist,"DATE-OBS",imgdet->dateobs);
      //cpl_propertylist_append_string(plist,"DATE",imgdet->date);
      cpl_propertylist_append_string(plist,"ESO DET DID",imgdet->dcsdictionaryid);
      cpl_propertylist_append_string(plist,"ESO DET ID",imgdet->dcsid);
      cpl_propertylist_append_int(plist,"NDETECT",imgdet->nbdetector);
      cpl_propertylist_append_int(plist,"NREGION",imgdet->nbregion);
      cpl_propertylist_append_int(plist,"MAXTEL",imgdet->nbtel);
    }
  // Create an IMAGING_DETECTOR table
  table= cpl_table_new(imgdet->nbregion);
  // Add all columns to the table
  cpl_table_new_column(table, "REGION", CPL_TYPE_INT);
  cpl_table_new_column(table, "DETECTOR", CPL_TYPE_INT);
  cpl_table_new_column_array(table, "PORTS", CPL_TYPE_INT, imgdet->nbtel);
  cpl_table_new_column(table, "CORRELATION", CPL_TYPE_INT);
  cpl_table_new_column(table, "REGNAME", CPL_TYPE_STRING);
  cpl_table_new_column_array(table, "CORNER", CPL_TYPE_INT, 2);
  cpl_table_new_column(table, "GAIN", CPL_TYPE_DOUBLE);
  cpl_table_new_column_array(table, "NAXIS", CPL_TYPE_INT, 2);
  cpl_table_new_column_array(table, "CRVAL", CPL_TYPE_DOUBLE, 2);
  cpl_table_new_column_array(table, "CRPIX", CPL_TYPE_FLOAT, 2);
  cpl_table_new_column_array(table, "CTYPE", CPL_TYPE_STRING, 2);
  // Set data in the table
  for(i = 0; i < imgdet->nbregion; i++)
    {
      cpl_table_set_int(table,"REGION",i,imgdet->list_region[i]->numregion);
      cpl_table_set_int(table,"DETECTOR",i,
			imgdet->list_region[i]->numdetector);
      array= cpl_array_new(imgdet->nbtel, CPL_TYPE_INT);
      cpl_table_set_array(table,"PORTS",i, array);
      for(j = 0; j < imgdet->nbtel; j++)
	{
	  cpl_array_set_int((cpl_array *)cpl_table_get_array(table,"PORTS",i), j,
			    imgdet->list_region[i]->numbeam[j]);
	}
      cpl_array_delete(array);
      cpl_table_set_int(table,"CORRELATION",i,
			imgdet->list_region[i]->correlation);
      cpl_table_set_string(table,"REGNAME",i,
			   imgdet->list_region[i]->regionname);
      array= cpl_array_new(2, CPL_TYPE_INT);
      cpl_table_set_array(table,"CORNER",i, array);
      cpl_array_delete(array);
      for(j = 0; j < 2; j++)
	{
	  cpl_array_set_int((cpl_array *)cpl_table_get_array(table,"CORNER",i),j,
			    imgdet->list_region[i]->corner[j]);
	}
      cpl_table_set_double(table,"GAIN",i,imgdet->list_region[i]->gain);
      array= cpl_array_new(2, CPL_TYPE_INT);
      cpl_table_set_array(table,"NAXIS",i, array);
      cpl_array_delete(array);
      for(j = 0; j < 2; j++)
	{
	  cpl_array_set_int((cpl_array *)cpl_table_get_array(table,"NAXIS",i),
			    j, imgdet->list_region[i]->naxis[j]);
	}
      array= cpl_array_new(2, CPL_TYPE_DOUBLE);
      cpl_table_set_array(table,"CRVAL",i, array);
      cpl_array_delete(array);
      for(j = 0; j < 2; j++)
	{
	  cpl_array_set_double((cpl_array *)cpl_table_get_array(table,"CRVAL",i),
			       j, imgdet->list_region[i]->crval[j]);
	}
      array= cpl_array_new(2, CPL_TYPE_FLOAT);
      cpl_table_set_array(table,"CRPIX",i, array);
      cpl_array_delete(array);
      for(j = 0; j < 2; j++)
	{
	  cpl_array_set_float((cpl_array *)cpl_table_get_array(table,"CRPIX",i),
			      j, imgdet->list_region[i]->crpix[j]);
	}
      array= cpl_array_new(2, CPL_TYPE_STRING);
      cpl_table_set_array(table,"CTYPE",i, array);
      cpl_array_delete(array);
      for(j = 0; j < 2; j++)
	{
	 /* cpl_array_set_string((cpl_array *)cpl_table_get_array(table,"CTYPE",i), */
         /*                    j, imgdet->list_region[i]->ctype[j]); */
	}
    }
  return table;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup imgdet
  @brief Save a mat_imagingdetector structure in a FITS file
  @param imgdet          mat_imagingdetector structure to save
  @param plist           primary header of the FITS file
  @param filename        FITS file name
  @return cpl_error_code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_imagingdetector_save(mat_imagingdetector *imgdet,
                                        cpl_propertylist *plist,
                                        const char* filename)
{

  int i=0;
  char *keyExtname=NULL;
  cpl_table *table=NULL;
  cpl_propertylist *keywords=NULL;
  cpl_frame *frame=NULL;
  cpl_propertylist *plist_tmp=NULL;
  FILE *fp=NULL;

  // Check input parameters
  mat_assert_value((filename!=NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_UNSPECIFIED,"no filename given");
  if (imgdet == NULL) {
    return CPL_ERROR_UNSPECIFIED;
  }


  frame = cpl_frame_new();
  cpl_frame_set_filename(frame, filename);

// Return an error if a table IMAGING_DETECTOR is already there
  fp=fopen(filename,"r");
  if (fp!=NULL) {
    int nbExtent=0;
    fclose(fp);
    nbExtent=cpl_frame_get_nextensions(frame);
    for(i=0;i<nbExtent;i++) {
      plist_tmp=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);

      keyExtname=(char *)cpl_propertylist_get_string(plist_tmp,"EXTNAME");
      if (keyExtname != NULL) {
	if (!strcmp(keyExtname,"IMAGING_DETECTOR")) {
	  cpl_propertylist_delete(plist_tmp);
	  cpl_frame_delete(frame);
	  
	  return CPL_ERROR_UNSPECIFIED;
	}
      }
      cpl_propertylist_delete(plist_tmp);
    }
  }
  //cpl_msg_info(cpl_func,"imagingdetector save, keywords");
// Create the IMAGING_DETECTOR keywords
  keywords= cpl_propertylist_new();
  table = mat_imagingdetector_to_table(imgdet, keywords);

  //cpl_msg_info(cpl_func,"imagingdetector save, table");

  //cpl_msg_info(cpl_func,"imagingdetector save, table remplie");
  fp=fopen(filename,"r");
  if (fp == NULL) {
    cpl_propertylist_erase(plist,"RADECSYS");
    cpl_table_save(table, plist, keywords, filename, CPL_IO_CREATE);
  } else {
    fclose(fp);
    cpl_table_save(table, NULL, keywords, filename, CPL_IO_EXTEND);			
  }



// Append the table to the FITS file if it exists
/*  error_code= cpl_table_save(table, NULL, keywords, filename, CPL_IO_EXTEND);
  if(error_code == CPL_ERROR_FILE_NOT_FOUND) {
    cpl_table_save(table, plist, keywords, filename, CPL_IO_CREATE);
    cpl_errorstate_set(prestate);
    }*/

// Free memory
  if (frame != NULL) cpl_frame_delete(frame);
  if (table != NULL) cpl_table_delete(table);
  if (keywords != NULL) cpl_propertylist_delete(keywords);


  return cpl_error_get_code();
}

