/* $Id: mat_imextended.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_imextended.c $
 */

#include <stdlib.h>
#include <stdio.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>

#include "mat_error.h"
#include "mat_utils.h"
#include "mat_imextended.h"
#include "mat_frame.h"
#include "mat_utils.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

/**
   @ingroup ime
   @brief Creates an extended detector monitoring map data structure assocciated with a detector.
   @param det This structure describes the detector of the extended detector monitoring map.
   @returns The new extended detector monitoring map of NULL on error.

   This function creates an extended detector monitoring map data structure using
   the detector specification as a template. The number of detector channels, and
   therefore the size of the detector channel specific members (channelgain,
   channeloffset and channelnoise), is determined by the det parameter.
 */
mat_imextended *mat_imextended_new(mat_detector *det)
{
  mat_imextended *ime = NULL;

  mat_assert_value((det != NULL), CPL_ERROR_NULL_INPUT, NULL, "no mat_detector (det) argument given");
  ime = (mat_imextended *)cpl_calloc(1, sizeof(mat_imextended));
  if (ime == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_imextended");
      return NULL;
    }
  ime->det = mat_detector_duplicate(det);
  if (ime->det == NULL)
    {
      cpl_msg_error(cpl_func, "could not duplicate mat_detector");
      mat_imextended_delete(ime);
      return NULL;
    }

  ime->keywords = cpl_propertylist_new();
  if (ime->keywords == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for cpl_propertylist");
      mat_imextended_delete(ime);
      return NULL;
    }
  ime->channelgain = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (ime->channelgain == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel gain");
      mat_imextended_delete(ime);
      return NULL;
    }
  ime->channeloffset = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (ime->channeloffset == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel offset");
      mat_imextended_delete(ime);
      return NULL;
    }
  ime->channelnoise = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (ime->channelnoise == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel noise");
      mat_imextended_delete(ime);
      return NULL;
    }
  ime->detectorgain = 1.0;
  ime->detectornoise = 0.0;
  ime->horizontalcorrelation = 0.0;
  ime->verticalcorrelation = 0.0;
  ime->allanvariancemin = 0.0;
  return ime;
}

/**
   @ingroup ime
   @brief This function deletes a mat_imextended structure.
   @param ime The extended detector monitoring map which will be deleted.

   This function deletes a mat_imextended structure.
   Deleting a extended detector monitoring data structure includes all members. If an
   error (for example during the creation of a man_imextended data structure)
   leads to an incomplete data structure (some members are NULL) this
   function detects this and deletes only the valid members.
 */
cpl_error_code mat_imextended_delete(mat_imextended *ime)
{
  mat_assert((ime != NULL), CPL_ERROR_NULL_INPUT, "no mat_imextended (ime) argument given");
  if (ime->keywords != NULL)
    {
      cpl_propertylist_delete(ime->keywords);
      ime->keywords = NULL;
    }
  if (ime->channelgain != NULL)
    {
      cpl_vector_delete(ime->channelgain);
      ime->channelgain = NULL;
    }
  if (ime->channeloffset != NULL)
    {
      cpl_vector_delete(ime->channeloffset);
      ime->channeloffset = NULL;
    }
  if (ime->channelnoise != NULL)
    {
      cpl_vector_delete(ime->channelnoise);
      ime->channelnoise = NULL;
    }
  if (ime->det != NULL)
    {
      mat_detector_delete(ime->det);
      ime->det = NULL;
    }
  cpl_free(ime);
  return CPL_ERROR_NONE;
}

/**
   @ingroup ime
   @brief This function saves a mat_imextended structure in a FITS file.
   @param ime The extended detector monitoring map of a detector.
   @param fname The name of the extended detector monitoring map FITS file.
   @param rname The name of the pipeline recipe.
   @param parlist The plugin parameter list.
   @param frameset The input frameset of the plugin (optional).
   @returns A cpl error code

   This function stores the contents of the mat_imextended data structure
   as an extended detector monitoring map in a FITS file. All QC1 parameters
   are stored in an empty primary header.

   The following QC1 parameters are stored in the primary header:

   - QC DETi GAINp The global detector gain (in electrons per ADU) will be used for scientific purposes.
   - QC DETi RONp The global read-out noise (in electrons).
   - QC DETi CHANNELj GAINp The detector channel specific gain (in electrons per ADU) will be used to detect detector and read-out electronics problems.
   - QC DETi CHANNELj OFFSETp The offset for the detector channels.
   - QC DETi CHANNELj RONp The read-out noise is given for each detector channel.
   - GC DETi HACORRp The Autocorrelation coefficient between two horizontally adjacent pixels.
   - GC DETi VACORRp The Autocorrelation coefficient between two vertically adjacent pixels.
   - GC DETi AVAR MIN Position of the turning point of the Allan variance plot.

   Where i is the detector number (1 = L/M-Band, 2 = N-Band), j is the detector channel
   number (1 .. 32 for the L/M-Band, 1 .. 64 for the N-Band), and p is the readout
   mode (HAWAII-2RG: 1 = slow readout, 2 = fast readout, Aquarius: 1 = low gain mode,
   2 = high gain mode). The detector number (used for DETi) will be always 1 for
   the L/M-Band detector and 2 for the N-Band detector. In addition, the DETi CHIP1 ID
   keyword allows to distinguish between different detector chips. This is necessary,
   to deal with a detector change after, for example, an instrument repair.
 */
cpl_error_code mat_imextended_save(mat_imextended *ime,
				   const char *fname,
				   const char *rname,
				   cpl_parameterlist *parlist,
				   cpl_frameset *frameset)
{
  int               c, nc;
  cpl_frame        *imeframe = NULL;
  cpl_propertylist *plist = NULL;
  char              kwd[64];

  cpl_msg_info(cpl_func, "storing the extended detector information in file %s", fname);
  cpl_error_reset();
  cpl_ensure_code((ime != NULL), CPL_ERROR_NULL_INPUT);
  /* Create product frame */
  imeframe = cpl_frame_new();
  cpl_frame_set_filename(imeframe, fname);
  cpl_frame_set_tag(imeframe, MATISSE_IME_PROCATG);
  cpl_frame_set_type(imeframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(imeframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(imeframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_frame_delete(imeframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  /*
    check if the input frameset is empty
    => function used in an artificial unit test (no real input files)
    => the DFS header cannot include the keywords from the first input file
  */
  if (!cpl_frameset_is_empty(frameset))
    {
      if (cpl_dfs_setup_product_header(plist, imeframe, frameset, parlist,
				       rname, PACKAGE "/" PACKAGE_VERSION, "?Dictionary?", NULL) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_error_reset();
	cpl_propertylist_delete(plist);
	cpl_frame_delete(imeframe);
	return CPL_ERROR_UNSPECIFIED;
      }
    }
  /* Add QC parameters in the header */
  /* readout mode is missing */
  snprintf(kwd, 64, "ESO QC DET%d GAIN%d", ime->det->nr, ime->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(ime->detectorgain, MAT_PREC_GAIN));
  snprintf(kwd, 64, "ESO QC DET%d RON%d", ime->det->nr, ime->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(ime->detectornoise, MAT_PREC_NOISE));
  nc = cpl_vector_get_size(ime->channelgain);
  for (c = 0; c < nc; c++)
    {
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d GAIN%d", ime->det->nr, c + 1, ime->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(ime->channelgain, c), MAT_PREC_GAIN));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d OFFSET%d", ime->det->nr, c + 1, ime->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(ime->channeloffset, c), MAT_PREC_INTENSITY));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d RON%d", ime->det->nr, c + 1, ime->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(ime->channelnoise, c), MAT_PREC_NOISE));
    }
  snprintf(kwd, 64, "ESO QC DET%d HACORR%d", ime->det->nr, ime->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(ime->horizontalcorrelation, MAT_PREC_CORRELATION));
  snprintf(kwd, 64, "ESO QC DET%d VACORR%d", ime->det->nr, ime->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(ime->verticalcorrelation, MAT_PREC_CORRELATION));
  snprintf(kwd, 64, "ESO QC DET%d AVAR MIN%d", ime->det->nr, ime->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(ime->allanvariancemin, MAT_PREC_VARIANCE));

  // Add Generic QC parameters
  cpl_propertylist *qclist=NULL;
  qclist = cpl_propertylist_new();
  mat_add_generic_qc(plist,qclist);
  cpl_propertylist_append(plist,qclist);
  cpl_propertylist_delete(qclist);


  /* Save the file */
  cpl_propertylist_erase(plist,"RADECSYS");
  if (cpl_propertylist_save(plist, fname, CPL_IO_CREATE) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_propertylist_delete(plist);
      cpl_frame_delete(imeframe);
      return CPL_ERROR_UNSPECIFIED;
    }
  cpl_propertylist_delete(plist);
  /* Log the saved file in the input frameset */
  cpl_frameset_insert(frameset, imeframe);
  //cpl_frame_delete(imeframe);
  return CPL_ERROR_NONE;
}
