/* $Id: mat_imgreg.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_imgreg.c $
 */

#include <stdlib.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>

#include "mat_error.h"
#include "mat_utils.h"
#include "mat_imgreg.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Functions
 -----------------------------------------------------------------------------*/

/**
   @brief Deletes a mat_imgreg data structure.
   @param region   A mat_imgreg data structure.
   @returns A cpl_error_code

   This function deletes all enclosed cpl_image data structures, deletes the vector
   of the cpl_image pointers and deletes the data structure itself.
 */
cpl_error_code mat_imgreg_delete(mat_imgreg *region)
{
  mat_assert_not_null(region, "no mat_imgreg (region) argument given");
  if (region->imgreg != NULL)
    {
      int i;
      for (i = 0; i < region->nbimgreg; i++)
	{
	  if (region->imgreg[i] != NULL)
	    {
	      cpl_image_delete(region->imgreg[i]);
	      region->imgreg[i] = NULL;
	    }
	}
      cpl_free(region->imgreg);
      region->nbimgreg = 0;
      region->imgreg = NULL;
    }
  cpl_free(region);
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Duplicate a mat_imgreg structure
  @param imgreg The mat_imgreg structure to duplicate.
  @param type   A cpl_tyep defining the data type of the created images.
  @return img_dup duplicated structure
 */
/*----------------------------------------------------------------------------*/
mat_imgreg *mat_imgreg_duplicate(const mat_imgreg *imgreg, cpl_type type)
{
  mat_imgreg *img_dup = NULL;
  int         k = 0;

  // Check input parameters
  mat_assert_value((imgreg!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no imgreg argument given");
  img_dup=cpl_calloc(1, sizeof(mat_imgreg));
  if (img_dup == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for img_dup");
      return NULL;
    }
  img_dup->numregion   = imgreg->numregion;
  img_dup->numdetector = imgreg->numdetector;
  img_dup->nbimgreg    = imgreg->nbimgreg;
  img_dup->imgreg      = cpl_calloc(imgreg->nbimgreg, sizeof(cpl_image *));
  // Copy images
  for(k = 0; k < img_dup->nbimgreg; k++)
    {
      cpl_image *srcimg = imgreg->imgreg[k];
      if (cpl_image_get_type(srcimg) == type)
	{
	  img_dup->imgreg[k] =  cpl_image_duplicate(srcimg);
	}
      else
	{
	  int x, y, nx, ny, rejected;
	  cpl_image *dstimg = NULL;
	  nx = cpl_image_get_size_x(srcimg);
	  ny = cpl_image_get_size_y(srcimg);
	  dstimg = cpl_image_new(nx, ny, type);
	  for (y = 1; y <= ny; y++)
	    {
	      for (x = 1; x <= nx; x++)
		{
		  cpl_image_set(dstimg, x, y, cpl_image_get(srcimg, x, y, &rejected));
		}
	    }
	  img_dup->imgreg[k] = dstimg;
	}
    }
  return img_dup;
}

