/* $Id: mat_kappamatrix.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_kappamatrix.h $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "mat_kappamatrix.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/
#define ERR_MAX_PIX 0.05
const int posShutN[]={3,2,0,1};
const int posShutL[]={0,1,3,2};



/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @ingroup kappa
  @brief Load a kappa matrix in mat_kappamatrix structure
  @param frame           current frame
  @return mat_kappamatrix
 */
/*-----------------------------------------------------------------------------*/
mat_kappamatrix * mat_kappamatrix_load(cpl_frame *frame) {

  int i=0;
  int j=0;
  int k=0;
  int l=0;
  int m=0;
  int index=0;
  int nbExtent=0;
  int flagDet=0;
  int flagKappa=0;
  char *keyExtname=NULL;
  mat_kappamatrix *kappa=NULL;  
  cpl_propertylist *plist=NULL;
  cpl_table *table=NULL;
  cpl_errorstate prestate = cpl_errorstate_get();
    
  // Check input parameters
  mat_assert_value((frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no cpl_frame (frame) argument given");

  // Allocate the structure and initialize its attributes
  kappa =cpl_malloc(sizeof(mat_kappamatrix));
  if (kappa == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for mat_kappamatrix");
    return NULL;
  }
  kappa->keywords= cpl_propertylist_load(cpl_frame_get_filename(frame),0);
  kappa->imgdet=NULL;
  
  // Find all extensions before KAPPA_MATRIX
  nbExtent=cpl_frame_get_nextensions(frame);
  for(i=0;i<nbExtent;i++) {
    plist=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1); 
    keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
    if (keyExtname != NULL) {
      table=cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
      // Load data in attribute imgdet
      if (!strcmp(keyExtname,"IMAGING_DETECTOR")) {
	kappa->imgdet = mat_imagingdetector_from_table(plist, table);
        /* Error handling */
        if (kappa->imgdet == NULL) { 
          cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
				"Could not load IMAGING_DETECTOR in frame: %s",
				cpl_frame_get_filename(frame));
        }
        else {
	  flagDet=1;
	}
      }
      cpl_table_delete(table);
    }
    cpl_propertylist_delete(plist);
  }
  
  // Now find  KAPPA_MATRIX
  for(i=0;i<nbExtent;i++) {
    plist=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
    keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
    if (keyExtname != NULL) {
      table=cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
      // Load data in attribute kappa 
      if (!strcmp(keyExtname,"KAPPA_MATRIX")) {
	// load kappa->nbbeam by computing the number of photomteric regions 
	// from kappa->imgdet
	kappa->nbbeam=0;
	for(j=0;j<kappa->imgdet->nbregion;j++) {
	  if ( kappa->imgdet->list_region[j]->correlation == 1 ) {
	    kappa->nbbeam++;
	  }
	}
	
	kappa->list_kappacoef=cpl_calloc(kappa->nbbeam,sizeof(mat_kappacoef *));
	if (kappa->list_kappacoef == NULL) {
	  cpl_msg_error(cpl_func,"could not allocate memory for mat_kappacoef");
	  return NULL;
	}
	for(j=0; j<kappa->nbbeam; j++) {
	  kappa->list_kappacoef[j] = cpl_malloc(sizeof(mat_kappacoef));
	  if (kappa->list_kappacoef[j] == NULL) {
	    cpl_msg_error(cpl_func,"could not allocate memory for kappa->list_kappacoef[j]");
	    return NULL;
	  }
	  kappa->list_kappacoef[j]->numregion = 
	    cpl_table_get_int(table,"REGION",j,NULL);
	  kappa->list_kappacoef[j]->numdetector = 
	    cpl_table_get_int(table,"DETECTOR",j,NULL);
	  
	  for(k=0;k<kappa->imgdet->nbregion;k++) {
	    if ( kappa->imgdet->list_region[k]->numregion == 
		 kappa->list_kappacoef[j]->numregion ) {
	      kappa->list_kappacoef[j]->nbchannel = 
		kappa->imgdet->list_region[k]->naxis[1];
	    }
	  }
	  kappa->list_kappacoef[j]->coef = 
	    cpl_matrix_new(3,kappa->list_kappacoef[j]->nbchannel);
	  kappa->list_kappacoef[j]->errcoef = 
	    cpl_matrix_new(3,kappa->list_kappacoef[j]->nbchannel);
	  
	  index=0;
	  
	  for(m=0;m<3;m++) {
	    for(l=0;l<kappa->list_kappacoef[j]->nbchannel;l++) {
	      cpl_matrix_set(kappa->list_kappacoef[j]->coef, m, l,
			     cpl_array_get_double
			     (cpl_table_get_array(table,"MATRIX",j),index,NULL));
	      cpl_matrix_set(kappa->list_kappacoef[j]->errcoef, m, l,
			     cpl_array_get_double
			     (cpl_table_get_array(table,"ERROR",j),index,NULL));
	      index++;
	    }
	  }
	}
	flagKappa=1;
      }
      cpl_table_delete(table);
    }
    cpl_propertylist_delete(plist);
  }
  // Check the flags
  if ( flagDet==0 || flagKappa==0) {
    cpl_error_set_message(cpl_func,CPL_ERROR_ILLEGAL_INPUT,
                          "Extension missing in frame: %s",
                          cpl_frame_get_filename(frame));
  }
  // Free memory
  
  /* Error handling */
  if (!cpl_errorstate_is_equal(prestate)) { 
    
    cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
    // Unload the mat_gendata structure
    mat_kappamatrix_free(kappa);
    return NULL;
  }
  
  return kappa;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup kappa
  @brief delete structure mat_kappamatrix
  @param  kappa          current structure
  @return error code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_kappamatrix_free(mat_kappamatrix *kappa) {
  int i=0;

  // Check input parameters
  mat_assert_value((kappa!=NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no cpl_framemat_kappamatrix (kappa) argument given");

  if (kappa->keywords!=NULL)
    {
      cpl_propertylist_delete(kappa->keywords);
      kappa->keywords = NULL;
    }
  if (kappa->imgdet!=NULL)
    {
      mat_imagingdetector_delete(kappa->imgdet);
      kappa->imgdet = NULL;
    }

  if (kappa->list_kappacoef != NULL)
    {
      for(i=0; i<kappa->nbbeam; i++)
	{
	  if (kappa->list_kappacoef[i] != NULL)
	    {
	      if (kappa->list_kappacoef[i]->coef != NULL)
		{
		  cpl_matrix_delete(kappa->list_kappacoef[i]->coef);
		  kappa->list_kappacoef[i]->coef = NULL;
		}
	      if (kappa->list_kappacoef[i]->errcoef != NULL)
		{
		  cpl_matrix_delete(kappa->list_kappacoef[i]->errcoef);
		  kappa->list_kappacoef[i]->errcoef = NULL;
		}
	      cpl_free(kappa->list_kappacoef[i]);
	      kappa->list_kappacoef[i] = NULL;
	    }
	}
      cpl_free(kappa->list_kappacoef);
      kappa->list_kappacoef = NULL;
    }
  cpl_free(kappa);

  return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup kappa
  @brief Save a mat_kappamatrix structure in a KAPPA_MATRIX FITS file
  @param kappa             mat_kappamatrix structure to save
  @param pheader           header of the binary table
  @param set_in            input frameset
  @param set_sel           selected frameset
  @param parlist           recipe parameter list
  @param recipe_name       name of recipe
  @param pro_cat           PRO.CATG
  @param pro_tech          PRO.TECH
  @param pro_sci           PRO.SCIENCE
  @param filename          name of the FITS file
  @return cpl_error_code
 */
/*-----------------------------------------------------------------------------*/

cpl_error_code mat_kappamatrix_save(
				    mat_kappamatrix *kappa,
				    cpl_propertylist *pheader, 
				    cpl_frameset *set_in,
				    cpl_frameset *set_sel,
				    const cpl_parameterlist *parlist,
				    const char *recipe_name,
				    const char *pro_cat,
				    const char *pro_tech,
				    const char *pro_sci,
				    char *filename) {
  
  int i=0;
  int j=0;
  int nbsubreg=0;
  int *id_region=NULL;
  double *matrix=NULL;
  double *error=NULL;
  char *str=NULL;
  cpl_table *kappa_map=NULL;
  cpl_propertylist *applist=NULL;
  cpl_array *matrix_array=NULL;
  cpl_array *error_array=NULL;
  cpl_errorstate prestate= cpl_errorstate_get();

  // Check input parameters
  mat_assert_value((kappa!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,"no mat_kappamatrix (kappa) argument given");
  mat_assert_value((pheader!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, "no cpl_propertylist (pheader) argument given");
  mat_assert_value((set_in!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, "no cpl_frameset (set_in) argument given");
  mat_assert_value((set_sel!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, "no cpl_frameset (set_sel) argument given");
  mat_assert_value((filename!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, "no filename given");


  nbsubreg = (*kappa).imgdet->nbregion/5;
  id_region = cpl_calloc(nbsubreg*4,sizeof(int));


  // Prepare the keywords of the KAPPA_MAP binary table

  str=cpl_malloc(50*sizeof(char));
  strncpy(str,(char *)cpl_propertylist_get_string(pheader,"ORIGIN"),50-1);
  str[50-1] = '\0';
  cpl_propertylist_erase(pheader,"ORIGIN");
  cpl_propertylist_append_string(pheader,"ORIGIN",str);
  strncpy(str,(char *)cpl_propertylist_get_string(pheader,"DATE"),50-1);
  str[50-1] = '\0';
  cpl_propertylist_erase(pheader,"DATE");
  cpl_propertylist_append_string(pheader,"DATE",str);
  cpl_free(str);
  cpl_propertylist_erase(pheader,"XTENSION");
  cpl_propertylist_erase(pheader,"BITPIX");
  cpl_propertylist_erase_regexp(pheader,"NAXIS",0);
  cpl_propertylist_erase_regexp(pheader,"COUNT",0);
  cpl_propertylist_erase(pheader,"TFIELDS");
  cpl_propertylist_erase_regexp(pheader,"SIM",0);
  cpl_propertylist_erase_regexp(pheader,"FORM",0);
  cpl_propertylist_erase_regexp(pheader,"TTYPE",0);
  cpl_propertylist_erase_regexp(pheader,"TDIM",0);


  // Remove the keywords from the DPR categories
  cpl_propertylist_erase_regexp(kappa->keywords, "DPR", 0);

  // Add the keywords from the PRO categories
  if (pro_tech != NULL) {
    cpl_propertylist_append_string(kappa->keywords, CPL_DFS_PRO_TECH, pro_tech);
  }
  if (pro_cat != NULL) {
    cpl_propertylist_append_string(kappa->keywords, CPL_DFS_PRO_CATG, pro_cat);
  }
  if (pro_sci != NULL) {
    /* cpl_propertylist_append_string(kappa->keywords, CPL_DFS_PRO_SCIENCE, pro_sci); */
    if (strstr(pro_sci,"F") != NULL)
      {
	cpl_propertylist_append_bool(kappa->keywords, CPL_DFS_PRO_SCIENCE, 0);
      }
    else
      {
	cpl_propertylist_append_bool(kappa->keywords, CPL_DFS_PRO_SCIENCE, 1);
      }
  }
  // Fill a cpl_table with kappa data that will replace the IMAGING_DATA
  //Determine the number of photometric regions and their index
  i=0;
  for(j=0;j<(*kappa).imgdet->nbregion;j++) {
    if (kappa->imgdet->list_region[j]->correlation == 1)
      {
	id_region[i]=kappa->list_kappacoef[i]->numregion+1;
	i++;
      }
  }
  kappa_map = cpl_table_new(nbsubreg*4);
  // Add columns
  cpl_table_new_column(kappa_map, "DETECTOR", CPL_TYPE_INT);
  cpl_table_new_column(kappa_map, "REGION", CPL_TYPE_INT);
  cpl_table_new_column_array(kappa_map, "MATRIX", CPL_TYPE_DOUBLE,
  			     cpl_matrix_get_nrow((*kappa).list_kappacoef[0]->coef) *
  			     cpl_matrix_get_ncol((*kappa).list_kappacoef[0]->coef));
  cpl_table_new_column_array(kappa_map, "ERROR", CPL_TYPE_DOUBLE,
  			     cpl_matrix_get_nrow((*kappa).list_kappacoef[0]->errcoef) *
  			     cpl_matrix_get_ncol((*kappa).list_kappacoef[0]->errcoef));

  // Get the data in a cpl_array
  for(i=0; i<nbsubreg*4; i++) {
    matrix = (double *)cpl_matrix_get_data((*kappa).list_kappacoef[i]->coef);
    error = (double *)cpl_matrix_get_data((*kappa).list_kappacoef[i]->errcoef);
    matrix_array =
      cpl_array_wrap_double(matrix,
  			    cpl_matrix_get_nrow((*kappa).list_kappacoef[i]->coef) *
  			    cpl_matrix_get_ncol((*kappa).list_kappacoef[i]->coef));
    error_array =
      cpl_array_wrap_double(error,
  			    cpl_matrix_get_nrow((*kappa).list_kappacoef[i]->errcoef) *
  			    cpl_matrix_get_ncol((*kappa).list_kappacoef[i]->errcoef));
    // Fill the columns with the data
    cpl_table_set_array(kappa_map, "MATRIX", i, matrix_array);
    cpl_table_set_array(kappa_map, "ERROR", i, error_array);
    cpl_table_set_int(kappa_map, "DETECTOR", i,
  		      (*kappa).imgdet->list_region[i]->numdetector);
    cpl_table_set_int(kappa_map,"REGION", i, id_region[i]);
    cpl_array_unwrap(matrix_array);
    cpl_array_unwrap(error_array);
  }


  // Make a new FITS file with kappa keywords and the created table
  applist=cpl_propertylist_duplicate(kappa->keywords);
  cpl_propertylist_erase(kappa->keywords,"RADECSYS");
  cpl_propertylist_erase(pheader,"RADECSYS");
  cpl_dfs_save_table(set_in, kappa->keywords, parlist, set_sel, NULL,
  		     kappa_map, pheader, recipe_name, applist, NULL,
  		     PACKAGE "/" PACKAGE_VERSION, filename);


 // Free memory
  if (kappa_map !=NULL) cpl_table_delete(kappa_map);
  if (id_region != NULL) cpl_free(id_region);
  if (applist != NULL) cpl_propertylist_delete(applist);


 
  if (!cpl_errorstate_is_equal(prestate)) { 
    
    cpl_errorstate_set(prestate);
    
    return CPL_ERROR_UNSPECIFIED;
  }
  return CPL_ERROR_NONE;

}


/*----------------------------------------------------------------------------*/
/**
  @ingroup kappa
  @brief Apply the shift, zoom, ratio coefficients for transforming a 
  row of a photomertic channel into a row in the interferometric one.
  @param p_row : row of the photometric channel                  
  @param p_param : parameters (shift,zoom,ratio)
  @param p_res : resulting row in the interferometric channel
  @return cpl_error_code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code interpolate(cpl_vector *p_row, cpl_vector *p_param, 
			   cpl_vector *p_res){
  int xmid=0;
  int Xmid=0;
  int i=0;
  int n=0;
  int npt=0;
  int N=0;
  cpl_vector *x=NULL;
  cpl_vector *X=NULL;
  cpl_vector *v=NULL;
  cpl_bivector *p_fout=NULL;
  cpl_bivector *p_fref=NULL;
  
  // Check input parameters
  mat_assert_value((p_row!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no p_row argument given");
  mat_assert_value((p_param!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no p_param argument given");
  mat_assert_value((p_res!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no p_res argument given");
  
  n = cpl_vector_get_size(p_row);
  x = cpl_vector_new(n) ;
  
  npt = cpl_vector_get_size(p_res);
  N = 4*npt/3;
  X = cpl_vector_new(npt);
  v = cpl_vector_new(npt);
  
  xmid = ceil(n/2);
  Xmid = ceil(N/2);
  for(i = 0;i<n;i++) {
    cpl_vector_set(x,i,(i-xmid)/(double)n);
  }
  for(i = 0;i<npt;i++) {  
    cpl_vector_set(X,i,cpl_vector_get
    		   (p_param,2)+((i+(N/8)-Xmid)/
    				((double)n *
    				 cpl_vector_get(p_param,1))));
  }

  p_fref = cpl_bivector_wrap_vectors(x,p_row);
  p_fout = cpl_bivector_wrap_vectors(X,v);
  cpl_bivector_interpolate_linear(p_fout,p_fref);
  

  for(i = 0;i<npt;i++) {
    cpl_vector_set(p_res,i,cpl_vector_get(v,i)*
		   cpl_vector_get(p_param,0)*((double)n)/((double)N));
  }
  if (x != NULL) cpl_vector_delete(x);
  if (X != NULL) cpl_vector_delete(X);
  if (v != NULL) cpl_vector_delete(v);
  if (p_fout != NULL) cpl_free(p_fout);
  if (p_fref != NULL) cpl_free(p_fref);

  return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
  @ingroup kappa
  @brief Calculates the derivativeof the (shift,zoom,ratio) transformation.
  @param p_row : row of the photometric channel                  
  @param p_param : parameters (shift,zoom,ratio)
  @param p_dres : transformation derivatives
  @return cpl_error_code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code derivative(cpl_vector *p_row, cpl_vector *p_param, 
			  cpl_matrix *p_dres){
  cpl_vector *p1=NULL;
  cpl_vector *p_res=NULL;
  cpl_vector *p_res1=NULL;
  cpl_vector *p_res2=NULL;
  int N=0;
  int i=0;
  double aux=0.;
  double aux1=0.;
  double aux2=0. ;

  // Check input parameters
  mat_assert_value((p_row!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no p_row argument given");
  mat_assert_value((p_param!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no p_param argument given");
  mat_assert_value((p_dres!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no p_dres argument given");

  aux = 0;
  aux1 = 0;
  aux2 = 0;
  
  p1 = cpl_vector_new(3);
  cpl_vector_set(p1,0,1.0);
  cpl_vector_set(p1,1,cpl_vector_get(p_param,1));
  cpl_vector_set(p1,2,cpl_vector_get(p_param,2)); 
  
  N = cpl_matrix_get_nrow(p_dres);
  p_res = cpl_vector_new(N);
  p_res1 = cpl_vector_new(N);
  p_res2 = cpl_vector_new(N);  
  interpolate(p_row,p1,p_res);
  for(i = 0;i<N;i++)
    {
      cpl_matrix_set(p_dres,i,0,cpl_vector_get(p_res,i));
    }
  
  cpl_vector_set(p1,0,cpl_vector_get(p_param,0));
  cpl_vector_set(p1,1,cpl_vector_get(p_param,1)+0.01);
  cpl_vector_set(p1,2,cpl_vector_get(p_param,2));
  interpolate(p_row,p1,p_res1);
  
  cpl_vector_set(p1,0,cpl_vector_get(p_param,0));
  cpl_vector_set(p1,1,cpl_vector_get(p_param,1)-0.01);
  cpl_vector_set(p1,2,cpl_vector_get(p_param,2));
  interpolate(p_row,p1,p_res2);
  
  for(i = 0;i<N;i++)
    {
      aux1 = cpl_vector_get(p_res1,i); 
      aux2 = cpl_vector_get(p_res2,i);
      aux = (aux1-aux2)/0.02;
      cpl_matrix_set(p_dres,i,1,aux);
    }
  
  cpl_vector_set(p1,0,cpl_vector_get(p_param,0));
  cpl_vector_set(p1,1,cpl_vector_get(p_param,1));
  cpl_vector_set(p1,2,cpl_vector_get(p_param,2)+0.01);
  interpolate(p_row,p1,p_res1);
  
  cpl_vector_set(p1,0,cpl_vector_get(p_param,0));
  cpl_vector_set(p1,1,cpl_vector_get(p_param,1));
  cpl_vector_set(p1,2,cpl_vector_get(p_param,2)-0.01);
  interpolate(p_row,p1,p_res2);
  
  for(i = 0;i<N;i++)
    {
      aux1 = cpl_vector_get(p_res1,i); 
      aux2 = cpl_vector_get(p_res2,i);
      aux = (aux1-aux2)/0.02;
      cpl_matrix_set(p_dres,i,2,aux);
    }
  if (p1 != NULL) cpl_vector_delete(p1);
  if (p_res != NULL) cpl_vector_delete(p_res);
  if (p_res1 != NULL) cpl_vector_delete(p_res1);  
  if (p_res2 != NULL) cpl_vector_delete(p_res2);
  return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup kappa
   @brief Compute the Kappa Matrix and stores it in a mat_kappamatrix structure
   @param pp_shutter1 : data with all shutters closed except beam 1
   @param pp_shutter2 : data with all shutters closed except beam 2
   @param pp_shutter3 : data with all shutters closed except beam 3
   @param pp_shutter4 : data with all shutters closed except beam 4
   @param pp_sky_shutter1 : Background data with all shutters closed except beam 1
   @param pp_sky_shutter2 : Background data with all shutters closed except beam 2
   @param pp_sky_shutter3 : Background data with all shutters closed except beam 3
   @param pp_sky_shutter4 : Background data with all shutters closed except beam 4
   @param nbFile : array of int containing the number of frames for each 
   kinf of data (dimension = 5)
   @return p_kappamatrix : the computed kappa matrix
*/
/*-----------------------------------------------------------------------------*/
mat_kappamatrix *mat_compute_kappamatrix(mat_gendata ** pp_shutter1,
					 mat_gendata ** pp_shutter2, 
					 mat_gendata ** pp_shutter3, 
					 mat_gendata ** pp_shutter4, 
					 mat_gendata ** pp_sky_shutter1,
					 mat_gendata ** pp_sky_shutter2,
					 mat_gendata ** pp_sky_shutter3,
					 mat_gendata ** pp_sky_shutter4,
					 int   nbFile[8]) {
  mat_kappamatrix *p_kappamatrix=NULL;
  cpl_image **pp_shutters_raw[8];
  int nbregion=0;
  int i=0;
  int k=0;
  int l=0;
  int j=0;
  /* int y=0; */
  /* int nphot=0; */
  /* int ninterf=0; */
  int yphot=0;
  /* int neight=0; */
  /* int nthreequarter=0; */
  /* int nbiter=0; */
  int *wI=NULL;
  int flag_rejected = 0;
  int *wS=NULL;
  int *p_nbimg=NULL;
  int s=0;
  int nbsubreg=0;
  /* cpl_matrix *p_matA=NULL; */
  /* cpl_matrix *p_matB=NULL; */
  /* cpl_matrix *p_row_interf=NULL; */
  /* cpl_matrix *p_mat_coef=NULL; */
  /* cpl_matrix *p_aux=NULL; */
  /* cpl_matrix *p_matA_trans=NULL; */
  /* cpl_matrix *normal=NULL; */
  /* cpl_matrix *invnormal=NULL; */
  /* cpl_matrix *p_err=NULL; */
  cpl_matrix *pmat_ratio=NULL;
  cpl_matrix *pmat_zoom=NULL;
  cpl_matrix *pmat_shift=NULL;
  /* cpl_vector *p_param=NULL; */
  /* cpl_vector *p_row_phot=NULL; */
  /* cpl_vector *p_res=NULL; */
  /* cpl_vector *p_vectB=NULL; */
  int x_size=0;
  int y_size=0;
  int nb_frame=0;
  /* double max=0.; */
  double *stdev_ratio=NULL;
  double *stdev_zoom=NULL;
  double *stdev_shift=NULL;
  double *mean_ratio=NULL;
  double *mean_zoom=NULL;
  double *mean_shift=NULL;
  double *flux=NULL;
  double stdev_ratio_m=0.;
  double stdev_zoom_m=0.;
  double stdev_shift_m=0;
  double mean_ratio_m=0;
  double mean_zoom_m=0;
  double mean_shift_m=0;
  char *qcname_mean_shift=NULL;
  char *qcname_mean_iratio=NULL;
  char *qcname_mean_zoom=NULL;
  char *chipName=NULL;
  char *qcname_stdev_shift=NULL;
  char *qcname_stdev_iratio=NULL;
  char *qcname_stdev_zoom=NULL;
  char *qcname_flux=NULL;
  cpl_propertylist *qclist=NULL;
  /* double bias=0.; */
  int posShut[4];
  /* int indexMaxPhot=0; */
  /* int indexMaxInterf=0; */
  /* double valmax=0.; */

  cpl_vector *vec1,*vec2,*vecx1,*vecx2;
  double xc1,xc2,sigma1,sigma2,area1,area2,offset;
  double zoom,ratio;
  double errPix1,errPix2;
  double val=0.;
    /* double m0vec1,m1vec1,m2vec1; */
    /* double m0vec2,m1vec2,m2vec2; */
    /* double valmax1,valmax2; */
  int sizex1,sizex2;
  int err1,err2;
  /* FILE *fp; */
 

  // Check input parameters
  mat_assert_value((pp_shutter1!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no pp_shutter1 argument given");
  mat_assert_value((pp_shutter2!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no pp_shutter2 argument given");
  mat_assert_value((pp_shutter3!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no pp_shutter3 argument given");
  mat_assert_value((pp_shutter4!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no pp_shutter4 argument given");
  mat_assert_value((pp_sky_shutter1!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no pp_sky_shutter1 argument given");
  mat_assert_value((pp_sky_shutter2!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no pp_sky_shutter2 argument given");
  mat_assert_value((pp_sky_shutter3!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no pp_sky_shutter3 argument given");
  mat_assert_value((pp_sky_shutter4!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no pp_sky_shutter4 argument given");



  if (pp_shutter1[0]->imgdet->list_region[0]->numdetector == 2) {
    cpl_msg_info(cpl_func,"Detector: AQUARIUS");
    for(i=0;i<4;i++) {
      posShut[i]=posShutN[i];
    }
  } else {
    cpl_msg_info(cpl_func,"Detector: HAWAI_2RG");
    for(i=0;i<4;i++) {
      posShut[i]=posShutL[i];
    }
  }

/*   pp_shutters[0] = pp_sky; */
/*   pp_shutters[1] = pp_shutter1; */
/*   pp_shutters[2] = pp_shutter2; */
/*   pp_shutters[3] = pp_shutter3; */
/*   pp_shutters[4] = pp_shutter4; */
  j = 0;
  k = 0;
  nbregion = pp_shutter1[0]->imgdet->nbregion; 
  nbsubreg = nbregion/5;
  wI = cpl_calloc(nbsubreg,sizeof(int));
  wS = cpl_calloc(4*nbsubreg,sizeof(int));
  for(i=0; i<nbregion; i++)
    {
      if (pp_shutter1[0]->imgdet->list_region[i]->correlation == 2)
	{
	  //interferometry indices
	  wI[k] = i;
	  k++;
	}
      else
	{
	  //photometry indices
	  wS[j] = i;
	  j++;
	}
    }
  // Initialize the cpl_images
  for(j=0;j<8;j++) {
    pp_shutters_raw[j] = cpl_calloc( nbregion, sizeof(cpl_image *));
  }
    
  for(i = 0; i<nbregion; i++) {
    x_size = pp_shutter1[0]->imgdet->list_region[i]->naxis[0];
    y_size = pp_shutter1[0]->imgdet->list_region[i]->naxis[1];
    for(j=0;j<8;j++) {
      pp_shutters_raw[j][i] =  cpl_image_new
	(x_size, y_size,cpl_image_get_type
	 (pp_shutter1[0]->imgdata->list_frame[0]->list_subwin[i]->imgreg[0]));
    }
  }

  //  Copy the images from mat_gendata parameter: add the data for each region and each frame
  p_nbimg = cpl_calloc(8, sizeof(int));
  for (s=0; s<8; s++) {
    k = 0;
    p_nbimg[s]=0;
    while(k < nbFile[s]) {
      for(i=0; i<nbregion; i++) {
	if (s == 0) {
	  nb_frame = pp_sky_shutter1[k]->imgdata->nbframe;
	} else if (s == 1) {
	  nb_frame = pp_sky_shutter2[k]->imgdata->nbframe;
	} else if (s == 2) {
	  nb_frame = pp_sky_shutter3[k]->imgdata->nbframe;
	} else if (s == 3) {
	  nb_frame = pp_sky_shutter4[k]->imgdata->nbframe;
	} else if (s == 4) {
	  nb_frame = pp_shutter1[k]->imgdata->nbframe;
	} else if (s == 5) {
	  nb_frame = pp_shutter2[k]->imgdata->nbframe;
	} else if (s == 6) {
	  nb_frame = pp_shutter3[k]->imgdata->nbframe;
	} else if (s == 7) {
	  nb_frame = pp_shutter4[k]->imgdata->nbframe;
	}
	for(j = 0; j<nb_frame; j++) {
	  if (s == 0) {
	    cpl_image_add(pp_shutters_raw[s][i],
			  pp_sky_shutter1[k]->imgdata->list_frame[j]
			  ->list_subwin[i]->imgreg[0]);
	  } else if (s == 1) {
	    cpl_image_add(pp_shutters_raw[s][i],
			  pp_sky_shutter2[k]->imgdata->list_frame[j]
			  ->list_subwin[i]->imgreg[0]);
	  } else if (s == 2) {
	    cpl_image_add(pp_shutters_raw[s][i],
			  pp_sky_shutter3[k]->imgdata->list_frame[j]
			  ->list_subwin[i]->imgreg[0]);
	  } else if (s == 3) {
	    cpl_image_add(pp_shutters_raw[s][i],
			  pp_sky_shutter4[k]->imgdata->list_frame[j]
			  ->list_subwin[i]->imgreg[0]);
	  } else if (s == 4) {
	    cpl_image_add(pp_shutters_raw[s][i],
			  pp_shutter1[k]->imgdata->list_frame[j]
			  ->list_subwin[i]->imgreg[0]);
	  } else if (s == 5) {
	    cpl_image_add(pp_shutters_raw[s][i],
			  pp_shutter2[k]->imgdata->list_frame[j]
			  ->list_subwin[i]->imgreg[0]);
	  } else if (s == 6) {
	    cpl_image_add(pp_shutters_raw[s][i],
			  pp_shutter3[k]->imgdata->list_frame[j]
			  ->list_subwin[i]->imgreg[0]);
	  } else if (s == 7) {
	    cpl_image_add(pp_shutters_raw[s][i],
			  pp_shutter4[k]->imgdata->list_frame[j]
			  ->list_subwin[i]->imgreg[0]);
	  }
	  p_nbimg[s]++;
	}
      }
      k++;
    }
    p_nbimg[s]/=nbregion;
    //    Make the average image from the images added for each region
    for(i=0; i<nbregion; i++) {
      cpl_image_multiply_scalar(pp_shutters_raw[s][i],
				1/(double)p_nbimg[s]);
    }
  }

  /*  cpl_image_save(pp_shutters_raw[5][2], */
  /* 		 "test1-lampbkg.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[5][0], */
  /* 		 "test2-lampbkg.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[5][1], */
  /* 		 "test3-lampbkg.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[5][3], */
  /* 		 "test4-lampbkg.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[5][4], */
  /* 		 "test5-lampbkg.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */

  //  Substract or add the background to the raw data
  for(j=0; j<4; j++) {
    for(i=0; i<nbregion; i++) {
      // AQUARIUS
      if (pp_shutter1[0]->imgdet->list_region[0]->numdetector == 2) {
  	cpl_image_subtract(pp_shutters_raw[j+4][i], pp_shutters_raw[j][i]); 
  	//cpl_image_add(pp_shutters_raw[j+4][i], pp_shutters_raw[j][i]);
      //HAWAI-2RG
      } else {
	cpl_image_subtract(pp_shutters_raw[j+4][i], pp_shutters_raw[j][i]);
      }
 
      for(k=0;k<cpl_image_get_size_x(pp_shutters_raw[j+4][i]);k++) {
	for(l=0;l<cpl_image_get_size_y(pp_shutters_raw[j+4][i]);l++) {
	  if ( isnan(cpl_image_get(pp_shutters_raw[j+4][i],k+1,l+1,&flag_rejected)) != 0) {
	    cpl_image_set(pp_shutters_raw[j+4][i],k+1,l+1,0.);
	    cpl_msg_warning(cpl_func,"NAN value detected - beam %d - pixel (%d,%d)",i+1,k+1,l+1);
	  }
	}
      }
      

   }
  }

  

  /*  cpl_image_save(pp_shutters_raw[4][2], */
  /* 		 "test1_4.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[4][0], */
  /* 		 "test2_4.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[4][1], */
  /* 		 "test3_4.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[4][3], */
  /* 		 "test4_4.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[4][4], */
  /* 		 "test5_4.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */

  /*  cpl_image_save(pp_shutters_raw[5][2], */
  /* 		 "test1_5.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[5][0], */
  /* 		 "test2_5.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[5][1], */
  /* 		 "test3_5.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[5][3], */
  /* 		 "test4_5.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[5][4], */
  /* 		 "test5_5.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */

  /*  cpl_image_save(pp_shutters_raw[6][2], */
  /* 		 "test1_6.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[6][0], */
  /* 		 "test2_6.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[6][1], */
  /* 		 "test3_6.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[6][3], */
  /* 		 "test4_6.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[6][4], */
  /* 		 "test5_6.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */

  /*  cpl_image_save(pp_shutters_raw[7][2], */
  /* 		 "test1_7.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[7][0], */
  /* 		 "test2_7.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[7][1], */
  /* 		 "test3_7.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[7][3], */
  /* 		 "test4_7.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* cpl_image_save(pp_shutters_raw[7][4], */
  /* 		 "test5_7.fits", CPL_BPP_IEEE_FLOAT, */
  /* 		 NULL, CPL_IO_CREATE); */
  /* exit(1); */
  /* cpl_plot_image("set view map;","with pm3d","",pp_shutters_raw[4][0]); */
  /* cpl_plot_image("set view map;","with pm3d","",pp_shutters_raw[4][1]); */
  /* cpl_plot_image("set view map;","with pm3d","",pp_shutters_raw[4][3]); */
  /* cpl_plot_image("set view map;","with pm3d","",pp_shutters_raw[4][4]); */

  /* nphot = cpl_image_get_size_x (pp_shutters_raw[4][wS[0]]);//-30; */
  yphot = cpl_image_get_size_y (pp_shutters_raw[4][wS[0]]);
  /* p_row_phot = cpl_vector_new(nphot); */
  /* ninterf = cpl_image_get_size_x (pp_shutters_raw[1][wI[0]]);//-85; */
  /* nthreequarter = 3*(ninterf)/4; */
  /* neight = ninterf/8; */

  /* p_row_interf = cpl_matrix_new(nthreequarter,1); */
  //Initialize the kappamatrix
  p_kappamatrix = cpl_malloc(sizeof(mat_kappamatrix));
  if (p_kappamatrix == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for p_kappamatrix");
    cpl_free(wS);
    cpl_free(wI);
    cpl_free(p_nbimg);
    return NULL;
  }
  p_kappamatrix->nbbeam = 4*nbsubreg;
  p_kappamatrix->keywords = cpl_propertylist_duplicate(pp_shutter1[0]->keywords);
  p_kappamatrix->imgdet = mat_imagingdetector_duplicate(pp_shutter1[0]->imgdet);
  p_kappamatrix->list_kappacoef=cpl_calloc(p_kappamatrix->nbbeam, sizeof(mat_kappacoef *));
  if (p_kappamatrix->list_kappacoef == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for p_kappamatrix->list_kappacoef");
    cpl_free(wS);
    cpl_free(wI);
    cpl_free(p_nbimg);
    return NULL;
  }
  
  for (i=0; i<p_kappamatrix->nbbeam; i++) {
    p_kappamatrix->list_kappacoef[i] = cpl_malloc(sizeof(mat_kappacoef));
    if (p_kappamatrix->list_kappacoef[i] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for p_kappamatrix->list_kappacoef[i]");
      cpl_free(p_kappamatrix);
      cpl_free(wS);
      cpl_free(wI);
      cpl_free(p_nbimg);
      return NULL;
    }
    p_kappamatrix->list_kappacoef[i]->numdetector = pp_shutter1[0]->imgdet->list_region[i]->numdetector;
    p_kappamatrix->list_kappacoef[i]->numregion = wS[posShut[i%4]];
    p_kappamatrix->list_kappacoef[i]->nbchannel = yphot;
    p_kappamatrix->list_kappacoef[i]->coef = cpl_matrix_new(3,yphot);
    p_kappamatrix->list_kappacoef[i]->errcoef = cpl_matrix_new(3,yphot);
  }
  /* p_aux = cpl_matrix_new(3,yphot); */
  /* p_err = cpl_matrix_new(3,yphot); */

  /* p_param = cpl_vector_new(3); */

  double pixlow=0.;
  double pixhigh=0.;
  int aa=0;
  
    
  for(s=4; s<8; s++) {
    for(k=0; k<nbsubreg; k++) {

    sizex1=cpl_image_get_size_x(pp_shutters_raw[s][wS[posShut[s-4]+k*4]]);
    sizex2=cpl_image_get_size_x(pp_shutters_raw[s][wI[k]]);
    vecx1=cpl_vector_new(sizex1);
    vecx2=cpl_vector_new(sizex2);    
    vec1=cpl_vector_new(sizex1);
    vec2=cpl_vector_new(sizex2);
    for(j=0;j<sizex1;j++)
      {
	cpl_vector_set(vecx1,j,j);
      }
    for(j=0;j<sizex2;j++)
      {
	cpl_vector_set(vecx2,j,j);
      }

    /* if (s==4) { */
    /*   fp = fopen("ratio.dat", "w+"); */
    /* } */
    for(i=0; i<yphot; i++) {
      for(j=1;j<=sizex1;j++) {
	cpl_vector_set(vec1,j-1,cpl_image_get(pp_shutters_raw[s][wS[posShut[s-4]+k*4]],j,i+1,&flag_rejected));
      }
      for(j=1;j<=sizex2;j++) {
	cpl_vector_set(vec2,j-1,cpl_image_get(pp_shutters_raw[s][wI[k]],j,i+1,&flag_rejected));
      }


      /* area1=cpl_vector_get_sum(vec1); */
      /* area2=cpl_vector_get_sum(vec2); */
      /* ratio=area2/area1; */

      /* valmax1=cpl_vector_get_max(vec1); */
      /* for(j=0;j<sizex1;j++) { */
      /* 	/\* if (s==4 && i==480) printf("%f\n",cpl_vector_get(vec1,j)); *\/ */
      /* 	if (cpl_vector_get(vec1,j) < 0.05*valmax1) { */
      /* 	  cpl_vector_set(vec1,j,0.); */
      /* 	} */
      /* } */
      /* valmax2=cpl_vector_get_max(vec2); */
      /* for(j=0;j<sizex2;j++) { */
      /* 	/\* if (s==4 && i==480) printf("%f\n",cpl_vector_get(vec2,j)); *\/ */
      /* 	if (cpl_vector_get(vec2,j) < 0.05*valmax2) { */
      /* 	  cpl_vector_set(vec2,j,0.); */
      /* 	} */
      /* } */

      /* m0vec1=mat_vector_get_moment(vec1,0); */
      /* m1vec1=mat_vector_get_moment(vec1,1); */
      /* m2vec1=mat_vector_get_moment(vec1,2); */
      /* m0vec2=mat_vector_get_moment(vec2,0); */
      /* m1vec2=mat_vector_get_moment(vec2,1); */
      /* m2vec2=mat_vector_get_moment(vec2,2); */
      /* xc1=m1vec1/m0vec1; */
      /* xc2=m1vec2/m0vec2; */
      /* sigma1=sqrt((m2vec1/m0vec1)-xc1*xc1); */
      /* sigma2=sqrt((m2vec2/m0vec2)-xc2*xc2); */

      err1=cpl_vector_fit_gaussian(vecx1,NULL,vec1,NULL,CPL_FIT_ALL,&xc1,&sigma1,&area1,&offset,NULL,NULL,NULL);
      if (err1 != CPL_ERROR_NONE) {
      	cpl_msg_warning(cpl_func,"GAUSS FIT PHOTOMETRY - beam %d - row %d - ERR %d %d",s-3,i,err1, CPL_ERROR_SINGULAR_MATRIX);
      }
      err2=cpl_vector_fit_gaussian(vecx2,NULL,vec2,NULL,CPL_FIT_ALL,&xc2,&sigma2,&area2,&offset,NULL,NULL,NULL);
      if (s==4 && k==0)
	{
	  // Low resolution
	  if (yphot < 200)
	    {
	      aa=100-7;
	      if ( abs(i-aa) <= 7 )
		{
		  pixhigh+=xc2;
		}
	      if ( abs(i-12) <= 7 )
		{
		  pixlow+=xc2;
		}
	    }
	  else
	    {
	      aa=yphot-10-7;
	      if ( abs(i-aa) <= 7 )
		{
		  pixhigh+=xc2;
		}
	      if ( abs(i-12) <= 7 )
		{
		  pixlow+=xc2;
		}
	    }
	}
      
      if (err2 != CPL_ERROR_NONE) {
      	cpl_msg_warning(cpl_func,"GAUSS FIT INTERFEROMETRY - beam %d - row %d - ERR %d",s-3,i,err2);
      }

      
      if (sigma1 < 1.e-6) {
	sigma1=1.E-3;
      }
      if (area1 < 1.e-6) {
	area1=1.E-3;
      }
      zoom=sigma2/sigma1;
      if (zoom < 1.e-6) zoom=1.E-3;
      ratio=area2/area1;
      if (ratio > 20.) ratio=20.;
      if (ratio < 0.) ratio=0.;
      if (zoom > 20.) zoom=20.;

      if (isnan(ratio) != 0) {
	ratio=20.;
      }
      if (isnan(zoom) != 0) {
	zoom=20.;
      }
      
      
      
      cpl_matrix_set(p_kappamatrix->list_kappacoef[s-4+k*4]->coef,0,i,ratio);
      cpl_matrix_set(p_kappamatrix->list_kappacoef[s-4+k*4]->coef,1,i,zoom);
      cpl_matrix_set(p_kappamatrix->list_kappacoef[s-4+k*4]->coef,2,i,(xc1-sizex1/2.)+(xc2-sizex2/2.)/zoom);
      /* if (i==100) { */
      /* 	cpl_vector *ii=NULL; */
      /* 	cpl_vector *jj=NULL; */
      /* 	jj=cpl_vector_new(sizex2); */
      /* 	ii=cpl_vector_new(3); */
      /* 	cpl_vector_set(ii,0,ratio); */
      /* 	cpl_vector_set(ii,1,zoom); */
      /* 	cpl_vector_set(ii,2,(xc1-sizex1/2.)+(xc2-sizex2/2.)/zoom); */
      /* 	mat_apply_kappamatrix(vec1,ii,jj); */
      /* 	for(int kk=0;kk<sizex2;kk++) { */
      /* 	  printf("%f %f\n",cpl_vector_get(jj,kk),cpl_vector_get(vec2,kk)); */
      /* 	} */
      /* 	exit(1); */
      /* } */
      errPix1=sqrt(sigma1*sigma1/fabs(area1));
      errPix2=sqrt(sigma2*sigma2/fabs(area2));

      cpl_matrix_set(p_kappamatrix->list_kappacoef[s-4+k*4]->errcoef,0,i,
      		     ratio*((sqrt(fabs(cpl_vector_get(vec1,(int)xc1)))/area1)+(sqrt(fabs(cpl_vector_get(vec2,(int)xc2)))/area2))
      		     );
      cpl_matrix_set(p_kappamatrix->list_kappacoef[s-4+k*4]->errcoef,1,i,
      		     zoom*((errPix1/sigma1)+(errPix2/sigma2))
      		     );
      // See doc of mat_vector_fit_gaussian
      cpl_matrix_set(p_kappamatrix->list_kappacoef[s-4+k*4]->errcoef,2,i,sqrt(errPix1*errPix1+errPix2*errPix2));

    }
    cpl_vector_delete(vec1);
    cpl_vector_delete(vec2);
    cpl_vector_delete(vecx1);
    cpl_vector_delete(vecx2);
    


    /* if (s==4) { */
    /*   fclose(fp); */
    /* } */

      
      /* for(i=0; i<yphot; i++) { */
      /* 	cpl_matrix_set(p_aux, 0, i, 2.0); */
      /* 	cpl_matrix_set(p_aux, 1, i, 6.0); */
      /* 	cpl_matrix_set(p_aux, 2, i, 0.0); */
      /* } */
      /* for (y=0; y<yphot; y++) { */
      /* 	cpl_vector_set(p_param, 0, 2.0); */
      /* 	cpl_vector_set(p_param, 1, 6.0); */
      /* 	cpl_vector_set(p_param, 2, 2.0); */
      /* 	bias=0.; */
      /* 	for(i=0;i<10;i++) { */
      /* 	  bias+=cpl_image_get(pp_shutters_raw[s][wS[posShut[s-4]+k*4]], nphot-i, y+1,&flag_rejected); */
      /* 	} */
      /* 	bias/=10.; */
      /* 	for (i=0; i<nphot; i++) { */
      /* 	  cpl_vector_set(p_row_phot, i, cpl_image_get(pp_shutters_raw[s][wS[posShut[s-4]+k*4]], nphot-i, y+1,&flag_rejected)-bias); */
      /* 	} */
      /* 	bias=0.; */
      /* 	for(i=0;i<60;i++) { */
      /* 	  bias+=cpl_image_get(pp_shutters_raw[s][wI[k]],i+1, y+1,&flag_rejected); */
      /* 	} */
      /* 	bias/=60.; */
      /* 	for (i=0; i<nthreequarter; i++) { */
      /* 	  cpl_matrix_set(p_row_interf, i, 0, (-1.)*cpl_image_get(pp_shutters_raw[s][wI[k]],i+1+neight,y+1, &flag_rejected)+bias); */
      /* 	} */


      /* 	p_res = cpl_vector_new(nthreequarter); */
      /* 	p_matA = cpl_matrix_new(nthreequarter,3); */
      /* 	p_matB = cpl_matrix_new(nthreequarter,1); */
      /* 	p_vectB = cpl_vector_new(nthreequarter); */
      /* 	nbiter = 0; */



      /* 	do { */
      /* 	  derivative(p_row_phot, p_param, p_matA); */
      /* 	  interpolate(p_row_phot, p_param, p_vectB); */

      /* 	  for (j=0; j<nthreequarter; j++) { */
      /* 	    cpl_matrix_set(p_matB, j, 0,cpl_vector_get(p_vectB,j)); */
      /* 	  } */
      /* 	  cpl_matrix_add(p_matB, p_row_interf); */

      /* 	  p_mat_coef = cpl_matrix_solve_normal(p_matA, p_matB); */
      /* 	  for(j=0; j<3; j++) { */
      /* 	    cpl_vector_set(p_param, j,cpl_vector_get(p_param,j)-cpl_matrix_get(p_mat_coef, j, 0)); */
      /* 	  } */
	  
      /* 	  max=0.; */
      /* 	  for(j=1; j<3; j++) { */
      /* 	    if( fabs(cpl_matrix_get(p_aux, j, y)-cpl_vector_get(p_param, j)) > max ) { */
      /* 	      max = fabs(cpl_matrix_get(p_aux, j, y)-cpl_vector_get(p_param, j)); */
      /* 	    } */
      /* 	  } */
      /* 	  cpl_matrix_delete( p_mat_coef); */
      /* 	  nbiter++; */
      /* 	  cpl_matrix_set(p_aux, 0, y, cpl_vector_get(p_param, 0)); */
      /* 	  cpl_matrix_set(p_aux, 1, y, cpl_vector_get(p_param, 1)); */
      /* 	  cpl_matrix_set(p_aux, 2, y, cpl_vector_get(p_param, 2)); */
      /* 	} */
      /* 	while(max>0.01 && nbiter<200); */
	
      /* 	/\* if (y == 50 && s == 5) { *\/ */
      /* 	/\*   for (i=0; i<nthreequarter; i++) { *\/ */
      /* 	/\*     printf("%f %f\n",-1*cpl_matrix_get(p_row_interf, i, 0),cpl_vector_get(p_vectB,i)); *\/ */
      /* 	/\*   } *\/ */
      /* 	/\*   exit(1); *\/ */
      /* 	/\* } *\/ */
      /* 	if (nbiter>=200 || fabs(cpl_vector_get(p_param, 0)) > 10 || fabs(cpl_vector_get(p_param, 1)) > 10) { */
      /* 	  cpl_msg_info(cpl_func,"S %d parameters diverge ligne %d", s, y); */
      /* 	  cpl_matrix_set(p_aux, 0, y, 2.0); */
      /* 	  cpl_matrix_set(p_aux, 1, y, 6.0); */
      /* 	  cpl_matrix_set(p_aux, 2, y, 0.0); */
      /* 	} */

      /* 	// Error computation */
      /* 	p_matA_trans = cpl_matrix_transpose_create(p_matA); */
      /* 	normal = cpl_matrix_product_create(p_matA_trans,p_matA); */
      /* 	invnormal = cpl_matrix_invert_create(normal); */
      /* 	for(j=0;j<3;j++) { */
      /* 	  cpl_matrix_set(p_err,j,y,ERR_MAX_PIX*sqrt(cpl_matrix_get(invnormal,j,j))); */
      /* 	} */
      
      /* 	cpl_matrix_delete(p_matA_trans); */
      /* 	cpl_matrix_delete(normal); */
      /* 	cpl_matrix_delete(invnormal); */
      /* 	cpl_matrix_delete(p_matA); */
      /* 	cpl_matrix_delete(p_matB); */
      /* 	cpl_vector_delete(p_res); */
      /* 	cpl_vector_delete(p_vectB); */
	
      /* } */
      /* cpl_matrix_copy(p_kappamatrix->list_kappacoef[s-4+k*4]->coef, p_aux, 0, 0); */
      /* cpl_matrix_copy(p_kappamatrix->list_kappacoef[s-4+k*4]->errcoef, p_err, 0, 0); */
    }
  }
  pixhigh/=15;
  pixlow/=15;

// Compute the QC parameters
// Store the results in a cpl_propertylist
  qclist = cpl_propertylist_new();
  stdev_ratio = cpl_calloc(4*nbsubreg,sizeof(double));
  stdev_zoom =cpl_calloc(4*nbsubreg,sizeof(double));
  stdev_shift = cpl_calloc(4*nbsubreg,sizeof(double));
  mean_ratio = cpl_calloc(4*nbsubreg,sizeof(double));
  mean_zoom = cpl_calloc(4*nbsubreg,sizeof(double));
  mean_shift = cpl_calloc(4*nbsubreg,sizeof(double));
  flux = cpl_calloc(4*nbsubreg,sizeof(double));
  chipName = (char *)cpl_propertylist_get_string(pp_shutter1[0]->keywords,"ESO DET CHIP NAME");
  for(i=0; i<(*p_kappamatrix).nbbeam; i++) {
    flux[i] = 0.0;
    pmat_ratio = cpl_matrix_extract_row (p_kappamatrix->list_kappacoef[i]->coef,0);
    pmat_zoom = cpl_matrix_extract_row (p_kappamatrix->list_kappacoef[i]->coef,1);
    pmat_shift = cpl_matrix_extract_row (p_kappamatrix->list_kappacoef[i]->coef,2);
    mean_ratio[i] = cpl_matrix_get_mean (pmat_ratio);
    stdev_ratio[i] = cpl_matrix_get_stdev (pmat_ratio);
    mean_zoom[i] = cpl_matrix_get_mean (pmat_zoom);
    stdev_zoom[i] = cpl_matrix_get_stdev (pmat_zoom);
    mean_shift[i] = cpl_matrix_get_mean (pmat_shift);
    stdev_shift[i] = cpl_matrix_get_stdev (pmat_shift);

    if (isnan(mean_ratio[i]) != 0) {
      mean_ratio[i]=0;
    }
    if (isnan(mean_zoom[i]) != 0) {
      mean_zoom[i]=0;
    }
    if (isnan(mean_shift[i]) != 0) {
      mean_shift[i]=0;
    }
    if (isnan(stdev_ratio[i]) != 0) {
      stdev_ratio[i]=0;
    }
    if (isnan(stdev_zoom[i]) != 0) {
      stdev_zoom[i]=0;
    }
    if (isnan(stdev_shift[i]) != 0) {
      stdev_shift[i]=0;
    }
    cpl_matrix_delete(pmat_ratio);
    cpl_matrix_delete(pmat_zoom);
    cpl_matrix_delete(pmat_shift);
  }

  for(j=0; j<4; j++){
    flux[j]=0.;
    for(i=0; i<nbsubreg; i++) {
      val=cpl_image_get_flux (pp_shutters_raw[j+4][wS[posShut[j]]]);
      flux[j] = flux[j] + val;
    }
    
    if (isnan(flux[j]) != 0) {
      flux[j]=0.;
    }
  }
  for(i=0; i<4; i++) {
    if(!strcmp(chipName, "AQUARIUS")) {
      qcname_mean_shift = cpl_sprintf("ESO QC DET%d SHIFT%d MEAN", 2, i+1);
      qcname_stdev_shift = cpl_sprintf("ESO QC DET%d SHIFT%d STDEV", 2, i+1);
      qcname_mean_zoom = cpl_sprintf("ESO QC DET%d ZOOM%d MEAN", 2, i+1);
      qcname_stdev_zoom = cpl_sprintf("ESO QC DET%d ZOOM%d STDEV", 2, i+1);
      qcname_mean_iratio = cpl_sprintf("ESO QC DET%d IRATIO%d MEAN", 2, i+1);
      qcname_stdev_iratio = cpl_sprintf("ESO QC DET%d IRATIO%d STDEV", 2, i+1);
      qcname_flux = cpl_sprintf("ESO QC DET%d FLUX%d", 2, i+1);
    }
    if(!strcmp(chipName, "HAWAII-2RG")) {
      qcname_mean_shift = cpl_sprintf("ESO QC DET%d SHIFT%d MEAN", 1, i+1);
      qcname_stdev_shift = cpl_sprintf("ESO QC DET%d SHIFT%d STDEV", 1, i+1);
      qcname_mean_zoom = cpl_sprintf("ESO QC DET%d ZOOM%d MEAN", 1, i+1);
      qcname_stdev_zoom = cpl_sprintf("ESO QC DET%d ZOOM%d STDEV", 1, i+1);
      qcname_mean_iratio = cpl_sprintf("ESO QC DET%d IRATIO%d MEAN", 1, i+1);
      qcname_stdev_iratio = cpl_sprintf("ESO QC DET%d IRATIO%d STDEV", 1, i+1);
      qcname_flux = cpl_sprintf("ESO QC DET%d FLUX%d", 1, i+1);
    }
    if ((*p_kappamatrix).nbbeam==4){
      cpl_propertylist_append_double(qclist, qcname_mean_iratio, mean_ratio[i]);
      cpl_propertylist_append_double(qclist, qcname_mean_zoom, mean_zoom[i]);
      cpl_propertylist_append_double(qclist, qcname_mean_shift, mean_shift[i]);
      cpl_propertylist_append_double(qclist, qcname_stdev_iratio, stdev_ratio[i]);
      cpl_propertylist_append_double(qclist, qcname_stdev_zoom, stdev_zoom[i]);
      cpl_propertylist_append_double(qclist, qcname_stdev_shift, stdev_shift[i]);
      cpl_propertylist_append_double(qclist, qcname_flux, flux[i]);
    }
    else{
      mean_ratio_m = 0;
      mean_zoom_m = 0;
      mean_shift_m = 0;
      stdev_ratio_m = 0;
      stdev_zoom_m = 0;
      stdev_shift_m = 0;
      k=0;
      while(k<nbsubreg){
	mean_ratio_m = mean_ratio_m + mean_ratio[i+k*4];
	mean_shift_m = mean_shift_m+mean_shift[i+k*4];
	mean_zoom_m = mean_zoom_m+mean_zoom[i+k*4];
	stdev_ratio_m = stdev_ratio_m+stdev_ratio[i+k*4];
	stdev_shift_m = stdev_shift_m+stdev_shift[i+k*4];
	stdev_zoom_m = stdev_zoom_m+stdev_zoom[i+k*4];
	k=k+1;
      }
      mean_ratio_m = mean_ratio_m/nbsubreg;
      mean_shift_m = mean_shift_m/nbsubreg;
      mean_zoom_m = mean_zoom_m/nbsubreg;
      stdev_ratio_m = (stdev_ratio_m*stdev_ratio_m)/nbsubreg;
      stdev_zoom_m = (stdev_zoom_m*stdev_zoom_m)/nbsubreg;
      stdev_shift_m = (stdev_shift_m*stdev_shift_m)/nbsubreg;
      cpl_propertylist_append_double(qclist, qcname_mean_iratio, mean_ratio_m);
      cpl_propertylist_append_double(qclist, qcname_mean_zoom, mean_zoom_m);
      cpl_propertylist_append_double(qclist, qcname_mean_shift, mean_shift_m);
      cpl_propertylist_append_double(qclist, qcname_stdev_iratio, stdev_ratio_m);
      cpl_propertylist_append_double(qclist, qcname_stdev_zoom, stdev_zoom_m);
      cpl_propertylist_append_double(qclist, qcname_stdev_shift, stdev_shift_m);
      cpl_propertylist_append_double(qclist, qcname_flux, flux[i]);
    }
    if (qcname_stdev_shift != NULL) cpl_free(qcname_stdev_shift);
    if (qcname_stdev_iratio != NULL) cpl_free(qcname_stdev_iratio);
    if (qcname_stdev_zoom != NULL) cpl_free(qcname_stdev_zoom);
    if (qcname_flux != NULL) cpl_free(qcname_flux);
    if (qcname_mean_shift != NULL) cpl_free(qcname_mean_shift);
    if (qcname_mean_iratio != NULL) cpl_free(qcname_mean_iratio);
    if (qcname_mean_zoom != NULL) cpl_free(qcname_mean_zoom);
  }

  mat_add_generic_qc(p_kappamatrix->keywords,qclist);
  cpl_propertylist_append_double(qclist,"ESO QC DELTA SHIFT",pixhigh-pixlow);
  
// Add QC parametr to the keyword of the primary header
  cpl_propertylist_append(p_kappamatrix->keywords,qclist);
  cpl_propertylist_delete(qclist);
 
  // Free Memory
  for (s=0;s<8;s++) {
    for(i=0; i<nbregion; i++) {
      cpl_image_delete(pp_shutters_raw[s][i]);
    }
    cpl_free(pp_shutters_raw[s]);
  }
  
  if (wI != NULL) cpl_free(wI);
  if (wS != NULL) cpl_free(wS);
  if (p_nbimg != NULL) cpl_free(p_nbimg);

  if (stdev_ratio != NULL) cpl_free(stdev_ratio);
  if (stdev_zoom != NULL) cpl_free(stdev_zoom);
  if (stdev_shift != NULL) cpl_free(stdev_shift);
  if (mean_ratio != NULL) cpl_free(mean_ratio);
  if (mean_zoom != NULL) cpl_free(mean_zoom);
  if (mean_shift != NULL) cpl_free(mean_shift);
  if (flux != NULL) cpl_free(flux);


  /* if (p_param != NULL) cpl_vector_delete(p_param); */
  /* if (p_row_phot != NULL) cpl_vector_delete(p_row_phot); */
  /* if (p_row_interf != NULL) cpl_matrix_delete(p_row_interf); */
  /* if (p_aux != NULL) cpl_matrix_delete(p_aux); */
  /* if (p_err != NULL) cpl_matrix_delete(p_err); */
  return p_kappamatrix;
}



