/* $Id: mat_obsflat.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_obsflat.c $
 */

#include <stdlib.h>
#include <stdio.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/

#include <string.h>

#include "mat_error.h"
#include "mat_obsflat.h"
#include "mat_flatfield.h"
#include "mat_frame.h"
#include "mat_utils.h"


#define MATISSE_OFM_FIRSTCOLUMN_NAME "TYPE"

static cpl_error_code load_region(const char* regionname, /*const*/ cpl_table* table, const int row, cpl_image* image);

/**
   @brief Creates an empty obsflat map.
   @param det     describes the detector
   @param imgdet  contains the IMAGING_DETECTOR binary table
   @return A newly created obsflat map or NULL on error

   Creates an empty obsflat map. The det argument is copyied into the obsflat map structure.

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
*/
mat_obsflat *mat_obsflat_new(mat_detector *det,
				  mat_imagingdetector *imgdet)
{
  mat_obsflat *ofm = NULL;
  int            r;

  mat_assert_value((det != NULL), CPL_ERROR_NULL_INPUT, NULL, "no mat_detector (det) argument given");
  mat_assert_value((imgdet != NULL), CPL_ERROR_NULL_INPUT, NULL, "no mat_imagingdetector (imgdet) argument given");
  mat_assert_value((imgdet->list_region != NULL), CPL_ERROR_NULL_INPUT, NULL, "no valid mat_imagingdetector (imgdet->list_region) argument given");
  ofm = (mat_obsflat *)cpl_calloc(1, sizeof(mat_obsflat));
  if (ofm == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield");
      return NULL;
    }
  ofm->det = mat_detector_duplicate(det);
  if (ofm->det == NULL)
    {
      cpl_msg_error(cpl_func, "could not duplicate mat_detector");
      mat_obsflat_delete(ofm);
      return NULL;
    }

  ofm->keywords = cpl_propertylist_new();
  if (ofm->keywords == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for cpl_propertylist");
      mat_obsflat_delete(ofm);
      return NULL;
    }
  ofm->imgdet = mat_imagingdetector_duplicate(imgdet);
  if (ofm->imgdet == NULL)
    {
      cpl_msg_error(cpl_func, "could not duplicate the mat_imagingdetector");
      mat_obsflat_delete(ofm);
      return NULL;
    }
  ofm->channelgain = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (ofm->channelgain == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel gain");
      mat_obsflat_delete(ofm);
      return NULL;
    }
  ofm->channeloffset = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (ofm->channeloffset == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel offset");
      mat_obsflat_delete(ofm);
      return NULL;
    }
  ofm->channelnoise = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (ofm->channelnoise == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel noise");
      mat_obsflat_delete(ofm);
      return NULL;
    }
  ofm->list_obsflat = (cpl_image **)cpl_calloc(imgdet->nbregion, sizeof(cpl_image *));
  if (ofm->list_obsflat == NULL)
    {
      cpl_msg_error(cpl_func, "could not create a cpl_image list (flatfield)");
      mat_obsflat_delete(ofm);
      return NULL;
    }
  ofm->list_bias = (cpl_image **)cpl_calloc(imgdet->nbregion, sizeof(cpl_image *));
  if (ofm->list_bias == NULL)
    {
      cpl_msg_error(cpl_func, "could not create a cpl_image list (error)");
      mat_obsflat_delete(ofm);
      return NULL;
    }
  for (r = 0; r < imgdet->nbregion; r++)
    {
      ofm->list_obsflat[r] = cpl_image_new(imgdet->list_region[r]->naxis[0], imgdet->list_region[r]->naxis[1], CPL_TYPE_DOUBLE);
      if (ofm->list_obsflat[r] == NULL)
	{
	  cpl_msg_error(cpl_func, "could not create a cpl_image for a flatfield region");
	  mat_obsflat_delete(ofm);
	  return NULL;
	}
      ofm->list_bias[r] = cpl_image_new(imgdet->list_region[r]->naxis[0], imgdet->list_region[r]->naxis[1], CPL_TYPE_DOUBLE);
      if (ofm->list_bias[r] == NULL)
	{
	  cpl_msg_error(cpl_func, "could not create a cpl_image for a stdev region");
	  mat_obsflat_delete(ofm);
	  return NULL;
	}
    }
  ofm->detectorgain = 1.0;
  ofm->detectornoise = 0.0;
  ofm->flatfieldstdev = 0.0;
  return ofm;
}

/**
   @brief Deletes a obsflat map.
   @param ofm the observation flatfield map which is deleted
   @return the cpl_error_code or CPL_ERROR_NONE

   A previously created observation flatfield map is deleted from memory.

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
   - CPL_ERROR_UNSPECIFIED if the internal data structure is corrupt (should not happen!)
*/
cpl_error_code mat_obsflat_delete(mat_obsflat *ofm)
{
  int         r;

  mat_assert((ofm != NULL), CPL_ERROR_NULL_INPUT, "no mat_flatfield (ofm) argument given");
  mat_assert(((ofm->list_obsflat == NULL) || (ofm->imgdet != NULL)), CPL_ERROR_UNSPECIFIED, "could not get the number of flatfield masks since ofm->imgdet is empty -> memory leak!");
  if (ofm->keywords != NULL)
    {
      cpl_propertylist_delete(ofm->keywords);
      ofm->keywords = NULL;
    }
  if (ofm->list_obsflat != NULL)
    {
      for (r = 0; r < ofm->imgdet->nbregion; r++)
	{
	  if (ofm->list_obsflat[r] != NULL)
	    {
	      cpl_image_delete(ofm->list_obsflat[r]);
	      ofm->list_obsflat[r] = NULL;
	    }
	}
      cpl_free(ofm->list_obsflat);
      ofm->list_obsflat = NULL;
    }
  if (ofm->list_bias != NULL)
    {
      for (r = 0; r < ofm->imgdet->nbregion; r++)
	{
	  if (ofm->list_bias[r] != NULL)
	    {
	      cpl_image_delete(ofm->list_bias[r]);
	      ofm->list_bias[r] = NULL;
	    }
	}
      cpl_free(ofm->list_bias);
      ofm->list_bias = NULL;
    }
  if (ofm->channelgain != NULL)
    {
      cpl_vector_delete(ofm->channelgain);
      ofm->channelgain = NULL;
    }
  if (ofm->channeloffset != NULL)
    {
      cpl_vector_delete(ofm->channeloffset);
      ofm->channeloffset = NULL;
    }
  if (ofm->channelnoise != NULL)
    {
      cpl_vector_delete(ofm->channelnoise);
      ofm->channelnoise = NULL;
    }
  if (ofm->imgdet != NULL)
    {
      mat_imagingdetector_delete(ofm->imgdet);
      ofm->imgdet = NULL;
    }
  if (ofm->det != NULL)
    {
      mat_detector_delete(ofm->det);
      ofm->det = NULL;
    }
  cpl_free(ofm);
  return CPL_ERROR_NONE;
}

/**
   @brief Stores a observation flatfield map in a FITS file
   @param ofm the observation flatfield map
   @param fname the name of the FITS file
   @param rname the name of the pipeline recipe
   @param parlist the propertylist containing the recipe parameters
   @param frameset the frameset of the recipe
   @return the cpl_error_code or CPL_ERROR_NONE

   The observation flatfield map is written to a FITS file.

   QC parameters:
   QC DETi GAIN  The global detector gain
   QC DETi RON  The global read-out noise of the detector
   QC DETi FFM STDEV  The standard deviation of the computed flatfield map

   QC DETi CHANNELj GAIN : The detector channel specific gain (in electrons per ADU, specific for a readout mode) will be used to detect detector and read-out electronics problems.
   QC DETi CHANNELj OFFSET : The offset for the detector channels.
   QC DETi CHANNELj RON : The read-out noise is given for each detector channel.

   i the detector number (HAWAII-2RG -> 1, Aquarius -> 2)
   j is the detector channel number (1 .. 32 for the L/M-Band, 1 .. 64 for the N-Band)

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
   - CPL_ERROR_UNSPECIFIED if it is not possible to fill the DFS header
   or store the bad pixel map in a file
*/
cpl_error_code mat_obsflat_save(mat_obsflat *ofm,
				     const char *fname,
				     const char *rname,
				     cpl_parameterlist *parlist,
				     cpl_frameset *frameset)
{
  int               c, cr;
  int               i = 0;
  int               arraysize = 0;
  cpl_frame         *ofmframe = NULL;
  cpl_propertylist  *plist = NULL;
  cpl_array         *image_array = NULL;
  cpl_array			*image_size = NULL;
  cpl_table         *obstable = NULL;
  double            *p_image_data = NULL;
  char              kwd[64];
  char              regionname[16];

  cpl_msg_info(cpl_func, "storing the flatfield map in file %s", fname);
  cpl_error_reset();
  cpl_ensure_code((ofm != NULL), CPL_ERROR_NULL_INPUT);
  /* Create product frame */
  ofmframe = cpl_frame_new();
  cpl_frame_set_filename(ofmframe, fname);
  cpl_frame_set_tag(ofmframe, MATISSE_OFM_PROCATG);
  cpl_frame_set_type(ofmframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(ofmframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(ofmframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_frame_delete(ofmframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  plist = cpl_propertylist_new();
  if(!cpl_frameset_is_empty(frameset)){
    /* Add DataFlow keywords */
    if (cpl_dfs_setup_product_header(plist, ofmframe, frameset, parlist,
				     rname, PACKAGE "/" PACKAGE_VERSION, "?Dictionary?", NULL) != CPL_ERROR_NONE) {
      cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_error_reset();
      cpl_propertylist_delete(plist);
      cpl_frame_delete(ofmframe);
      return CPL_ERROR_UNSPECIFIED;
    }
  } else {
    /**
     * Add the detector keyword
     */
    snprintf(kwd, 64, "ESO DET CHIP NAME");
    cpl_propertylist_append_string(plist,kwd , "AQ");
    ofm->det->nr = 2;
  }
  /* Add QC parameters in the header */
  /* readout mode is missing */
  snprintf(kwd, 64, "ESO QC DET%d GAIN%d", ofm->det->nr, ofm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, ofm->detectorgain);
  snprintf(kwd, 64, "ESO QC DET%d RON%d", ofm->det->nr, ofm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, ofm->detectornoise);
  cr = cpl_vector_get_size(ofm->channelgain);

  for (c = 0; c < cr; c++)
    {
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d GAIN%d", ofm->det->nr, c + 1, ofm->det->read_id);
      cpl_propertylist_append_double(plist, kwd, cpl_vector_get(ofm->channelgain, c));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d OFFSET%d", ofm->det->nr, c + 1, ofm->det->read_id);
      cpl_propertylist_append_double(plist, kwd, cpl_vector_get(ofm->channeloffset, c));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d RON%d", ofm->det->nr, c + 1, ofm->det->read_id);
      cpl_propertylist_append_double(plist, kwd, cpl_vector_get(ofm->channelnoise, c));
    }


  snprintf(kwd, 64, "ESO QC DET%d FFM STDEV%d", ofm->det->nr, ofm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, ofm->flatfieldstdev);

  // Add Generic QC parameters
  cpl_propertylist *qclist=NULL;
  qclist = cpl_propertylist_new();
  mat_add_generic_qc(plist,qclist);
  cpl_propertylist_append(plist,qclist);
  cpl_propertylist_delete(qclist);
  
  
  //write primary header
  cpl_propertylist_save(plist, fname, CPL_IO_CREATE);
  cpl_propertylist_delete(plist);

  //create propertylist for obsflat extension
  plist = cpl_propertylist_new();
  cpl_propertylist_append_string(plist, "EXTNAME", MATISSE_OFM_OBSFLAT_EXT);

  //create the binary table for the obsflat and fill it with the values
  obstable = cpl_table_new(2);
  cpl_table_new_column(obstable,MATISSE_OFM_FIRSTCOLUMN_NAME, CPL_TYPE_STRING);
  cpl_table_set_string(obstable, MATISSE_OFM_FIRSTCOLUMN_NAME, 0, "F");
  cpl_table_set_string(obstable, MATISSE_OFM_FIRSTCOLUMN_NAME, 1, "B");
  image_size = cpl_array_new(2, CPL_TYPE_INT);

  for(i = 0; i < ofm->imgdet->nbregion; ++i){

    arraysize = ofm->imgdet->list_region[i]->naxis[0]*ofm->imgdet->list_region[i]->naxis[1];
    snprintf(regionname, 16, "DATA%d", i+1);
    cpl_table_new_column_array(obstable,regionname, CPL_TYPE_DOUBLE, arraysize);
    cpl_array_set_int(image_size, 0, ofm->imgdet->list_region[i]->naxis[0]);
    cpl_array_set_int(image_size, 1, ofm->imgdet->list_region[i]->naxis[1]);
    cpl_table_set_column_dimensions(obstable, regionname,image_size);

    p_image_data = cpl_image_get_data_double(ofm->list_obsflat[i]);
    image_array = cpl_array_wrap_double(p_image_data, arraysize);
    if(image_array == NULL){
      cpl_msg_error(cpl_func, "Could not wrap array to save, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_propertylist_delete(plist);
      cpl_frame_delete(ofmframe);
      return CPL_ERROR_UNSPECIFIED;

    }
    if(cpl_table_set_array(obstable,regionname, 0, image_array ) != CPL_ERROR_NONE){
      cpl_msg_error(cpl_func, "Could not write array to table, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_propertylist_delete(plist);
      cpl_frame_delete(ofmframe);
      return CPL_ERROR_UNSPECIFIED;
    }

    cpl_array_unwrap(image_array);
    if(cpl_error_get_code() != CPL_ERROR_NONE){
      cpl_msg_error(cpl_func, "Could not unwrap obsflat array to save, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_propertylist_delete(plist);
      cpl_frame_delete(ofmframe);
      return CPL_ERROR_UNSPECIFIED;
    }

    p_image_data = cpl_image_get_data_double(ofm->list_bias[i]);
    image_array = cpl_array_wrap_double(p_image_data, arraysize);
    if(image_array == NULL){
      cpl_msg_error(cpl_func, "Could not wrap array to save, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_propertylist_delete(plist);
      cpl_frame_delete(ofmframe);
      return CPL_ERROR_UNSPECIFIED;

    }

    if(cpl_table_set_array(obstable,regionname, 1, image_array ) != CPL_ERROR_NONE){
      cpl_msg_error(cpl_func, "Could not write array to table, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_propertylist_delete(plist);
      cpl_frame_delete(ofmframe);
      return CPL_ERROR_UNSPECIFIED;
    }

    cpl_array_unwrap(image_array);
    if(cpl_error_get_code() != CPL_ERROR_NONE){
      cpl_msg_error(cpl_func, "Could not unwrap obsbias array to save, code = %d, message = %s",
		    cpl_error_get_code(),
		    cpl_error_get_message());
      cpl_propertylist_delete(plist);
      cpl_frame_delete(ofmframe);
      return CPL_ERROR_UNSPECIFIED;
    }
  }

  /* Save the file */
  if(cpl_table_save(obstable, NULL, plist, fname,CPL_IO_EXTEND) != CPL_ERROR_NONE){
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_frame_delete(ofmframe);
    return CPL_ERROR_UNSPECIFIED;
  }

  //clear the used memory
  cpl_table_delete(obstable);
  cpl_propertylist_delete(plist);
  cpl_array_delete(image_size);

  //append the other extensions to the file
  mat_imagingdetector_save(ofm->imgdet, NULL, fname);
  /* Log the saved file in the input frameset */
  cpl_frameset_insert(frameset, ofmframe);
  //cpl_frame_delete(ofmframe);
  return CPL_ERROR_NONE;
}

/**
   @brief Loads a observation flatfield map in a FITS file
   @param ofmframe the observation flatfield map
   @return the new allocated and filles mat_obsflat structure or NULL

   The observation flatfield map is loaded from a FITS.

*/
mat_obsflat *mat_obsflat_load(cpl_frame *ofmframe)
{
  int i = 0;
  mat_detector     *det = NULL;
  mat_imagingdetector   *imdetector = NULL;
  cpl_propertylist *plist = NULL;
  cpl_propertylist *imaging_detector_plist = NULL;
  cpl_image        *ofm_image = NULL;
  cpl_image        *bias_image = NULL;
  mat_obsflat    *ofm = NULL;
  int nextensions = 0;
  int extnumber = -1;
  int               cr, c;
  cpl_table         *obsflattable = NULL;
  cpl_table         *imaging_detector_table = NULL;
  char              kwd[64];
  char              regionname[16];

  cpl_msg_info(cpl_func, "load the flatfield map from file %s", cpl_frame_get_filename(ofmframe));
  /* create a mat_detector structure, load the primary keywords and decode the detector */
  det = mat_detector_new();
  if (det == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield->det");
      return NULL;
    }
  plist = cpl_propertylist_load(cpl_frame_get_filename(ofmframe), 0);
  if (plist == NULL)
    {
      cpl_msg_error(cpl_func, "could not load the primary keywords");
      mat_detector_delete(det);
      return NULL;
    }
  mat_detector_decode(det, plist);

  //load the imaging detector
  extnumber = cpl_fits_find_extension(cpl_frame_get_filename(ofmframe), "IMAGING_DETECTOR");
  if(extnumber < 1){
    //error or not found
    cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield");
    cpl_propertylist_delete(plist);
    return NULL;
  }

  imaging_detector_plist = cpl_propertylist_load(cpl_frame_get_filename(ofmframe), extnumber);
  if(imaging_detector_plist == NULL){
    cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield");
    cpl_propertylist_delete(plist);
    return NULL;
  }
  imaging_detector_table = cpl_table_load(cpl_frame_get_filename(ofmframe), extnumber, 0);
  if(imaging_detector_table == NULL){
    cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield");
    cpl_propertylist_delete(plist);
    cpl_propertylist_delete(imaging_detector_plist);
    return NULL;

  }
  imdetector = mat_imagingdetector_from_table(imaging_detector_plist, imaging_detector_table);
  if(imdetector == NULL){
    cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield");
    cpl_propertylist_delete(plist);
    cpl_propertylist_delete(imaging_detector_plist);
    return NULL;
  }
  cpl_propertylist_delete(imaging_detector_plist);
  /* create and initialize the mat_flatfield structure */
  ofm = mat_obsflat_new(det, imdetector); /* no mat_imagingdetector means the whole detector */
  mat_detector_delete(det); /* we can delete the mat_detector, because we have a copy in ofm */
  mat_imagingdetector_delete(imdetector);/* we can delete the mat_detector, because we have a copy in ofm */
  det = NULL;
  if (ofm == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield");
      cpl_propertylist_delete(plist);
      return NULL;
    }
  /* move the propertylist into the ofm */
  cpl_propertylist_delete(ofm->keywords);
  ofm->keywords = plist;
  /* get all important keyword values from the propertylist */
  snprintf(kwd, 64, "ESO QC DET%d GAIN%d", ofm->det->nr, ofm->det->read_id);
  ofm->detectorgain = cpl_propertylist_get_double(ofm->keywords, kwd);
  snprintf(kwd, 64, "ESO QC DET%d RON%d", ofm->det->nr, ofm->det->read_id);
  ofm->detectornoise = cpl_propertylist_get_double(ofm->keywords, kwd);
  snprintf(kwd, 64, "ESO QC DET%d FFM STDEV%d", ofm->det->nr, ofm->det->read_id);
  ofm->flatfieldstdev = cpl_propertylist_get_double(ofm->keywords, kwd);
  cr = cpl_vector_get_size(ofm->channelgain);
  for (c = 0; c < cr; c++)
    {
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d GAIN%d", ofm->det->nr, c + 1, ofm->det->read_id);
      cpl_vector_set(ofm->channelgain, c, cpl_propertylist_get_double(ofm->keywords, kwd));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d OFFSET%d", ofm->det->nr, c + 1, ofm->det->read_id);
      cpl_vector_set(ofm->channeloffset, c, cpl_propertylist_get_double(ofm->keywords, kwd));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d RON%d", ofm->det->nr, c + 1, ofm->det->read_id);
      cpl_vector_set(ofm->channelnoise, c, cpl_propertylist_get_double(ofm->keywords, kwd));
    }

  //load the imaging detector
  nextensions = cpl_fits_count_extensions(cpl_frame_get_filename(ofmframe));
  if(nextensions <= 1){
    cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield");
    //      cpl_propertylist_delete(plist);
    mat_obsflat_delete(ofm);
    return NULL;
  }

  extnumber = cpl_fits_find_extension(cpl_frame_get_filename(ofmframe), MATISSE_OFM_OBSFLAT_EXT);
  if(extnumber < 1){
    //error or not found
    cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield");
    //      cpl_propertylist_delete(plist);
    mat_obsflat_delete(ofm);
    return NULL;
  }
  obsflattable = cpl_table_load(cpl_frame_get_filename(ofmframe), extnumber, 0);

  if(obsflattable == NULL){
    cpl_msg_error(cpl_func, "could not load obsflat binary table");
    //      cpl_propertylist_delete(plist);
    mat_obsflat_delete(ofm);
    return NULL;
  }

  for(i = 0; i < ofm->imgdet->nbregion; ++i){
    /*collect information about the dimensions of a column*/
    snprintf(regionname, 16, "DATA%d", i+1);
    ofm_image = ofm->list_obsflat[i];
    if(load_region(regionname, obsflattable, 0,  ofm_image) != CPL_ERROR_NONE){
      cpl_msg_error(cpl_func, "could not load obsflat binary table");
      //		cpl_propertylist_delete(plist);
      cpl_table_delete(obsflattable);
      mat_obsflat_delete(ofm);
      return NULL;
    }

    bias_image = ofm->list_bias[i];
    if(load_region(regionname, obsflattable, 1,  bias_image) != CPL_ERROR_NONE){
      cpl_msg_error(cpl_func, "could not load obsflat binary table");
      //		cpl_propertylist_delete(plist);
      cpl_table_delete(obsflattable);
      mat_obsflat_delete(ofm);
      return NULL;
    }
  }
  cpl_table_delete(obsflattable);
  cpl_table_delete(imaging_detector_table);

  return ofm;
}

/**
   @brief Loads a region from a cpl table into a cpl_image
   @param regionname the name of the region to be loaded into the image
   @param table the cpl_table containing the region which should be loaded
   @param row the row number in the table
   @param image pointer to a preallocated cpl_image of the same size and type as the region data
   @return the cpl_error_code or CPL_ERROR_NONE


   This is a helper method which copies the data of a cpl_table cell, specified by the regionname (column) and the
   given row number, into a preallocated cpl_image. The dimensions and the type of the image has to match the
   cells dimensions. The image has to be preallocated by using mat_obsflat_new(det, imdetector);


   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
   - CPL_ERROR_UNSPECIFIED if the region doesn't exist or the dimensions didn't match

*/
static cpl_error_code load_region(const char* regionname, /*const*/ cpl_table* table, const int row, cpl_image* image){
  cpl_image        *wraped_image = NULL;
  cpl_image        *ofm_image = NULL;
  double            *p_array_data = NULL;
  cpl_array         **rows_singleregion = NULL;
  cpl_array         *region = NULL;
  int nrows = 0;
  int coldepth = 0;
  int coldimx = 0;
  int coldimy = 0;
  /* int rejected; */

  //cpl_msg_info(cpl_func, "load_region(%s, .., %d, ..)", regionname, row);
  coldepth = cpl_table_get_column_depth(table, regionname);
  if(coldepth !=0){
    coldimx = cpl_table_get_column_dimension(table, regionname, 0);
    coldimy = cpl_table_get_column_dimension(table, regionname, 1);
  }
  nrows = cpl_table_get_nrow(table);

  rows_singleregion = cpl_table_get_data_array(table, regionname);
  if(rows_singleregion == NULL){
    cpl_msg_error(cpl_func, "could not load region from obsflattable");
    return CPL_ERROR_UNSPECIFIED;
  }

  if(row < nrows){
    region = rows_singleregion[row];
  }

  if(region == NULL){
    cpl_msg_error(cpl_func, "can't get array row %d, region %s from table", row, regionname);
    return CPL_ERROR_UNSPECIFIED;
  }


  ofm_image = image;
  if(cpl_image_get_size_x(ofm_image) != coldimx ||
     cpl_image_get_size_y(ofm_image) != coldimy){
    cpl_msg_error(cpl_func, "image dimensions are not equal");
    return CPL_ERROR_UNSPECIFIED;
  }
  p_array_data = cpl_array_get_data_double(region);
  if(p_array_data == NULL){
    cpl_msg_error(cpl_func, "can't get array data from table");
    return CPL_ERROR_UNSPECIFIED;
  }
  wraped_image = cpl_image_wrap_double(coldimx, coldimy,p_array_data);
  if(wraped_image == NULL){
    cpl_msg_error(cpl_func, "can't wrap image");
    return CPL_ERROR_UNSPECIFIED;
  }

  if(cpl_image_copy(ofm_image, wraped_image, 1, 1) != CPL_ERROR_NONE){
    cpl_msg_error(cpl_func, "can't copy image");
    return CPL_ERROR_UNSPECIFIED;
  }
  /*
  cpl_msg_info(cpl_func, "   cpl_image_get(image, %d, %d, ...) = %g",
	       (int)cpl_image_get_size_x(image)/2,
	       (int)cpl_image_get_size_y(image)/2,
	       cpl_image_get(image, cpl_image_get_size_x(image)/2, cpl_image_get_size_y(image)/2, &rejected));
  */
  cpl_image_unwrap(wraped_image);
  return CPL_ERROR_NONE;
}

/**
   @brief This function remaps a OBS_FLATFIELD map to a given sub-window layout.
   @param ofm    The original OBS_FLATFIELD map, usually with only one spectral band recorded in SiPhot mode (so 21 sub-windows).
   @param imgdet This structure describes the sub-window layout.
   @returns The mapped OBS_FALTFIELD map or NULL.

   This function uses the sub-window layout of the imgdet argument to create
   a new OBS_FLATFIELD map (using the function mat_obsflat_new and the layout in imgdet).
   In a second step, the sub-window specific parts of the obsflat map are extracted
   and stored in the newly created obsflat map by using a temporarily unmapped obsflat.
 */
mat_obsflat *mat_obsflat_map(mat_obsflat *ofm, mat_imagingdetector *imgdet)
{
  mat_obsflat *nofm = NULL;
  cpl_image   *dflat = NULL;
  cpl_image   *dbias = NULL;
  int          r, i;
  /* int          rejected; */
  double      *pflat;
  double      *pbias;

  if (ofm == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_obsflat (ofm) argument given");
      return NULL;
    }
  if (ofm->det == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (ofm->det) argument given");
      return NULL;
    }
  if (imgdet == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_imagingdetector (imgdet) argument given");
      return NULL;
    }
  if (imgdet->list_region == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no valid mat_imagingdetector (imgdet->list_region) argument given");
      return NULL;
    }
  // create the new obsflat according to the requested sub-window setup
  nofm = mat_obsflat_new(ofm->det, imgdet);
  if (nofm == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_obsflat");
      return NULL;
    }
  /* put a copy of the propertylist into the newly created obsflat map */
  if (nofm->keywords != NULL)
    {
      cpl_propertylist_delete(nofm->keywords);
      nofm->keywords = NULL;
      nofm->keywords = cpl_propertylist_duplicate(ofm->keywords);
    }
  /* copy the QC parameters into the new obsflat map */
  nofm->detectorgain = ofm->detectorgain;
  nofm->detectornoise = ofm->detectornoise;
  nofm->flatfieldstdev = ofm->flatfieldstdev;
  cpl_vector_copy(nofm->channelgain, ofm->channelgain);
  cpl_vector_copy(nofm->channeloffset, ofm->channeloffset);
  cpl_vector_copy(nofm->channelnoise, ofm->channelnoise);
  // allocate and initialize the temporary unmapped flat and bias images
  dflat = cpl_image_new(ofm->det->nx, ofm->det->ny, CPL_TYPE_DOUBLE);
  if (dflat == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "could not allocate temporary unmapped obsflat");
      mat_obsflat_delete(nofm);
      return NULL;
    }
  dbias = cpl_image_new(ofm->det->nx, ofm->det->ny, CPL_TYPE_DOUBLE);
  if (dbias == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "could not allocate temporary unmapped obsbias");
      mat_obsflat_delete(nofm);
      cpl_image_delete(dflat);
      return NULL;
    }
  pflat = cpl_image_get_data_double(dflat);
  pbias = cpl_image_get_data_double(dbias);
  for (i = 0; i < ofm->det->nx*ofm->det->ny; i++)
    {
      *pflat++ = 1.0;
      *pbias++ = 0.0;
    }
  // unmap the original obsflat (flatfield and bias) to the temporary images
  for (r = 0; r < ofm->imgdet->nbregion; r++)
    {
      mat_region *reg = ofm->imgdet->list_region[r];
      /*
      cpl_msg_info(cpl_func, "  original flat[%d,%d] = %g",
		   (int)cpl_image_get_size_x(ofm->list_obsflat[r])/2,
		   (int)cpl_image_get_size_y(ofm->list_obsflat[r])/2,
		   cpl_image_get(ofm->list_obsflat[r], cpl_image_get_size_x(ofm->list_obsflat[r])/2, cpl_image_get_size_y(ofm->list_obsflat[r])/2, &rejected));
      */
      cpl_image_copy(dflat, ofm->list_obsflat[r], reg->corner[0], reg->corner[1]);
      /*
      cpl_msg_info(cpl_func, "  original bias[%d,%d] = %g",
		   (int)cpl_image_get_size_x(ofm->list_bias[r])/2,
		   (int)cpl_image_get_size_y(ofm->list_bias[r])/2,
		   cpl_image_get(ofm->list_bias[r], cpl_image_get_size_x(ofm->list_bias[r])/2, cpl_image_get_size_y(ofm->list_bias[r])/2, &rejected));
      */
      cpl_image_copy(dbias, ofm->list_bias[r], reg->corner[0], reg->corner[1]);
    }
  /* extract only the sub-window equivalents of the obsflat map */
  for (r = 0; r < nofm->imgdet->nbregion; r++)
    {
      mat_region *reg = nofm->imgdet->list_region[r];
      int         llx = reg->corner[0];
      int         lly = reg->corner[1];
      int         urx = reg->corner[0] + reg->naxis[0] - 1;
      int         ury = reg->corner[1] + reg->naxis[1] - 1;
      if (nofm->list_obsflat[r] != NULL)
	{
	  cpl_image_delete(nofm->list_obsflat[r]);
	  nofm->list_obsflat[r] = NULL;
	}
      nofm->list_obsflat[r] = cpl_image_extract(dflat, llx, lly, urx, ury);
      /*
      cpl_msg_info(cpl_func, "  remapped flat[%d,%d] = %g",
		   (int)cpl_image_get_size_x(nofm->list_obsflat[r])/2,
		   (int)cpl_image_get_size_y(nofm->list_obsflat[r])/2,
		   cpl_image_get(nofm->list_obsflat[r], cpl_image_get_size_x(nofm->list_obsflat[r])/2, cpl_image_get_size_y(nofm->list_obsflat[r])/2, &rejected));
      */
      if (nofm->list_bias[r] != NULL)
	{
	  cpl_image_delete(nofm->list_bias[r]);
	  nofm->list_bias[r] = NULL;
	}
      nofm->list_bias[r] = cpl_image_extract(dbias, llx, lly, urx, ury);
      /*
      cpl_msg_info(cpl_func, "  remapped bias[%d,%d] = %g",
		   (int)cpl_image_get_size_x(nofm->list_bias[r])/2,
		   (int)cpl_image_get_size_y(nofm->list_bias[r])/2,
		   cpl_image_get(nofm->list_bias[r], cpl_image_get_size_x(nofm->list_bias[r])/2, cpl_image_get_size_y(nofm->list_bias[r])/2, &rejected));
      */
    }
  cpl_image_delete(dflat);
  cpl_image_delete(dbias);
  return nofm;
}
