/* $Id: mat_oifits_init_from_corrflux.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oifits_init_from_corrflux.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#include <string.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/
#include <math.h>
#include "hdrl.h"
#include "mat_oifits_init_from_corrflux.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
  Define
  ----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/


/**
   @brief Transfrom mjd in local sideral time
   @param longitude           contains the longitude in degree
   @param mjd                 contains the mjd
   @return lst in hours 
*/
// longitude in degrees, lst in hours
double mjd2lst(double longitude, double mjd) {
    double lst;
    double c[4]={280.46061837,360.98564736629,0.000387933,38710000.0};
    double mjd2000=51544.5;
    double t0;
    double t;
    double theta;
    double decimal=0.;

    t0=mjd-mjd2000;
    t=t0/36535.;
    theta=c[0]+(c[1]*t0)+pow(t,2.0)*(c[2]-t/c[3]);
    lst=(theta+longitude)/15.0;
    if ( lst < 0.) {
	decimal=lst-(int)lst;
	lst=24.0+((int)lst % 24)+decimal;
    }
    decimal=lst-(int)lst;
    lst=((int)lst % 24)+decimal;

    return lst;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup opdwvpo
   @brief Fill the uv coordinates of the mat_oivis2 part of a 
   mat_oifits stucture
   @param oifits             contains the mat_oifits structure
   @return 0 if no error 
*/
/*----------------------------------------------------------------------------*/
int mat_oivis2_filluv(mat_oifits *oifits)
{
  int i=0;
  int j=0;
  double lst=0.;
  double obsLat;
  double obsLon;
  double base[3]={0.,0.,0.};
  double decl;
  double ah=0.;
  double xyz1[3];
  double xyz2[3];

  obsLat = cpl_propertylist_get_double(oifits->keywords, "ESO ISS GEOLAT");
  obsLat *= CPL_MATH_PI/180.0;
  obsLon = cpl_propertylist_get_double(oifits->keywords, "ESO ISS GEOLON");
  decl = oifits->oitarget->decep0*CPL_MATH_PI/180.;
  for (i = 0; i < oifits->oivis2->nbvis2; i++)
    {
      lst = mjd2lst(obsLon,oifits->oivis2->list_vis2[i]->dateobsmjd);//time);
      ah = ((lst*15.)-oifits->oitarget->raep0)*CPL_MATH_PI/180.;
      mat_get_staxyz_from_staindex(oifits->array,oifits->oivis2->list_vis2[i]->stationindex[0],xyz1);
      mat_get_staxyz_from_staindex(oifits->array,oifits->oivis2->list_vis2[i]->stationindex[1],xyz2);
      for (j = 0; j < 3; j++)
	{
	  base[j]=xyz1[j]-xyz2[j];
	    /* oifits->array->list_station[oifits->oivis2->list_vis2[i]->stationindex[0] - 1]->staxyz[j] */
	    /* - */
	    /* oifits->array->list_station[oifits->oivis2->list_vis2[i]->stationindex[1] - 1]->staxyz[j]; */
	}
      oifits->oivis2->list_vis2[i]->ucoord =
	-(base[0]*cos(ah))+(base[1]*sin(obsLat)*sin(ah))
	-
	(base[2]*cos(obsLat)*sin(ah));
      oifits->oivis2->list_vis2[i]->vcoord =
	-(base[0]*sin(decl)*sin(ah))
	-
	(base[1]*((sin(obsLat)*sin(decl)*cos(ah))+(cos(obsLat)*cos(decl))))
	+
	(base[2]*((cos(obsLat)*sin(decl)*cos(ah))-(sin(obsLat)*cos(decl))));
    }
  return 0;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup opdwvpo
   @brief Fill the uv coordinates of the mat_oivis part of a 
   mat_oifits stucture
   @param oifits             contains the mat_oifits structure
   @return 0 if no error 
*/
/*----------------------------------------------------------------------------*/
int mat_oivis_filluv(mat_oifits *oifits) {
    int i=0;
    int j=0;
    double lst=0.;
    double obsLat;
    double obsLon;
    double base[3]={0.,0.,0.};
    double decl;
    double ah=0.;
    double xyz1[3];
    double xyz2[3];

    obsLat=cpl_propertylist_get_double(oifits->keywords,
				       "ESO ISS GEOLAT");
    obsLat*=CPL_MATH_PI/180.0;
    obsLon=cpl_propertylist_get_double(oifits->keywords,
				       "ESO ISS GEOLON");
    decl=oifits->oitarget->decep0*CPL_MATH_PI/180.;
    for(i=0;i<oifits->oivis->nbvis;i++) {
      lst=mjd2lst(obsLon,oifits->oivis->list_vis[i]->dateobsmjd);//time);
      ah=((lst*15.)-oifits->oitarget->raep0)*CPL_MATH_PI/180.;
      mat_get_staxyz_from_staindex(oifits->array,oifits->oivis->list_vis[i]->stationindex[0],xyz1);
      mat_get_staxyz_from_staindex(oifits->array,oifits->oivis->list_vis[i]->stationindex[1],xyz2);
      for(j=0;j<3;j++){
	base[j]=xyz1[j]-xyz2[j];
	/* oifits->array-> */
	/* list_station[oifits->oivis->list_vis[i]->stationindex[0]-1]-> */
	/* staxyz[j]-oifits->array-> */
	/* list_station[oifits->oivis->list_vis[i]->stationindex[1]-1]-> */
	/* staxyz[j]; */
      }
      oifits->oivis->list_vis[i]->ucoord=
	-(base[0]*cos(ah))+(base[1]*sin(obsLat)*sin(ah))-
	(base[2]*cos(obsLat)*sin(ah));
      oifits->oivis->list_vis[i]->vcoord=-(base[0]*sin(decl)*sin(ah))-
	(base[1]*((sin(obsLat)*sin(decl)*cos(ah))+(cos(obsLat)*cos(decl))))+
	(base[2]*((cos(obsLat)*sin(decl)*cos(ah))-(sin(obsLat)*cos(decl))));
    }
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
   @ingroup opdwvpo
   @brief Fill the uv coordinates of the mat_oit3 part of a 
   mat_oifits stucture
   @param oifits             contains the mat_oifits structure
   @return 0 if no error 
*/
/*----------------------------------------------------------------------------*/
int mat_oit3_filluv(mat_oifits *oifits)
{
  int i=0;
  int j=0;
  double lst=0.;
  double obsLat;
  double obsLon;
  double base[3]={0.,0.,0.};
  double decl;
  double ah=0.;
  double xyz1[3];
  double xyz2[3];
  double xyz3[3];

    obsLat = cpl_propertylist_get_double(oifits->keywords, "ESO ISS GEOLAT");
    obsLat *= CPL_MATH_PI/180.0;
    obsLon = cpl_propertylist_get_double(oifits->keywords, "ESO ISS GEOLON");
    decl = oifits->oitarget->decep0*CPL_MATH_PI/180.;
    for (i = 0; i < oifits->oit3->nbt3; i++)
      {
	lst = mjd2lst(obsLon,oifits->oivis2->list_vis2[i]->dateobsmjd);//time);
	ah = ((lst*15.)-oifits->oitarget->raep0)*CPL_MATH_PI/180.;
	mat_get_staxyz_from_staindex(oifits->array,oifits->oit3->list_t3[i]->stationindex[0],xyz1);
	mat_get_staxyz_from_staindex(oifits->array,oifits->oit3->list_t3[i]->stationindex[1],xyz2);
	mat_get_staxyz_from_staindex(oifits->array,oifits->oit3->list_t3[i]->stationindex[2],xyz3);
	for (j = 0; j < 3; j++)
	  {
	    base[j] = xyz1[j]-xyz2[j];
	      /* oifits->array->list_station[oifits->oit3->list_t3[i]->stationindex[0] -1 ]->staxyz[j] */
	      /* - */
	      /* oifits->array->list_station[oifits->oit3->list_t3[i]->stationindex[1] -1 ]->staxyz[j]; */
	  }
	oifits->oit3->list_t3[i]->u1coord = 
	  -(base[0]*cos(ah))+(base[1]*sin(obsLat)*sin(ah))
	  -
	  (base[2]*cos(obsLat)*sin(ah));
	oifits->oit3->list_t3[i]->v1coord =
	  -(base[0]*sin(decl)*sin(ah))
	  -
	  (base[1]*((sin(obsLat)*sin(decl)*cos(ah))+(cos(obsLat)*cos(decl))))
	  +
	  (base[2]*((cos(obsLat)*sin(decl)*cos(ah))-(sin(obsLat)*cos(decl))));
	for (j = 0; j < 3; j++)
	  {
	    base[j] = xyz2[j]-xyz3[j];
	      /* oifits->array->list_station[oifits->oit3->list_t3[i]->stationindex[1] - 1]->staxyz[j] */
	      /* - */
	      /* oifits->array->list_station[oifits->oit3->list_t3[i]->stationindex[2] - 1]->staxyz[j]; */
	}
	oifits->oit3->list_t3[i]->u2coord =
	  -(base[0]*cos(ah))+(base[1]*sin(obsLat)*sin(ah))
	  -
	  (base[2]*cos(obsLat)*sin(ah));
	oifits->oit3->list_t3[i]->v2coord =
	  -(base[0]*sin(decl)*sin(ah))
	  -
	  (base[1]*((sin(obsLat)*sin(decl)*cos(ah))+(cos(obsLat)*cos(decl))))
	  +
	  (base[2]*((cos(obsLat)*sin(decl)*cos(ah))-(sin(obsLat)*cos(decl))));
    }
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
   @brief Initialize a mat_oifits structure from a mat_corrflux structure
   @param *corrFlux           contains the mat_corrflux structure 
   @param *oifits             contains the mat_oifits structure
   @return 0 if no error 
*/
int mat_oifits_init_from_corrflux(mat_corrflux *corrFlux,mat_oifits *oifits) {
#include "mat_station_default.h"
    char insname[]="MATISSE";
    char arrayname[]="VLTI";
    char cooframe[]="GEOCENTRIC";
    double arrayx=1951952.;
    double arrayy=-5483173.;
    double arrayz=-2668147.;
    const char *charFoo=NULL;
    char keyName[100]="";
    /* char *bcd1Status=NULL; */
    /* char *bcd2Status=NULL; */
    char *bcdMode=NULL;
    int j=0;
    int k=0;
    int cpt=0;
    double val;
    double dispCoef[N_DEG_DISPERSION_LAW+1];
    /* char str8[9]=""; */
    int science;
    char *detName=NULL;
    int nbFrameTarget = 0;
    double exptime=0.;

    struct stat s;

    // Retrieve EOP parameters for uv coordiantes compuatation
    cpl_table * eop_table;
    if (stat("matis_eop.fits",&s) == 0) {
      cpl_msg_info(cpl_func,"EOP file found in the current directory.");
    } else {
      cpl_msg_info(cpl_func,"No EOP file in the current directory.");
      cpl_msg_info(cpl_func,"Connection to IERS server.");
      cpl_msg_info (cpl_func, "Retrieving EOP file ");
      size_t  data_length;
      char * eop_data = hdrl_download_url_to_buffer("https://datacenter.iers.org/products/eop/rapid/standard/finals2000A.data", &data_length);
      //char * eop_data = mat_eop_download_finals2000A ("ftp.iers.org", "/products/eop/rapid/standard/finals2000A.data", &data_length);
      eop_table = hdrl_eop_data_totable (eop_data, (int)data_length);
     //eop_table = mat_eop_data_totable (eop_data, (int)data_length);
      cpl_table_save (eop_table, NULL, NULL, "matis_eop.fits", CPL_IO_CREATE);
      cpl_table_delete(eop_table);      
      free(eop_data);
    }
    eop_table = cpl_table_load("matis_eop.fits", 1, 0);
    /* Removing the file to avoid errors reported by the regression tests as the
     *  content of the file changes over time */
    if (cpl_msg_get_level() == CPL_MSG_INFO && stat("matis_eop.fits", &s) == 0) {
    	remove("matis_eop.fits");
    }

    nbFrameTarget = cpl_propertylist_get_int(corrFlux->keywords,"PRO FRAME TARGET");
    
    oifits->nbarray=0;
    oifits->nbwave=0;
    oifits->nbtupel=0;
    oifits->nbspect=0;

    oifits->oivis=NULL;
    oifits->oivis2=NULL;
    oifits->oit3=NULL;
    oifits->oispect=NULL;
    oifits->oiwave=NULL;
    oifits->oitf2=NULL;
    oifits->array_list=NULL;
    oifits->wave_list=NULL;
    oifits->spect_list=NULL;
    oifits->tupel_list=NULL;


    dispCoef[0] = cpl_propertylist_get_double(corrFlux->keywords, "PRO DISP COEF0");
    dispCoef[1] = cpl_propertylist_get_double(corrFlux->keywords, "PRO DISP COEF1");
    dispCoef[2] = cpl_propertylist_get_double(corrFlux->keywords, "PRO DISP COEF2");
    dispCoef[3] = cpl_propertylist_get_double(corrFlux->keywords, "PRO DISP COEF3");
    dispCoef[4] = cpl_propertylist_get_double(corrFlux->keywords, "PRO DISP COEF4");
    oifits->keywords=cpl_propertylist_duplicate(corrFlux->keywords);
    /* bcd1Status = (char *)cpl_propertylist_get_string(oifits->keywords,"ESO INS BCD1 ID"); */
    /* bcd2Status = (char *)cpl_propertylist_get_string(oifits->keywords,"ESO INS BCD2 ID"); */
    bcdMode = (char *)cpl_propertylist_get_string(oifits->keywords,"ESO CFG BCD MODE");
    detName=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO DET CHIP NAME");
    if (detName == NULL) {
      cpl_msg_error(cpl_func,"could not read ESO DET CHIP NAME keyword");
      return CPL_ERROR_NULL_INPUT;
    }
    exptime = (corrFlux->list_corr[nbFrameTarget-1]->time-corrFlux->list_corr[0]->time) * 86400.;
    
    // OIARRAY
    oifits->array=NULL;
    oifits->array= (mat_array *)cpl_calloc(1, sizeof(mat_array));
    oifits->array->arrayname = cpl_strdup(arrayname);
    oifits->array->cooframe = cpl_strdup(cooframe);
    oifits->array->arrayx=arrayx;
    oifits->array->arrayy=arrayy;
    oifits->array->arrayz=arrayz;
    oifits->array->telescope = NULL;
    oifits->array->dcsdictionaryid = NULL;
    oifits->array->dcsid = NULL;
    oifits->array->nbstation=cpl_propertylist_has(oifits->keywords,"ESO ISS CONF T1NAME");
    oifits->array->nbstation+=cpl_propertylist_has(oifits->keywords,"ESO ISS CONF T2NAME");
    oifits->array->nbstation+=cpl_propertylist_has(oifits->keywords,"ESO ISS CONF T3NAME");
    oifits->array->nbstation+=cpl_propertylist_has(oifits->keywords,"ESO ISS CONF T4NAME");
    
    if (oifits->array->nbstation == 0) {
      cpl_msg_warning(cpl_func,"Keyword ESO ISS CONF T?NAME not found");
      cpl_msg_warning(cpl_func,"We fix nbstation to 4");
      oifits->array->nbstation=4;
    }

    oifits->array->list_station= (mat_station **)cpl_calloc(oifits->array->nbstation, sizeof(mat_station *));
    for (k = 0; k < oifits->array->nbstation; k++)
      {
    	oifits->array->list_station[k]= (mat_station *)cpl_calloc(1, sizeof(mat_station));
    	sprintf(keyName,"ESO ISS CONF T%dNAME",k+1);
    	charFoo=(char *)cpl_propertylist_get_string(oifits->keywords,keyName);
    	if (charFoo == NULL)
    	  {
    	    /* sprintf(oifits->array->list_station[k]->telname,"%s","foo"); */
    	    sprintf(oifits->array->list_station[k]->telname,"%s",tel_name_default[k]);
    	  }
    	else
    	  {
    	    sprintf(oifits->array->list_station[k]->telname,"%s",charFoo);
    	  }
    	sprintf(keyName,"ESO ISS CONF STATION%d",k+1);
    	charFoo=(char *)cpl_propertylist_get_string(oifits->keywords,keyName);
    	if (charFoo == NULL)
    	  {
    	    /* sprintf(oifits->array->list_station[k]->staname,"%s","foo"); */
    	    sprintf(oifits->array->list_station[k]->staname,"%s",sta_name_default[k]);
    	  }
    	else
    	  {
    	    sprintf(oifits->array->list_station[k]->staname,"%s",charFoo);
    	  }
    	/* oifits->array->list_station[k]->staindex = k + 1; */
	oifits->array->list_station[k]->staindex=mat_get_sta_index(oifits->array->list_station[k]->staname);

    	sprintf(keyName,"ESO ISS CONF T%dX",k+1);
	if (cpl_propertylist_has(oifits->keywords,keyName)) {
	  oifits->array->list_station[k]->staxyz[0] = cpl_propertylist_get_double(oifits->keywords,keyName);
	} else {
	  oifits->array->list_station[k]->staxyz[0] = sta_x_default[k];
	}
    	sprintf(keyName,"ESO ISS CONF T%dY",k+1);
	if (cpl_propertylist_has(oifits->keywords,keyName)) {
	  oifits->array->list_station[k]->staxyz[1] = cpl_propertylist_get_double(oifits->keywords,keyName);
	} else {
	  oifits->array->list_station[k]->staxyz[1] = sta_y_default[k];
	}
    	sprintf(keyName,"ESO ISS CONF T%dZ",k+1);
	if (cpl_propertylist_has(oifits->keywords,keyName)) {
	  oifits->array->list_station[k]->staxyz[2] = cpl_propertylist_get_double(oifits->keywords,keyName);
	} else {
	  oifits->array->list_station[k]->staxyz[2] = sta_z_default[k];
	}

    	if (oifits->array->list_station[k]->telname[0] == 'U')
    	  {
    	    oifits->array->list_station[k]->diameter=8.2;
    	  }
    	else
    	  {
    	    oifits->array->list_station[k]->diameter=1.8;
    	  }
    	oifits->array->list_station[k]->fov=1.0;
    	sprintf(oifits->array->list_station[k]->fovtype,"%s","RADIUS");

    }
    // OITARGET
    oifits->oitarget=NULL;
    oifits->oitarget=(mat_oitarget *)cpl_calloc(1, sizeof(mat_oitarget));
    oifits->oitarget->targetid=1;
    charFoo=(char *)cpl_propertylist_get_string(oifits->keywords,"ESO OBS TARG NAME");
    if (charFoo != NULL) {
    	oifits->oitarget->targetname = cpl_strdup(charFoo);
    } else {
    	cpl_msg_warning(cpl_func,"Keyword ESO OBS TARG NAME not found");
    }
    oifits->oitarget->equinox=0.;
    oifits->oitarget->raep0=cpl_propertylist_get_double(oifits->keywords,"RA");
    oifits->oitarget->decep0=cpl_propertylist_get_double(oifits->keywords,"DEC");
    oifits->oitarget->raerr=0.;
    oifits->oitarget->decerr=0.;
    oifits->oitarget->sysvel=0.;
    oifits->oitarget->veltyp = cpl_strdup("HELIOCEN");
    oifits->oitarget->veldef = cpl_sprintf("%8s","OPTICAL");//cpl_strdup("OPTICAL \0");

    oifits->oitarget->pmra=0.;
    oifits->oitarget->pmdec=0.;
    oifits->oitarget->pmraerr=0.;
    oifits->oitarget->pmdecerr=0.;
    oifits->oitarget->parallax=0.;
    oifits->oitarget->paraerr=0.;
    oifits->oitarget->spectyp = cpl_strdup("UNKNOWN");

    science=cpl_propertylist_get_bool(corrFlux->keywords,"ESO PRO SCIENCE");
    if ( science == 0 ) {
      oifits->oitarget->category = cpl_strdup("CAL");
    } else {
      oifits->oitarget->category = cpl_strdup("SCI");
    }

    // OIWAVE
    oifits->oiwave=NULL;
    oifits->oiwave= (mat_oiwavelength *)cpl_calloc(1, sizeof(mat_oiwavelength));
    oifits->oiwave->insname = cpl_strdup(insname);
    oifits->oiwave->nbchannel=0;
    for(k = 0; k < corrFlux->imgdet->nbregion; k++)
      {
    	oifits->oiwave->nbchannel += corrFlux->imgdet->list_region[k]->naxis[1];
      }
    oifits->oiwave->effwave=cpl_calloc(oifits->oiwave->nbchannel,sizeof(double));
    oifits->oiwave->effband=cpl_calloc(oifits->oiwave->nbchannel,sizeof(double));
    cpt=0;
    for(k = 0; k < corrFlux->imgdet->nbregion; k++)
      {
    	mat_region *region = corrFlux->imgdet->list_region[k];
    	for(j = 0; j < region->naxis[1]; j++)
    	  {
    	    oifits->oiwave->effwave[cpt] =1.e-6*(dispCoef[0]+dispCoef[1]*(j+region->corner[1])+
						 dispCoef[2]*pow((double)(j + region->corner[1]), 2.0)+
						 dispCoef[3]*pow((double)(j + region->corner[1]), 3.0)+
						 dispCoef[4]*pow((double)(j + region->corner[1]), 4.0));
    	    oifits->oiwave->effband[cpt]=1.e-6*fabs(dispCoef[1]*(region->naxis[1] - 1 + region->corner[1])
						    +dispCoef[2]*pow((double)(region->naxis[1] - 1 + region->corner[1]), 2.0)
						    +dispCoef[3]*pow((double)(region->naxis[1] - 1 + region->corner[1]), 3.0)
						    +dispCoef[4]*pow((double)(region->naxis[1] - 1 + region->corner[1]), 4.0)
						    -dispCoef[1]*(region->corner[1])
						    -dispCoef[2]*pow((double)(region->corner[1]), 2.0)
						    -dispCoef[3]*pow((double)(region->corner[1]), 3.0)
						    -dispCoef[4]*pow((double)(region->corner[1]), 4.0)
						    )
	      /corrFlux->imgdet->list_region[k]->naxis[1];
    	    cpt++;
    	  }
      }

    // OIVIS
    // Allocate structure
    oifits->oivis = NULL;
    oifits->oivis = (mat_oivis *)cpl_calloc(1, sizeof(mat_oivis));
    oifits->oivis->dateobs   = cpl_strdup(corrFlux->imgdet->dateobs);
/* mat_propertylist_get_string_default(NULL, 0, oifits->keywords, "ESO TPL START", NULL); */
    oifits->oivis->insname   = cpl_strdup(insname);
    oifits->oivis->arrayname = cpl_strdup(arrayname);

    oifits->oivis->nbchannel = oifits->oiwave->nbchannel;
    oifits->oivis->nbvis     = (oifits->array->nbstation *
				(oifits->array->nbstation-1))/2;
    oifits->oivis->list_vis  =  (mat_viselem **)cpl_calloc(oifits->oivis->nbvis, sizeof(mat_viselem *));
    val=0.;
    //    for(k = 0; k < corrFlux->nbframe / corrFlux->imgdet->nbregion; k++)
    for(k = 0; k < nbFrameTarget;k++)
    {
    	val+=corrFlux->list_corr[k]->time;
    }
    //  val /= corrFlux->nbframe/corrFlux->imgdet->nbregion;
    val /= nbFrameTarget;
    for(k = 0; k < oifits->oivis->nbvis; k++)
    {
    	oifits->oivis->list_vis[k]= (mat_viselem *)cpl_calloc(1, sizeof(mat_viselem));
    	oifits->oivis->list_vis[k]->targetid  = 1;
    	oifits->oivis->list_vis[k]->time      = (val-(int)val)*86400.;
    	oifits->oivis->list_vis[k]->dateobsmjd= val;//corrFlux->imgdet->dateobsmjd;
    	oifits->oivis->list_vis[k]->exptime   = exptime;//corrFlux->list_corr[0]->exptime;
    	oifits->oivis->list_vis[k]->visamp    = cpl_calloc(oifits->oivis->nbchannel, sizeof(double));
	
	/* oifits->oivis->list_vis[k]->cfxamp    = cpl_calloc(oifits->oivis->nbchannel, sizeof(double)); */
    	oifits->oivis->list_vis[k]->visphi    = cpl_calloc(oifits->oivis->nbchannel, sizeof(double));
	
    	oifits->oivis->list_vis[k]->visamperr = cpl_calloc(oifits->oivis->nbchannel,sizeof(double));
    	oifits->oivis->list_vis[k]->visphierr = cpl_calloc(oifits->oivis->nbchannel,sizeof(double));
    	oifits->oivis->list_vis[k]->bandflag  = cpl_calloc(oifits->oivis->nbchannel,sizeof(cpl_boolean));
    }
    /* if (strcmp(bcd1Status,"OUT")) { */
    /* 	if (strcmp(bcd2Status,"OUT")) { */
    if (strcmp(bcdMode, "IN-IN") == 0)
      {
	// IN-IN
	if (oifits->array->nbstation==4)
	  {
	    oifits->oivis->list_vis[0]->stationindex[0]=oifits->array->list_station[3]->staindex;
	    oifits->oivis->list_vis[0]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[1]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[1]=oifits->array->list_station[3]->staindex;
	    oifits->oivis->list_vis[3]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[3]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[4]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[4]->stationindex[1]=oifits->array->list_station[3]->staindex;
	    oifits->oivis->list_vis[5]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[5]->stationindex[1]=oifits->array->list_station[2]->staindex;
	  }
	else if (oifits->array->nbstation==3)
	  {
	    oifits->oivis->list_vis[0]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[0]->stationindex[1]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[1]=oifits->array->list_station[2]->staindex;
	  }
	else
	  {
	    oifits->oivis->list_vis[0]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[0]->stationindex[1]=oifits->array->list_station[0]->staindex;
	  }
      }
    if (strcmp(bcdMode, "IN-OUT") == 0)
      {
	if (oifits->array->nbstation==4)
	  {
	    oifits->oivis->list_vis[0]->stationindex[0]=oifits->array->list_station[3]->staindex;
	    oifits->oivis->list_vis[0]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[1]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[1]=oifits->array->list_station[3]->staindex;
	    oifits->oivis->list_vis[3]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[3]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[4]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[4]->stationindex[1]=oifits->array->list_station[3]->staindex;
	    oifits->oivis->list_vis[5]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[5]->stationindex[1]=oifits->array->list_station[2]->staindex;
	  }
	else if (oifits->array->nbstation==3)
	  {
	    oifits->oivis->list_vis[0]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[0]->stationindex[1]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[1]=oifits->array->list_station[2]->staindex;
	  }
	else
	  {
	    oifits->oivis->list_vis[0]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[0]->stationindex[1]=oifits->array->list_station[0]->staindex;
	  }
      }
    if (strcmp(bcdMode, "OUT-IN") == 0)
      {
	if (oifits->array->nbstation==4)
	  {
	    oifits->oivis->list_vis[0]->stationindex[0]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[0]->stationindex[1]=oifits->array->list_station[3]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[1]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[3]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[3]->stationindex[1]=oifits->array->list_station[3]->staindex;
	    oifits->oivis->list_vis[4]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[4]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[5]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[5]->stationindex[1]=oifits->array->list_station[3]->staindex;
	  }
	else if (oifits->array->nbstation==3)
	  {
	    oifits->oivis->list_vis[0]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[0]->stationindex[1]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[1]=oifits->array->list_station[2]->staindex;
	  }
	else
	  {
	    oifits->oivis->list_vis[0]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[0]->stationindex[1]=oifits->array->list_station[1]->staindex;
	  }
      }
    if (strcmp(bcdMode, "OUT-OUT") == 0)
      {
	/* } else { */
	// OUT-OUT
	if (oifits->array->nbstation==4)
	  {
	    oifits->oivis->list_vis[0]->stationindex[0]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[0]->stationindex[1]=oifits->array->list_station[3]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[1]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[3]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[3]->stationindex[1]=oifits->array->list_station[3]->staindex;
	    oifits->oivis->list_vis[4]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[4]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[5]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[5]->stationindex[1]=oifits->array->list_station[3]->staindex;
	  }
	else if (oifits->array->nbstation==3)
	  {
	    oifits->oivis->list_vis[0]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[0]->stationindex[1]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oivis->list_vis[1]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[2]->stationindex[1]=oifits->array->list_station[2]->staindex;
	  }
	else
	  {
	    oifits->oivis->list_vis[0]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oivis->list_vis[0]->stationindex[1]=oifits->array->list_station[1]->staindex;
	  }
      }
    mat_eop_uv_oivis(oifits->oivis,oifits->array,oifits->keywords,eop_table);
    /* mat_oivis_filluv(oifits); */

    // OIVIS2
    oifits->oivis2=NULL;
    oifits->oivis2= (mat_oivis2 *)cpl_calloc(1, sizeof(mat_oivis2));
    oifits->oivis2->dateobs = cpl_strdup(corrFlux->imgdet->dateobs);

/* mat_propertylist_get_string_default(NULL, 0, oifits->keywords, "ESO TPL START", NULL); */
/*     if (oifits->oivis2->dateobs == NULL) { */
/*       oifits->oivis2->dateobs = cpl_sprintf("%d",cpl_propertylist_get_int(oifits->keywords, "ESO TPL START"));	     */
/*     }  */
    oifits->oivis2->insname = cpl_strdup(insname);
    oifits->oivis2->arrayname = cpl_strdup(arrayname);

    oifits->oivis2->nbchannel = oifits->oiwave->nbchannel;
    oifits->oivis2->nbvis2=(oifits->array->nbstation *
			    (oifits->array->nbstation - 1))/2;
    oifits->oivis2->list_vis2= (mat_vis2elem **)cpl_calloc(oifits->oivis2->nbvis2, sizeof(mat_vis2elem *));
    val=0.;
    //  for(k = 0; k < corrFlux->nbframe/corrFlux->imgdet->nbregion; k++)
    for(k = 0; k < nbFrameTarget;k++)
      {
    	val += corrFlux->list_corr[k]->time;
      }
    //  val /= corrFlux->nbframe/corrFlux->imgdet->nbregion;
    val /= nbFrameTarget;
    for(k = 0; k < oifits->oivis2->nbvis2; k++)
      {
	oifits->oivis2->list_vis2[k]= (mat_vis2elem *)cpl_calloc(1, sizeof(mat_vis2elem));
    	oifits->oivis2->list_vis2[k]->targetid = 1;
    	oifits->oivis2->list_vis2[k]->time = (val-(int)val)*86400.;
    	oifits->oivis2->list_vis2[k]->dateobsmjd = val;//corrFlux->imgdet->dateobsmjd;
    	oifits->oivis2->list_vis2[k]->exptime = exptime;//corrFlux->list_corr[0]->exptime;

	oifits->oivis2->list_vis2[k]->vis2 = cpl_calloc(oifits->oivis2->nbchannel, sizeof(double));
    	oifits->oivis2->list_vis2[k]->vis2err = cpl_calloc(oifits->oivis2->nbchannel, sizeof(double));

	/* oifits->oivis2->list_vis2[k]->cfx2 = cpl_calloc(oifits->oivis2->nbchannel, sizeof(double)); */
    	/* oifits->oivis2->list_vis2[k]->cfx2err = cpl_calloc(oifits->oivis2->nbchannel, sizeof(double)); */
	
    	oifits->oivis2->list_vis2[k]->bandflag = cpl_calloc(oifits->oivis2->nbchannel, sizeof(cpl_boolean));
      }
    /* if (strcmp(bcd1Status,"OUT")) { */
    /* 	if (strcmp(bcd2Status,"OUT")) { */
    if (strcmp(bcdMode, "IN-IN") == 0) {
      // IN-IN
      if (oifits->array->nbstation==4) {
	oifits->oivis2->list_vis2[0]->stationindex[0]=oifits->array->list_station[3]->staindex;
	oifits->oivis2->list_vis2[0]->stationindex[1]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[1]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[1]=oifits->array->list_station[3]->staindex;
	oifits->oivis2->list_vis2[3]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[3]->stationindex[1]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[4]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[4]->stationindex[1]=oifits->array->list_station[3]->staindex;
	oifits->oivis2->list_vis2[5]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[5]->stationindex[1]=oifits->array->list_station[2]->staindex;
      } else if (oifits->array->nbstation==3) {
	oifits->oivis2->list_vis2[0]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[0]->stationindex[1]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[1]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[1]=oifits->array->list_station[2]->staindex;
      } else {
	oifits->oivis2->list_vis2[0]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[0]->stationindex[1]=oifits->array->list_station[0]->staindex;
      }
    }
    if (strcmp(bcdMode, "IN-OUT") == 0) {
    /* } else { */
    	    // IN-OUT
      if (oifits->array->nbstation==4) {
	oifits->oivis2->list_vis2[0]->stationindex[0]=oifits->array->list_station[3]->staindex;
	oifits->oivis2->list_vis2[0]->stationindex[1]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[1]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[1]=oifits->array->list_station[3]->staindex;
	oifits->oivis2->list_vis2[3]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[3]->stationindex[1]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[4]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[4]->stationindex[1]=oifits->array->list_station[3]->staindex;
	oifits->oivis2->list_vis2[5]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[5]->stationindex[1]=oifits->array->list_station[2]->staindex;
      } else if (oifits->array->nbstation==3) {
	oifits->oivis2->list_vis2[0]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[0]->stationindex[1]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[1]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[1]=oifits->array->list_station[2]->staindex;
      } else {
	oifits->oivis2->list_vis2[0]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[0]->stationindex[1]=oifits->array->list_station[0]->staindex;
      }
    }
    if (strcmp(bcdMode, "OUT-IN") == 0) {
    /* } else { */
    /* 	if (strcmp(bcd2Status,"OUT")) { */
    	    // OUT-IN
      if (oifits->array->nbstation==4) {
	oifits->oivis2->list_vis2[0]->stationindex[0]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[0]->stationindex[1]=oifits->array->list_station[3]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[1]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[1]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[3]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[3]->stationindex[1]=oifits->array->list_station[3]->staindex;
	oifits->oivis2->list_vis2[4]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[4]->stationindex[1]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[5]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[5]->stationindex[1]=oifits->array->list_station[3]->staindex;
      } else if (oifits->array->nbstation==3) {
	oifits->oivis2->list_vis2[0]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[0]->stationindex[1]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[1]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[1]=oifits->array->list_station[2]->staindex;
      } else {
	oifits->oivis2->list_vis2[0]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[0]->stationindex[1]=oifits->array->list_station[1]->staindex;
      }
    }
    if (strcmp(bcdMode, "OUT-OUT") == 0) {
    	/* } else { */
    	    // OUT-OUT
      if (oifits->array->nbstation==4) {
	oifits->oivis2->list_vis2[0]->stationindex[0]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[0]->stationindex[1]=oifits->array->list_station[3]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[1]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[1]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[3]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[3]->stationindex[1]=oifits->array->list_station[3]->staindex;
	oifits->oivis2->list_vis2[4]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[4]->stationindex[1]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[5]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[5]->stationindex[1]=oifits->array->list_station[3]->staindex;
      } else if (oifits->array->nbstation==3) {
	oifits->oivis2->list_vis2[0]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[0]->stationindex[1]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[0]=oifits->array->list_station[1]->staindex;
	oifits->oivis2->list_vis2[1]->stationindex[1]=oifits->array->list_station[2]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[2]->stationindex[1]=oifits->array->list_station[2]->staindex;
      } else {
	oifits->oivis2->list_vis2[0]->stationindex[0]=oifits->array->list_station[0]->staindex;
	oifits->oivis2->list_vis2[0]->stationindex[1]=oifits->array->list_station[1]->staindex;
      }
    }
    mat_eop_uv_oivis2(oifits->oivis2,oifits->array,oifits->keywords,eop_table);
    /* mat_oivis2_filluv(oifits); */
    // OIT3
    oifits->oit3=NULL;
    if (oifits->array->nbstation > 2)
      {
	oifits->oit3= (mat_oit3 *)cpl_calloc(1, sizeof(mat_oit3));
	oifits->oit3->dateobs = cpl_strdup(corrFlux->imgdet->dateobs);

/* mat_propertylist_get_string_default(NULL, 0, oifits->keywords, "ESO TPL START", NULL); */
/* 	if (oifits->oit3->dateobs == NULL) { */
/* 	  oifits->oit3->dateobs = cpl_sprintf("%d",cpl_propertylist_get_int(oifits->keywords, "ESO TPL START"));	     */
/* 	}  */
    	oifits->oit3->insname = cpl_strdup(insname);
    	oifits->oit3->arrayname = cpl_strdup(arrayname);
    	oifits->oit3->nbchannel = oifits->oiwave->nbchannel;
    	if (oifits->array->nbstation == 3)
	  {
    	    oifits->oit3->nbt3 = 1;
	  }
    	if (oifits->array->nbstation == 4 )
	  {
    	    oifits->oit3->nbt3 = 4;
	  }
    	oifits->oit3->list_t3 = (mat_t3elem **)cpl_calloc(oifits->oit3->nbt3, sizeof(mat_t3elem *));
    	val=0.;
	//  	for (k = 0; k < corrFlux->nbframe/corrFlux->imgdet->nbregion; k++)
    	for (k = 0; k < nbFrameTarget; k++)
	  {
    	    val += corrFlux->list_corr[k]->time;
	  }
	//    	val /= corrFlux->nbframe/corrFlux->imgdet->nbregion;
    	val /= nbFrameTarget;
    	for (k = 0; k < oifits->oit3->nbt3; k++)
	  {
	    oifits->oit3->list_t3[k] = (mat_t3elem *)cpl_calloc(1, sizeof(mat_t3elem));
    	    oifits->oit3->list_t3[k]->targetid = 1;
    	    oifits->oit3->list_t3[k]->time = (val-(int)val)*86400.;
    	    oifits->oit3->list_t3[k]->exptime = exptime;//corrFlux->list_corr[0]->exptime;
    	    oifits->oit3->list_t3[k]->dateobsmjd = val;//corrFlux->imgdet->dateobsmjd;
    	    oifits->oit3->list_t3[k]->t3phi = cpl_calloc(oifits->oit3->nbchannel, sizeof(double));
    	    oifits->oit3->list_t3[k]->t3phierr = cpl_calloc(oifits->oit3->nbchannel,sizeof(double));
    	    oifits->oit3->list_t3[k]->t3amp = cpl_calloc(oifits->oit3->nbchannel,sizeof(double));
    	    oifits->oit3->list_t3[k]->t3amperr = cpl_calloc(oifits->oit3->nbchannel,sizeof(double));
    	    oifits->oit3->list_t3[k]->bandflag = cpl_calloc(oifits->oit3->nbchannel,sizeof(cpl_boolean));
	  }
	if (strcmp(bcdMode, "IN-IN") == 0) {
    	/* if (strcmp(bcd1Status,"OUT")) { */
    	/*     if (strcmp(bcd2Status,"OUT")) { */
    		// IN-IN
	  if (oifits->array->nbstation==4) {
	    oifits->oit3->list_t3[0]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[1]=oifits->array->list_station[3]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[2]=oifits->array->list_station[2]->staindex;
	    oifits->oit3->list_t3[1]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[1]->stationindex[1]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[1]->stationindex[2]=oifits->array->list_station[3]->staindex;
	    oifits->oit3->list_t3[2]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[2]->stationindex[1]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[2]->stationindex[2]=oifits->array->list_station[2]->staindex;
	    oifits->oit3->list_t3[3]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[3]->stationindex[1]=oifits->array->list_station[3]->staindex;
	    oifits->oit3->list_t3[3]->stationindex[2]=oifits->array->list_station[2]->staindex;
	  } else {
	    oifits->oit3->list_t3[0]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[1]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[2]=oifits->array->list_station[2]->staindex;
	  }
	}
	if (strcmp(bcdMode, "IN-OUT") == 0) {
    	    /* } else { */
    		// IN-OUT
	  if (oifits->array->nbstation==4) {
	    oifits->oit3->list_t3[0]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[1]=oifits->array->list_station[3]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[2]=oifits->array->list_station[2]->staindex;
	    oifits->oit3->list_t3[1]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[1]->stationindex[1]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[1]->stationindex[2]=oifits->array->list_station[3]->staindex;
	    oifits->oit3->list_t3[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[2]->stationindex[1]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[2]->stationindex[2]=oifits->array->list_station[2]->staindex;
	    oifits->oit3->list_t3[3]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[3]->stationindex[1]=oifits->array->list_station[3]->staindex;
	    oifits->oit3->list_t3[3]->stationindex[2]=oifits->array->list_station[2]->staindex;
	  } else {
	    oifits->oit3->list_t3[0]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[1]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[2]=oifits->array->list_station[2]->staindex;
	  }
	}
	if (strcmp(bcdMode, "OUT-IN") == 0) {
    	/* } else { */
    	/*     if (strcmp(bcd2Status,"OUT")) { */
    		// OUT-IN
	  if (oifits->array->nbstation==4) {
	    oifits->oit3->list_t3[0]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[2]=oifits->array->list_station[3]->staindex;
	    oifits->oit3->list_t3[1]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[1]->stationindex[1]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[1]->stationindex[2]=oifits->array->list_station[2]->staindex;
	    oifits->oit3->list_t3[2]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[2]->stationindex[1]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[2]->stationindex[2]=oifits->array->list_station[3]->staindex;
	    oifits->oit3->list_t3[3]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[3]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oit3->list_t3[3]->stationindex[2]=oifits->array->list_station[3]->staindex;
	  } else {
	    oifits->oit3->list_t3[0]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[1]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[2]=oifits->array->list_station[2]->staindex;
	  }
	}
	if (strcmp(bcdMode, "OUT-OUT") == 0) {	
    	    /* } else { */
    		// OUT-OUT
	  if (oifits->array->nbstation==4) {
	    oifits->oit3->list_t3[0]->stationindex[0]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[2]=oifits->array->list_station[3]->staindex;
	    oifits->oit3->list_t3[1]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[1]->stationindex[1]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[1]->stationindex[2]=oifits->array->list_station[2]->staindex;
	    oifits->oit3->list_t3[2]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[2]->stationindex[1]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[2]->stationindex[2]=oifits->array->list_station[3]->staindex;
	    oifits->oit3->list_t3[3]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[3]->stationindex[1]=oifits->array->list_station[2]->staindex;
	    oifits->oit3->list_t3[3]->stationindex[2]=oifits->array->list_station[3]->staindex;
	  } else {
	    oifits->oit3->list_t3[0]->stationindex[0]=oifits->array->list_station[0]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[1]=oifits->array->list_station[1]->staindex;
	    oifits->oit3->list_t3[0]->stationindex[2]=oifits->array->list_station[2]->staindex;
	  }
	}
	mat_eop_uv_oit3(oifits->oit3,oifits->array,oifits->keywords,eop_table);
    	/* mat_oit3_filluv(oifits); */
    }
   // OISPECT
    oifits->oispect=NULL;
    oifits->oispect=mat_oispectrum_new(oifits->array->nbstation,oifits->oiwave->nbchannel);
    oifits->oispect->dateobs = cpl_strdup(corrFlux->imgdet->dateobs);

/* mat_propertylist_get_string_default(NULL, 0, oifits->keywords, "ESO TPL START", NULL); */
/*     if (oifits->oispect->dateobs == NULL) { */
/*       oifits->oispect->dateobs = cpl_sprintf("%d",cpl_propertylist_get_int(oifits->keywords, "ESO TPL START"));	     */
/*     }  */
    oifits->oispect->insname = cpl_strdup(insname);
    oifits->oispect->arrayname = cpl_strdup(arrayname);
    val=0.;
    //    for (k = 0; k < corrFlux->nbframe/corrFlux->imgdet->nbregion; k++)
    for (k = 0; k < nbFrameTarget; k++)
      {
	val += corrFlux->list_corr[k]->time;
      }
    //    val /= corrFlux->nbframe/corrFlux->imgdet->nbregion;
    val /= nbFrameTarget;
    for(k=0;k<oifits->oispect->nbflux;k++) {
      oifits->oispect->list_flux[k]->targetid = 1;
      oifits->oispect->list_flux[k]->exptime = exptime;//corrFlux->list_corr[0]->exptime;
      oifits->oispect->list_flux[k]->time = (val-(int)val)*86400.;
      oifits->oispect->list_flux[k]->dateobsmjd = val;//corrFlux->imgdet->dateobsmjd;
      oifits->oispect->list_flux[k]->stationindex=oifits->array->list_station[k]->staindex;
      /* oifits->oispect->list_flux[k]->bandflag  = cpl_calloc(oifits->oispect->nbchannel,sizeof(cpl_boolean)); */
    }

    if (strcmp(bcdMode, "IN-IN") == 0) {	
    /* if (strcmp(bcd1Status,"OUT")) { */
    /* 	if (strcmp(bcd2Status,"OUT")) { */
    	    // IN-IN
      if (oifits->array->nbstation==4) {
	if(!strcmp(detName, "AQUARIUS")) {
	  oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[3]->staindex;
	  oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[2]->staindex;
	  oifits->oispect->list_flux[2]->stationindex=oifits->array->list_station[0]->staindex;
	  oifits->oispect->list_flux[3]->stationindex=oifits->array->list_station[1]->staindex;
	} else {
	  oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[1]->staindex;
	  oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[0]->staindex;
	  oifits->oispect->list_flux[2]->stationindex=oifits->array->list_station[2]->staindex;
	  oifits->oispect->list_flux[3]->stationindex=oifits->array->list_station[3]->staindex;
	}
	  
      } else if (oifits->array->nbstation==3) {
	oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[2]->staindex;
	oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[0]->staindex;
	oifits->oispect->list_flux[2]->stationindex=oifits->array->list_station[1]->staindex;
      } else {
	oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[0]->staindex;
	oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[1]->staindex;
      }
    }
    if (strcmp(bcdMode, "IN-OUT") == 0) {	
    	/* } else { */
    	    // IN-OUT
      if (oifits->array->nbstation==4) {
	if(!strcmp(detName, "AQUARIUS")) {
	  oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[3]->staindex;
	  oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[2]->staindex;
	  oifits->oispect->list_flux[2]->stationindex=oifits->array->list_station[1]->staindex;
	  oifits->oispect->list_flux[3]->stationindex=oifits->array->list_station[0]->staindex;
	} else {
	  oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[0]->staindex;
	  oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[1]->staindex;
	  oifits->oispect->list_flux[2]->stationindex=oifits->array->list_station[2]->staindex;
	  oifits->oispect->list_flux[3]->stationindex=oifits->array->list_station[3]->staindex;
	}
	
      } else if (oifits->array->nbstation==3) {
	oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[2]->staindex;
	oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[0]->staindex;
	oifits->oispect->list_flux[2]->stationindex=oifits->array->list_station[1]->staindex;
      } else {
	oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[1]->staindex;
	oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[0]->staindex;
      }
    }
    if (strcmp(bcdMode, "OUT-IN") == 0) {    
    /* } else { */
    	/* if (strcmp(bcd2Status,"OUT")) { */
    	    // OUT-IN
      if (oifits->array->nbstation==4) {
	if(!strcmp(detName, "AQUARIUS")) {
	  oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[2]->staindex;
	  oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[3]->staindex;
	  oifits->oispect->list_flux[2]->stationindex=oifits->array->list_station[0]->staindex;
	  oifits->oispect->list_flux[3]->stationindex=oifits->array->list_station[1]->staindex;
	} else {
	  oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[1]->staindex;
	  oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[0]->staindex;
	  oifits->oispect->list_flux[2]->stationindex=oifits->array->list_station[3]->staindex;
	  oifits->oispect->list_flux[3]->stationindex=oifits->array->list_station[2]->staindex;
	}	  
      } else if (oifits->array->nbstation==3) {
	oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[0]->staindex;
	oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[2]->staindex;
	oifits->oispect->list_flux[2]->stationindex=oifits->array->list_station[1]->staindex;
      } else {
	oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[0]->staindex;
	oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[1]->staindex;
      }
    }
    if (strcmp(bcdMode, "OUT-OUT") == 0) {    
    	/* } else { */
    	    // OUT-OUT
      if (oifits->array->nbstation==4) {
	if(!strcmp(detName, "AQUARIUS")) {
	  oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[2]->staindex;
	  oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[3]->staindex;
	  oifits->oispect->list_flux[2]->stationindex=oifits->array->list_station[1]->staindex;
	  oifits->oispect->list_flux[3]->stationindex=oifits->array->list_station[0]->staindex;
	} else {
	  oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[0]->staindex;
	  oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[1]->staindex;
	  oifits->oispect->list_flux[2]->stationindex=oifits->array->list_station[3]->staindex;
	  oifits->oispect->list_flux[3]->stationindex=oifits->array->list_station[2]->staindex;
	}	  
      } else if (oifits->array->nbstation==3) {
	oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[0]->staindex;
	oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[1]->staindex;
	oifits->oispect->list_flux[2]->stationindex=oifits->array->list_station[2]->staindex;
      } else {
	oifits->oispect->list_flux[0]->stationindex=oifits->array->list_station[1]->staindex;
	oifits->oispect->list_flux[1]->stationindex=oifits->array->list_station[0]->staindex;
      }
    }





    /* oifits->oispect= (mat_oispectrum *)cpl_calloc(1, sizeof(mat_oispectrum)); */
    /* oifits->oispect->flux=cpl_calloc(oifits->oiwave->nbchannel,sizeof(double)); */
    /* oifits->oispect->fluxerr=cpl_calloc(oifits->oiwave->nbchannel,sizeof(double)); */
    /* if (charFoo != NULL) free(charFoo); */
    cpl_table_delete(eop_table);      
    
    return 0;
}

