/* $Id: mat_oitf2.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oitf2.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "mat_const.h"
#include "mat_error.h"
#include "mat_utils.h"
#include "mat_oitf2.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/**
  @ingroup oitf2
  @brief Initialize a mat_tf2elem structure
  @param nbchannel          Number of spectral channels
  @return mat_tf2elem
 */
/*-----------------------------------------------------------------------------*/
mat_tf2elem *mat_tf2elem_new(int nbchannel)
{
  mat_tf2elem  *el = (mat_tf2elem *)cpl_calloc(1, sizeof(mat_tf2elem));

  if (el == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_tf2elem");
      return NULL;
    }
  el->tf2 = cpl_calloc(nbchannel, sizeof(double));
  if (el->tf2 == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_tf2elem->tf2");
      mat_tf2elem_delete(el);
      return NULL;
    }
  el->tf2err = cpl_calloc(nbchannel, sizeof(double));
  if (el->tf2err == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_tf2elem->tf2err");
      mat_tf2elem_delete(el);
      return NULL;
    }
  el->tf = cpl_calloc(nbchannel, sizeof(double));
  if (el->tf == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_tf2elem->tf");
      mat_tf2elem_delete(el);
      return NULL;
    }
  el->tferr = cpl_calloc(nbchannel, sizeof(double));
  if (el->tferr == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_tf2elem->tferr");
      mat_tf2elem_delete(el);
      return NULL;
    }
  return el;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oitf2
  @brief Delete a mat_tf2elem structure
  @param el contains the mat_tf2elem structure
  @return cpl_error_code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_tf2elem_delete(mat_tf2elem *el)
{
  mat_assert_value((el != NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no mat_tf2elem (el) argument given");
  if (el->tf2 != NULL)
    {
      cpl_free(el->tf2);
      el->tf2 = NULL;
    }
  if (el->tf2err != NULL)
    {
      cpl_free(el->tf2err);
      el->tf2err = NULL;
    }
  if (el->tf != NULL)
    {
      cpl_free(el->tf);
      el->tf = NULL;
    }
  if (el->tferr != NULL)
    {
      cpl_free(el->tferr);
      el->tferr = NULL;
    }
  cpl_free(el);
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oitf2
  @brief Initialize a mat_oitf2 structure
  @param nbtf2 Number of baselines
  @param nbchannel Number of spectral channels
  @return mat_oitf2
 */
/*-----------------------------------------------------------------------------*/
mat_oitf2 *mat_oitf2_new(int nbtf2, int nbchannel)
{
  mat_oitf2 *oitf2;
  int         i;

  oitf2 = (mat_oitf2 *)cpl_calloc(1, sizeof(mat_oitf2));
  if (oitf2 == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oitf2");
      return NULL;
    }
  oitf2->nbtf2 = nbtf2;
  oitf2->nbchannel = nbchannel;
  oitf2->list_tf2 = (mat_tf2elem **)cpl_calloc(nbtf2, sizeof(mat_tf2elem *));
  if (oitf2->list_tf2 == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oitf2->list_tf2");
      mat_oitf2_delete_nt(oitf2);
      return NULL;
    }
  for (i = 0; i < nbtf2; i++) 
    {
      oitf2->list_tf2[i] = mat_tf2elem_new(nbchannel);
      if (oitf2->list_tf2[i] == NULL)
	{
	  mat_oitf2_delete_nt(oitf2);
	  return NULL;
	}
    }
  return oitf2;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oitf2
  @brief Delete a mat_oitf2 structure
  @param oitf2 contains the mat_oitf2 structure
  @return cpl_error_code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_oitf2_delete_nt(mat_oitf2 *oitf2)
{
  int i;

  mat_assert_value((oitf2!=NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no mat_oitf2 (oitf2) argument given");
  if (oitf2->dateobs != NULL)
    {
      cpl_free(oitf2->dateobs);
      oitf2->dateobs = NULL;
    }
  if (oitf2->arrayname != NULL)
    {
      cpl_free(oitf2->arrayname);
      oitf2->arrayname = NULL;
    }
  if (oitf2->insname != NULL)
    {
      cpl_free(oitf2->insname);
      oitf2->insname = NULL;
    }
  if (oitf2->list_tf2 != NULL)
    {
      for (i = 0; i < oitf2->nbtf2; i++)
	{
	  if (oitf2->list_tf2[i] != NULL)
	    {
	      mat_tf2elem_delete(oitf2->list_tf2[i]);
	      oitf2->list_tf2[i] = NULL;
	    }
	}
      cpl_free(oitf2->list_tf2);
      oitf2->list_tf2 = NULL;
      oitf2->nbtf2 = 0;
    }
  cpl_free(oitf2);
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oitf2
  @brief Load a mat_oitf2 structure from a cpl_table
  @param plist contains the cpl_propertylist 
  @param table contains the cpl_table
  @return mat_oitf2 or NULL if error
 */
/*-----------------------------------------------------------------------------*/
mat_oitf2 *mat_oitf2_from_table(cpl_propertylist *plist, cpl_table *table)
{
  mat_oitf2  *oitf2;
  int          nbtf2;
  int          nbchannel;
  int          j, k;

  nbtf2 = cpl_table_get_nrow(table);
  /* old code
  nbchannel = cpl_array_get_size(cpl_table_get_array (table, "TF2", 0));
  if (nbchannel == 0) nbchannel = 1; // no array!
  */
  if (cpl_table_get_column_depth(table, "TF2") == 0)
    {
      nbchannel = 1;
    }
  else
    {
      nbchannel = cpl_table_get_column_dimension(table, "TF2", 0);
    }
  cpl_msg_info(cpl_func, "TF2 table has %d wavelength data", nbchannel);
  oitf2 = mat_oitf2_new(nbtf2, nbchannel);
  if (oitf2 == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oitf2");
      return NULL;
    }

  oitf2->extver    = mat_propertylist_get_int_default(plist, "EXTVER", 1);
  oitf2->arrayname = mat_propertylist_get_string_default(NULL, 0, plist, "ARRNAME", "");
  oitf2->dateobs   = mat_propertylist_get_string_default(NULL, 0, plist, "DATE-OBS", "");
  oitf2->insname   = mat_propertylist_get_string_default(NULL, 0, plist, "INSNAME", "");

  for(j = 0; j < oitf2->nbtf2; j++) 
    {
      mat_tf2elem  *el = oitf2->list_tf2[j];
      // copy the values from the table into the tf2 element
      el->time = cpl_table_get_double (table, "TIME", j, NULL);
      el->dateobsmjd = cpl_table_get_double (table, "MJD", j, NULL);
      el->exptime = cpl_table_get_double (table, "INT_TIME", j, NULL);
      el->stationindex[0]= cpl_array_get_int(cpl_table_get_array(table, "STA_INDEX", j),0, NULL);
      el->stationindex[1] = cpl_array_get_int(cpl_table_get_array(table, "STA_INDEX", j), 1, NULL);
      if (oitf2->nbchannel == 1)
	{
	  el->tf2[0] = cpl_table_get_double(table, "TF2", j, NULL);
	  el->tf2err[0] = cpl_table_get_double(table, "TF2ERR", j, NULL);
	  el->tf[0] = cpl_table_get_double(table, "TF", j, NULL);
	  el->tferr[0] = cpl_table_get_double(table, "TFERR", j, NULL);
	}
      else
	{
	  for (k = 0; k < oitf2->nbchannel; k++) 
	    {
	      el->tf2[k] = cpl_array_get_double(cpl_table_get_array(table, "TF2", j), k, NULL);
	      el->tf2err[k] = cpl_array_get_double(cpl_table_get_array(table, "TF2ERR", j), k, NULL);
	      el->tf[k] = cpl_array_get_double(cpl_table_get_array(table, "TF", j), k, NULL);
	      el->tferr[k] = cpl_array_get_double(cpl_table_get_array(table, "TFERR", j), k, NULL);
	    }
	}
    }
  return oitf2;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oitf2
  @brief Save a mat_oitf2 structure in a cpl_table
  @param oitf2 contains the mat_oift2
  @param plist contains the cpl_propertylist 
  @return cpl_table or NULL if error
 */
/*-----------------------------------------------------------------------------*/
cpl_table *mat_oitf2_to_table(mat_oitf2 *oitf2, cpl_propertylist *plist)
{
  cpl_table    *table;
  cpl_array    *arr;
  int           i;

  if (plist != NULL)
    {
      cpl_propertylist_append_string(plist,"EXTNAME","TF2");
      cpl_propertylist_append_int(plist, "EXTVER", oitf2->extver);
      if ( cpl_propertylist_has(plist,"OI_REVN") )
	{
	  cpl_propertylist_erase(plist,"OI_REVN");
	}
      cpl_propertylist_append_int(plist,"OI_REVN",2);
      cpl_propertylist_append_string(plist,"DATE-OBS",oitf2->dateobs);
      cpl_propertylist_append_string(plist,"ARRNAME",oitf2->arrayname);
      cpl_propertylist_append_string(plist,"INSNAME",oitf2->insname);
    }
  table = cpl_table_new(oitf2->nbtf2);
  if (table == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_table");
      return NULL;
    }
  // Add columns
  cpl_table_new_column(table, "TIME", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "TIME","s");
  cpl_table_new_column(table, "MJD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "MJD","day");
  cpl_table_new_column(table, "INT_TIME", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "INT_TIME","s");
  if (oitf2->nbchannel == 1)
    {
      cpl_table_new_column(table, "TF2", CPL_TYPE_DOUBLE);
      cpl_table_new_column(table, "TF2ERR", CPL_TYPE_DOUBLE);
      cpl_table_new_column(table, "TF", CPL_TYPE_DOUBLE);
      cpl_table_new_column(table, "TFERR", CPL_TYPE_DOUBLE);
    }
  else
    {
      cpl_table_new_column_array(table, "TF2", CPL_TYPE_DOUBLE, oitf2->nbchannel);
      cpl_table_new_column_array(table, "TF2ERR", CPL_TYPE_DOUBLE, oitf2->nbchannel);
      cpl_table_new_column_array(table, "TF", CPL_TYPE_DOUBLE, oitf2->nbchannel);
      cpl_table_new_column_array(table, "TFERR", CPL_TYPE_DOUBLE, oitf2->nbchannel);
    }
  cpl_table_new_column_array(table,"STA_INDEX", CPL_TYPE_INT, 2);
  // Fill the table
  for (i = 0; i < oitf2->nbtf2; i++)
    {
      mat_tf2elem  *el = oitf2->list_tf2[i];
      cpl_table_set_double(table,"TIME",i, el->time);
      cpl_table_set_double(table,"MJD",i, el->dateobsmjd);
      cpl_table_set_double(table,"INT_TIME",i, el->exptime);
      if (oitf2->nbchannel == 1)
	{
	  cpl_table_set_double(table,"TF2",i, el->tf2[0]);
	  cpl_table_set_double(table,"TF2ERR",i, el->tf2err[0]);
	  cpl_table_set_double(table,"TF",i, el->tf[0]);
	  cpl_table_set_double(table,"TFERR",i, el->tferr[0]);
	}
      else
	{
	  arr = cpl_array_wrap_double(el->tf2, oitf2->nbchannel);
	  cpl_table_set_array(table,"TF2", i, arr);
	  cpl_array_unwrap(arr);
	  arr = cpl_array_wrap_double(el->tf2err, oitf2->nbchannel);
	  cpl_table_set_array(table,"TF2ERR",i,arr);
	  cpl_array_unwrap(arr);
	  arr = cpl_array_wrap_double(el->tf, oitf2->nbchannel);
	  cpl_table_set_array(table,"TF", i, arr);
	  cpl_array_unwrap(arr);
	  arr = cpl_array_wrap_double(el->tferr, oitf2->nbchannel);
	  cpl_table_set_array(table,"TFERR",i,arr);
	  cpl_array_unwrap(arr);
	}
      arr = cpl_array_wrap_int(el->stationindex, 2);
      cpl_table_set_array(table, "STA_INDEX", i, arr);
      cpl_array_unwrap(arr);
    }
  cpl_table_set_column_savetype(table,"STA_INDEX",CPL_TYPE_SHORT);
  return table;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oitf2
  @brief Load a frame in mat_oitf2 structure
  @param frame           current frame
  @return mat_oitf2
 */
/*-----------------------------------------------------------------------------*/
mat_oitf2 *mat_oitf2_load_nt(cpl_frame *frame)
{
  mat_oitf2 *oitf2 = NULL;  
  int nbExtent = 0;
  int i = 0;
  cpl_propertylist *plist = NULL;
  char *keyExtname = NULL;
  cpl_table *table = NULL;

  /*Allocate the structure and initialize its attributes*/
  mat_assert_value((frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_frame (frame) argument given");
  nbExtent = cpl_frame_get_nextensions(frame);
  for(i = 0; i<nbExtent; i++) 
    {
      plist = cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
      keyExtname = (char *)cpl_propertylist_get_string(plist,"EXTNAME");
      if (keyExtname != NULL) 
	{
	  table = cpl_table_load(cpl_frame_get_filename(frame), i+1, 
				 0);
	  /*Load data*/
	  /* if (!strcmp(keyExtname, "OI_RAWTF2") || !strcmp(keyExtname, "TF2"))  */
	  if (!strcmp(keyExtname, "TF2")) 
	    {
	      cpl_error_reset();
	      oitf2 = mat_oitf2_from_table(plist, table);
	    }
	  cpl_table_delete(table);
	}
      cpl_propertylist_delete(plist);
    }
  return oitf2;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup oitf2
   @brief Append an TF2 binary table to an existing OIFITS file
   @param filename          name of the FITS file
   @param oitf2            the mat_oitf2 structure containing the 
                           informations for filling the binary table
   @return cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code  mat_oitf2_append_nt(char *filename,mat_oitf2 *oitf2)
{
  cpl_table *oitf2Tab = NULL;
  cpl_propertylist *keyTab = NULL;
  cpl_errorstate prestate = cpl_errorstate_get();

  /*Check input parameters*/
  mat_assert_value((oitf2!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oitf2 (oitf2) argument given");
  mat_assert_value((filename!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no filename given");
  
  keyTab=cpl_propertylist_new();
  oitf2Tab = mat_oitf2_to_table(oitf2, keyTab);
  cpl_table_save(oitf2Tab,NULL,keyTab,filename,CPL_IO_EXTEND);

  cpl_propertylist_delete(keyTab);
  cpl_table_delete(oitf2Tab);
  if (!cpl_errorstate_is_equal(prestate))
    {
      cpl_errorstate_set(prestate);
      return CPL_ERROR_UNSPECIFIED;
    }
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup oitf2
  @brief Get the time span of the observation as MJD values (beginning and end).
  @param oitf2   the mat_oitf2 structure containing the informations for filling the binary table
  @param min_mjd The beginning of the observation.
  @param max_mjd The end of the observation.
  @return cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mat_oitf2_get_mjd_span(mat_oitf2 *oitf2, double *min_mjd, double *max_mjd)
{
  int i;

  /*Check input parameters*/
  mat_assert_value((oitf2!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oitf2 (oitf2) argument given");
  mat_assert_value((min_mjd!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no start MJD given");
  mat_assert_value((max_mjd!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no end MJD given");
  *min_mjd =  1.0e12;
  *max_mjd = -1.0e12;
  for (i = 0; i < oitf2->nbtf2; i++)
    {
      mat_tf2elem *el = oitf2->list_tf2[i];
      *min_mjd = fmin(*min_mjd, el->dateobsmjd);
      *max_mjd = fmax(*max_mjd, el->dateobsmjd);
    }
  return CPL_ERROR_NONE;
}
