/* $Id: mat_oivis2.c,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oivis2.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/

#include "mat_const.h"
#include "mat_error.h"
#include "mat_utils.h"
#include "mat_oivis2.h"

/*-----------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/

#define CHUNK_SIZE 1024

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/
/**
   @ingroup oivis2
   @brief This method creates a new mat_vis2elem structure with all pointers set to NULL
   @param nbchannel Number of spectral channels
   @returns Pointer to a new allocated mat_vis2elem structure on success or NULL on failure
 
   This method creates a new mat_vis2elem structure and sets all values to the default. For
   pointers the value is NULL for primitive data types the value is set to 0.
*/
mat_vis2elem *mat_vis2elem_new(int nbchannel)
{
  mat_vis2elem  *el = (mat_vis2elem *)cpl_calloc(1, sizeof(mat_vis2elem));

  if (el == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_vis2elem");
      return NULL;
    }
  
  el->vis2 = cpl_calloc(nbchannel, sizeof(double));
  if (el->vis2 == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_vis2elem->vis2");
      mat_vis2elem_delete(el);
      return NULL;
    }
  el->vis2err = cpl_calloc(nbchannel, sizeof(double));
  if (el->vis2err == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_vis2elem->vis2err");
      mat_vis2elem_delete(el);
      return NULL;
    }
  
  /* el->cfx2 = cpl_calloc(nbchannel, sizeof(double)); */
  /* if (el->cfx2 == NULL) */
  /*   { */
  /*     cpl_msg_error(cpl_func,"could not allocate memory for mat_vis2elem->cfx2"); */
  /*     mat_vis2elem_delete(el); */
  /*     return NULL; */
  /*   } */
  /* el->cfx2err = cpl_calloc(nbchannel, sizeof(double)); */
  /* if (el->cfx2err == NULL) */
  /*   { */
  /*     cpl_msg_error(cpl_func,"could not allocate memory for mat_vis2elem->vis2err"); */
  /*     mat_vis2elem_delete(el); */
  /*     return NULL; */
  /*   } */
  
  el->bandflag = cpl_calloc(nbchannel, sizeof(cpl_boolean));
  if (el->bandflag == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_vis2elem->bandflag");
      mat_vis2elem_delete(el);
      return NULL;
    }
  return el;
}

/**
   @ingroup oivis2
   @brief Free memory of a single mat_vis2elem data structure.
   @param el Pointer to one mat_vis2elem data structure.
   @return A cpl_error_code
 
   This method frees the memory of a mat_vis2elem data structure.
*/
cpl_error_code mat_vis2elem_delete(mat_vis2elem *el)
{
  mat_assert_value((el != NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no mat_vis2elem (el) argument given");
  
  if (el->vis2 != NULL)
    {
      cpl_free(el->vis2);
      el->vis2 = NULL;
    }
  if (el->vis2err != NULL)
    {
      cpl_free(el->vis2err);
      el->vis2err = NULL;
    }
  
  /* if (el->cfx2 != NULL) */
  /*   { */
  /*     cpl_free(el->cfx2); */
  /*     el->cfx2 = NULL; */
  /*   } */
  /* if (el->cfx2err != NULL) */
  /*   { */
  /*     cpl_free(el->cfx2err); */
  /*     el->cfx2err = NULL; */
  /*   } */
  
  if (el->bandflag != NULL)
    {
      cpl_free(el->bandflag);
      el->bandflag = NULL;
    }
  cpl_free(el);
  return CPL_ERROR_NONE;
}

/**
   @ingroup oivis2
   @brief This method creates a new mat_oivis2 structure with all pointers set to NULL.
   @param nbvis2     Number of measurements.
   @param nbchannel  Number of spectral channels.
   @returns Pointer to a new allocated mat_oivis2 structure on success or NULL on failure.
 
   This method creates a new mat_oivis2 structure and sets all values to the default. For
   pointers the value is NULL for primitive data types the value is set to 0. The size
   of the element list (member list_vis2 of the mat_oivis2 data structure) is a multiple
   of CHUNK_SIZE (1024). This allows that after creation of a mat_vis2 data structure
   new elements (mat_vis2elem) can be created an added to that structure.
*/
mat_oivis2 *mat_oivis2_new(int nbvis2, int nbchannel)
{
  mat_oivis2 *oivis2;
  int         i;

  oivis2 = (mat_oivis2 *)cpl_calloc(1, sizeof(mat_oivis2));
  if (oivis2 == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oivis2");
      return NULL;
    }
  oivis2->nbvis2 = nbvis2;
  oivis2->nbchannel = nbchannel;
  if (nbvis2 != 0)
    { /* ensure that the vector of pointers is a multiple of CHUNK_SIZE */
      size_t size = ((nbvis2 + CHUNK_SIZE - 1)/CHUNK_SIZE)*CHUNK_SIZE;
      oivis2->list_vis2 = (mat_vis2elem **)cpl_calloc(size, sizeof(mat_vis2elem *));
      if (oivis2->list_vis2 == NULL)
	{
	  cpl_msg_error(cpl_func,"could not allocate memory for mat_oivis2->list_vis2");
	  mat_oivis2_delete(oivis2);
	  return NULL;
	}
      for (i = 0; i < nbvis2; i++) 
	{
	  oivis2->list_vis2[i] = mat_vis2elem_new(nbchannel);
	  if (oivis2->list_vis2[i] == NULL)
	    {
	      mat_oivis2_delete(oivis2);
	      return NULL;
	    }
	}
    }
  return oivis2;
}

/**
   @ingroup oivis2
   @brief Creates and add a new mat_vis2elem data structure to an mat_oivis2 data structure.
   @param oivis2  This represents an existing OI_VIS2 binary table.
   @returns A newly created and initialized mat_vis2elem data structure.

   This function allocates a new mat_vis2elem data structure representing a squared visibility.
   The number of wavelengths is given by the mat_oivis2 data structure (member nbchannel).
   The new VIS2 element is added to the list of VIS2 elements in the specified OI_VIS2 binary
   table (parameter oivis2). The size of the element list (member list_vis2 of the mat_oivis2
   data structure) is a multiple of CHUNK_SIZE (1024).
*/
mat_vis2elem *mat_oivis2_elem_new(mat_oivis2 *oivis2)
{
  mat_vis2elem *elem = NULL;

  mat_assert_value((oivis2!=NULL),CPL_ERROR_NULL_INPUT,NULL,"no mat_oivis2 (oivis2) argument given");
  if ((oivis2->nbvis2 % CHUNK_SIZE) == 0)
    { /* make the vector for element pointers bigger if neccessary */
      mat_vis2elem **nlist = NULL;
      int            i;
      nlist = cpl_realloc(oivis2->list_vis2, (oivis2->nbvis2 + CHUNK_SIZE)*sizeof(mat_vis2elem *));
      if (nlist == NULL)
	{
	  cpl_msg_error(cpl_func,"could not reallocate memory for mat_oivis2->list_vis2");
	  return NULL;
	}
      /* the memory is not initialized after a realloc */
      for (i = oivis2->nbvis2; i < oivis2->nbvis2 + CHUNK_SIZE; i++)
	{
	  nlist[i] = NULL;
	}
      oivis2->list_vis2 = nlist;
    }
  elem = mat_vis2elem_new(oivis2->nbchannel);
  if (elem == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oivis2->list_vis2[%d]", oivis2->nbvis2);
      return NULL;
    }
  oivis2->list_vis2[oivis2->nbvis2] = elem;
  oivis2->nbvis2++;
  return elem;
}

/**
   @ingroup oivis2
   @brief Free memory of a single mat_oivis2 data structure.
   @param oivis2 Pointer to one mat_oivis data structure.
   @return A cpl_error_code
 
   This method frees the memory of a mat_oivis2 data structure. This includes
   all embedded data structures (strings and mat_vis2elem data structures).
*/
cpl_error_code mat_oivis2_delete(mat_oivis2 *oivis2)
{
  int i;

  mat_assert_value((oivis2!=NULL),CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT,"no mat_oivis2 (oivis2) argument given");
  if (oivis2->dateobs != NULL)
    {
      cpl_free(oivis2->dateobs);
      oivis2->dateobs = NULL;
    }
  if (oivis2->arrayname != NULL)
    {
      cpl_free(oivis2->arrayname);
      oivis2->arrayname = NULL;
    }
  if (oivis2->insname != NULL)
    {
      cpl_free(oivis2->insname);
      oivis2->insname = NULL;
    }
  if (oivis2->list_vis2 != NULL)
    {
      for (i = 0; i < oivis2->nbvis2; i++)
	{
	  if (oivis2->list_vis2[i] != NULL)
	    {
	      mat_vis2elem_delete(oivis2->list_vis2[i]);
	      oivis2->list_vis2[i] = NULL;
	    }
	}
      cpl_free(oivis2->list_vis2);
      oivis2->list_vis2 = NULL;
      oivis2->nbvis2 = 0;
    }
  cpl_free(oivis2);
  return CPL_ERROR_NONE;
}

/**
   @ingroup oivis2
   @brief Load a mat_oivis2 data structure from a cpl table and a propertylist
   @param plist  This CPL propertylist contains the header keywords.
   @param table  This CPL table contains the OI_VIS2 vinary table.
   @return A newly allocated an filled mat_oivis2 or NULL on failure.

   This function loads an entire OI_VIS2 binary table into an internal data structure.
   The data structure contains no members for the columns from the OI-FITS version 2 standard.
*/
mat_oivis2 *mat_oivis2_from_table(cpl_propertylist *plist, cpl_table *table)
{
  mat_oivis2  *oivis2;
  int          nbvis2;
  int          nbchannel;
  int          j, k;

  mat_assert_value((plist != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_propertylist (plist) argument given");
  mat_assert_value((table != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_table (table) argument given");

  nbvis2 = cpl_table_get_nrow(table);
  /* old code
  nbchannel = cpl_array_get_size(cpl_table_get_array (table, "VIS2DATA", 0));
  if (nbchannel == 0) nbchannel = 1; // no array!
  */
  /* new code */
  
  if (cpl_table_get_column_depth(table, "VIS2DATA") == 0)
    {
      nbchannel = 1;
    }
  else
    {
      nbchannel = cpl_table_get_column_dimension(table, "VIS2DATA", 0);
    }
  cpl_msg_info(cpl_func, "VIS2 table has %d wavelength data", nbchannel);
  oivis2 = mat_oivis2_new(nbvis2, nbchannel);
  if (oivis2 == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oivis2");
      return NULL;
    }

  oivis2->extver    = mat_propertylist_get_int_default(plist, "EXTVER", 1);
  oivis2->arrayname = mat_propertylist_get_string_default(NULL, 0, plist, "ARRNAME", "empty");
  oivis2->dateobs   = mat_propertylist_get_string_default(NULL, 0, plist, "DATE-OBS", "empty");
  oivis2->insname   = mat_propertylist_get_string_default(NULL, 0, plist, "INSNAME", "empty");

  for(j = 0; j < oivis2->nbvis2; j++) 
    {
      mat_vis2elem  *el = oivis2->list_vis2[j];
      // copy the values from the table into the vis2 element
      el->targetid   = cpl_table_get_int    (table, "TARGET_ID", j, NULL);
      el->time       = cpl_table_get_double (table, "TIME", j, NULL);
      el->dateobsmjd = cpl_table_get_double (table, "MJD", j, NULL);
      el->exptime    = cpl_table_get_double (table, "INT_TIME", j, NULL);
      el->ucoord     = cpl_table_get_double (table, "UCOORD", j, NULL);
      el->vcoord     = cpl_table_get_double (table, "VCOORD", j, NULL);
      el->stationindex[0]= cpl_array_get_int(cpl_table_get_array(table, "STA_INDEX", j),0, NULL);
      el->stationindex[1] = cpl_array_get_int(cpl_table_get_array(table, "STA_INDEX", j), 1, NULL);
      if (nbchannel == 1)
	{
	  el->vis2[0]     = cpl_table_get_double(table, "VIS2DATA",j, NULL);
	  el->vis2err[0]  = cpl_table_get_double(table, "VIS2ERR", j, NULL);
	  
	  /* el->vis2[0]     = cpl_table_get_double(table, "CFX2DATA",j, NULL); */
	  /* el->vis2err[0]  = cpl_table_get_double(table, "CFX2ERR", j, NULL); */
	  
	  el->bandflag[0] = (cpl_table_get_int(table,   "FLAG", j, NULL) == 1);
	}
      else
	{
	  for (k = 0; k < oivis2->nbchannel; k++) 
	    {
	      el->vis2[k] = cpl_array_get_double(cpl_table_get_array(table, "VIS2DATA", j), k, NULL);
	      el->vis2err[k] = cpl_array_get_double(cpl_table_get_array(table, "VIS2ERR", j), k, NULL);
	      
	      /* el->vis2[k] = cpl_array_get_double(cpl_table_get_array(table, "CFX2DATA", j), k, NULL); */
	      /* el->vis2err[k] = cpl_array_get_double(cpl_table_get_array(table, "CFX2ERR", j), k, NULL); */
	      
	      el->bandflag[k] = (cpl_array_get_int(cpl_table_get_array(table,"FLAG", j), k, NULL) == 1);
	    }
	}
    }
  return oivis2;
}

/**
   @ingroup oivis2
   @brief Convert a mat_oivis2 data structure into a CPL table and a propertylist.
   @param oivis2  The data structure containing a whole OI_VIS2 binary table.
   @param plist   This propertylist will contain the header keywords of the OI_VIS2 binary table.
   @return A CPL table containing the OI_VIS2 binary table or NULL on failure.

   A new CPL table will be created which will contain all squared visibilities. The header keywords of
   the OI_VIS2 binary table will be stored in the provided propertylist (parameter plist).
   The CPL table does not contain the columns from the OI-FITS version 2 standard.
*/
cpl_table *mat_oivis2_to_table(mat_oivis2 *oivis2, cpl_propertylist *plist)
{
  cpl_table    *table;
  cpl_array    *arr;
  int           i, j;
  int          *flags;

  if (plist != NULL)
    {
      cpl_propertylist_append_string(plist, "EXTNAME","OI_VIS2");
      cpl_propertylist_append_int   (plist, "EXTVER", oivis2->extver);
      if ( cpl_propertylist_has(plist,"OI_REVN") )
	{
	  cpl_propertylist_erase(plist,"OI_REVN");
	}
      cpl_propertylist_append_int   (plist, "OI_REVN",2);
      cpl_propertylist_append_string(plist, "DATE-OBS",oivis2->dateobs);
      cpl_propertylist_append_string(plist, "ARRNAME",oivis2->arrayname);
      cpl_propertylist_append_string(plist, "INSNAME",oivis2->insname);
    }
  flags = (int *)cpl_calloc(oivis2->nbchannel, sizeof(int));
  if (flags == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for flag vector");
      return NULL;
    }
  table = cpl_table_new(oivis2->nbvis2);
  if (table == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_table");
      cpl_free(flags);
      return NULL;
    }
  // Add columns
  cpl_table_new_column     (table, "TARGET_ID", CPL_TYPE_INT);
  cpl_table_new_column     (table, "TIME", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "TIME","s");
  cpl_table_new_column     (table, "MJD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "MJD","day");
  cpl_table_new_column     (table, "INT_TIME", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "INT_TIME","s");
  if (oivis2->nbchannel == 1)
    {
      cpl_table_new_column(table, "VIS2DATA", CPL_TYPE_DOUBLE);
      cpl_table_new_column(table, "VIS2ERR", CPL_TYPE_DOUBLE);
      
      /* cpl_table_new_column(table, "CFX2DATA", CPL_TYPE_DOUBLE); */
      /* cpl_table_new_column(table, "CFX2ERR", CPL_TYPE_DOUBLE); */
    }
  else
    {
      cpl_table_new_column_array(table, "VIS2DATA", CPL_TYPE_DOUBLE, oivis2->nbchannel);
      cpl_table_new_column_array(table, "VIS2ERR", CPL_TYPE_DOUBLE, oivis2->nbchannel);
      
      /* cpl_table_new_column_array(table, "CFX2DATA", CPL_TYPE_DOUBLE, oivis2->nbchannel); */
      /* cpl_table_new_column_array(table, "CFX2ERR", CPL_TYPE_DOUBLE, oivis2->nbchannel); */
    }
  cpl_table_new_column      (table, "UCOORD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit (table, "UCOORD","m");
  cpl_table_new_column      (table, "VCOORD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit (table, "VCOORD","m");
  cpl_table_new_column_array(table,"STA_INDEX", CPL_TYPE_INT, 2);
  if (oivis2->nbchannel == 1)
    {
      cpl_table_new_column(table, "FLAG", CPL_TYPE_INT);
    }
  else
    {
      cpl_table_new_column_array(table, "FLAG", CPL_TYPE_INT, oivis2->nbchannel);
    }
  // Fill the table
  for(i=0;i<oivis2->nbvis2;i++)
    {
    
      mat_vis2elem  *el = oivis2->list_vis2[i];
      cpl_table_set_int   (table, "TARGET_ID", i, el->targetid);
      cpl_table_set_double(table, "TIME",i, el->time);
      cpl_table_set_double(table, "MJD",i, el->dateobsmjd);
      cpl_table_set_double(table, "INT_TIME",i, el->exptime);
      if (oivis2->nbchannel == 1)
	{
	  cpl_table_set_double(table,"VIS2DATA",i, el->vis2[0]);
	  cpl_table_set_double(table,"VIS2ERR",i,  el->vis2err[0]);
	  
	  /* cpl_table_set_double(table,"CFX2DATA",i, el->cfx2[0]); */
	  /* cpl_table_set_double(table,"CFX2ERR",i,  el->cfx2err[0]); */
	}
      else
	{
	  arr = cpl_array_wrap_double(el->vis2, oivis2->nbchannel);
	  cpl_table_set_array(table,"VIS2DATA", i, arr);
	  cpl_array_unwrap(arr);
	  arr = cpl_array_wrap_double(el->vis2err, oivis2->nbchannel);
	  cpl_table_set_array(table,"VIS2ERR",i,arr);
	  cpl_array_unwrap(arr);
	  
	  /* arr = cpl_array_wrap_double(el->cfx2, oivis2->nbchannel); */
	  /* cpl_table_set_array(table,"CFX2DATA", i, arr); */
	  /* cpl_array_unwrap(arr); */
	  /* arr = cpl_array_wrap_double(el->cfx2err, oivis2->nbchannel); */
	  /* cpl_table_set_array(table,"CFX2ERR",i,arr); */
	  /* cpl_array_unwrap(arr); */
	}
      cpl_table_set_double(table, "UCOORD",i,el->ucoord);
      cpl_table_set_double(table, "VCOORD",i,el->vcoord);
      arr = cpl_array_wrap_int(el->stationindex, 2);
      cpl_table_set_array(table,  "STA_INDEX", i, arr);
      cpl_array_unwrap(arr);
      if (oivis2->nbchannel == 1)
	{
	  cpl_table_set_int(table, "FLAG", i, (el->bandflag[0] != 0));
	}
      else
	{
	  for(j=0;j<oivis2->nbchannel;j++)
	    {
	      if (el->bandflag[j])
		{
		  flags[j] = 1;
		}
	      else
		{
		  flags[j] = 0;
		}
	    }
	  arr = cpl_array_wrap_int(flags, oivis2->nbchannel);
	  cpl_table_set_array(table,"FLAG",i, arr);
	  cpl_array_unwrap(arr);
	}
    }
  cpl_table_set_column_savetype(table, "TARGET_ID", CPL_TYPE_SHORT);
  cpl_table_set_column_savetype(table, "STA_INDEX", CPL_TYPE_SHORT);
  cpl_table_set_column_savetype(table, "FLAG",      CPL_TYPE_BOOL);
  cpl_free(flags);
  return table;
}

/**
   @ingroup oivis2
   @brief Load a OI_VIS2 binary table from a frame into a mat_oivis2 structure.
   @param frame  A frame containing a OI_VIS2 binary table.
   @return A newly allocated an filled mat_oivis2 or NULL on failure.

   This function scans all extensions inside a frame and loads the first OI_VIS2 binary table
   into a newly created mat_oivis2 data structure. If the frame does not contain any
   OI_VIS2 binary table, NULL is returned.
*/
mat_oivis2 *mat_oivis2_load(cpl_frame *frame)
{
  int nbExtent;
  int  iExtent;

  mat_assert_value((frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_frame (frame) argument given");
  nbExtent = cpl_frame_get_nextensions(frame);
  for(iExtent = 1; iExtent <= nbExtent; iExtent++)
    {
      cpl_propertylist *plist = cpl_propertylist_load(cpl_frame_get_filename(frame), iExtent);
      char *keyExtname = (char *)cpl_propertylist_get_string(plist, "EXTNAME");
      if ((keyExtname != NULL) && (strcmp(keyExtname, "OI_VIS2") == 0))
	{ /* Only the first OI_VIS2 binary table is loaded! */
	  cpl_table *table = cpl_table_load(cpl_frame_get_filename(frame), iExtent, 0);
	  mat_oivis2 *oivis2 = mat_oivis2_from_table(plist, table);
	  cpl_table_delete(table);
	  cpl_propertylist_delete(plist);
	  return oivis2;
	}
      cpl_propertylist_delete(plist);
    }
  return NULL;
}

/**
   @ingroup oivis2
   @brief Append an OI_VIS2 binary table to an existing OIFITS file.
   @param filename  Name of the FITS file.
   @param oivis2    The mat_oivis2 structure containing a whole OI_VIS2 binary table.
   @return cpl_error_code

   This function converts a mat_oivis2 data structure into a CPL binary table and
   appends this at the end of an already existing FITS file.
*/
cpl_error_code  mat_oivis2_append(char *filename,mat_oivis2 *oivis2)
{
  cpl_table *oivis2Tab = NULL;
  cpl_propertylist *keyTab = NULL;
  cpl_errorstate prestate = cpl_errorstate_get();

  /*Check input parameters*/
  mat_assert_value((oivis2!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oivis2 (oivis2) argument given");
  mat_assert_value((filename!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no filename given");
  
  keyTab=cpl_propertylist_new();
  oivis2Tab = mat_oivis2_to_table(oivis2, keyTab);
  cpl_table_save(oivis2Tab,NULL,keyTab,filename,CPL_IO_EXTEND);

  cpl_propertylist_delete(keyTab);
  cpl_table_delete(oivis2Tab);
  if (!cpl_errorstate_is_equal(prestate))
    {
      cpl_errorstate_set(prestate);
      return CPL_ERROR_UNSPECIFIED;
    }
  return CPL_ERROR_NONE;
}

/**
   @ingroup oivis2
   @brief Get the time span of the observation as MJD values (beginning and end).
   @param oivis2   The mat_oivis2 structure containing a OI_VIS2 binary table.
   @param min_mjd  The beginning of the observation.
   @param max_mjd  The end of the observation.
   @return cpl_error_code

   The time span of a binary table is used to group OI_VIS, OI_VIS2 and OI_T3 binary
   tables into one tupel (see the mat_oifits data structure).
*/
cpl_error_code mat_oivis2_get_mjd_span(mat_oivis2 *oivis2, double *min_mjd, double *max_mjd)
{
  int i;

  /*Check input parameters*/
  mat_assert_value((oivis2!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oivis2 (oivis2) argument given");
  mat_assert_value((min_mjd!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no start MJD given");
  mat_assert_value((max_mjd!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no end MJD given");
  *min_mjd =  1.0e12;
  *max_mjd = -1.0e12;
  for (i = 0; i < oivis2->nbvis2; i++)
    {
      mat_vis2elem *el = oivis2->list_vis2[i];
      *min_mjd = fmin(*min_mjd, el->dateobsmjd);
      *max_mjd = fmax(*max_mjd, el->dateobsmjd);
    }
  return CPL_ERROR_NONE;
}
