/* $Id: mat_oivis2.h,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License,	or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_oivis2.h $
 */

#ifndef MAT_OIVIS2_H
#define MAT_OIVIS2_H

/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/

#include "cpl.h"

/*-----------------------------------------------------------------------------
  Defines
  ----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  Low Level Structures
  ----------------------------------------------------------------------------*/

/* Structure mat_vis2elem */
/**
   @ingroup oivis2
   @brief Structure containing one measurement of an OI_VIS2.

   This data structure represents one entry of an OI_VIS binary table.
*/
typedef struct {
    int		 targetid;      /*!< Target number as index into the OI_TARGET binary table (always 1 for MATISSE OI-FITS files). */
    double	 time;          /*!< UTC time of observation. */
    double	 dateobsmjd;    /*!< UTC date of observation (MJD). */
    double	 exptime;       /*!< Integration time per measurement in seconds. */
    double	*vis2;          /*!< Array of squared visibilities. */
    double	*vis2err;       /*!< Array of errors in squared visibilities. */
    
  /* double      *cfx2; */            /*!< Array of squared correlated fluxes. */
  /* double      *cfx2err; */         /*!< Array of errors in squared correlated fluxes. */
    
    double       ucoord;          /*!< U coordinate of the data in meter. */
    double       vcoord;          /*!< V coordinate of the data in meter. */
    int          stationindex[2]; /*!< Station numbers contributing to the data. */
    cpl_boolean *bandflag;        /*!< Flag if a data point is invalid (should be ignored). */
} mat_vis2elem;

/* Structure mat_oivis */
/**
   @ingroup oivis2
   @brief Structure containing an OI_VIS2 binary table.

   This data structure represents one OI_VIS2 binary table of an OI-FITS file. A standard compliant OI-FITS file
   can contain several of these binary tables. This is implemented in the mat_oifits data structure.
*/
typedef struct {
    int   extver;			/*!< Extension version (optional). */
    char          *dateobs;	/*!< UTC start date of observations. */
    char          *arrayname;	/*!< Identifies the corresponding OI_ARRAY binary table. */
    char          *insname;	/*!< Identifies the corresponding OI_WAVELENGTH binary table. */
    int            nbchannel;	/*!< Number of spectral channels (equivalent to the corresponding OI_WAVELENGTH binary table). */
    int            nbvis2;	/*!< Number of squared visibilities in this binary table (equivalent with the number of rows). */
    mat_vis2elem **list_vis2;	/*!< List containing all squared visibilities stored in mat_vis2elem data structures. */
} mat_oivis2;

/*-----------------------------------------------------------------------------
  High Level Structures
  ----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/
mat_vis2elem	*mat_vis2elem_new(int nbchannel);

cpl_error_code	mat_vis2elem_delete(mat_vis2elem *el);

mat_oivis2	*mat_oivis2_new(int nbvis2, int nbchannel);

mat_vis2elem *mat_oivis2_elem_new(mat_oivis2 *oivis2);

cpl_error_code	mat_oivis2_delete(mat_oivis2 *oivis2);

mat_oivis2 *mat_oivis2_from_table(cpl_propertylist	*plist,
				  cpl_table		*table);

cpl_table *mat_oivis2_to_table(mat_oivis2	*oivis2,
			       cpl_propertylist *plist);

mat_oivis2	*mat_oivis2_load(cpl_frame *frame);

cpl_error_code  mat_oivis2_append(char *filename,mat_oivis2 *oivis2);

cpl_error_code mat_oivis2_get_mjd_span(mat_oivis2 *oivis2, double *min_mjd, double *max_mjd);


#endif
