/* $Id: mat_opd_wvpo.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 //  *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_opd_wvpo.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#include <string.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/
#include <math.h>

#include "mat_opd_wvpo.h"
#include "mat_error.h"
#include <sys/time.h>

/* must not be followed by a semicolon!
 * gcc on mac has omp but it doesn't work for nontrivial cases as libc lacks
 * alloca */
#if defined (_OPENMP) && !defined( __APPLE__)
  #define HDRL_OMP(x) _Pragma (#x)
#else
  #define HDRL_OMP(x)
#endif

/*-----------------------------------------------------------------------------
  Define
  ----------------------------------------------------------------------------*/
#define NB_ITERATION_MAX 20
#define EPS_MAX 1.E-3

/*-----------------------------------------------------------------------------
  Functions prototypes
  ----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
   @ingroup opdwvpo
   @brief Apply zoom coefficients for transforming a 
   spectral channel of the correlated flux.
   @param p_row : spectral channel of the correlated flux                  
   @param zoom : zooming parameters 
   @param p_res : result
   @return cpl_error_code
*/
/*----------------------------------------------------------------------------*/
cpl_error_code interpolate_corrflux(cpl_vector	*p_row, double zoom, 
				    cpl_vector	*p_res)
{
  int			 xmid	= 0;
  int			 i	= 0;
  int			 n	= 0;
  int			 npt	= 0;
  cpl_vector		*x	= NULL;
  cpl_vector		*X	= NULL;
  cpl_vector		*v	= NULL;
  cpl_bivector	*p_fout	= NULL;
  cpl_bivector	*p_fref	= NULL;
  
  // Check input parameters
  mat_assert_value((p_row!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no p_row argument given");
  mat_assert_value((p_res!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no p_res argument given");
  
  n = cpl_vector_get_size(p_row);
  x = cpl_vector_new(n) ;
  
  npt	 = cpl_vector_get_size(p_res);
  X	 = cpl_vector_new(npt);
  v	 = cpl_vector_new(npt);
  xmid = ceil(n/2);
  for(i = 0;i<n;i++)
    {
      cpl_vector_set(x,i,(i-xmid)/(double)n);
    }
  for(i = 0;i<npt;i++)
    {  
      cpl_vector_set(X,i,cpl_vector_get(x,i)*zoom);		   
    }

  p_fref = cpl_bivector_wrap_vectors(x,p_row);
  p_fout = cpl_bivector_wrap_vectors(X,v);
  
  if (cpl_bivector_interpolate_linear(p_fout,p_fref) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func,"Error during the interpolation of the correlated flux");
      cpl_vector_delete(x);
      cpl_vector_delete(X);
      cpl_vector_delete(v);
      return CPL_ERROR_UNSPECIFIED;
    }
  for(i = 0;i<npt;i++) {
    cpl_vector_set(p_res,i,cpl_vector_get(v,i));
  }
  if (x != NULL) cpl_vector_delete(x);
  if (X != NULL) cpl_vector_delete(X);
  if (v != NULL) cpl_vector_delete(v);
  if (p_fout != NULL) cpl_free(p_fout);
  if (p_fref != NULL) cpl_free(p_fref);

  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup opdwvpo
   @brief Compute the OPD and Water Vapor Offset from a mat_corrflux structure. 
   @param corrFlux           Correlated Flux
   @param oiopdwvpo          mat_oiopdwvpo structure contaning the results
   @return 0 if no error 
*/
/* @param chromaticOpdFit    True if Fit of Humidty, Temperature, pression    */
/* int mat_opd_wvpo(mat_corrflux *corrFlux,mat_oiopdwvpo *oiopdwvpo, cpl_parameter *chromaticOpdFit) */
int mat_opd_wvpo(mat_corrflux *corrFlux,mat_oiopdwvpo *oiopdwvpo, char *fileGRAVITY)
{
    /* Ranges and indices */
    int	iReg = 0,iFrame=0,iWlen=0,iWlen0=0, iBase=0;
    int	nbRegion;
    int	nbWlen;
    
  int		  i		  = 0;
  int		  j		  = 0;
  int		  k		  = 0;
  /* int		  l		  = 0; */
  /* int		  n		  = 0; */
  //  int		  cptWave	  = 0;
  int		  nbWave	  = 0;
  int		  uPixelMin	  = 0;
  int		  uPixelMax	  = 0;
  int		  pis		  = 0;
  cpl_vector	 *dispCoef	  = NULL;
  double	  lambda	  = 0.;
  /* double	  dOverLambda	  = 0.; */
  double	  nrjReal	  = 0.;
  double	  nrjImag	  = 0.;
  cpl_vector	**opd		  = NULL;
  cpl_vector	**opdErr	  = NULL;
  cpl_vector	**tempoff	  = NULL;
  cpl_vector	**tempoffErr	  = NULL;
  cpl_vector	**humoff	  = NULL;
  cpl_vector	**humoffErr	  = NULL;
  cpl_vector	 *wavelength	  = NULL;
  /* int    cptEnd      = 0; */
  /* double temp        = 0.; */
  /* double funcObj     = 0.; */
  /* double funcObjCur  = 0.; */
  /* double funcObjPrec = 0.; */
  /* double funcObjMin  = 0.; */
  cpl_vector	 *foo		  = NULL;
  /* const gsl_rng_type *T; */
  /* gsl_rng *r; */
  double	  alpha		  = 0.;
  double	  zr		  = 0.;
  double	  zi		  = 0.;
  /* double piston      = 0.; */
  /* double pistonCur   = 0.; */
  /* double pistonFinal = 0.; */
  /* double deno        = 0.; */
  int		  nbBase	  = 0;
  int		  cpt		  = 0;
  double	  pressure	  = 0.;
  double	  temperature	  = 0.;
  double	  humidity	  = 0.;
  cpl_vector	 *DLstatic	  = NULL;
  cpl_vector	 *DLopl		  = NULL;
  double	  argOpdChrom	  = 0.;
  double	  refractiveIndex = 0.;
  cpl_vector	 *z		  = NULL;
  /* asacg_parm	  cgParm; */
  /* asa_parm	  asaParm; */
  double	 *x		  = NULL;
  double	 *lo		  = NULL;
  double	 *hi		  = NULL;
  int		  nx		  = 3;
  /* cpl_vector	 *asaUser	  = NULL; */
  int		  icpt0		  = 0;
  int		  ncpt0		  = 0;
  /* int		  icpt1		  = 0; */
  /* int		  ncpt1		  = 0; */
  /* int		  icpt2		  = 0; */
  /* int		  ncpt2		  = 0; */
  /* double	 *xmin		  = NULL; */
  double	 *xminFirst	  = NULL;
  double	 *xminErr	  = NULL;
  double	  objFuncCurrent  = 0.;
  double	 *objFuncMin	  = NULL;
  /* double	 *vecDiagCov	  = NULL; */
  cpl_vector	 *chi2		  = NULL;
  int		  icptOne1	  = 0;
  int		  icptOne2	  = 0;
  char	 *detName	  = NULL;
  char	  band;
  char	  key[15];
  float	  lambdamin	  = 0.;
  float	  lambdamax	  = 0.;
  float	  wlenmin	  = 1e99;
  float	  wlenmax	  = 0.;
  int		  detNum	  = 0;
  int		  resolution	  = 0;
  char	 *specResData	  = NULL;
  double	  posFringePeak	  = 0.;
  double	  modulus	  = 0.;

  cpl_vector *phasorRealRef=NULL;
  cpl_vector *phasorImagRef=NULL;
  double waveRef=0.;
  double val;
  int nbFrameOverPhot=0;
  int corner=0;
  /* int dimSpatial=0; */
  int dimSpectral=0;
  cpl_image *ftReal=NULL;
  cpl_image *ftImag=NULL;
  double phiRealRef=0.;
  double phiImagRef=0.;
  double factor=0.;
  double org=0.;
  int iWlenRef=0;
  double valmin=0.;
  int algoCophasing=1;
  int nbFrameTarget=0;
  int expno = 0;
  char *output=0;

  
  expno = cpl_propertylist_get_int(corrFlux->keywords,"ESO TPL EXPNO");
    
  if (cpl_propertylist_has(corrFlux->keywords,"PRO FRAME TARGET"))
    {
      nbFrameTarget=cpl_propertylist_get_int(corrFlux->keywords,"PRO FRAME TARGET");
    }
  else
    {
      nbFrameTarget=corrFlux->nbframe;
    }


  // Check input parameters
  mat_assert((corrFlux!=NULL),CPL_ERROR_NULL_INPUT, "no mat_corrflux	\
argument given");
  mat_assert((oiopdwvpo!=NULL),CPL_ERROR_NULL_INPUT, "no mat_oiopdwvpo	\
argument given");


  nbFrameOverPhot=corrFlux->nbframe/corrFlux->imgdet->nbregion;
  foo=cpl_vector_new(10);
  phasorRealRef=cpl_vector_new(6);
  phasorImagRef=cpl_vector_new(6);

  /* Read the detector type (and hence 1st approx of band) */
  detName=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO DET CHIP NAME");
  if (detName == NULL)
    {
      cpl_msg_error(cpl_func,
		    "no ESO DET CHIP NAME keyword in frame");
      cpl_vector_delete(phasorRealRef);
      cpl_vector_delete(phasorImagRef);
      cpl_vector_delete(foo);
      return -1;
    }
  mat_identify_det(detName, &band, &detNum);

  if (fileGRAVITY != NULL && detNum == 2)
    {
      algoCophasing=3;
      cpl_msg_info(cpl_func,"GRAAMAT dat found -> K4N cophasing");
    }
  
  /* Read the spectral resolution */
  sprintf(key, "ESO INS DI%c ID",band);
  specResData  = (char *)cpl_propertylist_get_string(corrFlux->keywords,key);
  if (specResData == NULL) {
      cpl_msg_error(cpl_func,
		    "no ESO INS DI%c ID keyword in frame",band);
    cpl_vector_delete(phasorRealRef);
    cpl_vector_delete(phasorImagRef);
    cpl_vector_delete(foo);
    return -1;
  }
  char *fil = NULL;
  fil = (char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS FIL NAME");
  mat_identify_res(specResData, fil, detNum, &resolution);
  
  
  // Loading of the dispersion coefficients
  dispCoef	= cpl_vector_new(N_DEG_DISPERSION_LAW+1);
  cpl_vector_set(dispCoef,0,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF0"));
  cpl_vector_set(dispCoef,1,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF1"));
  cpl_vector_set(dispCoef,2,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF2"));
  cpl_vector_set(dispCoef,3,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF3"));
  cpl_vector_set(dispCoef,4,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF4"));


      
  /* Set the useful range of wavelengths to compute OPD */
  /* First of all retrieve the actual wavelength range */
  nbRegion = corrFlux->imgdet->nbregion;
  for(iReg = 0;iReg<nbRegion;iReg++)
  {
      cpl_msg_info(cpl_func,"Determining wavelength range");
      nbWlen = corrFlux->imgdet->list_region[iReg]->naxis[1];
        iWlen0 = corrFlux->imgdet->list_region[iReg]->corner[1];
        for(iWlen = 0;iWlen<nbWlen;iWlen++)
	{
            /* Compute wavelength and set wlen interval */
            lambda = mat_get_lambda(dispCoef, iWlen + iWlen0);
	    if(lambda < wlenmin)
		wlenmin = lambda;
	    if(lambda > wlenmax)
		wlenmax = lambda;
	}
  }
  mat_identify_wlrng(detNum, wlenmin, wlenmax, resolution, &lambdamin, &lambdamax);
  


  
  if (oiopdwvpo->array != NULL) {
    nbBase = (oiopdwvpo->array->nbstation*(oiopdwvpo->array->nbstation - 1))/2;
  } else {
    nbBase   = 6;
  }
  cpl_msg_info(cpl_func,"Number of baselines : %d",nbBase);



  
  // Loading the ambient conditions and the DL positions
  pressure	= cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS AMBI PRES");
  temperature	= cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS AMBI TEMP");
  humidity	= cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS AMBI RHUM");
  cpl_msg_info(cpl_func,"Ambient Pressure = %f",pressure);
  cpl_msg_info(cpl_func,"Ambient Temperature = %f",temperature);
  cpl_msg_info(cpl_func,"Ambient Relative Humidity = %f",humidity);

  DLstatic = cpl_vector_new(MAT_MAXTEL);
  DLopl    = cpl_vector_new(MAT_MAXTEL);
  for(i=0;i<MAT_MAXTEL;i++) {
    cpl_vector_set(DLstatic,i,0.);
    cpl_vector_set(DLopl,i,0.);
  }
  if ( cpl_propertylist_has(oiopdwvpo->keywords,"ESO ISS CONF A1L") )
    {
      cpl_vector_set(DLstatic,0,cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS CONF A1L"));
      cpl_vector_set(DLopl,0,(cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT1 OPL START")
			      +cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT1 OPL END"))/2.);
    }
  if ( cpl_propertylist_has(oiopdwvpo->keywords,"ESO ISS CONF A2L") ) {
    cpl_vector_set(DLstatic,1,cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS CONF A2L"));
    cpl_vector_set(DLopl,1,(cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT2 OPL START")
			    +cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT2 OPL END"))/2.);
  }
  if ( cpl_propertylist_has(oiopdwvpo->keywords,"ESO ISS CONF A3L") ) {
    cpl_vector_set(DLstatic,2,cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS CONF A3L"));
    cpl_vector_set(DLopl,2,(cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT3 OPL START")
			    +cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT3 OPL END"))/2.);
  }
  if ( cpl_propertylist_has(oiopdwvpo->keywords,"ESO ISS CONF A4L") ) {
    cpl_vector_set(DLstatic,3,cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS CONF A4L"));
    cpl_vector_set(DLopl,3,(cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT4 OPL START")
			    +cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT4 OPL END"))/2.);
  }
  for(i=0;i<MAT_MAXTEL;i++) {
    cpl_msg_info(cpl_func,"DL%d	STATIC = %f",i+1,cpl_vector_get(DLstatic,i));
    cpl_msg_info(cpl_func,"DL%d	OPL = %f",i+1,cpl_vector_get(DLopl,i));
  }



  cpl_msg_info(cpl_func,"Computation of the OPD and WVPO");
  nbWave=0;
  for(j=0;j<corrFlux->imgdet->nbregion;j++) {
    nbWave+=corrFlux->imgdet->list_region[j]->naxis[1];
  }

  opd        = cpl_malloc(nbBase*sizeof(cpl_vector *));
  opdErr     = cpl_malloc(nbBase*sizeof(cpl_vector *));
  tempoff    = cpl_malloc(nbBase*sizeof(cpl_vector *));
  tempoffErr = cpl_malloc(nbBase*sizeof(cpl_vector *));
  humoff     = cpl_malloc(nbBase*sizeof(cpl_vector *));
  humoffErr  = cpl_malloc(nbBase*sizeof(cpl_vector *));
  z = cpl_vector_new(nbBase);
  for(iBase=0;iBase<nbBase;iBase++)
    {
      opd[iBase]        = cpl_vector_new(nbFrameTarget);
      opdErr[iBase]     = cpl_vector_new(nbFrameTarget);
      tempoff[iBase]    = cpl_vector_new(nbFrameTarget);
      tempoffErr[iBase] = cpl_vector_new(nbFrameTarget);
      humoff[iBase]     = cpl_vector_new(nbFrameTarget);
      humoffErr[iBase]  = cpl_vector_new(nbFrameTarget);
      int idxStation[2] = {0,0};
      for(int iStation=0;iStation<MAT_MAXTEL;iStation++)
	{
	  if (oiopdwvpo->oiopd->list_opd[iBase]->stationindex[0] == oiopdwvpo->array->list_station[iStation]->staindex) {
	    idxStation[0]=iStation;
	  }
	  if (oiopdwvpo->oiopd->list_opd[iBase]->stationindex[1] == oiopdwvpo->array->list_station[iStation]->staindex) {
	    idxStation[1]=iStation;
	  }
	}
      cpl_vector_set(z,iBase,cpl_vector_get(DLstatic,idxStation[1])+cpl_vector_get(DLopl,idxStation[1])-
		     cpl_vector_get(DLstatic,idxStation[0])-cpl_vector_get(DLopl,idxStation[0]));
      /* printf("z= %f\n",cpl_vector_get(z,iBase)); */
    }
  wavelength    = cpl_vector_new(nbWave);

  cpl_imagelist * imglistReal = cpl_imagelist_new();
  cpl_imagelist * imglistImag = cpl_imagelist_new();
  cpl_image ** imgcurrentReal = NULL;
  cpl_image ** imgcurrentImag = NULL;
  imgcurrentReal = cpl_calloc((nbFrameTarget+1),sizeof(cpl_image *));
  imgcurrentImag = cpl_calloc((nbFrameTarget+1),sizeof(cpl_image *));
  for(iFrame=0;iFrame<=nbFrameTarget;iFrame++) {
    imgcurrentReal[iFrame] = cpl_image_new(nbWave,8,CPL_TYPE_DOUBLE);
    imgcurrentImag[iFrame] = cpl_image_new(nbWave,8,CPL_TYPE_DOUBLE);
  }
  int cptWaveList=0;
   
  for(iFrame=0;iFrame<nbFrameTarget;iFrame++) {
    if (iFrame % 100 == 0) cpl_msg_info(cpl_func,"Frame number %d",iFrame);
    cptWaveList=0;
    for(iReg=0;iReg<corrFlux->imgdet->nbregion;iReg++) {
      corner=corrFlux->imgdet->list_region[iReg]->corner[1];
      /* dimSpatial=corrFlux->imgdet->list_region[iReg]->naxis[0]; */
      dimSpectral=corrFlux->imgdet->list_region[iReg]->naxis[1];
      ftReal = corrFlux->list_corr[iReg*nbFrameOverPhot+iFrame]->imgreg[0];
      ftImag = corrFlux->list_corr[iReg*nbFrameOverPhot+iFrame]->imgreg[1];
      // Compute the energy of each fringe peak
      for(iWlen=0;iWlen<dimSpectral;iWlen++) {
	lambda=mat_get_lambda(dispCoef,iWlen+corner);
	cpl_vector_set(wavelength,cptWaveList,lambda);
	for(iBase=0;iBase<nbBase;iBase++) {
	  posFringePeak=mat_get_pos_fringe_peak(lambda,detNum,resolution,iBase);
	  uPixelMin=(int)posFringePeak;
	  uPixelMax=(int)posFringePeak+1;
		  
	  nrjReal=cpl_image_get(ftReal,uPixelMin+1,iWlen+1,&pis)*(uPixelMax-posFringePeak)+
	    cpl_image_get(ftReal,uPixelMax+1,iWlen+1,&pis)*(posFringePeak-uPixelMin);
	  nrjImag=cpl_image_get(ftImag,uPixelMin+1,iWlen+1,&pis)*(uPixelMax-posFringePeak)+
	    cpl_image_get(ftImag,uPixelMax+1,iWlen+1,&pis)*(posFringePeak-uPixelMin);

	  
	  // Correction of the chromatic OPD (following Mathar)
	  ////////////////////////////////////////////////////////////////////////////
	  // Compute the chromatic OPD
	  // Apply the correction to (phasorReal,phasorImag)
	  // PhasorRealCorr+j*PhasorImagCorr=(PhasorReal+j*PhasorImag)*exp(j*2*pi*OPDchrom/lambda)
	  refractiveIndex = mat_oiopd_mathar(lambda,pressure*100.,temperature+273.15,humidity);
    	  argOpdChrom = 0.*2*CPL_MATH_PI*refractiveIndex*cpl_vector_get(z,iBase)*1.E6/lambda;
	  //if (iFrame==0) printf("%d %f \n",iBase,refractiveIndex*cpl_vector_get(z,iBase));
	  //argOpdChrom=0.;
	  cpl_image_set(imgcurrentReal[iFrame],cptWaveList+1,iBase+2,
			nrjReal*cos(argOpdChrom)+nrjImag*sin(argOpdChrom));
	  cpl_image_set(imgcurrentImag[iFrame],cptWaveList+1,iBase+2,
			nrjImag*cos(argOpdChrom)-nrjReal*sin(argOpdChrom));

	  for(k=0;k<10;k++) {
	    cpl_vector_set(foo,k,cpl_image_get(ftReal,k+1,iWlen+1,&pis));
	  }
	  if (iBase==0) {
	    cpl_image_set(imgcurrentReal[iFrame],cptWaveList+1,8,cpl_vector_get_stdev(foo));
	  }
	  for(k=0;k<10;k++) {
	    cpl_vector_set(foo,k,cpl_image_get(ftImag,k+1,iWlen+1,&pis));
	  }
	  if (iBase==0) {
	    cpl_image_set(imgcurrentImag[iFrame],cptWaveList+1,8,cpl_vector_get_stdev(foo));
	  }
	  cpl_image_set(imgcurrentReal[iFrame],cptWaveList+1,1,lambda);///(1.-0.0041*lambda));
	  cpl_image_set(imgcurrentImag[iFrame],cptWaveList+1,1,lambda);///(1.-0.0041*lambda));
	}
	cptWaveList++;
      }
    }
  }

  if (algoCophasing == 2) {
    int nbInterp=0;
    int nbRef=0;
    int nbPadding=4096*2;
    cpl_vector *waveUniform=NULL;
    cpl_vector *vecwaveRef=NULL;
    cpl_vector *vecRealRef=NULL;
    cpl_vector *vecImagRef=NULL;
    cpl_vector *vecRealInterp=NULL;
    cpl_vector *vecImagInterp=NULL;
    cpl_image *imgPadding=NULL;
    cpl_image *fftPadding=NULL;

    cpl_bivector *foutReal=NULL;
    cpl_bivector *frefReal=NULL;
    cpl_bivector *foutImag=NULL;
    cpl_bivector *frefImag=NULL;
    
    nbInterp=(int)pow(2.,(int)(log(nbWave)/log(2.))+1);
    /* printf("nbInterp=%d\n",nbInterp); */
    for(iWlen=0;iWlen<nbWave;iWlen++) {
      lambda=cpl_vector_get(wavelength,iWlen);
      if (lambda>=lambdamin && lambda<=lambdamax) {
	nbRef++;
      }
    }
    vecwaveRef=cpl_vector_new(nbRef);

    if (detNum == 1) {
      iWlenRef=nbRef-1;
      for(iWlen=0;iWlen<nbWave;iWlen++) {
	lambda=cpl_vector_get(wavelength,iWlen);
	if (lambda>=lambdamin && lambda<=lambdamax) {
	  cpl_vector_set(vecwaveRef,iWlenRef,lambda);
	  iWlenRef--;
	}
      }
    } else {
      iWlenRef=0;
      for(iWlen=0;iWlen<nbWave;iWlen++) {
	lambda=cpl_vector_get(wavelength,iWlen);
	if (lambda>=lambdamin && lambda<=lambdamax) {
	  cpl_vector_set(vecwaveRef,iWlenRef,lambda);
	  iWlenRef++;
	}
      }
    }
    
    waveUniform=cpl_vector_new(nbInterp);
    for(iWlen=0;iWlen<nbInterp;iWlen++) {
      cpl_vector_set(waveUniform,iWlen,cpl_vector_get(vecwaveRef,0)+
		     iWlen*(cpl_vector_get(vecwaveRef,nbRef-1)-cpl_vector_get(vecwaveRef,0))/nbInterp);
    }
    
    vecRealRef=cpl_vector_new(nbRef);
    vecImagRef=cpl_vector_new(nbRef);
    vecRealInterp=cpl_vector_new(nbInterp);
    vecImagInterp=cpl_vector_new(nbInterp);
    imgPadding=cpl_image_new(nbPadding,1,CPL_TYPE_DOUBLE_COMPLEX);
    fftPadding=cpl_image_new(nbPadding,1,CPL_TYPE_DOUBLE_COMPLEX);
    // Algorithm TF
    for(iFrame=0;iFrame<nbFrameTarget;iFrame++) {
      for(iBase=0;iBase<nbBase;iBase++) {
	if (detNum == 1) {
	  iWlenRef=nbRef-1;
	  for(iWlen=0;iWlen<nbWave;iWlen++) {
	    lambda=cpl_vector_get(wavelength,iWlen);
	    if (lambda>=lambdamin && lambda<=lambdamax) {
	      cpl_vector_set(vecRealRef,iWlenRef,cpl_image_get(imgcurrentReal[iFrame],iWlen+1,2+iBase,&pis));
	      cpl_vector_set(vecImagRef,iWlenRef,cpl_image_get(imgcurrentImag[iFrame],iWlen+1,2+iBase,&pis));
	      iWlenRef--;
	    }
	  }
	} else {
	  iWlenRef=0;
	  for(iWlen=0;iWlen<nbWave;iWlen++) {
	    lambda=cpl_vector_get(wavelength,iWlen);
	    if (lambda>=lambdamin && lambda<=lambdamax) {
	      cpl_vector_set(vecRealRef,iWlenRef,cpl_image_get(imgcurrentReal[iFrame],iWlen+1,2+iBase,&pis));
	      cpl_vector_set(vecImagRef,iWlenRef,cpl_image_get(imgcurrentImag[iFrame],iWlen+1,2+iBase,&pis));
	      iWlenRef++;
	    }
	  }
	}
	frefReal = cpl_bivector_wrap_vectors(vecwaveRef,vecRealRef);
        foutReal = cpl_bivector_wrap_vectors(waveUniform,vecRealInterp);
	cpl_bivector_interpolate_linear(foutReal,frefReal);
	frefImag = cpl_bivector_wrap_vectors(vecwaveRef,vecImagRef);
        foutImag = cpl_bivector_wrap_vectors(waveUniform,vecImagInterp);
	cpl_bivector_interpolate_linear(foutImag,frefImag);

	for(iWlen=0;iWlen<nbInterp;iWlen++) {
	  cpl_image_set_complex(imgPadding,1+iWlen+(nbPadding-nbInterp)/2,1,
				cpl_vector_get(vecRealInterp,iWlen)+I*cpl_vector_get(vecImagInterp,iWlen));
	}
	//FFT
	cpl_fft_image(fftPadding,imgPadding,CPL_FFT_FORWARD | CPL_FFT_FIND_MEASURE);
	double valmax=0.;
	int indexMax=0;
	for(iWlen=0;iWlen<nbPadding;iWlen++) {
	  val=pow(creal(cpl_image_get_complex(fftPadding,1+iWlen,1,&pis)),2.0)+
	    pow(cimag(cpl_image_get_complex(fftPadding,1+iWlen,1,&pis)),2.0);
	  if (val > valmax) {
	    valmax=val;
	    indexMax=iWlen;
	  }
	}
	if (indexMax > nbPadding/2) {
	  indexMax-=nbPadding;
	}
	cpl_vector_set(opd[iBase],iFrame,-1.*(pow(cpl_vector_get(waveUniform,nbInterp/2),2.0)/
					  (cpl_vector_get(waveUniform,nbInterp-1)-cpl_vector_get(waveUniform,0)))
		       *(indexMax*nbInterp)/nbPadding);
	cpl_vector_set(opdErr[iBase],iFrame,0.);
	cpl_vector_set(tempoff[iBase],iFrame,0.);
	cpl_vector_set(tempoffErr[iBase],iFrame,0.);
	cpl_vector_set(humoff[iBase],iFrame,0.);
	cpl_vector_set(humoffErr[iBase],iFrame,0.);
	  

	
	cpl_free(foutReal);
	cpl_free(frefReal);
	cpl_free(foutImag);
	cpl_free(frefImag);
	

	
      }
    }
    cpl_vector_delete(vecRealRef);
    cpl_vector_delete(vecImagRef);
    cpl_vector_delete(vecRealInterp);
    cpl_vector_delete(vecImagInterp);
    cpl_image_delete(imgPadding);
    cpl_image_delete(fftPadding);
    cpl_vector_delete(vecwaveRef);
    cpl_vector_delete(waveUniform);
	    
  } else if (algoCophasing == 1) {
    // Cophasing Algorithm (CRAL)
    cpl_vector *vecalpha[6];
    cpl_vector *vecbeta[6];
    cpl_vector *vecphiStatic[6];
    cpl_vector *vecalphaErr[6];
    for(iBase=0;iBase<6;iBase++) {
      vecalphaErr[iBase]=cpl_vector_new(nbFrameTarget);
    }
    mat_apply_cophasing_algo(imgcurrentReal,imgcurrentImag,
			     nbWave,nbFrameTarget,
			     vecalpha,vecbeta,vecphiStatic,
			     lambdamin,lambdamax);


    cpl_vector *vecSub=cpl_vector_new(10);
    for(iBase=0;iBase<6;iBase++) {
      for(iFrame=0;iFrame<nbFrameTarget;iFrame++) {
	for(int iSub=-5;iSub<5;iSub++) {
	  if (iFrame >= 5) {
	    cpl_vector_set(vecSub,iSub+5,cpl_vector_get(vecalpha[iBase],iFrame+iSub));
	  } else if (iFrame < 5) {
	    cpl_vector_set(vecSub,iSub+5,cpl_vector_get(vecalpha[iBase],iFrame+iSub+5));
	  } else if (iFrame >= nbFrameTarget-5) {
	    cpl_vector_set(vecSub,iSub+5,cpl_vector_get(vecalpha[iBase],iFrame+iSub-5));
	  }
	}
	cpl_vector_set(vecalphaErr[iBase],iFrame,cpl_vector_get_stdev(vecSub));

	
	cpl_vector_set(opd[iBase],iFrame,cpl_vector_get(vecalpha[iBase],iFrame)/2./CPL_MATH_PI);
	cpl_vector_set(opdErr[iBase],iFrame,cpl_vector_get(vecalphaErr[iBase],iFrame)/2./CPL_MATH_PI);
	cpl_vector_set(tempoff[iBase],iFrame,0.);
	cpl_vector_set(tempoffErr[iBase],iFrame,0.);
	cpl_vector_set(humoff[iBase],iFrame,0.);
	cpl_vector_set(humoff[iBase],iFrame,cpl_vector_get(vecbeta[iBase],iFrame));
	cpl_vector_set(humoffErr[iBase],iFrame,0.);


      }
    }
    cpl_vector_delete(vecSub);

    double phi=0.;
    double re[nbWave],im[nbWave];
    for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
      {
	if (iFrame==0)
	  {
	    for(int iWave=0;iWave<nbWave;iWave++)
	      {
		re[iWave]=0.;
		im[iWave]=0.;
	      }
	  }
	  
	for(int iWave=0;iWave<nbWave;iWave++)
	  {
	    phi=(2*CPL_MATH_PI*cpl_vector_get(opd[5],iFrame)/cpl_image_get(imgcurrentImag[iFrame],iWave+1,1,&pis))+
	      cpl_vector_get(vecbeta[5],iFrame)+
	      cpl_vector_get(vecphiStatic[5],iWave);
	    re[iWave]+=cpl_image_get(imgcurrentReal[iFrame],iWave+1,7,&pis)*cos(phi)+cpl_image_get(imgcurrentImag[iFrame],iWave+1,7,&pis)*sin(phi);
	    im[iWave]+=cpl_image_get(imgcurrentImag[iFrame],iWave+1,7,&pis)*cos(phi)-cpl_image_get(imgcurrentReal[iFrame],iWave+1,7,&pis)*sin(phi);
	  }
      }
    /* for(int iWave=0;iWave<nbWave;iWave++) */
    /*   { */
    /* 	printf("%f %f %f\n",cpl_image_get(imgcurrentImag[120],iWave+1,1,&pis),re[iWave]/corrFlux->nbframe,im[iWave]/corrFlux->nbframe); */
    /*   } */

    for(iBase=0;iBase<6;iBase++) {
      cpl_vector_delete(vecalpha[iBase]);
      cpl_vector_delete(vecbeta[iBase]);
      cpl_vector_delete(vecphiStatic[iBase]);
      cpl_vector_delete(vecalphaErr[iBase]);
    }
  } else if (algoCophasing == 3) {
    cpl_table *tbl_gra=NULL;
    cpl_propertylist *header_gra=NULL;
    cpl_matrix **fluxcorR=NULL;
    cpl_matrix **fluxcorI=NULL;
    cpl_vector *time_gra=NULL;
    cpl_vector *time_mt=NULL;
    cpl_matrix *gd=NULL;
    cpl_matrix *pa=NULL;
    int iCol_gra,nRow_gra;
    double mjdstart_gra=0.;
    int bcdmode=0;
    /* ------------------------------------- */
    /* Load GRAVITY FITS file in a cpl_table */
    /* ------------------------------------- */
    tbl_gra=cpl_table_load(fileGRAVITY,4,0);
    header_gra=cpl_propertylist_load(fileGRAVITY,0);
    nRow_gra=cpl_table_get_nrow(tbl_gra);
    mjdstart_gra=cpl_propertylist_get_double(header_gra,"MJD-OBS");
    /* -------------------------------------------------- */
    /* Allocate memory of fluxcorR, fluxcorI, and GDtotal */
    /* -------------------------------------------------- */
    fluxcorR=cpl_malloc(NB_BASE*sizeof(cpl_matrix*));
    fluxcorI=cpl_malloc(NB_BASE*sizeof(cpl_matrix*));
    for(iBase=0;iBase<NB_BASE;iBase++)
      {
    	fluxcorR[iBase]=cpl_matrix_new(nRow_gra,NB_WAVELENGTH);
    	fluxcorI[iBase]=cpl_matrix_new(nRow_gra,NB_WAVELENGTH);
      }

    /* --------------------------- */
    /* Read COHERENCE measurements */
    /* --------------------------- */
    for(int iRow=0;iRow<nRow_gra;iRow++)
      {
    	iCol_gra=0;
    	for(int iWave=0;iWave<NB_WAVELENGTH;iWave++)
    	  {
    	    iCol_gra+=4;
	  
    	    for(iBase=0;iBase<NB_BASE;iBase++)
    	      {
    		cpl_matrix_set(fluxcorR[iBase],iRow,iWave,
    			       cpl_array_get_float(cpl_table_get_array(tbl_gra,"COHERENCE",iRow),iCol_gra,NULL));
    		iCol_gra++;
    	      }
	  
    	    for(iBase=0;iBase<NB_BASE;iBase++)
    	      {
    		cpl_matrix_set(fluxcorI[iBase],iRow,iWave,
    			       cpl_array_get_float(cpl_table_get_array(tbl_gra,"COHERENCE",iRow),iCol_gra,NULL));
    		iCol_gra++;
    	      }
    	  }
      }
    /* ------------------------------------------------------------ */
    /* Get GRAVITY time vector (\micro s) and transform it into MJD */
    /* ------------------------------------------------------------ */
    time_gra=cpl_vector_new(nRow_gra);
    for(int iRow=0;iRow<nRow_gra;iRow++)
      {
    	cpl_vector_set(time_gra,iRow,cpl_table_get(tbl_gra,"TIME",iRow,NULL)*1E-6/86400.);
      }
    cpl_vector_add_scalar(time_gra,mjdstart_gra);

    /* ------------------------------------- */
    /* Calculate GD and PA for cophasing K4N */
    /* ------------------------------------- */
      if (strcmp(cpl_propertylist_get_string(corrFlux->keywords,"ESO CFG BCD MODE"),"OUT-OUT") == 0)
    	{
    	  bcdmode = 0;
    	}
      else if (strcmp(cpl_propertylist_get_string(corrFlux->keywords,"ESO CFG BCD MODE"),"IN-OUT") == 0)
    	{
    	  bcdmode = 1;
    	}
      else if(strcmp(cpl_propertylist_get_string(corrFlux->keywords,"ESO CFG BCD MODE"),"OUT-IN") == 0)
    	{
    	  bcdmode = 2;
    	}
      else if(strcmp(cpl_propertylist_get_string(corrFlux->keywords,"ESO CFG BCD MODE"),"IN-IN") == 0)
    	{
    	  bcdmode = 3;
    	}
    time_mt=cpl_vector_new(cpl_propertylist_get_int(corrFlux->keywords,"PRO FRAME TARGET"));
    for(int iRow=0;iRow<corrFlux->nbframe;iRow++)
      {
    	cpl_vector_set(time_mt,iRow,corrFlux->list_corr[iRow]->time);
      }
    gd=cpl_matrix_new(cpl_propertylist_get_int(corrFlux->keywords,"PRO FRAME TARGET"),NB_BASE);
    pa=cpl_matrix_new(cpl_propertylist_get_int(corrFlux->keywords,"PRO FRAME TARGET"),NB_BASE);
    mat_compute_GD_PA_K4N(time_gra,time_mt,fluxcorR,fluxcorI,bcdmode,gd,pa);


    

    /* ------------*/
    /* Free memory */
    /* -------------*/
    for(iBase=0;iBase<NB_BASE;iBase++)
      {
    	cpl_matrix_delete(fluxcorR[iBase]);
    	cpl_matrix_delete(fluxcorI[iBase]);
      }
    cpl_free(fluxcorR);
    cpl_free(fluxcorI);
    cpl_table_delete(tbl_gra);
    cpl_propertylist_delete(header_gra);
    cpl_vector_delete(time_gra);
    cpl_vector_delete(time_mt);


    for(iBase=0;iBase<NB_BASE;iBase++)
      {
	for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
	  {
	    cpl_vector_set(opd[iBase],iFrame,cpl_matrix_get(gd,iFrame,iBase));
	    cpl_vector_set(opdErr[iBase],iFrame,0.);
	    cpl_vector_set(tempoff[iBase],iFrame,0.);
	    cpl_vector_set(tempoffErr[iBase],iFrame,0.);
	    cpl_vector_set(humoff[iBase],iFrame,cpl_matrix_get(pa,iFrame,iBase));
	    cpl_vector_set(humoffErr[iBase],iFrame,0.);
	  }
      }
    cpl_matrix_delete(gd);
    cpl_matrix_delete(pa);

  } else {
    for(iFrame=0;iFrame<nbFrameTarget;iFrame++) {
      // Refrence Channel Determination
      waveRef=(lambdamax+lambdamin)/2.;
      valmin=1.E20;
      for(iWlen=0;iWlen<nbWave;iWlen++) {
	val=fabs(waveRef-cpl_vector_get(wavelength,iWlen));
	if (val < valmin) {
	  valmin=val;
	  iWlenRef=iWlen;
	}
      }
      for(iBase=0;iBase<6;iBase++) {
	modulus=pow(cpl_image_get(imgcurrentReal[iFrame],iWlenRef+1,iBase+2,&pis),2.0)+
	  pow(cpl_image_get(imgcurrentImag[iFrame],iWlenRef+1,iBase+2,&pis),2.0);								       
	cpl_vector_set(phasorRealRef,iBase,cpl_image_get(imgcurrentReal[iFrame],iWlenRef+1,iBase+2,&pis)/modulus);
	cpl_vector_set(phasorImagRef,iBase,cpl_image_get(imgcurrentImag[iFrame],iWlenRef+1,iBase+2,&pis)/modulus);
      }
      waveRef=cpl_vector_get(wavelength,iWlenRef);
      if (iFrame == 0) {
	cpl_msg_info(cpl_func,"Reference Wavelength %f",waveRef);
      }

      // 1st Step only OPD, the values of pressure, temperature and humidity
      // are kept to the values given in the keywords
      // However for pratical reasons, the fit is done with 3 parameters 
      // OPD + temp and hum offsets but the range for the 
      // temp and hum offsets are extremely small (+/-1E-6)
      xminFirst  = (double *)malloc(nx*nbBase*sizeof(double));
      xminErr    = (double *)malloc(nx*nbBase*sizeof(double));
      objFuncMin = (double *)malloc(nbBase*sizeof(double));
      lo         = (double *)malloc(nx*sizeof(double));
      hi         = (double *)malloc(nx*sizeof(double));
      // Estimation of the OPD
      for(iBase=0;iBase<nbBase;iBase++)
	{
	  phiRealRef = cpl_vector_get(phasorRealRef,iBase);
	  phiImagRef = cpl_vector_get(phasorImagRef,iBase);
	  if (detNum == 1 && resolution ==3) { 
	    ncpt0=3000;
	  } else {
	    ncpt0=1000;
	  }
	    
	  x=(double *)malloc(nx*sizeof(double));
	  if (detNum == 1 && resolution ==3) { 
	    lo[0]=-300.;
	    hi[0]=300.;
	  } else {
	    lo[0]=-100.;
	    hi[0]=100.;
	  }	    
	  objFuncMin[iBase]=1.E20;
	  for(icpt0=0;icpt0<ncpt0;icpt0++)
	    { 
	      x[0]=lo[0]+icpt0*(hi[0]-lo[0])/ncpt0;
	      factor=2.*CPL_MATH_PI*x[0];
	      if (waveRef == 0.) {
		org=0.;
	      } else {
		org=-1.*factor/waveRef;
	      }
	      objFuncCurrent=0.;
	      for(iWlen=0;iWlen<nbWave;iWlen++)
		{
		  lambda=cpl_vector_get(wavelength,iWlen);
		  if (lambda>=lambdamin && lambda<=lambdamax)
		    {
		      alpha=(factor/lambda)+org;
		      zr=(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis)*phiRealRef+
			  cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis)*phiImagRef);///modulus;
		      zi=(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis)*phiRealRef-
			  cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis)*phiImagRef);///modulus;		      
		      objFuncCurrent+=pow(zi*cos(alpha)+zr*sin(alpha),2.0)/
			(pow(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,8,&pis),2.0)+
			 pow(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,8,&pis),2.0));
		    }
		}
	      if (objFuncCurrent<objFuncMin[iBase])
		{
		  xminFirst[nbBase*0+iBase]=x[0];
		  objFuncMin[iBase]=objFuncCurrent;
		}
	    }
	  ncpt0=40;
	  lo[0]=xminFirst[nbBase*0+iBase]-4;
	  hi[0]=xminFirst[nbBase*0+iBase]+4;
	  objFuncMin[iBase]=1.E20;
	  for(icpt0=0;icpt0<ncpt0;icpt0++)
	    { 
	      x[0]=lo[0]+icpt0*(hi[0]-lo[0])/ncpt0;	  
	      factor=2.*CPL_MATH_PI*x[0];
	      if (waveRef == 0.) {
		org=0.;
	      } else {
		org=-1.*factor/waveRef;
	      }
	      objFuncCurrent=0.;
	      for(iWlen=0;iWlen<nbWave;iWlen++)
		{
		  lambda=cpl_vector_get(wavelength,iWlen);
		  if (lambda>=lambdamin && lambda<=lambdamax)
		    {
		      /* modulus=pow(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis),2.0)+ */
		      /*   pow(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis),2.0);					 */			       
		      alpha=(factor/lambda)+org;
		      zr=(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis)*phiRealRef+
			  cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis)*phiImagRef);
		      zi=(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis)*phiRealRef-
			  cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis)*phiImagRef);
		      objFuncCurrent+=pow(zi*cos(alpha)+zr*sin(alpha),2.0)/
			(pow(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,8,&pis),2.0)+
			 pow(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,8,&pis),2.0));
		    }
		}
	      if (objFuncCurrent<objFuncMin[iBase])
		{
		  xminFirst[nbBase*0+iBase]=x[0];
		  objFuncMin[iBase]=objFuncCurrent;
		}
	    }

	  ncpt0=100;
	  chi2=cpl_vector_new(ncpt0);
	  lo[0]=xminFirst[nbBase*0+iBase]-0.4;
	  hi[0]=xminFirst[nbBase*0+iBase]+0.4;
	  objFuncMin[iBase]=1.E20;
	  for(icpt0=0;icpt0<ncpt0;icpt0++)
	    { 
	      x[0]=lo[0]+icpt0*(hi[0]-lo[0])/ncpt0;	  
	      factor=2.*CPL_MATH_PI*x[0];
	      if (waveRef == 0.) {
		org=0.;
	      } else {
		org=-1.*factor/waveRef;
	      }
	      objFuncCurrent=0.;
	      for(iWlen=0;iWlen<nbWave;iWlen++)
		{
		  lambda=cpl_vector_get(wavelength,iWlen);
		  if (lambda>=lambdamin && lambda<=lambdamax)
		    {
		      /* modulus=pow(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis),2.0)+ */
		      /*   pow(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis),2.0);*/			       
		      alpha=(factor/lambda)+org;
		      zr=(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis)*phiRealRef+
			  cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis)*phiImagRef);
		      zi=(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis)*phiRealRef-
			  cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis)*phiImagRef);
		      objFuncCurrent+=pow(zi*cos(alpha)+zr*sin(alpha),2.0)/
			(pow(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,8,&pis),2.0)+
			 pow(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,8,&pis),2.0));
		    }
		}
	      if (objFuncCurrent<objFuncMin[iBase])
		{
		  xminFirst[nbBase*0+iBase]=x[0];
		  objFuncMin[iBase]=objFuncCurrent;
		}
	      cpl_vector_set(chi2,icpt0,objFuncCurrent);
	    }
	  /* if (iFrame == 1 && iBase==0) { */
	  /*   printf("OPD %f\n",xminFirst[nbBase*0+iBase]); */
	  /*   factor=2.*CPL_MATH_PI*xminFirst[nbBase*0+iBase]; */
	  /*   org=-1.*factor/waveRef; */
	  /*   for(iWlen=0;iWlen<nbWave;iWlen++) { */
	  /*     lambda=cpl_vector_get(wavelength,iWlen); */
	  /*     alpha=(factor/lambda)+org; */
	  /*     zr=(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis)*phiRealRef+ */
	  /* 	  cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis)*phiImagRef); */
	  /*     zi=(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis)*phiRealRef- */
	  /* 	  cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis)*phiImagRef); */
	  /*     printf("%f %f %f %f %f %f\n",lambda,zr,zi,cos(alpha),sin(alpha),zi*cos(alpha)+zr*sin(alpha)); */
	  /*   } */
	  /*   exit(0); */
	  /* } */
	
	  for(icpt0=0;icpt0<ncpt0;icpt0++)
	    {
	      if (  cpl_vector_get(chi2,icpt0)-objFuncMin[iBase]<0.2*nbWave )
		{
		  icptOne1=icpt0;
		  break;
		}
	    }
	  for(icpt0=icptOne1;icpt0<ncpt0;icpt0++)
	    {
	      if (  cpl_vector_get(chi2,icpt0)-objFuncMin[iBase]>0.2*nbWave )
		{
		  icptOne2=icpt0;
		  break;
		}
	    }
	  icptOne1--;
	  xminErr[iBase]=(icptOne2-icptOne1)*(hi[0]-lo[0])/ncpt0;
	  cpl_vector_delete(chi2);    
	
	  free(x);
	  cpl_vector_set(opd[iBase],iFrame,-1.*xminFirst[iBase]);
	  cpl_vector_set(opdErr[iBase],iFrame,xminErr[iBase]);
	  cpl_vector_set(tempoff[iBase],iFrame,0.);
	  cpl_vector_set(tempoffErr[iBase],iFrame,0.);
	  cpl_vector_set(humoff[iBase],iFrame,0.);
	  cpl_vector_set(humoffErr[iBase],iFrame,0.);
	}

      free(lo);
      free(hi);
      free(xminFirst);
      free(xminErr);
      free(objFuncMin);
    }
  }




  for(iFrame=0;iFrame<=nbFrameTarget;iFrame++) {
    cpl_imagelist_set(imglistReal,imgcurrentReal[iFrame],iFrame);
    cpl_imagelist_set(imglistImag,imgcurrentImag[iFrame],iFrame);
  }

  output=cpl_sprintf("nrjReal_%d.fits",expno);
  cpl_imagelist_save(imglistReal,output,CPL_TYPE_DOUBLE,NULL,CPL_IO_CREATE);
  cpl_free(output);
  output=cpl_sprintf("nrjImag_%d.fits",expno);
  cpl_imagelist_save(imglistImag,output,CPL_TYPE_DOUBLE,NULL,CPL_IO_CREATE);
  cpl_free(output);

  
  /* if (cpl_parameter_get_bool(chromaticOpdFit)) { */
  /*   double objFuncTotal=0.; */
    
  /*   for(iFrame=0;iFrame<nbFrameTarget;iFrame++) { */
  /*     printf("iFrame: %d\n",iFrame); */

  /*     ncpt0=100; */
  /*     ncpt1=20; */
  /*     ncpt2=100; */
  /*     objFuncMin=(double *)malloc(nbBase*sizeof(double)); */
  /*     lo=(double *)malloc(nx*sizeof(double)); */
  /*     hi=(double *)malloc(nx*sizeof(double)); */
      
  /*     // Limits for temperature Offset */
  /*     lo[0]=-10.; */
  /*     hi[0]=10.; */
  /*     // Limits for humidity Offset */
  /*     lo[1]=-20.; */
  /*     hi[1]=20.; */
  /*     xmin=(double *)malloc((2+nbBase)*sizeof(double)); */
  /*     xminFirst=(double *)malloc(nbBase*sizeof(double)); */
  /*     x=(double *)malloc(nx*sizeof(double)); */

  /*     objFuncTotal=1.E20; */
  /*     for(icpt0=0;icpt0<ncpt0;icpt0++) { */
  /* 	x[0]=lo[0]+icpt0*(hi[0]-lo[0])/ncpt0; */
  /* 	for(icpt1=0;icpt1<ncpt1;icpt1++) { */
  /* 	  x[1]=lo[1]+icpt1*(hi[1]-lo[1])/ncpt1; */
	  
  /* 	  cptWaveList=0; */
  /* 	  for(iReg=0;iReg<corrFlux->imgdet->nbregion;iReg++) { */
  /* 	    corner=corrFlux->imgdet->list_region[iReg]->corner[1]; */
  /* 	    dimSpectral=corrFlux->imgdet->list_region[iReg]->naxis[1]; */
  /* 	    ftReal=corrFlux->list_corr[iReg*nbFrameOverPhot+iFrame]->imgreg[0]; */
  /* 	    ftImag=corrFlux->list_corr[iReg*nbFrameOverPhot+iFrame]->imgreg[1]; */
  /* 	    // Compute the energy of each fringe peak */
  /* 	    for(iWlen=0;iWlen<dimSpectral;iWlen++) { */
  /* 	      lambda=cpl_vector_get(wavelength,cptWaveList); */
  /* 	      for(iBase=0;iBase<nbBase;iBase++) { */
  /* 		posFringePeak=mat_get_pos_fringe_peak(lambda,detNum,resolution,iBase); */
  /* 		uPixelMin=(int)posFringePeak; */
  /* 		uPixelMax=(int)posFringePeak+1; */
		
  /* 		nrjReal=cpl_image_get(ftReal,uPixelMin+1,iWlen+1,&pis)*(uPixelMax-posFringePeak)+ */
  /* 		  cpl_image_get(ftReal,uPixelMax+1,iWlen+1,&pis)*(posFringePeak-uPixelMin); */
  /* 		nrjImag=cpl_image_get(ftImag,uPixelMin+1,iWlen+1,&pis)*(uPixelMax-posFringePeak)+ */
  /* 		  cpl_image_get(ftImag,uPixelMax+1,iWlen+1,&pis)*(posFringePeak-uPixelMin); */
		
		
  /* 		// Correction of the chromatic OPD (following Mathar) */
  /* 		//////////////////////////////////////////////////////////////////////////// */
  /* 		// Compute the chromatic OPD */
  /* 		// Apply the correction to (phasorReal,phasorImag) */
  /* 		// PhasorRealCorr+j*PhasorImagCorr=(PhasorReal+j*PhasorImag)*exp(j*2*pi*OPDchrom/lambda) */
  /* 		refractiveIndex=mat_oiopd_mathar(lambda,pressure*100.,temperature+273.15+x[0],humidity+x[1]); */
  /* 		argOpdChrom=0*2*CPL_MATH_PI*refractiveIndex*cpl_vector_get(z,iBase)*1.E6/lambda; */
  /* 		cpl_image_set(imgcurrentReal[iFrame],cptWaveList+1,iBase+2, */
  /* 			      nrjReal*cos(argOpdChrom)-nrjImag*sin(argOpdChrom)); */
  /* 		cpl_image_set(imgcurrentImag[iFrame],cptWaveList+1,iBase+2, */
  /* 			      nrjImag*cos(argOpdChrom)+nrjReal*sin(argOpdChrom)); */
		
  /* 	      } */

  /* 	      cptWaveList++; */
  /* 	    } */
  /* 	  } */

  /* 	  for(iBase=0;iBase<nbBase;iBase++) { */
  /* 	    phiRealRef=cpl_vector_get(phasorRealRef,iBase); */
  /* 	    phiImagRef=cpl_vector_get(phasorImagRef,iBase); */
  /* 	    lo[2]=-1*cpl_vector_get(opd[iBase],iFrame)-2.; */
  /* 	    hi[2]=-1*cpl_vector_get(opd[iBase],iFrame)+2.; */
  /* 	    objFuncMin[iBase]=1.E20; */
  /* 	    for(icpt2=0;icpt2<ncpt2;icpt2++) {  */
  /* 	      x[2]=lo[2]+icpt2*(hi[2]-lo[2])/ncpt2; */
  /* 	      factor=2.*CPL_MATH_PI*x[2]; */
  /* 	      if (waveRef == 0.) { */
  /* 		org=0.; */
  /* 	      } else { */
  /* 		org=-1.*factor/waveRef; */
  /* 	      } */
  /* 	      objFuncCurrent=0.; */
  /* 	      for(iWlen=0;iWlen<nbWave;iWlen++) { */
  /* 		lambda=cpl_vector_get(wavelength,iWlen); */
  /* 		if (lambda>=lambdamin && lambda<=lambdamax) { */
  /* 		  alpha=(factor/lambda)+org; */
  /* 		  zr=(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis)*phiRealRef+ */
  /* 		      cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis)*phiImagRef);///modulus; */
  /* 		  zi=(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis)*phiRealRef- */
  /* 		      cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis)*phiImagRef);///modulus; */
  /* 		  objFuncCurrent+=pow(zi*cos(alpha)+zr*sin(alpha),2.0)/ */
  /* 		    (pow(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,8,&pis),2.0)+ */
  /* 		     pow(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,8,&pis),2.0)); */
  /* 		} */
  /* 	      } */
	      
  /* 	      if (objFuncCurrent<objFuncMin[iBase]) { */
  /* 		xminFirst[iBase]=x[2]; */
  /* 		objFuncMin[iBase]=objFuncCurrent; */
  /* 	      } */
  /* 	    } */
  /* 	  } */
  /* 	  val=0.; */
  /* 	  for(iBase=0;iBase<nbBase;iBase++) { */
  /* 	    val+=objFuncMin[iBase]; */
  /* 	    /\* printf("%f(%f) %f ",xminFirst[iBase],cpl_vector_get(opd[iBase],iFrame),objFuncMin[iBase]); *\/ */
  /* 	  } */
  /* 	  /\* printf("\n"); *\/ */
	  


	      
  /* 	  /\* printf("%d %d %f %f %e\n",icpt0,icpt1,x[0],x[1],val); *\/ */

  /* 	  if (val<objFuncTotal) { */
  /* 	    xmin[0]=x[0]; */
  /* 	    xmin[1]=x[1]; */
  /* 	    for(iBase=0;iBase<nbBase;iBase++) { */
  /* 	      xmin[2+iBase]=xminFirst[iBase]; */
  /* 	    } */
  /* 	    objFuncTotal=val; */
  /* 	  } */
	  

  /* 	} */
  /*     } */
  /*     printf("%f %f ",xmin[0],xmin[1]); */
  /*     for(iBase=0;iBase<nbBase;iBase++) { */
  /* 	printf("%f(%f) ",xmin[2+iBase],cpl_vector_get(opd[iBase],iFrame)); */
  /*     } */
  /*     printf("\n"); */

  /*     /\* exit(1); *\/ */
















      
  /*     /\* // 2nd Step OPD + Temperature and humidity offsets *\/ */
  /*     /\* // The initial value of OPD comes from the 1st Step *\/ */
  /*     /\* // with a small range.  *\/ */
  /*     /\* ncpt0=1;//5; *\/ */
  /*     /\* ncpt1=1;//10; *\/ */
  /*     /\* ncpt2=1;//10; *\/ */
  /*     /\* objFuncMin=(double *)malloc(nbBase*sizeof(double)); *\/ */
  /*     /\* vecDiagCov=(double *)malloc(nx*sizeof(double)); *\/ */
  /*     /\* lo=(double *)malloc(nx*sizeof(double)); *\/ */
  /*     /\* hi=(double *)malloc(nx*sizeof(double)); *\/ */
      
  /*     /\* // Limits for temperature Offset *\/ */
  /*     /\* lo[1]=-0.5; *\/ */
  /*     /\* hi[1]=0.5; *\/ */
  /*     /\* // Limits for humidity Offset *\/ */
  /*     /\* lo[2]=-5.; *\/ */
  /*     /\* hi[2]=5.; *\/ */
  /*     /\* xmin=(double *)malloc(nx*nbBase*sizeof(double)); *\/ */
  /*     /\* xminErr=(double *)malloc(nx*nbBase*sizeof(double)); *\/ */
  /*     /\* // Estimation of the OPD *\/ */
  /*     /\* for(iBase=0;iBase<nbBase;iBase++){ *\/ */
  /*     /\* 	// Limits for OPD based on 1st Step *\/ */
  /*     /\* 	lo[0]=xminFirst[iBase]-2.; *\/ */
  /*     /\* 	hi[0]=xminFirst[iBase]+2.; *\/ */
  /*     /\* 	// ASA_CG *\/ */
  /*     /\* 	x=(double *)malloc(nx*sizeof(double)); *\/ */
  /*     /\* 	asa_cg_default(&cgParm); *\/ */
  /*     /\* 	asa_default(&asaParm); *\/ */
	
  /*     /\* 	cgParm.PrintParms=FALSE; *\/ */
  /*     /\* 	cgParm.PrintLevel=0; *\/ */
  /*     /\* 	cgParm.maxit_fac=100.; *\/ */
  /*     /\* 	asaParm.PrintParms=FALSE; *\/ */
  /*     /\* 	asaParm.PrintLevel=0; *\/ */
  /*     /\* 	asaParm.PrintFinal=0; *\/ */
  /*     /\* 	asaParm.max_backsteps=100; *\/ */
  /*     /\* 	asaParm.maxit_fac=100.; *\/ */
  /*     /\* 	asaParm.AArmijoFac=0.; *\/ */
	
  /*     /\* 	asaUser=cpl_vector_new(5*nbWave+3+1); *\/ */
  /*     /\* 	for(iWlen=0;iWlen<nbWave;iWlen++) { *\/ */
  /*     /\* 	  modulus=pow(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis),2.0)+ *\/ */
  /*     /\* 	    pow(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis),2.0);								        *\/ */
  /*     /\* 	  cpl_vector_set(asaUser,iWlen,cpl_vector_get(wavelength,iWlen)); *\/ */
  /*     /\* 	  cpl_vector_set(asaUser,nbWave+iWlen,cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis)/modulus); *\/ */
  /*     /\* 	  cpl_vector_set(asaUser,2*nbWave+iWlen,cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis)/modulus); *\/ */
  /*     /\* 	  cpl_vector_set(asaUser,3*nbWave+iWlen,pow(cpl_image_get(imgcurrentReal[iFrame],iWlen+1,8,&pis),2.0)/pow(modulus,2.)); *\/ */
  /*     /\* 	  cpl_vector_set(asaUser,4*nbWave+iWlen,pow(cpl_image_get(imgcurrentImag[iFrame],iWlen+1,8,&pis),2.0)/pow(modulus,2.)); *\/ */
  /*     /\* 	} *\/ */
  /*     /\* 	cpl_vector_set(asaUser,5*nbWave,pressure); *\/ */
  /*     /\* 	cpl_vector_set(asaUser,5*nbWave+1,temperature); *\/ */
  /*     /\* 	cpl_vector_set(asaUser,5*nbWave+2,humidity); *\/ */
  /*     /\* 	cpl_vector_set(asaUser,5*nbWave+3,cpl_vector_get(z,iBase)); *\/ */
	
  /*     /\* 	objFuncMin[iBase]=1.E20; *\/ */
  /*     /\* 	printf("Iframe,Ibase %d %d \n",iFrame,iBase); *\/ */
  /*     /\* 	for(icpt0=0;icpt0<ncpt0;icpt0++) { *\/ */
  /*     /\* 	  for(icpt1=0;icpt1<ncpt1;icpt1++) { *\/ */
  /*     /\* 	    for(icpt2=0;icpt2<ncpt2;icpt2++) { *\/ */
	      
  /*     /\* 	      x[0]=lo[0]+icpt0*(hi[0]-lo[0])/ncpt0; *\/ */
  /*     /\* 	      x[1]=lo[1]+icpt1*(hi[1]-lo[1])/ncpt1; *\/ */
  /*     /\* 	      x[2]=lo[2]+icpt2*(hi[2]-lo[2])/ncpt2; *\/ */
	      
  /*     /\* 	      asa_cg(x,lo,hi,nx,NULL,&cgParm,&asaParm,1.E-4,mat_opd_asavalue, *\/ */
  /*     /\* 		     mat_opd_asagrad,mat_opd_asavalgrad,NULL,asaUser); *\/ */
  /*     /\* 	      objFuncCurrent=mat_opd_value(x,asaUser); *\/ */
  /*     /\* 	      if (objFuncCurrent < objFuncMin[iBase]) { *\/ */
  /*     /\* 		objFuncMin[iBase]=objFuncCurrent; *\/ */
  /*     /\* 		mat_opd_cov(x,nx,asaUser,vecDiagCov); *\/ */
  /*     /\* 		for(n=0;n<nx;n++) { *\/ */
  /*     /\* 		  xmin[nbBase*n+iBase]=x[n]; *\/ */
  /*     /\* 		  xminErr[nbBase*n+iBase]=sqrt(objFuncMin[iBase])*sqrt(vecDiagCov[n]); *\/ */
  /*     /\* 		} *\/ */
  /*     /\* 	      } *\/ */
  /*     /\* 	    } *\/ */
  /*     /\* 	  } *\/ */
  /*     /\* 	} *\/ */
  /*     /\* 	cpl_vector_delete(asaUser); *\/ */
	
  /*     for(iBase=0;iBase<nbBase;iBase++) {	 */
  /* 	cpl_vector_set(opd[iBase],iFrame,-1.*xmin[2+iBase]); */
  /* 	cpl_vector_set(opdErr[iBase],iFrame,0.);//xminErr[iBase]); */
  /* 	cpl_vector_set(tempoff[iBase],iFrame,xmin[0]); */
  /* 	cpl_vector_set(tempoffErr[iBase],iFrame,0.);//xminErr[nbBase+iBase]); */
  /* 	cpl_vector_set(humoff[iBase],iFrame,xmin[1]); */
  /* 	cpl_vector_set(humoffErr[iBase],iFrame,0.);//xminErr[2*nbBase+iBase]); */
  /*     } */
  /*     /\* for(iBase=0;iBase<nbBase;iBase++) { *\/ */
  /*     /\* 	for(n=0;n<nx;n++) { *\/ */
  /*     /\* 	  printf("BASE= %d, X%d = %e +/- %e, ObjFunc = %f \n",iBase+1,n+1,xmin[nbBase*n+iBase], *\/ */
  /*     /\* 		 xminErr[nbBase*n+iBase],objFuncMin[iBase]); *\/ */
  /*     /\* 	} *\/ */
  /*     /\* } *\/ */
  /*     free(x); */
  /*     free(lo); */
  /*     free(hi); */
  /*     free(xmin); */
  /*     free(xminFirst); */
  /*     /\* free(xminErr); *\/ */
  /*     free(objFuncMin); */
  /*   } */
  /* }  */

  /* double opdVal,phi,phasorRe,phasorIm,cfluxRe,cfluxIm; */
  /* double modul; */

  /* for(iFrame=0;iFrame<corrFlux->nbframe;iFrame++) { */
  /*   for(iBase=0;iBase<nbBase;iBase++){ */
  /*     opdVal=cpl_vector_get(opd[iBase],iFrame); */
  /*     for(iWlen=0;iWlen<nbWave;iWlen++) { */
  /*       lambda=cpl_vector_get(wavelength,iWlen); */
  /*       phi = 2 * CPL_MATH_PI * opdVal / lambda; */
  /* 	phasorRe = cos(phi); */
  /* 	phasorIm = sin(phi); */
  /* 	cfluxRe=cpl_image_get(imgcurrentReal[iFrame],iWlen+1,iBase+2,&pis); */
  /* 	cfluxIm=cpl_image_get(imgcurrentImag[iFrame],iWlen+1,iBase+2,&pis); */
  /* 	modul=sqrt(cfluxRe*cfluxRe+cfluxIm*cfluxIm); */
	
  /* 	cpl_image_set(imgcurrentReal[iFrame],iWlen+1,iBase+1,cfluxRe*phasorRe+cfluxIm*phasorIm); */
  /* 	cpl_image_set(imgcurrentImag[iFrame],iWlen+1,iBase+1,cfluxIm*phasorRe-cfluxRe*phasorIm); */
  /* 	if (iBase==1 ) printf("%f %f %f %f\n",cfluxRe,cfluxIm,phasorRe*modul,phasorIm*modul); */
  /*     } */
  /*   } */
  /*   exit(0); */
  /* } */
  /*   exit(0); */
  

  cpl_vector_delete(phasorRealRef);
  cpl_vector_delete(phasorImagRef);
  cpl_imagelist_delete(imglistReal);
  cpl_imagelist_delete(imglistImag);
  cpl_free(imgcurrentReal);
  cpl_free(imgcurrentImag);

  // Storing results into the structure oiopdwvpo
  cpt=0;
  for(i=0;i<oiopdwvpo->oiopd->nbopd/nbBase;i++) {
    for(j=0;j<nbBase;j++) {
      oiopdwvpo->oiopd->list_opd[cpt]->opd=cpl_vector_get(opd[j],i);//
      oiopdwvpo->oiopd->list_opd[cpt]->opderr=cpl_vector_get(opdErr[j],i);
      oiopdwvpo->oiopd->list_opd[cpt]->tempoff=cpl_vector_get(tempoff[j],i);
      oiopdwvpo->oiopd->list_opd[cpt]->tempofferr=cpl_vector_get(tempoffErr[j],i);
      oiopdwvpo->oiopd->list_opd[cpt]->humoff=cpl_vector_get(humoff[j],i);
      oiopdwvpo->oiopd->list_opd[cpt]->humofferr=cpl_vector_get(humoffErr[j],i);
      cpt++;
    }
  }
  cpl_msg_info(cpl_func,"OPD estimation done");
  cpl_vector_delete(z);
  cpl_vector_delete(wavelength);
  for(iBase=0;iBase<nbBase;iBase++) {
    cpl_vector_delete(opd[iBase]);
    cpl_vector_delete(opdErr[iBase]);
    cpl_vector_delete(tempoff[iBase]);
    cpl_vector_delete(tempoffErr[iBase]);
    cpl_vector_delete(humoff[iBase]);
    cpl_vector_delete(humoffErr[iBase]);
  }
  cpl_free(opd);
  cpl_free(opdErr);
  cpl_free(tempoff);
  cpl_free(tempoffErr);
  cpl_free(humoff);
  cpl_free(humoffErr);

  cpl_vector_delete(dispCoef);
  cpl_vector_delete(foo);
  cpl_vector_delete(DLstatic);
  cpl_vector_delete(DLopl);

  return(0);
}

inline double my_f(const gsl_vector *v, void *params) {
  double *p = (double *)params;
  int nWave=(int)p[0];
  double lambdamin=p[1];
  double lambdamax=p[2];
  double x=0.;
  double y=0.;
  double eta,phi,alpha;
  alpha=gsl_vector_get(v, 0);
  for(int i=0;i<nWave;i++){
    if (fabs(p[3+3*nWave+i]-(lambdamax+lambdamin)/2) < (lambdamax-lambdamin)/2)
      {
	eta=p[3+i]*p[3+i]+p[3+nWave+i]*p[3+nWave+i];
	phi=atan2(p[3+nWave+i],p[3+i]);
	x+=eta*cos(phi-p[3+2*nWave+i]-(alpha/p[3+3*nWave+i]));
	y+=eta*sin(phi-p[3+2*nWave+i]-(alpha/p[3+3*nWave+i]));
      }
  }
  return -1.*sqrt(x*x+y*y);
}

int mat_apply_cophasing_algo(cpl_image **real,cpl_image **imag,
			     int nWave, int nFrame,
			     cpl_vector *alpha[], cpl_vector *beta[],
			     cpl_vector *phiStatic[],
			     float lambdamin, float lambdamax) {

  struct timeval tv1, tv2; // To measure timing of parallel section
  cpl_vector *phiprev=NULL;
  double eps=0;
  
  for(int iBase=0;iBase<6;iBase++) {
    alpha[iBase]=cpl_vector_new(nFrame);
    beta[iBase]=cpl_vector_new(nFrame);
    phiStatic[iBase]=cpl_vector_new(nWave);
    for(int iWave=0;iWave<nWave;iWave++) {
      cpl_vector_set(phiStatic[iBase],iWave,0.);
    }
    cpl_msg_info(cpl_func, "Starting parallel loops");
    gettimeofday(&tv1, NULL);

    // HDRL_OMP(omp parallel for) -> results change
    for(int iConv=0;iConv<NB_ITERATION_MAX;iConv++) {
      HDRL_OMP(omp parallel for)
      for(int iFrame=0;iFrame<nFrame;iFrame++) {
	  int pis;
	  double par[3+4*nWave];
	  par[0]=(double)nWave;
	  par[1]=(double)lambdamin;
	  par[2]=(double)lambdamax;
    
	for(int iWave=0;iWave<nWave;iWave++) {
	  par[3+iWave]=cpl_image_get(real[iFrame],iWave+1,iBase+2,&pis);
	  par[3+nWave+iWave]=cpl_image_get(imag[iFrame],iWave+1,iBase+2,&pis);
	  par[3+2*nWave+iWave]=cpl_vector_get(phiStatic[iBase],iWave);
	  par[3+3*nWave+iWave]=cpl_image_get(real[iFrame],iWave+1,1,&pis);
	}


	const gsl_multimin_fminimizer_type *T = gsl_multimin_fminimizer_nmsimplex2;
	gsl_multimin_fminimizer *s = NULL;
	gsl_vector *ss, *x;
	gsl_multimin_function minex_func;
      
	int iter = 0;
	int status;
	double size;

	/* Starting point */
	x = gsl_vector_alloc (1);
	gsl_vector_set (x, 0, 0.0);	

	/* if (iBase == 3 && iFrame == 20 && iConv== 19) */
	/*   { */
	/*     for(int iii=-1000;iii<=1000;iii++) */
	/*       { */
	/* 	gsl_vector_set (x, 0, iii*0.1); */
	/* 	printf("%d %f\n",iii,my_f(x,par)); */
	/*       } */
	/*     exit(1); */
	/*   }; */
	/* Set initial step sizes to 1 */
	ss = gsl_vector_alloc (1);
	gsl_vector_set_all (ss, 1.0);
      
	/* Initialize method and iterate */
	minex_func.n = 1;
	minex_func.f = my_f;
	minex_func.params = par;
      
	s = gsl_multimin_fminimizer_alloc (T, 1);
	gsl_multimin_fminimizer_set (s, &minex_func, x, ss);
      
	do {
	  iter++;
	  status = gsl_multimin_fminimizer_iterate(s);
	
	  if (status) 
	    break;
	
	  size = gsl_multimin_fminimizer_size (s);
	  status = gsl_multimin_test_size (size, 1e-4);
	  /* printf ("%5d %10.3e f() = %7.3f size = %.3f\n", */
	  /* 	iter, */
	  /* 	gsl_vector_get (s->x, 0), */
	  /* 	s->fval, size); */
	}
	while (status == GSL_CONTINUE && iter < 500);
	if (status != GSL_SUCCESS) {
	  printf ("Frame:%d Base:%d No convergence\n", iFrame, iBase);
	}	
	/* exit(1); */
	double alphaEstim=gsl_vector_get (s->x, 0);

	cpl_vector_set(alpha[iBase],iFrame,alphaEstim);
	double xval=0.;
	double yval=0.;
	for(int iWave=0;iWave<nWave;iWave++){
	  if (fabs(par[3+3*nWave+iWave]-(lambdamax+lambdamin)/2) < (lambdamax-lambdamin)/2)
	    {
	      double eta=sqrt(par[3+iWave]*par[3+iWave]+par[3+nWave+iWave]*par[3+nWave+iWave]);
	      double phi=atan2(par[3+nWave+iWave],par[3+iWave]);
	      xval+=eta*cos(phi-par[3+2*nWave+iWave]-(alphaEstim/par[3+3*nWave+iWave]));
	      yval+=eta*sin(phi-par[3+2*nWave+iWave]-(alphaEstim/par[3+3*nWave+iWave]));
	    }
	}
	double betaEstim=atan2(yval,xval);
	cpl_vector_set(beta[iBase],iFrame,betaEstim);
	
	/* if (iBase==0 && iConv==1) */
	/*   printf("%f\n",gsl_vector_get (s->x, 0)); */

	gsl_vector_free(x);
	gsl_vector_free(ss);
	gsl_multimin_fminimizer_free (s);
      }
      /* cpl_vector *out=NULL; */
      /* out=cpl_vector_duplicate(alpha[iBase]); */
      /* for(int iFrame=2;iFrame<nFrame-2;iFrame++) { */
      /* 	cpl_vector_set(out,iFrame,(cpl_vector_get(alpha[iBase],iFrame-2)+ */
      /* 				   cpl_vector_get(alpha[iBase],iFrame-1)+ */
      /* 				   cpl_vector_get(alpha[iBase],iFrame)+ */
      /* 				   cpl_vector_get(alpha[iBase],iFrame+1)+ */
      /* 				   cpl_vector_get(alpha[iBase],iFrame+2))/5.); */
      /* } */
      /* for(int iFrame=0;iFrame<nFrame;iFrame++) { */
      /* 	cpl_vector_set(alpha[iBase],iFrame,cpl_vector_get(out,iFrame)); */
      /* } */
      /* cpl_vector_delete(out); */
      /* for(int iFrame=0;iFrame<nFrame;iFrame++) { */
      /* 	for(int iWave=0;iWave<nWave;iWave++) { */
      /* 	  par[3+iWave]=cpl_image_get(real[iFrame],iWave+1,iBase+2,&pis); */
      /* 	  par[3+nWave+iWave]=cpl_image_get(imag[iFrame],iWave+1,iBase+2,&pis); */
      /* 	  par[3+2*nWave+iWave]=cpl_vector_get(phiStatic[iBase],iWave); */
      /* 	  par[3+3*nWave+iWave]=cpl_image_get(real[iFrame],iWave+1,1,&pis); */
      /* 	} */
      /* 	xval=0.; */
      /* 	yval=0.; */
      /* 	alphaEstim=cpl_vector_get(alpha[iBase],iFrame); */
      /* 	for(int iWave=0;iWave<nWave;iWave++){ */
      /* 	  if (fabs(par[3+3*nWave+iWave]-(lambdamax+lambdamin)/2) < (lambdamax-lambdamin)/2) */
      /* 	    { */
      /* 	      eta=sqrt(par[3+iWave]*par[3+iWave]+par[3+nWave+iWave]*par[3+nWave+iWave]); */
      /* 	      phi=atan2(par[3+nWave+iWave],par[3+iWave]); */
      /* 	      xval+=eta*cos(phi-par[3+2*nWave+iWave]-(alphaEstim/par[3+3*nWave+iWave])); */
      /* 	      yval+=eta*sin(phi-par[3+2*nWave+iWave]-(alphaEstim/par[3+3*nWave+iWave])); */
      /* 	    } */
      /* 	} */
      /* 	betaEstim=atan2(yval,xval); */
      /* 	cpl_vector_set(beta[iBase],iFrame,betaEstim); */
      /* } */

      for(int iWave=0;iWave<nWave;iWave++){
	double xval=0.;
	double yval=0.;
	int pis;
	for(int iFrame=0;iFrame<nFrame;iFrame++) {
	  double eta=sqrt((cpl_image_get(real[iFrame],iWave+1,iBase+2,&pis))*(cpl_image_get(real[iFrame],iWave+1,iBase+2,&pis))+
		   (cpl_image_get(imag[iFrame],iWave+1,iBase+2,&pis))*(cpl_image_get(imag[iFrame],iWave+1,iBase+2,&pis)));
	  double phi=atan2(cpl_image_get(imag[iFrame],iWave+1,iBase+2,&pis),cpl_image_get(real[iFrame],iWave+1,iBase+2,&pis));
	  xval+=eta*cos(phi-cpl_vector_get(beta[iBase],iFrame)-(cpl_vector_get(alpha[iBase],iFrame)/cpl_image_get(real[iFrame],iWave+1,1,&pis)));
	  yval+=eta*sin(phi-cpl_vector_get(beta[iBase],iFrame)-(cpl_vector_get(alpha[iBase],iFrame)/cpl_image_get(real[iFrame],iWave+1,1,&pis)));
	}
	cpl_vector_set(phiStatic[iBase],iWave,atan2(yval,xval));
	eps+=pow(atan2(yval,xval)-cpl_vector_get(phiprev,iWave),2.0);
	cpl_vector_set(phiprev,iWave,atan2(yval,xval));
      }
      if (sqrt(eps)/sqrt(nWave) < EPS_MAX) {
	cpl_msg_info(cpl_func,"Convergence achieved at iteration %d",iConv+1);
	break;
      }
    }
  if (sqrt(eps)/sqrt(nWave) >= EPS_MAX) {
    cpl_msg_warning(cpl_func,"No convergence - Stop at iteration %d",NB_ITERATION_MAX);
  }
  gettimeofday(&tv2, NULL);
  cpl_msg_info(cpl_func, "Wall time for parallel loop = %f seconds\n",
               (double) (tv2.tv_usec - tv1.tv_usec) / 1000000 +
               (double) (tv2.tv_sec - tv1.tv_sec));
  }
  cpl_vector_delete(phiprev);
  return 0;
}


/*----------------------------------------------------------------------------*/
/**
   @ingroup opdwvpo
   @brief Corrects the OPD and Water Vapor Offset of a mat_corrflux structure. 
   @param corrFlux           Correlated Flux
   @param oiopdwvpo             mat_oiopdwvpo structure contaning the results
   @return 0 if no error 
*/
int mat_corr_opd_wvpo(mat_oiopdwvpo	*oiopdwvpo, mat_corrflux *corrFlux)
{
    /* Ranges and indices */
    int	iReg = 0,iFrame=0,iWlen=0,iWlen0=0,iBase=0;
    int	nbRegion, nbFrames;
    int	nbWlen, nbFreq, nbBases = 6;
    
  double	  lambda     = 0.;
  double	  dOverLambda= 0.;
  int		   i;
  int		  wPixel = 0;
  int		  pis        = 0;
  double	  phi        = 0.;
  double	  phasorIm   = 0.;
  double	  phasorRe   = 0.;
  double	  cfluxIm    = 0.;
  double	  cfluxRe    = 0.;
  cpl_image	**workChanRe = NULL;
  cpl_image     **workChanIm = NULL;

  cpl_vector *dispCoef   = NULL;
  int         uPixelMin  = 0;
  int         uPixelMax  = 0;
  int         cpt        = 0;
  double      opd        = 0;
  int detNum=0;
  int resolution=0;
  char *specResData=NULL;
  double posFringePeak=0.;
  char *detName=NULL;
  char	  band;
  char	  key[15];
  float  lambdamin	  = 0.;
  float  lambdamax	  = 0.;
  float	  wlenmin	  = 1e99;
  float	  wlenmax	  = 0.;

  double	  pressure	  = 0.;
  double	  temperature	  = 0.;
  double	  humidity	  = 0.;
  cpl_vector	 *DLstatic	  = NULL;
  cpl_vector	 *DLopl		  = NULL;
  cpl_vector	 *z		  = NULL;
  double	  argOpdChrom	  = 0.;
  double	  refractiveIndex = 0.;

  // Check input parameters
  mat_assert((corrFlux!=NULL),CPL_ERROR_NULL_INPUT, "no mat_corrflux \
argument given");
  mat_assert((oiopdwvpo!=NULL),CPL_ERROR_NULL_INPUT, "no mat_oiopdwvpo \
argument given");
  
  nbRegion   = corrFlux->imgdet->nbregion;
  nbFrames   = corrFlux->nbframe/nbRegion;
  int nbFrameTarget=nbFrames;
  if (cpl_propertylist_has(corrFlux->keywords,"PRO FRAME TARGET")) {
    nbFrameTarget=cpl_propertylist_get_int(corrFlux->keywords,"PRO FRAME TARGET");
  }
 
  /* Read the detector type (and hence band) */
  detName=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO DET CHIP NAME");
  if (detName == NULL)
    {
      cpl_msg_error(cpl_func,
		    "no ESO DET CHIP NAME keyword in frame");
      return -1;
    }
  mat_identify_det(detName, &band, &detNum);

  /* Read the spectral resolution */
  sprintf(key, "ESO INS DI%c ID",band);
  specResData  = (char *)cpl_propertylist_get_string(corrFlux->keywords,key);
  char *fil = NULL;
  fil = (char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS FIL NAME");
  mat_identify_res(specResData, fil, detNum, &resolution);

  

  // Loading of the dispersion coefficients
  dispCoef=cpl_vector_new(N_DEG_DISPERSION_LAW+1);
  cpl_vector_set(dispCoef,0,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF0"));
  cpl_vector_set(dispCoef,1,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF1"));
  cpl_vector_set(dispCoef,2,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF2"));
  cpl_vector_set(dispCoef,3,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF3"));
  cpl_vector_set(dispCoef,4,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF4"));


      
  /* Set the useful range of wavelengths to compute OPD */
  /* First of all retrieve the actual wavelength range */
  nbRegion = corrFlux->imgdet->nbregion;
  for(iReg = 0;iReg<nbRegion;iReg++)
  {
      cpl_msg_info(cpl_func,"Determining wavelength range");
      nbWlen = corrFlux->imgdet->list_region[iReg]->naxis[1];
        iWlen0 = corrFlux->imgdet->list_region[iReg]->corner[1];
        for(iWlen = 0;iWlen<nbWlen;iWlen++)
	{
            /* Compute wavelength and set wlen interval */
            lambda = mat_get_lambda(dispCoef, iWlen + iWlen0);
	    if(lambda < wlenmin)
		wlenmin = lambda;
	    if(lambda > wlenmax)
		wlenmax = lambda;
	}
  }
  mat_identify_wlrng(detNum, wlenmin, wlenmax, resolution, &lambdamin, &lambdamax);



  if (oiopdwvpo->array != NULL) {
    nbBases = (oiopdwvpo->array->nbstation*(oiopdwvpo->array->nbstation - 1))/2;
  } else {
    nbBases   = 6;
  }
   
  /* First initialize arrays */
  workChanIm  = cpl_calloc(nbRegion, sizeof(cpl_image *));
  workChanRe  = cpl_calloc(nbRegion, sizeof(cpl_image *));
  

  // Loading the ambient conditions and the DL positions
  pressure	= cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS AMBI PRES");
  temperature	= cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS AMBI TEMP");
  humidity	= cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS AMBI RHUM");
  cpl_msg_info(cpl_func,"Ambient Pressure = %f",pressure);
  cpl_msg_info(cpl_func,"Ambient Temperature = %f",temperature);
  cpl_msg_info(cpl_func,"Ambient Relative Humidity = %f",humidity);

  DLstatic = cpl_vector_new(MAT_MAXTEL);
  DLopl    = cpl_vector_new(MAT_MAXTEL);
  for(i=0;i<MAT_MAXTEL;i++) {
    cpl_vector_set(DLstatic,i,0.);
    cpl_vector_set(DLopl,i,0.);
  }
  if ( cpl_propertylist_has(oiopdwvpo->keywords,"ESO ISS CONF A1L") )
    {
      cpl_vector_set(DLstatic,0,cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS CONF A1L"));
      cpl_vector_set(DLopl,0,(cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT1 OPL START")
			      +cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT1 OPL END"))/2.);
    }
  if ( cpl_propertylist_has(oiopdwvpo->keywords,"ESO ISS CONF A2L") ) {
    cpl_vector_set(DLstatic,1,cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS CONF A2L"));
    cpl_vector_set(DLopl,1,(cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT2 OPL START")
			    +cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT2 OPL END"))/2.);
  }
  if ( cpl_propertylist_has(oiopdwvpo->keywords,"ESO ISS CONF A3L") ) {
    cpl_vector_set(DLstatic,2,cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS CONF A3L"));
    cpl_vector_set(DLopl,2,(cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT3 OPL START")
			    +cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT3 OPL END"))/2.);
  }
  if ( cpl_propertylist_has(oiopdwvpo->keywords,"ESO ISS CONF A4L") ) {
    cpl_vector_set(DLstatic,3,cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO ISS CONF A4L"));
    cpl_vector_set(DLopl,3,(cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT4 OPL START")
			    +cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO DEL DLT4 OPL END"))/2.);
  }
  for(i=0;i<MAT_MAXTEL;i++) {
    cpl_msg_info(cpl_func,"DL%d	STATIC = %f",i+1,cpl_vector_get(DLstatic,i));
    cpl_msg_info(cpl_func,"DL%d	OPL = %f",i+1,cpl_vector_get(DLopl,i));
  }
  
  z = cpl_vector_new(nbBases);
  for(iBase=0;iBase<nbBases;iBase++)
    {
      int idxStation[2] = {0,0};
      for(int iStation=0;iStation<MAT_MAXTEL;iStation++)
	{
	  if (oiopdwvpo->oiopd->list_opd[iBase]->stationindex[0] == oiopdwvpo->array->list_station[iStation]->staindex) {
	    idxStation[0]=iStation;
	  }
	  if (oiopdwvpo->oiopd->list_opd[iBase]->stationindex[1] == oiopdwvpo->array->list_station[iStation]->staindex) {
	    idxStation[1]=iStation;
	  }
	}
      cpl_vector_set(z,iBase,cpl_vector_get(DLstatic,idxStation[1])+cpl_vector_get(DLopl,idxStation[1])-
		     cpl_vector_get(DLstatic,idxStation[0])-cpl_vector_get(DLopl,idxStation[0]));
    }

  cpl_msg_info(cpl_func,"Correction of the OPD and WVPO");


  // Initialize arrays
  for(iReg=0;iReg<nbRegion;iReg++)
    {
      nbFreq = corrFlux->imgdet->list_region[iReg]->naxis[0];
      nbWlen = corrFlux->imgdet->list_region[iReg]->naxis[1];

      workChanRe[iReg]  = cpl_image_new(nbFreq, nbWlen, CPL_TYPE_FLOAT);
      workChanIm[iReg]  = cpl_image_new(nbFreq, nbWlen, CPL_TYPE_FLOAT);
	
      for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
	{
	  /* Get real part and imaginary part of coherent flux */
	  /*  and put it in work channel                       */
	  /* W = Ci                                            */
	  cpl_image_copy(workChanRe[iReg], corrFlux->
			 list_corr[iReg*nbFrames+iFrame]->imgreg[0],1,1);
	  cpl_image_copy(workChanIm[iReg], corrFlux->
			 list_corr[iReg*nbFrames+iFrame]->imgreg[1],1,1);
	    
	  /* Loop on wavelengths */
	  for(iWlen=0;iWlen<nbWlen;iWlen++)
	    {
	      /* Compute wavelength */	      
	      lambda=mat_get_lambda(dispCoef,iWlen+corrFlux->imgdet->list_region[iReg]->corner[1]);
	      /* Pick-up phasor at the peak frequency and store it */
	      for(iBase=0;iBase<6;iBase++)
		{
		  // Do the OPD correction on a range of frequencies
		  posFringePeak=mat_get_pos_fringe_peak(lambda,detNum,resolution,iBase);
		  dOverLambda=(posFringePeak-corrFlux->imgdet->list_region[iReg]->naxis[0]/2)/(3*(iBase+1));
		  uPixelMin=(int)posFringePeak-(int)(dOverLambda*1.5)-1;
		  uPixelMax=(int)posFringePeak+(int)(dOverLambda*1.5)+1;


		  // Correction of the chromatic OPD (following Mathar)
		  /////////////////////////////////////////////////////////////
		  refractiveIndex = mat_oiopd_mathar(lambda,pressure*100.,temperature+273.15,humidity);
		  argOpdChrom = 0.*2*CPL_MATH_PI*refractiveIndex*cpl_vector_get(z,iBase)*1.E6/lambda;

		  
		  // Get the OPD value
		  cpt = iFrame*nbBases + iBase;
		  opd = oiopdwvpo->oiopd->list_opd[cpt]->opd;
		  // Compute the phase associated with the OPD
		  phi = 2 * CPL_MATH_PI * opd / lambda + argOpdChrom;
			
		  /* Buildup a phasor with 1 amplitude */
		  /* out of the phase */
		  phasorRe = cos(phi);
		  phasorIm = sin(phi);

		  for(wPixel=uPixelMin;wPixel<=uPixelMax;wPixel++)
		    {
		      /* Get the coherent flux value */
		      cfluxRe=cpl_image_get(workChanRe[iReg],
					    wPixel+1,iWlen+1,&pis);
		      cfluxIm=cpl_image_get(workChanIm[iReg],
					    wPixel+1,iWlen+1,&pis);
			
	     	
		      /* Subtract phase phasor from the complex
			 coherent flux to obtain a clean coherent flux
			 which is stored back into the coherent flux itself */
		      /* Real part */
		      cpl_image_set(workChanRe[iReg],wPixel+1,iWlen+1,
				    cfluxRe*phasorRe+cfluxIm*phasorIm);
		      /* Imaginary part */
		      cpl_image_set(workChanIm[iReg],wPixel+1,iWlen+1,
				    cfluxIm*phasorRe-cfluxRe*phasorIm);
		    }
		}
	    }

	  /* Store real part and imaginary part of coherent flux */
	  cpl_image_copy(corrFlux->
			 list_corr[iReg*nbFrames+iFrame]->imgreg[0],
			 workChanRe[iReg], 1,1);
	  cpl_image_copy(corrFlux->
			 list_corr[iReg*nbFrames+iFrame]->imgreg[1],
			 workChanIm[iReg], 1,1);
	}
    }


  for(iReg=0;iReg<nbRegion;iReg++)
    {
      cpl_image_delete(workChanRe[iReg]);
      cpl_image_delete(workChanIm[iReg]);
    }
  cpl_free(workChanRe);
  cpl_free(workChanIm);

  cpl_vector_delete(dispCoef);
  cpl_vector_delete(DLstatic);
  cpl_vector_delete(DLopl);
  cpl_vector_delete(z);

  return 0;
}

/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
    
/*----------------------------------------------------------------------------*/
/**
   @ingroup opdwvpo
   @brief Initialize a mat_oiopdwvpo structure from a mat_corrflux structure
   @param corrFlux           contains the mat_corrflux structure 
   @param oiopdwvpo             contains the mat_oiopdwvpo structure
   @return 0 if no error 
*/
int mat_oiopdwvpo_init_from_corrflux(mat_corrflux *corrFlux,mat_oiopdwvpo *oiopdwvpo)
{
#include "mat_station_default.h"
  char insname[]   = "MATISSE";
  char arrayname[] = "VLTI";
  char cooframe[]  = "GEOCENTRIC";
  double arrayx =  1951952.;
  double arrayy = -5483173.;
  double arrayz = -2668147.;
  const char *charFoo=NULL;
  char keyName[100]="";
  /* char *bcd1Status=NULL; */
  /* char *bcd2Status=NULL; */
  char *bcdMode=NULL;
  /* int j=0; */
  int k=0;
  int l=0;
  /* int cpt=0; */
  int nbBase=0;
  double val;
  /* double dispCoef[3]; */
  //  int nbstation;
  int science;
    

  int nbFrameTarget=0;
  if (cpl_propertylist_has(corrFlux->keywords,"PRO FRAME TARGET"))
    {
      nbFrameTarget=cpl_propertylist_get_int(corrFlux->keywords,"PRO FRAME TARGET");
    }
  else
    {
      nbFrameTarget=corrFlux->nbframe;
    }


  oiopdwvpo->keywords=cpl_propertylist_duplicate(corrFlux->keywords);
  if ( cpl_propertylist_has(oiopdwvpo->keywords,"ESO CFG BCD MODE") ) {
    bcdMode = (char *)cpl_propertylist_get_string(oiopdwvpo->keywords,"ESO CFG BCD MODE");
  } 
  // OIARRAY
  oiopdwvpo->array=NULL;
  oiopdwvpo->array= (mat_array *)cpl_calloc(1, sizeof(mat_array));
  oiopdwvpo->array->arrayname = cpl_strdup(arrayname);
  oiopdwvpo->array->cooframe = cpl_strdup(cooframe);
  oiopdwvpo->array->arrayx=arrayx;
  oiopdwvpo->array->arrayy=arrayy;
  oiopdwvpo->array->arrayz=arrayz;
  oiopdwvpo->array->telescope = NULL;
  oiopdwvpo->array->dcsdictionaryid = NULL;
  oiopdwvpo->array->dcsid = NULL;
  oiopdwvpo->array->nbstation=cpl_propertylist_has(oiopdwvpo->keywords,"ESO ISS CONF T1NAME");
  oiopdwvpo->array->nbstation+=cpl_propertylist_has(oiopdwvpo->keywords,"ESO ISS CONF T2NAME");
  oiopdwvpo->array->nbstation+=cpl_propertylist_has(oiopdwvpo->keywords,"ESO ISS CONF T3NAME");
  oiopdwvpo->array->nbstation+=cpl_propertylist_has(oiopdwvpo->keywords,"ESO ISS CONF T4NAME");
    
  if (oiopdwvpo->array->nbstation == 0) {
    cpl_msg_warning(cpl_func,"Keyword ESO ISS CONF T?NAME not found");
    cpl_msg_warning(cpl_func,"We fix nbstation to 4");
    oiopdwvpo->array->nbstation=4;
  }

  oiopdwvpo->array->list_station= (mat_station **)cpl_calloc(oiopdwvpo->array->nbstation, sizeof(mat_station *));
  for (k = 0; k < oiopdwvpo->array->nbstation; k++)
    {
      oiopdwvpo->array->list_station[k]= (mat_station *)cpl_calloc(1, sizeof(mat_station));
      sprintf(keyName,"ESO ISS CONF T%dNAME",k+1);
      charFoo=(char *)cpl_propertylist_get_string(oiopdwvpo->keywords,keyName);
      if (charFoo == NULL)
	{
	  /* sprintf(oiopdwvpo->array->list_station[k]->telname,"%s","foo"); */
	  sprintf(oiopdwvpo->array->list_station[k]->telname,"%s",tel_name_default[k]);
	}
      else
	{
	  sprintf(oiopdwvpo->array->list_station[k]->telname,"%s",charFoo);
	}
      sprintf(keyName,"ESO ISS CONF STATION%d",k+1);
      charFoo=(char *)cpl_propertylist_get_string(oiopdwvpo->keywords,keyName);
      if (charFoo == NULL)
	{
	  /* sprintf(oiopdwvpo->array->list_station[k]->staname,"%s","foo"); */
	  sprintf(oiopdwvpo->array->list_station[k]->staname,"%s",sta_name_default[k]);
	}
      else
	{
	  sprintf(oiopdwvpo->array->list_station[k]->staname,"%s",charFoo);
	}
      /* oiopdwvpo->array->list_station[k]->staindex = k + 1; */
      oiopdwvpo->array->list_station[k]->staindex=mat_get_sta_index(oiopdwvpo->array->list_station[k]->staname);

      sprintf(keyName,"ESO ISS CONF T%dX",k+1);
      if (cpl_propertylist_has(oiopdwvpo->keywords,keyName)) {
	oiopdwvpo->array->list_station[k]->staxyz[0] = cpl_propertylist_get_double(oiopdwvpo->keywords,keyName);
      } else {
	oiopdwvpo->array->list_station[k]->staxyz[0] = sta_x_default[k];
      }
      sprintf(keyName,"ESO ISS CONF T%dY",k+1);
      if (cpl_propertylist_has(oiopdwvpo->keywords,keyName)) {
	oiopdwvpo->array->list_station[k]->staxyz[1] = cpl_propertylist_get_double(oiopdwvpo->keywords,keyName);
      } else {
	oiopdwvpo->array->list_station[k]->staxyz[1] = sta_y_default[k];
      }
      sprintf(keyName,"ESO ISS CONF T%dZ",k+1);
      if (cpl_propertylist_has(oiopdwvpo->keywords,keyName)) {
	oiopdwvpo->array->list_station[k]->staxyz[2] = cpl_propertylist_get_double(oiopdwvpo->keywords,keyName);
      } else {
	oiopdwvpo->array->list_station[k]->staxyz[2] = sta_z_default[k];
      }

      if (oiopdwvpo->array->list_station[k]->telname[0] == 'U')
	{
	  oiopdwvpo->array->list_station[k]->diameter=8.2;
	}
      else
	{
	  oiopdwvpo->array->list_station[k]->diameter=1.8;
	}
      oiopdwvpo->array->list_station[k]->fov=1.0;
      sprintf(oiopdwvpo->array->list_station[k]->fovtype,"%s","RADIUS");

    }
  // OITARGET
  oiopdwvpo->oitarget=NULL;
  if (cpl_propertylist_has(oiopdwvpo->keywords,"ESO OBS TARG NAME")) {
    charFoo=(char *)cpl_propertylist_get_string(oiopdwvpo->keywords,"ESO OBS TARG NAME");
  } else {
    charFoo=(char *)cpl_propertylist_get_string(oiopdwvpo->keywords,"ESO OBS TARGET NAME");
  }
  science=cpl_propertylist_get_bool(corrFlux->keywords,"ESO PRO SCIENCE");
      
  if (charFoo != NULL) {
    oiopdwvpo->oitarget=(mat_oitarget *)cpl_calloc(1, sizeof(mat_oitarget));
    oiopdwvpo->oitarget->targetname = cpl_strdup(charFoo);
    oiopdwvpo->oitarget->equinox=0.;
    oiopdwvpo->oitarget->raep0=cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO COU GUID RA");
    oiopdwvpo->oitarget->decep0=cpl_propertylist_get_double(oiopdwvpo->keywords,"ESO COU GUID DEC");
    oiopdwvpo->oitarget->raerr=0.;
    oiopdwvpo->oitarget->decerr=0.;
    oiopdwvpo->oitarget->sysvel=0.;
    oiopdwvpo->oitarget->veltyp = cpl_strdup("HELIOCEN");
    oiopdwvpo->oitarget->veldef = cpl_strdup("OPTICAL");
    oiopdwvpo->oitarget->pmra=0.;
    oiopdwvpo->oitarget->pmdec=0.;
    oiopdwvpo->oitarget->pmraerr=0.;
    oiopdwvpo->oitarget->pmdecerr=0.;
    oiopdwvpo->oitarget->parallax=0.;
    oiopdwvpo->oitarget->paraerr=0.;
    oiopdwvpo->oitarget->spectyp = cpl_strdup("UNKNOWN");
    if ( science == 0 ) {
      oiopdwvpo->oitarget->category = cpl_strdup("CAL");
    } else {
      oiopdwvpo->oitarget->category = cpl_strdup("SCI");
    }
  } else {
    cpl_msg_warning(cpl_func,"Keyword ESO OBS TARG NAME not found");
    oiopdwvpo->oitarget=(mat_oitarget *)cpl_calloc(1, sizeof(mat_oitarget));
    oiopdwvpo->oitarget->targetname = cpl_strdup("FOO");
    oiopdwvpo->oitarget->equinox=0.;
    oiopdwvpo->oitarget->raep0=0.;
    oiopdwvpo->oitarget->decep0=0.;
    oiopdwvpo->oitarget->raerr=0.;
    oiopdwvpo->oitarget->decerr=0.;
    oiopdwvpo->oitarget->sysvel=0.;
    oiopdwvpo->oitarget->veltyp = cpl_strdup("HELIOCEN");
    oiopdwvpo->oitarget->veldef = cpl_strdup("OPTICAL");
    oiopdwvpo->oitarget->pmra=0.;
    oiopdwvpo->oitarget->pmdec=0.;
    oiopdwvpo->oitarget->pmraerr=0.;
    oiopdwvpo->oitarget->pmdecerr=0.;
    oiopdwvpo->oitarget->parallax=0.;
    oiopdwvpo->oitarget->paraerr=0.;
    oiopdwvpo->oitarget->spectyp = cpl_strdup("UNKNOWN");
    if ( science == 0 ) {
      oiopdwvpo->oitarget->category = cpl_strdup("CAL");
    } else {
      oiopdwvpo->oitarget->category = cpl_strdup("SCI");
    }
  }

  // OIOPD
  oiopdwvpo->oiopd=NULL;
  printf("\n");
  oiopdwvpo->oiopd= (mat_oiopd *)cpl_calloc(1, sizeof(mat_oiopd));
  oiopdwvpo->oiopd->dateobs = mat_propertylist_get_string_default(NULL, 0,
								  oiopdwvpo->keywords, "ESO TPL START", NULL);
  oiopdwvpo->oiopd->insname = cpl_strdup(insname);
  oiopdwvpo->oiopd->arrayname = cpl_strdup(arrayname);
  if (oiopdwvpo->array != NULL) {
    nbBase=(oiopdwvpo->array->nbstation*(oiopdwvpo->array->nbstation - 1))/2;
  } else {
    nbBase=6;
  }
  //  oiopdwvpo->oiopd->nbopd = (corrFlux->nbframe/corrFlux->imgdet->nbregion) * nbBase;
  oiopdwvpo->oiopd->nbopd = nbFrameTarget * nbBase;
  oiopdwvpo->oiopd->list_opd= (mat_opdelem **)cpl_calloc(oiopdwvpo->oiopd->nbopd, sizeof(mat_opdelem *));
  for(k = 0; k < nbFrameTarget; k++) {
    val = corrFlux->list_corr[k]->time;
    for (l= 0; l< nbBase; l++) {
      oiopdwvpo->oiopd->list_opd[k*nbBase+l]=
	(mat_opdelem *)cpl_calloc(1, sizeof(mat_opdelem));
      oiopdwvpo->oiopd->list_opd[k*nbBase+l]->time =
	(val-(int)val)*86400.;
      oiopdwvpo->oiopd->list_opd[k*nbBase+l]->dateobsmjd = corrFlux->list_corr[k]->time;
      /* corrFlux->imgdet->dateobsmjd; */
      oiopdwvpo->oiopd->list_opd[k*nbBase+l]->exptime =
	corrFlux->list_corr[0]->exptime;
      oiopdwvpo->oiopd->list_opd[k*nbBase+l]->opd = 0.;
      oiopdwvpo->oiopd->list_opd[k*nbBase+l]->opderr = 0.;
      oiopdwvpo->oiopd->list_opd[k*nbBase+l]->tempoff = 0.;
      oiopdwvpo->oiopd->list_opd[k*nbBase+l]->tempofferr = 0.;
      oiopdwvpo->oiopd->list_opd[k*nbBase+l]->humoff = 0.;
      oiopdwvpo->oiopd->list_opd[k*nbBase+l]->humofferr = 0.;
    }
  }
  if (oiopdwvpo->array != NULL) {
    for (k=0;k<nbFrameTarget;k++) {
      /* if (strcmp(bcd1Status,"OUT")) { */
      /* 	if (strcmp(bcd2Status,"OUT")) { */
      if (strcmp(bcdMode,"IN-IN")==0) {
	// IN-IN
	if (oiopdwvpo->array->nbstation==4) {
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[0]=oiopdwvpo->array->list_station[3]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[1]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[1]=oiopdwvpo->array->list_station[3]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+3]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+3]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+4]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+4]->stationindex[1]=oiopdwvpo->array->list_station[3]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+5]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+5]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	} else if (oiopdwvpo->array->nbstation==3) {
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[1]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	} else {
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[1]=oiopdwvpo->array->list_station[0]->staindex;
	}
      }
      if (strcmp(bcdMode,"IN-OUT")==0) {    
	/* } else { */
	  // IN-OUT
	if (oiopdwvpo->array->nbstation==4) {
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[0]=oiopdwvpo->array->list_station[3]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[1]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[1]=oiopdwvpo->array->list_station[3]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+3]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+3]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+4]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+4]->stationindex[1]=oiopdwvpo->array->list_station[3]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+5]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+5]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	} else if (oiopdwvpo->array->nbstation==3) {
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[1]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	} else {
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[1]=oiopdwvpo->array->list_station[0]->staindex;
	}
      }
      
      if (strcmp(bcdMode,"OUT-IN")==0) {    
      /* } else { */
      /* 	if (strcmp(bcd2Status,"OUT")) { */
	  // OUT-IN
	if (oiopdwvpo->array->nbstation==4) {
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[0]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[1]=oiopdwvpo->array->list_station[3]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[1]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+3]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+3]->stationindex[1]=oiopdwvpo->array->list_station[3]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+4]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+4]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+5]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+5]->stationindex[1]=oiopdwvpo->array->list_station[3]->staindex;
	} else if (oiopdwvpo->array->nbstation==3) {
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[1]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	} else {
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[1]=oiopdwvpo->array->list_station[1]->staindex;
	}
      }
      if (strcmp(bcdMode,"OUT-OUT")==0) {         
	/* } else { */
	  // OUT-OUT
	if (oiopdwvpo->array->nbstation==4) {
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[0]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[1]=oiopdwvpo->array->list_station[3]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[1]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+3]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+3]->stationindex[1]=oiopdwvpo->array->list_station[3]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+4]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+4]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+5]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+5]->stationindex[1]=oiopdwvpo->array->list_station[3]->staindex;
	} else if (oiopdwvpo->array->nbstation==3) {
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[1]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[0]=oiopdwvpo->array->list_station[1]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+1]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+2]->stationindex[1]=oiopdwvpo->array->list_station[2]->staindex;
	} else {
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[0]=oiopdwvpo->array->list_station[0]->staindex;
	  oiopdwvpo->oiopd->list_opd[k*nbBase+0]->stationindex[1]=oiopdwvpo->array->list_station[1]->staindex;
	}
      }
    }
  }
  return 0;
}

/*----------------------------------------------------------------------------*/

cpl_error_code mat_oiopd_delete(mat_oiopd *oiopd)
{
  int i=0;
  if (oiopd == NULL) return CPL_ERROR_NONE;
  if (oiopd->list_opd != NULL) 
    {
      if (oiopd->dateobs != NULL)
	{
	  cpl_free(oiopd->dateobs);
	  oiopd->dateobs = NULL;
	}
      if (oiopd->arrayname != NULL)
	{
	  cpl_free(oiopd->arrayname);
	  oiopd->arrayname = NULL;
	}
      if (oiopd->insname != NULL)
	{
	  cpl_free(oiopd->insname);
	  oiopd->insname = NULL;
	}
      
      for(i=0;i<oiopd->nbopd;i++) {
	cpl_free(oiopd->list_opd[i]);
      }
      cpl_free(oiopd->list_opd);
      oiopd->list_opd=NULL;
    }
  cpl_free(oiopd);
  /* oiopd=NULL; */
  return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
   @ingroup opdwvpo
   @brief delete structure mat_oiopdwvpo
   @param  oiopdwvpo          current structure
   @return error code
*/
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_oiopdwvpo_delete(mat_oiopdwvpo *oiopdwvpo)
{
  if (oiopdwvpo == NULL) return CPL_ERROR_NONE;
  if (oiopdwvpo->keywords != NULL)
    {
      cpl_propertylist_delete(oiopdwvpo->keywords);
      oiopdwvpo->keywords = NULL;
    }
  if (oiopdwvpo->array != NULL) 
    {
      mat_array_delete(oiopdwvpo->array);
      oiopdwvpo->array = NULL;
    }
  if (oiopdwvpo->oitarget != NULL) 
    {
      mat_oitarget_delete(oiopdwvpo->oitarget);
      oiopdwvpo->oitarget = NULL;
    }
  if (oiopdwvpo->oiopd != NULL) 
    {
      mat_oiopd_delete(oiopdwvpo->oiopd);
      oiopdwvpo->oiopd = NULL;
    }
  cpl_free(oiopdwvpo);
  return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
   @ingroup opdwvpo
   @brief delete a list of structure mat_oiopdwvpo
   @param  oiopdwvpo         list
   @param nboiopd            number of structures
   @return error code
*/
/*-----------------------------------------------------------------------------*/

cpl_error_code mat_oiopdwvpo_list_delete(mat_oiopdwvpo **oiopdwvpo, int nboiopd) {
  int i = 0;

  if(oiopdwvpo == NULL) return CPL_ERROR_NONE;

  for(i = 0; i < nboiopd; ++i)
    {
      if (oiopdwvpo[i] != NULL)
	{
	  mat_oiopdwvpo_delete(oiopdwvpo[i]);
	  oiopdwvpo[i] = NULL;
	}
    }
  cpl_free(oiopdwvpo);
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup opdwvpo
   @brief Save a mat_oiopwvpo structure in a FITS file. This function saves the 
   the oitarget only. The other tables should be appened with the 
   function mat_oiopd_append for example.
   @param oiopdwvpo            data structure to save
   @param frameset          input frameset
   @param usedframes        selected frameset
   @param parlist           recipe parameter list
   @param recipe_name       name of recipe
   @param filename          name of the FITS file
   @return cpl_error_code
*/
/*----------------------------------------------------------------------------*/
cpl_error_code  mat_oiopdwvpo_save(mat_oiopdwvpo           *oiopdwvpo,
				   cpl_frameset            *frameset,
				   cpl_frameset            *usedframes ,
				   const cpl_parameterlist *parlist,
				   const char              *recipe_name,
				   char                    *filename)
{
  cpl_table        *oitargetTab = NULL;
  cpl_propertylist *applist     = NULL;
  cpl_propertylist *keyTab      = NULL;
  cpl_errorstate    prestate    = cpl_errorstate_get();
  /* const char       *protech     = NULL; */

  /*Check input parameters*/
  mat_assert_value((oiopdwvpo!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   " no mat_oiopwvpo argument given");
  mat_assert_value((frameset!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, 
		   "no cpl_frameset (frameset) argument given");
  mat_assert_value((usedframes!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, 
		   "no cpl_frameset (usedframes) argument given");
  mat_assert_value((filename!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no filename given");
  mat_assert_value((recipe_name!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no recipe_name given");

    
  keyTab=cpl_propertylist_new();
  /*Remove the keywords from the DPR categories*/
  /* protech=cpl_propertylist_get_string(oiopdwvpo->keywords,"ESO PRO TECH"); */
  cpl_propertylist_erase_regexp(oiopdwvpo->keywords, "DPR", 0);
  cpl_propertylist_erase_regexp(oiopdwvpo->keywords, "PRO CATG", 0);
  cpl_propertylist_erase_regexp(oiopdwvpo->keywords, "PRO TECH", 0);
  /*Add the keywords from the PRO categories*/
  /* cpl_propertylist_append_string(oiopdwvpo->keywords, CPL_DFS_PRO_SCIENCE,  */
  /* 				   "T"); */
  cpl_propertylist_append_string(oiopdwvpo->keywords, CPL_DFS_PRO_TECH, "IMAGE");
  cpl_propertylist_append_string(oiopdwvpo->keywords, CPL_DFS_PRO_CATG, "OI_OPDWVPO");

  oitargetTab = mat_oitarget_to_table(oiopdwvpo->oitarget, keyTab);


  /*Make a new FITS file */
  cpl_propertylist_erase(oiopdwvpo->keywords,"RADECSYS");
  applist = cpl_propertylist_duplicate(oiopdwvpo->keywords);
  if (cpl_dfs_save_table(frameset,NULL,parlist,usedframes,NULL,
			 oitargetTab, keyTab,recipe_name,applist,
			 NULL,PACKAGE "/" PACKAGE_VERSION, filename) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func,"Unable to create OIOPDWVO fits file");
    }
  else
    {
      if (oiopdwvpo->array != NULL)
	{
	  mat_array_append(filename,oiopdwvpo->array);
	}
      if (oiopdwvpo->oiopd != NULL)
	{
	  int nbFrameTarget=-1;
	  if ( cpl_propertylist_has(oiopdwvpo->keywords,"PRO FRAME TARGET") ) {
	    nbFrameTarget = cpl_propertylist_get_int(oiopdwvpo->keywords,"PRO FRAME TARGET");
	  }
	  mat_oiopd_append(filename,oiopdwvpo->oiopd,nbFrameTarget);
	}
    }

  /*Free memory*/
  cpl_propertylist_delete(applist);
  cpl_propertylist_delete(keyTab);
  cpl_table_delete(oitargetTab);

  if (!cpl_errorstate_is_equal(prestate))
    {
      cpl_errorstate_set(prestate);
      return CPL_ERROR_UNSPECIFIED;
    } 
  return CPL_ERROR_NONE;
    
}

/**
   @ingroup opdwvpo
   @brief copy a  mat_oiopd data structure into a cpl table and a propertylist
   @param oiopd     data structure to copy
   @param plist          propertylist containing the header 
   @return cpl table containing the data
*/
cpl_table *mat_oiopd_to_table(mat_oiopd *oiopd, cpl_propertylist *plist)
{
  cpl_table    *table;
  cpl_array    *arr;
  int           i;
  int nbFrameTarget=-1;

  if (plist != NULL)
    {
      if ( cpl_propertylist_has(plist,"OI_REVN") )
	{
	  cpl_propertylist_erase(plist,"OI_REVN");
	}
      if ( cpl_propertylist_has(plist,"PRO FRAME TARGET") )
	{
	  nbFrameTarget = cpl_propertylist_get_int(plist,"PRO FRAME TARGET");
	} 
      cpl_propertylist_append_string(plist,"EXTNAME","OPD");
      cpl_propertylist_append_int(plist,   "OI_REVN",2);
      cpl_propertylist_append_string(plist,"DATE-OBS",oiopd->dateobs);
      cpl_propertylist_append_string(plist,"ARRNAME",oiopd->arrayname);
      cpl_propertylist_append_string(plist,"INSNAME",oiopd->insname);
    }

  if (nbFrameTarget > 0)
    {
      table = cpl_table_new(nbFrameTarget);
    }
  else
    {
      table = cpl_table_new(oiopd->nbopd/6);
    }
  if (table == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_table");
      return NULL;
    }
  /*Add columns*/
  cpl_table_new_column(      table, "TIME", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit( table, "TIME","s");
  cpl_table_new_column(      table, "MJD", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit( table, "MJD","day");
  cpl_table_new_column(      table, "INT_TIME", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit( table, "INT_TIME","s");
  cpl_table_new_column_array(table, "OPD", CPL_TYPE_DOUBLE,6);
  cpl_table_set_column_unit( table, "OPD","micrometer");
  cpl_table_new_column_array(table, "OPDERR", CPL_TYPE_DOUBLE,6);
  cpl_table_set_column_unit( table, "OPDERR","micrometer");
  cpl_table_new_column_array(table, "TEMPOFF", CPL_TYPE_DOUBLE,6);
  cpl_table_new_column_array(table, "TEMPOFFERR", CPL_TYPE_DOUBLE,6);
  cpl_table_new_column_array(table, "HUMOFF", CPL_TYPE_DOUBLE,6);
  cpl_table_new_column_array(table, "HUMOFFERR", CPL_TYPE_DOUBLE,6);
  cpl_table_new_column_array(table,"STA_INDEX", CPL_TYPE_INT,12);
  // Fill the table
  for(i = 0; i < oiopd->nbopd; i+=6)
    {
      if (nbFrameTarget < 0 || i/6 < nbFrameTarget)
	{
	  mat_opdelem  *el = oiopd->list_opd[i];
	  cpl_table_set_double(table,"TIME",i/6, el->time);
	  cpl_table_set_double(table,"MJD",i/6, el->dateobsmjd);
	  cpl_table_set_double(table,"INT_TIME",i/6, el->exptime);
      

	  double vecopd[6];
	  double vecopderr[6];
	  double vectemp[6];
	  double vectemperr[6];
	  double vechum[6];
	  double vechumerr[6];
	  int vecsta[12];
	  for(int iBase=0;iBase<6;iBase++) {
	    vecopd[iBase]=oiopd->list_opd[i+iBase]->opd;
	    vecopderr[iBase]=oiopd->list_opd[i+iBase]->opderr;
	    vectemp[iBase]=oiopd->list_opd[i+iBase]->tempoff;
	    vectemperr[iBase]=oiopd->list_opd[i+iBase]->tempofferr;
	    vechum[iBase]=oiopd->list_opd[i+iBase]->humoff;
	    vechumerr[iBase]=oiopd->list_opd[i+iBase]->humofferr;
	    vecsta[iBase*2]=oiopd->list_opd[i+iBase]->stationindex[0];
	    vecsta[iBase*2+1]=oiopd->list_opd[i+iBase]->stationindex[1];
	  }

	  arr = cpl_array_wrap_double(vecopd, 6);
	  cpl_table_set_array(table,"OPD",i/6, arr);
	  cpl_array_unwrap(arr);

	  arr = cpl_array_wrap_double(vecopderr, 6);
	  cpl_table_set_array(table,"OPDERR",i/6, arr);
	  cpl_array_unwrap(arr);

	  arr = cpl_array_wrap_double(vectemp, 6);
	  cpl_table_set_array(table,"TEMPOFF",i/6, arr);
	  cpl_array_unwrap(arr);

	  arr = cpl_array_wrap_double(vectemperr, 6);
	  cpl_table_set_array(table,"TEMPOFFERR",i/6, arr);
	  cpl_array_unwrap(arr);

	  arr = cpl_array_wrap_double(vechum, 6);
	  cpl_table_set_array(table,"HUMOFF",i/6, arr);
	  cpl_array_unwrap(arr);
	  
	  arr = cpl_array_wrap_double(vechumerr, 6);
	  cpl_table_set_array(table,"HUMOFFERR",i/6, arr);
	  cpl_array_unwrap(arr);

	  arr = cpl_array_wrap_int(vecsta, 12);
	  cpl_table_set_array(table, "STA_INDEX", i/6, arr);
	  cpl_array_unwrap(arr);
	}
    }
  /* cpl_table_set_column_savetype(table,"TARGET_ID",CPL_TYPE_SHORT); */
  cpl_table_set_column_savetype(table,"STA_INDEX",CPL_TYPE_SHORT);
  return table;
}


/*----------------------------------------------------------------------------*/
/**
   @ingroup opdwvpo
   @brief Append an OPD binary table to an existing OIFITS file
   @param filename          name of the FITS file
   @param oiopd             the mat_oiopd structure containing the 
   informations for filling the binary table
   @param nbTarget          number of target frames
   @return cpl_error_code
*/
/*----------------------------------------------------------------------------*/
cpl_error_code  mat_oiopd_append(char *filename,mat_oiopd *oiopd, int nbTarget) {
  cpl_table *oiopdTab = NULL;
  cpl_propertylist *keyTab = NULL;
  cpl_errorstate prestate = cpl_errorstate_get();

  /*Check input parameters*/
  mat_assert_value((oiopd!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_oiopd (oiopd) argument given");
  mat_assert_value((filename!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no filename given");

  keyTab=cpl_propertylist_new();
  if (nbTarget >= 0) {
    cpl_propertylist_append_int(keyTab,"PRO FRAME TARGET",nbTarget);
  }
  oiopdTab = mat_oiopd_to_table(oiopd, keyTab);
  cpl_table_save(oiopdTab,NULL,keyTab,filename,CPL_IO_EXTEND);

  cpl_propertylist_delete(keyTab);
  cpl_table_delete(oiopdTab);
  if (!cpl_errorstate_is_equal(prestate))
    {
      cpl_errorstate_set(prestate);
      return CPL_ERROR_UNSPECIFIED;
    }
    
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup opdwvpo
   @brief Load OBJ_OPDWVPO data in mat_oiopdwvpo structure
   @param frame           current frame
   @return mat_oiopdwvpo
*/
/*-----------------------------------------------------------------------------*/
mat_oiopdwvpo *mat_oiopdwvpo_load(cpl_frame *frame)
{
  mat_oiopdwvpo    *oiopdwvpo;
  const char       *fname;
  int               nbextent;
  int               i;
  const char       *extname;
  cpl_propertylist *plist;
  cpl_table        *table;
    
  cpl_msg_info(cpl_func,"Loading oiopdwvpo frame set");
    
  mat_assert_value((frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_frame (frame) argument given");
    
  // Allocate the structure and initialize its attributes
  oiopdwvpo = (mat_oiopdwvpo *)cpl_calloc(1, sizeof(mat_oiopdwvpo));
  if (oiopdwvpo == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for oiopdwvpo");
      return NULL;
    }
    
  // load the primary keywords (always)
  oiopdwvpo->keywords = cpl_propertylist_load(cpl_frame_get_filename(frame), 0);
        
  // we load everything which is in the input file and listed in the mat_oiopdwvpo data structure
  fname    = cpl_frame_get_filename(frame);
  nbextent = cpl_frame_get_nextensions(frame);
  for(i = 0; i < nbextent; i++)
    {
      plist = cpl_propertylist_load(fname, i + 1);
      extname = cpl_propertylist_get_string(plist,"EXTNAME");
      if (extname == NULL)
	{
	  cpl_propertylist_delete(plist);
	  continue;
	}
      if (!strcmp(extname,"OI_TARGET"))
	{
	  table = cpl_table_load(fname, i + 1, 0);
	  oiopdwvpo->oitarget = mat_oitarget_from_table(plist, table);
	  cpl_table_delete(table);
	}
      if (!strcmp(extname,"OI_ARRAY") || !strcmp(extname,"ARRAY_GEOMETRY"))
	{
	  table = cpl_table_load(fname, i + 1, 0);
	  oiopdwvpo->array = mat_array_from_table(plist, table);

	  cpl_table_delete(table);
	}
      if (!strcmp(extname,"OPD"))
	{
	  table = cpl_table_load(fname, i + 1, 0);
	  
	  oiopdwvpo->oiopd = mat_oiopd_from_table(plist, table);

	  cpl_table_delete(table);
	}
      cpl_propertylist_delete(plist);
    }
	
  /* oiopdwvpo->keywords= cpl_propertylist_load(cpl_frame_get_filename(frame),0); */
   
  return oiopdwvpo;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup oiopd
   @brief Load a mat_oiopd data structure from a cpl table and a propertylist
   @param plist  This CPL propertylist contains the header keywords.
   @param table  This CPL table contains the OI_VIS2 vinary table.
   @return A newly allocated an filled mat_oiopd or NULL on failure.

   This function loads an entire OI_VIS2 binary table into an internal data structure.
   The data structure contains no members for the columns from the OI-FITS version 2 standard.
*/
mat_oiopd *mat_oiopd_from_table(cpl_propertylist *plist, cpl_table *table)
{
  mat_oiopd *oiopd;
  int        nbopd;
  int        nbchannel;
  int        j;

  mat_assert_value((plist != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_propertylist (plist) argument given");
  mat_assert_value((table != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_table (table) argument given");

  nbopd     = cpl_table_get_nrow(table)*6;
  
  // Remove nbchannel but it will come back when water vapor calculation comes in
  // nbchannel = cpl_array_get_size(cpl_table_get_array (table, "VIS2DATA", 0));
  // if (nbchannel == 0)
  //  nbchannel = 1; // no array!
  //cpl_msg_info(cpl_func, "VIS2 table has %d wavelength data", nbchannel);
  nbchannel = 1;
  oiopd     = mat_oiopd_new(nbopd, nbchannel);
  
  if (oiopd == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oiopd");
      return NULL;
    }

  //oiopd->extver    = mat_propertylist_get_int_default(plist, "EXTVER", 1);
  oiopd->arrayname = mat_propertylist_get_string_default(NULL, 0, plist, "ARRNAME", "empty");
  oiopd->dateobs   = mat_propertylist_get_string_default(NULL, 0, plist, "DATE-OBS", "empty");
  oiopd->insname   = mat_propertylist_get_string_default(NULL, 0, plist, "INSNAME", "empty");

  for(j = 0; j < oiopd->nbopd; j+=6) 
    {
      for(int iBase=0;iBase<6;iBase++) {
	mat_opdelem  *el = oiopd->list_opd[j+iBase];
	// copy the values from the table into the opd element
	el->time            = cpl_table_get_double (table, "TIME", j/6, NULL);
	el->dateobsmjd      = cpl_table_get_double (table, "MJD", j/6, NULL);
	el->exptime         = cpl_table_get_double (table, "INT_TIME", j/6, NULL);

	el->stationindex[0] = cpl_array_get_int(cpl_table_get_array(table, "STA_INDEX", j/6), 2*iBase, NULL);
	el->stationindex[1] = cpl_array_get_int(cpl_table_get_array(table, "STA_INDEX", j/6), 2*iBase+1, NULL);
	
	el->opd = cpl_array_get_double(cpl_table_get_array(table, "OPD", j/6), iBase, NULL);
	el->opderr = cpl_array_get_double(cpl_table_get_array(table, "OPDERR", j/6), iBase, NULL);


	if (nbchannel == 1)
	  {
	    el->tempoff = cpl_array_get_double(cpl_table_get_array(table, "TEMPOFF", j/6), iBase, NULL);
	    el->tempofferr = cpl_array_get_double(cpl_table_get_array(table, "TEMPOFFERR", j/6), iBase, NULL);
	    el->humoff = cpl_array_get_double(cpl_table_get_array(table, "HUMOFF", j/6), iBase, NULL);
	    el->humofferr = cpl_array_get_double(cpl_table_get_array(table, "HUMOFFERR", j/6), iBase, NULL);
	  }
	else
	  {
	    //  for (k = 0; k < oiopd->nbchannel; k++) 
	    //  {
	    //  el->wv[k]    = cpl_array_get_double(cpl_table_get_array(table, "WV", j), k, NULL);
	    // el->wverr[k] = cpl_array_get_double(cpl_table_get_array(table, "WVERR", j), k, NULL);
	    // }
	  }
      }
    }
  return oiopd;
}

/**
   @ingroup oiopd
   @brief This method creates a new mat_oiopd structure with all pointers set to NULL.
   @param nbopd     Number of measurements.
   @param nbchannel  Number of spectral channels.
   @returns Pointer to a new allocated mat_oiopd structure on success or NULL on failure.
 
   This method creates a new mat_oiopd structure and sets all values to the default. For
   pointers the value is NULL for primitive data types the value is set to 0. The size
   of the element list (member list_opd of the mat_oiopd data structure) is a multiple
   of CHUNK_SIZE (1024). This allows that after creation of a mat_opd data structure
   new elements (mat_opdelem) can be created an added to that structure.
*/
mat_oiopd *mat_oiopd_new(int nbopd, int nbchannel)
{
  mat_oiopd  *oiopd;
  int         i;

  /* unused instructions. Only to avoid warning during compilation */
  /* nbchannel parameter will be used later with wvpo */
  i=nbchannel;

  oiopd = (mat_oiopd *)cpl_calloc(1, sizeof(mat_oiopd));
  if (oiopd == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_oiopd");
      return NULL;
    }
  oiopd->nbopd = nbopd;
  // oiopd->nbchannel = nbchannel;
  if (nbopd != 0)
    {
      /* ensure that the vector of pointers is a multiple of CHUNK_SIZE */
      size_t size     = ((nbopd + CHUNK_SIZE - 1)/CHUNK_SIZE)*CHUNK_SIZE;
      oiopd->list_opd = (mat_opdelem **)cpl_calloc(size, sizeof(mat_opdelem *));
      if (oiopd->list_opd == NULL)
	{
	  cpl_msg_error(cpl_func,"could not allocate memory for mat_oiopd->list_opd");
	  mat_oiopd_delete(oiopd);
	  return NULL;
	}
      for (i = 0; i < nbopd; i++) 
	{
	  oiopd->list_opd[i] = (mat_opdelem *)cpl_calloc(1, sizeof(mat_opdelem));
	  if (oiopd->list_opd[i] == NULL)
	    {
	      mat_oiopd_delete(oiopd);
	      return NULL;
	    }
	}
    }
  return oiopd;
}


/*----------------------------------------------------------------------------*/
/**
   @ingroup oiopd
   @brief Compute the refractive index following Ciddor's model
   @param lambda  current wavelength.
   @param pressure  ambient pressure (in hPa).
   @param temp      ambient temperature (in K).
   @param humidity  ambient relative humidity.
   @return refractive index.

   This function computes the refractive index following Ciddor's model.
   In fact, this function gives the refractive index with respect to 
   the index at a reference wavelength (3.9micron in L/M band and 
   10.5micron in N band)
*/
double mat_oiopd_ciddor(double lambda, double pressure,
			double temp, double humidity) 
{
  double A=1.2378847E-5;
  double B=-1.9121316E-2;
  double C=33.93711047;
  double Dg=-6.3431645E3;
  double alpha=1.00062;
  double beta=3.14E-8;
  double gamma=5.6E-7;
  double k0=238.0185;
  double k1=5792105.;
  double k2=57.362;
  double k3=167917.;
  double a0=1.58123E-6;
  double a1=-2.9331E-8;
  double a2=1.1043E-10;
  double b0=5.707E-6;
  double b1=-2.051E-8;
  double c0=1.9898E-4;
  double c1=-2.376E-6;
  double d=1.83E-11;
  double e=-0.765E-8;
  double R=8.314510;
  double w0=295.235;
  double w1=2.6422;
  double w2=-0.03238;
  double w3=0.004028;
  double cf=1.022;
  
  // To be fixed to the Paranal value
  double xc=400.;

  double T,P,H;
  double sigma,tm,svp,f,xw,nasm1,naxsm1;
  double Ma,Mw,Pa,Ta,tam,xwa,Za;
  double Pw,Tw,twm,xww,Zw;
  double rhoaxs,rhows,Z,rhoa,rhow,nwsm1;
  double nprop,lambdaref,npropref;
  
  T=temp;
  P=pressure;
  H=humidity;
  
  sigma=1./lambda;
  tm=T-273.15;
  svp=exp(A*T*T+B*T+C+Dg/T);
  f=alpha+beta*P+gamma*tm*tm;
  xw=f*H*svp/P;
  nasm1=1E-8*( (k1/(k0-sigma*sigma)) + (k3/(k2-sigma*sigma)) );
  naxsm1=(nasm1)*(1.+0.534E-6*(xc-450.));

  Ma=1.E-3*(28.9635+12.011E-6*(xc-400.));
  Mw=0.018015;
  Pa=101325.;
  Ta=288.15;
  tam=Ta-273.15;
  xwa=0.;
  Za=1.-(Pa/Ta)*(a0+a1*tam+a2*tam*tam+(b0+b1*tam)*xwa+(c0+c1*tam)*xwa*xwa)+(Pa/Ta)*(Pa/Ta)*(d+e*xwa*xwa);
  Pw=1333.;
  Tw=293.15;
  twm=Tw-273.15;
  xww=1.;
  Zw=1.-(Pw/Tw)*(a0+a1*twm+a2*twm*twm+(b0+b1*twm)*xww+(c0+c1*twm)*xww*xww)+(Pw/Tw)*(Pw/Tw)*(d+e*xww*xww);
  
  rhoaxs=Pa*Ma/(Za*R*Ta);
  
  rhows=Pw*Mw/(Zw*R*Tw);
 
  Z=1.-(P/T)*(a0+a1*tm+a2*tm*tm+(b0+b1*tm)*xw+(c0+c1*tm)*xw*xw)+(P/T)*(P/T)*(d+e*xw*xw);
  
  rhoa=P*Ma*(1.-xw)/(Z*R*T);
  rhow=P*Mw*xw/(Z*R*T);
  
  nwsm1=1E-8*cf*(w0+w1*pow(sigma,2.)+w2*pow(sigma,4.)+w3*pow(sigma,6.));
  
  nprop=(rhoa/rhoaxs)*(naxsm1)+(rhow/rhows)*(nwsm1);

  // refractive index at the reference wavelength
  if (lambda < 6.) {
    lambdaref=3.9;
  } else {
    lambdaref=10.5;
  }
  sigma=1./lambdaref;
  nasm1=1E-8*( (k1/(k0-sigma*sigma)) + (k3/(k2-sigma*sigma)) );
  naxsm1=(nasm1)*(1.+0.534E-6*(xc-450.));
  nwsm1=1E-8*cf*(w0+w1*pow(sigma,2.)+w2*pow(sigma,4.)+w3*pow(sigma,6.));
  npropref=(rhoa/rhoaxs)*(naxsm1)+(rhow/rhows)*(nwsm1);
    
  return nprop-npropref;
}



/*----------------------------------------------------------------------------*/
/**
   @ingroup oiopd
   @brief Compute the refractive index following Mathar's model
   @param lambda  current wavelength.
   @param pressure  ambient pressure (in hPa).
   @param temp      ambient temperature (in K).
   @param humidity  ambient relative humidity.
   @return refractive index.

   This function computes the refractive index following Mathar's model.
   In fact, this function gives the refractive index with respect to 
   the index at a reference wavelength (3.9micron in L/M band and 
   10.5micron in N band)
*/
double mat_oiopd_mathar(double lambda, double pressure,
			double temp, double humidity) 
{
  double Tref=290.65;
  double Pref=75000.0;
  double Href=10.0;

  double nuRefL;
  double nuRefN;
  double cRefL[6]={0.200049E-3,0.145221E-9,0.250951E-12,-0.745834E-15,-0.161432E-17,0.352780E-20};
  double cTL[6]={0.588431E-1,-0.825182E-7,0.137982E-9,0.35242E-13,-0.730651E-15,-0.167911E-18};
  double cTTL[6]={-3.13579,0.694124E-3,-0.500604E-6,-0.116668E-8,0.209644E-11,0.591037E-14};
  double cHL[6]={-0.108142E-7,0.230102E-11,-0.154652E-14,-0.323014E-17,0.630616E-20,0.173880E-22};
  double cHHL[6]={0.586812E-12,0.312198E-16,-0.197792E-19,-0.461945E-22,0.788398E-25,0.24558E-27};
  double cPL[6]={0.2669E-8,0.168162E-14,0.353075E-17,-0.963455E-20,-0.223079E-22,0.453166E-25};
  double cPPL[6]={0.60886E-17,0.46156E-22,0.184282E-24,-0.524471E-27,-0.121299E-29,0.246512E-32};
  double cTHL[6]={0.517962E-4,-0.112149E-7,0.776507E-11,0.172569E-13,-0.320582E-16,-0.899435E-19};
  double cTPL[6]={0.778638E-6,0.446396E-12,0.7846E-15,-0.195151E-17,-0.542083E-20,0.10353E-22};
  double cHPL[6]={-0.217243E-15,0.104747E-20,-0.523689E-23,0.817386E-26,0.309913E-28,-0.363491E-31};

  double cRefN[6]={0.199885E-3,0.344739E-9,-0.273714E-12,0.393383E-15,-0.569488E-17,0.164556E-20};
  double cTN[6]={0.5939E-1,-0.172226E-5,0.237654E-8,-0.381812E-11,0.30505E-14,-0.157464E-16};
  double cTTN[6]={-6.50355,0.10383E-1,-0.139464E-4,-0.220077E-7,-0.272412E-10,0.126364E-12};
  double cHN[6]={-0.221938E-7,0.347377E-10,-0.465991E-13,0.735848E-16,-0.897119E-19,0.380817E-21};
  double cHHN[6]={0.3935242E-12,0.464083E-15,-0.621764E-18,0.981126E-21,-0.121384E-23,0.515111E-26};
  double cPN[6]={0.266809E-8,0.695247E-15,0.159070E-17,-0.303451E-20,-0.661489E-22,0.178226E-24};
  double cPPN[6]={0.610508E-17,0.227694E-22,0.786323E-25,-0.174448E-27,-0.359791E-29,0.978307E-32};
  double cTHN[6]={0.106776E-3,-0.168516E-6,0.226201E-9,-0.356457E-12,0.43798E-15,-0.194545E-17};
  double cTPN[6]={0.778368E-6,0.216404E-12,0.581805E-15,-0.189618E-17,-0.198869E-19,0.589381E-22};
  double cHPN[6]={-0.206365E-15,0.300234E-19,-0.426519E-22,0.684306E-25,-0.46732E-29,0.126117E-30};
  
  double lambdaref=0.;
  double cN[6],cL[6];
  double refractiveIndex=0.;
  double refractiveIndexRef=0.;
  int i=0.;

  
  nuRefL=1.E4/3.4;
  nuRefN=1.E4/10.1;

  if (lambda < 6.) {
    lambdaref=3.5;
    refractiveIndex=0.;
    refractiveIndexRef=0.;
    for(i=0;i<6;i++) {
      cL[i]=cRefL[i]+cTL[i]*((1./temp)-(1./Tref))+cTTL[i]*pow(((1./temp)-(1./Tref)),2.0)+
	cHL[i]*(humidity-Href)+cHHL[i]*pow((humidity-Href),2.0)+
	cPL[i]*(pressure-Pref)+cPPL[i]*pow((pressure-Pref),2.0)+cTHL[i]*((1./temp)-(1./Tref))*(humidity-Href)+
	cTPL[i]*((1./temp)-(1./Tref))*(pressure-Pref)+cHPL[i]*(humidity-Href)*(pressure-Pref);
      refractiveIndex+=cL[i]*pow(((1E4/lambda)-nuRefL),i);
      refractiveIndexRef+=cL[i]*pow(((1E4/lambdaref)-nuRefL),i);
    }
  } else {
    lambdaref=10.5;
    /* lambdaref=3.5; */
    refractiveIndex=0.;
    refractiveIndexRef=0.;
    for(i=0;i<6;i++) {
      cN[i]=cRefN[i]+cTN[i]*((1./temp)-(1./Tref))+cTTN[i]*pow(((1./temp)-(1./Tref)),2.0)+
	cHN[i]*(humidity-Href)+cHHN[i]*pow((humidity-Href),2.0)+
	cPN[i]*(pressure-Pref)+cPPN[i]*pow((pressure-Pref),2.0)+cTHN[i]*((1./temp)-(1./Tref))*(humidity-Href)+
	cTPN[i]*((1./temp)-(1./Tref))*(pressure-Pref)+cHPN[i]*(humidity-Href)*(pressure-Pref);
      refractiveIndex+=cN[i]*pow(((1E4/lambda)-nuRefN),i);
      refractiveIndexRef+=cN[i]*pow(((1E4/lambdaref)-nuRefN),i);
    }
  }
  return refractiveIndex-refractiveIndexRef;
  //return refractiveIndex-1.;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup oiopd
   @brief Compute the derivative of the refractive index wrt humidity 
   @brief following Mathar's model
   @param lambda  current wavelength.
   @param pressure  ambient pressure (in hPa).
   @param temp      ambient temperature (in K).
   @param humidity  ambient relative humidity.
   @return derivative of the refractive index wrt humidity.

   This function computes the derivative of the refractive index wrt humidity
   following Mathar's model.
*/
double mat_oiopd_mathar_der_hum(double lambda, double pressure,
				double temp, double humidity) 
{
  double Tref=290.65;
  double Pref=75000.0;
  double Href=10.0;

  double nuRefL;
  double nuRefN;
  double cHL[6]={-0.108142E-7,0.230102E-11,-0.154652E-14,-0.323014E-17,0.630616E-20,0.173880E-22};
  double cHHL[6]={0.586812E-12,0.312198E-16,-0.197792E-19,-0.461945E-22,0.788398E-25,0.24558E-27};
  double cTHL[6]={0.517962E-4,-0.112149E-7,0.776507E-11,0.172569E-13,-0.320582E-16,-0.899435E-19};
  double cHPL[6]={-0.217243E-15,0.104747E-20,-0.523689E-23,0.817386E-26,0.309913E-28,-0.363491E-31};

  double cHN[6]={-0.221938E-7,0.347377E-10,-0.465991E-13,0.735848E-16,-0.897119E-19,0.380817E-21};
  double cHHN[6]={0.3935242E-12,0.464083E-15,-0.621764E-18,0.981126E-21,-0.121384E-23,0.515111E-26};
  double cTHN[6]={0.106776E-3,-0.168516E-6,0.226201E-9,-0.356457E-12,0.43798E-15,-0.194545E-17};
  double cHPN[6]={-0.206365E-15,0.300234E-19,-0.426519E-22,0.684306E-25,-0.46732E-29,0.126117E-30};
  
  double lambdaref=0.;
  double cN[6],cL[6];
  double refractiveIndex=0.;
  double refractiveIndexRef=0.;
  int i=0.;

  
  nuRefL=1.E4/3.4;
  nuRefN=1.E4/10.1;

  if (lambda < 6.) {
    lambdaref=3.5;
    refractiveIndex=0.;
    refractiveIndexRef=0.;
    for(i=0;i<6;i++) {
      cL[i]=cHL[i]+2.*cHHL[i]*(humidity-Href)+
	cTHL[i]*((1./temp)-(1./Tref))+cHPL[i]*(pressure-Pref);
      refractiveIndex+=cL[i]*pow(((1E4/lambda)-nuRefL),i);
      refractiveIndexRef+=cL[i]*pow(((1E4/lambdaref)-nuRefL),i);
    }
  } else {
    lambdaref=10.5;
    refractiveIndex=0.;
    refractiveIndexRef=0.;
    for(i=0;i<6;i++) {
      cN[i]=cHN[i]+2.*cHHN[i]*(humidity-Href)+
	cTHN[i]*((1./temp)-(1./Tref))+cHPN[i]*(pressure-Pref);
      refractiveIndex+=cN[i]*pow(((1E4/lambda)-nuRefN),i);
      refractiveIndexRef+=cN[i]*pow(((1E4/lambdaref)-nuRefN),i);
    }
  }
  return refractiveIndex-refractiveIndexRef;
}
/*----------------------------------------------------------------------------*/
/**
   @ingroup oiopd
   @brief Compute the derivative of the refractive index wrt temperature 
   @brief following Mathar's model
   @param lambda  current wavelength.
   @param pressure  ambient pressure (in hPa).
   @param temp      ambient temperature (in K).
   @param humidity  ambient relative humidity.
   @return derivative of the refractive index wrt temperature.

   This function computes the derivative of the refractive index wrt temperature
   following Mathar's model.
*/
double mat_oiopd_mathar_der_temp(double lambda, double pressure,
				 double temp, double humidity) 
{
  double Tref=290.65;
  double Pref=75000.0;
  double Href=10.0;

  double nuRefL;
  double nuRefN;
  double cTL[6]	 = {0.588431E-1,-0.825182E-7,0.137982E-9,0.35242E-13,-0.730651E-15,-0.167911E-18};
  double cTTL[6] = {-3.13579,0.694124E-3,-0.500604E-6,-0.116668E-8,0.209644E-11,0.591037E-14};
  double cTHL[6] = {0.517962E-4,-0.112149E-7,0.776507E-11,0.172569E-13,-0.320582E-16,-0.899435E-19};
  double cTPL[6] = {0.778638E-6,0.446396E-12,0.7846E-15,-0.195151E-17,-0.542083E-20,0.10353E-22};

  double cTN[6]	 = {0.5939E-1,-0.172226E-5,0.237654E-8,-0.381812E-11,0.30505E-14,-0.157464E-16};
  double cTTN[6] = {-6.50355,0.10383E-1,-0.139464E-4,-0.220077E-7,-0.272412E-10,0.126364E-12};
  double cTHN[6] = {0.106776E-3,-0.168516E-6,0.226201E-9,-0.356457E-12,0.43798E-15,-0.194545E-17};
  double cTPN[6] = {0.778368E-6,0.216404E-12,0.581805E-15,-0.189618E-17,-0.198869E-19,0.589381E-22};
  
  double lambdaref=0.;
  double cN[6],cL[6];
  double refractiveIndex=0.;
  double refractiveIndexRef=0.;
  int i=0.;

  
  nuRefL=1.E4/3.4;
  nuRefN=1.E4/10.1;

  if (lambda < 6.) {
    lambdaref=3.5;
    refractiveIndex=0.;
    refractiveIndexRef=0.;
    for(i=0;i<6;i++) {
      cL[i]=cTL[i]*(-1./pow(temp,2.0))+2.*cTTL[i]*(-1./pow(temp,2.0))*((1./temp)-(1./Tref))+
	cTHL[i]*(-1./pow(temp,2.0))*(humidity-Href)+cTPL[i]*(-1./pow(temp,2.0))*(pressure-Pref);
      refractiveIndex+=cL[i]*pow(((1E4/lambda)-nuRefL),i);
      refractiveIndexRef+=cL[i]*pow(((1E4/lambdaref)-nuRefL),i);
    }
  } else {
    lambdaref=10.5;
    refractiveIndex=0.;
    refractiveIndexRef=0.;
    for(i=0;i<6;i++) {
      cN[i]=cTN[i]*(-1./pow(temp,2.0))+2.*cTTN[i]*(-1./pow(temp,2.0))*((1./temp)-(1./Tref))+
	cTHN[i]*(-1./pow(temp,2.0))*(humidity-Href)+cTPN[i]*(-1./pow(temp,2.0))*(pressure-Pref);
      refractiveIndex+=cN[i]*pow(((1E4/lambda)-nuRefN),i);
      refractiveIndexRef+=cN[i]*pow(((1E4/lambdaref)-nuRefN),i);
    }
  }
  return refractiveIndex-refractiveIndexRef;
}

double mat_opd_asavalue(asa_objective *asa) {
  cpl_vector *vec = (cpl_vector *)(asa->user);
  double f, lambda, *x, alpha, z;
  int i, n, nUser;
  double modCurrent=0.;
  double modMax=0.;
  double temperature;
  double pressure;
  double humidity;

  nUser	      =	cpl_vector_get_size(vec);
  n	      = (nUser-4)/5;
  pressure    =	cpl_vector_get(vec,nUser-4)*100.;
  temperature =	cpl_vector_get(vec,nUser-3)+273.15;
  humidity    =	cpl_vector_get(vec,nUser-2);
  z	      =	cpl_vector_get(vec,nUser-1);

  x = asa->x;
  f=0.;
  // i=i+4 should replaced by i++ when real data
  for(i=0;i<n;i=i+4) {
    lambda=cpl_vector_get(vec,i);
    alpha=2*CPL_MATH_PI*(x[0]+
			 z*1.E6*x[1]*mat_oiopd_mathar_der_temp(lambda,pressure,temperature,humidity)+
			 z*1.E6*x[2]*mat_oiopd_mathar_der_hum(lambda,pressure,temperature,humidity)
			 )/lambda;
    f+=pow(cpl_vector_get(vec,i+2*n)*cos(alpha)+cpl_vector_get(vec,i+n)*sin(alpha),2.0)/
      (pow(cpl_vector_get(vec,i+4*n),2.)+pow(cpl_vector_get(vec,i+3*n),2.))/n;
    modCurrent=(pow(cpl_vector_get(vec,i+2*n),2.)+pow(cpl_vector_get(vec,i+n),2.));
    if (modCurrent > modMax) {
      modMax=modCurrent;
    }
  }
  f/=modMax;
  return f;
}
double mat_opd_value(double x[], cpl_vector *vec) {
  double f, lambda, alpha, z;
  int i, n, nUser;
  double temperature;
  double pressure;
  double humidity;

  nUser	      =	cpl_vector_get_size(vec);
  n	      = (nUser-4)/5;
  pressure    =	cpl_vector_get(vec,nUser-4)*100.;
  temperature =	cpl_vector_get(vec,nUser-3)+273.15;
  humidity    =	cpl_vector_get(vec,nUser-2);
  z	      =	cpl_vector_get(vec,nUser-1);

  f=0.;
  // i=i+4 should replaced by i++ when real data
  for(i=0;i<n;i++) {
    lambda=cpl_vector_get(vec,i);
    alpha=2*CPL_MATH_PI*(x[0]+
			 z*1.E6*x[1]*mat_oiopd_mathar_der_temp(lambda,pressure,temperature,humidity)+
			 z*1.E6*x[2]*mat_oiopd_mathar_der_hum(lambda,pressure,temperature,humidity)
			 )/lambda;
    f+=pow(cpl_vector_get(vec,i+2*n)*cos(alpha)+cpl_vector_get(vec,i+n)*sin(alpha),2.0)/
      // n/4 should be replaced by n when real data
      (pow(cpl_vector_get(vec,i+4*n),2.)+pow(cpl_vector_get(vec,i+3*n),2.))/(n);
  }
  return f;
}

void mat_opd_cov(double x[],int nParam,cpl_vector *vec, double *g) {
  double	lambda, alpha, z;
  int		i, n, nUser;
  double	temperature;
  double	pressure;
  double	humidity;

  nUser=cpl_vector_get_size(vec);
  n = (nUser-4)/5;
  pressure=cpl_vector_get(vec,nUser-4)*100.;
  temperature=cpl_vector_get(vec,nUser-3)+273.15;
  humidity=cpl_vector_get(vec,nUser-2);
  z=cpl_vector_get(vec,nUser-1);

  for(i=0;i<nParam;i++) {
    g[i]=0.;
  }
  // i=i+4 should replaced by i++ when real data
  for(i=0;i<n;i++) {
    lambda=cpl_vector_get(vec,i);
    alpha=2*CPL_MATH_PI*(x[0]+
			 z*1.E6*x[1]*mat_oiopd_mathar_der_temp(lambda,pressure,temperature,humidity)+
			 z*1.E6*x[2]*mat_oiopd_mathar_der_hum(lambda,pressure,temperature,humidity)
			 )/lambda;
    g[0]+=pow((2.*CPL_MATH_PI/lambda)*
	      (cpl_vector_get(vec,i+n)*cos(alpha)-cpl_vector_get(vec,i+2*n)*sin(alpha)),2.)/
      (pow(cpl_vector_get(vec,i+3*n),2.)+pow(cpl_vector_get(vec,i+4*n),2.));
    g[1]+=pow((2.*CPL_MATH_PI*z*1.E6*mat_oiopd_mathar_der_temp(lambda,pressure,temperature,humidity)/lambda)*
	      (cpl_vector_get(vec,i+n)*cos(alpha)-cpl_vector_get(vec,i+2*n)*sin(alpha)),2.)/
      (pow(cpl_vector_get(vec,i+3*n),2.)+pow(cpl_vector_get(vec,i+4*n),2.));
    g[2]+=pow((2.*CPL_MATH_PI*z*1.E6*mat_oiopd_mathar_der_hum(lambda,pressure,temperature,humidity)/lambda)*
	      (cpl_vector_get(vec,i+n)*cos(alpha)-cpl_vector_get(vec,i+2*n)*sin(alpha)),2.)/
      (pow(cpl_vector_get(vec,i+3*n),2.)+pow(cpl_vector_get(vec,i+4*n),2.));
  }
  for(i=0;i<nParam;i++) {
    g[i]=1./g[i];
  }
  return;
}

void mat_opd_asagrad(asa_objective *asa) {
  cpl_vector *vec = (cpl_vector *)(asa->user);
  double	lambda, *x, alpha, *g, z;
  int		i, n, nUser, nParam;
  double	modCurrent  = 0.;
  double	modMax	    = 0.;
  double	temperature;
  double	pressure;
  double	humidity;

  nUser=cpl_vector_get_size(vec);
  n = (nUser-4)/5;
  pressure=cpl_vector_get(vec,nUser-4)*100.;
  temperature=cpl_vector_get(vec,nUser-3)+273.15;
  humidity=cpl_vector_get(vec,nUser-2);
  z=cpl_vector_get(vec,nUser-1);

  x = asa->x;
  g = asa->g;
  nParam = asa->n;
  for(i=0;i<nParam;i++) {
    g[i]=0.;
  }
  // i=i+4 should replaced by i++ when real data
  for(i=0;i<n;i++) {
    lambda=cpl_vector_get(vec,i);
    alpha=2*CPL_MATH_PI*(x[0]+
			 z*1.E6*x[1]*mat_oiopd_mathar_der_temp(lambda,pressure,temperature,humidity)+
			 z*1.E6*x[2]*mat_oiopd_mathar_der_hum(lambda,pressure,temperature,humidity)
			 )/lambda;
    g[0]+=2.*(2.*CPL_MATH_PI/lambda)*
      (cpl_vector_get(vec,i+n)*cos(alpha)-cpl_vector_get(vec,i+2*n)*sin(alpha))*
      (cpl_vector_get(vec,i+2*n)*cos(alpha)+cpl_vector_get(vec,i+n)*sin(alpha))/
      (pow(cpl_vector_get(vec,i+4*n),2.)+pow(cpl_vector_get(vec,i+3*n),2.))/n;
    g[1]+=2.*(2.*CPL_MATH_PI*z*1.E6*mat_oiopd_mathar_der_temp(lambda,pressure,temperature,humidity)/lambda)*
      (cpl_vector_get(vec,i+n)*cos(alpha)-cpl_vector_get(vec,i+2*n)*sin(alpha))*
      (cpl_vector_get(vec,i+2*n)*cos(alpha)+cpl_vector_get(vec,i+n)*sin(alpha))/
      (pow(cpl_vector_get(vec,i+4*n),2.)+pow(cpl_vector_get(vec,i+3*n),2.))/n;
    g[2]+=2.*(2.*CPL_MATH_PI*z*1.E6*mat_oiopd_mathar_der_hum(lambda,pressure,temperature,humidity)/lambda)*
      (cpl_vector_get(vec,i+n)*cos(alpha)-cpl_vector_get(vec,i+2*n)*sin(alpha))*
      (cpl_vector_get(vec,i+2*n)*cos(alpha)+cpl_vector_get(vec,i+n)*sin(alpha))/
      (pow(cpl_vector_get(vec,i+4*n),2.)+pow(cpl_vector_get(vec,i+3*n),2.))/n;
    modCurrent=(pow(cpl_vector_get(vec,i+2*n),2.)+pow(cpl_vector_get(vec,i+n),2.));
    if (modCurrent > modMax) {
      modMax=modCurrent;
    }
  }
  for(i=0;i<nParam;i++) {
    g[i]/=modMax;
  }
  return ;
}

double mat_opd_asavalgrad(asa_objective *asa) {
  cpl_vector *vec = (cpl_vector *)(asa->user);
  double	f, lambda, *x, alpha, *g, z;
  int		i, n, nUser, nParam;
  double	modCurrent  = 0.;
  double	modMax	    = 0.;
  double	temperature;
  double	pressure;
  double	humidity;

  nUser=cpl_vector_get_size(vec);
  n = (nUser-4)/5;
  pressure=cpl_vector_get(vec,nUser-4)*100.;
  temperature=cpl_vector_get(vec,nUser-3)+273.15;
  humidity=cpl_vector_get(vec,nUser-2);
  z=cpl_vector_get(vec,nUser-1);
  x = asa->x;
  g = asa->g;
  f=0.;
  nParam = asa->n;
  for(i=0;i<nParam;i++) {
    g[i]=0.;
  }
  // i=i+4 should replaced by i++ when real data
  for(i=0;i<n;i++) {
    lambda=cpl_vector_get(vec,i);
    alpha=2*CPL_MATH_PI*(x[0]+
			 z*1.E6*x[1]*mat_oiopd_mathar_der_temp(lambda,pressure,temperature,humidity)+
			 z*1.E6*x[2]*mat_oiopd_mathar_der_hum(lambda,pressure,temperature,humidity)
			 )/lambda;
    g[0]+=2.*(2.*CPL_MATH_PI/lambda)*
      (cpl_vector_get(vec,i+n)*cos(alpha)-cpl_vector_get(vec,i+2*n)*sin(alpha))*
      (cpl_vector_get(vec,i+2*n)*cos(alpha)+cpl_vector_get(vec,i+n)*sin(alpha))/
      (pow(cpl_vector_get(vec,i+4*n),2.)+pow(cpl_vector_get(vec,i+3*n),2.))/n;
    g[1]+=2.*(2.*CPL_MATH_PI*z*1.E6*mat_oiopd_mathar_der_temp(lambda,pressure,temperature,humidity)/lambda)*
      (cpl_vector_get(vec,i+n)*cos(alpha)-cpl_vector_get(vec,i+2*n)*sin(alpha))*
      (cpl_vector_get(vec,i+2*n)*cos(alpha)+cpl_vector_get(vec,i+n)*sin(alpha))/
      (pow(cpl_vector_get(vec,i+4*n),2.)+pow(cpl_vector_get(vec,i+3*n),2.))/n;
    g[2]+=2.*(2.*CPL_MATH_PI*z*1.E6*mat_oiopd_mathar_der_hum(lambda,pressure,temperature,humidity)/lambda)*
      (cpl_vector_get(vec,i+n)*cos(alpha)-cpl_vector_get(vec,i+2*n)*sin(alpha))*
      (cpl_vector_get(vec,i+2*n)*cos(alpha)+cpl_vector_get(vec,i+n)*sin(alpha))/
      (pow(cpl_vector_get(vec,i+4*n),2.)+pow(cpl_vector_get(vec,i+3*n),2.))/n;
    f+=pow(cpl_vector_get(vec,i+2*n)*cos(alpha)+cpl_vector_get(vec,i+n)*sin(alpha),2.0)/
      (pow(cpl_vector_get(vec,i+3*n),2.)+pow(cpl_vector_get(vec,i+4*n),2.))/n;
    modCurrent=(pow(cpl_vector_get(vec,i+2*n),2.)+pow(cpl_vector_get(vec,i+n),2.));
    if (modCurrent > modMax) {
      modMax=modCurrent;
    }
  }
  f/=modMax;
  for(i=0;i<nParam;i++) {
    g[i]/=modMax;
  }
  return f;
}
