/* $Id: mat_photbeams.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_photbeams.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "mat_photbeams.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/



/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/**
  @ingroup photbeams
  @brief Load PHOTBEAMS data in mat_photbeams structure
  @param frame           current frame
  @return mat_photbeams
 */
/*-----------------------------------------------------------------------------*/
mat_photbeams * mat_photbeams_load(cpl_frame *frame)
{
  mat_photbeams *phot=NULL;
  int i=0;
  int j=0;
  int m=0;
  int l=0;
  int index=0;
  int nbExtent=0;
  int flagDet=0;
  int flagPhot=0;
  int dimx=0;
  int dimy=0;
  int cpt=0;
  char str[50];
  char *keyExtname=NULL;
  cpl_type type;
  cpl_array *arr=NULL;

  cpl_propertylist *plist=NULL;
  cpl_table *table=NULL;


 //test if frame is empty
  if (frame == NULL){
    cpl_msg_info(cpl_func,"frame empty");
    return NULL;
  }

  // Allocate the structure and initialize its attributes
  phot = cpl_calloc(1, sizeof(mat_photbeams));
  if (phot == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for phot");
    return NULL;
  }
  phot->keywords= cpl_propertylist_load(cpl_frame_get_filename(frame),0);
  phot->imgdet=NULL;
  phot->list_phot=NULL;

  // Load IMAGING_DETECTOR
  nbExtent=cpl_frame_get_nextensions(frame);
  for(i=0;i<nbExtent;i++) {
    plist=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
    keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");

    if (keyExtname != NULL) {
      table=cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
      // Load data in attribute imgdet
      if (!strcmp(keyExtname,"IMAGING_DETECTOR")) {
	cpl_msg_info(cpl_func, "loading the IMAGING_DETECTOR extension");
	phot->imgdet = mat_imagingdetector_from_table(plist, table);
        /* Error handling */
        if (phot->imgdet == NULL) { 
          cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
			                 "Could not load IMAGING_DETECTOR in frame: %s",
			                 cpl_frame_get_filename(frame));
        }
        else {
	  flagDet=1;
        }
      }
      cpl_table_delete(table);
    }
    cpl_propertylist_delete(plist);
  }

  if (flagDet == 1) {

    for(i=0;i<nbExtent;i++) {
      plist=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
      keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
      
      if (keyExtname != NULL) {
	table=cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
	if (!strcmp(keyExtname,"PHOT_BEAMS")) {
	  cpl_msg_info(cpl_func, "loading the PHOT_BEAMS extension");
	  phot->nbphotxframe=cpl_propertylist_get_long(plist,"NAXIS2")*
	    phot->imgdet->nbregion;
	  phot->list_phot=cpl_malloc(phot->nbphotxframe*sizeof(mat_imgreg *));
	  if (phot->list_phot==NULL) {
	    cpl_msg_error(cpl_func,"could not allocate memory for phot->list_phot");
	    return NULL;
	  }
	  cpt=0;
	  for(i=0;i<phot->imgdet->nbregion;i++) {
	    for(j=0;j<phot->nbphotxframe/phot->imgdet->nbregion;j++) {
	      phot->list_phot[cpt]=cpl_malloc(sizeof(mat_imgreg));
	      if (phot->list_phot[cpt]==NULL) {
		cpl_msg_error(cpl_func,"could not allocate memory for phot->list_phot[i]");
		return NULL;
	      }
	      phot->list_phot[cpt]->numregion=phot->imgdet->list_region[i]->numregion;
	      phot->list_phot[cpt]->numdetector=phot->imgdet->list_region[i]->numdetector;
	      phot->list_phot[cpt]->nbimgreg=1;
	      phot->list_phot[cpt]->imgreg=cpl_calloc(phot->list_phot[cpt]->nbimgreg,sizeof(cpl_image *));

	      /* snprintf(str,50,"DATATOTAL%d",phot->list_phot[cpt]->numregion); */
	      /* dimx=phot->imgdet->list_region[i]->naxis[0]; */
	      /* dimy=phot->imgdet->list_region[i]->naxis[1]; */
	      /* arr=(cpl_array *)cpl_table_get_array(table,str,j); */
	      /* type=cpl_array_get_type(arr); */
	      /* phot->list_phot[cpt]->imgreg[0]=cpl_image_new(dimx,dimy,type); */
	      
	      /* index=0; */
	      /* for(m=0;m<dimy;m++) { */
	      /* 	for(l=0;l<dimx;l++) { */
	      /* 	  cpl_image_set(phot->list_phot[cpt]->imgreg[0],l+1,m+1, */
	      /* 			cpl_array_get_float(arr,index,NULL)); */
	      /* 	  index++; */
	      /* 	} */
	      /* } */
	      /* //cpl_array_delete(arr); */
	      
	      snprintf(str,50,"DATANOSKY%d",phot->list_phot[cpt]->numregion);
	      dimx=phot->imgdet->list_region[i]->naxis[0];
	      dimy=phot->imgdet->list_region[i]->naxis[1];
	      arr=(cpl_array *)cpl_table_get_array(table,str,j);
	      type=cpl_array_get_type(arr);
	      phot->list_phot[cpt]->imgreg[0]=cpl_image_new(dimx,dimy,type);
	      
	      index=0;
	      for(m=0;m<dimy;m++) {
		for(l=0;l<dimx;l++) {
		  cpl_image_set(phot->list_phot[cpt]->imgreg[0],l+1,m+1,
				cpl_array_get_float(arr,index,NULL));
		  index++;
		}
	      }



	      // cpl_array_delete(arr);
	      
	      cpt++;
	    }
	  }
	  /* Error handling */
	  if (phot->list_phot == NULL) { 
	    cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
				  "Could not load PHOT_BEAMS in frame: %s",
				  cpl_frame_get_filename(frame));
	  } else {
	    flagPhot=1;
	  }

	}
	cpl_table_delete(table);
      }
      cpl_propertylist_delete(plist);
    }
  }
  if ( flagDet==0 || flagPhot==0) {
    cpl_error_set_message(cpl_func,CPL_ERROR_ILLEGAL_INPUT,
                          "Extension missing in frame: %s",
                          cpl_frame_get_filename(frame));
  }
  
  return phot;
}



/*----------------------------------------------------------------------------*/
/**
  @ingroup photbeams
  @brief delete structure mat_photbeams
  @param  phot          current structure
  @return error code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_photbeams_free(mat_photbeams *phot) {
  int i=0;

  // Check input parameters
  mat_assert_value((phot!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,"no mat_photbeams (phot) argument given");
  
  if (phot->keywords != NULL) cpl_propertylist_delete(phot->keywords);
  if (phot->imgdet != NULL) mat_imagingdetector_delete(phot->imgdet);

  for(i=0; i<phot->nbphotxframe; i++) {
    if (phot->list_phot[i] != NULL) {
      if (phot->list_phot[i]->imgreg != NULL) {
	cpl_image_delete(phot->list_phot[i]->imgreg[0]);
	/* cpl_image_delete(phot->list_phot[i]->imgreg[1]); */
	cpl_free(phot->list_phot[i]->imgreg);
      }
      cpl_free(phot->list_phot[i]);
    }
  }
  if (phot->list_phot != NULL) cpl_free(phot->list_phot);
  cpl_free(phot);

  return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup photbeams
  @brief Save a mat_photbeams structure in a PHOT_BEAMS FITS file
  @param set_in            input frameset
  @param set_sel           selected frameset
  @param parlist           recipe parameter list
  @param recipe_name       name of recipe
  @param pro_cat           PRO.CATG
  @param pro_tech          PRO.TECH
  @param pro_sci           PRO.SCIENCE
  @param phot             mat_photbeams structure to save
  @param pheader           header of the binary table
  @param filename          name of the FITS file
  @return cpl_error_code
 */
/*-----------------------------------------------------------------------------*/

cpl_error_code mat_photbeams_save(
				  mat_photbeams *phot,
				  cpl_propertylist *pheader, 
				  cpl_frameset *set_in,
				  cpl_frameset *set_sel,
				  const cpl_parameterlist *parlist,
				  const char *recipe_name,
				  const char *pro_cat,
				  const char *pro_tech,
				  const char *pro_sci,
				  char *filename) {
  
  int i=0;
  int j=0;
  int cpt=0;
//  int nbSubWin=0;
  int k=0;
  float *map=NULL;
  char str[50];
  /* char *insMode=NULL; */
  char *sliderPhot=NULL;
  char *detName=NULL;
  char mode[3];
  cpl_table *phot_map=NULL;
  cpl_propertylist *applist=NULL;
  cpl_propertylist *plist=NULL;
  cpl_array *map_array=NULL;
  cpl_array *arrDim=NULL;
  cpl_errorstate prestate= cpl_errorstate_get();
  (void) pro_sci;
  // Check input parameters
  mat_assert_value((phot!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_photbeams (phot) argument given");
  mat_assert_value((pheader!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, 
		   "no cpl_propertylist (pheader) argument given");
  mat_assert_value((set_in!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, 
		   "no cpl_frameset (set_in) argument given");
  mat_assert_value((set_sel!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, 
		   "no cpl_frameset (set_sel) argument given");
  mat_assert_value((filename!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, 
		   "no filename given");

  /* insMode=(char *)cpl_propertylist_get_string(phot->keywords, */
  /* 						    "ESO INS MODE"); */
  /* if (insMode == NULL) { */
  /*   cpl_msg_error(cpl_func,"could not read ESO INS MODE keyword"); */
  /*   return CPL_ERROR_NULL_INPUT; */
  /* } */
  detName=(char *)cpl_propertylist_get_string(phot->keywords,
  					      "ESO DET CHIP NAME");
  if (detName == NULL) {
    cpl_msg_error(cpl_func,"could not read ESO DET CHIP NAME keyword");
    return CPL_ERROR_NULL_INPUT;
  }

  if(!strcmp(detName, "AQUARIUS")) {
    sliderPhot=
      (char *)cpl_propertylist_get_string(phot->keywords, 
					  "ESO INS PIN NAME");	
  }
  else {
    sliderPhot=
      (char *)cpl_propertylist_get_string(phot->keywords, 
					  "ESO INS PIL NAME");	
  }
  if (!strcmp(sliderPhot, "PHOTO")) {
    sprintf(mode,"s");
  } else {
    sprintf(mode,"h");
  }
  /* if(!strcmp(detName, "AQUARIUS")) { */
  /*   snprintf(mode,3,"%s",insMode+8); */
  /* } */
  /* else { */
  /*   snprintf(mode,3,"%s",insMode+4); */
  /* } */
  
  cpl_propertylist_erase(pheader,"XTENSION");
  cpl_propertylist_erase(pheader,"BITPIX");
  cpl_propertylist_erase_regexp(pheader,"NAXIS",0);
  cpl_propertylist_erase_regexp(pheader,"COUNT",0);
  cpl_propertylist_erase(pheader,"TFIELDS");
  cpl_propertylist_erase_regexp(pheader,"SIM",0);
  cpl_propertylist_erase_regexp(pheader,"FORM",0);
  cpl_propertylist_erase_regexp(pheader,"TTYPE",0);
  cpl_propertylist_erase_regexp(pheader,"TDIM",0);
  cpl_propertylist_erase_regexp(pheader,"ESO PRO",0);
  // Add the keywords from the PRO categories
  if (pro_tech != NULL) {
    cpl_propertylist_append_string(phot->keywords, CPL_DFS_PRO_TECH, pro_tech);
  }
  if (pro_cat != NULL) {
    cpl_propertylist_append_string(phot->keywords, CPL_DFS_PRO_CATG, pro_cat);
  }
  if (pro_sci != NULL) {
    if (strstr(pro_sci,"F") != NULL)
      {
	cpl_propertylist_append_bool(phot->keywords, CPL_DFS_PRO_SCIENCE, 0);
      }
    else
      {
	cpl_propertylist_append_bool(phot->keywords, CPL_DFS_PRO_SCIENCE, 1);
      }
  }

  // Prepare the keywords of the PHOT_BEAMS binary table
  plist=cpl_propertylist_new();
  cpl_propertylist_append_string(plist,"EXTNAME","PHOT_BEAMS");
  //cpl_propertylist_append_string(plist,"ORIGIN",phot->imgdet->origin);
  cpl_propertylist_append_string(plist,"INSTRUME",phot->imgdet->instrument);
  cpl_propertylist_append_double(plist,"MJD-OBS",phot->imgdet->dateobsmjd);
  cpl_propertylist_append_string(plist,"DATE-OBS",phot->imgdet->dateobs);
  //cpl_propertylist_append_string(plist,"DATE",phot->imgdet->date);
  cpl_propertylist_append_string(plist,"ESO DET DID",
  				 phot->imgdet->dcsdictionaryid);
  cpl_propertylist_append_string(plist,"ESO DET ID",phot->imgdet->dcsid);
  cpl_propertylist_append_int(plist,"NREGION",phot->imgdet->nbregion);
  cpl_propertylist_append_int(plist,"MAXTEL",phot->imgdet->nbtel);


  // Fill a cpl_table with photbeams data that will replace the IMAGING_DATA
  // Add columns
  if (strstr(mode,"s") == NULL) {
    phot_map = cpl_table_new(2);
    // HighSens Case
    for(k=0;k<4;k++) {
      for(j=0;j<phot->imgdet->nbregion/4;j++) {
  	/* snprintf(str,50,"DATATOTAL%d",j*4+k+1); */
  	/* cpl_table_new_column_array(phot_map, str, CPL_TYPE_FLOAT, */
  	/* 			   cpl_image_get_size_x( */
	/* 			       phot->list_phot[0]->imgreg[0]) * */
  	/* 			   cpl_image_get_size_y( */
	/* 			       phot->list_phot[0]->imgreg[0])); */
  	/* arrDim=cpl_array_new(2,CPL_TYPE_INT); */
  	/* cpl_array_set(arrDim,0,cpl_image_get_size_x( */
	/* 		  phot->list_phot[0]->imgreg[0])); */
  	/* cpl_array_set(arrDim,1,cpl_image_get_size_y( */
	/* 		  phot->list_phot[0]->imgreg[0])); */
  	/* cpl_table_set_column_dimensions(phot_map,str,arrDim); */
  	/* cpl_array_delete(arrDim); */
  	snprintf(str,50,"DATANOSKY%d",j*4+k+1);
  	cpl_table_new_column_array(phot_map, str, CPL_TYPE_FLOAT,
  				   cpl_image_get_size_x(
				       phot->list_phot[0]->imgreg[0]) *
  				   cpl_image_get_size_y(
				       phot->list_phot[0]->imgreg[0]));
  	arrDim=cpl_array_new(2,CPL_TYPE_INT);
  	cpl_array_set(arrDim,0,cpl_image_get_size_x(
			  phot->list_phot[0]->imgreg[0]));
  	cpl_array_set(arrDim,1,cpl_image_get_size_y(
			  phot->list_phot[0]->imgreg[0]));
  	cpl_table_set_column_dimensions(phot_map,str,arrDim);
  	cpl_array_delete(arrDim);
      }
    }
  } else {
    /* nbSubWin=phot->imgdet->nbregion/5; */
    /* phot_map = cpl_table_new(phot->nbphotxframe/(phot->imgdet->nbregion-nbSubWin)); */
    phot_map = cpl_table_new(phot->nbphotxframe/phot->imgdet->nbregion);
    // SiPhot Case
    for(j=0;j<phot->imgdet->nbregion;j++) {
      if ( phot->imgdet->list_region[j]->correlation == 1 ) {
  	/* snprintf(str,50,"DATATOTAL%d",j+1); */
  	/* cpl_table_new_column_array(phot_map, str, CPL_TYPE_FLOAT, */
  	/* 			   cpl_image_get_size_x( */
	/* 			       phot->list_phot[0]->imgreg[0]) * */
  	/* 			   cpl_image_get_size_y( */
	/* 			       phot->list_phot[0]->imgreg[0])); */
  	/* arrDim=cpl_array_new(2,CPL_TYPE_INT); */
  	/* cpl_array_set(arrDim,0,cpl_image_get_size_x( */
	/* 		  phot->list_phot[0]->imgreg[0])); */
  	/* cpl_array_set(arrDim,1,cpl_image_get_size_y( */
	/* 		  phot->list_phot[0]->imgreg[0])); */
  	/* cpl_table_set_column_dimensions(phot_map,str,arrDim); */
  	/* cpl_array_delete(arrDim); */


  	snprintf(str,50,"DATANOSKY%d",j+1);
  	cpl_table_new_column_array(phot_map, str, CPL_TYPE_FLOAT,
  				   cpl_image_get_size_x(
				       phot->list_phot[0]->imgreg[0]) *
  				   cpl_image_get_size_y(
				       phot->list_phot[0]->imgreg[0]));
  	arrDim=cpl_array_new(2,CPL_TYPE_INT);
  	cpl_array_set(arrDim,0,cpl_image_get_size_x(
			  phot->list_phot[0]->imgreg[0]));
  	cpl_array_set(arrDim,1,cpl_image_get_size_y(
			  phot->list_phot[0]->imgreg[0]));
  	cpl_table_set_column_dimensions(phot_map,str,arrDim);
  	cpl_array_delete(arrDim);
      }
    }
  }
  // Get the data in a cpl_array
  cpt=0;
  if (strstr(mode,"s") == NULL ) {
    // HighSens Case
    for(k=0;k<4;k++) {
      for(j=0;j<phot->imgdet->nbregion/4;j++) {
    	/* map = (float *)cpl_image_get_data(phot->list_phot[j*4+k]->imgreg[0]); */
    	/* map_array = cpl_array_wrap_float */
    	/*   (map,cpl_image_get_size_x(phot->list_phot[0]->imgreg[0]) * */
    	/*    cpl_image_get_size_y(phot->list_phot[0]->imgreg[0])); */
    	/* snprintf(str,50,"DATATOTAL%d",j*4+k+1); */
    	/* cpl_table_set_array(phot_map,str,0,map_array); */
    	/* cpl_array_unwrap(map_array); */
	
   	map = (float *)cpl_image_get_data(phot->list_phot[j*4+k]->imgreg[0]);
    	map_array = cpl_array_wrap_float
    	  (map,cpl_image_get_size_x(phot->list_phot[0]->imgreg[0]) *
    	   cpl_image_get_size_y(phot->list_phot[0]->imgreg[0]));
    	snprintf(str,50,"DATANOSKY%d",j*4+k+1);
    	cpl_table_set_array(phot_map,str,0,map_array);
    	cpl_array_unwrap(map_array);

    	/* map = (float *)cpl_image_get_data(phot->list_phot[(phot->nbphotxframe/2) */
	/* 						  +j*4+k]->imgreg[0]); */
    	/* map_array = cpl_array_wrap_float */
    	/*   (map,cpl_image_get_size_x(phot->list_phot[0]->imgreg[0]) * */
    	/*    cpl_image_get_size_y(phot->list_phot[0]->imgreg[0])); */
    	/* snprintf(str,50,"DATATOTAL%d",j*4+k+1); */
    	/* cpl_table_set_array(phot_map,str,1,map_array); */
    	/* cpl_array_unwrap(map_array); */
	
    	map = (float *)cpl_image_get_data(phot->list_phot[(phot->nbphotxframe/2)
							  +j*4+k]->imgreg[0]);
    	map_array = cpl_array_wrap_float
    	  (map,cpl_image_get_size_x(phot->list_phot[0]->imgreg[0]) *
    	   cpl_image_get_size_y(phot->list_phot[0]->imgreg[0]));
    	snprintf(str,50,"DATANOSKY%d",j*4+k+1);
    	cpl_table_set_array(phot_map,str,1,map_array);
    	cpl_array_unwrap(map_array);
      }
    }

  } else {
    // SiPhot Case
    /* for(i=0; i<phot->nbphotxframe/ */
    /* 	  (phot->imgdet->nbregion-nbSubWin); i++) { */
    for(i=0; i<phot->nbphotxframe/phot->imgdet->nbregion; i++) {
      for(j=0;j<phot->imgdet->nbregion;j++) {
	if (cpt%100==0) {
	  printf("%2d percent  done\n",(int)((100.*cpt)/phot->nbphotxframe));
	}
  	if ( phot->imgdet->list_region[j]->correlation == 1 ) {
  	  /* map = (float *)cpl_image_get_data(phot->list_phot[cpt]->imgreg[0]); */
  	  /* map_array = cpl_array_wrap_float */
  	  /*   (map,cpl_image_get_size_x(phot->list_phot[0]->imgreg[0]) * */
  	  /*    cpl_image_get_size_y(phot->list_phot[0]->imgreg[0])); */
  	  /* snprintf(str,50,"DATATOTAL%d",j+1); */
  	  /* cpl_table_set_array(phot_map,str,i,map_array); */
  	  /* cpl_array_unwrap(map_array); */

  	  map = (float *)cpl_image_get_data(phot->list_phot[cpt]->imgreg[0]);
  	  map_array = cpl_array_wrap_float
  	    (map,cpl_image_get_size_x(phot->list_phot[0]->imgreg[0]) *
  	     cpl_image_get_size_y(phot->list_phot[0]->imgreg[0]));
  	  snprintf(str,50,"DATANOSKY%d",j+1);
  	  cpl_table_set_array(phot_map,str,i,map_array);
  	  cpl_array_unwrap(map_array);
  	  cpt++;
  	}
      }
    }
  }

  // Make a new FITS file with kappa keywords and the created table
  cpl_propertylist_erase(phot->keywords,"RADECSYS");
  applist=cpl_propertylist_duplicate(phot->keywords);
  cpl_dfs_save_table(set_in, pheader, parlist, set_sel, NULL,
  		     phot_map, plist, recipe_name, applist, NULL,
		     PACKAGE "/" PACKAGE_VERSION, filename);
  /* // Append the IMAGING_DETECTOR table */
  /* mat_imagingdetector_save(phot->imgdet,phot->keywords,filename); */

  cpl_table_delete(phot_map);
  cpl_propertylist_delete(applist);
  cpl_propertylist_delete(plist);


 
  if (!cpl_errorstate_is_equal(prestate)) { 
    
    cpl_errorstate_set(prestate);
    
    return CPL_ERROR_UNSPECIFIED;
  }
  return CPL_ERROR_NONE;

}



/*----------------------------------------------------------------------------*/
/**
  @brief Add the phot_beams at the end of existing FITS file
  @param phot              mat_photbeams structure to save
  @param filename          name of the FITS file
  @return cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mat_photbeams_append(mat_photbeams *phot,char *filename) {
  /* int status=0; */
  int i=0;
  int j=0;
  int cpt=0;
  char *sliderPhot=NULL;
  char *detName=NULL;
  char mode[3];
  float *map=NULL;
  cpl_array *map_array=NULL;
  cpl_array *arrDim=NULL;
  char str[50];
  cpl_table *phot_map=NULL;
  cpl_propertylist *plist=NULL;
  /* cpl_errorstate prestate= cpl_errorstate_get(); */
  
  // Check input parameters
  mat_assert_value((phot!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_photbeams (phot) argument given");
  mat_assert_value((filename!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "No filename given");

  detName=(char *)cpl_propertylist_get_string(phot->keywords, 
					      "ESO DET CHIP NAME");
  if (detName == NULL) {
    cpl_msg_error(cpl_func,"could not read ESO DET CHIP NAME keyword");
    return CPL_ERROR_NULL_INPUT;
  }

  if(!strcmp(detName, "AQUARIUS")) {
    sliderPhot=
      (char *)cpl_propertylist_get_string(phot->keywords, 
					  "ESO INS PIN NAME");	
  }
  else {
    sliderPhot=
      (char *)cpl_propertylist_get_string(phot->keywords, 
					  "ESO INS PIL NAME");	
  }
  if (!strcmp(sliderPhot, "PHOTO")) {
    sprintf(mode,"s");
  } else {
    sprintf(mode,"h");
  }


  if (strcmp(mode,"s")) {
    // HighSens Case
    cpl_msg_info(cpl_func,"mat_photbeams_append not used in HighSes mode");
  } else {
    // SiPhot Case
    // Prepare the keywords of the IMAGING_DATA binary table
    plist=cpl_propertylist_new();  
    cpl_propertylist_append_string(plist,"EXTNAME ","PHOT_BEAMS");

    phot_map = cpl_table_new(phot->nbphotxframe/phot->imgdet->nbregion);
    // SiPhot Case
    for(j=0;j<phot->imgdet->nbregion;j++) {
      if ( phot->imgdet->list_region[j]->correlation == 1 ) {
  	snprintf(str,50,"DATATOTAL%d",j+1);
  	cpl_table_new_column_array(phot_map, str, CPL_TYPE_FLOAT,
  				   cpl_image_get_size_x(
							phot->list_phot[0]->imgreg[0]) *
  				   cpl_image_get_size_y(
							phot->list_phot[0]->imgreg[0]));
  	arrDim=cpl_array_new(2,CPL_TYPE_INT);
  	cpl_array_set(arrDim,0,cpl_image_get_size_x(
						    phot->list_phot[0]->imgreg[0]));
  	cpl_array_set(arrDim,1,cpl_image_get_size_y(
						    phot->list_phot[0]->imgreg[0]));
  	cpl_table_set_column_dimensions(phot_map,str,arrDim);
  	cpl_array_delete(arrDim);


  	snprintf(str,50,"DATANOSKY%d",j+1);
  	cpl_table_new_column_array(phot_map, str, CPL_TYPE_FLOAT,
  				   cpl_image_get_size_x(
							phot->list_phot[0]->imgreg[1]) *
  				   cpl_image_get_size_y(
							phot->list_phot[0]->imgreg[1]));
  	arrDim=cpl_array_new(2,CPL_TYPE_INT);
  	cpl_array_set(arrDim,0,cpl_image_get_size_x(
						    phot->list_phot[0]->imgreg[0]));
  	cpl_array_set(arrDim,1,cpl_image_get_size_y(
						    phot->list_phot[0]->imgreg[0]));
  	cpl_table_set_column_dimensions(phot_map,str,arrDim);
  	cpl_array_delete(arrDim);
      }
    }

    cpt=0;

    for(i=0; i<phot->nbphotxframe/phot->imgdet->nbregion; i++) {
      for(j=0;j<phot->imgdet->nbregion;j++) {
  	if ( phot->imgdet->list_region[j]->correlation == 1 ) {
  	  map = (float *)cpl_image_get_data(phot->list_phot[cpt]->imgreg[0]);
  	  map_array = cpl_array_wrap_float
  	    (map,cpl_image_get_size_x(phot->list_phot[0]->imgreg[0]) *
  	     cpl_image_get_size_y(phot->list_phot[0]->imgreg[0]));
  	  snprintf(str,50,"DATATOTAL%d",j+1);
  	  cpl_table_set_array(phot_map,str,i,map_array);
  	  cpl_array_unwrap(map_array);

  	  map = (float *)cpl_image_get_data(phot->list_phot[cpt]->imgreg[1]);
  	  map_array = cpl_array_wrap_float
  	    (map,cpl_image_get_size_x(phot->list_phot[0]->imgreg[1]) *
  	     cpl_image_get_size_y(phot->list_phot[0]->imgreg[1]));
  	  snprintf(str,50,"DATANOSKY%d",j+1);
  	  cpl_table_set_array(phot_map,str,i,map_array);
  	  cpl_array_unwrap(map_array);
  	  cpt++;
  	}
      }
    }
    cpl_table_save(phot_map,NULL,plist,filename,CPL_IO_APPEND);
    if (phot_map != NULL) cpl_table_delete(phot_map);
    if (plist != NULL) cpl_propertylist_delete(plist);
  }

  /* if (!cpl_errorstate_is_equal(prestate)) {  */
    
  /*   cpl_errorstate_set(prestate); */
    
  /*   return CPL_ERROR_UNSPECIFIED; */
  /* } */

  return CPL_ERROR_NONE;

}


/*----------------------------------------------------------------------------*/
/**
  @ingroup photbeams
  @brief duplicate structure mat_photbeams
  @param  phot          current structure
  @return duplicated mat_photbeams stucture
 */
/*-----------------------------------------------------------------------------*/
mat_photbeams *mat_photbeams_duplicate(mat_photbeams *phot)
{
  mat_photbeams *photdup = NULL;

  photdup = (mat_photbeams *)cpl_calloc(1, sizeof(mat_photbeams));
  if (photdup == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_photbeams");
      return NULL;
    }
  photdup->keywords=cpl_propertylist_duplicate(phot->keywords);
  photdup->nbphotxframe=phot->nbphotxframe;
  photdup->list_phot=cpl_malloc(photdup->nbphotxframe*sizeof(mat_imgreg *));
  if (photdup->list_phot==NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for photdup->list_phot");
      mat_photbeams_free(photdup);
      return NULL;
    }
  for (int i=0;i<photdup->nbphotxframe;i++)
    {
      photdup->list_phot[i]=mat_imgreg_duplicate(phot->list_phot[i],
						 cpl_image_get_type(phot->list_phot[0]->imgreg[0]));
    }
  photdup->imgdet=mat_imagingdetector_duplicate(phot->imgdet);
  return photdup;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup photbeams
  @brief transfomr structure mat_photbeams wrt BCD 
  @param  phot          current structure in OUT-OUT
  @param bcd1status     BCD1 position
  @param bcd2status     BCD1 position
  @return transformed mat_photbeams stucture
 */
/*-----------------------------------------------------------------------------*/
mat_photbeams *mat_photbeams_transform_bcd(mat_photbeams *phot,
					   const char *bcd1status,
					   const char *bcd2status)
{
  mat_photbeams *photdup = NULL;
  int cpt = 0;
  photdup = (mat_photbeams *)cpl_calloc(1, sizeof(mat_photbeams));
  if (photdup == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_photbeams");
      return NULL;
    }
  photdup->keywords=cpl_propertylist_duplicate(phot->keywords);
  cpl_propertylist_set_string(photdup->keywords,"ESO INS BCD1 ID",bcd1status);
  cpl_propertylist_set_string(photdup->keywords,"ESO INS BCD2 ID",bcd2status);
  photdup->nbphotxframe=phot->nbphotxframe;
  photdup->list_phot=cpl_malloc(photdup->nbphotxframe*sizeof(mat_imgreg *));
  if (photdup->list_phot==NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for photdup->list_phot");
      mat_photbeams_free(photdup);
      return NULL;
    }
  cpt=0;
  for (int iReg=0;iReg<phot->imgdet->nbregion;iReg++)
    {
      for (int iFrame=0;iFrame<photdup->nbphotxframe/phot->imgdet->nbregion;iFrame++)
	{
	  if ( strcmp(bcd1status, "OUT") == 0 && strcmp(bcd2status, "OUT") )
	    {
	      photdup->list_phot[cpt]=mat_imgreg_duplicate(phot->list_phot[cpt],
							   cpl_image_get_type(phot->list_phot[0]->imgreg[0]));
	    }
	  if ( strcmp(bcd1status, "IN") == 0 && strcmp(bcd2status, "OUT") )
	    {
	      if (iReg == 0)
		{
		  photdup->list_phot[cpt]=mat_imgreg_duplicate(phot->list_phot[photdup->nbphotxframe/phot->imgdet->nbregion+iFrame],
							       cpl_image_get_type(phot->list_phot[0]->imgreg[0]));
		}
	      else if (iReg == 1)
		{
		  photdup->list_phot[cpt]=mat_imgreg_duplicate(phot->list_phot[iFrame],
							       cpl_image_get_type(phot->list_phot[0]->imgreg[0]));
		}
	      else
		{
		  photdup->list_phot[cpt]=mat_imgreg_duplicate(phot->list_phot[cpt],
							       cpl_image_get_type(phot->list_phot[0]->imgreg[0]));
		}
	    }
	      
	  if ( strcmp(bcd1status, "OUT") == 0 && strcmp(bcd2status, "IN") )
	    {
	      if (iReg == 2)
		{
		  photdup->list_phot[cpt]=mat_imgreg_duplicate(phot->list_phot[3*photdup->nbphotxframe/phot->imgdet->nbregion+iFrame],
							       cpl_image_get_type(phot->list_phot[0]->imgreg[0]));
		}
	      else if (iReg == 3)
		{
		  photdup->list_phot[cpt]=mat_imgreg_duplicate(phot->list_phot[2*photdup->nbphotxframe/phot->imgdet->nbregion+iFrame],
							       cpl_image_get_type(phot->list_phot[0]->imgreg[0]));
		}
	      else
		{
		  photdup->list_phot[cpt]=mat_imgreg_duplicate(phot->list_phot[cpt],
							       cpl_image_get_type(phot->list_phot[0]->imgreg[0]));
		}
	    }
	  if ( strcmp(bcd1status, "IN") == 0 && strcmp(bcd2status, "IN") )
	    {
	      if (iReg == 0)
		{
		  photdup->list_phot[cpt]=mat_imgreg_duplicate(phot->list_phot[photdup->nbphotxframe/phot->imgdet->nbregion+iFrame],
							       cpl_image_get_type(phot->list_phot[0]->imgreg[0]));
		}
	      else if (iReg == 1)
		{
		  photdup->list_phot[cpt]=mat_imgreg_duplicate(phot->list_phot[iFrame],
							       cpl_image_get_type(phot->list_phot[0]->imgreg[0]));
		}
	      else if (iReg == 2)
		{
		  photdup->list_phot[cpt]=mat_imgreg_duplicate(phot->list_phot[3*photdup->nbphotxframe/phot->imgdet->nbregion+iFrame],
							       cpl_image_get_type(phot->list_phot[0]->imgreg[0]));
		}
	      else if (iReg == 3)
		{
		  photdup->list_phot[cpt]=mat_imgreg_duplicate(phot->list_phot[2*photdup->nbphotxframe/phot->imgdet->nbregion+iFrame],
							       cpl_image_get_type(phot->list_phot[0]->imgreg[0]));
		}
	    }
	  cpt++;
	}
    }
  photdup->imgdet=mat_imagingdetector_duplicate(phot->imgdet);
  return photdup;
  
}
