/* $Id: mat_proc_incoher_lib.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_proc_incoher_lib.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#include <string.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/

#include "mat_proc_incoher_lib.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
  Define
  ----------------------------------------------------------------------------*/
#define NB_BCD 4

/*-----------------------------------------------------------------------------
  Functions prototypes
  ----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
   @brief Apply the incoherent processing (mat_proc_incoher)
   @param frameset      contains the frames
   @param parlist      contains the recipes parameters
   @param recname      Name of the recipes calling this function
   @return 0 if no error 
*/
int mat_proc_incoher_lib(cpl_frameset *frameset,
			 cpl_parameterlist *parlist,
			 const char *recname)
{
  
  cpl_frameset *framesetsorted  = NULL;
  cpl_parameter *correlatedFlux = NULL;
    
  cpl_frame *cur_frame=NULL;
  cpl_frame *sel_frame=NULL;
  cpl_frame *dup_frame=NULL;
  char *pszFileName=NULL;
  char *pszFileTag=NULL;
  char *detName=NULL;
  char *sliderPhot=NULL;
  char mode[3]=" ";
  char modeFirst[3]=" ";
  /* char *bcdStatus=NULL; */
  /* char *bcd1Status=NULL; */
  /* char *bcd2Status=NULL; */
  char *bcdMode=NULL;
  /* char bcd1StatusFirst[100]=""; */
  /* char bcd2StatusFirst[100]=""; */
  cpl_propertylist *plist=NULL;
  int i=0;
  int k=0;
  int ix=0;
  int cpt=0;
  int nbCorrFlux=0;
  int keyNexpCorr=0;
  cpl_vector *listExpnoCorr=NULL;
  int nbPhot=0;
  int keyNexpPhot=0;
  cpl_vector *listExpnoPhot=NULL;
  int ixPhot=0;
  int ixCorrFlux=0;
  int ixOiVis=0;
  int ixOiOpd=0;
  mat_photbeams **phot=NULL;
  mat_photbeams *photOutOut=NULL;
  mat_corrflux **corrFlux=NULL;
  mat_oifits **oivis=NULL;
  mat_oiopdwvpo **oiopd=NULL;
  mat_corrflux *nofringe=NULL;
  // cpl_matrix *dispCoef=NULL;
  mat_oifits **oifits=NULL;
  cpl_frameset *usedframes=NULL;
  char *filename=NULL;
  cpl_frame *frameCorrFlux=NULL;
  cpl_frame *framePhot=NULL;
  cpl_frameset_iterator *it=NULL;
  cpl_errorstate prestate = cpl_errorstate_get();
  int flagProcess=1;
  int indexCorrFlux=0;
  int nexpReal=0;
  int indexFileVis2=1;
  int indexFileT3=1;
  int indexFileSpec=1;
  int flagPhotOutOut=0;
  int flagnofringe=0;
  cpl_propertylist *header=NULL;
  char *protech=NULL;
  int expno_corr=0;
  int expno=0;
  int ivis=0;
  int iopd=0;
  
  //	 Check input parameters
  mat_assert_value((frameset != NULL),CPL_ERROR_NULL_INPUT,
		   CPL_ERROR_NULL_INPUT, "no frameset argument given");
  // 0. Retrieve parameter
  correlatedFlux=cpl_parameterlist_find(parlist,
					"matisse.mat_proc_incoher.corrFlux");

  cur_frame = cpl_frameset_find(frameset,"PHOT_BEAMS");
  if (cur_frame == NULL) {
    cpl_parameter_set_bool(correlatedFlux,1);
  }

  
  // Sort the data versus BCD
  for(int ibcd=0;ibcd<NB_BCD;ibcd++)
    {
      flagProcess = 1;
      flagnofringe = 0;

      framesetsorted=cpl_frameset_new();
		
      it = cpl_frameset_iterator_new(frameset);
      do {
	cur_frame = cpl_frameset_iterator_get(it);
	if (cur_frame != NULL)
	  {
	    pszFileName = (char *)cpl_frame_get_filename( cur_frame );
	    pszFileTag  = (char *)cpl_frame_get_tag( cur_frame );
	    plist=cpl_propertylist_load(pszFileName,0);
	    if (strcmp(pszFileTag,"PHOT_BEAMS") == 0 || strcmp(pszFileTag,"OBJ_CORR_FLUX") == 0 ||
		strcmp(pszFileTag,"RAW_DPHASE") == 0 || strcmp(pszFileTag,"OI_OPDWVPO") == 0)
	      {
		if ( cpl_propertylist_has(plist,"ESO INS BCD1 ID") && cpl_propertylist_has(plist,"ESO INS BCD2 ID") )
		  {
		    bcdMode = (char *) cpl_propertylist_get_string(plist,"ESO CFG BCD MODE");
		    if (strcmp(bcdMode, "OUT-OUT") == 0 && ibcd==0)
		      {
			dup_frame=cpl_frame_duplicate(cur_frame);
			cpl_frameset_insert(framesetsorted,dup_frame);
		      }	
		    if (strcmp(bcdMode, "IN-IN") == 0 && ibcd==1)
		      {
			dup_frame=cpl_frame_duplicate(cur_frame);
			cpl_frameset_insert(framesetsorted,dup_frame);
		      }
		    if (strcmp(bcdMode, "OUT-IN") == 0 && ibcd==2)
		      {
			dup_frame=cpl_frame_duplicate(cur_frame);
			cpl_frameset_insert(framesetsorted,dup_frame);
		      }
		    if (strcmp(bcdMode, "IN-OUT") == 0 && ibcd==3)
		      {
			dup_frame=cpl_frame_duplicate(cur_frame);
			cpl_frameset_insert(framesetsorted,dup_frame);
		      }
		    header = cpl_propertylist_load(pszFileName,0);
		    protech=(char *)cpl_propertylist_get_string(header,"ESO PRO TECH");
		    if (protech == NULL) {
		      cpl_msg_error(cpl_func,"No PRO TYPE keyword");
		      return -1;
		    }
		    if (flagnofringe == 0 && strstr(protech,"NOFRINGE") != NULL)
		      {
			flagnofringe=1;
		      }
		    cpl_propertylist_delete(header);		    
		  }
	      }
	    // this is used in case the value of NEXP keyword is wrong
	    ix=mat_get_expno(plist);
	    if (ix > nexpReal)
	      {
		nexpReal=ix;
	      }
		
	    cpl_propertylist_delete(plist);
	  }
      } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
      cpl_frameset_iterator_delete(it);

      cpl_msg_info(cpl_func,"NEXPreal=%d",nexpReal);
	  
      if ( (cpl_frameset_count_tags(framesetsorted,"OBJ_CORR_FLUX") == 0 && flagnofringe == 0) ||
	   (cpl_frameset_count_tags(framesetsorted,"OBJ_CORR_FLUX") == 1 && flagnofringe == 1) )
	{
	  if (ibcd==0)
	    {
	      cpl_msg_warning(cpl_func,"No OBJ_CORR_FLUX frame to process for BCD OUT-OUT");
	      flagProcess=0;
	    }
	  if (ibcd==1)
	    {
	      cpl_msg_warning(cpl_func,"No OBJ_CORR_FLUX frame to process for BCD IN-IN");
	      flagProcess=0;
	    }
	  if (ibcd==2)
	    {
	      cpl_msg_warning(cpl_func,"No OBJ_CORR_FLUX frame to process for BCD OUT-IN");
	      flagProcess=0;
	    }
	  if (ibcd==3)
	    {
	      cpl_msg_warning(cpl_func,"No OBJ_CORR_FLUX frame to process for BCD IN-OUT");
	      flagProcess=0;
	    }
	}
      else
	{
          if ( !cpl_parameter_get_bool(correlatedFlux) && cpl_frameset_count_tags(framesetsorted,"PHOT_BEAMS") == 0)
	    {
	      if (ibcd == 0)
		{
		  flagPhotOutOut = 1;
		  cpl_msg_warning(cpl_func,"No PHOT_BEAMS frame to process for BCD OUT-OUT");
		  flagProcess=0;
		}
	      else if (ibcd==1)
		{
		  if (flagPhotOutOut == 1)
		    {
		      cpl_msg_warning(cpl_func,"No PHOT_BEAMS frame to process for BCD IN-IN");
		      flagProcess=0;
		    }
		}
	      else if (ibcd==2)
		{
		  if (flagPhotOutOut == 1)
		    {
		      cpl_msg_warning(cpl_func,"No PHOT_BEAMS frame to process for BCD OUT-IN");
		      flagProcess=0;
		    }
		}
	      else if (ibcd==3)
		{
		  if (flagPhotOutOut == 1)
		    {
		      cpl_msg_warning(cpl_func,"No PHOT_BEAMS frame to process for BCD IN-OUT");
		      flagProcess=0;
		    }
		}
	    }
	}
		
      if (flagProcess==1)
	{
	  // 1. Check the data type of the input frames
	  cpt=0;
	  prestate=cpl_errorstate_get();
	  it= cpl_frameset_iterator_new(framesetsorted);
	  do {
	    cur_frame = cpl_frameset_iterator_get(it);
	    if (cur_frame != NULL)
	      {
		pszFileName = (char*)cpl_frame_get_filename( cur_frame );
		pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
		plist=cpl_propertylist_load(pszFileName,0);
		protech=(char *)cpl_propertylist_get_string(plist,"ESO PRO TECH");
		if (protech == NULL) {
		  cpl_msg_error(cpl_func,"No PRO TYPE keyword");
		  return -1;
		}
		if (( strcmp(pszFileTag,"PHOT_BEAMS") == 0 || strcmp(pszFileTag,"OBJ_CORR_FLUX") == 0 ) &&
		    strstr(protech,"NOFRINGE") == NULL )
		  {
		    detName=(char *)cpl_propertylist_get_string(plist,"ESO DET CHIP NAME");
		    if (detName == NULL)
		      {
			cpl_msg_error(cpl_func,"no ESO DET CHIP NAME keyword in frame");
			return -1;
		      }
		    if(!strcmp(detName, "AQUARIUS"))
		      {
			sliderPhot=(char *)cpl_propertylist_get_string(plist, "ESO INS PIN NAME");	
		      }
		    else
		      {
			sliderPhot=(char *)cpl_propertylist_get_string(plist,"ESO INS PIL NAME");	
		      }
		    if (!strcmp(sliderPhot, "PHOTO"))
		      {
			sprintf(mode,"s");
		      }
		    else
		      {
			sprintf(mode,"h");
		      }
		    if (cpt == 0 )
		      {
			strncpy(modeFirst,mode,3);
		      } 

		    if (strcmp(pszFileTag,"PHOT_BEAMS") == 0)
		      {
			if (nbPhot == 0)
			  {
			    keyNexpPhot=nexpReal;
			    if (keyNexpPhot > 0)
			      {
				listExpnoPhot=cpl_vector_new(keyNexpPhot);
				for(ix=0;ix<keyNexpPhot;ix++)
				  {
				    cpl_vector_set(listExpnoPhot,ix,0);
				  }
			      }
			    else
			      {
				cpl_msg_error(cpl_func,"no ESO TPL NEXP keyword in frame");
				return -1;
			      }
			  }
			if (strstr(modeFirst,"s") != NULL)
			  {
			    ix=mat_get_expno(plist);
			    if (ix > 0)
			      {
				cpl_vector_set(listExpnoPhot,ix-1,1);
			      }
			  }
			else
			  {
			    cpl_vector_set(listExpnoPhot,nbPhot,1);
			  }
			nbPhot++;
		      }
		    if (strcmp(pszFileTag,"OBJ_CORR_FLUX") == 0)
		      {
			if (nbCorrFlux == 0)
			  {
			    keyNexpCorr=nexpReal;
			    if (keyNexpCorr > 0)
			      {
				listExpnoCorr=cpl_vector_new(keyNexpCorr);
				for(ix=0;ix<keyNexpCorr;ix++)
				  {
				    cpl_vector_set(listExpnoCorr,ix,0);
				  }
			      }
			    else
			      {
				cpl_msg_error(cpl_func,"no ESO TPL NEXP keyword in frame");
				cpl_vector_delete(listExpnoPhot);
				return -1;
			      }
			  }
			if (strstr(modeFirst,"s") != NULL)
			  {
			    ix=mat_get_expno(plist);
			    if (ix > 0)
			      {
				cpl_vector_set(listExpnoCorr,ix-1,1);
			      }
			  }
			else
			  {
			    cpl_vector_set(listExpnoCorr,nbCorrFlux,1);
			  }
			nbCorrFlux++;
		      }
		    cpt++;
		  }
		// Get next frame from SOF
		cpl_propertylist_delete(plist);
	      }
	  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
	  cpl_frameset_iterator_delete(it);
	  if (!cpl_errorstate_is_equal(prestate))
	    {
	      cpl_errorstate_set(prestate);
	    }

	  
	  if ( !cpl_parameter_get_bool(correlatedFlux) )
	    {
	      if (keyNexpPhot > 0 && strstr(modeFirst,"s") != NULL &&
		  nbPhot != keyNexpPhot)
		{
		  cpl_msg_warning(cpl_func,
				  "The number of PHOT_BEAMS does not correspond to the expected number of exposures (ESO TPL NEXP)");
		  nbPhot=keyNexpPhot;
		}
	      if ( nbPhot == 0 && flagPhotOutOut == 0 )
		{
		  phot=cpl_calloc(1,sizeof(mat_photbeams *));
		}
	      else
		{
		  phot=cpl_calloc(nbPhot,sizeof(mat_photbeams *));
		}
	      if (phot == NULL)
		{
		  cpl_msg_error(cpl_func,"could not allocate memory for phot");
		  return -1;
		}
	    }
	  if (keyNexpCorr > 0 && strstr(modeFirst,"s") != NULL && nbCorrFlux != keyNexpCorr)
	    {
	      cpl_msg_warning(cpl_func,"The number of OBJ_CORR_FLUX does not correspond to the expected number of exposures (ESO TPL NEXP)");
	      nbCorrFlux=keyNexpCorr;
	    }

	  corrFlux=cpl_calloc(nbCorrFlux,sizeof(mat_corrflux *));
	  oivis=cpl_calloc(nbCorrFlux,sizeof(mat_oivis *));
	  oiopd=cpl_calloc(nbCorrFlux,sizeof(mat_oiopdwvpo *));
	  if (corrFlux == NULL)
	    {
	      cpl_msg_error(cpl_func,"could not allocate memory for corrFlux");
	      cpl_free(phot);
	      return -1;
	    }
	  cpl_msg_info(cpl_func,"NBCORRFLUX= %d",nbCorrFlux);
	  // 2.Load PHOT_BEAMS and OBJ_CORR_FLUX
	  //  Initialization of the mat_oifits structure
	  oifits     = cpl_calloc(nbCorrFlux,sizeof(mat_oifits *));
	  ixPhot     = 0;
	  ixCorrFlux = 0;
	  ixOiVis = 0;
	  ixOiOpd = 0;
	  cpl_msg_info(cpl_func,"Loading PHOT_BEAMS and OBJ_CORR_FLUX");
	  
	  prestate=cpl_errorstate_get();
	  it= cpl_frameset_iterator_new(framesetsorted);
	  do {
	    cur_frame = cpl_frameset_iterator_get(it);
	    if (cur_frame != NULL)
	      {
		pszFileName = (char*)cpl_frame_get_filename( cur_frame );
		pszFileTag  = (char *)cpl_frame_get_tag( cur_frame );
		plist=cpl_propertylist_load(pszFileName,0);
		protech=(char *)cpl_propertylist_get_string(plist,"ESO PRO TECH");
		if (protech == NULL) {
		  cpl_msg_error(cpl_func,"No PRO TYPE keyword");
		  return -1;
		}
		if ( !cpl_parameter_get_bool(correlatedFlux) )
		  {
		    if (strcmp(pszFileTag,"PHOT_BEAMS") == 0)
		      {
			cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
							
			if (strstr(modeFirst,"s") == NULL)
			  {
			    phot[ixPhot] = mat_photbeams_load(cur_frame);
			    // Duplicate 1st photometry in OUT-OUT
			    // could be used for other BCD configuration in case of
			    // lack of photometry in the other BCD configurations
			    if (ibcd == 0 && photOutOut == NULL)
			      {
				photOutOut=mat_photbeams_duplicate(phot[ixPhot]);
			      }
			    ixPhot++;
			  }
		      }
		  }
		if ( strcmp(pszFileTag,"OBJ_CORR_FLUX") == 0 && strstr(protech,"NOFRINGE") == NULL )
		  {
		    cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
		    if (strstr(modeFirst,"s") != NULL)
		      {
			ixCorrFlux=mat_get_expno(plist)-1;
		      }
						
		    if (strstr(modeFirst,"s") == NULL)
		      {
			corrFlux[ixCorrFlux] = mat_corrflux_load(cur_frame);
			oifits[ixCorrFlux]=cpl_malloc(sizeof(mat_oifits));
			mat_oifits_init_from_corrflux(corrFlux[ixCorrFlux],oifits[ixCorrFlux]);
			ixCorrFlux++;
		      }
		  }
		
		if ( strcmp(pszFileTag,"OBJ_CORR_FLUX") == 0 && strstr(protech,"NOFRINGE") != NULL )
		  {
		    cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
		    nofringe = mat_corrflux_load(cur_frame);
		  }

		// If N band (or HighSens) and RMNREC available then load OIOPDWVPO and RAW_DPHASE
		if (strstr(modeFirst,"s") == NULL)
		  {
		    if ( strcmp(pszFileTag,"RAW_DPHASE") == 0 )
		      {
			if (ixOiVis == 0 )
			  {
			    cpl_msg_info(cpl_func,"Loading RAW_DPHASE files");
			  }
			oivis[ixOiVis] = mat_oifits_load(cur_frame);
			ixOiVis++;
		      }
		    if ( strcmp(pszFileTag,"OI_OPDWVPO") == 0 )
		      {
			if (ixOiOpd == 0 )
			  {
			    cpl_msg_info(cpl_func,"Loading OI_OPDWVPO files");
			  }
			oiopd[ixOiOpd] = mat_oiopdwvpo_load(cur_frame);
			ixOiOpd++;
		      }
		  }
		
		cpl_propertylist_delete(plist);
	      }
	  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
	  cpl_frameset_iterator_delete(it);
	  if (!cpl_errorstate_is_equal(prestate))
	    {
	      cpl_errorstate_set(prestate);
	    }


	  
	  // In HighSens mode, if photometry missing in a BCD configuration
	  // Use photOutOut if availableD
	  if ( !cpl_parameter_get_bool(correlatedFlux) )
	    {
	      if (strstr(modeFirst,"s") == NULL && ibcd > 0 && nbPhot == 0)
		{
		  cpl_msg_warning(cpl_func," No photometry in the current BCD configuration");
		  cpl_msg_warning(cpl_func," Use photometry in BCD configuration OUT-OUT if available");		      
		  if (photOutOut != NULL)
		    {
		      nbPhot=1;
		      if (ibcd == 1)
			{
			  phot[0]=mat_photbeams_transform_bcd(photOutOut,"IN","IN");
			}
		      else if (ibcd == 2)
			{
			  phot[0]=mat_photbeams_transform_bcd(photOutOut,"OUT","IN");
			}
		      else if (ibcd == 3)
			{
			  phot[0]=mat_photbeams_transform_bcd(photOutOut,"IN","OUT");
			}
		    }
		  else
		    {
		      cpl_msg_error(cpl_func,"No photometry in BCD configuration OUT-OUT");
		      for(int ic=0;ic<nbCorrFlux;ic++)
			{
			  if (oifits[ic] != NULL)
			    {
			      mat_oifits_delete(oifits[ic]);
			    }
			}
		      cpl_free(oifits);
		      return -1;
		    }
		}
	    }
	  // Checking the nummber of PHOTBEAMS frames with respect to the mode
	  // only if corrFlux=FALSE
	  if ( !cpl_parameter_get_bool(correlatedFlux) )
	    {
	      if (strstr(modeFirst,"s") == NULL)
		{
		  cpl_msg_info(cpl_func,"HighSens mode");
		  // In the case of HighSens only 1 PHOTBEAMS is necessary
		  // If several PHOTBEAMS, only the firts PHOTBEAMS will be used
		  switch (nbPhot)
		    {
		    case 0:
		      cpl_msg_error(cpl_func,"No PHOTBEAMS frame found in the sof file");
		      if (nofringe != NULL)
			{
			  mat_corrflux_free(nofringe);
			}
		      cpl_free(corrFlux);
		      cpl_free(oifits);
		      cpl_free(phot);
		      return CPL_ERROR_INCOMPATIBLE_INPUT;
		      break;
		    case 1:
		      cpl_msg_info(cpl_func,"1 PHOTBEAMS frame found in the sof file");
		      break;
		    default:
		      cpl_msg_warning(cpl_func,"%d PHOTBEAMS frame found in the sof file",nbPhot);
		      cpl_msg_warning(cpl_func,"Only the first PHOTBEAMS frame will be considered in the following");
		    }
		}
	      else
		{
		  cpl_msg_info(cpl_func,"SiPhot mode");
		  if (nbPhot != nbCorrFlux)
		    {
		      cpl_msg_error(cpl_func,"The number of PHOT_BEAMS files should be equal to the number of OBJ_CORR_FLUX files");
		      cpl_free(corrFlux);
		      if (nofringe != NULL)
			{
			  mat_corrflux_free(nofringe);
			}
		      cpl_free(oifits);
		      return CPL_ERROR_INCOMPATIBLE_INPUT;
		    }
		  else
		    {
		      for(ix=0;ix<nbCorrFlux;ix++)
			{
			  if (cpl_vector_get(listExpnoCorr,ix) != cpl_vector_get(listExpnoPhot,ix))
			    {
			      cpl_msg_error(cpl_func,"The exposures do not correspond between PHOT_BEAMS and OBJ_CORR_FLUX");
			      cpl_free(corrFlux);
			      if (nofringe != NULL)
				{
				  mat_corrflux_free(nofringe);
				}
			      cpl_free(oifits);
			      return CPL_ERROR_INCOMPATIBLE_INPUT;
			    }
			}
		    }
		}
	    }

	  // Start estimation of interferometric observables
	  for(i=0;i<nbCorrFlux;i++)
	    {
	      if (cpl_vector_get(listExpnoCorr,i) == 1)
		{
		  k=0;
		  indexCorrFlux=0;
		  prestate=cpl_errorstate_get();
		  it= cpl_frameset_iterator_new(framesetsorted);
		  do {
		    cur_frame = cpl_frameset_iterator_get(it);
		    if (cur_frame != NULL)
		      {
			pszFileName = (char *)cpl_frame_get_filename( cur_frame );
			pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
			plist=cpl_propertylist_load(pszFileName,0);
			protech=(char *)cpl_propertylist_get_string(plist,"ESO PRO TECH");
			if (protech == NULL) {
			  cpl_msg_error(cpl_func,"No PRO TECH keyword");
			  return -1;
			}
			if ( !cpl_parameter_get_bool(correlatedFlux) )
			  {
			    if (strcmp(pszFileTag,"PHOT_BEAMS") == 0)
			      {
								
				//Load the data for SIPHOT mode
				if (strstr(modeFirst,"s") != NULL)
				  {
				    ixPhot=mat_get_expno(plist)-1;
				    if (ixPhot == i)
				      {
					cpl_msg_info(cpl_func,"Loading PHOT_BEAMS file %s",pszFileName);
					phot[i] = mat_photbeams_load(cur_frame);
				      }
				  }								
									
				if (strstr(modeFirst,"s") == NULL)
				  {
				    if (k==0)
				      {
					framePhot=cpl_frame_new();
					cpl_frame_set_filename(framePhot,cpl_frame_get_filename(cur_frame));
					cpl_frame_set_group(framePhot,cpl_frame_get_group(cur_frame));
					cpl_frame_set_level(framePhot,cpl_frame_get_level(cur_frame));
					cpl_frame_set_tag(framePhot,pszFileTag);
					cpl_frame_set_type(framePhot,cpl_frame_get_type(cur_frame));
				      }
				  }
				else
				  {
				    if (i+1 == mat_get_expno(plist))
				      {
					framePhot=cpl_frame_new();
					cpl_frame_set_filename(framePhot,cpl_frame_get_filename(cur_frame));
					cpl_frame_set_group(framePhot,cpl_frame_get_group(cur_frame));
					cpl_frame_set_level(framePhot,cpl_frame_get_level(cur_frame));
					cpl_frame_set_tag(framePhot,pszFileTag);
					cpl_frame_set_type(framePhot,cpl_frame_get_type(cur_frame));
				      }
				  }
				k++;
			      }
			  }
			if ( strcmp(pszFileTag,"OBJ_CORR_FLUX") == 0 && strstr(protech,"NOFRINGE") == NULL )
			  {
								
			    // Load the data for SIPHOT and HIGHSENS modes 
			    // Init oifits
			    if (strstr(modeFirst,"s") != NULL)
			      {
				ixCorrFlux=mat_get_expno(plist)-1;
				if (ixCorrFlux == i)
				  {
				    cpl_msg_info(cpl_func,"Loading OBJ_CORR_FLUX file %s",pszFileName);
				    corrFlux[i] = mat_corrflux_load(cur_frame);
				    oifits[i]=cpl_malloc(sizeof(mat_oifits));
				    mat_oifits_init_from_corrflux(corrFlux[i],oifits[i]);
				  }
			      }
								
			    if (strstr(modeFirst,"s") == NULL)
			      {
				if (indexCorrFlux == i)
				  {
				    frameCorrFlux=cpl_frame_new();
				    cpl_frame_set_filename(frameCorrFlux,cpl_frame_get_filename(cur_frame));
				    cpl_frame_set_group(frameCorrFlux,cpl_frame_get_group(cur_frame));
				    cpl_frame_set_level(frameCorrFlux,cpl_frame_get_level(cur_frame));
				    cpl_frame_set_tag(frameCorrFlux,pszFileTag);
				    cpl_frame_set_type(frameCorrFlux,cpl_frame_get_type(cur_frame));
				  }
			      }
			    else
			      {
				if (i+1 == mat_get_expno(plist))
				  {
				    frameCorrFlux=cpl_frame_new();
				    cpl_frame_set_filename(frameCorrFlux,cpl_frame_get_filename(cur_frame));
				    cpl_frame_set_group(frameCorrFlux,cpl_frame_get_group(cur_frame));
				    cpl_frame_set_level(frameCorrFlux,cpl_frame_get_level(cur_frame));
				    cpl_frame_set_tag(frameCorrFlux,pszFileTag);
				    cpl_frame_set_type(frameCorrFlux,cpl_frame_get_type(cur_frame));
				  }
			      }
			    indexCorrFlux++;
			  }
			cpl_propertylist_delete(plist);
		      }
		  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
		  cpl_frameset_iterator_delete(it);
		  if (!cpl_errorstate_is_equal(prestate))
		    {
		      cpl_errorstate_set(prestate);
		    }
		  //3. Squared Visibility estimation
		  if (strstr(modeFirst,"s") == NULL)
		    {
		      if ( !cpl_parameter_get_bool(correlatedFlux) )
			{
			  mat_compute_vis2(corrFlux[i],phot[0],oifits[i],nofringe);
			}
		      else
			{
			  mat_compute_vis2(corrFlux[i],NULL,oifits[i],nofringe);
			}
		    }
		  else
		    {
		      if ( !cpl_parameter_get_bool(correlatedFlux) )
			{
			  mat_compute_vis2(corrFlux[i],phot[i],oifits[i],nofringe);
			}
		      else
			{
			  mat_compute_vis2(corrFlux[i],NULL,oifits[i],nofringe);
			}
		    }
		  //4. Closure Phase estimation
		  if (oifits[i]->oit3 != NULL)
		    {
		      ivis=-1;
		      iopd=-1;
		      if (ixOiVis == nbCorrFlux && ixOiOpd == nbCorrFlux)
			{
			  expno_corr = cpl_propertylist_get_int(corrFlux[i]->keywords,"ESO TPL EXPNO");
			  for(int ii=0;ii<nbCorrFlux;ii++)
			    {
			      expno = cpl_propertylist_get_int(oivis[i]->keywords,"ESO TPL EXPNO");
			      if (expno == expno_corr)
				{
				  ivis=ii;
				}
			      expno = cpl_propertylist_get_int(oiopd[i]->keywords,"ESO TPL EXPNO");
			      if (expno == expno_corr)
				{
				  iopd=ii;
				}
			    }
			  if (ivis == -1 || iopd == -1)
			    {
			      mat_compute_cphase(corrFlux[i],oifits[i],nofringe,NULL,NULL);
			    }
			  else
			    {
			      mat_compute_cphase(corrFlux[i],oifits[i],nofringe,oivis[ivis],oiopd[iopd]);
			    }
			}
		      else
			{
			  mat_compute_cphase(corrFlux[i],oifits[i],nofringe,NULL,NULL);
			}
		    }

		  //5. Spectrum estimation
		  if ( !cpl_parameter_get_bool(correlatedFlux) )
		    {
		      if (strstr(modeFirst,"s") == NULL)
			{
			  mat_compute_spectrum(phot[0],oifits[i]);
			}
		      else
			{
			  mat_compute_spectrum(phot[i],oifits[i]);
			}
		    }

		  //6. Save oifits
		  if (cpl_frameset_find(frameset,"CALIB_RAW") != NULL ||
		      cpl_frameset_find(frameset,"TARGET_RAW") != NULL ||
		      cpl_frameset_find(frameset,"CALIB_SRC_RAW") != NULL)
		    {
		      usedframes=cpl_frameset_new();
		      it= cpl_frameset_iterator_new(frameset);
		      do
			{
			  cur_frame = cpl_frameset_iterator_get(it);
			  if (cur_frame != NULL)
			    {
			      pszFileName = (char*)cpl_frame_get_filename( cur_frame );
			      pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
			      if (strcmp(pszFileTag, "CALIB_RAW") == 0 || strcmp(pszFileTag, "TARGET_RAW") == 0 || strcmp(pszFileTag, "CALIB_SRC_RAW") == 0)
				{
				  plist=cpl_propertylist_load(pszFileName,0);
				  if (strcmp(cpl_propertylist_get_string(oifits[i]->keywords,"ESO CFG BCD MODE"),"OUT-OUT") == 0)
				    {
				      if (strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD1 NAME"),"OUT") == 0 &&
					  strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD2 NAME"),"OUT") == 0 &&
					  cpl_propertylist_get_int(plist,"ESO TPL EXPNO") == cpl_propertylist_get_int(oifits[i]->keywords,"ESO TPL EXPNO"))
					{
					  sel_frame=cpl_frame_duplicate(cur_frame);
					  cpl_frameset_insert(usedframes,sel_frame);
					}
				    }
				  else if(strcmp(cpl_propertylist_get_string(oifits[i]->keywords,"ESO CFG BCD MODE"),"IN-OUT") == 0)
				    {
				      if (strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD1 NAME"),"IN") == 0 &&
					  strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD2 NAME"),"OUT") == 0 &&
					  cpl_propertylist_get_int(plist,"ESO TPL EXPNO") == cpl_propertylist_get_int(oifits[i]->keywords,"ESO TPL EXPNO"))
					{
					  sel_frame=cpl_frame_duplicate(cur_frame);
					  cpl_frameset_insert(usedframes,sel_frame);
					}
				    }
				  else if(strcmp(cpl_propertylist_get_string(oifits[i]->keywords,"ESO CFG BCD MODE"),"OUT-IN") == 0)
				    {
				      if (strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD1 NAME"),"OUT") == 0 &&
					  strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD2 NAME"),"IN") == 0 &&
					  cpl_propertylist_get_int(plist,"ESO TPL EXPNO") == cpl_propertylist_get_int(oifits[i]->keywords,"ESO TPL EXPNO"))
					{
					  sel_frame=cpl_frame_duplicate(cur_frame);
					  cpl_frameset_insert(usedframes,sel_frame);
					}
				    }
				  else if(strcmp(cpl_propertylist_get_string(oifits[i]->keywords,"ESO CFG BCD MODE"),"IN-IN") == 0)
				    {
				      if (strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD1 NAME"),"IN") == 0 &&
					  strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD2 NAME"),"IN") == 0 &&
					  cpl_propertylist_get_int(plist,"ESO TPL EXPNO") == cpl_propertylist_get_int(oifits[i]->keywords,"ESO TPL EXPNO"))
					{
					  sel_frame=cpl_frame_duplicate(cur_frame);
					  cpl_frameset_insert(usedframes,sel_frame);
					}
				    }
				  cpl_propertylist_delete(plist);
				}
			    }
			} while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
		      cpl_frameset_iterator_delete(it);
		      it= cpl_frameset_iterator_new(frameset);
		      do
			{
			  cur_frame = cpl_frameset_iterator_get(it);
			  if (cur_frame != NULL)
			    {
			      pszFileName = (char*)cpl_frame_get_filename( cur_frame );
			      pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
			      if (strcmp(pszFileTag, "CALIB_RAW") == 0 || strcmp(pszFileTag, "TARGET_RAW") == 0 || strcmp(pszFileTag, "CALIB_SRC_RAW") == 0)
				{
				  plist=cpl_propertylist_load(pszFileName,0);
				  if (strcmp(cpl_propertylist_get_string(oifits[i]->keywords,"ESO CFG BCD MODE"),"OUT-OUT") == 0)
				    {
				      if (strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD1 NAME"),"OUT") != 0 ||
					  strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD2 NAME"),"OUT") != 0 ||
					  cpl_propertylist_get_int(plist,"ESO TPL EXPNO") != cpl_propertylist_get_int(oifits[i]->keywords,"ESO TPL EXPNO"))
					{
					  sel_frame=cpl_frame_duplicate(cur_frame);
					  cpl_frameset_insert(usedframes,sel_frame);
					}
				    }
				  else if(strcmp(cpl_propertylist_get_string(oifits[i]->keywords,"ESO CFG BCD MODE"),"IN-OUT") == 0)
				    {
				      if (strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD1 NAME"),"IN") != 0 ||
					  strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD2 NAME"),"OUT") != 0 ||
					  cpl_propertylist_get_int(plist,"ESO TPL EXPNO") != cpl_propertylist_get_int(oifits[i]->keywords,"ESO TPL EXPNO"))
					{
					  sel_frame=cpl_frame_duplicate(cur_frame);
					  cpl_frameset_insert(usedframes,sel_frame);
					}
				    }
				  else if(strcmp(cpl_propertylist_get_string(oifits[i]->keywords,"ESO CFG BCD MODE"),"OUT-IN") == 0)
				    {
				      if (strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD1 NAME"),"OUT") != 0 ||
					  strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD2 NAME"),"IN") != 0 ||
					  cpl_propertylist_get_int(plist,"ESO TPL EXPNO") != cpl_propertylist_get_int(oifits[i]->keywords,"ESO TPL EXPNO"))
					{
					  sel_frame=cpl_frame_duplicate(cur_frame);
					  cpl_frameset_insert(usedframes,sel_frame);
					}
				    }
				  else if(strcmp(cpl_propertylist_get_string(oifits[i]->keywords,"ESO CFG BCD MODE"),"IN-IN") == 0)
				    {
				      if (strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD1 NAME"),"IN") != 0 ||
					  strcmp(cpl_propertylist_get_string(plist,"ESO INS BCD2 NAME"),"IN") != 0 ||
					  cpl_propertylist_get_int(plist,"ESO TPL EXPNO") != cpl_propertylist_get_int(oifits[i]->keywords,"ESO TPL EXPNO"))
					{
					  sel_frame=cpl_frame_duplicate(cur_frame);
					  cpl_frameset_insert(usedframes,sel_frame);
					}
				    }
				  cpl_propertylist_delete(plist);
				}
			      else if (strcmp(pszFileTag, "BADPIX") == 0 || strcmp(pszFileTag, "NONLINEARITY") == 0 ||
				       strcmp(pszFileTag, "SHIFT_MAP") == 0 || strcmp(pszFileTag, "KAPPA_MATRIX") == 0 ||
				       strcmp(pszFileTag, "OBS_FLATFIELD") == 0 || strcmp(pszFileTag, "JSDC_CAT") == 0)
				{
				  sel_frame=cpl_frame_duplicate(cur_frame);
				  cpl_frameset_insert(usedframes,sel_frame);
				}
		       
			    }
			} while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
		      cpl_frameset_iterator_delete(it);
		      it= cpl_frameset_iterator_new(frameset);
		      do
			{
			  cur_frame = cpl_frameset_iterator_get(it);
			  if (cur_frame != NULL)
			    {
			      pszFileName = (char*)cpl_frame_get_filename( cur_frame );
			      pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
			      if (strcmp(pszFileTag, "SKY_RAW") == 0 || strcmp(pszFileTag, "HOT_DARK") == 0 )
				{
				  sel_frame=cpl_frame_duplicate(cur_frame);
				  cpl_frameset_insert(usedframes,sel_frame);
				}
			    }
			} while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
		      cpl_frameset_iterator_delete(it);
		    }
		  else
		    {
		      usedframes=cpl_frameset_new();
		      cpl_frameset_insert(usedframes,frameCorrFlux);
		      if (framePhot != NULL)
			{
			  cpl_frameset_insert(usedframes,framePhot);
			}
		    }
		  filename=cpl_sprintf("RAW_VIS2_%04d.fits",indexFileVis2);
		  mat_oifits_save(oifits[i],frameset,usedframes,parlist,recname,filename,1);
		  indexFileVis2++;
		  cpl_free(filename);
		  if (oifits[i]->oit3 != NULL)
		    {
		      filename=cpl_sprintf("RAW_CPHASE_%04d.fits",indexFileT3);
		      mat_oifits_save(oifits[i],frameset,usedframes,parlist,recname,filename,2);
		      indexFileT3++;
		      cpl_free(filename);
		    }
			 
		  if ( !cpl_parameter_get_bool(correlatedFlux) )
		    {
		      filename=cpl_sprintf("RAW_SPECTRUM_%04d.fits",indexFileSpec);
		      mat_oifits_save(oifits[i],frameset,usedframes,parlist,recname,filename,3);
		      indexFileSpec++;
		      cpl_free(filename);
		    }
		  cpl_frameset_delete(usedframes);
		  framePhot=NULL;
					
		  // Delete phot and corr data for SIPHOT mode and corr for HIGHSENS mode
		  if (strstr(modeFirst,"s") != NULL)
		    {
		      if (!cpl_parameter_get_bool(correlatedFlux))
			{
			  if (phot[i] != NULL )
			    {
			      mat_photbeams_free(phot[i]);
			      phot[i]=NULL;
			    }
			}
		      if (corrFlux[i] != NULL)
			{
			  mat_corrflux_free(corrFlux[i]);
			  corrFlux[i]=NULL;
			}	
		    }
		}
	    }
	  // Delete phot data for HIGHSENS mode
	  // ?. Free Memory
	  if (listExpnoCorr != NULL)
	    {
	      cpl_vector_delete(listExpnoCorr);
	      listExpnoCorr=NULL;
	    }
	  if (listExpnoPhot != NULL)
	    {
	      cpl_vector_delete(listExpnoPhot);
	      listExpnoPhot=NULL;
	    }
	  if ( !cpl_parameter_get_bool(correlatedFlux) )
	    {
	      for(i=0;i<nbPhot;i++)
		{
		  if (phot[i] != NULL)
		    {
		      mat_photbeams_free(phot[i]);
		    }
		}
	      if (phot != NULL)
		{
		  cpl_free(phot);
		  phot=NULL;
		}
	    }
	  for(i=0;i<nbCorrFlux;i++)
	    {
	      if (corrFlux[i] != NULL)
		{
		  mat_corrflux_free(corrFlux[i]);
		}
	      if (oivis[i] != NULL)
		{
		  mat_oifits_delete(oivis[i]);
		}
	      if (oiopd[i] != NULL)
		{
		  mat_oiopdwvpo_delete(oiopd[i]);
		}
	    }
	  if (corrFlux != NULL)
	    {
	      cpl_free(corrFlux);
	      corrFlux=NULL;
	    }
	  if (oivis != NULL)
	    {
	      cpl_free(oivis);
	      oivis=NULL;
	    }
	  if (oiopd != NULL)
	    {
	      cpl_free(oiopd);
	      oiopd=NULL;
	    }
	  if (nofringe != NULL)
	     {
	       mat_corrflux_free(nofringe);
	       nofringe=NULL;
	     }
	  mat_oifits_list_delete(oifits,nbCorrFlux);

	  nbCorrFlux=0;
	  nbPhot=0;
	}
      cpl_frameset_delete(framesetsorted);
    }

		  
  // Delete photoutout
  if (photOutOut != NULL)
    {
      mat_photbeams_free(photOutOut);
    }

  return 0;
}
