/* $Id: mat_region.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_region.c $
 */

#include <stdlib.h>
#include <stdio.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>
#include <cpl.h>

#include "mat_error.h"
#include "mat_utils.h"
#include "mat_region.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Functions
 -----------------------------------------------------------------------------*/

/**
   @brief Creates a new mat_region data structure.
   @param nr     The region number.
   @param nbtel  The number of telescopes.
   @param x      The lower left corner of the sub-window.
   @param y      The lower left corner of the sub-window.
   @param nx     The sub-window width.
   @param ny     Th sub-window height.
   @returns A new mat_region data structure or NULL.

   This function creates a new mat_regin data structure including space
   for the beam numbers and for all strings. The sub-window position and
   dimension is filled with the parameters (x, y, nx and ny). All other
   elements are initialized with default values.
 */
mat_region *mat_region_new(int nr, int nbtel, int x, int y, int nx, int ny)
{
  char        regionname[32];
  int         i;
  mat_region *region = (mat_region*)cpl_calloc(1, sizeof(mat_region));

  if (region == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for region %d", nr);
      return NULL;
    }
  region->numregion = nr;
  /* the following values are only provisional */
  snprintf(regionname, 32, "IMG%d", nr);
  region->regionname = cpl_strdup(regionname);
  if (region->regionname == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for region->regionname");
      mat_region_delete(region);
      return NULL;
    }
  region->numdetector = 1; /* only provisional */
  region->numbeam = cpl_calloc(nbtel, sizeof(int));
  if (region->numbeam == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for region->numbeam");
      mat_region_delete(region);
      return NULL;
    }
  for (i = 0; i < nbtel; i++)
    {
      region->numbeam[i] = 0;
    }
  region->correlation = 0;
  region->corner[0] = x;
  region->corner[1] = y;
  region->gain = 1.0;
  region->crval[0] = 0.0;
  region->crval[1] = 0.0;
  region->crpix[0] = 0.0;
  region->crpix[1] = 0.0;
  region->ctype[0] = cpl_calloc(SIZE_MAX_KEYWORDS, sizeof(char));
  if (region->ctype[0] == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for region->ctype[0]");
      mat_region_delete(region);
      return NULL;
    }
  strncpy(region->ctype[0], "PIXEL", SIZE_MAX_KEYWORDS);
  region->ctype[1] = cpl_calloc(SIZE_MAX_KEYWORDS, sizeof(char));
  if (region->ctype[0] == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for region->ctype[1]");
      mat_region_delete(region);
      return NULL;
    }
  strncpy(region->ctype[1], "PIXEL", SIZE_MAX_KEYWORDS);
  region->naxis[0] = nx;
  region->naxis[1] = ny;
  return region;
}

/**
   @brief Deletes a mat_region data structure.
   @param region   The mat_region data structure.
   @returns a cpl_error_code

   This function deletes a mat_regin data structure and the enclosed memory space for
   the region name, the beam numbers and the coordinate types (ctype element).
 */
cpl_error_code mat_region_delete(mat_region *region)
{
  if (region == NULL) return CPL_ERROR_NONE;
  if (region->regionname != NULL)
    {
      cpl_free(region->regionname);
      region->regionname = NULL;
    }
  if (region->numbeam != NULL)
    {
      cpl_free(region->numbeam);
      region->numbeam = NULL;
    }
  if (region->ctype[0] != NULL)
    {
      cpl_free(region->ctype[0]);
      region->ctype[0] = NULL;
    }
  if (region->ctype[1] != NULL)
    {
      cpl_free(region->ctype[1]);
      region->ctype[1] = NULL;
    }
  cpl_free(region);
  return CPL_ERROR_NONE;
}

/**
   @brief Sets the name of a region.
   @param region    The mat_region data structure.
   @param name      The name of the region (mandatory).
   @param nr        An optional name appendix or -1.
   @returns A cpl_error_code.

   This function deletes the previous region name and creates and stores a new one.
   If the name appendix is greater than 0, the new region name is the given name
   with the number appended.
 */
cpl_error_code mat_region_set_name(mat_region *region, const char *name, int nr)
{
  if (region == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_region (region) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (name == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no char * (name) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  if (region->regionname != NULL)
    {
      cpl_free(region->regionname);
      region->regionname = NULL;
    }
  if (nr <= 0)
    {
      region->regionname = cpl_strdup(name);
    }
  else
    {
      char rname[32];
      snprintf(rname, 32, "%s%d", name, nr);
      rname[31] = '\0';
      region->regionname = cpl_strdup(rname);
    }
  if (region->regionname == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for region->regionname");
      return CPL_ERROR_UNSPECIFIED;
    }
  return CPL_ERROR_NONE;
}

/**
   @brief Sets the position and size of a sub-window.
   @param region    A sub-window as mat_region data structure.
   @param x      The lower left corner of the sub-window.
   @param y      The lower left corner of the sub-window.
   @param nx     The sub-window width.
   @param ny     Th sub-window height.
   @returns A cpl_error_code.

   This function sets the position and size of a sub-window.
 */
cpl_error_code mat_region_set_area(mat_region *region, int x, int y, int nx, int ny)
{
  if (region == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_region (region) argument given");
      return CPL_ERROR_NULL_INPUT;
    }
  region->corner[0] = x;
  region->corner[1] = y;
  region->naxis[0] = nx;
  region->naxis[1] = ny;
  return CPL_ERROR_NONE;
}

/**
   @brief Copies a mat_region data structure into a given obe or a new allocated data structure.
   @param dst     The destination data structure or NULL.
   @param src     The mat_region data structure used as source.
   @param nbtel   The number of telescopes used for the beam number vector.
   @returns A mat_region data structure or NULL.

   This function uses a given (parameter dst) data structure or allocates a new one to copy
   the given mat_region data structure (parameter src) into.
 */
mat_region *mat_region_copy(mat_region *dst, const mat_region *src, int nbtel)
{
  int   created = 0;
  int   i;

  cpl_errorstate prestate = cpl_errorstate_get();
  mat_assert_value((src!=NULL),CPL_ERROR_NULL_INPUT,NULL,"no mat_region (src) argument given");
  mat_assert_value((nbtel>0),CPL_ERROR_NULL_INPUT,NULL,"Number of telecopes sould be greater than or equal to 1");
  /* if the destination does not exist, create an new region */
  if (dst == NULL)
    {
      dst = (mat_region *)cpl_calloc(1, sizeof(mat_region));
      if (dst == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for new mat_region (dst)");
	return NULL;
      }
      created = 1;
    }
  /* copy all properties of the source region into the destination region (beware of the arrays!) */
  dst->numregion = src->numregion;
  mat_region_set_name(dst, src->regionname, 0);
  dst->numdetector = src->numdetector;
  dst->numbeam = cpl_realloc(dst->numbeam, nbtel*sizeof(int));
  if (dst->numbeam == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for new beam numbers (region->numbeam)");
      if (created)
	{
	  mat_region_delete(dst);
	}
      return NULL;
    }
  for (i = 0; i < nbtel; i++)
    {
      dst->numbeam[i] = src->numbeam[i];
    }
  dst->correlation = src->correlation;
  dst->corner[0] = src->corner[0];
  dst->corner[1] = src->corner[1];
  dst->gain = src->gain;
  dst->crval[0] = src->crval[0];
  dst->crval[1] = src->crval[1];
  dst->crpix[0] = src->crpix[0];
  dst->crpix[1] = src->crpix[1];
  if (dst->ctype[0] != NULL)
    {
      cpl_free(dst->ctype[0]);
      dst->ctype[0] = NULL;
    }
  dst->ctype[0] = cpl_strdup(src->ctype[0]);
  if (dst->ctype[1] != NULL)
    {
      cpl_free(dst->ctype[1]);
      dst->ctype[1] = NULL;
    }
  dst->ctype[1] = cpl_strdup(src->ctype[1]);
  dst->naxis[0] = src->naxis[0];
  dst->naxis[1] = src->naxis[1];
  /* Error handling */
  if (!cpl_errorstate_is_equal(prestate)) { 
    cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
    if (created)
      {
	mat_region_delete(dst);
      }
    return NULL;
  }
  return dst;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Duplicates a mat_region structure
  @param region         mat_region structure to duplicate
  @param nbtel          number of telescopes
  @return mat_region
 */
/*-----------------------------------------------------------------------------*/
mat_region* mat_region_duplicate (const mat_region * region, int nbtel)
{
  return mat_region_copy(NULL, region, nbtel);
}

