/* $Id: mat_shift.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_shift.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "mat_shift.h"
#include "mat_error.h"
#include <sys/time.h>
/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/
#define EPSILON 0.000001
#define ERR_MAX_PIX 0.01
#define COEF_GAMMA 0.

/* must not be followed by a semicolon!
 * gcc on mac has omp but it doesn't work for nontrivial cases as libc lacks
 * alloca */
#if defined (_OPENMP) && !defined( __APPLE__)
#define HDRL_OMP(x) _Pragma (#x)
#else
#define HDRL_OMP(x)
#endif
static struct timeval tv1, tv2;
/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @ingroup shift
  @brief Load a shiftmap in a mat_shiftmap structure
  @param frame : current frame contaning the shiftmap
  @return mat_shiftmap: loaded structure
 */
/*-----------------------------------------------------------------------------*/
mat_shiftmap * mat_shiftmap_load(cpl_frame *frame) {

  mat_shiftmap *shift=NULL;
  int i=0;
  int j=0;
  int l=0;
  int m=0;
  int index=0;
  int nbExtent=0;
  int flagDet=0;
  int flagShift=0;
  char *keyExtname=NULL;
  char str[32];

  cpl_propertylist *plist=NULL;
  cpl_table *table=NULL;

  cpl_errorstate prestate = cpl_errorstate_get();

  // Check input parameters
  mat_assert_value((frame!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no cpl_frame (frame) argument given");

  // Allocate the structure and initialize its attributes
  shift =cpl_malloc(sizeof(mat_shiftmap));
  if (shift==NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for shift");
    return NULL;
  }
  (*shift).keywords= cpl_propertylist_load(cpl_frame_get_filename(frame),0);
  (*shift).imgdet=NULL;
  // Find all extensions before IMAGING_DATA
  nbExtent=cpl_frame_get_nextensions(frame);
  for(i=0;i<nbExtent;i++) {
    plist=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1); 
    keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
    if (keyExtname != NULL) {
      table=cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
      // Load data in attribute imgdet
      if (!strcmp(keyExtname,"IMAGING_DETECTOR")) {
	(*shift).imgdet = mat_imagingdetector_from_table(plist, table);
        /* Error handling */
        if ((*shift).imgdet == NULL) { 
          cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
				"Could not load IMAGING_DETECTOR in frame: %s",
				cpl_frame_get_filename(frame));
        }
        else {
	  flagDet=1;
	}
      }
      cpl_table_delete(table);
    }
    //cpl_msg_info(cpl_func,"flag %d",flagDet);
    cpl_propertylist_delete(plist);
  }
  if ( flagDet==1 ) {

    // Now find  SHIFT_DATA
    for(i=0;i<nbExtent;i++) {
      plist=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
      keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
      if (keyExtname != NULL) {
	table=cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
	// Load data in attribute shift 
	if (!strcmp(keyExtname,"SHIFT_MAP")) {
	  (*shift).nbreg=cpl_propertylist_get_int(plist,"NREGION");
	  (*shift).list_shiftcoefx=cpl_calloc((*shift).nbreg,sizeof(mat_shiftcoef *));
	  if (shift->list_shiftcoefx==NULL) {
	    cpl_msg_error(cpl_func,"could not allocate memory for shift->list_shiftcoefx");
	    return NULL;
	  }
	  (*shift).list_shiftcoefy=cpl_calloc((*shift).nbreg,sizeof(mat_shiftcoef *));
	  if (shift->list_shiftcoefy==NULL) {
	    cpl_msg_error(cpl_func,"could not allocate memory for shift->list_shiftcoefy");
	    return NULL;
	  }
	  for(j=0; j<(*shift).nbreg; j++) {
	    // Shift in X direction
	    (*shift).list_shiftcoefx[j] = cpl_malloc(sizeof(mat_shiftcoef));
	    if (shift->list_shiftcoefx[j]==NULL) {
	      cpl_msg_error(cpl_func,"could not allocate memory for shift->list_shiftcoefx[j]");
	      return NULL;
	    }
	    (*shift).list_shiftcoefx[j]->numregion = j;
	    (*shift).list_shiftcoefx[j]->numdetector =
	      (*shift).imgdet->list_region[j]->numdetector;
	    (*shift).list_shiftcoefx[j]->order = 3;
	    (*shift).list_shiftcoefx[j]->nlig = (*shift).imgdet->list_region[j]->naxis[1];
	    (*shift).list_shiftcoefx[j]->coefdistor =
	      cpl_matrix_new(3,(*shift).imgdet->list_region[j]->naxis[1]);
	    (*shift).list_shiftcoefx[j]->errdistor =
	      cpl_matrix_new(3,(*shift).imgdet->list_region[j]->naxis[1]);

	    snprintf(str,sizeof(str),"MAPX_%d",j+1);
	    index=0;
	    
	    for(m=0;m<(*shift).list_shiftcoefx[j]->order;m++) {
	      for(l=0;l<(*shift).imgdet->list_region[j]->naxis[1];l++) {
	    	cpl_matrix_set(shift->list_shiftcoefx[j]->coefdistor, m, l,
	    		       cpl_array_get_double
	    		       (cpl_table_get_array(table,str,0),index,NULL));
	    	cpl_matrix_set(shift->list_shiftcoefx[j]->errdistor, m, l,
	    		       cpl_array_get_double
	    		       (cpl_table_get_array(table,str,1),index,NULL));
	    	index++;
	      }
	    }
	    
	    // Shift in Y direction
	    //cpl_msg_info(cpl_func,"extensions frame not shiftmap shift y");
	    (*shift).list_shiftcoefy[j] = cpl_malloc(sizeof(mat_shiftcoef));
	    if (shift->list_shiftcoefy[j]==NULL) {
	      cpl_msg_error(cpl_func,"could not allocate memory for shift->list_shiftcoefy[j]");
	      return NULL;
	    }
	    (*shift).list_shiftcoefy[j]->numregion = j;
	    (*shift).list_shiftcoefy[j]->numdetector =
	      (*shift).imgdet->list_region[j]->numdetector;
	    (*shift).list_shiftcoefy[j]->order = 3;
	    (*shift).list_shiftcoefy[j]->nlig =
	      (*shift).imgdet->list_region[j]->naxis[0];
	    (*shift).list_shiftcoefy[j]->coefdistor =
	      cpl_matrix_new(3,(*shift).imgdet->list_region[j]->naxis[0]);
	    (*shift).list_shiftcoefy[j]->errdistor =
	      cpl_matrix_new(3,(*shift).imgdet->list_region[j]->naxis[0]);
	    snprintf(str,sizeof(str),"MAPY_%d",j+1);
	    index=0;
	    for(m=0;m<(*shift).list_shiftcoefy[j]->order;m++) {
	      for(l=0;l<(*shift).imgdet->list_region[j]->naxis[0];l++) {
	    	cpl_matrix_set(shift->list_shiftcoefy[j]->coefdistor, m, l,
	    		       cpl_array_get_double(cpl_table_get_array(table,str,0),
	    					    index,NULL));
	    	cpl_matrix_set(shift->list_shiftcoefy[j]->errdistor, m, l,
	    		       cpl_array_get_double(cpl_table_get_array(table,str,1),
	    					    index,NULL));
	    	index++;
	      }
	    }
	  }
	  
	  // Dispersion coefficient
	  snprintf(str,sizeof(str),"DISP");
	  for(m=0;m<N_DEG_DISPERSION_LAW+1;m++) {
	    (*shift).dispcoef[m]=0.;
	    (*shift).errdispcoef[m]=0.;
	    if (m < cpl_table_get_column_dimension(table,str,0))
	      {
		(*shift).dispcoef[m]=cpl_array_get_double(cpl_table_get_array(table,str,0),m,NULL);
		(*shift).errdispcoef[m]=cpl_array_get_double(cpl_table_get_array(table,str,1),m,NULL);
	      }
	  }
	  
	  flagShift=1;
	}
	cpl_table_delete(table);
      }
      cpl_propertylist_delete(plist);
    }
  }
  // Check the flags
  if ( flagDet==0 || flagShift==0) {
    cpl_error_set_message(cpl_func,CPL_ERROR_ILLEGAL_INPUT,
                          "Extension missing in frame: %s",
                          cpl_frame_get_filename(frame));
  }
  // Free memory
  /* Error handling */
  if (!cpl_errorstate_is_equal(prestate)) { 
    
    cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
    return NULL;
  }
  return shift;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup shift
  @brief delete a structure mat_shiftmap
  @param  shift : current structure
  @return error code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_shiftmap_free(mat_shiftmap *shift) {
  int i=0;

  // Check input parameters
  mat_assert_value((shift!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,"no mat_shiftmap (shift) argument given");

  if (shift->keywords != NULL) cpl_propertylist_delete((*shift).keywords);
  if (shift->imgdet != NULL) mat_imagingdetector_delete((*shift).imgdet);

  for(i=0; i<(*shift).nbreg; i++) {
    if (shift->list_shiftcoefx[i] != NULL) {
      cpl_matrix_delete((*shift).list_shiftcoefx[i]->coefdistor);
      cpl_matrix_delete((*shift).list_shiftcoefx[i]->errdistor);
      cpl_free((*shift).list_shiftcoefx[i]);
    }
    if (shift->list_shiftcoefy[i] != NULL) {
      cpl_matrix_delete((*shift).list_shiftcoefy[i]->coefdistor);
      cpl_matrix_delete((*shift).list_shiftcoefy[i]->errdistor);
      cpl_free((*shift).list_shiftcoefy[i]);
    }
  }
  if (shift->list_shiftcoefx != NULL) cpl_free(shift->list_shiftcoefx);
  if (shift->list_shiftcoefy != NULL) cpl_free(shift->list_shiftcoefy);
  cpl_free(shift);
  return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup shift
  @brief Save a mat_shiftmap structure in a SHIFT_MAP FITS file
  @param shift             mat_shiftmap structure to save
  @param set_in            input frameset of the recipes
  @param set_sel           selected frameset
  @param parlist           recipes parameter list
  @param recipe_name       name of recipes
  @param pro_cat           PRO.CATG
  @param pro_tech          PRO.TECH
  @param pro_sci           PRO.SCIENCE
  @param pheader           header of the binary table
  @param filename          name of the FITS file
  @return cpl_error_code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_shiftmap_save(
				mat_shiftmap *shift,
				cpl_propertylist *pheader,
				cpl_frameset *set_in,
				cpl_frameset *set_sel,
				const cpl_parameterlist *parlist,
				const char *recipe_name,
				const char *pro_cat,
				const char *pro_tech,
				const char *pro_sci,
				char *filename) {


  int i=0;
  double *mapx=NULL;
  double *mapy=NULL;
  /* char *str=NULL; */
  cpl_table *shift_map=NULL;
  cpl_propertylist *applist=NULL;
  cpl_array *disp_array=NULL;
  cpl_array *mapx_array[5];
  cpl_array *mapy_array[5];
  cpl_errorstate prestate= cpl_errorstate_get();


  // Check input parameters
  mat_assert_value((shift!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,"no mat_shiftmap (shift) argument given");
  mat_assert_value((pheader!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, "no cpl_propertylist (pheader) argument given");
  mat_assert_value((set_in!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, "no cpl_frameset (set_in) argument given");
  mat_assert_value((set_sel!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, "no cpl_frameset (set_sel) argument given");
  mat_assert_value((filename!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, "no filename given");


  // Prepare the keywords of the SHIFT_MAP binary table
  /*
  str=malloc(50*sizeof(char));
  strncpy(str,(char *)cpl_propertylist_get_string(pheader,"ORIGIN"),50);
  cpl_propertylist_erase(pheader,"ORIGIN");
  cpl_propertylist_append_string(pheader,"ORIGIN",str);
  strncpy(str,(char *)cpl_propertylist_get_string(pheader,"DATE"),50);
  cpl_propertylist_erase(pheader,"DATE");
  cpl_propertylist_append_string(pheader,"DATE",str);
  free(str);
  */
  cpl_propertylist_erase(pheader,"XTENSION");
  cpl_propertylist_erase(pheader,"BITPIX");
  cpl_propertylist_erase_regexp(pheader,"NAXIS",0);
  cpl_propertylist_erase_regexp(pheader,"COUNT",0);
  cpl_propertylist_erase(pheader,"TFIELDS");
  cpl_propertylist_erase_regexp(pheader,"SIM",0);
  cpl_propertylist_erase_regexp(pheader,"FORM",0);
  cpl_propertylist_erase_regexp(pheader,"TTYPE",0);
  cpl_propertylist_erase_regexp(pheader,"TDIM",0);


  // Remove the keywords from the DPR categories
  cpl_propertylist_erase_regexp(shift->keywords, "DPR", 0);

  // Add the keywords from the PRO categories
  if (pro_tech != NULL) {
    cpl_propertylist_append_string(shift->keywords, CPL_DFS_PRO_TECH, pro_tech);
  }
  if (pro_cat != NULL) {
    cpl_propertylist_append_string(shift->keywords, CPL_DFS_PRO_CATG, pro_cat);
  }
  if (pro_sci != NULL) {
    //cpl_propertylist_append_string(shift->keywords, CPL_DFS_PRO_SCIENCE, pro_sci);
    if (strstr(pro_sci,"F") != NULL)
      {
	cpl_propertylist_append_bool(shift->keywords, CPL_DFS_PRO_SCIENCE, 0);
      }
    else
      {
	cpl_propertylist_append_bool(shift->keywords, CPL_DFS_PRO_SCIENCE, 1);
      }
       
  }


  // Add the keywords from the PRO categories
  //cpl_propertylist_append_string(shift->keywords, "PRO TECH", "IMAGE");
  //cpl_propertylist_append_string(shift->keywords, "PRO CATG", "SHIFT_MAP");
  //cpl_propertylist_append_string(shift->keywords, "PRO SCIENCE", "F");

  // Fill a cpl_table with shift data that will replace the IMAGING_DATA
  shift_map = cpl_table_new(2);

  // Add a column for the detector number
  cpl_table_new_column(shift_map, "DETECTOR", CPL_TYPE_INT);
  cpl_table_set_int(shift_map, "DETECTOR", 0,
                   (*shift).imgdet->list_region[0]->numdetector);
   cpl_table_set_int(shift_map, "DETECTOR", 1,
                   (*shift).imgdet->list_region[0]->numdetector);

  // Add 5 columns for the spatial distorsion coefficients (1 for each region)
  cpl_table_new_column_array(shift_map, "MAPX_1", CPL_TYPE_DOUBLE, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefx[0]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefx[0]->coefdistor));
  cpl_table_new_column_array(shift_map, "MAPX_2", CPL_TYPE_DOUBLE, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefx[1]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefx[1]->coefdistor));
  cpl_table_new_column_array(shift_map, "MAPX_3", CPL_TYPE_DOUBLE, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefx[2]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefx[2]->coefdistor));
  cpl_table_new_column_array(shift_map, "MAPX_4", CPL_TYPE_DOUBLE, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefx[3]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefx[3]->coefdistor));
  cpl_table_new_column_array(shift_map, "MAPX_5", CPL_TYPE_DOUBLE, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefx[4]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefx[4]->coefdistor));

  // Add 5 columns for the spectral distorsion coefficients (1 for each region)
  cpl_table_new_column_array(shift_map, "MAPY_1", CPL_TYPE_DOUBLE, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefy[0]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefy[0]->coefdistor));
  cpl_table_new_column_array(shift_map, "MAPY_2", CPL_TYPE_DOUBLE, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefy[1]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefy[1]->coefdistor));
  cpl_table_new_column_array(shift_map, "MAPY_3", CPL_TYPE_DOUBLE, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefy[2]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefy[2]->coefdistor));
  cpl_table_new_column_array(shift_map, "MAPY_4", CPL_TYPE_DOUBLE, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefy[3]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefy[3]->coefdistor));
  cpl_table_new_column_array(shift_map, "MAPY_5", CPL_TYPE_DOUBLE, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefy[4]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefy[4]->coefdistor));

  // Get the data in a cpl_array
  for(i=0; i<5; i++) {
    mapx = (double *)cpl_matrix_get_data((*shift).list_shiftcoefx[i]->coefdistor);
    mapy = (double *)cpl_matrix_get_data((*shift).list_shiftcoefy[i]->coefdistor);
    mapx_array[i]= cpl_array_wrap_double(mapx, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefx[i]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefx[i]->coefdistor));
    mapy_array[i]= cpl_array_wrap_double(mapy, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefy[i]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefy[i]->coefdistor));
  }

  // Fill the columns with the data
  cpl_table_set_array(shift_map, "MAPX_1", 0, mapx_array[0]);
  cpl_table_set_array(shift_map, "MAPX_2", 0, mapx_array[1]);
  cpl_table_set_array(shift_map, "MAPX_3", 0, mapx_array[2]);
  cpl_table_set_array(shift_map, "MAPX_4", 0, mapx_array[3]);
  cpl_table_set_array(shift_map, "MAPX_5", 0, mapx_array[4]);
  cpl_table_set_array(shift_map, "MAPY_1", 0, mapy_array[0]);
  cpl_table_set_array(shift_map, "MAPY_2", 0, mapy_array[1]);
  cpl_table_set_array(shift_map, "MAPY_3", 0, mapy_array[2]);
  cpl_table_set_array(shift_map, "MAPY_4", 0, mapy_array[3]);
  cpl_table_set_array(shift_map, "MAPY_5", 0, mapy_array[4]);

// Free memory
  for(i=0; i<5; i++) {
    cpl_array_unwrap(mapx_array[i]);
    cpl_array_unwrap(mapy_array[i]);
  }

  // Get the data in a cpl_array
  for(i=0; i<5; i++) {
    mapx = (double *)cpl_matrix_get_data((*shift).list_shiftcoefx[i]->errdistor);
    mapy = (double *)cpl_matrix_get_data((*shift).list_shiftcoefy[i]->errdistor);
    mapx_array[i]= cpl_array_wrap_double(mapx, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefx[i]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefx[i]->coefdistor));
    mapy_array[i]= cpl_array_wrap_double(mapy, 
      cpl_matrix_get_nrow((*shift).list_shiftcoefy[i]->coefdistor) *
      cpl_matrix_get_ncol((*shift).list_shiftcoefy[i]->coefdistor));
  }

  // Fill the columns with the data
  cpl_table_set_array(shift_map, "MAPX_1", 1, mapx_array[0]);
  cpl_table_set_array(shift_map, "MAPX_2", 1, mapx_array[1]);
  cpl_table_set_array(shift_map, "MAPX_3", 1, mapx_array[2]);
  cpl_table_set_array(shift_map, "MAPX_4", 1, mapx_array[3]);
  cpl_table_set_array(shift_map, "MAPX_5", 1, mapx_array[4]);
  cpl_table_set_array(shift_map, "MAPY_1", 1, mapy_array[0]);
  cpl_table_set_array(shift_map, "MAPY_2", 1, mapy_array[1]);
  cpl_table_set_array(shift_map, "MAPY_3", 1, mapy_array[2]);
  cpl_table_set_array(shift_map, "MAPY_4", 1, mapy_array[3]);
  cpl_table_set_array(shift_map, "MAPY_5", 1, mapy_array[4]);

  // Add a column for the dispersion law coefficients
  cpl_table_new_column_array(shift_map, "DISP", CPL_TYPE_DOUBLE, N_DEG_DISPERSION_LAW+1);
  disp_array= cpl_array_new(N_DEG_DISPERSION_LAW+1, CPL_TYPE_DOUBLE);
  for(i=0; i<N_DEG_DISPERSION_LAW+1; i++) {
    cpl_array_set_double(disp_array, i, (*shift).dispcoef[i]);
  }
  cpl_table_set_array(shift_map, "DISP", 0, disp_array);
  cpl_array_delete(disp_array);

  disp_array= cpl_array_new(N_DEG_DISPERSION_LAW+1, CPL_TYPE_DOUBLE);
  for(i=0; i<N_DEG_DISPERSION_LAW+1; i++) {
    cpl_array_set_double(disp_array, i, (*shift).errdispcoef[i]);
  }
  cpl_table_set_array(shift_map, "DISP", 1, disp_array);

  // Make a new FITS file with shift keywords and the created table
  //cpl_table_save(shift_map, shift->keywords, pheader, filename, CPL_IO_CREATE);
  // Make a new FITS file with gendata keywords and the created table
  cpl_propertylist_erase(shift->keywords,"RADECSYS");
  applist=cpl_propertylist_duplicate(shift->keywords);
  cpl_dfs_save_table(set_in, shift->keywords, parlist, set_sel, NULL, 
		     shift_map, pheader, recipe_name, applist, NULL, 
					  PACKAGE "/" PACKAGE_VERSION, filename);

  

// Free memory
  cpl_propertylist_delete(applist);
  for(i=0; i<5; i++) {
    cpl_array_unwrap(mapx_array[i]);
    cpl_array_unwrap(mapy_array[i]);
  }
  cpl_array_delete(disp_array);
  cpl_table_delete(shift_map);

  if (!cpl_errorstate_is_equal(prestate)) { 
    
    cpl_errorstate_set(prestate);
    
    return CPL_ERROR_UNSPECIFIED;
  }

  return CPL_ERROR_NONE;

}


/*----------------------------------------------------------------------------*/
/**
  @ingroup shift
  @brief Compute the SHIFT_MAP and stores it in a mat_shiftmap structure. This 
  function computes also the QC DET<i> PICOV<l> (the relative coverage of the 
  interferometric channel by the photometric channels).</em>
  @param distor_hotdark    DISTOR_HOTDARK data (one or several frames for each
  pinhole)
  @param spectra_hotdark   SPECTRA_HOTDARK data (one or several frames)
  @param distor            DISTOR_IMAGES data (one or several frames for each 
  pinhole)
  @param spectra           SPECTRA_IMAGES data (one or several frames)
  @param nbframes[]        array of int representing the dimension of the 4 
  mat_gendata passed to this function (number of frames)
  @param debug             if set, intermediate product files are saved for debugging
  @param debug             if set, observationnal correction are applied
  @return mat_shiftmap
 */
/*-----------------------------------------------------------------------------*/
mat_shiftmap *mat_compute_shift(mat_gendata **distor_hotdark, 
                                mat_gendata **spectra_hotdark,
                                mat_gendata **distor,
                                mat_gendata **spectra,
                                int nbframes[4],
				cpl_parameter *debug,
				cpl_parameter *obsCorrection) {

  mat_shiftmap *shiftmap=NULL;

  // Get the number of regions
  int i=0;
  int j=0;
  int k=0;
  int l=0;
  int n=0;
  int imx=0; 
  int cpt=0;
  int nboutliers=0;
  int nboutprec=0;
  int boxsz=0;
  int nbmaxmin=0;
  int nbmaxminrejected=0;
  int *boundaries=NULL;
  int nInterf=0;
  int flag=0;
  int out=0;
  int index=0;

  /* cpl_errorstate prestate= cpl_errorstate_get(); */

  cpl_image **distor_h1_raw=NULL;
  cpl_image **distor_h2_raw=NULL;
  cpl_image **distor_h3_raw=NULL;
  cpl_image **distor_background=NULL;
  cpl_image **spectra_raw=NULL;
  cpl_image **spectra_background=NULL;
  /* cpl_image *imgInteg=NULL; */
  cpl_image *foo=NULL;

  double *nbimg=NULL;
  double valmax=0.;
  double valmin=0.;
  double params[3];
  double res=0;
  double val=0;
  double val_cur=0.;
  double chi2=0.;
  double poserr=0.;
  double *poserravg=NULL;
  double zero_rate=0;
  double refpix[10];
  /* double foo1=0.; */
  /* double foo2=0.; */
  /* double foo3=0.; */
  /* double foo4=0.; */
  /* double coefb0=0.; */
  /* double coefb1=0.; */
  /* double coef_a[3]; */
  /* double coef_b[3]; */
  /* double coef_c[3]; */
  /* double coef_d[3]; */
  /* double coef_e[3]; */
  /* double coef_f[3]; */


  cpl_vector *row=NULL;
  cpl_vector **spectraObs=NULL;
  /* cpl_vector *specFeature=NULL; */
  cpl_vector *fitvals=NULL;
  cpl_vector *fitvals2=NULL;
  cpl_vector *outliers=NULL;
  cpl_vector *residual=NULL;

  cpl_matrix *coeff=NULL;
  cpl_matrix *coefftrans=NULL;
  cpl_matrix *rhs=NULL;
  cpl_matrix *normal=NULL;
  cpl_matrix *invnormal=NULL;
  cpl_matrix *matrix_temp=NULL;
  cpl_matrix *sol=NULL;
  cpl_matrix *samppos=NULL;

  mat_gendata **spectra_shift=NULL;
  mat_gendata **spectra_hd_shift=NULL;
  /* mat_gendata **spectra_corrected=NULL; */
  /* mat_gendata **distor_corrected=NULL; */

  char *chipName=NULL;
  char *resolution=NULL;
  /* char *filename=NULL; */
  char *qcname=NULL;

  FILE *file = NULL;

  cpl_size mindeg,maxdeg,deg1,deg2;

  cpl_array *typemaxmin=NULL;

  cpl_polynomial *polysmooth=NULL;
  
  cpl_boolean sampsym=TRUE;

  cpl_propertylist *qclist=NULL;
  /* cpl_propertylist *nrtslist=NULL; */

  gsl_vector *vector=NULL;



// Check input parameters
  mat_assert_value((distor_hotdark!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no mat_gendata (distor_hotdark) argument given");
  mat_assert_value((spectra_hotdark!=NULL),CPL_ERROR_NULL_INPUT, NULL, "no mat_gendata (spectra_hotdark) argument given");
  mat_assert_value((distor!=NULL),CPL_ERROR_NULL_INPUT, NULL, "no mat_gendata (distor) argument given");
  mat_assert_value((spectra!=NULL),CPL_ERROR_NULL_INPUT, NULL, "no mat_gendata (spectra) argument given");
  for (i=0;i<4;i++) {
    mat_assert_value((nbframes[i] > 0), CPL_ERROR_NULL_INPUT, NULL, "number of each kind of frames should be at least 1");
  }

  int nbregion = distor[0]->imgdet->nbregion;
  int nb_row[nbregion];
  int nb_column[nbregion]; 
  cpl_matrix *distor_index_max[nbregion];
  int nPhot[nbregion-1];
  double mjdobs=0.;

  // Get the MJD of observations
  mjdobs=cpl_propertylist_get_float(distor[0]->keywords,"MJD-OBS");
  
  // Get the band and the resolution
  chipName= (char *)cpl_propertylist_get_string(spectra[0]->keywords,
                                                  "ESO DET CHIP NAME");
  if (chipName == NULL) {
    cpl_msg_error(cpl_func,"No ESO DET CHIP NAME keyword in the primary header of spectra[0]");
    return NULL;
  }
  if(!strcmp(chipName, "AQUARIUS")) {
    resolution= (char *)cpl_propertylist_get_string(spectra[0]->keywords,
                                                    "ESO INS DIN ID");
  } else {
    resolution= (char *)cpl_propertylist_get_string(spectra[0]->keywords,
                                                    "ESO INS DIL ID");
  }
  if (resolution == NULL) {
    cpl_msg_error(cpl_func,"No ESO INS DIN ID keyword in the primary header of spectra[0]");
    return NULL;
  }

// Memory allocation
  shiftmap = cpl_malloc(sizeof(mat_shiftmap));
  if (shiftmap == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for mat_shiftmap");
    return NULL;
  }
  (*shiftmap).keywords = cpl_propertylist_duplicate(spectra[0]->keywords);
  if (shiftmap->keywords == NULL) {
    cpl_msg_error(cpl_func,"could not duplicate a cpl_propertylist");
    return NULL;
  }
  (*shiftmap).imgdet = mat_imagingdetector_duplicate(distor[0]->imgdet);
  if (shiftmap->imgdet == NULL) {
    cpl_msg_error(cpl_func,"could not duplicate a mat_imagingdetector");
    return NULL;
  }
  (*shiftmap).nbreg = nbregion;
  (*shiftmap).list_shiftcoefx = cpl_calloc(nbregion, sizeof(mat_shiftcoef *));
  if (shiftmap->list_shiftcoefx == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for mat_shiftcoef");
    return NULL;
  }
  (*shiftmap).list_shiftcoefy = cpl_calloc(nbregion, sizeof(mat_shiftcoef *));
  if (shiftmap->list_shiftcoefy == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for mat_shiftcoef");
    return NULL;
  }
  for(i=0; i<N_DEG_DISPERSION_LAW+1; i++) {
    (*shiftmap).dispcoef[i]=0;
  }
  nbimg = cpl_calloc(nbregion, sizeof(double));
  if (nbimg == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for double");
    cpl_free(shiftmap);
    return NULL;
  }


  // Initialize the cpl_image tables
  distor_h1_raw = cpl_calloc(nbregion, sizeof(cpl_image *));
  if (distor_h1_raw == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
    cpl_free(nbimg);
    cpl_free(shiftmap);
    return NULL;
  }
  distor_h2_raw = cpl_calloc(nbregion, sizeof(cpl_image *));
  if (distor_h2_raw == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
    cpl_free(distor_h1_raw);
    cpl_free(nbimg);
    cpl_free(shiftmap);
    return NULL;
  }
  distor_h3_raw = cpl_calloc(nbregion, sizeof(cpl_image *));
  if (distor_h3_raw == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
    cpl_free(distor_h1_raw);
    cpl_free(distor_h2_raw);
    cpl_free(nbimg);
    cpl_free(shiftmap);
    return NULL;
  }
  spectra_raw = cpl_calloc(nbregion, sizeof(cpl_image *));
  if (spectra_raw == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
    cpl_free(distor_h1_raw);
    cpl_free(distor_h2_raw);
    cpl_free(distor_h3_raw);
    cpl_free(nbimg);
    cpl_free(shiftmap);
    return NULL;
  }
  distor_background = cpl_calloc(nbregion, sizeof(cpl_image *));
  if (distor_background == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
    cpl_free(distor_h1_raw);
    cpl_free(distor_h2_raw);
    cpl_free(distor_h3_raw);
    cpl_free(spectra_raw);
    cpl_free(nbimg);
    cpl_free(shiftmap);
    return NULL;
  }
  spectra_background = cpl_calloc(nbregion, sizeof(cpl_image *));
  if (spectra_background == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
    cpl_free(distor_h1_raw);
    cpl_free(distor_h2_raw);
    cpl_free(distor_h3_raw);
    cpl_free(spectra_raw);
    cpl_free(distor_background);
    cpl_free(nbimg);
    cpl_free(shiftmap);
    return NULL;
  }

  for(i=0; i<nbregion; i++) {
    distor_h1_raw[i]= cpl_image_new(
                          distor[0]->imgdet->list_region[i]->naxis[0],
                          distor[0]->imgdet->list_region[i]->naxis[1],
                          cpl_image_get_type(distor[0]->imgdata->
                              list_frame[0]->list_subwin[i]->imgreg[0]));
    if (distor_h1_raw[i] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
    distor_h2_raw[i]= cpl_image_new(
                          distor[0]->imgdet->list_region[i]->naxis[0],
                          distor[0]->imgdet->list_region[i]->naxis[1],
                          cpl_image_get_type(distor[0]->imgdata->
                              list_frame[0]->list_subwin[i]->imgreg[0]));
    if (distor_h2_raw[i] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
    distor_h3_raw[i]= cpl_image_new(
                          distor[0]->imgdet->list_region[i]->naxis[0],
                          distor[0]->imgdet->list_region[i]->naxis[1],
                          cpl_image_get_type(distor[0]->imgdata->
                              list_frame[0]->list_subwin[i]->imgreg[0]));
    if (distor_h3_raw[i] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
    spectra_raw[i]= cpl_image_new(
                      spectra[0]->imgdet->list_region[i]->naxis[0], 
                      spectra[0]->imgdet->list_region[i]->naxis[1],
                      cpl_image_get_type(spectra[0]->imgdata->
                          list_frame[0]->list_subwin[i]->imgreg[0]));
    if (spectra_raw[i] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
  
    distor_background[i]= 
          cpl_image_new(distor_hotdark[0]->imgdet->list_region[i]->naxis[0], 
                        distor_hotdark[0]->imgdet->list_region[i]->naxis[1],
                        cpl_image_get_type(distor_hotdark[0]->imgdata->
                            list_frame[0]->list_subwin[i]->imgreg[0]));
    if (distor_background[i] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }  
    spectra_background[i]= 
          cpl_image_new(spectra_hotdark[0]->imgdet->list_region[i]->naxis[0],
                        spectra_hotdark[0]->imgdet->list_region[i]->naxis[1],
                        cpl_image_get_type(spectra_hotdark[0]->imgdata->
                            list_frame[0]->list_subwin[i]->imgreg[0]));
    if (spectra_background[i] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }

  }

// Others memory allocation and initialization using nbregion 
  for(i=0; i<nbregion; i++) {
    (*shiftmap).list_shiftcoefx[i] = cpl_malloc(sizeof(mat_shiftcoef));
    if (shiftmap->list_shiftcoefx[i] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_shiftcoef");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
    (*shiftmap).list_shiftcoefy[i] = cpl_malloc(sizeof(mat_shiftcoef));
    if (shiftmap->list_shiftcoefy[i] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_shiftcoef");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }

    nbimg[i] = 0;

    nb_row[i]= cpl_image_get_size_y(distor_h1_raw[i]);
    nb_column[i]= cpl_image_get_size_x(distor_h1_raw[i]);

    // this variable contains the coordinate in the spatial direction 
    // of the maximum for each pinehole (3), for each spectral channel (nb_row) 
    // and for each region.
    distor_index_max[i] = cpl_matrix_new(nb_row[i], 3);
    if (distor_index_max[i] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }

    (*shiftmap).list_shiftcoefx[i]->numregion = i;
    (*shiftmap).list_shiftcoefx[i]->numdetector = 
        (*shiftmap).imgdet->list_region[i]->numdetector;
    (*shiftmap).list_shiftcoefx[i]->order = 3;
    (*shiftmap).list_shiftcoefx[i]->nlig = nb_row[i];
    (*shiftmap).list_shiftcoefx[i]->coefdistor = 
        cpl_matrix_new(3, nb_row[i]);
    if (shiftmap->list_shiftcoefx[i]->coefdistor == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
    (*shiftmap).list_shiftcoefx[i]->errdistor = 
        cpl_matrix_new(3, nb_row[i]);
    if (shiftmap->list_shiftcoefx[i]->errdistor == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
    // Inital value of the coefficients (a,b,c)
    // the coefficients are used in this way X=a+bx+cx**2
    // where x is the intial coordinates et X is the corrected ones
    for(j=0; j<shiftmap->list_shiftcoefx[i]->nlig; j++) {
      cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor, 0, j, 0);
      cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor, 1, j, 1);
      cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor, 2, j, 0);
    }


    (*shiftmap).list_shiftcoefy[i]->numregion = i;
    (*shiftmap).list_shiftcoefy[i]->numdetector = 
        (*shiftmap).imgdet->list_region[i]->numdetector;
    (*shiftmap).list_shiftcoefy[i]->order = 3;
    (*shiftmap).list_shiftcoefy[i]->nlig = nb_column[i];
    (*shiftmap).list_shiftcoefy[i]->coefdistor = 
        cpl_matrix_new(3, nb_column[i]);
    if (shiftmap->list_shiftcoefy[i]->coefdistor == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
    (*shiftmap).list_shiftcoefy[i]->errdistor = 
        cpl_matrix_new(3, nb_column[i]);
    if (shiftmap->list_shiftcoefy[i]->errdistor == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
    // Inital value of the coefficients (a,b,c)
    // the coefficients are used in this way Y=a+by+cy**2
    // where y is the intial coordinates et Y is the corrected ones
    for(j=0; j<shiftmap->list_shiftcoefy[i]->nlig; j++) {
      cpl_matrix_set(shiftmap->list_shiftcoefy[i]->coefdistor, 0, j, 0);
      cpl_matrix_set(shiftmap->list_shiftcoefy[i]->coefdistor, 1, j, 1);
      cpl_matrix_set(shiftmap->list_shiftcoefy[i]->coefdistor, 2, j, 0);
    }
  }
  coeff= cpl_matrix_new(3, 3);
  if (coeff == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
    cpl_free(distor_h1_raw);
    cpl_free(distor_h2_raw);
    cpl_free(distor_h3_raw);
    cpl_free(spectra_raw);
    cpl_free(distor_background);
    cpl_free(spectra_background);
    cpl_free(nbimg);
    cpl_free(shiftmap);
    return NULL;
  }
  rhs= cpl_matrix_new(3, 1);
  if (rhs == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
    cpl_matrix_delete(coeff);
    cpl_free(distor_h1_raw);
    cpl_free(distor_h2_raw);
    cpl_free(distor_h3_raw);
    cpl_free(spectra_raw);
    cpl_free(distor_background);
    cpl_free(spectra_background);
    cpl_free(nbimg);
    cpl_free(shiftmap);
    return NULL;
  }
// Copy the images from mat_gendata parameters
  // Hole 1 : Add the distor raw data for each region and each frame
  k=0;
  while(k< nbframes[0]) {
    for(i=0; i<nbregion; i++) {
      cpl_image_add(distor_h1_raw[i],distor[k]->imgdata
		    ->list_frame[0]->list_subwin[i]->imgreg[0]);
      nbimg[i]++;
    }
    k++;
  }
  // Make the average image from the images added for each region
  for(i=0; i<nbregion; i++) {
    if (nbimg[i] == 0) {
      cpl_msg_error(cpl_func,"0 images found for DISTOR HOLE 1");
      cpl_matrix_delete(rhs);
      cpl_matrix_delete(coeff);
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
    cpl_image_multiply_scalar(distor_h1_raw[i], 1/nbimg[i]);
    nbimg[i]=0;
  }

  // Hole 2 : Add the distor raw data for each region and each frame
  k=nbframes[0];
  while(k< nbframes[0] * 2) {
    for(i=0; i<nbregion; i++) {
      cpl_image_add(distor_h2_raw[i],distor[k]->imgdata
		    ->list_frame[0]->list_subwin[i]->imgreg[0]);
      nbimg[i]++;
    }
    k++;
  }
  // Make the average image from the images added for each region
  for(i=0; i<nbregion; i++) {
    if (nbimg[i] == 0) {
      cpl_msg_error(cpl_func,"0 images found for DISTOR HOLE 2");
      cpl_matrix_delete(rhs);
      cpl_matrix_delete(coeff);
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
    cpl_image_multiply_scalar(distor_h2_raw[i], 1/nbimg[i]);
    nbimg[i]=0;
  }

  // Hole 3 : Add the distor raw data for each region and each frame
  k=nbframes[0] * 2;
  while(k< nbframes[0] * 3) {
    for(i=0; i<nbregion; i++) {
      cpl_image_add(distor_h3_raw[i],distor[k]->imgdata
		    ->list_frame[0]->list_subwin[i]->imgreg[0]);
      nbimg[i]++;
    }
    k++;
  }
  // Make the average image from the images added for each region
  for(i=0; i<nbregion; i++) {
    if (nbimg[i] == 0) {
      cpl_msg_error(cpl_func,"0 images found for DISTOR HOLE 3");
      cpl_matrix_delete(rhs);
      cpl_matrix_delete(coeff);
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
    cpl_image_multiply_scalar(distor_h3_raw[i], 1/nbimg[i]);
    nbimg[i]=0;
  }


  // Add the hotdark (background) data for each region and each frame
  k=0;
  while(k< nbframes[1]) {
    for(i=0; i<nbregion; i++) {
      cpl_image_add(distor_background[i],distor_hotdark[k]
		    ->imgdata->list_frame[0]->list_subwin[i]->imgreg[0]);
      nbimg[i]++;
    }
    k++;
  }
  // Make the average image from the images added for each region
  for(i=0; i<nbregion; i++) {
    if (nbimg[i] == 0) {
      cpl_msg_error(cpl_func,"0 images found for DISTOR BACKGROUND");
      cpl_matrix_delete(rhs);
      cpl_matrix_delete(coeff);
      cpl_free(distor_h1_raw);
      cpl_free(distor_h2_raw);
      cpl_free(distor_h3_raw);
      cpl_free(spectra_raw);
      cpl_free(distor_background);
      cpl_free(spectra_background);
      cpl_free(nbimg);
      cpl_free(shiftmap);
      return NULL;
    }
    cpl_image_multiply_scalar(distor_background[i], 1/nbimg[i]);
    nbimg[i]=0;
  }
 

// Substract the laboratory background to the raw data (DISTOR)
  for(i=0; i<nbregion; i++) {
    cpl_image_subtract(distor_h1_raw[i], distor_background[i]);
    cpl_image_subtract(distor_h2_raw[i], distor_background[i]);
    cpl_image_subtract(distor_h3_raw[i], distor_background[i]);
  }

  if ( cpl_parameter_get_bool(debug) ) {
    cpl_image_save(distor_h1_raw[2], "image1.fits",
		   CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_CREATE);
    cpl_image_save(distor_h2_raw[2], "image2.fits",
		   CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_CREATE);
    cpl_image_save(distor_h3_raw[2], "image3.fits",
		   CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_CREATE);
  }


// Find and fit the index of the maximum value for each column and each hole
  poserravg=calloc(nbregion,sizeof(double));
  // Find each key position in each row depending on the band and 
  // the resolution


  for(i=0; i<nbregion; i++) {
    // Set the coefficients which are used to fit the results
    params[0]=0.0;
    params[1]=0.0;
    params[2]=10.0;
    // Fit each maximum in each column
    for(j=0; j< nb_row[i]; j++) {
      

      // Fit the indexes of the max values in this hole
      for(k=0; k<3; k++) {
        if(k == 0) {
          row= cpl_vector_new_from_image_row(distor_h1_raw[i], j+1);
        }
        else if(k == 1) {
          row= cpl_vector_new_from_image_row(distor_h2_raw[i], j+1);
        }
        else {
          row= cpl_vector_new_from_image_row(distor_h3_raw[i], j+1);
        }
	if (row == NULL) {
	  cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
	  return NULL;
	}


	//!!!!!!!!!!!!!!!!!!!!!!
	/* if (cpl_vector_get_size(row) > 100) { */
	/*   boxsz=30; */
	/* } else { */
	/*   boxsz=6; */
	/* } */
	/* polysmooth= cpl_polynomial_new(1); */
	/* if (polysmooth == NULL) { */
	/*   cpl_msg_error(cpl_func,"could not allocate memory for cpl_polynomial"); */
	/*   return NULL; */
	/* } */

	/* samppos= cpl_matrix_new(1, boxsz); */
	/* if (samppos == NULL) { */
	/*   cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix"); */
	/*   return NULL; */
	/* } */
	/* fitvals2=cpl_vector_new(boxsz); */
	/* if (fitvals2 == NULL) { */
	/*   cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector"); */
	/*   return NULL; */
	/* } */
	/* sampsym = CPL_TRUE; */
	/* mindeg= 0; */
	/* maxdeg= 2; */
	/* deg1=1; */
	/* deg2=2; */
	//!!!!!!!!!!!!!!!!!!!!!!





        // Get the max value corresponding to the amplitude of 
	// the gaussian curve
        params[0]=cpl_vector_get_max(row);

	valmax=0.;
        // Find the indexes of the maximum
        for(l=0; l< cpl_vector_get_size(row); l++) {
          if(cpl_vector_get(row, l) > valmax) {
            valmax=cpl_vector_get(row, l);
            params[1]=l;
          }
        }


	//!!!!!!!!!!!!!!!!!!!!!!!

	/* // Fit a second order polynomial around imx */
	/* // and determine the maximum/minimum of the polynomial */
	/* for(l=0; l<boxsz;l++) { */
	/*     cpl_matrix_set(samppos, 0, l, l); */
	/*     cpl_vector_set(fitvals2, l, cpl_vector_get(row,params[1]-(boxsz/2)+l)); */
	/* } */
	/* if (cpl_polynomial_fit(polysmooth, samppos, &sampsym, fitvals2, NULL, */
	/* 		       CPL_FALSE, &mindeg, &maxdeg) != CPL_ERROR_NONE) { */
	/*   cpl_msg_error(cpl_func,"could not fir a polynomial"); */
	/*   return NULL; */
	/* } */
	/* // we determine the new coordinate of teh spectral feature with */
	/* // the coordinates of the max/min of the second order polynomial */
	/* res=params[1]-(boxsz/2) + (-1* cpl_polynomial_get_coeff(polysmooth,&deg1)/ */
	/* (2.*cpl_polynomial_get_coeff(polysmooth,&deg2))); */

	//!!!!!!!!!!!!!!!!!!!!!!!









        // Fit the indexes of the maximum
	vector= gsl_vector_alloc(nb_column[i]);
        for(l=0; l<nb_column[i]; l++) {
          gsl_vector_set(vector, l, (double)cpl_vector_get(row, l));
        }
	res = mat_fit_gauss(nb_column[i], vector, params, &poserr);
	if (res < 0){
	  cpl_msg_warning(cpl_func,"could not fit a gaussian - reg:%d - row:%d - hole:%d",i,j,k);
	  /* return NULL; */
	}


        // Store the indexes found in a matrix
        cpl_matrix_set(distor_index_max[i], j, k, res);
	cpl_vector_delete(row);
        gsl_vector_free(vector);


	//!!!!!!!!!!!!!!!!!!!!!!!
	/* cpl_matrix_delete(samppos); */
        /* cpl_vector_delete(fitvals2); */
        /* cpl_polynomial_delete(polysmooth); */
	//!!!!!!!!!!!!!!!!!!!!!!!
      }
    }
    poserravg[i]/=(nb_row[i]*3);
  }
		   

  // The estimation of the maximum coordinates in the spatial direction
  // could be noisy. So we decided to smooth the variation of theses
  // coordinates wrt the spectral channels.
  // Smooth the list of pixel of hole centers and remove outliers
  if ( cpl_parameter_get_bool(debug) ) {
    file = fopen("row.dat", "w+");
  }
  sampsym = CPL_TRUE;
  mindeg= 0;
  maxdeg= 2;
  for(i=0; i<nbregion; i++) {
    polysmooth= cpl_polynomial_new(1);
    if (polysmooth == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_polynomial");
      return NULL;
    }

    outliers=cpl_vector_new(nb_row[i]);
    if (outliers == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
      return NULL;
    }

    residual=cpl_vector_new(nb_row[i]);
    if (residual == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
      cpl_vector_delete(outliers);
      return NULL;
    }
  
    for(k=0; k < 3; k++) {

      for(l=0; l<nb_row[i];l++) {
    	cpl_vector_set(outliers, l, 0);
      }
      nboutliers=0;
      if(!strcmp(chipName, "AQUARIUS") && !strcmp(resolution, "LOW")) { 
	nboutliers=110;
	for(l=0; l<110;l++) {
	  cpl_vector_set(outliers, l, 2);
	}
      }
      nboutprec=-1;
      cpt=0;
      // iterations until no new outliers or iteration max achieved
      while (nboutliers > nboutprec || cpt < cptmax) {
    	samppos= cpl_matrix_new(1,nb_row[i]-nboutliers);
    	if (samppos == NULL) {
    	  cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
    	  return NULL;
    	}

    	fitvals2=cpl_vector_new(nb_row[i]-nboutliers);
    	if (fitvals2 == NULL) {
    	  cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
	  cpl_matrix_delete(samppos);
    	  return NULL;
    	}
    	// Fit a second order polynomial to the maximum coordinates wrt the 
	// spectral channels
    	j=0;
    	for(l=0; l<nb_row[i];l++) {
    	  if (cpl_vector_get(outliers,l) == 0) {
    	    cpl_matrix_set(samppos, 0, j, l);
    	    cpl_vector_set(fitvals2, j, cpl_matrix_get(distor_index_max[i], l, k));
    	    j++;
    	  }
    	}
    	if (cpl_polynomial_fit(polysmooth, samppos, &sampsym, fitvals2, NULL,
    			       CPL_FALSE, &mindeg, &maxdeg) != CPL_ERROR_NONE) {
    	  cpl_msg_error(cpl_func,"could not fit a polynomial");
    	  return NULL;
    	}
    	if (cpl_vector_fill_polynomial_fit_residual(residual, fitvals2, NULL,
    						    polysmooth, samppos, &chi2) != CPL_ERROR_NONE) {
    	   cpl_msg_error(cpl_func,"could not fill a vector with the residuals of a polynomial fit");
    	  return NULL;
    	}
	// Determination of the outliers wrt the fitted second order polynomial
    	nboutprec=nboutliers;
    	nboutliers=0;
    	for(l=0; l<nb_row[i];l++) {
    	  if (fabs(cpl_matrix_get(distor_index_max[i], l, k)-
    		   cpl_polynomial_eval_1d(polysmooth,(double) l,NULL)) > 3*sqrt(chi2) ) {
    	    cpl_vector_set(outliers, l, 1);
    	    nboutliers++;
    	  } else {
	    if (cpl_vector_get(outliers, l) != 2) {
	      cpl_vector_set(outliers, l, 0);
	    } else {
	      nboutliers++;
	    }
    	  }
    	}
    	cpl_matrix_delete(samppos);
    	cpl_vector_delete(fitvals2);
    	cpt++;
      }
      // when no more outliers, we take the second order polynomial as the 
      // new values for the maximum coodinates.
      for(l=0; l<nb_row[i];l++) {
      	  cpl_matrix_set(distor_index_max[i], l, k,
      		       cpl_polynomial_eval_1d(polysmooth,(double) l,NULL));
      }
    }

    if ( cpl_parameter_get_bool(debug) ) {
      for(l=0; l<nb_row[i];l++) {
	fprintf(file, "%d\t", l); 
	for(k=0; k< 3; k++) {
	  fprintf(file, "%lf", cpl_matrix_get(distor_index_max[i], l, k));
	  if(k< 2) {
	    fprintf(file, "\t");
	  }
	  else {
	    fprintf(file, "\n");
	  }
	}
      }
    }
    cpl_polynomial_delete(polysmooth);
    cpl_vector_delete(residual);
    cpl_vector_delete(outliers);
  }
  if ( cpl_parameter_get_bool(debug) ) {
    fclose(file);
  }

// Determine the distortion coefficients of the interferometric channel and 
// (up to) four photometric channels
// Store these results in mat_shiftmap structure
// Compute the distorsion coefficients in spatial direction (DISTOR_IMAGES)
  for(i=0; i<nbregion; i++) {  
    // For each spectral channel
    for(j=0; j< nb_row[i]; j++) {
      // For each reference coordinate
      for(k=0; k< (*shiftmap).list_shiftcoefx[i]->order; k++) {
	// coeff is the design matrix of the least square estimation.
	// It is a 3x3 matrix since we have 3 measurements (coordinates of
	// the maximum in the spatial direction for the 3 pineholes) and 
	// we have 3 parameters (X=a+bx+cx**2)
        cpl_matrix_set(coeff, k, 0, 1);
        val= cpl_matrix_get(distor_index_max[i], j, k);
        cpl_matrix_set(coeff, k, 1, val);
        cpl_matrix_set(coeff, k, 2, pow(val, 2));

	// We choose the central spectral channel (nb_row/2)
	// as the reference channel. The coordiantes in the 
	// spatial direction of this channel will not change 
	// when the distrosion will be corrected.
        val= cpl_matrix_get(distor_index_max[i], nb_row[i]/2, k);
        cpl_matrix_set(rhs, k, 0, val);
      }
      // Solve the linear system to find the distorsion coefficients
      // and append the result to the matrix
      coefftrans=cpl_matrix_transpose_create(coeff);
      if (coefftrans == NULL) {
	cpl_msg_error(cpl_func,"could not tranpose a cpl_matrix");
	return NULL;
      }
      normal=cpl_matrix_product_create(coefftrans,coeff);
      if (normal == NULL) {
	cpl_msg_error(cpl_func,"could not multply 2 cpl_matrix");
	return NULL;
      }
      invnormal=cpl_matrix_invert_create(normal);
      if (invnormal == NULL) {
	cpl_msg_error(cpl_func,"could not invert a cpl_matrix");
	return NULL;
      }
      cpl_matrix_delete(normal);
      matrix_temp=cpl_matrix_product_create(coefftrans,rhs);
      if (matrix_temp == NULL) {
	cpl_msg_error(cpl_func,"could not multply 2 cpl_matrix");
	return NULL;
      }
      cpl_matrix_delete(coefftrans);
      // solution of the coefficients (a,b,c) of the least square estimation
      sol=cpl_matrix_product_create(invnormal,matrix_temp);
      if (sol == NULL) {
	cpl_msg_error(cpl_func,"could not multply 2 cpl_matrix");
	return NULL;
      }
      cpl_matrix_delete(matrix_temp);
      cpl_matrix_copy((*shiftmap).list_shiftcoefx[i]->coefdistor,
                        sol, 0, j);





      if(!strcmp(chipName, "AQUARIUS") && !strcmp(resolution, "HIGH")) { 
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,0,j,0.);
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,1,j,1.);
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,2,j,0.);
      }
      if(!strcmp(chipName, "AQUARIUS") && !strcmp(resolution, "LOW")) { 
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,0,j,0.);
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,1,j,1.);
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,2,j,0.);
      }
      if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "LOW")) { 
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,0,j,0.);
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,1,j,1.);
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,2,j,0.);
      }
      if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "MED")) { 
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,0,j,0.);
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,1,j,1.);
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,2,j,0.);
      }
      if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "HIGH")) { 
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,0,j,0.);
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,1,j,1.);
	cpl_matrix_set(shiftmap->list_shiftcoefx[i]->coefdistor,2,j,0.);
      }






      for(k=0;k<3;k++) {
	cpl_matrix_set((*shiftmap).list_shiftcoefx[i]->errdistor,
		       k,j,ERR_MAX_PIX*sqrt(cpl_matrix_get(invnormal,k,k)));
	//		       k,j,poserravg[i]*sqrt(cpl_matrix_get(invnormal,k,k)));
      }
      cpl_matrix_delete(invnormal);
      cpl_matrix_delete(sol);
    }
  }
  cpl_matrix_delete(coeff);
  cpl_matrix_delete(rhs);
  for(i=0; i<nbregion; i++) {
    cpl_matrix_delete(distor_index_max[i]);
  }

  free(poserravg);

//////////////////////////////////////////////////////////////////////////
// the distrosion coefficients estimation in the spatial direction is done 
  cpl_msg_info(cpl_func,
	       "The distrosion coefficients estimation in the spatial direction is done");


  spectra_shift= cpl_calloc(nbframes[2], sizeof(mat_gendata *));
  if (spectra_shift == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for mat_gendata");
    return NULL;
  }
  spectra_hd_shift= cpl_calloc(nbframes[3], sizeof(mat_gendata *));
  if (spectra_hd_shift == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for mat_gendata");
    cpl_free(spectra_shift);
    return NULL;
  }
  // Apply the distorsion coefficients to SPECTRA_IMAGES and SPECTRA_HOTDARK
  for(i=0; i< nbframes[2]; i++) {
    spectra_shift[i]= mat_apply_shift(spectra[i], shiftmap, izero); 
    if (spectra_shift[i] == NULL) {
      cpl_msg_error(cpl_func,"could not apply the shift correction");
      cpl_free(spectra_shift);
      cpl_free(spectra_hd_shift);
      return NULL;
    }
  }
  for(i=0; i< nbframes[3]; i++) {
    spectra_hd_shift[i]= mat_apply_shift(spectra_hotdark[i], shiftmap, izero); 
    if (spectra_hd_shift[i] == NULL) {
      cpl_msg_error(cpl_func,"could not apply the shift correction");
      cpl_free(spectra_shift);
      cpl_free(spectra_hd_shift);
      return NULL;
    }
  }
  cpl_msg_info(cpl_func,
	       "Plastic foils frames: Correction of the distrosion in the spatial direction done");


  // Add the spectra raw data for each region and each frame
  k=0;
  while(k< nbframes[2]) {
    for(i=0; i<nbregion; i++) {
      cpl_image_add(spectra_raw[i],spectra_shift[k]->imgdata
		    ->list_frame[0]->list_subwin[i]->imgreg[0]);
      nbimg[i]++;
    }
    k++;
  }

  // Make the average image from the images added for each region
  for(i=0; i<nbregion; i++) {
    if (nbimg[i] == 0) {
      cpl_msg_error(cpl_func,"0 images found for SPECTRA");
      cpl_free(spectra_hd_shift);
      cpl_free(spectra_shift);
      return NULL;
    }
    cpl_image_multiply_scalar(spectra_raw[i], 1/nbimg[i]);
    nbimg[i]=0;
  }

  // Add the spectra (background) data for each region and each frame
  k=0;
  while(k< nbframes[3]) {
    for(i=0; i<nbregion; i++) {
      cpl_image_add(spectra_background[i],spectra_hd_shift[k]
		    ->imgdata->list_frame[0]->list_subwin[i]->imgreg[0]);
      nbimg[i]++;
    }
    k++;
  }
  // Make the average image from the images added for each region
  for(i=0; i<nbregion; i++) {
    if (nbimg[i] == 0) {
      cpl_msg_error(cpl_func,"0 images found for SPECTRA BACKGROUND");
      cpl_free(spectra_shift);
      cpl_free(spectra_hd_shift);
      return NULL;
    }
    cpl_image_multiply_scalar(spectra_background[i], 1/nbimg[i]);
    nbimg[i]=0;
  }

  if(!strcmp(chipName, "HAWAII-2RG")) {
    // Divide the laboratory background to the raw data (SPECTRA)
    if (!strcmp(resolution, "MED") ||!strcmp(resolution, "HIGH") ) {
      for(i=0; i<nbregion; i++) {
	cpl_image_divide(spectra_raw[i], spectra_background[i]);
      }
    }
  }

  if ( cpl_parameter_get_bool(debug) ) {
    foo=cpl_image_new(cpl_image_get_size_x(spectra_raw[0])+
		      cpl_image_get_size_x(spectra_raw[1])+
		      cpl_image_get_size_x(spectra_raw[2])+
		      cpl_image_get_size_x(spectra_raw[3])+
		      cpl_image_get_size_x(spectra_raw[4]),
		      cpl_image_get_size_y(spectra_raw[0]),
		      cpl_image_get_type(spectra_raw[0]));

    for(i=1;i<=cpl_image_get_size_x(spectra_raw[0]);i++) {
      for(j=1;j<=cpl_image_get_size_y(spectra_raw[0]);j++) {
	cpl_image_set(foo,i,j,cpl_image_get(spectra_raw[0],i,j,&out));
      }
    }
    for(i=1;i<=cpl_image_get_size_x(spectra_raw[1]);i++) {
      for(j=1;j<=cpl_image_get_size_y(spectra_raw[1]);j++) {
	cpl_image_set(foo,cpl_image_get_size_x(spectra_raw[0])+
		      i,j,cpl_image_get(spectra_raw[1],i,j,&out));
      }
    }
    for(i=1;i<=cpl_image_get_size_x(spectra_raw[2]);i++) {
      for(j=1;j<=cpl_image_get_size_y(spectra_raw[2]);j++) {
	cpl_image_set(foo,cpl_image_get_size_x(spectra_raw[0])+
		      cpl_image_get_size_x(spectra_raw[1])+
		      i,j,cpl_image_get(spectra_raw[2],i,j,&out));
      }
    }
    for(i=1;i<=cpl_image_get_size_x(spectra_raw[3]);i++) {
      for(j=1;j<=cpl_image_get_size_y(spectra_raw[3]);j++) {
	cpl_image_set(foo,cpl_image_get_size_x(spectra_raw[0])+
		      cpl_image_get_size_x(spectra_raw[1])+
		      cpl_image_get_size_x(spectra_raw[2])+
		      i,j,cpl_image_get(spectra_raw[3],i,j,&out));
      }
    }
    for(i=1;i<=cpl_image_get_size_x(spectra_raw[4]);i++) {
      for(j=1;j<=cpl_image_get_size_y(spectra_raw[4]);j++) {
	cpl_image_set(foo,cpl_image_get_size_x(spectra_raw[0])+
		      cpl_image_get_size_x(spectra_raw[1])+
		      cpl_image_get_size_x(spectra_raw[2])+
		      cpl_image_get_size_x(spectra_raw[3])+
		      i,j,cpl_image_get(spectra_raw[4],i,j,&out));
      }
    }
    cpl_image_save(foo, "imageraw.fits",
		   CPL_BPP_IEEE_FLOAT, spectra[0]->keywords, CPL_IO_CREATE); 
        
    cpl_image_delete(foo);
  }


  for(i=0; i< nbframes[2]; i++) {
    mat_gendata_delete(spectra_shift[i]);
  }
  for(i=0; i< nbframes[3]; i++) {
    mat_gendata_delete(spectra_hd_shift[i]);
  }
  cpl_free(spectra_shift);
  cpl_free(spectra_hd_shift);



 

  
  //////////////////////////////////////////////////////////
// Compute the distorsion coefficients in spectral direction

  int corner=shiftmap->imgdet->list_region[0]->corner[1];
  /* // Get the band and the resolution */
  /* chipName= (char *)cpl_propertylist_get_string(spectra[0]->keywords, */
  /*                                                 "ESO DET CHIP NAME"); */
  /* if (chipName == NULL) { */
  /*   cpl_msg_error(cpl_func,"No ESO DET CHIP NAME keyword in the primary header of spectra[0]"); */
  /*   return NULL; */
  /* } */

  // a: band N
  if(!strcmp(chipName, "AQUARIUS")) {
    /* resolution= (char *)cpl_propertylist_get_string(spectra[0]->keywords, */
    /*                                                 "ESO INS DIN ID"); */
    /* if (resolution == NULL) { */
    /*   cpl_msg_error(cpl_func,"No ESO INS DIN ID keyword in the primary header of spectra[0]"); */
    /*   return NULL; */
    /* } */
    // a1: band N, low resolution
    if(!strcmp(resolution, "LOW")) {
      nbmaxmin= 3;
      typemaxmin= cpl_array_new(nbmaxmin, CPL_TYPE_INT);
      if (typemaxmin == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_array");
	return NULL;
      }

      cpl_array_set(typemaxmin, 0, 0);
      cpl_array_set(typemaxmin, 1, 0);
      cpl_array_set(typemaxmin, 2, 0);
      
      boundaries= cpl_calloc(2*nbmaxmin, sizeof(int));
      if (boundaries == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for int");
	return NULL;
      }
      boundaries[0] = 50+(385-corner);
      boundaries[1] = 100+(385-corner);
      boundaries[2] = 145+(385-corner);
      boundaries[3] = 170+(385-corner);
      boundaries[4] = 170+(385-corner);
      boundaries[5] = 250+(385-corner);
      fitvals= cpl_vector_new(nbmaxmin);
      if (fitvals == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
	cpl_free(boundaries);
	return NULL;
      }
      /* cpl_vector_set(fitvals, 0, 8.7); */
      /* cpl_vector_set(fitvals, 1, 11.02); */
      /* cpl_vector_set(fitvals, 2, 12.76); */
      /* cpl_vector_set(fitvals, 0, 8.59); */
      /* cpl_vector_set(fitvals, 1, 11.06); */
      /* cpl_vector_set(fitvals, 2, 12.74); */
      cpl_vector_set(fitvals, 0, 8.6549);
      cpl_vector_set(fitvals, 1, 11.0336);
      cpl_vector_set(fitvals, 2, 12.7002);
      boxsz=5;
    }
    // a2: band N, high resolution
    else if(!strcmp(resolution, "HIGH")) {
      nbmaxmin= 5;
      typemaxmin= cpl_array_new(nbmaxmin, CPL_TYPE_INT);
      if (typemaxmin == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_array");
	return NULL;
      }
      cpl_array_set(typemaxmin, 0, 0);
      cpl_array_set(typemaxmin, 1, 1);
      cpl_array_set(typemaxmin, 2, 0);
      cpl_array_set(typemaxmin, 3, 0);
      cpl_array_set(typemaxmin, 4, 1);
      boundaries= cpl_calloc(2*nbmaxmin, sizeof(int));
      if (boundaries == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for int");
	return NULL;
      }
      boundaries[0] = 120+(53-corner);
      boundaries[1] = 220+(53-corner);
      boundaries[2] = 300+(53-corner);
      boundaries[3] = 400+(53-corner);
      boundaries[4] = 480+(53-corner);
      boundaries[5] = 520+(53-corner);
      boundaries[6] = 600+(53-corner);
      boundaries[7] = 650+(53-corner);
      boundaries[8] = 840+(53-corner);
      boundaries[9] = 880+(53-corner);
      fitvals= cpl_vector_new(nbmaxmin);
      if (fitvals == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
	cpl_free(boundaries);
	return NULL;
      }
      cpl_vector_set(fitvals, 0, 8.7);
      cpl_vector_set(fitvals, 1, 9.86);
      cpl_vector_set(fitvals, 2, 10.80);
      cpl_vector_set(fitvals, 3, 11.53);
      cpl_vector_set(fitvals, 4, 13.0);
      boxsz=20;
    }
    else {
      cpl_msg_error(cpl_func, "AQUARIUS : Can't find the resolution.");
      return NULL;
    }
  }
  // b: band L
  else if(!strcmp(chipName, "HAWAII-2RG")) {
    /* resolution= (char *)cpl_propertylist_get_string(spectra[0]->keywords, */
    /*                                                 "ESO INS DIL ID"); */
    // b1: band L, low resolution
    if(!strcmp(resolution, "LOW")) {
      nbmaxmin= 3;
      typemaxmin= cpl_array_new(nbmaxmin, CPL_TYPE_INT);
      if (typemaxmin == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_array");
	return NULL;
      }
      cpl_array_set(typemaxmin, 0, 1);
      cpl_array_set(typemaxmin, 1, 0);
      cpl_array_set(typemaxmin, 2, 1);
      boundaries= cpl_calloc(2*nbmaxmin, sizeof(int));
      if (boundaries == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for int");
	return NULL;
      }
      if (mjdobs > 58810. && mjdobs < 59615)
	{
	  boundaries[0] = 5;
	  boundaries[1] = 20;
	  boundaries[2] = 5;
	  boundaries[3] = 20;
	  boundaries[4] = 80;
	  boundaries[5] = 100;
	}
      else if (mjdobs >= 59615.)
	{
	  boundaries[0] = 5;
	  boundaries[1] = 20;
	  boundaries[2] = 5;
	  boundaries[3] = 15;
	  boundaries[4] = 80;
	  boundaries[5] = 100;
	}
      else
	{
	  boundaries[0] = 75+(77-corner);
	  boundaries[1] = 95+(77-corner);
	  boundaries[2] = 100+(77-corner);
	  boundaries[3] = 130+(77-corner);
	  boundaries[4] = 115+(77-corner);
	  boundaries[5] = 135+(77-corner);
	}
      fitvals= cpl_vector_new(nbmaxmin);
      if (fitvals == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
	cpl_free(boundaries);
	return NULL;
      }
      cpl_vector_set(fitvals, 0, 3.43728);
      cpl_vector_set(fitvals, 1, 4.00231);
      cpl_vector_set(fitvals, 2, 4.49815);
      boxsz=5;
    }

    // b2: band L, medium resolution
    else if(!strcmp(resolution, "MED")) {
      nbmaxmin= 5;
      typemaxmin= cpl_array_new(nbmaxmin, CPL_TYPE_INT);
      if (typemaxmin == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_array");
	return NULL;
      }
      cpl_array_set(typemaxmin, 0, 1);
      cpl_array_set(typemaxmin, 1, 0);
      cpl_array_set(typemaxmin, 2, 0);
      cpl_array_set(typemaxmin, 3, 1);
      cpl_array_set(typemaxmin, 4, 1);
      boundaries= cpl_calloc(2*nbmaxmin, sizeof(int));
      if (boundaries == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for int");
	return NULL;
      }
      boundaries[0] = 220+(41-corner);
      boundaries[1] = 300+(41-corner);
      boundaries[2] = 800+(41-corner);
      boundaries[3] = 860+(41-corner);
      boundaries[4] = 1250+(41-corner);
      boundaries[5] = 1280+(41-corner);
      boundaries[6] = 1440+(41-corner);
      boundaries[7] = 1500+(41-corner);
      boundaries[8] = 1710+(41-corner);
      boundaries[9] = 1730+(41-corner);
      fitvals= cpl_vector_new(nbmaxmin);
      if (fitvals == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
	cpl_free(boundaries);
	return NULL;
      }
      if ( cpl_parameter_get_bool(obsCorrection) )
	{
	  cpl_vector_set(fitvals, 0, 4.85756359073307);//4.8581030328859205);
	  cpl_vector_set(fitvals, 1, 4.122120503896885);//4.122387479629846);
	  cpl_vector_set(fitvals, 2, 3.5004435635974236);//3.501619287541611);
	  cpl_vector_set(fitvals, 3, 3.2275969804640687);//3.229502299619296);
	  cpl_vector_set(fitvals, 4, 2.860891231296692);//2.86409588177675);
	}
      else
	{
	  cpl_vector_set(fitvals, 0, 4.87);
	  cpl_vector_set(fitvals, 1, 4.122);
	  cpl_vector_set(fitvals, 2, 3.5);
	  cpl_vector_set(fitvals, 3, 3.23);
	  cpl_vector_set(fitvals, 4, 2.87);
	}

      boxsz=20;
    }

    // b3: band L, high resolution
    else if(!strcmp(resolution, "HIGH")) {
      nbmaxmin= 4;
      typemaxmin= cpl_array_new(nbmaxmin, CPL_TYPE_INT);
      if (typemaxmin == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_array");
	return NULL;
      }
      cpl_array_set(typemaxmin, 0, 0);
      cpl_array_set(typemaxmin, 1, 0);
      cpl_array_set(typemaxmin, 2, 1);
      cpl_array_set(typemaxmin, 3, 1);
      boundaries= cpl_calloc(2*nbmaxmin, sizeof(int));
      if (boundaries == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for int");
	return NULL;
      }
      boundaries[0] = 100+(41-corner);
      boundaries[1] = 200+(41-corner);
      boundaries[2] = 960+(41-corner);
      boundaries[3] = 1050+(41-corner);
      /* boundaries[4] = 1100; */
      /* boundaries[5] = 1150; */
      boundaries[4] = 1320+(41-corner);
      boundaries[5] = 1380+(41-corner);
      boundaries[6] = 1800+(41-corner);
      boundaries[7] = 1880+(41-corner);
      /* boundaries[8] = 1790; */
      /* boundaries[9] = 1850; */
      /* boundaries[10] = 1910; */
      /* boundaries[11] = 1940; */
      /* boundaries[4] = 1800; */
      /* boundaries[5] = 1950; */
      fitvals= cpl_vector_new(nbmaxmin);
      if (fitvals == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
	cpl_free(boundaries);
	return NULL;
      }
      if ( cpl_parameter_get_bool(obsCorrection) )
	{
	  cpl_vector_set(fitvals, 0, 4.119578708925919);//4.117106211061543);
	  cpl_vector_set(fitvals, 1, 3.5053939205752114);//3.502545446109689);
	  cpl_vector_set(fitvals, 2, 3.244389116712771);//3.2292073564914516);
	  cpl_vector_set(fitvals, 3, 2.9069850642303035);//2.863466994992536);
	}
      else
	{
	  cpl_vector_set(fitvals, 0, 4.11);
	  cpl_vector_set(fitvals, 1, 3.5);
	  cpl_vector_set(fitvals, 2, 3.23);
	  cpl_vector_set(fitvals, 3, 2.87);
	}	  
      boxsz=20;
    }
    else {
      cpl_msg_error(cpl_func, "HAWAII2RG : Can't find the resolution.");
      return NULL;
    }
  }
  else {
    cpl_msg_error(cpl_func, "Chip name incorrect or missing.");
    return NULL;
  }
  // Find each key position in each row depending on the band and 
  // the resolution
  polysmooth= cpl_polynomial_new(1);
  if (polysmooth == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_polynomial");
    cpl_vector_delete(fitvals);
    cpl_free(boundaries);
    return NULL;
  }

  samppos= cpl_matrix_new(1, boxsz);
  if (samppos == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
    cpl_vector_delete(fitvals);
    cpl_free(boundaries);
    return NULL;
  }
  fitvals2=cpl_vector_new(boxsz);
  if (fitvals2 == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
    cpl_vector_delete(fitvals);
    cpl_matrix_delete(samppos);
    cpl_free(boundaries);
    return NULL;
  }
  sampsym = CPL_TRUE;
  mindeg= 0;
  maxdeg= 2;
  deg1=1;
  deg2=2;
  for(i=0; i<nbregion; i++) {
    distor_index_max[i] = cpl_matrix_new(nb_column[i], nbmaxmin);
    if (distor_index_max[i] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
      return NULL;
    }
    for(j=0; j< nb_column[i]; j++) {
      // Load one row of pixel values from the spectra_raw image
      row= cpl_vector_new_from_image_column(spectra_raw[i], j+1);
      if (row == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
	return NULL;
      }
      
      // we determine the coordinate in the spectral direction  of the 
      // spectral feature, as the pixel of the max or min.
      for(k=0; k < nbmaxmin; k++) {
        // Find a maximum
        if(cpl_array_get_int(typemaxmin, k, NULL) ==0) {
	  valmax=0.;
          for(l=boundaries[2*k]; l < boundaries[2*k+1]; l++) {
	    val_cur=cpl_vector_get(row, l);
	    if (val_cur > valmax) {
	      imx=l;
	      valmax=val_cur;
	    }
          }
       }
        // Find a minimum
        else {
	  valmin=1.E15;
          for(l=boundaries[2*k]; l < boundaries[2*k+1]; l++) {
	    if (cpl_vector_get(row, l) < valmin ) {
	      imx=l;
	      valmin=cpl_vector_get(row, l);
	    }
	  }
        }
	// If the coordinate of the spectral feature is too close 
	// to the edge of the region, we fix it wrt to boxsz.
	// Let's remember that boxsz is the size in pixels of the box 
	// in which we will fit a second order polynomial in order 
	// to improve the determination of the coordinates of the 
	// spectral features.
	if (imx < boxsz/2) {
	  imx=boxsz/2;
	}

	if (imx > nb_row[i]-boxsz/2-1) {
	  imx=nb_row[i]-boxsz/2-1;
	}

	// Fit a second order polynomial around imx 
	// and determine the maximum/minimum of the polynomial
	for(l=0; l<boxsz;l++) {
	    cpl_matrix_set(samppos, 0, l, l);
	    cpl_vector_set(fitvals2, l, cpl_vector_get(row,imx-(boxsz/2)+l));
	}
	if (cpl_polynomial_fit(polysmooth, samppos, &sampsym, fitvals2, NULL, 
			       CPL_FALSE, &mindeg, &maxdeg) != CPL_ERROR_NONE) {
	  cpl_msg_error(cpl_func,"could not fir a polynomial");
	  return NULL;
	}

	// we determine the new coordinate of teh spectral feature with 
	// the coordinates of the max/min of the second order polynomial
	if (cpl_polynomial_get_coeff(polysmooth,&deg2) != 0.) {
	  valmax=imx-(boxsz/2) + (-1* cpl_polynomial_get_coeff(polysmooth,&deg1)/
				  (2.*cpl_polynomial_get_coeff(polysmooth,&deg2)));
	  if (isnan(valmax) != 0) {
	    valmax=imx*1.0;
	  }
	} else {
	  valmax=imx*1.0;
	}

	// Set the position of the maximum or minimum found in a matrix
	cpl_matrix_set(distor_index_max[i], j, k, valmax);
      }
      cpl_vector_delete(row);
    }
  }
  cpl_matrix_delete(samppos);
  cpl_vector_delete(fitvals2);
  cpl_polynomial_delete(polysmooth);



  // Same procedure than for the distorsion coefficients in the spatial direction
  // Smooth the list of pixel of platic foils features and remove outliers
  if ( cpl_parameter_get_bool(debug) ) {
    file = fopen("column.dat", "w+");
  }
  sampsym = CPL_TRUE;
  mindeg= 0;
  maxdeg=2;
  if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "HIGH") ) { 
    maxdeg= 0;
  }
  if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "MED") ) { 
    maxdeg= 0;
  }
  if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "LOW") ) { 
    maxdeg= 0;
  }
  for(i=0; i<nbregion; i++) {
    polysmooth= cpl_polynomial_new(1);
    if (polysmooth == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_polynomial");
      return NULL;
    }

    outliers=cpl_vector_new(nb_column[i]);
    if (outliers == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
      return NULL;
    }
    residual=cpl_vector_new(nb_column[i]);
    if (residual == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
      cpl_vector_delete(outliers);
      return NULL;
    }
  
    for(k=0; k < nbmaxmin; k++) {

      for(l=0; l<nb_column[i];l++) {
	cpl_vector_set(outliers, l, 0);
      }
      nboutliers=0;
 
      if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "HIGH")) { 
	if (i==0 || i==1 || i==3 || i==4) {
	  /* nboutliers=nb_column[i]-5; */
	  /* for(l=0; l<78;l++) { */
	  /*   cpl_vector_set(outliers, l, 1); */
	  /* } */
	  /* for(l=83; l<nb_column[i];l++) { */
	  /*   cpl_vector_set(outliers, l, 1); */
	  /* } */
	  nboutliers=nb_column[i]-10;
	  for(l=0; l<nb_column[i]/2-15-5;l++) {
	    cpl_vector_set(outliers, l, 1);
	  }
	  for(l=nb_column[i]/2-15+5; l<nb_column[i];l++) {
	    cpl_vector_set(outliers, l, 1);
	  }
	} else {
	  /* nboutliers=nb_column[i]-30; */
	  /* for(l=0; l<255;l++) { */
	  /*   cpl_vector_set(outliers, l, 1); */
	  /* } */
	  /* for(l=285; l<nb_column[i];l++) { */
	  /*   cpl_vector_set(outliers, l, 1); */
	  /* } */
	  nboutliers=nb_column[i]-60;
	  for(l=0; l<nb_column[i]/2-60-30;l++) {
	    cpl_vector_set(outliers, l, 1);
	  }
	  for(l=nb_column[i]/2-60+30; l<nb_column[i];l++) {
	    cpl_vector_set(outliers, l, 1);
	  }
	}
      }
      if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "MED")) { 
	if (i==0 || i==1 || i==3 || i==4) {
	  /* nboutliers=nb_column[i]-50; */
	  /* for(l=0; l<30;l++) { */
	  /*   cpl_vector_set(outliers, l, 1); */
	  /* } */
	  /* for(l=80; l<nb_column[i];l++) { */
	  /*   cpl_vector_set(outliers, l, 1); */
	  /* } */
	  nboutliers=nb_column[i]-10;
	  for(l=0; l<nb_column[i]/2-15-5;l++) {
	    cpl_vector_set(outliers, l, 1);
	  }
	  for(l=nb_column[i]/2-15+5; l<nb_column[i];l++) {
	    cpl_vector_set(outliers, l, 1);
	  }
	} else {
	  /* nboutliers=nb_column[i]-424; */
	  /* for(l=0; l<100;l++) { */
	  /*   cpl_vector_set(outliers, l, 1); */
	  /* } */
	  /* for(l=524; l<nb_column[i];l++) { */
	  /*   cpl_vector_set(outliers, l, 1); */
	  /* } */
	  nboutliers=nb_column[i]-60;
	  for(l=0; l<nb_column[i]/2-60-30;l++) {
	    cpl_vector_set(outliers, l, 1);
	  }
	  for(l=nb_column[i]/2-60+30; l<nb_column[i];l++) {
	    cpl_vector_set(outliers, l, 1);
	  }
	}
      }
      if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "LOW")) { 
	if (i==0 || i==1 || i==3 || i==4) {
	  /* nboutliers=nb_column[i]-40; */
	  /* for(l=0; l<45;l++) { */
	  /*   cpl_vector_set(outliers, l, 1); */
	  /* } */
	  /* for(l=85; l<nb_column[i];l++) { */
	  /*   cpl_vector_set(outliers, l, 1); */
	  /* } */
	  nboutliers=nb_column[i]-10;
	  for(l=0; l<nb_column[i]/2-15-5;l++) {
	    cpl_vector_set(outliers, l, 1);
	  }
	  for(l=nb_column[i]/2-15+5; l<nb_column[i];l++) {
	    cpl_vector_set(outliers, l, 1);
	  }
	} else {
	  /* nboutliers=nb_column[i]-280; */
	  /* for(l=0; l<40;l++) { */
	  /*   cpl_vector_set(outliers, l, 1); */
	  /* } */
	  /* for(l=320; l<nb_column[i];l++) { */
	  /*   cpl_vector_set(outliers, l, 1); */
	  /* } */
	  nboutliers=nb_column[i]-60;
	  for(l=0; l<nb_column[i]/2-60-30;l++) {
	    cpl_vector_set(outliers, l, 1);
	  }
	  for(l=nb_column[i]/2-60+30; l<nb_column[i];l++) {
	    cpl_vector_set(outliers, l, 1);
	  }
	}
      }
      

      nboutprec=-1;
      cpt=0;
      while (nboutliers > nboutprec || cpt < cptmax) {
	samppos= cpl_matrix_new(1,nb_column[i]-nboutliers);
	if (samppos == NULL) {
	  cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
	  return NULL;
	}

	fitvals2=cpl_vector_new(nb_column[i]-nboutliers);
	if (fitvals2 == NULL) {
	  cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
	  cpl_matrix_delete(samppos);
	  return NULL;
	}
	// Fit a second order polynomial
	j=0;
	for(l=0; l<nb_column[i];l++) {
	  if (cpl_vector_get(outliers,l) == 0) {
	    cpl_matrix_set(samppos, 0, j, l);
	    cpl_vector_set(fitvals2, j, cpl_matrix_get(distor_index_max[i], l, k));
	    j++;
	  }
	}
	if (cpl_polynomial_fit(polysmooth, samppos, &sampsym, fitvals2, NULL, 
			       CPL_FALSE, &mindeg, &maxdeg) != CPL_ERROR_NONE) {
	  cpl_msg_error(cpl_func,"could not fit a polynomial");
	  return NULL;
	}
	if (cpl_vector_fill_polynomial_fit_residual(residual, fitvals2, NULL, 
						    polysmooth, samppos, &chi2) != CPL_ERROR_NONE) {
	  cpl_msg_error(cpl_func,"could not fill a vector from the residuals of a polynomial fit");
	  return NULL;
	}
	nboutprec=nboutliers;
	nboutliers=0;
	for(l=0; l<nb_column[i];l++) {
	  if (fabs(cpl_matrix_get(distor_index_max[i], l, k)-
		   cpl_polynomial_eval_1d(polysmooth,(double) l,NULL)) > 3*sqrt(chi2)) {
	    cpl_vector_set(outliers, l, 1);
	    nboutliers++;
	  } else {
	    cpl_vector_set(outliers, l, 0);
	  }
	}
	cpl_matrix_delete(samppos);
	cpl_vector_delete(fitvals2);
	cpt++;
      }      
      for(l=0; l<nb_column[i];l++) {
	val=cpl_polynomial_eval_1d(polysmooth,(double) l,NULL);
	if (isnan(val) == 0) {
      	  cpl_matrix_set(distor_index_max[i], l, k,val);
	} 
      }
    }
 
    if ( cpl_parameter_get_bool(debug) ) {
      for(l=0; l<nb_column[i];l++) {
	fprintf(file, "%d\t", l); 
	for(k=0; k< nbmaxmin; k++) {
	  fprintf(file, "%lf", cpl_matrix_get(distor_index_max[i], l, k));
	  if(k< nbmaxmin-1) {
	    fprintf(file, "\t");
	  }
	  else {
	    fprintf(file, "\n");
	  }
	}
      }
    }
    cpl_polynomial_delete(polysmooth);
    cpl_vector_delete(residual);
    cpl_vector_delete(outliers);
  }

  if ( cpl_parameter_get_bool(debug) ) {
    fclose(file);
  }
/* // Determine the reference pixel of each plastic foils features  */
/* // in order to retrieve a linear dispersion law. There are nbmaxmin  */
/* // features.  */
/* // We choose the central row (nb_column/2) of the first region as reference for  */
/* // determining the distorsion in the spectral direction */
/* // We fit a second order polynome "by hand" */
/* // And we fix the coordinates of each spectral feature in the variable refpix  */
/*   for(i=0;i<nbmaxmin;i++) { */
/*     foo3+=cpl_vector_get(fitvals,i); */
/*     foo4+=cpl_matrix_get(distor_index_max[0], nb_column[0]/2, i); */
/*   } */
/*   foo3/=nbmaxmin; */
/*   foo4/=nbmaxmin; */
/*   for(i=0;i<nbmaxmin;i++) { */
/*     foo1+=(cpl_vector_get(fitvals,i)-foo3)*(cpl_matrix_get(distor_index_max[0], nb_column[0]/2, i)-foo4); */
/*     foo2+=(cpl_vector_get(fitvals,i)-foo3)*(cpl_vector_get(fitvals,i)-foo3); */
/*   } */
/*   if (foo2 == 0.) { */
/*     cpl_msg_error(cpl_func,"could not fit a 2nd order polynome to the reference pixels"); */
/*     return NULL; */
/*   } */
/*   coefb1=foo1/foo2; */
/*   coefb0=foo4-foo3*coefb1; */

/*   // reference pixel of spectral features computed from seond order polynomial */
/*   for(i=0;i<nbmaxmin;i++) { */
/*     refpix[i]=cpl_vector_get(fitvals,i)*coefb1+coefb0; */
/*   } */

  for(i=0;i<nbmaxmin;i++) {
    refpix[i]=cpl_matrix_get(distor_index_max[2], nb_column[2]/2, i);
  }
  
  
  
  // Compute the distorsion coefficients in spectral direction (DISTOR_IMAGES)
  if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "LOW") ) { 
    for(i=0; i<nbregion; i++) {
      coeff= cpl_matrix_new(nbmaxmin, 1);
      if (coeff == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
	return NULL;
      }
      rhs= cpl_matrix_new(nbmaxmin, 1);
      if (rhs == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
	cpl_matrix_delete(coeff);
	return NULL;
      }
      // For each spatial channel
      for(j=0; j< nb_column[i]; j++) {
	// For each reference coordinate
	for(k=0; k< nbmaxmin; k++) {
	  cpl_matrix_set(coeff, k, 0, 1);
	  cpl_matrix_set(rhs, k, 0, refpix[k]-cpl_matrix_get(distor_index_max[i], j, k));
	}
	// Solve a linear system to find the distorsion coefficients
	// and append the result to the matrix
	coefftrans=cpl_matrix_transpose_create(coeff);
	if (coefftrans == NULL) {
	  cpl_msg_error(cpl_func,"could not transpose a cpl_matrix");
	  return NULL;
	}

	normal=cpl_matrix_product_create(coefftrans,coeff);
	if (normal == NULL) {
	  cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix");
	  return NULL;
	}
	invnormal=cpl_matrix_invert_create(normal);
	if (invnormal == NULL) {
	  cpl_msg_error(cpl_func,"could not invert cpl_matrix");
	  return NULL;
	}
	cpl_matrix_delete(normal);
	matrix_temp=cpl_matrix_product_create(coefftrans,rhs);
	if (matrix_temp == NULL) {
	  cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix");
	  return NULL;
	}
	sol=cpl_matrix_product_create(invnormal,matrix_temp);
	if (sol == NULL) {
	  cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix");
	  return NULL;
	}
	cpl_matrix_delete(matrix_temp);
	cpl_matrix_delete(coefftrans);
	cpl_matrix_copy(shiftmap->list_shiftcoefy[i]->coefdistor,
                        sol, 0, j);
	cpl_matrix_set(shiftmap->list_shiftcoefy[i]->coefdistor,1,j,1.);
	cpl_matrix_set(shiftmap->list_shiftcoefy[i]->coefdistor,2,j,0.);

	for(k=0;k<1;k++) {
	  cpl_matrix_set(shiftmap->list_shiftcoefy[i]->errdistor,
			 k,j,ERR_MAX_PIX*sqrt(cpl_matrix_get(invnormal,k,k)));
	}
	cpl_matrix_set(shiftmap->list_shiftcoefy[i]->errdistor,1,j,0.);
	cpl_matrix_set(shiftmap->list_shiftcoefy[i]->errdistor,2,j,0.);
	cpl_matrix_delete(invnormal);
	cpl_matrix_delete(sol);
      }
      cpl_matrix_delete(coeff);
      cpl_matrix_delete(rhs);
    }  
  } else {
    for(i=0; i<nbregion; i++) {
      coeff= cpl_matrix_new(nbmaxmin, 3);
      if (coeff == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
	return NULL;
      }
      rhs= cpl_matrix_new(nbmaxmin, 1);
      if (rhs == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
	cpl_matrix_delete(coeff);
	return NULL;
      }
      // For each spatial channel
      for(j=0; j< nb_column[i]; j++) {
	// For each reference coordinate
	for(k=0; k< nbmaxmin; k++) {
	  // Fill a matrix of coefficients
	  // coeff is the design matrix of the least square estimation.
	  // It is a 3xnbmaxmin matrix since we have 3 measurements (coordinates
	  // in the spectral direction of each spectral feature) and 
	  // we have 3 parameters (Y=a+by+cy**2)
	  cpl_matrix_set(coeff, k, 0, 1);
	  val= cpl_matrix_get(distor_index_max[i], j, k);
	  cpl_matrix_set(coeff, k, 1, val);
	  cpl_matrix_set(coeff, k, 2, pow(val, 2));

	  // Fill a matrix with the position of maxima and minima
	  // Reference pixel of teh central row of the first region
	  cpl_matrix_set(rhs, k, 0, refpix[k]);

	}
	// Solve a linear system to find the distorsion coefficients
	// and append the result to the matrix
	coefftrans=cpl_matrix_transpose_create(coeff);
	if (coefftrans == NULL) {
	  cpl_msg_error(cpl_func,"could not transpose a cpl_matrix");
	  return NULL;
	}

	normal=cpl_matrix_product_create(coefftrans,coeff);
	if (normal == NULL) {
	  cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix");
	  return NULL;
	}
	invnormal=cpl_matrix_invert_create(normal);
	if (invnormal == NULL) {
	  cpl_msg_error(cpl_func,"could not invert cpl_matrix");
	  return NULL;
	}
	cpl_matrix_delete(normal);
	matrix_temp=cpl_matrix_product_create(coefftrans,rhs);
	if (matrix_temp == NULL) {
	  cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix");
	  return NULL;
	}
	sol=cpl_matrix_product_create(invnormal,matrix_temp);
	if (sol == NULL) {
	  cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix");
	  return NULL;
	}
	cpl_matrix_delete(matrix_temp);
	cpl_matrix_delete(coefftrans);
	cpl_matrix_copy((*shiftmap).list_shiftcoefy[i]->coefdistor,
                        sol, 0, j);
	/* cpl_matrix_set(shiftmap->list_shiftcoefy[i]->coefdistor,0,j,0.); */
	/* cpl_matrix_set(shiftmap->list_shiftcoefy[i]->coefdistor,1,j,1.); */
	/* cpl_matrix_set(shiftmap->list_shiftcoefy[i]->coefdistor,2,j,0.); */

	for(k=0;k<3;k++) {
	  cpl_matrix_set((*shiftmap).list_shiftcoefy[i]->errdistor,
			 k,j,ERR_MAX_PIX*sqrt(cpl_matrix_get(invnormal,k,k)));
	}
	cpl_matrix_delete(invnormal);
	cpl_matrix_delete(sol);
      }
      cpl_matrix_delete(coeff);
      cpl_matrix_delete(rhs);
    }
  }


  for(i=0; i<nbregion; i++) {
    cpl_matrix_delete(distor_index_max[i]);
  }


/* // Correct Raw Frames  */
/*   spectra_corrected= cpl_calloc(nbframes[2], sizeof(mat_gendata *)); */
/*   if (spectra_corrected == NULL) { */
/*     cpl_msg_error(cpl_func,"could not allocate memory for mat_gendata"); */
/*     return NULL; */
/*   } */
/*   //distor_corrected= cpl_calloc(nbframes[0]*3, sizeof(mat_gendata *)); */
/*   for(i=0; i< nbframes[2]; i++) { */
/*     spectra_corrected[i]= mat_apply_shift(spectra[i], shiftmap, ione);  */
/*     if (spectra_corrected[i] == NULL) { */
/*       cpl_msg_error(cpl_func,"could not apply shift correction"); */
/*       return NULL; */
/*     } */
/*   } */
/*   //for(i=0; i< 3*nbframes[0]; i++) { */
/*   //  distor_corrected[i]= mat_apply_shift(distor[i], shiftmap, ione);  */
/*   //} */




  /* // Apply Observationnal Corrections */
  /* if ( cpl_parameter_get_bool(obsCorrection) ) */
  /*   { */
  /*     double a1=0.,b1=0.,c1=0.; */
  /*     double coefP[nbregion][3]; */
  /*     if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "MED")) */
  /* 	{  */
  /* 	  coefP[0][0]=-0.60784959579434839;//-0.4384; */
  /* 	  coefP[0][1]=1.0001699468570202;//0.9998; */
  /* 	  coefP[0][2]=7.8209670731696472e-08;//3.551e-07; */
  /* 	  coefP[1][0]=-0.65699445668002299;//-0.02365; */
  /* 	  coefP[1][1]=0.99987159830807204;//0.9985; */
  /* 	  coefP[1][2]=2.1696107346454268e-07;//9.987e-07; */
  /* 	  coefP[2][0]=-0.59060159884876873;//-0.6832; */
  /* 	  coefP[2][1]=0.9995825519537711;//0.9997; */
  /* 	  coefP[2][2]=5.1755121685629005e-07;//5.949e-07; */
  /* 	  coefP[3][0]=0.9059738728043697;//0.5359; */
  /* 	  coefP[3][1]=0.99873054031942465;//0.9991; */
  /* 	  coefP[3][2]=3.6608416181709993e-07;//2.712e-07; */
  /* 	} */
  /*     else if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "HIGH")) */
  /* 	{ */
  /* 	  coefP[0][0]=-2.9148358666628518;//-3.0834468794175445;//-1.1087652218859325; */
  /* 	  coefP[0][1]=1.0048233165212161;//1.0045306855566838;//1.0031316878480552; */
  /* 	  coefP[0][2]=-1.9332464045229592e-06;//-1.5769198134524529e-06;//-1.0919690738521317e-06; */
  /* 	  coefP[1][0]=-3.1011334268681097;//-3.3102273933315307;//-1.3207828441075131; */
  /* 	  coefP[1][1]=1.0038955900874327;//1.0037191583984488;//1.0035334135211373; */
  /* 	  coefP[1][2]=-1.5176908648936047e-06;//-7.884049999075656e-07;//-9.5119723975411167e-07; */
  /* 	  coefP[2][0]=-1.5099344840844222;//-2.2678917781500587;//-0.61672427790318973; */
  /* 	  coefP[2][1]=1.0004979238013443;//1.00377793356238;//1.0030958256736233; */
  /* 	  coefP[2][2]=9.0793576683026313e-07;//-1.528927963857534e-06;//-1.0637667120464666e-06; */
  /* 	  coefP[3][0]=0.90224588347357049;//1.2954185743594173;//0.66240228781800947; */
  /* 	  coefP[3][1]=0.99908864472450354;//0.99802685250134504;//0.99823009292465381; */
  /* 	  coefP[3][2]=2.9250346468281351e-07;//8.2621429875721231e-07;//6.4082407095449372e-07; */
  /* 	} */
  /*     else */
  /* 	{ */
  /* 	  coefP[0][0]=0.; */
  /* 	  coefP[0][1]=1.; */
  /* 	  coefP[0][2]=0.; */
  /* 	  coefP[1][0]=0.; */
  /* 	  coefP[1][1]=1.; */
  /* 	  coefP[1][2]=0.; */
  /* 	  coefP[2][0]=0.; */
  /* 	  coefP[2][1]=1.; */
  /* 	  coefP[2][2]=0.; */
  /* 	  coefP[3][0]=0.; */
  /* 	  coefP[3][1]=1.; */
  /* 	  coefP[3][2]=0.; */
  /* 	} */

  /*     for(i=0; i<4; i++) */
  /* 	{ */
  /* 	  // Not apply to the interferometric region */
  /* 	  if (i < 2) */
  /* 	    { */
  /* 	      k=i; */
  /* 	    } */
  /* 	  else */
  /* 	    { */
  /* 	      k=i+1; */
  /* 	    } */
  /* 	  for(j=0; j< nb_column[i]; j++) */
  /* 	    { */
  /* 	      a1=cpl_matrix_get(shiftmap->list_shiftcoefy[k]->coefdistor,0,j); */
  /* 	      b1=cpl_matrix_get(shiftmap->list_shiftcoefy[k]->coefdistor,1,j); */
  /* 	      c1=cpl_matrix_get(shiftmap->list_shiftcoefy[k]->coefdistor,2,j); */
  /* 	      cpl_matrix_set(shiftmap->list_shiftcoefy[k]->coefdistor,0,j,coefP[i][0]+coefP[i][1]*a1+coefP[i][2]*a1*a1); */
  /* 	      cpl_matrix_set(shiftmap->list_shiftcoefy[k]->coefdistor,1,j,coefP[i][1]*b1+2*a1*b1*coefP[i][2]); */
  /* 	      cpl_matrix_set(shiftmap->list_shiftcoefy[k]->coefdistor,2,j,coefP[i][1]*c1+coefP[i][2]*b1*b1+2*coefP[i][2]*a1*c1); */
  /* 	    } */
  /* 	}  */
  /*   } */







  for(i=0; i<nbregion; i++) {
    cpl_image_delete(spectra_raw[i]);
    cpl_image_delete(spectra_background[i]);
  }
  cpl_free(spectra_raw);
  cpl_free(spectra_background);
  spectra_raw = cpl_calloc(nbregion, sizeof(cpl_image *));
  if (spectra_raw == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
    return NULL;
  }
  spectra_background = cpl_calloc(nbregion, sizeof(cpl_image *));
  if (spectra_background == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_image");
    cpl_free(spectra_raw);
    cpl_free(spectra_background);
    return NULL;
  }
  for(i=0; i<nbregion; i++) {
    spectra_raw[i]= cpl_image_new(
				  spectra[0]->imgdet->list_region[i]->naxis[0], 
				  spectra[0]->imgdet->list_region[i]->naxis[1],
				  cpl_image_get_type(spectra[0]->imgdata->
						     list_frame[0]->list_subwin[i]->imgreg[0]));
    spectra_background[i]= 
      cpl_image_new(spectra_hotdark[0]->imgdet->list_region[i]->naxis[0],
		    spectra_hotdark[0]->imgdet->list_region[i]->naxis[1],
		    cpl_image_get_type(spectra_hotdark[0]->imgdata->
				       list_frame[0]->list_subwin[i]->imgreg[0]));
  }

  spectra_shift= cpl_calloc(nbframes[2], sizeof(mat_gendata *));
  if (spectra_shift == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for mat_gendata");
    cpl_free(spectra_raw);
    cpl_free(spectra_background);
    return NULL;
  }
  spectra_hd_shift= cpl_calloc(nbframes[3], sizeof(mat_gendata *));
  if (spectra_hd_shift == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for mat_gendata");
    cpl_free(spectra_shift);
    cpl_free(spectra_raw);
    return NULL;
  }
  // Apply the distorsion coefficients to SPECTRA_IMAGES and SPECTRA_HOTDARK
  for(i=0; i< nbframes[2]; i++) {
    spectra_shift[i]= mat_apply_shift(spectra[i], shiftmap, izero);
    if (spectra_shift[i] == NULL) {
      cpl_msg_error(cpl_func,"could not apply the shift correction");
      cpl_free(spectra_shift);
      cpl_free(spectra_hd_shift);
      cpl_free(spectra_raw);
      return NULL;
    }
  }
  for(i=0; i< nbframes[3]; i++) {
    spectra_hd_shift[i]= mat_apply_shift(spectra_hotdark[i], shiftmap, izero);
    if (spectra_hd_shift[i] == NULL) {
      cpl_msg_error(cpl_func,"could not apply the shift correction");
      cpl_free(spectra_shift);
      cpl_free(spectra_hd_shift);
      cpl_free(spectra_raw);
      return NULL;
    }
  }
  cpl_msg_info(cpl_func,
	       "Plastic foils frames: Correction of the distrosion in the spatial direction done");


  // Add the spectra raw data for each region and each frame
  k=0;
  while(k< nbframes[2]) {
    for(i=0; i<nbregion; i++) {
      cpl_image_add(spectra_raw[i],spectra_shift[k]->imgdata
		    ->list_frame[0]->list_subwin[i]->imgreg[0]);
      nbimg[i]++;
    }
    k++;
  }

  // Make the average image from the images added for each region
  for(i=0; i<nbregion; i++) {
    if (nbimg[i] == 0) {
      cpl_msg_error(cpl_func,"0 images found for SPECTRA");
      cpl_free(spectra_shift);
      cpl_free(spectra_hd_shift);
      cpl_free(spectra_raw);
      return NULL;
    }
    cpl_image_multiply_scalar(spectra_raw[i], 1/nbimg[i]);
    nbimg[i]=0;
  }

  // Add the spectra (background) data for each region and each frame
  k=0;
  while(k< nbframes[3]) {
    for(i=0; i<nbregion; i++) {
      cpl_image_add(spectra_background[i],spectra_hd_shift[k]
		    ->imgdata->list_frame[0]->list_subwin[i]->imgreg[0]);
      nbimg[i]++;
    }
    k++;
  }
  // Make the average image from the images added for each region
  for(i=0; i<nbregion; i++) {
    if (nbimg[i] == 0) {
      cpl_msg_error(cpl_func,"0 images found for SPECTRA BACKGROUND");
      cpl_free(spectra_shift);
      cpl_free(spectra_hd_shift);
      cpl_free(spectra_raw);
      return NULL;
    }
    cpl_image_multiply_scalar(spectra_background[i], 1/nbimg[i]);
    nbimg[i]=0;
  }

  if(!strcmp(chipName, "HAWAII-2RG")) {
    // Divide the laboratory background to the raw data (SPECTRA)
    if (!strcmp(resolution, "MED") ||!strcmp(resolution, "HIGH") ) {
      for(i=0; i<nbregion; i++) {
	cpl_image_divide(spectra_raw[i], spectra_background[i]);
      }
    }
  }


  if ( cpl_parameter_get_bool(debug) ) {
    foo=cpl_image_new(cpl_image_get_size_x(spectra_raw[0])+
		      cpl_image_get_size_x(spectra_raw[1])+
		      cpl_image_get_size_x(spectra_raw[2])+
		      cpl_image_get_size_x(spectra_raw[3])+
		      cpl_image_get_size_x(spectra_raw[4]),
		      cpl_image_get_size_y(spectra_raw[0]),
		      cpl_image_get_type(spectra_raw[0]));

    for(i=1;i<=cpl_image_get_size_x(spectra_raw[0]);i++) {
      for(j=1;j<=cpl_image_get_size_y(spectra_raw[0]);j++) {
	cpl_image_set(foo,i,j,cpl_image_get(spectra_raw[0],i,j,&out));
      }
    }
    for(i=1;i<=cpl_image_get_size_x(spectra_raw[1]);i++) {
      for(j=1;j<=cpl_image_get_size_y(spectra_raw[1]);j++) {
	cpl_image_set(foo,cpl_image_get_size_x(spectra_raw[0])+
		      i,j,cpl_image_get(spectra_raw[1],i,j,&out));
      }
    }
    for(i=1;i<=cpl_image_get_size_x(spectra_raw[2]);i++) {
      for(j=1;j<=cpl_image_get_size_y(spectra_raw[2]);j++) {
	cpl_image_set(foo,cpl_image_get_size_x(spectra_raw[0])+
		      cpl_image_get_size_x(spectra_raw[1])+
		      i,j,cpl_image_get(spectra_raw[2],i,j,&out));
      }
    }
    for(i=1;i<=cpl_image_get_size_x(spectra_raw[3]);i++) {
      for(j=1;j<=cpl_image_get_size_y(spectra_raw[3]);j++) {
	cpl_image_set(foo,cpl_image_get_size_x(spectra_raw[0])+
		      cpl_image_get_size_x(spectra_raw[1])+
		      cpl_image_get_size_x(spectra_raw[2])+
		      i,j,cpl_image_get(spectra_raw[3],i,j,&out));
      }
    }
    for(i=1;i<=cpl_image_get_size_x(spectra_raw[4]);i++) {
      for(j=1;j<=cpl_image_get_size_y(spectra_raw[4]);j++) {
	cpl_image_set(foo,cpl_image_get_size_x(spectra_raw[0])+
		      cpl_image_get_size_x(spectra_raw[1])+
		      cpl_image_get_size_x(spectra_raw[2])+
		      cpl_image_get_size_x(spectra_raw[3])+
		      i,j,cpl_image_get(spectra_raw[4],i,j,&out));
      }
    }
    cpl_image_save(foo, "imagecor.fits",
		   CPL_BPP_IEEE_FLOAT, spectra[0]->keywords, CPL_IO_CREATE); 
        
    cpl_image_delete(foo);
  }





  for(i=0; i< nbframes[2]; i++) {
    mat_gendata_delete(spectra_shift[i]);
  }
  for(i=0; i< nbframes[3]; i++) {
    mat_gendata_delete(spectra_hd_shift[i]);
  }
  cpl_free(spectra_shift);
  cpl_free(spectra_hd_shift);




  ///////////////////////////////////////
  // Dispersion Law Computation

  // We first compute the spectra of the plastic foils 
  // (integration of the corrected images and projection along the spatial 
  // direction). all the computations are done in the interferometric region.
  spectraObs=cpl_malloc(nbregion*sizeof(cpl_vector *));
  for(j=0;j<nbregion;j++) {
      /* imgInteg =  */
      /* 	cpl_image_new(cpl_image_get_size_x(spectra_corrected[0]->imgdata-> */
      /* 					   list_frame[0]->list_subwin[j]->imgreg[0]), */
      /* 		      cpl_image_get_size_y(spectra_corrected[0]->imgdata-> */
      /* 					   list_frame[0]->list_subwin[j]->imgreg[0]), */
      /* 		      cpl_image_get_type(spectra_corrected[0]->imgdata-> */
      /* 					 list_frame[0]->list_subwin[j]->imgreg[0])); */
      /* if (imgInteg == NULL) { */
      /* 	cpl_msg_error(cpl_func,"could not allocate memory for cpl_image"); */
      /* 	return NULL; */
      /* } */
    
      /* for (i=0;i<spectra_corrected[0]->imgdata->nbframe;i++) { */
      /* 	cpl_image_add(imgInteg,spectra_corrected[0]->imgdata->list_frame[i]-> */
      /* 		      list_subwin[j]->imgreg[0]); */
      /* } */
      /* foo=cpl_image_collapse_create(imgInteg, 1); */
    foo=cpl_image_collapse_create(spectra_raw[j], 1);
    if (foo == NULL) {
      cpl_msg_error(cpl_func,"could not collapse cpl_image");
      return NULL;
    }
    // Spectra of the plastic foils
    spectraObs[j] = cpl_vector_new_from_image_column(foo,1);
    val=cpl_vector_get_sum(spectraObs[j]);
    cpl_vector_divide_scalar(spectraObs[j],val);
    
    if (spectraObs[j] == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
      return NULL;
    }
    cpl_image_delete(foo);
    /* cpl_image_delete(imgInteg); */
  }



  if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "LOW") )
    {
      cpl_free(boundaries);
      cpl_array_delete(typemaxmin);
      cpl_vector_delete(fitvals);
      if (mjdobs > 58810.)
	{
	  nbmaxmin = 5;
	}
      else
	{
	  nbmaxmin = 6;
	}
      fitvals= cpl_vector_new(nbmaxmin);
      if (fitvals == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
	return NULL;
      }
      // Valure from CalibratePlasticFoils.py
      // pixesl = [126.84082593 146.57535341 168.48303426 192.08181818 208.2766323 ]
      // lambda = [4.92938323 4.60883764 4.21042808 3.7354085  3.36314942]
      if (mjdobs > 58810.)
	{
	  cpl_vector_set(fitvals, 0, 4.92938323);
	  cpl_vector_set(fitvals, 1, 4.60883764);
	  cpl_vector_set(fitvals, 2, 3.7354085);
	  cpl_vector_set(fitvals, 3, 3.36120672);
	  cpl_vector_set(fitvals, 4, 3.04453497);
	  typemaxmin= cpl_array_new(nbmaxmin, CPL_TYPE_INT);
	  cpl_array_set(typemaxmin,0,1);
	  cpl_array_set(typemaxmin,1,0);
	  cpl_array_set(typemaxmin,2,0);
	  cpl_array_set(typemaxmin,3,1);
	  cpl_array_set(typemaxmin,4,0);
	  boundaries= cpl_calloc(2*nbmaxmin, sizeof(int));
	  boundaries[0]=3;
	  boundaries[1]=20;
	  boundaries[2]=27;
	  boundaries[3]=40;
	  boundaries[4]=70;
	  boundaries[5]=85;
	  boundaries[6]=80;
	  boundaries[7]=96;
	  boundaries[8]=100;
	  boundaries[9]=110;
	}
      else
	{
	  cpl_vector_set(fitvals, 0, 4.92938323);
	  cpl_vector_set(fitvals, 1, 4.60883764);
	  cpl_vector_set(fitvals, 2, 4.21011428);
	  cpl_vector_set(fitvals, 3, 3.7354085);
	  cpl_vector_set(fitvals, 4, 3.36120672);
	  cpl_vector_set(fitvals, 5, 3.04453497);
   
	  typemaxmin= cpl_array_new(nbmaxmin, CPL_TYPE_INT);
	  cpl_array_set(typemaxmin,0,1);
	  cpl_array_set(typemaxmin,1,0);
	  cpl_array_set(typemaxmin,2,1);
	  cpl_array_set(typemaxmin,3,0);
	  cpl_array_set(typemaxmin,4,1);
	  cpl_array_set(typemaxmin,5,0);
	  boundaries= cpl_calloc(2*nbmaxmin, sizeof(int));
	  boundaries[0]=46+(77-corner);
	  boundaries[1]=60+(77-corner);
	  boundaries[2]=67+(77-corner);
	  boundaries[3]=75+(77-corner);
	  boundaries[4]=85+(77-corner);
	  boundaries[5]=95+(77-corner);
	  boundaries[6]=112+(77-corner);
	  boundaries[7]=118+(77-corner);
	  boundaries[8]=127+(77-corner);
	  boundaries[9]=133+(77-corner);
	  boundaries[10]=139+(77-corner);
	  boundaries[11]=148+(77-corner);
	}
      // Load one row of pixel values from the spectra_raw image
      for(k=0; k < nbmaxmin; k++) {
	// Find a maximum
        if(cpl_array_get_int(typemaxmin, k, NULL) == 0) {
	  valmax=0.;
	  for(l=boundaries[2*k]; l < boundaries[2*k+1]; l++) {
	    val_cur=cpl_vector_get(spectraObs[2], l);
	    if (val_cur > valmax) {
	      refpix[k]=l*1.0;
	      valmax=val_cur;
	    }
	  }
	}
	// Find a minimum
	else {
	  valmin=1.E15;
	  for(l=boundaries[2*k]; l < boundaries[2*k+1]; l++) {
	    val_cur=cpl_vector_get(spectraObs[2], l);
	    if (val_cur < valmin ) {
	      refpix[k]=l*1.0;
	      valmin=val_cur;
	    }
	  }
	}

	boxsz=5;
	polysmooth= cpl_polynomial_new(1);
	if (polysmooth == NULL) {
	  cpl_msg_error(cpl_func,"could not allocate memory for cpl_polynomial");
	  return NULL;
	}
      
	samppos= cpl_matrix_new(1, boxsz);
	if (samppos == NULL) {
	  cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
	  return NULL;
	}
	fitvals2=cpl_vector_new(boxsz);
	if (fitvals2 == NULL) {
	  cpl_msg_error(cpl_func,"could not allocate memory for cpl_vector");
	  cpl_matrix_delete(samppos);
	  return NULL;
	}
	sampsym = CPL_TRUE;
	mindeg= 0;
	maxdeg= 2;
	deg1=1;
	deg2=2;
	// Fit a second order polynomial around imx 
	// and determine the maximum/minimum of the polynomial
	for(l=0; l<boxsz;l++) {
	  cpl_matrix_set(samppos, 0, l, l);
	  cpl_vector_set(fitvals2, l, cpl_vector_get(spectraObs[2],refpix[k]-(boxsz/2)+l));
	}
	if (cpl_polynomial_fit(polysmooth, samppos, &sampsym, fitvals2, NULL, 
			       CPL_FALSE, &mindeg, &maxdeg) != CPL_ERROR_NONE) {
	  cpl_msg_error(cpl_func,"could not fir a polynomial");
	  return NULL;
	}
	// we determine the new coordinate of teh spectral feature with 
	// the coordinates of the max/min of the second order polynomial
	if (cpl_polynomial_get_coeff(polysmooth,&deg2) != 0) {
	  valmax=refpix[k]-(boxsz/2) + (-1* cpl_polynomial_get_coeff(polysmooth,&deg1)/
					(2.*cpl_polynomial_get_coeff(polysmooth,&deg2)));
	  if (isnan(valmax) != 0) {
	    valmax=refpix[k];
	  }
	} else {
	  valmax=refpix[k];
	}
	cpl_matrix_delete(samppos);
	cpl_vector_delete(fitvals2);
	cpl_polynomial_delete(polysmooth);
	// Set the position of the maximum or minimum found in a matrix
	refpix[k]= valmax;
      }


      // Compute the coefficients giving the dispersion law
      // by Fitting a polynomial 
      coeff= cpl_matrix_new(nbmaxmin, N_DEG_DISPERSION_LAW+1);
      if (coeff == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
	return NULL;
      }
      rhs= cpl_matrix_new(nbmaxmin, 1);
      if (rhs == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
	cpl_matrix_delete(coeff);
	return NULL;
      }
  
      for(k=0; k< nbmaxmin; k++) {
	// Fill a matrix of coefficients
	cpl_matrix_set(coeff, k, 0, 1);
	/* val= cpl_vector_get(specFeature, k ); */
	val= refpix[k];
	val+=shiftmap->imgdet->list_region[0]->corner[1];
	cpl_matrix_set(coeff, k, 1, val);
	cpl_matrix_set(coeff, k, 2, pow(val, 2));
	cpl_matrix_set(coeff, k, 3, pow(val, 3));
	cpl_matrix_set(coeff, k, 4, pow(val, 4));
	// Fill a matrix with the position of maxima and minima
	// val= cpl_matrix_get(distor_index_max[0], nb_column[0]/2, k);
	cpl_matrix_set(rhs, k, 0, cpl_vector_get(fitvals,k));
      }
      // Solve a linear system to find the distorsion dispersion coefficients
      // and append the result to the matrix
      coefftrans=cpl_matrix_transpose_create(coeff);
      if (coefftrans == NULL) {
	cpl_msg_error(cpl_func,"could not transpose a cpl_matrix");
	cpl_matrix_delete(rhs);
	return NULL;
      }
      normal=cpl_matrix_product_create(coefftrans,coeff);
      if (normal == NULL) {
	cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix");
	cpl_matrix_delete(rhs);
	return NULL;
      }
      invnormal=cpl_matrix_invert_create(normal);
      if (invnormal == NULL) {
	cpl_msg_error(cpl_func,"could not invert a cpl_matrix");
	cpl_matrix_delete(rhs);
	return NULL;
      }
      cpl_matrix_delete(normal);
      matrix_temp=cpl_matrix_product_create(coefftrans,rhs);
      if (matrix_temp == NULL) {
	cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix");
	return NULL;
      }
      sol=cpl_matrix_product_create(invnormal,matrix_temp);
      if (sol == NULL) {
	cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix");
	return NULL;
      }
      cpl_matrix_delete(matrix_temp);
      cpl_matrix_delete(coefftrans);
      for(k=0;k<N_DEG_DISPERSION_LAW+1;k++) {
	shiftmap->dispcoef[k]=cpl_matrix_get(sol,k,0);
	shiftmap->errdispcoef[k]=ERR_MAX_PIX*sqrt(cpl_matrix_get(invnormal,k,k));
      }
      cpl_matrix_delete(invnormal);
      cpl_matrix_delete(sol);
      cpl_matrix_delete(coeff);
      cpl_matrix_delete(rhs);

      cpl_vector_delete(fitvals);
      cpl_free(boundaries);
      cpl_array_delete(typemaxmin);






      
      // New dispersion law - Hardcoded - No way to estimate it from plastic foils
      /*      shiftmap->dispcoef[0]=1.999;
      shiftmap->dispcoef[1]=0.1021;
      shiftmap->dispcoef[2]=-0.001051;
      shiftmap->dispcoef[3]=4.205e-06;
      shiftmap->dispcoef[4]=-6.537e-09;
      for(k=0;k<N_DEG_DISPERSION_LAW+1;k++)
	{
	  shiftmap->errdispcoef[k]=0.;
	  }*/
    }

  else
    {
      nbmaxminrejected=0;
      if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "MED") ) {
	nbmaxminrejected=1;
      }
      if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "HIGH") ) {
	nbmaxminrejected=1;
      }
      if(!strcmp(chipName, "AQUARIUS") && !strcmp(resolution, "HIGH") ) {
	nbmaxminrejected=1;
      }
      // Compute the coefficients giving the dispersion law
      // by Fitting a polynomial 
      coeff= cpl_matrix_new(nbmaxmin-nbmaxminrejected, 3);
      if (coeff == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
	return NULL;
      }
      rhs= cpl_matrix_new(nbmaxmin-nbmaxminrejected, 1);
      if (rhs == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
	cpl_matrix_delete(coeff);
	return NULL;
      }
  
      for(k=0; k< nbmaxmin-nbmaxminrejected; k++) {
	index=k;
	if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "MED") ) {
	  index=k+1;
	}
	if(!strcmp(chipName, "HAWAII-2RG") && !strcmp(resolution, "HIGH") ) {
	  index=k+1;
	}
	// Fill a matrix of coefficients
	cpl_matrix_set(coeff, k, 0, 1);
	/* val= cpl_vector_get(specFeature, k ); */
	val= refpix[index];
	val+=shiftmap->imgdet->list_region[0]->corner[1];
	cpl_matrix_set(coeff, k, 1, val);
	cpl_matrix_set(coeff, k, 2, pow(val, 2));
	// Fill a matrix with the position of maxima and minima
	// val= cpl_matrix_get(distor_index_max[0], nb_column[0]/2, k);
	cpl_matrix_set(rhs, k, 0, cpl_vector_get(fitvals,index));
      }
      // Solve a linear system to find the distorsion dispersion coefficients
      // and append the result to the matrix
      coefftrans=cpl_matrix_transpose_create(coeff);
      if (coefftrans == NULL) {
	cpl_msg_error(cpl_func,"could not transpose a cpl_matrix");
	cpl_matrix_delete(rhs);
	return NULL;
      }
      normal=cpl_matrix_product_create(coefftrans,coeff);
      if (normal == NULL) {
	cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix");
	cpl_matrix_delete(rhs);
	return NULL;
      }
      invnormal=cpl_matrix_invert_create(normal);
      if (invnormal == NULL) {
	cpl_msg_error(cpl_func,"could not invert a cpl_matrix");
	cpl_matrix_delete(rhs);
	return NULL;
      }
      cpl_matrix_delete(normal);
      matrix_temp=cpl_matrix_product_create(coefftrans,rhs);
      if (matrix_temp == NULL) {
	cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix");
	return NULL;
      }
      sol=cpl_matrix_product_create(invnormal,matrix_temp);
      if (sol == NULL) {
	cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix");
	return NULL;
      }
      cpl_matrix_delete(matrix_temp);
      cpl_matrix_delete(coefftrans);
      for(k=0;k<3;k++) {
	shiftmap->dispcoef[k]=cpl_matrix_get(sol,k,0);
	shiftmap->errdispcoef[k]=ERR_MAX_PIX*sqrt(cpl_matrix_get(invnormal,k,k));
      }
      for(k=3;k<N_DEG_DISPERSION_LAW;k++) {
	shiftmap->dispcoef[k]=0.;
	shiftmap->errdispcoef[k]=0.;
      }
      cpl_matrix_delete(invnormal);
      cpl_matrix_delete(sol);
      cpl_matrix_delete(coeff);
      cpl_matrix_delete(rhs);
 
      cpl_vector_delete(fitvals);
      cpl_free(boundaries);
      cpl_array_delete(typemaxmin);
    }

  // Save spectrum of Plastic Foil
  if ( cpl_parameter_get_bool(debug) ) {
    file = fopen("spectrum_plasticfoil.dat", "w+");
    /* for(i=0;i<cpl_image_get_size_y(spectra_corrected[0]->imgdata-> */
    /* 				   list_frame[0]->list_subwin[2]->imgreg[0]);i++) { */
    for(i=0;i<cpl_image_get_size_y(spectra_raw[2]);i++) {
      k=shiftmap->imgdet->list_region[0]->corner[1]+i;
      fprintf(file,"%f ",shiftmap->dispcoef[0]+shiftmap->dispcoef[1]*k+
	      shiftmap->dispcoef[2]*k*k+shiftmap->dispcoef[3]*k*k*k+
	      shiftmap->dispcoef[4]*k*k*k*k);
      for(j=0;j<nbregion;j++) {
	fprintf(file,"%f ",cpl_vector_get(spectraObs[j],i));
      }
      fprintf(file,"\n");
    }
    fclose(file);
  }
  for (i=0;i<nbregion;i++) {
    cpl_vector_delete(spectraObs[i]);
  }
  cpl_free(spectraObs);


// Compute the QC1 parameters
  zero_rate= 0.5;
  nInterf= 0;
  for(i=0; i<nbregion-1; i++) {
    nPhot[i]=0;
  }
  // Find the "I" interferometer region
  for(i=0; i<nbregion; i++) {
    /* if( spectra_corrected[0]->imgdet->list_region[i]->correlation == 2 ) { */
    /*   for(j=0; j<(*spectra_corrected[0]).imgdet->list_region[i]->naxis[1]; j++) { */
    if( spectra[0]->imgdet->list_region[i]->correlation == 2 ) {
      for(j=0; j<(*spectra[0]).imgdet->list_region[i]->naxis[1]; j++) {
        cpt=0;
        // Count the number of zero values for each row
        /* for(k=0; k<(*spectra_corrected[0]).imgdet->list_region[i]->naxis[0]; k++) { */
        for(k=0; k<(*spectra[0]).imgdet->list_region[i]->naxis[0]; k++) {
      
         /* if(cpl_image_get( */
         /*   (*spectra_corrected[0]).imgdata->list_frame[0]->list_subwin[i]->imgreg[0], */
         /*    k+1, j+1, &out) < EPSILON) { */
         if(cpl_image_get(spectra_raw[i],k+1, j+1, &out) < EPSILON) {
            cpt++;
          }
        }
        // Increment nInterf if this number is below the zero_rate
        /* if(cpt < (*spectra_corrected[0]).imgdet->list_region[i]->naxis[0] * zero_rate) { */
        if(cpt < (*spectra[0]).imgdet->list_region[i]->naxis[0] * zero_rate) {
          nInterf++;
          // -------------------------------------------------------------
          // Verify in the photometric regions that the number of zeros
          // of this row is below the zero_rate too
          flag=0;
          for(l=0; l<nbregion; l++) {
            /* if( spectra_corrected[0]->imgdet->list_region[l]->correlation == 2 ) { */
            if( spectra[0]->imgdet->list_region[l]->correlation == 2 ) {
              flag=1;
            }
            /* if( spectra_corrected[0]->imgdet->list_region[l]->correlation == 1 ) { */
            if( spectra[0]->imgdet->list_region[l]->correlation == 1 ) {
	      cpt=0;
	      // Count the number of zero values for each row
	      /* for(n=0; n<(*spectra_corrected[0]).imgdet->list_region[l]->naxis[0]; n++) { */
	      for(n=0; n<(*spectra[0]).imgdet->list_region[l]->naxis[0]; n++) {
		/* if (cpl_image_get((*spectra_corrected[0]).imgdata->list_frame[0] */
                /*                   ->list_subwin[l]->imgreg[0], n+1, j+1, &out) < EPSILON) { */
		if (cpl_image_get(spectra_raw[l], n+1, j+1, &out) < EPSILON) {
		  cpt++;
		}
	      }
	      /* if(cpt < (*spectra_corrected[0]).imgdet->list_region[l]->naxis[0] * zero_rate) { */
	      if(cpt < (*spectra[0]).imgdet->list_region[l]->naxis[0] * zero_rate) {
		nPhot[l-flag]++;
	      }
	    }
          }
          // -------------------------------------------------------------
        }
      }
    }
  }

// Store the results in a cpl_propertylist
  qclist= cpl_propertylist_new();
  if (qclist == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for cpl_propertylist");
    return NULL;
  }

  flag=0;
  // Add the "QC DETi PICOVj" values as keyword3s in the property list
  for(i=0; i<nbregion; i++) {
    /* if( spectra_corrected[0]->imgdet->list_region[i]->correlation == 1 ) { */
    if( spectra[0]->imgdet->list_region[i]->correlation == 1 ) {
      // case band N :
      if(!strcmp(chipName, "AQUARIUS")) {
	qcname=cpl_sprintf("ESO QC DET%d PICOV%d", 2, i+1-flag);
      }
      // case band L :
      if(!strcmp(chipName, "HAWAII-2RG")) {
	qcname=cpl_sprintf("ESO QC DET%d PICOV%d", 1, i+1-flag);
      }
      if(nInterf == 0) {
	cpl_propertylist_append_double(qclist, qcname, 0.);
      } else {
	cpl_propertylist_append_double(qclist, qcname, (nPhot[i-flag]*1.0)/nInterf);
      }
      cpl_free(qcname);
    } else {
      flag=1;
    }
    
  }
  
  mat_add_generic_qc(shiftmap->keywords,qclist);

  // Add QC parametr to the keyword of the primary header
  cpl_propertylist_append(shiftmap->keywords,qclist);
  cpl_propertylist_delete(qclist);



/* // Compute the CRVAL, DMC and CD coefficients */
/* // and store them as keywords of the primary header */

/*   for(i=0; i<nbregion; i++) { */

/*     // Distorsion in the X direction */
/*     for (j=0;j<3;j++) { */
/*       // Fit a 2nd order polynome to each column of shiftcoefx and shiftcoefy */
/*       coeff= cpl_matrix_new(nb_row[i], 3); */
/*       if (coeff == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       rhs= cpl_matrix_new(nb_row[i], 1); */
/*       if (rhs == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       for(k=0; k< nb_row[i]; k++) { */
/* 	// Fill a matrix of coefficients */
/* 	cpl_matrix_set(coeff, k, 0, 1); */
/* 	cpl_matrix_set(coeff, k, 1, (double)k ); */
/* 	cpl_matrix_set(coeff, k, 2, pow(k, 2)); */
/* 	// Fill a matrix with the values of  */
/* 	cpl_matrix_set(rhs, k, 0,  */
/* 		       cpl_matrix_get(shiftmap->list_shiftcoefx[i]->coefdistor,j,k)); */
/*       } */
/*       // Solve the linear system  */
/*       coefftrans=cpl_matrix_transpose_create(coeff); */
/*       if (coefftrans == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not transpose cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       normal=cpl_matrix_product_create(coefftrans,coeff); */
/*       if (normal == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       invnormal=cpl_matrix_invert_create(normal); */
/*       if (invnormal == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not invert a cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       cpl_matrix_delete(normal); */
/*       matrix_temp=cpl_matrix_product_create(coefftrans,rhs); */
/*       if (matrix_temp == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       sol=cpl_matrix_product_create(invnormal,matrix_temp); */
/*       if (sol == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       cpl_matrix_delete(matrix_temp); */
/*       cpl_matrix_delete(coefftrans); */
/*       for(k=0;k<3;k++) { */
/* 	if (j == 0) { */
/* 	  coef_a[k]=cpl_matrix_get(sol,k,0); */
/* 	} */
/* 	if (j == 1) { */
/* 	  coef_b[k]=cpl_matrix_get(sol,k,0); */
/* 	} */
/* 	if (j == 2) { */
/* 	  coef_c[k]=cpl_matrix_get(sol,k,0); */
/* 	} */
/*       } */
/*       cpl_matrix_delete(invnormal); */
/*       cpl_matrix_delete(sol); */
/*       cpl_matrix_delete(coeff); */
/*       cpl_matrix_delete(rhs); */
/*     } */
/*     // Distorsion in the Y direction */
/*     for (j=0;j<3;j++) { */
/*       // Fit a 2nd order polynome to each column of shiftcoefy */
/*       coeff= cpl_matrix_new(nb_column[i], 3); */
/*       if (coeff == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       rhs= cpl_matrix_new(nb_column[i], 1); */
/*       if (rhs == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       for(k=0; k< nb_column[i]; k++) { */
/* 	// Fill a matrix of coefficients */
/* 	cpl_matrix_set(coeff, k, 0, 1); */
/* 	cpl_matrix_set(coeff, k, 1, (double)k ); */
/* 	cpl_matrix_set(coeff, k, 2, pow(k, 2)); */
/* 	// Fill a matrix with the values of  */
/* 	cpl_matrix_set(rhs, k, 0,  */
/* 		       cpl_matrix_get(shiftmap->list_shiftcoefy[i]->coefdistor,j,k)); */
/*       } */
/*       // Solve the linear system  */
/*       coefftrans=cpl_matrix_transpose_create(coeff); */
/*       if (coefftrans == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not transpose cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       normal=cpl_matrix_product_create(coefftrans,coeff); */
/*       if (normal == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       invnormal=cpl_matrix_invert_create(normal); */
/*       if (invnormal == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not invert a cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       cpl_matrix_delete(normal); */
/*       matrix_temp=cpl_matrix_product_create(coefftrans,rhs); */
/*       if (matrix_temp == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       sol=cpl_matrix_product_create(invnormal,matrix_temp); */
/*       if (sol == NULL) { */
/* 	cpl_msg_error(cpl_func,"could not multiply 2 cpl_matrix"); */
/* 	return NULL; */
/*       } */
/*       cpl_matrix_delete(matrix_temp); */
/*       cpl_matrix_delete(coefftrans); */
/*       for(k=0;k<3;k++) { */
/* 	if (j == 0) { */
/* 	  coef_d[k]=cpl_matrix_get(sol,k,0); */
/* 	} */
/* 	if (j == 1) { */
/* 	  coef_e[k]=cpl_matrix_get(sol,k,0); */
/* 	} */
/* 	if (j == 2) { */
/* 	  coef_f[k]=cpl_matrix_get(sol,k,0); */
/* 	} */
/*       } */
/*       cpl_matrix_delete(invnormal); */
/*       cpl_matrix_delete(sol); */
/*       cpl_matrix_delete(coeff); */
/*       cpl_matrix_delete(rhs); */
/*     } */
/*   } */
  

  /* nrtslist= cpl_propertylist_new();  */
  /* if (nrtslist == NULL) { */
  /*   cpl_msg_error(cpl_func,"could not allocate memory for cpl_propertylist"); */
  /*   return NULL; */
  /* } */
  /* flag=0; */
  /* for(i=0; i<nbregion; i++) { */
  /*   if( spectra_corrected[0]->imgdet->list_region[i]->correlation == 2 ) { */
  /*     flag=1; */
  /*     qcname=cpl_sprintf("ESO PRO INTERF CRVAL1"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, shiftmap->dispcoef[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF CD00"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, COEF_GAMMA); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF CD11"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, shiftmap->dispcoef[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC00"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_a[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC01"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_b[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC02"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_a[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC03"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_b[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC04"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_c[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC05"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_a[2]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC06"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_c[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC07"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_b[2]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC08"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_c[2]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC10"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_d[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC11"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_e[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC12"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_d[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC13"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_e[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC14"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_f[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC15"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_d[2]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC16"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_f[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC17"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_e[2]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO INTERF DMC18"); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_f[2]); */
  /*     cpl_free(qcname); */
  /*   }  */
  /*   if( spectra_corrected[0]->imgdet->list_region[i]->correlation == 1 ) { */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d CRVAL1", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, shiftmap->dispcoef[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d CD00", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, COEF_GAMMA); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d CD11", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, shiftmap->dispcoef[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC00", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_a[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC01", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_b[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC02", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_a[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC03", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_b[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC04", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_c[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC05", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_a[2]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC06", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_c[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC07", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_b[2]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC08", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_c[2]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC10", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_d[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC11", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_e[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC12", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_d[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC13", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_e[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC14", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_f[0]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC15", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_d[2]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC16", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_f[1]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC17", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_e[2]); */
  /*     cpl_free(qcname); */
  /*     qcname=cpl_sprintf("ESO PRO PHOT%d DMC18", i+1-flag); */
  /*     cpl_propertylist_append_double(nrtslist, qcname, coef_f[2]); */
  /*     cpl_free(qcname); */
  /*   } */
  
  /* } */
  /* cpl_propertylist_append(shiftmap->keywords,nrtslist); */
  /* cpl_propertylist_delete(nrtslist); */

// Free Memory
  for(i=0; i<nbregion; i++) {
    cpl_image_delete(distor_h1_raw[i]);
    cpl_image_delete(distor_h2_raw[i]);
    cpl_image_delete(distor_h3_raw[i]);
    cpl_image_delete(distor_background[i]);
    cpl_image_delete(spectra_raw[i]);
    cpl_image_delete(spectra_background[i]);

  }
  cpl_free(distor_h1_raw);
  cpl_free(distor_h2_raw);
  cpl_free(distor_h3_raw);
  cpl_free(distor_background);
  cpl_free(spectra_raw);
  cpl_free(spectra_background);
  cpl_free(nbimg);

  /* for(i=0; i< nbframes[2]; i++) { */
  /*   mat_gendata_delete(spectra_corrected[i]); */
  /* } */
  /* //  for(i=0; i< nbframes[0]*3; i++) { */
  /* //    mat_gendata_delete(distor_corrected[i]); */
  /* //  } */
  /* cpl_free(spectra_corrected); */
  /* cpl_free(distor_corrected); */
  
  /* if (!cpl_errorstate_is_equal(prestate)) {  */
  /*   cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one); */
  /*   cpl_errorstate_set(prestate);     */

  /*   return NULL; */
  /* } */
  return shiftmap;
}



/*----------------------------------------------------------------------------*/
/**
  @ingroup shift
  @brief Correct data from distorsion
  @param gendata : contains the data to correct
  @param shift : contains the distorsion coefficients
  @param border            0: extrapolation at the border 
                           1: border filled by zero
  @return mat_gendata : corrected data
 */
/*----------------------------------------------------------------------------*/
mat_gendata *mat_apply_shift(mat_gendata *gendata, 
                             mat_shiftmap *shift,
                             int border) {
    mat_gendata *gendata_corrected=NULL;
/*
    int i=0;
    int j=0;
    int k=0;
    int l=0;
    int m=0;
    int n=0;
    int mi=0;
    register int xmax=0;
    register int ymax=0;
    register int cornerx=0;
    register int cornery=0;
    register int cornerxShift=0;
    register int corneryShift=0;
    register int ic=0;
    register int jc=0;
    register int jtemp=0;
    register int ic1=0;
    register int jc1=0;
    register int ic2=0;
    register int jc2=0;
    register int k1=0;
    register int l1=0;
    int out=0;

    register int xliminf=0;
    register int xlimsup=0;
    register int yliminf=0;
    register int ylimsup=0;
    register int boxszx=0;
    register int boxszy=0;
    register int flagout=0;
    register int flagbox=0;
    register double valmin=0.;
    register double val=0.;
    register double dist=0.;
    register double a=0.;
    register double b=0.;
    register double c=0.;
    register double d=0.;
    register double e=0.;
    register double f=0.;
     register double valmax=0;
    register double distmean=0;

 */
    char *detNameData=NULL;
    char *detNameShift=NULL;
    char *specResData=NULL;
    char *specResShift=NULL;
    int flagSciPhot=0;
    cpl_errorstate prestate = cpl_errorstate_get();

    // Check input parameters
    mat_assert_value((gendata!=NULL),CPL_ERROR_NULL_INPUT, NULL,"no mat_gendata argument given");
    mat_assert_value((shift!=NULL),CPL_ERROR_NULL_INPUT, NULL, "no mat_shiftmap argument given");
    detNameData=(char *)cpl_propertylist_get_string(gendata->keywords,"ESO DET CHIP NAME");
    detNameShift=(char *)cpl_propertylist_get_string(shift->keywords,"ESO DET CHIP NAME");
    if ( strcmp(detNameData,detNameShift) ) {
        cpl_msg_error(cpl_func,"The SHIFT_MAP is not compatible with the data (Detector)");
        return NULL;
    }
    // Check detector and spectral resolution
    if(!strcmp(detNameData, "AQUARIUS")) {
        if (cpl_propertylist_has(gendata->keywords,"ESO INS DIN POS")) {
            specResData=(char *)cpl_propertylist_get_string(gendata->keywords,"ESO INS DIN POS");
        } else {
            specResData=(char *)cpl_propertylist_get_string(gendata->keywords,"ESO INS DIN ID");
        }
        if (cpl_propertylist_has(shift->keywords,"ESO INS DIN POS")) {
            specResShift=(char *)cpl_propertylist_get_string(shift->keywords,"ESO INS DIN POS");
        } else {
            specResShift=(char *)cpl_propertylist_get_string(shift->keywords,"ESO INS DIN ID");
        }
    } else {
        if (cpl_propertylist_has(gendata->keywords,"ESO INS DIL POS")) {
            specResData=(char *)cpl_propertylist_get_string(gendata->keywords,"ESO INS DIL POS");
        } else {
            specResData=(char *)cpl_propertylist_get_string(gendata->keywords,"ESO INS DIL ID");
        }
        if (cpl_propertylist_has(shift->keywords,"ESO INS DIL POS")) {
            specResShift=(char *)cpl_propertylist_get_string(shift->keywords,"ESO INS DIL POS");
        } else {
            specResShift=(char *)cpl_propertylist_get_string(shift->keywords,"ESO INS DIL ID");
        }
    }
    if ( strcmp(specResData,specResShift) ) {
        cpl_msg_warning(cpl_func,"The SHIFT_MAP is not compatible with the data (spectral resolution)");
        /* return NULL; */
    }
    for(cpl_size i=0; i<gendata->imgdet->nbregion; i++) {
        if (gendata->imgdet->list_region[i]->correlation == 1) {
            flagSciPhot=1;
        }
    }

    gendata_corrected = mat_gendata_duplicate(gendata, CPL_TYPE_FLOAT);
    if (gendata_corrected == NULL) {
        cpl_msg_error(cpl_func,"could not duplicate a mat_gendata");
        return NULL;
    }
    // Set default values for distorsion coefficients if they are missing
    for(cpl_size i=0; i<shift->nbreg; i++) {
        if(shift->list_shiftcoefx[i]->coefdistor == NULL) {
            shift->list_shiftcoefx[i]->coefdistor =
                            cpl_matrix_new(shift->list_shiftcoefx[i]->order,
                                            shift->list_shiftcoefx[i]->nlig);
            if (shift->list_shiftcoefx[i]->coefdistor == NULL) {
                cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
                return NULL;
            }

            for(cpl_size j=0; j<shift->list_shiftcoefx[i]->nlig; j++) {
                cpl_matrix_set(shift->list_shiftcoefx[i]->coefdistor, 0, j, 0);
                cpl_matrix_set(shift->list_shiftcoefx[i]->coefdistor, 1, j, 1);
                cpl_matrix_set(shift->list_shiftcoefx[i]->coefdistor, 2, j, 0);
            }
        }
    }
    for(cpl_size i=0; i<shift->nbreg; i++) {
        if(shift->list_shiftcoefy[i]->coefdistor == NULL) {
            shift->list_shiftcoefy[i]->coefdistor =
                            cpl_matrix_new(shift->list_shiftcoefy[i]->order,
                                            shift->list_shiftcoefy[i]->nlig);
            if (shift->list_shiftcoefx[i]->coefdistor == NULL) {
                cpl_msg_error(cpl_func,"could not allocate memory for cpl_matrix");
                return NULL;
            }

            for(cpl_size j=0; j<shift->list_shiftcoefy[i]->nlig; j++) {
                cpl_matrix_set(shift->list_shiftcoefy[i]->coefdistor, 0, j, 0);
                cpl_matrix_set(shift->list_shiftcoefy[i]->coefdistor, 1, j, 1);
                cpl_matrix_set(shift->list_shiftcoefy[i]->coefdistor, 2, j, 0);
            }
        }
    }
    // Compute the new intensities' location for each image

    for(cpl_size m=0; m<gendata->imgdet->nbregion; ++m) {
        // Get dimension of this image
        const int xmax= gendata->imgdet->list_region[m]->naxis[0];
        const int ymax= gendata->imgdet->list_region[m]->naxis[1];
        const int cornerx=gendata->imgdet->list_region[m]->corner[0]-1;
        const int cornery=gendata->imgdet->list_region[m]->corner[1]-1;
        int mi = 0;
        if (flagSciPhot == 1) {
            mi=m-(m/5)*5;
        } else {
            mi=2;
        }
        const int cornerxShift=shift->imgdet->list_region[mi]->corner[0]-1;
        const int corneryShift=shift->imgdet->list_region[mi]->corner[1]-1;
        double * pX=calloc(xmax*ymax,sizeof(double));
        if (pX == NULL) {
            cpl_msg_error(cpl_func,"could not allocate memory for double");
            return NULL;
        }
        double * pY=calloc(xmax*ymax,sizeof(double));
        if (pY == NULL) {
            cpl_msg_error(cpl_func,"could not allocate memory for double");
            free(pX);
            return NULL;
        }
        // Apply the distorsion map
        double distmean=0.;
        for(cpl_size i=0; i<xmax; i++) {
            for(cpl_size j=0; j<ymax; j++) {
                const int k=i+cornerx-cornerxShift;
                const int l=j+cornery-corneryShift;
                pX[j*xmax+i]=cpl_matrix_get(shift->list_shiftcoefx[mi]->coefdistor, 0, l) +
                                (cpl_matrix_get(shift->list_shiftcoefx[mi]->coefdistor, 1, l) * k) +
                                (cpl_matrix_get(shift->list_shiftcoefx[mi]->coefdistor, 2, l)*pow(k,2))-
                                (cornerx-cornerxShift);
                pY[j*xmax+i]=cpl_matrix_get(shift->list_shiftcoefy[mi]->coefdistor, 0, k) +
                                (cpl_matrix_get(shift->list_shiftcoefy[mi]->coefdistor, 1, k) * l) +
                                (cpl_matrix_get(shift->list_shiftcoefy[mi]->coefdistor, 2, k)*pow(l,2))-
                                (cornery-corneryShift);
                const double dist=sqrt(pow(k-pX[j*xmax+i],2.0)+pow(l-pY[j*xmax+i],2.0));
                distmean+=dist;
            }
        }
        /* printf("DISTORTION MEAN=%f\n",distmean/(xmax*ymax)); */

        cpl_msg_info(cpl_func,"Starting parallel loop of %d", xmax);
        gettimeofday(&tv1, NULL);

        HDRL_OMP(omp parallel for)
        for(cpl_size k=0; k<xmax; ++k) {
            int boxszx=4;
            int boxszy=6;
            for(cpl_size l=0; l<ymax; ++l) {
                // Compute the new coordinates of each pixel
                double valmin=1.E10;
                int ic=0;
                int jc=0;

                // The size of the box is increased in both directions
                // if the limit is reached.
                boxszx-=2;
                boxszy-=2;
                if (boxszx < 2 ) boxszx=2;
                if (boxszy < 4 ) boxszy=4;;

                int flagbox=1;
                while (flagbox == 1) {
                    int xlimsup=k+boxszx;
                    if (xlimsup > xmax-1) xlimsup=xmax-1;
                    int xliminf=k-boxszx;
                    if (xliminf < 0) xliminf=0;
                    int ylimsup=l+boxszy;
                    if (ylimsup > ymax-1) ylimsup=ymax-1;
                    int yliminf=l-boxszy;
                    if (yliminf < 0) yliminf=0;

                    for(cpl_size j=yliminf; j<=ylimsup; ++j) {
                        cpl_size jtemp=j*xmax;
                        for(cpl_size i=xliminf; i<=xlimsup; ++i) {
                            const double dist=sqrt(pow(k-pX[jtemp+i], 2) + pow(l-pY[jtemp+i], 2));
                            if(dist< valmin) {
                                valmin= dist;
                                ic= i;
                                jc= j;
                            }
                        }
                    }
                    if (llabs(k-ic) == boxszx) {
                        boxszx+=2;
                        flagbox=1;
                    } else {
                        flagbox=0;
                    }
                    if (llabs(l-jc) == boxszy) {
                        boxszy+=2;
                        flagbox=1;
                    }
                }
                int flagout=0;

                if(k < pX[jc*xmax+ic]) {
                    ic--;
                    if(ic< 0) {
                        ic= 0;
                        flagout=1;
                    }
                }
                if(l < pY[jc*xmax+ic]) {
                    jc--;
                    if(jc< 0) {
                        jc= 0;
                        flagout=1;
                    }
                }
                if(ic >= xmax-1) {
                    ic=xmax-2;
                    flagout=1;
                }
                if(jc >= ymax-1) {
                    jc=ymax-2;
                    flagout=1;
                }

                const int ic1=ic+1;
                const int jc1=jc+1;
                const int ic2=ic1+1;
                const int jc2=jc1+1;
                const double a=pX[jc*xmax+ic];
                const double b=pX[jc*xmax+ic1];
                const double c=pY[jc1*xmax+ic];
                const double d=pY[jc*xmax+ic];
                const double e=pY[jc1*xmax+ic1];
                const double f=pY[jc*xmax+ic1];
                const int k1=k+1;
                const int l1=l+1;
                // Set the data corrected in a new mat_gendata structurecpl_msg_info

                for(cpl_size n=0; n<gendata->imgdata->nbframe; ++n) {
                    if ( (flagout == 0) || (border == 0) ) {
                        int out = 0 ;
                        double val=(b-k)*((c-l)*(cpl_image_get(gendata->imgdata->list_frame[n]
                                                                                     ->list_subwin[m]->imgreg[0], ic1, jc1, &out))
                                                                                     + (l-d)*(cpl_image_get(gendata->imgdata->list_frame[n]
                                                                                                                                         ->list_subwin[m]->imgreg[0], ic1, jc2, &out)))
                                                                                                                                         +(k-a)*((e-l)*(cpl_image_get(gendata->imgdata->list_frame[n]
                                                                                                                                                                                                   ->list_subwin[m]->imgreg[0], ic2, jc1, &out))
                                                                                                                                                                                                   + (l-f)*(cpl_image_get(gendata->imgdata->list_frame[n]
                                                                                                                                                                                                                                                       ->list_subwin[m]->imgreg[0], ic2, jc2, &out)));
                        if (isnan(val) != 0) {
                            val=0.;
                        }
                        cpl_image_set(gendata_corrected->imgdata->list_frame[n]->list_subwin[m]->imgreg[0],k1, l1,val);

                    } else {
                        cpl_image_set(gendata_corrected->imgdata->list_frame[n]->list_subwin[m]->imgreg[0],k1, l1,0.);
                    }
                }
            }
        }
        gettimeofday(&tv2, NULL);
        cpl_msg_info(cpl_func, "Wall time for parallel loop = %f seconds\n",
                     (double) (tv2.tv_usec - tv1.tv_usec) / 1000000 +
                     (double) (tv2.tv_sec - tv1.tv_sec));

        free(pX);
        free(pY);

    }

    /* Error handling */
    if (!cpl_errorstate_is_equal(prestate)) {

        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        cpl_errorstate_set(prestate);

        return NULL;
    }

    return gendata_corrected;
}
