
/* $Id: mat_utils.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_utils.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/
#include <ctype.h>
#include <string.h>
#include <math.h>

#include "mat_utils.h"

/*-----------------------------------------------------------------------------
  Define
  ----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  Functions prototypes
  ----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
int mat_get_sta_index(char * name) {
  const char * sta_name_list[]={"A0","A1","B0","B1","B2","B3","B4","B5","C0","C1",
				"C2","C3","D0","D1","D2","E0","G0","G1","G2","H0",
				"I1","J1","J2","J3","J4","J5","J6","K0","L0","M0",
				"I2","U1","U2","U3","U4"};
  for(int i=0;i<35;i++) {
    if (strstr(name,sta_name_list[i]) != NULL) {
      return (i+1);
    }
  }
  return 0;    
}

cpl_error_code mat_get_staxyz_from_staindex(mat_array *oiarray,int staindex,double xyz[3]) {
  for(int i=0;i<oiarray->nbstation;i++) {
    if (staindex == oiarray->list_station[i]->staindex) {
      xyz[0]=oiarray->list_station[i]->staxyz[0];
      xyz[1]=oiarray->list_station[i]->staxyz[1];
      xyz[2]=oiarray->list_station[i]->staxyz[2];
      return CPL_ERROR_NONE;
    }
  }
  cpl_msg_warning(cpl_func,"Staion not found");
  return CPL_ERROR_DATA_NOT_FOUND;
}

/**
   @brief Remove all spaces from a (non-const) string
   @param *str          string
   @return name of the file string without spaces
*/
void remove_spaces(char *str) {
  char *src=str;
  char *dst=src;
  while (*src != '\0') {
    if (*src != ' ') {
      *dst++=*src;
    }
    src++;
  }
  *dst='\0';
}

/**
   @brief Extract the filename in a path
   @param *name          full path
   @return name of the file 
*/
char * basename(const char *name) {
  const char *base = name;
  while (*name) {
    if (*name++ == '/') {
      base=name;
    }
  }
  return (char *) base;
}

/**
   @brief Gets a string from a property list
   @param dst    The memory where the string is stored
   @param size   The available size (including '\0')
   @param plist  The property list
   @param name   The property name

   This function gets a string value from a property list and stores
   it in a buffer. The buffer size limits the length of the string value.
   It is guaranteed that the buffer will contain a '\0'-terminated string.
   If the property does not exist, the buffer will contain an empty string.
*/
void mat_propertylist_get_string(char *dst, int size, cpl_propertylist *plist, 
                                 const char *name)
{
  const char *value = cpl_propertylist_get_string(plist, name);
  if (value == NULL)
    {
      cpl_msg_error(cpl_func, "keyword %s is not in the property list", name);
      *dst = '\0';
    }
  else
    {
      strncpy(dst, value, size - 1);
      dst[size - 1] = '\0';
    }
}

/**
   @brief Gets a string from a property list or uses a default value.
   @param dst    A already allocated string (optional).
   @param size   The size of the preallocated string (or the new string or 0).
   @param plist  The property list containing (hopefully) the value.
   @param name   The name of the property.
   @param defval The default value (can be NULL).
   @returns The string value or NULL.

   This function gets a keyword value as string from a property list.
*/
char *mat_propertylist_get_string_default(char *dst, int size,
                                          cpl_propertylist *plist, 
                                          const char *name,
                                          const char *defval)
{
  const char *value = cpl_propertylist_get_string(plist, name);
  if (value == NULL)
    { // we did not find the property in the list -> use the default value (this can also be NULL)
      cpl_error_reset();
      value = defval;
    }
  if (value != NULL)
    { // we have found the keyword or we have a default value
      if (dst == NULL)
	{ // we have no preallocated string
	  if (size == 0)
	    { // the size is 0 -> use the original string size
	      size = strlen(value) + 1;
	    }
	  // allocate the space for the result
	  dst = (char *)cpl_calloc(size, sizeof(char));
	}
      strncpy(dst, value, size);
      dst[size - 1] = '\0';
      return dst;
    }
  else
    { // we do not have the requested keyword in the property list and we have no default value
      if (dst == NULL)
        { // we have no preallocated string
	  if (size == 0)
            { // No size is given -> return NULL;
	      return NULL;
            }
	  else
            { // a size for the restul is given -> cpl_calloc()
	      return cpl_calloc(size, sizeof(char));
            }
        }
      else
        { // we have a preallocated string => '\0'
	  *dst = '\0';
	  return dst;
        }
    }
}

/**
   @brief Gets an integer from a property list or uses a default value.
   @param plist  The property list containing (hopefully) the value.
   @param name   The name of the property.
   @param defval The default value.
   @returns The default value or the value from the property.

   This function gets a keyword value as integer from a property list.
*/
int mat_propertylist_get_int_default(cpl_propertylist *plist, 
                                     const char *name,
                                     int defval)
{
  const cpl_property *prop = cpl_propertylist_get_property_const(plist, name);
  if (prop == NULL)
    { // no property with the given name found => default
      return defval;
    }
  else
    {
      switch (cpl_property_get_type(prop))
        {
	case CPL_TYPE_INT:
	  return cpl_property_get_int(prop);
	case CPL_TYPE_LONG:
	  return (int)cpl_property_get_long(prop);
	case CPL_TYPE_LONG_LONG:
	  return (int)cpl_property_get_long_long(prop);
	case CPL_TYPE_FLOAT:
	  return (int)rintf(cpl_property_get_float(prop));
	case CPL_TYPE_DOUBLE:
	  return (int)rint(cpl_property_get_double(prop));
	default:
	  return defval;
        }
    }
}

/**
   @brief Gets an double from a property list or uses a default value.
   @param plist  The property list containing (hopefully) the value.
   @param name   The name of the property.
   @param defval The default value.
   @returns The default value or the value from the property.

   This function gets a keyword value as double from a property list.
*/
double mat_propertylist_get_double_default(cpl_propertylist *plist, 
                                           const char *name,
                                           double defval)
{
  const cpl_property *prop = cpl_propertylist_get_property_const(plist, name);
  if (prop == NULL)
    { // no property with the given name found => default
      return defval;
    }
  else
    {
      switch (cpl_property_get_type(prop))
        {
	case CPL_TYPE_INT:
	  return (double)cpl_property_get_int(prop);
	case CPL_TYPE_LONG:
	  return (double)cpl_property_get_long(prop);
	case CPL_TYPE_LONG_LONG:
	  return (double)cpl_property_get_long_long(prop);
	case CPL_TYPE_FLOAT:
	  return (double)cpl_property_get_float(prop);
	case CPL_TYPE_DOUBLE:
	  return cpl_property_get_double(prop);
	default:
	  return defval;
        }
    }
}

/**
   @brief Extracts a string from a property list and make a copy
   @param plist  The property list
   @param name   The property name
   @returns An allocated string (maybe empty)

   This function gets a string value from a property list and makes a copy.
   If the property does not exist, an empty string ("") is returned. The caller
   must free the string memory.
*/
char *mat_propertylist_copy_string(cpl_propertylist *plist, const char *name)
{
  const char *value = cpl_propertylist_get_string(plist, name); 

  if (value == NULL)
    {
      cpl_msg_error(cpl_func, "keyword %s is not in the property list", name);
      return cpl_strdup("");
    }
  else
    {
      /* cpl_msg_info(cpl_func, "keyword %s = %s", name, value); */
      return cpl_strdup(value);
    }
}

/**
   @brief Find the sub-window which can be used for detector monitoring
   @param det       Contains the detector specification including dimensions.
   @param imgdet    Contains the sub-window setup.
   @returns The index of the sub-window used for detector monitoring or -1

   This function uses the detector specification to find a sub-window which
   covers all detector channels. This sub-window must cover the whole detector
   width and the position must be in the vertical center. Such a sub-window
   covers automatically all detector channels for a HAWAII-2RG and a Aquarius
   detector.
*/
int mat_find_central_region(mat_detector *det, mat_imagingdetector *imgdet)
{
  int             nr, r;
  int             found = -1;
  int             dist  = det->ny;

  /*
   * Because the sub-windows in the raw input data may covers only some 
   * regions
   * on the detector, try to find the sub-window which is located in the
   * central part of the detector and spans the whole width of the detector.
   * Such a sub-window covers automatically all detector channels for a 
   * HAWAII-2RG
   * and a Aquarius detector.
   */
  nr = imgdet->nbregion;
  for (r = 0; r < nr; r++)
    {
      mat_region *region = imgdet->list_region[r];
      /*
       * The sub-window must span the whole detector width.
       */
      if (region->naxis[0] != det->nx) continue;
      /*
       * The sub-window height must be at least det->channel_nx * 
       * det->channel_nrows,
       * because we use a square part of a detector channel 
       * (info->det->channel_nx * info->det->channel_nx)
       * for calculating the autocorrelation.
       */
      if (region->naxis[1] < (det->channel_nx * det->channel_nrows)) continue;
      /*
       * The sub-window position must be in the central part of the detector.
       * If a sub-window covers the whole detector, this is also ok.
       */
      if (det->channel_nrows == 1)
        { // the detector channels span the whole height -> look for the most central sub-window
	  int hdist = abs(det->ny/2 - (region->corner[1] + region->naxis[1]/2));
	  if (hdist < dist)
            {
	      found = r;
	      dist = hdist;
            }
        }
      else
        {
	  if (region->corner[1] != ((det->ny - region->naxis[1])/2) + 1) continue;
        }
      return r;
    }
  if (found != -1)
    { // the detector channels span the whole height and we found a useful sub-window
      return found;
    }
  cpl_msg_error(cpl_func, 
		"FITS file does not contain a sub-window which covers all detector channels");
  return -1;
}

/**
   @brief Extract two, comma separated, integers from a parameter.
   @param param  The parameter
   @param name   The parameter name (only for debugging)
   @param i1     The first integer in the list
   @param i2     The second integer in the list

   This function splits the string value of a parameter into three integers.
*/
void mat_parameter_get_int_double(cpl_parameter *param, const char *name, 
                                  int *i1, int *i2)
{
  const char       *arg = NULL;
  int               i, j;
  char              str[32];

  if (param == NULL)
    {
      return;
    }
  arg = cpl_parameter_get_string(param);
  i = 0;
  /* 1. Extract the region number from the poi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i1 = atoi(str);
  /* 2. Skip the comma character between the region number and 
     the x coordinate */
  if (arg[i] == ',')
    {
      i++;
    }
  /* 3. Extract the x coordinate from the poi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i2 = atoi(str);
  cpl_msg_info(cpl_func, "-%s=%s => %d,%d", name, arg, *i1, *i2);
}

/**
   @brief Extract three, comma separated, integers from a parameter.
   @param param  The parameter
   @param name   The parameter name (only for debugging)
   @param i1     The first integer in the list
   @param i2     The second integer in the list
   @param i3     The third integer in the list

   This function splits the string value of a parameter into three integers.
*/
void mat_parameter_get_int_triple(cpl_parameter *param, const char *name, 
                                  int *i1, int *i2, int *i3)
{
  const char       *arg = NULL;
  int               i, j;
  char              str[32];

  if (param == NULL)
    {
      return;
    }
  arg = cpl_parameter_get_string(param);
  i = 0;
  /* 1. Extract the region number from the poi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i1 = atoi(str);
  /* 2. Skip the comma character between the region number and 
     the x coordinate */
  if (arg[i] == ',')
    {
      i++;
    }
  /* 3. Extract the x coordinate from the poi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i2 = atoi(str);
  /* 4. Skip the comma character between the x coordinate 
     and the y coordinate */
  if (arg[i] == ',')
    {
      i++;
    }
  /* 5. Extract the y coordinate from the poi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i3 = atoi(str);
  cpl_msg_info(cpl_func, "-%s=%s => %d,%d,%d", name, arg, *i1, *i2, *i3);
}

/**
   @brief Extract four, comma separated, integers from a parameter.
   @param param  The parameter
   @param name   The parameter name (only for debugging)
   @param i1     The first integer in the list
   @param i2     The second integer in the list
   @param i3     The third integer in the list
   @param i4     The fourth integer in the list

   This function splits the string value of a parameter into four integers.
*/
void mat_parameter_get_int_quadruple(cpl_parameter *param, const char *name, 
                                     int *i1, int *i2, int *i3, int *i4)
{
  const char       *arg = NULL;
  int               i, j;
  char              str[32];

  if (param == NULL)
    {
      return;
    }
  arg = cpl_parameter_get_string(param);
  i = 0;
  /* 1. Extract the x position from the roi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i1 = atoi(str);
  /* 2. Skip the comma character */
  if (arg[i] == ',')
    {
      i++;
    }
  /* 3. Extract the y position from the roi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i2 = atoi(str);
  /* 4. Skip the comma character */
  if (arg[i] == ',')
    {
      i++;
    }
  /* 5. Extract the region width from the roi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i3 = atoi(str);
  /* 6. Skip the comma character */
  if (arg[i] == ',')
    {
      i++;
    }
  /* 7. Extract the region height from the roi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i4 = atoi(str);
  cpl_msg_info(cpl_func, "-%s=%s => %d,%d,%d,%d", name, arg, 
	       *i1, *i2, *i3, *i4);
}

/**
   @brief Extract five, comma separated, integers from a parameter.
   @param param  The parameter
   @param name   The parameter name (only for debugging)
   @param i1     The first integer in the list
   @param i2     The second integer in the list
   @param i3     The third integer in the list
   @param i4     The fourth integer in the list
   @param i5     The fifth integer in the list

   This function splits the string value of a parameter into five integers.
*/
void mat_parameter_get_int_quintuple(cpl_parameter *param, const char *name, 
                                     int *i1, int *i2, int *i3, int *i4, int *i5)
{
  const char       *arg = NULL;
  int               i, j;
  char              str[32];

  if (param == NULL)
    {
      return;
    }
  arg = cpl_parameter_get_string(param);
  i = 0;
  /* 1. Extract the region number from the roi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i1 = atoi(str);
  /* 2. Skip the comma character */
  if (arg[i] == ',')
    {
      i++;
    }
  /* 3. Extract the x position from the roi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i2 = atoi(str);
  /* 4. Skip the comma character */
  if (arg[i] == ',')
    {
      i++;
    }
  /* 5. Extract the y position from the roi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i3 = atoi(str);
  /* 6. Skip the comma character */
  if (arg[i] == ',')
    {
      i++;
    }
  /* 7. Extract the region width from the roi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i4 = atoi(str);
  /* 8. Skip the comma character */
  if (arg[i] == ',')
    {
      i++;
    }
  /* 9. Extract the region height from the roi string */
  j = 0;
  while (isdigit(arg[i]) || (arg[i] == '-'))
    {
      if (j < 30)
        {
	  str[j++] = arg[i];
        }
      i++;
    }
  str[j] = '\0';
  *i5 = atoi(str);
  cpl_msg_info(cpl_func, "-%s=%s => %d,%d,%d,%d,%d", name, arg, 
	       *i1, *i2, *i3, *i4, *i5);
}

/**
   @brief Rounds a floating point value to a given precision.
   @param x The floating point value which will be rounded.
   @param precision The final precision of the value.
   @returns The rounded value.

   This function rounds a floating point value to a specific precision. 
   The precision parameter specifies the
   number of digits after (positive precision value) of before 
   (negative precision value) the decimal point.
*/
double mat_round(double x, int precision)
{
  int i = 0;

  while ((precision - i) > 10)
    {
      x *= 1.0e10;
      i += 10;
    }
  while ((precision - i) > 0)
    {
      x *= 10.0;
      i += 1;
    }
  while ((precision - i) < -10)
    {
      x /= 1.0e10;
      i -= 10;
    }
  while ((precision - i) < 0)
    {
      x /= 10.0;
      i -= 1;
    }
  x = round(x);
  i = 0;
  while ((precision - i) > 10)
    {
      x /= 1.0e10;
      i += 10;
    }
  while ((precision - i) > 0)
    {
      x /= 10.0;
      i += 1;
    }
  while ((precision - i) < -10)
    {
      x *= 1.0e10;
      i -= 10;
    }
  while ((precision - i) < 0)
    {
      x *= 10.0;
      i -= 1;
    }
  return x;
}

/**
   @brief Rounds a floating point value to a given relative precision.
   @param x The floating point value which will be rounded.
   @param precision The final relative precision of the value.
   @returns The rounded value.

   This function rounds a floating point value to a specific precision. 
   The precision parameter specifies the number of significant digits.
*/
double mat_round_relative(double x, int precision)
{
  int  l;
  int neg = 0;

  if (x == 0.0) return 0.0;
  if (x < 0.0)
    {
      neg = 1;
      x = -x;
    }
  l = (int)ceil(log10(x));
  if (l > 0)
    {
      if (l > precision)
        { // more digits are before the decimal point than the precision
	  // -> divide - round - multiply
	  double scale = pow(10.0, (double)(l - precision));
	  x /= scale;
	  x = round(x);
	  x *= scale;
        }
      else
        {
	  double scale = pow(10.0, (double)(precision - l));
	  x *= scale;
	  x = round(x);
	  x /= scale;
        }
    }
  else
    {
      double scale = pow(10.0, (double)(precision - l));
      x *= scale;
      x = round(x);
      x /= scale;
    }
  if (neg)
    {
      return -x;
    }
  else
    {
      return x;
    }
}

/**
   @brief calculates a 2-order polynome (y = q + s*(x - p)**2) through three points, gives directly the position of the maximum/minimum
   @param x  the three x values
   @param y  the three y values
   @param p  the x-position of the maximum/minimum
   @param q  the y-position of the maximum/minimum
   @param s  form factor, if positive: minimum, if negative: maxiumum
   @return error code (if the three points are on a straigt line, ...)

   This function calculates a simple 2-order polynome going through three points. The returned parameters give directly the position and type of the extrema.
*/
cpl_boolean mat_calc_poly2_peak(double *x, double *y, double *p, double *q, double *s)
{
  /*
    y = q + s*(x - p)**2
    ->
    y1 = q + s*(x1 - p)**2
    y2 = q + s*(x2 - p)**2
    y3 = q + s*(x3 - p)**2
    =>
    q = y1 - s*(x1 - p)**2

    y2 - y1
    s = -------------------------
    (x2 - p)**2 - (x1 - p)**2

    1   x1**2*(y2 - y3) + x2**2*(y3 - y1) + x3**2*(y1 - y2)
    p = - * ---------------------------------------------------
    2      x1*(y2 - y3) +    x2*(y3 - y1) +    x3*(y1 - y2)
  */
  double a, b;

  a = x[0]*x[0]*(y[1] - y[2]) + x[1]*x[1]*(y[2] - y[0]) + x[2]*x[2]*(y[0] - y[1]);
  b = x[0]*(y[1] - y[2]) + x[1]*(y[2] - y[0]) + x[2]*(y[0] - y[1]);
  if (fabs(b) < 1e-6)
    {
      *p = 0.0;
      *q = 0.0;
      *s = 0.0;
      return CPL_FALSE;
    }
  *p = 0.5*(a/b);
  a = y[1] - y[0];
  b = (x[1] - (*p))*(x[1] - (*p)) - (x[0] - (*p))*(x[0] - (*p));
  if (fabs(b) < 1e-6)
    {
      *p = 0.0;
      *q = 0.0;
      *s = 0.0;
      return CPL_FALSE;
    }
  *s = a/b;
  *q = y[0] - (*s)*(x[0] - (*p))*(x[0] - (*p));
  return CPL_TRUE;
  
}

/**
   @brief 1D polynomial Fit
   @param vecy vector to fit.
   @param vecw weigths of each data
   @param deg polynomial degree
   @param fit fitted polynome
   @param even if even!=0 polynome=sigma_k x**2k otherwise sigma_k x**k
   @returns 0
*/
int mat_polyfit_1d(cpl_vector *vecy,cpl_vector *vecw,int deg,cpl_vector *fit, 
                   int even) {

  int i=0;
  int k=0;
  double xi=0.;
  double chisq=0.;
  double val=0.;
  gsl_matrix *X, *cov;
  gsl_vector *vy, *w, *c;
  cpl_size size;
  gsl_multifit_linear_workspace *workbf=NULL;
  
  if (vecy == NULL || vecw == NULL || fit == NULL || deg<0) {
    return -1;
  }
  if (cpl_vector_get_size(vecy) != cpl_vector_get_size(vecw)) {
    return -1;
  }
  if (cpl_vector_get_size(vecy) != cpl_vector_get_size(fit)) {
    return -1;
  }
  if (cpl_vector_get_size(fit) != cpl_vector_get_size(vecw)) {
    return -1;
  }
  
  size=cpl_vector_get_size(vecy);
  X = gsl_matrix_alloc(size,deg+1);
  vy = gsl_vector_alloc(size);
  w = gsl_vector_alloc(size);
  
  c = gsl_vector_alloc(deg+1);
  cov = gsl_matrix_alloc(deg+1,deg+1);
  
  for(i=0;i<size;i++) {
    xi=(i-size/2.)/size;
    for(k=0;k<=deg;k++){
      if (even) {
	gsl_matrix_set(X,i,k,pow(xi,2*k));
      } else {
	gsl_matrix_set(X,i,k,pow(xi,k));
      }
    }
    gsl_vector_set(vy,i,cpl_vector_get(vecy,i));
    gsl_vector_set(w,i,cpl_vector_get(vecw,i));
  }
  /* gsl_multifit_linear_workspace * workbf =  */
  /*  gsl_multifit_linear_alloc(size,deg+1); */
  workbf=gsl_multifit_linear_alloc(size,deg+1);
  gsl_multifit_wlinear(X,w,vy,c,cov,&chisq,workbf);
  gsl_multifit_linear_free(workbf);
  
  for(i=0;i<size;i++) {
    xi=(i-size/2.)/size;
    val=0.0;
    for(k=0;k<=deg;k++){
      if (even) {
	val+=gsl_vector_get(c,k)*pow(xi,2*k);
      } else {
	val+=gsl_vector_get(c,k)*pow(xi,k);
      }
    }
    cpl_vector_set(fit,i,val);
  }

  gsl_vector_free(c);
  gsl_vector_free(vy);
  gsl_vector_free(w);
  gsl_matrix_free(X);
  gsl_matrix_free(cov);

  return 0;
}


/**
   @brief Return the number of telecopes used in a mat_oivis2 structure
   @param oivis2 mat_oivis2 strcuture
   @returns an integer corresponding to the number of telescopes
*/
int mat_get_nbtel_from_oivis2(mat_oivis2 *oivis2) {
    
  int i=0;
  int j=0;
  int k=0;
  int flag=0;
  int nbtel=0;
  int *tel=NULL;
  
  if (oivis2 == NULL) {
    return -1;
  }
  if (oivis2->nbvis2 < 1) {
    return -1;
  }
  
  tel=cpl_calloc(2*oivis2->nbvis2,sizeof(int));
  
  for(i=0;i<oivis2->nbvis2;i++) {
    for(j=0;j<2;j++) {
      flag=0;
      for(k=0;k<nbtel;k++) {
	if (oivis2->list_vis2[i]->stationindex[j] == tel[k]) {
	  flag=1;
	}
      }
      if (flag==0) {
	tel[nbtel]=oivis2->list_vis2[i]->stationindex[j];
	nbtel++;
      }
    }
  }
  cpl_free(tel);
  return nbtel;
}

/**
   @brief Return the number of telecopes used in a mat_oivis structure
   @param oivis mat_oivis strcuture
   @returns an integer corresponding to the number of telescopes
*/
int mat_get_nbtel_from_oivis(mat_oivis *oivis) {
    
  int i=0;
  int j=0;
  int k=0;
  int flag=0;
  int nbtel=0;
  int *tel=NULL;

  if (oivis == NULL) {
    return -1;
  }
  if (oivis->nbvis < 1) {
    return -1;
  }
  tel=cpl_calloc(2*oivis->nbvis,sizeof(int));

  for(i=0;i<oivis->nbvis;i++) {
    for(j=0;j<2;j++) {
      flag=0;
      for(k=0;k<nbtel;k++) {
	if (oivis->list_vis[i]->stationindex[j] == tel[k]) {
	  flag=1;
	}
      }
      if (flag==0) {
	tel[nbtel]=oivis->list_vis[i]->stationindex[j];
	nbtel++;
      }
    }
  }
  cpl_free(tel);
  return nbtel;
}

/**
   @brief Return the number of telecopes used in a mat_oit3 structure
   @param oit3 mat_oit3 strcuture
   @returns an integer corresponding to the number of telescopes
*/
int mat_get_nbtel_from_oit3(mat_oit3 *oit3) {
    
  int i=0;
  int j=0;
  int k=0;
  int flag=0;
  int nbtel=0;
  int *tel=NULL;

  if (oit3 == NULL) {
    return -1;
  }
  if (oit3->nbt3 < 1) {
    return -1;
  }
  tel=cpl_calloc(3*oit3->nbt3,sizeof(int));

  for(i=0;i<oit3->nbt3;i++) {
    for(j=0;j<3;j++) {
      flag=0;
      for(k=0;k<nbtel;k++) {
	if (oit3->list_t3[i]->stationindex[j] == tel[k]) {
	  flag=1;
	}
      }
      if (flag==0) {
	tel[nbtel]=oit3->list_t3[i]->stationindex[j];
	nbtel++;
      }
    }
  }
  cpl_free(tel);
  return nbtel;
}


/**
   @brief Compute a combination
   @param n total number of elements
   @param p number of elements to choose
   @returns an integer corresponding to the results of the combination
*/
int mat_combinaison(int n, int p) {
  int i=0;
  int deno=0;
  int num=0;

  if (p == 0 || p > n) {
    return -1;
  }
    
  num=1;
  for(i=0;i<p;i++) {
    num*=(n-i);
  }
  deno=1;
  for(i=1;i<=p;i++) {
    deno*=i;
  }
  return (int)(num/deno);
}

/**
   @brief Return the value of the "ESO TPL NEXP" keyword
   @param plist a cpl_propertylist
   @returns an integer corresponding to "ESO TPL NEXP"
*/
int mat_get_nexp(cpl_propertylist *plist) {
  int nexp=0;
  if (cpl_propertylist_has(plist,"ESO TPL NEXP")) {
    nexp=cpl_propertylist_get_int(plist,"ESO TPL NEXP");
  } else {
    nexp=-1;
  }
  return nexp;
}

/**
   @brief Return the value of the "ESO TPL NEXPNO" keyword
   @param plist a cpl_propertylist
   @returns an integer corresponding to "ESO TPL NEXPNO"
*/
int mat_get_expno(cpl_propertylist *plist) {
  int nexp=0;
  if (cpl_propertylist_has(plist,"ESO TPL EXPNO")) {
    nexp=cpl_propertylist_get_int(plist,"ESO TPL EXPNO");
  } else {
    nexp=-1;
  }
  return nexp;
}

/**
   @brief Divide a frameset into several framesets sorted by TPL START specific keyword
   @param *frameset          Initial set of frames
   @param **listFrameSet     List of produced framesets
   @returns nbFrameSet        Number of produced framesets
*/
int mat_frameset_sort_by_tpl_start(cpl_frameset *frameset, cpl_frameset *listFrameSet[]) {
  cpl_frame *cur_frame=NULL;
  cpl_frame *sel_frame=NULL;
  cpl_frame *cur_frameSub=NULL;
  cpl_frameset_iterator *it=NULL;
  cpl_frameset_iterator *itsub=NULL;
  int nbFrameSet=0;
  int flagNotInserted=1;
  int i=0;
  char *keyValue=NULL;
  char *keyValueRef=NULL;
  cpl_errorstate prestate = cpl_errorstate_get();
  
  it= cpl_frameset_iterator_new(frameset);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      keyValue=mat_frame_get_tpl_start(cur_frame);
      flagNotInserted=1;
      for(i=0;i<nbFrameSet;i++) {
	itsub= cpl_frameset_iterator_new(listFrameSet[i]);
	cur_frameSub = cpl_frameset_iterator_get(itsub);
	keyValueRef=mat_frame_get_tpl_start(cur_frameSub);
	cpl_frameset_iterator_delete(itsub);
	if (strcmp(keyValue,keyValueRef) == 0) {
	  sel_frame=cpl_frame_duplicate(cur_frame);
	  cpl_frameset_insert(listFrameSet[i],sel_frame);
	  flagNotInserted=0;
	}   
	cpl_free(keyValueRef);
      }
      if (flagNotInserted) {
	listFrameSet[i]=cpl_frameset_new();
	sel_frame=cpl_frame_duplicate(cur_frame);
	cpl_frameset_insert(listFrameSet[i],sel_frame);
	nbFrameSet++;
      }
      cpl_free(keyValue);
    } else {
      cpl_frameset_erase_frame(frameset,cur_frame);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }

  /* for(i=0;i<nbFrameSet;i++) { */
  /*   printf("NUMBER %d\n",i); */
  /*   cpl_frameset_dump(listFrameSet[i],NULL); */
  /* } */
  
  cpl_frameset_iterator_delete(it);
  return nbFrameSet;
}

/**
   @brief Return the value of the "ESO TPL START" keyword
   @param frame a cpl_frame
   @returns a string corresponding to "ESO TPL START"
*/
char *mat_frame_get_tpl_start(cpl_frame *frame) {
  char *pszFileName=NULL;
  cpl_propertylist *plist=NULL;
  char *keyValue=NULL;
  cpl_type keyType=0;

  pszFileName = (char*)cpl_frame_get_filename(frame);
  plist=cpl_propertylist_load(pszFileName,0);
  keyType=cpl_propertylist_get_type(plist,"ESO TPL START");
  if (keyType == CPL_TYPE_INT) {
    keyValue=cpl_sprintf("%d",cpl_propertylist_get_int(plist,"ESO TPL START"));
  } else {
    keyValue=(char *)cpl_propertylist_get_string(plist,"ESO TPL START");
  }
  cpl_propertylist_delete(plist);
  return keyValue;
}


/*----------------------------------------------------------------------------*/
/**
   @brief Check if the frame is a photometry recording (HighSens mode)
   @param frame    current frame                  
   @return TRUE or FALSE
*/
/*----------------------------------------------------------------------------*/
cpl_boolean mat_check_photometry(cpl_frame *frame) {
  char *pszFileName=NULL;
  char *detName=NULL;
  cpl_propertylist *plist=NULL;
  int ixShut=0;
  /* char *p_shutter = NULL; */
  pszFileName = (char*)cpl_frame_get_filename(frame);
  plist=cpl_propertylist_load(pszFileName,0);
  detName=(char *)cpl_propertylist_get_string(plist,"ESO DET CHIP NAME");
  if (detName == NULL) {
    cpl_msg_error(cpl_func,"no ESO DET CHIP NAME keyword in frame");
    return -1;
  }

  if(strcmp(detName, "AQUARIUS") == 0) {
    if (cpl_propertylist_get_bool(plist,"ESO INS BSN1 ST") == 1)
      ixShut++;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSN2 ST") == 1)
      ixShut++;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSN3 ST") == 1)
      ixShut++;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSN4 ST") == 1)
      ixShut++;

    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSN1 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSN1 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut++; */
    /* } */
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSN2 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSN2 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut++; */
    /* } */
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSN3 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSN3 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut++; */
    /* } */
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSN4 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSN4 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut++; */
    /* } */
  } else if(strcmp(detName, "HAWAII-2RG") == 0) {

    if (cpl_propertylist_get_bool(plist,"ESO INS BSL1 ST") == 1)
      ixShut++;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSL2 ST") == 1)
      ixShut++;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSL3 ST") == 1)
      ixShut++;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSL4 ST") == 1)
      ixShut++;

    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSL1 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSL1 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut++; */
    /* } */
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSL2 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSL2 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut++; */
    /* } */
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSL3 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSL3 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut++; */
    /* } */
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSL4 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSL4 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut++; */
    /* } */
  }
  if (plist != NULL) cpl_propertylist_delete(plist);
  if (ixShut == 1) {
    return CPL_TRUE;
  } else {
    return CPL_FALSE;
  }
}


/*----------------------------------------------------------------------------*/
/**
   @brief Retrun the index of the open shutter for a given detector
   @param plist a cpl_propertylist
   @param detNameFirst Name fo teh detector
   @return an integer corresponding to the shutter index
*/
/*----------------------------------------------------------------------------*/
int mat_find_open_shutter(cpl_propertylist *plist, char *detNameFirst) {
  int ixShut=0;
  /* char *p_shutter = NULL; */
  if(strcmp(detNameFirst, "AQUARIUS") == 0) {
    if (cpl_propertylist_get_bool(plist,"ESO INS BSN1 ST") == 1)
      ixShut=3;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSN2 ST") == 1)
      ixShut=2;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSN3 ST") == 1)
      ixShut=0;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSN4 ST") == 1)
      ixShut=1;
    /* p_shutter=(char *) cpl_propertylist_get_string(plist,"ESO INS BSN1 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSN1 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut=0; */
    /* } */
    /* p_shutter =(char *)cpl_propertylist_get_string(plist,"ESO INS BSN2 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSN2 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut=1; */
    /* } */
    /* p_shutter = (char *)cpl_propertylist_get_string(plist,"ESO INS BSN3 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSN3 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut=2; */
    /* } */
    /* p_shutter = (char *)cpl_propertylist_get_string(plist,"ESO INS BSN4 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSN4 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut=3; */
    /* } */
  } else if(strcmp(detNameFirst, "HAWAII-2RG") == 0) {
    if (cpl_propertylist_get_bool(plist,"ESO INS BSL1 ST") == 1)
      ixShut=0;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSL2 ST") == 1)
      ixShut=1;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSL3 ST") == 1)
      ixShut=3;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSL4 ST") == 1)
      ixShut=2;



    /* p_shutter = (char *)cpl_propertylist_get_string(plist,"ESO INS BSL1 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSL1 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut=0; */
    /* } */
    /* p_shutter = (char *)cpl_propertylist_get_string(plist,"ESO INS BSL2 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSL2 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut=1; */
    /* } */
    /* p_shutter = (char *)cpl_propertylist_get_string(plist,"ESO INS BSL3 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSL3 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut=2; */
    /* } */
    /* p_shutter = (char *)cpl_propertylist_get_string(plist,"ESO INS BSL4 ST"); */
    /* if (p_shutter == NULL) { */
    /*     cpl_msg_error(cpl_func,"no ESO INS BSL4 ST in FITS file header"); */
    /*     return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*     ixShut=3; */
    /* } */




  }
  return ixShut;
}


/*----------------------------------------------------------------------------*/
/**
   @brief Initialize a mat_photbeams structure
   @param gendata raw data
   @param photBeams structure to initialize
   @param highSens 1 if HighSens mode, 0 otherwise
   @param p_nbPhot Number of photometric regions
   @param nbTarget numbre of frames (used in SiPhot mode only)
   @return 1 if no error
*/
/*----------------------------------------------------------------------------*/
int mat_init_photbeams(mat_gendata *gendata, mat_photbeams *photBeams, int highSens, int *p_nbPhot, int nbTarget) {
  int i=0;
  int j=0;
  int k=0;
  int nbPhot=0;
  int indreg=0;

  if (highSens == 1) {

    // In the case of HighSens, there is only one region per
    // spectral band. 
    // As we integrate the photometry of each telescope over
    // the whole exposure, 
    // photBeams->nbphotxframe is always equal to 4*number 
    // of spectral bands.
    nbPhot=gendata->imgdet->nbregion;
    photBeams->nbphotxframe=4*nbPhot;

    photBeams->keywords=cpl_propertylist_duplicate(gendata->keywords);


    photBeams->imgdet=mat_imagingdetector_duplicate(gendata->imgdet);
    // remove all regions
    for (i=0;i<photBeams->imgdet->nbregion;i++) {
      cpl_free(photBeams->imgdet->list_region[i]->numbeam);
      cpl_free(photBeams->imgdet->list_region[i]->regionname);
      for(j=0;j<2;j++) {
	cpl_free(photBeams->imgdet->list_region[i]->ctype[j]);
      }
      cpl_free(photBeams->imgdet->list_region[i]);
    }
    cpl_free(photBeams->imgdet->list_region);
    // Create 4 identical regions for each spectral band
    photBeams->imgdet->nbregion*=4;
    photBeams->imgdet->list_region=cpl_malloc(photBeams->imgdet->nbregion*sizeof(mat_region *));
    if (photBeams->imgdet->list_region == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for photBeams->imgdet->list_region");
      return -1;
    } 

    for (i=0;i<photBeams->imgdet->nbregion;i++) {
      photBeams->imgdet->list_region[i]=cpl_malloc(sizeof(mat_region));
      if (photBeams->imgdet->list_region[i] == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for photBeams->imgdet->list_region[i]");
	return -1;
      } 
      photBeams->imgdet->list_region[i]->numregion=i+1;
      photBeams->imgdet->list_region[i]->regionname=cpl_malloc(10*sizeof(char));
      strncpy(photBeams->imgdet->list_region[i]->regionname,(char *)gendata->imgdet->list_region[i/4]->regionname,10);
      photBeams->imgdet->list_region[i]->numdetector=gendata->imgdet->list_region[i/4]->numdetector;
      photBeams->imgdet->list_region[i]->numbeam=cpl_calloc(gendata->imgdet->nbtel,sizeof(int));
      for(j=0;j<gendata->imgdet->nbtel;j++) {
	photBeams->imgdet->list_region[i]->numbeam[j]=gendata->imgdet->list_region[i/4]->numbeam[j];
      }
      photBeams->imgdet->list_region[i]->correlation=gendata->imgdet->list_region[i/4]->correlation;
      for(j=0;j<2;j++) {
	photBeams->imgdet->list_region[i]->corner[j]=gendata->imgdet->list_region[i/4]->corner[j];
      }
      photBeams->imgdet->list_region[i]->gain=gendata->imgdet->list_region[i/4]->gain;
      for(j=0;j<2;j++) {
	photBeams->imgdet->list_region[i]->crval[j]=gendata->imgdet->list_region[i/4]->crval[j];
      }
      for(j=0;j<2;j++) {
	photBeams->imgdet->list_region[i]->crpix[j]=gendata->imgdet->list_region[i/4]->crpix[j];
      }
      for(j=0;j<2;j++) {
	photBeams->imgdet->list_region[i]->ctype[j]=cpl_malloc(10*sizeof(char));
	strncpy(photBeams->imgdet->list_region[i]->ctype[j],(char *)gendata->imgdet->list_region[i/4]->ctype[j],10);
      }
      for(j=0;j<2;j++) {
	photBeams->imgdet->list_region[i]->naxis[j]=gendata->imgdet->list_region[i/4]->naxis[j];
      }
    }
     
    // *2 because we compute the stdev of the images
    photBeams->list_phot=(mat_imgreg **)cpl_calloc(photBeams->nbphotxframe*2,sizeof(mat_imgreg *));
    if (photBeams->list_phot==NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for photBeams->list_phot");
      return -1;
    }

    *p_nbPhot=nbPhot;
     
  } else {
    nbPhot=0;
    // we count the number of photometric regions 
    // (normaly equal to 4)
    for(k=0;k<gendata->imgdet->nbregion;k++) {
      if (gendata->imgdet->list_region[k]->correlation == 1 ) {
	nbPhot++;
      }
    }
    photBeams->list_phot=NULL;
    photBeams->nbphotxframe=nbTarget*nbPhot;
    photBeams->keywords=cpl_propertylist_duplicate(gendata->keywords);
    photBeams->imgdet=mat_imagingdetector_duplicate(gendata->imgdet);

    // remove all regions
    for (i=0;i<photBeams->imgdet->nbregion;i++) {
      cpl_free(photBeams->imgdet->list_region[i]->numbeam);
      cpl_free(photBeams->imgdet->list_region[i]->regionname);
      for(j=0;j<2;j++) {
    	cpl_free(photBeams->imgdet->list_region[i]->ctype[j]);
      }
      cpl_free(photBeams->imgdet->list_region[i]);
    }
    cpl_free(photBeams->imgdet->list_region);
    // Keep interferometric regions only
    photBeams->imgdet->nbregion=gendata->imgdet->nbregion-gendata->imgdet->nbregion/5;
    photBeams->imgdet->list_region=cpl_malloc(photBeams->imgdet->nbregion*sizeof(mat_region *));
    if (photBeams->imgdet->list_region == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for photBeams->imgdet->list_region");
      return -1;
    }
    indreg=0;
    for (i=0;i<gendata->imgdet->nbregion;i++) {
      if (gendata->imgdet->list_region[i]->correlation == 1) {
    	photBeams->imgdet->list_region[indreg]=cpl_malloc(sizeof(mat_region));
    	if (photBeams->imgdet->list_region[indreg]==NULL) {
    	  cpl_msg_error(cpl_func,"could not allocate memory for photBeams->imgdet->list_region[indreg]");
    	  return -1;
    	}
    	photBeams->imgdet->list_region[indreg]->numregion=indreg+1;
    	photBeams->imgdet->list_region[indreg]->regionname=cpl_malloc(10*sizeof(char));
    	strncpy(photBeams->imgdet->list_region[indreg]->regionname,(char *)gendata->imgdet->list_region[i]->regionname,10);
    	photBeams->imgdet->list_region[indreg]->numdetector=gendata->imgdet->list_region[i]->numdetector;
    	photBeams->imgdet->list_region[indreg]->numbeam=cpl_calloc(gendata->imgdet->nbtel,sizeof(int));
    	for(j=0;j<gendata->imgdet->nbtel;j++) {
    	  photBeams->imgdet->list_region[indreg]->numbeam[j]=gendata->imgdet->list_region[i]->numbeam[j];
    	}
    	photBeams->imgdet->list_region[indreg]->correlation=gendata->imgdet->list_region[i]->correlation;
    	for(j=0;j<2;j++) {
    	  photBeams->imgdet->list_region[indreg]->corner[j]=gendata->imgdet->list_region[i]->corner[j];
    	}
    	photBeams->imgdet->list_region[indreg]->gain=gendata->imgdet->list_region[i]->gain;
    	for(j=0;j<2;j++) {
    	  photBeams->imgdet->list_region[indreg]->crval[j]=gendata->imgdet->list_region[i]->crval[j];
    	}
    	for(j=0;j<2;j++) {
    	  photBeams->imgdet->list_region[indreg]->crpix[j]=gendata->imgdet->list_region[i]->crpix[j];
    	}
    	for(j=0;j<2;j++) {
    	  photBeams->imgdet->list_region[indreg]->ctype[j]=cpl_malloc(10*sizeof(char));
    	  strncpy(photBeams->imgdet->list_region[indreg]->ctype[j],(char *)gendata->imgdet->list_region[i]->ctype[j],10);
    	}
    	for(j=0;j<2;j++) {
    	  photBeams->imgdet->list_region[indreg]->naxis[j]=gendata->imgdet->list_region[5*(i/5)+2]->naxis[j];
    	}
    	indreg++;
      }
    }
    photBeams->list_phot=cpl_calloc(photBeams->nbphotxframe,sizeof(mat_imgreg *));
    if (photBeams->list_phot == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for photBeams->list_phot");
      return -1;
    }

    
    *p_nbPhot=nbPhot;
  }
  return 1;
}



/*----------------------------------------------------------------------------*/
/**
   @brief Compute the interpolation with ratio, zoom, offset
   @param p_row    known data points                  
   @param p_param  parameters to apply to row in order to calculate new data
   @param p_res    the result
   @return cpl_error_code
*/
/*----------------------------------------------------------------------------*/
cpl_error_code mat_apply_kappamatrix(cpl_vector *p_row, cpl_vector *p_param, 
                                     cpl_vector *p_res){
  int xmid=0;
  int Xmid=0;
  int i=0;
  int n=0;
  int npt=0;
  cpl_vector *x=NULL;
  cpl_vector *X=NULL;
  int jbeg=0;
  int j=0;
  float deno=0.;
  float num=0;
  float factor=0.;
  float offset=0.;
  float val=0.;
 
  // Check input parameters
  /* mat_assert_value((p_row != NULL),CPL_ERROR_NULL_INPUT,  */
  /*      CPL_ERROR_NULL_INPUT, "no p_row argument given"); */
  /* mat_assert_value((p_param != NULL),CPL_ERROR_NULL_INPUT,  */
  /*      CPL_ERROR_NULL_INPUT, "no p_param argument given"); */
  /* mat_assert_value((p_res != NULL),CPL_ERROR_NULL_INPUT,  */
  /*      CPL_ERROR_NULL_INPUT, "no p_res argument given"); */
  n = cpl_vector_get_size(p_row);
  x = cpl_vector_new(n) ;
  
  npt = cpl_vector_get_size(p_res);
  X = cpl_vector_new(npt);




  /* Remove Offsets for p_row */
  offset=0.;
  for(i=0;i<20;i++) {
    offset+=cpl_vector_get(p_row,i)+cpl_vector_get(p_row,n-i);
  }
  offset/=40.;
  for(i=0;i<n;i++) {
    val=cpl_vector_get(p_row,i);
    cpl_vector_set(p_row,i,val-offset);
  }



  
  xmid = ceil(n/2);
  Xmid = ceil(npt/2);
  for(i=0;i<n;i++){
    cpl_vector_set(x,i,(i-xmid)/(double)n);
  }
  for(i=0;i<npt;i++) {  
    // Dividing cpl_vector_get(p_param,2) by n, new way of estimation of the offset
    // Xmid-i instead of i-Xmid
    cpl_vector_set(X,i,(cpl_vector_get(p_param,2)/(double)n)+
		   ((Xmid-i)/((double)n * cpl_vector_get(p_param,1))));		   
    if (cpl_vector_get(X,i) > cpl_vector_get(x,n-1) ) {
      cpl_vector_set(X,i,cpl_vector_get(x,n-1));
    }
  }

  jbeg=0;
  for(i=0;i<npt;i++) {  
    /* for(j=0;j<n;j++) { */
    for(j=jbeg;j<n;j++) {
      if ( cpl_vector_get(X,i) < cpl_vector_get(x,j) ) {
	jbeg=j-1;
	break;
      }
    }
    if (jbeg < 0) {
      cpl_vector_set(p_res,i,cpl_vector_get(p_row,0)*
		     cpl_vector_get(p_param,0)*((double)n)/((double)npt));

    } else if (jbeg >= npt-1) {
      cpl_vector_set(p_res,i,cpl_vector_get(p_row,npt-1)*
		     cpl_vector_get(p_param,0)*((double)n)/((double)npt));
    } else {
      num=(cpl_vector_get(x,jbeg+1)-cpl_vector_get(X,i))*
	cpl_vector_get(p_row,jbeg);
      num+=(cpl_vector_get(X,i)-cpl_vector_get(x,jbeg))*
	cpl_vector_get(p_row,jbeg+1);
      deno=(cpl_vector_get(x,jbeg+1)-cpl_vector_get(x,jbeg));
      cpl_vector_set(p_res,i,(num/deno)*
		     cpl_vector_get(p_param,0)*((double)n)/((double)npt));
    }
  }

  // Remove Offsets for p_row
  /* offset=0.; */
  /* for(i=0;i<20;i++) { */
  /*   offset+=cpl_vector_get(p_row,i)+cpl_vector_get(p_row,n-i); */
  /* } */
  /* offset/=40.; */
  /* for(i=0;i<n;i++) { */
  /*   val=cpl_vector_get(p_row,i); */
  /*   cpl_vector_set(p_row,i,val-offset); */
  /* } */
  /* // Remove Offsets for p_res */
  /* offset=0.; */
  /* for(i=0;i<100;i++) { */
  /*   offset+=cpl_vector_get(p_res,i)+cpl_vector_get(p_res,npt-i); */
  /* } */
  /* offset/=200.; */
  /* for(i=0;i<npt;i++) { */
  /*   val=cpl_vector_get(p_res,i); */
  /*   cpl_vector_set(p_res,i,val-offset); */
  /* } */
  
  factor=cpl_vector_get(p_param,0)*fabs(cpl_vector_get_sum(p_row)/cpl_vector_get_sum(p_res));
  cpl_vector_multiply_scalar(p_res,factor);

  if (x != NULL) cpl_vector_delete(x);
  if (X != NULL) cpl_vector_delete(X);
  return cpl_error_get_code();
}



/**
   @brief Compute an average sky
   @param gendata contains the mat_gendata structure
   @param avgSky array containing the averaged sky
   @return 0 if no error
*/

int mat_compute_avg_sky(mat_gendata *gendata,cpl_image **avgSky) {
  int cpt=0;
  int k=0;
  int ix=0;
  int j=0;

  cpt=0;
  for(k=0;k<gendata->imgdet->nbregion;k++) {
    if (gendata->imgdet->list_region[k]->correlation == 2) {
      avgSky[cpt]=cpl_image_new(gendata->imgdet->list_region[k]->naxis[0],gendata->imgdet->list_region[k]->naxis[1],CPL_TYPE_FLOAT);
      cpt++;
    }
  }
  cpt=0;
  ix=0;
  for(k=0;k<gendata->imgdet->nbregion;k++) {
    if (gendata->imgdet->list_region[k]->correlation == 2) {
      for(j=0;j<gendata->imgdata->nbframe;j++) {
	if ( !strcmp(gendata->imgdata->list_frame[j]->tartype,"S") ) {	    
	  cpl_image_add(avgSky[cpt],gendata->imgdata->list_frame[j]->list_subwin[k]->imgreg[0]);
	  ix++;
	}
      }
      if (ix == 0) {
	cpl_msg_warning(cpl_func,"No Sky found");
	return -1;
      } else {
	cpl_image_divide_scalar(avgSky[cpt],ix);
      }
      cpt++;
    }
  }
  cpl_msg_info(cpl_func,"Average sky done");
  return 0;
}
/**
   @brief Compute an average sky
   @param gendata contains the mat_gendata structure
   @param avgSky array containing the averaged sky
   @return 0 if no error
*/

int mat_compute_avg_img(mat_gendata *gendata,cpl_image **avgImg) {
  int cpt=0;
  int k=0;
  int ix=0;
  int j=0;

  cpt=0;
  for(k=0;k<gendata->imgdet->nbregion;k++) {
    if (gendata->imgdet->list_region[k]->correlation == 2) {
      avgImg[cpt]=cpl_image_new(gendata->imgdet->list_region[k]->naxis[0],gendata->imgdet->list_region[k]->naxis[1],CPL_TYPE_FLOAT);
      cpt++;
    }
  }
  cpt=0;
  ix=0;
  for(k=0;k<gendata->imgdet->nbregion;k++) {
    if (gendata->imgdet->list_region[k]->correlation == 2) {
      for(j=0;j<gendata->imgdata->nbframe;j++) {
	if ( !strcmp(gendata->imgdata->list_frame[j]->tartype,"T") ) {	    
	  cpl_image_add(avgImg[cpt],gendata->imgdata->list_frame[j]->list_subwin[k]->imgreg[0]);
	  ix++;
	}
      }
      if (ix == 0) {
	cpl_msg_warning(cpl_func,"No Image found");
	return -1;
      } else {
	cpl_image_divide_scalar(avgImg[cpt],ix);
      }
      cpt++;
    }
  }
  cpl_msg_info(cpl_func,"Average Image done");
  return 0;
}


/*----------------------------------------------------------------------------*/
/**
   @brief Check if the frame is not a photometry recording (HighSens mode)
   @param frame    current frame                  
   @return TRUE or FALSE
*/
/*----------------------------------------------------------------------------*/
cpl_boolean mat_check_interf(cpl_frame *frame) {
  char *pszFileName=NULL;
  char *detName=NULL;
  cpl_propertylist *plist=NULL;
  int ixShut=0;
  /* char *p_shutter = NULL; */

  pszFileName = (char*)cpl_frame_get_filename(frame);
  plist=cpl_propertylist_load(pszFileName,0);
  detName=(char *)cpl_propertylist_get_string(plist,"ESO DET CHIP NAME");
  if (detName == NULL) {
    cpl_msg_error(cpl_func,"no ESO DET CHIP NAME keyword in frame");
    return -1;
  }

  if(strcmp(detName, "AQUARIUS") == 0) {
    if (cpl_propertylist_get_bool(plist,"ESO INS BSN1 ST") == 1)
      ixShut++;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSN2 ST") == 1)
      ixShut++;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSN3 ST") == 1)
      ixShut++;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSN4 ST") == 1)
      ixShut++;
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSN1 ST"); */
    /* if (p_shutter == NULL) { */
    /*   cpl_msg_error(cpl_func,"no ESO INS BSN1 ST in FITS file header"); */
    /*   return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*   ixShut++; */
    /* } */
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSN2 ST"); */
    /* if (p_shutter == NULL) { */
    /*   cpl_msg_error(cpl_func,"no ESO INS BSN2 ST in FITS file header"); */
    /*   return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*   ixShut++; */
    /* } */
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSN3 ST"); */
    /* if (p_shutter == NULL) { */
    /*   cpl_msg_error(cpl_func,"no ESO INS BSN3 ST in FITS file header"); */
    /*   return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*   ixShut++; */
    /* } */
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSN4 ST"); */
    /* if (p_shutter == NULL) { */
    /*   cpl_msg_error(cpl_func,"no ESO INS BSN4 ST in FITS file header"); */
    /*   return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*   ixShut++; */
    /* } */
  } else if(strcmp(detName, "HAWAII2RG") == 0) {
    if (cpl_propertylist_get_bool(plist,"ESO INS BSL1 ST") == 1)
      ixShut++;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSL2 ST") == 1)
      ixShut++;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSL3 ST") == 1)
      ixShut++;
    if (cpl_propertylist_get_bool(plist,"ESO INS BSL4 ST") == 1)
      ixShut++;
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSL1 ST"); */
    /* if (p_shutter == NULL) { */
    /*   cpl_msg_error(cpl_func,"no ESO INS BSL1 ST in FITS file header"); */
    /*   return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*   ixShut++; */
    /* } */
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSL2 ST"); */
    /* if (p_shutter == NULL) { */
    /*   cpl_msg_error(cpl_func,"no ESO INS BSL2 ST in FITS file header"); */
    /*   return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*   ixShut++; */
    /* } */
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSL3 ST"); */
    /* if (p_shutter == NULL) { */
    /*   cpl_msg_error(cpl_func,"no ESO INS BSL3 ST in FITS file header"); */
    /*   return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*   ixShut++; */
    /* } */
    /* p_shutter=(char *)cpl_propertylist_get_string(plist,"ESO INS BSL4 ST"); */
    /* if (p_shutter == NULL) { */
    /*   cpl_msg_error(cpl_func,"no ESO INS BSL4 ST in FITS file header"); */
    /*   return -1; */
    /* } */
    /* if (strcmp(p_shutter,"T") == 0) { */
    /*   ixShut++; */
    /* } */
  }
  if (plist != NULL) cpl_propertylist_delete(plist);

  if (ixShut == 1) {
    return CPL_FALSE;
  } else {
    return CPL_TRUE;
  }
}
/*----------------------------------------------------------------------------*/
/**
   @brief Demodulate the fringe signal with lsq method for the last frame
   @param gendata contains the mat_gendata structure
   @param corrFlux contains a mat_corrflux structure
   @param modeFirst "s" or "h"
   @param dispCoef dispersion law coefficient
   @param filledInterfCycle index of the good image in one opd modulation cycle
   @param imgInterfCycle array containing the image ins one opd modulation cycle
   @param localOpd opd applied by the opd modulation system
   @param nbImgInterfCycle number of image in one opd modulation cycle
   @return 0 if no error
*/
int mat_lsq_opdmod_last_frame(mat_gendata *gendata,mat_corrflux *corrFlux, char modeFirst[], 
                              cpl_vector *dispCoef, int *filledInterfCycle, cpl_image **imgInterfCycle, 
                              cpl_vector *localOpd, int nbImgInterfCycle, int *jstep) {
  int k=0;
  int i=0;
  int l=0;
  int ix=0;
  int jrow=0;
  int dimSpatial=0;
  int dimSpectral=0;
  int flag_rejected=0;
  double lambda=0.;
  /* double localOpdFrame[4]={0.,0.,0.,0.}; */
  cpl_vector *outOpdMod=NULL;
  cpl_vector *ftReal=NULL;
  cpl_vector *ftImag=NULL;
  double complex *in;
  double complex *out;
  cpl_image *self=NULL;
  cpl_image *other=NULL;
  char *detName;
  char *specResData;
  char *fil;
  int resolution;
  int nbdetector;
  /* double posFringePeak; */
  /* double dOverLambda; */
  int ier=0;
  /* cpl_vector *vecWindow=NULL; */
  /* cpl_vector *vecFoo=NULL; */
  int flagBcdIN=0;
  char *bcd1Status;
  
  bcd1Status = (char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS BCD1 ID");
  if(!strcmp(bcd1Status, "IN")) {
    flagBcdIN=1;
  } else {
    flagBcdIN=0;
  }

  detName=(char *) cpl_propertylist_get_string(corrFlux->keywords,"ESO DET CHIP NAME");
  if (detName == NULL) {
    cpl_msg_error(cpl_func,
		  "no ESO DET CHIP NAME keyword in frame");
    return -1;
  }
  
  
  if(!strcmp(detName, "AQUARIUS")) {
    nbdetector=2;
    specResData=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS DIN ID");
    if (!strcmp(specResData,"LOW")) {
      resolution=1;
    }
    if (!strcmp(specResData,"HIGH")) {
      resolution=2;
    }
    
  }
  else {
    nbdetector=1;
    specResData=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS DIL ID");
    fil=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS fIL ID");
    if (!strcmp(specResData,"LOW")) {
      resolution=1;
    }
    if (!strcmp(specResData,"MED")) {
      resolution=2;
    }
    if (!strcmp(specResData,"HIGH")) {
      resolution=3;
    }
    if (!strcmp(specResData,"HIGH+") && !strcmp(fil,"L")) {
      resolution=4;
    }
    if (!strcmp(specResData,"HIGH+") && !strcmp(fil,"M")) {
      resolution=5;
    }
  }

  for(k=0;k<corrFlux->imgdet->nbregion;k++) {
    if (strstr(modeFirst,"s") != NULL) {
      ix=5*k+2;
    } else {
      ix=k;
    }
    dimSpatial = gendata->imgdet->list_region[ix]->naxis[0];
    dimSpectral = gendata->imgdet->list_region[ix]->naxis[1];
    corrFlux->list_corr[corrFlux->nbframe]=cpl_malloc(sizeof(mat_imgcorr));
    corrFlux->list_corr[corrFlux->nbframe]->numregion=k;
    corrFlux->list_corr[corrFlux->nbframe]->numdetector=gendata->imgdet->list_region[k]->numdetector;
    corrFlux->list_corr[corrFlux->nbframe]->time=gendata->imgdata->list_frame[*jstep]->time;
    corrFlux->list_corr[corrFlux->nbframe]->exptime=gendata->imgdata->list_frame[*jstep]->exptime;
    corrFlux->list_corr[corrFlux->nbframe]->nbimgreg=2;
    corrFlux->list_corr[corrFlux->nbframe]->imgreg=cpl_calloc(2,sizeof(cpl_image *));
    corrFlux->list_corr[corrFlux->nbframe]->imgreg[0]=cpl_image_new(dimSpatial,dimSpectral,CPL_TYPE_FLOAT);
    corrFlux->list_corr[corrFlux->nbframe]->imgreg[1]=cpl_image_new(dimSpatial,dimSpectral,CPL_TYPE_FLOAT);
    ftReal=cpl_vector_new(gendata->imgdata->maxstep*dimSpatial);
    ftImag=cpl_vector_new(gendata->imgdata->maxstep*dimSpatial);
    outOpdMod=cpl_vector_new(2*dimSpatial);
    in=(double complex *)malloc(sizeof(double complex)*dimSpatial);
    self=cpl_image_new(dimSpatial,1,CPL_TYPE_DOUBLE | CPL_TYPE_COMPLEX);
    // Compute the FFT of each spctral channel
    // of each step of teh OPD modulation
    for(i=0;i<dimSpectral;i++) {
      for(l=0;l<gendata->imgdata->maxstep;l++) {
	if (filledInterfCycle[l]) {
        
          // Before FFT, shift image by minus half spatial size
          for(jrow=0;jrow<dimSpatial;jrow++) {
            in[jrow]=cpl_image_get(imgInterfCycle[k*gendata->imgdata->maxstep+l],(jrow+dimSpatial/2)%dimSpatial+1,i+1,&flag_rejected);
          }

          // Perform FFT
	  other=cpl_image_wrap_double_complex(dimSpatial,1,in);
	  if (i == 0) {
	    cpl_fft_image(self,other,CPL_FFT_FORWARD | CPL_FFT_FIND_MEASURE);
	  } else {
	    cpl_fft_image(self,other,CPL_FFT_FORWARD);
	  }
	  out=cpl_image_get_data_double_complex(self);

          // After FFT, shift image by plus half spatial size
          for(jrow=0;jrow<dimSpatial;jrow++) {
            cpl_vector_set(ftReal,l*dimSpatial+(jrow+dimSpatial/2)%dimSpatial,creal(out[jrow]));
            cpl_vector_set(ftImag,l*dimSpatial+(jrow+dimSpatial/2)%dimSpatial,cimag(out[jrow]));
          }

	  cpl_image_unwrap(other);
	}
      }
      // LSQ method using OPD modulation
      lambda=mat_get_lambda(dispCoef,i+gendata->imgdet->list_region[ix]->corner[1]);
      /* lambda=dispCoef[0]+dispCoef[1]*(i+gendata->imgdet->list_region[ix]->corner[1]); */
      /* lambda+=dispCoef[2]*pow((double)(i+gendata->imgdet->list_region[ix]->corner[1]),2.0); */
      /* posFringePeak=(int)mat_get_pos_fringe_peak(lambda,nbdetector,resolution,5); */
      /* dOverLambda=(posFringePeak-corrFlux->imgdet->list_region[k]->naxis[0]/2)/(18); */
      ier=mat_apply_opdmod(gendata->imgdata->maxstep,ftReal,ftImag,filledInterfCycle,localOpd,lambda,outOpdMod,resolution,nbdetector,flagBcdIN);
      if (ier==0) {
	// Store results in 2 images 
	// (real and imaginary parts)
	for(jrow=0;jrow<dimSpatial;jrow++) {
	  cpl_image_set(corrFlux->list_corr[corrFlux->nbframe]->imgreg[0],jrow+1,i+1,cpl_vector_get(outOpdMod,jrow));
	  cpl_image_set(corrFlux->list_corr[corrFlux->nbframe]->imgreg[1],jrow+1,i+1,cpl_vector_get(outOpdMod,jrow+dimSpatial));
	}
      }
    }
    cpl_image_delete(self);
    if (in != NULL) free(in);
    if (outOpdMod != NULL) cpl_vector_delete(outOpdMod);
    if (ftReal != NULL) cpl_vector_delete(ftReal);
    if (ftImag != NULL) cpl_vector_delete(ftImag);
    if (ier==0) {
      corrFlux->nbframe++;
    } else {
      cpl_image_delete(corrFlux->list_corr[corrFlux->nbframe]->imgreg[0]);
      cpl_image_delete(corrFlux->list_corr[corrFlux->nbframe]->imgreg[1]);
      cpl_free(corrFlux->list_corr[corrFlux->nbframe]->imgreg);
      cpl_free(corrFlux->list_corr[corrFlux->nbframe]);
    }
	
  }	      
  // RAZ
  for(k=0;k<nbImgInterfCycle;k++) {
    if (filledInterfCycle[k/corrFlux->imgdet->nbregion] == 1) {
      cpl_image_delete(imgInterfCycle[k]);
    }
  }
  for(l=0;l<gendata->imgdata->maxstep;l++) {
    filledInterfCycle[l]=0;
  }

  return 0;
}


/*
   @brief Demodulate the fringe signal with lsq method
   @param gendata contains the mat_gendata structure
   @param corrFlux contains a mat_corrflux structure
   @param modeFirst "s" or "h"
   @param jframe current frame index
   @param dispCoef dispersion law coefficient
   @param localOpdBcd opd applied by the opd modulation system corrected with BCD
   @param filledInterfCycle index of the good image in one opd modulation cycle
   @param imgInterfCycle array containing the image ins one opd modulation cycle
   @param localOpd opd applied by the opd modulation system
   @param cp counter
   @param nbImgInterfCycle number of image in one opd modulation cycle
   @return 0 if no error
*/
int mat_lsq_opdmod(mat_gendata *gendata,mat_corrflux *corrFlux, char modeFirst[], 
                   int jframe, cpl_vector *dispCoef, cpl_matrix *localOpdBcd,
                   int *filledInterfCycle, cpl_image **imgInterfCycle, 
                   cpl_vector *localOpd, int *cp, int nbImgInterfCycle,int *jstep) {
  int k=0;
  int i=0;
  int l=0;
  int ix=0;
  int jrow=0;
  int dimSpatial=0;
  int dimSpectral=0;
  int flag_rejected=0;
  double lambda=0.;
  /* double localOpdFrame[4]={0.,0.,0.,0.}; */
  cpl_vector *outOpdMod=NULL;
  cpl_vector *ftReal=NULL;
  cpl_vector *ftImag=NULL;
  double complex *in;
  double complex *out;
  cpl_image *self=NULL;
  cpl_image *other=NULL;
  char *detName;
  char *specResData;
  char *fil;
  int resolution;
  int nbdetector;
  /* double posFringePeak; */
  /* double dOverLambda; */
  int ier=0;
  int flagBcdIN=0;
  char *bcd1Status;
  
  bcd1Status = (char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS BCD1 ID");
  if(!strcmp(bcd1Status, "IN")) {
    flagBcdIN=1;
  } else {
    flagBcdIN=0;
  }

  detName=(char *) cpl_propertylist_get_string(corrFlux->keywords,"ESO DET CHIP NAME");
  if (detName == NULL) {
    cpl_msg_error(cpl_func,
		  "no ESO DET CHIP NAME keyword in frame");
    return -1;
  }
  
  
  if(!strcmp(detName, "AQUARIUS")) {
    nbdetector=2;
    specResData=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS DIN ID");
    if (!strcmp(specResData,"LOW")) {
      resolution=1;
    }
    if (!strcmp(specResData,"HIGH")) {
      resolution=2;
    }
    
  } else {
    nbdetector=1;
    specResData=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS DIL ID");
    fil=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS FIL ID");
    if (!strcmp(specResData,"LOW")) {
      resolution=1;
    }
    if (!strcmp(specResData,"MED")) {
      resolution=2;
    }
    if (!strcmp(specResData,"HIGH")) {
      resolution=3;
    }
    if (!strcmp(specResData,"HIGH+") && !strcmp(fil,"L")) {
      resolution=4;
    }
    if (!strcmp(specResData,"HIGH+") && !strcmp(fil,"M")) {
      resolution=5;
    }
  }
  

  if (gendata->imgdata->list_frame[jframe]->stepphase > *cp) {
    // load the maxstep images
    (*cp)=gendata->imgdata->list_frame[jframe]->stepphase;
    /* if ((*cp)==(int)(gendata->imgdata->maxstep/2)) { */
    /*   *jstep=jframe; */
    /* } */
    filledInterfCycle[(*cp)-1]=1;
    for(k=0;k<corrFlux->imgdet->nbregion;k++) {
      if (strstr(modeFirst,"s") != NULL) {
	ix=5*k+2;
      } else {
	ix=k;
      }
      imgInterfCycle[k*gendata->imgdata->maxstep+(*cp)-1]=cpl_image_duplicate(gendata->imgdata->list_frame[jframe]->list_subwin[ix]->imgreg[0]);
    }
    for(k=0;k<4;k++) {
      cpl_vector_set(localOpd,k*gendata->imgdata->maxstep+(*cp)-1,cpl_matrix_get(localOpdBcd,jframe,k));
    }
  } else {
    // process the maxstep images
    //printf("Processing of an OPD cycle ...\n");
    for(k=0;k<corrFlux->imgdet->nbregion;k++) {
      if (strstr(modeFirst,"s") != NULL) {
	ix=5*k+2;
      } else {
	ix=k;
      }
      dimSpatial = gendata->imgdet->list_region[ix]->naxis[0];
      dimSpectral = gendata->imgdet->list_region[ix]->naxis[1];
      corrFlux->list_corr[corrFlux->nbframe]=cpl_malloc(sizeof(mat_imgcorr));
      corrFlux->list_corr[corrFlux->nbframe]->numregion=k+1;
      corrFlux->list_corr[corrFlux->nbframe]->numdetector=gendata->imgdet->list_region[k]->numdetector;
      corrFlux->list_corr[corrFlux->nbframe]->time=gendata->imgdata->list_frame[*jstep]->time;
      corrFlux->list_corr[corrFlux->nbframe]->exptime=gendata->imgdata->list_frame[*jstep]->exptime;
      corrFlux->list_corr[corrFlux->nbframe]->nbimgreg=2;
      corrFlux->list_corr[corrFlux->nbframe]->imgreg=cpl_calloc(2,sizeof(cpl_image *));
      corrFlux->list_corr[corrFlux->nbframe]->imgreg[0]=cpl_image_new(dimSpatial,dimSpectral,CPL_TYPE_FLOAT);
      corrFlux->list_corr[corrFlux->nbframe]->imgreg[1]=cpl_image_new(dimSpatial,dimSpectral,CPL_TYPE_FLOAT);
      ftReal=cpl_vector_new(gendata->imgdata->maxstep*dimSpatial);
      ftImag=cpl_vector_new(gendata->imgdata->maxstep*dimSpatial);
      outOpdMod=cpl_vector_new(2*dimSpatial);
      in=(double complex *)malloc(sizeof(double complex)*dimSpatial);
      self=cpl_image_new(dimSpatial,1,CPL_TYPE_DOUBLE | CPL_TYPE_COMPLEX);
      // Compute the FFT of each spectral channel
      // of each step of teh OPD modulation
      for(i=0;i<dimSpectral;i++) {
	for(l=0;l<gendata->imgdata->maxstep;l++) {
	  if (filledInterfCycle[l]) {

            // Before FFT, shift image by minus half spatial size
	    for(jrow=0;jrow<dimSpatial;jrow++) {
              in[jrow]=cpl_image_get(imgInterfCycle[k*gendata->imgdata->maxstep+l],(jrow+dimSpatial/2)%dimSpatial+1,i+1,&flag_rejected);
            }

            // Perform FFT
	    other=cpl_image_wrap_double_complex(dimSpatial,1,in);
	    if (i == 0) {
	      cpl_fft_image(self,other,CPL_FFT_FORWARD | CPL_FFT_FIND_MEASURE);
	    } else {
	      cpl_fft_image(self,other,CPL_FFT_FORWARD);
	    }
	    out=cpl_image_get_data_double_complex(self);

            // After FFT, shift image by plus half spatial size
            for(jrow=0;jrow<dimSpatial;jrow++) {
              cpl_vector_set(ftReal,l*dimSpatial+(jrow+dimSpatial/2)%dimSpatial,creal(out[jrow]));
              cpl_vector_set(ftImag,l*dimSpatial+(jrow+dimSpatial/2)%dimSpatial,cimag(out[jrow]));
            }

	    cpl_image_unwrap(other);

	  } else {
	    for(jrow=0;jrow<dimSpatial;jrow++) {
	      cpl_vector_set(ftReal,l*dimSpatial+jrow,0.);
	      cpl_vector_set(ftImag,l*dimSpatial+jrow,0.);
	    }
	  }
	}
	// LSQ method using OPD modulation
	lambda=mat_get_lambda(dispCoef,i+gendata->imgdet->list_region[ix]->corner[1]);
	/* lambda=dispCoef[0]+dispCoef[1]*(i+gendata->imgdet->list_region[ix]->corner[1]); */
	/* lambda+=dispCoef[2]*pow((double)(i+gendata->imgdet->list_region[ix]->corner[1]),2.0); */
	/* posFringePeak=mat_get_pos_fringe_peak(lambda,nbdetector,resolution,5); */
	/* dOverLambda=(posFringePeak-corrFlux->imgdet->list_region[k]->naxis[0]/2)/(18); */
	ier=mat_apply_opdmod(gendata->imgdata->maxstep,ftReal,ftImag,filledInterfCycle,localOpd,lambda,outOpdMod,resolution,nbdetector,flagBcdIN);
	if (ier == 0) {
	  // Store results in 2 images
	  // (real and imaginary parts)
	  for(jrow=0;jrow<dimSpatial;jrow++) {
	    cpl_image_set(corrFlux->list_corr[corrFlux->nbframe]->imgreg[0],jrow+1,i+1,cpl_vector_get(outOpdMod,jrow));
	    cpl_image_set(corrFlux->list_corr[corrFlux->nbframe]->imgreg[1],jrow+1,i+1,cpl_vector_get(outOpdMod,jrow+dimSpatial));
	  }
	} 
      }
      cpl_image_delete(self);
      if (in != NULL) free(in);
      if (outOpdMod != NULL)
	cpl_vector_delete(outOpdMod);
      if (ftReal != NULL)
	cpl_vector_delete(ftReal);
      if (ftImag != NULL)
	cpl_vector_delete(ftImag);
      if (ier == 0) {
	corrFlux->nbframe++;
      } else {
	cpl_image_delete(corrFlux->list_corr[corrFlux->nbframe]->imgreg[0]);
	cpl_image_delete(corrFlux->list_corr[corrFlux->nbframe]->imgreg[1]);
	cpl_free(corrFlux->list_corr[corrFlux->nbframe]->imgreg);
	cpl_free(corrFlux->list_corr[corrFlux->nbframe]);
      }

    }
    //RAZ
    for(k=0;k<nbImgInterfCycle;k++) {
      if (filledInterfCycle[k/corrFlux->imgdet->nbregion] == 1) {
	cpl_image_delete(imgInterfCycle[k]);
      }
    }
    for(l=0;l<gendata->imgdata->maxstep;l++) {
      filledInterfCycle[l]=0;
    }
		    
    // store the following images
    (*cp)=gendata->imgdata->list_frame[jframe]->stepphase;
    *jstep=jframe;
    for(k=0;k<4;k++) {
      cpl_vector_set(localOpd,k*gendata->imgdata->maxstep+(*cp)-1,cpl_matrix_get(localOpdBcd,jframe,k));
    }
    filledInterfCycle[(*cp)-1]=1;
    for(k=0;k<corrFlux->imgdet->nbregion;k++) {
      if (strstr(modeFirst,"s") != NULL) {
	ix=5*k+2;
      } else {
	ix=k;
      }
      imgInterfCycle[k*gendata->imgdata->maxstep+(*cp)-1]=cpl_image_duplicate(gendata->imgdata->list_frame[jframe]->list_subwin[ix]->imgreg[0]);
    }		  
  }
  return 0;
}

/**
   @brief Demodulate the fringe signal
   @param gendata contains the mat_gendata structure
   @param corrFlux contains a mat_corrflux structure
   @param modeFirst "s" or "h"
   @param jframe current frame index
   @param dispCoef dispersion law coefficient
   @param localOpdBcd opd applied by teh opd modulation system
   @return 0 if no error
*/
int mat_demodulate_fringes(mat_gendata *gendata,mat_corrflux *corrFlux, char modeFirst[], 
                           int jframe, cpl_vector *dispCoef, cpl_matrix *localOpdBcd) {
  int k=0;
  int i=0;
  int ix=0;
  int jrow=0;
  int dimSpatial=0;
  int dimSpectral=0;
  int flag_rejected=0;
  double lambda=0.;
  double localOpdFrame[4]={0.,0.,0.,0.};
  cpl_vector *outOpdMod=NULL;
  cpl_vector *ftReal=NULL;
  cpl_vector *ftImag=NULL;
  double complex *in;
  double complex *out;
  cpl_image *self=NULL;
  cpl_image *other=NULL;
  char *detName;
  char *specResData;
  char *fil;
  int resolution;
  int nbdetector;
  /* double posFringePeak; */
  /* double dOverLambda; */
  /* cpl_vector *vecWindow=NULL; */
  /* cpl_vector *vecFoo=NULL; */
  char *bcd1Status;
  int flagBcdIN=0;
  
  bcd1Status = (char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS BCD1 ID");
  if(!strcmp(bcd1Status, "IN")) {
    flagBcdIN=1;
  } else {
    flagBcdIN=0;
  }
  

  detName=(char *) cpl_propertylist_get_string(corrFlux->keywords,"ESO DET CHIP NAME");
  if (detName == NULL) {
    cpl_msg_error(cpl_func,
		  "no ESO DET CHIP NAME keyword in frame");
    return -1;
  }
  
  
  if(!strcmp(detName, "AQUARIUS")) {
    nbdetector=2;
    specResData=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS DIN ID");
    if (!strcmp(specResData,"LOW")) {
      resolution=1;
    }
    if (!strcmp(specResData,"HIGH")) {
      resolution=2;
    }
    
  }
  else {
    nbdetector=1;
    specResData=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS DIL ID");
    fil=(char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS FIL ID");
    if (!strcmp(specResData,"LOW")) {
      resolution=1;
    }
    if (!strcmp(specResData,"MED")) {
      resolution=2;
    }
    if (!strcmp(specResData,"HIGH")) {
      resolution=3;
    }
    if (!strcmp(specResData,"HIGH+") && !strcmp(fil,"L")) {
      resolution=4;
    }
    if (!strcmp(specResData,"HIGH+") && !strcmp(fil,"M")) {
      resolution=5;
    }
  }

  for(k=0;k<corrFlux->imgdet->nbregion;k++) {
    if (strstr(modeFirst,"s") != NULL) {
      ix=5*k+2;
    } else {
      ix=k;
    }
    dimSpatial = gendata->imgdet->list_region[ix]->naxis[0];
    dimSpectral = gendata->imgdet->list_region[ix]->naxis[1];
    corrFlux->list_corr[corrFlux->nbframe]=cpl_malloc(sizeof(mat_imgcorr));
    corrFlux->list_corr[corrFlux->nbframe]->numregion=k+1;
    corrFlux->list_corr[corrFlux->nbframe]->numdetector=gendata->imgdet->list_region[k]->numdetector;
    corrFlux->list_corr[corrFlux->nbframe]->time=gendata->imgdata->list_frame[jframe]->time;
    corrFlux->list_corr[corrFlux->nbframe]->exptime=gendata->imgdata->list_frame[jframe]->exptime;
    corrFlux->list_corr[corrFlux->nbframe]->nbimgreg=2;
    corrFlux->list_corr[corrFlux->nbframe]->imgreg=cpl_calloc(2,sizeof(cpl_image *));
    corrFlux->list_corr[corrFlux->nbframe]->imgreg[0]=cpl_image_new(dimSpatial,dimSpectral,CPL_TYPE_FLOAT);
    corrFlux->list_corr[corrFlux->nbframe]->imgreg[1]=cpl_image_new(dimSpatial,dimSpectral,CPL_TYPE_FLOAT);
    ftReal=cpl_vector_new(dimSpatial);
    ftImag=cpl_vector_new(dimSpatial);

    /* vecWindow=cpl_vector_new(dimSpatial); */
    /* vecFoo=cpl_vector_new(dimSpatial); */
    /* cpl_vector_fill_kernel_profile(vecFoo,CPL_KERNEL_HAMMING,CPL_KERNEL_DEF_WIDTH); */
    /* for(jrow=0;jrow<dimSpatial;jrow++) { */
    /*   if (jrow<dimSpatial/2) { */
    /* 	cpl_vector_set(vecWindow,jrow,cpl_vector_get(vecFoo,dimSpatial/2-jrow)); */
    /*   } else { */
    /* 	cpl_vector_set(vecWindow,jrow,cpl_vector_get(vecFoo,jrow-dimSpatial/2)); */
    /*   } */
    /* } */
    /* cpl_vector_delete(vecFoo); */
    /* cpl_plot_vector(";;", */
    /* 		    "with lines","",vecWindow); */

    /* exit(1); */

    outOpdMod=cpl_vector_new(2*dimSpatial);
    in=(double complex *)malloc(sizeof(double complex)*dimSpatial);
    self=cpl_image_new(dimSpatial,1,CPL_TYPE_DOUBLE | CPL_TYPE_COMPLEX);
    // Compute the FFT of each spctral channel of
    // each step of the OPD modulation

    /* printf("%f\n",cpl_image_get_flux(gendata->imgdata->list_frame[jframe]->list_subwin[ix]->imgreg[0])); */

    for(i=0;i<dimSpectral;i++) {

      // Before FFT, shift image by minus half spatial size
      for(jrow=0;jrow<dimSpatial;jrow++) {
        in[jrow]=cpl_image_get(gendata->imgdata->list_frame[jframe]->list_subwin[ix]->imgreg[0],(jrow+dimSpatial/2)%dimSpatial+1,i+1,&flag_rejected);
      }

      // Perform FFT
      other=cpl_image_wrap_double_complex(dimSpatial,1,in);
      if (i == 0) {
	cpl_fft_image(self,other,CPL_FFT_FORWARD | CPL_FFT_FIND_MEASURE);
      } else {
	cpl_fft_image(self,other,CPL_FFT_FORWARD);
      }
      out=cpl_image_get_data_double_complex(self);

      // After FFT, shift image by plus half spatial size
      for(jrow=0;jrow<dimSpatial;jrow++) {
        cpl_vector_set(ftReal,(jrow+dimSpatial/2)%dimSpatial,creal(out[jrow]));
        cpl_vector_set(ftImag,(jrow+dimSpatial/2)%dimSpatial,cimag(out[jrow]));
      }

      cpl_image_unwrap(other);
      lambda=mat_get_lambda(dispCoef,i+gendata->imgdet->list_region[ix]->corner[1]);
      int flagUp=1;
      if (jframe == 0) {
        if (cpl_matrix_get(localOpdBcd,jframe,0)-cpl_matrix_get(localOpdBcd,jframe+1,0) > 0) {
          flagUp=0;
	}
      } else {
        if (cpl_matrix_get(localOpdBcd,jframe,0)-cpl_matrix_get(localOpdBcd,jframe-1,0) < 0) {
          flagUp=0;
        }
      }
      
      for(jrow=0;jrow<MAT_MAXTEL;jrow++) {
	localOpdFrame[jrow]=cpl_matrix_get(localOpdBcd,jframe,jrow);
      }
      mat_apply_demod(ftReal,ftImag,localOpdFrame,lambda,outOpdMod,resolution,nbdetector,flagUp,flagBcdIN);
      // Store results in 2 images
      // (real and imaginary parts)
      for(jrow=0;jrow<dimSpatial;jrow++) {
	cpl_image_set(corrFlux->list_corr[corrFlux->nbframe]->imgreg[0],jrow+1,i+1,cpl_vector_get(outOpdMod,jrow));
	cpl_image_set(corrFlux->list_corr[corrFlux->nbframe]->imgreg[1],jrow+1,i+1,cpl_vector_get(outOpdMod,jrow+dimSpatial));
	/* cpl_image_set(corrFlux->list_corr[corrFlux->nbframe]->imgreg[0],jrow+1,i+1,cpl_vector_get(ftReal,jrow)); */
	/* cpl_image_set(corrFlux->list_corr[corrFlux->nbframe]->imgreg[1],jrow+1,i+1,cpl_vector_get(ftImag,jrow)); */
      }
    }

    /* cpl_image_save(corrFlux->list_corr[corrFlux->nbframe]->imgreg[0],"real.fits", CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE); */
    /* cpl_image_save(corrFlux->list_corr[corrFlux->nbframe]->imgreg[1],"imag.fits", CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE); */
    /* exit(1); */

    cpl_image_delete(self);
    if (in != NULL) free(in);
    if (outOpdMod != NULL) {
      cpl_vector_delete(outOpdMod);
    }
    if (ftReal != NULL) cpl_vector_delete(ftReal);
    if (ftImag != NULL) cpl_vector_delete(ftImag);
    /* if (vecWindow != NULL) cpl_vector_delete(vecWindow); */
    corrFlux->nbframe++;
  }
  return 0;
}

/**
   @brief Remove an averaged sky from an image
   @param gendata contains the mat_gendata structure
   @param avgSky image containing the average sky
   @param modeFirst "s" or "h"
   @return 0 if no error
*/
int mat_remove_avgsky(mat_gendata *gendata, cpl_image **avgSky, char modeFirst[]) {
  int k=0;
  int ix=0;
  int j=0;
  int nbregion=0;
  if (strstr(modeFirst,"s") != NULL) {
    nbregion=gendata->imgdet->nbregion/5;
  } else {
    nbregion=gendata->imgdet->nbregion;
  }
  

  for(k=0;k<nbregion;k++) {
    if (strstr(modeFirst,"s") != NULL) {
      ix=5*k+2;
    } else {
      ix=k;
    }
    for(j=0;j<gendata->imgdata->nbframe;j++) {
      cpl_image_subtract(gendata->imgdata->list_frame[j]->list_subwin[ix]->imgreg[0],avgSky[k]);
    }
  }
  return 0;
}
/**
   @brief Remove an averaged image from an image
   @param gendata contains the mat_gendata structure
   @param avgImg image containing the average image
   @param modeFirst "s" or "h"
   @return 0 if no error
*/
int mat_remove_avgimg(mat_gendata *gendata, cpl_image **avgImg, char modeFirst[]) {
  int k=0;
  int ix=0;
  int j=0;
  int nbregion=0;
  if (strstr(modeFirst,"s") != NULL) {
    nbregion=gendata->imgdet->nbregion/5;
  } else {
    nbregion=gendata->imgdet->nbregion;
  }
  

  for(k=0;k<nbregion;k++) {
    if (strstr(modeFirst,"s") != NULL) {
      ix=5*k+2;
    } else {
      ix=k;
    }
    for(j=0;j<gendata->imgdata->nbframe;j++) {
      if ( strcmp(gendata->imgdata->list_frame[j]->tartype,"T") == 0 ) {	    
	cpl_image_subtract(gendata->imgdata->list_frame[j]->list_subwin[ix]->imgreg[0],avgImg[k]);
      }
    }
  }
  return 0;
}



/**
   @brief Switch the localOpd vector of a mat_gendata structure wrt to bcd1 and bcd2
   @param gendata contains the mat_gendata structure
   @return a cpl_matrix (nbframexnbtel) which contains the localOpd 
*/
cpl_matrix *mat_switch_localopd_wrt_bcd(mat_gendata *gendata) {
  cpl_matrix *localOpdBcd=NULL;
  double mjd=0.;
  int j=0;
  /* int i=0; */
  mjd=cpl_propertylist_get_double(gendata->keywords,"MJD-OBS");
  localOpdBcd=cpl_matrix_new(gendata->imgdata->nbframe,MAT_MAXTEL);
  if ( gendata->imgdet->nbdetector==1 ) {
    for(j=0;j<gendata->imgdata->nbframe;j++) {
      if (mjd < 58204.5) {
      	if (j<gendata->imgdata->nbframe-1 ) {
      	  cpl_matrix_set(localOpdBcd,j,0,gendata->imgdata->list_frame[j+1]->localopd[0]);
      	  cpl_matrix_set(localOpdBcd,j,1,gendata->imgdata->list_frame[j+1]->localopd[1]);
      	  cpl_matrix_set(localOpdBcd,j,2,gendata->imgdata->list_frame[j+1]->localopd[2]);
      	  cpl_matrix_set(localOpdBcd,j,3,gendata->imgdata->list_frame[j+1]->localopd[3]);
      	} else {
      	  cpl_matrix_set(localOpdBcd,j,0,gendata->imgdata->list_frame[j]->localopd[0]);
      	  cpl_matrix_set(localOpdBcd,j,1,gendata->imgdata->list_frame[j]->localopd[1]);
      	  cpl_matrix_set(localOpdBcd,j,2,gendata->imgdata->list_frame[j]->localopd[2]);
      	  cpl_matrix_set(localOpdBcd,j,3,gendata->imgdata->list_frame[j]->localopd[3]);
      	}
      } else if (mjd >= 58240.5 && mjd < 58255.5)  {
      	if (j>0 ) {
      	  cpl_matrix_set(localOpdBcd,j,0,gendata->imgdata->list_frame[j-1]->localopd[0]);
      	  cpl_matrix_set(localOpdBcd,j,1,gendata->imgdata->list_frame[j-1]->localopd[1]);
      	  cpl_matrix_set(localOpdBcd,j,2,gendata->imgdata->list_frame[j-1]->localopd[2]);
      	  cpl_matrix_set(localOpdBcd,j,3,gendata->imgdata->list_frame[j-1]->localopd[3]);
      	} else {
      	  cpl_matrix_set(localOpdBcd,j,0,gendata->imgdata->list_frame[j]->localopd[0]);
      	  cpl_matrix_set(localOpdBcd,j,1,gendata->imgdata->list_frame[j]->localopd[1]);
      	  cpl_matrix_set(localOpdBcd,j,2,gendata->imgdata->list_frame[j]->localopd[2]);
      	  cpl_matrix_set(localOpdBcd,j,3,gendata->imgdata->list_frame[j]->localopd[3]);
      	}
      } else  {
	cpl_matrix_set(localOpdBcd,j,0,gendata->imgdata->list_frame[j]->localopd[0]);
	cpl_matrix_set(localOpdBcd,j,1,gendata->imgdata->list_frame[j]->localopd[1]);
	cpl_matrix_set(localOpdBcd,j,2,gendata->imgdata->list_frame[j]->localopd[2]);
	cpl_matrix_set(localOpdBcd,j,3,gendata->imgdata->list_frame[j]->localopd[3]);
      }
    }
  }
  if ( gendata->imgdet->nbdetector==2 ) {
    for(j=0;j<gendata->imgdata->nbframe;j++) {
      // Ordre a verifier
      if (j<gendata->imgdata->nbframe-1 && mjd < 58204.5) {
	cpl_matrix_set(localOpdBcd,j,0,gendata->imgdata->list_frame[j+1]->localopd[0]);
	cpl_matrix_set(localOpdBcd,j,1,gendata->imgdata->list_frame[j+1]->localopd[1]);
	cpl_matrix_set(localOpdBcd,j,2,gendata->imgdata->list_frame[j+1]->localopd[2]);
	cpl_matrix_set(localOpdBcd,j,3,gendata->imgdata->list_frame[j+1]->localopd[3]);
      } else {
	cpl_matrix_set(localOpdBcd,j,0,gendata->imgdata->list_frame[j]->localopd[0]);
	cpl_matrix_set(localOpdBcd,j,1,gendata->imgdata->list_frame[j]->localopd[1]);
	cpl_matrix_set(localOpdBcd,j,2,gendata->imgdata->list_frame[j]->localopd[2]);
	cpl_matrix_set(localOpdBcd,j,3,gendata->imgdata->list_frame[j]->localopd[3]);
      }
    }
  }
  /* if(strcmp(bcd1StatusFirst, "OUT") == 0) { */
  /*   cpl_matrix_set(localOpdBcd,j,0,gendata->imgdata->list_frame[j]->localopd[0]); */
  /*   cpl_matrix_set(localOpdBcd,j,1,gendata->imgdata->list_frame[j]->localopd[1]); */
  /* } else { */
  /*   cpl_matrix_set(localOpdBcd,j,1,gendata->imgdata->list_frame[j]->localopd[0]); */
  /*   cpl_matrix_set(localOpdBcd,j,0,gendata->imgdata->list_frame[j]->localopd[1]); */
  /* } */
  /* if(strcmp(bcd2StatusFirst, "OUT") == 0) { */
  /*   cpl_matrix_set(localOpdBcd,j,2,gendata->imgdata->list_frame[j]->localopd[2]); */
  /*   cpl_matrix_set(localOpdBcd,j,3,gendata->imgdata->list_frame[j]->localopd[3]); */
  /* } else { */
  /*   cpl_matrix_set(localOpdBcd,j,3,gendata->imgdata->list_frame[j]->localopd[2]); */
  /*   cpl_matrix_set(localOpdBcd,j,2,gendata->imgdata->list_frame[j]->localopd[3]); */
  /* } */
  /* } */
  return localOpdBcd;
}



/**
   @brief Initialize a mat_corrflux structure from a mat_gendata structure
   @param gendata contains the mat_gendata structure
   @param modeFirst "s" or "h"
   @return mat_corrflux
*/
mat_corrflux *mat_init_corrflux(mat_gendata *gendata, char modeFirst[]) {
  int i=0;
  int j=0;
  mat_corrflux *corrFlux=NULL;

  // Initialize mat_corrflux structure
  corrFlux=cpl_malloc(sizeof(mat_corrflux));
  if (corrFlux == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for corrFlux");
    return NULL;
  }
  corrFlux->keywords=cpl_propertylist_duplicate(gendata->keywords);
  if (corrFlux->keywords == NULL) {
    cpl_msg_error(cpl_func,"could not duplicate corrFlux->keywords");
    mat_corrflux_free(corrFlux);
    return NULL;
  }
  corrFlux->imgdet=mat_imagingdetector_duplicate(gendata->imgdet);
  if (corrFlux->imgdet == NULL) {
    cpl_msg_error(cpl_func,"could not duplicate corrFlux->imgdet");
    mat_corrflux_free(corrFlux);
    return NULL;
  }
	
  if (strstr(modeFirst,"s") != NULL) {
    // remove all regions
    for (i=0;i<corrFlux->imgdet->nbregion;i++) {
      cpl_free(corrFlux->imgdet->list_region[i]->numbeam);
      cpl_free(corrFlux->imgdet->list_region[i]->regionname);
      for(j=0;j<2;j++) {
	cpl_free(corrFlux->imgdet->list_region[i]->ctype[j]);
      }
      cpl_free(corrFlux->imgdet->list_region[i]);
    }
    cpl_free(corrFlux->imgdet->list_region);
    // Keep interferometric regions only
    corrFlux->imgdet->nbregion=gendata->imgdet->nbregion/5;
    corrFlux->imgdet->list_region=cpl_malloc(corrFlux->imgdet->nbregion*sizeof(mat_region *));
    if (corrFlux->imgdet->list_region == NULL) {
      cpl_msg_error(cpl_func,"could not allocate memory for corrFlux->imgdet->list_region");
      mat_corrflux_free(corrFlux);
      return NULL;
    }
    for (i=0;i<corrFlux->imgdet->nbregion;i++) {
      corrFlux->imgdet->list_region[i]=
	cpl_malloc(sizeof(mat_region));
      if (corrFlux->imgdet->list_region[i] == NULL) {
	cpl_msg_error(cpl_func,"could not allocate memory for corrFlux->imgdet->list_region[i]");
	mat_corrflux_free(corrFlux);
	return NULL;
      }
      corrFlux->imgdet->list_region[i]->numregion=i+1;
      corrFlux->imgdet->list_region[i]->regionname=cpl_malloc(10*sizeof(char));
      strncpy(corrFlux->imgdet->list_region[i]->regionname,(char *)gendata->imgdet->list_region[i*5+2]->regionname,10);
      corrFlux->imgdet->list_region[i]->numdetector=gendata->imgdet->list_region[i*5+2]->numdetector;
      corrFlux->imgdet->list_region[i]->numbeam=cpl_calloc(gendata->imgdet->nbtel,sizeof(int));
      for(j=0;j<gendata->imgdet->nbtel;j++) {
	corrFlux->imgdet->list_region[i]->numbeam[j]=gendata->imgdet->list_region[i*5+2]->numbeam[j];
      }
      corrFlux->imgdet->list_region[i]->correlation=gendata->imgdet->list_region[i*5+2]->correlation;
      for(j=0;j<2;j++) {
	corrFlux->imgdet->list_region[i]->corner[j]=gendata->imgdet->list_region[i*5+2]->corner[j];
      }
      corrFlux->imgdet->list_region[i]->gain=gendata->imgdet->list_region[i*5+2]->gain;
      for(j=0;j<2;j++) {
	corrFlux->imgdet->list_region[i]->crval[j]=gendata->imgdet->list_region[i*5+2]->crval[j];
      }
      for(j=0;j<2;j++) {
	corrFlux->imgdet->list_region[i]->crpix[j]=gendata->imgdet->list_region[i*5+2]->crpix[j];
      }
      for(j=0;j<2;j++) {
	corrFlux->imgdet->list_region[i]->ctype[j]=cpl_malloc(10*sizeof(char));
	strncpy(corrFlux->imgdet->list_region[i]->ctype[j],(char *)gendata->imgdet->list_region[i*5+2]->ctype[j],10);
      }
      for(j=0;j<2;j++) {
	corrFlux->imgdet->list_region[i]->naxis[j]=gendata->imgdet->list_region[i*5+2]->naxis[j];
      }
    }
  }
  corrFlux->nbframe=0; 	  // Initialization
  // we use the nb of frame of gendata because we do not know the
  // number of OPD modulation cycle here
  corrFlux->list_corr=cpl_calloc(gendata->imgdata->nbframe*corrFlux->imgdet->nbregion,sizeof(mat_imgcorr *));
  if (corrFlux->list_corr == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for corrFlux->list_corr");
    mat_corrflux_free(corrFlux);
    return NULL;
  }
  return corrFlux;
}


/**
   @brief Compute the moments (p,q) of an image
   @param img image to process
   @param p moment order in axis X
   @param q moment order in axis Y
   @param x0 photocenter in axis X (required to compute central moments)
   @param y0 photocenter in axis Y (required to compute central moments)
   @return moment value
*/
double mat_image_get_moment(cpl_image *img, int p, int q, double x0, double y0) {
  int i=0;
  int j=0;
  double moment=0.;
  int is_rejected=0;

  for(i=1;i<=cpl_image_get_size_x(img);i++) {
    for(j=1;j<=cpl_image_get_size_y(img);j++) {
      moment+=pow((double)(i-x0),(double)p)*pow((double)(j-y0),(double)q)*
	cpl_image_get(img,i,j,&is_rejected);
    }
  }

  return moment;
}

/**
   @brief Compute the moments (p,q) of a vector
   @param vec vector to process
   @param p moment order in axis X
   @return moment value
*/
double mat_vector_get_moment(cpl_vector *vec, int p) {
  int i=0;
  double moment=0.;

  for(i=0;i<cpl_vector_get_size(vec);i++) {
    moment+=pow((double)(i),(double)p)*cpl_vector_get(vec,i);
  }

  return moment;
}

/**
   @brief Remove the origin phase
   @param corrFlux contains a mat_corrflux structure
   @param shiftMean This parameter contains the mean offset
   @return 0 if no error
*/
int mat_remove_origin_phase(mat_corrflux *corrFlux,double shiftMean) {
  int dimSpatial=0;
  int dimSpectral=0;
  int i=0;
  int j=0;
  int l=0;
  double alpha=0.;
  int rej=0;
  double valre=0.;
  double valim=0.;
  
  for(l=0;l<corrFlux->nbframe;l++) {
    dimSpatial = cpl_image_get_size_x(corrFlux->list_corr[l]->imgreg[0]);
    dimSpectral = cpl_image_get_size_y(corrFlux->list_corr[l]->imgreg[0]);
    for(i=0;i<dimSpectral;i++) {
      for(j=0;j<dimSpatial;j++) {
	alpha=2.*CPL_MATH_PI*(j-(dimSpatial/2))*shiftMean/dimSpatial;
	valre=cpl_image_get(corrFlux->list_corr[l]->imgreg[0],j+1,i+1,&rej);
	valim=cpl_image_get(corrFlux->list_corr[l]->imgreg[1],j+1,i+1,&rej);
	cpl_image_set(corrFlux->list_corr[l]->imgreg[0],j+1,i+1,valre*cos(alpha)+valim*sin(alpha));
	cpl_image_set(corrFlux->list_corr[l]->imgreg[1],j+1,i+1,valim*cos(alpha)-valre*sin(alpha));
      }
    }
  }
  return 0;
}

/**
   @brief Define the origin phase
   @param image contains an image of the FFT of an interferogram
   @param dispCoef coefficient for spectral dispersion law
   @param corner corner pixel of the image in the spectral direction
   @param nbdetector detector number
   @param resolution resolution number
   @return 0 if no error
*/
double mat_get_origin_phase(cpl_image *image, cpl_vector *dispCoef, int corner, int nbdetector, int resolution) {
  int dimSpatial=0;
  int dimSpectral=0;
  /* int k=0; */
  int i=0;
  int j=0;
  /* int l=0; */
  int m=0;
  int jrow=0;
  double lambda=0.;
  double dOverLambda=0.;
  double posFringePeak=0.;
  double valmin=0.;
  double shift=0.;
  double shiftMin=0.; 
  cpl_vector *shiftMinMean=NULL;
  double alpha=0.;
  double obj=0.;
  int rej=0;
  /* double valre=0.; */
  /* double valim=0.; */
  cpl_vector *ftReal=NULL;
  cpl_vector *ftImag=NULL;
  double complex *in;
  double complex *out;
  cpl_image *self=NULL;
  cpl_image *other=NULL;
  
  dimSpatial = cpl_image_get_size_x(image);
  dimSpectral = cpl_image_get_size_y(image);
  shiftMinMean=cpl_vector_new(dimSpectral);
  ftReal=cpl_vector_new(dimSpatial);
  ftImag=cpl_vector_new(dimSpatial);
  in=(double complex *)malloc(sizeof(double complex)*dimSpatial);
  self=cpl_image_new(dimSpatial,1,CPL_TYPE_DOUBLE | CPL_TYPE_COMPLEX);
  // Compute the FFT of each spctral channel of
  // each step of the OPD modulation


  /* shiftMinMean=0.; */
  for(i=0;i<dimSpectral;i++) {

    // Before FFT, shift image by minus half spatial size
    for(jrow=0;jrow<dimSpatial;jrow++) {
      in[jrow]=cpl_image_get(image,(jrow+dimSpatial/2)%dimSpatial+1,i+1,&rej);
    }

    // Perform FFT
    other=cpl_image_wrap_double_complex(dimSpatial,1,in);
    if (i == 0) {
      cpl_fft_image(self,other,CPL_FFT_FORWARD | CPL_FFT_FIND_MEASURE);
    } else {
      cpl_fft_image(self,other,CPL_FFT_FORWARD);
    }
    out=cpl_image_get_data_double_complex(self);

    // After FFT, shift image by plus half spatial size
    for(jrow=0;jrow<dimSpatial;jrow++) {
      cpl_vector_set(ftReal,(jrow+dimSpatial/2)%dimSpatial,creal(out[jrow]));
      cpl_vector_set(ftImag,(jrow+dimSpatial/2)%dimSpatial,cimag(out[jrow]));
    }

    cpl_image_unwrap(other);

    lambda=mat_get_lambda(dispCoef,i+corner);
    posFringePeak=mat_get_pos_fringe_peak(lambda,nbdetector,resolution,5);
    dOverLambda=(posFringePeak-dimSpatial/2)/(18);
    shiftMin=0.;
    valmin=1.E+10;
    for(m=0;m<2000;m++) {
      shift=-100.+m/10.;
      obj=0.;
      for(j=0;j<dimSpatial;j++) {
	if ( fabs(j-dimSpatial/2.) < dOverLambda*0.7 && j != (dimSpatial/2.) ) {
	  alpha=2.*CPL_MATH_PI*(j-(dimSpatial/2))*shift/dimSpatial;
	  obj+=pow(cpl_vector_get(ftImag,j)*cos(alpha)-cpl_vector_get(ftReal,j)*sin(alpha),2.0);
	}
      }
      if (obj < valmin) {
	shiftMin=shift;
	valmin=obj;
      }
    }
    /* shiftMinMean+=shiftMin; */
    cpl_vector_set(shiftMinMean,i,shiftMin);
  }
  /* shiftMinMean/=dimSpectral; */
  shiftMin=cpl_vector_get_median(shiftMinMean);
  cpl_vector_delete(shiftMinMean);
  cpl_image_delete(self);
  if (in != NULL) free(in);
  if (ftReal != NULL) cpl_vector_delete(ftReal);
  if (ftImag != NULL) cpl_vector_delete(ftImag);
  return shiftMin;
}

/*----------------------------------------------------------------------------*/

double mat_get_lambda(cpl_vector *dispCoef, int iWlen)
{
  return (cpl_vector_get(dispCoef,0) +
	  cpl_vector_get(dispCoef,1)*iWlen+
	  cpl_vector_get(dispCoef,2)*pow((double)(iWlen),2.0)+
	  cpl_vector_get(dispCoef,3)*pow((double)(iWlen),3.0)+
	  cpl_vector_get(dispCoef,4)*pow((double)(iWlen),4.0));
  /* /(1-0.0197*(cpl_vector_get(dispCoef,0) + */
  /*     cpl_vector_get(dispCoef,1)*iWlen+ */
  /*    cpl_vector_get(dispCoef,2)*pow((double)(iWlen),2.0))); */
}

/*----------------------------------------------------------------------------*/

double mat_get_pos_fringe_peak(double lambda, int nbdetector, int resolution, int nbPeak)
{ 
    
  int    ndeg = 4;
  double pos  = 0.0;
  //LOW Lband
  /* double coefLowL[6][4]={{-0.95944131,   13.41486666,  -66.78534397,  446.09234419}, */
  /* 			 {-1.81480496,   25.22107656, -125.17884483,  566.49904442}, */
  /* 			 {-2.9182797,    40.4952395,  -200.17486549,  714.75630716}, */
  /* 			 {-3.82377804,   53.226404,   -264.13141458,  845.05281112}, */
  /* 			 {-4.6901372,    65.15079323, -322.95736098,  965.97793634}, */
  /* 			 {-5.54047411,    77.1336955,   -383.46780211, 1090.90570748}}; */
 
  double coefLowL[6][4]={{-0.4556,7.119,-40.41,409},
			 {-0.8461,13.15,-74.79,496.2},
			 {-1.276,19.75,-112.2,589.2},
			 {-1.726,26.8,-152.4,686.1},
			 {-2.061,32.08,-183.4,768.2},
			 {-2.466,38.46,-220.2,859.6}};


  //MED Lband
  double coefMedL[6][4]={{-0.40458533,6.29982113,-35.93176046,400.51307076},
			 {-0.768369,12.15301024,-70.67980912,490.80810891},
			 {-1.1838513,18.68916979,-108.42171393,585.63277772},
			 {-1.58397496,24.94976593,-144.37789969,674.71401742},
			 {-1.92893278,30.51967041,-177.59978399,762.15923057},
			 {-2.46631289,38.46283974,-220.20890889,859.57343339}};
  //HIGH Lband
  double coefHighL[6][4]={{-7.76325999e-02,2.85803397e+00,-2.38896136e+01,3.86541237e+02},
  			  {-1.13615654,16.06496795,-84.35454809,506.49602832},
  			  {-2.15840394,28.87467559,-143.72867819,626.28407559},
  			  {-2.57088889,35.05430564,-178.53862469,712.87290071},
  			  {-3.41297488,46.10499035,-231.84891029,824.80604373},
  			  {-4.03534524,54.94605543,-277.68461223,926.20906903}};
  //HIGH+ L Lband
  /* double coefHighPL[6][4]={{151.62040081 ,-1840.78317251  ,7444.42906601 ,-9696.97074995}, */
  /* 			   {-22.25800777   ,270.83742164 ,-1108.16613918  ,1876.72441597}, */
  /* 			   {69.9127686   ,-857.90262575  ,3494.2210153  ,-4350.19872164}, */
  /* 			   {81.65283091  ,-990.56299244  ,3985.21267341 ,-4922.95093638}, */
  /* 			   {-50.0890604    ,622.5113151  ,-2602.91780212  ,4072.98773669}, */
  /* 			   {-124.23058251  ,1494.43034753 ,-6019.38712562  ,8552.92592427}}; */
  double coefHighPL[6][4]={{46.92381825  ,-575.83292663  ,2350.46771501 ,-2859.76121054},
			   {37.85018685  ,-457.31852196  ,1831.95952621 ,-2080.20518923},
			   {135.13876018 ,-1645.17983715  ,6661.27261425 ,-8596.43141241},
			   {74.1450283   ,-897.8869477   ,3603.96969576 ,-4400.28625648},
			   {-74.19344804   ,914.33963092 ,-3780.58861381  ,5657.10747326},
			   {-297.43442211   ,3613.92469913 ,-14665.79945733  ,20311.8765064}};

  //HIGH+ M Lband
  double coefHighPM[6][4]={{-4.04997778e-02   ,1.08717444e+00  ,-1.11757475e+01   ,3.61653205e+02},
			   {-7.97838664e-02   ,2.04182331e+00  ,-2.11370752e+01   ,4.09604587e+02},
			   {-4.37181854e-01   ,7.78418877e+00  ,-5.53540456e+01   ,4.99290911e+02},
			   {-0.62634194   ,11.05653781  ,-77.24346968  ,566.48130022},
			   {3.55555270e-01  ,-2.91074724e+00  ,-1.45110489e+01   ,4.96362387e+02},
			   {-3.82269755e-01   ,8.72164019e+00  ,-7.87577351e+01   ,6.34900337e+02}};
  //LOW N Band
  //NICE
  /* double coefLowN[6][4]={{-2.98163531e-02,1.08358795e+00,-1.41482831e+01,3.12092829e+02}, */
  /* 			 {-4.90964755e-02,1.85980128e+00,-2.54293085e+01,3.81750155e+02}, */
  /* 			 {-8.85899105e-02,3.25959766e+00,-4.32375370e+01,4.75420374e+02}, */
  /* 			 {-8.06383990e-02,3.15070154e+00,-4.49620810e+01,5.10764458e+02}, */
  /* 			 {-1.26728561e-01,4.76146126e+00,-6.48931726e+01,6.10396192e+02}, */
  /* 			 {-1.59168773e-01,5.94189838e+00,-8.01528090e+01,6.92391294e+02}}; */
  //PARANAL
  double coefLowN[6][4]={{-3.95483998e-02,   1.37564618e+00,  -1.70577288e+01,   3.21894600e+02},
			 {-4.76003497e-02,   1.79970437e+00,  -2.47761612e+01,   3.80077092e+02},
			 {-7.40343534e-02,   2.81699698e+00,  -3.88658935e+01,   4.61598450e+02},
			 {-1.00340512e-01,   3.81626748e+00,  -5.24457584e+01,   5.39108924e+02},
			 {-1.26314812e-01,   4.77665904e+00,  -6.54137415e+01,   6.14006202e+02},
			 {-1.56535415e-01,   5.88898888e+00,  -8.00954824e+01,   6.95157234e+02}};
  //HIGH N band
  double coefHighN[6][4]={{-3.43196804e-02,1.20512640e+00,-1.52169779e+01,3.15136231e+02},
			  {-2.97143582e-02,1.22299128e+00,-1.85789319e+01,3.57707050e+02},
			  {-5.00991891e-02,2.02164866e+00,-3.00991437e+01,4.29379529e+02},
			  {-5.80957887e-02,2.41603396e+00,-3.70764878e+01,4.82912685e+02},
			  {-7.10830976e-02,2.96606534e+00,-4.57422329e+01,5.42871607e+02},
			  {-8.73819754e-02,3.63087513e+00,-5.56164682e+01,6.06546595e+02}};

  if (nbdetector == 1) {
    switch(resolution) {
    case 1:
      pos=0.;
      for(int i=0;i<ndeg;i++) {
	pos+=coefLowL[nbPeak][i]*pow(lambda,ndeg-1-i);
      }
      break;
    case 2:
      pos=0.;
      for(int i=0;i<ndeg;i++) {
	pos+=coefMedL[nbPeak][i]*pow(lambda,ndeg-1-i);
      }
      break;

    case 3:
      pos=0.;
      for(int i=0;i<ndeg;i++) {
	pos+=coefHighL[nbPeak][i]*pow(lambda,ndeg-1-i);
      }
      break;

    case 4:
      pos=0.;
      for(int i=0;i<ndeg;i++) {
	pos+=coefHighPL[nbPeak][i]*pow(lambda,ndeg-1-i);
      }
      break;
      
    case 5:
      pos=0.;
      for(int i=0;i<ndeg;i++) {
	pos+=coefHighPM[nbPeak][i]*pow(lambda,ndeg-1-i);
      }
      break;

    default:
      break;
    }
  } else if (nbdetector == 2) {
    switch(resolution) {
    case 1:
      pos=0.;
      for(int i=0;i<ndeg;i++) {
	pos+=coefLowN[nbPeak][i]*pow(lambda,ndeg-1-i);
      }
      break;
    case 2:
      pos=0.;
      for(int i=0;i<ndeg;i++) {
	pos+=coefHighN[nbPeak][i]*pow(lambda,ndeg-1-i);
      }
      break;

    default:
      break;
    }
  } else {
    cpl_msg_error(cpl_func,"Detector not known in mat_get_pos_fringe_peak");
    return -1;
  }
  

  return pos;
}

/*----------------------------------------------------------------------------*/

double mat_abacus_err_phi(double x)
/****************************************************
 * Copy/paste of amdlib's function for phase errors *
 ****************************************************
 *
 * Estimate true phase rms from the cross-spectrum variance.
 * see Petrov, Roddier and Aime, JOSAA vol 3, N°5, may 1986 p 634.
 * and Petrov's Thesis, p. 50 ff.
 * I replace the piecewise interpolation usually used by a polynomial
 * approximation of the function:
 * if z=log10(y), 
 * then z=(C[1]*X^7+C[2]*X^6+C[3]*X^5+C[4]*X^4+C[5]*X^3+C[6]*X^2+C[7]*X+C[8])
 * and y=10^z.
 * where
 * C[01]= 2.71918080109099
 * C[02]=-17.1901043936273
 * C[03]= 45.0654103760899
 * C[04]=-63.4441678243197
 * C[05]= 52.3098941426378
 * C[06]=-25.8090699917488
 * C[07]= 7.84352873962491
 * C[08]=-1.57308595820081
 * This interpolation is valid in the range x=[0.1,1.74413]. 
 * Error is 1% everywhere except above x=1.73 where it is 10%
 * Below x=0.1: y=x
 * Above x=M_PI/sqrt(3.0), y = blanking (impossible value)
 * Above x=1.74413, we take: y=0.691/(pi/sqrt(3.0)-x)
 */
{
  const double Asymptot=CPL_MATH_PI/sqrt(3.0);
  double c[8]={2.7191808010909,
	       -17.1901043936273,
	       45.0654103760899,
	       -63.4441678243197,
	       52.3098941426378,
	       -25.8090699917488,
	       7.84352873962491,
	       -1.57308595820081};
    
  double x2,x3,x4,x5,x6,x7,z;
    
  if(x>Asymptot)
    return((double)4*CPL_MATH_PI);
  else if(x>1.74413)
    return(0.691/(Asymptot-x));
  else if(x<0.1)
    return(x);
    
  x2=x*x;
  x3=x2*x;
  x4=x2*x2;
  x5=x3*x2;
  x6=x3*x3;
  x7=x6*x;
  z=c[0]*x7+c[1]*x6+c[2]*x5+c[3]*x4+c[4]*x3+c[5]*x2+c[6]*x+c[7];
    
  return pow(10,z);
}

/*----------------------------------------------------------------------------*/

int mat_get_err_phi(cpl_image *err_in, cpl_image *err_out)
{
  int iWlen=0,iBase=0;
  int pis = 0;
  int nbWlen, nbBases;
  nbBases = cpl_image_get_size_x (err_in);
  nbWlen  = cpl_image_get_size_y (err_in);
    
  for(iWlen=0;iWlen<nbWlen;iWlen++)
    {
      for(iBase=0;iBase<nbBases;iBase++)
        {     
	  cpl_image_set(err_out, iBase+1, iWlen+1, 
			mat_abacus_err_phi(cpl_image_get(err_in,iBase+1,iWlen+1,&pis)));
        }
    }
  return 0;
}

/*----------------------------------------------------------------------------*/

int mat_identify_det(char *detName, char *band, int *detNum)
/* Identifies detector and band ('L' for L, M or LM */
{
  cpl_msg_info(cpl_func, "Looking for detector..."); 
  if(!strcmp(detName, "AQUARIUS"))
    {
      /* cpl_msg_info(cpl_func,detName, "AQUARIUS"); */
      *detNum    = 2;
      *band = 'N';
    }
  else
    {
      /* cpl_msg_info(cpl_func,detName, "HAWAI-2RG"); */
      *detNum    = 1;
      *band = 'L';
    }
  return(0);
}

/*----------------------------------------------------------------------------*/

int mat_identify_wlrng(int detNum, float wlenMin, float wlenMax, int resolution, float *lambdaMin, float *lambdaMax)
/* Identifies the wavelength range */
{
  cpl_msg_info(cpl_func, "Looking for wavelength range..."); 
  cpl_msg_info(cpl_func, "wlenMin: %f",wlenMin); 
  cpl_msg_info(cpl_func, "wlenMax: %f",wlenMax); 
  if(detNum == 1)
    {
      if (resolution == 4)
	{
	  *lambdaMin = 3.98;
	  *lambdaMax = 4.1;
	}
      else if (resolution == 5)
	{
	  *lambdaMin = 4.6;
	  *lambdaMax = 5.1;
	}
      else
	{
	  
	  /* Hawaii = L, M or LM */
	  /* cpl_msg_info(cpl_func,detName, "HAWAI-2RG"); */

	  /* L only: wavelength should be < 4.5 microns */
	  /* LM: wavelength max should be > 4.5 microns & wlenMin < 3.8 microns */
	  /* We use the L band for OPD & reference channel */
	  if((wlenMin < 4.5) || ((wlenMax > 4.5) && (wlenMin < 3.8)))
	    {
	      if (resolution == 3)
		{
		  *lambdaMin = 3.0;
		  *lambdaMax = 4.2;
		}
	      else if(resolution == 2)
		{
		  *lambdaMin = 3.0;
		  *lambdaMax = 4.2;      
		}
	      else
		{
		  *lambdaMin = 3.0;
		  *lambdaMax = 3.8;
		}	    
	    }
	  /* M only: wavelength should be > 4.5 microns */
	  else if(wlenMin >= 4.5)
	    {
	      *lambdaMin = 4.5;
	      *lambdaMax = 5.0;
	    }
	}
    }
  else if(detNum == 2)
    {
      /* cpl_msg_info(cpl_func,detName, "AQUARIUS"); */
      *lambdaMin = 8.;
      *lambdaMax = 13.;
    }
  else
    cpl_msg_error(cpl_func, "Detector is unknown!"); 

  return(0);
}

/*----------------------------------------------------------------------------*/

int mat_identify_res(char *specResData, char *fil, int detNum, int *resolution)
/* Identifies the spectral resolution */
{
  cpl_msg_info(cpl_func, "Looking for resoltution..."); 
  if (detNum == 1)
    {
      if (!strcmp(specResData,"LOW"))
	{
	  *resolution = 1;
	}
      else if (!strcmp(specResData,"MED"))
	{
	  *resolution=2;
	}
      else if (!strcmp(specResData,"HIGH"))
	{
	  *resolution=3;
	}
      else if (!strcmp(specResData,"HIGH+") && !strcmp(fil,"L"))
	{
	  *resolution = 4;
	}
      else if (!strcmp(specResData,"HIGH+") && !strcmp(fil,"M"))
	{
	  *resolution = 5;
	}
    }
  else if(detNum == 2)
    {
      if (!strcmp(specResData,"LOW"))
	{
	  *resolution = 1;
	}
      else if (!strcmp(specResData,"HIGH"))
	{
	  *resolution=2;
	}
    }  
  else
    cpl_msg_error(cpl_func, "Detector is unknown!"); 
    
  return(0);
}

/*----------------------------------------------------------------------------*/






mat_gendata * mat_gendata_select_band(mat_gendata *genData,cpl_vector *Coef) {
  mat_gendata *genDataSelected = NULL;
  int flagMin=0;
  int flagMax=0;
  int pxMax=0;
  int pxMin=0;
  cpl_vector *dispCoef=NULL;
  int x, y, y1, rejected;
  int i,j,k,r;
  int px;
  char *fil = NULL;
  char *res = NULL;

  dispCoef=cpl_vector_new(N_DEG_DISPERSION_LAW+1);
  if (Coef != NULL) {
    cpl_vector_set(dispCoef,0,cpl_vector_get(Coef,0));  
    cpl_vector_set(dispCoef,1,cpl_vector_get(Coef,1));  
    cpl_vector_set(dispCoef,2,cpl_vector_get(Coef,2));  
    cpl_vector_set(dispCoef,3,cpl_vector_get(Coef,3));  
    cpl_vector_set(dispCoef,4,cpl_vector_get(Coef,4));  
  } else {
    cpl_vector_set(dispCoef,0,cpl_propertylist_get_double(genData->keywords,"PRO DISP COEF0"));  
    cpl_vector_set(dispCoef,1,cpl_propertylist_get_double(genData->keywords,"PRO DISP COEF1"));  
    cpl_vector_set(dispCoef,2,cpl_propertylist_get_double(genData->keywords,"PRO DISP COEF2"));  
    cpl_vector_set(dispCoef,3,cpl_propertylist_get_double(genData->keywords,"PRO DISP COEF3"));  
    cpl_vector_set(dispCoef,4,cpl_propertylist_get_double(genData->keywords,"PRO DISP COEF4"));  
  }

  // select limit in pixels
  for(i=0;i<genData->imgdet->list_region[0]->naxis[1];i++) {
    px=genData->imgdet->list_region[0]->corner[1]+i;
    if (genData->imgdet->list_region[0]->numdetector == 1) {
      res=(char *)cpl_propertylist_get_string(genData->keywords,"ESO INS DIL ID");
      fil=(char *)cpl_propertylist_get_string(genData->keywords,"ESO INS FIL NAME");
      if (!strcmp(res,"HIGH+") && !strcmp(fil,"L"))
	{
	  if (mat_get_lambda(dispCoef,px) <= 3.98 && flagMin == 0) {
	    pxMax=px+1;
	    flagMin=1;
	  }
	  if (mat_get_lambda(dispCoef,px) <= 4.1 && flagMax == 0) {
	    pxMin=px;
	    flagMax=1;
	  }
	}
      else if (!strcmp(res,"HIGH+") && !strcmp(fil,"M"))
	{
	  if (mat_get_lambda(dispCoef,px) <= 4.55 && flagMin == 0) {
	    pxMax=px+1;
	    flagMin=1;
	  }
	  if (mat_get_lambda(dispCoef,px) <= 5.15 && flagMax == 0) {
	    pxMin=px;
	    flagMax=1;
	  }
	}
      else
	{
	  if (mat_get_lambda(dispCoef,px) <= 2.8 && flagMin == 0) {
	    pxMax=px+1;
	    flagMin=1;
	  }
	  if (mat_get_lambda(dispCoef,px) <= 5.1 && flagMax == 0) {
	    pxMin=px;
	    flagMax=1;
	  }
	}
    } else {
      if (mat_get_lambda(dispCoef,px) > 7.9 && flagMin == 0) {
	pxMin=px;
	flagMin=1;
      }
      if (mat_get_lambda(dispCoef,px) > 13.1 && flagMax == 0) {
	pxMax=px;
	flagMax=1;
      }
    }
  }
  if (flagMin == 0) {
    pxMin=genData->imgdet->list_region[0]->corner[1];
  }
  if (flagMax == 0) {
    pxMax=genData->imgdet->list_region[0]->corner[1]+genData->imgdet->list_region[0]->naxis[1]-1;
  }
  cpl_vector_delete(dispCoef);
  //test if mat_gendata is empty
  if (genData == NULL)
    {
      return NULL;
    }

  genDataSelected = mat_gendata_duplicate(genData,cpl_image_get_type(genData->imgdata->list_frame[0]->list_subwin[0]->imgreg[0]));
  if (pxMin > genDataSelected->imgdet->list_region[0]->corner[1] && pxMax < genDataSelected->imgdet->list_region[0]->corner[1]+genDataSelected->imgdet->list_region[0]->naxis[1] ) {
    for (r = 0; r < genDataSelected->imgdet->nbregion; r++) {
      genDataSelected->imgdet->list_region[r]->corner[1] = pxMin;
      genDataSelected->imgdet->list_region[r]->naxis[1] = pxMax-pxMin+1;
    }

    for (i=0;i<genDataSelected->imgdata->nbframe;i++) {
      for(j=0;j<genDataSelected->imgdata->list_frame[i]->nbsubwin;j++) {
	for(k = 0; k < genDataSelected->imgdata->list_frame[i]->list_subwin[j]->nbimgreg; k++) {
	  cpl_image *dstimg = NULL;
	  cpl_image_delete(genDataSelected->imgdata->list_frame[i]->list_subwin[j]->imgreg[k]);
	  dstimg = cpl_image_new(genDataSelected->imgdet->list_region[j]->naxis[0], 
				 genDataSelected->imgdet->list_region[j]->naxis[1],
				 cpl_image_get_type(genData->imgdata->list_frame[i]->list_subwin[j]->imgreg[k]));
	  for (y = pxMin-genData->imgdet->list_region[j]->corner[1]; y<=pxMax-genData->imgdet->list_region[j]->corner[1]; y++) {
	    for (x = 1; x <= genData->imgdet->list_region[j]->naxis[0]; x++) {
	      y1=y-(pxMin-genData->imgdet->list_region[j]->corner[1])+1;
	      cpl_image_set(dstimg, x, y1, cpl_image_get(genData->imgdata->list_frame[i]->list_subwin[j]->imgreg[k], x, y, &rejected));
	    }
	  }
	  genDataSelected->imgdata->list_frame[i]->list_subwin[j]->imgreg[k] = dstimg;
	}
      }
    }
  }

  return genDataSelected;
}


mat_gendata * mat_gendata_binning(mat_gendata *genData, int nbbin) {
  mat_gendata *genDataBin = NULL;
  int x, y, y1, rejected;
  int i,j,k,r;

  //test if mat_gendata is empty
  if (genData == NULL)
    {
      return NULL;
    }

  genDataBin = mat_gendata_duplicate(genData,cpl_image_get_type(genData->imgdata->list_frame[0]->list_subwin[0]->imgreg[0]));
  for (r = 0; r < genDataBin->imgdet->nbregion; r++)
    {
      genDataBin->imgdet->list_region[r]->corner[1] += (nbbin-1)/2;
      genDataBin->imgdet->list_region[r]->naxis[1] = genData->imgdet->list_region[r]->naxis[1]/nbbin;
    }
  for (i=0;i<genDataBin->imgdata->nbframe;i++)
    {
      for(j=0;j<genDataBin->imgdata->list_frame[i]->nbsubwin;j++)
	{
	  for(k = 0; k < genDataBin->imgdata->list_frame[i]->list_subwin[j]->nbimgreg; k++)
	    {
	      cpl_image *dstimg = NULL;
	      cpl_image_delete(genDataBin->imgdata->list_frame[i]->list_subwin[j]->imgreg[k]);
	      dstimg = cpl_image_new(genDataBin->imgdet->list_region[j]->naxis[0], 
				     genDataBin->imgdet->list_region[j]->naxis[1],
				     cpl_image_get_type(genData->imgdata->list_frame[i]->list_subwin[j]->imgreg[k]));
	      for (y = 1; y<= genDataBin->imgdet->list_region[j]->naxis[1]; y++)
		{
		  for (x = 1; x <= genDataBin->imgdet->list_region[j]->naxis[0]; x++)
		    {
		      cpl_image_set(dstimg, x, y, cpl_image_get(genData->imgdata->list_frame[i]->list_subwin[j]->imgreg[k], x, (y-1)*nbbin, &rejected));
		      for (y1=1;y1<nbbin;y1++)
			{
			  cpl_image_set(dstimg, x, y, cpl_image_get(dstimg, x, y, &rejected)+
					cpl_image_get(genData->imgdata->list_frame[i]->list_subwin[j]->imgreg[k], x, (y-1)*nbbin+y1, &rejected));
			}
		    }
		}
	      genDataBin->imgdata->list_frame[i]->list_subwin[j]->imgreg[k] = dstimg;
	    }
	}
    }
  
  return genDataBin;
}



cpl_vector *mat_vector_hampel_filter(cpl_vector *in, int k, int t)
{

  cpl_vector *out=NULL;
  cpl_vector *subvec=NULL;
  double L = 1.4826;
  int i,n;
  double med,mad;
   
  
  out = cpl_vector_duplicate(in);
  n=cpl_vector_get_size(in);
  // Make sure k is not larger than n
  k = CPL_MIN(k, n);
  
  for(i=0; i<n;i++)
    {
      if (i < k/2)
	{
	  subvec=cpl_vector_extract(in,0,k-1,1);
	  med=cpl_vector_get_median(subvec);
	  cpl_vector_subtract_scalar(subvec,med);
	  cpl_vector_multiply(subvec,subvec);
	  cpl_vector_sqrt(subvec);
	  mad=cpl_vector_get_median(subvec);
	  cpl_vector_delete(subvec);    
	}
      else if (i >= n-k/2)
	{
	  subvec=cpl_vector_extract(in,n-k,n-1,1);
	  med=cpl_vector_get_median(subvec);
	  cpl_vector_subtract_scalar(subvec,med);
	  cpl_vector_multiply(subvec,subvec);
	  cpl_vector_sqrt(subvec);
	  mad=cpl_vector_get_median(subvec);
	  cpl_vector_delete(subvec);    
	}
      else
	{
	  subvec=cpl_vector_extract(in,i-k/2,i+k/2-1,1);
	  med=cpl_vector_get_median(subvec);
	  cpl_vector_subtract_scalar(subvec,med);
	  cpl_vector_multiply(subvec,subvec);
	  cpl_vector_sqrt(subvec);
	  mad=cpl_vector_get_median(subvec);
	  cpl_vector_delete(subvec);    
	}
      if ( fabs(cpl_vector_get(in,i) - med) > t*L*mad )
	{
	  cpl_vector_set(out,i,med);
	}
    }
  return out;
}


/* -------------------------------- */
/* ---   mat_get_fname_gra (DS) --- */
/* -------------------------------- */

char *mat_get_fname_gra(int expno_mat, cpl_frameset *frameset)
{

  cpl_frameset_iterator *it=NULL;
  cpl_frame *frame=NULL;
  int expno_gra=0;
  char *fname_gra=NULL;

  /* ------------------------------------------------------------------- */
  /* Loop through the frameset to check for a corresponding Gravity file */
  /* ------------------------------------------------------------------- */
  it = cpl_frameset_iterator_new(frameset);
  frame = cpl_frameset_iterator_get(it);
  while (frame != NULL)
    {
      if  (strcmp(cpl_frame_get_tag(frame),"RMNREC") == 0)
	{
	  cpl_propertylist *header_gra = NULL;
	  header_gra=cpl_propertylist_load(cpl_frame_get_filename(frame),0);
	  if (strcmp( cpl_propertylist_get_string(header_gra,"ESO DEL FT SENSOR"), "GRAVITY")==0 &&
	      strcmp( cpl_propertylist_get_string(header_gra,"ESO DEL FT STATUS"), "ON")==0)
	    {
	      expno_gra=cpl_propertylist_get_int(header_gra,"ESO TPL EXPNO");
	      if ((expno_mat == expno_gra+1) || (expno_mat == expno_gra+2))
		{	   	     
		  fname_gra=(char *)cpl_frame_get_filename(frame);
		  cpl_msg_info(cpl_func,"Found Gravity file: %s, (EXPNO %d)\n",fname_gra,expno_gra);
		  cpl_propertylist_delete(header_gra); header_gra = NULL;
		  break;
		}
	    }
	   cpl_propertylist_delete(header_gra);
	}
      cpl_frameset_iterator_advance(it, 1);
      frame = cpl_frameset_iterator_get(it);
    }
  
  /* ----------- */
  /* Free memory */
  /* ----------- */
  cpl_frameset_iterator_delete(it); 

  
  return fname_gra;
}

/* -------------------------------- */
/* ---   mat_flag_frames_L (DS) --- */
/* -------------------------------- */

void mat_flag_frames_L(mat_gendata *gendata, char *fileGRAVITY, int debug) 
{
  cpl_table *tbl_gra=NULL;
  cpl_propertylist *header_gra=NULL;

  cpl_matrix **fluxcorR=NULL;
  cpl_matrix **fluxcorI=NULL;

  cpl_vector *mjdstart_mat=NULL;
  cpl_vector *time_gra=NULL;
  cpl_vector *time_index=NULL;
  cpl_vector *time_GD=NULL;
  cpl_vector **GDsmth=NULL;
  cpl_vector **GDder=NULL;
  cpl_vector **GDtotal=NULL;
  cpl_vector **flag_frames=NULL;
  cpl_vector *flagGD=NULL;

  int iCol_gra,nRow_gra=0;
  int index_start=0;
  int index_end=0;
  int nRow_frame=0;
  int nbframes=0;
  int ilastzero=0;
  int bin_factor=0;
  int ipos_inf=0;
  int ipos_sup=0;

  float xr=0.;
  float xi=0.;
  float yr=0.;
  float yi=0.;
  float rms_GD=0.;
  float ratio_jump=0.;
  
  double mjdstart_gra;

  FILE *fichier=NULL;
  char fnameout[80];
  char *date_mat=NULL;
  
  //cpl_init(CPL_INIT_DEFAULT);
    
  /* ------------------------------------------------------------------- */
  /* Matisse data: Get the number of frames and the associated MJD_start */
  /* ------------------------------------------------------------------- */
  nbframes=gendata->imgdata->nbframe;
  date_mat = (char *)cpl_propertylist_get_string(gendata->keywords,"DATE-OBS"); 
  if (debug == 101)
    {
      cpl_msg_info(cpl_func,"Number of frames Matisse: %d",nbframes);
    }
  mjdstart_mat=cpl_vector_new(nbframes); 
  for(int iframe_mat=0;iframe_mat<nbframes;iframe_mat++)
    {
      cpl_vector_set(mjdstart_mat,iframe_mat,gendata->imgdata->list_frame[iframe_mat]->time);
      if (debug == 101)	
	{
	  cpl_msg_info(cpl_func,"Matisse Frame Number: %d, tartyp: %s, MJDstart: %12.10f",iframe_mat+1,gendata->imgdata->list_frame[iframe_mat]->tartype,gendata->imgdata->list_frame[iframe_mat]->time);
	}
    }
 
  /* ------------------------------------- */
  /* Load GRAVITY FITS file in a cpl_table */
  /* ------------------------------------- */
  tbl_gra=cpl_table_load(fileGRAVITY,4,0); 
  header_gra=cpl_propertylist_load(fileGRAVITY,0);
  nRow_gra=cpl_table_get_nrow(tbl_gra);
  mjdstart_gra=cpl_propertylist_get_double(header_gra,"MJD-OBS");
  if (debug == 101)
    {
      cpl_msg_info(cpl_func,"Number of lines GRAVITY: %d",nRow_gra);
      cpl_msg_info(cpl_func,"GRAVITY timestart MJD= %lf\n", mjdstart_gra);
    }

  /* -------------------------------------------------- */
  /* Allocate memory of fluxcorR, fluxcorI, and GDtotal */
  /* -------------------------------------------------- */
  fluxcorR=cpl_malloc(NB_BASE*sizeof(cpl_matrix*));
  fluxcorI=cpl_malloc(NB_BASE*sizeof(cpl_matrix*));
  GDtotal=cpl_malloc(NB_BASE*sizeof(cpl_vector*)); 
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      fluxcorR[iBase]=cpl_matrix_new(nRow_gra,NB_WAVELENGTH);
      fluxcorI[iBase]=cpl_matrix_new(nRow_gra,NB_WAVELENGTH);
      GDtotal[iBase]=cpl_vector_new(nRow_gra);
    } 

  /* --------------------------- */
  /* Read COHERENCE measurements */
  /* --------------------------- */
  for(int iRow=0;iRow<nRow_gra;iRow++)
    {
      iCol_gra=0;
      for(int iWave=0;iWave<NB_WAVELENGTH;iWave++)
	{
	  iCol_gra+=4;
	  
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    {
	      cpl_matrix_set(fluxcorR[iBase],iRow,iWave,cpl_array_get_float(cpl_table_get_array(tbl_gra,"COHERENCE",iRow),iCol_gra,NULL));     
	      iCol_gra++;   
	    }
	  
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    { 
	      cpl_matrix_set(fluxcorI[iBase],iRow,iWave,cpl_array_get_float(cpl_table_get_array(tbl_gra,"COHERENCE",iRow),iCol_gra,NULL)); 
	      iCol_gra++;
	    }
	}
    }
 
  /* --------------------- */
  /* Calculate the GDtotal */
  /* --------------------- */
  for(int iRow=0;iRow<nRow_gra;iRow++)
    {
      for(int iBase=0;iBase<NB_BASE;iBase++)
	{
	  yr=0.;
	  yi=0.;
	  xr=0.;
	  xi=0.;
	  
	  for(int iWave=0;iWave<NB_WAVELENGTH-1;iWave++)
	    {
	      xr+=cpl_matrix_get(fluxcorR[iBase],iRow,iWave)*cpl_matrix_get(fluxcorR[iBase],iRow,iWave+1)+
		cpl_matrix_get(fluxcorI[iBase],iRow,iWave)*cpl_matrix_get(fluxcorI[iBase],iRow,iWave+1);
	      xi+=cpl_matrix_get(fluxcorR[iBase],iRow,iWave+1)*cpl_matrix_get(fluxcorI[iBase],iRow,iWave)-
		cpl_matrix_get(fluxcorR[iBase],iRow,iWave)*cpl_matrix_get(fluxcorI[iBase],iRow,iWave+1);
	      yr+=cpl_matrix_get(fluxcorR[iBase],iRow,iWave);	      
	      yi+=cpl_matrix_get(fluxcorI[iBase],iRow,iWave);
	      cpl_vector_set(GDtotal[iBase], iRow, atan2(xi,xr)*((lambdag*resolg)/(2*CPL_MATH_PI)));	  
	    }
	}
    }
  flagGD=cpl_vector_new(NB_BASE);
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      cpl_vector_set(flagGD,iBase,0);
      if (cpl_vector_get_stdev(GDtotal[iBase]) > 1.8)
	{
	  cpl_vector_set(flagGD,iBase,1);
	}
      cpl_msg_info(cpl_func,"Base:%d RMS_GS=%f (microns)\n",iBase+1,cpl_vector_get_stdev(GDtotal[iBase]));
    }

  
  
  /* ----------------------------------- */
  /* Free memory (fluxcorR and fluxcorI) */
  /* ----------------------------------- */
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      cpl_matrix_delete(fluxcorR[iBase]);
      cpl_matrix_delete(fluxcorI[iBase]);
    }
  cpl_free(fluxcorR);
  cpl_free(fluxcorI);
 
  /* ------------------------------------------------------------ */
  /* Get GRAVITY time vector (\micro s) and transform it into MJD */
  /* ------------------------------------------------------------ */
  time_gra=cpl_vector_new(nRow_gra);
  for(int iRow=0;iRow<nRow_gra;iRow++)
    {
      cpl_vector_set(time_gra,iRow,cpl_table_get(tbl_gra,"TIME",iRow,NULL)*1E-6/86400.);
    }
  cpl_vector_add_scalar(time_gra,mjdstart_gra);

  /* ---------------------------------------------------------------------- */
  /* Find if GRAVITY observations are available during Matisse observations */
  /* ---------------------------------------------------------------------- */ 
  time_index=cpl_vector_new(nbframes);
  flag_frames=cpl_malloc(NB_BASE*sizeof(cpl_vector*));
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      flag_frames[iBase]=cpl_vector_new(nbframes);
      cpl_vector_fill(flag_frames[iBase],0);
    }
 
  for(int iframe_mat=0;iframe_mat<nbframes;iframe_mat++)
    {
      if (cpl_vector_get(time_gra,0) > cpl_vector_get(mjdstart_mat,iframe_mat))
	{
	  /* --- Flag here frame as jump or NO_TRACKING, change Tartype --- */
	  sprintf(gendata->imgdata->list_frame[iframe_mat]->tartype,"%s","N");
	  
	  if (debug == 101)
	    {
	      cpl_msg_info(cpl_func,"No GRAVITY observations during the frame Matisse. Change tartype. Frame: %d\n\n",iframe_mat+1);
	    }
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    {
	      cpl_vector_set(flag_frames[iBase],iframe_mat,100);
	    }
	}
      else if (cpl_vector_get(time_gra,nRow_gra-1) < (cpl_vector_get(mjdstart_mat,iframe_mat)+(cpl_vector_get(mjdstart_mat,1)-cpl_vector_get(mjdstart_mat,0))))
	{
	  /* --- Flag here frame as jump or NO_TRACKING, change Tartype --- */
	  sprintf(gendata->imgdata->list_frame[iframe_mat]->tartype,"%s","N");
	  
	  if (debug == 101)
	    {
	      cpl_msg_info(cpl_func,"No GRAVITY observations during the frame Matisse. Change tartype. Frame: %d\n\n",iframe_mat+1);
	    }
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    {
	      cpl_vector_set(flag_frames[iBase],iframe_mat,100);
	    }
	} 
      
      /* ----------------------------------------- */
      /* Find indices "time_index" of GRAVITY data */
      /* ----------------------------------------- */
      else
	{ 
	  if (iframe_mat < nbframes-1)
	    {
	      index_start = cpl_vector_find(time_gra,cpl_vector_get(mjdstart_mat,iframe_mat));
	      index_end = cpl_vector_find(time_gra,cpl_vector_get(mjdstart_mat,iframe_mat+1));
	      if (debug == 101)
		{
		  cpl_msg_info(cpl_func,"Index GRAVITY starting time %d / ending time %d",index_start,index_end);
		  cpl_msg_info(cpl_func,"GRAVITY Starting time %12.10f / ending time %12.10f",cpl_vector_get(time_gra,index_start),cpl_vector_get(time_gra,index_end));
		  cpl_msg_info(cpl_func,"Matisse Starting time frame Matisse %12.10f / ending time %12.10f\n",cpl_vector_get(mjdstart_mat,iframe_mat),cpl_vector_get(mjdstart_mat,iframe_mat+1));
		}
	      cpl_vector_set(time_index,iframe_mat,index_start);
	    }
	  else if (iframe_mat == nbframes-1)
	    {
	      index_start = cpl_vector_find(time_gra,cpl_vector_get(mjdstart_mat,iframe_mat));
	      index_end = index_start + nRow_frame;
	      cpl_vector_set(time_index,iframe_mat,index_start);
	    }   

	  /* ---------------- */
	  /* Extract time MJD */
	  /* ---------------- */
	  nRow_frame=index_end-index_start+1;
	  //time_GD=cpl_vector_new(nRow_frame);	      
	  time_GD=cpl_vector_extract(time_gra,index_start,index_end,1);
	  
	  /* ------------------------------------------------------ */
	  /* To calculate GRAVITY Group Delay (GD) by frame Matisse */
	  /* ------------------------------------------------------ */
	  GDsmth=cpl_malloc(NB_BASE*sizeof(cpl_vector*));
	  GDder=cpl_malloc(NB_BASE*sizeof(cpl_vector*));
	  
	  /* ----------------------------- */
	  /* Cut in subseries GRAVITY data */
	  /* ----------------------------- */	 
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    {
	      //GDsmth[iBase]=cpl_vector_new(nRow_frame);
	      GDder[iBase]=cpl_vector_new(nRow_frame);
	    }
	  
	  /* ------------------------------------------------------------------ */
	  /* To detect jumps in GRAVITY Group Delay (GD) frame by frame Matisse */
	  /* ------------------------------------------------------------------ */
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    {

	      cpl_vector * tmp_vec =
		  cpl_vector_extract(GDtotal[iBase],index_start,index_end,1);
	      rms_GD=cpl_vector_get_stdev(tmp_vec);
	      cpl_vector_delete(tmp_vec);

	      if (rms_GD > lambdag/2)
		{
		  rms_GD=lambdag/2;
		}
	      bin_factor=round((18*pow(rms_GD,2))/(pow(lambdag,2)*pow(alphag,2)));
	
	      if (bin_factor == 1)
		{
		  bin_factor=bin_factor+1;
		}
	 
	      if (debug == 101)
		{
		  cpl_msg_info(cpl_func,"Base: %d, Frame: %d", iBase+1, iframe_mat+1); 
		  cpl_msg_info(cpl_func,"RMS of GD: %f",rms_GD);	  
		  cpl_msg_info(cpl_func,"Binning factor: %d\n",bin_factor);
		}	  
	 
	      /* -------------------------------- */
	      /* Compute low-pass filtering of GD */
	      /* -------------------------------- */

	      tmp_vec = cpl_vector_extract(GDtotal[iBase],index_start,index_end,1);
	      GDsmth[iBase]=cpl_vector_filter_lowpass_create(tmp_vec, CPL_LOWPASS_LINEAR,floor(bin_factor/2));
	      cpl_vector_delete(tmp_vec);
	      /* /\* -----------------------------*\/ */
	      /* /\* Compute the derivative of GD *\/ */
	      /* /\* -----------------------------*\/ */
	      for(int iRow_frame=0;iRow_frame<nRow_frame;iRow_frame++)
		{
		  if (iRow_frame < floor(bin_factor/2))
		    {
		      cpl_vector_set(GDder[iBase],iRow_frame,cpl_vector_get(GDsmth[iBase],iRow_frame)-cpl_vector_get(GDsmth[iBase],iRow_frame+bin_factor));
		    }
		  else if (iRow_frame > nRow_frame-floor(bin_factor/2)-1)
		    {
		      cpl_vector_set(GDder[iBase],iRow_frame,cpl_vector_get(GDsmth[iBase],iRow_frame-bin_factor)-cpl_vector_get(GDsmth[iBase],iRow_frame));
		    }
		  else
		    {
		      cpl_vector_set(GDder[iBase],iRow_frame,
				     cpl_vector_get(GDsmth[iBase],iRow_frame-floor(bin_factor/2))-cpl_vector_get(GDsmth[iBase],iRow_frame+floor(bin_factor/2)));
		    }
		}      
	    }
	
	  /* ----------------------------------- */
	  /* Test to detect jumps and bad frames */
	  /* ----------------------------------- */	  
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    {
	      for(int iRow_frame=0;iRow_frame<nRow_frame-1;iRow_frame++)
		{	  
		  if (fabs(cpl_vector_get(GDder[iBase],iRow_frame)) >= factorgL * lambdag)	
		    {
		      ipos_inf = iRow_frame-101;
		      if (ipos_inf < 0)
			{
			  ipos_inf=0;
			}
		      ipos_sup = iRow_frame+101;
		      if (ipos_sup > nRow_frame)
			{
			  ipos_sup=nRow_frame;
			}
		      
		      cpl_vector * tmp_vec1 =
			  cpl_vector_extract(GDsmth[iBase], iRow_frame+1, ipos_sup,1);
		      cpl_vector * tmp_vec2 =
			  cpl_vector_extract(GDsmth[iBase], ipos_inf, iRow_frame-1,1);

		      ratio_jump=(cpl_vector_get_mean(tmp_vec1)
				  - cpl_vector_get_mean(tmp_vec2))/lambdag;

		      cpl_vector_delete(tmp_vec1);
		      cpl_vector_delete(tmp_vec2);

		      /* -------------------- */
		      /* Test if "real jumps" */
		      /* -------------------- */
		      if (cpl_vector_get(flagGD,iBase) == 0)
			{
			  if (fabs(ratio_jump) >= threshold_ratio_jumpL)
			    {
			      /* --- IF JUMPS, change Tartype --- */
			      /* iRow_jump = index_start+iRow_frame;  */
			      cpl_vector_set(flag_frames[iBase],iframe_mat,-10);	 
			      sprintf(gendata->imgdata->list_frame[iframe_mat]->tartype,"%s","J");
			  
			      if (debug==101)
				{
				  cpl_msg_info(cpl_func,"Jump in frame: Der %f, ratio_jump %f, Base %d, Frame %d, Index %d",cpl_vector_get(GDder[iBase],iRow_frame),ratio_jump,iBase+1,iframe_mat+1,iRow_frame+1);
				}
			      break;
			    }
			  else if (fabs(ratio_jump) < threshold_ratio_jumpL)
			    {
			      /* --- IF BAD frame, change Tartype --- */
			      cpl_vector_set(flag_frames[iBase],iframe_mat,20);
			      /* sprintf(gendata->imgdata->list_frame[iframe_mat]->tartype,"%s","B"); */
			  
			      if (debug==101)
				{
				  cpl_msg_info(cpl_func,"Bad data in frame: Der %f, ratio_jump %f, Base %d, Frame %d, Index %d",cpl_vector_get(GDder[iBase],iRow_frame),ratio_jump,iBase+1,iframe_mat+1,iRow_frame+1);
				}
			    }
			}
		    }	
		}	      
	    }

	  /* ----------- */
	  /* Free memory */
	  /* ----------- */
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    {
	      cpl_vector_delete(GDsmth[iBase]);
	      cpl_vector_delete(GDder[iBase]);
	    }
	  cpl_free(GDsmth);
	  cpl_free(GDder);
	  cpl_vector_delete(time_GD); 
	}
    }
 
  /* --------------------- */
  /* Edit flag_frame table */
  /* --------------------- */
  GDsmth=cpl_malloc(NB_BASE*sizeof(cpl_vector*));
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      //GDsmth[iBase]=cpl_vector_new(nRow_gra);
      rms_GD=cpl_vector_get_stdev(GDtotal[iBase]);
      if (rms_GD > lambdag/2)
      	{
      	  rms_GD=lambdag/2;
      	}
      bin_factor=round((18*pow(rms_GD,2))/(pow(lambdag,2)*pow(alphag,2)));
      if (bin_factor == 1)
      	{
      	  bin_factor=bin_factor+1;
      	}
      
      GDsmth[iBase]=cpl_vector_filter_lowpass_create(GDtotal[iBase],CPL_LOWPASS_LINEAR,floor(bin_factor/2));
      
      for(int iframe_mat=1;iframe_mat<nbframes-1;iframe_mat++)
    	{
    	  if (cpl_vector_get(flag_frames[iBase],iframe_mat)==0)
    	    {
    	      if (cpl_vector_get(flag_frames[iBase],iframe_mat-1)==-10 || cpl_vector_get(flag_frames[iBase],iframe_mat-1)==20)
    		{
    		  index_start = cpl_vector_find(time_gra,cpl_vector_get(mjdstart_mat,iframe_mat));

    		  cpl_vector * tmp_vec1 = cpl_vector_extract(GDsmth[iBase], index_start, index_start+nRow_frame-1,1);
    		  cpl_vector * tmp_vec2 = cpl_vector_extract(GDsmth[iBase],index_start-(iframe_mat-ilastzero-1)*nRow_frame-nRow_frame,index_start-(iframe_mat-ilastzero-1)*nRow_frame-1,1);
    		  ratio_jump = (cpl_vector_get_mean(tmp_vec1)
    		      - cpl_vector_get_mean(tmp_vec2))/lambdag;
    		  cpl_vector_delete(tmp_vec1);
    		  cpl_vector_delete(tmp_vec2);

    		  if (ratio_jump >= 0.5 && ratio_jump < 1.5)
    		    {
    		      cpl_vector_set(flag_frames[iBase],iframe_mat,1);
    		      if (debug==101)
    			{
    			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J1, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
    			}
    		    }
    		  if (ratio_jump >= 1.5 && ratio_jump < 2.5)
    		    {
    		      cpl_vector_set(flag_frames[iBase],iframe_mat,2);
    		      if (debug==101)
    			{
    			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J2, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
    			}
    		    }
    		  if (ratio_jump >= 2.5 && ratio_jump < 3.5)
    		    {
    		      cpl_vector_set(flag_frames[iBase],iframe_mat,3);
    		      if (debug==101)
    			{
    			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J3, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
    			}
    		    }
    		  if (ratio_jump >= 3.5 && ratio_jump < 4.5)
    		    {
    		      cpl_vector_set(flag_frames[iBase],iframe_mat,4);
    		      if (debug==101)
    			{
    			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J4, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
    			}
    		    }
    		  if (ratio_jump <= -0.5 && ratio_jump > -1.5)
    		    {
    		      cpl_vector_set(flag_frames[iBase],iframe_mat,-1);
    		      if (debug==101)
    			{
    			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J-1, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
    			}
    		    }
    		  if (ratio_jump <= -1.5 && ratio_jump > -2.5)
    		    {
    		      cpl_vector_set(flag_frames[iBase],iframe_mat,-2);
    		      if (debug==1)
    			{
    			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J-2, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
    			}
    		    }
    		  if (ratio_jump <= -2.5 && ratio_jump > -3.5)
    		    {
    		      cpl_vector_set(flag_frames[iBase],iframe_mat,-3);
    		      if (debug==101)
    			{
    			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J-3, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
    			}
    		    }
    		  if (ratio_jump <= -3.5 && ratio_jump > -4.5)
    		    {
    		      cpl_vector_set(flag_frames[iBase],iframe_mat,-4);
    		      if (debug==101)
    			{
    			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J-4, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
    			}
    		    }
    		}
    	      ilastzero = iframe_mat;
    	    }
    	}
      ilastzero=0;
    }

  /* -------------------------------------- */
  /* Save flag_frame table as an ASCII file */
  /* -------------------------------------- */
  if (debug == 1)
    {
      sprintf(fnameout,"FLAG_%s.txt",date_mat);
      cpl_msg_info(cpl_func,"Saving %s\n",fnameout);
      fichier = fopen(fnameout,"w");
      for(int iframe_mat=0;iframe_mat<nbframes;iframe_mat++)
  	{
  	  //fprintf(fichier,"%d",iframe_mat);
  	  fprintf(fichier,"%d %12.10f",iframe_mat,cpl_vector_get(mjdstart_mat,iframe_mat));
  	  for(int iBase=0;iBase<NB_BASE;iBase++)
  	    {
  	      fprintf(fichier,"%4.0f",cpl_vector_get(flag_frames[iBase],iframe_mat));
  	    }
  	  fprintf(fichier,"\n");
  	}
      fclose(fichier);
    }
  
  /* ----------- */
  /* Free memory */
  /* ----------- */
  cpl_table_delete(tbl_gra);
  cpl_propertylist_delete(header_gra);
  cpl_vector_delete(mjdstart_mat);
  cpl_vector_delete(time_gra);
  cpl_vector_delete(time_index);
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      cpl_vector_delete(GDtotal[iBase]);
      cpl_vector_delete(flag_frames[iBase]);
      cpl_vector_delete(GDsmth[iBase]);
    }
  cpl_free(GDtotal);
  cpl_free(flag_frames);
  cpl_free(GDsmth);
  cpl_vector_delete(flagGD);
}


/* -------------------------------- */
/* ---   mat_flag_frames_N (DS) --- */
/* -------------------------------- */

void mat_flag_frames_N(mat_gendata *gendata, char *fileGRAVITY, int debug) 
{
  cpl_table *tbl_gra=NULL;
  cpl_propertylist *header_gra=NULL;

  cpl_matrix **fluxcorR=NULL;
  cpl_matrix **fluxcorI=NULL;

  cpl_vector *mjdstart_mat=NULL;
  cpl_vector *time_gra=NULL;
  cpl_vector *time_index=NULL;
  cpl_vector *time_GD=NULL;
  cpl_vector **GDsmth=NULL;
  cpl_vector **GDder=NULL;
  cpl_vector **GDtotal=NULL;
  cpl_vector *step_phase=NULL;
  cpl_vector **flag_frames=NULL;
  cpl_vector *flagGD=NULL;

  int iCol_gra,nRow_gra;
  int index_start=0;
  int index_end=0;
  int iframe_mat_start=0;
  int iframe_mat_end=0;
  int nRow_frame=0;
  int nbframes;
  int nbcycles;
  //int iRow_jump=0;
  int ilastzero=0;
  int counter2=0;
  int ipos_inf=0;
  int ipos_sup=0;
  int bin_factor;
  int maxstep;
  int step_nb=2;

  float xr;
  float xi;
  float yr;
  float yi;
  float rms_GD;
  float ratio_jump=0;
 
  double mjdstart_gra;

  FILE *fichier=NULL;
  char fnameout[80];	
  char *date_mat=NULL;
  
  cpl_init(CPL_INIT_DEFAULT);
  
  /* ------------------------------------------------------------------- */
  /* Matisse data: Get the number of frames and the associated MJD_start */
  /* ------------------------------------------------------------------- */
  nbframes=gendata->imgdata->nbframe;
  maxstep=gendata->imgdata->maxstep;
  date_mat = (char *)cpl_propertylist_get_string(gendata->keywords,"DATE-OBS");
  if (debug == 101)
    {
      cpl_msg_info(cpl_func,"Number of frames Matisse: %d",nbframes);
    }
  mjdstart_mat=cpl_vector_new(nbframes); 
  step_phase=cpl_vector_new(nbframes);
  for(int iframe_mat=0;iframe_mat<nbframes;iframe_mat++)
    {
      cpl_vector_set(mjdstart_mat,iframe_mat,gendata->imgdata->list_frame[iframe_mat]->time);
      cpl_vector_set(step_phase,iframe_mat,gendata->imgdata->list_frame[iframe_mat]->stepphase);
      if (debug == 101)	
	{
	  cpl_msg_info(cpl_func,"Matisse Frame Number: %d, tartyp: %s, step_phase: %d, MJDstart: %12.10f",iframe_mat+1,gendata->imgdata->list_frame[iframe_mat]->tartype,gendata->imgdata->list_frame[iframe_mat]->stepphase,gendata->imgdata->list_frame[iframe_mat]->time);
	}
    }
  if ((int)cpl_vector_get_max(step_phase) != maxstep)
    {
      maxstep=(int)cpl_vector_get_max(step_phase);
    }

  /* ------------------------------------- */
  /* Load GRAVITY FITS file in a cpl_table */
  /* ------------------------------------- */
  tbl_gra=cpl_table_load(fileGRAVITY,4,0);
  header_gra=cpl_propertylist_load(fileGRAVITY,0);
  nRow_gra=cpl_table_get_nrow(tbl_gra);
  mjdstart_gra=cpl_propertylist_get_double(header_gra,"MJD-OBS");
  if (debug == 101)
    {
      cpl_msg_info(cpl_func,"Number of lines GRAVITY: %d",nRow_gra);
      cpl_msg_info(cpl_func,"GRAVITY timestart MJD= %lf\n", mjdstart_gra);
    }
  
  /* -------------------------------------------------- */
  /* Allocate memory of fluxcorR, fluxcorI, and GDtotal */
  /* -------------------------------------------------- */
  fluxcorR=cpl_malloc(NB_BASE*sizeof(cpl_matrix*));
  fluxcorI=cpl_malloc(NB_BASE*sizeof(cpl_matrix*));
  GDtotal=cpl_malloc(NB_BASE*sizeof(cpl_vector*));
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      fluxcorR[iBase]=cpl_matrix_new(nRow_gra,NB_WAVELENGTH);
      fluxcorI[iBase]=cpl_matrix_new(nRow_gra,NB_WAVELENGTH);
      GDtotal[iBase]=cpl_vector_new(nRow_gra);
    } 

  /* --------------------------- */
  /* Read COHERENCE measurements */
  /* --------------------------- */
  for(int iRow=0;iRow<nRow_gra;iRow++)
    {
      iCol_gra=0;
      for(int iWave=0;iWave<NB_WAVELENGTH;iWave++)
	{
	  iCol_gra+=4;
	  
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    {
	      cpl_matrix_set(fluxcorR[iBase],iRow,iWave,
			     cpl_array_get_float(cpl_table_get_array(tbl_gra,"COHERENCE",iRow),iCol_gra,NULL));     
	      iCol_gra++;   
	    }
	  
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    { 
	      cpl_matrix_set(fluxcorI[iBase],iRow,iWave,
			     cpl_array_get_float(cpl_table_get_array(tbl_gra,"COHERENCE",iRow),iCol_gra,NULL)); 
	      iCol_gra++;
	    }
	}
    }

  /* --------------------- */
  /* Calculate the GDtotal */
  /* --------------------- */
  for(int iRow=0;iRow<nRow_gra;iRow++)
    {
      for(int iBase=0;iBase<NB_BASE;iBase++)
	{
	  yr=0.;
	  yi=0.;
	  xr=0.;
	  xi=0.;
	  
	  for(int iWave=0;iWave<NB_WAVELENGTH-1;iWave++)
	    {
	      xr+=cpl_matrix_get(fluxcorR[iBase],iRow,iWave)*cpl_matrix_get(fluxcorR[iBase],iRow,iWave+1)+
		cpl_matrix_get(fluxcorI[iBase],iRow,iWave)*cpl_matrix_get(fluxcorI[iBase],iRow,iWave+1);
	      xi+=cpl_matrix_get(fluxcorR[iBase],iRow,iWave+1)*cpl_matrix_get(fluxcorI[iBase],iRow,iWave)-
		cpl_matrix_get(fluxcorR[iBase],iRow,iWave)*cpl_matrix_get(fluxcorI[iBase],iRow,iWave+1);
	      yr+=cpl_matrix_get(fluxcorR[iBase],iRow,iWave);	      
	      yi+=cpl_matrix_get(fluxcorI[iBase],iRow,iWave);
	      cpl_vector_set(GDtotal[iBase], iRow, atan2(xi,xr)*((lambdag*resolg)/(2*CPL_MATH_PI)));	  
	    }
	}
    }
  flagGD=cpl_vector_new(NB_BASE);
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      cpl_vector_set(flagGD,iBase,0);
      if (cpl_vector_get_stdev(GDtotal[iBase]) > 1.8)
	{
	  cpl_vector_set(flagGD,iBase,1);
	}
      cpl_msg_info(cpl_func,"Base:%d RMS_GS=%f (microns)\n",iBase+1,cpl_vector_get_stdev(GDtotal[iBase]));
    }


  /* ------------------------------------------------------------ */
  /* Get GRAVITY time vector (\micro s) and transform it into MJD */
  /* ------------------------------------------------------------ */
  time_gra=cpl_vector_new(nRow_gra);
  for(int iRow=0;iRow<nRow_gra;iRow++)
    {
      cpl_vector_set(time_gra,iRow,cpl_table_get(tbl_gra,"TIME",iRow,NULL)*1E-6/86400.);
    }
  cpl_vector_add_scalar(time_gra,mjdstart_gra);

  /* /\* ------------------------------------- *\/ */
  /* /\* Calculate GD and PA for cophasing K4N *\/ */
  /* /\* ------------------------------------- *\/ */
  /* mat_compute_GD_PA_K4N(time_gra,fluxcorR,fluxcorI,cpl_propertylist_get_int(gendata->keywords,"ESO TPL EXPNO")); */

  /* ----------------------------------- */
  /* Free memory (fluxcorR and fluxcorI) */
  /* ----------------------------------- */
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      cpl_matrix_delete(fluxcorR[iBase]);
      cpl_matrix_delete(fluxcorI[iBase]);
    }
  cpl_free(fluxcorR);
  cpl_free(fluxcorI);

  /* ---------------------------------------------------------------------- */
  /* Find if GRAVITY observations are available during Matisse observations */
  /* ---------------------------------------------------------------------- */
  time_index=cpl_vector_new(nbframes);
  flag_frames=cpl_malloc(NB_BASE*sizeof(cpl_vector*));
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      flag_frames[iBase]=cpl_vector_new(nbframes);
      cpl_vector_fill(flag_frames[iBase],0);
    }

  for(int iframe_mat=0;iframe_mat<nbframes;iframe_mat++)
    {
      if (cpl_vector_get(time_gra,0) > cpl_vector_get(mjdstart_mat,iframe_mat))
	{
	  /* --- NO_TRACKING, change Tartype --- */
	  sprintf(gendata->imgdata->list_frame[iframe_mat]->tartype,"%s","N");
	  
	  if (debug == 101)
	    {
	      cpl_msg_info(cpl_func,"No GRAVITY observations during the frame Matisse. Change tartype. Frame: %d", iframe_mat+1);
	    }
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    {
	      cpl_vector_set(flag_frames[iBase],iframe_mat,100);
	    }
	  iframe_mat_start++; 
	}
      else if (cpl_vector_get(time_gra,nRow_gra-1) < (cpl_vector_get(mjdstart_mat,iframe_mat)+(cpl_vector_get(mjdstart_mat,1)-cpl_vector_get(mjdstart_mat,0))))
	{
	  /* --- NO_TRACKING, change Tartype --- */
	  sprintf(gendata->imgdata->list_frame[iframe_mat]->tartype,"%s","N");
	  
	  if (debug == 101)
	    {
	      cpl_msg_info(cpl_func,"No GRAVITY observations during the frame Matisse. Change tartype. Frame: %d", iframe_mat+1);
	    }
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    {
	      cpl_vector_set(flag_frames[iBase],iframe_mat,100);
	    }
	  counter2++;
	}
    }

  /* ------------------------- */
  /* Test start stepping_phase */
  /* ------------------------- */
  if (cpl_vector_get(step_phase,iframe_mat_start) != 1)
    {
      while (cpl_vector_get(step_phase,iframe_mat_start) != 1)
  	{
  	  /* --- 1st step_phase(s) different from 1, change Tartype --- */
  	  sprintf(gendata->imgdata->list_frame[iframe_mat_start]->tartype,"%s","Z");
	  
  	  if (debug == 101)
  	    {
	      cpl_msg_info(cpl_func,"Change tartype: Uncomplete cycle at start: %d, Step: %f",iframe_mat_start,cpl_vector_get(step_phase,iframe_mat_start));
  	    }
  	  for(int iBase=0;iBase<NB_BASE;iBase++)
  	    {
  	      cpl_vector_set(flag_frames[iBase],iframe_mat_start,200);
  	    }
  	  iframe_mat_start++;
  	}
    }
  
  
  /* ----------------------- */
  /* Test end stepping_phase */
  /* ----------------------- */
  iframe_mat_end=nbframes-1-counter2;
  if (cpl_vector_get(step_phase,nbframes-1) != maxstep)
    {
      
      while (cpl_vector_get(step_phase,iframe_mat_end) != maxstep)
  	{
  	  /* --- Last step_phase(s) different from maxstep, change Tartype --- */
  	  sprintf(gendata->imgdata->list_frame[iframe_mat_end]->tartype,"%s","Z");
	  
  	  if (debug == 101)
  	    {
  	      cpl_msg_info(cpl_func,"Change tartype: Uncomplete cycle at end: %d, Step: %f",iframe_mat_end,cpl_vector_get(step_phase,iframe_mat_end));
  	    }
  	  for(int iBase=0;iBase<NB_BASE;iBase++)
  	    {
  	      cpl_vector_set(flag_frames[iBase],iframe_mat_end,200);
  	    }
  	  iframe_mat_end--;
  	}
    }
 
  /* -------------------------------------- */
  /* Start loop on blocks of stepping phase */
  /* -------------------------------------- */
  maxstep=maxstep*step_nb;
  nbcycles = (iframe_mat_end-iframe_mat_start+1)/maxstep;
  for(int icycle_mat=0;icycle_mat<nbcycles;icycle_mat++)
    {
      if (debug == 101)
	{
	  cpl_msg_info(cpl_func,"Available GRAVITY observations during the frame Matisse: %d", iframe_mat_start);
	}

      /* -------------------------------------------------------------------------------------------- */
      /* Find indices "start" and "end" of GRAVITY data corresponding to each block of stepping phase */
      /* -------------------------------------------------------------------------------------------- */
      if (icycle_mat < nbcycles-1)
	{	    
	  index_start = cpl_vector_find(time_gra,cpl_vector_get(mjdstart_mat,iframe_mat_start));
	  index_end = cpl_vector_find(time_gra,cpl_vector_get(mjdstart_mat,iframe_mat_start+maxstep))-1;	 

	  if (debug==101)
	    {
	      cpl_msg_info(cpl_func,"Index GRAVITY starting index %d / ending index %d",index_start,index_end);
	      cpl_msg_info(cpl_func,"GRAVITY Starting time %12.10f / ending time %12.10f",cpl_vector_get(time_gra,index_start),cpl_vector_get(time_gra,index_end));
	      cpl_msg_info(cpl_func,"Matisse Starting time frame Matisse %12.10f / ending time %12.10f\n",cpl_vector_get(mjdstart_mat,iframe_mat_start),cpl_vector_get(mjdstart_mat,iframe_mat_start+maxstep+1));
	    }

	  cpl_vector_set(time_index,iframe_mat_start,index_start);
	  iframe_mat_start = iframe_mat_start+maxstep;
	}
      else if (icycle_mat == nbcycles-1)
	{	 
	  index_start = cpl_vector_find(time_gra,cpl_vector_get(mjdstart_mat,iframe_mat_start));	 
	  index_end = index_start + nRow_frame-1;
	  cpl_vector_set(time_index,iframe_mat_start,index_start);
	}
	  	  
      /* ---------------- */
      /* Extract time MJD */
      /* ---------------- */
      nRow_frame=index_end-index_start+1;
      //time_GD=cpl_vector_new(nRow_frame);	      
      time_GD=cpl_vector_extract(time_gra,index_start,index_end,1);
      
      /* ------------------------------------------------------ */
      /* To calculate GRAVITY Group Delay (GD) by frame Matisse */
      /* ------------------------------------------------------ */      
      GDsmth=cpl_malloc(NB_BASE*sizeof(cpl_vector*));
      GDder=cpl_malloc(NB_BASE*sizeof(cpl_vector*));
     
      /* ----------------------------- */
      /* Cut in subseries GRAVITY data */
      /* ----------------------------- */
      for(int iBase=0;iBase<NB_BASE;iBase++)
	{
	  //GDsmth[iBase]=cpl_vector_new(nRow_frame);
	  GDder[iBase]=cpl_vector_new(nRow_frame);
	}
    		       	  
      /* --------------------------------------------------------------------------- */
      /* To detect jumps in GRAVITY Group Delay (GD) by Matisse stepping phase block */
      /* --------------------------------------------------------------------------- */
      for(int iBase=0;iBase<NB_BASE;iBase++)
	{
	  rms_GD=cpl_vector_get_stdev(cpl_vector_extract(GDtotal[iBase],index_start,index_end,1));
	  if (rms_GD > lambdag/2)
	    {
	      rms_GD=lambdag/2;
	    }
	  bin_factor=round((18*pow(rms_GD,2))/(pow(lambdag,2)*pow(alphag,2)));
	
	  if (bin_factor == 1)
	    {
	      bin_factor=bin_factor+1;
	    }
	 
	  if (debug == 101)
	    {
	      cpl_msg_info(cpl_func,"Base: %d, Cycle: %d", iBase+1, icycle_mat+1); 
	      cpl_msg_info(cpl_func,"RMS of GD: %f",rms_GD);	  
	      cpl_msg_info(cpl_func,"Binning factor: %d\n",bin_factor);
	    }

	  /* -------------------------------- */
	  /* Compute low-pass filtering of GD */
	  /* -------------------------------- */
	  GDsmth[iBase]=cpl_vector_filter_lowpass_create(cpl_vector_extract(GDtotal[iBase],index_start,index_end,1),CPL_LOWPASS_LINEAR,floor(bin_factor/2));

	  /* -----------------------------*/
	  /* Compute the derivative of GD */
	  /* -----------------------------*/
	  for(int iRow_frame=0;iRow_frame<nRow_frame;iRow_frame++)
	    {
	      if (iRow_frame < floor(bin_factor/2))
		{
		  cpl_vector_set(GDder[iBase],iRow_frame,cpl_vector_get(GDsmth[iBase],iRow_frame)-cpl_vector_get(GDsmth[iBase],iRow_frame+bin_factor));
		}
	      else if (iRow_frame > nRow_frame-floor(bin_factor/2)-1)
		{
		  cpl_vector_set(GDder[iBase],iRow_frame,cpl_vector_get(GDsmth[iBase],iRow_frame-bin_factor)-cpl_vector_get(GDsmth[iBase],iRow_frame));
		}
	      else
		{
		  cpl_vector_set(GDder[iBase],iRow_frame,
				 cpl_vector_get(GDsmth[iBase],iRow_frame-floor(bin_factor/2))-cpl_vector_get(GDsmth[iBase],iRow_frame+floor(bin_factor/2)));
		}
	    }      
	}

      /* ----------------------------------- */
      /* Test to detect jumps and bad frames */
      /* ----------------------------------- */
      for(int iBase=0;iBase<NB_BASE;iBase++)
	{
	  for(int iRow_frame=0;iRow_frame<nRow_frame-1;iRow_frame++)
	    {	    
	      if (fabs(cpl_vector_get(GDder[iBase],iRow_frame)) >= factorgN * lambdag)	
		{	     		  
		  ipos_inf = iRow_frame-101;
		  if (ipos_inf < 0)
		    {
		      ipos_inf=0;
		    }
		  ipos_sup = iRow_frame+101;
		  if (ipos_sup > nRow_frame)
		    {
		      ipos_sup=nRow_frame;
		    }
		 
		  ratio_jump=(cpl_vector_get_mean(cpl_vector_extract(GDsmth[iBase], iRow_frame+1, ipos_sup,1))
		  		  - cpl_vector_get_mean(cpl_vector_extract(GDsmth[iBase], ipos_inf, iRow_frame-1,1)))/lambdag;

		  
		  /* -------------------- */
		  /* Test if "real jumps" */
		  /* -------------------- */
		  if (cpl_vector_get(flagGD,iBase) == 0)
		    {
		      if (fabs(ratio_jump) >= threshold_ratio_jumpN)
			{
			  /* --- IF JUMPS, change Tartype for all the frames in one stepping-phase cycle --- */ 
			  //iRow_jump=index_start+iRow_frame;
			  if ((float)(iRow_frame+1+2)/(float)nRow_frame < 0.5)	     
			    {
			      for (int istep=0;istep<maxstep/step_nb;istep++)
				{
				  cpl_vector_set(flag_frames[iBase],iframe_mat_start-maxstep+istep,-10);
				  sprintf(gendata->imgdata->list_frame[iframe_mat_start-maxstep+istep]->tartype,"%s","J"); 
				}
			    }
		      
			  else if ((float)(iRow_frame+1+2)/(float)nRow_frame > 0.5)
			    {
			      for (int istep=maxstep/step_nb;istep<maxstep;istep++)
				{
				  cpl_vector_set(flag_frames[iBase],iframe_mat_start-maxstep+istep,-10);
				  sprintf(gendata->imgdata->list_frame[iframe_mat_start-maxstep+istep]->tartype,"%s","J"); 
				}
			    }		     
		      
			  if (debug==101)
			    {
			      cpl_msg_info(cpl_func,"Jump in stepped-frame: Der %f, ratio_jump %f, Base %d, Cycle %d, Index %d",cpl_vector_get(GDder[iBase],iRow_frame),ratio_jump,iBase+1,icycle_mat+1,iRow_frame+1);
			    }
			  //break;
			}
		      else if (fabs(ratio_jump) < threshold_ratio_jumpN)
			{
			  /* --- IF BAD frame, change Tartype for all the frames in one stepping-phase cycle --- */
			  //iRow_jump=index_start+iRow_frame; 
			  if ((float)(iRow_frame+1+2)/(float)nRow_frame < 0.5)	     
			    {
			      for (int istep=0;istep<maxstep/step_nb;istep++)
				{
				  cpl_vector_set(flag_frames[iBase],iframe_mat_start-maxstep+istep,20);
				  /* sprintf(gendata->imgdata->list_frame[iframe_mat_start-maxstep+istep]->tartype,"%s","B");	       */
				}
			    }
		      
			  else if ((float)(iRow_frame+1+2)/(float)nRow_frame > 0.5)
			    {
			      for (int istep=maxstep/step_nb;istep<maxstep;istep++)
				{
				  cpl_vector_set(flag_frames[iBase],iframe_mat_start-maxstep+istep,20);
				  /* sprintf(gendata->imgdata->list_frame[iframe_mat_start-maxstep+istep]->tartype,"%s","B"); */
				}
			    }

			  if (debug==101)
			    {
			      cpl_msg_info(cpl_func,"Bad data in stepped-frame: Der %f, ratio_jump %f, Base %d, Cycle %d, Index %d",cpl_vector_get(GDder[iBase],iRow_frame),ratio_jump,iBase+1,icycle_mat+1,iRow_frame+1);
			    } 
			}
		    }
		}	    
	    }	
	}

      /* ----------- */
      /* Free memory */
      /* ----------- */
      for(int iBase=0;iBase<NB_BASE;iBase++)
	{
	  cpl_vector_delete(GDsmth[iBase]);
	  cpl_vector_delete(GDder[iBase]);
	}
      cpl_free(GDsmth);
      cpl_free(GDder);
      cpl_vector_delete(time_GD);        
    }
	   
  /* --------------------- */
  /* Edit flag_frame table */
  /* --------------------- */
  maxstep=floor(maxstep/step_nb);
  nRow_frame=floor(nRow_frame/step_nb);
  
  GDsmth=cpl_malloc(NB_BASE*sizeof(cpl_vector*));
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      //GDsmth[iBase]=cpl_vector_new(nRow_gra);
      rms_GD=cpl_vector_get_stdev(GDtotal[iBase]);
      if (rms_GD > lambdag/2)
  	{
  	  rms_GD=lambdag/2;
  	}
      bin_factor=round((18*pow(rms_GD,2))/(pow(lambdag,2)*pow(alphag,2)));
      if (bin_factor == 1)
  	{
  	  bin_factor=bin_factor+1;
  	}
      
      GDsmth[iBase]=cpl_vector_filter_lowpass_create(GDtotal[iBase],CPL_LOWPASS_LINEAR,floor(bin_factor/2));
      
      for(int iframe_mat=1;iframe_mat<nbframes-1;iframe_mat++)
  	{
  	  if (cpl_vector_get(flag_frames[iBase],iframe_mat)==0)
  	    {
  	      if (cpl_vector_get(flag_frames[iBase],iframe_mat-1)==-10 || cpl_vector_get(flag_frames[iBase],iframe_mat-1)==20)
  		{
  		  index_start = cpl_vector_find(time_gra,cpl_vector_get(mjdstart_mat,iframe_mat));

		  ratio_jump = (cpl_vector_get_median(cpl_vector_extract(GDsmth[iBase], index_start, index_start+nRow_frame-1,1))
				- cpl_vector_get_median(cpl_vector_extract(GDsmth[iBase],index_start-(iframe_mat-ilastzero-1)/maxstep*nRow_frame-nRow_frame,index_start-(iframe_mat-ilastzero-1)/maxstep*nRow_frame-1,1)))/lambdag;
		 
		  
  		  if (ratio_jump >= 0.5 && ratio_jump < 1.5)
  		    {
  		      cpl_vector_set(flag_frames[iBase],iframe_mat,1);
  		      if (debug==101)
  			{
  			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J1, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
  			}
  		    }
  		  if (ratio_jump >= 1.5 && ratio_jump < 2.5)
  		    {
  		      cpl_vector_set(flag_frames[iBase],iframe_mat,2);
  		      if (debug==101)
  			{
  			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J2, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
  			}
  		    }
  		  if (ratio_jump >= 2.5 && ratio_jump < 3.5)
  		    {
  		      cpl_vector_set(flag_frames[iBase],iframe_mat,3);
  		      if (debug==101)
  			{
  			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J3, Base %d, Frame %d\n",ratio_jump,iBase+1,iframe_mat+1);
  			}
  		    }
  		  if (ratio_jump >= 3.5 && ratio_jump < 4.5)
  		    {
  		      cpl_vector_set(flag_frames[iBase],iframe_mat,4);
  		      if (debug==101)
  			{
  			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J4, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
  			}
  		    }
  		  if (ratio_jump <= -0.5 && ratio_jump > -1.5)
  		    {
  		      cpl_vector_set(flag_frames[iBase],iframe_mat,-1);
  		      if (debug==101)
  			{
  			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J-1, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
  			}
  		    }
  		  if (ratio_jump <= -1.5 && ratio_jump > -2.5)
  		    {
  		      cpl_vector_set(flag_frames[iBase],iframe_mat,-2);
  		      if (debug==101)
  			{
  			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J-2, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
  			}
  		    }
  		  if (ratio_jump <= -2.5 && ratio_jump > -3.5)
  		    {
  		      cpl_vector_set(flag_frames[iBase],iframe_mat,-3);
  		      if (debug==101)
  			{
  			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J-3, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
  			}
  		    }
  		  if (ratio_jump <= -3.5 && ratio_jump > -4.5)
  		    {
  		      cpl_vector_set(flag_frames[iBase],iframe_mat,-4);
  		      if (debug==101)
  			{
  			  cpl_msg_info(cpl_func,"ratio_jump: %f, flag: J-4, Base %d, Frame %d",ratio_jump,iBase+1,iframe_mat+1);
  			}
  		    }
  		}
  	      ilastzero = iframe_mat;
  	    }
  	}
      ilastzero=0;
    }

  /* -------------------------------------- */
  /* Save flag_frame table as an ASCII file */
  /* -------------------------------------- */
  if (debug == 1)
    {
      sprintf(fnameout,"FLAG_%s.txt",date_mat);
      fichier = fopen(fnameout,"w");
      for(int iframe_mat=0;iframe_mat<nbframes;iframe_mat++)
	{
	  //fprintf(fichier,"%d",iframe_mat);
	  //fprintf(fichier,"%d %12.10f %f",iframe_mat,cpl_vector_get(mjdstart_mat,iframe_mat),cpl_vector_get(step_phase,iframe_mat));
	  fprintf(fichier,"%d %12.10f",iframe_mat,cpl_vector_get(mjdstart_mat,iframe_mat));
	  for(int iBase=0;iBase<NB_BASE;iBase++)
	    {
	      fprintf(fichier,"%4.0f",cpl_vector_get(flag_frames[iBase],iframe_mat));
	    }
	  fprintf(fichier,"\n");
	}
      fclose(fichier);
    }
  
  /* ----------- */
  /* Free Memory */
  /* ----------- */
  cpl_table_delete(tbl_gra);
  cpl_propertylist_delete(header_gra);
  cpl_vector_delete(mjdstart_mat);
  cpl_vector_delete(time_gra);
  cpl_vector_delete(step_phase);
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      cpl_vector_delete(GDtotal[iBase]);
      cpl_vector_delete(flag_frames[iBase]);
       cpl_vector_delete(GDsmth[iBase]);
    }
  cpl_free(GDtotal);
  cpl_free(flag_frames);
  cpl_free(GDsmth);
  cpl_vector_delete(flagGD);
}

 
void mat_compute_GD_PA_K4N(cpl_vector *time_gra, cpl_vector *time_mt, cpl_matrix **fluxcorR,
			   cpl_matrix **fluxcorI, int bcdmode, cpl_matrix *gdestimated, cpl_matrix *paestimated)
{
  cpl_matrix **fluxcorRsmooth=NULL;
  cpl_matrix **fluxcorIsmooth=NULL;
  cpl_matrix **fluxcorRsmoothB=NULL;
  cpl_matrix **fluxcorIsmoothB=NULL;
  cpl_vector **gd=NULL;
  cpl_vector **gd_copy=NULL;
  cpl_matrix *pdc=NULL;
  cpl_matrix *pac=NULL;
  cpl_matrix *pac_copy=NULL;
  cpl_matrix *Sigma=NULL;
  cpl_matrix *papredicted=NULL;
  cpl_matrix *pdpredicted=NULL;
  cpl_matrix *gdpredicted=NULL;
  cpl_vector *vec_in,*vec_out;
  int size_kernel=0;
  int nRow_gra=0;
  double fooR=0.;
  double fooI=0.;
  cpl_vector *vecfooR=NULL;
  cpl_vector *vecfooI=NULL;
  double factor=0.;
  double medR=0.;
  double medI=0.;
  FILE *fichier=NULL;
  char fnameout[80];
  double lambda_gra[4]={(2.07767+2.06029)/2.,
			(2.17257+2.15652)/2.,
			(2.27180+2.26070)/2.,
			(2.35531+2.35021)/2.};
  // OUT-OUT, IN-OUT OUT-IN, IN-IN
  int indexBasefromGV2MT[4][6]={{0,5,3,1,4,2},{0,5,1,3,2,4},{0,5,4,2,3,1},{0,5,2,4,1,3}};
  int signfromGV2MT[4][6]={{1,1,1,1,1,1},{-1,1,1,1,1,1},{1,-1,1,1,1,1},{-1,-1,1,1,1,1}};
  double gammaK=-1.1E24;
  double gammaN=1./(10.5*(5.035E-24-7.756E-24)/4.);
  double indexN=6.491E-24;
  double indexK=9.029E-24;
  double waveN0=10.5;
 
  cpl_msg_info(cpl_func,"Computing GD and PA for cophasing K4N");
  nRow_gra=cpl_matrix_get_nrow(fluxcorR[0]);
  factor=lambdag/(2.*CPL_MATH_PI);

  // Computing Phase Delay
  pdc=cpl_matrix_new(nRow_gra,NB_BASE);
  vecfooR=cpl_vector_new(nRow_gra);
  vecfooI=cpl_vector_new(nRow_gra);
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
     for(int iFrame=0;iFrame<nRow_gra;iFrame++)
  	{
  	  cpl_vector_set(vecfooR,iFrame,0.);
  	  cpl_vector_set(vecfooI,iFrame,0.);
  	  for(int iWave=0;iWave<NB_WAVELENGTH;iWave++)
  	    {
  	      fooR=cpl_vector_get(vecfooR,iFrame);
  	      cpl_vector_set(vecfooR,iFrame,cpl_matrix_get(fluxcorR[iBase],iFrame,iWave)+fooR);
  	      fooI=cpl_vector_get(vecfooI,iFrame);
  	      cpl_vector_set(vecfooI,iFrame,cpl_matrix_get(fluxcorI[iBase],iFrame,iWave)+fooI);
  	    }
  	}
     medR=cpl_vector_get_median(vecfooR);
     medI=cpl_vector_get_median(vecfooI);
     for(int iFrame=0;iFrame<nRow_gra;iFrame++)
     	{
     	  fooR=cpl_vector_get(vecfooR,iFrame)*medR+cpl_vector_get(vecfooI,iFrame)*medI;
     	  fooI=cpl_vector_get(vecfooI,iFrame)*medR-cpl_vector_get(vecfooR,iFrame)*medI;
     	  cpl_matrix_set(pdc,iFrame,iBase,atan2(fooI,fooR)*factor);
     	}
    }
  cpl_vector_delete(vecfooR);
  cpl_vector_delete(vecfooI);
  
  // Smoothing GRAVITY Correlated Flux corrected from PD
  size_kernel=40;
  fluxcorRsmooth=cpl_malloc(NB_BASE*sizeof(cpl_matrix*));
  fluxcorIsmooth=cpl_malloc(NB_BASE*sizeof(cpl_matrix*));
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      fluxcorRsmooth[iBase]=cpl_matrix_new(nRow_gra,NB_WAVELENGTH);
      fluxcorIsmooth[iBase]=cpl_matrix_new(nRow_gra,NB_WAVELENGTH);
    } 
  vec_in=cpl_vector_new(nRow_gra);
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      for(int iWave=0;iWave<NB_WAVELENGTH;iWave++)
	{
	  for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	    {
	      fooR=cpl_matrix_get(fluxcorR[iBase],iFrame,iWave)*cos(cpl_matrix_get(pdc,iFrame,iBase)/factor)+
		cpl_matrix_get(fluxcorI[iBase],iFrame,iWave)*sin(cpl_matrix_get(pdc,iFrame,iBase)/factor);
	      cpl_vector_set(vec_in,iFrame,fooR);
	    }
	  vec_out=cpl_vector_filter_median_create(vec_in,size_kernel/2);
	  for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	    {
	      cpl_matrix_set(fluxcorRsmooth[iBase],iFrame,iWave,cpl_vector_get(vec_out,iFrame));
	    }
	  cpl_vector_delete(vec_out);
	  for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	    {
	      fooI=cpl_matrix_get(fluxcorI[iBase],iFrame,iWave)*cos(cpl_matrix_get(pdc,iFrame,iBase)/factor)-
		cpl_matrix_get(fluxcorR[iBase],iFrame,iWave)*sin(cpl_matrix_get(pdc,iFrame,iBase)/factor);
	      cpl_vector_set(vec_in,iFrame,fooI);
	    }
	  vec_out=cpl_vector_filter_median_create(vec_in,size_kernel/2);
	  for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	    {
	      cpl_matrix_set(fluxcorIsmooth[iBase],iFrame,iWave,cpl_vector_get(vec_out,iFrame));
	    }
	  cpl_vector_delete(vec_out);
	}
    }

  // Computing Group Delay
  gd=cpl_malloc(NB_BASE*sizeof(cpl_vector*));
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      gd[iBase]=cpl_vector_new(nRow_gra);
    } 
  for(int iRow=0;iRow<nRow_gra;iRow++)
    {
      for(int iBase=0;iBase<NB_BASE;iBase++)
	{
	  fooR=0.;
	  fooI=0.;
	  for(int iWave=0;iWave<NB_WAVELENGTH-1;iWave++)
	    {
	      fooR+=cpl_matrix_get(fluxcorRsmooth[iBase],iRow,iWave)*cpl_matrix_get(fluxcorRsmooth[iBase],iRow,iWave+1)+
		cpl_matrix_get(fluxcorIsmooth[iBase],iRow,iWave)*cpl_matrix_get(fluxcorIsmooth[iBase],iRow,iWave+1);
	      fooI+=cpl_matrix_get(fluxcorRsmooth[iBase],iRow,iWave+1)*cpl_matrix_get(fluxcorIsmooth[iBase],iRow,iWave)-
		cpl_matrix_get(fluxcorRsmooth[iBase],iRow,iWave)*cpl_matrix_get(fluxcorIsmooth[iBase],iRow,iWave+1);
	      cpl_vector_set(gd[iBase], iRow, atan2(fooI,fooR)*((lambdag*resolg)/(2*CPL_MATH_PI)));	  
	    }
	}
    }

  // Computing Achromatic Phase Term
  size_kernel=3000;
  pac=cpl_matrix_new(nRow_gra,NB_BASE);
  fluxcorRsmoothB=cpl_malloc(NB_BASE*sizeof(cpl_matrix*));
  fluxcorIsmoothB=cpl_malloc(NB_BASE*sizeof(cpl_matrix*));
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      fluxcorRsmoothB[iBase]=cpl_matrix_new(nRow_gra,NB_WAVELENGTH);
      fluxcorIsmoothB[iBase]=cpl_matrix_new(nRow_gra,NB_WAVELENGTH);
    } 
  vec_in=cpl_vector_new(nRow_gra);
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      for(int iWave=0;iWave<NB_WAVELENGTH;iWave++)
	{
	  factor=2*CPL_MATH_PI/lambda_gra[iWave];
	  for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	    {
	      fooR=cpl_matrix_get(fluxcorR[iBase],iFrame,iWave)*cos(cpl_vector_get(gd[iBase],iFrame)*factor)+
		cpl_matrix_get(fluxcorI[iBase],iFrame,iWave)*sin(cpl_vector_get(gd[iBase],iFrame)*factor);
	      cpl_vector_set(vec_in,iFrame,fooR);
	    }
	  vec_out=cpl_vector_filter_lowpass_create(vec_in,CPL_LOWPASS_LINEAR,size_kernel/2);
	  for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	    {
	      cpl_matrix_set(fluxcorRsmoothB[iBase],iFrame,iWave,cpl_vector_get(vec_out,iFrame));
	    }
	  cpl_vector_delete(vec_out);
	  for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	    {
	      fooI=cpl_matrix_get(fluxcorI[iBase],iFrame,iWave)*cos(cpl_vector_get(gd[iBase],iFrame)*factor)-
		cpl_matrix_get(fluxcorR[iBase],iFrame,iWave)*sin(cpl_vector_get(gd[iBase],iFrame)*factor);
	      cpl_vector_set(vec_in,iFrame,fooI);
	    }
	  vec_out=cpl_vector_filter_lowpass_create(vec_in,CPL_LOWPASS_LINEAR,size_kernel/2);
	  for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	    {
	      cpl_matrix_set(fluxcorIsmoothB[iBase],iFrame,iWave,cpl_vector_get(vec_out,iFrame));
	    }
	  cpl_vector_delete(vec_out);
	}
      for(int iFrame=0;iFrame<nRow_gra;iFrame++)
  	{
  	  fooR=0.;
  	  fooI=0.;
  	  for(int iWave=0;iWave<NB_WAVELENGTH;iWave++)
  	    {
  	      fooR+=cpl_matrix_get(fluxcorRsmoothB[iBase],iFrame,iWave);
  	      fooI+=cpl_matrix_get(fluxcorIsmoothB[iBase],iFrame,iWave);
  	    }
	  cpl_matrix_set(pac,iFrame,iBase,atan2(fooI,fooR));
  	}
      // Unwrap pac
      cpl_vector *vecin=NULL;
      cpl_vector *vecout=NULL;
      vecin=cpl_vector_new(nRow_gra);
      vecout=cpl_vector_new(nRow_gra);
      for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	{
	  cpl_vector_set(vecin,iFrame,cpl_matrix_get(pac,iFrame,iBase));
	}
      mat_vector_unwrap(vecin,vecout);
      for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	{
	  cpl_matrix_set(pac,iFrame,iBase,cpl_vector_get(vecout,iFrame));
	}
      cpl_vector_delete(vecin);
      cpl_vector_delete(vecout);
    }
  // Saving PA GRAVITY
  sprintf(fnameout,"GRA_%i.txt",bcdmode);
  fichier = fopen(fnameout,"w");
  for(int iFrame=0;iFrame<nRow_gra;iFrame++)
    {
      fprintf(fichier,"%e ",(cpl_vector_get(time_gra,iFrame)-cpl_vector_get(time_gra,0))*86400.);
      /* for(int iBase=0;iBase<NB_BASE;iBase++) */
      /* 	{ */
      /* 	  fprintf(fichier,"%e ",gammaK*(lambdag*1.E-6/(2*CPL_MATH_PI))*cpl_matrix_get(pac,iFrame,iBase)); */
      /* 	} */
      for(int iBase=0;iBase<NB_BASE;iBase++)
	{
	  fprintf(fichier,"%e ",gammaK*(1.e-4*cpl_vector_get(gd[iBase],iFrame)-(lambdag*1.e-4/(2*CPL_MATH_PI))*cpl_matrix_get(pdc,iFrame,iBase)));
	}
      fprintf(fichier,"\n");
    }
  fclose(fichier);


  
  // Arrange GRA4MAT data in the MATISSE order depending on BCD
  pac_copy=cpl_matrix_duplicate(pac);
  gd_copy=cpl_malloc(NB_BASE*sizeof(cpl_vector*));
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      gd_copy[iBase]=cpl_vector_duplicate(gd[iBase]);
    } 
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	{
	  cpl_matrix_set(pac,iFrame,iBase,
			 cpl_matrix_get(pac_copy,iFrame,indexBasefromGV2MT[bcdmode][iBase])*
			 signfromGV2MT[bcdmode][iBase]);
	  cpl_vector_set(gd[iBase],iFrame,cpl_vector_get(gd_copy[indexBasefromGV2MT[bcdmode][iBase]],iFrame)*
			 signfromGV2MT[bcdmode][iBase]);
	}
    } 
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      cpl_vector_delete(gd_copy[iBase]);
    }
  cpl_free(gd_copy);
  cpl_matrix_delete(pac_copy);

  // Predict GD and PA in N band
  Sigma=cpl_matrix_duplicate(pac);
  pdpredicted=cpl_matrix_duplicate(pac);
  gdpredicted=cpl_matrix_duplicate(pac);
  papredicted=cpl_matrix_duplicate(pac);
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	{
	  cpl_matrix_set(pdc,iFrame,iBase,(cpl_matrix_get(pac,iFrame,iBase)*lambdag/(2.*CPL_MATH_PI))
			 +cpl_vector_get(gd[iBase],iFrame));
	  cpl_matrix_set(Sigma,iFrame,iBase,gammaK*(cpl_matrix_get(pdc,iFrame,iBase)-
						    cpl_vector_get(gd[iBase],iFrame)));
	  cpl_matrix_set(pdpredicted,iFrame,iBase,(cpl_matrix_get(pdc,iFrame,iBase)+
						    (indexN-indexK)*cpl_matrix_get(Sigma,iFrame,iBase)));
	  cpl_matrix_set(gdpredicted,iFrame,iBase,(cpl_vector_get(gd[iBase],iFrame)+(indexN-indexK-(1./gammaN)+(1./gammaK))*
						    cpl_matrix_get(Sigma,iFrame,iBase)));
	  cpl_matrix_set(papredicted,iFrame,iBase,(cpl_matrix_get(pdpredicted,iFrame,iBase)-cpl_matrix_get(gdpredicted,iFrame,iBase))
			 *(2.*CPL_MATH_PI/waveN0));
	}
    }
  // Interpolate on MATISSE frames
  cpl_bivector *fref=NULL;
  cpl_bivector *fout=NULL;
  cpl_vector *vecout=NULL;
  cpl_vector *vecref=NULL;
  int nRow_mt=cpl_vector_get_size(time_mt);
  cpl_vector *time_mt_interp=NULL;
  
  time_mt_interp=cpl_vector_duplicate(time_mt);
  for(int iFrame=0;iFrame<nRow_mt;iFrame++)
    {
      if (cpl_vector_get(time_mt_interp,iFrame) < cpl_vector_get(time_gra,0))
	{
	  cpl_vector_set(time_mt_interp,iFrame,cpl_vector_get(time_gra,0));
	}
      if (cpl_vector_get(time_mt_interp,iFrame) > cpl_vector_get(time_gra,nRow_gra-1))
	{
	  cpl_vector_set(time_mt_interp,iFrame,cpl_vector_get(time_gra,nRow_gra-1));
	}
    }
      
  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      //PA
      vecref=cpl_vector_new(nRow_gra);
      for(int iFrame=0;iFrame<nRow_gra;iFrame++)
	{
	  cpl_vector_set(vecref,iFrame,cpl_matrix_get(papredicted,iFrame,iBase));
	}
      fref=cpl_bivector_wrap_vectors(time_gra,vecref);
      vecout=cpl_vector_new(nRow_mt);
      fout=cpl_bivector_wrap_vectors(time_mt_interp,vecout);
      cpl_bivector_interpolate_linear(fout,fref);
      vecout=cpl_bivector_get_y(fout);
      for(int iFrame=0;iFrame<nRow_mt;iFrame++)
      	{
      	  cpl_matrix_set(paestimated,iFrame,iBase,cpl_vector_get(vecout,iFrame));
      	}
      cpl_bivector_unwrap_vectors(fref);
      cpl_bivector_unwrap_vectors(fout);
      cpl_vector_delete(vecout);

      //GD
      for(int iFrame=0;iFrame<nRow_gra;iFrame++)
      	{
      	  cpl_vector_set(vecref,iFrame,cpl_matrix_get(gdpredicted,iFrame,iBase));
      	}
      fref=cpl_bivector_wrap_vectors(time_gra,vecref);
      vecout=cpl_vector_new(cpl_vector_get_size(time_mt));
      fout=cpl_bivector_wrap_vectors(time_mt_interp,vecout);
      cpl_bivector_interpolate_linear(fout,fref);
      vecout=cpl_bivector_get_y(fout);
      for(int iFrame=0;iFrame<nRow_mt;iFrame++)
      	{
      	  cpl_matrix_set(gdestimated,iFrame,iBase,cpl_vector_get(vecout,iFrame));
      	}
      cpl_bivector_unwrap_vectors(fref);
      cpl_bivector_unwrap_vectors(fout);
      cpl_vector_delete(vecref);
      cpl_vector_delete(vecout);
    }


  // Saving predicted GD and PA
  sprintf(fnameout,"K4N_%i.txt",bcdmode);
  fichier = fopen(fnameout,"w");
  for(int iFrame=0;iFrame<cpl_vector_get_size(time_mt);iFrame++)
    {
      fprintf(fichier,"%12.10f",cpl_vector_get(time_mt,iFrame));
      for(int iBase=0;iBase<NB_BASE;iBase++)
  	{
  	  fprintf(fichier," %f",cpl_matrix_get(paestimated,iFrame,iBase));
  	}
      for(int iBase=0;iBase<NB_BASE;iBase++)
  	{
  	  fprintf(fichier," %f",cpl_matrix_get(paestimated,iFrame,iBase));
  	}
      fprintf(fichier,"\n");
    }
  fclose(fichier);
  

  for(int iBase=0;iBase<NB_BASE;iBase++)
    {
      cpl_matrix_delete(fluxcorRsmooth[iBase]);
      cpl_matrix_delete(fluxcorIsmooth[iBase]);
      cpl_matrix_delete(fluxcorRsmoothB[iBase]);
      cpl_matrix_delete(fluxcorIsmoothB[iBase]);
      cpl_vector_delete(gd[iBase]);
    }
  cpl_free(fluxcorRsmooth);
  cpl_free(fluxcorIsmooth);
  cpl_free(fluxcorRsmoothB);
  cpl_free(fluxcorIsmoothB);
  cpl_free(gd);
  cpl_vector_delete(vec_in);
  cpl_matrix_delete(pdc);
  cpl_matrix_delete(pac);
  cpl_matrix_delete(Sigma);
  cpl_matrix_delete(papredicted);
  cpl_matrix_delete(pdpredicted);
  cpl_matrix_delete(gdpredicted);
  cpl_vector_delete(time_mt_interp);
}

void mat_vector_unwrap(cpl_vector *in, cpl_vector *out) {
  int len = cpl_vector_get_size(in);
  cpl_vector_set(out,0,cpl_vector_get(in,0));
  for (int i = 1; i < len; i++) {
    float d = cpl_vector_get(in,i) - cpl_vector_get(in,i-1);
    d = d > M_PI ? d - 2*M_PI : (d < -M_PI ? d + 2*M_PI : d);
    cpl_vector_set(out,i,cpl_vector_get(out,i-1)+d);
  }
}

void mat_add_generic_qc(cpl_propertylist *keywords, cpl_propertylist *qc) {
  if (cpl_propertylist_has(keywords,"ESO DET NAME"))
    {
      if (strcmp(cpl_propertylist_get_string(keywords,"ESO DET NAME"),"MATISSE-N")==0)
	{
	  cpl_propertylist_append_string(qc, "ESO QC INS DISP",
				       cpl_propertylist_get_string(keywords,"ESO INS DIN ID"));
	  cpl_propertylist_append_string(qc, "ESO QC INS FILT",
				       cpl_propertylist_get_string(keywords,"ESO INS FIN ID"));
	  cpl_propertylist_append_string(qc, "ESO QC INS POLARIZER",
				       cpl_propertylist_get_string(keywords,"ESO INS PON ID"));
	  cpl_propertylist_append_string(qc, "ESO QC INS PHOT SLIDER",
				       cpl_propertylist_get_string(keywords,"ESO INS PIN ID"));
	  cpl_propertylist_append_string(qc, "ESO QC INS SPAT FILTER",
				       cpl_propertylist_get_string(keywords,"ESO INS SFN ID"));
	  
	  if (cpl_propertylist_has(keywords,"ESO INS FIN ENC"))
	    {
	      cpl_propertylist_append_int(qc, "ESO QC INS FILT ENC",
					  cpl_propertylist_get_int(keywords,"ESO INS FIN ENC"));
	    }
	  if (cpl_propertylist_has(keywords,"ESO INS DIN ENC"))
	    {
	      cpl_propertylist_append_int(qc, "ESO QC INS DISP ENC",
					  cpl_propertylist_get_int(keywords,"ESO INS DIN ENC"));
	    }
	  cpl_propertylist_append_string(qc, "ESO QC DET1 NAME","");
	  cpl_propertylist_append_string(qc, "ESO QC DET2 NAME",
				       cpl_propertylist_get_string(keywords,"ESO DET NAME"));	  
	}
      else
	{
	  cpl_propertylist_append_string(qc, "ESO QC INS DISP",
				       cpl_propertylist_get_string(keywords,"ESO INS DIL ID"));
	  cpl_propertylist_append_string(qc, "ESO QC INS FILT",
				       cpl_propertylist_get_string(keywords,"ESO INS FIL ID"));
	  cpl_propertylist_append_string(qc, "ESO QC INS POLARIZER",
				       cpl_propertylist_get_string(keywords,"ESO INS POL ID"));
	  cpl_propertylist_append_string(qc, "ESO QC INS PHOT SLIDER",
				       cpl_propertylist_get_string(keywords,"ESO INS PIL ID"));
	  cpl_propertylist_append_string(qc, "ESO QC INS SPAT FILTER",
				       cpl_propertylist_get_string(keywords,"ESO INS SFL ID"));
	  if (cpl_propertylist_has(keywords,"ESO INS FIL ENC"))
	    {
	      cpl_propertylist_append_int(qc, "ESO QC INS FILT ENC",
					  cpl_propertylist_get_int(keywords,"ESO INS FIL ENC"));
	    }
	  if (cpl_propertylist_has(keywords,"ESO INS DIL ENC"))
	    {
	      cpl_propertylist_append_int(qc, "ESO QC INS DISP ENC",
					  cpl_propertylist_get_int(keywords,"ESO INS DIL ENC"));
	    }
	  cpl_propertylist_append_string(qc, "ESO QC DET1 NAME",
				       cpl_propertylist_get_string(keywords,"ESO DET NAME"));	  
	  cpl_propertylist_append_string(qc, "ESO QC DET2 NAME","");
	}
    }
}

void mat_add_qc_raw_estimates(cpl_propertylist *keywords, cpl_parameterlist *parlist) {
	// QC AMBI  TRANS
	if (cpl_propertylist_has(keywords,"ESO OBS AMBI TRANS") == 1)
	{
		if (strcmp(cpl_propertylist_get_string(keywords,"ESO OBS AMBI TRANS"),"1PHO") == 0)
		{
			cpl_propertylist_append_int(keywords,"ESO QC AMBI TRANS",1);
		}
		else if (strcmp(cpl_propertylist_get_string(keywords,"ESO OBS AMBI TRANS"),"1CLR") == 0)
		{
			cpl_propertylist_append_int(keywords,"ESO QC AMBI TRANS",2);
		}
		else if (strcmp(cpl_propertylist_get_string(keywords,"ESO OBS AMBI TRANS"),"3THN") == 0)
		{
			cpl_propertylist_append_int(keywords,"ESO QC AMBI TRANS",3);
		}
		else if (strcmp(cpl_propertylist_get_string(keywords,"ESO OBS AMBI TRANS"),"4THK") == 0)
		{
			cpl_propertylist_append_int(keywords,"ESO QC AMBI TRANS",4);
		}
	}
	// Depending on corrFlux parameter
	// QC FLAG FLUX
	// QC FLAG FLUXSNR
	cpl_parameter *correlatedFlux = NULL;
	correlatedFlux=cpl_parameterlist_find(parlist,
			"matisse.mat_proc_incoher.corrFlux");
	if ( cpl_parameter_get_bool(correlatedFlux) )
	{
		cpl_propertylist_append_bool(keywords,"ESO QC CORR FLUX",1);
		cpl_propertylist_append_int(keywords,"ESO QC FLAG FLUX",2);
		cpl_propertylist_append_int(keywords,"ESO QC FLAG FLUXSNR",2);
	}
	else
	{
		cpl_propertylist_append_bool(keywords,"ESO QC CORR FLUX",0);
		if (cpl_propertylist_has(keywords,"ESO QC DET1 FLUX IP1") == 1 ||
				cpl_propertylist_has(keywords,"ESO QC DET2 FLUX IP1") == 1)
		{
			if (cpl_propertylist_has(keywords,"ESO QC DET1 FLUX IP1") == 1	)
			{
				if (cpl_propertylist_get_double(keywords,"ESO QC DET1 FLUX IP1") > 0 &&
						cpl_propertylist_get_double(keywords,"ESO QC DET1 FLUX IP3") > 0 &&
						cpl_propertylist_get_double(keywords,"ESO QC DET1 FLUX IP5") > 0 &&
						cpl_propertylist_get_double(keywords,"ESO QC DET1 FLUX IP7") > 0)
				{
					cpl_propertylist_append_int(keywords,"ESO QC FLAG FLUX",0);
				}
				else
				{
					cpl_propertylist_append_int(keywords,"ESO QC FLAG FLUX",10);
				}
			}
			else if (cpl_propertylist_has(keywords,"ESO QC DET2 FLUX IP1") == 1	)
			{
				if (cpl_propertylist_get_double(keywords,"ESO QC DET2 FLUX IP1") > 0 &&
						cpl_propertylist_get_double(keywords,"ESO QC DET2 FLUX IP3") > 0 &&
						cpl_propertylist_get_double(keywords,"ESO QC DET2 FLUX IP5") > 0 &&
						cpl_propertylist_get_double(keywords,"ESO QC DET2 FLUX IP7") > 0)
				{
					cpl_propertylist_append_int(keywords,"ESO QC FLAG FLUX",0);
				}
				else
				{
					cpl_propertylist_append_int(keywords,"ESO QC FLAG FLUX",10);
				}
			}
		}
		else
		{
			cpl_propertylist_append_int(keywords,"ESO QC FLAG FLUX",2);
			cpl_propertylist_append_int(keywords,"ESO QC FLAG FLUXSNR",2);
		}

		/*--------------------------------------------------------------------*/
		/* QC FLAG FLUXSNR */
		if(cpl_propertylist_has(keywords,"ESO QC CORR FLUX") &&
				cpl_propertylist_get_bool(keywords, "ESO QC CORR FLUX") == 1){

			if(cpl_propertylist_has(keywords,"ESO QC FLUX SNR") &&
					cpl_propertylist_get_double(keywords, "ESO QC FLUX SNR") >= 2){
				cpl_propertylist_append_int(keywords,"ESO QC FLAG FLUXSNR", 1);
			}
			if(cpl_propertylist_has(keywords,"ESO QC FLUX SNR") &&
					cpl_propertylist_get_double(keywords, "ESO QC FLUX SNR") < 2){
				cpl_propertylist_append_int(keywords,"ESO QC FLAG FLUXSNR", 11);
			}
		}

		if(cpl_propertylist_has(keywords,"ESO QC CORR FLUX") &&
				cpl_propertylist_get_bool(keywords, "ESO QC CORR FLUX") != 1){

			if(cpl_propertylist_has(keywords,"ESO QC FLUX SNR") &&
					cpl_propertylist_get_double(keywords, "ESO QC FLUX SNR") >= 2){
				cpl_propertylist_append_int(keywords,"ESO QC FLAG FLUXSNR", 0);
			}
			if(cpl_propertylist_has(keywords,"ESO QC FLUX SNR") &&
					cpl_propertylist_get_double(keywords, "ESO QC FLUX SNR") < 2){
				cpl_propertylist_append_int(keywords,"ESO QC FLAG FLUXSNR", 10);
			}
		}
		/*--------------------------------------------------------------------*/
	}
	// QC FLAG VISSNR
	if (cpl_propertylist_has(keywords,"ESO QC DET1 TFSQR1 MEAN") == 1 ||
			cpl_propertylist_has(keywords,"ESO QC DET2 TFSQR1 MEAN") == 1)
	{
		if (cpl_propertylist_get_double(keywords,"ESO QC TF SNR") >=2)
		{
			cpl_propertylist_append_int(keywords,"ESO QC FLAG VISSNR",0);
		}
		else
		{
			cpl_propertylist_append_int(keywords,"ESO QC FLAG VISSNR",10);
		}
	}
	else
	{
		if (cpl_propertylist_get_double(keywords,"ESO QC VIS SNR") >=2)
		{
			cpl_propertylist_append_int(keywords,"ESO QC FLAG VISSNR",1);
		}
		else
		{
			cpl_propertylist_append_int(keywords,"ESO QC FLAG VISSNR",11);
		}
	}
	// QC MAG
	if ( strcmp(cpl_propertylist_get_string(keywords,"ESO DET NAME"),"MATISSE-N") == 0)
	{
		cpl_propertylist_append_double(keywords,"ESO QC MAG",cpl_propertylist_get_double(keywords,"ESO PRO JSDC NMAG"));
	}
	else
	{
		cpl_propertylist_append_double(keywords,"ESO QC MAG",cpl_propertylist_get_double(keywords,"ESO PRO JSDC LMAG"));
	}
	// QC OBJ FLUX
	if ( strcmp(cpl_propertylist_get_string(keywords,"ESO DET NAME"),"MATISSE-N") == 0)
	{
		if (cpl_propertylist_has(keywords,"ESO SEQ TARG FLUX N"))
		{
			cpl_propertylist_append_double(keywords,"ESO QC OBJ FLUX",
					cpl_propertylist_get_double(keywords,"ESO SEQ TARG FLUX N"));
		}
		else
		{
			cpl_propertylist_append_double(keywords,"ESO QC OBJ FLUX",-99);
		}
	}
	else
	{
		if (cpl_propertylist_has(keywords,"ESO SEQ TARG FLUX L"))
		{
			cpl_propertylist_append_double(keywords,"ESO QC OBJ FLUX",
					cpl_propertylist_get_double(keywords,"ESO SEQ TARG FLUX L"));
		}
		else
		{
			cpl_propertylist_append_double(keywords,"ESO QC OBJ FLUX",-99);
		}
	}
}
